/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   DSM-CC SI support
 * @file    dsm_si.c
 * @date    24 April 2014
 * @author  Adam Sturtridge
 */
/*---includes for this file--------------------------------------------------*/
#include <string.h>

#include "stb_os.h"
#include "stb_filter.h"
#include "dsm_control.h"
#include "dm_debug.h"
#include "dsm_si.h"

/*---constant definitions for this file--------------------------------------*/

#define PAT_PID     0
#define PAT_TID     0
#define PMT_TID     2
#define TIMEOUT_PAT 700
#define TIMEOUT_PMT 1000
#define TIMEOUT_DEFAULT 0x7FFF

#define DSI_RQST_MAGIC 0xBA000000

#ifndef NDEBUG
#define dsmctrl dsi->ctrl
#endif

/*---local typedef structs for this file-------------------------------------*/

typedef struct s_tablerequest
{
   struct s_tablerequest *next;
   U16BIT pid;
   U16BIT xid;
   U8BIT tid;
   U8BIT version;
   U32BIT timeout;
} S_TableRequest;

typedef struct s_DsiInstance
{
   F_MemAlloc memAlloc;
   F_MemFree memFree;
   H_DsmControl ctrl;
   F_FILTER_CALLBACK cbfunc;
   void *mutex;
   S_TableRequest *tables;
   PIDFILT pflt;
   SECFILT sflt;
   U32BIT rid;
   DMXREF dmxref;
   U8BIT version;
} S_DsiInstance;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

/*------------------- Local function definitions ----------------------------*/

U32BIT TimeoutPeriod(U8BIT tid)
{
   switch (tid)
   {
      case PAT_TID:
         return TIMEOUT_PAT;

      case PMT_TID:
         return TIMEOUT_PMT;

      default:
         return TIMEOUT_DEFAULT;
   }
}

/**
 * @brief   Utility function to setup a section filter
 * @param   dmx demux reference
 * @param   pfid PID filter Id
 * @param   pfid Handle to allocated PID filter
 * @param   tid  Table Id(s) required
 * @param   tidMask Mask for tid byte
 * @param   tidExt Table Id Extension(s)
 * @param   tidExtMask Mask for tidExt bytes
 * @return  SECFILT  Handle to the allocated section filter.
 *                   INVALID_SECFILT - failure.
 */
static SECFILT SectionFilterAdd( DMXREF dmx, PIDFILT pfid, U8BIT tid,
   U8BIT tidMask, U16BIT tidExt, U16BIT tidExtMask )
{
   U8BIT match[SECFILT_MASK_SIZE];
   U8BIT mmask[SECFILT_MASK_SIZE];
   SECFILT sfid;
   U16BIT i;
   sfid = STB_DMXGrabSectFilter(dmx, pfid);
   if (sfid != INVALID_SECFILT)
   {
      match[0] = tid;
      match[1] = (U8BIT)(tidExt >> 8);
      match[2] = (U8BIT)(tidExt & 0x00ff);
      mmask[0] = tidMask;
      mmask[1] = (U8BIT)(tidExtMask >> 8);
      mmask[2] = (U8BIT)(tidExtMask & 0x00ff);
      for (i = 3; i != SECFILT_MASK_SIZE; i++)
      {
         match[i] = 0;
         mmask[i] = 0;
      }
      STB_DMXSetupSectFilter(dmx, sfid, match, mmask, 0, TRUE /*crc checked*/);
   }
   return sfid;
}

static void ReleaseSi(H_DsiInstance dsi)
{
   DBGPRINT(DM_SI, "pflt=%x", dsi->pflt)
   STB_DMXStopPIDFilter(dsi->dmxref, dsi->pflt);
   STB_DMXReleaseSectFilter(dsi->dmxref, dsi->sflt);
   dsi->sflt = INVALID_SECFILT;
   STB_DMXReleasePIDFilter(dsi->dmxref, dsi->pflt);
   dsi->pflt = INVALID_PIDFILT;
}

static BOOLEAN RequestSi(H_DsiInstance dsi, U16BIT pid, U8BIT tid, U16BIT xid, U16BIT xmsk)
{
   DMXREF dmxref;
   BOOLEAN ok;

   ASSERT(dsi->pflt == INVALID_PIDFILT);
   ASSERT(dsi->sflt == INVALID_SECFILT);

   dmxref = dsi->dmxref;
   dsi->pflt = STB_DMXGrabPIDFilter(dmxref, pid, dsi->cbfunc);
   if (dsi->pflt == INVALID_PIDFILT)
   {
      ERRPRINT("Grab PID filter fail")
      ok = FALSE;
   }
   else
   {
      dsi->sflt = SectionFilterAdd( dmxref, dsi->pflt, tid, 0xFF, xid, xmsk);
      if (dsi->sflt == INVALID_SECFILT)
      {
         ERRPRINT("Grab SEC filter fail")
         STB_DMXReleasePIDFilter(dmxref, dsi->pflt);
         dsi->pflt = INVALID_PIDFILT;
         ok = FALSE;
      }
      else
      {
         STB_DMXStartPIDFilter(dmxref, dsi->pflt);
         DBGPRINT(DM_SI, "tid=%d xid=%d pflt=%x", tid, xid, dsi->pflt)
         ok = TRUE;
      }
   }
   return ok;
}

S_TableRequest* FindTableRqst(H_DsiInstance dsi, U16BIT xid, U8BIT tid)
{
   S_TableRequest *rqst;
   rqst = dsi->tables;
   while (rqst != NULL && (rqst->xid != xid || rqst->tid != tid))
   {
      rqst = rqst->next;
   }
   return rqst;
}

/*------------------- Global function definitions ---------------------------*/


/**
 * @brief   Create instance of DSI
 * @param   cbfunc Function to process SI section data
 * @param   config Configuration of DSM-CC
 * @return  DSI Instance handle
 */
H_DsiInstance DSI_CreateInstance(H_DsmControl ctrl, F_FILTER_CALLBACK cbfunc, S_DsmccConfig *config)
{
   H_DsiInstance dsi;
   dsi = config->memAlloc(sizeof(S_DsiInstance));
   if (dsi != NULL)
   {
      dsi->mutex = STB_OSCreateMutex();
      if (dsi->mutex == NULL)
      {
         ERRPRINT("mutex create fail")
         config->memFree(dsi);
         dsi = NULL;
      }
      else
      {
         dsi->memAlloc = config->memAlloc;
         dsi->memFree = config->memFree;
         dsi->cbfunc = cbfunc;
         dsi->tables = NULL;
         dsi->pflt = INVALID_PIDFILT;
         dsi->sflt = INVALID_SECFILT;
         dsi->ctrl = ctrl;
      }
   }
   return dsi;
}

/**
 * @brief   Destroy instance created by SIQ_CreateInstance
 * @param   H_DsiInstance dsi DSI instance handle.
 * @return
 */
void DSI_DestroyInstance( H_DsiInstance dsi )
{
   STB_OSDeleteMutex(dsi->mutex);
   dsi->memFree( dsi );
}

/**
 * @brief   Start getting SI data from Demux
 * @param   dsi DSI instance handle.
 * @param   transId Transport stream ID
 * @param   dmxref Demux reference handle
 * @return
 */
BOOLEAN DSI_Start( H_DsiInstance dsi, U16BIT transId, DMXREF dmxref )
{
   dsi->dmxref = dmxref;
   return DSI_RequestTable( dsi, PAT_PID, transId, PAT_TID ) ? TRUE : FALSE;
}

/**
 * @brief   Stop getting SI data from Demux
 * @param   H_DsiInstance  dsi        DSI instance handle.
 * @return
 */
void DSI_Stop( H_DsiInstance dsi )
{
   S_TableRequest *rqst, *next;
   if (dsi->pflt != INVALID_PIDFILT)
   {
      ReleaseSi(dsi);
      dsi->rid = 0;
   }
   STB_OSMutexLock(dsi->mutex);
   rqst = dsi->tables;
   while (rqst != NULL)
   {
      next = rqst->next;
      dsi->memFree(rqst);
      rqst = next;
   }
   dsi->tables = NULL;
   STB_OSMutexUnlock(dsi->mutex);
}

/**
 * @brief   Request Table data. It supplies Table soon after the
 *          call to this function, and then, whenever the Table
 *          version changes.
 *          This state continues until DSI_CancelTable is called.
 * @param   dsi DSI instance handle
 * @param   pid PID for this table
 * @param   xid Table extension ID
 * @param   tid Table ID (e.g. PMT is 0x02)
 * @return  zero is failure, non-zero is success
 */
U32BIT DSI_RequestTable( H_DsiInstance dsi, U16BIT pid, U16BIT xid, U8BIT tid )
{
   S_TableRequest *rqst;
   U32BIT result;

   STB_OSMutexLock(dsi->mutex);
   rqst = FindTableRqst(dsi, xid, tid);
   STB_OSMutexUnlock(dsi->mutex);
   if (rqst != NULL)
   {
      rqst->version = 0xFF;
      rqst->timeout = STB_OSGetClockMilliseconds();
      result = DSI_RQST_MAGIC | (((U32BIT)tid) << 16) | xid;
   }
   else
   {
      rqst = dsi->memAlloc(sizeof(S_TableRequest));
      if (rqst == NULL)
      {
         ERRPRINT("memory fail for serviceId=%d", xid);
         result = 0;
      }
      else
      {
         rqst->xid = xid;
         rqst->pid = pid;
         rqst->tid = tid;
         rqst->version = 0xFF;
         rqst->timeout = STB_OSGetClockMilliseconds();
         STB_OSMutexLock(dsi->mutex);
         rqst->next = dsi->tables;
         dsi->tables = rqst;
         STB_OSMutexUnlock(dsi->mutex);
         result = DSI_RQST_MAGIC | (((U32BIT)tid) << 16) | xid;
      }
   }
   if (result && dsi->pflt == INVALID_PIDFILT)
   {
      if ( RequestSi(dsi, pid, tid, xid, 0xFFFF) )
      {
         dsi->rid = result;
      }
      rqst->timeout += TimeoutPeriod(tid);
   }
   return result;
}

/**
 * @brief   Cancel request for table data made by DSI_RequestTable()
 * @param   dsi DSI instance handle
 * @param   rid Request Id returned by DSI_RequestTable()
 * @return  void
 */
void DSI_CancelTable( H_DsiInstance dsi, U32BIT rid )
{
   S_TableRequest **pprqst, *rqst;
   U16BIT xid;
   U8BIT tid;
   ASSERT((rid & 0xFF000000) == DSI_RQST_MAGIC);
   if (rid == dsi->rid)
   {
      ASSERT(dsi->pflt != INVALID_PIDFILT);
      ReleaseSi(dsi);
      dsi->rid = 0;
   }
   tid = (rid >> 16) & 0xFF;
   xid = rid & 0xFFFF;
   STB_OSMutexLock(dsi->mutex);
   pprqst = &(dsi->tables);
   while (*pprqst != NULL)
   {
      rqst = *pprqst;
      if (rqst->xid == xid && rqst->tid == tid)
      {
         *pprqst = rqst->next;
         dsi->memFree(rqst);
         break;
      }
      pprqst = &(rqst->next);
   }
   STB_OSMutexUnlock(dsi->mutex);
}

/**
 * @brief   Check outstanding PMT (and PAT) requests to see whether
 *          any need to be refreshed
 * @param   dsi DSI instance handle
 * @return  U32BIT time when need to call this function again
 */
U32BIT DSI_CheckRequests( H_DsiInstance dsi, U32BIT now )
{
   U32BIT tdiff, least, tout;
   S_TableRequest *rqst;
   U16BIT pid, xid;
   U8BIT tid;
   DBGPRINT(DM_SI, "rid=%x pflt=%x", dsi->rid, dsi->pflt)
   if (!dsi->rid && dsi->pflt != INVALID_PIDFILT)
   {
      ReleaseSi(dsi);
   }
   if (dsi->pflt == INVALID_PIDFILT)
   {
      STB_OSMutexLock(dsi->mutex);
      rqst = dsi->tables;
      while (rqst != NULL)
      {
         if (rqst->timeout <= now)
         {
            pid = rqst->pid;
            xid = rqst->xid;
            tid = rqst->tid;
            break;
         }
         rqst = rqst->next;
      }
      STB_OSMutexUnlock(dsi->mutex);
      if (rqst != NULL)
      {
         if (RequestSi(dsi, pid, tid, xid, 0xFFFF))
         {
            dsi->rid = DSI_RQST_MAGIC | (((U32BIT)tid) << 16) | xid;
            rqst->timeout = now + TimeoutPeriod(tid);
         }
      }
   }
   least = TIMEOUT_DEFAULT;
   STB_OSMutexLock(dsi->mutex);
   rqst = dsi->tables;
   while (rqst != NULL)
   {
      tdiff = rqst->timeout - now;
      tout = TimeoutPeriod(rqst->tid);
      if (least > tdiff)
      {
         least = tdiff;
      }
      else if (tdiff > tout)
      {
         /*time wrap*/
         rqst->timeout = now + tout;
      }
      rqst = rqst->next;
   }
   STB_OSMutexUnlock(dsi->mutex);
   return now + least;
}

/**
 * @brief   Check whether SI table section data is required
 * @param   dsi DSI instance handle
 * @param   data pointer to section data (starting at table Id byte
 * @param   pXid pointer to return table extension
 * @param   pVer pointer to return table version
 * @return  BOOLEAN - TRUE if required, otherwise FALSE
 */
BOOLEAN DSI_RequireTable( H_DsiInstance dsi, U8BIT *hdr, U16BIT *pXid, U8BIT *pVer )
{
   S_TableRequest *rqst;
   BOOLEAN result;
   U16BIT xid;
   U8BIT tid;
   U8BIT version;
   tid = hdr[0];
   version = (hdr[5] >> 1) & 0x1f;
   xid = hdr[3] << 8 | hdr[4];

   STB_OSMutexLock(dsi->mutex);
   rqst = FindTableRqst(dsi, xid, tid);
   if (rqst != NULL && rqst->version != version)
   {
      DBGPRINT(DM_SI, "PMT xid=%d pflt=%x ver=%d", xid, dsi->pflt, version)
      result = TRUE;
      *pXid = xid;
      *pVer = version;
      rqst->version = version;
      rqst->timeout = STB_OSGetClockMilliseconds() + TimeoutPeriod(tid);
   }
   else
   {
      result = FALSE;
   }
   dsi->rid = 0;
   STB_OSMutexUnlock(dsi->mutex);

   return result;
}

