/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Internal header for DSM-CC si support
 * @file    dsm_si.h
 * @date    24 April 2014
 * @author  Adam Sturtridge
 */
#ifndef _DSM_SI_H
#define _DSM_SI_H

typedef struct s_DsiInstance *H_DsiInstance;

/**
 * @brief   Create instance of DSI
 * @param   cbfunc Function to process SI section data
 * @param   config Configuration of DSM-CC
 * @return  DSI Instance handle
 */
H_DsiInstance DSI_CreateInstance(H_DsmControl dsmctrl, F_FILTER_CALLBACK cbfunc, S_DsmccConfig *config);

/**
 * @brief   Destroy instance created by SIQ_CreateInstance
 * @param   H_DsiInstance dsi DSI instance handle.
 * @return
 */
void DSI_DestroyInstance( H_DsiInstance dsi );

/**
 * @brief   Start getting SI data from Demux
 * @param   H_DsiInstance  dsi        DSI instance handle.
 * @param   DMXREF       dmxref       Demux reference handle
 * @return
 */
BOOLEAN DSI_Start( H_DsiInstance dsi, U16BIT transId, DMXREF dmxref );

/**
 * @brief   Stop getting SI data from Demux
 * @param   H_DsiInstance  dsi        DSI instance handle.
 * @return
 */
void DSI_Stop( H_DsiInstance dsi );

/**
 * @brief   Request Table data. It supplies Table soon after the
 *          call to this function, and then, whenever the Table
 *          version changes.
 *          This state continues until DSI_CancelTable is called.
 * @param   dsi DSI instance handle
 * @param   pid PID for this table
 * @param   xid Table extension ID
 * @param   tid Table ID (e.g. PMT is 0x02)
 * @return  zero is failure, non-zero is success
 */
U32BIT DSI_RequestTable( H_DsiInstance dsi, U16BIT pid, U16BIT xid, U8BIT tid );

/**
 * @brief   Cancel request for table data made by DSI_RequestTable()
 * @param   dsi DSI instance handle
 * @param   rid Request Id returned by DSI_RequestTable()
 * @return  void
 */
void DSI_CancelTable( H_DsiInstance dsi, U32BIT rid );

/**
 * @brief   Check outstanding PMT (and PAT) requests to see whether
 *          any need to be refreshed
 * @param   dsi DSI instance handle
 * @return  U32BIT time when need to call this function again
 */
U32BIT DSI_CheckRequests( H_DsiInstance dsi, U32BIT now );

/**
 * @brief   Check whether SI table section data is required
 * @param   dsi DSI instance handle
 * @param   data pointer to section data (starting at table Id byte
 * @param   pXid pointer to return table extension
 * @param   pVer pointer to return table version
 * @return  BOOLEAN - TRUE if required, otherwise FALSE
 */
BOOLEAN DSI_RequireTable( H_DsiInstance dsi, U8BIT *data, U16BIT *pXid, U8BIT *pVer );

#endif /*_DSM_SI_H*/
