/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Database access public functions
 * @file    dba_nvm.c
 * @date    11/11/2013
 * @author  Ocean Blue
 */

//---includes for this file------------------------------------------------------------------------
#include <string.h>

#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"

#include "dba.h"
#include "dba_nvm.h"
#include "stbdbram.h"
#include "stbdbnvm.h"
#include "stbnvm.h"
#include "stbcsum.h"

#include "stbheap.h"
#include "stbuni.h"

//---constant definitions for this file-------------------------------------------------------------
#define FORMAT_VERSION_NUM       4
#define FORMAT_ID_STRING         "STB DB 01"
#define FORMAT_ID_STRING_SIZE    10

#define NUM_FIELDS(X)            (sizeof(X) / sizeof(FIELD_ACCESS))

/* Length of various strings held in the database */
#define DBA_GROUP_NAME_LEN       35
#define DBA_SERV_NAME_LEN        32
#define DBA_TIMER_NAME_LEN       255   /* Max length of an event name is 255 chars */
#define DBA_FAVLIST_NAME_LEN     128   /* Arbitrary length */
#define DBA_TIMER_ADDINFO_LEN    255
#define DBA_LNB_NAME_LEN         24

#define DBA_CRID_LEN             65

/* CI+ specific strings */
#define DBA_PROFILE_NAME_LEN     255

//---local typedefs, structs, enumerations for this file--------------------------------------------

typedef struct
{
   U8BIT id_str[FORMAT_ID_STRING_SIZE];
   U8BIT spare[4];
   U8BIT version;
   U8BIT checksum;
} DATABASE_FORMAT;

typedef struct
{
   U16BIT desc_size;
   U8BIT num_records;
   U8BIT padding[1];
} DATABASE_DESCRIPTOR;

typedef struct
{
   U8BIT record_id;
   U8BIT num_fields;
   U16BIT first_block;
} RECORD_DESCRIPTOR;

typedef struct
{
   U16BIT field_id;
   U16BIT field_offset;
   U16BIT field_size;
   U8BIT padding[2];
} FIELD_DESCRIPTOR;

typedef struct
{
   U16BIT field_id;
   E_DBA_FIELD_TYPE type;
} FIELD_TYPE;

typedef struct
{
   U16BIT field_id;
   U16BIT nvm_bit_offset;
   U16BIT nvm_bit_size;
   U16BIT ram_byte_offset;
   U16BIT ram_byte_size;
   U32BIT default_value;
} FIELD_ACCESS;

typedef struct
{
   U8BIT num_fields;
   const FIELD_ACCESS *field;
} RECORD_ACCESS;


//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)
static const FIELD_TYPE database_field_type[] =
{
   {DBA_FIELD_PARENT, UNUM_TYPE},
   {DBA_FIELD_REC_NAME, STR_TYPE},
   {DBA_FIELD_ORIG_NET_ID, UNUM_TYPE},
   {DBA_FIELD_NET_ID, UNUM_TYPE},
   {DBA_FIELD_TRANSPORT_ID, UNUM_TYPE},
   {DBA_FIELD_SERVICE_ID, UNUM_TYPE},
   {DBA_FIELD_VERSION, UNUM_TYPE},

   {DBA_FIELD_LNB_TYPE, UNUM_TYPE},
   {DBA_FIELD_LNB_22K, UNUM_TYPE},
   {DBA_FIELD_LNB_12V, UNUM_TYPE},
   {DBA_FIELD_LNB_PULSEPOSN, UNUM_TYPE},
   {DBA_FIELD_LNB_DISPOSN, UNUM_TYPE},
   {DBA_FIELD_LNB_DISTONE, UNUM_TYPE},
   {DBA_FIELD_LNB_DISCSWITCH, UNUM_TYPE},
   {DBA_FIELD_LNB_DISUSWITCH, UNUM_TYPE},
   {DBA_FIELD_LNB_DISSMATV, UNUM_TYPE},
   {DBA_FIELD_LNB_DISREPEAT, UNUM_TYPE},
   {DBA_FIELD_LNB_UNICABLEFREQ, UNUM_TYPE},
   {DBA_FIELD_LNB_UNICABLECHAN, UNUM_TYPE},
   {DBA_FIELD_LNB_POWER, UNUM_TYPE},
   {DBA_FIELD_LNB_NAME, STR_TYPE},

   {DBA_FIELD_SAT_DISH, UNUM_TYPE},
   {DBA_FIELD_SAT_LONGWE, UNUM_TYPE},
   {DBA_FIELD_SAT_LONGPOS, UNUM_TYPE},

   {DBA_FIELD_TRAN_FREQ, UNUM_TYPE},
   {DBA_FIELD_TRAN_SRATE, UNUM_TYPE},
   {DBA_FIELD_TRAN_SIGNAL_STRENGTH, UNUM_TYPE},
   {DBA_FIELD_TRAN_SIGNAL_QUALITY, UNUM_TYPE},

   {DBA_FIELD_STRAN_POL, UNUM_TYPE},
   {DBA_FIELD_STRAN_FEC, UNUM_TYPE},
   {DBA_FIELD_STRAN_DVBS2, UNUM_TYPE},
   {DBA_FIELD_STRAN_MODULATION, UNUM_TYPE},

   {DBA_FIELD_TTRAN_MODE, UNUM_TYPE},
   {DBA_FIELD_TTRAN_TERR_TYPE, UNUM_TYPE},
   {DBA_FIELD_TTRAN_PLP_ID, UNUM_TYPE},
   {DBA_FIELD_TTRAN_BWIDTH, UNUM_TYPE},

   {DBA_FIELD_CTRAN_MODE, UNUM_TYPE},

   {DBA_FIELD_SERV_ID, UNUM_TYPE},
   {DBA_FIELD_SERV_TYPE, UNUM_TYPE},
   {DBA_FIELD_SERV_LCN, UNUM_TYPE},
   {DBA_FIELD_SERV_REQ_LCN, UNUM_TYPE},
   {DBA_FIELD_SERV_HIDDEN, UNUM_TYPE},
   {DBA_FIELD_SERV_SELECTABLE, UNUM_TYPE},
   {DBA_FIELD_SERV_LOCKED, UNUM_TYPE},
   {DBA_FIELD_SERV_SCHED_DISABLED, UNUM_TYPE},
   {DBA_FIELD_SERV_NOWNEXT_DISABLED, UNUM_TYPE},
   {DBA_FIELD_SERV_FAV_GROUPS, UNUM_TYPE},
   {DBA_FIELD_SERV_FREESAT_ID, UNUM_TYPE},
   {DBA_FIELD_SERV_REGION_ID, UNUM_TYPE},
   {DBA_FIELD_SERV_LCN_EDITABLE, UNUM_TYPE},
   {DBA_FIELD_SERV_DELETED, UNUM_TYPE},

   {DBA_FIELD_TIMER_CRID, STR_TYPE},
   {DBA_FIELD_TIMER_DISKID, UNUM_TYPE},
   {DBA_FIELD_TIMER_OTHERCRID, STR_TYPE},
   {DBA_FIELD_TIMER_HANDLE, UNUM_TYPE},
   {DBA_FIELD_TIMER_STARTTIME, UNUM_TYPE},
   {DBA_FIELD_TIMER_DURATION, UNUM_TYPE},
   {DBA_FIELD_TIMER_TYPE, UNUM_TYPE},
   {DBA_FIELD_TIMER_FREQUENCY, UNUM_TYPE},
   {DBA_FIELD_TIMER_RAMPVOLUME, UNUM_TYPE},
   {DBA_FIELD_TIMER_EVENTID, UNUM_TYPE},
   {DBA_FIELD_TIMER_MISSED, UNUM_TYPE},
   {DBA_FIELD_TIMER_EVENT_TRIGGERED, UNUM_TYPE},
   {DBA_FIELD_TIMER_NOTIFY_TIME, UNUM_TYPE},
   {DBA_FIELD_TIMER_ADDITIONAL_INFO, STR_TYPE},
   {DBA_FIELD_TIMER_START_PADDING, UNUM_TYPE},
   {DBA_FIELD_TIMER_END_PADDING, UNUM_TYPE},
   {DBA_FIELD_TIMER_DO_NOT_DELETE, UNUM_TYPE},

   {DBA_FIELD_CRID_SERIES, UNUM_TYPE},
   {DBA_FIELD_CRID_RECOMMENDED, UNUM_TYPE},
   {DBA_FIELD_CRID_EIT_DATE, UNUM_TYPE},
   {DBA_FIELD_CRID_DO_NOT_DELETE, UNUM_TYPE},

   {DBA_FIELD_FAVLIST_ID, UNUM_TYPE},
   {DBA_FIELD_FAVLIST_INDEX, UNUM_TYPE},
   {DBA_FIELD_FAVLIST_USER_DATA, UNUM_TYPE},

   {DBA_FIELD_BAND_POLARITY, UNUM_TYPE},
   {DBA_FIELD_BAND_MIN_FREQUENCY, UNUM_TYPE},
   {DBA_FIELD_BAND_MAX_FREQUENCY, UNUM_TYPE},
   {DBA_FIELD_BAND_LOCAL_OSC_FREQUENCY, UNUM_TYPE},
   {DBA_FIELD_BAND_LNB_VOLTAGE, UNUM_TYPE},
   {DBA_FIELD_BAND_22_KHZ, UNUM_TYPE}
};

// String fields must start on a byte boundary
// String fields MUST be defined in RAM
// format is : FIELD NAME, NVM BIT OFFSET, NVM BIT SIZE, RAM BYTE OFFSET, RAM BYTE SIZE, DEFAULT VALUE

/* LNB record */
static const FIELD_ACCESS nvm_lnb_rec[] =
{
   {DBA_FIELD_LNB_TYPE, 0, 2, 0, 0, 0},
   {DBA_FIELD_LNB_POWER, 2, 2, 0, 0, 0},
   {DBA_FIELD_LNB_22K, 4, 1, 0, 0, 0},
   {DBA_FIELD_LNB_12V, 5, 1, 0, 0, 0},
   {DBA_FIELD_LNB_PULSEPOSN, 6, 1, 0, 0, 0},
   {DBA_FIELD_LNB_DISPOSN, 7, 1, 0, 0, 0},
   {DBA_FIELD_LNB_DISTONE, 8, 2, 0, 0, 0},
   {DBA_FIELD_LNB_DISCSWITCH, 10, 3, 0, 0, 0},
   {DBA_FIELD_LNB_DISUSWITCH, 13, 5, 0, 0, 0},
   {DBA_FIELD_LNB_DISSMATV, 18, 1, 0, 0, 0},
   {DBA_FIELD_LNB_DISREPEAT, 19, 4, 0, 0, 0},
   {DBA_FIELD_LNB_UNICABLEFREQ, 23, 32, 0, 0, 0},
   {DBA_FIELD_LNB_UNICABLECHAN, 55, 3, 0, 0, 0},
   {DBA_FIELD_LNB_NAME, 64, (DBA_LNB_NAME_LEN * 8), 0, DBA_LNB_NAME_LEN, 0}
   /* 2 blocks used - 64 bits spare here */
};

/* LNB band record */
static const FIELD_ACCESS nvm_lnb_band_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_BAND_POLARITY, 16, 2, 0, 0, 0},
   {DBA_FIELD_BAND_MIN_FREQUENCY, 18, 16, 0, 0, 0},
   {DBA_FIELD_BAND_MAX_FREQUENCY, 34, 16, 0, 0, 0},
   {DBA_FIELD_BAND_LOCAL_OSC_FREQUENCY, 50, 16, 0, 0, 0},
   {DBA_FIELD_BAND_LNB_VOLTAGE, 66, 2, 0, 0, 0},
   {DBA_FIELD_BAND_22_KHZ, 68, 1, 0, 0, 0}
   /* 1 block used - 91 bits spare here */
};

/* Satellite record */
static const FIELD_ACCESS nvm_sat_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_SAT_DISH, 16, 16, 0, 0, 0},
   {DBA_FIELD_SAT_LONGWE, 32, 1, 0, 0, 0},
   {DBA_FIELD_SAT_LONGPOS, 33, 15, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 48, (DBA_GROUP_NAME_LEN * 8), 0, DBA_GROUP_NAME_LEN, 0}
   /* 3 blocks used - 152 bits spare */
};

/* Network record */
static const FIELD_ACCESS nvm_net_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_NET_ID, 16, 16, 0, 0, 0},
   {DBA_FIELD_VERSION, 32, 8, 0, 0, 0},
   {DBA_FIELD_ORIG_NET_ID, 40, 16, 0, 0, 0},
   {DBA_FIELD_PROFILE_TYPE, 56, 2, 0, 0, 0},
   {DBA_FIELD_PROFILE_CAM_ID, 58, 32, 0, 0, 0},
   {DBA_FIELD_OPERATOR_SEARCH, 90, 1, 0, 0, 0},
   {DBA_FIELD_OP_SEARCH_DATE, 91, 16, 0, 0, 0},
   {DBA_FIELD_OP_SEARCH_TIME, 107, 16, 0, 0, 0},
   {DBA_FIELD_FAVLIST_ID, 123, 8, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 136, (DBA_GROUP_NAME_LEN * 8), 0, DBA_GROUP_NAME_LEN, 0},
   {DBA_FIELD_PROFILE_NAME, 136 + (DBA_GROUP_NAME_LEN * 8), DBA_PROFILE_NAME_LEN * 8,
    DBA_GROUP_NAME_LEN, DBA_PROFILE_NAME_LEN, 0}
   /* 16 blocks used - 72 bits spare */
};

/* Satellite transponder record */
static const FIELD_ACCESS nvm_stran_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_TRAN_FREQ, 16, 32, 0, 0, 0},
   {DBA_FIELD_TRAN_SRATE, 48, 16, 0, 0, 0},
   {DBA_FIELD_STRAN_POL, 64, 2, 0, 0, 0},
   {DBA_FIELD_STRAN_FEC, 66, 3, 0, 0, 0},
   {DBA_FIELD_TRANSPORT_ID, 69, 16, 0, 0, 0},
   {DBA_FIELD_ORIG_NET_ID, 85, 16, 0, 0, 0},
   {DBA_FIELD_STRAN_MODULATION, 101, 2, 0, 0, 0},
   {DBA_FIELD_STRAN_DVBS2, 103, 1, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_STRENGTH, 104, 8, 0, 0, 0},
   {DBA_FIELD_VERSION, 112, 8, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_QUALITY, 120, 8, 0, 0, 0}
};

/* Terrestrial transport record */
static const FIELD_ACCESS nvm_ttran_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_TRANSPORT_ID, 16, 16, 0, 0, 0},
   {DBA_FIELD_ORIG_NET_ID, 32, 16, 0, 0, 0},
   {DBA_FIELD_TRAN_FREQ, 48, 32, 0, 0, 0},
   {DBA_FIELD_TTRAN_MODE, 80, 2, 0, 0, 0},
   {DBA_FIELD_TTRAN_BWIDTH, 82, 3, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_STRENGTH, 85, 8, 0, 0, 0},
   {DBA_FIELD_TTRAN_TERR_TYPE, 93, 2, 0, 0, 0},
   {DBA_FIELD_TTRAN_PLP_ID, 95, 8, 0, 0, 0},
   {DBA_FIELD_VERSION, 103, 8, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_QUALITY, 111, 8, 0, 0, 0}
};

/* Cable transport record */
static const FIELD_ACCESS nvm_ctran_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_TRANSPORT_ID, 16, 16, 0, 0, 0},
   {DBA_FIELD_ORIG_NET_ID, 32, 16, 0, 0, 0},
   {DBA_FIELD_TRAN_FREQ, 48, 32, 0, 0, 0},
   {DBA_FIELD_TRAN_SRATE, 80, 16, 0, 0, 0},
   {DBA_FIELD_CTRAN_MODE, 96, 3, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_STRENGTH, 99, 8, 0, 0, 0},
   {DBA_FIELD_VERSION, 107, 8, 0, 0, 0},
   {DBA_FIELD_TRAN_SIGNAL_QUALITY, 115, 8, 0, 0, 0}
};

/* Service record (used for satellite, cable and terrestrial) */
static const FIELD_ACCESS nvm_serv_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},
   {DBA_FIELD_SERV_ID, 16, 16, 0, 0, 0},
   {DBA_FIELD_SERV_TYPE, 46, 8, 0, 0, 0},
   {DBA_FIELD_SERV_LCN, 32, 14, 0, 0, 0},
   {DBA_FIELD_SERV_REQ_LCN, 54, 16, 0, 0, 0},
   {DBA_FIELD_SERV_FAV_GROUPS, 70, 8, 0, 0, 0},
   {DBA_FIELD_SERV_HIDDEN, 78, 1, 0, 0, 0},
   {DBA_FIELD_SERV_SELECTABLE, 79, 1, 0, 0, 0},
   {DBA_FIELD_SERV_LOCKED, 80, 1, 0, 0, 0},
   {DBA_FIELD_SERV_SCHED_DISABLED, 81, 1, 0, 0, 0},
   {DBA_FIELD_SERV_NOWNEXT_DISABLED, 82, 1, 0, 0, 0},
   {DBA_FIELD_SERV_FREESAT_ID, 83, 16, 0, 0, 0},
   {DBA_FIELD_SERV_REGION_ID, 99, 16, 0, 0, 0},
   {DBA_FIELD_SERV_LCN_EDITABLE, 115, 1, 0, 0, 0},
   {DBA_FIELD_SERV_DELETED, 116, 1, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 120, (DBA_SERV_NAME_LEN * 8), 0, DBA_SERV_NAME_LEN, 0}
   /* 3 blocks used - 103 bits spare here */
};

/* Timer record (also used for PVR recording) */
static const FIELD_ACCESS nvm_timer_rec[] =
{
   {DBA_FIELD_TIMER_HANDLE, 0, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_DISKID, 32, 16, 0, 0, 0},
   {DBA_FIELD_TIMER_EVENTID, 48, 16, 0, 0, 0},
   {DBA_FIELD_ORIG_NET_ID, 64, 16, 0, 0, 0},
   {DBA_FIELD_TRANSPORT_ID, 80, 16, 0, 0, 0},
   {DBA_FIELD_SERVICE_ID, 96, 16, 0, 0, 0},
   {DBA_FIELD_TIMER_STARTTIME, 112, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_DURATION, 144, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_TYPE, 176, 4, 0, 0, 0},
   {DBA_FIELD_TIMER_FREQUENCY, 180, 4, 0, 0, 0},
   {DBA_FIELD_TIMER_RAMPVOLUME, 184, 1, 0, 0, 0},
   {DBA_FIELD_TIMER_MISSED, 185, 1, 0, 0, 0},
   {DBA_FIELD_CRID_RECOMMENDED, 186, 1, 0, 0, 0},
   {DBA_FIELD_TIMER_EVENT_TRIGGERED, 187, 1, 0, 0, 0},
   {DBA_FIELD_TIMER_NOTIFY_TIME, 188, 16, 0, 0, 0},
   {DBA_FIELD_TIMER_START_PADDING, 204, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_END_PADDING, 236, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_DO_NOT_DELETE, 268, 1, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 272, (DBA_TIMER_NAME_LEN * 8), 0, DBA_TIMER_NAME_LEN, 0},
   {DBA_FIELD_TIMER_CRID, 272 + (DBA_TIMER_NAME_LEN * 8), (DBA_CRID_LEN * 8),
    DBA_TIMER_NAME_LEN, DBA_CRID_LEN, 0},
   {DBA_FIELD_TIMER_OTHERCRID, 272 + (DBA_TIMER_NAME_LEN * 8) + (DBA_CRID_LEN * 8),
    (DBA_CRID_LEN * 8), DBA_TIMER_NAME_LEN + DBA_CRID_LEN, DBA_CRID_LEN, 0},
   {DBA_FIELD_TIMER_ADDITIONAL_INFO, 272 + (DBA_TIMER_NAME_LEN * 8) + (DBA_CRID_LEN * 8) + (DBA_CRID_LEN * 8),
    (DBA_TIMER_ADDINFO_LEN * 8), DBA_TIMER_NAME_LEN + DBA_CRID_LEN + DBA_CRID_LEN, DBA_TIMER_ADDINFO_LEN, 0}
   /* 34 blocks used - 104 spare bits here */
};

/* CRID record */
static const FIELD_ACCESS nvm_crid_rec[] =
{
   {DBA_FIELD_CRID_EIT_DATE, 0, 16, 0, 0, 0},
   {DBA_FIELD_TIMER_STARTTIME, 16, 32, 0, 0, 0},
   {DBA_FIELD_SERVICE_ID, 48, 16, 0, 0, 0},
   {DBA_FIELD_CRID_SERIES, 64, 1, 0, 0, 0},
   {DBA_FIELD_CRID_RECOMMENDED, 65, 1, 0, 0, 0},
   {DBA_FIELD_CRID_DO_NOT_DELETE, 66, 1, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 72, (DBA_TIMER_NAME_LEN * 8), 0, DBA_TIMER_NAME_LEN, 0},
   {DBA_FIELD_TIMER_CRID, 72 + (DBA_TIMER_NAME_LEN * 8), (DBA_CRID_LEN * 8),
    DBA_TIMER_NAME_LEN, DBA_CRID_LEN, 0}
   /* 17 blocks used - 88 spare bits */
};

/* Favourite list record */
static const FIELD_ACCESS nvm_fav_list_rec[] =
{
   {DBA_FIELD_FAVLIST_ID, 0, 8, 0, 0, 0},
   {DBA_FIELD_FAVLIST_INDEX, 8, 8, 0, 0, 0},
   {DBA_FIELD_FAVLIST_USER_DATA, 16, 32, 0, 0, 0},
   {DBA_FIELD_REC_NAME, 48, (DBA_FAVLIST_NAME_LEN * 8), 0, DBA_FAVLIST_NAME_LEN, 0}
   /* 7 blocks used - 48 spare bits */
};

/* Favourite list service record */
static const FIELD_ACCESS nvm_fav_serv_rec[] =
{
   {DBA_FIELD_PARENT, 0, 16, 0, 0, 0},                           /* This is used to link to the service */
   {DBA_FIELD_FAVLIST_ID, 16, 8, 0, 0, 0},
   {DBA_FIELD_FAVLIST_INDEX, 24, 16, 0, 0, 0}
   /* 1 block used - 120 spare bits */
};

static const FIELD_ACCESS nvm_cicam_timer[] =
{
   {DBA_FIELD_PROFILE_CAM_ID, 0, 32, 0, 0, 0},
   {DBA_FIELD_TIMER_HANDLE, 32, 32, 0, 0, 0}
   /* 1 block used - 96 spare bits */
};

/* Event record - not stored in this database */
/* static const FIELD_ACCESS nvm_event_rec[0]; */

static const RECORD_ACCESS database_access_lut[DBA_NUM_RECORDS] =
{
   {NUM_FIELDS(nvm_lnb_rec), nvm_lnb_rec},
   {NUM_FIELDS(nvm_sat_rec), nvm_sat_rec},
   {NUM_FIELDS(nvm_net_rec), nvm_net_rec},
   {NUM_FIELDS(nvm_stran_rec), nvm_stran_rec},
   {NUM_FIELDS(nvm_ttran_rec), nvm_ttran_rec},
   {NUM_FIELDS(nvm_ctran_rec), nvm_ctran_rec},
   {NUM_FIELDS(nvm_serv_rec), nvm_serv_rec},
   {NUM_FIELDS(nvm_timer_rec), nvm_timer_rec},
   {NUM_FIELDS(nvm_crid_rec), nvm_crid_rec},
   {NUM_FIELDS(nvm_fav_list_rec), nvm_fav_list_rec},
   {NUM_FIELDS(nvm_fav_serv_rec), nvm_fav_serv_rec},
   {NUM_FIELDS(nvm_lnb_band_rec), nvm_lnb_band_rec},
   {NUM_FIELDS(nvm_cicam_timer), nvm_cicam_timer}/*,
   {NUM_FIELDS(nvm_event_rec), nvm_event_rec}*/
};

static BOOLEAN dba_initialised = FALSE;
static void *dba_sem;
static U8BIT version_string[FORMAT_ID_STRING_SIZE + 1];

static U8BIT *backup_database;

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static BOOLEAN ReadFormat(U8BIT *data_ptr, U8BIT *version);
static void WriteFormat(U8BIT *data_ptr);
static void BuildDatabase(U8BIT *data_ptr);
static void ConvertDatabase(U8BIT *data_ptr);
static void ClearDatabase(void);

static U8BIT GetFieldType(U16BIT field_id);
static U16BIT GetNVMRecSize(U8BIT rec_id);
static U16BIT GetRAMRecSize(U8BIT rec_id);

static BOOLEAN GetNVMFieldParams(U8BIT rec_id, U16BIT field_id, U16BIT *offset, U16BIT *size);
static BOOLEAN GetRAMFieldParams(U8BIT rec_id, U16BIT field_id, U16BIT *offset, U16BIT *size);

static void AddToNVMList(void *rec_ptr);
static void SetStartNVMList(U8BIT rec_id, U16BIT block_no);
static U16BIT GetStartNVMList(U8BIT *data_ptr, U8BIT rec_id);
static void RemoveFromNVMList(void *rec_ptr);

static void DestroyRecord(void *rec_ptr);


//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Performs any initialisation required prior to the database being loaded
 * @return  TRUE if initialisation is successful, FALSE otherwise
 */
BOOLEAN DBA_Initialise(void)
{
   FUNCTION_START(DBA_Initialise);

   if (!dba_initialised)
   {
      dba_sem = STB_OSCreateSemaphore();
      if (dba_sem != NULL)
      {
         dba_initialised = TRUE;
      }

      STB_NVMInitialise();

      backup_database = NULL;
   }

   FUNCTION_FINISH(DBA_Initialise);

   return(dba_initialised);
}

/**
 * @brief   Releases any resources acquired by initialisation and clears any currently loaded
 *          database, including a backup, if one has been created.
 */
void DBA_Terminate(void)
{
   FUNCTION_START(DBA_Terminate);

   if (dba_initialised)
   {
      if (dba_sem != NULL)
      {
         STB_OSDeleteSemaphore(dba_sem);
         dba_sem = NULL;
      }

      if (backup_database != NULL)
      {
         STB_FreeMemory(backup_database);
         backup_database = NULL;
      }

      dba_initialised = FALSE;
   }

   FUNCTION_FINISH(DBA_Terminate);
}

/**
 * @brief   Reads a database from non-volatile storage, creating any structures in memory that
 *          will be required to access it and the records it contains and makes this the working
 *          database. If the database is found to be invalid, or doesn't exist at all, then an
 *          empty database should be created.
 * @param   pathname full pathname of the file to be loaded containing the database.
 *                     This argument is only relevant where the system being used is file based.
 * @return  TRUE if a database is loaded successfully, FALSE otherwise
 */
BOOLEAN DBA_LoadDatabase(U8BIT *pathname)
{
   U32BIT fsize;
   U16BIT i, j, k;
   U8BIT *ram_cache;
   U8BIT version;
   BOOLEAN retval;

   FUNCTION_START(DBA_LoadDatabase);

   /* pathname isn't used for NVM database */
   USE_UNWANTED_PARAM(pathname);

   retval = FALSE;

   if (dba_initialised)
   {
      /* Calc format size */
      fsize = (U16BIT)sizeof(DATABASE_FORMAT);
      fsize += (U16BIT)sizeof(DATABASE_DESCRIPTOR);

      for (i = 0; i < DBA_NUM_RECORDS; i++)
      {
         k = 0;
         for (j = 0; j < database_access_lut[i].num_fields; j++)
         {
            if (database_access_lut[i].field[j].nvm_bit_size > 0)
            {
               k++;
               fsize += (U16BIT)sizeof(FIELD_DESCRIPTOR);
            }
         }

         if (k > 0)
         {
            fsize += (U16BIT)sizeof(RECORD_DESCRIPTOR);
         }
      }

      /* Initialise DB access */
      STB_InitNVMAccess(fsize);
      STB_InitRAMAccess();

      fsize = STB_NVMGetSTBSize();

      /* Only want to setup a RAM area for the DB if one isn't currently setup */
      if ((ram_cache = STB_GetNVMAccessRAM()) == NULL)
      {
         /* Allocate RAM that can be used as a cache by the database */
         ram_cache = (U8BIT *)STB_GetMemory(fsize);
      }

      if (ram_cache != NULL)
      {
         if (STB_NVMSTBRead(0, STB_NVMGetSTBSize(), ram_cache))
         {
            /* Validate database */
            if (ReadFormat(ram_cache, &version))
            {
               /* Valid - check format version */
               if (version == FORMAT_VERSION_NUM)
               {
                  /* Correct format - make sure checksums are correct */
                  if (STB_CheckNVMDatabaseIntegrity())
                  {
                     /* Correct format - build RAM database */
                     BuildDatabase(ram_cache);
                  }
                  else
                  {
                     /* Block checksums aren't correct - use an empty database */
                     memset(ram_cache, 0xff, fsize);

                     WriteFormat(ram_cache);
                     ClearDatabase();
                     BuildDatabase(ram_cache);
                     STB_NVMSTBWrite(0, STB_NVMGetSTBSize(), ram_cache);
                  }
               }
               else
               {
                  /* Wrong format - convert, build and save RAM database */
                  ConvertDatabase(ram_cache);
                  BuildDatabase(ram_cache);
                  STB_NVMSTBWrite(0, STB_NVMGetSTBSize(), ram_cache);
               }
            }
            else
            {
               /* Invalid - use an empty database */
               memset(ram_cache, 0xff, fsize);

               WriteFormat(ram_cache);
               ClearDatabase();
               BuildDatabase(ram_cache);
               STB_NVMSTBWrite(0, STB_NVMGetSTBSize(), ram_cache);
            }

            /* Use the data in memory as a ram cache for the database */
            STB_SetNVMAccessRAM(ram_cache);

            retval = TRUE;
         }
         else
         {
            /* Failed to read database, maybe the wrong size, so create an empty one */
            memset(ram_cache, 0xff, fsize);

            WriteFormat(ram_cache);
            ClearDatabase();
            BuildDatabase(ram_cache);
            STB_NVMSTBWrite(0, STB_NVMGetSTBSize(), ram_cache);
         }
      }
   }

   FUNCTION_FINISH(DBA_LoadDatabase);

   return(retval);
}

/**
 * @brief   Saves any changes made to the working database to non-volatile storage.
 *          If saving to a file, the pathname used to open it will be used.
 * @return  TRUE if the database is saved successfully, FALSE otherwise
 */
BOOLEAN DBA_SaveDatabase(void)
{
   BOOLEAN retval;

   FUNCTION_START(DBA_SaveDatabase);

   if (dba_initialised)
   {
      STB_NVMSave();
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(DBA_SaveDatabase);

   return(retval);
}

/**
 * @brief   Clears the working database of all records. Following this call, it should still
 *          be possible to access the database, but it will be empty, and new records can be
 *          added to it.
 * @return  TRUE if the database is cleared, FALSE otherwise
 */
BOOLEAN DBA_ClearDatabase(void)
{
   BOOLEAN retval;

   FUNCTION_START(DBA_ClearDatabase);

   if (dba_initialised)
   {
      ClearDatabase();
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(DBA_ClearDatabase);

   return(retval);
}

/**
 * @brief   Creates a backup copy of the working database. Whether the backup database is saved
 *          to non-volatile storage is implementation dependant and so the backup database may
 *          not survive a system reboot.
 * @param   pathname full pathname of the file to save the backup to if this is supported
 *                     by the database implementation.
 * @return  TRUE if a backup is created, FALSE otherwise
 */
BOOLEAN DBA_BackupDatabase(U8BIT *pathname)
{
   BOOLEAN retval;
   U32BIT db_size;
   U8BIT *current_db;

   FUNCTION_START(DBA_BackupDatabase);

   /* pathname isn't used for NVM database */
   USE_UNWANTED_PARAM(pathname);

   retval = FALSE;

   if (backup_database != NULL)
   {
      /* Already have a backup, delete it first */
      STB_FreeMemory(backup_database);
      backup_database = NULL;
   }

   /* Only backup if there's a current database */
   if ((current_db = STB_GetNVMAccessRAM()) != NULL)
   {
      STB_NVMFlushCache(FALSE);

      /* Allocate memory to backup to */
      db_size = STB_NVMGetSTBSize();
      if ((backup_database = (U8BIT *)STB_GetMemory(db_size)) != NULL)
      {
         memcpy(backup_database, current_db, db_size);
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(DBA_BackupDatabase);

   return(retval);
}

/**
 * @brief   Checks whether the working database can be restored from a backup copy
 * @return  TRUE if there is a backup copy and it can be restored, FALSE otherwise
 */
BOOLEAN DBA_CanRestoreDatabase(void)
{
   BOOLEAN retval = FALSE;
   U8BIT *current_db;

   FUNCTION_START(DBA_CanRestoreDatabase);

   current_db = STB_GetNVMAccessRAM();

   if ((backup_database != NULL) && (current_db != NULL))
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(DBA_CanRestoreDatabase);

   return(retval);
}

/**
 * @brief   Restores the working database from a previously made backup copy.
 * @return  TRUE if the working database is restored from the backup, FALSE otherwise
 */
BOOLEAN DBA_RestoreDatabase(void)
{
   BOOLEAN retval = FALSE;
   U32BIT db_size;
   U8BIT *current_db;

   FUNCTION_START(DBA_RestoreDatabase);

   current_db = STB_GetNVMAccessRAM();

   if ((backup_database != NULL) && (current_db != NULL))
   {
      STB_NVMFlushCache(TRUE);

      /* Overwrite the existing RAM cache database with the backup copy */
      db_size = STB_NVMGetSTBSize();
      memcpy(current_db, backup_database, db_size);

      /* Force the database status to ensure it's saved the next time STB_NVMSave is called */
      STB_NVMChanged(TRUE);

      retval = TRUE;
   }

   FUNCTION_FINISH(DBA_RestoreDatabase);

   return(retval);
}

/**
 * @brief   Erases the backup copy of the database. If data was stored in non-volatile storage
 *          then this should be deleted too.
 */
void DBA_EraseBackupDatabase(void)
{
   FUNCTION_START(DBA_EraseBackupDatabase);

   if (backup_database != NULL)
   {
      STB_FreeMemory(backup_database);
      backup_database = NULL;
   }

   FUNCTION_FINISH(DBA_EraseBackupDatabase);
}

/**
 * @brief   Export the working database to an XML file.
 * @param   xml_pathname full pathname of the file to export to
 * @return  TRUE if the database is exported successfully, FALSE otherwise
 */
BOOLEAN DBA_ExportToXML(U8BIT *xml_pathname)
{
   FUNCTION_START(DBA_ExportToXML);
   USE_UNWANTED_PARAM(xml_pathname);
   FUNCTION_FINISH(DBA_ExportToXML);
   return(FALSE);
}

/**
 * @brief   Imports records from the given XML file into the working database. If a record already
 *          exists in the database then it will be updated, and if it doesn't then a new record
 *          will be created.
 * @param   xml_pathname full pathname of the file to be imported
 * @return  TRUE if the file is imported successfully, FALSE otherwise
 */
BOOLEAN DBA_ImportFromXML(U8BIT *xml_pathname)
{
   FUNCTION_START(DBA_ImportFromXML);
   USE_UNWANTED_PARAM(xml_pathname);
   FUNCTION_FINISH(DBA_ImportFromXML);
   return(FALSE);
}

/**
 * @brief   Locks the database to prevent access from other threads or processes.
 */
void DBA_LockDatabase(void)
{
   FUNCTION_START(DBA_LockDatabase);
   STB_OSSemaphoreWait(dba_sem);
   FUNCTION_FINISH(DBA_LockDatabase);
}

/**
 * @brief   Unlocks the database to allow other threads or processes to access it.
 */
void DBA_UnlockDatabase(void)
{
   FUNCTION_START(DBA_UnlockDatabase);
   STB_OSSemaphoreSignal(dba_sem);
   FUNCTION_FINISH(DBA_UnlockDatabase);
}

/**
 * @brief   Returns a version string representing the working database.
 * @return  '\0' terminated string in UTF-8 format
 */
U8BIT* DBA_DatabaseVersion(void)
{
   DATABASE_FORMAT db_format;

   FUNCTION_START(DBA_DatabaseVersion);

   /* Read format header */
   if (STB_ReadNVMData(0, sizeof(DATABASE_FORMAT), (U8BIT *)&db_format))
   {
      /* Copy version string into static array */
      strncpy((char *)version_string, (char *)db_format.id_str, FORMAT_ID_STRING_SIZE + 1);
   }
   else
   {
      version_string[0] = '\0';
   }

   FUNCTION_FINISH(DBA_DatabaseVersion);

   return(version_string);
}

/**
 * @brief   Returns the size in bytes of the database as stored in non-volatile storage
 * @param   max_size returns the maximum size in bytes that the database can grow to.
 *                   This value may be returned as 0 if this can't be determined.
 * @return  current size of the working database in bytes
 */
U32BIT DBA_DatabaseFileSize(U32BIT *max_size)
{
   U32BIT size;

   FUNCTION_START(DBA_DatabaseFileSize);

   size = STB_GetNVMBlocksUsed() * STB_GetNVMBlockSize();

   if (max_size != NULL)
   {
      *max_size = STB_GetNVMBlockCount() * STB_GetNVMBlockSize();
   }

   FUNCTION_FINISH(DBA_DatabaseFileSize);

   return(size);
}

/**
 * @brief   Creates a new record of the given type, adding it to the database as a child of the
 *          given parent record, if provided.
 * @param   record_id type of record to be created
 * @param   parent create a record that is a child of this parent, can be passed as NULL
 * @return  handle of new record, or NULL if creation fails
 */
void* DBA_CreateRecord(U32BIT record_id, void *parent)
{
   U16BIT nvm_size, ram_size, nvm_block;
   U16BIT field_offset, field_size;
   void *rec_ptr;

   FUNCTION_START(DBA_CreateRecord);

   rec_ptr = NULL;

   if (dba_initialised && (record_id < DBA_NUM_RECORDS) /*&& (record_id != DBA_RECORD_EVENT)*/)
   {
      nvm_size = GetNVMRecSize(record_id);
      ram_size = GetRAMRecSize(record_id);

      /* If record has an NVM record */
      if (nvm_size > 0)
      {
         /* Try to create NVM record */
         nvm_block = STB_CreateNVMRecord(record_id, nvm_size);
         if (nvm_block != NVM_INVALID_BLOCK_ID)
         {
            /* Try to create RAM record */
            rec_ptr = STB_CreateRAMRecord(record_id, ram_size, nvm_block, parent);
            if (rec_ptr == NULL)
            {
               /* Failed - destroy NVM record */
               STB_DestroyNVMRecord(nvm_block);
            }
            else
            {
               /* Add to linked list */
               AddToNVMList(rec_ptr);

               if (parent != NULL)
               {
                  /* Set parent link */
                  if (GetNVMFieldParams(record_id, DBA_FIELD_PARENT, &field_offset,
                         &field_size))
                  {
                     STB_SetNVMRecordNumber(nvm_block, field_offset, field_size,
                        (U32BIT)STB_GetRAMRecordNVMBlock(parent));
                  }
               }
               else
               {
                  if (GetNVMFieldParams(record_id, DBA_FIELD_PARENT, &field_offset,
                         &field_size))
                  {
                     STB_SetNVMRecordNumber(nvm_block, field_offset, field_size,
                        NVM_INVALID_BLOCK_ID);
                  }
               }
            }
         }
      }
      else
      {
         /* Only a RAM record - try to create it */
         rec_ptr = STB_CreateRAMRecord(record_id, ram_size, NVM_INVALID_BLOCK_ID, parent);
      }
   }

   FUNCTION_FINISH(DBA_CreateRecord);

   return(rec_ptr);
}

/**
 * @brief   Destroys the given record, removing it from the database and freeing any memory
 *          associated with it.
 * @param   record database record to be deleted
 */
void DBA_DestroyRecord(void *record)
{
   U8BIT count;
   void *crec_ptr;
   void *cnext_ptr;

   FUNCTION_START(DBA_DestroyRecord);

   if (dba_initialised && (record != NULL))
   {
      /* Destroy any children of this record */
      count = DBA_NUM_RECORDS - 1;
      while (count > STB_GetRAMRecordId(record))
      {
         crec_ptr = STB_FindRAMRecordFromId(count, record, NULL);
         while (crec_ptr != NULL)
         {
            cnext_ptr = STB_FindRAMRecordFromId(count, record, crec_ptr);
            DestroyRecord(crec_ptr);
            crec_ptr = cnext_ptr;
         }

         count--;
      }

      /* Destroy the record */
      DestroyRecord(record);
   }

   FUNCTION_FINISH(DBA_DestroyRecord);
}

/**
 * @brief   Finds the next record, of the given type, that comes after last_rec. last_rec
 *          must be the same type of record as the one being found. Parent is optional, but
 *          if provided, a record will only be returned if the parent of the one found is the same.
 * @param   record_id - type of record to look for
 * @param   parent if not NULL, this must be the parent of the record found for it to be returned
 * @param   last_rec previous record of the type being looked for, if NULL then the first record
 *                     will be returned.
 * @return  handle of the record, or NULL if none found
 */
void* DBA_FindRecord(U32BIT record_id, void *parent, void *last_rec)
{
   void *rec_ptr;

   FUNCTION_START(DBA_FindRecord);

   if (dba_initialised && (record_id < DBA_NUM_RECORDS))
   {
      rec_ptr = STB_FindRAMRecordFromId(record_id, parent, last_rec);
   }
   else
   {
      rec_ptr = NULL;
   }

   FUNCTION_FINISH(DBA_FindRecord);

   return(rec_ptr);
}

/**
 * @brief   Set of change the parent of the given record
 * @param   record handle of the record whose parent is to be changed
 * @param   parent handle of the new parent record, can be NULL
 */
void DBA_SetRecordParent(void *record, void *parent)
{
   U8BIT rec_id;
   U16BIT nvm_block;
   U32BIT link_val;
   U16BIT field_offset, field_size;

   FUNCTION_START(DBA_SetRecordParent);

   ASSERT(record != NULL);

   if (dba_initialised && (record != NULL))
   {
      /* Get record id of record */
      rec_id = STB_GetRAMRecordId(record);

      /* Set parent pointer in RAM record */
      STB_SetRAMRecordParent(record, parent);

      /* Save NVM parent link... */
      if (parent == NULL)
      {
         link_val = NVM_INVALID_BLOCK_ID;
      }
      else
      {
         link_val = (U32BIT)STB_GetRAMRecordNVMBlock(parent);
      }

      if (GetRAMFieldParams(rec_id, DBA_FIELD_PARENT, &field_offset, &field_size))
      {
         STB_SetRAMRecordNumber(record, field_offset, field_size, link_val);
      }

      if (GetNVMFieldParams(rec_id, DBA_FIELD_PARENT, &field_offset, &field_size))
      {
         nvm_block = STB_GetRAMRecordNVMBlock(record);
         STB_SetNVMRecordNumber(nvm_block, field_offset, field_size, link_val);
      }
   }

   FUNCTION_FINISH(DBA_SetRecordParent);
}

/**
 * @brief   Returns the handle to the parent of the given record
 * @param   record handle of record whose parent is to be returned
 * @return  handle of the record's parent
 */
void* DBA_GetRecordParent(void *record)
{
   void *parent;

   FUNCTION_START(DBA_GetRecordParent);

   ASSERT(record != NULL);

   if (dba_initialised && (record != NULL))
   {
      parent = STB_GetRAMRecordParent(record);
   }
   else
   {
      parent = NULL;
   }

   FUNCTION_FINISH(DBA_GetRecordParent);

   return(parent);
}

/**
 * @brief   Forces a record to be saved to non-volatile storage. Depending on the implementation,
 *          this function may not do anything if the data is updated to non-volatile storage
 *          as any records and/or fields are created or updated.
 * @param   record handle of record to be saved
 */
void DBA_SaveRecord(void *record)
{
   FUNCTION_START(DBA_SaveRecord);

   /* Not implemented for NVM storage */
   USE_UNWANTED_PARAM(record);

   FUNCTION_FINISH(DBA_SaveRecord);
}

/**
 * @brief   Set the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field is a string value.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   value new value of the field
 * @return  TRUE if the value is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldValue(void *record, U32BIT field_id, U32BIT value)
{
   U8BIT rec_id, type;
   U16BIT field_size, field_offset;
   BOOLEAN retval;

   FUNCTION_START(DBA_SetFieldValue);

   ASSERT(record != NULL);

   retval = FALSE;

   if (dba_initialised)
   {
      /* Get record id of record and look for field id */
      rec_id = STB_GetRAMRecordId(record);
      type = GetFieldType(field_id);
      if (type != STR_TYPE)
      {
         /* Check if this field is in RAM */
         if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            STB_SetRAMRecordNumber(record, field_offset, field_size, value);
         }

         /* Check if this field is in NVM */
         if (GetNVMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            STB_SetNVMRecordNumber(STB_GetRAMRecordNVMBlock(record), field_offset,
               field_size, value);
         }

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(DBA_SetFieldValue);

   return(retval);
}

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field is an integer value.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldString(void *record, U32BIT field_id, U8BIT *string, U16BIT num_bytes)
{
   U8BIT rec_id, type;
   U16BIT field_size, field_offset;
   BOOLEAN retval;

   FUNCTION_START(DBA_SetFieldString);

   ASSERT(record != NULL);

   retval = FALSE;

   if (dba_initialised)
   {
      /* Get record id of record and look for field id */
      rec_id = STB_GetRAMRecordId(record);
      type = GetFieldType(field_id);
      if (type == STR_TYPE)
      {
         /* Check if this field is in RAM */
         if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            if (num_bytes < field_size)
            {
               field_size = num_bytes;
            }

            STB_SetRAMRecordString(record, field_offset, field_size, string);
         }

         /* Check if this field is in NVM */
         if (GetNVMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            /* NVM field size is in bits, hence *8 */
            if ((num_bytes * 8) < field_size)
            {
               field_size = num_bytes * 8;
            }

            STB_SetNVMRecordString(STB_GetRAMRecordNVMBlock(record), field_offset,
               field_size, string);
         }

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(DBA_SetFieldString);

   return(retval);
}

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   lang_code language code of the string
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT *string,
   U16BIT num_bytes)
{
   FUNCTION_START(DBA_SetFieldLangString);

   /* Not implemented for NVM storage */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(lang_code);
   USE_UNWANTED_PARAM(string);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_SetFieldLangString);

   return(FALSE);
}

/**
 * @brief   Set a variable amount of data of a record's field. The function will fail if the
 *          record doesn't exist, the record doesn't include the given field, or the field
 *          doesn't hold data. The data provided will be copied.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   data data to be stored in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if the data is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldData(void *record, U32BIT field_id, U8BIT *data, U16BIT num_bytes)
{
   FUNCTION_START(DBA_SetFieldData);

   /* Not implemented for NVM storage */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(data);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_SetFieldData);

   return(FALSE);
}

/**
 * @brief   Gets the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field is a string value.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   value pointer to the returned value if function returns TRUE
 * @return  TRUE if a value is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldValue(void *record, U32BIT field_id, U32BIT *value)
{
   U8BIT rec_id, type;
   U16BIT field_size, field_offset;
   BOOLEAN retval;

   FUNCTION_START(DBA_GetFieldValue);

   ASSERT(record != NULL);
   ASSERT(value != NULL);

   retval = FALSE;

   if (dba_initialised)
   {
      /* Get record id of record and look for field id */
      rec_id = STB_GetRAMRecordId(record);
      type = GetFieldType(field_id);
      if (type != STR_TYPE)
      {
         /* Check if this field is in RAM */
         if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            *value = STB_GetRAMRecordNumber(record, field_offset, field_size);
            retval = TRUE;
         }
         else
         {
            /* Check if this field is in NVM */
            if (GetNVMFieldParams(rec_id, field_id, &field_offset, &field_size))
            {
               *value = STB_GetNVMRecordNumber(STB_GetRAMRecordNVMBlock(record), field_offset,
                     field_size);
               retval = TRUE;
            }
            else
            {
               *value = 0;
            }
         }
      }
   }

   FUNCTION_FINISH(DBA_GetFieldValue);

   return(retval);
}

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field is an integer value.
 *          The pointer to the string returned will be to the data held by the database, so the
 *          data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldString(void *record, U32BIT field_id, U8BIT **string, U16BIT *num_bytes)
{
   U8BIT rec_id, type;
   U16BIT field_size, field_offset;
   BOOLEAN retval;

   FUNCTION_START(DBA_GetFieldString);

   ASSERT(record != NULL);

   retval = FALSE;

   if (dba_initialised)
   {
      /* Get record id of record and look for field id */
      rec_id = STB_GetRAMRecordId(record);
      type = GetFieldType(field_id);
      if (type == STR_TYPE)
      {
         /* Check if this field is in RAM */
         if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            *string = STB_GetRAMRecordString(record, field_offset, field_size);
            retval = TRUE;
         }
         else
         {
            /* Check if this field is in NVM */
            if (GetNVMFieldParams(rec_id, field_id, &field_offset, &field_size))
            {
               *string = STB_GetNVMRecordString(STB_GetRAMRecordNVMBlock(record),
                     field_offset, field_size);
               retval = TRUE;
            }
         }

         if (*string != NULL)
         {
            *num_bytes = STB_GetNumBytesInString(*string);
         }
      }
   }

   FUNCTION_FINISH(DBA_GetFieldString);

   return(retval);
}

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field is an integer value.
 *          The pointer to the string returned will be to the data held by the database, so the
 *          data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   lang_code language code of the string to be returned
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT **string,
   U16BIT *num_bytes)
{
   FUNCTION_START(DBA_GetFieldLangString);

   /* Not implemented for NVM storage */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(lang_code);
   USE_UNWANTED_PARAM(string);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_GetFieldLangString);

   return(FALSE);
}

/**
 * @brief   Gets the data of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a data field.
 *          The pointer to the data returned will be to the data held by the database, so the
 *          data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   data pointer to the returned data in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if data is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldData(void *record, U32BIT field_id, U8BIT **data, U16BIT *num_bytes)
{
   FUNCTION_START(DBA_GetFieldData);

   /* Not implemented for NVM storage */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(data);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_GetFieldData);

   return(FALSE);
}

/**
 * @brief   Returns the number of bytes available for the given data block
 * @param   data_block_id the data block whose size is to be returned
 * @return  size of bytes of the data block, or 0 if block not found
 */
U32BIT DBA_DataBlockSize(U32BIT data_block_id)
{
   U32BIT size;

   FUNCTION_START(DBA_DataBlockSize);

   size = STB_NVMGetDataBlockSize(data_block_id);

   FUNCTION_FINISH(DBA_DataBlockSize);

   return(size);
}

/**
 * @brief   Read a block of data from the database into the given buffer
 * @param   data_block_id id of the data block to be read
 * @param   data pointer to the buffer into which the data will be read
 * @param   num_bytes the number of bytes to be read
 * @return  number of bytes read into the buffer
 */
U32BIT DBA_DataBlockRead(U32BIT data_block_id, U8BIT *data, U32BIT num_bytes)
{
   U32BIT bytes_read;
   U32BIT block_size;

   FUNCTION_START(DBA_DataBlockRead);

   bytes_read = 0;

   block_size = STB_NVMGetDataBlockSize(data_block_id);
   if (block_size < num_bytes)
   {
      num_bytes = block_size;
   }

   if (STB_NVMDataBlockRead(data_block_id, num_bytes, data))
   {
      bytes_read = num_bytes;
   }

   FUNCTION_FINISH(DBA_DataBlockRead);

   return(bytes_read);
}

/**
 * @brief   Writes a block of data into the database from the given buffer
 * @param   data_block_id id of the data block being written
 * @param   data pointer to the data to be written
 * @param   num_bytes number of bytes of data to be written
 * @return  TRUE if the data is written, FALSE otherwise
 */
BOOLEAN DBA_DataBlockWrite(U32BIT data_block_id, U8BIT *data, U32BIT num_bytes)
{
   BOOLEAN retval;

   FUNCTION_START(DBA_DataBlockWrite);

   retval = STB_NVMDataBlockWrite(data_block_id, num_bytes, data);;

   FUNCTION_FINISH(DBA_DataBlockWrite);

   return(retval);
}

//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------

static BOOLEAN ReadFormat(U8BIT *data_ptr, U8BIT *version)
{
   DATABASE_FORMAT db_format;
   BOOLEAN retval = FALSE;

   FUNCTION_START(ReadFormat);

   // read format header
   memcpy((void *)&db_format, (void *)data_ptr, (size_t)sizeof(DATABASE_FORMAT));

   // validate checksum
   if (STB_CheckChecksum((U8BIT *)&db_format, (U32BIT)sizeof(DATABASE_FORMAT)))
   {
      // validate format string
      if (strcmp((char *)db_format.id_str, (char *)FORMAT_ID_STRING) == 0)
      {
         // return version number
         *version = db_format.version;
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(ReadFormat);

   return(retval);
}

static void WriteFormat(U8BIT *data_ptr)
{
   U8BIT i, j, rcount;
   U8BIT fcount[DBA_NUM_RECORDS];
   U16BIT size;
   DATABASE_FORMAT db_format;
   DATABASE_DESCRIPTOR db_desc;
   RECORD_DESCRIPTOR rec_desc;
   FIELD_DESCRIPTOR fld_desc;

   FUNCTION_START(WriteFormat);

   // calc num of recs in NVM, number of NVM fields for each record and total descriptor size
   rcount = 0;
   size = (U16BIT)sizeof(DATABASE_DESCRIPTOR);
   for (i = 0; i < DBA_NUM_RECORDS; i++)
   {
      fcount[i] = 0;
      for (j = 0; j < database_access_lut[i].num_fields; j++)
      {
         if (database_access_lut[i].field[j].nvm_bit_size > 0)
         {
            fcount[i]++;
            size += (U16BIT)sizeof(FIELD_DESCRIPTOR);
         }
      }
      if (fcount[i] > 0)
      {
         rcount++;
         size += (U16BIT)sizeof(RECORD_DESCRIPTOR);
      }
   }

   // write format header
   memset(&db_format, 0, sizeof(db_format));
   strncpy((char *)db_format.id_str, (char *)FORMAT_ID_STRING, FORMAT_ID_STRING_SIZE);
   db_format.version = FORMAT_VERSION_NUM;
   db_format.checksum = STB_CalcChecksum((U8BIT *)&db_format, (U32BIT)sizeof(DATABASE_FORMAT));
   memcpy((void *)data_ptr, (void *)&db_format, (size_t)sizeof(DATABASE_FORMAT));
   data_ptr += (U16BIT)sizeof(DATABASE_FORMAT);

   // write database descriptor
   // endian independent version of:
   // db_desc.desc_size = size;
   STB_SetBE16Bit(&db_desc.desc_size, size);
   db_desc.num_records = rcount;
   db_desc.padding[0] = 0;
   memcpy((void *)data_ptr, (void *)&db_desc, (size_t)sizeof(DATABASE_DESCRIPTOR));
   data_ptr += (U16BIT)sizeof(DATABASE_DESCRIPTOR);

   // write record and field descriptors
   for (i = 0; i < DBA_NUM_RECORDS; i++)
   {
      // if this record has NVM fields
      if (fcount[i] > 0)
      {
         rec_desc.record_id = i;
         rec_desc.num_fields = fcount[i];
         // endian independent version of:
         // rec_desc.first_block = NVM_INVALID_BLOCK_ID;
         STB_SetBE16Bit(&rec_desc.first_block, NVM_INVALID_BLOCK_ID);
         memcpy((void *)data_ptr, (void *)&rec_desc, (size_t)sizeof(RECORD_DESCRIPTOR));
         data_ptr += (U16BIT)sizeof(RECORD_DESCRIPTOR);

         for (j = 0; j < database_access_lut[i].num_fields; j++)
         {
            if (database_access_lut[i].field[j].nvm_bit_size > 0)
            {
               // endian independent version of:
               // fld_desc.field_id = j;
               // fld_desc.field_offset = database_access_lut[i].field[j].nvm_bit_offset;
               STB_SetBE16Bit(&fld_desc.field_id, j);
               STB_SetBE16Bit(&fld_desc.field_offset, database_access_lut[i].field[j].nvm_bit_offset);
               fld_desc.field_size = database_access_lut[i].field[j].nvm_bit_size;
               fld_desc.padding[0] = 0;
               memcpy((void *)data_ptr, (void *)&fld_desc, (size_t)sizeof(FIELD_DESCRIPTOR));
               data_ptr += (U16BIT)sizeof(FIELD_DESCRIPTOR);
            }
         }
      }
   }

   FUNCTION_FINISH(WriteFormat);
}

static void BuildDatabase(U8BIT *data_ptr)
{
   U8BIT rec_id, type;
   U16BIT offset, size, field_id;
   U16BIT i, nvm_blk;
   U32BIT value;
   void *rec_ptr;
   void *par_ptr;
   BOOLEAN reset_ram_access = FALSE;

   FUNCTION_START(BuildDatabase);

   if (data_ptr != STB_GetNVMAccessRAM())
   {
      // force NVM access to temp area
      STB_SetNVMAccessRAM(data_ptr);
      reset_ram_access = TRUE;
   }

   // build NVM map in RAM
   STB_InitNVMMap();

   // for every record type in the list
   for (rec_id = 0; rec_id < DBA_NUM_RECORDS; rec_id++)
   {
      // get first occurence of record in NVM and size of RAM record
      nvm_blk = GetStartNVMList(data_ptr, rec_id);
      while (nvm_blk != NVM_INVALID_BLOCK_ID)
      {
         // if NVM has a parent, get RAM parent record pointer
         if (GetNVMFieldParams(rec_id, DBA_FIELD_PARENT, &offset, &size))
         {
            value = STB_GetNVMRecordNumber(nvm_blk, offset, size);
            par_ptr = STB_FindRAMRecordFromNVMBlock((U16BIT)value);
         }
         else
         {
            par_ptr = NULL;
         }

         // create RAM record of same type
         rec_ptr = STB_CreateRAMRecord(rec_id, GetRAMRecSize(rec_id), nvm_blk, par_ptr);
         if (rec_ptr != NULL)
         {
            // copy any fields that exist in both NVM and RAM
            for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
            {
               field_id = database_access_lut[rec_id].field[i].field_id;
               if (GetNVMFieldParams(rec_id, field_id, &offset, &size))
               {
                  type = GetFieldType(field_id);
                  switch (type)
                  {
                     case UNUM_TYPE:
                     {
                        value = STB_GetNVMRecordNumber(nvm_blk, offset, size);
                        if ((value != 0) &&
                            GetRAMFieldParams(rec_id, field_id, &offset, &size))
                        {
                           STB_SetRAMRecordNumber(rec_ptr, offset, size, value);
                        }
                        break;
                     }
                     case STR_TYPE:
                     {
                        U8BIT *str = STB_GetNVMRecordString(nvm_blk, offset, size);
                        if ((str != NULL) &&
                            GetRAMFieldParams(rec_id, field_id, &offset, &size))
                        {
                           STB_SetRAMRecordString(rec_ptr, offset, size, str);
                        }
                        break;
                     }
                     default:;
                  }
               }
            }
         }

         // get next NVM record
         nvm_blk = STB_GetNextNVMBlock(nvm_blk);
      }
   }

   if (reset_ram_access)
   {
      // restore NVM access to device
      STB_SetNVMAccessRAM(NULL);
   }

   FUNCTION_FINISH(BuildDatabase);
}

static void ConvertDatabase(U8BIT *data_ptr)
{
   U8BIT i, j;
   DATABASE_DESCRIPTOR *db_ptr;
   RECORD_DESCRIPTOR *rd_ptr;
   FIELD_DESCRIPTOR *fd_ptr;
   DATABASE_DESCRIPTOR db_desc;
   RECORD_DESCRIPTOR *rec_desc;
   FIELD_DESCRIPTOR **fld_desc;

   FUNCTION_START(ConvertDatabase);

   // parse database descriptors
   db_ptr = (DATABASE_DESCRIPTOR *)(data_ptr + (U32BIT)sizeof(DATABASE_FORMAT));
   // endian independent version of:
   // db_desc.desc_size = db_ptr->desc_size;
   db_desc.desc_size = STB_GetBE16Bit(&db_ptr->desc_size);
   db_desc.num_records = db_ptr->num_records;
   db_ptr++;

   rec_desc = (RECORD_DESCRIPTOR *)STB_GetMemory((U32BIT)(sizeof(RECORD_DESCRIPTOR) * db_desc.num_records));
   fld_desc = (FIELD_DESCRIPTOR **)STB_GetMemory((U32BIT)(sizeof(RECORD_DESCRIPTOR *) * db_desc.num_records));
   if ((rec_desc != NULL) && (fld_desc != NULL))
   {
      rd_ptr = (RECORD_DESCRIPTOR *)db_ptr;
      for (i = 0; i < db_desc.num_records; i++)
      {
         rec_desc[i].record_id = rd_ptr->record_id;
         rec_desc[i].num_fields = rd_ptr->num_fields;
         rd_ptr++;

         fd_ptr = (FIELD_DESCRIPTOR *)rd_ptr;
         fld_desc[i] = (FIELD_DESCRIPTOR *)STB_GetMemory((U32BIT)(sizeof(FIELD_DESCRIPTOR) * rec_desc[i].num_fields));
         if (fld_desc[i] != NULL)
         {
            for (j = 0; j < rec_desc[i].num_fields; j++)
            {
               // endian independent version of:
               // fld_desc[i][j].field_id = fd_ptr->field_id;
               // fld_desc[i][j].field_offset = fd_ptr->field_offset;
               fld_desc[i][j].field_id = STB_GetBE16Bit(&fd_ptr->field_id);
               fld_desc[i][j].field_offset = STB_GetBE16Bit(&fd_ptr->field_offset);
               fld_desc[i][j].field_size = fd_ptr->field_size;
               fd_ptr++;
            }
         }
         rd_ptr = (RECORD_DESCRIPTOR *)fd_ptr;
      }
   }

   // delete memory used
   if (fld_desc != NULL)
   {
      for (i = 0; i < db_desc.num_records; i++)
      {
         STB_FreeMemory((void *)fld_desc[i]);
      }
      STB_FreeMemory((void *)fld_desc);
   }
   STB_FreeMemory((void *)rec_desc);

   FUNCTION_FINISH(ConvertDatabase);
}

static void ClearDatabase(void)
{
   U8BIT rec_id, i;
   U16BIT field_id, field_offset, field_size;
   void *field_val;
   void *rec_ptr;

   FUNCTION_START(ClearDatabase);

   /* For every record type in the list */
   for (rec_id = 0; rec_id < DBA_NUM_RECORDS; rec_id++)
   {
      for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
      {
         field_id = database_access_lut[rec_id].field[i].field_id;
         if (GetFieldType(field_id) == PTR_TYPE)
         {
            if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
            {
               ASSERT(field_size == sizeof(void*));
               /* Free memory used by any pointer fields */
               rec_ptr = STB_FindRAMRecordFromId(rec_id, NULL, NULL);
               while (rec_ptr != NULL)
               {
                  field_val = STB_GetRAMRecordPointer(rec_ptr, field_offset);
                  STB_FreeMemory(field_val);
                  rec_ptr = STB_FindRAMRecordFromId(rec_id, NULL, rec_ptr);
               }
            }
         }
      }
   }

   /* Delete any existing RAM records */
   STB_PurgeRAMRecords();

   FUNCTION_FINISH(ClearDatabase);
}

static U8BIT GetFieldType(U16BIT field_id)
{
   U16BIT i;
   U16BIT num_fields = sizeof(database_field_type) / sizeof(database_field_type[0]);
   U8BIT ret_val = 0;

   FUNCTION_START(GetFieldType);

   // find field id in type LUT
   for (i = 0; i < num_fields; i++)
   {
      if (database_field_type[i].field_id == field_id)
      {
         ret_val = (U8BIT)database_field_type[i].type;
         break;
      }
   }

   FUNCTION_FINISH(GetFieldType);

   return(ret_val);
}

static U16BIT GetNVMRecSize(U8BIT rec_id)
{
   U16BIT rec_size, max_size;
   U8BIT i;

   FUNCTION_START(GetNVMRecSize);

   /* Get maximum number of bits used for a field (offset + size) */
   max_size = 0;
   for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
   {
      rec_size = database_access_lut[rec_id].field[i].nvm_bit_offset;
      rec_size += (U16BIT)database_access_lut[rec_id].field[i].nvm_bit_size;
      if (rec_size > max_size)
      {
         max_size = rec_size;
      }
   }

   /* Round up to number of complete bytes */
   if ((max_size % 8) > 0)
   {
      max_size += (8 - (max_size % 8));
   }

   /* Convert to bytes */
   max_size = (max_size >> 3);

   FUNCTION_FINISH(GetNVMRecSize);

   return(max_size);
}

static U16BIT GetRAMRecSize(U8BIT rec_id)
{
   U16BIT rec_size, max_size;
   U8BIT i;

   FUNCTION_START(GetRAMRecSize);

   /* Get maximum number of bytes used for a field (offset + size) */
   max_size = 0;
   for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
   {
      rec_size = database_access_lut[rec_id].field[i].ram_byte_offset;
      rec_size += (U16BIT)database_access_lut[rec_id].field[i].ram_byte_size;
      if (rec_size > max_size)
      {
         max_size = rec_size;
      }
   }

   FUNCTION_FINISH(GetRAMRecSize);

   return(max_size);
}

static BOOLEAN GetNVMFieldParams(U8BIT rec_id, U16BIT field_id, U16BIT *offset, U16BIT *size)
{
   U8BIT i;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(GetNVMFieldParams);

   ASSERT(rec_id < DBA_NUM_RECORDS);
   ASSERT(offset != NULL);
   ASSERT(size != NULL);

   for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
   {
      if (database_access_lut[rec_id].field[i].field_id == field_id)
      {
         *offset = database_access_lut[rec_id].field[i].nvm_bit_offset;
         *size = database_access_lut[rec_id].field[i].nvm_bit_size;
         if (*size > 0)
         {
            ret_val = TRUE;
         }
         break;
      }
   }

   FUNCTION_FINISH(GetNVMFieldParams);

   return(ret_val);
}

static BOOLEAN GetRAMFieldParams(U8BIT rec_id, U16BIT field_id, U16BIT *offset, U16BIT *size)
{
   U8BIT i;
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(GetRAMFieldParams);

   ASSERT(rec_id < DBA_NUM_RECORDS);
   ASSERT(offset != NULL);
   ASSERT(size != NULL);

   for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
   {
      if (database_access_lut[rec_id].field[i].field_id == field_id)
      {
         *offset = database_access_lut[rec_id].field[i].ram_byte_offset;
         *size = database_access_lut[rec_id].field[i].ram_byte_size;
         if (*size > 0)
         {
            ret_val = TRUE;
         }
         break;
      }
   }

   FUNCTION_FINISH(GetRAMFieldParams);

   return(ret_val);
}

static void AddToNVMList(void *rec_ptr)
{
   U8BIT rec_id;
   U16BIT nvm_block;

   FUNCTION_START(AddToNVMList);

   ASSERT(rec_ptr != NULL);

   rec_id = STB_GetRAMRecordId(rec_ptr);
   nvm_block = STB_GetRAMRecordNVMBlock(rec_ptr);
   if (STB_GetRAMRecordPrevNVMBlock(rec_ptr) == NVM_INVALID_BLOCK_ID)
   {
      SetStartNVMList(rec_id, nvm_block);
      STB_SetNextNVMBlock(nvm_block, STB_GetRAMRecordNextNVMBlock(rec_ptr));
   }
   else
   {
      STB_SetNextNVMBlock(STB_GetRAMRecordPrevNVMBlock(rec_ptr), nvm_block);
      STB_SetNextNVMBlock(nvm_block, STB_GetRAMRecordNextNVMBlock(rec_ptr));
   }

   FUNCTION_FINISH(AddToNVMList);
}

static void SetStartNVMList(U8BIT rec_id, U16BIT block_no)
{
   U8BIT i;
   U16BIT offset;
   DATABASE_DESCRIPTOR db_desc;
   RECORD_DESCRIPTOR rec_desc;

   FUNCTION_START(SetStartNVMList);

   ASSERT(rec_id < DBA_NUM_RECORDS);

   offset = (U16BIT)sizeof(DATABASE_FORMAT);
   if (STB_ReadNVMData(offset, sizeof(DATABASE_DESCRIPTOR), (U8BIT *)&db_desc))
   {
      offset += (U16BIT)sizeof(DATABASE_DESCRIPTOR);
      for (i = 0; i < db_desc.num_records; i++)
      {
         if (STB_ReadNVMData(offset, sizeof(RECORD_DESCRIPTOR), (U8BIT *)&rec_desc))
         {
            if (rec_desc.record_id == rec_id)
            {
               // endian independent version of:
               // rec_desc.first_block = block_no;
               STB_SetBE16Bit(&rec_desc.first_block, block_no);
               STB_WriteNVMData(offset, sizeof(RECORD_DESCRIPTOR), (U8BIT *)&rec_desc);
               break;
            }
            offset += (U16BIT)sizeof(RECORD_DESCRIPTOR);
            offset += (U16BIT)(sizeof(FIELD_DESCRIPTOR) * rec_desc.num_fields);
         }
      }
   }

   FUNCTION_FINISH(SetStartNVMList);
}

static U16BIT GetStartNVMList(U8BIT *data_ptr, U8BIT rec_id)
{
   U8BIT i;
   U16BIT offset;
   DATABASE_DESCRIPTOR db_desc;
   RECORD_DESCRIPTOR rec_desc;
   U16BIT ret_val = NVM_INVALID_BLOCK_ID;

   FUNCTION_START(GetStartNVMList);

   ASSERT(rec_id < DBA_NUM_RECORDS);

   if (data_ptr == NULL)
   {
      // read fron NVM
      offset = (U16BIT)sizeof(DATABASE_FORMAT);
      if (STB_ReadNVMData(offset, sizeof(DATABASE_DESCRIPTOR), (U8BIT *)&db_desc))
      {
         offset += (U16BIT)sizeof(DATABASE_DESCRIPTOR);
         for (i = 0; i < db_desc.num_records; i++)
         {
            if (STB_ReadNVMData(offset, sizeof(RECORD_DESCRIPTOR), (U8BIT *)&rec_desc))
            {
               if (rec_desc.record_id == rec_id)
               {
                  // endian independent version of:
                  // ret_val = rec_desc.first_block;
                  ret_val = STB_GetBE16Bit(&rec_desc.first_block);
                  break;
               }
               offset += (U16BIT)sizeof(RECORD_DESCRIPTOR);
               offset += (U16BIT)(sizeof(FIELD_DESCRIPTOR) * rec_desc.num_fields);
            }
         }
      }
   }
   else
   {
      // read fron RAM buffer
      data_ptr += (U16BIT)sizeof(DATABASE_FORMAT);
      memcpy((void *)&db_desc, (void *)data_ptr, (size_t)sizeof(DATABASE_DESCRIPTOR));
      data_ptr += (U16BIT)sizeof(DATABASE_DESCRIPTOR);
      for (i = 0; i < db_desc.num_records; i++)
      {
         memcpy((void *)&rec_desc, (void *)data_ptr, (size_t)sizeof(RECORD_DESCRIPTOR));
         if (rec_desc.record_id == rec_id)
         {
            // endian independent version of:
            // ret_val = rec_desc.first_block;
            ret_val = STB_GetBE16Bit(&rec_desc.first_block);
            break;
         }
         data_ptr += (U16BIT)sizeof(RECORD_DESCRIPTOR);
         data_ptr += (U16BIT)(sizeof(FIELD_DESCRIPTOR) * rec_desc.num_fields);
      }
   }

   FUNCTION_FINISH(GetStartNVMList);

   return(ret_val);
}

static void RemoveFromNVMList(void *rec_ptr)
{
   U8BIT rec_id;

   FUNCTION_START(RemoveFromNVMList);

   ASSERT(rec_ptr != NULL);

   rec_id = STB_GetRAMRecordId(rec_ptr);
   if (STB_GetRAMRecordPrevNVMBlock(rec_ptr) == NVM_INVALID_BLOCK_ID)
   {
      SetStartNVMList(rec_id, STB_GetRAMRecordNextNVMBlock(rec_ptr));
   }
   else
   {
      STB_SetNextNVMBlock(STB_GetRAMRecordPrevNVMBlock(rec_ptr), STB_GetRAMRecordNextNVMBlock(rec_ptr));
   }

   FUNCTION_FINISH(RemoveFromNVMList);
}

static void DestroyRecord(void *rec_ptr)
{
   U8BIT rec_id, i;
   U16BIT nvm_block, field_id, field_offset, field_size;
   void *field_val;

   FUNCTION_START(DestroyRecord);

   ASSERT(rec_ptr != NULL);

   /* Free memory used by any pointer fields */
   rec_id = STB_GetRAMRecordId(rec_ptr);
   for (i = 0; i < database_access_lut[rec_id].num_fields; i++)
   {
      field_id = database_access_lut[rec_id].field[i].field_id;
      if (GetFieldType(field_id) == PTR_TYPE)
      {
         if (GetRAMFieldParams(rec_id, field_id, &field_offset, &field_size))
         {
            ASSERT(field_size == sizeof(void*));
            field_val = STB_GetRAMRecordPointer(rec_ptr, field_offset);

            if (field_val != NULL)
            {
               STB_FreeMemory(field_val);
            }
         }
      }
   }

   // if it has a NVM record
   nvm_block = STB_GetRAMRecordNVMBlock(rec_ptr);
   if (nvm_block != NVM_INVALID_BLOCK_ID)
   {
      // repair linked list
      RemoveFromNVMList(rec_ptr);

      // destroy NVM record
      STB_DestroyNVMRecord(nvm_block);
   }

   // destroy RAM record
   STB_DestroyRAMRecord(rec_ptr);

   FUNCTION_FINISH(DestroyRecord);
}

