/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Check sum calculation routines
 *
 * @file    stbcsum.c
 * @date    14/10/2000
 */

//---includes for this file----------------------------------------------------
// compiler library header files

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

//---constant definitions for this file----------------------------------------

//---local typedef structs for this file---------------------------------------

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

//---local function definitions------------------------------------------------


//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Calculates the checksum to zero for the data block provided.
 *
 * @param   U8BIT* data_ptr - the pointer to the data block
 * @param   U32BIT data_size - size of the data
 *
 * @return   U8BIT - checksum value.
 *
 */
U8BIT STB_CalcChecksum(U8BIT *data_ptr, U32BIT data_size)
{
   U8BIT cs;
   U32BIT i;

//   FUNCTION_START(STB_CalcChecksum);

   ASSERT(data_ptr != NULL);
   ASSERT(data_size > 0);

   cs = 0;
   for (i = 0; i < (data_size - 1); i++)
   {
      cs += *(data_ptr + i);
   }
   cs = (0xff - cs + 1);

//   FUNCTION_FINISH(STB_CalcChecksum);

   return(cs);
}

/**
 *

 *
 * @brief   Validates the checksum to zero for the block pointer provided.
 *
 * @param   U8BIT* data_ptr - the pointer to the data block
 * @param   U32BIT data_size - size of the data
 *
 * @return   BOOLEAN - TRUE if checksum value is valid.
 *
 */
BOOLEAN STB_CheckChecksum(U8BIT *data_ptr, U32BIT data_size)
{
   U8BIT cs;
   U32BIT i;
   BOOLEAN ret_val;

//   FUNCTION_START(STB_CheckChecksum);

   ASSERT(data_ptr != NULL);
   ASSERT(data_size > 0);

   cs = 0;
   for (i = 0; i < data_size; i++)
   {
      cs += *(data_ptr + i);
   }

   if (cs == 0)
   {
      ret_val = TRUE;
   }
   else
   {
      ret_val = FALSE;
   }

//   FUNCTION_FINISH(STB_CheckChecksum);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns 16bit int from address, assuming value is stored in Big endian format.
 *
 * @param   U16BIT* addr - pointer to 16bit int
 *
 * @return   U16BIT - integer value.
 *
 */
U16BIT STB_GetBE16Bit(U16BIT *addr)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetBE16Bit);

   ASSERT(addr != NULL);

   ret_val = (U16BIT)((U8BIT *)addr)[0];
   ret_val <<= 8;
   ret_val |= (U16BIT)((U8BIT *)addr)[1];

   FUNCTION_FINISH(STB_GetBE16Bit);

   return(ret_val);
}

/**
 *

 *
 * @brief   Stores 16bit int in address, forcing value to be stored in Big endian format.
 *
 * @param   U16BIT* addr - pointer to 16bit int
 * @param   U16BIT value - integer value
 *

 *
 */
void STB_SetBE16Bit(U16BIT *addr, U16BIT value)
{
   FUNCTION_START(STB_SetBE16Bit);

   ASSERT(addr != NULL);

   ((U8BIT *)addr)[0] = (U8BIT)(value >> 8);
   ((U8BIT *)addr)[1] = (U8BIT)(value & 0x00ff);

   FUNCTION_FINISH(STB_SetBE16Bit);
}

//*****************************************************************************
// End of file
//*****************************************************************************

