/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Non-volatile memory database functions
 *
 * @file    stbdbnvm.c
 * @date    06/09/2000
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "dba.h"
#include "dba_nvm.h"
#include "stbdbnvm.h"
#include "stbnvm.h"
#include "stbcsum.h"

#include "stbheap.h"

//---constant definitions for this file----------------------------------------

#define BLOCK_PAYLOAD_SIZE    20
#define FIRST_BLOCK_MASK      0x80
#define LINKED_BLOCK_MASK     0x40
#define RECID_BLOCK_MASK      0x3f

//---local typedef structs for this file---------------------------------------

typedef struct
{
   U8BIT payload[BLOCK_PAYLOAD_SIZE];
   U16BIT next_blk;
   U8BIT block_type;
   U8BIT checksum;
} NVM_BLOCK_REC;

typedef struct
{
   U16BIT blk_id;
   NVM_BLOCK_REC blk_data;
   BOOLEAN writing;
} NVM_RECOVER_REC;

typedef struct
{
   U8BIT *blk_map;
   U8BIT *ram_data;
   U16BIT blk_off;
   U16BIT num_blks;
   BOOLEAN changed;
} NVM_DB_STATUS_REC;

typedef struct
{
   U8BIT *data;
   U16BIT size;
   U16BIT blk_no;
   BOOLEAN changed;
} NVM_CACHE_STATUS_REC;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static NVM_DB_STATUS_REC db_status;
static NVM_CACHE_STATUS_REC cache_status;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static U16BIT GetBlockMap(U16BIT last_blk, BOOLEAN setclr);
static void SetBlockMap(U16BIT blk, BOOLEAN setclr);

static BOOLEAN ReadBlock(U16BIT blk_no, NVM_BLOCK_REC *blk_ptr, BOOLEAN check);
static BOOLEAN WriteBlock(U16BIT blk_no, NVM_BLOCK_REC *blk_ptr, BOOLEAN check);

static BOOLEAN CacheNVMRecord(U16BIT block_no);
static BOOLEAN SaveNVMCache(void);

//---local function definitions------------------------------------------------

/**
 *

 *
 * @brief   Finds the next block following last block in block map matching given value.
 *
 * @param   U16BIT last_blk - the number of the last block found (NVM_INVALID_BLOCK_ID if none)
 * @param   BOOLEAN setclr - TRUE to test for bit set, FALSE to test for bit clear
 *
 * @return   U16BIT - number of next block.
 *
 */
static U16BIT GetBlockMap(U16BIT last_blk, BOOLEAN setclr)
{
   U16BIT byte;
   U8BIT bit;
   U16BIT blk;
   U16BIT next_blk = NVM_INVALID_BLOCK_ID;

   FUNCTION_START(GetBlockMap);

   if (last_blk == NVM_INVALID_BLOCK_ID)
   {
      // start from first block
      blk = 0;
   }
   else
   {
      // or one after last block
      blk = last_blk + 1;
   }

   while (blk < db_status.num_blks)
   {
      byte = (U16BIT)(blk / 8);
      bit = (U8BIT)(blk % 8);
      if ((db_status.blk_map[byte] & (0x80 >> bit)) == 0x00)
      {
         if (setclr == FALSE)
         {
            next_blk = blk;
            break;
         }
      }
      else
      {
         if (setclr == TRUE)
         {
            next_blk = blk;
            break;
         }
      }
      blk++;
   }

   FUNCTION_FINISH(GetBlockMap);

   return(next_blk);
}

/**
 *

 *
 * @brief   Sets the given block number in block map to given value.
 *
 * @param   U16BIT blk - the number of the block
 * @param   BOOLEAN setclr - TRUE to set bit, FALSE to clear bit
 *

 *
 */
static void SetBlockMap(U16BIT blk, BOOLEAN setclr)
{
   U16BIT byte;
   U8BIT bit;

   FUNCTION_START(SetBlockMap);

   ASSERT(blk < db_status.num_blks);

   byte = (U16BIT)(blk / 8);
   bit = (U8BIT)(blk % 8);

   if (setclr == TRUE)
   {
      db_status.blk_map[byte] |= (0x80 >> bit);
   }
   else
   {
      db_status.blk_map[byte] &= ~(0x80 >> bit);
   }

   FUNCTION_FINISH(SetBlockMap);
}

/**
 *

 *
 * @brief   Reads data from the block number of NVM given into the block pointer provided.
 *
 * @param   U16BIT blk_no - id of block in NVM
 * @param   NVM_BLOCK_REC* blk_ptr - the pointer to the NVM block
 * @param   BOOLEAN check - TRUE to validate checksum, else FALSE
 *
 * @return   BOOLEAN - TRUE on success, else FALSE.
 *
 */
static BOOLEAN ReadBlock(U16BIT blk_no, NVM_BLOCK_REC *blk_ptr, BOOLEAN check)
{
   U32BIT offset;
   BOOLEAN ret_val = FALSE;

//   FUNCTION_START(ReadBlock);

   ASSERT(blk_no < db_status.num_blks);
   ASSERT(blk_ptr != NULL);

   // get NVM address
   offset = (U32BIT)db_status.blk_off +
            ((U32BIT)blk_no * (U32BIT)sizeof(NVM_BLOCK_REC));
   if (db_status.ram_data == NULL)
   {
      // read block
      if (STB_NVMSTBRead(offset, (U32BIT)sizeof(NVM_BLOCK_REC), (U8BIT *)blk_ptr))
      {
         ret_val = TRUE;
      }
   }
   else
   {
      // copy data
      memcpy((void *)blk_ptr, (void *)(db_status.ram_data + offset), (size_t)sizeof(NVM_BLOCK_REC));
      ret_val = TRUE;
   }

   // calculate checksum and compare
   if (check && ret_val && (blk_ptr->block_type != 0xff))
   {
      if (STB_CheckChecksum((U8BIT *)blk_ptr, (U32BIT)sizeof(NVM_BLOCK_REC)) == FALSE)
      {
         ret_val = FALSE;
      }
   }

//   FUNCTION_FINISH(ReadBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes data to the block number of NVM given from the block pointer provided.
 *
 * @param   U16BIT blk_no - id of block in NVM
 * @param   NVM_BLOCK_REC* blk_ptr - the pointer to the NVM block
 * @param   BOOLEAN check - TRUE to calc checksum, else FALSE
 *
 * @return   BOOLEAN - TRUE on success, else FALSE.
 *
 */
static BOOLEAN WriteBlock(U16BIT blk_no, NVM_BLOCK_REC *blk_ptr, BOOLEAN check)
{
   U32BIT offset;
   BOOLEAN ret_val = FALSE;

//   FUNCTION_START(WriteBlock);

   ASSERT(blk_no < db_status.num_blks);
   ASSERT(blk_ptr != NULL);

   // calculate checksum
   if (check)
   {
      blk_ptr->checksum = STB_CalcChecksum((U8BIT *)blk_ptr, (U32BIT)sizeof(NVM_BLOCK_REC));
   }

   // get NVM address
   offset = (U32BIT)db_status.blk_off +
            ((U32BIT)blk_no * (U32BIT)sizeof(NVM_BLOCK_REC));
   if (db_status.ram_data == NULL)
   {
      // write block
      if (STB_NVMSTBWrite(offset, (U32BIT)sizeof(NVM_BLOCK_REC), (U8BIT *)blk_ptr))
      {
         ret_val = TRUE;
      }
   }
   else
   {
      // copy data
      memcpy((void *)(db_status.ram_data + offset), (void *)blk_ptr, (size_t)sizeof(NVM_BLOCK_REC));
      db_status.changed = TRUE;
      ret_val = TRUE;
   }

//   FUNCTION_FINISH(WriteBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads the NVM record starting at the specified block number into RAM cache.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 *
 * @return   TRUE - on success, else FALSE.
 *
 */
static BOOLEAN CacheNVMRecord(U16BIT block_no)
{
   NVM_BLOCK_REC curr_blk;
   BOOLEAN ret_val = TRUE;

   FUNCTION_START(CacheNVMRecord);

   ASSERT(block_no < db_status.num_blks);

   // if this record is not already cached
   if ((cache_status.blk_no != block_no) || (cache_status.data == NULL))
   {
      // save cache in NVM and delete
      SaveNVMCache();

      if (cache_status.data != NULL)
      {
         STB_FreeMemory((void *)cache_status.data);
      }

      //update flags for new record
      cache_status.data = NULL;
      cache_status.size = 0;
      cache_status.blk_no = block_no;
      cache_status.changed = FALSE;

      // read payloads of all blocks of this record into cache
      while (block_no != NVM_INVALID_BLOCK_ID)
      {
         // read contents of block
         if (ReadBlock(block_no, &curr_blk, TRUE))
         {
            // get memory for new cache data area
            cache_status.data = (U8BIT *)STB_ResizeMemory((void *)cache_status.data,
                  (U32BIT)(cache_status.size + BLOCK_PAYLOAD_SIZE));
            if (cache_status.data != NULL)
            {
               // copy new payload
               memcpy((void *)(cache_status.data + cache_status.size),
                  (void *)curr_blk.payload, (size_t)BLOCK_PAYLOAD_SIZE);
               cache_status.size += BLOCK_PAYLOAD_SIZE;

               // get next block number
               if ((curr_blk.block_type & LINKED_BLOCK_MASK) != 0)
                  // endian independent version of:
                  // block_no = curr_blk.next_blk;
                  block_no = STB_GetBE16Bit(&curr_blk.next_blk);
               else
                  block_no = NVM_INVALID_BLOCK_ID;
            }
            else
            {
               // malloc failed so cache failed
               ret_val = FALSE;
               break;
            }
         }
         else
         {
            // read failed so cache failed
            ret_val = FALSE;
            break;
         }
      }
   }

   FUNCTION_FINISH(CacheNVMRecord);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes the NVM record in RAM cache back into NVM.
 *

 *
 * @return   TRUE - on success, else FALSE.
 *
 */
static BOOLEAN SaveNVMCache(void)
{
   U8BIT *src_ptr;
   U16BIT block_no;
   NVM_BLOCK_REC curr_blk;
   BOOLEAN ret_val = TRUE;

   FUNCTION_START(SaveNVMCache);

   // if cache is valid and has been changed
   if ((cache_status.changed == TRUE) && (cache_status.data != NULL))
   {
      src_ptr = cache_status.data;
      cache_status.changed = FALSE;

      // write cache back into payloads of all blocks of this record
      block_no = cache_status.blk_no;
      while (block_no != NVM_INVALID_BLOCK_ID)
      {
         // read contents of block
         if (ReadBlock(block_no, &curr_blk, TRUE))
         {
            // copy payload into block and back into NVM
            memcpy((void *)curr_blk.payload, (void *)src_ptr, (size_t)BLOCK_PAYLOAD_SIZE);
            if (WriteBlock(block_no, &curr_blk, TRUE))
            {
               // get next block number
               src_ptr += BLOCK_PAYLOAD_SIZE;
               if ((curr_blk.block_type & LINKED_BLOCK_MASK) != 0)
                  // endian independent version of:
                  // block_no = curr_blk.next_blk;
                  block_no = STB_GetBE16Bit(&curr_blk.next_blk);
               else
                  block_no = NVM_INVALID_BLOCK_ID;
            }
            else
            {
               // write failed so save failed
               ret_val = FALSE;
               break;
            }
         }
         else
         {
            // read failed so save failed
            ret_val = FALSE;
            break;
         }
      }
   }

   FUNCTION_FINISH(SaveNVMCache);

   return(ret_val);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises parameters needed for NVM block access
 *
 * @param   U16BIT offset - offset to NVM blocks in bytes
 *

 *
 */
void STB_InitNVMAccess(U16BIT offset)
{
   U32BIT size;

   FUNCTION_START(STB_InitNVMAccess);

   // calculate size of NVM available for blocks
   size = STB_NVMGetSTBSize();
   size -= (U32BIT)offset;
   size -= (U32BIT)sizeof(NVM_RECOVER_REC);

   // setup status vars
   db_status.blk_map = NULL;
   db_status.ram_data = NULL;
   db_status.blk_off = offset;
   db_status.num_blks = (U16BIT)(size / (U32BIT)sizeof(NVM_BLOCK_REC));
   db_status.changed = FALSE;

   FUNCTION_FINISH(STB_InitNVMAccess);
}

/**
 *

 *
 * @brief   Initialises parameters needed for NVM block access (calcs num blocks if required)
 *

 *

 *
 */
void STB_InitNVMMap(void)
{
   U32BIT size;
   U16BIT byte, i;
   U8BIT bit;
   NVM_BLOCK_REC curr_blk;

   FUNCTION_START(STB_InitNVMMap);

   // calculate size of RAM needed for map
   size = (U32BIT)(db_status.num_blks / 8);
   if ((db_status.num_blks % 8) > 0)
      size++;

   // delete old block map
   if (db_status.blk_map != NULL)
   {
      STB_FreeMemory((void *)db_status.blk_map);
   }

   // create new block map
   db_status.blk_map = (U8BIT *)STB_GetMemory(size);
   if (db_status.blk_map != NULL)
   {
      memset((void *)db_status.blk_map, 0xff, (size_t)size);

      for (i = 0; i < db_status.num_blks; i++)
      {
         // read a block
         if (ReadBlock(i, &curr_blk, FALSE))
         {
            // if block is spare
            if (curr_blk.block_type == 0xff)
            {
               // clear this bit of the map
               byte = (U16BIT)(i / 8);
               bit = (U8BIT)(i % 8);
               db_status.blk_map[byte] &= ~(0x80 >> bit);
            }
         }
      }
   }

   FUNCTION_FINISH(STB_InitNVMMap);
}

/**
 *

 *
 * @brief   Reads each database block from NVM and checks its checksum. If the data has already
 *                 been read into the RAM cache then the data will be checked from here, but this data
 *                 has been read from NVM and is an exact copy, so the check is still valid.
 *                 Note: It's assumed that STB_InitNVMAccess has already been called.
 *

 *
 * @return   TRUE if all checksums are valid, FALSE otherwise
 *
 */
BOOLEAN STB_CheckNVMDatabaseIntegrity(void)
{
   BOOLEAN retval;
   U16BIT i;
   NVM_BLOCK_REC blk;

   FUNCTION_START(STB_CheckNVMDatabaseIntegrity);

   retval = TRUE;

   /* Read each block to check it's checksum */
   for (i = 0; (i < db_status.num_blks) && retval; i++)
   {
      if (!ReadBlock(i, &blk, TRUE))
      {
         retval = FALSE;
      }
   }

   FUNCTION_FINISH(STB_CheckNVMDatabaseIntegrity);

   return(retval);
}

/**
 *

 *
 * @brief   Sets all DB NVM access to RAM block supplied instead of NVM (cancel if NULL).
 *
 * @param   U8BIT* ram_ptr - pointer to RAM area
 *

 *
 */
void STB_SetNVMAccessRAM(U8BIT *ram_ptr)
{
   FUNCTION_START(STB_SetNVMAccessRAM);

   db_status.ram_data = ram_ptr;

   FUNCTION_FINISH(STB_SetNVMAccessRAM);
}

/**
 *

 *
 * @brief   Returns the current RAM pointer used for DB NVM access
 *

 *
 * @return   U8BIT* - pointer to RAM area
 *
 */
U8BIT* STB_GetNVMAccessRAM(void)
{
   FUNCTION_START(STB_GetNVMAccessRAM);
   FUNCTION_FINISH(STB_GetNVMAccessRAM);
   return(db_status.ram_data);
}

/**
 *

 *
 * @brief   Returns total number of NVM database blocks.
 *

 *
 * @return   U16BIT - number of blocks.
 *
 */
U16BIT STB_GetNVMBlockCount(void)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetNVMBlockCount);

   ret_val = db_status.num_blks;

   FUNCTION_FINISH(STB_GetNVMBlockCount);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns total number of NVM database blocks in use.
 *

 *
 * @return   U16BIT - number of blocks.
 *
 */
U16BIT STB_GetNVMBlocksUsed(void)
{
   U16BIT block_no, ret_val;

   FUNCTION_START(STB_GetNVMBlocksUsed);

   ret_val = 0;
   if (db_status.blk_map != NULL)
   {
      block_no = GetBlockMap(NVM_INVALID_BLOCK_ID, TRUE);
      while (block_no != NVM_INVALID_BLOCK_ID)
      {
         ret_val++;
         block_no = GetBlockMap(block_no, TRUE);
      }
   }
   FUNCTION_FINISH(STB_GetNVMBlocksUsed);

   return(ret_val);
}

/**
 * @brief   Returns the size of an NVM database block, in bytes
 * @return  size of block in bytes
 */
U16BIT STB_GetNVMBlockSize(void)
{
   FUNCTION_START(STB_GetNVMBlockSize);
   FUNCTION_FINISH(STB_GetNVMBlockSize);
   return(sizeof(NVM_BLOCK_REC));
}

/**
 *

 *
 * @brief   Returns number of NVM database blocks needed for specified number of bytes.
 *
 * @param   U16BIT size - the size (in bytes)
 *
 * @return   U16BIT - number of blocks.
 *
 */
U16BIT STB_GetNVMBlocksNeeded(U16BIT size)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetNVMBlocksNeeded);

   ret_val = (size / BLOCK_PAYLOAD_SIZE);
   if ((size % BLOCK_PAYLOAD_SIZE) > 0)
      ret_val++;

   FUNCTION_FINISH(STB_GetNVMBlocksNeeded);

   return(ret_val);
}

/**
 *

 *
 * @brief   Creates a record of the type given in NVM by overwriting invalid blocks.
 *
 * @param   U8BIT rec_id - the ID of the record to be created
 * @param   U16BIT size - the size the record to be created (in bytes)
 *
 * @return   U16BIT - block number created if OK, else NVM_INVALID_BLOCK_ID.
 *
 */
U16BIT STB_CreateNVMRecord(U8BIT rec_id, U16BIT size)
{
   U16BIT block_no, next_block, first_block, num_blks;
   NVM_BLOCK_REC curr_blk;

   FUNCTION_START(STB_CreateNVMRecord);

   ASSERT(size > 0);
   ASSERT(db_status.blk_map != NULL);

   // decide how many blocks we need for this size record
   num_blks = STB_GetNVMBlocksNeeded(size);

   // search NVM for spare block(s) required
   first_block = NVM_INVALID_BLOCK_ID;
   block_no = GetBlockMap(NVM_INVALID_BLOCK_ID, FALSE);
   while (block_no != NVM_INVALID_BLOCK_ID)
   {
      // setup contents of linked block
      memset((void *)&curr_blk, 0, (size_t)sizeof(NVM_BLOCK_REC));
      curr_blk.block_type = rec_id;

      // mark first block and remember block number
      if (first_block == NVM_INVALID_BLOCK_ID)
      {
         curr_blk.block_type |= FIRST_BLOCK_MASK;
         first_block = block_no;
      }

      // mark all but last block and add link to next block
      if (num_blks > 1)
      {
         curr_blk.block_type |= LINKED_BLOCK_MASK;
         next_block = GetBlockMap(block_no, FALSE);
      }
      else
      {
         next_block = NVM_INVALID_BLOCK_ID;
      }
      // endian independent version of:
      // curr_blk.next_blk = next_block;
      STB_SetBE16Bit(&curr_blk.next_blk, next_block);

      // write contents of new NVM block
      if (WriteBlock(block_no, &curr_blk, TRUE))
      {
         // mark block as used
         SetBlockMap(block_no, TRUE);
         num_blks--;
         block_no = next_block;
      }
      else
      {
         // write failed so create failed
         break;
      }
   }

   FUNCTION_FINISH(STB_CreateNVMRecord);

   return(first_block);
}

/**
 *

 *
 * @brief   Destroys NVM record by marking all NVM blocks used by the record as invalid.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 *

 *
 */
void STB_DestroyNVMRecord(U16BIT block_no)
{
   U16BIT next_block;
   NVM_BLOCK_REC curr_blk;

   FUNCTION_START(STB_DestroyNVMRecord);

   ASSERT(block_no < db_status.num_blks);

   // for each block in the chain
   while (block_no != NVM_INVALID_BLOCK_ID)
   {
      // read contents of block
      if (ReadBlock(block_no, &curr_blk, TRUE))
      {
         // get next block number
         if ((curr_blk.block_type & LINKED_BLOCK_MASK) != 0)
            // endian independent version of:
            // next_block = curr_blk.next_blk;
            next_block = STB_GetBE16Bit(&curr_blk.next_blk);
         else
            next_block = NVM_INVALID_BLOCK_ID;

         // write old NVM block as unused
         memset((void *)&curr_blk, 0xff, (size_t)sizeof(NVM_BLOCK_REC));
         if (WriteBlock(block_no, &curr_blk, FALSE))
         {
            // mark block as empty
            SetBlockMap(block_no, FALSE);
            block_no = next_block;
         }
         else
         {
            // write failed so destroy failed
            break;
         }
      }
      else
      {
         // read failed so destroy failed
         break;
      }
   }

   FUNCTION_FINISH(STB_DestroyNVMRecord);
}

/**
 *

 *
 * @brief   Repairs NVM linked list by pointing the NVM record starting at the specified
 *                 block number to the record starting at the specified next block number.
 *
 * @param   U16BIT block_no - the NVM block number of the current record
 * @param   U16BIT next_block - the NVM block number of the next record
 *

 *
 */
void STB_SetNextNVMBlock(U16BIT block_no, U16BIT next_block)
{
   NVM_BLOCK_REC curr_blk;

   FUNCTION_START(STB_SetNextNVMBlock);

   ASSERT(block_no < db_status.num_blks);

   // for each block in the chain
   while (block_no != NVM_INVALID_BLOCK_ID)
   {
      // read contents of block
      if (ReadBlock(block_no, &curr_blk, TRUE))
      {
         if ((curr_blk.block_type & LINKED_BLOCK_MASK) != 0)
         {
            // get next block number
            // endian independent version of:
            // block_no = curr_blk.next_blk;
            block_no = STB_GetBE16Bit(&curr_blk.next_blk);
         }
         else
         {
            // or point current record to next record
            // endian independent version of:
            // curr_blk.next_blk = next_block;
            STB_SetBE16Bit(&curr_blk.next_blk, next_block);
            WriteBlock(block_no, &curr_blk, TRUE);
            block_no = NVM_INVALID_BLOCK_ID;
         }
      }
      else
      {
         // read failed so repair failed
         break;
      }
   }

   FUNCTION_FINISH(STB_SetNextNVMBlock);
}

/**
 *

 *
 * @brief   Returns the block number of the next NVM record pointed to by the NVM record
 *                 starting at the specified block number.
 *
 * @param   U16BIT block_no - the NVM block number of the current record
 *
 * @return   U16BIT - the NVM block number of the next record.
 *
 */
U16BIT STB_GetNextNVMBlock(U16BIT block_no)
{
   NVM_BLOCK_REC curr_blk;
   U16BIT ret_val = NVM_INVALID_BLOCK_ID;

   FUNCTION_START(STB_GetNextNVMBlock);

   ASSERT(block_no < db_status.num_blks);

   // for each block in the chain
   while (block_no != NVM_INVALID_BLOCK_ID)
   {
      // read contents of block
      if (ReadBlock(block_no, &curr_blk, TRUE))
      {
         // get next block number
         // endian independent version of:
         // block_no = curr_blk.next_blk;
         block_no = STB_GetBE16Bit(&curr_blk.next_blk);

         if ((curr_blk.block_type & LINKED_BLOCK_MASK) == 0)
         {
            // or get next record from current record
            ret_val = block_no;
            block_no = NVM_INVALID_BLOCK_ID;
         }
      }
      else
      {
         // read failed so repair failed
         break;
      }
   }

   FUNCTION_FINISH(STB_GetNextNVMBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Finds next record of the type given in NVM.
 *
 * @param   U8BIT rec_id - the ID of the record
 * @param   U16BIT last_blk - last block found (NVM_INVALID_BLOCK_ID if none)
 *
 * @return   U16BIT - block number found if OK, else NVM_INVALID_BLOCK_ID.
 *
 */
U16BIT STB_FindNVMRecordFromId(U8BIT rec_id, U16BIT last_blk)
{
   U16BIT block_no;
   NVM_BLOCK_REC curr_blk;
   U16BIT ret_val = NVM_INVALID_BLOCK_ID;

   FUNCTION_START(STB_FindNVMRecordFromId);

   ASSERT(rec_id < DBA_NUM_RECORDS);
   ASSERT(db_status.blk_map != NULL);

   // search NVM for used blocks from last position
   block_no = GetBlockMap(last_blk, TRUE);
   while (block_no != NVM_INVALID_BLOCK_ID)
   {
      // read a block
      if (ReadBlock(block_no, &curr_blk, TRUE))
      {
         // if first block in linked list
         if ((curr_blk.block_type & FIRST_BLOCK_MASK) != 0)
         {
            // if block of given type
            if ((curr_blk.block_type & RECID_BLOCK_MASK) == rec_id)
            {
               ret_val = block_no;
               break;
            }
         }
      }
      block_no = GetBlockMap(block_no, TRUE);
   }

   FUNCTION_FINISH(STB_FindNVMRecordFromId);

   return(ret_val);
}

void STB_NVMChanged(BOOLEAN state)
{
   FUNCTION_START(STB_NVMChanged);
   db_status.changed = state;
   FUNCTION_FINISH(STB_NVMChanged);
}

/**
 * @brief   Flush cached changes
 * @param   clear TRUE clear the cache after flushing cached changes, FALSE do not clear the cache
 */
void STB_NVMFlushCache(BOOLEAN clear)
{
   FUNCTION_START(STB_NVMFlushCache);

   SaveNVMCache();

   if (clear)
   {
      if (cache_status.data != NULL)
      {
         STB_FreeMemory((void *)cache_status.data);
         cache_status.data = NULL;
      }
      cache_status.changed = FALSE;
   }

   FUNCTION_FINISH(STB_NVMFlushCache);
}

/**
 *

 *
 * @brief   Saves the RAM cache data to NVM
 *

 *

 *
 */
void STB_NVMSave(void)
{
   FUNCTION_START(STB_NVMSave);

   /* Save current cache block */
   SaveNVMCache();

   /* Save the DB data to NVM if it's being cached in RAM and has been changed */
   if ((db_status.ram_data != NULL) && db_status.changed)
   {
      STB_NVMSTBWrite(0, STB_NVMGetSTBSize(), db_status.ram_data);
      db_status.changed = FALSE;
   }

   FUNCTION_FINISH(STB_NVMSave);
}

/**
 *
 * @brief   Writes the specified value into a field of a NVM record.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 * @param   U16BIT offset - field bit offset into the record
 * @param   U16BIT size - field bit size in the record
 * @param   U8BIT* string - the string value of the field
 *
 */
void STB_SetNVMRecordString(U16BIT block_no, U16BIT offset, U16BIT size, U8BIT *string)
{
   U16BIT i;
   U8BIT *data_ptr;

   FUNCTION_START(STB_SetNVMRecordString);

   ASSERT(block_no < db_status.num_blks);

   // read NVM record into cache
   if (CacheNVMRecord(block_no))
   {
      // find start of field and offset in first byte
      data_ptr = cache_status.data;
      data_ptr += (offset / 8);

      if (size != 0)
      {
         // for string type, copy data from pointer
         for (i = 0; i < (size / 8); i++)
         {
            *data_ptr++ = string[i];
         }
         *(data_ptr - 1) = '\0';      // ensure string is null terminated
      }
      else
      {
         *data_ptr = '\0';      // ensure string is null terminated
      }

      cache_status.changed = TRUE;
   }

   FUNCTION_FINISH(STB_SetNVMRecordString);
}

/**
 *
 * @brief   Writes the specified value into a field of a NVM record.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 * @param   U16BIT offset - field bit offset into the record
 * @param   U16BIT size - field bit size in the record
 * @param   U32BIT value - the numeric value of the field
 *
 */
void STB_SetNVMRecordNumber(U16BIT block_no, U16BIT offset, U16BIT size, U32BIT value)
{
   U8BIT *data_ptr;

   FUNCTION_START(STB_SetNVMRecordNumber);

   ASSERT(block_no < db_status.num_blks);
   ASSERT(size > 0);

   // read NVM record into cache
   if (CacheNVMRecord(block_no))
   {
      // find end of field and offset in last byte
      data_ptr = cache_status.data;
      data_ptr += ((offset + size - 1) / 8);
      offset = (7 - ((offset + size - 1) % 8));

      // or for any other type, copy value LSB to field in reverse order
      // forces data into big endian format
      while (size > 0)
      {
         *data_ptr &= (U8BIT) ~(0x00000001 << offset);
         *data_ptr |= (U8BIT)((value & 0x00000001) << offset);
         value = (value >> 1);
         // at start of byte, get previous byte
         offset++;
         if (offset > 7)
         {
            offset = 0;
            data_ptr--;
         }
         size--;
      }

      cache_status.changed = TRUE;
   }

   FUNCTION_FINISH(STB_SetNVMRecordNumber);
}

/**
 *
 * @brief   Reads string of a field from a NVM record.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 *
 * @return  U8BIT* - string of the field.
 *
 */
U8BIT* STB_GetNVMRecordString(U16BIT block_no, U16BIT offset, U16BIT size)
{
   U8BIT *data_ptr;

   FUNCTION_START(STB_GetNVMRecordString);
   USE_UNWANTED_PARAM(size);

   ASSERT(block_no < db_status.num_blks);

   // read NVM record into cache
   if (CacheNVMRecord(block_no))
   {
      // find start of field and offset in first byte
      data_ptr = cache_status.data;
      data_ptr += (offset / 8);
   }
   else
   {
      data_ptr = NULL;
   }

   FUNCTION_FINISH(STB_GetNVMRecordString);

   return(data_ptr);
}

/**
 *
 * @brief   Reads the specified value of a field from a NVM record.
 *
 * @param   U16BIT block_no - the first NVM block number of the record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 *
 * @return   U32BIT - the numeric value of the field.
 *
 */
U32BIT STB_GetNVMRecordNumber(U16BIT block_no, U16BIT offset, U16BIT size)
{
   U8BIT *data_ptr;
   U32BIT value = 0;

   FUNCTION_START(STB_GetNVMRecordNumber);

   ASSERT(block_no < db_status.num_blks);
   ASSERT(size > 0);

   // read NVM record into cache
   if (CacheNVMRecord(block_no))
   {
      // find start of field and offset in first byte
      data_ptr = cache_status.data;
      data_ptr += (offset / 8);
      offset = (offset % 8);

      // or for any other type, copy each field bit to LSB of value
      // expects data in big endian format
      value = 0;
      while (size > 0)
      {
         value = (value << 1);
         value |= (U32BIT)((*data_ptr >> (7 - offset)) & 0x01);
         // at end of byte, get next byte
         offset++;
         if (offset > 7)
         {
            offset = 0;
            data_ptr++;
         }
         size--;
      }
   }

   FUNCTION_FINISH(STB_GetNVMRecordNumber);

   return(value);
}

/**
 *

 *
 * @brief   Writes the given data to NVM.
 *
 * @param   U16BIT offset - byte offset at which to write the data
 * @param   U16BIT size - number of bytes of data to write
 * @param   U8BIT* data_ptr - pointer to the data to be written
 *

 *
 */
void STB_WriteNVMData(U16BIT offset, U16BIT size, U8BIT *data_ptr)
{
   PU8BIT addr;

   FUNCTION_START(STB_WriteNVMData);

   if (db_status.ram_data == NULL)
   {
      /* Write the data directly to NVM */
      STB_NVMSTBWrite((U32BIT)offset, (U32BIT)size, data_ptr);
   }
   else
   {
      /* Copy the data into the RAM cache */
      addr = db_status.ram_data + offset;
      memcpy((void *)addr, (void *)data_ptr, (size_t)size);
      db_status.changed = TRUE;
   }

   FUNCTION_FINISH(STB_WriteNVMData);
}

/**
 *

 *
 * @brief   Reads data from the NVM into the given buffer.
 *
 * @param   U16BIT offset - byte offset from which to read the data
 * @param   U16BIT size - number of bytes of data to read
 * @param   U8BIT* data_ptr - address of buffer to read data into
 *
 * @return   BOOLEAN - TRUE on success, else FALSE.
 *
 */
BOOLEAN STB_ReadNVMData(U16BIT offset, U16BIT size, U8BIT *data_ptr)
{
   BOOLEAN ret_val = FALSE;

   FUNCTION_START(STB_ReadNVMData);

   if (db_status.ram_data == NULL)
   {
      /* Read the data directly to NVM */
      if (STB_NVMSTBRead((U32BIT)offset, (U32BIT)size, data_ptr))
      {
         ret_val = TRUE;
      }
   }
   else
   {
      /* Copy the data from the RAM cache */
      memcpy((void *)data_ptr, (void *)(db_status.ram_data + offset), (size_t)size);
      ret_val = TRUE;
   }

   FUNCTION_FINISH(STB_ReadNVMData);

   return(ret_val);
}

//*****************************************************************************
// End of file
//*****************************************************************************

