/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   RAM database functions
 *
 * @file    stbdbram.c
 * @date    06/09/2000
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "dba.h"
#include "dba_nvm.h"
#include "stbdbram.h"
#include "stbdbnvm.h"
#include "stbllist.h"
#include "stbheap.h"

//---constant definitions for this file----------------------------------------

//---local typedef structs for this file---------------------------------------

typedef struct
{
   LINK_LIST_PTR_BLK list_ptrs;
   U8BIT rec_id;
   U16BIT nvm_block;
   void *parent;
} RAM_DB_HEADER;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static LINK_LIST_HEADER *llist_hdr_array;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

//---local function definitions------------------------------------------------

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises parameters needed for RAM record access
 *

 *

 *
 */
void STB_InitRAMAccess(void)
{
   U8BIT i;

   FUNCTION_START(STB_InitRAMAccess);

   // create and init linked list headers
   llist_hdr_array = (LINK_LIST_HEADER *)STB_GetMemory((U32BIT)(sizeof(LINK_LIST_HEADER) * DBA_NUM_RECORDS));
   if (llist_hdr_array != NULL)
   {
      for (i = 0; i < DBA_NUM_RECORDS; i++)
      {
         llist_hdr_array[i].null_next = NULL;
         llist_hdr_array[i].first = &llist_hdr_array[i];
         llist_hdr_array[i].last = &llist_hdr_array[i];
      }
   }

   FUNCTION_FINISH(STB_InitRAMAccess);
}

/**
 *

 *
 * @brief   Initialises RAM database by destroying all records and linked lists.
 *

 *

 *
 */
void STB_PurgeRAMRecords(void)
{
   U8BIT i;
   void *rec_ptr;
   void *next_ptr;

   FUNCTION_START(STB_PurgeRAMRecords);

   // for each record list
   for (i = 0; i < DBA_NUM_RECORDS; i++)
   {
      // get first record in list
      rec_ptr = (void *)STB_LLGetFirstBlock(&llist_hdr_array[i]);
      while (rec_ptr != NULL)
      {
         // get next record in list
         next_ptr = (void *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)rec_ptr);

         // remove record from linked list
         STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)rec_ptr);

         // delete memory
         STB_FreeMemory(rec_ptr);

         rec_ptr = next_ptr;
      }
   }

   FUNCTION_FINISH(STB_PurgeRAMRecords);
}

/**
 *

 *
 * @brief   Creates a record of the type given in RAM (mallocs block).
 *
 * @param   U8BIT rec_id - the ID of the record to be created
 * @param   U16BIT size - the size the record to be created (in bytes)
 * @param   U16BIT nvm_block - the ID of the corresponding NVM block
 * @param   void* parent - the pointer to record parent
 *
 * @return   void* - pointer to record created, NULL if failed.
 *
 */
void* STB_CreateRAMRecord(U8BIT rec_id, U16BIT size, U16BIT nvm_block, void *parent)
{
   void *rec_ptr;

   FUNCTION_START(STB_CreateRAMRecord);

   ASSERT(rec_id < DBA_NUM_RECORDS);

   // add space for rec header
   size += (U16BIT)sizeof(RAM_DB_HEADER);

   // get memory in RAM
   rec_ptr = STB_GetMemory((U32BIT)size);
   if (rec_ptr != NULL)
   {
      // setup contents of new memory block
      memset(rec_ptr, 0, size);
      ((RAM_DB_HEADER *)rec_ptr)->rec_id = rec_id;
      ((RAM_DB_HEADER *)rec_ptr)->nvm_block = nvm_block;
      ((RAM_DB_HEADER *)rec_ptr)->parent = parent;

      // add record to linked list
      STB_LLAddBlockToEnd(&llist_hdr_array[rec_id], (LINK_LIST_PTR_BLK *)rec_ptr);
   }

   FUNCTION_FINISH(STB_CreateRAMRecord);

   return(rec_ptr);
}

/**
 *

 *
 * @brief   Destroys record given in RAM (frees block).
 *
 * @param   void* rec_ptr - pointer to the record to be deleted
 *

 *
 */
void STB_DestroyRAMRecord(void *rec_ptr)
{
   FUNCTION_START(STB_DestroyRAMRecord);

   ASSERT(rec_ptr != NULL);

   // remove record from linked list
   STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)rec_ptr);

   // delete memory
   STB_FreeMemory(rec_ptr);

   FUNCTION_FINISH(STB_DestroyRAMRecord);
}

/**
 *

 *
 * @brief   Returns record type id for given record pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 *
 * @return   U8BIT - record id.
 *
 */
U8BIT STB_GetRAMRecordId(void *rec_ptr)
{
   U8BIT ret_val;

   FUNCTION_START(STB_GetRAMRecordId);

   ASSERT(rec_ptr != NULL);

   ret_val = ((RAM_DB_HEADER *)rec_ptr)->rec_id;

   FUNCTION_FINISH(STB_GetRAMRecordId);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns NVM block number for given record pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 *
 * @return   U16BIT - NVM block.
 *
 */
U16BIT STB_GetRAMRecordNVMBlock(void *rec_ptr)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetRAMRecordNVMBlock);

   ASSERT(rec_ptr != NULL);

   ret_val = ((RAM_DB_HEADER *)rec_ptr)->nvm_block;

   FUNCTION_FINISH(STB_GetRAMRecordNVMBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns NVM block number for previous record of given pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 *
 * @return   U16BIT - NVM block.
 *
 */
U16BIT STB_GetRAMRecordPrevNVMBlock(void *rec_ptr)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetRAMRecordPrevNVMBlock);

   ASSERT(rec_ptr != NULL);

   rec_ptr = (void *)STB_LLGetPrevBlock((LINK_LIST_PTR_BLK *)rec_ptr);
   if (rec_ptr == NULL)
   {
      ret_val = NVM_INVALID_BLOCK_ID;
   }
   else
   {
      ret_val = ((RAM_DB_HEADER *)rec_ptr)->nvm_block;
   }

   FUNCTION_FINISH(STB_GetRAMRecordPrevNVMBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns NVM block number for next record of given pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 *
 * @return   U16BIT - NVM block.
 *
 */
U16BIT STB_GetRAMRecordNextNVMBlock(void *rec_ptr)
{
   U16BIT ret_val;

   FUNCTION_START(STB_GetRAMRecordNextNVMBlock);

   ASSERT(rec_ptr != NULL);

   rec_ptr = (void *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)rec_ptr);
   if (rec_ptr == NULL)
   {
      ret_val = NVM_INVALID_BLOCK_ID;
   }
   else
   {
      ret_val = ((RAM_DB_HEADER *)rec_ptr)->nvm_block;
   }

   FUNCTION_FINISH(STB_GetRAMRecordNextNVMBlock);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns parent pointer for given record pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 *
 * @return   void* - pointer to the parent.
 *
 */
void* STB_GetRAMRecordParent(void *rec_ptr)
{
   void *ret_val;

   FUNCTION_START(STB_GetRAMRecordParent);

   ASSERT(rec_ptr != NULL);

   ret_val = ((RAM_DB_HEADER *)rec_ptr)->parent;

   FUNCTION_FINISH(STB_GetRAMRecordParent);

   return(ret_val);
}

/**
 *

 *
 * @brief   Sets parent pointer for given record pointer.
 *
 * @param   void* rec_ptr - pointer to the record
 * @param   void* parent  - pointer to the required parent. Could be NULL for no parent.
 *

 *
 */
void STB_SetRAMRecordParent(void *rec_ptr, void *parent)
{
   FUNCTION_START(STB_SetRAMRecordParent);

   ASSERT(rec_ptr != NULL);

   ((RAM_DB_HEADER *)rec_ptr)->parent = parent;

   FUNCTION_FINISH(STB_SetRAMRecordParent);
}

/**
 *

 *
 * @brief   Moves RAM record before the specified record in the RAM linked list
 *
 * @param   void* rec_ptr - pointer to the record
 * @param   void* dst_ptr - pointer to destination
 *

 *
 */
void STB_MoveRAMRecordBefore(void *rec_ptr, void *dst_ptr)
{
   FUNCTION_START(STB_MoveRAMRecordBefore);

   ASSERT(rec_ptr != NULL);
   ASSERT(dst_ptr != NULL);

   STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)rec_ptr);
   STB_LLAddBlockBefore((LINK_LIST_PTR_BLK *)dst_ptr, (LINK_LIST_PTR_BLK *)rec_ptr);

   FUNCTION_FINISH(STB_MoveRAMRecordBefore);
}

/**
 *

 *
 * @brief   Moves RAM record after the specified record in the RAM linked list
 *
 * @param   void* rec_ptr - pointer to the record
 * @param   void* dst_ptr - pointer to destination
 *

 *
 */
void STB_MoveRAMRecordAfter(void *rec_ptr, void *dst_ptr)
{
   FUNCTION_START(STB_MoveRAMRecordAfter);

   ASSERT(rec_ptr != NULL);
   ASSERT(dst_ptr != NULL);

   STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)rec_ptr);
   STB_LLAddBlockAfter((LINK_LIST_PTR_BLK *)dst_ptr, (LINK_LIST_PTR_BLK *)rec_ptr);

   FUNCTION_FINISH(STB_MoveRAMRecordAfter);
}

/**
 *

 *
 * @brief   Returns pointer to RAM structure for the given record type.
 *                 Finds the next record in the list after the last_rec or first if NULL
 *
 * @param   U8BIT rec_id - the ID of the record
 * @param   void* parent - the parent of the record
 * @param   void* last_rec - last record found (NULL in none)
 *
 * @return   void* - pointer to next record (NULL if no more).
 *
 */
void* STB_FindRAMRecordFromId(U8BIT rec_id, void *parent, void *last_rec)
{
   void *rec_ptr;
   void *tmp_ptr;
   void *ret_val = NULL;

   FUNCTION_START(STB_FindRAMRecordFromId);

   ASSERT(rec_id < DBA_NUM_RECORDS);

   if (last_rec == NULL)
   {
      // if first time, start with first record in list
      rec_ptr = (void *)STB_LLGetFirstBlock(&llist_hdr_array[rec_id]);
   }
   else
   {
      // else, start with next record in list
      rec_ptr = (void *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)last_rec);
   }

   // from first record in list
   while ((rec_ptr != NULL) && (ret_val == NULL))
   {
      if (((RAM_DB_HEADER *)rec_ptr)->rec_id == rec_id)
      {
         if (parent != NULL)
         {
            // search up tree for parent
            tmp_ptr = rec_ptr;
            while (tmp_ptr != NULL)
            {
               if (((RAM_DB_HEADER *)tmp_ptr)->parent == parent)
               {
                  ret_val = rec_ptr;
                  break;
               }
               tmp_ptr = ((RAM_DB_HEADER *)tmp_ptr)->parent;
            }
         }
         else
         {
            ret_val = rec_ptr;
         }
      }
      // get next record in list
      rec_ptr = (void *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)rec_ptr);
   }

   FUNCTION_FINISH(STB_FindRAMRecordFromId);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns pointer to RAM structure which relates to the given NVM record block no.
 *
 * @param   U16BIT nvm_blk - the block number of the NVM record
 *
 * @return   void* - pointer to RAM record (NULL if not found).
 *
 */
void* STB_FindRAMRecordFromNVMBlock(U16BIT nvm_block)
{
   U8BIT i;
   void *rec_ptr;
   void *ret_val = NULL;

   FUNCTION_START(STB_FindRAMRecordFromNVMBlock);

   // for each record list
   for (i = 0; i < DBA_NUM_RECORDS; i++)
   {
      // get first record in list
      rec_ptr = (void *)STB_LLGetFirstBlock(&llist_hdr_array[i]);
      while (rec_ptr != NULL)
      {
         if (((RAM_DB_HEADER *)rec_ptr)->nvm_block == nvm_block)
         {
            ret_val = rec_ptr;
            break;
         }
         // get next record in list
         rec_ptr = (void *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)rec_ptr);
      }
      if (ret_val != NULL)
         break;
   }

   FUNCTION_FINISH(STB_FindRAMRecordFromNVMBlock);

   return(ret_val);
}

/**
 *
 * @brief   Writes the specified string into a field of a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 * @param   U8BIT* string - pointer to string field
 *
 */
void STB_SetRAMRecordString(void *rec_ptr, U16BIT offset, U16BIT size, U8BIT *string)
{
   U16BIT i;
   U8BIT *data_ptr;

   FUNCTION_START(STB_SetRAMRecordString);

   ASSERT(rec_ptr != NULL);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   if (size != 0)
   {
      // for string type, copy data from pointer
      for (i = 0; i != size; i++)
      {
         *data_ptr++ = string[i];
      }
      *(data_ptr - 1) = '\0';      // ensure string is null terminated
   }
   else
   {
      *data_ptr = '\0';      // ensure string is null terminated
   }

   FUNCTION_FINISH(STB_SetRAMRecordString);
}

/**
 *
 * @brief   Writes the specified value into a field of a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 * @param   U32BIT value - the numeric value of the field
 *
 */
void STB_SetRAMRecordNumber(void *rec_ptr, U16BIT offset, U16BIT size, U32BIT value )
{
   U16BIT i;
   U8BIT *data_ptr;

   FUNCTION_START(STB_SetRAMRecordNumber);

   ASSERT(rec_ptr != NULL);
   ASSERT(size > 0);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   // for number type, copy part/all of U32BIT into field
   // forces data into big endian format
   data_ptr += (size - 1);
   for (i = 0; i != size; i++)
   {
      *data_ptr-- = (value & 0x000000ff);
      value = (value >> 8);
   }

   FUNCTION_FINISH(STB_SetRAMRecordNumber);
}

/**
 *
 * @brief   Writes the specified value into a field of a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 * @param   U32BIT value - the numeric value of the field
 *
 */
void STB_SetRAMRecordPointer(void *rec_ptr, U16BIT offset, void* ptr)
{
   U16BIT i;
   U8BIT *data_ptr;
   long value = (long)ptr;

   FUNCTION_START(STB_SetRAMRecordNumber);

   ASSERT(rec_ptr != NULL);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   data_ptr += (sizeof(void*) - 1);
   for (i = 0; i != sizeof(void*); i++)
   {
      *data_ptr-- = (value & 0x000000ff);
      value = (value >> 8);
   }

   FUNCTION_FINISH(STB_SetRAMRecordNumber);
}

/**
 *
 * @brief   Reads the specified value of a field from a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 * @param   U8BIT type - the datatype of 'value'
 *
 * @return   U32BIT - the numeric value of the field.
 *
 */
U8BIT* STB_GetRAMRecordString(void *rec_ptr, U16BIT offset, U16BIT size)
{
   U8BIT *data_ptr;

   FUNCTION_START(STB_GetRAMRecordString);
   USE_UNWANTED_PARAM(size);

   ASSERT(rec_ptr != NULL);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   FUNCTION_FINISH(STB_GetRAMRecordString);

   return(data_ptr);
}

/**
 *
 * @brief   Reads the specified value of a field from a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 *
 * @return   U32BIT - the numeric value of the field.
 *
 */
U32BIT STB_GetRAMRecordNumber(void *rec_ptr, U16BIT offset, U16BIT size)
{
   U16BIT i;
   U8BIT *data_ptr;
   U32BIT value = 0;

   FUNCTION_START(STB_GetRAMRecordNumber);

   ASSERT(rec_ptr != NULL);
   ASSERT(size > 0);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   // or for any other type, copy field into part/all of U32BIT return value
   // expects data in big endian format
   for (i = 0; i != size; i++)
   {
      value = (value << 8);
      value |= *data_ptr++;
   }

   FUNCTION_FINISH(STB_GetRAMRecordNumber);

   return(value);
}

/**
 *
 * @brief   Reads the specified value of a field from a RAM record.
 *
 * @param   void* rec_ptr - pointer to the start of the RAM record
 * @param   U16BIT offset - field byte offset into the record
 * @param   U16BIT size - field byte size in the record
 *
 * @return   U32BIT - the numeric value of the field.
 *
 */
void *STB_GetRAMRecordPointer(void *rec_ptr, U16BIT offset)
{
   U16BIT i;
   U8BIT *data_ptr;
   long value = 0;

   FUNCTION_START(STB_GetRAMRecordPointer);

   ASSERT(rec_ptr != NULL);

   // find start of field
   data_ptr = (U8BIT *)rec_ptr;
   data_ptr += sizeof(RAM_DB_HEADER);
   data_ptr += offset;

   // or for any other type, copy field into part/all of U32BIT return value
   // expects data in big endian format
   for (i = 0; i != sizeof(void*); i++)
   {
      value = (value << 8);
      value |= *data_ptr++;
   }

   FUNCTION_FINISH(STB_GetRAMRecordPointer);

   return((void*)value);
}

//*****************************************************************************
// End of file
//*****************************************************************************

