/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   NVM control functions
 *
 * @file    stbnvm.c
 * @date    26/09/2000
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbnvm.h"
#include "stbcsum.h"
#include "stbheap.h"
#include "stbhwmem.h"

//---macro definitions for this file-------------------------------------------
#ifdef STB_NV_PRINT_REQUIRED
#define STB_NV_PRINT(x) DEBUG_PRINTX_CONDITIONAL(DEBUG_STB_NV) x
#else
#define STB_NV_PRINT(x)
#endif

//---constant definitions for this file----------------------------------------

#define FORMAT_VERSION_NUM       1
#define FORMAT_ID_STRING         "DTVKit NV"
#define FORMAT_ID_STRING_SIZE    10

//---local typedef structs for this file---------------------------------------

// if this format structure is changed, the version number MUST be incremented!
typedef struct
{
   U8BIT id_str[FORMAT_ID_STRING_SIZE];
   U8BIT version;
   U8BIT num_data_blocks;
} NVM_FORMAT_REC;

typedef struct
{
   U32BIT data_block_id;
   U32BIT data_block_size;
} S_DATA_BLOCK_REC;

typedef struct s_data_block_list
{
   S_DATA_BLOCK_REC block;
   U32BIT nvm_addr;
   struct s_data_block_list *next;
   struct s_data_block_list *prev;
} S_DATA_BLOCK_LIST;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)
static NVM_FORMAT_REC nvm_format;
static S_DATA_BLOCK_LIST *data_block_list;
static U32BIT stb_area_offset;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static U32BIT CalcNvmOffset(U32BIT offset);
static void MoveBlock(U32BIT src_addr, U32BIT dst_addr, U32BIT size);
static BOOLEAN LoadFormat(void);
static void SaveFormat(void);

//---local function definitions------------------------------------------------

static U32BIT CalcNvmOffset(U32BIT offset)
{
   return(offset + (offset % STB_MEMGetNVMAlign()));
}

/**
 *

 *
 * @brief   Moves block of NVM data from/to the given addresses.
 *
 * @param   U32BIT src_addr - pointer to write data from
 * @param   U32BIT dst_addr - pointer to write data to
 * @param   U32BIT bytes - number of bytes to move
 *

 *
 */
static void MoveBlock(U32BIT src_addr, U32BIT dst_addr, U32BIT bytes)
{
   U8BIT *temp_array;

   FUNCTION_START(MoveBlock);

   if ((dst_addr + bytes) > STB_MEMGetNVMSize())
   {
      bytes = (STB_MEMGetNVMSize() - dst_addr);
   }

   temp_array = (U8BIT *)STB_GetMemory(bytes);
   if (temp_array != NULL)
   {
      if (STB_MEMReadNVM(src_addr, bytes, temp_array))
      {
         STB_MEMWriteNVM(dst_addr, bytes, temp_array);
      }
      STB_FreeMemory(temp_array);
   }

   FUNCTION_FINISH(MoveBlock);
}

/**
 *

 *
 * @brief   Loads NVM format info into RAM and validates
 *

 *
 * @return   BOOLEAN - TRUE if format OK, else FALSE.
 *
 */
static BOOLEAN LoadFormat(void)
{
   BOOLEAN ret_val = FALSE;
   U32BIT offset;
   S_DATA_BLOCK_LIST *rec_ptr;
   S_DATA_BLOCK_LIST *last_rec;
   U8BIT i;

   FUNCTION_START(LoadFormat);

   // read format info into RAM
   if (STB_MEMReadNVM(STB_MEMGetNVMOffset(), (U32BIT)sizeof(NVM_FORMAT_REC), (U8BIT *)&nvm_format))
   {
      // validate format string
      if (strcmp((char *)nvm_format.id_str, (char *)FORMAT_ID_STRING) == 0)
      {
         // check version number
         if (nvm_format.version == FORMAT_VERSION_NUM)
         {
            ret_val = TRUE;

            /* Read any data blocks */
            offset = CalcNvmOffset(STB_MEMGetNVMOffset() + sizeof(NVM_FORMAT_REC));

            for (i = 0, last_rec = NULL; (i < nvm_format.num_data_blocks) && ret_val; i++)
            {
               rec_ptr = (S_DATA_BLOCK_LIST *)STB_GetMemory(sizeof(S_DATA_BLOCK_LIST));
               if (rec_ptr != NULL)
               {
                  if (STB_MEMReadNVM(offset, sizeof(S_DATA_BLOCK_REC), (U8BIT *)&rec_ptr->block))
                  {
                     rec_ptr->nvm_addr = CalcNvmOffset(offset + sizeof(S_DATA_BLOCK_REC));
                     rec_ptr->next = NULL;

                     if (data_block_list == NULL)
                     {
                        data_block_list = rec_ptr;
                     }
                     else if (last_rec)
                     {
                        last_rec->next = rec_ptr;
                     }

                     rec_ptr->prev = last_rec;
                     last_rec = rec_ptr;

                     /* Offset to the next data block is size of header + size of this data block */
                     offset = CalcNvmOffset(offset + sizeof(S_DATA_BLOCK_REC) + rec_ptr->block.data_block_size);
                  }
                  else
                  {
                     ret_val = FALSE;
                  }
               }
            }

            /* Offset should now be pointing at the start of the service database, so save this value */
            stb_area_offset = offset;
         }
      }
   }

   FUNCTION_FINISH(LoadFormat);

   return(ret_val);
}

/**
 *

 *
 * @brief   Builds new format info and saves into NVM.
 *

 *
 */
static void SaveFormat(void)
{
   S_DATA_BLOCK_LIST *rec_ptr;
   U32BIT offset;

   FUNCTION_START(SaveFormat);

   /* Setup new format info and checksum */
   strncpy((char *)nvm_format.id_str, (char *)FORMAT_ID_STRING, FORMAT_ID_STRING_SIZE);
   nvm_format.version = FORMAT_VERSION_NUM;
#if 0
   /* Calculate the offset of each area in NVM, adjusting it according to the alignment
    * required by the physical storage media */
   nvm_format.app_area_offset = (U32BIT)sizeof(NVM_FORMAT_REC);
   nvm_format.app_area_offset += (nvm_format.app_area_offset % STB_MEMGetNVMAlign());

   nvm_format.dvb_area_offset = nvm_format.app_area_offset + nvm_format.app_area_size;
   nvm_format.dvb_area_offset += (nvm_format.dvb_area_offset % STB_MEMGetNVMAlign());

   nvm_format.stb_area_offset = (nvm_format.dvb_area_offset + nvm_format.dvb_area_size);
   nvm_format.stb_area_offset += (nvm_format.stb_area_offset % STB_MEMGetNVMAlign());

   nvm_format.checksum = STB_CalcChecksum((U8BIT *)&nvm_format, (U32BIT)sizeof(NVM_FORMAT_REC));
#endif
   /* Write header format info into NVM */
   offset = STB_MEMGetNVMOffset();

   if (STB_MEMWriteNVM(offset, (U32BIT)sizeof(NVM_FORMAT_REC), (U8BIT *)&nvm_format))
   {
      /* Write the header for each data block */
      offset = CalcNvmOffset(offset + sizeof(NVM_FORMAT_REC));

      for (rec_ptr = data_block_list; rec_ptr != NULL; rec_ptr = rec_ptr->next)
      {
         STB_MEMWriteNVM(offset, (U32BIT)sizeof(S_DATA_BLOCK_REC), (U8BIT *)&rec_ptr->block);

         /* Offset to the next data block is size of header + size of this data block */
         offset = CalcNvmOffset(offset + sizeof(S_DATA_BLOCK_REC) + rec_ptr->block.data_block_size);
      }

      stb_area_offset = offset;
   }

   FUNCTION_FINISH(SaveFormat);
}

//---global function definitions-----------------------------------------------

/**
 *

 *
 * @brief   Initialises NVM control.
 *
 * @param   U32BIT dvb_size - the number of bytes required in NVM by the application
 *

 *
 */
void STB_NVMInitialise(void)
{
   FUNCTION_START(STB_NVMInitialise);

   if (!LoadFormat())
   {
      memset(&nvm_format, 0, sizeof(nvm_format));
      SaveFormat();
   }

   FUNCTION_FINISH(STB_NVMInitialise);
}

/**
 * @brief   Returns the number of bytes of data stored for the given data block
 * @param   data_block_id data block identifier
 * @return  size in bytes
 */
U32BIT STB_NVMGetDataBlockSize(U32BIT data_block_id)
{
   U32BIT size;
   S_DATA_BLOCK_LIST *rec_ptr;

   FUNCTION_START(STB_NVMGetDataBlockSize);

   size = 0;

   for (rec_ptr = data_block_list; rec_ptr != NULL; rec_ptr = rec_ptr->next)
   {
      if (rec_ptr->block.data_block_id == data_block_id)
      {
         size = rec_ptr->block.data_block_size;
         break;
      }
   }

   FUNCTION_FINISH(STB_NVMGetDataBlockSize);

   return(size);
}

/**
 * @brief   Reads data bytes for the given data block from NVM
 * @param   data_block_id data block from which the data is to be read
 * @param   num_bytes number of bytes to be read
 * @param   dest_addr buffer to read the data into
 * @return  TRUE if the data is read, FALSE otherwise
 */
BOOLEAN STB_NVMDataBlockRead(U32BIT data_block_id, U32BIT num_bytes, U8BIT *dest_addr)
{
   BOOLEAN ret_val;
   S_DATA_BLOCK_LIST *rec_ptr;

   FUNCTION_START(STB_NVMDataBlockRead);

   ret_val = FALSE;

   /* Find the info for the data block to be read */
   for (rec_ptr = data_block_list; rec_ptr != NULL; rec_ptr = rec_ptr->next)
   {
      if (rec_ptr->block.data_block_id == data_block_id)
      {
         ret_val = STB_MEMReadNVM(rec_ptr->nvm_addr, num_bytes, dest_addr);
         break;
      }
   }

   FUNCTION_FINISH(STB_NVMDataBlockRead);

   return(ret_val);
}

/**
 * @brief   Writes data bytes for the given data block to NVM
 * @param   data_block_id data block to be written
 * @param   num_bytes number of bytes to be written
 * @param   src_addr data to be written
 * @return  TRUE if the data is written, FALSE otherwise
 */
BOOLEAN STB_NVMDataBlockWrite(U32BIT data_block_id, U32BIT num_bytes, U8BIT *src_addr)
{
   BOOLEAN ret_val;
   S_DATA_BLOCK_LIST *rec_ptr;
   S_DATA_BLOCK_LIST *last_rec;
   S_DATA_BLOCK_LIST *bptr;
   U32BIT offset;
   BOOLEAN block_changed;

   FUNCTION_START(STB_NVMDataBlockWrite);

   ret_val = FALSE;

   /* Find the info for the data block to be read */
   for (rec_ptr = data_block_list, last_rec = NULL; rec_ptr != NULL; rec_ptr = rec_ptr->next)
   {
      if (rec_ptr->block.data_block_id == data_block_id)
      {
         break;
      }

      last_rec = rec_ptr;
   }

   block_changed = FALSE;

   if (rec_ptr == NULL)
   {
      /* This is a new data block so create a record for it */
      rec_ptr = (S_DATA_BLOCK_LIST *)STB_GetMemory(sizeof(S_DATA_BLOCK_LIST));
      if (rec_ptr != NULL)
      {
         memset(rec_ptr, 0, sizeof(*rec_ptr));

         rec_ptr->block.data_block_id = data_block_id;
         rec_ptr->block.data_block_size = num_bytes;

         /* Add it to the end of the list */
         if (data_block_list == NULL)
         {
            data_block_list = rec_ptr;
         }
         else
         {
            last_rec->next = rec_ptr;
         }

         rec_ptr->prev = last_rec;

         nvm_format.num_data_blocks++;
         block_changed = TRUE;
      }
   }

   if (rec_ptr != NULL)
   {
      /* The data block is new or has increased in size so all following blocks need to be
       * moved, in reverse order */
      if (rec_ptr->block.data_block_size != num_bytes)
      {
         block_changed = TRUE;
         rec_ptr->block.data_block_size = num_bytes;
      }

      /* Calculate the new offset */
      offset = STB_MEMGetNVMOffset() + sizeof(NVM_FORMAT_REC);;
      for (bptr = data_block_list, last_rec = NULL; bptr != NULL; bptr = bptr->next)
      {
         offset = CalcNvmOffset(offset + sizeof(S_DATA_BLOCK_REC) + bptr->block.data_block_size);
         last_rec = bptr;
      }

      /* Now have the new offset for the service database, check it needs to be moved */
      if (offset != stb_area_offset)
      {
         /* Move the service database */
         MoveBlock(stb_area_offset, offset, STB_NVMGetSTBSize());
         stb_area_offset = offset;

         /* Move all data blocks in reverse order until the one that's changed or been added */
         for (bptr = last_rec; (bptr != NULL) && (bptr != rec_ptr); bptr = bptr->prev)
         {
            /* Calc the new offset for this block */
            offset = CalcNvmOffset(offset - sizeof(S_DATA_BLOCK_REC) - bptr->block.data_block_size);
            MoveBlock(bptr->nvm_addr, offset, sizeof(S_DATA_BLOCK_REC) + bptr->block.data_block_size);
            bptr->nvm_addr = CalcNvmOffset(offset + sizeof(S_DATA_BLOCK_REC));
         }
      }

      if (block_changed)
      {
         rec_ptr->nvm_addr = CalcNvmOffset(offset - rec_ptr->block.data_block_size);
         SaveFormat();
      }

      /* Save the data */
      ret_val = STB_MEMWriteNVM(rec_ptr->nvm_addr, num_bytes, src_addr);
   }

   FUNCTION_FINISH(STB_NVMDataBlockWrite);

   return(ret_val);
}

/**
 *

 *
 * @brief   Returns size of STB database storage are (in bytes).
 *

 *
 * @return   U32BIT - Number of bytes.
 *
 */
U32BIT STB_NVMGetSTBSize(void)
{
   U32BIT ret_val;

   FUNCTION_START(STB_NVMGetSTBSize);

   ASSERT(STB_MEMGetNVMSize() > (STB_MEMGetNVMOffset()+stb_area_offset));

   ret_val = (STB_MEMGetNVMSize() - STB_MEMGetNVMOffset() - stb_area_offset);

   FUNCTION_FINISH(STB_NVMGetSTBSize);

   return(ret_val);
}

/**
 *

 *
 * @brief   Reads bytes from the given position of the STB area of NVM.
 *
 * @param   U32BIT offset - offset into STB NVM area
 * @param   U32BIT bytes - number of bytes to read
 * @param   U8BIT* dest_addr - pointer to read data into
 *
 * @return   BOOLEAN - TRUE if NVM read OK, else FALSE.
 *
 */
BOOLEAN STB_NVMSTBRead(U32BIT offset, U32BIT bytes, U8BIT *dest_addr)
{
   BOOLEAN ret_val;
   U32BIT nvm_addr;

   FUNCTION_START(STB_NVMSTBRead);

   ASSERT(bytes > 0);
   ASSERT(dest_addr != NULL);

   nvm_addr = STB_MEMGetNVMOffset();
   nvm_addr += stb_area_offset;
   nvm_addr += offset;
   ret_val = STB_MEMReadNVM(nvm_addr, bytes, dest_addr);

   FUNCTION_FINISH(STB_NVMSTBRead);

   return(ret_val);
}

/**
 *

 *
 * @brief   Writes bytes into the given position of the STB area of NVM.
 *
 * @param   U32BIT offset - offset into STB NVM area
 * @param   U32BIT bytes - number of bytes to write
 * @param   U8BIT* src_addr - pointer to write data from
 *
 * @return   BOOLEAN - TRUE if NVM write OK, else FALSE.
 *
 */
BOOLEAN STB_NVMSTBWrite(U32BIT offset, U32BIT bytes, U8BIT *src_addr)
{
   BOOLEAN ret_val;
   U32BIT nvm_addr;

   FUNCTION_START(STB_NVMSTBWrite);

   ASSERT(bytes > 0);
   ASSERT(src_addr != NULL);

   nvm_addr = STB_MEMGetNVMOffset();
   nvm_addr += stb_area_offset;
   nvm_addr += offset;
   ret_val = STB_MEMWriteNVM(nvm_addr, bytes, src_addr);

   FUNCTION_FINISH(STB_NVMSTBWrite);

   return(ret_val);
}

//*****************************************************************************
// End of file
//*****************************************************************************

