/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Reference counted buffer that is automatically deleted once the count becomes zero
 * @file    buffer.h
 * @date    01/01/2017
 * @author  Ocean Blue
 */

#include <string.h>

#include "techtype.h"
#include "dbgfuncs.h"
#include "stbheap.h"

#include "buffer.h"

/**
 * @brief   Create buffer of given 'size' initialised to 'data'. The count is initialised to one
 * @param   size Size
 * @param   data Source data. Can be NULL
 * @return  Buffer data
 */
void * Buffer_New(U32BIT size, const void *data)
{
   void *result = NULL;
   S_BUFFER *buffer;

   FUNCTION_START(Buffer_New);

   if ((buffer = STB_GetMemory(sizeof(S_BUFFER) + size)))
   {
      buffer->size = size;
      buffer->refcount = 1;
      result = &buffer[1];
      if (data)
      {
         memcpy(result, data, size);
      }
   }

   FUNCTION_FINISH(Buffer_New);

   return result;
}

/**
 * @brief   Increment the count by one
 * @param   buffer_data Buffer data
 */
void Buffer_Ref(void *buffer_data)
{
   S_BUFFER *buffer;

   FUNCTION_START(Buffer_Ref);

   if (buffer_data)
   {
      buffer = &((S_BUFFER *)buffer_data)[-1];
      buffer->refcount++;
   }

   FUNCTION_FINISH(Buffer_Ref);

   return;
}

/**
 * @brief   Decrement the count by one. The buffer is automatically deleted once the count becomes zero
 * @param   buffer_data Buffer data
 */
void Buffer_Unref(void *buffer_data)
{
   S_BUFFER *buffer;

   FUNCTION_START(Buffer_Unref);

   if (buffer_data)
   {
      buffer = &((S_BUFFER *)buffer_data)[-1];
      if (buffer->refcount > 1)
      {
         buffer->refcount--;
      }
      else
      {
         STB_FreeMemory(buffer);
      }
   }

   FUNCTION_FINISH(Buffer_Unref);

   return;
}

