/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Records can be created, updated and deleted in-memory, these operations may optionally
 *          be recorded so that changes can be compared with and saved into another database
 * @file    database.c
 * @date    01/01/2017
 * @author  Ocean Blue
 */

#include <string.h>
#include <sqlite3.h>

#include "techtype.h"
#include "dbgfuncs.h"
#include "stbllist.h"
#include "stbheap.h"

#include "database.h"
#include "buffer.h"

static void AddChild(S_RECORD *parent, S_RECORD *child);
static void RemoveChild(S_RECORD *parent, S_RECORD *child);

static LINK_LIST_HEADER *database = NULL;

/**
 * @brief   Initialise runtime database
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN Database_Initialise(void)
{
   BOOLEAN result = FALSE;
   U32BIT i;

   FUNCTION_START(Database_Initialise);

   if (!database)
   {
      database = STB_GetMemory(sizeof(LINK_LIST_HEADER) * database_layout_length);
      if (database)
      {
         for (i = 0; i < database_layout_length; i++)
         {
            database[i].null_next = NULL;
            database[i].first = &database[i];
            database[i].last = &database[i];
         }

         result = TRUE;
      }
   }

   FUNCTION_FINISH(Database_Initialise);

   return result;
}

/**
 * @brief   Uninitialise runtime database
 */
void Database_Uninitialise(void)
{
   FUNCTION_START(Database_Uninitialise);

   if (database)
   {
      Database_Clear();
      STB_FreeMemory(database);
      database = NULL;
   }

   FUNCTION_FINISH(Database_Uninitialise);

   return;
}

/**
 * @brief   Clear runtime database
 */
void Database_Clear(void)
{
   U32BIT i;
   S_RECORD *next_record, *record;

   FUNCTION_START(Database_Clear);

   ASSERT(database);

   for (i = 0; i < database_layout_length; i++)
   {
      next_record = (S_RECORD *)STB_LLGetFirstBlock(&database[i]);
      while ((record = next_record))
      {
         next_record = (S_RECORD *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)record);
         if (!Database_DeleteRecord(record))
         {
            Database_DestroyRecord(record);
         }
      }
   }

   FUNCTION_FINISH(Database_Clear);

   return;
}

/**
 * @brief   Get table with given 'record_type'
 * @param   record_type Record type (see database/inc/dba.h)
 * @return  Database table, NULL if not found
 */
const S_TABLE * Database_GetTable(U32BIT record_type)
{
   const S_TABLE *result = NULL;
   U32BIT i;

   FUNCTION_START(Database_GetTable);

   for (i = 0; i < database_layout_length; i++)
   {
      if (database_layout[i].record_type == record_type)
      {
         result = &database_layout[i];
         break;
      }
   }

   FUNCTION_FINISH(Database_GetTable);

   return result;
}

/**
 * @brief   Get column in 'table' with given 'field_type'
 * @param   table Database table
 * @param   field_type Field type (see database/inc/dba.h)
 * @return  Database column, NULL if not found
 */
const S_COLUMN * Database_GetColumn(const S_TABLE *table, U32BIT field_type)
{
   const S_COLUMN *result = NULL;
   U32BIT i;

   FUNCTION_START(Database_GetColumn);

   ASSERT(table);

   for (i = 0; i < table->length; i++)
   {
      if (table->columns[i].field_type == field_type)
      {
         result = &table->columns[i];
         break;
      }
   }

   FUNCTION_FINISH(Database_GetColumn);

   return result;
}

/**
 * @brief   Get record in 'table' after 'prev' or first record if 'prev' is NULL
 * @param   table Database table
 * @param   prev Database record or NULL
 * @param   exclude_delete Exclude deleted records
 * @return  Database record, NULL if not found
 */
S_RECORD * Database_GetRecord(const S_TABLE *table, const S_RECORD *prev, BOOLEAN exclude_deleted)
{
   S_RECORD *result = NULL;

   FUNCTION_START(Database_GetRecord);

   ASSERT(database);
   ASSERT(table);

   if (prev)
   {
      result = (S_RECORD *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)prev);
   }
   else
   {
      result = (S_RECORD *)STB_LLGetFirstBlock(&database[table - database_layout]);
   }

   if (exclude_deleted)
   {
      while (result && (result->operation == OPERATION_DELETE))
      {
         result = (S_RECORD *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)result);
      }
   }

   FUNCTION_FINISH(Database_GetRecord);

   return result;
}

/**
 * @brief   Find record in 'table' with given 'id'
 * @param   table Database table
 * @param   id Record id
 * @return  Database record, NULL if not found
 */
S_RECORD * Database_FindRecord(const S_TABLE *table, sqlite3_int64 id)
{
   S_RECORD *result = NULL;

   FUNCTION_START(Database_FindRecord);

   ASSERT(table);

   while ((result = Database_GetRecord(table, result, TRUE)))
   {
      if ((result->operation != OPERATION_DELETE) && (result->id == id))
      {
         break;
      }
   }

   FUNCTION_FINISH(Database_FindRecord);

   return result;
}

/**
 * @brief   Get field from 'column' in given 'record' 
 * @param   record Database record (must not be deleted)
 * @param   column Database column (must exist in 'record')
 * @return  Database field
 */
S_FIELD * Database_GetField(S_RECORD *record, const S_COLUMN *column)
{
   S_FIELD *result;

   FUNCTION_START(Database_GetField);

   ASSERT(record);
   ASSERT(record->operation != OPERATION_DELETE);
   ASSERT(column);

   result = &(record->fields[column - record->table->columns]);

   FUNCTION_FINISH(Database_GetField);

   return result;
}

/**
 * @brief   Create database record of type 'table' with optional 'parent'
 * @param   table Database table
 * @param   parent Parent database record (suitable for type 'table') or NULL
 * @return  Database record, NULL on failure
 */
S_RECORD * Database_CreateRecord(const S_TABLE *table, S_RECORD *parent)
{
   S_RECORD *result = NULL;

   FUNCTION_START(Database_CreateRecord);

   ASSERT(database);
   ASSERT(table);

   if ((result = STB_GetMemory(sizeof(S_RECORD) + (sizeof(S_FIELD) * table->length))))
   {
      memset(result, 0, sizeof(S_RECORD) + (sizeof(S_FIELD) * table->length));
      result->table = table;
      result->parent = parent;

      if (parent)
      {
         ASSERT(parent->table < table);
         AddChild(parent, result);
      }

      STB_LLAddBlockToEnd(&database[table - database_layout], (LINK_LIST_PTR_BLK *)result);
   }

   FUNCTION_FINISH(Database_CreateRecord);

   return result;
}

/**
 * @brief   Free 'record' buffers and mark as deleted. If not created in SQLite 'record' is destroyed
 *          automatically, otherwise it must be manually destroyed once deleted from SQLite
 * @param   record Database record
 * @return  TRUE if 'record' destroyed automatically, FALSE otherwise
 */
BOOLEAN Database_DeleteRecord(S_RECORD *record)
{
   BOOLEAN result = FALSE;
   U32BIT i;

   FUNCTION_START(Database_DeleteRecord);

   ASSERT(record);

   if (record->operation == OPERATION_CREATE)
   {
      result = TRUE;
   }

   while (record->child)
   {
      Database_DeleteRecord(record->child);
   }

   if (record->parent)
   {
      RemoveChild(record->parent, record);
      record->parent = NULL;
   }

   for (i = 0; i < record->table->length; i++)
   {
      if (record->table->columns[i].data_type == &database_type_text)
      {
         Buffer_Unref(record->fields[i].value.text);
         record->fields[i].value.text = NULL;
         record->fields[i].size = 0;
      }
      else if (record->table->columns[i].data_type == &database_type_blob)
      {
         Buffer_Unref(record->fields[i].value.blob);
         record->fields[i].value.blob = NULL;
         record->fields[i].size = 0;
      }
   }

   record->operation = OPERATION_DELETE;
   if (result)
   {
      Database_DestroyRecord(record);
   }

   FUNCTION_FINISH(Database_DeleteRecord);

   return result;
}

/**
 * @brief   Destroy deleted 'record'
 * @param   record Database record (must be deleted but not already destroyed)
 */
void Database_DestroyRecord(S_RECORD *record)
{
   FUNCTION_START(Database_DestroyRecord);

   ASSERT(record);
   ASSERT(record->operation == OPERATION_DELETE);

   STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)record);
   STB_FreeMemory(record);

   FUNCTION_FINISH(Database_DestroyRecord);

   return;
}

/**
 * @brief   Add record operation (except delete). Operations are merged
 * @param   record Database record (must not be not deleted)
 * @param   operation Record operation (must not be delete)
 */
void Database_AddRecordOperation(S_RECORD *record, OPERATION_TYPE operation)
{
   FUNCTION_START(Database_AddRecordOperation);

   ASSERT(record);
   ASSERT(record->operation != OPERATION_DELETE);
   ASSERT(operation != OPERATION_DELETE);

   if (record->operation < operation)
   {
      record->operation = operation;
   }

   FUNCTION_FINISH(Database_AddRecordOperation);

   return;
}

/**
 * @brief   Clear the record operation
 * @param   record Database record (must not be not deleted)
 */
void Database_ClearRecordOperation(S_RECORD *record)
{
   FUNCTION_START(Database_ClearRecordOperation);

   ASSERT(record);
   ASSERT(record->operation != OPERATION_DELETE);

   record->operation = OPERATION_NONE;

   FUNCTION_FINISH(Database_ClearRecordOperation);

   return;
}

/**
 * @brief   Get parent of 'record'
 * @param   record Database record (must not be not deleted)
 * @return  Database record, NULL if no parent
 */
S_RECORD * Database_GetRecordParent(S_RECORD *record)
{
   S_RECORD *result;

   FUNCTION_START(Database_GetRecordParent);

   ASSERT(record);
   ASSERT(record->operation != OPERATION_DELETE);

   result = record->parent;

   FUNCTION_FINISH(Database_GetRecordParent);

   return result;
}

/**
 * @brief   Set the parent of 'record'
 * @param   record Database record (must not be not deleted)
 * @param   parent Database record (must not be not deleted) or NULL
 */
void Database_SetRecordParent(S_RECORD *record, S_RECORD *parent)
{
   FUNCTION_START(Database_SetRecordParent);

   ASSERT(record);
   ASSERT(record->operation != OPERATION_DELETE);

   if (record->parent)
   {
      RemoveChild(record->parent, record);
   }

   record->parent = parent;
   if (record->parent)
   {
      ASSERT(record->parent->table < record->table);
      ASSERT(record->parent->operation != OPERATION_DELETE);

      AddChild(record->parent, record);
   }

   FUNCTION_FINISH(Database_SetRecordParent);

   return;
}

/**
 * @brief   Get uint value of 'field'
 * @param   field Database field
 * @return  Uint Value
 */
U32BIT Database_GetRecordFieldUInt(S_FIELD *field)
{
   U32BIT result;

   FUNCTION_START(Database_GetRecordFieldUInt);

   ASSERT(field);

   result = field->value.uint;

   FUNCTION_FINISH(Database_GetRecordFieldUInt);

   return result;
}

/**
 * @brief   Set uint value of 'field'
 * @param   field Database field
 * @param   value Uint value
 */
void Database_SetRecordFieldUInt(S_FIELD *field, U32BIT value)
{
   FUNCTION_START(Database_SetRecordFieldUInt);

   ASSERT(field);

   field->value.uint = value;
   field->size = sizeof(field->value.uint);

   FUNCTION_FINISH(Database_SetRecordFieldUInt);

   return;
}

/**
 * @brief   Get text value of 'field'
 * @param   field Database field
 * @param   size Size out
 * @return  Text value
 */
const char * Database_GetRecordFieldText(S_FIELD *field, U32BIT *size)
{
   const char *result;

   FUNCTION_START(Database_GetRecordFieldText);

   ASSERT(field);

   result = field->value.text;
   if (size)
   {
      *size = field->size;
   }

   FUNCTION_FINISH(Database_GetRecordFieldText);

   return result;
}

/**
 * @brief   Set text value of 'field'
 * @param   field Database field
 * @param   value Text value
 * @param   size Text value size (including terminating null character)
 */
void Database_SetRecordFieldText(S_FIELD *field, const unsigned char *value, U32BIT size)
{
   FUNCTION_START(Database_SetRecordFieldText);

   ASSERT(field);

   if (field->value.text)
   {
      Buffer_Unref(field->value.text);
      field->value.text = NULL;
      field->size = 0;
   }

   if (size)
   {
      ASSERT(value);

      if ((field->value.text = Buffer_New(size, value)))
      {
         field->size = size;
      }
   }

   FUNCTION_FINISH(Database_SetRecordFieldText);

   return;
}

/**
 * @brief   Get blob value of 'field'
 * @param   field Database field
 * @param   size Size out
 * @return  Blob value
 */
const void * Database_GetRecordFieldBlob(S_FIELD *field, U32BIT *size)
{
   const void *result;

   FUNCTION_START(Database_GetRecordFieldBlob);

   ASSERT(field);

   result = field->value.blob;
   if (size)
   {
      *size = field->size;
   }

   FUNCTION_FINISH(Database_GetRecordFieldBlob);

   return result;
}

/**
 * @brief   Set blob value of 'field'
 * @param   field Database field
 * @param   value Blob value
 * @param   size Blob value size
 */
void Database_SetRecordFieldBlob(S_FIELD *field, const void *value, U32BIT size)
{
   FUNCTION_START(Database_SetRecordFieldBlob);

   ASSERT(field);

   if (field->value.blob)
   {
      Buffer_Unref(field->value.blob);
      field->value.blob = NULL;
      field->size = 0;
   }

   if (size)
   {
      ASSERT(value);

      if ((field->value.blob = Buffer_New(size, value)))
      {
         field->size = size;
      }
   }

   FUNCTION_FINISH(Database_SetRecordFieldBlob);

   return;
}

/**
 * @brief   Add 'child' to 'parent'
 * @param   parent Database record (must not be deleted)
 * @param   child Database record (must not be not deleted)
 */
static void AddChild(S_RECORD *parent, S_RECORD *child)
{
   S_RECORD **record;

   FUNCTION_START(AddChild);

   ASSERT(parent);

   record = &parent->child;
   while (*record)
   {
      record = &(*record)->sibling;
   }

   *record = child;

   FUNCTION_FINISH(AddChild);

   return;
}

/**
 * @brief   Remove 'child' from 'parent'
 * @param   parent Database record (must not be not deleted)
 * @param   child Database record
 */
static void RemoveChild(S_RECORD *parent, S_RECORD *child)
{
   S_RECORD **record;

   FUNCTION_START(RemoveChild);

   ASSERT(parent);

   record = &parent->child;
   while (*record)
   {
      if (*record == child)
      {
         *record = (*record)->sibling;
         break;
      }

      record = &(*record)->sibling;
   }

   FUNCTION_FINISH(RemoveChild);

   return;
}

