/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Records can be created, updated and deleted in-memory, these operations may optionally
 *          be recorded so that changes can be compared with and saved into another database
 * @file    database.h
 * @date    01/01/2017
 * @author  Ocean Blue
 */

#ifndef DBA_DATABASE_H
#define DBA_DATABASE_H

#include <sqlite3.h>

#include "stbllist.h"

#define ARRAY_LENGTH(X) (sizeof(X)/sizeof(*X))
#define INITIAL_VERSION 0

typedef enum
{
   OPERATION_NONE,
   OPERATION_UPDATE,
   OPERATION_CREATE,
   OPERATION_DELETE
} OPERATION_TYPE;

typedef struct
{
   U32BIT sqlite;
   const char *sql;
} S_DATA_TYPE;

typedef struct
{
   U32BIT field_type;
   const S_DATA_TYPE *data_type;
} S_COLUMN;

typedef struct
{
   U32BIT record_type;
   const S_COLUMN *columns;
   U32BIT length;
} S_TABLE;

typedef struct
{
   union
   {
      U32BIT uint;
      char *text;
      void *blob;
   } value;
   U32BIT size;
} S_FIELD;

typedef struct s_record
{
   LINK_LIST_PTR_BLK ptrs;
   const S_TABLE *table;
   sqlite3_int64 id;
   struct s_record *parent;
   struct s_record *sibling;
   struct s_record *child;
   OPERATION_TYPE operation;
   S_FIELD fields[];
} S_RECORD;

extern const S32BIT database_layout_version;
extern const S_DATA_TYPE database_type_uint;
extern const S_DATA_TYPE database_type_text;
extern const S_DATA_TYPE database_type_blob;
extern const S_TABLE database_layout[];
extern const U32BIT database_layout_length;

/**
 * @brief   Initialise runtime database
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN Database_Initialise(void);

/**
 * @brief   Uninitialise runtime database
 */
void Database_Uninitialise(void);

/**
 * @brief   Clear runtime database
 */
void Database_Clear(void);

/**
 * @brief   Get table with given 'record_type'
 * @param   record_type Record type (see database/inc/dba.h)
 * @return  Database table, NULL if not found
 */
const S_TABLE * Database_GetTable(U32BIT record_type);

/**
 * @brief   Get column in 'table' with given 'field_type'
 * @param   table Database table
 * @param   field_type Field type (see database/inc/dba.h)
 * @return  Database column, NULL if not found
 */
const S_COLUMN * Database_GetColumn(const S_TABLE *table, U32BIT field_type);

/**
 * @brief   Get record in 'table' after 'prev' or first record if 'prev' is NULL
 * @param   table Database table
 * @param   prev Database record or NULL
 * @param   exclude_delete Exclude deleted records
 * @return  Database record, NULL if not found
 */
S_RECORD * Database_GetRecord(const S_TABLE *table, const S_RECORD *prev, BOOLEAN exclude_deleted);

/**
 * @brief   Find record in 'table' with given 'id'
 * @param   table Database table
 * @param   id Record id
 * @return  Database record, NULL if not found
 */
S_RECORD * Database_FindRecord(const S_TABLE *table, sqlite3_int64 id);

/**
 * @brief   Get field from 'column' in given 'record' 
 * @param   record Database record (must not be deleted)
 * @param   column Database column (must exist in 'record')
 * @return  Database field
 */
S_FIELD * Database_GetField(S_RECORD *record, const S_COLUMN *column);

/**
 * @brief   Create database record of type 'table' with optional 'parent'
 * @param   table Database table
 * @param   parent Parent database record (suitable for type 'table') or NULL
 * @return  Database record, NULL on failure
 */
S_RECORD * Database_CreateRecord(const S_TABLE *table, S_RECORD *parent);

/**
 * @brief   Free 'record' buffers and mark as deleted. If not created in SQLite 'record' is destroyed
 *          automatically, otherwise it must be manually destroyed once deleted from SQLite
 * @param   record Database record
 * @return  TRUE if 'record' destroyed automatically, FALSE otherwise
 */
BOOLEAN Database_DeleteRecord(S_RECORD *record);

/**
 * @brief   Destroy deleted 'record'
 * @param   record Database record (must be deleted but not already destroyed)
 */
void Database_DestroyRecord(S_RECORD *record);

/**
 * @brief   Add record operation (except delete). Operations are merged
 * @param   record Database record (must not be not deleted)
 * @param   operation Record operation (must not be delete)
 */
void Database_AddRecordOperation(S_RECORD *record, OPERATION_TYPE operation);

/**
 * @brief   Clear the record operation
 * @param   record Database record (must not be not deleted)
 */
void Database_ClearRecordOperation(S_RECORD *record);

/**
 * @brief   Get parent of 'record'
 * @param   record Database record (must not be not deleted)
 * @return  Database record, NULL if no parent
 */
S_RECORD * Database_GetRecordParent(S_RECORD *record);

/**
 * @brief   Set the parent of 'record'
 * @param   record Database record (must not be not deleted)
 * @param   parent Database record (must not be not deleted) or NULL
 */
void Database_SetRecordParent(S_RECORD *record, S_RECORD *parent);

/**
 * @brief   Get uint value of 'field'
 * @param   field Database field
 * @return  Uint Value
 */
U32BIT Database_GetRecordFieldUInt(S_FIELD *field);

/**
 * @brief   Set uint value of 'field'
 * @param   field Database field
 * @param   value Uint value
 */
void Database_SetRecordFieldUInt(S_FIELD *field, U32BIT value);

/**
 * @brief   Get text value of 'field'
 * @param   field Database field
 * @param   size Size out
 * @return  Text value
 */
const char * Database_GetRecordFieldText(S_FIELD *field, U32BIT *size);

/**
 * @brief   Set text value of 'field'
 * @param   field Database field
 * @param   value Text value
 * @param   size Text value size (including terminating null character)
 */
void Database_SetRecordFieldText(S_FIELD *field, const unsigned char *value, U32BIT size);

/**
 * @brief   Get blob value of 'field'
 * @param   field Database field
 * @param   size Size out
 * @return  Blob value
 */
const void * Database_GetRecordFieldBlob(S_FIELD *field, U32BIT *size);

/**
 * @brief   Set blob value of 'field'
 * @param   field Database field
 * @param   value Blob value
 * @param   size Blob value size
 */
void Database_SetRecordFieldBlob(S_FIELD *field, const void *value, U32BIT size);

#endif

