/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Database access public functions
 * @file    dba_sqlite.c
 * @date    01/01/2017
 * @author  Ocean Blue
 */

#include <string.h>
#include <stdio.h>
#include <sqlite3.h>

#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"

#include "dba.h"
#include "database.h"
#include "sqlite.h"
#include "buffer.h"

static BOOLEAN initialised = FALSE;
static BOOLEAN loaded = FALSE;
static void *database_semaphore = NULL;

/**
 * @brief   Performs any initialisation required prior to the database being loaded
 * @return  TRUE if initialisation is successful, FALSE otherwise
 */
BOOLEAN DBA_Initialise(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_Initialise);

   if (initialised)
   {
      result = TRUE;
   }
   else
   {
      if (SQLite_Initialise())
      {
         if ((database_semaphore = STB_OSCreateSemaphore()))
         {
            initialised = TRUE;
            result = TRUE;
         }
         else
         {
            SQLite_Uninitialise();
         }
      }
   }

   FUNCTION_FINISH(DBA_Initialise);

   return result;
}

/**
 * @brief   Releases any resources acquired by initialisation and clears any currently loaded
 *          database, including a backup, if one has been created.
 */
void DBA_Terminate(void)
{
   FUNCTION_START(DBA_Terminate);

   initialised = FALSE;
   loaded = FALSE;

   if (database_semaphore)
   {
      STB_OSDeleteSemaphore(database_semaphore);
      database_semaphore = NULL;
   }

   SQLite_Uninitialise();

   FUNCTION_FINISH(DBA_Terminate);

   return;
}

/**
 * @brief   Reads a database from non-volatile storage, creating any structures in memory that
 *          will be required to access it and the records it contains and makes this the working
 *          database. If the database is found to be invalid, or doesn't exist at all, then an
 *          empty database should be created.
 * @param   pathname full pathname of the file to be loaded containing the database.
 *                     This argument is only relevant where the system being used is file based.
 * @return  TRUE if a database is loaded successfully, FALSE otherwise
 */
BOOLEAN DBA_LoadDatabase(U8BIT *pathname)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_LoadDatabase);

   if (initialised)
   {
      if (pathname)
      {
         result = SQLite_Open((char *)pathname);
      }
      else
      {
         result = SQLite_Open(DEFAULT_SQLITE_FILE);
      }
   }

   loaded = result;

   FUNCTION_FINISH(DBA_LoadDatabase);

   return result;
}

/**
 * @brief   Saves any changes made to the working database to non-volatile storage.
 *          If saving to a file, the pathname used to open it will be used.
 * @return  TRUE if the database is saved successfully, FALSE otherwise
 */
BOOLEAN DBA_SaveDatabase(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_SaveDatabase);

   if (loaded)
   {
      result = SQLite_SaveDatabaseRecords();
   }

   FUNCTION_FINISH(DBA_SaveDatabase);

   return result;
}

/**
 * @brief   Clears the working database of all records. Following this call, it should still
 *          be possible to access the database, but it will be empty, and new records can be
 *          added to it.
 * @return  TRUE if the database is cleared, FALSE otherwise
 */
BOOLEAN DBA_ClearDatabase(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_ClearDatabase);

   if (loaded)
   {
      if (!(result = SQLite_Reset()))
      {
         loaded = FALSE;
      }
   }

   FUNCTION_FINISH(DBA_ClearDatabase);

   return result;
}

/**
 * @brief   Creates a backup copy of the working database. Whether the backup database is saved
 *          to non-volatile storage is implementation dependant and so the backup database may
 *          not survive a system reboot.
 * @param   pathname full pathname of the file to save the backup to if this is supported
 *                     by the database implementation.
 * @return  TRUE if a backup is created, FALSE otherwise
 */
BOOLEAN DBA_BackupDatabase(U8BIT *pathname)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_BackupDatabase);
   USE_UNWANTED_PARAM(pathname);

   if (loaded)
   {
      result = SQLite_CreateBackup(NULL);
   }

   FUNCTION_FINISH(DBA_BackupDatabase);

   return result;
}

/**
 * @brief   Checks whether the working database can be restored from a backup copy
 * @return  TRUE if there is a backup copy and it can be restored, FALSE otherwise
 */
BOOLEAN DBA_CanRestoreDatabase(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_CanRestoreDatabase);

   if (loaded)
   {
      result = SQLite_RestoreBackup(NULL, TRUE);
   }

   FUNCTION_FINISH(DBA_CanRestoreDatabase);

   return result;
}

/**
 * @brief   Restores the working database from a previously made backup copy.
 * @return  TRUE if the working database is restored from the backup, FALSE otherwise
 */
BOOLEAN DBA_RestoreDatabase(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_RestoreDatabase);

   if (loaded)
   {
      result = SQLite_RestoreBackup(NULL, FALSE);
   }

   FUNCTION_FINISH(DBA_RestoreDatabase);

   return result;
}

/**
 * @brief   Erases the backup copy of the database. If data was stored in non-volatile storage
 *          then this should be deleted too.
 */
void DBA_EraseBackupDatabase(void)
{
   FUNCTION_START(DBA_EraseBackupDatabase);

   if (loaded)
   {
      SQLite_RemoveBackup(NULL);
   }

   FUNCTION_FINISH(DBA_EraseBackupDatabase);

   return;
}

/**
 * @brief   Export the working database to an XML file.
 * @param   xml_pathname full pathname of the file to export to
 * @return  TRUE if the database is exported successfully, FALSE otherwise
 */
BOOLEAN DBA_ExportToXML(U8BIT *xml_pathname)
{
   BOOLEAN result;

   FUNCTION_START(DBA_ExportToXML);

   /* Not implemented */
   USE_UNWANTED_PARAM(xml_pathname);

   result = FALSE;

   FUNCTION_FINISH(DBA_ExportToXML);

   return result;
}

/**
 * @brief   Imports records from the given XML file into the working database. If a record already
 *          exists in the database then it will be updated, and if it doesn't then a new record
 *          will be created.
 * @param   xml_pathname full pathname of the file to be imported
 * @return  TRUE if the file is imported successfully, FALSE otherwise
 */
BOOLEAN DBA_ImportFromXML(U8BIT *xml_pathname)
{
   BOOLEAN result;

   FUNCTION_START(DBA_ImportFromXML);

   /* Not implemented */
   USE_UNWANTED_PARAM(xml_pathname);

   result = FALSE;

   FUNCTION_FINISH(DBA_ImportFromXML);

   return result;
}

/**
 * @brief   Locks the database to prevent access from other threads or processes.
 */
void DBA_LockDatabase(void)
{
   FUNCTION_START(DBA_LockDatabase);

   if (initialised)
   {
      STB_OSSemaphoreWait(database_semaphore);
   }

   FUNCTION_FINISH(DBA_LockDatabase);

   return;
}

/**
 * @brief   Unlocks the database to allow other threads or processes to access it.
 */
void DBA_UnlockDatabase(void)
{
   FUNCTION_START(DBA_UnlockDatabase);

   if (initialised)
   {
      STB_OSSemaphoreSignal(database_semaphore);
   }

   FUNCTION_FINISH(DBA_UnlockDatabase);

   return;
}

/**
 * @brief   Returns a version string representing the working database.
 * @return  '\0' terminated string in UTF-8 format
 */
U8BIT* DBA_DatabaseVersion(void)
{
   static char result[20] = {0};

   FUNCTION_START(DBA_DatabaseVersion);

   if (!result[0])
   {
      snprintf(result, sizeof(result), "%ld", (long)database_layout_version);
   }

   FUNCTION_FINISH(DBA_DatabaseVersion);

   return (U8BIT *)result;
}

/**
 * @brief   Returns the size in bytes of the database as stored in non-volatile storage
 * @param   max_size returns the maximum size in bytes that the database can grow to.
 *                     This value may be returned as 0 if this can't be determined.
 * @return  current size of the working database in bytes
 */
U32BIT DBA_DatabaseFileSize(U32BIT *max_size)
{
   U32BIT result = 0;

   FUNCTION_START(DBA_DatabaseFileSize);

   /* Not implemented */

   *max_size = 0;

   FUNCTION_FINISH(DBA_DatabaseFileSize);

   return result;
}

/**
 * @brief   Creates a new record of the given type, adding it to the database as a child of the
 *          given parent record, if provided.
 * @param   record_id type of record to be created
 * @param   parent create a record that is a child of this parent, can be passed as NULL
 * @return  handle of new record, or NULL if creation fails
 */
void* DBA_CreateRecord(U32BIT record_id, void *parent)
{
   S_RECORD *result = NULL;
   const S_TABLE *table;

   FUNCTION_START(DBA_CreateRecord);

   ASSERT(loaded);

   if (initialised)
   {
      table = Database_GetTable(record_id);
      if (table)
      {
         result = Database_CreateRecord(table, parent);
         if (result)
         {
            Database_AddRecordOperation(result, OPERATION_CREATE);
         }
      }
   }

   FUNCTION_FINISH(DBA_CreateRecord);

   return result;
}

/**
 * @brief   Destroys the given record, removing it from the database and freeing any memory
 *          associated with it.
 * @param   record database record to be deleted
 */
void DBA_DestroyRecord(void *record)
{
   FUNCTION_START(DBA_DestroyRecord);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         Database_DeleteRecord(record);
      }
   }

   FUNCTION_FINISH(DBA_DestroyRecord);

   return;
}

/**
 * @brief   Finds the next record, of the given type, that comes after last_rec. last_rec
 *          must be the same type of record as the one being found. Parent is optional, but
 *          if provided, a record will only be returned if the parent of the one found is the same.
 * @param   record_id type of record to look for
 * @param   parent if not NULL, this must be the parent of the record found for it to be returned
 * @param   last_rec previous record of the type being looked for, if NULL then the first record
 *                     will be returned.
 * @return  handle of the record, or NULL if none found
 */
void* DBA_FindRecord(U32BIT record_id, void *parent, void *last_rec)
{
   S_RECORD *result = NULL;
   const S_TABLE *table;

   FUNCTION_START(DBA_FindRecord);

   ASSERT(loaded);

   if (initialised)
   {
      if (last_rec)
      {
         table = ((S_RECORD *)last_rec)->table;
      }
      else
      {
         table = Database_GetTable(record_id);
      }

      if (table)
      {
         result = Database_GetRecord(table, last_rec, TRUE);

         if (parent)
         {
            while (result && (Database_GetRecordParent(result) != parent))
            {
               result = Database_GetRecord(table, result, TRUE);
            }
         }
      }
   }

   FUNCTION_FINISH(DBA_FindRecord);

   return result;
}

/**
 * @brief   Set of change the parent of the given record
 * @param   record handle of the record whose parent is to be changed
 * @param   parent handle of the new parent record, can be NULL
 */
void DBA_SetRecordParent(void *record, void *parent)
{
   FUNCTION_START(DBA_SetRecordParent);

   ASSERT(loaded);

   if (initialised)
   {
      Database_SetRecordParent(record, parent);
      Database_AddRecordOperation(record, OPERATION_UPDATE);
   }

   FUNCTION_FINISH(DBA_SetRecordParent);

   return;
}

/**
 * @brief   Returns the handle to the parent of the given record
 * @param   record handle of record whose parent is to be returned
 * @return  handle of the record's parent
 */
void* DBA_GetRecordParent(void *record)
{
   S_RECORD *result = NULL;

   FUNCTION_START(DBA_GetRecordParent);

   ASSERT(loaded);

   if (initialised)
   {
      result = Database_GetRecordParent(record);
   }

   FUNCTION_FINISH(DBA_GetRecordParent);

   return result;
}

/**
 * @brief   Forces a record to be saved to non-volatile storage. Depending on the implementation,
 *          this function may not do anything if the data is updated to non-volatile storage
 *          as any records and/or fields are created or updated.
 * @param   record handle of record to be saved
 */
void DBA_SaveRecord(void *record)
{
   FUNCTION_START(DBA_SaveRecord);

   /* Leave not implemented. Breaks atomicity. Not implemented in other database */
   USE_UNWANTED_PARAM(record);

   FUNCTION_FINISH(DBA_SaveRecord);

   return;
}

/**
 * @brief   Set the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a value field.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   value new value of the field
 * @return  TRUE if the value is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldValue(void *record, U32BIT field_id, U32BIT value)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;

   FUNCTION_START(DBA_SetFieldValue);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_uint)
         {
            Database_SetRecordFieldUInt(Database_GetField(record, column), value);
            Database_AddRecordOperation(record, OPERATION_UPDATE);
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_SetFieldValue);

   return result;
}

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldString(void *record, U32BIT field_id, U8BIT *string, U16BIT num_bytes)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;

   FUNCTION_START(DBA_SetFieldString);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_text)
         {
            Database_SetRecordFieldText(Database_GetField(record, column), string, num_bytes);
            Database_AddRecordOperation(record, OPERATION_UPDATE);
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_SetFieldString);

   return result;
}

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   lang_code language code of the string
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT *string,
   U16BIT num_bytes)
{
   FUNCTION_START(DBA_SetFieldLangString);

   /* Not implemented */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(lang_code);
   USE_UNWANTED_PARAM(string);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_SetFieldLangString);

   return FALSE;
}

/**
 * @brief   Set a variable amount of data of a record's field. The function will fail if the
 *          record doesn't exist, the record doesn't include the given field, or the field
 *          doesn't hold data. The data provided will be copied.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   data data to be stored in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if the data is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldData(void *record, U32BIT field_id, U8BIT *data, U16BIT num_bytes)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;

   FUNCTION_START(DBA_SetFieldData);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_blob)
         {
            Database_SetRecordFieldBlob(Database_GetField(record, column), data, num_bytes);
            Database_AddRecordOperation(record, OPERATION_UPDATE);
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_SetFieldData);

   return result;
}

/**
 * @brief   Gets the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a value field.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   value pointer to the returned value if function returns TRUE
 * @return  TRUE if a value is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldValue(void *record, U32BIT field_id, U32BIT *value)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;

   FUNCTION_START(DBA_GetFieldValue);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_uint)
         {
            *value = Database_GetRecordFieldUInt(Database_GetField(record, column));
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_GetFieldValue);

   return result;
}

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The pointer to the string returned will be to the string data held by the database,
 *          so the data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldString(void *record, U32BIT field_id, U8BIT **string, U16BIT *num_bytes)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;
   U32BIT bytes;
   static void *blob_ref = NULL;

   FUNCTION_START(DBA_GetFieldString);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_text)
         {
            Buffer_Unref(blob_ref);
            blob_ref = (void *)Database_GetRecordFieldText(Database_GetField(record, column), &bytes);
            Buffer_Ref(blob_ref);
            *string = blob_ref;
            *num_bytes = bytes;
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_GetFieldString);

   return result;
}

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The pointer to the string returned will be to the string data held by the database,
 *          so the data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   lang_code language code of the string to be returned
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT **string,
   U16BIT *num_bytes)
{
   FUNCTION_START(DBA_GetFieldLangString);

   /* Not implemented */
   USE_UNWANTED_PARAM(record);
   USE_UNWANTED_PARAM(field_id);
   USE_UNWANTED_PARAM(lang_code);
   USE_UNWANTED_PARAM(string);
   USE_UNWANTED_PARAM(num_bytes);

   FUNCTION_FINISH(DBA_GetFieldLangString);

   return FALSE;
}

/**
 * @brief   Gets the data of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a data field.
 *          The pointer to the data returned will be to the data held by the database, so the
 *          data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   data pointer to the returned data in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if data is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldData(void *record, U32BIT field_id, U8BIT **data, U16BIT *num_bytes)
{
   BOOLEAN result = FALSE;
   const S_COLUMN *column;
   U32BIT bytes;
   static void *blob_ref = NULL;

   FUNCTION_START(DBA_GetFieldData);

   ASSERT(loaded);

   if (initialised)
   {
      if (record != NULL)
      {
         column = Database_GetColumn(((S_RECORD *)record)->table, field_id);
         if (column != NULL && column->data_type == &database_type_blob)
         {
            Buffer_Unref(blob_ref);
            blob_ref = (void *)Database_GetRecordFieldBlob(Database_GetField(record, column), &bytes);
            Buffer_Ref(blob_ref);
            *data = blob_ref;
            *num_bytes = bytes;
            result = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBA_GetFieldData);

   return result;
}

/**
 * @brief   Returns the number of bytes available for the given data block
 * @param   data_block_id the data block whose size is to be returned
 * @return  size of bytes of the data block, or 0 if block not found
 */
U32BIT DBA_DataBlockSize(U32BIT data_block_id)
{
   U32BIT result = 0;

   FUNCTION_START(DBA_DataBlockSize);

   if (loaded)
   {
      result = SQLite_GetBlobSize(data_block_id);
   }

   FUNCTION_FINISH(DBA_DataBlockSize);

   return result;
}

/**
 * @brief   Read a block of data from the database into the given buffer
 * @param   data_block_id id of the data block to be read
 * @param   data pointer to the buffer into which the data will be read
 * @param   max_num_bytes the maximum number of bytes to be read
 * @return  number of bytes read into the buffer
 */
U32BIT DBA_DataBlockRead(U32BIT data_block_id, U8BIT *data, U32BIT max_num_bytes)
{
   U32BIT result = 0;

   FUNCTION_START(DBA_DataBlockRead);

   if (loaded)
   {
      if (max_num_bytes)
      {
         result = SQLite_ReadBlob(data_block_id, data, max_num_bytes);
      }
   }

   FUNCTION_FINISH(DBA_DataBlockRead);

   return result;
}

/**
 * @brief   Writes a block of data into the database from the given buffer
 * @param   data_block_id id of the data block being written
 * @param   data pointer to the data to be written
 * @param   num_bytes number of bytes of data to be written
 * @return  TRUE if the data is written, FALSE otherwise
 */
BOOLEAN DBA_DataBlockWrite(U32BIT data_block_id, U8BIT *data, U32BIT num_bytes)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(DBA_DataBlockWrite);

   if (loaded)
   {
      result = SQLite_WriteBlob(data_block_id, data, num_bytes);
   }

   FUNCTION_FINISH(DBA_DataBlockWrite);

   return result;
}

