/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Interface to SQLite. Loads database records into the in-memory database and saves
 *          in-memory database changes into the database. Reads and writes database blobs
 * @file    sqlite.h
 * @date    01/01/2017
 * @author  Ocean Blue
 */

/*#define DEBUG_SQLITE*/
/*#define DEBUG_SQL*/

#include <string.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <sqlite3.h>

#include "techtype.h"
#include "dbgfuncs.h"
#include "stbllist.h"
#include "stbheap.h"
#include "stbhwc.h"

#include "database.h"
#include "sqlite.h"
#include "buffer.h"

#ifdef DEBUG_SQLITE
#define DBG_SQLITE(X) STB_SPDebugWrite X
#else
#define DBG_SQLITE(X)
#endif

#ifdef DEBUG_SQL
#define DBG_SQL(X) STB_SPDebugWrite X
#else
#define DBG_SQL(X)
#endif

#define STRING_INITIAL_SIZE 512

typedef struct
{
   char *string;
   U32BIT size;
   U32BIT length;
} S_STRING;

static BOOLEAN InitialiseDatabase(void *db);
static BOOLEAN CreateDatabaseTable(void *db, const S_TABLE *table, const char *prefix);
static BOOLEAN CreateDatabaseBlobsTable(void *db);
static BOOLEAN CreateDatabaseRecord(void *db, S_RECORD *record);
static BOOLEAN DeleteDatabaseRecord(void *db, S_RECORD *record);
static BOOLEAN UpdateDatabaseRecord(void *db, S_RECORD *record);
static BOOLEAN LoadDatabaseRecords(void);
static BOOLEAN UpgradeDatabase(void *db);
static S_STRING GetUpgradeableColumnNames(void *db, const S_TABLE *table);
static BOOLEAN GetUserVersion(void *db, S32BIT *user_version);
static BOOLEAN SetUserVersion(void *db, S32BIT user_version);
static BOOLEAN IntegrityCheck(void *db);
static void StartTransaction(void *db);
static void EndTransaction(void *db, BOOLEAN commit);
static void CreateString(S_STRING *string, const char *format, ...);
static void SetString(S_STRING *string, const char *format, ...);
static void AppendString(S_STRING *string, const char *format, ...);
static void AppendStringArgs(S_STRING *string, const char *format, va_list args);
static void DeleteString(S_STRING *string);

static BOOLEAN sqlite_initialised = FALSE;
static sqlite3_mutex *transaction = NULL;
static sqlite3 *db_memory_backup = NULL;
static BOOLEAN sqlite_db_open = FALSE;
static sqlite3 *db = NULL;
static char *db_filename = NULL;

/**
 * @brief   Initialise SQLite interface
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Initialise(void)
{
   BOOLEAN result = FALSE;

   FUNCTION_START(SQLite_Initialise);

   if (!sqlite_initialised)
   {
      if (Database_Initialise())
      {
         if ((transaction = sqlite3_mutex_alloc(SQLITE_MUTEX_RECURSIVE)))
         {
            sqlite_initialised = TRUE;
         }
         else
         {
            Database_Uninitialise();
         }
      }
   }

   if (sqlite_initialised)
   {
      result = TRUE;
   }

   FUNCTION_FINISH(SQLite_Initialise);

   return result;
}

/**
 * @brief   Uninitialise SQLite interface
 */
void SQLite_Uninitialise(void)
{
   FUNCTION_START(SQLite_Uninitialise);

   if (sqlite_initialised)
   {
      SQLite_Close();

      if (db_memory_backup)
      {
         sqlite3_close(db_memory_backup);
         db_memory_backup = NULL;
      }

      sqlite3_mutex_free(transaction);
      transaction = NULL;
      Database_Uninitialise();

      sqlite_initialised = FALSE;
   }

   FUNCTION_FINISH(SQLite_Uninitialise);

   return;
}

/**
 * @brief   Open database 'filename'. The database is created and initialised if it does not
 *          exist. The database is upgraded if it is older than the current version. The in-memory
 *          database is cleared and database records are loaded into it
 * @param   filename Database file
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Open(const char *filename)
{
   BOOLEAN result = FALSE;
   S32BIT user_version;

   FUNCTION_START(SQLite_Open);

   ASSERT(sqlite_initialised);
   ASSERT(filename);

   sqlite3_mutex_enter(transaction);

   /* check not already open */
   if (sqlite_db_open)
   {
      if (strcmp(filename, db_filename) == 0)
      {
         result = TRUE;
      }
   }

   if (!result)
   {
      SQLite_Close();
      Database_Clear();

      /* open */
      if ((db_filename = STB_GetMemory(strlen(filename) + 1)))
      {
         strcpy(db_filename, filename);
         if (sqlite3_open(db_filename, &db) == SQLITE_OK)
         {
            sqlite_db_open = TRUE;
         }
         else
         {
            DBG_SQLITE(("%s", sqlite3_errmsg(db)));

            /* could not open. tidy up as closed */
            db = NULL;
            STB_FreeMemory(db_filename);
            db_filename = NULL;
         }
      }

      if (sqlite_db_open)
      {
         /* prepare and load, reset or close */
         if (IntegrityCheck(db) && GetUserVersion(db, &user_version))
         {
            StartTransaction(db);

            if (user_version == INITIAL_VERSION)
            {
               result = InitialiseDatabase(db);
            }
            else if (user_version < database_layout_version)
            {
               result = UpgradeDatabase(db);
            }
            else if (user_version == database_layout_version)
            {
               result = TRUE;
            }

            EndTransaction(db, result);

            if (result)
            {
               result = LoadDatabaseRecords();
            }
         }

         if (!result)
         {
            if (!(result = SQLite_Reset()))
            {
               SQLite_Close();
            }
         }
      }
   }

   sqlite3_mutex_leave(transaction);

   FUNCTION_FINISH(SQLite_Open);

   return result;
}

/**
 * @brief   Close database
 */
void SQLite_Close(void)
{
   S32BIT rc;

   FUNCTION_START(SQLite_Close);

   ASSERT(sqlite_initialised);

   sqlite3_mutex_enter(transaction);

   if (sqlite_db_open)
   {
      rc = sqlite3_close(db);
      ASSERT(rc == SQLITE_OK);
      USE_UNWANTED_PARAM(rc);

      db = NULL;
      STB_FreeMemory(db_filename);
      db_filename = NULL;

      sqlite_db_open = FALSE;
   }

   sqlite3_mutex_leave(transaction);

   FUNCTION_FINISH(SQLite_Close);

   return;
}

/**
 * @brief   Reset database and clear in-memory database. Database closed on failure
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Reset(void)
{
   BOOLEAN result = FALSE;
   S32BIT rc;
   S32BIT user_version;

   FUNCTION_START(SQLite_Reset);

   ASSERT(sqlite_initialised);

   sqlite3_mutex_enter(transaction);

   Database_Clear();

   if (sqlite_db_open)
   {
      /* close and remove */
      rc = sqlite3_close(db);
      ASSERT(rc == SQLITE_OK);
      USE_UNWANTED_PARAM(rc);
      remove(db_filename);

      /* reopen */
      if (sqlite3_open(db_filename, &db) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));

         /* could not reopen. tidy up as closed */
         db = NULL;
         STB_FreeMemory(db_filename);
         db_filename = NULL;

         sqlite_db_open = FALSE;
      }

      if (sqlite_db_open)
      {
         /* prepare or close */
         if (IntegrityCheck(db) && GetUserVersion(db, &user_version))
         {
            StartTransaction(db);

            if (user_version == INITIAL_VERSION)
            {
               result = InitialiseDatabase(db);
            }

            EndTransaction(db, result);
         }

         if (!result)
         {
            SQLite_Close();
         }
      }
   }

   sqlite3_mutex_leave(transaction);

   FUNCTION_FINISH(SQLite_Reset);

   return result;
}

/**
 * @brief   Save in-memory database changes into the database
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_SaveDatabaseRecords(void)
{
   BOOLEAN result = TRUE;
   U32BIT i;
   S_RECORD *record;
   S_RECORD *next_record;

   FUNCTION_START(SQLite_SaveDatabaseRecords);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   for (i = 0; (i < database_layout_length) && result; i++)
   {
      next_record = Database_GetRecord(&database_layout[i], NULL, FALSE);

      while ((record = next_record) && result)
      {
         next_record = Database_GetRecord(&database_layout[i], record, FALSE);

         if (record->operation == OPERATION_NONE)
         {
            continue;
         }
         else if (record->operation == OPERATION_UPDATE)
         {
            result = UpdateDatabaseRecord(db, record);
         }
         else if (record->operation == OPERATION_CREATE)
         {
            result = CreateDatabaseRecord(db, record);
         }
         else if (record->operation == OPERATION_DELETE)
         {
            result = DeleteDatabaseRecord(db, record);
         }
      }
   }

   if (result)
   {
      for (i = 0; i < database_layout_length; i++)
      {
         next_record = Database_GetRecord(&database_layout[i], NULL, FALSE);

         while ((record = next_record))
         {
            next_record = Database_GetRecord(&database_layout[i], record, FALSE);

            if (record->operation == OPERATION_NONE)
            {
               continue;
            }
            else if (record->operation == OPERATION_DELETE)
            {
               Database_DestroyRecord(record);
            }
            else
            {
               Database_ClearRecordOperation(record);
            }
         }
      }
   }

   EndTransaction(db, result);

   FUNCTION_FINISH(SQLite_SaveDatabaseRecords);

   return result;
}

/**
 * @brief   Get size of blob 'id'
 * @param   id Blob id
 * @return  Size on success, 0 on failure
 */
U32BIT SQLite_GetBlobSize(U32BIT id)
{
   U32BIT result = 0;
   S_STRING sql;
   sqlite3_stmt *stmt;

   static const char sql_select_blob[] = "SELECT blob FROM blobs WHERE id = %lu";

   FUNCTION_START(SQLite_GetBlobSize);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   CreateString(&sql, sql_select_blob, id);
   DBG_SQL((sql.string));

   if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      if (sqlite3_step(stmt) == SQLITE_ROW)
      {
         result = sqlite3_column_bytes(stmt, 0);
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      }
   }
   else
   {
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
   }

   DeleteString(&sql);

   EndTransaction(db, TRUE);

   FUNCTION_FINISH(SQLite_GetBlobSize);

   return result;
}

/**
 * @brief   Read blob 'id' into 'data'
 * @param   id Blob id
 * @param   data Data buffer
 * @param   data_size Size of 'data'
 * @return  Size read on success, 0 on failure
 */
U32BIT SQLite_ReadBlob(U32BIT id, void *data, U32BIT data_size)
{
   U32BIT result = 0;
   S_STRING sql;
   sqlite3_stmt *stmt;

   static const char sql_select_blob[] = "SELECT blob FROM blobs WHERE id = %lu";

   FUNCTION_START(SQLite_ReadBlob);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   CreateString(&sql, sql_select_blob, id);
   DBG_SQL((sql.string));

   if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      if (sqlite3_step(stmt) == SQLITE_ROW)
      {
         result = sqlite3_column_bytes(stmt, 0);
         if (data_size < result)
         {
            result = data_size;
         }

         if (data)
         {
            memcpy(data, sqlite3_column_blob(stmt, 0), result);
         }
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      }
   }
   else
   {
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
   }

   DeleteString(&sql);

   EndTransaction(db, TRUE);

   FUNCTION_FINISH(SQLite_ReadBlob);

   return result;
}

/**
 * @brief   Write 'data' into blob 'id'
 * @param   id Blob id
 * @param   data Data buffer
 * @param   data_size Size of 'data'
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_WriteBlob(U32BIT id, const void *data, U32BIT data_size)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   char *errmsg;
   sqlite3_stmt *stmt;
   void *buffer;

   static const char sql_delete_blob[] = "DELETE FROM blobs WHERE id = %lu";
   static const char sql_insert_blob[] = "INSERT INTO blobs VALUES (%lu, ?)";

   FUNCTION_START(SQLite_WriteBlob);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   CreateString(&sql, sql_delete_blob, id);
   DBG_SQL((sql.string));

   if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
      result = FALSE;
   }

   if (result && (data_size > 0))
   {
      SetString(&sql, sql_insert_blob, id);
      DBG_SQL((sql.string));

      if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
      {
         ASSERT(stmt);

         if ((buffer = Buffer_New(data_size, data)))
         {
            if (sqlite3_bind_blob(stmt, 1, buffer, data_size, Buffer_Unref) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", sqlite3_errmsg(db)));
               result = FALSE;
            }

            if (result && (sqlite3_step(stmt) != SQLITE_DONE))
            {
               result = FALSE;
            }

            if (sqlite3_finalize(stmt) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", sqlite3_errmsg(db)));
               result = FALSE;
            }
         }
         else
         {
            result = FALSE;
         }
      }
      else
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
         result = FALSE;
      }
   }

   DeleteString(&sql);

   EndTransaction(db, result);

   FUNCTION_FINISH(SQLite_WriteBlob);

   return result;
}

/**
 * @brief   Create backup 'filename'
 * @param   filename Database file or NULL for memory backup
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_CreateBackup(const char *filename)
{
   BOOLEAN result = FALSE;
   sqlite3 *db_backup;
   sqlite3_backup *backup;

   FUNCTION_START(SQLite_CreateBackup);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   /* get connection for file or in-memory backup */
   if (filename)
   {
      if (sqlite3_open(filename, &db_backup) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db_backup)));
         db_backup = NULL;
      }
   }
   else
   {
      if (!db_memory_backup)
      {
         if (sqlite3_open(":memory:", &db_memory_backup) != SQLITE_OK)
         {
            DBG_SQLITE(("%s", sqlite3_errmsg(db_memory_backup)));
            db_memory_backup = NULL;
         }
      }

      db_backup = db_memory_backup;
   }

   if (db_backup)
   {
      /* create backup */
      if ((backup = sqlite3_backup_init(db_backup, "main", db, "main")))
      {
         sqlite3_backup_step(backup, -1);
         sqlite3_backup_finish(backup);
      }

      if (sqlite3_errcode(db_backup) == SQLITE_OK)
      {
         result = TRUE;
         if (filename)
         {
            sqlite3_close(db_backup);
         }
      }
      else
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db_backup)));
         sqlite3_close(db_backup);
         if (!filename)
         {
            db_memory_backup = NULL;
         }
      }
   }

   EndTransaction(db, result);

   FUNCTION_FINISH(SQLite_CreateBackup);

   return result;
}

/**
 * @brief   Restore backup 'filename'
 * @param   filename Database file or NULL for memory backup
 * @param   check_exists_only Only check whether the backup exists
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_RestoreBackup(const char *filename, BOOLEAN check_exists_only)
{
   BOOLEAN result = FALSE;
   S32BIT user_version;
   sqlite3 *db_backup;
   sqlite3_backup *backup;

   FUNCTION_START(SQLite_RestoreBackup);

   ASSERT(sqlite_db_open);

   sqlite3_mutex_enter(transaction);

   /* get handle for file or in-memory backup */
   if (filename)
   {
      if (sqlite3_open_v2(filename, &db_backup, SQLITE_OPEN_READONLY, NULL) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db_backup)));
         db_backup = NULL;
      }
   }
   else
   {
      db_backup = db_memory_backup;
   }

   if (db_backup)
   {
      StartTransaction(db);

      /* restore backup */
      if ((backup = sqlite3_backup_init(db, "main", db_backup, "main")))
      {
         sqlite3_backup_step(backup, -1);
         sqlite3_backup_finish(backup);
      }

      if (sqlite3_errcode(db_backup) == SQLITE_OK)
      {
         /* prepare */
         if (IntegrityCheck(db) && GetUserVersion(db, &user_version))
         {
            if (user_version == INITIAL_VERSION)
            {
               result = InitialiseDatabase(db);
            }
            else if (user_version < database_layout_version)
            {
               result = UpgradeDatabase(db);
            }
            else if (user_version == database_layout_version)
            {
               result = TRUE;
            }
         }
      }
      else
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db_backup)));
      }

      if (check_exists_only)
      {
         EndTransaction(db, FALSE);
      }
      else
      {
         EndTransaction(db, result);

         /* load */
         if (result)
         {
            Database_Clear();
            if (!(result = LoadDatabaseRecords()))
            {
               Database_Clear();
            }
         }
      }

      if (filename)
      {
         sqlite3_close(db_backup);
      }
   }

   sqlite3_mutex_leave(transaction);

   FUNCTION_FINISH(SQLite_RestoreBackup);

   return result;
}

/**
 * @brief   Remove backup 'filename'
 * @param   filename Database file or NULL for memory backup
 * @return  TRUE on success, FALSE on failure
 */
void SQLite_RemoveBackup(const char *filename)
{
   FUNCTION_START(SQLite_RemoveBackup);

   ASSERT(sqlite_db_open);

   StartTransaction(db);

   if (filename)
   {
      remove(filename);
   }
   else
   {
      if (db_memory_backup)
      {
         sqlite3_close(db_memory_backup);
         db_memory_backup = NULL;
      }
   }

   EndTransaction(db, TRUE);

   FUNCTION_FINISH(SQLite_RemoveBackup);

   return;
}


/**
 * @brief   Initialise empty 'db'
 * @param   db SQLite database handle
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN InitialiseDatabase(void *db)
{
   BOOLEAN result = TRUE;
   U32BIT i;

   FUNCTION_START(InitialiseDatabase);

   for (i = 0; (i < database_layout_length) && result; i++)
   {
      result = CreateDatabaseTable(db, &database_layout[i], NULL);
   }

   CreateDatabaseBlobsTable(db);

   SetUserVersion(db, database_layout_version);

   FUNCTION_FINISH(InitialiseDatabase);

   return result;
}

/**
 * @brief   Create 'table' in 'db'
 * @param   db SQLite database handle
 * @param   table Database table
 * @param   prefix Prefix on table name or NULL
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN CreateDatabaseTable(void *db, const S_TABLE *table, const char *prefix)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   U32BIT i;
   char *errmsg;

   static const char sql_create_table_1[] = "CREATE TABLE %st%lu(parenttable INT DEFAULT NULL, parentid INT";
   static const char sql_create_table_2[] = ", c%lu %s";
   static const char sql_create_table_3[] = ")";

   FUNCTION_START(CreateDatabaseTable);

   ASSERT(table);

   if (!prefix)
   {
      prefix = "";
   }

   /* create sql string */
   CreateString(&sql, sql_create_table_1, prefix, table->record_type);
   for (i = 0; i < table->length; i++)
   {
      AppendString(&sql, sql_create_table_2, table->columns[i].field_type, table->columns[i].data_type->sql);
   }
   AppendString(&sql, sql_create_table_3);
   DBG_SQL((sql.string));

   /* execute */
   if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
      result = FALSE;
   }

   DeleteString(&sql);

   FUNCTION_FINISH(CreateDatabaseTable);

   return result;
}

/**
 * @brief   Create blobs table in 'db'
 * @param   db SQLite database handle
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN CreateDatabaseBlobsTable(void *db)
{
   BOOLEAN result = TRUE;
   char *errmsg;

   static const char sql[] = "CREATE TABLE blobs(id INT, blob BLOB)";

   FUNCTION_START(CreateDatabaseBlobsTable);

   DBG_SQL((sql));

   if (sqlite3_exec(db, sql, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
      result = FALSE;
   }

   FUNCTION_FINISH(CreateDatabaseBlobsTable);

   return result;
}

/**
 * @brief   Insert 'record' into 'db'
 * @param   db SQLite database handle
 * @param   record Database record
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN CreateDatabaseRecord(void *db, S_RECORD *record)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   char *errmsg;

   static const char sql_insert_table_1[] = "INSERT INTO t%lu (parenttable, parentid) VALUES ('t%d', %lld)";
   static const char sql_insert_table_2[] = "INSERT INTO t%lu DEFAULT VALUES";

   FUNCTION_START(CreateDatabaseRecord);

   ASSERT(record);

   /* create sql string */
   CreateString(&sql, NULL);
   if (record->parent)
   {
      ASSERT(record->parent->operation != OPERATION_CREATE);
      AppendString(&sql, sql_insert_table_1, record->table->record_type,
         record->parent->table->record_type, record->parent->id);
   }
   else
   {
      AppendString(&sql, sql_insert_table_2, record->table->record_type);
   }
   DBG_SQL((sql.string));

   /* execute */
   if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) == SQLITE_OK)
   {
      record->id = sqlite3_last_insert_rowid(db);
      result = UpdateDatabaseRecord(db, record);
   }
   else
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
      result = FALSE;
   }

   DeleteString(&sql);

   FUNCTION_FINISH(CreateDatabaseRecord);

   return result;
}

/**
 * @brief   Delete 'record' from 'db'
 * @param   db SQLite database handle
 * @param   record Database record
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN DeleteDatabaseRecord(void *db, S_RECORD *record)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   char *errmsg;

   static const char sql_delete_table[] = "DELETE FROM t%lu WHERE rowid = %lld";

   FUNCTION_START(DeleteDatabaseRecord);

   ASSERT(record);

   CreateString(&sql, sql_delete_table, record->table->record_type, record->id);
   DBG_SQL((sql.string));

   if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
      result = FALSE;
   }

   DeleteString(&sql);

   FUNCTION_FINISH(DeleteDatabaseRecord);

   return result;
}

/**
 * @brief   Update 'record' in 'db'
 * @param   db SQLite database handle
 * @param   record Database record
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN UpdateDatabaseRecord(void *db, S_RECORD *record)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   U32BIT i;
   sqlite3_stmt *stmt;

   static const char sql_update_table_1[] = "UPDATE t%d SET";
   static const char sql_update_table_2[] = " parenttable = %lu, parentid = %lld";
   static const char sql_update_table_3[] = " parenttable = NULL";
   static const char sql_update_table_4[] = ", c%lu = ?";
   static const char sql_update_table_5[] = " WHERE rowid = %lld";

   FUNCTION_START(UpdateDatabaseRecord);

   ASSERT(record);

   /* create sql string */
   CreateString(&sql, sql_update_table_1, record->table->record_type);
   if (record->parent)
   {
      AppendString(&sql, sql_update_table_2, record->parent->table->record_type, record->parent->id);
   }
   else
   {
      AppendString(&sql, sql_update_table_3);
   }
   for (i = 0; i < record->table->length; i++)
   {
      AppendString(&sql, sql_update_table_4, record->table->columns[i].field_type);
   }
   AppendString(&sql, sql_update_table_5, record->id);
   DBG_SQL((sql.string));

   /* prepare */
   if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      /* bind field values */
      for (i = 0; (i < record->table->length) && result; i++)
      {
         if (record->table->columns[i].data_type == &database_type_uint)
         {
            if (sqlite3_bind_int64(stmt, i + 1, record->fields[i].value.uint) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", sqlite3_errmsg(db)));
               result = FALSE;
            }
         }
         else if (record->table->columns[i].data_type == &database_type_text)
         {
            Buffer_Ref(record->fields[i].value.text);
            if (sqlite3_bind_text(stmt, i + 1, record->fields[i].value.text,
                  record->fields[i].size, Buffer_Unref) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", sqlite3_errmsg(db)));
               result = FALSE;
            }
         }
         else if (record->table->columns[i].data_type == &database_type_blob)
         {
            Buffer_Ref(record->fields[i].value.blob);
            if (sqlite3_bind_blob(stmt, i + 1, record->fields[i].value.blob,
                  record->fields[i].size, Buffer_Unref) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", sqlite3_errmsg(db)));
               result = FALSE;
            }
         }
      }

      /* execute */
      if (result && (sqlite3_step(stmt) != SQLITE_DONE))
      {
         result = FALSE;
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
         result = FALSE;
      }
   }
   else
   {
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      result = FALSE;
   }

   DeleteString(&sql);

   FUNCTION_FINISH(UpdateDatabaseRecord);

   return result;
}

/**
 * @brief   Load database records into the in-memory database
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN LoadDatabaseRecords(void)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   U32BIT i, j, col;
   sqlite3_stmt *stmt;
   S_RECORD *record;
   const S_TABLE *parent_table;
   S_RECORD *parent;

   static const char sql_select_table[] = "SELECT rowid, * FROM t%lu";

   FUNCTION_START(LoadDatabaseRecords);

   ASSERT(sqlite_db_open);

   CreateString(&sql, NULL);

   /* for each table type */
   for (i = 0; (i < database_layout_length) && result; i++)
   {
      SetString(&sql, sql_select_table, database_layout[i].record_type);
      DBG_SQL((sql.string));

      if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
      {
         ASSERT(stmt);

         /* for each row in table */
         while ((sqlite3_step(stmt) == SQLITE_ROW) && result)
         {
            /* create in-memory database record */
            if ((record = Database_CreateRecord(&database_layout[i], NULL)))
            {
               record->id = sqlite3_column_int64(stmt, 0); /* (0) rowid */

               /* set parent */
               if (sqlite3_column_type(stmt, 1) != SQLITE_NULL)
               {
                  /* (1) parenttable, (2) parentid */
                  parent = NULL;
                  if ((parent_table = Database_GetTable(sqlite3_column_int64(stmt, 1))) != NULL)
                  {
                     parent = Database_FindRecord(parent_table, sqlite3_column_int64(stmt, 2));
                  }

                  if (parent != NULL)
                  {
                     Database_SetRecordParent(record, parent);
                  }
                  else
                  {
                     result = FALSE;
                  }
               }

               /* set fields ((3) column, ...) */
               for (j = 0, col = 3; (j < record->table->length) && result; j++, col++)
               {
                  if (record->table->columns[j].data_type == &database_type_uint)
                  {
                     Database_SetRecordFieldUInt(&record->fields[j], sqlite3_column_int64(stmt, col));
                  }
                  else if (record->table->columns[j].data_type == &database_type_text)
                  {
                     Database_SetRecordFieldText(&record->fields[j], sqlite3_column_text(stmt, col),
                        sqlite3_column_bytes(stmt, col));
                  }
                  else if (record->table->columns[j].data_type == &database_type_blob)
                  {
                     Database_SetRecordFieldBlob(&record->fields[j], sqlite3_column_blob(stmt, col),
                        sqlite3_column_bytes(stmt, col));
                  }
               }
            }
            else
            {
               result = FALSE;
            }
         }

         if (sqlite3_finalize(stmt) != SQLITE_OK)
         {
            DBG_SQLITE(("%s", sqlite3_errmsg(db)));
            result = FALSE;
         }
      }
      else
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
         result = FALSE;
      }
   }

   DeleteString(&sql);

   FUNCTION_FINISH(LoadDatabaseRecords);

   return result;
}

/**
 * @brief   Upgrade 'db' to current version
 * @param   db SQLite database handle
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN UpgradeDatabase(void *db)
{
   BOOLEAN result = TRUE;
   S_STRING sql;
   U32BIT i, type;
   S_STRING names;
   char *errmsg;

   static const char sql_insert[] = "INSERT INTO new_t%lu (rowid, parenttable, parentid%s) "
      "SELECT rowid, parenttable, parentid%s FROM t%lu; ";
   static const char sql_drop_rename[] = "DROP TABLE t%lu; ALTER TABLE new_t%lu RENAME TO t%lu";

   FUNCTION_START(UpgradeDatabase);

   CreateString(&sql, NULL);

   /* for each table type */
   for (i = 0; (i < database_layout_length) && result; i++)
   {
      /* create new table */
      if ((result = CreateDatabaseTable(db, &database_layout[i], "new_")))
      {
         /* get column names that exist in present and current layouts */
         names = GetUpgradeableColumnNames(db, &database_layout[i]);
         if (names.string)
         {
            type = database_layout[i].record_type;
            /* copy data into new table, drop old table and rename new table */
            SetString(&sql, sql_insert, type, names.string, names.string, type);
            AppendString(&sql, sql_drop_rename, type, type, type);
            DBG_SQL((sql.string));

            if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) != SQLITE_OK)
            {
               DBG_SQLITE(("%s", errmsg));
               sqlite3_free(errmsg);
               result = FALSE;
            }

            DeleteString(&names);
         }
      }
   }

   SetUserVersion(db, database_layout_version);

   DeleteString(&sql);

   FUNCTION_FINISH(UpgradeDatabase);

   return result;
}

/**
 * @brief   Get list of upgradeable column names in 'table'
 * @param   db SQLite database handle
 * @param   table Database table
 * @return  String on success, deleted string on failure
 */
static S_STRING GetUpgradeableColumnNames(void *db, const S_TABLE *table)
{
   S_STRING result;
   S_STRING sql;
   U32BIT i;
   sqlite3_stmt *stmt;
   const char *name;
   char *endptr;
   U32BIT field_type;

   static const char sql_table_info[] = "PRAGMA table_info(t%lu)";

   FUNCTION_START(GetUpgradeableColumnNames);

   CreateString(&result, NULL);

   /* get columns in present schema */
   CreateString(&sql, sql_table_info, table->record_type);
   DBG_SQL((sql.string));

   if (sqlite3_prepare_v2(db, sql.string, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      while (sqlite3_step(stmt) == SQLITE_ROW)
      {
         /* check column name prefix */
         name = (char *)sqlite3_column_text(stmt, 1);
         if (name[0] != 'c')
         {
            continue;
         }

         /* get field type from column name */
         field_type = strtoul(name + 1, &endptr, 10);
         if (endptr <= (name + 1))
         {
            continue;
         }

         /* append name if column also exists in current layout */
         for (i = 0; i < table->length; i++)
         {
            if (table->columns[i].field_type == field_type)
            {
               AppendString(&result, ", c%lu", field_type);
               continue;
            }
         }
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DeleteString(&result);
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      }
   }
   else
   {
      DeleteString(&result);
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
   }

   DeleteString(&sql);

   FUNCTION_FINISH(GetUpgradeableColumnNames);

   return result;
}

/**
 * @brief   Get database user version
 * @param   db SQLite database handle
 * @param   user_version User version out
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN GetUserVersion(void *db, S32BIT *user_version)
{
   BOOLEAN result = FALSE;
   sqlite3_stmt *stmt;

   static const char sql[] = "PRAGMA user_version";

   FUNCTION_START(GetUserVersion);

   ASSERT(user_version);

   DBG_SQL((sql));

   if (sqlite3_prepare_v2(db, sql, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      if (sqlite3_step(stmt) == SQLITE_ROW)
      {
         *user_version = sqlite3_column_int(stmt, 0);
         result = TRUE;
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      }
   }
   else
   {
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
   }

   FUNCTION_FINISH(GetUserVersion);

   return result;
}

/**
 * @brief   Set database user version to 'user_version'
 * @param   db SQLite database handle
 * @param   user_version User version
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN SetUserVersion(void *db, S32BIT user_version)
{
   BOOLEAN result = FALSE;
   S_STRING sql;
   char *errmsg;

   static const char sql_set_user_version[] = "PRAGMA user_version = %d";

   FUNCTION_START(SetUserVersion);

   CreateString(&sql, sql_set_user_version, user_version);
   DBG_SQL((sql.string));

   if (sqlite3_exec(db, sql.string, NULL, NULL, &errmsg) == SQLITE_OK)
   {
      result = TRUE;
   }
   else
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
   }

   DeleteString(&sql);

   FUNCTION_FINISH(SetUserVersion);

   return result;
}

/**
 * @brief   Check database integrity
 * @param   db SQLite database handle
 * @return  TRUE on success, FALSE on failure
 */
static BOOLEAN IntegrityCheck(void *db)
{
   BOOLEAN result = TRUE;
   sqlite3_stmt *stmt;

   static const char sql[] = "PRAGMA integrity_check";

   FUNCTION_START(IntegrityCheck);

   DBG_SQL((sql));

   if (sqlite3_prepare_v2(db, sql, -1, &stmt, NULL) == SQLITE_OK)
   {
      ASSERT(stmt);

      while ((sqlite3_step(stmt) == SQLITE_ROW))
      {
         if (strcmp((const char *)sqlite3_column_text(stmt, 0), "ok") != 0)
         {
            DBG_SQLITE(("%s", sqlite3_column_text(stmt, 0)));
            result = FALSE;
         }
      }

      if (sqlite3_finalize(stmt) != SQLITE_OK)
      {
         DBG_SQLITE(("%s", sqlite3_errmsg(db)));
         result = FALSE;
      }
   }
   else
   {
      DBG_SQLITE(("%s", sqlite3_errmsg(db)));
      result = FALSE;
   }

   FUNCTION_FINISH(IntegrityCheck);

   return result;
}

/**
 * @brief   Start database transaction
 * @param   db SQLite database handle
 */
static void StartTransaction(void *db)
{
   char *errmsg;

   static const char sql[] = "BEGIN";

   FUNCTION_START(StartTransaction);

   sqlite3_mutex_enter(transaction);

   DBG_SQL((sql));

   if (sqlite3_exec(db, sql, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
   }

   FUNCTION_FINISH(StartTransaction);

   return;
}

/**
 * @brief   End database transaction
 * @param   db SQLite database handle
 */
static void EndTransaction(void *db, BOOLEAN commit)
{
   char *errmsg;
   const char *sql;

   static const char sql_commit_transaction[] = "COMMIT";
   static const char sql_rollback_transaction[] = "ROLLBACK";

   FUNCTION_START(EndTransaction);

   if (commit)
   {
      sql = sql_commit_transaction;
   }
   else
   {
      sql = sql_rollback_transaction;
   }

   DBG_SQL((sql));

   if (sqlite3_exec(db, sql, NULL, NULL, &errmsg) != SQLITE_OK)
   {
      DBG_SQLITE(("%s", errmsg));
      sqlite3_free(errmsg);
   }

   sqlite3_mutex_leave(transaction);

   FUNCTION_FINISH(EndTransaction);

   return;
}

/**
 * @brief   Create string
 * @param   string String object to initialise
 * @param   format Format string or NULL
 * @param   ... Arguments
 */
static void CreateString(S_STRING *string, const char *format, ...)
{
   va_list args;

   FUNCTION_START(CreateString);

   ASSERT(string);

   string->size = 0;
   string->length = 0;

   ASSERT(STRING_INITIAL_SIZE > 0);
   if ((string->string = STB_GetMemory(STRING_INITIAL_SIZE)))
   {
      string->size = STRING_INITIAL_SIZE;
      string->string[0] = '\0';

      if (format)
      {
         va_start(args, format);
         AppendStringArgs(string, format, args);
         va_end(args);
      }
   }

   FUNCTION_FINISH(CreateString);

   return;
}

/**
 * @brief   Set 'string' to 'format'
 * @param   string String object
 * @param   format Format string or NULL
 * @param   ... Arguments
 */
static void SetString(S_STRING *string, const char *format, ...)
{
   va_list args;

   FUNCTION_START(SetString);

   ASSERT(string);

   string->length = 0;
   if (string->string)
   {
      string->string[0] = '\0';

      if (format)
      {
         va_start(args, format);
         AppendStringArgs(string, format, args);
         va_end(args);
      }
   }

   FUNCTION_FINISH(SetString);

   return;
}

/**
 * @brief   Append 'format' to 'string'
 * @param   string String object
 * @param   format Format string
 * @param   ... Arguments
 */
static void AppendString(S_STRING *string, const char *format, ...)
{
   va_list args;

   FUNCTION_START(AppendString);

   va_start(args, format);
   AppendStringArgs(string, format, args);
   va_end(args);

   FUNCTION_FINISH(AppendString);

   return;
}

/**
 * @brief   Append 'format' to 'string'
 * @param   string String object
 * @param   format Format string
 * @param   ... Arguments
 */
static void AppendStringArgs(S_STRING *string, const char *format, va_list args)
{
   S32BIT required;

   FUNCTION_START(AppendStringArgs);

   ASSERT(string);

   if (string->string)
   {
      required = vsnprintf(string->string + string->length, string->size - string->length, format, args);

      /* if string too small resize and repeat */
      if (required >= (S32BIT)(string->size - string->length))
      {
         string->size = string->length + required + 1;
         if ((string->string = STB_ResizeMemory(string->string, string->size)))
         {
            required = vsnprintf(string->string + string->length, string->size - string->length, format, args);
         }
         else
         {
            required = -1;
         }
      }

      if (required >= 0)
      {
         string->length += required;
      }
      else
      {
         DeleteString(string);
      }
   }

   FUNCTION_FINISH(AppendStringArgs);

   return;
}

/**
 * @brief   Delete string
 * @param   string String object
 */
static void DeleteString(S_STRING *string)
{
   FUNCTION_START(DeleteString);

   ASSERT(string);

   string->size = 0;
   string->length = 0;
   if (string->string)
   {
      STB_FreeMemory(string->string);
      string->string = NULL;
   }

   FUNCTION_FINISH(DeleteString);

   return;
}

