/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Interface to SQLite. Loads database records into the in-memory database and saves
 *          in-memory database changes into the database. Reads and writes database blobs
 * @file    sqlite.h
 * @date    01/01/2017
 * @author  Ocean Blue
 */

#ifndef DBA_SQLITE_H
#define DBA_SQLITE_H

#include "techtype.h"

/**
 * @brief   Initialise SQLite interface
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Initialise(void);

/**
 * @brief   Uninitialise SQLite interface
 */
void SQLite_Uninitialise(void);

/**
 * @brief   Open database 'filename'. The database is created and initialised if it does not
 *          exist. The database is upgraded if it is older than the current version. The in-memory
 *          database is cleared and database records are loaded into it
 * @param   filename Database file
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Open(const char *filename);

/**
 * @brief   Close database
 */
void SQLite_Close(void);

/**
 * @brief   Reset database and clear in-memory database. Database closed on failure
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_Reset(void);

/**
 * @brief   Save in-memory database changes into the database
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_SaveDatabaseRecords(void);

/**
 * @brief   Get size of blob 'id'
 * @param   id Blob id
 * @return  Size on success, 0 on failure
 */
U32BIT SQLite_GetBlobSize(U32BIT id);

/**
 * @brief   Read blob 'id' into 'data'
 * @param   id Blob id
 * @param   data Data buffer
 * @param   data_size Size of 'data'
 * @return  Size read on success, 0 on failure
 */
U32BIT SQLite_ReadBlob(U32BIT id, void *data, U32BIT data_size);

/**
 * @brief   Write 'data' into blob 'id'
 * @param   id Blob id
 * @param   data Data buffer
 * @param   data_size Size of 'data'
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_WriteBlob(U32BIT id, const void *data, U32BIT data_size);

/**
 * @brief   Create backup 'filename'
 * @param   filename Database file or NULL for memory backup
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_CreateBackup(const char *filename);

/**
 * @brief   Restore backup 'filename'
 * @param   filename Database file or NULL for memory backup
 * @param   check_exists_only Only check whether the backup exists
 * @return  TRUE on success, FALSE on failure
 */
BOOLEAN SQLite_RestoreBackup(const char *filename, BOOLEAN check_exists_only);

/**
 * @brief   Remove backup 'filename'
 * @param   filename Database file or NULL for memory backup
 */
void SQLite_RemoveBackup(const char *filename);

#endif

