/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Database access defines, structures and public functions
 * @file    dba.h
 * @date    11/11/2013
 * @author  Ocean Blue
 */

/* Pre-processor mechanism so multiple inclusions don't cause compilation error */

#ifndef _DBA_H
#define _DBA_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

/* Field IDs MUST be unique but need not be contiguous within records range 0x0000 to 0xffff
 * some fields are used in more than one record spare IDs are left for future expansion. */

/* Fields common to more than one record type */
#define DBA_FIELD_PARENT                        0x0100
#define DBA_FIELD_REC_NAME                      0x0101
#define DBA_FIELD_ORIG_NET_ID                   0x0102
#define DBA_FIELD_NET_ID                        0x0103
#define DBA_FIELD_TRANSPORT_ID                  0x0104
#define DBA_FIELD_SERVICE_ID                    0x0105
#define DBA_FIELD_VERSION                       0x0106

/* Fields in LNB records */
#define DBA_FIELD_LNB_TYPE                      0x0202
#define DBA_FIELD_LNB_22K                       0x0203
#define DBA_FIELD_LNB_12V                       0x0204
#define DBA_FIELD_LNB_PULSEPOSN                 0x0205
#define DBA_FIELD_LNB_DISPOSN                   0x0206
#define DBA_FIELD_LNB_DISTONE                   0x0207
#define DBA_FIELD_LNB_DISCSWITCH                0x0208
#define DBA_FIELD_LNB_DISUSWITCH                0x0209
#define DBA_FIELD_LNB_DISSMATV                  0x020a
#define DBA_FIELD_LNB_DISREPEAT                 0x020b
#define DBA_FIELD_LNB_UNICABLEFREQ              0x020c
#define DBA_FIELD_LNB_UNICABLECHAN              0x020d
#define DBA_FIELD_LNB_POWER                     0x020e
#define DBA_FIELD_LNB_NAME                      0x020f

/* Fields in satellite records */
#define DBA_FIELD_SAT_DISH                      0x0300
#define DBA_FIELD_SAT_LONGWE                    0x0301
#define DBA_FIELD_SAT_LONGPOS                   0x0302

/* Fields in network records */
#define DBA_FIELD_PROFILE_TYPE                  0x0401
#define DBA_FIELD_PROFILE_CAM_ID                0x0402
#define DBA_FIELD_PROFILE_NAME                  0x0403
#define DBA_FIELD_OPERATOR_SEARCH               0x0404
#define DBA_FIELD_OP_SEARCH_DATE                0x0405
#define DBA_FIELD_OP_SEARCH_TIME                0x0406

/* Fields common to transport records */
#define DBA_FIELD_TRAN_FREQ                     0x0500
#define DBA_FIELD_TRAN_SRATE                    0x0501
#define DBA_FIELD_TRAN_SIGNAL_STRENGTH          0x0502
#define DBA_FIELD_TRAN_SIGNAL_QUALITY           0x0503

/* Fields in satellite transport records */
#define DBA_FIELD_STRAN_POL                     0x0600
#define DBA_FIELD_STRAN_FEC                     0x0601
#define DBA_FIELD_STRAN_DVBS2                   0x0602
#define DBA_FIELD_STRAN_MODULATION              0x0603

/* Fields in terrestrial transport records */
#define DBA_FIELD_TTRAN_MODE                    0x0700
#define DBA_FIELD_TTRAN_TERR_TYPE               0x0701
#define DBA_FIELD_TTRAN_PLP_ID                  0x0702
#define DBA_FIELD_TTRAN_BWIDTH                  0x0703

/* Fields in cable transport records */
#define DBA_FIELD_CTRAN_MODE                    0x0800

/* Fields common to service records */
#define DBA_FIELD_SERV_ID                       0x0900
#define DBA_FIELD_SERV_TYPE                     0x0901
#define DBA_FIELD_SERV_LCN                      0x0902
#define DBA_FIELD_SERV_REQ_LCN                  0x0903
#define DBA_FIELD_SERV_HIDDEN                   0x0904
#define DBA_FIELD_SERV_SELECTABLE               0x0905
#define DBA_FIELD_SERV_LOCKED                   0x0906
#define DBA_FIELD_SERV_SCHED_DISABLED           0x0907
#define DBA_FIELD_SERV_NOWNEXT_DISABLED         0x0908
#define DBA_FIELD_SERV_FAV_GROUPS               0x0909
#define DBA_FIELD_SERV_FREESAT_ID               0x090a
#define DBA_FIELD_SERV_REGION_ID                0x090b
#define DBA_FIELD_SERV_LCN_EDITABLE             0x090c
#define DBA_FIELD_SERV_DELETED                  0x090d
/* reserved */
#define DBA_FIELD_SERV_GROUP_MASK               0x09a0

/* Fields in timer records */
#define DBA_FIELD_TIMER_HANDLE                  0x0a00
#define DBA_FIELD_TIMER_STARTTIME               0x0a01
#define DBA_FIELD_TIMER_DURATION                0x0a02
#define DBA_FIELD_TIMER_TYPE                    0x0a03
#define DBA_FIELD_TIMER_FREQUENCY               0x0a04
#define DBA_FIELD_TIMER_RAMPVOLUME              0x0a05
#define DBA_FIELD_TIMER_EVENTID                 0x0a06
#define DBA_FIELD_TIMER_NOTIFY_TIME             0x0a07
#define DBA_FIELD_TIMER_CRID                    0x0a08
#define DBA_FIELD_TIMER_DISKID                  0x0a09
#define DBA_FIELD_TIMER_OTHERCRID               0x0a0a
#define DBA_FIELD_TIMER_MISSED                  0x0a0b
#define DBA_FIELD_TIMER_EVENT_TRIGGERED         0x0a0c
#define DBA_FIELD_TIMER_ADDITIONAL_INFO         0x0a0d
#define DBA_FIELD_TIMER_START_PADDING           0x0a0e
#define DBA_FIELD_TIMER_END_PADDING             0x0a0f
#define DBA_FIELD_TIMER_DO_NOT_DELETE           0x0a10

/* Fields in CRID records */
#define DBA_FIELD_CRID_EIT_DATE                 0x0b00
#define DBA_FIELD_CRID_SERIES                   0x0b01
#define DBA_FIELD_CRID_RECOMMENDED              0x0b02
#define DBA_FIELD_CRID_DO_NOT_DELETE            0x0b03

/* Fields in favourite lists and favourite services */
#define DBA_FIELD_FAVLIST_ID                    0x0c00
#define DBA_FIELD_FAVLIST_INDEX                 0x0c01
#define DBA_FIELD_FAVLIST_USER_DATA             0x0c02

/* Fields used to store event information */
#define DBA_FIELD_EVENT_STARTTIME               0x0d00
#define DBA_FIELD_EVENT_DURATION                0x0d01
#define DBA_FIELD_EVENT_ID                      0x0d02
#define DBA_FIELD_EVENT_CONTENT                 0x0d03
#define DBA_FIELD_EVENT_AGE_RATING              0x0d04
#define DBA_FIELD_EVENT_SCRAMBLED               0x0d05
#define DBA_FIELD_EVENT_SUBTITLES               0x0d06
#define DBA_FIELD_EVENT_AUDIO_DESC              0x0d07
#define DBA_FIELD_EVENT_FREE_TO_AIR             0x0d08
#define DBA_FIELD_EVENT_DO_NOT_SCRAMBLE         0x0d09
#define DBA_FIELD_EVENT_NAME                    0x0d0a
#define DBA_FIELD_EVENT_DESCRIPTION             0x0d0b
#define DBA_FIELD_EVENT_EXTENDED_DESC           0x0d0c
#define DBA_FIELD_EVENT_GUIDANCE                0x0d0d
#define DBA_FIELD_EVENT_CONTENT_DATA            0x0d0e

/* Fields in LNB band records */
#define DBA_FIELD_BAND_POLARITY                 0x0e00
#define DBA_FIELD_BAND_MIN_FREQUENCY            0x0e01
#define DBA_FIELD_BAND_MAX_FREQUENCY            0x0e02
#define DBA_FIELD_BAND_LOCAL_OSC_FREQUENCY      0x0e03
#define DBA_FIELD_BAND_LNB_VOLTAGE              0x0e04
#define DBA_FIELD_BAND_22_KHZ                   0x0e05


//---Enumerations for public use-----------------------------------------------

/* Set of record types supported by the database, though not all types may be supported
 * by all database implementations */
typedef enum
{
   DBA_RECORD_LNB = 0,
   DBA_RECORD_SATELLITE,
   DBA_RECORD_NETWORK,
   DBA_RECORD_SAT_TRANSPORT,
   DBA_RECORD_TERR_TRANSPORT,
   DBA_RECORD_CAB_TRANSPORT,
   DBA_RECORD_SERVICE,
   DBA_RECORD_TIMER,
   DBA_RECORD_CRID,
   DBA_RECORD_FAV_LIST,
   DBA_RECORD_FAV_SERV,
   DBA_RECORD_LNB_BAND,
   DBA_RECORD_CICAM_TIMER,
   /*DBA_RECORD_EVENT,*/
   DBA_NUM_RECORDS               /* This must be the last value in the enum */
} E_DBA_RECORDS;


//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Performs any initialisation required prior to the database being loaded
 * @return  TRUE if initialisation is successful, FALSE otherwise
 */
BOOLEAN DBA_Initialise(void);

/**
 * @brief   Releases any resources acquired by initialisation and clears any currently loaded
 *          database, including a backup, if one has been created.
 */
void DBA_Terminate(void);

/**
 * @brief   Reads a database from non-volatile storage, creating any structures in memory that
 *          will be required to access it and the records it contains and makes this the working
 *          database. If the database is found to be invalid, or doesn't exist at all, then an
 *          empty database should be created.
 * @param   pathname full pathname of the file to be loaded containing the database.
 *                     This argument is only relevant where the system being used is file based.
 * @return  TRUE if a database is loaded successfully, FALSE otherwise
 */
BOOLEAN DBA_LoadDatabase(U8BIT *pathname);

/**
 * @brief   Saves any changes made to the working database to non-volatile storage.
 *          If saving to a file, the pathname used to open it will be used.
 * @return  TRUE if the database is saved successfully, FALSE otherwise
 */
BOOLEAN DBA_SaveDatabase(void);

/**
 * @brief   Clears the working database of all records. Following this call, it should still
 *          be possible to access the database, but it will be empty, and new records can be
 *          added to it.
 * @return  TRUE if the database is cleared, FALSE otherwise
 */
BOOLEAN DBA_ClearDatabase(void);

/**
 * @brief   Creates a backup copy of the working database. Whether the backup database is saved
 *          to non-volatile storage is implementation dependant and so the backup database may
 *          not survive a system reboot.
 * @param   pathname full pathname of the file to save the backup to if this is supported
 *                     by the database implementation.
 * @return  TRUE if a backup is created, FALSE otherwise
 */
BOOLEAN DBA_BackupDatabase(U8BIT *pathname);

/**
 * @brief   Checks whether the working database can be restored from a backup copy
 * @return  TRUE if there is a backup copy and it can be restored, FALSE otherwise
 */
BOOLEAN DBA_CanRestoreDatabase(void);

/**
 * @brief   Restores the working database from a previously made backup copy.
 * @return  TRUE if the working database is restored from the backup, FALSE otherwise
 */
BOOLEAN DBA_RestoreDatabase(void);

/**
 * @brief   Erases the backup copy of the database. If data was stored in non-volatile storage
 *          then this should be deleted too.
 */
void DBA_EraseBackupDatabase(void);

/**
 * @brief   Export the working database to an XML file.
 * @param   xml_pathname full pathname of the file to export to
 * @return  TRUE if the database is exported successfully, FALSE otherwise
 */
BOOLEAN DBA_ExportToXML(U8BIT *xml_pathname);

/**
 * @brief   Imports records from the given XML file into the working database. If a record already
 *          exists in the database then it will be updated, and if it doesn't then a new record
 *          will be created.
 * @param   xml_pathname full pathname of the file to be imported
 * @return  TRUE if the file is imported successfully, FALSE otherwise
 */
BOOLEAN DBA_ImportFromXML(U8BIT *xml_pathname);

/**
 * @brief   Locks the database to prevent access from other threads or processes.
 */
void DBA_LockDatabase(void);

/**
 * @brief   Unlocks the database to allow other threads or processes to access it.
 */
void DBA_UnlockDatabase(void);

/**
 * @brief   Returns a version string representing the working database.
 * @return  '\0' terminated string in UTF-8 format
 */
U8BIT* DBA_DatabaseVersion(void);

/**
 * @brief   Returns the size in bytes of the database as stored in non-volatile storage
 * @param   max_size returns the maximum size in bytes that the database can grow to.
 *                     This value may be returned as 0 if this can't be determined.
 * @return  current size of the working database in bytes
 */
U32BIT DBA_DatabaseFileSize(U32BIT *max_size);

/**
 * @brief   Creates a new record of the given type, adding it to the database as a child of the
 *          given parent record, if provided.
 * @param   record_id type of record to be created
 * @param   parent create a record that is a child of this parent, can be passed as NULL
 * @return  handle of new record, or NULL if creation fails
 */
void* DBA_CreateRecord(U32BIT record_id, void *parent);

/**
 * @brief   Destroys the given record, removing it from the database and freeing any memory
 *          associated with it.
 * @param   record database record to be deleted
 */
void DBA_DestroyRecord(void *record);

/**
 * @brief   Finds the next record, of the given type, that comes after last_rec. last_rec
 *          must be the same type of record as the one being found. Parent is optional, but
 *          if provided, a record will only be returned if the parent of the one found is the same.
 * @param   record_id type of record to look for
 * @param   parent if not NULL, this must be the parent of the record found for it to be returned
 * @param   last_rec previous record of the type being looked for, if NULL then the first record
 *                     will be returned.
 * @return  handle of the record, or NULL if none found
 */
void* DBA_FindRecord(U32BIT record_id, void *parent, void *last_rec);

/**
 * @brief   Set of change the parent of the given record
 * @param   record handle of the record whose parent is to be changed
 * @param   parent handle of the new parent record, can be NULL
 */
void DBA_SetRecordParent(void *record, void *parent);

/**
 * @brief   Returns the handle to the parent of the given record
 * @param   record handle of record whose parent is to be returned
 * @return  handle of the record's parent
 */
void* DBA_GetRecordParent(void *record);

/**
 * @brief   Forces a record to be saved to non-volatile storage. Depending on the implementation,
 *          this function may not do anything if the data is updated to non-volatile storage
 *          as any records and/or fields are created or updated.
 * @param   record handle of record to be saved
 */
void DBA_SaveRecord(void *record);

/**
 * @brief   Set the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a value field.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   value new value of the field
 * @return  TRUE if the value is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldValue(void *record, U32BIT field_id, U32BIT value);

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldString(void *record, U32BIT field_id, U8BIT *string, U16BIT num_bytes);

/**
 * @brief   Set the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The string data provided will be copied and no interpretation is made of the format
 *          of the string.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   lang_code language code of the string
 * @param   string string value of the field
 * @param   num_bytes number of bytes in the string, should include null terminator if present
 * @return  TRUE if the string is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT *string,
   U16BIT num_bytes);

/**
 * @brief   Set a variable amount of data of a record's field. The function will fail if the
 *          record doesn't exist, the record doesn't include the given field, or the field
 *          doesn't hold data. The data provided will be copied.
 * @param   record handle of the record to be changed
 * @param   field_id field within the record to be changed
 * @param   data data to be stored in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if the data is set, FALSE otherwise
 */
BOOLEAN DBA_SetFieldData(void *record, U32BIT field_id, U8BIT *data, U16BIT num_bytes);

/**
 * @brief   Gets the value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a value field.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   value pointer to the returned value if function returns TRUE
 * @return  TRUE if a value is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldValue(void *record, U32BIT field_id, U32BIT *value);

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The pointer to the string returned will be to the string data held by the database,
 *          so the data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldString(void *record, U32BIT field_id, U8BIT **string, U16BIT *num_bytes);

/**
 * @brief   Gets the string value of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a string field.
 *          The pointer to the string returned will be to the string data held by the database,
 *          so the data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   lang_code language code of the string to be returned
 * @param   string pointer to the returned string value in the field
 * @param   num_bytes number of bytes in the string
 * @return  TRUE if a string is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldLangString(void *record, U32BIT field_id, U32BIT lang_code, U8BIT **string,
   U16BIT *num_bytes);

/**
 * @brief   Gets the data of a record's field. The function will fail if the record doesn't
 *          exist, the record doesn't include the given field, or the field isn't a data field.
 *          The pointer to the data returned will be to the data held by the database, so the
 *          data must not be changed.
 * @param   record handle of the record being queried
 * @param   field_id field within the record being queried
 * @param   data pointer to the returned data in the field
 * @param   num_bytes number of bytes of data
 * @return  TRUE if data is returned, FALSE otherwise
 */
BOOLEAN DBA_GetFieldData(void *record, U32BIT field_id, U8BIT **data, U16BIT *num_bytes);

/**
 * @brief   Returns the number of bytes available for the given data block
 * @param   data_block_id the data block whose size is to be returned
 * @return  size of bytes of the data block, or 0 if block not found
 */
U32BIT DBA_DataBlockSize(U32BIT data_block_id);

/**
 * @brief   Read a block of data from the database into the given buffer
 * @param   data_block_id id of the data block to be read
 * @param   data pointer to the buffer into which the data will be read
 * @param   max_num_bytes the maximum number of bytes to be read
 * @return  number of bytes read into the buffer
 */
U32BIT DBA_DataBlockRead(U32BIT data_block_id, U8BIT *data, U32BIT max_num_bytes);

/**
 * @brief   Writes a block of data into the database from the given buffer
 * @param   data_block_id id of the data block being written
 * @param   data pointer to the data to be written
 * @param   num_bytes number of bytes of data to be written
 * @return  TRUE if the data is written, FALSE otherwise
 */
BOOLEAN DBA_DataBlockWrite(U32BIT data_block_id, U8BIT *data, U32BIT num_bytes);

#endif /* _DBA_H */

