/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application configuration
 * @file    ap_cfg.h
 * @date    5/04/2004
 * @author  Ocean Blue
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _AP_CFG_H
#define _AP_CFG_H

#include "techtype.h"
#include "stbdpc.h"
#include "stbsitab.h"

//---Constant and macro definitions for public use-----------------------------

#define ACFG_INVALID_REGION   255
#define ACFG_INVALID_LANG     255

#define ACFG_REGION_TEST      (ACFG_INVALID_REGION - 1)

/* Country codes - please add new countries in alphabetical order */

#define COUNTRY_CODE_ARGENTINA    (U32BIT)(('a' << 16) | ('r' << 8) | 'g')
#define COUNTRY_CODE_AUSTRALIA    (U32BIT)(('a' << 16) | ('u' << 8) | 's')
#define COUNTRY_CODE_AUSTRIA      (U32BIT)(('a' << 16) | ('u' << 8) | 't')
#define COUNTRY_CODE_BELGIUM      (U32BIT)(('b' << 16) | ('e' << 8) | 'l')
#define COUNTRY_CODE_BOLIVIA      (U32BIT)(('b' << 16) | ('o' << 8) | 'l')
#define COUNTRY_CODE_BRAZIL       (U32BIT)(('b' << 16) | ('r' << 8) | 'a')
#define COUNTRY_CODE_CHILE        (U32BIT)(('c' << 16) | ('h' << 8) | 'l')
#define COUNTRY_CODE_CHINA        (U32BIT)(('c' << 16) | ('h' << 8) | 'i')
#define COUNTRY_CODE_COLOMBIA     (U32BIT)(('c' << 16) | ('o' << 8) | 'l')
#define COUNTRY_CODE_COSTARICA    (U32BIT)(('c' << 16) | ('r' << 8) | 'i')
#define COUNTRY_CODE_CROATIA      (U32BIT)(('h' << 16) | ('r' << 8) | 'v')
#define COUNTRY_CODE_CZECHREP     (U32BIT)(('c' << 16) | ('z' << 8) | 'e')
#define COUNTRY_CODE_DENMARK      (U32BIT)(('d' << 16) | ('n' << 8) | 'k')
#define COUNTRY_CODE_DOMINICANREP (U32BIT)(('d' << 16) | ('m' << 8) | 'a')
#define COUNTRY_CODE_ECUADOR      (U32BIT)(('e' << 16) | ('c' << 8) | 'u')
#define COUNTRY_CODE_ELSALVADOR   (U32BIT)(('s' << 16) | ('l' << 8) | 'v')
#define COUNTRY_CODE_FINLAND      (U32BIT)(('f' << 16) | ('i' << 8) | 'n')
#define COUNTRY_CODE_FRANCE       (U32BIT)(('f' << 16) | ('r' << 8) | 'a')
#define COUNTRY_CODE_GERMANY      (U32BIT)(('d' << 16) | ('e' << 8) | 'u')
#define COUNTRY_CODE_GUATEMALA    (U32BIT)(('g' << 16) | ('t' << 8) | 'm')
#define COUNTRY_CODE_HONDURAS     (U32BIT)(('h' << 16) | ('n' << 8) | 'd')
#define COUNTRY_CODE_HUNGARY      (U32BIT)(('h' << 16) | ('u' << 8) | 'n')
#define COUNTRY_CODE_IRELAND      (U32BIT)(('i' << 16) | ('r' << 8) | 'l')
#define COUNTRY_CODE_ITALY        (U32BIT)(('i' << 16) | ('t' << 8) | 'a')
#define COUNTRY_CODE_LATVIA       (U32BIT)(('l' << 16) | ('v' << 8) | 'a')
#define COUNTRY_CODE_LUXEMBOURG   (U32BIT)(('l' << 16) | ('u' << 8) | 'x')
#define COUNTRY_CODE_MEXICO       (U32BIT)(('m' << 16) | ('e' << 8) | 'x')
#define COUNTRY_CODE_NETHERLANDS  (U32BIT)(('n' << 16) | ('l' << 8) | 'd')
#define COUNTRY_CODE_NICARAGUA    (U32BIT)(('n' << 16) | ('i' << 8) | 'c')
#define COUNTRY_CODE_NORWAY       (U32BIT)(('n' << 16) | ('o' << 8) | 'r')
#define COUNTRY_CODE_PANAMA       (U32BIT)(('p' << 16) | ('a' << 8) | 'n')
#define COUNTRY_CODE_PERU         (U32BIT)(('p' << 16) | ('e' << 8) | 'r')
#define COUNTRY_CODE_POLAND       (U32BIT)(('p' << 16) | ('o' << 8) | 'l')
#define COUNTRY_CODE_PORTUGAL     (U32BIT)(('p' << 16) | ('o' << 8) | 'r')
#define COUNTRY_CODE_RUSSIA       (U32BIT)(('r' << 16) | ('u' << 8) | 's')
#define COUNTRY_CODE_SERBIA       (U32BIT)(('s' << 16) | ('r' << 8) | 'b')
#define COUNTRY_CODE_SLOVAKIA     (U32BIT)(('s' << 16) | ('v' << 8) | 'k')
#define COUNTRY_CODE_SLOVENIA     (U32BIT)(('s' << 16) | ('v' << 8) | 'n')
#define COUNTRY_CODE_SOUTHAFRICA  (U32BIT)(('z' << 16) | ('a' << 8) | 'f')
#define COUNTRY_CODE_SPAIN        (U32BIT)(('e' << 16) | ('s' << 8) | 'p')
#define COUNTRY_CODE_SWEDEN       (U32BIT)(('s' << 16) | ('w' << 8) | 'e')
#define COUNTRY_CODE_SWITZERLAND  (U32BIT)(('c' << 16) | ('h' << 8) | 'e')
#define COUNTRY_CODE_TAIWAN       (U32BIT)(('t' << 16) | ('w' << 8) | 'n')
#define COUNTRY_CODE_TAIWAN_NUM   (U32BIT)(('1' << 16) | ('5' << 8) | '8')
#define COUNTRY_CODE_UK           (U32BIT)(('g' << 16) | ('b' << 8) | 'r')
#define COUNTRY_CODE_UKRAINE      (U32BIT)(('u' << 16) | ('k' << 8) | 'r')
#define COUNTRY_CODE_VENEZUELA    (U32BIT)(('v' << 16) | ('e' << 8) | 'n')

#define COUNTRY_CODE_USERDEFINED  (U32BIT)(('u' << 24) | ('s' << 16) | ('e' << 8) | 'r')


// define language codes. (note these should all be lower-case - si handling converts received
// codes into lower-case when the tables are parsed)

#define ENGLISH_LANG_CODE              (('e' << 16) | ('n' << 8) | 'g')
#define WELSH1_LANG_CODE               (('w' << 16) | ('e' << 8) | 'l')
#define WELSH2_LANG_CODE               (('c' << 16) | ('y' << 8) | 'm')
#define GAELIC_LANG_CODE               (('g' << 16) | ('l' << 8) | 'a')
#define IRISH1_LANG_CODE               (('i' << 16) | ('r' << 8) | 'i')
#define IRISH2_LANG_CODE               (('g' << 16) | ('l' << 8) | 'e')
#define DUTCH1_LANG_CODE               (('n' << 16) | ('l' << 8) | 'd')
#define DUTCH2_LANG_CODE               (('d' << 16) | ('u' << 8) | 't')
#define GERMAN1_LANG_CODE              (('g' << 16) | ('e' << 8) | 'r')
#define GERMAN2_LANG_CODE              (('d' << 16) | ('e' << 8) | 'u')
#define FRENCH1_LANG_CODE              (('f' << 16) | ('r' << 8) | 'e')
#define FRENCH2_LANG_CODE              (('f' << 16) | ('r' << 8) | 'a')
#define RUSSIAN_LANG_CODE              (('r' << 16) | ('u' << 8) | 's')
#define SIMPLIFIED_CHINESE_LANG_CODE   (('c' << 16) | ('h' << 8) | 'i')
#define TRADITIONAL_CHINESE_LANG_CODE  (('c' << 16) | ('h' << 8) | 'i')
#define FINNISH_LANG_CODE              (('f' << 16) | ('i' << 8) | 'n')
#define SWEDISH_LANG_CODE              (('s' << 16) | ('w' << 8) | 'e')
#define NORWEGIAN_LANG_CODE            (('n' << 16) | ('o' << 8) | 'r')
#define DANISH_LANG_CODE               (('d' << 16) | ('a' << 8) | 'n')
#define MANDARIN_LANG_CODE             (('c' << 16) | ('m' << 8) | 'n')
#define CANTONESE_LANG_CODE            (('y' << 16) | ('u' << 8) | 'e')
#define MAORI1_LANG_CODE               (('m' << 16) | ('a' << 8) | 'o')
#define MAORI2_LANG_CODE               (('m' << 16) | ('r' << 8) | 'i')
#define JAPANESE_LANG_CODE             (('j' << 16) | ('p' << 8) | 'n')
#define ITALIAN_LANG_CODE              (('i' << 16) | ('t' << 8) | 'a')
#define SPANISH1_LANG_CODE             (('s' << 16) | ('p' << 8) | 'a')
#define SPANISH2_LANG_CODE             (('e' << 16) | ('s' << 8) | 'l')
#define KOREAN_LANG_CODE               (('k' << 16) | ('o' << 8) | 'r')
#define HINDI_LANG_CODE                (('h' << 16) | ('i' << 8) | 'n')
#define CZECH1_LANG_CODE               (('c' << 16) | ('z' << 8) | 'e')
#define CZECH2_LANG_CODE               (('c' << 16) | ('e' << 8) | 's')
#define SLOVAK1_LANG_CODE              (('s' << 16) | ('l' << 8) | 'k')
#define SLOVAK2_LANG_CODE              (('s' << 16) | ('l' << 8) | 'o')
#define AFRIKAANS_LANG_CODE            (('a' << 16) | ('f' << 8) | 'r')
#define NDEBELE_LANG_CODE              (('n' << 16) | ('b' << 8) | 'l')
#define NORTHERN_SOTHO_LANG_CODE       (('n' << 16) | ('s' << 8) | 'o')
#define SOUTHERN_SOTHO_LANG_CODE       (('s' << 16) | ('o' << 8) | 't')
#define SWATI_LANG_CODE                (('s' << 16) | ('s' << 8) | 'w')
#define TSONGA_LANG_CODE               (('t' << 16) | ('s' << 8) | 'o')
#define TSWANA_LANG_CODE               (('t' << 16) | ('s' << 8) | 'n')
#define VENDA_LANG_CODE                (('v' << 16) | ('e' << 8) | 'n')
#define XHOSA_LANG_CODE                (('x' << 16) | ('h' << 8) | 'o')
#define ZULU_LANG_CODE                 (('z' << 16) | ('u' << 8) | 'l')
#define CROATIAN_LANG_CODE             (('h' << 16) | ('r' << 8) | 'v')
#define HUNGARIAN_LANG_CODE            (('h' << 16) | ('u' << 8) | 'n')
#define LATVIAN_LANG_CODE              (('l' << 16) | ('a' << 8) | 'v')
#define LUXEMBOURGISH_LANG_CODE        (('l' << 16) | ('t' << 8) | 'z')
#define POLISH_LANG_CODE               (('p' << 16) | ('o' << 8) | 'l')
#define PORTUGUESE_LANG_CODE           (('p' << 16) | ('o' << 8) | 'r')
#define SERBIAN_LANG_CODE              (('s' << 16) | ('r' << 8) | 'p')
#define ROMANIAN1_LANG_CODE            (('r' << 16) | ('o' << 8) | 'n')
#define ROMANIAN2_LANG_CODE            (('r' << 16) | ('u' << 8) | 'm')
#define ALBANIAN1_LANG_CODE            (('s' << 16) | ('q' << 8) | 'i')
#define ALBANIAN2_LANG_CODE            (('a' << 16) | ('l' << 8) | 'b')
#define SLOVENE_LANG_CODE              (('s' << 16) | ('l' << 8) | 'v')
#define CATALAN_LANG_CODE              (('c' << 16) | ('a' << 8) | 't')
#define GALICIAN_LANG_CODE             (('g' << 16) | ('l' << 8) | 'g')
#define BASQUE1_LANG_CODE              (('e' << 16) | ('u' << 8) | 's')
#define BASQUE2_LANG_CODE              (('b' << 16) | ('a' << 8) | 'q')
#define UKRAINIAN_LANG_CODE            (('u' << 16) | ('k' << 8) | 'r')
#define UND_LANG_CODE                  (('u' << 16) | ('n' << 8) | 'd')
#define ARABIC_LANG_CODE               (('a' << 16) | ('r' << 8) | 'a')

//---Enumerations for public use-----------------------------------------------

/* Language IDs */
enum E_DVB_LANGUAGE_IDS
{
   DVB_LANGUAGE_ENGLISH,
   DVB_LANGUAGE_WELSH,
   DVB_LANGUAGE_GAELIC,
   DVB_LANGUAGE_IRISH,
   DVB_LANGUAGE_FRENCH,
   DVB_LANGUAGE_GERMAN,
   DVB_LANGUAGE_DUTCH,
   DVB_LANGUAGE_RUSSIAN,
   DVB_LANGUAGE_SIMPLIFIED_CHINESE,
   DVB_LANGUAGE_TRADITIONAL_CHINESE,
   DVB_LANGUAGE_FINNISH,
   DVB_LANGUAGE_SWEDISH,
   DVB_LANGUAGE_NORWEGIAN,
   DVB_LANGUAGE_DANISH,
   DVB_LANGUAGE_MANDARIN,
   DVB_LANGUAGE_CANTONESE,
   DVB_LANGUAGE_MAORI,
   DVB_LANGUAGE_JAPANESE,
   DVB_LANGUAGE_ITALIAN,
   DVB_LANGUAGE_SPANISH,
   DVB_LANGUAGE_KOREAN,
   DVB_LANGUAGE_HINDI,
   DVB_LANGUAGE_CZECH,
   DVB_LANGUAGE_SLOVAK,
   DVB_LANGUAGE_AFRIKAANS,
   DVB_LANGUAGE_NDEBELE,
   DVB_LANGUAGE_NORTHERN_SOTHO,
   DVB_LANGUAGE_SOUTHERN_SOTHO,
   DVB_LANGUAGE_SWATI,
   DVB_LANGUAGE_TSONGA,
   DVB_LANGUAGE_TSWANA,
   DVB_LANGUAGE_VENDA,
   DVB_LANGUAGE_XHOSA,
   DVB_LANGUAGE_ZULU,
   DVB_LANGUAGE_CROATIAN,
   DVB_LANGUAGE_HUNGARIAN,
   DVB_LANGUAGE_LATVIAN,
   DVB_LANGUAGE_LUXEMBOURGISH,
   DVB_LANGUAGE_POLISH,
   DVB_LANGUAGE_PORTUGUESE,
   DVB_LANGUAGE_SERBIAN,
   DVB_LANGUAGE_ROMANIAN,
   DVB_LANGUAGE_ALBANIAN,
   DVB_LANGUAGE_SLOVENE,
   DVB_LANGUAGE_CATALAN,
   DVB_LANGUAGE_GALICIAN,
   DVB_LANGUAGE_BASQUE,
   DVB_LANGUAGE_UKRAINIAN,
   DVB_LANGUAGE_ARABIC
};

/* Database lang definitions */
enum E_DB_LANGS
{
   ACFG_DB_LANG_ENGLISH,
   ACFG_DB_LANG_WELSH1,
   ACFG_DB_LANG_WELSH2,
   ACFG_DB_LANG_GAELIC,
   ACFG_DB_LANG_IRISH1,
   ACFG_DB_LANG_IRISH2,
   ACFG_DB_LANG_DUTCH1,
   ACFG_DB_LANG_DUTCH2,
   ACFG_DB_LANG_GERMAN1,
   ACFG_DB_LANG_GERMAN2,
   ACFG_DB_LANG_FRENCH1,
   ACFG_DB_LANG_FRENCH2,
   ACFG_DB_LANG_RUSSIAN,
   ACFG_DB_LANG_SIMPLIFIED_CHINESE,
   ACFG_DB_LANG_TRADITIONAL_CHINESE,
   ACFG_DB_LANG_FINNISH,
   ACFG_DB_LANG_SWEDISH,
   ACFG_DB_LANG_NORWEGIAN,
   ACFG_DB_LANG_DANISH,
   ACFG_DB_LANG_MANDARIN,
   ACFG_DB_LANG_CANTONESE,
   ACFG_DB_LANG_MAORI1,
   ACFG_DB_LANG_MAORI2,
   ACFG_DB_LANG_JAPANESE,
   ACFG_DB_LANG_ITALIAN,
   ACFG_DB_LANG_SPANISH1,
   ACFG_DB_LANG_SPANISH2,
   ACFG_DB_LANG_KOREAN,
   ACFG_DB_LANG_HINDI,
   ACFG_DB_LANG_CZECH1,
   ACFG_DB_LANG_CZECH2,
   ACFG_DB_LANG_SLOVAK1,
   ACFG_DB_LANG_SLOVAK2,
   ACFG_DB_LANG_AFRIKAANS,
   ACFG_DB_LANG_NDEBELE,
   ACFG_DB_LANG_NORTHERN_SOTHO,
   ACFG_DB_LANG_SOUTHERN_SOTHO,
   ACFG_DB_LANG_SWATI,
   ACFG_DB_LANG_TSONGA,
   ACFG_DB_LANG_TSWANA,
   ACFG_DB_LANG_VENDA,
   ACFG_DB_LANG_XHOSA,
   ACFG_DB_LANG_ZULU,
   ACFG_DB_LANG_CROATIAN,
   ACFG_DB_LANG_HUNGARIAN,
   ACFG_DB_LANG_LATVIAN,
   ACFG_DB_LANG_LUXEMBOURGISH,
   ACFG_DB_LANG_POLISH,
   ACFG_DB_LANG_PORTUGUESE,
   ACFG_DB_LANG_SERBIAN,
   ACFG_DB_LANG_ROMANIAN1,
   ACFG_DB_LANG_ROMANIAN2,
   ACFG_DB_LANG_ALBANIAN1,
   ACFG_DB_LANG_ALBANIAN2,
   ACFG_DB_LANG_SLOVENE,
   ACFG_DB_LANG_CATALAN,
   ACFG_DB_LANG_GALICIAN,
   ACFG_DB_LANG_BASQUE1,
   ACFG_DB_LANG_BASQUE2,
   ACFG_DB_LANG_UKRAINIAN,
   ACFG_DB_LANG_UND,
   ACFG_DB_LANG_ARABIC,
   ACFG_NUM_DB_LANGUAGES
};

/* Types of SI updates that are to be supported without requiring a scan to update the service database */
typedef enum
{
   ACFG_DYNAMIC_SI_UPDATE_NONE = 0,
   ACFG_DYNAMIC_SI_UPDATE_LCNS = 0x0001,             /* Dynamic reassigmment of LCNs */
   ACFG_DYNAMIC_SI_UPDATE_SERVICE_ADD = 0x0002,
   ACFG_DYNAMIC_SI_UPDATE_SERVICE_REMOVE = 0x0004,
   ACFG_DYNAMIC_SI_UPDATE_SERVICE_MOVE = 0x0008
} E_ACFG_DYNAMIC_SI_UPDATE_TYPE;


//---Global type defs for public use-------------------------------------------

typedef struct ana_rf_channel_data
{
   U8BIT *name;
   U32BIT freq_hz;
} ACFG_ANA_RF_CHANNEL_DATA;

typedef struct ter_rf_channel_data
{
   U8BIT *name;
   U32BIT freq_hz;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TMODE mode;
   E_STB_DP_TTYPE type;
} ACFG_TER_RF_CHANNEL_DATA;

typedef struct cab_rf_channel_data
{
   U8BIT *name;
   U32BIT freq_hz;
   E_STB_DP_CMODE mode;
   U16BIT symbol_rate;
} ACFG_CAB_RF_CHANNEL_DATA;

typedef struct
{
   U16BIT update_types;
   U16BIT original_network_id;
} ACFG_DYNAMIC_SI_UPDATES;

typedef BOOLEAN (*AllocLcnFunc)(E_STB_DP_SIGNAL_TYPE tuner_type);
typedef void (*DBTidyFunc)(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual);

typedef struct region_data_entry
{
   U8BIT *name;
   U8BIT code;
} ACFG_REGION_DATA;

typedef struct
{
   U16BIT language_id;
   U8BIT ids[3];
} ACFG_LANGUAGE_IDS;

typedef struct
{
   U8BIT table_id;
   STB_SI_USER_DEF_DESCRIP_FUNCTION func;
} ACFG_COUNTRY_SPECIFIC_DESCRIPTOR;


typedef struct
{
   U8BIT *country_name;
   U32BIT country_code;
   U32BIT private_data_specifier;
   U16BIT terr_first_unallocated_lcn;
   U16BIT terr_last_unallocated_lcn;
   U16BIT cab_first_unallocated_lcn;
   U16BIT cab_last_unallocated_lcn;
   U16BIT sat_first_unallocated_lcn;
   U16BIT sat_last_unallocated_lcn;
   AllocLcnFunc terr_lcn_func;
   AllocLcnFunc cab_lcn_func;
   AllocLcnFunc sat_lcn_func;
   DBTidyFunc terr_db_tidy_func;
   DBTidyFunc cab_db_tidy_func;
   DBTidyFunc sat_db_tidy_func;
   U16BIT start_watershed;
   U16BIT end_watershed;
   U8BIT num_regions;
   ACFG_REGION_DATA *region_data;
   U8BIT *event_content_types;
   U16BIT num_ana_rf_channels;
   ACFG_ANA_RF_CHANNEL_DATA *ana_rf_channel_table_ptr;
   U16BIT num_ter_rf_channels;
   ACFG_TER_RF_CHANNEL_DATA *ter_rf_channel_table_ptr;
   U16BIT num_cab_rf_channels;
   ACFG_CAB_RF_CHANNEL_DATA *cab_rf_channel_table_ptr;
   U16BIT min_sat_freq;
   U16BIT max_sat_freq;
   U16BIT sat_freq_inc;
   U8BIT num_symbol_rates;
   U16BIT sat_symbol_rates[8];
   BOOLEAN include_dvbs2;
   BOOLEAN aerial_power_option_reqd;
   BOOLEAN default_aerial_power;
   BOOLEAN aerial_tuning_screen_reqd;
   BOOLEAN nordig_compliant;
   BOOLEAN terr_nordig_service;
   BOOLEAN cab_nordig_service;
   BOOLEAN sat_nordig_service;
   U8BIT num_db_langs;
   ACFG_LANGUAGE_IDS *db_lang_data;
   U8BIT eit_searches_per_day;                  /* Number of times per day to perform an EIT search when in stand-by */
   U8BIT eit_update_time;                       /* Maximum time needed to receive all the events on a transport */
   ACFG_COUNTRY_SPECIFIC_DESCRIPTOR *si_descs;  /* Array of SI descriptors, terminated by an entry with table ID 0 */
   U8BIT default_secondary_language;            /* Fall back language ID where a secondary language isn't defined */
   ACFG_DYNAMIC_SI_UPDATES *terr_dynamic_si_updates;
   ACFG_DYNAMIC_SI_UPDATES *cab_dynamic_si_updates;
   ACFG_DYNAMIC_SI_UPDATES *sat_dynamic_si_updates;
} ACFG_COUNTRY_CONFIG;

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Returns the number of country configurations included in the DVB stack
 * @return  number of countries
 */
U8BIT ACFG_GetNumCountries(void);

/**
 * @brief   Returns the country code the DVB is configured for
 * @return  country code
 */
U32BIT ACFG_GetCountry(void);

/**
 * @brief   Returns an array containing the names of all the countries included
 *          in the DVB. The index into this array is referred to as the country_id.
 *          The returned array should be freed using ACFG_ReleaseCountryList.
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_countries_ptr pointer to the number of countries in the returned array
 */
void ACFG_GetCountryList(U8BIT ***str_array_ptr, U8BIT *num_countries_ptr);

/**
 * @brief   Frees a country list previously acquired using ACFG_GetCountryList
 * @param   str_array array to be freed
 * @param   num_countries number of items in the array
 */
void ACFG_ReleaseCountryList(U8BIT **str_array, U8BIT num_countries);

/**
 * @brief   Copies the country configuration for the given country into the provided structure.
 *          None of the included arrays should be freed.
 * @param   country_code 3 character country code
 * @param   config structure into which the configuration data will be copied
 * @return  TRUE if the country code is valid, FALSE otherwise
 */
BOOLEAN ACFG_GetCountryConfig(U32BIT country_code, ACFG_COUNTRY_CONFIG *config);

/**
 * @brief   Sets the country configuration settings that will be used for given country.
 *          This could be used to override the built-in configuration for a country but would
 *          more normally be used to set the configuration for the user defined country, if it's
 *          been included. If the config is set for the current country then ACFG_SetCountryIds
 *          will be called to ensure any country settings are changed if necessary.
 * @param   country_code 3 character country code
 * @param   config configuration settings to be used for the country
 * @return  TRUE if the country code is valid, FALSE otherwise
 */
BOOLEAN ACFG_SetCountryConfig(U32BIT country_code, ACFG_COUNTRY_CONFIG *config);

/**
 * @brief   Returns the number of regions in the given country
 * @param   country_code country code
 * @return  number of regions
 */
U8BIT ACFG_GetNumRegions(U32BIT country_code);

/**
 * @brief   Returns an array of pointers to strings containing the region names for
 *          the given country. The index into this array is the region_id.
 *          The array is allocated by this function and should be freed using
 *          ACFG_ReleaseRegionList.
 * @param   country_code country code
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_regions_ptr pointer to the number of regions in the returned array
 * @return  TRUE if the country_code is valid
 */
BOOLEAN ACFG_GetRegionList(U32BIT country_code, U8BIT ***str_array_ptr, U8BIT *num_regions_ptr);

/**
 * @brief   Returns the region code that identifies the given region
 * @param   country_code country code
 * @param   region_id region index
 * @param   code_ptr pointer to value in which the code will be returned
 * @return  TRUE if the country and region indices are valid, FALSE otherwise
 */
BOOLEAN ACFG_GetRegionCode(U32BIT country_code, U8BIT region_id, U8BIT *code_ptr);

/**
 * @brief   Frees the region array previously acquired using ACFG_GetRegionList
 * @param   str_array array to be freed
 * @param   num_regions number of items in the array
 */
void ACFG_ReleaseRegionList(U8BIT **str_array, U8BIT num_regions);

/**
 * @brief   Returns the index in the list of countries known by the DVB, of the given country
 * @param   country_code country code
 * @return  country index, value returned by ACFG_GetNumCountries() if the country isn't found
 */
U8BIT ACFG_GetCountryIndex(U32BIT country_code);

/**
 * @brief   Returns the internal ID of the region the DVB is configured for.
 * @return  region id
 */
U8BIT ACFG_GetRegionId(void);

/**
 * @brief   Sets the current country and sets default values for region and language
 * @param   country_code country code, as defined above (see COUNTRY_CODE_* defines)
 * @return  TRUE if country is known, FALSE otherwise
 */
BOOLEAN ACFG_SetCountry(U32BIT country_code);

/**
 * @brief   Returns the number of languages defined for the given country that
 *          can be used for audio and subtitles/teletext.
 * @param   country_code country code
 * @return  number of available languages
 */
U8BIT ACFG_GetNumDbLanguages(U32BIT country_code);

/**
 * @brief   Sets the languages that can be used for setting the default primary and secondary audio,
 *          subtitle and teletext languages. These will override any languages that are defined in
 *          the country configuration. The currently set primary and secondary audio language ids
 *          will be reset to 0 if they're greater than the number of languages in the new array.
 * @param   country_code country code of the country whose languages are to be set
 * @param   languages array of languages. This array will be copied.
 * @param   num_languages number of items in the above language array
 * @return  TRUE if the country is found and the languages are set, FALSE otherwise
 */
BOOLEAN ACFG_SetDbLanguages(U32BIT country_code, ACFG_LANGUAGE_IDS *languages, U8BIT num_languages);

/**
 * @brief   Returns an array of pointers to strings containing the available language names for
 *          the given country. The index into this array is the lang_id used when
 *          getting and setting audio and text language ids.
 *          The array is allocated by this function and should be freed using
 *          ACFG_ReleaseDbLangList.
 * @param   country_code country code
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_langs_ptr pointer to the number of languages in the returned array
 * @return  TRUE if the country_code is valid
 */
BOOLEAN ACFG_GetDbLangList(U32BIT country_code, U8BIT ***str_array_ptr, U8BIT *num_langs_ptr);

/**
 * @brief   Returns an array of language ids for a given country and index into the
 *          language array. A language id is one of the ACFG_DB_LANG_XXXX values above and there
 *          might be more than one for a given language. For example german language has two
 *          ids associated to it: ACFG_DB_LANG_GERMAN1 and ACFG_DB_LANG_GERMAN2 because german
 *          language can have two different codes, 'deu' and 'ger'. See also functions
 *          ACFG_ConvertLangIdToCode and ACFG_ConvertLangCodeToId. The returned pointer is an
 *          internal array and must not be freed.
 * @param   country_code country code
 * @param   lang_entry index into the language array
 * @return  pointer to the array of language ids, or NULL if either of the indices are invalid.
 */
U8BIT* ACFG_GetDbLangId(U32BIT country_code, U8BIT lang_entry);

/**
 * @brief   Frees the language array previously acquired using ACFG_GetDbLangList
 * @param   str_array array of lang names to be freed
 * @param   num_langs number of names in the array
 */
void ACFG_ReleaseDbLangList(U8BIT **str_array, U8BIT num_langs);

/**
 * @brief   Returns the language id for the given language code
 * @param   lang_code language code
 * @return  language id, or ACFG_INVALID_LANG if language code isn't found
 */
U8BIT ACFG_ConvertLangCodeToId(U32BIT lang_code);

/**
 * @brief   Returns the language code for the given language id
 * @param   lang_id language id
 * @return  language code, or 0 if language id isn't valid
 */
U32BIT ACFG_ConvertLangIdToCode(U8BIT lang_id);

/**
 * @brief   Saves the configured country and changes settings related to the country
 * @param   country_code country code
 * @param   region_id ID of region in country, if appropriate
 * @param   audio_lang_id ID of language to be used for audio
 * @param   sub_lang_id ID of language to be used for subtitles/teletext
 */
void ACFG_SetCountryIds(U32BIT country_code, U8BIT region_id, U8BIT audio_lang_id, U8BIT sub_lang_id);

/**
 * @brief   Returns the primary audio language ID
 * @return  language id
 */
U8BIT ACFG_GetPrimaryAudioLangId(void);

/**
 * @brief   Returns the secondary audio language ID
 * @return  language id
 */
U8BIT ACFG_GetSecondaryAudioLangId(void);

/**
 * @brief   Returns the primary DVB subtitle/teletext language ID
 * @return  language id
 */
U8BIT ACFG_GetPrimaryTextLangId(void);

/**
 * @brief   Returns the secondary DVB subtitle/teletext language ID
 * @return  language id
 */
U8BIT ACFG_GetSecondaryTextLangId(void);

/**
 * @brief   Sets the primary language id to be used for audio. This is the language
 *          that will be chosen first, if available
 * @param   lang_id language id
 */
void ACFG_SetPrimaryAudioLangId(U8BIT lang_id);

/**
 * @brief   Sets the secondary language id to be used for audio.
 * @param   lang_id language id
 */
void ACFG_SetSecondaryAudioLangId(U8BIT lang_id);

/**
 * @brief   Sets the primary language id to be used for teletext/subtitles.
 * @param   lang_id language id
 */
void ACFG_SetPrimaryTextLangId(U8BIT lang_id);

/**
 * @brief   Sets the secondary language id to be used for teletext/subtitles.
 * @param   lang_id language id
 */
void ACFG_SetSecondaryTextLangId(U8BIT lang_id);

/**
 * @brief   Returns the default secondary language for audio and subtitles as
 *          defined for the selected country. This is used for countries that
 *          don't use the secondary language, but expect a defined language to be
 *          used if the primary one isn't available.
 * @return  language id, or ACFG_INVALID_LANG when the value is not defined
 */
U8BIT ACFG_GetDefaultSecondaryLangId(void);

/**
 * @brief   Returns a pointer to the channel table for an analog tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetAnaRfChannelTable(U32BIT country_code, ACFG_ANA_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr);

/**
 * @brief   Returns a pointer to the channel table for a terrestrial tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetTerRfChannelTable(U32BIT country_code, ACFG_TER_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr);

/**
 * @brief   Returns a pointer to the channel table for a cable tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetCabRfChannelTable(U32BIT country_code, ACFG_CAB_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr);

/**
 * @brief   Replaces the cable tuning table for the current country with the one provided
 * @param   cable_channel_data pointer to the new table to point
 * @param   number_channels number of entries in the channel table
 */
void ACFG_SetCableChannelTable(ACFG_CAB_RF_CHANNEL_DATA *cable_channel_data, U16BIT number_channels);

/**
 * @brief   Returns the country code for the given country index
 * @param   country_id country index
 * @return  country code, or 0 if country id isn't valid
 */
U32BIT ACFG_GetCountryCode(U8BIT country_id);

/**
 * @brief   Returns the private data specifier value for the given country
 * @param   country_code coutry code
 * @return  private data specifier value; 0 if one isn't defined for the country
 *          or country code isn't valid
 */
U32BIT ACFG_GetPrivateDataSpecifier(U32BIT country_code);

/**
 * @brief   Returns the first LCN that should be used when assigning LCNs to services
 *          that don't appear in an LCN descriptor, or can't be assigned their desired LCN
 * @param   country_code country code
 * @param   tuner_type tuner type of the value to be returned
 * @return  LCN; 0 if country or tuner type aren't valid
 */
U16BIT ACFG_GetFirstUnallocatedLcn(U32BIT country_code, E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Returns the last LCN that should be used when assigning LCNs to services
 *          that don't appear in an LCN descriptor, or can't be assigned their desired LCN
 * @param   country_code country code
 * @param   tuner_type tuner type of the value to be returned
 * @return  LCN; 0 if country or tuner type aren't valid
 */
U16BIT ACFG_GetLastUnallocatedLcn(U32BIT country_code, E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Get the start and end times of the watershed hours for a country
 * @param   country_code country code
 * @param   start_time returns the start time in minutes since midnight
 * @param   end_time returns the end time in minutes since midnight
 * @return  TRUE if the start and end times are valid and have been returned, FALSE otherwise
 */
BOOLEAN ACFG_GetWatershedTimes(U32BIT country_code, U16BIT *start_time, U16BIT *end_time);

/**
 * @brief   Returns the minimum frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetMinSatelliteScanFreq(U32BIT country_code);

/**
 * @brief   Returns the maximum frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetMaxSatelliteScanFreq(U32BIT country_code);

/**
 * @brief   Returns the increment frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetSatelliteScanFreqInc(U32BIT country_code);

/**
 * @brief   Returns a fixed array of symbol rates to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  array of symbol rates in Kbps; NULL if undefined or country is invalid
 */
U16BIT* ACFG_GetSatelliteScanSymbolRates(U32BIT country_code);

/**
 * @brief   Returns the number of items in the fixed array of symbol rates to be
 *          used when performing a satellite based frequency scan in the given country
 * @param   country_code country code
 * @return  number of symbol rates; 0 if undefined or country is invalid
 */
U8BIT ACFG_GetSatelliteScanNumSymbolRates(U32BIT country_code);

/**
 * @brief   Returns whether DVB-S2 should be included when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  TRUE if DVB-S2 should be used, FALSE otherwise
 */
BOOLEAN ACFG_GetSatelliteScanDvbS2(U32BIT country_code);

/**
 * @brief   Returns whether the aerial power option is required for DVB-T/T2
 *          for the given country
 * @param   country_code country code
 * @return  TRUE if required, FALSE otherwise
 */
BOOLEAN ACFG_GetAerialPowerOptionReqd(U32BIT country_code);

/**
 * @brief   Returns the default aerial power option setting for DVB-T/T2 for
 *          the given country
 * @param   country_code country code
 * @return  TRUE or FALSE
 */
BOOLEAN ACFG_GetDefaultAerialPower(U32BIT country_code);

/**
 * @brief   Returns whether an aerial tuning screen should be presented by the interface
 *          before performing a DVB-T/T2 scan for services for the given country
 * @param   country_code country code
 * @return  TRUE if required, FALSE otherwise
 */
BOOLEAN ACFG_GetAerialTuningScreenReqd(U32BIT country_code);

/**
 * @brief   Returns the event content types for the given country. This defines how
 *          the content type value broadcast as part of the EIT is to be interpreted
 * @param   country_code country code
 * @return  pointer to the content type array. The index of this array is level_1 from the
 *          event content descriptors in the EIT and its elements are of type ADB_EVENT_CONTENT,
 *          defined in ap_dbacc.h
 */
U8BIT* ACFG_GetEventContentTypes(U32BIT country_code);

/**
 * @brief   Sets the start and end times during which background searches are allowed
 *          run when in standby
 * @param   start_time start time in minutes since midnight
 * @param   end_time end time in minutes
 */
void ACFG_SetBackgroundSearchTime(U16BIT start_time, U16BIT end_time);

/**
 * @brief   Gets the background start and end search times
 * @param   start_time return start time in minutes since midnight
 * @param   end_time return end time in minutes
 */
void ACFG_GetBackgroundSearchTime(U16BIT *start_time, U16BIT *end_time);

/**
 * @brief   Enables or disables the background service search when in standby
 * @param   enabled TRUE to enable, FALSE to disable
 */
void ACFG_SetBackgroundServiceSearch(BOOLEAN enabled);

/**
 * @brief   Returns whether the background service search is enabled or not
 * @return  TRUE if enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetBackgroundServiceSearch(void);

/**
 * @brief   Enables or disables the background SSU  search when in standby
 * @param   enabled TRUE to enable, FALSE to disable
 */
void ACFG_SetBackgroundSSUSearch(BOOLEAN enabled);

/**
 * @brief   Returns whether the background SSU search is enabled or not
 * @return  TRUE if enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetBackgroundSSUSearch(void);

/**
 * @brief   Returns whether the current country requires Nordig compliance for SI
 * @return  TRUE if Nordig, FALSE otherwise
 */
BOOLEAN ACFG_IsNordigCountry(void);

/**
 * @brief   Returns whether the broadcast SI data for the given tuner type is to be treated
 *          as Nordig compliant. This is in relation to the current country
 * @param   tuner_type tuner type to be checked; SIGNAL_NONE is not a valid argument
 * @return  TRUE if SI data should comply with Nordig specifications, FALSE otherwise
 */
BOOLEAN ACFG_IsNordigService(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Returns the number of minutes after which all the events should have
 *          been received during an EIT search. This is the EIT repetition time.
 * @return  time in minutes
 */
U8BIT ACFG_GetEitUpdateTime(void);

/**
 * @brief   Returns the number of EIT searches that should be performed per day
 *          when the box is in standby
 * @return  number of times
 */
U8BIT ACFG_GetEitSearchesPerDay(void);

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-T/T2 scan for the given country
 * @param   country_code country id
 * @return  pointer to function, or NULL
 */
AllocLcnFunc ACFG_GetTerrestrialLcnFunction(U32BIT country_code);

/**
 * @brief   Overrides an existing DVB-T/T2 LCN allocation function for the given country
 * @param   country_code country id
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetTerrestrialLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr);

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-C scan for the given country
 * @param   country_code country id
 * @return  pointer to function, or NULL
 */
AllocLcnFunc ACFG_GetCableLcnFunction(U32BIT country_code);

/**
 * @brief   Overrides an existing DVB-C LCN allocation function for the given country
 * @param   country_code country id
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetCableLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr);

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-S/S2 scan for the given country
 * @param   country_code country id
 * @return  pointer to function, or NULL
 */
AllocLcnFunc ACFG_GetSatelliteLcnFunction(U32BIT country_code);

/**
 * @brief   Overrides an existing DVB-S/S2 LCN allocation function for the given country
 * @param   country_code country id
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetSatelliteLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr);

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-T/T2 scan for the given country
 * @param   country_code country code
 * @return  pointer to function, or NULL
 */
DBTidyFunc ACFG_GetTerrestrialDBTidyFunction(U32BIT country_code);

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-C scan for the given country
 * @param   country_code country code
 * @return  pointer to function, or NULL
 */
DBTidyFunc ACFG_GetCableDBTidyFunction(U32BIT country_code);

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-S/S2 scan for the given country
 * @param   country_code country code
 * @return  pointer to function, or NULL
 */
DBTidyFunc ACFG_GetSatelliteDBTidyFunction(U32BIT country_code);

/**
 * @brief   Sets the language code to be used when the SI descriptors contain 'und', 'qaa' or don't
 *          contain any language descriptors.
 * @param   lang_code Language code to be use when the language is undefined. ILanguage codes that
 *          are not present in db_lang_list[] (defined in ap_cfdat.h), except ADB_LANG_CODE_UNDEF,
 *          will be ignored.
 */
void ACFG_SetUndefinedLanguageBehaviour(U32BIT lang_code);

/**
 * @brief   Gets the language code set by ACFG_SetUndefinedLanguageBehaviour, or 'und' if that
 *          function has not been called.
 * @return  Language code currently used when the SI descriptors don't define a language
 */
U32BIT ACFG_GetUndefinedLanguageBehaviour(void);

/**
 * @brief   Sets the host and country module for CI stack. Value zero for disable.
 * @param   module Module Id given by CI stack
 */
void ACFG_SetHostCountryLanguageModuleId(U32BIT module);

/**
 * @brief   Use to check whether a dynamic SI update is enabled for the currently configured
 *          country and given signal type and original network ID. If an array item is found with an
 *          original network ID of ADB_INVALID_DVB_ID then this item will be used.
 * @param   signal_type terrestrial, satellite or cable
 * @param   onet_id original network ID to be checked
 * @param   update_type see whether this type of update is enabled
 * @return  TRUE if the update is enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetDynamicSIUpdate(E_STB_DP_SIGNAL_TYPE signal_type, U16BIT onet_id,
   E_ACFG_DYNAMIC_SI_UPDATE_TYPE update_type);

/**
 * @brief   Sets the dynamic SI updates for the given tuner type of the currently configured country
 * @param   signal_type terrestrial, satellite or cable
 * @param   dynamic_updates array of SI update types; the content of this array isn't copied so must
 *                          continue to exist after it's set. The last element of the array must be an
 *                          entry with the update type of ACFG_DYNAMIC_SI_UPDATE_NONE; the original
 *                          network ID for this entry will be ignored so can be any value.
 * @param   num_updates number of SI update entries in the provided array
 * @return  TRUE if the country and signal type are valid, FALSE otherwise
 */
BOOLEAN ACFG_SetDynamicSIUpdates(E_STB_DP_SIGNAL_TYPE signal_type, ACFG_DYNAMIC_SI_UPDATES *dynamic_updates,
   U8BIT num_updates);

#endif //  _AP_CFG_H
