/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CI control functions
 * @file    ap_ci.h
 * @date    February 2011
 * @author  Ocean Blue
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error
#ifndef _AP_CI_H
#define _AP_CI_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------


//---Enumerations for public use-----------------------------------------------

/* Replies for starting an operator profile search */

typedef enum {
  CIP_START_OPERATOR_SEARCH_NO,
  CIP_START_OPERATOR_SEARCH_YES,
  CIP_START_OPERATOR_SEARCH_ASK
} E_CIP_START_OPERATOR_SEARCH;

typedef enum {
   CIP_NORMAL_TUNE,
   CIP_QUIET_TUNE,
   CIP_KEEP_APP_RUNING,
   CIP_QUIET_KEEP_APP
} E_CIP_TUNE_FLAGS;

//---Global type defs for public use-------------------------------------------

/**
 * @brief   Notify that DVB is about to tune to service or transport stream.
 *          This function would be called as a result of STB_CITuneBroadcastRequest(),
 *          STB_CITuneLCNRequest(), STB_CITuneTripletRequest(), or STB_CITuneIPRequest()
 *          but before tuning occurs.
 * @param   onet_id original network id
 * @param   trans_id transport id
 * @param   serv_id service id
 * @param   tune flags Presentation engine tuning flags (Quiet tune / Keep app running)
 * @return  n/a
 */
typedef void (*F_NotifyStartTuning)(U32BIT module, U16BIT onet_id, U16BIT trans_id, U16BIT serv_id, E_CIP_TUNE_FLAGS flags);

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Return CAM upgrade option (Yes/No/Ask)
 * @return  U8BIT the cams current upgrade mode
 */
U8BIT ACI_GetCamUpgradeMode(void);

/**
 * @brief   Sets the CAM upgrade option (Yes/No/Ask)
 * @param   upgrade_mode cam upgrade mode
 */
void ACI_SetCamUpgradeMode(U8BIT upgrade_mode);

/**
 * @brief   Return the option for starting an operator profile search (yes/no/ask)
 * @return  The current setting
 */
E_CIP_START_OPERATOR_SEARCH ACI_GetOperatorSearchMode(void);

/**
 * @brief   Sets the operator profile search mode (Yes/No/Ask). This should be used
 *          by the app to control whether a search can be started, or the user
 *          should be asked first, or can't run.
 * @param   search_mode search mode
 */
void ACI_SetOperatorSearchMode(E_CIP_START_OPERATOR_SEARCH search_mode);

/**
 * @brief   Sets whether an operator profile search needs to be run
 * @param   module module requesting or cancelling an operator search
 * @param   required TRUE if a search is required, FALSE otherwise
 */
void ACI_OperatorSearchRequired(U32BIT module, BOOLEAN required);

/**
 * @brief   Sets the date/time that an operator search should be started for the given module
 * @param   cicam_id ID of the CICAM requesting an operator search
 * @param   date UTC MJD date code
 * @param   hour UTC hour
 * @param   min UTC minute
 */
void ACI_ScheduleOperatorSearch(U32BIT cicam_id, U16BIT date, U8BIT hour, U8BIT min);

/**
 * @brief   Returns whether an operator profile search has been requested
 * @return  TRUE if a search has been requested, FALSE otherwise
 */
BOOLEAN ACI_IsOperatorSearchRequired(void);

/**
 * @brief   Returns the module performing or requiring and operator search
 * @return  module
 */
U32BIT ACI_GetOperatorSearchModule(void);

/**
 * @brief   Checks all the CI+ profiles to see if any have requested an update search
 * @param   module pointer to return the handle of the first module requiring an update
 *          Only valid if the function returns TRUE
 * @return  TRUE if a profile needs updating, FALSE otherwise
 */
BOOLEAN ACI_GetFirstOperatorSearchModule(U32BIT *module);

/**
 * @brief   Checks all the CI+ profiles to find the one with the earliest scheduled search.
 *          All returned values are only valid if the function returns TRUE.
 * @param   module pointer to return the handle of the first module requiring an update
 * @param   date pointer to return the date code of the search
 * @param   hours pointer to return the hour of the search
 * @param   mins pointer to return the minute of the search
 * @return  TRUE if a scheduled search is found for a profile, FALSE otherwise
 */
BOOLEAN ACI_GetFirstScheduledOperatorSearch(U32BIT *module, U16BIT *date, U8BIT *hours, U8BIT *mins);

/**
 * @brief   Called by the app to start an operator profile search for the given module
 * @param   module operator module
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACI_StartOperatorSearchForModule(U32BIT module);

/**
 * @brief   Acquires a CI slot for the given path on the given service after
 *          releasing any slot already being used by the path
 * @param   path decode path
 * @param   s_ptr service to be used on the path
 * @return  TRUE if a slot is acquired, FALSE otherwise
 */
BOOLEAN ACI_AcquireCISlot(U8BIT path, void *s_ptr);

/**
 * @brief   Acquires a CI slot for the recording path. This function may need to "steal" the CI
 *          slot from the live path (if they are not the same path).
 * @param   path decode path for recording
 * @param   s_ptr service to be used on the path
 */
BOOLEAN ACI_AcquireCISlotForRecording(U8BIT path, void* s_ptr);

/**
 * @brief   Looks for a CAM that's able to descramble the given service and returns its slot id
 * @param   serv_ptr service
 * @return  slot id, if CAM is found, or INVALID_RES_ID if not
 */
U8BIT ACI_FindCISlotForService(void *serv_ptr);

/**
 * @brief   Checks if a pin has been saved for the CAM in the given slot and returns it
 * @param   slot_id CI slot
 * @return  pin value, or -1 if there's no CAM in the slot or no pin has been saved
 */
S32BIT ACI_ReadPinForSlot(U8BIT slot_id);

/**
 * @brief   Saves the given pin associated with the CAM in the given slot.
 *          If the pin for this CAM has previously been saved then the saved value will be updated.
 *          If all entries are used then the oldest will be overwritten
 * @param   slot_id CI slot
 * @param   pin pin value to be saved
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN ACI_WritePinForSlot(U8BIT slot_id, S32BIT pin);

/**
 * @brief   Converts the pin value to ASCII and sends it to the CAM
 *          An STB_EVENT_CI_PIN_STATUS event will be sent to notify
 *          the validity, or otherwise, of the pin.
 * @param   slot_id - slot Id
 * @param   pin - pin value
 * @return  TRUE if the pin was sent, FALSE otherwise
 */
BOOLEAN ACI_SendPinToCam(U8BIT slot_id, U32BIT pin);

/**
 * @brief   Register callback function to notify start of tuning process due to CI
 *          tune request
 * @param   cb_func Callback function - NULL disables the call.
 */
void ACI_RegisterStartTuneNotifyCallback(F_NotifyStartTuning cb_func);

#endif
