/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2010 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   CI plus support functions
 * @file    ap_ci_support.h
 * @date    5th January 2014
 * @author  Adam Sturtridge
 */

#ifndef AP_CI_SUPPORT_H
#define AP_CI_SUPPORT_H
#ifdef COMMON_INTERFACE

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "stbci.h"
#include "ap_ci.h"

/*---Constant and macro definitions for public use-----------------------------*/

/*---Enumerations for public use-----------------------------------------------*/

/*---Global type defs for public use-------------------------------------------*/

/*---Global Function prototypes for public use---------------------------------*/

/**
 * @brief    Handle Tune request from the CAM
 * @param    module Host control module
 * @param    nid network ID
 * @param    onid original network ID
 * @param    tsid transport stream ID
 * @param    sid service ID
 */
void ACI_HcTune(U32BIT module, U16BIT nid, U16BIT onid, U16BIT tsid, U16BIT sid);

/**
 * @brief   This function is used by the CI stack to notify the host that a session
 *          with the host countrol resource has been opened by the module.
 *          This signals to the host that the module intends to request that the
 *          host tunes to other transport streams / services.
 *          Following this notification, the host must call the function
 *          ACI_HcAskRelease to request that the module closes the session.
 *          The function ACI_HcAskReleaseReply will be called to let the host know
 *          whether the control over the tuner has been released or not.
 * @param   module host control module
 *
 */
void ACI_HcNotifyHostControlSession(U32BIT module);

/**
 * @brief   This function is used by the CI stack to notify the host that a session
 *          with the host control resource has been closed by the module.
 *          This signals to the host that it has controls over the tuner and it
 *          does not need to call STB_CIAskRelease before using it.
 *          Following this call, any PIDs that have been replaced by STB_CIReplace
 *          should be restored, if possible.
 * @param   module - host control module
 */
void ACI_HcNotifyHostControlSessionClosed(U32BIT module);

/**
 * @brief   This function is called by the CI+ stack to send the reply of a
 *          release request to the host.
 *          This function is called in response to a call to STB_CIAskRelease.
 *          The host must not assume that it has control over the tuner until
 *          this function is called.
 * @param   module specifies module required
 * @param   release_reply the reply to the release request
 */
void ACI_HcAskReleaseReply(U32BIT module, U8BIT release_reply);

/**
 * @brief   This function is used by the CI+ stack to request that the host tunes
 *          to a transport stream using the given parameters.
 *          If the service_id is zero, the host should not report a PMT using
 *          STB_CIUpdatePmt().
 *          If the PMT is provided, it should be used when selecting elementary
 *          streams on the service. The PMT is not provided if service_id is zero.
 *          Following the tune operation (whether successful or not), the host
 *          must call the function CIP_TuneReply.
 * @param   module host control module
 * @param   service_id the service to tune to
 * @param   desc_loop_len length of descriptor loop in bytes
 * @param   desc_loop the descriptor loop
 * @param   pmt the PMT to use (or NULL)
 * @param   flags flags to pass to presentation engine
 */
void ACI_HcTuneBroadcastRequest(U32BIT module, U16BIT service_id,
   U16BIT desc_loop_len, U8BIT *desc_loop, U8BIT *pmt, E_CIP_TUNE_FLAGS flags);

/**
 * @brief   Handle tune to service request from the CAM
 * @param   module Host control module
 * @param   s_ptr service
 * @param   flags flags to pass to presentation engine
 */
void ACI_HcTuneService(U32BIT module, void *s_ptr, E_CIP_TUNE_FLAGS flags);

/**
 * @brief   This function is called by the CI+ stack to deliver the operator
 *          status to the host.
 *          This should be called if the CICAM supports the Operator Profile.
 *          This provides the module to be used for other Operator Profile
 *          related calls.
 * @param   module operator profile module
 * @param   status operator status
 */
void ACI_OpNotifyOperatorStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status);

/**
 * @brief   This function is called by the CI+ stack to deliver the operator
 *          information to the host.
 * @param   module operator profile module
 * @param   info_version operator information version
 * @param   info operator information or NULL (if not valid)
 */
void ACI_OpNotifyOperatorInfo(U32BIT module, U8BIT info_version,
   S_STB_CI_OPERATOR_INFO *info);

/**
 * @brief   This function is called by the CI+ stack to request that the host
 *          performs the tune request as described in the provided descriptors.
 *          The descriptor loop contains system delivery descriptors which the
 *          host must try in order until either an error condition is raised,
 *          the tune operation is successful or the descriptor loop is exhausted.
 *          The operation of the host should follow the pseudo code in section 14.7.5.11.
 *          When the tune attempt is done, the host must call the function
 *          ACI_OpSendOperatorTuneStatus with the results of the tune attempt.
 *          This function may be called during a profile search sequence.
 * @param   module operator profile module
 * @param   desc_loop_len length of descriptor loop in bytes
 * @param   desc_loop the descriptor loop
 */
void ACI_OpNotifyOperatorTune(U32BIT module, U16BIT desc_loop_len, U8BIT *desc_loop);

/**
 * @brief   This function is called by the CI+ stack to deliver the operator search
 *          status to the host.
 *          This function will be called at the end of a profile search sequence.
 * @param   module operator profile module
 * @param   status operator status
 */
void ACI_OpNotifyOperatorSearchStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status);

/**
 * @brief   This function is called by the CI+ stack to deliver the operator NIT
 *          to the host.
 *          The CICAM delivers zero or more NIT sections to the host. Each section
 *          is a complete NIT section including the CRC-32 field, and the sections
 *          are provided without any padding between them.
 *          The NIT section data must be copied if it used after this function returns
 * @param   module - operator profile module
 * @param   nit_loop_length - length of NIT loop in bytes (may be 0)
 * @param   nit_sections - NIT section(s), if any
 */
void ACI_OpNotifyOperatorNit(U32BIT module, U16BIT nit_loop_length,
   U8BIT *nit_sections);

#endif /*COMMON_INTERFACE*/
#endif /* AP_CI_SUPPORT_H */

/******************************************************************************
** End of file
******************************************************************************/
