/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application stb layer control
 * @file    ap_cntrl.h
 * @date    18/03/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _AP_CNTRL_H
#define _AP_CNTRL_H

#include "techtype.h"
#include "ap_common.h"
#include "stbhwav.h"
#include "vtctype.h"
#include "stbdpc.h"

#ifdef INCLUDE_OTA_SSU
#include "stbota.h"
#endif

#include "ap_si.h"

//---Constant and macro definitions for public use-----------------------------


//---Enumerations for public use-----------------------------------------------

typedef enum
{
   ACTL_SI_NO_SEARCH,
   ACTL_SI_CHANNEL_SEARCH,
   ACTL_SI_CHANNEL_SEARCH_NO_NIT,
   ACTL_SI_STARTUP_SEARCH,
   ACTL_SI_EVENT_PF_SEARCH,
   ACTL_SI_EVENT_SCHED_SEARCH,
   ACTL_SI_EVENT_PF_SCHED_SEARCH,
   ACTL_SI_DVB_SSU_SEARCH,
   ACTL_SI_TOT_SEARCH,
   ACTL_SI_USER_DEFINED
} E_ACTL_SI_SRCH_REQD;

typedef enum
{
   ACTL_FREQ_SEARCH,
   ACTL_NETWORK_SEARCH
} E_ACTL_SEARCH_TYPE;

typedef enum
{
   ACTL_EVENT_SEARCH_PF,
   ACTL_EVENT_SEARCH_SCHED,
   ACTL_EVENT_SEARCH_PF_SCHED
} E_ACTL_EVENT_SEARCH;

typedef enum
{
   ACTL_STANDBY_MODE,
   ACTL_STANDBY_SCART_MODE,
   ACTL_TV_MODE,
   ACTL_SCART_MODE,
   ACTL_INT_DVD_MODE
} E_ACTL_AV_MODE;

typedef enum
{
   ACTL_NO_CHANGE,
   ACTL_AUDIO_CHANGE,
   ACTL_ADESC_CHANGE,
   ACTL_BOTH_CHANGED
} E_ACTL_DECODE_CHANGE;


//---Global type defs for public use-------------------------------------------
typedef struct
{
   S8BIT offset;
   E_STB_DP_ANALOG_VIDEO_TYPE vtype;
} S_MANUAL_ANA_TUNING_PARAMS;

typedef struct
{
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TMODE mode;
   E_STB_DP_TTYPE type;
   U8BIT plp_id;
} S_MANUAL_TERR_TUNING_PARAMS;

typedef struct
{
   E_STB_DP_CMODE mode;
   U16BIT symbol_rate;
   E_STB_DP_TBWIDTH bwidth;
} S_MANUAL_CABLE_TUNING_PARAMS;

typedef struct
{
   void *satellite;
   E_STB_DP_POLARITY polarity;
   U16BIT symbol_rate;
   E_STB_DP_FEC fec;
   BOOLEAN dvb_s2;
   E_STB_DP_MODULATION modulation;
} S_MANUAL_SAT_TUNING_PARAMS;

typedef struct
{
   U32BIT freq;
   union
   {
      S_MANUAL_ANA_TUNING_PARAMS ana;
      S_MANUAL_TERR_TUNING_PARAMS terr;
      S_MANUAL_CABLE_TUNING_PARAMS cab;
      S_MANUAL_SAT_TUNING_PARAMS sat;
   } u;
   U32BIT end_freq;
} S_MANUAL_TUNING_PARAMS;


typedef struct
{
   E_STB_DP_RES_OWNER owner;
   void *data;
   U32BIT data_size;
} S_ACTL_OWNER_INFO;

// =====================================================
#define ACTL_MAXNUM_COMPONENT     5
#define ACTL_MAXLEN_LOCKMESS      40

// SCDCAComponent equivalent
typedef struct {
    U16BIT   pid;
    U16BIT   ecm_pid;
    U8BIT    type;
    U8BIT    reserved[3];
}S_ACTL_COMPONENT;

// SCDCALockService equivalent
typedef struct {
    U32BIT   frequency;
    U32BIT   symbol_rate;
    U16BIT   pcr_pid;
    U8BIT    modulation;
    U8BIT    componentNum;
    S_ACTL_COMPONENT components[ACTL_MAXNUM_COMPONENT];
    U8BIT    fec_outer;
    U8BIT    fec_inner;
    char     beforeInfo[ACTL_MAXLEN_LOCKMESS+1];
    char     quitInfo[ACTL_MAXLEN_LOCKMESS+1];
    char     endInfo[ACTL_MAXLEN_LOCKMESS+1];
}S_ACTL_SERVICE_PARAMS;
// =====================================================


//---Global type defs for public use-------------------------------------------

/**
 * @brief   Prototype for function that's called when a file requested from a
 *          DSM-CC carousel is received.
 * @param   user_data user defined data passed when the file was requested
 * @param   file_data data
 * @param   data_size number of bytes of data
 */
typedef void (*DSM_FILE_CALLBACK)(void *user_data, U8BIT *file_data, U32BIT data_size);

/**
 * @brief   Function pointer definition:
 *          Determines subtitle PID and whether it is DVB or teletext, for the given service
 * @param   serv_ptr service
 * @param   subt_info pointer to ADB_SUBT_INFO for extra info on subtitles
 * @return  PID to collect subtitles data, zero if none available
 */
typedef U16BIT (*F_ServiceGetSubtitlePid)(void *serv_ptr, ADB_SUBT_INFO *subt_info);

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Entry point for starting a service search for a full retune or to update
 *          the existing service lineup.
 * @param   tuner_type tuner type that defines the search to be performed
 * @param   type type of service search to be performed
 * @return  TRUE if search starts successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartServiceSearch(E_STB_DP_SIGNAL_TYPE tuner_type, E_ACTL_SEARCH_TYPE type);

/**
 * @brief   When the search has completed, this function should be called to see
 *          whether the target region UI should be presented. This is only needed
 *          for DVB-T/T2 searches.
 * @return  TRUE if target region UI is required, FALSE otherwise
 */
BOOLEAN ACTL_IsTargetRegionRequired(void);

/**
 * @brief   Function to stop the service search before it completes
 */
void ACTL_StopServiceSearch(void);

/**
 * @brief   Function to start the startup search when booting from cold. This
 *          search checks the validity of the services contained in the database.
 *          It's only valid for DVB-T/T2 and DVB-C.
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartStartupSearch(void);

#ifdef INCLUDE_OTA_SSU
/**
 * @brief   Starts looking for an OTA update on the given transport. If no decode path is provided,
 *          or it is and it isn't tuned or is tuned to a different transport, then it will be tuned
 *          to the given transport before it starts looking for an OTA update.
 * @param   path decode path to be used to tune to the given transport. If INVALID_RES_ID is used
 *               then the live path will be used or a decode path will be acquired
 * @param   transport transport to tune to to check whether an OTA is available
 * @param   serv_id ID of the service on teh transport that contains the OTA
 * @param   version_callback callback function that will be called if an OTA is found that should
 *                           check whether the update is for this system and the version is later
 *                           than the current version
 * @return  TRUE if the search is started, which may mean a tune request has been made, FALSE otherwise
 */
BOOLEAN ACTL_StartOTAUpdate(U8BIT path, void *transport, U16BIT serv_id,
   F_SSU_VERSION_CALLBACK version_callback);

/**
 * @brief   Once an OTA update has been found or not then this function must be called
 *          to continue or terminate the update.
 * @param   do_update TRUE if the update is to be downloaded and installed, FALSE otherwise
 */
void ACTL_ContinueOTAUpdate(BOOLEAN do_update);

/**
 * @brief   Stops an OTA update
 */
void ACTL_StopOTAUpdate(void);

/**
 * @brief   Function to start a search to see if there's an SSU available
 * @param   tuner_type restrict SSU search to transports of the given type;
 *                     SIGNAL_NONE means all available transports will be used.
 * @param   version_callback callback function that will be called if an OTA is found that should
 *                           check whether the update is for this system and the version is later
 *                           than the current version
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartSSUSearch(E_STB_DP_SIGNAL_TYPE tuner_type, F_SSU_VERSION_CALLBACK version_callback);

#endif /* INCLUDE_OTA_SSU */

/**
 * @brief   Function to prepare a SSU data download
 *
 * @return  TRUE if the download is prepared, FALSE otherwise
 */
BOOLEAN ACTL_PrepareSSUDownload(void);

/**
 * @brief   Function to start a SSU data download
 *
 * @return  TRUE if the download is started, FALSE otherwise
 */
BOOLEAN ACTL_StartSSUDownload(U8BIT path);

/**
 * @brief   Function to get the SSU data download progress
 *
 * @return  Download progress as a percentage [0 - 100]
 */
U8BIT ACTL_GetSSUDownloadProgress(void);

/**
 * @brief   Function to stop a SSU data download
 *
 * @return  TRUE if the download is stoped, FALSE otherwise
 */
BOOLEAN ACTL_StopSSUDownload(void);

/**
 * @brief   Function to get a SSU update information
 *
 * @return  JSON string
 */
const U8BIT * ACTL_GetSSUInfo(void);

/**
 * @brief   Function to start a TOT search which is required to set the system clock
 *          when starting from power off if a real-time clock isn't available.
 * @param   tuner_type restrict TOT search to transports of the given type;
 *                     SIGNAL_NONE means all available transports will be used.
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartTotSearch(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Function to stop the TOT search before it completes
 */
void ACTL_StopTotSearch(void);

/**
 * @brief   Entry point for starting an EIT search
 * @param   tuner_type limit the search to transports of this tuner type;
 *                     pass as SIGNAL_NONE for all transports
 * @param   search_type event search type to be performed, p/f only, sched only, or both
 * @return  TRUE if search starts successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartEitSearch(E_STB_DP_SIGNAL_TYPE tuner_type, E_ACTL_EVENT_SEARCH search_type);

/**
 * @brief   Function to stop the EIT search before it completes
 */
void ACTL_StopEitSearch(void);

/**
 * @brief   Returns TRUE if current search has finished.
 *          This works for any of the available types of searches.
 * @return  TRUE if search has finished, FALSE otherwise
 */
BOOLEAN ACTL_IsSearchComplete(void);

/**
 * @brief   Returns search progress as a percentage.
 *          This works for any of the available types of searches, but the
 *          scope for providing progress on anything other than the service
 *          searches is fairly limited.
 * @return  Percent complete, 0 if not started
 */
U8BIT ACTL_GetSearchProgress(void);

/**
 * @brief   Returns path used buy currently running search.
 *          This works for any of the available types of searches, but the
 *          scope for providing path on anything other than the service
 *          searches is fairly limited.
 * @return  Search path or INVALID_RES_ID if no search is currently running
 */
U8BIT ACTL_GetServiceSearchPath(void);


/**
 * @brief   Start a service search on, or just tune to, a transport, using chan_id
 *          as an index into the country's tuning table.
 * @param   tuner_type tuner type to be used for the search
 * @param   chan_id index into the country's channel array to tune to
 * @param   start_search TRUE if a service search is to be started, otherwise just tune
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN ACTL_StartManualSearchById(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT chan_id,
   BOOLEAN start_search);

/**
 * @brief   Start a service search on, or just tune to, the transport defined
 *          by the given tuning parameters
 * @param   tuner_type tuner type to be used for the search
 * @param   tuning_params user defined tuning parameters
 * @param   type type of service search to be performed
 * @return  TRUE if the search is started successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartManualSearch(E_STB_DP_SIGNAL_TYPE tuner_type,
   S_MANUAL_TUNING_PARAMS *tuning_params, E_ACTL_SEARCH_TYPE type);

/**
 * @brief   Function to be called when a manual service search has completed,
 *          or is being stopped.
 */
void ACTL_FinishManualSearch(void);

/**
 * @brief   Start the signal detection, or just tune to, the transport defined
 *          by the given tuning parameters
 * @param   tuner_type tuner type to be used for the search
 * @param   tuning_params user defined tuning parameters
 * @return  TRUE successful, FALSE otherwise
 */
BOOLEAN ACTL_StartSignalDetection(E_STB_DP_SIGNAL_TYPE tuner_type, S_MANUAL_TUNING_PARAMS *tuning_params);

/**
 * @brief   Function to be called to stop the signal detection.
 */
void ACTL_StopSignalDetection(void);

/**
 * @brief   Acquires a decode path suitable for tuning to the given transport
 * @param   t_ptr the transport the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be acquired with the path, FALSE otherwise
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the acquired path, should be NULL for live TV (used by CI+)
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT ACTL_AcquirePathForTransport(void *t_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info);

/**
 * @brief   Acquires a decode path suitable for tuning to the given service
 * @param   s_ptr the service the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be acquired with the path, FALSE otherwise
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the acquired path, should be NULL for live TV (used by CI+)
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT ACTL_AcquirePathForService(void *s_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner);

/**
 * @brief   Finds the path being used to view the given service and releases it
 * @param   s_ptr service pointer
 * @param   owner owner requesting the path to be released
 */
void ACTL_ReleaseLivePathForService(void *s_ptr, E_STB_DP_RES_OWNER owner);

/**
 * @brief   Releases the decode path and all resources no longer needed.
 *          The path may not be released if the path is owned by a module
 * @param   path decode path
 * @return  TRUE if the path is released, FALSE otherwise
 */
BOOLEAN ACTL_ReleasePath(U8BIT path);

/**
 * @brief   Attempts to take ownership of the given path (used by CI+)
 * @param   path decode path to be owned
 * @param   owner_info owner info for the path
 * @return  TRUE if the ownership is set, FALSE otherwise
 */
BOOLEAN ACTL_AcquirePathOwnership(U8BIT path, S_ACTL_OWNER_INFO *owner_info);

/**
 * @brief   Releases ownership of the path, and frees any associated data,
 *          if the given owner is the path's owner (used by CI+)
 * @param   path decode path
 * @param   owner owner releasing ownership
 * @return  TRUE if the ownership is released or the path isn't owned, FALSE otherwise
 */
BOOLEAN ACTL_ReleasePathOwnership(U8BIT path, E_STB_DP_RES_OWNER owner);

/**
 * @brief   Checks whether there's a tuner available to view the given service.
 *          This takes into account whether tuners are being used for recording
 *          or have been acquired by CI+.
 * @param   s_ptr service being checked
 * @return  TRUE if service can be viewed, FALSE otherwise
 */
BOOLEAN ACTL_CanServiceBeViewed(void *s_ptr);

/**
 * @brief   Returns the current satellite being used by the given decode path
 * @param   path decode path
 * @return  Pointer to satellite
 */
void* ACTL_GetCurrentSatellite(U8BIT path);

/**
 * @brief   Tunes to the given transport and sets the type of SI monitoring that
 *          will be started when the tuning completes.
 * @param   path decode path, use INVALID_RES_ID to allow the DVB to acquire an appropriate path
 * @param   owner_info owner info for the path, can be NULL (used by CI+)
 * @param   t_ptr transport to tune to
 * @param   reqd_si type of SI monitoring to be started when tuned
 * @param   relock_on TRUE if the tuner is to attempt to relock if the signal is lost
 * @return  decode path that is being used to tune to the transport.
 *          Returning INVALID_RES_ID means the path couldn't be acquired immediately and
 *          doesn't mean that it has failed
 */
U8BIT ACTL_TuneToTransport(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *t_ptr,
   E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on);

/**
 * @brief   Starts the process of tuning to the specified service. If the service is to be
 *          tuned on the live path, then path should be passed as INVALID_RES_ID to ensure
 *          that any CI+ resource handling is performed.
 * @param   path decode path, INVALID_RES_ID for the live path
 * @param   owner_info owner module requesting the tune, can be NULL (used by CI+)
 * @param   s_ptr required service
 * @param   override_lock TRUE if parental lock settings are to be ignored.
 * @param   for_live FALSE for recording, TRUE otherwise
 * @return  decode path that is being used to tune to the service.
 *          Returning INVALID_RES_ID means the path couldn't be acquired immediately and
 *          doesn't mean that it has failed
 */
U8BIT ACTL_TuneToService(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *s_ptr,
   BOOLEAN override_lock, BOOLEAN for_live);

/**
 * @brief   Starts the process of tuning to a given transport or service that's
 *          defined by the given delivery system descriptor. If the DSD defines
 *          a transport that doesn't currently exist, then one will be created,
 *          and if a service_id is given and a service with this ID doesn't exist
 *          on the transport, then one will be created, but will be marked as hidden
 *          and non-selectable so it won't appear to a user.
 * @param   path decode path to be tuned, must be valid
 * @param   dsd_type delivery system descriptor type (T, C or S)
 * @param   dsd delivery system descriptor
 * @param   service_id pass as 0 if just tuning to a transport
 * @param   reqd_si SI processing mode that should be selected when tuned.
 *          If tuning to a service the normal UPDATE mode will be used
 *          and this setting will be ignored.
 * @return  TRUE if the tune operation is started, FALSE otherwise
 */
BOOLEAN ACTL_TuneUsingDSD(U8BIT path, SI_DELIVERY_SYS_DESC_TYPE dsd_type,
   SI_DELIVERY_SYS_DESC *dsd, U16BIT service_id, E_ACTL_SI_SRCH_REQD reqd_si);

/**
 * @brief   Returns whether the given path is currently tuned
 * @param   path decode path
 * @return  TRUE if tuned, FALSE otherwise
 */
BOOLEAN ACTL_IsTuned(U8BIT path);

/**
 * @brief   Stops tuning on the given path
 * @param   path decode path
 */
void ACTL_TuneOff(U8BIT path);

/**
 * @brief   Stops decoding on the given path
 * @param   path decode path
 */
void ACTL_DecodeOff(U8BIT path);

/**
 * @brief   Stops and restarts audio decoding on the live path. This may be required
 *          due to a change in language preferences, or some other audio setting.
 */
void ACTL_ReTuneAudio(void);

/**
 * @brief   Stops and restarts subtitle decoding on the live path. This may be required
 *          due to a change in language preferences, or some other setting.
 */
void ACTL_ReTuneSubtitles(void);

/**
 * @brief   Returns whether decoding has been started on the given path
 * @param   path decode path
 * @return  TRUE if decoding has started, FALSE otherwise
 */
BOOLEAN ACTL_HasDecodingStarted(U8BIT path);

/**
 * @brief   Returns whether decoding is locked, due to parental locking, on the given path
 * @param   path decode path
 * @return  TRUE if decoding is locked, FALSE otherwise
 */
BOOLEAN ACTL_IsDecodingLocked(U8BIT path);

/**
 * @brief   Releases the lock on a channel after decoding has been blocked due
 *          to the service being parental locked, after which decoding will be started.
 */
void ACTL_ReleaseChannelLock(void);

/**
 * @brief   Tunes to the specified rf channel array entry for DVB-T and DVB-C
 * @param   path decode path
 * @param   id index in the current channel table (for the current signal type and country)
 * @param   reqd_si the type of SI handling required
 * @param   relock_on if FALSE auto relock will be turned off before tuning, otherwise it will be on
 * @return  TRUE if id valid, FALSE otherwise
 */
BOOLEAN ACTL_TuneToRfChanArrayEntry(U8BIT path, U16BIT id, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on);

/**
 * @brief   Tunes to the specified rf channel array entry in analogue mode
 * @param   path decode path
 * @brief   id index in the analogue channel table for the current country
 * @return  TRUE if id valid, FALSE otherwise
 */
BOOLEAN ACTL_TuneToRfChanArrayAnalogEntry(U8BIT path, U16BIT id);

/**
 * @brief   Tunes to the given set of tuning parameters
 * @param   path decode path
 * @param   tuning_params tuning parameters to be used
 * @param   reqd_si SI search mode to be used if tuning is successful
 * @param   relock_on defines whether the tuner is to attempt to relock if lock is lost
 * @return  TRUE if tuning is started, FALSE otherwise
 */
BOOLEAN ACTL_TuneToUserDefinedParams(U8BIT path, S_MANUAL_TUNING_PARAMS *tuning_params,
   E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on);

/**
 * @brief   Sets a grace period during which a system will enter active standby before going
 *          into passive or low power standby. This allows a system to be brought back out of
 *          standby quickly for a period of time.
 * @param   num_seconds grace period in seconds
 */
void ACTL_SetStandbyGracePeriod(U16BIT num_seconds);

/**
 * @brief   Puts DVB into standby mode. It will continue to monitor SI for recordings,
 *          SSU updates, etc., unless it goes into low power standby
 */
void ACTL_EnterStandby(void);

/**
 * @brief   Brings the DVB out of standby mode
 * @param   s_ptr service to be tuned to for live viewing. If NULL then the last
 *          service viewed will be restored
 * @param   tune_to_service if s_ptr is NULL then this boolean is used to define
 *          whether to tune to the last service viewed
 */
void ACTL_LeaveStandby(void *s_ptr, BOOLEAN tune_to_service);

/**
 * @brief   Reports the standby state to the A/V output controller
 * @param   state TRUE for standby is on, FALSE for standby is off
 */
void ACTL_SetStandbyState(BOOLEAN state);

/**
 * @brief   Reports standby state to the a/v output control state machine
 */
void ACTL_SetStandbyVCRActive(void);

/**
 * @brief   Sets the video window to the size specified. Coordinates are relative
 *          to the screen resolution
 * @param   win_x window X position
 * @param   win_y window Y position
 * @param   win_width window width (0 for full screen)
 * @param   win_height window height (0 for full screen)
 */
void ACTL_SetVideoWindow(S16BIT win_x, S16BIT win_y, U16BIT win_width, U16BIT win_height);

/**
 * @brief   Used to set the TV aspect ratio
 * @param   aspect_ratio TV aspect ratio, e.g. 4:3
 */
void ACTL_SetTVAspectRatio(E_STB_AV_ASPECT_RATIO aspect_ratio);

/**
 * @brief   Returns the current TV aspect ratio
 * @return  aspect ratio
 */
E_STB_AV_ASPECT_RATIO ACTL_GetTVAspectRatio(void);

/**
 * @brief   Used to set the TV aspect mode which defines how the video will be
 *          displayed based on the aspect ratio of the TV and video, along with
 *          some other factors.
 * @param   aspect_mode TV aspect mode, e.g. letterbox
 */
void ACTL_SetTVAspectMode(E_STB_AV_ASPECT_MODE aspect_mode);

/**
 * @brief   Returns the current TV aspect mode
 * @return  aspect mode
 */
E_STB_AV_ASPECT_MODE ACTL_GetTVAspectMode(void);

/**
 * @brief   Used to set the aspect conversion applied to the video based on the
 *          TV aspect ratio and aspect mode to be applied. The given settings will
 *          also be saved.
 * @param   aspect_ratio TV aspect ratio, e.g. 4:3
 * @param   aspect_mode TV aspect mode, e.g. letterbox
 */
void ACTL_SetTVAspectConversion(E_STB_AV_ASPECT_RATIO aspect_ratio, E_STB_AV_ASPECT_MODE aspect_mode);

/**
 * @brief   Reads the saved video format and returns the best mode available if it's
 *          set to AUTO or is invalid for the currently connected TV.
 * @param   width returns the new video width if saved mode isn't valid, can be NULL
 * @param   height returns the new video height if saved mode isn't valid, can be NULL
 * @return  current video mode, or best video mode available
 */
E_STB_AV_VIDEO_FORMAT ACTL_GetActualVideoMode(U16BIT *width, U16BIT *height);

/**
 * @brief   Checks that the selected HDMI resolution mode is supported and, if not,
 *          chooses the best one available. This function is called by the event task when
 *          the HDMI connected event is received.
 */
void ACTL_HDMIConnected(void);

/**
 * @brief   Sets flag to indicate HDMI is now disconnected.
 *          This function is called by the event task when the HDMI disconnected event is received.
 */
void ACTL_HDMIDisconnected(void);

/**
 * @brief   Returns whether the HDMI is connected or not.
 * @return  TRUE if the HDMI is connected, FALSE otherwise
 */
BOOLEAN ACTL_IsHDMIConnected(void);

/**
 * @brief   Returns an array of valid HDMI resolutions and the index of the
 *          currently selected format. The final entry in each array will be
 *          the auto video format.
 * @param   video_formats pointer to an array of video formats.
 * @param   current_index used to return index of currently selected video
 *          format
 * @return  Size of returned arrays, will always be at least 1
 */
U16BIT ACTL_GetHDMIResolutions(E_STB_AV_VIDEO_FORMAT **video_formats, U16BIT *current_index);

/**
 * @brief   Update video mode sets Voyager resolution as well as platform
 *          Also, stops subtitles during the operation.
 *          If HDMI resolution mode is not supported, it chooses the best
 *          one available.
 * @param   aspect TV aspect ratio, e.g. 4:3 or 16:9
 * @param   force forces resolution update when TRUE
 */
void ACTL_UpdateVideoMode(E_STB_AV_ASPECT_RATIO aspect, BOOLEAN force);

/**
 * @brief   Checks content protection requirements for the current event on the
 *          given service to determine whether HDCP has to be used. The HDMI output
 *          will be enabled/disabled accordingly.
 * @param   s_ptr service
 */
void ACTL_ApplyHDCP(void *s_ptr);

/**
 * @brief   Returns av_mode
 * @return  Scart AV status
 */
E_ACTL_AV_MODE ACTL_GetAvModeStatus(void);

/* RF channel details utility functions */

/**
 * @brief   Returns the number of entries in the rf channel table
 * @param   tuner_type tuner type
 * @return  number of entries in the channel table for the present country
 */
U16BIT ACTL_GetNumRfChanArrayEntries(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Returns a pointer to the channel name
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  pointer to the const name string
 */
U8BIT* ACTL_GetRfChanName(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id);

/**
 * @brief   Returns the rf name appropriate to the frequency specified
 * @param   tuner_type tuner type
 * @param   freq_hz frequency in hz
 * @return  pointer to the const name string, or NULL if not found
 */
U8BIT* ACTL_GetRfNameFromFreq(E_STB_DP_SIGNAL_TYPE tuner_type, U32BIT freq_hz);

/**
 * @brief   Returns a pointer to the channel name
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  pointer to the const name string
 */
U32BIT ACTL_GetRfChanFreqHz(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id);

/**
 * @brief   Returns the channel symbol rate
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  symbol rate of selected id
 */
U16BIT ACTL_GetRfSymbolRate(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id);

/**
 * @brief   Returns the modulation mode
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  Modulation mode (E_STB_DP_TMODE or E_STB_DP_CMODE depending on the tuner type
 */
U8BIT ACTL_GetRfModulation(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id);

/**
 * @brief   Returns the terrestrial type (T or T2) for the given channel id
 * @param   id index in the terrestrial channel table for the current country
 * @param   terrestrial type for the channel
 */
E_STB_DP_TTYPE ACTL_GetTerRfChanType(U16BIT id);

/**
 * @brief   Returns the current signal strength
 * @param   path decode path
 * @return  the signal strength as percentage of maximum (0-100)
 */
U8BIT ACTL_GetSignalStrength(U8BIT path);

/**
 * @brief   Returns the Bit Error Rate
 * @param   path decode path
 * @return  bit_error_ratio (BER) in 1.0E-7(i.e. 10^-7) unit
 */
U32BIT ACTL_GetSignalBitErrorRate(U8BIT path);

/**
 * @brief   Returns the Signal Noise Ratio
 * @param   path decode path
 * @return  the unit of snr is 0.1dB
 */
U32BIT ACTL_GetSignalNoiseRatio(U8BIT path);

#ifdef COMMON_INTERFACE
/* CI functions */

/**
 * @brief   Starts the process of tuning to the specified service provided by CI+.
 *          The same rules apply as when tuning to a normal service using ACTL_TuneToService.
 * @param   path decode path, INVALID_RES_ID for the live path
 * @param   owner_info owner module requesting the tune, can be NULL
 * @param   s_ptr required service
 * @param   si_mode SI search mode to be started on successful tuning
 */
void ACTL_TuneToCIService(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *s_ptr, E_APP_SI_MODE si_mode);

/**
 * @brief   Called when a reply is received from the CI+ stack in response to the request
 *          for it to release control of the live path and tuner. If it has released the
 *          path, then tuning can continue as requested.
 * @param   path Path associated with the response
 * @param   continue_tuning TRUE if tuning should continue, FALSE otherwise
 */
void ACTL_ContinueCIPLUSTune(U8BIT path, BOOLEAN continue_tuning);
#endif

/**
 * @brief   Enables CI module
 */
void ACTL_EnableCiModule(void);

/**
 * @brief   Disables CI module
 */
void ACTL_DisableCiModule(void);

/**
 * @brief   Returns state of ci_ui_required flag
 * @return  TRUE if CI OPEN event has been received
 */
BOOLEAN ACTL_IsCiUiRequired(void);

/**
 * @brief   Gets user paused flag value
 * @brief   path decode path
 * @return  TRUE if the subtitles are in paused mode, FALSE otherwise
 */
BOOLEAN ACTL_GetDecodePausedState(U8BIT path);

/**
 * @brief   Starts subtitle processing and display if the current service has
 *          valid subtitle data. If DVB subtitles aren't available, teletext will
 *          be used if available.
 * @return  TRUE if started, FALSE otherwise
 */
BOOLEAN ACTL_StartSubtitles(void);

/**
 * @brief   Disables the display of subtitles but processing continues
 */
void ACTL_PauseSubtitles(void);

/**
 * @brief   Resumes the display of subtitles after they've been paused
 */
void ACTL_ResumeSubtitles(void);

/**
 * @brief   Stops subtitles from being displayed and processed
 */
void ACTL_StopSubtitles(void);

/**
 * @brief   Stops subtitles being displayed if another OSD requires them to be
            hidden, such as CI+
 * @return  TRUE if suppressed, FALSE otherwise
 */
BOOLEAN ACTL_SuppressSubtitles(BOOLEAN suppress);

/**
 * @brief   Returns whether subtitles are being displayed
 */
BOOLEAN ACTL_AreSubtitlesDisplayed(void);

/**
 * @brief   Returns whether subtitles have been started, even if they aren't being displayed
 */
BOOLEAN ACTL_AreSubtitlesStarted(void);

/**
 * @brief   Returns whether audio description has been turned on
 * @return  TRUE if AD is on
 */
BOOLEAN ACTL_IsAudioDescriptionOn(void);

/**
 * @brief   Starts decoding an audio description stream, if available, on the given path.
 *          If AD isn't currently available it will be marked as enabled and started
 *          when it becomes available.
 * @param   path decode path
 * @return  TRUE if AD is turned on, FALSE otherwise
 */
BOOLEAN ACTL_StartAudioDescription(U8BIT path);

/**
 * @brief   Stops AD decoding
 * @param   path decode path
 */
void ACTL_StopAudioDescription(U8BIT path);

/**
 * @brief   Sets the AD volume
 * @param   volume AD volume as a percentage
 */
void ACTL_SetADVolume(U8BIT volume);

/**
 * @brief   Enables or disables parental control. This enables or disables
 *          locking on a per channel basis. If an age has previously been set,
 *          this will override the setting, effectively turning off age related
 *          blockign of channels.
 * @param   enabled TRUE if parental control is to be enabled
 */
void ACTL_SetParentalControl(BOOLEAN enabled);

/**
 * @brief   Returns whether parental control is enabled. This will also return
 *          TRUE if parental control has been set to a valid age.
 * @return  TRUE if parental control is enabled, FALSE otherwise
 */
BOOLEAN ACTL_ParentalControlEnabled(void);

/**
 * @brief   Sets the age (valid values 4-18) at which parental control will be
 *          will be applied. If the age is invalid, no change will be made to
 *          the current setting.
 * @param   age DVB SI age to be set
 */
void ACTL_SetParentalControlAge(U8BIT age);

/**
 * @brief   Returns the current age set for parental control. 0 will be returned
 *          if parental control is disabled or no age is set.
 * @return  age in the range 4-18, or 0
 */
U8BIT ACTL_GetParentalControlAge(void);

/**
 * @brief   Checks the parental control for the current event on the
 *          given service to determine whether decoding should be locked
 * @param   path decode path
 * @param   s_ptr service
 */
void ACTL_ApplyParentalControl(U8BIT path, void *s_ptr);

#ifdef FREEVIEW_PLAYBACK
/**
 * @brief   Returns whether trailer booking is available based on whether it's enabled
 *          and the number of links currently available on the given service
 * @param   serv_ptr service on which trailer booking is being queried
 * @return  TRUE if trailer booking is available, FALSE otherwise
 */
BOOLEAN ACTL_IsTrailerBookingAvailable(void *serv_ptr);
#endif

/**
 * @brief   Starts the SI to acquire the PMT and fill the ip service in
 */
void ACTL_SetupPlayback(void);

/**
 * @brief   Start streaming the specified service
 * @param   s_ptr the pointer to the service structure containing all the information
 */
void ACTL_StartPlayback(void *s_ptr);

/**
 * @brief   Returns the active path (live, playback, etc...), i.e. the one using the decoders
 * @return  active path
 */
U8BIT ACTL_GetActivePath(void);

/**
 * @brief   Returns the search path
 * @return  search path
 */
U8BIT ACTL_GetSearchPath(void);

/**
 * @brief   Sets up and starts decoding for the given service.
 *          This is used for PVR playback, but could also be used to start decoding
 *          for other non-broadcast services.
 * @param   path decode path
 * @param   s_ptr service
 */
void ACTL_StartDecoding(U8BIT path, void *s_ptr);

/**
 * @brief   Sets the MHEG5 audio volume adjust to be applied
 * @param   control TRUE if MHEG has control of Audio/Video
 */
void ACTL_SetMhegAVControl(BOOLEAN control);

/**
 * @brief   Sets function retrieve Subtitle PID and info. Allows external MHEG5 module to
 *          override the default function ADB_ServiceGetSubtitlePid.
 * @param   func Function to get subtitle info, NULL if use default
 */
void ACTL_SetMhegSubtitlePidFunc(F_ServiceGetSubtitlePid func);

/**
 * @brief   Sets the MHEG5 audio volume adjust to be applied
 * @param   scaling the volume scaling from -100% to 100%
 */
void ACTL_SetVolumeAdjustment(S8BIT scaling);

/**
 * @brief   Returns the current audio volume
 * @return  The current volume on a scale of 0-100.
 */
U8BIT ACTL_GetVolume(void);

/**
 * @brief   Sets the main audio volume and returns the new volume
 * @param   volume unsigned value giving volume to be applied
 * @return  The new volume on a scale of 0-100.
 */
U8BIT ACTL_SetVolume(U8BIT volume);

/**
 * @brief   Changes the main audio volume and returns the new volume
 * @param   volume_change signed value giving change in volume to be applied
 * @return  The new volume on a scale of 0-100.
 */
U8BIT ACTL_ChangeVolume(S8BIT volume_change);

/**
 * @brief   Sets the audio mute state
 * @param   mute TRUE to set mute, FALSE otherwise
 */
void ACTL_SetMute(BOOLEAN mute);

/**
 * @brief   Toggles the current mute state and returns the new mute setting
 * @return  The new mute setting, TRUE = muted
 */
BOOLEAN ACTL_ToggleMute(void);

/**
 * @brief   Returns the muted state of the audio
 * @return  TRUE if muted, FALSE otherwise
 */
BOOLEAN ACTL_IsMuted(void);

/**
 * @brief   Sets the current profile. (CI+ only)
 *          If the profile is being set to CI+ profile and the necessary CAM isn't
 *          present, then setting the profile will fail.
 * @param   profile profile to be set
 * @return  TRUE if the profile is set successfully, FALSE otherwise
 */
BOOLEAN ACTL_SetActiveProfile(void *profile);


/**
 * @brief   Setup default audio decoding params on current tuned service
 *          Can be used by external module (e.g. MHEG5)
 * @param   path decode path to be owned
 * @param   s_ptr service pointer
 * @return  E_ACTL_DECODE_CHANGE - indicate whether been change of decode params
 */
E_ACTL_DECODE_CHANGE ACTL_SetupAudioDecoding(U8BIT path, void *s_ptr);

/**
 * @brief   Switch the live-path for fast-channel-change
 * @param   fromPath the origin live-path
 * @param   toPath the new live-path
 * @return  TRUE if FCC switch successfully, FALSE otherwise
 */
BOOLEAN ACTL_SwitchLivePath(U8BIT fromPath, U8BIT toPath);


//---Function prototypes for internal use---------------------------------

/**
 * @brief   Control system initialisation
 */
void ACTL_InitialiseAppControl(void);

/**
 * @brief   Actions external events
 * @param   event the event to be handled
 */
void ACTL_ActionEvent(U32BIT event, void *event_data);

/**
 * @brief   Checks whether the service tuned to on the live path has changed
 *          and informs the UI if it has.
 * @return  TRUE if the service has changed, FALSE otherwise
 */
BOOLEAN ACTL_CheckLiveServiceChange(void);

/**
 * @brief   Handles all the private timer events
 * @param   timer_handle timer handle
 * @return  TRUE if the timer event has been handled, FALSE otherwise
 */
BOOLEAN ACTL_HandlePrivateTimerEvent(U32BIT timer_handle);

void ACTL_SetAnalogChanIdString(U8BIT *str_ptr);

/**
 * @brief   Enables or disables analog video display
 * @param   allow_analog_video if TRUE analog video is enabled, otherwise disabled
 */
void ACTL_AllowAnalogVideo(BOOLEAN allow_analog_video);

/**
 * @brief   set up the locked service for the next tuning
 * @param   the parameters of the locked service
 */
void ACTL_SetLockedService(S_ACTL_SERVICE_PARAMS * serviceParams);

/**
 * @brief   get the locked service
 * @param   none
 */
void * ACTL_GetLockedService(void);

#endif //  _AP_CNTRL_H
