/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application database access functions
 * @file    ap_dbacc.h
 * @date    24/03/2003
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _AP_DBACC_H
#define _AP_DBACC_H

#include "techtype.h"
#include "ap_common.h"
#include "stbhwtun.h"
#include "stbdpc.h"     // required for E_STB_DP_TMODE and E_STB_DP_BWIDTH
#include "stbsitab.h"   /* Required for E_ICON_COORD_SYSTEM */
#include "stbgc.h"

//---Constant and macro definitions for public use-----------------------------

#define ADB_INVALID_DVB_ID                ((U16BIT)0xffff)

#define LINK_TYPE_CA_REPLACEMENT_SERVICE  0x03
#define LINK_TYPE_SERVICE_REPLACEMENT     0x05

// number of service names (default name + preferred names)
#define ADB_NUM_SERV_NAME_IDS    6

//maximum number of favgroups that can be handled by the database
#define ADB_MAX_FAVGROUPS        8

/* LNB freq bands */
#define LNB_C_BAND_FREQUENCY              5150

#define LNB_KU_BAND_FREQUENCY_MINIMUM     9750
#define LNB_KU_BAND_FREQUENCY_MAXIMUM     12000

#define LNB_KU_BAND_FREQUENCY_LOW         LNB_KU_BAND_FREQUENCY_MINIMUM
#define LNB_KU_BAND_FREQUENCY_HIGH        10600

//---Enumerations for public use-----------------------------------------------
typedef enum
{
   ADB_TRANSPORT_LIST_ALL = 0xff,
   ADB_TRANSPORT_LIST_NOT_TUNED = 0x01,
   ADB_TRANSPORT_LIST_TUNED = 0x02
} ADB_TRANSPORT_LIST_TYPE;

typedef enum
{
   ADB_SERVICE_LIST_UNKNOWN = 0x0000,
   ADB_SERVICE_LIST_TV = 0x0001,               // these definitions can be ORed together to make
   ADB_SERVICE_LIST_RADIO = 0x0002,            // combinations, e.g.
   ADB_SERVICE_LIST_DATA = 0x0004,             // ADB_SERVICE_LIST_TV | ADB_SERVICE_LIST_DATA
   ADB_SERVICE_LIST_DIGITAL = 0x0007,
   ADB_SERVICE_LIST_FREESAT = 0x0040,         // Flag to say that MUST be valid freesat services
   ADB_SERVICE_LIST_ANALOG = 0x0080,
   ADB_SERVICE_LIST_TV_DATA = 0x0085,
   ADB_SERVICE_LIST_FAV_GROUP_A = 0x0100,
   ADB_SERVICE_LIST_FAV_GROUP_B = 0x0200,
   ADB_SERVICE_LIST_FAV_GROUP_C = 0x0400,
   ADB_SERVICE_LIST_FAV_GROUP_D = 0x0800,
   ADB_SERVICE_LIST_ALL = (ADB_SERVICE_LIST_DIGITAL | ADB_SERVICE_LIST_ANALOG),
   ADB_SERVICE_LIST_FAV_LIST = 0x8000
} ADB_SERVICE_LIST_TYPE;

/* Macro to define an ADB_SERVICE_LIST_TYPE for a favourite list */
#define ADB_LIST_TYPE_FROM_FAVLIST(id)  (((id) << 16) | ADB_SERVICE_LIST_FAV_LIST)
#define ADB_FAVLIST_FROM_LIST_TYPE(L)  (U8BIT)((L) >> 16)

typedef enum /* These values come from the 8bit service type field in the service descriptor */
{
   ADB_SERVICE_TYPE_TV = 0x01,
   ADB_SERVICE_TYPE_RADIO = 0x02,
   ADB_SERVICE_TYPE_TELETEXT = 0x03,
   ADB_SERVICE_TYPE_NVOD_REF = 0x04,
   ADB_SERVICE_TYPE_NVOD_TIMESHIFT = 0x05,
   ADB_SERVICE_TYPE_MOSAIC = 0x06,
   ADB_SERVICE_TYPE_AVC_RADIO = 0x0a,
   ADB_SERVICE_TYPE_AVC_MOSAIC = 0x0b,
   ADB_SERVICE_TYPE_DATA = 0x0c,
   ADB_SERVICE_TYPE_MPEG2_HD = 0x11,
   ADB_SERVICE_TYPE_AVC_SD_TV = 0x16,
   ADB_SERVICE_TYPE_AVC_SD_NVOD_TIMESHIFT = 0x17,
   ADB_SERVICE_TYPE_AVC_SD_NVOD_REF = 0x18,
   ADB_SERVICE_TYPE_HD_TV = 0x19,
   ADB_SERVICE_TYPE_AVC_HD_NVOD_TIMESHIFT = 0x1a,
   ADB_SERVICE_TYPE_AVC_HD_NVOD_REF = 0x1b,
   ADB_SERVICE_TYPE_UHD_TV = 0x1f,
   /* Values beloow are chosen so that they do not conflict with broadcast values */
   ADB_SERVICE_TYPE_ANALOG = 0x100,
   ADB_SERVICE_TYPE_DSD = 0x101,
   ADB_SERVICE_TYPE_VIRTUAL = 0x102
} ADB_SERVICE_TYPE;

typedef enum
{
   /* Video stream types, from least to most preferred */
   ADB_VIDEO_STREAM,
   ADB_H264_VIDEO_STREAM,
   ADB_H265_VIDEO_STREAM,

   /* Audio stream types */
   ADB_AUDIO_STREAM,
   ADB_AAC_AUDIO_STREAM,
   ADB_HEAAC_AUDIO_STREAM,
   ADB_HEAACv2_AUDIO_STREAM,
   ADB_AC3_AUDIO_STREAM,
   ADB_EAC3_AUDIO_STREAM,

   /* Other stream types */
   ADB_SUBTITLE_STREAM,
   ADB_DATA_STREAM,
   ADB_TTEXT_STREAM
} ADB_STREAM_TYPE;

typedef enum
{
   ADB_STREAM_LIST_ALL = 0xff,
   ADB_VIDEO_LIST_STREAM = 0x01,
   ADB_AUDIO_LIST_STREAM = 0x02,
   ADB_SUBTITLE_LIST_STREAM = 0x04,
   ADB_DATA_LIST_STREAM = 0x08,
   ADB_TTEXT_LIST_STREAM = 0x10,
   ADB_TTEXT_SUBT_LIST_STREAM = 0x20
} ADB_STREAM_LIST_TYPE;


typedef enum
{
   ADB_AUDIO_TYPE_UNDEFINED = 0,
   ADB_AUDIO_TYPE_NORMAL = 0,       /* This name is deprecated and was incorrectly defined as 1
                                     * but is retained for backwards compatibility */
   ADB_AUDIO_TYPE_CLEAN_EFFECTS = 1,
   ADB_AUDIO_TYPE_FOR_HEARING_IMPAIRED = 2,
   ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED = 3
} ADB_AUDIO_TYPE;


typedef enum
{
   ADB_SUBTITLE_TYPE_DVB = 0x10,
   ADB_SUBTITLE_TYPE_DVB_4_3 = 0x11,
   ADB_SUBTITLE_TYPE_DVB_16_9 = 0x12,
   ADB_SUBTITLE_TYPE_DVB_221_1 = 0x13,
   ADB_SUBTITLE_TYPE_DVB_HD = 0x14,
   ADB_SUBTITLE_TYPE_DVB_HARD_HEARING = 0x20,
   ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3 = 0x21,
   ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9 = 0x22,
   ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_221_1 = 0x23,
   ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_HD = 0x24
} ADB_SUBTITLE_TYPE;

typedef enum
{
   ADB_TELETEXT_TYPE_INITIAL = 0x01,
   ADB_TELETEXT_TYPE_SUBTITLE = 0x02,
   ADB_TELETEXT_TYPE_ADDITIONAL_INFO = 0x03,
   ADB_TELETEXT_TYPE_SCHEDULE = 0x04,
   ADB_TELETEXT_TYPE_SUBTITLE_HARD_HEARING = 0x05
} ADB_TELETEXT_TYPE;

typedef enum
{
   ADB_EVENT_CONTENT_UNCLASSIFIED = 0x00,
   ADB_EVENT_CONTENT_MOVIE = 0x10,
   ADB_EVENT_CONTENT_NEWS = 0x20,
   ADB_EVENT_CONTENT_ENTERTAINMENT = 0x30,
   ADB_EVENT_CONTENT_SPORT = 0x40,
   ADB_EVENT_CONTENT_CHILD = 0x50,
   ADB_EVENT_CONTENT_MUSIC = 0x60,
   ADB_EVENT_CONTENT_ARTS = 0x70,
   ADB_EVENT_CONTENT_SOCIAL = 0x80,
   ADB_EVENT_CONTENT_EDUCATION = 0x90,
   ADB_EVENT_CONTENT_LEISURE = 0xA0,
   ADB_EVENT_CONTENT_SPECIAL = 0xB0,
   ADB_EVENT_CONTENT_RESERVED1 = 0xC0,
   ADB_EVENT_CONTENT_RESERVED2 = 0xD0,
   ADB_EVENT_CONTENT_RESERVED3 = 0xE0,
   ADB_EVENT_CONTENT_USERDEFINED = 0xF0,
} ADB_EVENT_CONTENT;

typedef enum
{
   FAV_GROUP_A = 0x01,
   FAV_GROUP_B = 0x02,
   FAV_GROUP_C = 0x04,
   FAV_GROUP_D = 0x08,
   FAV_GROUP_ALL = 0x0f
} ADB_FAV_GROUP;

typedef enum
{
   ADB_PROFILE_TYPE_BROADCAST,
   ADB_PROFILE_TYPE_CIPLUS
} ADB_PROFILE_TYPE;

//---Global type defs for public use-------------------------------------------

typedef struct
{
   E_STB_DP_LNB_TYPE type;
   E_STB_DP_LNB_POWER power;
   E_STB_DP_DISEQC_TONE diseqc_tone;
   E_STB_DP_DISEQC_CSWITCH c_switch;
   BOOLEAN is_22k;
   BOOLEAN is_12v;
   BOOLEAN is_pulse_posn;
   BOOLEAN is_diseqc_posn;
   BOOLEAN is_smatv;
   U8BIT diseqc_repeats;      // 0 to 3 inclusive
   U8BIT u_switch;            // 0 to 16 inclusive
   U8BIT unicable_chan;
   U32BIT unicable_if;
   U8BIT *name;
} ADB_LNB_SETTINGS;

/**
 * @brief   Structure representing the component information as found in the EIT component_descriptor
 */
typedef struct
{
   U8BIT stream_content;
   U8BIT component_type;
   U8BIT component_tag;
   U8BIT language_code[3];
} ADB_EVENT_COMPONENT_INFO;

typedef struct
{
  U8BIT *item_description;
  U8BIT *item;
} ADB_EVENT_ITEMIZED_INFO;

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Clears the service database and resets everything to defaults values
 */
void ADB_ResetDatabase(void);

/**
 * @brief   Saves the database to non-volatile memory
 */
void ADB_SaveDatabase(void);

/**
 * @brief   Saves the event schedule for all services for the given type of tuner
 *          to the database. Note: the database has to support this operation.
 * @param   tuner_type tuner type that the services must be on, SIGNAL_NONE will save the
 *                     schedule data for all services
 * @param   satellite if tuner_type is SIGNAL_QPSK then only events for services on
 *                    the given satellite will be saved. If NULL, events for services
 *                    on all satellites will be saved.
 */
void ADB_SaveEventSchedule(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

/**
 * @brief   Deletes all networks, transports and services related to the given type of tuner
 * @param   tuner_type tuner type
 * @param   satellite if tuner_type is SIGNAL_QPSK then only services on the given satellite
 *          will be deleted. If NULL, services on all satellite will be deleted
 */
void ADB_DeleteServices(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

/**
 * @brief   Frees the memory used by any of the name lists (e.g. networks, services, etc)
 * @param   name_list name list to be freed
 * @param   num_names number of items in the name list
 */
void ADB_ReleaseNameList(U8BIT **name_list, U16BIT num_names);

/**
 * @brief   Returns the number of LNB bands in the database
 * @return  Number of LNB bands
 */
U16BIT ADB_GetNumLNBBands(void);

/**
 * @brief   Creates a new LNB band in the database with the given settings
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @param   lnb_ptr pointer to an LNB already present in the database
 * @return  pointer to new LNB band
 */
void* ADB_AddLNBBand(S_STB_DP_LNB_BAND *band_parameters, void *lnb_ptr);

/**
 * @brief   Returns the next LNB band from the database
 * @param   band_ptr current LNB band pointer, NULL to get the first LNB band
 * @return  pointer to LNB band
 */
void* ADB_GetNextLNBBand(void *band_ptr);

/**
 * @brief   Returns the LNB associated with the band
 * @param   band_ptr LNB band
 * @return  pointer to LNB
 */
void* ADB_GetLNBBandLNB(void *band_ptr);

/**
 * @brief   Returns the structure describing the band and the expected LNB behaviour in it.
 * @param   band_ptr LNB band
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @return  TRUE if the band has been found, FALSE otherwise
 */
BOOLEAN ADB_GetLNBBandParameters(void *band_ptr, S_STB_DP_LNB_BAND *params);

/**
 * @brief   Deletes LNB bands associated with a given LNB
 * @param   lnb_ptr a pointer to an LNB record, LNB bands associated with this LNB are deleted
 */
void ADB_DeleteAssociatedLNBBands(void *lnb_ptr);

/**
 * @brief   Returns the number of satellites in the database
 * @return  Number of satellites
 */
U16BIT ADB_GetNumSatellites(void);

/**
 * @brief   Creates a new satellite in the database with the given settings
 * @param   name_str name to be given to the satellite, must be a DVB format string
 * @param   dish_pos
 * @param   long_pos longitudinal position of the satellite in 1/10ths degree (e.g. 19.2 would be
 *          192)
 * @param   east_west whether the position of defined from east (TRUE) or west (FALSE)
 * @param   lnb_ptr pointer to an LNB already present in the database
 * @return  pointer to new satellite
 */
void* ADB_AddSatellite(U8BIT *name_str, U16BIT dish_pos, U16BIT long_pos, BOOLEAN east_west, void *lnb_ptr);

/**
 * @brief   Returns the next satellite from the database
 * @param   sat_ptr current satellite pointer, NULL to get the first satellite
 * @return  pointer to satellite
 */
void* ADB_GetNextSatellite(void *sat_ptr);

/**
 * @brief   Returns the pointer to the name of the satellite
 * @param   sat_ptr satellite
 * @return  pointer to satellite name
 */
U8BIT* ADB_GetSatelliteName(void *sat_ptr);

/**
 * @brief   Returns the LNB associated with the given satellite
 * @param   sat_ptr satellite
 * @return  pointer to LNB
 */
void* ADB_GetSatelliteLNB(void *sat_ptr);

/**
 * @brief   Returns the longitudinal position of the given satellite in 1/10ths degree
 * @param   sat_ptr satellite
 * @return  satellite position in 1/10ths degree
 */
U16BIT ADB_GetSatelliteLongitude(void *sat_ptr);

/**
 * @brief   Returns the position direction of the given satellite
 * @param   sat_ptr satellite
 * @return  TRUE=east, FALSE=west
 */
BOOLEAN ADB_GetSatelliteDirection(void *sat_ptr);

/**
 * @brief   Returns the number of LNBs in the database
 * @return  Number of LNBs
 */
U16BIT ADB_GetNumLNBs(void);

/**
 * @brief   Creates a new LNB in the database with the given settings
 * @param   settings settings to be given to the new LNB
 * @return  pointer to the new LNB, NULL if creation fails
 */
void* ADB_AddLNB(ADB_LNB_SETTINGS *settings);

/**
 * @brief   Returns the next LNB from the database
 * @param   lnb_ptr current LNB pointer, NULL to get the first LNB
 * @return  pointer to LNB
 */
void* ADB_GetNextLNB(void *lnb_ptr);

/**
 * @brief   Returns the current settings for the given LNB
 * @param   lnb_ptr LNB
 * @param   settings structure in which the values are returned
 * @return  TRUE if the LNB is valid and values are returned, FALSE otherwise
 */
BOOLEAN ADB_GetLNBSettings(void *lnb_ptr, ADB_LNB_SETTINGS *settings);

/**
 * @brief   Returns the number of network records in the database
 * @return  Number of networks
 */
U16BIT ADB_GetNumNetworks(void);

/**
 * @brief   Allocates and returns a list of the network records in the database.
 * @param   nlist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no networks
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no networks
 */
void ADB_GetNetworkList(void ***nlist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Frees a network list returned by ADB_GetNetworkList
 * @param   nlist network list to be freed
 */
void ADB_ReleaseNetworkList(void **nlist);

/**
 * @brief   Returns a list of names, in UTF-8 format, corresponding to the given network list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   nlist the array of network records
 * @param   num_entries number of items in the list
 * @param   short_names TRUE if the short network names should be returned
 * @return  pointer to the list of name strings
 */
U8BIT** ADB_GetNetworkListNames(void **nlist, U16BIT num_entries, BOOLEAN short_names);

/**
 * @brief   Allocates and returns an array of network ids for the given networks
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   nlist the array of network records
 * @param   num_entries number of items in the list
 * @return  pointer to the network id array, NULL if no networks are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetNetworkListIds(void **nlist, U16BIT num_entries);

/**
 * @brief   Returns the name, in UTF-8 format, of the given network.
 *          The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   n_ptr network
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetNetworkName(void *n_ptr);

/**
 * @brief   Returns the name in the language defined by lang, in UTF-8 format, of the
 *          given network. The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   n_ptr network
 * @param   lang language id
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetNetworkNameByLang(void *n_ptr, U8BIT lang);

/**
 * @brief   Returns the network id of the given network
 * @param   n_ptr network
 * @return  network id, 0 if network isn't found
 */
U16BIT ADB_GetNetworkId(void *n_ptr);

/**
 * @brief   Returns a combined array of country codes for all discovered networks
 * @param   code_array pointer to an array that will be allocated in UI temp memory
 * @return  Number of codes returned in the array
 */
U8BIT ADB_GetNetworkTargetCountries(U32BIT **code_array);

/**
 * @brief   Returns an array of primary region codes and names for the given country
 * @param   country_code code of country
 * @param   code_array pointer to an array of primary region codes that will be
 *          allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *          of the primary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkPrimaryTargetRegions(U32BIT country_code, U8BIT **code_array, U8BIT ***name_array);

/**
 * @brief   Returns an array of secondary region codes and names for the given country
 *          and primary region.
 * @param   country_code code of country
 * @param   primary_region primary region code
 * @param   code_array pointer to an array of secondary region codes that will be
 *          allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *          of the secondary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkSecondaryTargetRegions(U32BIT country_code, U8BIT primary_region,
   U8BIT **code_array, U8BIT ***name_array);

/**
 * @brief   Returns an array of tertiary region codes and names for the given country,
 *          primary region and secondary region.
 * @param   country_code code of country
 * @param   primary_region primary region code
 * @param   secondary_region secondary region code
 * @param   code_array pointer to an array of tertiary region codes that will be
 *                       allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *                       of the tertiary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkTertiaryTargetRegions(U32BIT country_code, U8BIT primary_region,
   U8BIT secondary_region, U16BIT **code_array, U8BIT ***name_array);

/**
 * @brief   Returns the network currently tuned to on the given decode path
 * @param   path decode path
 * @return  network pointer, or NULL if invalid path of not tuned
 */
void* ADB_GetTunedNetwork(U8BIT path);

/**
 * @brief   Returns the number of transports in the database
 * @return  Number of transports
 */
U16BIT ADB_GetNumTransports(void);

/**
 * @brief   Allocates and returns a list of all transport records in the database.
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *                      be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *                            value will be 0 on return if there are no transports
 */
void ADB_GetTransportList(void ***tlist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief
 * @brief   Allocates and returns a list of all transport records in the database
 *          for the given type of tuner.
 * @param   tuner_type transports of this tuner type will be returned; if SIGNAL_NONE is given
 *                     then all transports will be returned
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *                    be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *                          value will be 0 on return if there are no transports
 */
void ADB_GetTransportListForTunerType(E_STB_DP_SIGNAL_TYPE tuner_type, void ***tlist_ptr,
   U16BIT *num_entries_ptr);

/**
 * @brief   Allocates and returns a list of all transport records in the database
 *          for the given network.
 * @param   n_ptr network pointer
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no transports
 */
void ADB_GetNetworkTransportList(void *n_ptr, void ***tlist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Frees a transport list returned from one of the ADB_GetTransportList functions
 * @param   tlist transport list to be freed
 * @param   num_transports number of transports in the list
 */
void ADB_ReleaseTransportList(void **tlist, U16BIT num_transports);

/**
 * @brief   Returns a list of names, in UTF-8 format, corresponding to the given transport list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the list of name strings
 */
U8BIT** ADB_GetTransportListNames(void **tlist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of transport ids for the given transports.
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the transport id array, NULL if no transports are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetTransportListTids(void **tlist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of percentage values representing the
 *          signal strength when each transport in the given list was tuned to.
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the signal strength array, NULL if no transports are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetTransportListTunedStrengths(void **tlist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of original network ids for each transport
 *          in the given list was tuned to. The returned array should be freed using
 *          STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the array of original network ids, NULL if no transports are
 *          provided or memory allocation fails
 */
U32BIT* ADB_GetTransportListOriginalNetworkIds(void **tlist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of tuning frequencies for each transport
 *          in the given list. The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the array of frequencies, NULL if no transports are
 *          provided or memory allocation fails
 */
U32BIT* ADB_GetTransportListFreqs(void **tlist, U16BIT num_entries);

/**
 * @brief   Returns the signal type for the given transport
 * @param   t_ptr transport
 * @return  signal type
 */
E_STB_DP_SIGNAL_TYPE ADB_GetTransportSignalType(void *t_ptr);

/**
 * @brief   Returns the name, in UTF-8 format, of the given transport.
 *          The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   t_ptr transport
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetTransportName(void *t_ptr);

/**
 * @brief   Returns the transport id of the given transport
 * @param   t_ptr transport
 * @return  transport id, 0 if transport isn't found
 */
U16BIT ADB_GetTransportTid(void *t_ptr);

/**
 * @brief   Returns the original network id of the given transport
 * @param   t_ptr transport
 * @return  original network id, 0 if transport isn't found
 */
U16BIT ADB_GetTransportOriginalNetworkId(void *t_ptr);

/**
 * @brief   Returns the network of the given transport
 * @param   t_ptr transport
 * @return  pointer to the network, possibly NULL no NIT was found for the transport
 */
void* ADB_GetTransportNetworkPtr(void *t_ptr);

/**
 * @brief   Returns the frequency, in Hz for DVB-T/T2 and DVB-C, and MHz for DVB-S/S2,
 *          of the given transport
 * @param   t_ptr transport
 * @return  frequency, 0 if transport not found
 */
U32BIT ADB_GetTransportFreqHz(void *t_ptr);

/**
 * @brief   Returns the bandwidth value of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  bandwidth, return value is undefined if transport isn't DVB-T/T2
 */
U8BIT ADB_GetTransportBwidth(void *t_ptr);

/**
 * @brief   Returns the tuning offset of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  bandwidth, return value is undefined if transport isn't DVB-T/T2
 */
S8BIT ADB_GetTransportOffset(void *t_ptr);

/**
 * @brief   Returns the constellation of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCONST_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCONST ADB_GetTransportConstellation(void *t_ptr);

/**
 * @brief   Returns the terrestrial mode of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  mode, returns MODE_COFDM_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_DP_TMODE ADB_GetTransportTerrMode(void *t_ptr);

/**
 * @brief   Returns the LP coderate of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCODERATE_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCODERATE ADB_GetTransportLpCodeRate(void *t_ptr);

/**
 * @brief   Returns the HP coderate of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCODERATE_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCODERATE ADB_GetTransportHpCodeRate(void *t_ptr);

/**
 * @brief   Returns the guard interval of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  guard interval, returns TUNE_TGUARDINT_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TGUARDINT ADB_GetTransportGuardInt(void *t_ptr);

/**
 * @brief   Returns the hierarchy of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  hierarchy, returns TUNE_THIERARCHY_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_THIERARCHY ADB_GetTransportHierarchy(void *t_ptr);

/**
 * @brief   Returns the QAM mode of the given cable transport
 * @param   t_ptr cable transport
 * @return  QAM mode, returns MODE_QAM_AUTO if transport isn't DVB-C
 */
E_STB_DP_CMODE ADB_GetTransportCableMode(void *t_ptr);

/**
 * @brief   Returns the symbol rate of the given cable transport
 * @param   t_ptr cable transport
 * @return  symbol rate, returns 0 if transport isn't DVB-C
 */
U16BIT ADB_GetTransportCableSymbolRate(void *t_ptr);

/**
 * @brief   Returns the symbol rate of the given satellite transponder
 * @param   t_ptr satellite transponder
 * @return  symbol rate, 0 if transport isn't DVB-S
 */
U16BIT ADB_GetTransportSatelliteSymbolRate(void *t_ptr);

/**
 * @brief   Returns the signal strength, as a percentage, of the tuned transport
 * @param   t_ptr cable transport
 * @return  signal strength as a percentage
 */
U8BIT ADB_GetTransportTunedStrength(void *t_ptr);

/**
 * @brief   Returns the signal quality, as a percentage, of the tuned transport
 * @param   t_ptr cable transport
 * @return  signal quality as a percentage
 */
U8BIT ADB_GetTransportTunedQuality(void *t_ptr);

/**
 * @brief   Returns a transport that is signalled as containing a complete set of
 *          SI data via an NIT linkage descripter
 * @param   n_ptr network pointer
 * @return  pointer to transport
 */
void* ADB_GetFullSITransport(void *n_ptr);

/**
 * @brief   Returns a service that is signalled as containing EPG data via an NIT linkage descriptor
 * @param   n_ptr Network pointer
 * @return  pointer to service
 */
void* ADB_GetEPGService(void *n_ptr);

/**
 * @brief   Returns the settings to tune to the given DVB-T/T2 transport
 * @param   t_ptr transport to get the tuning parameters for
 * @param   terr_type pointer to return the terrestrial type, TERR_TYPE_UNKNOWN on error
 * @param   freq_hz pointer to return the tuning frequency in Hz, 0 on error
 * @param   mode pointer to return the terrestrial mode, MODE_COFDM_UNDEFINED on error
 * @param   bwidth pointer to return the bandwidth
 * @param   plp_id pointer to return the PLP value for DVB-T2, will be 0 for DVB-T
 */
void ADB_GetTransportTerrestrialTuningParams(void *t_ptr, E_STB_DP_TTYPE *terr_type, U32BIT *freq_hz,
   E_STB_DP_TMODE *mode, E_STB_DP_TBWIDTH *bwidth, U8BIT *plp_id);

/**
 * @brief   Returns the settings to tune to the given terrestrial analog transport
 * @param   t_ptr transport to get the tuning parameters for
 * @param   freq_hz pointer to return the tuning frequency in Hz, 0 on error
 * @param   offset pointer to return offset value, 0 on error
 * @param   vtype pointer to return the video type
 */
void ADB_GetTransportAnalogTuningParams(void *t_ptr, U32BIT *freq_hz, S8BIT *offset,
   E_STB_DP_ANALOG_VIDEO_TYPE *vtype);

/**
 * @brief   Returns the parameters needed to tune a cable tuner to a transport
 * @param   t_ptr transport to tune to
 * @param   freq_hz pointer to the returned freq of the transport
 * @param   mode pointer to the returned cable mode of the transport
 * @param   symrate pointer to the returned symbol rate of the transport
 */
void ADB_GetTransportCableTuningParams(void *t_ptr, U32BIT *freq_hz, E_STB_DP_CMODE *mode,
   U16BIT *symrate);

/**
 * @brief   Returns the parameters needed to tune a satellite tuner to a transport
 * @param   t_ptr transport to tune to
 * @param   freq_hz pointer to the returned freq of the transport
 * @param   polarity pointer to the returned polarity of the transport
 * @param   symrate pointer to the returned symbol rate of the transport
 * @param   fec pointer to the returned FEC of the transport
 * @param   dvb_s2 returned as TRUE if a DVB_S2 transport
 * @param   modulation pointer to returned modulation used by the transport
 */
void ADB_GetTransportSatTuningParams(void *t_ptr, U32BIT *freq_hz, E_STB_DP_POLARITY *polarity,
   U16BIT *symrate, E_STB_DP_FEC *fec, BOOLEAN *dvb_s2, E_STB_DP_MODULATION *modulation);

/**
 * @brief   Returns the parent satellite for the given transport
 * @param   t_ptr transport
 * @return  pointer to satellite
 */
void* ADB_GetTransportSatellite(void *t_ptr);

/**
 * @brief   Finds the transport with the given ids
 * @param   net_id network id, can be ADB_INVALID_DVB_ID
 * @param   onet_id original network id
 * @param   tran_id transport id
 * @return  transport, or NULL if not found
 */
void* ADB_GetTransportFromIds(U16BIT net_id, U16BIT onet_id, U16BIT tran_id);

/**
 * @brief   Sets the given transport as the one tuned to on the given decode path.
 *          The transport's network is also set as the tuned network, if valid
 * @param   path decode path
 * @param   t_ptr transport
 */
void ADB_SetTunedTransport(U8BIT path, void *t_ptr);

/**
 * @brief   Returns the transport that's tuned to on the given decode path
 * @param   path decode path
 * @return  transport, NULL if not tuned
 */
void* ADB_GetTunedTransport(U8BIT path);

/**
 * @brief   Add service record defined by the given delivery system descriptor.
 *          If the DSD defines a transport that doesn't currently exist, then one
 *          will be created, and if a service_id is given and a service with this ID
 *          doesn't exist on the transport, then one will be created, but will be
 *          marked as hidden and non-selectable so it won't appear to a user.
 * @param   data Sequence of bytes representing a system delivery descriptor
 *          as defined in EN300468 section 6.2.13
 * @param   service_id service ID
 * @return  service, or NULL if service not found or added
 */
void *ADB_AddServiceUsingDsd(U8BIT *data, U16BIT service_id);

/**
 * @brief   Returns the total number of services in the database
 * @return  number of services
 */
U16BIT ADB_GetNumServices(void);

/**
 * @brief   Returns the number of services in the database that would be returned
 *          with the given list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   inc_hidden TRUE if hidden services should be included in the returned count
 * @return  number of services
 */
U16BIT ADB_GetNumServicesInList(U32BIT list_type, BOOLEAN inc_hidden);

/**
 * @brief   Allocates and returns a list of all services in the database for the
 *          given list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetServiceList(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Allocates and returns a list of all services in the database for the
 *          given list type, including hidden services
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 * @param   ignore_selectable defines whether the returned service list should include services
 *          that are signalled as being unselectable. Use TRUE to ignore a service's selectable flag
 */
void ADB_GetServiceListIncludingHidden(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr,
   BOOLEAN ignore_selectable);

/**
 * @brief   Returns the number of services marked as deleted with the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @return  number of services marked as deleted
 */
U16BIT ADB_GetNumDeletedServices(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Returns a list of services that have been marked as 'deleted' for the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @param   slist_ptr pointer to an array that will be allocated, value will be NULL on return if
 *                    there are no services. The list must be freed using ADB_ReleaseServiceList
 * @return  number of services in the returned list
 */
U16BIT ADB_GetDeletedServiceList(E_STB_DP_SIGNAL_TYPE tuner_type, void ***slist_ptr);

/**
 * @brief   Allocates and returns a list of all services in the database on the given transport
 * @param   t_ptr transport to be queried
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetTransportServiceList(void *t_ptr, void ***slist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Allocates and returns a list of all services in the database on all
 *          transports on the given network
 * @param   n_ptr network to be queried
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetNetworkServiceList(void *n_ptr, void ***slist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Allocates and returns a list of all services that are marked as being unavailable.
 *          A service becomes unavailable when it disappears from the SDT.
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetUnavailableServiceList(void ***slist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Allocates and returns a list of all services that are marked as locked
 *          for parental control purposes
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetLockedServiceList(void ***slist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Frees a list of services returned from one of the functions that
 *          returns a service list, such as ADB_GetServiceList
 * @param   slist service list to be freed
 * @param   num_servs number of services in the list
 */
void ADB_ReleaseServiceList(void **slist, U16BIT num_servs);

/**
 * @brief   Creates a subset of locked services from the supplied service list. Memory is allocated
 *          as TempMemory so will be released when the calling screen closes
 * @param   slist pointer to an array of service record pointers
 * @param   num_entries number of entries in the list
 * @param   locked_slist_ptr address for the return of the locked slist
 * @param   locked_num_entries_ptr address for the return of the number of entries in the locked list
 */
void ADB_GetServiceListLockedSubset(void **slist, U16BIT num_entries,
   void ***locked_slist_ptr, U16BIT *locked_num_entries_ptr);

/**
 * @brief   Sorts the given list of services into alphabetical order. Case is significant
 * @param   slist service list to be sorted
 * @param   num_entries number of services in the list
 * @param   short_name TRUE if the short name for a service should be used
 */
void ADB_SortServiceListAlphabetically(void **slist, U16BIT num_entries, BOOLEAN short_name);

/**
 * @brief   Checks whether the given service is in the current service list
 * @param   serv_ptr service to be checked
 * @return  TRUE if the service is found, FALSE otherwise
 */
BOOLEAN ADB_IsValidService(void *serv_ptr);

/**
 * @brief   Returns the signal type for transport of given service. It also indicates
 *          the mode (T2, S2 or C2)
 * @param   s_ptr service
 * @param   is_sig2 return signal mode, TRUE means T2, S2 or C2 (depending on signal type)
 * @return  signal type
 */
E_STB_DP_SIGNAL_TYPE ADB_GetServiceSignalType(void *s_ptr, BOOLEAN *is_sig2);

/**
 * @brief   Returns the next service, in LCN order, for the given list type, starting
 *          from the given service. If serv is NULL then the first service is returned.
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   serv starting service, NULL to get the first service
 * @return  next service, or NULL if no more services
 */
void* ADB_GetNextServiceInList(U32BIT list_type, void *serv);

/**
 * @brief   Returns the previous service, in LCN order, for the given list type, starting
 *          from the given service. If serv is NULL then the last service is returned.
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   serv starting service, NULL to get the last service
 * @return  previous service, or NULL if no more services
 */
void* ADB_GetPrevServiceInList(U32BIT list_type, void *serv);

/**
 * @brief   Returns the service matching the given LCN and list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   lcn locgical channel number
 * @param   show_unselectable used by MHEG to allow it to find services that
 *          would normally not be selectable by a user
 * @return  service, or NULL if not found
 */
void* ADB_FindServiceByLcn(U32BIT list_type, U16BIT lcn, BOOLEAN show_unselectable);

/**
 * @brief   Returns the service matching the given, or preceding, LCN and list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   lcn logical channel number
 * @return  service, or NULL if not found
 */
void* ADB_FindServiceByNearestLcn(U32BIT list_type, U16BIT lcn);

/**
 * @brief   Returns a pointer to the service matching the given IDs.
 * @param   onet_id original network id
 * @param   tid transport id
 * @param   sid service id
 * @return  service pointer, or NULL if not found
 */
void* ADB_FindServiceByIds(U16BIT onet_id, U16BIT tid, U16BIT sid);

/**
 * @brief   Returns the index into the given list of services of the service with the given LCN
 * @param   lcn logical channel number to search for
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindLcnInServiceList(U16BIT lcn, void **slist_ptr, U16BIT num_entries);

/**
 * @brief   Returns the index in the given list of services of the service matching
 *          the given, or preceding, LCN
 * @param   lcn logical channel number
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindNearestLcnInServiceList(U16BIT lcn, void **slist_ptr, U16BIT num_entries);

/**
 * @brief   Returns the index in the given list of services of the given service
 * @param   s_ptr service to search for
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindServiceInList(void *s_ptr, void **slist_ptr, U16BIT num_entries);

/**
 * @brief   Allocates and returns a list of the fullnames for all the services in the given service
 *          list. The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   use_pref_name TRUE if the preferred names are to be returned.
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListFullNames(void **slist, U16BIT num_entries, BOOLEAN use_pref_names);

/**
 * @brief   Allocates and returns a list of the short names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   use_pref_name TRUE if the preferred names are to be returned.
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListShortNames(void **slist, U16BIT num_entries, BOOLEAN use_pref_names);

/**
 * @brief   Allocates and returns an array of the logical channel numbers that
 *          have been assigned to all the services in the given service list.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of LCNs, NULL if list is empty or memory allocation fails
 */
U16BIT* ADB_GetServiceListLcns(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of the logical channel numbers that
 *          each of the services in the given service list originally requested.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of LCNs, NULL if list is empty or memory allocation fails
 */
U16BIT* ADB_GetServiceListOrigLcns(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of the unavailable flags for each of
 *          the services in the given service list. A value of TRUE means the
 *          service is unavailable, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListUnavailableFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of the 'new' flags for each of the
 *          services in the given service list. A service is marked as 'new' when
 *          it's added to the service database following an update service scan.
 *          A value of TRUE means the service is new, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListNewFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of the 'locked' flags for each of the
 *          services in the given service list. A service is marked as 'locked'
 *          due to setup of parental control.
 *          A value of TRUE means the service is locked, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListLockedFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Sets the locked state of all services in the given service list
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   locked TRUE if the services are to be locked, or FALSE to unlock
 */
void ADB_SetServiceListLockedFlag(void **slist, U16BIT num_entries, BOOLEAN locked);

/**
 * @brief   Allocates and returns an array of the scrambled flags for each of the
 *          services in the given service list. A value of TRUE means the service
 *          is signalled as scrambled and FALSE means it's free-to-air.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListScrambledFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array indicating whether each service in the
 *          given service list is an HD service. A value of TRUE means the service
 *          is signalled as being HD and FALSE means it's SD.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListHdFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array indicating whether each service in the
 *          given service list is signalled as being hidden or not. A value of TRUE
 *          means the service is hidden and FALSE means it's visible.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListHiddenFlag(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns a list of network names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListNetworkNames(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns a list of transport names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListTransportNames(void **slist, U16BIT num_entries);

/**
 * @brief   Allocates and returns an array of the signal strengths of the transports
 *          containing the given list of services.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListTransportStrengths(void **slist, U16BIT num_entries);

/**
 * @brief   Sets the hidden status of the given service, which means the service
 *          would not be presented to the user in any list of available services.
 * @param   s_ptr service
 * @param   hidden TRUE to hide the service, FALSE to make it visible
 */
void ADB_SetServiceHiddenFlag(void *s_ptr, BOOLEAN hidden);

/**
 * @brief   Returns the hidden status of the given service. A hidden service
 *          should not be present to teh user in any list of available services.
 * @param   s_ptr service
 * @return  TRUE if the service is hidden, FALSE if it's visible
 */
BOOLEAN ADB_GetServiceHiddenFlag(void *s_ptr);

/**
 * @brief   Returns the selectable status of the given service
 * @param   s_ptr service
 * @return  TRUE if the service is valid and is selectable, FALSE otherwise
 */
BOOLEAN ADB_GetServiceSelectableFlag(void *s_ptr);

/**
 * @brief   Returns whether the LCN assigned to the given service can be changed
 * @param   s_ptr service
 * @return  TRUE if the service is valid and its LCN can be edited, FALSE otherwise
 */
BOOLEAN ADB_GetServiceLcnEditable(void *s_ptr);

/**
 * @brief   Sets the 'deleted' flag for the given service as a deleted service. If it's marked
 *          as deleted then it won't be included in any of the normal service lists
 * @param   s_ptr service to be changed
 * @param   deleted TRUE if the service is to be marked as deleted, FALSE to make it available
 */
void ADB_SetServiceDeletedFlag(void *s_ptr, BOOLEAN deleted);

/**
 * @brief   Returns the full name of the given service as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   use_pref_name TRUE if the preferred name should be returned
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceFullName(void *s_ptr, BOOLEAN use_pref_name);

/**
 * @brief   Returns the short name of the given service as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   use_pref_name TRUE if the preferred name should be returned
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceShortName(void *s_ptr, BOOLEAN use_pref_name);

/**
 * @brief   Returns the full name of the given service as a UTF-8 string using
 *          the given language and preferred name id, rather than the defaults.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language id
 * @param   pref_name_id preferred name id
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceFullNameByLangAndPrefId(void *s_ptr, U8BIT lang, U8BIT pref_name_id);

/**
 * @brief   Returns the short name of the given service as a UTF-8 string using
 *          the given language and preferred name id, rather than the defaults.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language id
 * @param   pref_name_id preferred name id
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceShortNameByLangAndPrefId(void *s_ptr, U8BIT lang, U8BIT pref_name_id);

/**
 * @brief   Returns the logical channel number assigned to the given service
 * @param   s_ptr service
 * @return  logical channel number
 */
U16BIT ADB_GetServiceLcn(void *s_ptr);

/**
 * @brief   Sets the assigned logical channel number of the given service.
 *          This will override any LCN assigned when performing a service scan.
 * @param   s_ptr service
 * @param   lcn logical channel number
 */
void ADB_SetServiceLcn(void *s_ptr, U16BIT lcn);

/**
 * @brief   Returns the signalled service id of the given service
 * @param   s_ptr service
 * @return  service id
 */
U16BIT ADB_GetServiceId(void *s_ptr);

/**
 * @brief   Returns the original network id, transport id and service id for the given service
 * @param   s_ptr service handle
 * @param   onet_id pointer for the return of the original network id
 * @param   trans_id pointer for the return of the transport id
 * @param   serv_id pointer for the return of the service id
 */
void ADB_GetServiceIds(void *s_ptr, U16BIT *onet_id, U16BIT *trans_id, U16BIT *serv_id);

/**
 * @brief   Sets whether the EIT schedule events for this service are held in memory.
 *          All services hold schedule events in memory by default
 * @param   s_ptr service
 * @param   state FALSE to disable, TRUE to enable
 */
void ADB_SetServiceScheduleState(void *s_ptr, BOOLEAN state);

/**
 * @brief   Returns whether the EIT schedule events are being held in memory for the service
 * @param   s_ptr service
 * @return  TRUE if data is being held, FALSE otherwise
 */
BOOLEAN ADB_GetServiceScheduleState(void *s_ptr);

/**
 * @brief   Sets whether the now/next EIT events for this service are held in memory.
 *          All services save now/next events by default
 * @param   s_ptr service
 * @param   state FALSE to disable, TRUE to enable
 */
void ADB_SetServiceNowNextState(void *s_ptr, BOOLEAN state);

/**
 * @brief   Returns whether the EIT now/next data is being held in memory for the service
 * @param   s_ptr service
 * @return  TRUE if data is being held, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNowNextState(void *s_ptr);

/**
 * @brief   Sort all services in the database according to their LCNs
 */
void ADB_SortServicesByLcn(void);

/**
 * @brief   Returns the logical channel number originally requested by the given service
 * @param   s_ptr service
 * @return  logical channel number
 */
U16BIT ADB_GetServiceOrigLcn(void *s_ptr);

/**
 * @brief   Returns the PCR PID for the given service
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServicePCRPid(void *s_ptr);

/**
 * @brief   Returns the audio PID for the given service. This will be the one chosen
 *          based on the settings for language, etc, or the one explicitly set (e.g.
 *          due to user selection).
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceAudioPid(void *s_ptr);

/**
 * @brief   Returns the audio type being used by the given service
 * @param   s_ptr service
 * @return  Audio type, defaults to ADB_AUDIO_STREAM (MPEG1)
 */
ADB_STREAM_TYPE ADB_GetServiceAudioType(void *s_ptr);

/**
 * @brief   Returns the video PID for the given service.
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceVideoPid(void *s_ptr);

/**
 * @brief   Returns the video type being used by the given service
 * @param   s_ptr service
 * @return  Video type, defaults to ADB_VIDEO_STREAM (MPEG2)
 */
ADB_STREAM_TYPE ADB_GetServiceVideoType(void *s_ptr);

/**
 * @brief   Returns the PID being used for DVB subtitles and/or teletext for the
 *          given service. This will be the one chosen based on the settings for
 *          language, etc, or the one explicitly set (e.g. due to user selection).
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceTextPid(void *s_ptr);

/**
 * @brief   Returns whether the service is signalled as an HD service
 * @param   s_ptr service
 * @return  TRUE if it can be determined that the service is HD, FALSE otherwise
 */
BOOLEAN ADB_GetServiceIsHd(void *s_ptr);

/**
 * @brief   Returns the audio PID to be used for the given service based on language
 *          settings, output format, etc.
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredAudioPid(void *s_ptr);

/**
 * @brief   Returns the PID to be used for audio description for the given service
 *          based on language settings, output format, etc.
 * @param   s_ptr service
 * @param   broadcast_mix returned as TRUE if the selected AD is broadcast mix
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredADPid(void *s_ptr, BOOLEAN *broadcast_mix);

/**
 * @brief   Returns the mode to be used for audio for the given service based on
 *          language settings, output format, etc.
 * @param   s_ptr service
 * @return  mode, default is AUDIO_STEREO
 */
E_STB_DP_AUDIO_MODE ADB_GetRequiredAudioMode(void *s_ptr);

/**
 * @brief   Returns the PID to be used for teletext for the given service based
 *          on language settings
 * @param   s_ptr service
 * @param   for_subtitles find the teletext service marked as being subtitles
 * @param   magazine returns the magazine part of the teletext page number
 * @param   page returns the page number part of the teletext page number
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredTtextPid(void *s_ptr, BOOLEAN for_subtitles, U8BIT *magazine, U8BIT *page);

/**
 * @brief   Returns the PID to be used for DVB subtitles for the given service based
 *          on language settings
 * @param   s_ptr service
 * @param   cpage_ptr returns the composition page number
 * @param   apage_ptr returns the ancilliary page number
 * @return  PID value, 0 if not found
 */
U16BIT ADB_GetRequiredSubtitleParams(void *s_ptr, U16BIT *cpage_ptr, U16BIT *apage_ptr);

/**
 * @brief   Returns the signalled type of the given service
 * @param   s_ptr service
 * @return  service type, returns 0 if unknown
 */
ADB_SERVICE_TYPE ADB_GetServiceType(void *s_ptr);

/**
 * @brief   Returns the service group mask of the given service
 * @param   s_ptr service
 * @return  service group mask
 */
U32BIT ADB_GetServiceGroupMask(void *s_ptr);

/**
 * @brief   Returns the provider name, as a UTF-8 string, using the default language,
 *          of the given service.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @return  UTF-8 provider name, or NULL
 */
U8BIT* ADB_GetServiceProviderName(void *s_ptr);

/**
 * @brief   Returns the provider name, as a UTF-8 string, using the given language,
 *          of the given service.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language identifier, one of the ACFG_DB_LANG_ values defined in ap_cfg.h
 * @return  UTF-8 provider name, or NULL
 */
U8BIT* ADB_GetServiceProviderNameByLang(void *s_ptr, U8BIT lang);

/**
 * @brief   Returns the status of the unavailable flag of the given service.
 *          This flag indicates whether a service is no longer signalled in the SDT.
 * @param   s_ptr service
 * @return  TRUE if the service is unavailable, FALSE otherwise
 */
BOOLEAN ADB_GetServiceUnavailableFlag(void *s_ptr);

/**
 * @brief   Returns the running status of the given service as reported by the SDT
 * @param   s_ptr service
 * @return  running status will be 0 if service isn't valid or SDT hasn't been parsed yet
 */
U8BIT ADB_GetServiceRunningStatus(void *s_ptr);

/**
 * @brief   Returns the status of the 'not running' flag of the given service.
 *          This flag indicates whether a service is signalled as not running
 *          in the SDT, or is removed from the PAT.
 * @param   s_ptr service
 * @return  TRUE if the service is not running, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNotRunningFlag(void *s_ptr);

/**
 * @brief   Returns the status of the 'new' flag of the given service.
 *          This flag indicates whether a service has been added to the service
 *          lineup following a service scan.
 * @param   s_ptr service
 * @return  TRUE if the service is new, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNewFlag(void *s_ptr);

/**
 * @brief   Returns the status of the 'locked' flag of the given service, which
 *          is used by the parental control.
 * @param   s_ptr service
 * @return  TRUE if the service is locked, FALSE otherwise
 */
BOOLEAN ADB_GetServiceLockedFlag(void *s_ptr);

/**
 * @brief   Changes the current state of the given service from locked to unlocked,
 *          or vide versa.
 * @param   s_ptr service
 */
void ADB_ToggleServiceLockedFlag(void *s_ptr);

/**
 * @brief   Locks or unlocks the given service.
 * @param   s_ptr service
 * @param   locked TRUE to lock the service, FALSE to unlock
 */
void ADB_SetServiceLockedFlag(void *s_ptr, BOOLEAN locked);

/**
 * @brief   Returns the status of the 'scrambled' flag of the given service.
 *          This flag is set depending on the scrambled state of all the streams
 *          that makeup the service being marked as free-to-air.
 * @param   s_ptr service
 * @return  TRUE if the service is scrambled, FALSE otherwise
 */
BOOLEAN ADB_GetServiceScrambledFlag(void *s_ptr);

/**
 * @brief   Used to query whether the given service has a CA descriptor
 * @param   s_ptr service
 * @return  TRUE if the service has a CA descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetServiceHasCaDesc(void *s_ptr);

/**
 * @brief   Returns the do_not_scramble flag for the service which is based on
 *          the presence of an FTA descriptor. The value returned will be found
 *          by looking backwards from the service to the transport, etc.
 * @param   s_ptr pointer to the service
 * @return  FALSE if the service is to be protected, TRUE otherwise
 */
BOOLEAN ADB_GetServiceDoNotScramble(void *s_ptr);

/**
 * @brief   Returns the content protection level value for the service that will have been
 *          set by the Nordig content protection descriptor in the PMT
 * @param   s_ptr service
 * @return  Value of the content protection level
 */
U8BIT ADB_GetServiceContentProtectionLevel(void *s_ptr);

/**
 * @brief   Returns a value indicating whether the given service is a Freesat service
 * @param   s_ptr service handle
 * @return  TRUE if service is a Freesat service, FALSE otherwise
 */
BOOLEAN ADB_IsFreesatService(void *s_ptr);

/**
 * @brief   Returns Freesat service ID of the given service
 * @param   s_ptr service handle
 * @return  Freesat ID of the service
 */
U16BIT ADB_GetFreesatServiceId(void *s_ptr);

/**
 * @brief   Returns a pointer to the service's parent transport record
 * @param   s_ptr pointer to the service
 * @return  transport, or NULL
 */
void* ADB_GetServiceTransportPtr(void *s_ptr);

/**
 * @brief   Returns a copy of the default authority CRID string for the given service.
 *          This string will be returned as broadcast and should be freed using STB_AppFreMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to a copy of the CRID string, or NULL
 */
U8BIT* ADB_GetServiceDefaultAuthority(void *serv_ptr);

/**
 * @brief   Adds the given RCT link info to the end of the list of existing RCT links
 *          already defined for the given service
 * @param   serv_ptr service to which link is to be added
 * @param   link_ptr link to be added
 */
void ADB_ServiceAddRCTLink(void *serv_ptr, void *link_ptr);

/**
 * @brief   Frees all RCT link info for the given service
 * @param   serv_ptr service from which links are to be cleared
 */
void ADB_ServiceReleaseRCTLinks(void *serv_ptr);

/**
 * @brief   Returns the number of RCT links for the given service
 * @param   serv_ptr service from which links are to be cleared
 * @return  Number of RCT links
 */
U8BIT ADB_GetServiceNumRCTLinks(void *serv_ptr);

/**
 * @brief   Returns a copy of the RCT links for the given service.
 *          The returned list should be freed using ADB_ReleaseRCTLinks.
 * @param   serv_ptr service from which links are to be copied
 * @param   num_links pointer to return the number of links
 * @return  Copy of links, or NULL
 */
void* ADB_GetServiceRCTLinks(void *serv_ptr, U8BIT *num_links);

/**
 * @brief   Returns the RCT link following the given link
 * @param   link_ptr pointer to an RCT link
 * @return  next RCT link, or NULL
 */
void* ADB_GetNextRCTLink(void *link_ptr);

/**
 * @brief   Frees the given list of RCT links
 * @param   links links to be freed
 */
void ADB_ReleaseRCTLinks(void *links);

/**
 * @brief   Returns the RCT link's promotional text based on the default language
 * @param   link pointer to link
 * @return  Pointer to promotional link string
 */
U8BIT* ADB_GetRCTLinkPromoText(void *link_ptr);

/**
 * @brief   Returns whether the given link is for a group trailer
 * @param   link_ptr pointer to an RCT link
 * @return  Link's is_group_trailer flag
 */
BOOLEAN ADB_IsRCTLinkGroupTrailer(void *link_ptr);

/**
 * @brief   Returns the name of the given RCT link
 * @param   link_ptr pointer to an RCT link
 * @return  pointer to the link's name - NOTE: this shouldn't be changed or freed!
 */
U8BIT* ADB_GetRCTLinkName(void *link_ptr);

/**
 * @brief   Returns the uri string of the given RCT link
 * @param   link_ptr pointer to an RCT link
 * @return  pointer to the link's uri - NOTE: this shouldn't be changed or freed!
 */
U8BIT* ADB_GetRCTLinkUriString(void *link_ptr);

/**
 * @brief   Checks all the RCT links for the given service to determine whether any
 *          of them specify that the default icon can be used.
 * @param   serv_ptr service containing the RCT links
 * @return  TRUE if default icon can be used, FALSE otherwise
 */
BOOLEAN ADB_ServiceRCTCanUseDefaultIcon(void *serv_ptr);

/**
 * @brief   Searches all the RCT links for a service to see if any of them define an
 *          icon to be used, and if one is found then all the data required to display
 *          the icon is returned.
 * @param   serv_ptr service
 * @param   icon_data address of pointer to icon data. This data shouldn't be freed.
 * @param   data_size pointer to variable containing amount of data
 * @param   pos_valid pointer to boolean indicating whether returned x,y values are valid
 * @param   x_pos pointer to returned X position to display the icon
 * @param   y_pos pointer to returned Y position to display the icon
 * @param   width pointer to returned width of the icon
 * @param   height pointer to returned height of the icon
 * @param   coord_system pointer to coord system of returned pos and size values
 * @return  TRUE if icon data is returned, FALSE otherwise
 */
BOOLEAN ADB_ServiceGetRCTIcon(void *serv_ptr, U8BIT **icon_data, U32BIT *data_size, BOOLEAN *pos_valid,
   U16BIT *x_pos, U16BIT *y_pos, U16BIT *width, U16BIT *height, E_ICON_COORD_SYSTEM *coord_system);

/**
 * @brief   Adds the given image icon to the end of the service's icon list.
 *          The icon id is checked and if it matches an icon already in the list
 *          then the new icon replaces the existing one.
 * @param   serv_ptr service to which link is to be added
 * @param   icon_ptr icon to be added
 * @return  TRUE if the icon is added, FALSE otherwise
 */
BOOLEAN ADB_ServiceAddImageIcon(void *serv_ptr, void *icon_ptr);

/**
 * @brief   Searches for a replacement service of the given type for the given service
 * @param   s_ptr service that a replacement is required for
 * @param   alt_serv_type type of replacement service to look for (e.g. CA replacement)
 * @return  alternative service found, or NULL
 */
void* ADB_GetAlternativeService(void *s_ptr, U8BIT alt_serv_type);

/**
 * @brief   Determines subtitle PID and whether it is DVB or teletext, for the given service
 * @param   serv_ptr service
 * @param   subt_info pointer to ADB_SUBT_INFO for extra info on subtitles
 * @return  PID to collect subtitles data, zero if none available
 */
U16BIT ADB_ServiceGetSubtitlePid(void *serv_ptr, ADB_SUBT_INFO *subt_info);

/**
 * @brief   Determines whether the given service has subtitles, DVB or teletext
 * @param   serv_ptr service
 * @param   dvb_subs returned as TRUE if the subtitles are DVB, FALSE for teletext
 * @return  TRUE if subtitles are available
 */
BOOLEAN ADB_ServiceHasSubtitles(void *serv_ptr, BOOLEAN *dvb_subs);

/**
 * @brief   Sets the tuned service for the given path, together with the associated
 *          tuned transport and network. The 'new' flag for the service is aso cleared.
 * @param   path decode path
 * @param   s_ptr service
 */
void ADB_SetTunedService(U8BIT path, void *s_ptr);

/**
 * @brief   Returns the tuned service for the given decode path.
 * @param   path decode path
 * @return  tuned service, or NULL
 */
void* ADB_GetTunedService(U8BIT path);

/**
 * @brief   Deletes the given service from the database
 * @param   s_ptr service to be deleted
 */
void ADB_DeleteServiceRec(void *s_ptr);

/**
 * @brief   Returns the number of streams of the given type for the given service
 * @param   serv_ptr service
 * @param   stream_list_type type of stream
 * @return  Number of streams
 */
U16BIT ADB_GetNumStreams(void *serv_ptr, ADB_STREAM_LIST_TYPE stream_list_type);

/**
 * @brief   Allocates and returns an array of the streams of the given type and
 *          for the given service.
 *          The returned array should be freed using ADB_ReleaseStreamList.
 * @param   serv_ptr service
 * @param   stream_list_type type of streams to be returned
 * @param   streamlist_ptr pointer to array that will be allocated
 * @param   num_entries_ptr pointer to number of items returned in the array
 */
void ADB_GetStreamList(void *serv_ptr, ADB_STREAM_LIST_TYPE stream_list_type, void ***streamlist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Frees the memory allocated for a stream list using ADB_GetStreamList
 * @param   streamlist_ptr stream list to be freed
 * @param   num_entries number of items in the stream list
 */
void ADB_ReleaseStreamList(void **streamlist_ptr, U16BIT num_entries);

/**
 * @brief   Returns the type of the given stream
 * @param   stream_ptr stream
 * @return  stream type
 */
ADB_STREAM_TYPE ADB_GetStreamType(void *stream_ptr);

/**
 * @brief   Returns the number of tag values defined for the given stream
 * @param   stream_ptr stream
 * @return  number of tag values
 */
U8BIT ADB_GetStreamNumTags(void *stream_ptr);

/**
 * @brief   Returns the requested tag value of the given stream
 * @param   stream_ptr stream
 * @param   index tag value to return, must be < ADB_GetStreamNumTags
 * @return  tag value
 */
U8BIT ADB_GetStreamTag(void *stream_ptr, U8BIT index);

/**
 * @brief   For the given stream and index into component tag array for stream,
 *          return stream content and component type as defined in the EITp/f's component descriptor
 *          See EN300468, sec 6.2.8 and table 26 for meaning of return values.
 * @param   stream_ptr stream
 * @param   index tag value to return, must be < ADB_GetStreamNumTags
 * @return  bits 15..12 - stream_content_ext, bits 11..8 - stream_content, bits 7..0 - component type
 */
U16BIT ADB_GetStreamTagContentType(void *stream_ptr, U8BIT index);

/**
 * @brief   Returns the PID for the given stream
 * @param   stream_ptr stream
 * @return  PID
 */
U16BIT ADB_GetStreamPID(void *stream_ptr);

/**
 * @brief   Returns whether the given stream is marked as being 'in use', which
 *          means it will be the stream being decoded for video, audio, etc.
 * @param   stream_ptr stream
 * @return  TRUE if stream is in use, FALSE otherwise
 */
BOOLEAN ADB_GetStreamInUse(void *stream_ptr);

/**
 * @brief   Returns whether the given stream has a CA descriptor
 * @param   stream_ptr stream
 * @return  TRUE if stream has a CA descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetStreamHasCaDesc(void *stream_ptr);

/**
 * @brief   Returns the component type as defined in the EITp/f's component descriptor
 *          for the given stream
 * @param   stream_ptr stream
 * @return  component type
 */
U8BIT ADB_GetStreamComponentType(void *stream_ptr);

/**
 * @brief   Returns the language code associated with the given audio stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetAudioStreamLangCode(void *stream_ptr);

/**
 * @brief   Returns the audio type of the given stream
 * @param   stream_ptr stream
 * @return  audio type
 */
ADB_AUDIO_TYPE ADB_GetAudioStreamType(void *stream_ptr);

/**
 * @brief   Returns the audio mode of the given stream
 * @param   stream_ptr stream
 * @return  audio mode
 */
E_STB_DP_AUDIO_MODE ADB_GetAudioStreamMode(void *stream_ptr);

/**
 * @brief   Returns the 3 char language code of the given teletext stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetTtextStreamLangCode(void *stream_ptr);

/**
 * @brief   Returns the type, as defined in the PMT, of the given teletext stream
 * @param   stream_ptr stream
 * @return  teletext stream type (ADB_TELETEXT_TYPE)
 */
U8BIT ADB_GetTtextStreamType(void *stream_ptr);

/**
 * @brief   Returns the magazine value of the given teletext stream
 * @param   stream_ptr stream
 * @return  magazine value
 */
U8BIT ADB_GetTtextStreamMagazine(void *stream_ptr);

/**
 * @brief   Returns the page value of the given teletext stream
 * @param   stream_ptr stream
 * @return  page value
 */
U8BIT ADB_GetTtextStreamPage(void *stream_ptr);

/**
 * @brief   Returns the 3 char language code of the given subtitle stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetSubtitleStreamLangCode(void *stream_ptr);

/**
 * @brief   Returns the type, as defined in the PMT, of the given subtitle stream
 * @param   stream_ptr stream
 * @return  subtitle stream type
 */
ADB_SUBTITLE_TYPE ADB_GetSubtitleStreamType(void *stream_ptr);

/**
 * @brief   Returns the composition page value of the given subtitle stream
 * @param   stream_ptr stream
 * @return  composition page value
 */
U16BIT ADB_GetSubtitleStreamCompositionPage(void *stream_ptr);

/**
 * @brief   Returns the ancillary page value of the given subtitle stream
 * @param   stream_ptr stream
 * @return  ancillary page value
 */
U16BIT ADB_GetSubtitleStreamAncillaryPage(void *stream_ptr);

/**
 * @brief   Sets the primary audio language to be used
 * @param   country_code current country code
 * @param   db_lang_id language id
 */
void ADB_SetAudioLang(U32BIT country_code, U8BIT lang_id);

/**
 * @brief   Sets the secondary audio language to be used. This will be selected
 *          if the primary audio language isn't available
 * @param   country_code current country code
 * @param   lang_id language id
 */
void ADB_SetSecondaryAudioLang(U32BIT country_code, U8BIT lang_id);

/**
 * @brief   Sets the primary subtitle and teletext language to be used
 * @param   country_code current country code
 * @param   lang_id language id
 */
void ADB_SetTextLang(U32BIT country_code, U8BIT lang_id);

/**
 * @brief   Sets the secondary subtitle and teletext language to be used. This
 *          will be used if the primary language isn't available
 * @param   country_code current country code
 * @param   lang_id language id
 */
void ADB_SetSecondaryTextLang(U32BIT country_code, U8BIT lang_id);

// now/next and schedule events

/**
 * @brief   Makes copies of the now and/or next events (EITp/f) for the given service.
 *          The returned events should be freed using ADB_ReleaseEventData.
 * @param   serv_ptr get the now/next events for this service
 * @param   now_event address in which to return a pointer to the now event
 *          This value can be NULL if the now event isn't to be returned
 * @param   next_event address in which to return a pointer to the next event
 *          This value can be NULL if the now event isn't to be returned
 */
void ADB_GetNowNextEvents(void *serv_ptr, void **now_event, void **next_event);

/**
 * @brief   Allocates and returns an array containing copies of events for the given
 *          service from the service's EIT schedule.
 *          The returned array should be freed using ADB_ReleaseEventList.
 * @param   include_old_events if TRUE, all events in the services schedule will be returned,
 *          but if FALSE then the first two events in the returned list
 *          will be the now/next events, followed by the remaining schedule
 * @param   serv_ptr service to get events for
 * @param   elist_ptr address in which to return a pointer to the alloctaed array of events
 * @param   num_entries_ptr address in which to return the number of events in the list
 */
void ADB_GetEventSchedule(BOOLEAN include_old_events, void *serv_ptr, void ***elist_ptr,
   U16BIT *num_entries_ptr);

/**
 * @brief   Creates a copy of the event schedule for the given service, but only includes
 *          events of the given genre type
 * @param   genre event genre to be matched
 * @param   include_old_events if FALSE, the first events returned will be the now/next events,
 *          if they match the genre, otherwise events that may have expired
 *          will also be included
 * @param   serv_ptr service
 * @param   elist_ptr returned with an allocated array of events
 * @param   num_entries_ptr returned with the number of entries in the returned array
 */
void ADB_GetEventScheduleByGenre(ADB_EVENT_CONTENT genre, BOOLEAN include_old_events,
   void *serv_ptr, void ***elist_ptr, U16BIT *num_entries_ptr);

/**
 * @brief   Frees any memory allocated for the given event and the event itself
 * @param   event_ptr event to be freed
 */
void ADB_ReleaseEventData(void *event_ptr);

/**
 * @brief   Frees all teh events in the given list and all associated memory for those events
 * @param   elist event list to be freed
 * @param   num_entries number of events in the list
 */
void ADB_ReleaseEventList(void **elist, U16BIT num_entries);

/**
 * @brief   Deletes all events and event related data for the given service
 * @param   serv_ptr service for the event
 * @param   delete_now_next specifies whether the now/next events should also be deleted
 */
void ADB_DeleteServiceEvents(void *serv_ptr, BOOLEAN delete_now_next);

/**
 * @brief   Deletes all events and event related data for all services
 * @param   delete_now_next specifies whether the now/next events should also be deleted
 */
void ADB_DeleteAllServiceEvents(BOOLEAN delete_now_next);

/**
 * @brief   Returns a copy of the event with the given event ID on the given service
 * @param   serv_ptr service for the event
 * @param   event_id ID of the event to be returned
 * @return  pointer to a copy of the event, or NULL if the event isn't found
 */
void* ADB_GetEvent(void *serv_ptr, U16BIT event_id);

/**
 * @brief   Returns a copy of the raw SI descriptor data with the given descriptor
 *          tag id and, optionally, extended descriptor tag id, for the event with
 *          the given event id.
 *          The data must be freed using ADB_ReleaseEventSIDescriptorData.
 * @param   serv_ptr service for the event
 * @param   event_id id of the event on the service
 * @param   dtag_id descriptor tag id
 * @param   ext_dtag_id extended descriptor tag id, or negative if not to be used
 * @param   desc_len number of bytes in the returned descriptor data
 * @return  pointer to the descriptor data, or NULL if event or descriptor isn't found
 */
U8BIT* ADB_GetEventSIDescriptorData(void *serv_ptr, U16BIT event_id, U8BIT dtag_id,
   S16BIT ext_dtag_id, U16BIT *desc_len);

/**
 * @brief   Frees the data returned by ADB_GetEventSIDescriptorData
 * @param   desc_data data to be freed
 * @param   desc_len number of bytes
 */
void ADB_ReleaseEventSIDescriptorData(U8BIT *desc_data, U16BIT desc_len);

/**
 * @brief   Returns a copy of the event preceeding the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   time Date/time from which to start searching
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_EarlierEvent(void *serv_ptr, U32DHMS time);

/**
 * @brief   Returns a copy of the event following the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   time Date/time from which to start searching
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_LaterEvent(void *serv_ptr, U32DHMS time);

/**
 * @brief   Returns a copy of the event preceeding the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   date MJD date code
 * @param   hour hour, 0-23
 * @param   min minute, 0-59
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_GetEarlierEvent(void *serv_ptr, U16BIT date, U8BIT hour, U8BIT min);

/**
 * @brief   Returns a copy of the event following the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   date MJD date code
 * @param   hour hour, 0-23
 * @param   min minute, 0-59
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_GetLaterEvent(void *serv_ptr, U16BIT date, U8BIT hour, U8BIT min);

/**
 * @brief   Finds the event scheduled to be broadcast at the specified time on the specified service
 * @param   time date/time when the event is scheduled for broadcast
 * @param   serv_ptr handle of the service on which the event is scheduled for broadcast
 * @return  event handler, NULL if not found
 * @note    the first two events searched will be the now/next events, followed by the remaining schedule
 */
void* ADB_FindEventFromTime(U32DHMS time, void *serv_ptr);

/**
 * @brief   Finds the event in the event schedule to be broadcast at the specified time on the specified service
 * @param   time date/time when the event is scheduled for broadcast
 * @param   serv_ptr handle of the service on which the event is scheduled for broadcast
 * @note    searches the complete schedule including old events
 */
void* ADB_FindEITScheduleEventFromTime(U32DHMS time, void *serv_ptr);

/**
 * @brief   Returns a value representing the date and time of the start of the given event
 * @param   event_ptr event
 * @return  date/time
 */
U32DHMS ADB_GetEventStartDateTime(void *event_ptr);

/**
 * @brief   Returns a value representing the duration of the given event
 * @param   event_ptr event
 * @return  duration
 */
U32DHMS ADB_GetEventDuration(void *event_ptr);

/**
 * @brief   Returns a value representing the date and time of the end of the given event
 * @param   event_ptr event
 * @return  date/time
 */
U32DHMS ADB_GetEventEndDateTime(void *event_ptr);

/**
 * @brief   Returns the MJD date code for the start of the given event
 * @param   event_ptr event
 * @return  date code
 */
U16BIT ADB_GetEventStartDate(void *event_ptr);

/**
 * @brief   Returns the hour (0-23) value for the start of the given event
 * @param   event_ptr event
 * @return  hour value (0-23)
 */
U8BIT ADB_GetEventStartHour(void *event_ptr);

/**
 * @brief   Returns the minute (0-59) value for the start of the given event
 * @param   event_ptr event
 * @return  minute value (0-59)
 */
U8BIT ADB_GetEventStartMin(void *event_ptr);

/**
 * @brief   Returns the seconds (0-59) value for the start of the given event
 * @param   event_ptr event
 * @return  seconds value (0-59)
 */
U8BIT ADB_GetEventStartSecs(void *event_ptr);

/**
 * @brief   Returns the hour value for the duration of the given event
 * @param   event_ptr event
 * @return  hour value
 */
U8BIT ADB_GetEventDurationHour(void *event_ptr);

/**
 * @brief   Returns the minute value (0-59) for the duration of the given event
 * @param   event_ptr event
 * @return  minute value (0-59)
 */
U8BIT ADB_GetEventDurationMin(void *event_ptr);

/**
 * @brief   Returns the seconds value (0-59) for the duration of the given event
 * @param   event_ptr event
 * @return  seconds value (0-59)
 */
U8BIT ADB_GetEventDurationSecs(void *event_ptr);

#if 0
/**
 * @brief   Returns end date/time for the given event
 * @param   event_ptr event
 * @param   end_date address of value to return the MJD end date value
 * @param   end_hour address of value to return the end hour value
 * @param   end_min address of value to return the end minute value
 * @return  TRUE if the event is valid and time is known, FALSE otherwise
 */
BOOLEAN ADB_GetEventEndDateTime(void *event_ptr, U16BIT *end_date, U8BIT *end_hour, U8BIT *end_min);
#endif

/**
 * @brief   Returns whether DVB subtitles are signalled as being available for the given event
 * @param   event_ptr event
 * @return  TRUE if subtitles are available, FALSE otherwise
 */
BOOLEAN ADB_GetEventSubtitlesAvailFlag(void *event_ptr);

/**
 * @brief   Returns the level 1 value of the first content identifier for the given event
 * @param   event_ptr event
 * @return  event content value, ADB_EVENT_CONTENT_UNCLASSIFIED if not available
 */
ADB_EVENT_CONTENT ADB_GetEventContentDesc(void *event_ptr);

/**
 * @brief   Returns the level 1 and 2 values for all content identifiers for the given event
 * @param   event_ptr event
 * @param   p_len address to return the number of bytes in the returned data
 * @return  pointer to the content data
 */
U8BIT* ADB_GetEventContentData(void *event_ptr, U8BIT *p_len);

/**
 * @brief   Returns the parental age value for the given event
 * @param   event_ptr event
 * @return  parental age value, 0 for invalid event
 */
U8BIT ADB_GetEventParentalAge(void *event_ptr);

/**
 * @brief   Returns the name of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventName(void *event_ptr);

/**
 * @brief   Returns the short event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventDescription(void *event_ptr);

/**
 * @brief   Checks whether the given event has any extended event descriptors
 * @param   event_ptr event
 * @return  TRUE if the event has at least one extended event descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetEventHasExtendedDescription(void *event_ptr);

/**
 * @brief   Returns the extended event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventExtendedDescription(void *event_ptr);

/**
 * @brief   Returns the guidance text for an event, either from the event itself
 *          or the event's service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   event_ptr event, can be NULL, in which case any guidance defined by the
 *                    service will be returned
 * @param   serv_ptr service, can be NULL to just get guidance text from the event
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* ADB_GetEventGuidance(void *event_ptr, void *serv_ptr, U8BIT *type, U8BIT *mode);

/**
 * @brief   Returns the event id for the given event
 * @param   event_ptr event
 * @return  event id
 */
U16BIT ADB_GetEventId(void *event_ptr);

/**
 * @brief   Returns whether audio description is signalled as being available for the given event
 * @param   event_ptr event
 * @return  TRUE if available, FALSE otherwise
 */
BOOLEAN ADB_GetEventAudioDescriptionFlag(void *event_ptr);

/**
 * @brief   Returns whether the event is signalled with none of its streams being CA scrambled
 * @param   event_ptr pointer to the event to check
 * @return  TRUE if the event all streams are free-to-air, FALSE otherwise
 */
BOOLEAN ADB_GetEventFreeToAir(void *event_ptr);

/**
 * @brief   Returns the do_not_scramble flag for the event, which will have been
 *          found by looking backwards in the event, SDT, transport and NIT, until
 *          found or not.
 * @param   event_ptr pointer to the event to check
 * @return  FALSE if the event is to be protected, TRUE otherwise
 */
BOOLEAN ADB_GetEventDoNotScramble(void *event_ptr);

/**
 * @brief   Returns whether there's an HD event linked to the SD event and returns
 *          the info for it. The DVB info is checked to ensure it relates to a known
 *          service.
 * @param   event_ptr pointer to the event to check
 * @param   verify_event TRUE, if the schedule should be checked for the target event
 * @param   only_simulcast TRUE, if only an HD simulcast event should be returned
 * @param   hd_serv_ptr pointer to return the service for the HD event, if found.
 *          This can be NULL if not required.
 * @param   hd_event_ptr pointer to return the HD event, if found. Note that this will only be
 *          filled in if verify_event is TRUE. This can be NULL if not required.
 * @return  TRUE if an event is found, FALSE otherwise
 */
BOOLEAN ADB_GetEventHDLinkageInfo(void *event_ptr, BOOLEAN verify_event, BOOLEAN only_simulcast,
   void **hd_serv_ptr, void **hd_event_ptr);

/**
 * @brief   Checks whether the two events are the same.
 *          Not all fields are checked, just date, time, duration and event ID
 * @param   event1_ptr pointer to the first of the events to check
 * @param   event2_ptr pointer to the second of the events to check
 * @return  TRUE if the two events are the same, FALSE otherwise
 */
BOOLEAN ADB_IsSameEvent(void *event1_ptr, void *event2_ptr);

/**
 * @brief   Returns the full programme CRID of the given event (including IMI).
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a programme CRID
 */
U8BIT* ADB_GetEventFullProgrammeCrid(void *serv_ptr, void *event_ptr);

/**
 * @brief   Returns the programme CRID of the given event (excluding IMI)
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a programme CRID
 */
U8BIT* ADB_GetEventProgrammeCrid(void *serv_ptr, void *event_ptr);

/**
 * @brief   Returns the full CRID for the given CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_str CRID string
 * @return  pointer to a string containing the CRID name
 */
U8BIT* ADB_GetFullCrid(void *serv_ptr, U8BIT *event_str);

/**
 * @brief   Returns the number of series CRIDs for the given event
 * @param   event_ptr pointer to the event
 * @return  number of series CRIDs
 */
U8BIT ADB_GetEventNumSeriesCrids(void *event_ptr);

/**
 * @brief   Returns the full series CRID of the given event
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   index 0 based index of series CRID to be returned
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a series CRID
 */
U8BIT* ADB_GetEventSeriesCrid(U8BIT index, void *serv_ptr, void *event_ptr);

/**
 * @brief   Returns the episode name of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  TRUE: valid series info, FALSE: invalid series info
 */
BOOLEAN ADB_GetEventSeriesInfo(void *event_ptr, U8BIT **episode_name, S32BIT *episode_number, U8BIT *content_status, BOOLEAN *last_episode);

/**
 * @brief   Returns the number of recommendation CRIDs for the given event
 * @param   event_ptr pointer to the event
 * @return  number of recommendation CRIDs
 */
U8BIT ADB_GetEventNumRecommendationCrids(void *event_ptr);

/**
 * @brief   Returns the full recommendation CRID of the given event
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   index 0 based index of recommendation CRID to be returned
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a CRID
 */
U8BIT* ADB_GetEventRecommendationCrid(U8BIT index, void *serv_ptr, void *event_ptr);

/**
 * @brief   Retrieves a list of components associated with the specified event, as described by
 *          component descriptors in the EIT.
 * @param   event_ptr Pointer to the event
 * @param   component_list Pointer to the returned list of components. The list must be freed by
 *          calling STB_AppFreeMemory, provided the returned number of components is greater than 0.
 * @return  Number of components associated with the specified event.
 */
U8BIT ADB_GetEventComponentList(void *event_ptr, ADB_EVENT_COMPONENT_INFO **component_list);

/**
 * @brief   Returns TRUE if the given CRID represents a split event (i.e. it
 *          contains an Instance Metadata Identifier (IMI)).
 * @param   crid CRID string
 * @return  TRUE if an IMI is found
 */
BOOLEAN ADB_IsSplitProgrammeCrid(U8BIT *crid);

/**
 * @brief   Returns a copy of the authority part of the CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   crid CRID string
 * @return  authority string
 */
U8BIT* ADB_GetAuthorityFromCrid(U8BIT *crid);

/**
 * @brief   Finds an alternative event for the given programme CRID and returns
 *          a pointer to a copy of the event and a pointer to the service
 * @param   prog_crid programme CRID to search for
 * @param   original_event_ptr pointer to the event for which an alternative is being sought, can be NULL
 * @param   serv_ptr pointer to return the service record if an alternate event is found
 * @return  pointer to a copy of the event, or NULL
 */
void* ADB_FindEventFromCrid(U8BIT *prog_crid, void *original_event_ptr, void **serv_ptr);

/**
 * @brief   Returns the favourite group value for the given service. This is an 8-bit
 *          value that can define whether the service is in any of upto 8 favourite
 *          groups. The value is a bit flag, so 1 means it's in group 1, 2 means it's
 *          in group 2, 4 means it's in group 3, 7 means it's in groups 1, 2 and 3, etc.
 * @param   s_ptr service
 * @return  favourite group bit value
 */
U8BIT ADB_GetServiceFavGroups(void *s_ptr);

/**
 * @brief   Sets the favourite group value for the given service. This is an 8-bit
 *          value that can define whether the service is in any of upto 8 favourite
 *          groups. The value is a bit flag, so 1 means it's in group 1, 2 means it's
 *          in group 2, 4 means it's in group 3, 7 means it's in groups 1, 2 and 3, etc.
 * @param   s_ptr service
 * @param   groups bit value, where a 1 means the service is in a group and 0 that it isn't
 */
void ADB_SetServiceFavGroups(void *s_ptr, U8BIT groups);

#if 0
/**
 * @brief   Removes the given service from the favourite groups defined by the fgrp_bit value.
 *          A '1' bit in the removes the service from that group and a '0' bit
 *          means there's no change.
 * @param   s_ptr service
 * @param   fgrp_bit bit value, where a 1 means the service is removed from a group
 */
void ADB_UnsetServiceFavGroupBit(void *s_ptr, U8BIT fgrp_bit);

/**
 * @brief   Sets the fav group bit setting of the specified service list according to settings of
 *          the tag array.
 *          1: Only the bit settings corresponding to non-zero values in the tag array will be set
 *          2: To set the fav group bit for the whole service database, then tag_array, slist and
 *          num_entries must all be set to NULL.
 * @param   tag_array array of tag values
 * @param   slist service list pointer
 * @param   num_entries service list size
 * @param   fgrp_bit value of fav group bit to be set
 */
void ADB_SetServiceTaggedFavGroupBit(U32BIT *tag_array, void **slist, U16BIT num_entries, U8BIT fgrp_bit);

/**
 * @brief   Unsets the fav group bit setting of the specified service list according to settings of
 *          the tag array.
 *          1: Only the bit settings corresponding to non-zero values in the tag array will be unset
 *          2: To unset the fav group bit for the whole database, then tag_array, slist and
 *          num_entries must all be set to NULL.
 * @param   tag_array array of tag values
 * @param   slist service list pointer
 * @param   num_entries service list size
 * @param   fgrp_bit value of fav group bit to be unset
 */
void ADB_UnsetServiceTaggedFavGroupBit(U32BIT *tag_array, void **slist, U16BIT num_entries, U8BIT fgrp_bit);
#endif

/**
 * @brief   Explicitly sets or clears the stream that will be used for audio on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and types, but if FALSE then the audio stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code audio language to be used for the selected stream if 'valid' is TRUE
 * @param   audio_type audio type to be used for the selected stream if 'valid' is TRUE
 * @param   stream_type stream type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdAudioStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code,
   ADB_AUDIO_TYPE audio_type, ADB_STREAM_TYPE stream_type);

/**
 * @brief   Explicitly sets or clears the stream that will be used for audio on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given PID, but if FALSE then the audio stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   pid PID to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdAudioStreamPID(void *s_ptr, BOOLEAN valid, U16BIT pid);

/**
 * @brief   Returns the settings defined by ADB_SetReqdAudioStreamSettings that will be
 *          used for audio on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set audio language code
 * @param   audio_type returns the set audio type
 * @param   stream_type returns the set stream type
 */
void ADB_GetReqdAudioStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code,
   ADB_AUDIO_TYPE *audio_type, ADB_STREAM_TYPE *stream_type);

/**
 * @brief   Explicitly sets or clears the stream that will be used for subtitles on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and type, but if FALSE then the stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code language to be used for the selected stream if 'valid' is TRUE
 * @param   subt_type subtitle type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdSubtitleStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code, ADB_SUBTITLE_TYPE subt_type);

/**
 * @brief   Returns the settings defined by ADB_SetReqdSubtitleStreamSettings that will be
 *          used for subtitles on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set language code
 * @param   subt_type returns the set subtitle type
 */
void ADB_GetReqdSubtitleStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code, ADB_SUBTITLE_TYPE *subt_type);

/**
 * @brief   Explicitly sets or clears the stream that will be used for teletext on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and type, but if FALSE then the stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code language to be used for the selected stream if 'valid' is TRUE
 * @param   ttext_type teletext type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdTeletextStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code, ADB_TELETEXT_TYPE ttext_type);

/**
 * @brief   Returns the settings defined by ADB_SetReqdTeletextStreamSettings that will be
 *          used for teletext on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set language code
 * @param   ttext_type returns the set teletext type
 */
void ADB_GetReqdTeletextStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code, ADB_TELETEXT_TYPE *ttext_type);

/**
 * @brief   Adds a CRID record to the database; use ADB_SaveCridRecord to store it
 *          If a CRID record with the given string already exists then a new record won't be
 *          created and the handle of the existing record will be returned.
 * @param   crid CRID string to be added
 * @param   series TRUE if this is a series CRID
 * @param   recommended TRUE if this is a recommended programme CRID
 * @return  CRID record handle, NULL if creation fails
 */
void* ADB_AddCridRecord(U8BIT *crid, BOOLEAN series, BOOLEAN recommended);

/**
 * @brief   Saves a CRID record to non-volatile storage
 * @param   c_ptr pointer to CRID record
 */
void ADB_SaveCridRecord(void *c_ptr);

/**
 * @brief   Searches for and returns the CRID record with the given CRID string
 * @param   crid_str pointer to CRID string
 * @return  pointer to the CRID record if one is found, or NULL
 */
void* ADB_FindCridRecord(U8BIT *crid_str);

/**
 * @brief   Sets the date and time fields on the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   datetime date and time to be saved in the record
 */
void ADB_SetCridDateTime(void *c_ptr, U32DHMS datetime);

/**
 * @brief   Sets the service ID on the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   serv_id service id
 */
void ADB_SetCridService(void *c_ptr, U16BIT serv_id);

/**
 * @brief   Sets the programme name field of the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   prog_name pointer to name string, the name is copied
 */
void ADB_SetCridProgrammeName(void *c_ptr, U8BIT *prog_name);

/**
 * @brief   Returns the programme name field of the given CRID record.
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to the programme name
 */
U8BIT* ADB_GetCridProgrammeName(void *c_ptr);

/**
 * @brief   Returns TRUE if the CRID record given represents a programme (split event)
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a programme
 */
BOOLEAN ADB_IsProgrammeCrid(void *c_ptr);

/**
 * @brief   Returns TRUE if the CRID record given represents a series
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a series
 */
BOOLEAN ADB_IsSeriesCrid(void *c_ptr);

/**
 * @brief   Returns TRUE if the CRID record given represents a recommendation
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a recommended event/series
 */
BOOLEAN ADB_IsRecommendationCrid(void *c_ptr);

/**
 * @brief   Sets the do not delete flag for the given CRID
 * @param   c_ptr pointer to CRID record
 * @param   do_not_delete value to set the flag to
 */
void ADB_SetCridDoNotDelete(void *c_ptr, BOOLEAN do_not_delete);

/**
 * @brief   Returns the value of the do not delete flag for the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  value of the flag, FALSE if the CRID record isn't valid
 */
BOOLEAN ADB_GetCridDoNotDelete(void *c_ptr);

/**
 * @brief   Updates the time the CRID was last seen in the EIT
 * @param   c_ptr pointer to CRID record
 */
void ADB_UpdateCridEitDate(void *c_ptr);

/**
 * @brief   Returns the CRID string from the given CRID record.
 *          The returned value should not be freed.
 * @param   c_ptr pointer to CRID record
 * @return  pointer to the CRID string
 */
U8BIT* ADB_GetCridString(void *c_ptr);

/**
 * @brief   Returns a value representing the date & time held in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  date/time value
 */
U32DHMS ADB_GetCridDateTime(void *c_ptr);

/**
 * @brief   Returns the service ID held in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  service id, or ADB_INVALID_DVB_ID if crid not valid
 */
U16BIT ADB_GetCridService(void *c_ptr);

/**
 * @brief   Returns the EIT update date contained in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  GMT date from the record
 */
U16BIT ADB_GetCridEitDate(void *c_ptr);

/**
 * @brief   Deletes the given CRID record from the database
 * @param   c_ptr pointer to CRID record
 */
void ADB_DeleteCridRecord(void *c_ptr);

/**
 * @brief   Creates an array of CRID record pointers to pass back to the caller
 *          with the number of entries in the array. Memory is allocated from
 *          temporary memory, so will be released when the calling screen closes.
 * @param   clist_ptr address for the return of the CRID array
 * @param   num_entries_ptr address for the return of the number of entries
 * @param   inc_series_crids TRUE if returned list is to include any series CRIDs
 * @param   inc_rec_crids TRUE if returned list is to include any recommendation CRIDs
 * @param   inc_prog_crids TRUE if returned list is to include any programme CRIDs
 */
void ADB_GetCridRecordList(void ***clist_ptr, U16BIT *num_entries_ptr, BOOLEAN inc_series_crids,
   BOOLEAN inc_rec_crids, BOOLEAN inc_prog_crids);

/**
 * @brief   Frees memory allocated for a CRID list due to a call to ADB_GetCridRecordList
 * @param   crid_list crid array
 * @param   num_entries number of items in the array
 */
void ADB_ReleaseCridRecordList(void **crid_list, U16BIT num_entries);

/**
 * @brief   Returns an array of available profiles. The array will be allocated
 *          within the function and should be released using ADB_ReleaseProfileList.
 *          This is a CI+ feature.
 * @param   profile_list pointer to the returned array of profiles.
 *                         Value isn't valid if 0 is returned
 * @param   active_profile pointer to return the index of the currently active profile
 * @return  Number of profiles
 */
U16BIT ADB_GetProfileList(void ***profile_list, U16BIT *active_profile);

/**
 * @brief   Frees a profile list acquired using ADB_GetProfileList
 * @param   profile_list pointer to the array of profiles.
 * @param   num_profiles number of profiles in the array
 */
void ADB_ReleaseProfileList(void **profile_list, U16BIT num_profiles);

/**
 * @brief   Returns the type of the given profile
 * @param   profile profile handle
 * @return  profile type
 */
ADB_PROFILE_TYPE ADB_GetProfileType(void *profile);

/**
 * @brief   Returns the default list type for the current profile. This function returns
 *          ADB_SERVICE_LIST_ALL if no CI+ profile is active, otherwise the favourite list
 *          containing the profile channels.
 * @return  Default service list type fot the current profile.
 */
ADB_SERVICE_LIST_TYPE ADB_GetServiceListTypeForCurrentProfile(void);

#ifdef COMMON_INTERFACE
/**
 * @brief   Finds the profile for the given CI+ operator module
 * @param   module operator module
 * @return  Profile handle, NULL if not found
 */
void* ADB_FindProfileForModule(U32BIT module);

/**
 * @brief   Deletes all records relating to the given profile from the database
 * @param   profile profile handle to be deleted
 * @return  TRUE if the profile is deleted, FALSE otherwise
 */
BOOLEAN ADB_DeleteProfile(void *profile);

/**
 * @brief   Returns the name of the given profile as a unicode string.
 *          The returned string must be freed using STB_ReleaseUnicodeString.
 * @param   profile profile handle
 * @return  profile name, can be NULL
 */
U8BIT* ADB_GetProfileName(void *profile);

/**
 * @brief   Mark CI module present for network specified by
 *          original network ID and CAM ID
 * @param   cicam_onet_id returned original network ID
 * @param   cicam_id returned CAM identifier
 */
void ADB_MarkCIModulePresent(U16BIT cicam_onet_id, U32BIT cicam_id);

/**
 * @brief   Finds the profile for the given CI+ operator module,
 *          and marks it not present
 * @param   module operator module
 */
void ADB_MarkCIModuleNotPresent(U32BIT module);

/**
 * @brief   Returns whether the CI+ module for the given profile is available
 * @param   profile profile handle
 * @return  TRUE if present, FALSE otherwise
 */
BOOLEAN ADB_GetProfileModulePresent(void *profile);

/**
 * @brief   Returns the operator module handle of the given profile
 * @param   profile profile handle
 * @return  module handle, or 0 if not available
 */
U32BIT ADB_GetProfileModule(void *profile);

/**
 * @brief   Returns the original network ID and CAM ID of the given profile
 * @param   profile profile handle
 * @param   cicam_onet_id returned original network ID
 * @param   cicam_id returned CAM identifier
 * @return  TRUE if the values are returned, FALSE otherwise
 */
BOOLEAN ADB_GetProfileCAMInfo(void *profile, U16BIT *cicam_onet_id, U32BIT *cicam_id);

/**
 * @brief   Set the flag indicating whether this operator profile needs to be updated with a scan
 * @param   profile profile handle
 * @param   search_required flag to be set
 * @param   date UTC MJD date code, if a scheduled search is required, 0 if not
 * @param   hour UTC hour, if a scheduled search is required, 0 if not
 * @param   min UTC minute, if a scheduled search is required, 0 if not
 * @return  TRUE if successful, FALSE if the profile is not valid
 */
BOOLEAN ADB_SetProfileSearchRequired(void *profile, BOOLEAN search_required, U16BIT date,
   U8BIT hour, U8BIT min);

/**
 * @brief   Returns whether the given operator profile requires a scan in order to be updated
 * @param   profile profile handle
 * @return  TRUE if a scan is required, FALSE otherwise
 */
BOOLEAN ADB_GetProfileSearchRequired(void *profile);

/**
 * @brief   Returns the scheduled operator search date/time for the given profile
 * @param   profile profile handle
 * @param   date pointer to return the date code as MJD
 * @param   hours pointer to return the search hour in GMT
 * @param   mins pointer to return the search minute in GMT
 * @return  TRUE if the profile is valid and a scheduled date/time is returned, FALSE otherwise
 */
BOOLEAN ADB_GetProfileSearchDateTime(void *profile, U16BIT *date, U8BIT *hours, U8BIT *mins);

/**
 * @brief   Sets the timer handle being used for the scheduled profile search
 * @param   profile profile handle
 * @param   timer_handle timer handle
 */
void ADB_SetProfileSearchTimer(U32BIT cicam_id, U32BIT timer_handle);

/**
 * @brief   Returns the operator search timer handle of the given profile
 * @param   profile profile handle
 * @return  timer handle, or INVALID_TIMER_HANDLE if not available
 */
U32BIT ADB_GetProfileSearchTimer(void *profile);

/**
 * @brief   Deletes the CICAM timer record with the specified timer handle
 * @param   timer_handle Timer handle to be searched for
 */
void ADB_DeleteCicamTimerbyHandle(U32BIT timer_handle);

/**
 * @brief   Returns whether the SDT has been received for the given service
 * @param   s_ptr service pointer
 * @return  TRUE if the SDT has been received
 */
BOOLEAN ADB_GetServiceSDTReceived(void *s_ptr);

/**
 * @brief   Returns the current CI protection descriptor for the given service
 * @param   s_ptr service pointer
 * @return  Protection descriptor data or NULL if not available
 */
U8BIT* ADB_GetServiceCIProtectionDesc(void *s_ptr);

/**
 * @brief   Checks whether the Ci protection desc needs to be updated for the given service
 * @param   s_ptr service pointer
 * @return  TRUE if the descriptor has expired and needs to be updated
 */
BOOLEAN ADB_HasCIProtectionExpired(void *s_ptr);
#endif

/**
 * @brief   Returns the current PMT data for the given service
 * @param   s_ptr service pointer
 * @param   data_len pointer to return the size of the PMT data, only valid
 *          return value isn't NULL
 * @return  PMT data or NULL if not available
 */
U8BIT* ADB_GetServicePMTData(void *s_ptr, U16BIT *data_len);

/**
 * @brief   Returns the current PMT PID for the given service
 * @param   s_ptr service pointer
 * @return  PMT PID
 */
U16BIT ADB_GetServicePmtPid(void *s_ptr);

/**
 * @brief   Returns the current AIT PID for the given service
 * @param   s_ptr service pointer
 * @return  AIT PID; zero PID means AIT table not present
 */
U16BIT ADB_GetServiceAitPid(void *s_ptr);

/**
 * @brief   Imports the database from an XML file
 * @param   filename full pathname of the file to import from
 * @return  TRUE if successful, otherwise FALSE
 */
BOOLEAN ADB_ImportDB(U8BIT *filename);

/**
 * @brief   Exports the database as an XML file
 * @param   filename full pathname of the file to export to
 * @return  TRUE if successful, otherwise FALSE
 */
BOOLEAN ADB_ExportDB(U8BIT *filename);

/**
 * @brief   Exports the current event schedule as an XML file
 * @param   filename full pathname of the file to export to
 * @param   utc_date_filter If non-zero, only events matching date will be exported
 * @param   type The service type(s) to export
 * @param   use_dvb_uri TRUE to use dvb:// uri for id, otherwise allocated lcn
 * @return  TRUE if successful, otherwise FALSE
 */
BOOLEAN ADB_ExportEPG(U8BIT *filename, U16BIT utc_date_filter, U32BIT type, BOOLEAN use_dvb_uri);

/**
 * @brief   Returns the number of favourite lists
 * @return  number of favourite lists
 */
U16BIT ADB_GetNumFavouriteLists(void);

/**
 * @brief   Creates a new favourite list and adds it to the existing lists, if any.
 * @param   name name to be given to the list
 * @param   user_data user defined value to be stored in the list
 * @param   index 0 based index where the new list should be inserted in any existing list.
 *          0 = at start, negative = at end, any other value = position
 * @param   list_id used to get the unique id assigned to the list, can be NULL
 * @return  TRUE if the list is created successfully, FALSE otherwise
 */
BOOLEAN ADB_AddFavouriteList(U8BIT *name, U32BIT user_data, S16BIT index, U8BIT *list_id);

/**
 * @brief   Returns the list id of the favourite list defined by the given index
 * @param   index list index
 * @return  list id, 0 if list isn't found
 */
U8BIT ADB_GetFavouriteListIdByIndex(U16BIT index);

/**
 * @brief   Returns the list id of the favourite list with the given user data value
 * @param   user_data user data value
 * @return  list id, 0 if list isn't found
 */
U8BIT ADB_GetFavouriteListByUserData(U32BIT user_data);

/**
 * @brief   Returns the name of the given favourite list, as a Unicode string
 * @param   list_id list id
 * @return  name of the favourite list as a unicode string, which must be
 *          released using STB_ReleaseUnicodeString
 */
U8BIT* ADB_GetFavouriteListName(U8BIT list_id);

/**
 * @brief   Set the name of the given favourite list
 * @param   list_id list id
 * @param   name of the favourite list
 * @return  TRUE if the name is set successfully, FALSE otherwise
 */
BOOLEAN ADB_SetFavouriteListName(U8BIT list_id, U8BIT *name);

/**
 * @brief   Returns the user data value of the given favourite list
 * @param   list_id list id
 * @return  user data value, will be 0 if list not found
 */
U32BIT ADB_GetFavouriteListUserData(U8BIT list_id);

/**
 * @brief   Set the user data of the given favourite list
 * @param   list_id list id
 * @param   user_data value
 */
void ADB_SetFavouriteListUserData(U8BIT list_id, U32BIT user_data);

/**
 * @brief   Change the order of the favourite lists by moving the given list to
 *          the given position
 * @param   list_id favourite list to be moved
 * @param   index position to move the list to, negative value will move the list to the end
 * @return  TRUE if the list is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListTo(U8BIT list_id, S16BIT index);

/**
 * @brief   Deletes the favourite list with the given list id
 * @param   list_id id of the favourite list to be deleted
 */
void ADB_DeleteFavouriteList(U8BIT list_id);

/**
 * @brief   Adds the given service to the favourite list defined by list_id,
 *          with the service being optionally added at a given position
 * @param   list_id favourite list id
 * @param   serv_ptr service to be added
 * @param   index position to add the service in the list,
 *          0 = at start, negative = at end, any other value is the position
 * @return  TRUE if the service is successfully added, FALSE otherwise
 */
BOOLEAN ADB_AddServiceToFavouriteList(U8BIT list_id, void *serv_ptr, S16BIT index);

/**
 * @brief   Change the order of the services in the given favourite lists by
 *          moving it to the given position
 * @param   list_id favourite list to be moved
 * @param   serv_ptr service to be moved
 * @param   index position in the list to move the service to,
 *          negative value will move service to the end of the list
 * @return  TRUE if the service is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListServiceTo(U8BIT list_id, void *serv_ptr, S16BIT index);

/**
 * @brief   Change the order of the services in the given favourite lists by
 *          moving it to the given position
 * @param   list_id favourite list to be moved
 * @param   current_index index of the service to be moved
 * @param   new_index position in the list to move the service to,
 *          negative value will move service to the end of the list
 * @return  TRUE if the service is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListServiceFromTo(U8BIT list_id, S16BIT current_index, S16BIT new_index);

/**
 * @brief   Removes the given service from the favourite list defined by the list id
 * @param   list_id favourite list id
 * @param   serv_ptr service to be removed from the list
 */
void ADB_DeleteServiceFromFavouriteList(U8BIT list_id, void *serv_ptr);

/**
 * @brief   Removes all services from the favourite list defined by list_id
 * @param   list_id favourite list id
 */
void ADB_DeleteAllServicesFromFavouriteList(U8BIT list_id);

#ifdef DEBUG_PRINT_DATABASE
/**
 * @brief   Outputs the contents of the service database to the console in human readable form
 */
void ADB_PrintDatabase(void);
#endif


//---Global Function prototypes for internal use---------------------------------
/**
 * @brief   Initialises database access
 */
void ADB_Initialise(void);

#if 0
/**
 * @brief   Reloads the contents of the database back into RAM
 */
void ADB_ReloadDBRam(void);
#endif

/**
 * @brief   Sets up the database for a search
 * @param   tuner_type type of tuner that will be used when doing the search
 * @param   satellite satellite that the search will be performed on, NULL if not relevant
 * @param   retune TRUE if a retune will be performed
 * @param   manual_search TRUE for manual search, FALSE otherwise
 */
void ADB_PrepareDatabaseForSearch(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN retune, BOOLEAN manual_search);

/**
 * @brief   Completes the database setup after a search has completed or been stopped
 * @param   save_changes TRUE if the modified database is to be saved
 * @param   tuner_type type of tuner that was used when doing the search
 * @param   satellite satellite that the search was performed on, NULL if not relevant
 * @param   search_completed TRUE if the search was completed, FALSE if terminated before completion
 * @param   clear_new_flags TRUE if all new flags are to be cleared (e.g. after a full search rather
 *          than an update search)
 * @param   manual_search TRUE for manual searches, FALSE otherwise
 */
void ADB_FinaliseDatabaseAfterSearch(BOOLEAN save_changes, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN search_completed, BOOLEAN clear_new_flags, BOOLEAN manual_search);

/**
 * @brief   Checks through all services after a service search and determines whether any of them
 *          require the same LCN. This function assumes that ADB_FinaliseDatabaseAfterSearch has
 *          already been called.
 * @return  TRUE if duplicates are found, FALSE otherwise
 */
BOOLEAN ADB_AreLcnsDuplicated(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Allocates LCNs to services after a service search. This function assumes that
 *          ADB_FinaliseDatabaseAfterSearch has already been called.
 */
void ADB_AllocateLcns(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Frees all data that's only required for service search. This should be called after
 *          the search is complete and LCNs have been assigned, etc.
 */
void ADB_ReleaseDatabaseSearchData(void);

/**
 * @brief   Set the signal level to 0 and searched flag to TRUE for the given transport
 * @param   t_ptr pointer to the transport
 */
void ADB_ReportNoSignalDuringSearch(void *t_ptr);

/**
 * @brief   Returns whether a transport has been used during a service scan operation
 * @param   t_ptr transport
 * @return  TRUE if the transport has been used, FALSE otherwise
 */
BOOLEAN ADB_GetTransportSearchFlag(void *t_ptr);

/**
 * @brief   Sets whether a transport has been used during a service scan operation
 * @param   t_ptr transport
 * @param   state set the flag to TRUE or FALSE
 */
void ADB_SetTransportSearchFlag(void *t_ptr, BOOLEAN state);

#if 0
void ADB_SetServiceListTaggedLockedFlag(U32BIT *tag_array, void **slist, U16BIT num_entries, BOOLEAN locked);
#endif

/**
 * @brief   Derive audio codec from stream
 * @param   audio_stream_type from which to derive the audio codec
 * @return  audio codec best corresponding to audio_stream_type
 */
E_STB_DP_AUDIO_CODEC ADB_GetAudioCodecFromStream(ADB_STREAM_TYPE audio_stream_type);

/**
 * @brief   Derive video codec from stream
 * @param   video_stream_type from which to derive the video codec
 * @return  video codec best corresponding to video_stream_type
 */
E_STB_DP_VIDEO_CODEC ADB_GetVideoCodecFromStream(ADB_STREAM_TYPE video_stream_type);

/**
 * @brief   Returns the service pointer of event
 * @param   event_ptr event pointer
 * @return  service pointer of event
 */
void* ADB_GetEventService(void *event_ptr);

/**
 * @brief   Returns the items of extended event descriptor as item descriptor and item itself.
 * @param   event_ptr event
 * @param   num_items_ptr address in which to return the number of items
 * @return  ADB_EVENT_ITEMIZED_INFO pointer to the list of items, NULL on error
 */
ADB_EVENT_ITEMIZED_INFO* ADB_GetEventItemizedDescription(void *event_ptr, U16BIT *num_items_ptr);

/**
 * @brief   Frees the memory used by event itemized data
 * @param   event_itemized_info pointer of info
 * @param   num_of_items number of items to release
 */
void ADB_ReleaseEventItemizedInfo(ADB_EVENT_ITEMIZED_INFO *event_itemized_info, U16BIT num_of_items);

#endif //  _AP_DBACC_H
