/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   macros and function prototypes for public use
 * @file    ap_epgsearch.h
 * @date    13/08/2011
 * @author  Ocean Blue
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _AP_EPGSEARCH_H
#define _AP_EPGSEARCH_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------
typedef enum e_ags_match_type
{
   AGS_MATCH_ANY_WORD = 0,        // EVENT CAN CONTAIN ANY WORD IN STRING
   AGS_MATCH_ALL_WORDS = 1,       // EVENT MUST CONTAIN ALL WORDS IN STRING (IN ANY ORDER)
   AGS_MATCH_EXACT_PHRASE = 2,    // EVENT MUST EXACTLY MATCH PHRASE (WORDS IN ORDER)
   AGS_MATCH_CONTAINS_PHRASE = 3, // EVENT MUST CONTAIN PHRASE (WORDS IN ORDER)
   AGS_MATCH_NOT_EQUAL = 4,       // OPPOSITE OF EXACT_PHRASE

   AGS_MATCH_IGNORE_CASE = 0x80
}
E_AGS_MATCH_TYPE;

typedef enum e_ags_match_action
{
   AGS_ACTION_NONE = 0,
   AGS_ACTION_SET_TIMER = 1,
   AGS_ACTION_SET_ALARM = 2
}
E_AGS_MATCH_ACTION;

typedef enum e_ags_search_fields
{
   AGS_SEARCH_ALL = 0xff,      // for convenience
   // These can be | (ORed) together
   AGS_SEARCH_TITLE = 0x01,
   AGS_SEARCH_DESC = 0x02,
   AGS_SEARCH_EXT_DESC = 0x04
}
E_AGS_SEARCH_FIELDS;

typedef enum e_ags_search_days
{
   AGS_DAYS_ALL = 0xff,
   AGS_DAYS_WEEKDAYS = 0x1f,  // for convenience
   AGS_DAYS_WEEKENDS = 0x60,  // for convenience
   // These can be | (ORed) together
   AGS_DAYS_MONDAY = 0x01,
   AGS_DAYS_TUESDAY = 0x02,
   AGS_DAYS_WEDNESDAY = 0x04,
   AGS_DAYS_THURSDAY = 0x08,
   AGS_DAYS_FRIDAY = 0x10,
   AGS_DAYS_SATURDAY = 0x20,
   AGS_DAYS_SUNDAY = 0x40
}
E_AGS_SEARCH_DAYS;

//---Global type defs for public use-------------------------------------------
typedef struct s_ags_time_range
{
   U8BIT min_hour;
   U8BIT min_minutes;
   U8BIT min_seconds;
   U8BIT max_hour;
   U8BIT max_minutes;
   U8BIT max_seconds;
} S_AGS_TIME_RANGE;


//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Opens an EPG Search
 * @param   search_str The search string to compare
 * @param   search_fields The EPG parts to match
 * @param   match_type The type of comparison to make on the string
 * @param   search_days Which days of the week to search
 * @param   search_times Time range in which to search (NULL for any)
 * @param   match_action The action to perform on a search match
 * @return  Handle for search, NULL if failed
 */
void* AGS_OpenSearch( U8BIT *search_str, E_AGS_SEARCH_FIELDS search_fields, E_AGS_MATCH_TYPE match_type,
   E_AGS_SEARCH_DAYS search_days, S_AGS_TIME_RANGE *search_times, E_AGS_MATCH_ACTION match_action);

/**
 * @brief   Gets the next matching result for an open search
 * @param   ags_handle The handle of the open search
 * @param   event_id Caller's event id var to be set with matching result
 * @param   serv_ptr Caller's service pointer to be set with matching result
 * @return  BOOLEAN, TRUE=Match Found, FALSE=No more matches
 */
BOOLEAN AGS_GetNextResult( void *ags_handle, U16BIT *event_id, void **serv_ptr );

/**
 * @brief   Closes an open search and frees all associated resources
 * @param   ags_handle The handle of the open search
 */
void AGS_CloseSearch( void *ags_handle);

#endif // _AP_EPGSEARCH_H
