/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   macros and function prototypes for public use
 * @file    ap_pvr.h
 * @date    11/05/2004
 * @author  Ocean Blue
 */
// pre-processor mechanism so multiple inclusions don't cause compilation error

#ifndef _AP_PVR_H

#define _AP_PVR_H

#include "techtype.h"

//---macro definitions for this file-------------------------------------------


//---Enumerations for public use-----------------------------------------------
typedef enum e_pvr_play_status
{
   PVR_PLAY_STATUS_NULL = 0,       // X     - no action
   PVR_PLAY_STATUS_STOP = 1,       // .     - stop
   PVR_PLAY_STATUS_PAUSE = 2,      // ||    - pause
   PVR_PLAY_STATUS_SF = 3,         // >|    - slow forward
   PVR_PLAY_STATUS_SR = 4,         // |<    - slow reverse
   PVR_PLAY_STATUS_NORMAL = 5,     // >     - play
   PVR_PLAY_STATUS_FF = 6,         // >>    - fast forward
   PVR_PLAY_STATUS_FR = 7,         // <<    - fast reverse
   PVR_PLAY_STATUS_QR = 8          // |<<   - quick repeat
} E_PVR_PLAY_STATUS;

typedef enum e_pvr_stop_type
{
   PVR_STOP_REC_ONLY = 0,
   PVR_STOP_PLAY = 1,
   PVR_STOP_PAUSE = 2
} E_PVR_STOP_TYPE;

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Use to query whether the PVR function is available and ready to be used
 * @return  TRUE if available and ready, FALSE otherwise
 */
BOOLEAN APVR_IsInitialised(void);

/**
 * @brief   Sets the time, in seconds, when an event will be sent as a notification that a
 *          recording is about to start. This setting will apply to all recording timers that
 *          don't have a notify time already set. If this value is set to 0 then events will
 *          only be sent for recording timers that have their own notify time set.
 * @param   notify_time number of seconds before the recording is due to start when the event
 *                      will be sent. Setting 0 disables the notification.
 */
void APVR_SetNotifyTime(U16BIT notify_time);

/**
 * @brief   Returns the current recording notification time in seconds
 * @return  notification time in seconds, 0 means it's disabled
 */
U16BIT APVR_GetNotifyTime(void);

/**
 * @brief   Sets the default time that will be added to the start of all recording timers created
 *          after this value is set. Positive values will start a recording before the scheduled
 *          start time and negative values will start it later than the scheduled time
 * @param   padding start padding time in seconds
 */
void APVR_SetStartPadding(S32BIT padding);

/**
 * @brief   Sets the default time that will be added to the end of all recording timers created
 *          after this value is set. Positive values will end a recording after the scheduled
 *          end time and negative values will end it before the scheduled end time
 * @param   padding end padding time in seconds
 */
void APVR_SetEndPadding(S32BIT padding);

/**
 * @brief   Returns the current setting for the start padding time added to all recording timers
 * @return  start padding value in seconds
 */
S32BIT APVR_GetStartPadding(void);

/**
 * @brief   Returns the current setting for the end padding time added to all recording timers
 * @return  end padding value in seconds
 */
S32BIT APVR_GetEndPadding(void);

// ----play list----

/**
 * @brief   Returns a list of all the existing recordings that can be played.
 *          All the arrays returned will be allocated by this function and
 *          should be freed by calling APVR_ReleasePlayList.
 * @param   handle_list address to return an array of recording handles
 * @param   name_list address to return an array of names for each recording,
 *          the U32BIT value will actually be a U8BIT* of a UTF-8 string
 * @param   rec_status_list address to return an array indicating whether each
 *          recording is currently being recorded, 0=no, 1=yes
 * @param   locked_list address to return an array indicating whether each
 *          recording is parentally locked, 0=no, 1=yes
 * @param   selected_list address to return an array indicating whether each
 *          recording is marked as selected, 0=no, 1=yes
 * @param   split_list address to return an array indicating whether each
 *          recording is part of a split recording, 0=no, 1=yes
 * @return  number of recordings
 */
U16BIT  APVR_GetPlayList(U32BIT **handle_list, U8BIT ***name_list, U32BIT **rec_status_list,
   U32BIT **locked_list, U32BIT **selected_list, U32BIT **split_list);

/**
 * @brief   Frees the arrays allocated by APVR_GetPlayList
 * @param   handle_list array of recording handles
 * @param   name_list array of recording names
 * @param   rec_status_list array of recording status'
 * @param   locked_list array of locked status'
 * @param   selected_list array of selected status'
 */
void  APVR_ReleasePlayList(U32BIT *handle_list, U8BIT **name_list, U32BIT *rec_status_list,
   U32BIT *locked_list, U32BIT *selected_list, U32BIT *split_list);

// ----playing----

/**
 * @brief   Starts playback of the given recording
 * @param   recording_handle handle of recorded programme to playback
 * @param   resume_playback TRUE if playback of the recording should resume from where
 *                          it was last stopped, FALSE to start from the beginning
 * @param   monitor_service service to be tuned to in order to monitor SI data whilst playback
 *                          is in progress. If this is NULL and the live path is still valid
 *                          and tuned to a service then this will be used to monitor SI data.
 *                          If neither are valid then no SI monitoring will be performed, which
 *                          may result in recordings failing to be started if they're set to
 *                          be event triggered.
 * @return  TRUE if playback starts, FALSE otherwise
 */
BOOLEAN APVR_PlayRecording(U32BIT recording_handle, BOOLEAN resume_playback, void *monitor_service);

/**
 * @brief   Returns the handle of the recording currently being played back
 * @return  playback handle, which may be invalid if playback isn't taking place
 */
U32BIT APVR_GetPlaybackHandle(void);

/**
 * @brief   Returns whether playback is in progress
 * @return  TRUE if playback has been started, FALSE otherwise
 */
BOOLEAN APVR_IsPlaying(void);

/**
 * @brief   Returns the playback progress in time and as a percentage
 * @param   handle recording handle being played
 * @param   hours pointer for return of playback hours
 * @param   mins pointer for return of playback minutes
 * @param   secs pointer for return of playback seconds
 * @param   progress pointer for return of position in playback as a percentage
 * @return  TRUE if values are returned
 */
BOOLEAN APVR_GetPlaybackElapsedTime(U32BIT handle, U8BIT *hours, U8BIT *mins, U8BIT *secs, U8BIT *progress);

/**
 * @brief   Stops playback and optionally returns to live TV
 * @param   return_to_live TRUE to return to live TV after stopping playback
 */
void APVR_StopPlay(BOOLEAN return_to_live);

/**
 * @brief   If playback has been paused or is subject to some trick mode, calling this
 *          function will result in playback resuming at normal playback speed (i.e. 100%)
 */
void APVR_NormalPlay(void);

/**
 * @brief   Pause playback
 */
void APVR_PausePlay(void);

/**
 * @brief   If paused, playback will go into the first available slow motion speed
 *          in a forwards direction. If speed is already > 0% then the next fast
 *          forward speed will be selected. If playback is in the reverse direction
 *          then the next slowest reverse play speed will be selected.
 */
void APVR_FFPlay(void);

/**
 * @brief   If paused, playback will go into the first available slow motion speed
 *          in a reverse direction. If speed is already < 0% then the next reverse
 *          speed will be selected. If playback is in the forwards direction then
 *          the next slowest forwards play speed will be selected.
 */
void APVR_FRPlay(void);

/**
 * @brief   Returns the current play mode according to the current play speed
 * @return  play mode
 */
E_PVR_PLAY_STATUS APVR_GetPlayMode(void);

/**
 * @brief   Returns the current play speed as a signed percentage value representing
 *          the speed, where 100% is normal playback speed.
 * @return  play speed
 */
S16BIT APVR_GetPlaySpeed(void);

/**
 * @brief   Set current play speed as a signed percentage value representing
 *          the speed, where 100% is normal playback speed.
 * @param   speed Play speed as a percentage
 * @return  TRUE if successful
 */
BOOLEAN APVR_SetPlaySpeed(S16BIT speed);

// ---- recording ----

/**
 * @brief   Acquires a decode path for recording the given service and tunes to it
 * @param   onet_id original network id for the service
 * @param   trans_id transport id for the service
 * @param   service_id service id for the service
 * @param   new_tuned_service returns TRUE if tuning is started.
 *          FALSE means the path is already tuned to the service
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT APVR_PrepareNewRecording(U16BIT onet_id, U16BIT trans_id, U16BIT service_id, BOOLEAN *new_tuned_service);

/**
 * @brief   Starts a recording after any tuning has completed and sets the info to be stored with it
 * @param   disk_id disk to record onto, default disk will be used if given disk id is invalid
 * @param   path decode path to use for recording
 * @param   recording_name name to be given to recording
 * @param   event_id event id which will be used to obtain info to save with the recording
 * @param   prog_crid programme crid (only relevant for systems that broadcast CRIDs)
 * @param   other_crid series or recommendation crid
 * @param   rec_handle pointer to returned recording handle
 * @return  TRUE if the recording is started successfully, FALSE otherwise
 */
BOOLEAN APVR_StartNewRecording(U16BIT disk_id, U8BIT path, U8BIT *recording_name, U16BIT event_id,
   U8BIT *prog_crid, U8BIT *other_crid, U32BIT *rec_handle);

/**
 * @brief   Stops the given recording
 * @param   recording_handle recording to be stopped
 * @return  TRUE if recording stopped ok, FALSE otherwise
 */
BOOLEAN APVR_StopRecording(U32BIT recording_handle);

/**
 * @brief   Returns an array of recordings currently in progress.
 *          Use APVR_ReleaseActiveRecordingList to free the returned array.
 * @param   handle_list address of an array allocated by the function containing
 *          the recording handles
 * @return  number of recordings in progress
 */
U8BIT APVR_GetActiveRecordingList(U32BIT **handle_list);

/**
 * @brief   Frees the arrays allocated by APVR_GetActiveRecordingList
 * @param   handle_list array of recording handles
 */
void APVR_ReleaseActiveRecordingList(U32BIT *handle_list);

/**
 * @brief   Get the name of the recording currently taking place on the given decode path
 * @param   path decode path being used for a recording
 * @param   name_ptr address of variable into which the name reference will be returned.
 *          This is actually a U8BIT*, but the string shouldn't be freed.
 * @return  TRUE if a recording is taking place on the given path and the name is returned,
 *          FALSE otherwise
 */
BOOLEAN APVR_GetPathRecordingName(U8BIT path, U8BIT **name_ptr);

/**
 * @brief   Delete the given recording, including all files associated with it
 *          and remove it from the database of available recordings that can be
 *          played back
 * @param   handle recording to be deleted
 * @return  TRUE if the recording is deleted, FALSE otherwise
 */
BOOLEAN APVR_DeleteRecording(U32BIT handle);

/**
 * @brief   Sets the size of the timeshift buffer so that it's able to store
 *          a recording of the given length before it has to loop round. Whether
 *          this value can be accurately achieved may vary according to the support
 *          provided by the platform.
 * @param   time_in_mins timeshift buffer size in minutes
 */
void APVR_SetTimeshiftBufferSize(U16BIT time_in_mins);

/**
 * @brief   Returns the maximum size of the timeshift buffer in minutes
 * @return  timeshift buffer size in minutes
 */
U16BIT APVR_GetTimeshiftBufferSize(void);

/**
 * @brief   Checks whether timeshift has been started, which includes both recording and playback
 * @return  TRUE if timeshift is started, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftStarted(void);

/**
 * @brief   Checks whether the given path is the one that's being used for timeshift recording
 * @param   path decode path to be checked
 * @return  TRUE if the given path is the one being used for timeshift recording, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftRecordingPath(U8BIT path);

/**
 * @brief   Checks whether the given recording handle is the timeshift recording handle
 * @param   handle recording handle to be checked
 * @return  TRUE if the given recording handle is the timeshift recording handle, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftRecording(U32BIT handle);

/**
 * @brief   Starts recording the live service ready for timeshift playback.
 *          The recording may not have started when this function returns, so
 *          an app should wait for the STB_EVENT_PVR_REC_START event before
 *          starting the paused playback using APVR_StartPausePlay.
 * @return  decode path used for the pause recording, INVALID_RES_ID on error
 */
U8BIT APVR_StartPauseRecord(void);

/**
 * @brief   Starts playback of a live timeshift recording started with APVR_StartPauseRecord.
 *          This function should not be called until STB_EVENT_PVR_REC_START has been
 *          received with the correct recording path as an argument (as returned by
 *          APVR_StartPauseRecord).
 * @return  decode path used for the playback, INVALID_RES_ID on error
 */
U8BIT APVR_StartPausePlay(void);

/**
 * @brief   Stops timeshifted playback and recording and optionally restarts decoding
 *          of the live signal.
 * @param   return_to_live TRUE to start decoding the live service
 */
void APVR_StopPauseRecord(BOOLEAN return_to_live);

/**
 * @brief   Returns TRUE if there are any recordings currently in progress.
 * @return  TRUE if recording in progress, FALSE otherwise
 */
BOOLEAN APVR_IsRecordingInProgress(void);

/**
 * @brief   Returns TRUE if given recording handle is still in progress on the
 *          given decode path.
 * @param   path decode path used for recording
 * @param   recording_handle recording
 * @return  TRUE if the recording is still in progress, FALSE otherwise
 */
BOOLEAN APVR_IsRecordingHandle(U8BIT path, U32BIT recording_handle);

/**
 * @brief   Returns the handle of the recording taking place on the given decode path
 * @param   path decode path used for recording
 * @param   recording_handle_ptr address to return the recording handle,
 *          only valid on return if a recording is taking place
 * @return  TRUE if a recording is taking place and handle is returned, FALSE otherwise
 */
BOOLEAN APVR_GetRecordingHandle(U8BIT path, U32BIT *recording_handle_ptr);

/**
 * @brief   Returns the length in time and size in KB of the recording with the given handle
 * @param   handle - handle of recording being queried
 * @param   length_hours - length of the recording in hours, returned
 * @param   length_mins - length of the recording in minutes, returned
 * @param   length_secs - length of the recording in seconds, returned
 * @param   rec_size_kb - size of the recording in KB, returned
 * @return  TRUE if the length and size are being returned, FALSE otherwise
 */
BOOLEAN APVR_RecordingGetLength(U32BIT recording_handle, U8BIT *length_hours, U8BIT *length_mins,
    U8BIT *length_secs, U32BIT *rec_size_kb );

// ---- disk ----
/**
 * @brief   get total percentage of recording space used
 * @param   total_space_used total percentage of recording space used
 */
U8BIT APVR_GetTotalSpacePercentUsed(U16BIT disk_id);

/**
 * @brief   Gets an estimate of disk space in time (hour / min): used, free and size
 * @param   disk_id ID of disk to be queried
 * @param   used_hour_ptr pointer to the variable containing the number of hours that the stored data is worth
 * @param   used_min_ptr pointer to the variable containing the number of minutes that the stored data is worth
 * @param   free_hour_ptr pointer to the variable containing the number of hours that the available space is worth
 * @param   free_min_ptr pointer to the variable containing the number of minutes that the available space is worth
 * @param   size_hour_ptr pointer to the variable containing the number of hours that the total disk size is worth
 * @param   size_min_ptr pointer to the variable containing the number of minutes that the total disk size is worth
 */
void APVR_GetDiskTime(U16BIT disk_id, U8BIT *used_hour_ptr, U8BIT *used_min_ptr,
   U8BIT *free_hour_ptr, U8BIT *free_min_ptr,
   U8BIT *size_hour_ptr, U8BIT *size_min_ptr);

/**
 * @brief   Ges disk space in Mbytes: used, free and size
 * @brief   disk_id ID of disk to be queried
 * @brief   used_mbyte_ptr amount of data stored on the disk
 * @brief   free_mbyte_ptr free space on the disk
 * @brief   size_mbyte_ptr total size of the disk
 */
void APVR_GetDiskMbyte(U16BIT disk_id, U32BIT *used_mbyte_ptr, U32BIT *free_mbyte_ptr, U32BIT *size_mbyte_ptr);

/**
 * @brief   Checks if there is enough space for the recording of the given event to take place
 * @brief   disk_id ID of disk to be queried
 * @param   event_ptr The event to check if there is any free space
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceForEvent(U16BIT disk_id, void *event_ptr);

/**
 * @brief   Check if there is enough space for the recording of the given duration to take place
 * @brief   disk_id ID of disk to be queried
 * @param   duration Recording duration expressed as U32DHMS
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceDuration(U16BIT disk_id, U32BIT duration);

/**
 * @brief   Check if there is enough space for the recording of the given duration to take place
 * @brief   disk_id ID of disk to be queried
 * @param   hours number of hours of the duration
 * @param   mins number of minutes of the duration
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceForDuration(U16BIT disk_id, U8BIT hours, U8BIT mins);

// ---- pause ----
/**
 * @brief   Calculates and returns the number of seconds behind live TV
 * @return  Number of seconds behind live
 */
U32BIT APVR_GetPauseProgress(void);

/**
 * @brief   Calculates and returns the number of seconds in playback buffer
 * @return  Number of seconds of playback
 */
U32BIT APVR_GetPlaybackTime(void);

/**
 * @brief   Sets returns the number of seconds in playback buffer
 * @param   position_in_secs number of seconds from start of playback
 * @return  TRUE if successfully changed play back position
 */
BOOLEAN APVR_SetPlaybackTime(U32BIT position_in_secs);

/**
 * @brief   Returns the path currently acquired fro playback
 * @return  Decode path for playback
 */
U8BIT APVR_GetPlaybackPath(void);

/**
 * @brief   Determines whether the given programme CRID is already in the list of recorded events
 * @param   prog_crid full CRID (i.e. including default authority) of event to be searched for
 * @return  TRUE if a recording with the given CRID is found, FALSE otherwise
 */
BOOLEAN APVR_IsEventInRecordList(U8BIT *prog_crid);

/**
 * @brief   Seaches the schedule for events that are part of the given series
 *          and that haven't already been recorded or aren't set to be recorded
 *          and creates a timer to record them.
 * @param   crid_ptr pointer to the CRID record
 */
void APVR_RecordSeries(void *crid_ptr);

/**
 * @brief   Searches the schedule for events that are part of the given recommendation
 *          and that haven't already been recorded or aren't set to be recorded
 *          and creates a timer to record them.
 * @param   crid_ptr pointer to the CRID record
 */
void APVR_RecordRecommendations(void *crid_ptr);

/**
 * @brief   If the current playback recording is a split event then the next
 *          chronological part of the event is found and its handle returned
 * @param   curr_handle handle of split recording when searching for the next part
 * @param   next_handle handle of the next recording to be played, returned
 * @return  TRUE if a recording is found, FALSE otherwise
 */
BOOLEAN APVR_FindNextSplitEvent(U32BIT curr_handle, U32BIT *next_handle);

/**
 * @brief   Ensures that any EIT updates are processed to handle scenarios such
 *          as checking for new series recordings, etc.
 */
void APVR_EitUpdated(void);

/**
 * @brief   This function should be called when there's an update to the PIDs
 *          for a service that's being recorded. The PIDs being recorded are
 *          checked and any that are no longer valid will no longer be recorded
 *          and any new PIDs will be added.
 * @param   path recording path
 */
void APVR_PidsUpdated(U8BIT path);

/**
 * @brief   A service instance is associated to the playback, this function returns its pointer.
 *          The handle can be used, for example, to call ADB_ServiceHasSubtitles as it's done for
 *          live channels.
 * @return  Playback service pointer
 */
void* APVR_GetPlaybackService(void);

/**
 * @brief   Deals with any private timers started by the PVR module
 * @param   timer_handle handle of the timer that's been triggered
 * @return  TRUE if the timer is a PVR timer and has been dealt with, FALSE otherwise
 */
BOOLEAN APVR_HandlePrivateTimer(U32BIT timer_handle);

#ifdef COMMON_INTERFACE
/**
 * @brief   When a recording is started that involves a CI CAM, a message is sent to check
 *          whether the recording can proceed, and this function handles the reply received
 * @param   start_succeeded TRUE if the reply indicates that the recording can proceed, FALSE
 *          otherwise
 * @param   slot_id the slot id associated with the reply
 */
void APVR_CIRecordReplyReceived(BOOLEAN start_succeeded, U8BIT slot_id);
#endif

//---Global Function prototypes for internal use---------------------------------
/**
 * @brief   Initialise all PVR resources: filters, recordings etc.
 */
void APVR_Initialise(void);

/**
 * @brief   Destroy all PVR resources: filters, recordings etc.
 */
void APVR_Terminate(void);

/**
 * @brief   Returns the PVR play status
 * @return  TRUE if a recoding is being played, FALSE otherwise
 */
BOOLEAN APVR_IsDecodingFile(void);

/**
 * @brief   Play in slow mothing mode
 */
void APVR_SlowMoPlay(void);

// quick replay current playing recording
void APVR_QRPlay(void);

// quick replay current playing recording
void APVR_JTLPlay(void);

/**
 * @brief   Mute for trick mode
 */
void APVR_TrickModeMuteOn(void);

/**
 * @brief   Unmute for trick mode
 */
void APVR_TrickModeMuteOff(void);

/**
 * @brief   Delete all the recordings marked as selected but not locked in the given list,
 *          including all files associated with it and remove it from the database
 *          of available recordings that can be played back.
 * @param   handles array of recording handles
 * @param   num_handles number of recordings in the array
 * @return  TRUE if at least one recording is deleted, FALSE otherwise
 */
BOOLEAN APVR_DeleteAllSelectedRecordings(U32BIT *handles, U16BIT num_handles);

/**
 * @brief   Un-select all recordings
 */
void APVR_UnselectAllRecordings(U32BIT *handles, U16BIT num_handles);

/**
 * @brief   Checks if any recording in a list is selected
 * @param   handles array of recording handles
 * @param   num_handles number of recording handles in the array
 * @return  TRUE if at least one recording is selected, FALSE otherwise
 */
BOOLEAN APVR_AreSelectedRecordings(U32BIT *handles, U16BIT num_handles);

/**
 * @brief   Used to set whether all recordings should be encrypted or not. The default is that
 *          recordings aren't encrypted, in which case the do_not_scramble flag in the SI
 *          free-to-air content management descriptor is used to determine whether a recording
 *          should be encrypted or not.
 *          This setting is not persistent, so needs to be set each time the DVB stack is initialised
 * @param   state TRUE if recordings are to be encrypted, FALSE if not
 */
void APVR_EncryptRecordings(BOOLEAN state);

#endif // _AP_PVR_H
