/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   application level SI task
 * @file    ap_si.h
 * @date    01/10/2002
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error
#ifndef _AP_SI_H
#define _AP_SI_H

#include "techtype.h"
#include "stbsiflt.h"
#include "stbsitab.h"

/* quality and strength are on a scale 0 to 10 */
#define GET_SIGNAL_STATUS(quality, strength) (((quality) << 8) | (strength))
#define BAD_SIGNAL_STATUS 0xffff

#define GET_SIGNAL_QUALITY(status) ((status) >> 8)
#define GET_SIGNAL_STRENGTH(status) ((status) & 0xff)

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------

typedef enum
{
   APP_SI_MODE_NO_SI,
   APP_SI_MODE_CHANNEL_SEARCH,            // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_CHANNEL_SEARCH_NO_NIT,     // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_STARTUP_SEARCH,            // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_EVENT_PF_SEARCH,           // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_EVENT_SCHED_SEARCH,        // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_EVENT_PF_SCHED_SEARCH,     // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_DVB_SSU_SEARCH,            // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_UPDATE,                    // use STB_DPSetSearchMode(0, FALSE);
   APP_SI_MODE_CIPLUS_UPDATE,             // use STB_DPSetSearchMode(0, FALSE);
   APP_SI_MODE_CIPLUS_NO_PAT_PMT,         // use STB_DPSetSearchMode(0, FALSE);
   APP_SI_MODE_TOT_SEARCH,                // use STB_DPSetSearchMode(0, TRUE);
   APP_SI_MODE_USER_DEFINED
} E_APP_SI_MODE;

typedef enum
{
   ASI_MODE_FLAG_IGNORE_NIT_TRANSPORT = 0,   // bit[0]: do'nt generate the transport records from the NIT
   ASI_MODE_FLAG_MAX
} E_ASI_MODE_FLAG;

typedef enum
{
   SEARCH_SERVICE_TYPE_ADVANCED_CODEC = 0x0001,    /* Advanced codec services - H264, AC-3, AAC, etc */
   SEARCH_SERVICE_TYPE_HD = 0x0002,                /* Services defined as HD, including MPEG2 HD */
   SEARCH_SERVICE_TYPE_HEVC = 0x0004,              /* HEVC codec services */

   SEARCH_SERVICE_TYPE_FREE_TO_AIR = 0x4000,
   SEARCH_SERVICE_TYPE_ENCRYPTED = 0x8000,

   SEARCH_SERVICE_TYPE_FTA = (SEARCH_SERVICE_TYPE_FREE_TO_AIR | 0x3fff),
   SEARCH_SERVICE_TYPE_SCRAMBLED = (SEARCH_SERVICE_TYPE_ENCRYPTED | 0x3fff),
   SEARCH_SERVICE_TYPE_ALL = 0xffff
} E_SEARCH_SERVICE_TYPE;

/* Events passed to SI managers */
typedef enum
{
   APP_SI_START_MANAGER = 0x0001,
   APP_SI_STOP_MANAGER = 0x0002,
   APP_SI_CHANNEL_CHANGE = 0x0003,

   APP_SI_PAT_RECEIVED = 0x0011,
   APP_SI_PMT_RECEIVED = 0x0012,
   APP_SI_SDT_RECEIVED = 0x0013,
   APP_SI_NIT_RECEIVED = 0x0014,
   APP_SI_EIT_RECEIVED = 0x0015,
   APP_SI_TOT_RECEIVED = 0x0016,
   APP_SI_TDT_RECEIVED = 0x0017,
   APP_SI_CAT_RECEIVED = 0x0018,
   APP_SI_RCT_RECEIVED = 0x0019,
   APP_SI_AIT_RECEIVED = 0x001A,
   APP_SI_BAT_RECEIVED = 0x0020,

   APP_SI_PAT_TIMEOUT = 0x0031,
   APP_SI_PMT_TIMEOUT = 0x0032,
   APP_SI_SDT_TIMEOUT = 0x0033,
   APP_SI_NIT_TIMEOUT = 0x0034,
   APP_SI_EIT_TIMEOUT = 0x0035,
   APP_SI_TOT_TIMEOUT = 0x0036,
   APP_SI_TDT_TIMEOUT = 0x0037,
   APP_SI_CAT_TIMEOUT = 0x0038,
   APP_SI_SCHED_TIMEOUT = 0x0039,
   APP_SI_BAT_TIMEOUT = 0x0040,

   APP_SI_PMT_UPDATE = 0x0051,
   APP_SI_UPDATE_DELAY_EXPIRED = 0x0052,
   APP_SI_STOP_PMT_REPORTING = 0x0053,

   APP_SI_USER_DEFINED_EVENT = 0x0100,
   /* SSU UNT */
   APP_SI_UNT_RECEIVED,
   APP_SI_UNT_TIMEOUT,
   APP_SI_UNT_UPDATE,

   APP_SI_INVALID
} E_APP_SI_EVENT_CODE;

typedef enum {
   APP_SI_EIT_JOURNAL_TYPE_CLEAR,
   APP_SI_EIT_JOURNAL_TYPE_ADD,
   APP_SI_EIT_JOURNAL_TYPE_DELETE,
   APP_SI_EIT_JOURNAL_TYPE_UPDATE,
   APP_SI_EIT_JOURNAL_TYPE_EXPIRE
} E_APP_SI_EIT_JOURNAL_TYPE;

typedef struct s_app_si_eit_journal
{
   E_APP_SI_EIT_JOURNAL_TYPE type;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U16BIT serv_id;
   U16BIT allocated_lcn;
   BOOLEAN is_sched;
   U16BIT event_id;
} S_APP_SI_EIT_SCHED_UPDATE;


//---Global type defs for public use-------------------------------------------
typedef BOOLEAN (*F_SIManager)(U8BIT, U32BIT, SI_TABLE_RECORD *);   /* Type of SI manager function */
typedef void (*F_EitParser)(U8BIT, U8BIT, U8BIT *, U32BIT, void *);
typedef void (*F_BatTableUpdate)(U8BIT, SI_BAT_TABLE *, SI_TABLE_RECORD *);
typedef void (*F_EitTableUpdate)(U8BIT, SI_TABLE_RECORD *);
typedef void (*F_NitTableUpdate)(U8BIT, SI_NIT_TABLE *, SI_TABLE_RECORD *);
typedef void (*F_PmtTableUpdate)(U8BIT, SI_PMT_TABLE *, SI_TABLE_RECORD *, void *);
typedef void (*F_SdtTableUpdate)(U8BIT, SI_SDT_TABLE *, SI_TABLE_RECORD *);
typedef void (*F_UntTableUpdate)(U8BIT, SI_UNT_TABLE *, SI_TABLE_RECORD *);

typedef void (*F_EitSchedUpdateCB)(S_APP_SI_EIT_SCHED_UPDATE *update);


//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Checks whether the NIT or SDT version numbers have changed, which
 *          may indicate a change to the service lineup requiring an update
 *          service scan.
 * @param   tuner_type check all networks and transports for this tuner type, or use
 *                     SIGNAL_NONE to check all networks and transports
 * @return  TRUE if one of the version numbers has changed, FALSE otherwise
 */
BOOLEAN ASI_CheckForServiceChange(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief   Set the type for services that should be added during a service search
 * @param   service_type all, free-to-air, or scrambled
 */
void ASI_SetSearchServiceType(E_SEARCH_SERVICE_TYPE service_type);

/**
 * @brief   Add the given service id to the list of services whose PMT will be requested
 *          with a higher priority than others
 * @param   service_id service id to be added to the list
 */
void ASI_AddServiceToPmtList(U16BIT service_id);

/**
 * @brief   Removes the service id from the PMT priority list
 * @param   service_id service id to be removed
 */
void ASI_RemoveServiceFromPmtList(U16BIT service_id);

/**
 * @brief   Clears all service ids from the PMT priority list
 */
void ASI_ClearPmtList(void);

/**
 * @brief   Takes data for a raw PMT for the given service and processes it as if
 *          it had been received from the demux, also passing it to anything monitoring
 *          PMTs and CI+.
 * @param   path decode path
 * @param   s_ptr service to be updated with info extracted from the PMT
 * @param   pmt_data raw PMT data
 */
void ASI_ProcessPmt(U8BIT path, void *s_ptr, U8BIT *pmt_data);

/**
 * @brief   Sets the number of hours of EIT data that's kept for each service
 *          that hasn't its had EIT schedule disabled.
 * @param   limit_hours number of hours to EIT data to be kept.
 *                        0 turns this feature off and all data will be kept.
 */
void ASI_SetEITScheduleLimit(U16BIT limit_hours);

/**
 * @brief   Returns the current setting for the number of hours of EIT data that's kept
 * @return  number of hours, 0 means all data is kept
 */
U16BIT ASI_GetEITScheduleLimit(void);

/**
 * @brief   Enables or disables the collection of BATs as part of the SI processing
 *          and allows the bouquet IDs of the BATs to be collected to be specified.
 * @param   use_bats TRUE to enable the collection and processing of BATs, FALSE to disable
 * @param   bouquet_ids an array of the bouquet IDs that are to be used.
 *                        If this is NULL then all BATs will be used.
 * @param   num_ids number of bouquet IDs in the array
 */
void ASI_EnableBatCollection(BOOLEAN use_bats, U16BIT *bouquet_ids, U16BIT num_ids);

/**
 * @brief   Returns the DVB triplet for the next location containing a possible OTA update.
 * @param   network network on which the OTA search is being made
 * @param   prev_location if this is NULL then info from the first OTA location on the network will
 *                        be returned, otherwise it should be the value returned when this function
 *                        was last called so the next possible OTA location will be returned
 * @param   onid_ptr pointer to return the value of the original network ID for the possible OTA
 * @param   tdi_ptr pointer to return the value of the transport ID for the possible OTA
 * @param   sid_ptr pointer to return the value of the service ID for the possible OTA
 * @return  If NULL then no OTA locations are available, otherwise a possible OTA is available and
 *          the returned value should be passed in the next time this function is called
 */
void* ASI_GetNextOTALocation(void *network, void *prev_location, U16BIT *onid_ptr,
   U16BIT *tid_ptr, U16BIT *sid_ptr);

/**
 * @brief   Sets the flag indicating whether the SSU has been refused or not
 * @param   refuse TRUE to refuse the SSU, FALSE to accept
 */
void ASI_RefuseSSU(BOOLEAN refuse);

/**
 * @brief   Returns the flag indicating whether the SSU was refused
 * @return  TRUE if the SSU was refused, FALSE otherwise
 */
BOOLEAN ASI_SSURefused(void);

/**
 * @brief   Sets the flag indicating whether the SSU can be refused by the user.
 * @param   mandatory TRUE if the SSU can't be refused by the user, FALSE otherwise
 */
void ASI_SSUSetMandatory(BOOLEAN mandatory);

/**
 * @brief   Returns the flag indicating whether the SSU is mandatory and so can't be refused
 * @return  TRUE if the SSU is mandatory, FALSE otherwise
 */
BOOLEAN ASI_SSUGetMandatory(void);


//---Global Function prototypes for internal use---------------------------------
/**
 * @brief   Initialises application SI handling
 */
void ASI_InitialiseAppSi(void);

/**
 * @brief   Sets application SI mode - used before STB_DPStartSI() is called.
 * @param   path decode path
 * @param   si_mode required mode
 */
void ASI_SetAppSiMode(U8BIT path, E_APP_SI_MODE si_mode);

/**
 * @brief   reset application SI mode flag
 * @param   path decode path
 *
 */
void ASI_ResetAppSiModeFlag(U8BIT path);

/**
 * @brief   Set application SI mode flags
 * @param   path decode path
 * @param   si_mode_flag required mode flag
 */
void ASI_SetAppSiModeFlag(U8BIT path, E_ASI_MODE_FLAG si_mode_flag, BOOLEAN enable);

/**
 * @brief   get application SI mode flag
 * @param   path decode path
 * @param   si_mode_flag required mode flag
 *
 * @return   TRUE or FALSE
 */
BOOLEAN ASI_GetAppSiModeFlag(U8BIT path, E_ASI_MODE_FLAG si_mode_flag);

/**
 * @brief   Returns the PID for the pmt of a given service (on live path)
 * @param   service_id provides the required service ID
 * @param   ts_id provides the required transport stream ID
 * @param   on_id provides the required original network ID
 * @return  the PID of the pmt for that service
 */
U16BIT ASI_GetPmtPid(U16BIT serv_id, U16BIT ts_id, U16BIT on_id);

/**
 * @brief   Prevents the current pmt being reported (e.g. to MHEG).
 */
void ASI_StopPmtReporting(U8BIT path);

/**
 * @brief   Returns TRUE if pmt has been reported to third parties
 * @param   path decode path
 * @return  TRUE if reported, FALSE otherwise
 */
BOOLEAN ASI_PmtReported(U8BIT path);

/**
 * @brief   Performs the neccessary actions for this module when entering/exiting standby
 *          according to the value passed in standby_state.
 * @param   standby_state if set TRUE indicates that the STB is entering standby
 *          mode and if set FALSE indicates shat the STB is exiting from standby mode.
 */
void ASI_SetStandbyState(BOOLEAN standby_state);

/**
 * @brief   Forces the SI demux filter collecting the CAT tables to be reset,
 *          so a previously processed version of the table won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartCatFilter(U8BIT path);

/**
 * @brief   Forces the SI demux filter collecting the NIT tables to be reset,
 *          so a previously processed version of the table won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartNitFilter(U8BIT path);

/**
 * @brief   Forces the SI demux filter collecting the SDT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartSdtFilter(U8BIT path);

/**
 * @brief   Forces the SI demux filter collecting the BAT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartBatFilter(U8BIT path);

/**
 * @brief   Forces the SI demux filter collecting the TOT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartTotFilter(U8BIT path);

/**
 * @brief   Forces the SI demux filter collecting the TDT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartTdtFilter(U8BIT path);

/**
 * @brief   Cancels any existing SI filters and starts a new one for the SDT, TDT, TOT,
 *          NIT, EIT (pf, pf++ and sched) and BAT SI tables.
 * @param   path decode path the filters are using
 * @param   use_standard_pids TRUE if SI tables on the DVB standard PIDs are to be collected,
 *                            FALSE will only start a filter if a PID is defined for that table
 *                            on the current service.
 */
void ASI_RestartSITables(U8BIT path, BOOLEAN use_standard_pids);

/**
 * @brief   Sets a function that will be called when parsing an EIT table and a
 *          descriptor is found that the standard code doesn't know how to parse
 * @param   parser_func function pointer
 */
void ASI_SetEITParserFunction(F_EitParser parser_func);

/**
 * @brief   Sets a function that will be called when a BAT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateBatFunction(F_BatTableUpdate update_func);

/**
 * @brief   Sets a function that will be called when an EIT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateEitFunction(F_EitTableUpdate update_func);

/**
 * @brief   Sets a function that will be called when an NIT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateNitFunction(F_NitTableUpdate update_func);

/**
 * @brief   Sets a function that will be called when a PMT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdatePmtFunction(F_PmtTableUpdate update_func);

/**
 * @brief   Sets a function that will be called when an SDT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateSdtFunction(F_SdtTableUpdate update_func);

/**
 * @brief   Sets a function that will be called when an UNT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateUntFunction(F_UntTableUpdate update_func);

/**
 * @brief   Processes an EIT table, partial or full, and updates the events of the service it is for
 * @param   table_rec SI table record containing the EIT data
 * @param   playback TRUE if the EIT is related to PVR playback
 */
void ASI_ProcessEitTable(SI_TABLE_RECORD *table_rec, BOOLEAN playback);

/**
 * @brief   Registers a function that will be called whenever an EIT event is added, updated,
 *          deleted or expires from a service's EIT schedule. The registered function should
 *          return control as soon as possible, otherwise processing of other EIT events will
 *          be affected.
 * @param   sched_update_callback function to be called
 */
void ASI_RegisterEitSchedUpdateCallback(F_EitSchedUpdateCB sched_update_callback);

/**      
 * @brief   Calls the EIT schedule update callback function, if one has been registered,
 *          with details of the service and event
 * @param   serv_ptr service whose EIT schedule has changed
 * @param   event_id ID of the event that's been added, updated, deleted or expired
 * @param   type type of the change. For APP_SI_EIT_JOURNAL_TYPE_CLEAR, the event_id is irrelevant
 */      
void ASI_NotifyEitSchedUpdate(void *serv_ptr, U16BIT event_id, E_APP_SI_EIT_JOURNAL_TYPE type);

/**
 * @brief   Processes the TOT table record to extract data for the database
 * @param   table_rec pointer to the TOT table record
 */
void ASI_ProcessTotTable(SI_TABLE_RECORD *table_rec);

/**
 * @brief   Processes the TDT table record to extract data for the database
 * @param   table_rec pointer to the TDT table record
 */
void ASI_ProcessTdtTable(SI_TABLE_RECORD *table_rec);

/**
 * @brief   Sets the network ID to be used for all future requests to get an NIT.
 *          If this value isn't set, the default behaviour is to request the NITactual,
 *          but when this value is set then the NIT received could be NITactual or NITother.
 *          To clear this setting this function should be called with a network ID of 0.
 * @param   network_id network ID of the NIT to be requested, or 0 to use NITactual
 */
void ASI_SetActiveNetworkId(U16BIT network_id);

/**
 * @brief   Sets whether updates should be allowed to the service database from standard SI
 *          tables that are received when the SI engine is running in UPDATE mode. The default
 *          setting is for updates to be allowed and this setting doesn't persist across
 *          power cycles so needs to be explicitly set if updates shouldn't be allowed.
 *          If updates aren't allowed then this will prevent dynamic service additions, deletions
 *          and moves from being performed, even if these have been enabled, but it doesn't affect
 *          updates to the database when some form of search is being performed, such as a
 *          service search. This setting is used when the SDT, NIT and BAT tables are received.
 * @param   signal_type specifies the signal type that the setting will apply to. If this value
 *                      is given as SIGNAL_NONE, then the given flag will be applied to all
 *                      signal types (DVB-T, DVB-C and DVB-S).
 * @param   allow flag that defines whether service database updates should be allowed or not
 */
void ASI_AllowDatabaseUpdates(E_STB_DP_SIGNAL_TYPE signal_type, BOOLEAN allow);

/**
 * @brief   Returns whether service database updates are allowed for the given signal type
 * @param   signal_type signal type being queried
 * @return  TRUE if service updates are allowed, FALSE if not, or if SIGNAL_NONE is given
 */
BOOLEAN ASI_DatabaseUpdatesAllowed(E_STB_DP_SIGNAL_TYPE signal_type);

#endif
