/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Header file for NVM data handling functions
 *
 * @file    app_nvm.h
 * @date    16/09/2003
 */

#ifndef _APP_NVM_H
#define _APP_NVM_H

#include "techtype.h"
#include "stbhwc.h"

//---constant definitions---------------------------------------------------------------------------
//#define MAX_FAV_GRPS             4

//---typedefs-structs-enumerations------------------------------------------------------------------

//**************************************************************************************************
//******   To add a new NVM item                                                              ******
//******   ---------------------------------------------------------------------------------  ******
//******                                                                                      ******
//******   1. Add new nvm item to bottom of e_nvm_items e.g. ITEM_NEW_NVM above NUM_ITEMS_NVM ******
//******   2. See notes in app_nvm.c                                                          ******
//**************************************************************************************************

//for use when using  APP_NvmReadDefault, APP_NvmRead, APP_NvmSave in screens
//to access required nvm items
typedef enum e_nvm_items
{
   FIRST_BOOT_NVM,
   STANDBY_POWERSAVE_NVM,
   VOLUME_NVM,
   ASPECT_MODE_NVM,
   SCART_INPUT_TYPE_NVM,
   PRIMARY_AUDIO_LANG_NVM,
   SECONDARY_AUDIO_LANG_NVM,
   AUDIO_TYPE_NVM,
   STANDBY_STATE_NVM,
   SEARCHING_STATE_NVM,
   PARENTAL_LOCK_NVM,
   ICS_PARENTAL_LOCK_NVM,
   SYSTEM_PIN_NO_NVM,
   AERIAL_POWER_ON_NVM,
   COUNTRY_CODE_NVM,
   REGION_ID_NVM,
   RF_CHANNEL_NUM_NVM,
   ASPECT_RATIO_NVM,
   TV_SCART_TYPE_NVM,
   WAKEUP_VOLUME_NVM,
   HDMI_RESOLUTION_NVM,
   NET_IF_TYPE_NVM,
   IP_MODE_NVM,
   IP_ADDRESS_NVM,
   SUBNET_MASK_NVM,
   GATEWAY_IP_NVM,
   DNS_SERVER_IP_NVM,
   ESSID_NVM,
   ESSID_PASSWORD_NVM,
   AD_VOLUME_NVM,
   SPDIF_OUTPUT_NVM,
   LIP_SYNC_ADJUSTMENT_NVM,
   TARGET_REGION_DEPTH_NVM,
   TARGET_REGION_COUNTRY_NVM,
   TARGET_REGION_PRIMARY_NVM,
   TARGET_REGION_SECONDARY_NVM,
   TARGET_REGION_TERTIARY_NVM,
   OTA_TYPE_NVM,
   OTA_LAST_UPDATE_SRCH_NVM,
   LAST_CHAN_SRCH_NVM,
   LAST_EIT_UPDATE_NVM,
   WATCH_HD_PROGRAMMES_NVM,
   RECORD_HD_PROGRAMMES_NVM,
   TIMESHIFT_BUFFER_SIZE_NVM,
   SUBTITLE_TYPE_NVM,
   BACKGROUND_SEARCH_START_NVM,
   BACKGROUND_SEARCH_END_NVM,
   SERVICE_SEARCH_ENABLED_NVM,
   SSU_SEARCH_ENABLED_NVM,
   LIVE_SERVICE_LCN_NVM,
   PRIMARY_TEXT_LANG_NVM,
   SECONDARY_TEXT_LANG_NVM,
   HDMI_AUDIO_OUTPUT_NVM,
   EIT_SCHED_LIMIT_NVM,
   PREFER_HD_AUDIO_NVM,
   PVR_NOTIFY_TIME_NVM,
   RECORD_START_PADDING_NVM,
   RECORD_END_PADDING_NVM,
#if defined(COMMON_INTERFACE)
   CICAM_ID0_0_NVM,
   CICAM_ID0_1_NVM,
   CICAM_PIN0_NVM,
   CICAM_PIN0_USED_TIME_NVM,
   CICAM_ID1_0_NVM,
   CICAM_ID1_1_NVM,
   CICAM_PIN1_NVM,
   CICAM_PIN1_USED_TIME_NVM,
   CICAM_ID2_0_NVM,
   CICAM_ID2_1_NVM,
   CICAM_PIN2_NVM,
   CICAM_PIN2_USED_TIME_NVM,
   CICAM_ID3_0_NVM,
   CICAM_ID3_1_NVM,
   CICAM_PIN3_NVM,
   CICAM_PIN3_USED_TIME_NVM,
#endif
   NUM_ITEMS_NVM
} E_NVM_ITEMS;

typedef enum
{
   STDBY_POWER_ON = 0,
   STDBY_STANDBY,
   STDBY_WAKE_FOR_SEARCH,
   STDBY_UPDATE_SEARCH,
   STDBY_WAKE_FOR_RECORDING,
   STDBY_RECORDING
} E_STANDBY_STATE;

typedef enum
{
   OTA_TYPE_AUTO = 0,
   OTA_TYPE_MANUAL
} E_OTA_TYPE;

typedef enum
{
   VIEW_REC_NEVER = 0,
   VIEW_REC_ASK,
   VIEW_REC_ALWAYS
} E_VIEW_REC_PREF;

typedef enum
{
   SUBTITLE_NORMAL = 0,
   SUBTITLE_HARD_OF_HEARING
} E_SUBTITLE_TYPE;

typedef enum
{
   PARENTAL_LOCK_OFF = 0,
   PARENTAL_LOCK_ON = 0xff
} E_PARENTAL_LOCK;

typedef enum
{
   NET_IF_NONE,
   NET_IF_WIRED,
   NET_IF_WIRELESS
} E_NET_IF_TYPE;

//---Global function prototypes--------------------------------------------------------------------

/**
 * @brief   Resets the DVB's settings to factory defaults
 */
void APP_NvmRestoreDefaults(void);

/**
 * @brief   Returns the factory default value for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  factory default value
 */
U32BIT APP_NvmReadDefault(E_NVM_ITEMS nvm_item);

/**
 * @brief   Returns the current value for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  current value
 */
U32BIT APP_NvmRead(E_NVM_ITEMS nvm_item);

/**
 * @brief   Returns pointer to current string for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  pointer to string
 */
U8BIT *APP_NvmReadString(E_NVM_ITEMS nvm_item);

/**
 * @brief   Sets the current value for the given DVB setting
 * @param   nvm_item - item to be read
 * @param   new_value - value for the item
 * @param   write_to_flash_now - if TRUE then all the current values will be saved.
 *                               When changing the values of several items, it will
 *                               be more efficient to only this this TRUE for the last item
 */
void APP_NvmSave(E_NVM_ITEMS nvm_item, U32BIT new_value, BOOLEAN write_to_flash_now);

/**
 * @brief   Sets the current value for the given DVB setting
 * @param   nvm_item - item to be written
 * @param   str_ptr - pointer to string for the item
 * @param   write_to_flash_now - if TRUE then all the current values will be saved.
 *                               When changing the values of several items, it will
 *                               be more efficient to only this this TRUE for the last item
 */
void APP_NvmSaveString(E_NVM_ITEMS nvm_item, U8BIT *str_ptr, BOOLEAN write_to_flash_now);

/**
 * @brief   Saves DVB values immediately
 */
void APP_NvmSaveAllNow(void);

/**
 * @brief   Returns the size in bytes the DVB module uses to save its settings
 * @return  size of DVB settings in bytes
 */
U32BIT APP_NvmGetDvbSize(void);

//---Global functions for internal use ------------------------------------------------------------

/**
 * @brief   Initialises the DVB core's NVM data
 */
void APP_NvmInitialise(void);

#endif // _APP_NVM_H
//**************************************************************************************************
// ---ooO END OF FILE Ooo---
//**************************************************************************************************
