/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CA control functions
 * @file    ap_ca.c
 * @date    October 2011
 * @author  Ocean Blue
 */

#define CA_DEBUG

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include "techtype.h"
#include "dbgfuncs.h"

#include "stbdpc.h"
#include "stbpvr.h"
#include "stbsiflt.h"
#include "stbsitab.h"

#include "ca_glue.h"

#include "ap_dbacc.h"

//---constant definitions for this file----------------------------------------
#ifdef CA_DEBUG
   #define AP_CA_PRINT(x) STB_SPDebugWrite x
#else
   #define AP_CA_PRINT(x)
#endif


//---local typedefs, structs, enumerations for this file--------------------------------------------


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/*!**************************************************************************
 * @brief   Acquires a CA descrambler for the given path on the given service if
 *          needed. No CA descrambler is acquired if a CI slot has been already
 *          acquired (CI takes precedence) or if a CA descrambler is already
 *          associated to this path.
 * @param   path - decode path
 * @param   s_ptr - service to be used on the path
 * @return  TRUE if the CA path has been successfully acquired, FALSE otherwise.
 ****************************************************************************/
BOOLEAN ACA_AcquireCADescrambler(U8BIT path, void *s_ptr)
{
   U8BIT *pmt_data;
   U16BIT data_len;
   BOOLEAN has_ca;
   U32BIT ca_handle;
   U8BIT running_status;
   BOOLEAN retval = FALSE;

   FUNCTION_START(ACA_AcquireCADescrambler);

   has_ca = STB_DPGetPathCADescrambler(path, &ca_handle);

#ifdef CA_DEBUG
#ifdef COMMON_INTERFACE
   AP_CA_PRINT(("%s(path=%u): CI slot=%u, CA=%u(0x%lx)", __FUNCTION__, path,
                STB_DPGetPathCISlot(path), has_ca, ca_handle));
#else
   AP_CA_PRINT(("%s(path=%u): CA=%u(0x%lx)", __FUNCTION__, path, has_ca, ca_handle));
#endif
#endif

   /* Only acquire a CA slot if there isn't one already and there's no CI slot */
   if (!has_ca
#ifdef COMMON_INTERFACE
       && (STB_DPGetPathCISlot(path) == INVALID_RES_ID)
#endif
       )
   {
      pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
      if (pmt_data != NULL)
      {
         if (STB_DPAcquireCADescramblerForPath(path, pmt_data, &ca_handle))
         {
            retval = TRUE;

            /* Pass the PMT to the CA system */
            STB_CAReportPMT(ca_handle, pmt_data, data_len);

            if ((running_status = ADB_GetServiceRunningStatus(s_ptr)) != 0)
            {
               /* Pass the running status to the CA system */
               STB_CANotifyRunningStatus(ca_handle, running_status);
            }
         }
#ifdef CA_DEBUG
         else
         {
            AP_CA_PRINT(("%s(path=%u): couldn't acquire a CA slot", __FUNCTION__, path));
         }
#endif
      }
#ifdef CA_DEBUG
      else
      {
         AP_CA_PRINT(("%s(path=%u): no pmt data available on service %p", __FUNCTION__, path, s_ptr));
      }
#endif
   }

   FUNCTION_FINISH(ACA_AcquireCADescrambler);

   return retval;
}

//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------
