/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application configuration functions
 * @file    ap_cfg.c
 * @date    05/04/2004
 * @author  Ocean Blue
 */

// debug macros

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbheap.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbllist.h"

#ifdef COMMON_INTERFACE
#include "stbci.h"
#endif

#include "app.h"
#include "ap_cfg.h"
#include "app_nvm.h"
#include "ap_tmr.h"
#include "ap_dbacc.h"
#include "ap_dbdef.h"

//---constant definitions for this file-------------------------------------------------------------

#define NUM_COUNTRIES (sizeof(country_data_table) / sizeof(ACFG_COUNTRY_CONFIG))

//---local typedefs, structs, enumerations for this file--------------------------------------------


//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

static U32BIT und_lang_code = ADB_LANG_CODE_UNDEF;
#ifdef COMMON_INTERFACE
static U32BIT ci_hcl_module = 0;
#endif

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static U8BIT GetCountryId(U32BIT country_code);


//--------------------------------------------------------------------------------------------------
// include file containing configuration data declarations. Must be included here AFTER typedefs.
#define DECLARE_DATA_TABLES
#define DECLARE_DB_LANG_LIST     // include db_lang_list[] defined in ap_cfdat.h
#include "ap_cfdat.h"
#undef DECLARE_DB_LANG_LIST
#undef DECLARE_DATA_TABLES
//--------------------------------------------------------------------------------------------------



//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------

#ifdef COMMON_INTERFACE
/**
 * @brief    Set the Host Country code in CI stack
 * @param    code - new country code (encoded in U32BIT)
 */
static void CISetHostCountryCode(U32BIT code)
{
   U8BIT country_code[3];
   FUNCTION_START(CISetHostCountryCode);
   if (ci_hcl_module)
   {
      /* Make sure the code in in uppercase */
      code &= ~0x00202020;
      country_code[0] = (U8BIT)(code >> 16);
      country_code[1] = (U8BIT)(code >> 8);
      country_code[2] = (U8BIT)code;
      STB_CISetHostCountryCode(ci_hcl_module, country_code);
   }
   FUNCTION_FINISH(CISetHostCountryCode);
}

/**
 * @brief    Set the Host Language code in CI stack
 * @param    code - new langauge code (encoded in U32BIT)
 */
static void CISetHostLanguageCode(U32BIT code)
{
   U8BIT lang_code[3];
   FUNCTION_START(CISetHostLanguageCode);
   if (ci_hcl_module)
   {
      lang_code[0] = (U8BIT)(code >> 16);
      lang_code[1] = (U8BIT)(code >> 8);
      lang_code[2] = (U8BIT)code;
      STB_CISetHostLanguageCode(ci_hcl_module, lang_code);
   }
   FUNCTION_FINISH(CISetHostLanguageCode);
}

#endif // COMMON_INTERFACE

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------


/**
 * @brief   Returns the number of country configurations included in the DVB stack
 * @return  number of countries
 */
U8BIT ACFG_GetNumCountries(void)
{
   FUNCTION_START(ACFG_GetNumCountries);
   FUNCTION_FINISH(ACFG_GetNumCountries);
   return(NUM_COUNTRIES);
}

/**
 * @brief   Returns the country code the DVB is configured for
 * @return  country code
 */
U32BIT ACFG_GetCountry(void)
{
   U32BIT country_code;

   FUNCTION_START(ACFG_GetCountry);
   country_code = APP_NvmRead(COUNTRY_CODE_NVM);
   FUNCTION_FINISH(ACFG_GetCountry);

   return(country_code);
}

/**
 * @brief   Returns the index in the list of countries known by the DVB, of the given country
 * @param   country_code country code
 * @return  country index, value returned by ACFG_GetNumCountries() if the country isn't found
 */
U8BIT ACFG_GetCountryIndex(U32BIT country_code)
{
   U8BIT index;

   FUNCTION_START(ACFG_GetCountryIndex);

   for (index = 0; index < NUM_COUNTRIES; index++)
   {
      if (country_data_table[index].country_code == country_code)
      {
         break;
      }
   }

   FUNCTION_FINISH(ACFG_GetCountryIndex);

   return(index);
}

/**
 * @brief   Returns the internal ID of the region the DVB is configured for.
 * @return  region id
 */
U8BIT ACFG_GetRegionId(void)
{
   U8BIT region_id;

   FUNCTION_START(ACFG_GetRegionId);
   region_id = (U8BIT)APP_NvmRead(REGION_ID_NVM);
   FUNCTION_FINISH(ACFG_GetRegionId);

   return(region_id);
}

/**
 * @brief   Returns the primary audio language ID
 * @return  language id
 */
U8BIT ACFG_GetPrimaryAudioLangId(void)
{
   U8BIT lang_id;

   FUNCTION_START(ACFG_GetPrimaryAudioLangId);
   lang_id = (U8BIT)APP_NvmRead(PRIMARY_AUDIO_LANG_NVM);
   FUNCTION_FINISH(ACFG_GetPrimaryAudioLangId);

   return(lang_id);
}

/**
 * @brief   Returns the secondary audio language ID
 * @return  language id
 */
U8BIT ACFG_GetSecondaryAudioLangId(void)
{
   U8BIT lang_id;

   FUNCTION_START(ACFG_GetSecondaryAudioLangId);
   lang_id = (U8BIT)APP_NvmRead(SECONDARY_AUDIO_LANG_NVM);
   FUNCTION_FINISH(ACFG_GetSecondaryAudioLangId);

   return(lang_id);
}

/**
 * @brief   Returns the primary DVB subtitle/teletext language ID
 * @return  language id
 */
U8BIT ACFG_GetPrimaryTextLangId(void)
{
   U8BIT lang_id;

   FUNCTION_START(ACFG_GetPrimaryTextLangId);
   lang_id = (U8BIT)APP_NvmRead(PRIMARY_TEXT_LANG_NVM);
   FUNCTION_FINISH(ACFG_GetPrimaryTextLangId);

   return(lang_id);
}

/**
 * @brief   Returns the secondary DVB subtitle/teletext language ID
 * @return  language id
 */
U8BIT ACFG_GetSecondaryTextLangId(void)
{
   U8BIT lang_id;

   FUNCTION_START(ACFG_GetSecondaryTextLangId);
   lang_id = (U8BIT)APP_NvmRead(SECONDARY_TEXT_LANG_NVM);
   FUNCTION_FINISH(ACFG_GetSecondaryTextLangId);

   return(lang_id);
}

/**
 * @brief   Sets the current country and sets default values for region and language
 * @param   country_code country code, as defined above (see COUNTRY_CODE_* defines)
 * @return  TRUE if country is known, FALSE otherwise
 */
BOOLEAN ACFG_SetCountry(U32BIT country_code)
{
   BOOLEAN result;
   U8BIT id;

   FUNCTION_START(ACFG_SetCountry);

   result = FALSE;

   for (id = 0; id < NUM_COUNTRIES; id++)
   {
      if (country_data_table[id].country_code == country_code)
      {
         /* Set this as the current country with default ids for the region and language */
         ACFG_SetCountryIds(country_code, 0, 0, 0);
         result = TRUE;
         break;
      }
   }

   FUNCTION_FINISH(ACFG_SetCountry);

   return(result);
}

/**
 * @brief   Saves the configured country and changes settings related to the country
 * @param   country_code country code
 * @param   region_id ID of region in country, if appropriate
 * @param   audio_lang_id ID of language to be used for audio
 * @param   sub_lang_id ID of language to be used for subtitles/teletext
 */
void ACFG_SetCountryIds(U32BIT country_code, U8BIT region_id, U8BIT audio_lang_id, U8BIT sub_lang_id)
{
   U8BIT country_id;
   BOOLEAN country_changed;
   BOOLEAN nvm_changed;
   U8BIT index;
#ifdef COMMON_INTERFACE
   U8BIT *lang_ids;
#endif

   FUNCTION_START(ACFG_SetCountryIds);

   country_changed = FALSE;
   nvm_changed = FALSE;

   for (country_id = 0; country_id < NUM_COUNTRIES; country_id++)
   {
      if (country_data_table[country_id].country_code == country_code)
      {
         if (country_code != APP_NvmRead(COUNTRY_CODE_NVM))
         {
            APP_NvmSave(COUNTRY_CODE_NVM, country_code, FALSE);

#ifdef COMMON_INTERFACE
            /* Update the host country */
            CISetHostCountryCode(country_code);
#endif
            country_changed = TRUE;
            nvm_changed = TRUE;
         }

         /* Update the aerial power option */
         if (ACFG_GetAerialPowerOptionReqd(country_code))
         {
            if (APP_NvmRead(AERIAL_POWER_ON_NVM) != ACFG_GetDefaultAerialPower(country_code))
            {
               APP_NvmSave(AERIAL_POWER_ON_NVM, ACFG_GetDefaultAerialPower(country_code), FALSE);
               nvm_changed = TRUE;
            }
         }
         else
         {
            if (APP_NvmRead(AERIAL_POWER_ON_NVM) != FALSE)
            {
               APP_NvmSave(AERIAL_POWER_ON_NVM, FALSE, FALSE);
               nvm_changed = TRUE;
            }
         }

         /* Set the private data specifier code for the selected country to ensure
          * the correct SI descriptors are parsed. */
         STB_SISetCountryPrivateDataSpecifier(ACFG_GetPrivateDataSpecifier(country_code));
         STB_SISetNordigPrivateDataSpecifierMode(ACFG_IsNordigCountry());

         /* Clear any existing user defined SI functions and set them for the selected country */
         STB_SIClearUserDefinedDescriptorFunctions();

         if (country_data_table[country_id].si_descs != NULL)
         {
            for (index = 0; country_data_table[country_id].si_descs[index].table_id != 0x00; index++)
            {
               STB_SISetUserDefinedDescriptorFunction(country_data_table[country_id].si_descs[index].table_id,
                  country_data_table[country_id].si_descs[index].func);
            }
         }

         if (country_changed || (audio_lang_id != (U8BIT)APP_NvmRead(PRIMARY_AUDIO_LANG_NVM)))
         {
            APP_NvmSave(PRIMARY_AUDIO_LANG_NVM, (U32BIT)audio_lang_id, FALSE);

            /* Set audio language */
            ADB_SetAudioLang(country_code, audio_lang_id);

#ifdef COMMON_INTERFACE
            /* Update the language code */
            lang_ids = ACFG_GetDbLangId(country_code, audio_lang_id);

            CISetHostLanguageCode(ACFG_ConvertLangIdToCode(lang_ids[0]));
#endif

            nvm_changed = TRUE;
         }

         if (country_changed || (sub_lang_id != (U8BIT)APP_NvmRead(PRIMARY_TEXT_LANG_NVM)))
         {
            APP_NvmSave(PRIMARY_TEXT_LANG_NVM, (U32BIT)sub_lang_id, FALSE);

            /* Set subtitle language */
            ADB_SetTextLang(country_code, sub_lang_id);

            nvm_changed = TRUE;
         }

         if (region_id != (U8BIT)APP_NvmRead(REGION_ID_NVM))
         {
            /* Save the region ID */
            APP_NvmSave(REGION_ID_NVM, (U32BIT)region_id, FALSE);
            nvm_changed = TRUE;
         }

         if (nvm_changed)
         {
            /* Save all updated info to NVM */
            APP_NvmSaveAllNow();
         }

         break;
      }
   }

   FUNCTION_FINISH(ACFG_SetCountryIds);
}

/**
 * @brief   Sets the primary language id to be used for audio. This is the language
 *          that will be chosen first, if available
 * @param   lang_id language id
 */
void ACFG_SetPrimaryAudioLangId(U8BIT lang_id)
{
   U32BIT country_code;

   FUNCTION_START(ACFG_SetPrimaryAudioLangId);

   if (lang_id != (U8BIT)APP_NvmRead(PRIMARY_AUDIO_LANG_NVM))
   {
      APP_NvmSave(PRIMARY_AUDIO_LANG_NVM, (U32BIT)lang_id, TRUE);

      /* Set audio and subtitle language */
      country_code = ACFG_GetCountry();
      if (GetCountryId(country_code) < NUM_COUNTRIES)
      {
         ADB_SetAudioLang(country_code, lang_id);
      }
   }

   FUNCTION_FINISH(ACFG_SetPrimaryAudioLangId);
}

/**
 * @brief   Sets the secondary language id to be used for audio.
 * @param   lang_id language id
 */
void ACFG_SetSecondaryAudioLangId(U8BIT lang_id)
{
   U32BIT country_code;

   FUNCTION_START(ACFG_SetSecondaryAudioLangId);

   if (lang_id != (U8BIT)APP_NvmRead(SECONDARY_AUDIO_LANG_NVM))
   {
      APP_NvmSave(SECONDARY_AUDIO_LANG_NVM, (U32BIT)lang_id, TRUE);

      /* Set secondary audio language */
      country_code = ACFG_GetCountry();
      if (GetCountryId(country_code) < NUM_COUNTRIES)
      {
         ADB_SetSecondaryAudioLang(country_code, lang_id);
      }
   }

   FUNCTION_FINISH(ACFG_SetSecondaryAudioLangId);
}

/**
 * @brief   Sets the primary language id to be used for teletext/subtitles.
 * @param   lang_id language id
 */
void ACFG_SetPrimaryTextLangId(U8BIT lang_id)
{
   U32BIT country_code;

   FUNCTION_START(ACFG_SetPrimaryTextLangId);

   if (lang_id != (U8BIT)APP_NvmRead(PRIMARY_TEXT_LANG_NVM))
   {
      APP_NvmSave(PRIMARY_TEXT_LANG_NVM, (U32BIT)lang_id, TRUE);

      /* Set subtitle language */
      country_code = ACFG_GetCountry();
      if (GetCountryId(country_code) < NUM_COUNTRIES)
      {
         ADB_SetTextLang(country_code, lang_id);
      }
   }

   FUNCTION_FINISH(ACFG_SetPrimaryTextLangId);
}

/**
 * @brief   Sets the secondary language id to be used for teletext/subtitles.
 * @param   lang_id language id
 */
void ACFG_SetSecondaryTextLangId(U8BIT lang_id)
{
   U32BIT country_code;

   FUNCTION_START(ACFG_SetSecondaryTextLangId);

   if (lang_id != (U8BIT)APP_NvmRead(SECONDARY_TEXT_LANG_NVM))
   {
      APP_NvmSave(SECONDARY_TEXT_LANG_NVM, (U32BIT)lang_id, TRUE);

      /* Set audio and subtitle language */
      country_code = ACFG_GetCountry();
      if (GetCountryId(country_code) < NUM_COUNTRIES)
      {
         ADB_SetSecondaryTextLang(country_code, lang_id);
      }
   }

   FUNCTION_FINISH(ACFG_SetSecondaryTextLangId);
}

/**
 * @brief   Returns an array containing the names of all the countries included
 *          in the DVB. The index into this array is referred to as the country_id.
 *          The returned array should be freed using ACFG_ReleaseCountryList.
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_countries_ptr pointer to the number of countries in the returned array
 */
void ACFG_GetCountryList(U8BIT ***str_array_ptr, U8BIT *num_countries_ptr)
{
   U8BIT **str_array;
   U8BIT num_entries;
   U8BIT i;

   FUNCTION_START(ACFG_GetCountryList);

   num_entries = NUM_COUNTRIES;
   str_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
   if (str_array != NULL)
   {
      for (i = 0; i < num_entries; i++)
      {
         str_array[i] = (U8BIT *)country_data_table[i].country_name;
      }
   }
   else
   {
      num_entries = 0;
   }

   *str_array_ptr = str_array;
   *num_countries_ptr = num_entries;

   FUNCTION_FINISH(ACFG_GetCountryList);
}

/**
 * @brief   Frees a country list previously acquired using ACFG_GetCountryList
 * @param   str_array array to be freed
 * @param   num_countries number of items in the array
 */
void ACFG_ReleaseCountryList(U8BIT **str_array, U8BIT num_countries)
{
   FUNCTION_START(ACFG_ReleaseCountryList);

   USE_UNWANTED_PARAM(num_countries);

   if (str_array != NULL)
   {
      STB_AppFreeMemory(str_array);
   }

   FUNCTION_FINISH(ACFG_ReleaseCountryList);
}

/**
 * @brief   Copies the country configuration for the given country into the provided structure.
 *          None of the included arrays should be freed.
 * @param   country_code 3 character country code
 * @param   config structure into which the configuration data will be copied
 * @return  TRUE if the country code is valid, FALSE otherwise
 */
BOOLEAN ACFG_GetCountryConfig(U32BIT country_code, ACFG_COUNTRY_CONFIG *config)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetCountryConfig);

   country_id = GetCountryId(country_code);
   if ((config != NULL) && (country_id < NUM_COUNTRIES))
   {
      memcpy(config, &country_data_table[country_id], sizeof(ACFG_COUNTRY_CONFIG));
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACFG_GetCountryConfig);

   return(retval);
}

/**
 * @brief   Sets the country configuration settings that will be used for given country.
 *          This could be used to override the built-in configuration for a country but would
 *          more normally be used to set the configuration for the user defined country, if it's
 *          been included. If the config is set for the current country then ACFG_SetCountryIds
 *          will be called to ensure any country settings are changed if necessary.
 * @param   country_code 3 character country code
 * @param   config configuration settings to be used for the country
 * @return  TRUE if the country code is valid, FALSE otherwise
 */
BOOLEAN ACFG_SetCountryConfig(U32BIT country_code, ACFG_COUNTRY_CONFIG *config)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_SetCountryConfig);

   country_id = GetCountryId(country_code);
   if ((config != NULL) && (country_id < NUM_COUNTRIES))
   {
      memcpy(&country_data_table[country_id], config, sizeof(ACFG_COUNTRY_CONFIG));

      if (ACFG_GetCountry() == country_code)
      {
         /* The country config that's been changed is the one that's selected;
          * set it again so default values are applied based on the config values that have just been set */
         ACFG_SetCountryIds(country_code, (U8BIT)APP_NvmRead(REGION_ID_NVM),
            (U8BIT)APP_NvmRead(PRIMARY_AUDIO_LANG_NVM), (U8BIT)APP_NvmRead(PRIMARY_TEXT_LANG_NVM));
      }

      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACFG_SetCountryConfig);

   return(retval);
}

/**
 * @brief   Returns the number of regions in the given country
 * @param   country_code country code
 * @return  number of regions
 */
U8BIT ACFG_GetNumRegions(U32BIT country_code)
{
   U8BIT num_regions;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetNumRegions);

   num_regions = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num_regions = country_data_table[country_id].num_regions;
   }

   FUNCTION_FINISH(ACFG_GetNumRegions);

   return(num_regions);
}

/**
 * @brief   Returns the number of languages defined for the given country that
 *          can be used for audio and subtitles/teletext.
 * @param   country_code country code
 * @return  number of available languages
 */
U8BIT ACFG_GetNumDbLanguages(U32BIT country_code)
{
   U8BIT num_db_languages;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetNumDbLanguages);

   num_db_languages = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num_db_languages = country_data_table[country_id].num_db_langs;
   }

   FUNCTION_FINISH(ACFG_GetNumDbLanguages);

   return(num_db_languages);
}

/**
 * @brief   Returns an array of pointers to strings containing the region names for
 *          the given country. The index into this array is the region_id.
 *          The array is allocated by this function and should be freed using
 *          ACFG_ReleaseRegionList.
 * @param   country_code country code
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_regions_ptr pointer to the number of regions in the returned array
 * @return  TRUE if the country_id is valid
 */
BOOLEAN ACFG_GetRegionList(U32BIT country_code, U8BIT ***str_array_ptr, U8BIT *num_regions_ptr)
{
   BOOLEAN retval;
   U8BIT num_regions;
   U8BIT num_entries;
   U8BIT **str_array;
   U8BIT country_id;
   ACFG_REGION_DATA *region_data;
   U8BIT i;

   FUNCTION_START(ACFG_GetRegionList);

   retval = FALSE;
   num_entries = 0;
   str_array = NULL;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num_regions = country_data_table[country_id].num_regions;
      region_data = country_data_table[country_id].region_data;
      if ((num_regions != 0) && (region_data != NULL))
      {
         str_array = STB_AppGetMemory(num_regions * sizeof(U8BIT *));
         if (str_array != NULL)
         {
            for (i = 0; i < num_regions; i++)
            {
               str_array[i] = region_data[i].name;
            }

            num_entries = num_regions;
         }
      }
      retval = TRUE;
   }

   *str_array_ptr = str_array;
   *num_regions_ptr = num_entries;

   FUNCTION_FINISH(ACFG_GetRegionList);
   return(retval);
}

/**
 * @brief   Frees the region array previously acquired using ACFG_GetRegionList
 * @param   str_array array to be freed
 * @param   num_regions number of items in the array
 */
void ACFG_ReleaseRegionList(U8BIT **str_array, U8BIT num_regions)
{
   FUNCTION_START(ACFG_ReleaseRegionList);

   USE_UNWANTED_PARAM(num_regions);

   if (str_array != NULL)
   {
      STB_AppFreeMemory(str_array);
   }

   FUNCTION_FINISH(ACFG_ReleaseRegionList);
}

/**
 * @brief   Sets the languages that can be used for setting the default primary and secondary audio,
 *          subtitle and teletext languages. These will override any languages that are defined in
 *          the country configuration. The currently set primary and secondary audio language ids
 *          will be reset to 0 if they're greater than the number of languages in the new array.
 * @param   country_code country code of the country whose languages are to be set
 * @param   languages array of languages. This array will be copied.
 * @param   num_languages number of items in the above language array
 * @return  TRUE if the country is found and the languages are set, FALSE otherwise
 */
BOOLEAN ACFG_SetDbLanguages(U32BIT country_code, ACFG_LANGUAGE_IDS *languages, U8BIT num_languages)
{
   BOOLEAN retval;
   U8BIT country_id;
   ACFG_LANGUAGE_IDS *lang_data;

   FUNCTION_START(ACFG_SetDbLanguages);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      if (num_languages != 0)
      {
         lang_data = STB_AppGetMemory(num_languages * sizeof(ACFG_LANGUAGE_IDS));
         if (lang_data != NULL)
         {
            memcpy(lang_data, languages, num_languages * sizeof(ACFG_LANGUAGE_IDS));

            country_data_table[country_id].num_db_langs = num_languages;
            country_data_table[country_id].db_lang_data = lang_data;
            retval = TRUE;
         }
      }
      else
      {
         country_data_table[country_id].num_db_langs = num_languages;
         country_data_table[country_id].db_lang_data = languages;
         retval = TRUE;
      }

      if (retval)
      {
         /* Check the language IDs that are set are still valid and reset to 0 if not */
         if (ACFG_GetPrimaryAudioLangId() >= num_languages)
         {
            ACFG_SetPrimaryAudioLangId(0);
         }

         if (ACFG_GetSecondaryAudioLangId() >= num_languages)
         {
            ACFG_SetSecondaryAudioLangId(0);
         }

         if (ACFG_GetPrimaryTextLangId() >= num_languages)
         {
            ACFG_SetPrimaryTextLangId(0);
         }

         if (ACFG_GetSecondaryTextLangId() >= num_languages)
         {
            ACFG_SetSecondaryTextLangId(0);
         }
      }
   }

   FUNCTION_FINISH(ACFG_SetDbLanguages);

   return(retval);
}

/**
 * @brief   Returns an array of pointers to strings containing the available language names for
 *          the given country. The index into this array is the lang_id used when
 *          getting and setting audio and text language ids.
 *          The array is allocated by this function and should be freed using
 *          ACFG_ReleaseDbLangList.
 * @param   country_code country code
 * @param   str_array_ptr pointer to an allocated array of static UTF-8 strings
 * @param   num_langs_ptr pointer to the number of languages in the returned array
 * @return  TRUE if the country_id is valid
 */
BOOLEAN ACFG_GetDbLangList(U32BIT country_code, U8BIT ***str_array_ptr, U8BIT *num_langs_ptr)
{
   BOOLEAN retval;
   U8BIT num_langs;
   U8BIT num_entries;
   U8BIT **str_array;
   U8BIT country_id;
   ACFG_LANGUAGE_IDS *lang_data;
   U8BIT i;

   FUNCTION_START(ACFG_GetDbLangList);

   retval = FALSE;
   num_entries = 0;
   str_array = NULL;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num_langs = country_data_table[country_id].num_db_langs;
      lang_data = country_data_table[country_id].db_lang_data;
      if ((num_langs != 0) && (lang_data != NULL))
      {
         str_array = STB_AppGetMemory(num_langs * sizeof(U8BIT *));
         if (str_array != NULL)
         {
            for (i = 0; i < num_langs; i++)
            {
               str_array[i] = (U8BIT *)language_strings[lang_data[i].language_id];
            }
            num_entries = num_langs;
         }
      }
      retval = TRUE;
   }

   *str_array_ptr = str_array;
   *num_langs_ptr = num_entries;

   FUNCTION_FINISH(ACFG_GetDbLangList);

   return(retval);
}

/**
 * @brief   Returns an array of language ids for a given country and index into the
 *          language array. A language id is one of the ACFG_DB_LANG_XXXX values above and there
 *          might be more than one for a given language. For example german language has two
 *          ids associated to it: ACFG_DB_LANG_GERMAN1 and ACFG_DB_LANG_GERMAN2 because german
 *          language can have two different codes, 'deu' and 'ger'. See also functions
 *          ACFG_ConvertLangIdToCode and ACFG_ConvertLangCodeToId. The returned pointer is an
 *          internal array and must not be freed.
 * @param   country_code country code
 * @param   lang_entry index into the language array
 * @return  pointer to the array of language ids, or NULL if either of the indices are invalid.
 */
U8BIT* ACFG_GetDbLangId(U32BIT country_code, U8BIT lang_entry)
{
   const U8BIT *lang_ids;
   U8BIT country_id;
   U8BIT num_langs;
   ACFG_LANGUAGE_IDS *lang_data;

   FUNCTION_START(ACFG_GetDbLangId);

   lang_ids = NULL;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num_langs = country_data_table[country_id].num_db_langs;
      lang_data = country_data_table[country_id].db_lang_data;
      if ((num_langs != 0) && (lang_data != NULL) && (lang_entry < num_langs))
      {
         lang_ids = lang_data[lang_entry].ids;
      }
   }

   FUNCTION_FINISH(ACFG_GetDbLangId);
   return (U8BIT*)lang_ids;
}

/**
 * @brief   Returns the language id for the given language code
 * @param   lang_code language code
 * @return  language id, or ACFG_INVALID_LANG if language code isn't found
 */
U8BIT ACFG_ConvertLangCodeToId(U32BIT lang_code)
{
   U8BIT lang_id;
   U8BIT i;

   FUNCTION_START(ACFG_ConvertLangCodeToId);

   lang_id = ACFG_INVALID_LANG;
   for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
   {
      if (db_lang_list[i] == lang_code)
      {
         lang_id = i;
         break;
      }
   }

   FUNCTION_FINISH(ACFG_ConvertLangCodeToId);
   return(lang_id);
}

/**
 * @brief   Returns the language code for the given language id
 * @param   lang_id language id
 * @return  language code, or 0 if language id isn't valid
 */
U32BIT ACFG_ConvertLangIdToCode(U8BIT lang)
{
   U32BIT lang_code;

   FUNCTION_START(ACFG_ConvertLangIdToCode);

   if (lang < ACFG_NUM_DB_LANGUAGES)
   {
      lang_code = db_lang_list[lang];
   }
   else
   {
      lang_code = 0;
   }
   FUNCTION_FINISH(ACFG_ConvertLangIdToCode);
   return(lang_code);
}

/**
 * @brief   Frees the language array previously acquired using ACFG_GetDbLangList
 * @param   str_array array of lang names to be freed
 * @param   num_langs number of names in the array
 */
void ACFG_ReleaseDbLangList(U8BIT **str_array, U8BIT num_langs)
{
   FUNCTION_START(ACFG_ReleaseDbLangList);

   USE_UNWANTED_PARAM(num_langs);

   if (str_array != NULL)
   {
      STB_AppFreeMemory(str_array);
   }

   FUNCTION_FINISH(ACFG_ReleaseDbLangList);
}

/**
 * @brief   Returns a pointer to the channel table for an analog tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetAnaRfChannelTable(U32BIT country_code, ACFG_ANA_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetAnaRfChannelTable);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      *rf_chan_data_ptr = (ACFG_ANA_RF_CHANNEL_DATA *) country_data_table[country_id].ana_rf_channel_table_ptr;
      *num_entries_ptr = country_data_table[country_id].num_ana_rf_channels;
      retval = TRUE;
   }

   FUNCTION_FINISH(ACFG_GetAnaRfChannelTable);
   return(retval);
}

/**
 * @brief   Returns a pointer to the channel table for a terrestrial tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetTerRfChannelTable(U32BIT country_code, ACFG_TER_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetTerRfChannelTable);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      *rf_chan_data_ptr = (ACFG_TER_RF_CHANNEL_DATA *) country_data_table[country_id].ter_rf_channel_table_ptr;
      *num_entries_ptr = country_data_table[country_id].num_ter_rf_channels;
      retval = TRUE;
   }

   FUNCTION_FINISH(ACFG_GetTerRfChannelTable);
   return(retval);
}

/**
 * @brief   Returns a pointer to the channel table for a cable tuner for the given country
 * @param   country_code country whose table is to be returned
 * @param   rf_chan_data_ptr returned pointer to the channel table
 * @param   num_entries_ptr number of entries in the returned channel table
 * @return  TRUE if the country is valid and data is returned, FALSE otherwise
 */
BOOLEAN ACFG_GetCabRfChannelTable(U32BIT country_code, ACFG_CAB_RF_CHANNEL_DATA **rf_chan_data_ptr,
   U16BIT *num_entries_ptr)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetCabRfChannelTable);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      *rf_chan_data_ptr = country_data_table[country_id].cab_rf_channel_table_ptr;
      *num_entries_ptr = country_data_table[country_id].num_cab_rf_channels;
      retval = TRUE;
   }

   FUNCTION_FINISH(ACFG_GetCabRfChannelTable);
   return(retval);
}

/**
 * @brief   Replaces the cable tuning table for the current country with the one provided
 * @param   cable_channel_data pointer to the new table to point
 * @param   number_channels number of entries in the channel table
 */
void ACFG_SetCableChannelTable(ACFG_CAB_RF_CHANNEL_DATA *cable_channel_data, U16BIT number_channels)
{
   U8BIT country_id;

   FUNCTION_START(ACFG_SetCableChannelTable);

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      country_data_table[country_id].cab_rf_channel_table_ptr = cable_channel_data;
      country_data_table[country_id].num_cab_rf_channels = number_channels;
   }

   FUNCTION_FINISH(ACFG_SetCableChannelTable);
}

/**
 * @brief   Returns the country code for the given country index
 * @param   country_id country index
 * @return  country code, or 0 if country id isn't valid
 */
U32BIT ACFG_GetCountryCode(U8BIT country_id)
{
   U32BIT code;

   FUNCTION_START(ACFG_GetCountryCode);

   code = 0;

   if (country_id < NUM_COUNTRIES)
   {
      code = country_data_table[country_id].country_code;
   }

   FUNCTION_FINISH(ACFG_GetCountryCode);
   return(code);
}

/**
 * @brief   Returns the private data specifier value for the given country
 * @param   country_code coutry code
 * @return  private data specifier value; 0 if one isn't defined for the country
 *          or country code isn't valid
 */
U32BIT ACFG_GetPrivateDataSpecifier(U32BIT country_code)
{
   U32BIT code;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetPrivateDataSpecifier);

   code = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      code = country_data_table[country_id].private_data_specifier;
   }

   FUNCTION_FINISH(ACFG_GetPrivateDataSpecifier);
   return(code);
}

/**
 * @brief   Returns the first LCN that should be used when assigning LCNs to services
 *          that don't appear in an LCN descriptor, or can't be assigned their desired LCN
 * @param   country_code country code
 * @param   tuner_type tuner type of the value to be returned
 * @return  LCN; 0 if country or tuner type aren't valid
 */
U16BIT ACFG_GetFirstUnallocatedLcn(U32BIT country_code, E_STB_DP_SIGNAL_TYPE tuner_type)
{
   U16BIT lcn;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetFirstUnallocatedLcn);

   lcn = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      switch (tuner_type)
      {
         case SIGNAL_COFDM:
            lcn = country_data_table[country_id].terr_first_unallocated_lcn;
            break;
         case SIGNAL_QAM:
            lcn = country_data_table[country_id].cab_first_unallocated_lcn;
            break;
         case SIGNAL_QPSK:
            lcn = country_data_table[country_id].sat_first_unallocated_lcn;
            break;
         default:
            break;
      }
   }

   FUNCTION_FINISH(ACFG_GetFirstUnallocatedLcn);
   return(lcn);
}

/**
 * @brief   Returns the last LCN that should be used when assigning LCNs to services
 *          that don't appear in an LCN descriptor, or can't be assigned their desired LCN
 * @param   country_code country code
 * @param   tuner_type tuner type of the value to be returned
 * @return  LCN; 0 if country or tuner type aren't valid
 */
U16BIT ACFG_GetLastUnallocatedLcn(U32BIT country_code, E_STB_DP_SIGNAL_TYPE tuner_type)
{
   U16BIT lcn;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetLastUnallocatedLcn);

   lcn = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      switch (tuner_type)
      {
         case SIGNAL_COFDM:
            lcn = country_data_table[country_id].terr_last_unallocated_lcn;
            break;
         case SIGNAL_QAM:
            lcn = country_data_table[country_id].cab_last_unallocated_lcn;
            break;
         case SIGNAL_QPSK:
            lcn = country_data_table[country_id].sat_last_unallocated_lcn;
            break;
         default:
            break;
      }
   }

   FUNCTION_FINISH(ACFG_GetLastUnallocatedLcn);
   return(lcn);
}

/**
 * @brief   Get the start and end times of the watershed hours for a country
 * @param   country_code country code
 * @param   start_time returns the start time in minutes since midnight
 * @param   end_time returns the end time in minutes since midnight
 * @return  TRUE if the start and end times are valid and have been returned, FALSE otherwise
 */
BOOLEAN ACFG_GetWatershedTimes(U32BIT country_code, U16BIT *start_time, U16BIT *end_time)
{
   BOOLEAN watershed_defined;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetWatershedTimes);

   watershed_defined = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      if ((country_data_table[country_id].start_watershed < 1440) &&
          (country_data_table[country_id].end_watershed < 1440))
      {
         *start_time = country_data_table[country_id].start_watershed;
         *end_time = country_data_table[country_id].end_watershed;
         watershed_defined = TRUE;
      }
   }

   FUNCTION_FINISH(ACFG_GetWatershedTimes);

   return(watershed_defined);
}

/**
 * @brief   Returns the minimum frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetMinSatelliteScanFreq(U32BIT country_code)
{
   U16BIT freq;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetMinSatelliteScanFreq);

   freq = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      freq = country_data_table[country_id].min_sat_freq;
   }

   FUNCTION_FINISH(ACFG_GetMinSatelliteScanFreq);

   return(freq);
}

/**
 * @brief   Returns the maximum frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetMaxSatelliteScanFreq(U32BIT country_code)
{
   U16BIT freq;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetMaxSatelliteScanFreq);

   freq = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      freq = country_data_table[country_id].max_sat_freq;
   }

   FUNCTION_FINISH(ACFG_GetMaxSatelliteScanFreq);

   return(freq);
}

/**
 * @brief   Returns the increment frequency to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  frequency in MHz; 0 if undefined or country is invalid
 */
U16BIT ACFG_GetSatelliteScanFreqInc(U32BIT country_code)
{
   U16BIT inc;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteScanFreqInc);

   inc = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      inc = country_data_table[country_id].sat_freq_inc;
   }

   FUNCTION_FINISH(ACFG_GetSatelliteScanFreqInc);

   return(inc);
}

/**
 * @brief   Returns a fixed array of symbol rates to be used when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  array of symbol rates in Kbps; NULL if undefined or country is invalid
 */
U16BIT* ACFG_GetSatelliteScanSymbolRates(U32BIT country_code)
{
   U16BIT *srates;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteScanSymbolRates);

   srates = NULL;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      srates = (U16BIT *)(country_data_table[country_id].sat_symbol_rates);
   }

   FUNCTION_FINISH(ACFG_GetSatelliteScanSymbolRates);

   return(srates);
}

/**
 * @brief   Returns the number of items in the fixed array of symbol rates to be
 *          used when performing a satellite based frequency scan in the given country
 * @param   country_code country code
 * @return  number of symbol rates; 0 if undefined or country is invalid
 */
U8BIT ACFG_GetSatelliteScanNumSymbolRates(U32BIT country_code)
{
   U8BIT num;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteScanNumSymbolRates);

   num = 0;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      num = country_data_table[country_id].num_symbol_rates;
   }

   FUNCTION_FINISH(ACFG_GetSatelliteScanNumSymbolRates);

   return(num);
}

/**
 * @brief   Returns whether DVB-S2 should be included when performing a satellite
 *          based frequency scan in the given country
 * @param   country_code country code
 * @return  TRUE if DVB-S2 should be used, FALSE otherwise
 */
BOOLEAN ACFG_GetSatelliteScanDvbS2(U32BIT country_code)
{
   BOOLEAN inc_dvbs2;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteScanDvbS2);

   inc_dvbs2 = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      inc_dvbs2 = country_data_table[country_id].include_dvbs2;
   }

   FUNCTION_FINISH(ACFG_GetSatelliteScanDvbS2);

   return(inc_dvbs2);
}

/**
 * @brief   Returns whether the aerial power option is required for DVB-T/T2
 *          for the given country
 * @param   country_code country code
 * @return  TRUE if required, FALSE otherwise
 */
BOOLEAN ACFG_GetAerialPowerOptionReqd(U32BIT country_code)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetAerialPowerOptionReqd);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].aerial_power_option_reqd;
   }

   FUNCTION_FINISH(ACFG_GetAerialPowerOptionReqd);
   return(retval);
}

/**
 * @brief   Returns the default aerial power option setting for DVB-T/T2 for
 *          the given country
 * @param   country_code country code
 * @return  TRUE or FALSE
 */
BOOLEAN ACFG_GetDefaultAerialPower(U32BIT country_code)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetDefaultAerialPower);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].default_aerial_power;
   }

   FUNCTION_FINISH(ACFG_GetDefaultAerialPower);
   return(retval);
}

/**
 * @brief   Returns whether an aerial tuning screen should be presented by the interface
 *          before performing a DVB-T/T2 scan for services for the given country
 * @param   country_code country code
 * @return  TRUE if required, FALSE otherwise
 */
BOOLEAN ACFG_GetAerialTuningScreenReqd(U32BIT country_code)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetAerialTuningScreenReqd);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].aerial_tuning_screen_reqd;
   }

   FUNCTION_FINISH(ACFG_GetAerialTuningScreenReqd);
   return(retval);
}

/**
 * @brief   Returns the region code that identifies the given region
 * @param   country_code country code
 * @param   region_id region index
 * @param   code_ptr pointer to value in which the code will be returned
 * @return  TRUE if the country and region indices are valid, FALSE otherwise
 */
BOOLEAN ACFG_GetRegionCode(U32BIT country_code, U8BIT region_id, U8BIT *code_ptr)
{
   BOOLEAN retval;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetRegionCode);

   retval = FALSE;

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      if (region_id < country_data_table[country_id].num_regions)
      {
         *code_ptr = country_data_table[country_id].region_data[region_id].code;
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(ACFG_GetRegionCode);
   return(retval);
}

/**
 * @brief   Returns the event content types for the given country. This defines how
 *          the content type value broadcast as part of the EIT is to be interpreted
 * @param   country_code country code
 * @return  pointer to the content type array. The index of this array is level_1 from the
 *          event content descriptors in the EIT and its elements are of type ADB_EVENT_CONTENT,
 *          defined in ap_dbacc.h
 */
U8BIT* ACFG_GetEventContentTypes(U32BIT country_code)
{
   const U8BIT *types;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetEventContentTypes);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      types = country_data_table[country_id].event_content_types;
   }
   else
   {
      types = NULL;
   }

   FUNCTION_FINISH(ACFG_GetEventContentTypes);

   return (U8BIT*)types;
}

/**
 * @brief   Sets the start and end times during which background searches are allowed
 *          run when in standby
 * @param   start_time start time in minutes since midnight
 * @param   end_time end time in minutes
 */
void ACFG_SetBackgroundSearchTime(U16BIT start_time, U16BIT end_time)
{
   FUNCTION_START(ACFG_SetBackgroundSearchTime);

   if ((end_time > start_time) && (end_time < (24 * 60)))
   {
      APP_NvmSave(BACKGROUND_SEARCH_START_NVM, (U32BIT)start_time, FALSE);
      APP_NvmSave(BACKGROUND_SEARCH_END_NVM, (U32BIT)end_time, TRUE);
   }

   FUNCTION_FINISH(ACFG_SetBackgroundSearchTime);
}

/**
 * @brief   Gets the background start and end search times
 * @param   start_time return start time in minutes since midnight
 * @param   end_time return end time in minutes
 */
void ACFG_GetBackgroundSearchTime(U16BIT *start_time, U16BIT *end_time)
{
   FUNCTION_START(ACFG_GetBackgroundSearchTime);

   if (start_time != NULL)
   {
      *start_time = (U16BIT)APP_NvmRead(BACKGROUND_SEARCH_START_NVM);
   }

   if (end_time != NULL)
   {
      *end_time = (U16BIT)APP_NvmRead(BACKGROUND_SEARCH_END_NVM);
   }

   FUNCTION_FINISH(ACFG_GetBackgroundSearchTime);
}

/**
 * @brief   Enables or disables the background service search when in standby
 * @param   enabled TRUE to enable, FALSE to disable
 */
void ACFG_SetBackgroundServiceSearch(BOOLEAN enabled)
{
   FUNCTION_START(ACFG_SetBackgroundServiceSearch);

   APP_NvmSave(SERVICE_SEARCH_ENABLED_NVM, (U32BIT)enabled, TRUE);

   FUNCTION_FINISH(ACFG_SetBackgroundServiceSearch);
}

/**
 * @brief   Returns whether the background service search is enabled or not
 * @return  TRUE if enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetBackgroundServiceSearch(void)
{
   BOOLEAN enabled;

   FUNCTION_START(ACFG_GetBackgroundServiceSearch);

   enabled = (BOOLEAN)APP_NvmRead(SERVICE_SEARCH_ENABLED_NVM);

   FUNCTION_FINISH(ACFG_GetBackgroundServiceSearch);

   return(enabled);
}

/**
 * @brief   Enables or disables the background SSU  search when in standby
 * @param   enabled TRUE to enable, FALSE to disable
 */
void ACFG_SetBackgroundSSUSearch(BOOLEAN enabled)
{
   FUNCTION_START(ACFG_SetBackgroundSSUSearch);

   APP_NvmSave(SSU_SEARCH_ENABLED_NVM, (U32BIT)enabled, TRUE);

   FUNCTION_FINISH(ACFG_SetBackgroundSSUSearch);
}

/**
 * @brief   Returns whether the background SSU search is enabled or not
 * @return  TRUE if enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetBackgroundSSUSearch(void)
{
   BOOLEAN enabled;

   FUNCTION_START(ACFG_GetBackgroundSSUSearch);

   enabled = (BOOLEAN)APP_NvmRead(SSU_SEARCH_ENABLED_NVM);

   FUNCTION_FINISH(ACFG_GetBackgroundSSUSearch);

   return(enabled);
}

/**
 * @brief   Returns whether the current country requires Nordig compliance for SI
 * @return  TRUE if Nordig, FALSE otherwise
 */
BOOLEAN ACFG_IsNordigCountry(void)
{
   U8BIT country_id;
   BOOLEAN retval;

   FUNCTION_START(ACFG_IsNordigCountry);

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].nordig_compliant;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACFG_IsNordigCountry);
   return(retval);
}

/**
 * @brief   Returns whether the broadcast SI data for the given tuner type is to be treated
 *          as Nordig compliant. This is in relation to the current country
 * @param   tuner_type tuner type to be checked; SIGNAL_NONE is not a valid argument
 * @return  TRUE if SI data should comply with Nordig specifications, FALSE otherwise
 */
BOOLEAN ACFG_IsNordigService(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   U8BIT country_id;
   BOOLEAN retval;

   FUNCTION_START(ACFG_IsNordigService);

   retval = FALSE;

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      switch (tuner_type)
      {
         case SIGNAL_COFDM:
            retval = country_data_table[country_id].terr_nordig_service;
            break;
         case SIGNAL_QAM:
            retval = country_data_table[country_id].cab_nordig_service;
            break;
         case SIGNAL_QPSK:
            retval = country_data_table[country_id].sat_nordig_service;
            break;
         default:
            break;
      }
   }

   FUNCTION_FINISH(ACFG_IsNordigService);

   return(retval);
}

/**
 * @brief   Returns the number of EIT searches that should be performed per day
 *          when the box is in standby
 * @return  number of times
 */
U8BIT ACFG_GetEitSearchesPerDay(void)
{
   U8BIT retval = 0;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetEitSearchesPerDay);

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].eit_searches_per_day;
   }

   FUNCTION_FINISH(ACFG_GetEitSearchesPerDay);

   return retval;
}

/**
 * @brief   Returns the number of minutes after which all the events should have
 *          been received during an EIT search. This is the EIT repetition time.
 * @return  time in minutes
 */
U8BIT ACFG_GetEitUpdateTime(void)
{
   U8BIT retval = 0;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetEitUpdateTime);

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].eit_update_time;
   }

   FUNCTION_FINISH(ACFG_GetEitUpdateTime);

   return retval;
}

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-T/T2 scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
AllocLcnFunc ACFG_GetTerrestrialLcnFunction(U32BIT country_code)
{
   AllocLcnFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetTerrestrialLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].terr_lcn_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetTerrestrialLcnFunction);

   return(func_ptr);
}

/**
 * @brief   Overrides an existing DVB-T/T2 LCN allocation function for the given country
 * @param   country_code country code
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetTerrestrialLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr)
{
   U8BIT country_id;

   FUNCTION_START(ACFG_SetTerrestrialLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      country_data_table[country_id].terr_lcn_func = func_ptr;
   }

   FUNCTION_FINISH(ACFG_SetTerrestrialLcnFunction);
}

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-C scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
AllocLcnFunc ACFG_GetCableLcnFunction(U32BIT country_code)
{
   AllocLcnFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetCableLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].cab_lcn_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetCableLcnFunction);

   return(func_ptr);
}

/**
 * @brief   Overrides an existing DVB-C LCN allocation function for the given country
 * @param   country_code country code
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetCableLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr)
{
   U8BIT country_id;

   FUNCTION_START(ACFG_SetCableLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      country_data_table[country_id].cab_lcn_func = func_ptr;
   }

   FUNCTION_FINISH(ACFG_SetCableLcnFunction);
}

/**
 * @brief   Returns a pointer to a function that's used to assign LCNs following
 *          a DVB-S/S2 scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
AllocLcnFunc ACFG_GetSatelliteLcnFunction(U32BIT country_code)
{
   AllocLcnFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].sat_lcn_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetSatelliteLcnFunction);

   return(func_ptr);
}

/**
 * @brief   Overrides an existing DVB-S/S2 LCN allocation function for the given country
 * @param   country_code country code
 * @param   pointer to function, can be NULL if no function is to be called
 */
void ACFG_SetSatelliteLcnFunction(U32BIT country_code, AllocLcnFunc func_ptr)
{
   U8BIT country_id;

   FUNCTION_START(ACFG_SetSatelliteLcnFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      country_data_table[country_id].sat_lcn_func = func_ptr;
   }

   FUNCTION_FINISH(ACFG_SetSatelliteLcnFunction);
}

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-T/T2 scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
DBTidyFunc ACFG_GetTerrestrialDBTidyFunction(U32BIT country_code)
{
   DBTidyFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetTerrestrialDBTidyFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].terr_db_tidy_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetTerrestrialDBTidyFunction);

   return(func_ptr);
}

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-C scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
DBTidyFunc ACFG_GetCableDBTidyFunction(U32BIT country_code)
{
   DBTidyFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetCableDBTidyFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].cab_db_tidy_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetCableDBTidyFunction);

   return(func_ptr);
}

/**
 * @brief   Returns a pointer to a function that's used to tidy up the database
 *          following a DVB-S/S2 scan for the given country
 * @param   country_code country code
 * @return  pointer to functions, or NULL
 */
DBTidyFunc ACFG_GetSatelliteDBTidyFunction(U32BIT country_code)
{
   DBTidyFunc func_ptr;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetSatelliteDBTidyFunction);

   country_id = GetCountryId(country_code);
   if (country_id < NUM_COUNTRIES)
   {
      func_ptr = country_data_table[country_id].sat_db_tidy_func;
   }
   else
   {
      func_ptr = NULL;
   }

   FUNCTION_FINISH(ACFG_GetSatelliteDBTidyFunction);

   return(func_ptr);
}

/**
 * @brief   Returns the default secondary language for audio and subtitles as
 *          defined for the selected country. This is used for countries that
 *          don't use the secondary language, but expect a defined language to be
 *          used if the primary one isn't available.
 * @return  language id, or ACFG_INVALID_LANG when the value is not defined
 */
U8BIT ACFG_GetDefaultSecondaryLangId(void)
{
   U8BIT retval = ACFG_INVALID_LANG;
   U8BIT country_id;

   FUNCTION_START(ACFG_GetDefaultSecondaryLangId);

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      retval = country_data_table[country_id].default_secondary_language;
   }

   FUNCTION_FINISH(ACFG_GetDefaultSecondaryLangId);

   return retval;
}

static U8BIT GetCountryId(U32BIT country_code)
{
   U8BIT id;

   FUNCTION_START(GetCountryId);

   for (id = 0; id < NUM_COUNTRIES; id++)
   {
      if (country_data_table[id].country_code == country_code)
      {
         break;
      }
   }

   FUNCTION_FINISH(GetCountryId);

   return(id);
}

/**
 * @brief   Sets the language code to be used when the SI descriptors contain 'und', 'qaa' or don't
 *          contain any language descriptors.
 * @param   lang_code Language code to be use when the language is undefined. ILanguage codes that
 *          are not present in db_lang_list[] (defined in ap_cfdat.h), except ADB_LANG_CODE_UNDEF,
 *          will be ignored.
 */
void ACFG_SetUndefinedLanguageBehaviour(U32BIT lang_code)
{
   U8BIT lang_id;

   FUNCTION_START(ACFG_SetUndefinedLanguageBehaviour);

   lang_id = ACFG_ConvertLangCodeToId(lang_code);
   if ((lang_id != ACFG_INVALID_DB_LANG) || (lang_code == ADB_LANG_CODE_UNDEF))
   {
      und_lang_code = lang_code;
   }

   FUNCTION_FINISH(ACFG_SetUndefinedLanguageBehaviour);
}

/**
 * @brief   Gets the language code set by ACFG_SetUndefinedLanguageBehaviour, or 'und' if that
 *          function has not been called.
 * @return  Language code currently used when the SI descriptors don't define a language
 */
U32BIT ACFG_GetUndefinedLanguageBehaviour(void)
{
   FUNCTION_START(ACFG_GetUndefinedLanguageBehaviour);
   FUNCTION_FINISH(ACFG_GetUndefinedLanguageBehaviour);
   return (und_lang_code);
}

/**
 * @brief   Sets the host and country module for CI stack. Value zero for disable.
 * @param   module Module Id given by CI stack
 */
void ACFG_SetHostCountryLanguageModuleId(U32BIT module)
{
   FUNCTION_START(ACFG_SetHostCountryLanguageModuleId);
#ifdef COMMON_INTERFACE
   ci_hcl_module = module;
#else
   USE_UNWANTED_PARAM(module);
#endif
   FUNCTION_FINISH(ACFG_SetHostCountryLanguageModuleId);
}

/**
 * @brief   Use to check whether a dynamic SI update type is enabled for the currently configured
 *          country and given signal type and original network ID. If an array item is found with an
 *          original network ID of ADB_INVALID_DVB_ID then this item will be used.
 * @param   signal_type terrestrial, satellite or cable
 * @param   onet_id original network ID to be checked
 * @param   update_type see whether this type of update is enabled
 * @return  TRUE if the update is enabled, FALSE otherwise
 */
BOOLEAN ACFG_GetDynamicSIUpdate(E_STB_DP_SIGNAL_TYPE signal_type, U16BIT onet_id,
   E_ACFG_DYNAMIC_SI_UPDATE_TYPE update_type)
{
   BOOLEAN enabled;
   U8BIT country_id;
   ACFG_DYNAMIC_SI_UPDATES *si_updates;
   U8BIT index;

   FUNCTION_START(ACFG_GetDynamicSIUpdate);

   enabled = FALSE;

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      switch (signal_type)
      {
         case SIGNAL_COFDM:
            si_updates = country_data_table[country_id].terr_dynamic_si_updates;
            break;
         case SIGNAL_QAM:
            si_updates = country_data_table[country_id].cab_dynamic_si_updates;
            break;
         case SIGNAL_QPSK:
            si_updates = country_data_table[country_id].sat_dynamic_si_updates;
            break;
         default:
            si_updates = NULL;
            break;
      }

      if (si_updates != NULL)
      {
         for (index = 0; si_updates[index].update_types != ACFG_DYNAMIC_SI_UPDATE_NONE; index++)
         {
            if ((si_updates[index].original_network_id == ADB_INVALID_DVB_ID) ||
               (si_updates[index].original_network_id == onet_id))
            {
               /* Found the required dynamic SI update settings */
               if ((si_updates[index].update_types & update_type) != 0)
               {
                  enabled = TRUE;
               }
               break;
            }
         }
      }
   }

   FUNCTION_FINISH(ACFG_GetDynamicSIUpdate);

   return(enabled);
}

/**
 * @brief   Sets the dynamic SI updates for the given tuner type of the currently configured country
 * @param   signal_type terrestrial, satellite or cable
 * @param   dynamic_updates array of SI update types; the content of this array isn't copied so must
 *                          continue to exist after it's set. The last element of the array must be an
 *                          entry with the update type of ACFG_DYNAMIC_SI_UPDATE_NONE; the original network ID
 *                          for this entry will be ignored so can be any value.
 * @param   num_updates number of SI update entries in the provided array
 * @return  TRUE if the country and signal type are valid, FALSE otherwise
 */
BOOLEAN ACFG_SetDynamicSIUpdates(E_STB_DP_SIGNAL_TYPE signal_type, ACFG_DYNAMIC_SI_UPDATES *dynamic_updates,
   U8BIT num_updates)
{
   BOOLEAN retval;
   U8BIT country_id;
   ACFG_DYNAMIC_SI_UPDATES **si_updates;

   FUNCTION_START(ACFG_SetDynamicSIUpdates);

   retval = FALSE;

   country_id = GetCountryId(APP_NvmRead(COUNTRY_CODE_NVM));
   if (country_id < NUM_COUNTRIES)
   {
      switch (signal_type)
      {
         case SIGNAL_COFDM:
            si_updates = &country_data_table[country_id].terr_dynamic_si_updates;
            break;
         case SIGNAL_QAM:
            si_updates = &country_data_table[country_id].cab_dynamic_si_updates;
            break;
         case SIGNAL_QPSK:
            si_updates = &country_data_table[country_id].sat_dynamic_si_updates;
            break;
         default:
            si_updates = NULL;
            break;
      }

      if (si_updates != NULL)
      {
         *si_updates = (ACFG_DYNAMIC_SI_UPDATES *)STB_AppGetMemory(num_updates * sizeof(ACFG_DYNAMIC_SI_UPDATES));
         if (*si_updates != NULL)
         {
            memcpy(*si_updates, dynamic_updates, num_updates * sizeof(ACFG_DYNAMIC_SI_UPDATES));
            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(ACFG_SetDynamicSIUpdates);

   return(retval);
}

