/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CI control functions
 * @file    ap_ci.c
 * @date    February 2011
 * @author  Ocean Blue
 */

/*#define CI_DEBUG*/

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwci.h"
#include "stbhwos.h"
#include "stbhwtun.h"

#include "stbdpc.h"
#include "stbpvr.h"
#include "stberc.h"
#include "stbheap.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbgc.h"
#include "stberc.h"
#include "ap_tmr.h"

#include "dba.h"

#include "app.h"
#include "ap_cfg.h"
#include "ap_events.h"
#include "ap_dbacc.h"
#include "ap_dbdef.h"
#include "ap_ci.h"
#include "ap_cntrl.h"
#include "ap_state.h"
#include "ap_pvr.h"
#include "app_nvm.h"

#include "stbci.h"
#include "stbcica.h"
#include "stbcicc.h"
#include "stbcikeys.h"
#include "ap_ci_int.h"
#include "ap_cihc.h"
#include "ap_ciop.h"

//---constant definitions for this file----------------------------------------
#ifdef CI_DEBUG
   #define AP_CI_PRINT(x)  STB_SPDebugWrite x
#else
   #define AP_CI_PRINT(x)
#endif

#define MAX_EVENTS         100

#define TASK_STACK_SIZE    4096
#define TASK_PRIORITY      11

//---local typedefs, structs, enumerations for this file--------------------------------------------

typedef enum
{
   CIPLUS_TUNE_TYPE_SERVICE,
   CIPLUS_TUNE_TYPE_TRANSPORT,
   CIPLUS_TUNE_TYPE_CI_SERVICE
} E_CIPLUS_TUNE_TYPE;

struct s_ciplus_tune_transport
{
   void *t_ptr;
   E_ACTL_SI_SRCH_REQD reqd_si;
};

struct s_ciplus_tune_service
{
   void *s_ptr;
   E_ACTL_SI_SRCH_REQD reqd_si;
   U8BIT *pmt;
};

typedef struct
{
   U32BIT module;
   E_CIPLUS_TUNE_TYPE type;
   E_CIP_TUNE_FLAGS flags;
   union
   {
      struct s_ciplus_tune_service service;
      struct s_ciplus_tune_transport transport;
   } u;
} S_CIPLUS_TUNE_DATA;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)
static U8BIT cam_upgrade_mode = STB_CI_UPGRADE_REPLY_ASK;
static E_CIP_START_OPERATOR_SEARCH operator_search_mode = CIP_START_OPERATOR_SEARCH_ASK;
static BOOLEAN operator_search_required = FALSE;
static U32BIT operator_search_module;
static void *operator_search_service = NULL;
static void *ciplus_event_queue = NULL;
static F_NotifyStartTuning NotifyStartTuning = NULL;


//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

/**
 * @brief  Applies usage rules to path
 * @param  path decode path
 * @param  service_id service ID for URI
 */
static void ApplyUsageRules(U8BIT path, void *s_ptr)
{
   U8BIT slot_id;
   U16BIT service_id;
   U8BIT *pmt_data;
   U16BIT data_len;
   U8BIT uri[CIP_URI_LEN] = { 0 };

   FUNCTION_START(ApplyUsageRules);

   AP_CI_PRINT(("[ACI] ApplyUsageRules: path=%d, service_id=0x%x", path, ADB_GetServiceId(s_ptr)));

   if (STB_DPIsDecodingPath(path))
   {
      slot_id = STB_DPGetPathCISlot(path);
      if (slot_id != INVALID_RES_ID && STB_CiCcAuthenticated(slot_id))
      {
         AP_CI_PRINT(("[ACI] Slot %u is used on path", slot_id));
         pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
         if (STB_CiCaSystemSupported(slot_id, pmt_data))
         {
            /* Service is scrambled, CAS supported and CAM is CI+, so
             * need to get the URI and apply it */
            AP_CI_PRINT(("[ACI] Service scrambled, CAS supported, CI+ CAM"));
            service_id = ADB_GetServiceId(s_ptr);
            STB_CiCcGetUsageRulesInfo(slot_id, service_id, uri);
         }
      }

      /* Apply the URI to the AV outputs (regardless of slot status) */
      STB_CiCcApplyUsageRulesInfo(uri);
   }

   FUNCTION_FINISH(ApplyUsageRules);
}

/**
 * @brief  Update the URI for any recording that is currently on the decode path
 * @param  path decode path
 * @param  service_id service ID for URI
 */
static void UpdateRecordingURI(U8BIT path, void *s_ptr)
{
   U32BIT rec_handle;
   U8BIT slot_id;
   U16BIT service_id;
   U8BIT *pmt_data;
   U16BIT data_len;
   U8BIT uri[CIP_URI_LEN] = { 0 };

   FUNCTION_START(UpdateRecordingURI);

   AP_CI_PRINT(("[ACI] UpdateRecordingURI: path = %d", path));

   /* Check recordings on the path */
   if (STB_DPIsRecording(path, &rec_handle))
   {
      AP_CI_PRINT(("[ACI] Recording on the path"));
      slot_id = STB_DPGetPathCISlot(path);
      if (slot_id != INVALID_RES_ID && STB_CiCcAuthenticated(slot_id))
      {
         AP_CI_PRINT(("[ACI] Slot %u used for recording", slot_id));
         pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
         if (STB_CiCaSystemSupported(slot_id, pmt_data))
         {
            /* Service is scrambled, CAS supported and CAM is CI+, so
             * need to get the URI and store it with the recording */
            AP_CI_PRINT(("[ACI] URI stored with recording"));
            service_id = ADB_GetServiceId(s_ptr);
            STB_CiCcGetUsageRulesInfo(slot_id, service_id, uri);
            STB_PVRRecordingAddURI(rec_handle, uri);
         }
      }
   }

   FUNCTION_FINISH(UpdateRecordingURI);
}

/**
 * @brief   Called when an operator search has finished. Returns the live path to
 *          the service that was being viewed before the search started.
 */
static void OperatorSearchFinished(void)
{
   U8BIT path;

   FUNCTION_START(OperatorSearchFinished);

   /* The live path used for searching will be owned by CI+, so release it before returning
    * to the previous service */
   for (path = 0; path < STB_DPGetNumPaths(); path++)
   {
      if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
      {
         (void)STB_DPReleasePath(path, RES_OWNER_CIPLUS);
      }
   }

   if (operator_search_service != NULL)
   {
      ACTL_TuneToService(INVALID_RES_ID, NULL, operator_search_service, FALSE, TRUE);
      operator_search_service = NULL;
   }

   FUNCTION_FINISH(OperatorSearchFinished);
}

/**
 * @brief  Apply the CC keys for the given slot
 * @param  slot_id slot id
 */
static void ApplyCCKeys(U8BIT slot_id)
{
   U8BIT path;
   void *s_ptr;
   U8BIT *pmt_data;
   U16BIT data_len;

   FUNCTION_START(ApplyCCKeys);
   AP_CI_PRINT(("[ACI] ApplyCCKeys(%u)", slot_id));
   if (slot_id != INVALID_RES_ID && STB_CiCcAuthenticated(slot_id))
   {
      /* A slot can be shared with more than one path,
       * so handle this event for all paths using this slot */
      path = INVALID_RES_ID;
      while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
      {
         s_ptr = ADB_GetTunedService(path);
         if (s_ptr != NULL)
         {
            pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
            if (STB_CiCaSystemSupported(slot_id, pmt_data))
            {
               /* Apply content control keys (if any) on path */
               STB_CiKeysApply(path, slot_id);
            }
         }
      }
   }
   FUNCTION_FINISH(ApplyCCKeys);
}

static void ReleaseCISlotFromPath(U8BIT path)
{
   void *service;
   U16BIT service_id;
   U8BIT other, slot_id;

   FUNCTION_START(ReleaseCISlotFromPath);

   STB_DPReleaseCISlotFromPath(path);
   ACTL_ReleasePathOwnership(path, RES_OWNER_CIPLUS);

   /* The CI slot is being used by at least one other path. The slot may have just been
    * released from a live path but is still in use on a recording path, so the recording
    * mode would need to be changed to indicate it is now unattended */
   slot_id = STB_DPGetPathCISlot(path);
   other = STB_DPIsCISlotInUse(0, slot_id, path);
   if (other != INVALID_RES_ID)
   {
      service = STB_DPGetTunedService(path);
      if (service != NULL)
      {
         service_id = ADB_GetServiceId(service);
      }
      else
      {
         service_id = 0xffff;
      }
      STB_CiCcSetRecordOperatingMode(slot_id, STB_DPIsLivePath(other)?
         STB_CI_MODE_WATCH_AND_BUFFER : STB_CI_MODE_UNATTENDED_RECORDING, service_id);
   }

   /* And clear any CI+ keys */
   STB_CiKeysClear(path, slot_id);

   FUNCTION_FINISH(ReleaseCISlotFromPath);
}

/**
 * @brief   CAM removed from CI slot
 * @param   slot_id slot on which cam is removed
 */
static void CISlotRemoved(U8BIT slot_id)
{
   U8BIT path;
   FUNCTION_START(CISlotRemoved);
   if (slot_id != INVALID_RES_ID)
   {
      ACI_OpSlotRemove(slot_id);
      path = INVALID_RES_ID;
      while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
      {
         /* Slot is being used but shouldn't be anymore */
         STB_DPReleaseCISlotFromPath(path);
         ACTL_ReleasePathOwnership(path, RES_OWNER_CIPLUS);
         STB_CiKeysClear(path, slot_id);
      }
      STB_CiCcRemove(slot_id);
   }
   FUNCTION_FINISH(CISlotRemoved);
}

/**
 * @brief   Re-evaluate current state follwing a change related to CI slots
 * @param   slot_id slot on which the status has changed
 */
static void CISlotStatusChanged(U8BIT slot_id)
{
   U8BIT path;
   void *s_ptr;
   U8BIT *pmt_data, *ci_prot;
   U16BIT data_len;

   FUNCTION_START(CISlotStatusChanged);

   if (slot_id != INVALID_RES_ID)
   {
      AP_CI_PRINT(("CISlotStatusChanged(%u)", slot_id));

      if (STB_CiCcIsSlotReady(slot_id))
      {
         /* See if the slot is being used */
         path = STB_DPIsCISlotInUse(INVALID_RES_ID, slot_id, INVALID_RES_ID);
         if (path != INVALID_RES_ID)
         {
            /* A slot can be shared with more than one path,
             * so handle this event for all paths using this slot */
            path = INVALID_RES_ID;
            while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
            {
               s_ptr = STB_DPGetTunedService(path);
               if (s_ptr != NULL)
               {
                  ci_prot = ADB_GetServiceCIProtectionDesc(s_ptr);
                  if (!STB_CiCcIsServiceAllowed(slot_id,ci_prot))
                  {
                     /* TS isn't allowed to be routed through this slot anymore */
                     ReleaseCISlotFromPath(path);
                  }
                  else if (STB_CiCcAuthenticated(slot_id))
                  {
                     pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
                     /* Check whether the keys need to be applied */
                     if (STB_CiCaSystemSupported(slot_id,pmt_data))
                     {
                        /* Apply content control keys (if any) on path */
                        STB_CiKeysApply(path, slot_id);
                     }
                  }
               }
            }
         }
         else
         {
            /* CAM inserted, check whether TS can be routed through it */
            if (STB_CiCcIsServiceAllowed(slot_id,NULL))
            {
               /* If recording of scrambled services is to be allowed then all paths should
                * be checked to see if any of them are being used for recording, whether they
                * are already using a CI slot, and if not, whether the CAM in this slot supports
                * the CA system for the service being recorded. If all this is true then the
                * slot should be assigned to that path, but is it valid to do this if a recording
                * has already been started?
                * Not being done at the moment, just apply it to the live path */
               if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
               {
                  if ((s_ptr = ADB_GetTunedService(path)) != NULL)
                  {
                     /* Route live path through the slot */
                     ACI_AcquireCISlot(path, s_ptr);
                  }
               }
            }
         }
      }
      else
      {
         /* CAM may have been removed from slot, check if any paths are using this slot */
         path = INVALID_RES_ID;
         while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
         {
            /* Slot is being used but shouldn't be anymore */
            ReleaseCISlotFromPath(path);
         }
      }
   }

   FUNCTION_FINISH(CISlotStatusChanged);
}

/**
 * @brief   Handles a CICAM licence update when recording, which results in the
 *          licence being saved with the recording for use during playback.
 * @param   path decode path
 * @param   slot_id CI slot
 */
static void RecordLicenceChanged(U8BIT path, U8BIT slot_id)
{
   U32BIT handle;
   U8BIT cicam_id[CIP_CICAM_ID_LEN];
   E_STB_CI_LICENSE_RCVD_STATUS status;
   U8BIT *licence;
   U16BIT licence_len;
   U8BIT licence_status;
   void *s_ptr;
   U8BIT *pmt_data;
   U16BIT data_len;
   U8BIT uri[CIP_URI_LEN];

   FUNCTION_START(RecordLicenceChanged);

   if (STB_DPIsRecording(path, &handle))
   {
      AP_CI_PRINT(("%s(path=%u, slot=%u): Updating recording 0x%08lx", __FUNCTION__, path,
                   slot_id, handle));

      status = STB_CI_LICENSE_RCVD_OK;

      if (STB_CIGetCicamId(slot_id, cicam_id))
      {
         /* Store the CICAM id with the recording */
         if (!STB_PVRRecordingSetCicamId(handle, cicam_id, sizeof(cicam_id)))
         {
            status = STB_CI_LICENSE_RCVD_HOST_ERROR;
         }
      }

      if (status == STB_CI_LICENSE_RCVD_OK)
      {
         if ((licence = STB_CiCcGetRecordingLicence(slot_id, &licence_status, &licence_len, uri)) != NULL)
         {
            s_ptr = ADB_GetTunedService(path);
            if (s_ptr != NULL)
            {
               if (STB_DPIsDecodingPath(path))
               {
                  /* Check the licence status to make sure it's still
                   * valid for the output to be viewed */
                  if (licence_status == STB_CI_LICENSE_OK)
                  {
                     /* In watch and buffer mode, the service can be viewed */
                     STB_AVBlankVideo(0, FALSE);
                  }
                  else
                  {
                     /* Problem with the licence, ensure the video is blanked and inform the app */
                     STB_AVBlankVideo(0, TRUE);
                  }
               }

               if (STB_CiCcAuthenticated(slot_id))
               {
                  pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
                  if (STB_CiCaSystemSupported(slot_id, pmt_data))
                  {
                     /* Service is scrambled, CAS supported and CAM is CI+, so
                      * URI needs to be applied and saved with the recording */
                     if (STB_DPIsDecodingPath(path))
                     {
                        STB_CiCcApplyUsageRulesInfo(uri);
                     }

                     AP_CI_PRINT(("%s(%u, %u): Adding URI to recording 0x%08lx", __FUNCTION__, path,
                                  slot_id, handle));

                     if (!STB_PVRRecordingAddURI(handle, uri))
                     {
                        status = STB_CI_LICENSE_RCVD_HOST_ERROR;
                        AP_CI_PRINT(("%s(%u, %u): Failed to save URI with recording 0x%08lx",
                                     __FUNCTION__, path, slot_id, handle));
                     }
                  }
               }
            }

            /* Store the licence with the recording */
            AP_CI_PRINT(("%s(%u, %u): Adding licence, len %u, to recording 0x%08lx", __FUNCTION__,
                         path, slot_id, licence_len, handle));

            if ((status == STB_CI_LICENSE_RCVD_OK) &&
                !STB_PVRRecordingAddLicence(handle, licence, licence_len))
            {
               status = STB_CI_LICENSE_RCVD_HOST_ERROR;
               AP_CI_PRINT(("%s(%u, %u): Failed to save licence with recording 0x%08lx",
                            __FUNCTION__, path, slot_id, handle));
            }
         }
         else
         {
            status = STB_CI_LICENSE_RCVD_HOST_ERROR;
            AP_CI_PRINT(("%s(%u, %u): Failed to get licence data from slot", __FUNCTION__, path, slot_id));
         }
      }

      /* Reply to the receipt of the licence */
      STB_CISendLicenseReceivedStatus(slot_id, status);
   }

   FUNCTION_FINISH(RecordLicenceChanged);
}

/**
 * @brief   Handles a CICAM licence update during playback, which may result in the
 *          app being informed that the licence is no longer valid for the recording
 * @param   slot_id CI slot
 */
static void PlaybackLicenceChanged(U8BIT slot_id)
{
   U8BIT path;
   U32BIT handle;
   U8BIT cicam_id[CIP_CICAM_ID_LEN];
   E_STB_CI_LICENSE_RCVD_STATUS status;
   U8BIT *licence;
   U16BIT licence_len;
   U8BIT licence_status;
   U8BIT uri[CIP_URI_LEN];

   FUNCTION_START(PlaybackLicenceChanged);

   if ((path = STB_DPGetPlaybackPath()) != INVALID_RES_ID)
   {
      handle = APVR_GetPlaybackHandle();

      /* Check that the CICAM id stored with the recording is the same
       * as the CICAM id that's provided the licence */
      if (STB_PVRRecordingGetCicamId(handle, cicam_id, sizeof(cicam_id)))
      {
         if (STB_CiCcFindSlotForCicamId(cicam_id) == slot_id)
         {
            AP_CI_PRINT(("%s(slot=%u): Playback path %u, recording 0x%08lx", __FUNCTION__, slot_id, path, handle));

            if ((licence = STB_CiCcGetPlaybackLicence(slot_id, &licence_status, &licence_len, uri)) != NULL)
            {
               /* Update the URI and licence stored with the recording */
               if (!STB_PVRRecordingUpdateURI(handle, uri) ||
                   !STB_PVRRecordingUpdateLicence(handle, licence, licence_len))
               {
                  status = STB_CI_LICENSE_RCVD_HOST_ERROR;
               }
               else
               {
                  if (licence_status != STB_CI_LICENSE_OK)
                  {
                     /* Playback isn't allowed, so send an event to the UI app to inform it of the
                      * playback status. It would be nice if the user was offered the option of jumping
                      * to the start of the next licenced section, if any, but this is up to the app */
                     AP_CI_PRINT(("%s(%u): Licence status %u", __FUNCTION__, path, licence_status));
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_PLAYBACK_LICENCE_STATUS,
                        &licence_status, sizeof(licence_status));
                  }
                  else
                  {
                     /* Apply the URI in the recording to the outputs */
                     STB_CiCcApplyUsageRulesInfoForPlayback(uri);
                  }

                  status = STB_CI_LICENSE_RCVD_OK;
               }
            }
            else
            {
               status = STB_CI_LICENSE_RCVD_HOST_ERROR;
            }
         }
         else
         {
            status = STB_CI_LICENSE_RCVD_INVALID_DATA;
         }
      }
      else
      {
         status = STB_CI_LICENSE_RCVD_HOST_ERROR;
         AP_CI_PRINT(("%s(%u): Failed to read CICAM id from recording 0x%08lx", __FUNCTION__,
                      slot_id, handle));
      }

      /* Reply to the receipt of the licence */
      STB_CISendLicenseReceivedStatus(slot_id, status);
   }

   FUNCTION_FINISH(PlaybackLicenceChanged);
}

/**
 * @brief   Handles a CICAM pin update when recording, which results in the
 *          pin info being saved with the recording for use during playback.
 * @param   path decode path
 * @param   slot_id CI slot
 */
static void RecordPin(U8BIT path, U8BIT slot_id)
{
   U32BIT handle;
   U8BIT cicam_id[CIP_CICAM_ID_LEN];
   U8BIT status;
   U8BIT age_rating;
   U8BIT *pin_private_data;
   U16BIT date_code;
   U8BIT hour, min, secs;

   FUNCTION_START(RecordPin);

   if (STB_DPIsRecording(path, &handle))
   {
      AP_CI_PRINT(("%s(%u): Updating recording 0x%08lx", __FUNCTION__, path, handle));

      if (STB_CIGetCicamId(slot_id, cicam_id))
      {
         /* Store the CICAM id with the recording */
         STB_PVRRecordingSetCicamId(handle, cicam_id, sizeof(cicam_id));

         /* Get the pin info to be stored with the recording */
         if (STB_CiCcGetRecordingPinInfo(slot_id, &status, &age_rating, &pin_private_data,
                &date_code, &hour, &min, &secs))
         {
            /* If the CAM is unable to descramble the content then the pin event doesn't
             * need to be saved with the recording (11.3.2.3 of CI+ v1.3.1 spec) */
            if (status != STB_CI_PIN_CONTENT_SCRAMBLED)
            {
               STB_PVRRecordingAddPin(handle, age_rating, pin_private_data,
                  date_code, hour, min, secs);
            }
         }
      }
   }
   FUNCTION_FINISH(RecordPin);
}

/**
* @brief The event notification callback function.
* @param event_code Event ID, as defined in stberc.h
* @param event_data Optional, event-specific data
* @param event_size size of the event_data
*/
static void CIPlusEventHandler(U32BIT event_code, void *event_data, U32BIT event_size)
{
   BOOLEAN retval;
   S_EVENT_INFO event_info;

   FUNCTION_START(CIPlusEventHandler);

   retval = TRUE;

   if ((event_data != NULL) && (event_size > 0))
   {
      /* Allocate memory for the event data to be passed on */
      if ((event_info.data = STB_AppGetMemory(event_size)) != NULL)
      {
         memcpy(event_info.data, event_data, event_size);
         event_info.data_size = event_size;
      }
      else
      {
         retval = FALSE;
      }
   }
   else
   {
      event_info.data = NULL;
      event_info.data_size = 0;
   }

   if (retval)
   {
      event_info.event_code = event_code;

      /* Send the event to the queue */
      STB_OSWriteQueue(ciplus_event_queue, (void *)&event_info, sizeof(S_EVENT_INFO), TIMEOUT_NEVER);
   }

   FUNCTION_FINISH(CIPlusEventHandler);
}

static void CIPlusEventTask(void *param)
{
   S_CIPLUS_TUNE_DATA *tune_data;
   S_CIP_RELEASE_REPLY *reply_data;
   S_ACTL_OWNER_INFO owner_info;
   void *serv_ptr;
   void *owner_data;
   U32BIT data_size;
   U8BIT slot_id, path;
   U8BIT *ciplus_pmt = NULL;
   S_EVENT_INFO event_info;
   U16BIT onet_id, trans_id, serv_id;

   FUNCTION_START(CIPlusEventTask);
   USE_UNWANTED_PARAM(param);

   while (1)
   {
      if (STB_OSReadQueue(ciplus_event_queue, (void *)&event_info, sizeof(S_EVENT_INFO), TIMEOUT_NEVER))
      {
         switch (event_info.event_code)
         {
            case STB_EVENT_CI_SLOT_STATUS_CHANGED:
            {
               CISlotStatusChanged(*(U8BIT *)event_info.data);
               break;
            }

            case STB_EVENT_CI_REMOVE:
            {
               CISlotRemoved(*(U8BIT *)event_info.data);
               break;
            }

            case STB_EVENT_CI_URI_CHANGED:
            {
               slot_id = *(U8BIT *)event_info.data;

               /* A slot can be shared with more than one path,
                * so handle this event for all paths using this slot */
               path = INVALID_RES_ID;
               while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
               {
                  ACI_UsageRulesStatusChanged(path);
               }
               break;
            }

            case STB_EVENT_CI_KEYS_CHANGED:
            {
               slot_id = *(U8BIT *)event_info.data;

               ApplyCCKeys(slot_id);
               break;
            }

            case STB_EVENT_CI_RECORD_LICENCE_CHANGED:
            {
               slot_id = *(U8BIT *)event_info.data;

               /* A slot can be shared with more than one path,
                * so handle this event for all paths using this slot */
               path = INVALID_RES_ID;
               while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
               {
                  RecordLicenceChanged(path, slot_id);
               }
               break;
            }

            case STB_EVENT_CI_RECORD_PIN:
            {
               slot_id = *(U8BIT *)event_info.data;

               /* A slot can be shared with more than one path,
                * so handle this event for all paths using this slot */
               path = INVALID_RES_ID;
               while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
               {
                  RecordPin(path, slot_id);
               }
               break;
            }

            case STB_EVENT_CI_PLAYBACK_LICENCE_CHANGED:
            {
               PlaybackLicenceChanged(*(U8BIT *)event_info.data);
               break;
            }

            case STB_EVENT_CI_PLAYBACK_BLANK_VIDEO:
            {
               /* Have to blank the video until the pin reply is received from the CAM */
               STB_AVBlankVideo(0, TRUE);
               break;
            }

            case STB_EVENT_CI_PLAYBACK_PIN_STATUS:
            {
               switch (*(E_STB_CI_PIN_STATUS *)event_info.data)
               {
                  case STB_CI_PIN_CODE_CORRECT:
                  case STB_CI_PIN_BLANK_NOT_REQUIRED:
                  {
                     /* Playback can continue */
                     STB_AVBlankVideo(0, FALSE);
                     break;
                  }

                  default:
                  {
                     /* Continue to blank the AV for any other pin status */
                     break;
                  }
               }
               break;
            }

            case STB_EVENT_CI_TUNE_REQUEST:
            {
               tune_data = (S_CIPLUS_TUNE_DATA *)event_info.data;
               if (tune_data != NULL)
               {
                  owner_info.owner = RES_OWNER_CIPLUS;
                  owner_info.data = &tune_data->module;
                  owner_info.data_size = sizeof(tune_data->module);

                  switch (tune_data->type)
                  {
                     case CIPLUS_TUNE_TYPE_TRANSPORT:
                     {
                        onet_id = ADB_GetTransportOriginalNetworkId(tune_data->u.transport.t_ptr);
                        trans_id = ADB_GetTransportTid(tune_data->u.transport.t_ptr);
                        if (NotifyStartTuning != NULL)
                        {
                           NotifyStartTuning(tune_data->module, onet_id, trans_id, 0, tune_data->flags);
                        }

                        AP_CI_PRINT(("%s: CIPLUS_TUNE_TYPE_TRANSPORT", __FUNCTION__));
                        /* Tune the live path to the transport requested by CI+ */
                        ACTL_TuneToTransport(INVALID_RES_ID, &owner_info, tune_data->u.transport.t_ptr,
                           APP_SI_MODE_NO_SI, TRUE);
                        break;
                     }

                     case CIPLUS_TUNE_TYPE_SERVICE:
                     {
                        ADB_GetServiceIds(tune_data->u.service.s_ptr, &onet_id, &trans_id, &serv_id);
                        if (NotifyStartTuning != NULL)
                        {
                           NotifyStartTuning(tune_data->module, onet_id, trans_id, serv_id, tune_data->flags);
                        }

                        AP_CI_PRINT(("%s: CIPLUS_TUNE_TYPE_SERVICE", __FUNCTION__));
                        /* Tune the live path to the service requested by CI+ */
                        ACTL_TuneToService(INVALID_RES_ID, &owner_info, tune_data->u.service.s_ptr, FALSE, TRUE);
                        break;
                     }

                     case CIPLUS_TUNE_TYPE_CI_SERVICE:
                     {
                        ADB_GetServiceIds(tune_data->u.service.s_ptr, &onet_id, &trans_id, &serv_id);
                        if (NotifyStartTuning != NULL)
                        {
                           NotifyStartTuning(tune_data->module, onet_id, trans_id, serv_id, tune_data->flags);
                        }

                        AP_CI_PRINT(("%s: CIPLUS_TUNE_TYPE_CI_SERVICE", __FUNCTION__));
                        if (ciplus_pmt != NULL)
                        {
                           STB_AppFreeMemory(ciplus_pmt);
                           ciplus_pmt = NULL;
                        }

                        if (tune_data->u.service.pmt != NULL)
                        {
                           /* Space for the PMT data was allocated when the event was sent
                            * and as it's stored as a pointer in the structure received with the
                            * event, the pointer just needs to be copied to ensure the data
                            * isn't lost. The PMT will be freed when the tune completes or when
                            * the next tune request is received (i.e. above). */
                           ciplus_pmt = tune_data->u.service.pmt;
                        }
                        /* Tune the live path to the service requested by CI+ */
                        ACTL_TuneToCIService(INVALID_RES_ID, &owner_info, tune_data->u.service.s_ptr,
                           tune_data->u.service.reqd_si);
                        break;
                     }
                  }
               }
               break;
            }

            case APP_EVENT_CIPLUS_TUNE_COMPLETED:
            {
               path = *(U8BIT *)event_info.data;
               if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
               {
                  ACI_TuneReply(path, *(U32BIT *)owner_data, CIP_TUNER_LOCKED);
               }

               if (ciplus_pmt != NULL)
               {
                  if ((serv_ptr = ADB_GetTunedService(path)) != NULL)
                  {
                     /* The PMT was provided by CI+, so it's now time to use it */
                     ASI_ProcessPmt(path, serv_ptr, ciplus_pmt);
                  }

                  /* PMT no longer required, so release it */
                  STB_AppFreeMemory(ciplus_pmt);
                  ciplus_pmt = NULL;
               }

               /* Inform the UI that the service has changed */
               if ((serv_ptr = ADB_GetTunedService(path)) != NULL)
               {
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED,
                     &serv_ptr, sizeof(void *));
               }

               break;
            }

            case STB_EVENT_CI_RELEASE_REPLY:
            {
               reply_data = (S_CIP_RELEASE_REPLY *)event_info.data;
               if (reply_data != NULL)
               {
                  AP_CI_PRINT(("%s: STB_EVENT_CI_RELEASE_REPLY %u", __FUNCTION__, reply_data->reply));
                  if (reply_data->reply == 0)
                  {
                     ACTL_ContinueCIPLUSTune(reply_data->path, TRUE);
                  }
                  else
                  {
                     ACTL_ContinueCIPLUSTune(reply_data->path, FALSE);
                  }
               }
               break;
            }

            case STB_EVENT_CI_OPERATOR_SEARCH_FINISHED:
            {
               OperatorSearchFinished();
               break;
            }

            case STB_EVENT_CI_RECORD_START:
            {
               APVR_CIRecordReplyReceived(TRUE, *(U8BIT *)event_info.data);
               break;
            }

            case STB_EVENT_CI_RECORD_START_FAILED:
            {
               APVR_CIRecordReplyReceived(FALSE, *(U8BIT *)event_info.data);
               break;
            }
            default:
            {
               break;
            }
         }

         /* The memory associated with the event can now be freed */
         if (event_info.data != NULL)
         {
            STB_AppFreeMemory(event_info.data);
         }
      }
   }

   FUNCTION_FINISH(CIPlusEventTask);
}

//----------------------------------------------------------------------------
//---Global function prototypes for this file---------------------------------
//----------------------------------------------------------------------------

void ACI_Initialise(void)
{
   ciplus_event_queue = STB_OSCreateQueue(sizeof(S_EVENT_INFO), MAX_EVENTS);
   STB_OSCreateTask(CIPlusEventTask, NULL, TASK_STACK_SIZE, TASK_PRIORITY, (U8BIT *)"CI+EventTask");
   STB_CiCcInitialise();
   ACI_HcInitialise();
   ACI_OpInitialise();
   APP_RegisterDVBEventHandler(CIPlusEventHandler);
   STB_SISetCiplusPrivateDataSpecifierMode(TRUE);
}

/**
 * @brief   Return CAM upgrade option (Yes/No/Ask)
 * @return  U8BIT the cams current upgrade mode
 */
U8BIT ACI_GetCamUpgradeMode(void)
{
   FUNCTION_START(ACI_GetCamUpgradeMode);
   FUNCTION_FINISH(ACI_GetCamUpgradeMode);
   return(cam_upgrade_mode);
}

/**
 * @brief   Sets the CAM upgrade option (Yes/No/Ask)
 * @param   upgrade_mode cam upgrade mode
 */
void ACI_SetCamUpgradeMode(U8BIT upgrade_mode)
{
   FUNCTION_START(ACI_SetCamUpgradeMode);
   cam_upgrade_mode = upgrade_mode;
   FUNCTION_FINISH(ACI_SetCamUpgradeMode);
}

/**
 * @brief   Return the option for starting an operator profile search (yes/no/ask)
 * @return  The current setting
 */
E_CIP_START_OPERATOR_SEARCH ACI_GetOperatorSearchMode(void)
{
   FUNCTION_START(ACI_GetOperatorSearchMode);
   FUNCTION_FINISH(ACI_GetOperatorSearchMode);
   return(operator_search_mode);
}

/**
 * @brief   Sets the operator profile search mode (Yes/No/Ask). This should be used
 *          by the app to control whether a search can be started, or the user
 *          should be asked first, or can't run.
 * @param   search_mode search mode
 */
void ACI_SetOperatorSearchMode(E_CIP_START_OPERATOR_SEARCH search_mode)
{
   FUNCTION_START(ACI_SetCamUpgradeMode);
   operator_search_mode = search_mode;
   FUNCTION_FINISH(ACI_SetOperatorSearchMode);
}

/**
 * @brief   Sets whether an operator profile search needs to be run
 * @param   module module requesting or cancelling an operator search
 * @param   required TRUE if a search is required, FALSE otherwise
 */
void ACI_OperatorSearchRequired(U32BIT module, BOOLEAN required)
{
   void *profile;

   FUNCTION_START(ACI_OperatorSearchRequired);

   if ((profile = ADB_FindProfileForModule(module)) != NULL)
   {
      ADB_SetProfileSearchRequired(profile, TRUE, 0, 0, 0);
   }
   else
   {
      /* A profile doesn't exist for this module yet (probably the initial search),
       * so store the info so the app can access it */
      operator_search_required = required;

      if (required)
      {
         operator_search_module = module;
      }
   }

   FUNCTION_FINISH(ACI_OperatorSearchRequired);
}

/**
 * @brief   Sets the date/time that an operator search should be started for the given module
 * @param   cicam_id ID of the CICAM requesting an operator search
 * @param   date UTC MJD date code
 * @param   hour UTC hour
 * @param   min UTC minute
 */
void ACI_ScheduleOperatorSearch(U32BIT cicam_id, U16BIT date, U8BIT hour, U8BIT min)
{
   U32BIT timer_handle;
   S_TIMER_INFO info;

   FUNCTION_START(ACI_ScheduleOperatorSearch);

   /* Create a timer to trigger when the search is due */
   info.start_time = DHMS_CREATE(date, hour, min, 0);
   info.type = TIMER_TYPE_PRIVATE;
   info.frequency = TIMER_FREQ_ONCE;
   timer_handle = ATMR_AddTimer(&info);
   if (timer_handle != INVALID_TIMER_HANDLE)
   {
      /* Save the timer / CICAM association */
      ADB_SetProfileSearchTimer(cicam_id, timer_handle);
      ADB_SaveDatabase();
   }

   FUNCTION_FINISH(ACI_ScheduleOperatorSearch);
}

/**
 * @brief   Returns whether an operator profile search has been requested
 * @return  TRUE if a search has been requested, FALSE otherwise
 */
BOOLEAN ACI_IsOperatorSearchRequired(void)
{
   FUNCTION_START(ACI_IsOperatorSearchRequired);
   FUNCTION_FINISH(ACI_IsOperatorSearchRequired);
   return(operator_search_required);
}

/**
 * @brief   Returns the module performing or requiring and operator search
 * @return  module
 */
U32BIT ACI_GetOperatorSearchModule(void)
{
   FUNCTION_START(ACI_GetOperatorSearchModule);
   FUNCTION_FINISH(ACI_GetOperatorSearchModule);
   return(operator_search_module);
}

/**
 * @brief   Checks all the CI+ profiles to see if any have requested an update search
 * @param   module pointer to return the handle of the first module requiring an update
 *          Only valid if the function returns TRUE
 * @return  TRUE if a profile needs updating, FALSE otherwise
 */
BOOLEAN ACI_GetFirstOperatorSearchModule(U32BIT *module)
{
   BOOLEAN retval;
   void **profile_list;
   U16BIT num_profiles;
   U16BIT active_profile;
   U16BIT i;

   FUNCTION_START(ACI_GetFirstOperatorSearchModule);

   retval = FALSE;

   if ((num_profiles = ADB_GetProfileList(&profile_list, &active_profile)) > 0)
   {
      for (i = 0; (i < num_profiles) && !retval; i++)
      {
         if ((ADB_GetProfileType(profile_list[i]) == ADB_PROFILE_TYPE_CIPLUS) &&
             ADB_GetProfileModulePresent(profile_list[i]) &&
             ADB_GetProfileSearchRequired(profile_list[i]))
         {
            /* Have found a module that needs to be updated */
            *module = ADB_GetProfileModule(profile_list[i]);
            retval = TRUE;
         }
      }

      ADB_ReleaseProfileList(profile_list, num_profiles);
   }

   FUNCTION_FINISH(ACI_GetFirstOperatorSearchModule);

   return(retval);
}

/**
 * @brief   Checks all the CI+ profiles to find the one with the earliest scheduled search.
 *          All returned values are only valid if the function returns TRUE.
 * @param   module pointer to return the handle of the first module requiring an update
 * @param   date pointer to return the date code of the search
 * @param   hours pointer to return the hour of the search
 * @param   mins pointer to return the minute of the search
 * @return  TRUE if a scheduled search is found for a profile, FALSE otherwise
 */
BOOLEAN ACI_GetFirstScheduledOperatorSearch(U32BIT *module, U16BIT *date, U8BIT *hours, U8BIT *mins)
{
   BOOLEAN retval;
   void **profile_list;
   U16BIT num_profiles;
   U16BIT active_profile;
   U16BIT i, profile_index;
   U16BIT min_date, profile_date;
   U8BIT min_hours, profile_hours;
   U8BIT min_mins, profile_mins;

   FUNCTION_START(ACI_GetFirstScheduledOperatorSearch);

   retval = FALSE;

   if ((num_profiles = ADB_GetProfileList(&profile_list, &active_profile)) > 0)
   {
      min_date = 0xffff;
      min_hours = 0xff;
      min_mins = 0xff;
      profile_index = num_profiles + 1;

      for (i = 0; i < num_profiles; i++)
      {
         if ((ADB_GetProfileType(profile_list[i]) == ADB_PROFILE_TYPE_CIPLUS) &&
             ADB_GetProfileModulePresent(profile_list[i]) &&
             ADB_GetProfileSearchDateTime(profile_list[i], &profile_date, &profile_hours, &profile_mins))
         {
            /* Have found a module that needs to be updated.
             * Check whether this is earlier than any previous module */
            if ((profile_date <= min_date) && (profile_hours <= min_hours) &&
                (profile_mins < min_mins))
            {
               min_date = profile_date;
               min_hours = profile_hours;
               min_mins = profile_mins;
               profile_index = i;
            }
         }
      }

      if (profile_index < num_profiles)
      {
         *module = ADB_GetProfileModule(profile_list[profile_index]);
         *date = min_date;
         *hours = min_hours;
         *mins = min_mins;
         retval = TRUE;
      }

      ADB_ReleaseProfileList(profile_list, num_profiles);
   }

   FUNCTION_FINISH(ACI_GetFirstScheduledOperatorSearch);

   return(retval);
}

/**
 * @brief   Called by the app to start an operator profile search for the given module
 * @param   module operator module
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACI_StartOperatorSearchForModule(U32BIT module)
{
   BOOLEAN retval;
   void *profile;
   U8BIT path;

   FUNCTION_START(ACI_StartOperatorSearchForModule);

   retval = FALSE;

   if ((profile = ADB_FindProfileForModule(module)) != NULL)
   {
      if (ADB_GetProfileSearchRequired(profile))
      {
         /* Search is being started so clear the required flag */
         ADB_SetProfileSearchRequired(profile, FALSE, 0, 0, 0);

         retval = ACI_StartOperatorSearch(module);
      }
   }
   else if (operator_search_required)
   {
      /* Search is being started so clear the required flag */
      ACI_OperatorSearchRequired(module, FALSE);

      retval = ACI_StartOperatorSearch(module);
   }

   if (retval)
   {
      /* Save the tuned service so it can be returned to when the operator search has completed */
      if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
      {
         operator_search_service = STB_DPGetTunedService(path);
      }
   }

   FUNCTION_FINISH(ACI_StartOperatorSearchForModule);

   return(retval);
}

/**
 * @brief   Checks whether the given timer is associated with any of the CI+ profiles
 *          and starts the operator search if possible
 * @param   timer_handle timer handle
 * @return  TRUE if the timer is associated with a profile, FALSE otherwise
 */
BOOLEAN ACI_HandlePrivateTimer(U32BIT timer_handle)
{
   BOOLEAN valid, retval;
   U32BIT module;
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(ACI_HandlePrivateTimer);

   AP_CI_PRINT(("ACI_HandlePrivateTimer(0x%x)", timer_handle));

   valid = FALSE;
   retval = FALSE;

   DBDEF_RequestAccess();
   t = DBDEF_FindCicamTimerByHandle(timer_handle);
   if (t != NULL)
   {
      retval = TRUE;
      if (ACI_FindOperatorProfileModule(t->cicam_identifier, &module))
      {
         /* Found the CICAM that requested the scheduled profile update, the search can only be
            performed if the module is present */
         valid = TRUE;
      }
   }
   DBDEF_ReleaseAccess();

   if (retval)
   {
      AP_CI_PRINT(("Timer is associated with CICAM 0x%x", t->cicam_identifier));
      /* Delete the timer and clear it from the profile */
      ATMR_DeleteTimer(timer_handle);
      ADB_DeleteCicamTimerbyHandle(timer_handle);
   }

   if (valid)
   {
      AP_CI_PRINT(("CI+ Operator profile module is 0x%x, search mode is %s", module,
         (operator_search_mode == CIP_START_OPERATOR_SEARCH_NO) ? "NO" :
         (operator_search_mode == CIP_START_OPERATOR_SEARCH_YES) ? "YES" :
         (operator_search_mode == CIP_START_OPERATOR_SEARCH_ASK) ? "ASK" : "Unexpected"));
      switch (operator_search_mode)
      {
         case CIP_START_OPERATOR_SEARCH_NO:
         {
            /* Search can't be started */
            break;
         }

         case CIP_START_OPERATOR_SEARCH_YES:
         {
            /* Search can be started, set flag to indicate a search is required */
            ACI_OperatorSearchRequired(module, TRUE);
            ACI_StartOperatorSearchForModule(module);
            break;
         }

         case CIP_START_OPERATOR_SEARCH_ASK:
         {
            /* Set flag to indicate a search is required */
            ACI_OperatorSearchRequired(module, TRUE);

            if (ASTE_InStandby())
            {
               AP_CI_PRINT(("ACI_HandlePrivateTimer: in stand-by"));
               /* In standby so search can be started */
               ACI_StartOperatorSearchForModule(module);
            }
            else
            {
               AP_CI_PRINT(("ACI_HandlePrivateTimer: ask the user"));
               /* Need to ask if it can be started now */
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_REQUEST_OPERATOR_SEARCH,
                  &module, sizeof(module));
            }
         }
      }
   }

   FUNCTION_FINISH(ACI_HandlePrivateTimer);

   return (retval);
}

/**
 * @brief   Ensures the TS is routed securely for CI+, either by setting the TS
 *          to pass through if a CI slot contains a CI+ CAM or bypass it if it doesn't.
 */
BOOLEAN ACI_SetSecureRouting(U8BIT path)
{
   BOOLEAN retval;
   U8BIT slot_id, num_slots;
   BOOLEAN pass_through;

   FUNCTION_START(ACI_SetSecureRouting);

   retval = TRUE;

   /* Ensure the TS for the path isn't being routed through a CI slot unless it contains a CI+ CAM */
   if ((num_slots = STB_CIGetSlotCount()) > 0)
   {
      for (slot_id = 0; (slot_id < num_slots) && retval; slot_id++)
      {
         pass_through = TRUE;

         /* IDs for the CI stack start at 1...this needs sorting out */
         if (STB_CiCcAuthenticated(slot_id))
         {
            /* Check whether the slot is already being used by another path */
            if (STB_DPIsCISlotInUse(INVALID_RES_ID, slot_id, path) != INVALID_RES_ID)
            {
               pass_through = FALSE;
            }
         }
         else
         {
            /* Nothing in the slot, or not a CI+ module, so bypass it */
            pass_through = FALSE;
         }

         if (!pass_through)
         {
            /* Bypass the slot to ensure it isn't modified */
            AP_CI_PRINT(("ACI_SetSecureRouting(%u): releasing slot %u from path", path, slot_id));
            ReleaseCISlotFromPath(path);
         }
      }
   }

   FUNCTION_FINISH(ACI_SetSecureRouting);

   return(retval);
}

/**
 * @brief   Acquires a CI slot for the given path on the given service after
 *          releasing any slot already being used by the path
 * @param   path decode path
 * @param   s_ptr service to be used on the path
 * @return  TRUE if a slot is acquired, FALSE otherwise
 */
BOOLEAN ACI_AcquireCISlot(U8BIT path, void *s_ptr)
{
   U8BIT slot_id;
   U8BIT *pmt_data;
   U16BIT service_id;
   U16BIT data_len;
   void *ci_prot;
   BOOLEAN retval;

   FUNCTION_START(ACI_AcquireCISlot);

   retval = FALSE;

   if ((STB_DPGetPathCISlot(path) != INVALID_RES_ID) && (STB_DPGetTunedService(path) != s_ptr))
   {
      AP_CI_PRINT(("ACI_AcquireCISlot(%u, %p): Releasing CAM %u", path, s_ptr,
                   STB_DPGetPathCISlot(path)));

      /* Release the CAM currently associated with the path as the
       * requirements for the CAM may be different for the new service */
      ReleaseCISlotFromPath(path);
   }

   if (s_ptr != NULL)
   {
      pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
      ci_prot = ADB_GetServiceCIProtectionDesc(s_ptr);
      service_id = ADB_GetServiceId(s_ptr);
   }
   else
   {
      pmt_data = NULL;
      ci_prot = NULL;
      service_id = 0xffff;
   }

   slot_id = STB_DPAcquireCISlotForPath(path, pmt_data, ci_prot);
   if (slot_id != INVALID_RES_ID)
   {
      /* Set the recording mode based on whether the slot is being used on the live path */
      if (STB_DPIsLivePath(path) || STB_DPIsLivePath(STB_DPIsCISlotInUse(0, slot_id, path)))
      {
         AP_CI_PRINT(("%s(%u): slot %u, record mode is WATCH for service %u",
                        __FUNCTION__, path, slot_id, service_id));
         STB_CiCcSetRecordOperatingMode(slot_id, STB_CI_MODE_WATCH_AND_BUFFER, service_id);
      }
      else
      {
         AP_CI_PRINT(("%s(%u): slot %u, record mode UNATTENDED for service %u",
                        __FUNCTION__, path, slot_id, service_id));
         STB_CiCcSetRecordOperatingMode(slot_id, STB_CI_MODE_UNATTENDED_RECORDING, service_id);
      }
      /* Apply any CI+ keys */
      STB_CiKeysApply(path, slot_id);
   }

   AP_CI_PRINT(("ACI_AcquireCISlot(%u, %p): Acquired slot %u, pmt=%p, ci_prot=%p",
                path, s_ptr, slot_id, pmt_data, ci_prot));

   if (slot_id != INVALID_RES_ID)
   {
      retval = TRUE;

      if (path == STB_DPGetLivePath())
      {
         /* Apply URI on the newly acquired path (if live) */
         ApplyUsageRules(path, s_ptr);
      }
   }

   FUNCTION_FINISH(ACI_AcquireCISlot);

   return(retval);
}

/**
 * @brief   Acquires a CI slot for the recording path. This function may need to "steal" the CI
 *          slot from the live path (if they are not the same path).
 * @param   path decode path for recording
 * @param   s_ptr service to be used on the path
 * @return  TRUE if a slot is acquired, FALSE otherwise
 */
BOOLEAN ACI_AcquireCISlotForRecording(U8BIT path, void* s_ptr)
{
   U8BIT slot_id;
   U8BIT *pmt_data;
   U16BIT data_len;
   U8BIT live_path;
   BOOLEAN slot_acquired;

   FUNCTION_START(ACI_AcquireCISlotForRecording);

   slot_acquired = ACI_AcquireCISlot(path, s_ptr);
   if (!slot_acquired)
   {
      /* See whether the slot needs to be taken away from the live path */
      if ((live_path = STB_DPGetLivePath()) != INVALID_RES_ID)
      {
         /* CI slot may need to be taken, but only if the service is
          * scrambled and the CAM supports the CAS (and the live is tuned
          * to a different service) */
         slot_id = STB_DPGetPathCISlot(live_path);
         if (slot_id != INVALID_RES_ID)
         {
            AP_CI_PRINT(("[ACI] Slot is used by live path"));
            pmt_data = ADB_GetServicePMTData(s_ptr, &data_len);
            if (STB_CiCcIsServiceAllowed(slot_id, NULL) &&
                STB_CiCaSystemSupported(slot_id, pmt_data))
            {
               /* Unfortunately the recording requires the CI slot */
               AP_CI_PRINT(("[ACI] slot is needed for recording, so take away from live path"));

               /* Clear keys (not required as the CAM is taken) */
               STB_CiKeysClear(live_path, slot_id);

               /* CI slot will be used by the recording path */
               ReleaseCISlotFromPath(live_path);

               /* Now try to acquire a slot again */
               slot_acquired = ACI_AcquireCISlot(path, s_ptr);

               /* TODO: Would be nice to let the user know what just happened */
            }
         }
         else
         {
            /* Slot must be used for another recording */
         }
      }
      else
      {
         /* Slot must be used for another recording */
      }
   }

   FUNCTION_FINISH(ACI_AcquireCISlotForRecording);

   return(slot_acquired);
}

/**
 * @brief   Looks for a CAM that's able to descramble the given service and returns its slot id
 * @param   serv_ptr service
 * @return  slot id, if CAM is found, or INVALID_RES_ID if not
 */
U8BIT ACI_FindCISlotForService(void *serv_ptr)
{
   U8BIT slot_id;
   U8BIT id, num_slots;
   void *pmt_data;
   U16BIT data_len;

   FUNCTION_START(ACI_FindCISlotForService);
   slot_id = INVALID_RES_ID;
   if ((num_slots = STB_CIGetSlotCount()) > 0)
   {
      if (serv_ptr != NULL)
      {
         pmt_data = ADB_GetServicePMTData(serv_ptr, &data_len);
         for (id = 0; id != num_slots; id++)
         {
            if (STB_CiCcAuthenticated(id) &&
                STB_CiCaSystemSupported(id, pmt_data))
            {
               /* Found a CI+ CAM that supports this service */
               slot_id = id;
               break;
            }
         }
      }
   }
   FUNCTION_FINISH(ACI_FindCISlotForService);
   return(slot_id);
}

/**
 * @brief   Re-evaluate current state follwing a change related to usage rules
 * @param   path decode path
 */
void ACI_UsageRulesStatusChanged(U8BIT path)
{
   void *s_ptr;

   FUNCTION_START(ACI_UsageRulesStatusChanged);
   if (STB_CIGetSlotCount() > 0)
   {
      /* URIs are associated with specific services, so if there is
       * no service, the URI is meaningless */
      s_ptr = ADB_GetTunedService(path);
      if (s_ptr != NULL)
      {
         /* The URI is applied to the CAM being used by the decoding path */
         ApplyUsageRules(path, s_ptr);

         /* Any recording on the path should be updated */
         UpdateRecordingURI(path, s_ptr);
      }
   }
   FUNCTION_FINISH(ACI_UsageRulesStatusChanged);
}

/**
 * @brief  Handle PMT change
 * @param  pmt the new PMT
 */
void ACI_ProgramMapTableChanged(U8BIT *pmt)
{
   U8BIT path;
   U8BIT num_paths;
   void *s_ptr;
   U8BIT slot_id;
   U16BIT service_id;

   FUNCTION_START(ACI_ProgramMapTableChanged);

   //AP_CI_PRINT(("[ACI] ACI_ProgramMapTableChanged"));

   if (STB_CIGetSlotCount() > 0)
   {
      num_paths = STB_DPGetNumPaths();
      for (path = 0; path < num_paths; path++)
      {
         slot_id = STB_DPGetPathCISlot(path);
         if (slot_id == INVALID_RES_ID || !STB_CiCcIsSlotReady(slot_id))
         {
            /* The path does not use the CI slot, or slot not ready, PMT is irrelevant */
            continue;
         }

         s_ptr = ADB_GetTunedService(path);
         if (s_ptr == NULL)
         {
            /* The PMT is irrelevant to the path */
            continue;
         }

         service_id = pmt[3] << 8 | pmt[4];
         if (ADB_GetServiceId(s_ptr) == service_id)
         {
            /* Update the CAM */
            //AP_CI_PRINT(("[ACI] Calling CIP_ReportPmt(%d, %p)", slot_id, pmt));
            STB_CiCaReportPmt(slot_id, pmt);
         }
      }
   }

   FUNCTION_FINISH(ACI_ProgramMapTableChanged);
}

/**
 * @brief   The given decode path is only trusted if it doesn't include a CI slot
 *          or the CI slot contains a CI+ CAM.
 * @param   path decode path
 * @return  TRUE if the decode path is trusted, FALSE otherwise
 */
BOOLEAN ACI_IsTrustedPath(U8BIT path)
{
   BOOLEAN trusted;
   U8BIT slot_id;

   FUNCTION_START(ACI_IsTrustedPath);

   trusted = FALSE;

   slot_id = STB_DPGetPathCISlot(path);

   if ((slot_id == INVALID_RES_ID) || STB_CiCcAuthenticated(slot_id))
   {
      trusted = TRUE;
   }

   FUNCTION_FINISH(ACI_IsTrustedPath);

   return(trusted);
}

/**
 * @brief   Checks if a pin has been saved for the CAM in the given slot and returns it
 * @param   slot_id CI slot
 * @return  pin value, or -1 if there's no CAM in the slot or no pin has been saved
 */
S32BIT ACI_ReadPinForSlot(U8BIT slot_id)
{
   S32BIT saved_pin;
   U8BIT cicam_id[CIP_CICAM_ID_LEN];
   U8BIT id_nvm[CIP_CICAM_ID_LEN];
   U32BIT nvm_value;
   U32DHMS used_time;

   FUNCTION_START(ACI_ReadPinForSlot);

   saved_pin = -1;

   /* Get the cicam id for the given slot */
   if (STB_CIGetCicamId(slot_id, cicam_id))
   {
      /* Check this against each saved cicam id to see if the pin is available */
      nvm_value = APP_NvmRead(CICAM_ID0_0_NVM);
      memcpy(&id_nvm[0], &nvm_value, 4);
      nvm_value = APP_NvmRead(CICAM_ID0_1_NVM);
      memcpy(&id_nvm[4], &nvm_value, 4);

      if (memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0)
      {
         /* This is the cicam id for the CAM in the given slot, so return this pin */
         saved_pin = APP_NvmRead(CICAM_PIN0_NVM);

         /* Update the date/time this pin was last used */
         used_time = STB_GCNowDHMSGmt();
         APP_NvmSave(CICAM_PIN0_USED_TIME_NVM, used_time, TRUE);
      }
      else
      {
         nvm_value = APP_NvmRead(CICAM_ID1_0_NVM);
         memcpy(&id_nvm[0], &nvm_value, 4);
         nvm_value = APP_NvmRead(CICAM_ID1_1_NVM);
         memcpy(&id_nvm[4], &nvm_value, 4);

         if (memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0)
         {
            /* This is the cicam id for the CAM in the given slot, so return this pin */
            saved_pin = APP_NvmRead(CICAM_PIN1_NVM);

            /* Update the date/time this pin was last used */
            used_time = STB_GCNowDHMSGmt();
            APP_NvmSave(CICAM_PIN1_USED_TIME_NVM, used_time, TRUE);
         }
         else
         {
            nvm_value = APP_NvmRead(CICAM_ID2_0_NVM);
            memcpy(&id_nvm[0], &nvm_value, 4);
            nvm_value = APP_NvmRead(CICAM_ID2_1_NVM);
            memcpy(&id_nvm[4], &nvm_value, 4);

            if (memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0)
            {
               /* This is the cicam id for the CAM in the given slot, so return this pin */
               saved_pin = APP_NvmRead(CICAM_PIN2_NVM);

               /* Update the date/time this pin was last used */
               used_time = STB_GCNowDHMSGmt();
               APP_NvmSave(CICAM_PIN2_USED_TIME_NVM, used_time, TRUE);
            }
            else
            {
               nvm_value = APP_NvmRead(CICAM_ID3_0_NVM);
               memcpy(&id_nvm[0], &nvm_value, 4);
               nvm_value = APP_NvmRead(CICAM_ID3_1_NVM);
               memcpy(&id_nvm[4], &nvm_value, 4);

               if (memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0)
               {
                  /* This is the cicam id for the CAM in the given slot, so return this pin */
                  saved_pin = APP_NvmRead(CICAM_PIN3_NVM);

                  /* Update the date/time this pin was last used */
                  used_time = STB_GCNowDHMSGmt();
                  APP_NvmSave(CICAM_PIN3_USED_TIME_NVM, used_time, TRUE);
               }
            }
         }
      }
   }

   FUNCTION_FINISH(ACI_ReadPinForSlot);

   return(saved_pin);
}

/**
 * @brief   Saves the given pin associated with the CAM in the given slot.
 *          If the pin for this CAM has previously been saved then the saved value will be updated.
 *          If all entries are used then the oldest will be overwritten
 * @param   slot_id CI slot
 * @param   pin pin value to be saved
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN ACI_WritePinForSlot(U8BIT slot_id, S32BIT pin)
{
   BOOLEAN retval;
   U8BIT cicam_id[CIP_CICAM_ID_LEN];
   U8BIT id_nvm[CIP_CICAM_ID_LEN];
   U32BIT nvm_value;
   U32DHMS used_time;
   U32DHMS used0, used1, used2, used3;

   FUNCTION_START(ACI_WritePinForSlot);

   retval = FALSE;

   /* Get the cicam id for the given slot */
   if (STB_CIGetCicamId(slot_id, cicam_id))
   {
      /* There's a CAM in the given slot, so the pin will be saved */
      retval = TRUE;

      /* Check whether there's already an entry for this cicam id.
       * If the cicam id isn't found but an empty entry is then the pin for this cicam
       * hasn't previously been saved and the entry can be used */
      nvm_value = APP_NvmRead(CICAM_ID0_0_NVM);
      memcpy(&id_nvm[0], &nvm_value, 4);
      nvm_value = APP_NvmRead(CICAM_ID0_1_NVM);
      memcpy(&id_nvm[4], &nvm_value, 4);

      if ((memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0) ||
          ((S32BIT)APP_NvmRead(CICAM_PIN0_NVM) < 0))
      {
         /* This entry can be used to save the pin */
         nvm_value = *(U32BIT *)&cicam_id[0];
         APP_NvmSave(CICAM_ID0_0_NVM, nvm_value, FALSE);
         nvm_value = *(U32BIT *)&cicam_id[4];
         APP_NvmSave(CICAM_ID0_1_NVM, nvm_value, FALSE);

         APP_NvmSave(CICAM_PIN0_NVM, pin, FALSE);

         /* Update the date/time this pin was last used */
         used_time = STB_GCNowDHMSGmt();
         APP_NvmSave(CICAM_PIN0_USED_TIME_NVM, used_time, TRUE);
      }
      else
      {
         nvm_value = APP_NvmRead(CICAM_ID1_0_NVM);
         memcpy(&id_nvm[0], &nvm_value, 4);
         nvm_value = APP_NvmRead(CICAM_ID1_1_NVM);
         memcpy(&id_nvm[4], &nvm_value, 4);

         if ((memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0) ||
             ((S32BIT)APP_NvmRead(CICAM_PIN1_NVM) < 0))
         {
            /* This entry can be used to save the pin */
            nvm_value = *(U32BIT *)&cicam_id[0];
            APP_NvmSave(CICAM_ID1_0_NVM, nvm_value, FALSE);
            nvm_value = *(U32BIT *)&cicam_id[4];
            APP_NvmSave(CICAM_ID1_1_NVM, nvm_value, FALSE);

            APP_NvmSave(CICAM_PIN1_NVM, pin, TRUE);

            /* Update the date/time this pin was last used */
            used_time = STB_GCNowDHMSGmt();
            APP_NvmSave(CICAM_PIN1_USED_TIME_NVM, used_time, TRUE);
         }
         else
         {
            nvm_value = APP_NvmRead(CICAM_ID2_0_NVM);
            memcpy(&id_nvm[0], &nvm_value, 4);
            nvm_value = APP_NvmRead(CICAM_ID2_1_NVM);
            memcpy(&id_nvm[4], &nvm_value, 4);

            if ((memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0) ||
                ((S32BIT)APP_NvmRead(CICAM_PIN2_NVM) < 0))
            {
               /* This entry can be used to save the pin */
               nvm_value = *(U32BIT *)&cicam_id[0];
               APP_NvmSave(CICAM_ID2_0_NVM, nvm_value, FALSE);
               nvm_value = *(U32BIT *)&cicam_id[4];
               APP_NvmSave(CICAM_ID2_1_NVM, nvm_value, FALSE);

               APP_NvmSave(CICAM_PIN2_NVM, pin, TRUE);

               /* Update the date/time this pin was last used */
               used_time = STB_GCNowDHMSGmt();
               APP_NvmSave(CICAM_PIN2_USED_TIME_NVM, used_time, TRUE);
            }
            else
            {
               nvm_value = APP_NvmRead(CICAM_ID3_0_NVM);
               memcpy(&id_nvm[0], &nvm_value, 4);
               nvm_value = APP_NvmRead(CICAM_ID3_1_NVM);
               memcpy(&id_nvm[4], &nvm_value, 4);

               if ((memcmp(cicam_id, id_nvm, sizeof(cicam_id)) == 0) ||
                   ((S32BIT)APP_NvmRead(CICAM_PIN3_NVM) < 0))
               {
                  /* This entry can be used to save the pin */
                  nvm_value = *(U32BIT *)&cicam_id[0];
                  APP_NvmSave(CICAM_ID3_0_NVM, nvm_value, FALSE);
                  nvm_value = *(U32BIT *)&cicam_id[4];
                  APP_NvmSave(CICAM_ID3_1_NVM, nvm_value, FALSE);

                  APP_NvmSave(CICAM_PIN3_NVM, pin, TRUE);

                  /* Update the date/time this pin was last used */
                  used_time = STB_GCNowDHMSGmt();
                  APP_NvmSave(CICAM_PIN3_USED_TIME_NVM, used_time, TRUE);
               }
               else
               {
                  /* Pin for this CAM hasn't previously been saved and all entries are being
                   * used, so find the oldest entry and overwrite that */
                  used0 = (U32DHMS)APP_NvmRead(CICAM_PIN0_USED_TIME_NVM);
                  used1 = (U32DHMS)APP_NvmRead(CICAM_PIN1_USED_TIME_NVM);
                  used2 = (U32DHMS)APP_NvmRead(CICAM_PIN2_USED_TIME_NVM);
                  used3 = (U32DHMS)APP_NvmRead(CICAM_PIN3_USED_TIME_NVM);

                  if ((used0 < used1) && (used0 < used2) && (used0 < used3))
                  {
                     /* This entry can be used to save the pin */
                     nvm_value = *(U32BIT *)&cicam_id[0];
                     APP_NvmSave(CICAM_ID0_0_NVM, nvm_value, FALSE);
                     nvm_value = *(U32BIT *)&cicam_id[4];
                     APP_NvmSave(CICAM_ID0_1_NVM, nvm_value, FALSE);

                     APP_NvmSave(CICAM_PIN0_NVM, pin, TRUE);

                     /* Update the date/time this pin was last used */
                     used_time = STB_GCNowDHMSGmt();
                     APP_NvmSave(CICAM_PIN0_USED_TIME_NVM, used_time, TRUE);
                  }
                  else if ((used1 < used0) && (used1 < used2) && (used1 < used3))
                  {
                     /* This entry can be used to save the pin */
                     nvm_value = *(U32BIT *)&cicam_id[0];
                     APP_NvmSave(CICAM_ID1_0_NVM, nvm_value, FALSE);
                     nvm_value = *(U32BIT *)&cicam_id[4];
                     APP_NvmSave(CICAM_ID1_1_NVM, nvm_value, FALSE);

                     APP_NvmSave(CICAM_PIN1_NVM, pin, TRUE);

                     /* Update the date/time this pin was last used */
                     used_time = STB_GCNowDHMSGmt();
                     APP_NvmSave(CICAM_PIN1_USED_TIME_NVM, used_time, TRUE);
                  }
                  else if ((used2 < used0) && (used2 < used1) && (used2 < used3))
                  {
                     /* This entry can be used to save the pin */
                     nvm_value = *(U32BIT *)&cicam_id[0];
                     APP_NvmSave(CICAM_ID2_0_NVM, nvm_value, FALSE);
                     nvm_value = *(U32BIT *)&cicam_id[4];
                     APP_NvmSave(CICAM_ID2_1_NVM, nvm_value, FALSE);

                     APP_NvmSave(CICAM_PIN2_NVM, pin, TRUE);

                     /* Update the date/time this pin was last used */
                     used_time = STB_GCNowDHMSGmt();
                     APP_NvmSave(CICAM_PIN2_USED_TIME_NVM, used_time, TRUE);
                  }
                  else
                  {
                     /* This entry can be used to save the pin */
                     nvm_value = *(U32BIT *)&cicam_id[0];
                     APP_NvmSave(CICAM_ID3_0_NVM, nvm_value, FALSE);
                     nvm_value = *(U32BIT *)&cicam_id[4];
                     APP_NvmSave(CICAM_ID3_1_NVM, nvm_value, FALSE);

                     APP_NvmSave(CICAM_PIN3_NVM, pin, TRUE);

                     /* Update the date/time this pin was last used */
                     used_time = STB_GCNowDHMSGmt();
                     APP_NvmSave(CICAM_PIN3_USED_TIME_NVM, used_time, TRUE);
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(ACI_WritePinForSlot);

   return(retval);
}


/**
 * @brief   Schedule a tune to a transport
 * @param   module module id
 * @param   t_ptr target transport pointer
 */
void ACI_TuneToTransport(U32BIT module, void *t_ptr)
{
   S_CIPLUS_TUNE_DATA tune_data;

   FUNCTION_START(ACI_TuneToTransport);

   tune_data.module = module;
   tune_data.type = CIPLUS_TUNE_TYPE_TRANSPORT;
   tune_data.u.transport.t_ptr = t_ptr;
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_TUNE, &tune_data, sizeof(tune_data));

   FUNCTION_FINISH(ACI_TuneToTransport);
}

/**
 * @brief   Schedule a tune to a service
 * @param   module module id
 * @param   s_ptr target service pointer
 * @param   flags flags to pass to presentation engine
 */
void ACI_TuneToService(U32BIT module, void *s_ptr, E_CIP_TUNE_FLAGS flags)
{
   S_CIPLUS_TUNE_DATA tune_data;

   FUNCTION_START(ACI_TuneToService);

   tune_data.module = module;
   tune_data.type = CIPLUS_TUNE_TYPE_SERVICE;
   tune_data.flags = flags;
   tune_data.u.service.s_ptr = s_ptr;
   tune_data.u.service.reqd_si = APP_SI_MODE_UPDATE;
   tune_data.u.service.pmt = NULL;
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_TUNE, &tune_data, sizeof(tune_data));

   FUNCTION_FINISH(ACI_TuneToService);
}

/**
 * @brief   Tune to a service/transport defined by a delivery system descriptor,
 *          possibly with PMT data, if it's a service.
 * @param   module - module id
 * @param   tune - tuning parameters
 * @param   flags flags to pass to presentation engine
 */
void ACI_TuneToDelSysDesc(U32BIT module, S_CIP_TUNE_DEL_SYS_DESC *tune, E_CIP_TUNE_FLAGS flags)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;
   S_CIPLUS_TUNE_DATA tune_data;
   U16BIT pmt_size;

   FUNCTION_START(ACI_TuneToDelSysDesc);

   DBDEF_RequestAccess();
   switch (tune->type)
   {
      case SI_DEL_SYS_DESC_TYPE_TERR:
      {
         if (tune->desc->terr.is_t2)
         {
            t_ptr = NULL;
         }
         else
         {
            if ((t_ptr = DBDEF_FindTerrestrialTransportRec(tune->desc->terr.u.t1.freq_hz, 0)) == NULL)
            {
               AP_CI_PRINT(("%s: new frequency", __FUNCTION__));
               t_ptr = DBDEF_AddTerrestrialTransportRec(tune->desc->terr.u.t1.freq_hz, 0, NULL);
            }
            AP_CI_PRINT(("%s: Terr freq=%d t_ptr=%p", __FUNCTION__, tune->desc->terr.u.t1.freq_hz, t_ptr));
         }
         break;
      }
      case SI_DEL_SYS_DESC_TYPE_SAT:
      {
         if ((t_ptr = DBDEF_FindSatTransportRec(tune->desc->sat.freq_hz, tune->desc->sat.sym_rate,
                    tune->desc->sat.polarity, tune->desc->sat.dvb_s2, tune->desc->sat.modulation, NULL)) == NULL)
         {
            AP_CI_PRINT(("%s: new frequency", __FUNCTION__));
            t_ptr = DBDEF_AddSatTransportRec(tune->desc->sat.freq_hz, tune->desc->sat.sym_rate,
                  tune->desc->sat.polarity, tune->desc->sat.dvb_s2, tune->desc->sat.modulation, NULL);
         }
         AP_CI_PRINT(("%s: Sat freq=%d t_ptr=%p", __FUNCTION__, tune->desc->sat.freq_hz, t_ptr));
         break;
      }
      case SI_DEL_SYS_DESC_TYPE_CABLE:
      {
         if ((t_ptr = DBDEF_FindCableTransportRec(tune->desc->cable.freq_hz, tune->desc->cable.symbol_rate)) == NULL)
         {
            AP_CI_PRINT(("%s: new frequency", __FUNCTION__));
            t_ptr = DBDEF_AddCableTransportRec(tune->desc->cable.freq_hz, tune->desc->cable.symbol_rate, NULL);
         }
         AP_CI_PRINT(("%s: Cab freq=%d t_ptr=%p", __FUNCTION__, tune->desc->cable.freq_hz, t_ptr));
         break;
      }
      default:
      {
         /* Unrecognised tuner type */
         AP_CI_PRINT(("%s: Unrecognised tuner type 0x%x", __FUNCTION__, tune->type));
         ACI_TuneReply(INVALID_RES_ID, module, CIP_TUNER_UNSUPPORTED_SYSTEM);
         t_ptr = NULL;
         break;
      }
   }

   DBDEF_ReleaseAccess();

   /* The delivery descriptor can now be freed */
   STB_SIReleaseDelSysDesc(tune->desc, tune->type);

   if (t_ptr != NULL)
   {
      tune_data.module = module;
      tune_data.u.transport.t_ptr = t_ptr;

      if (tune->service_id != 0)
      {
         DBDEF_RequestAccess();

         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                  t_ptr->orig_net_id, t_ptr->tran_id, tune->service_id);
         if (s_ptr == NULL)
         {
            /* Create a service record */
            if ((s_ptr = DBDEF_AddServiceRec(tune->service_id, t_ptr)) != NULL)
            {
               AP_CI_PRINT(("%s: new service with ID 0x%x", __FUNCTION__, tune->service_id));

               /* Make the service hidden and non-selectable so it isn't available to the user */
               s_ptr->hidden = TRUE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);
               s_ptr->selectable = FALSE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);
               DBA_SaveRecord(s_ptr->dba_rec);

               DBDEF_SetServiceType(s_ptr, tune->service_type);
               if (tune->service_name != NULL)
               {
                  DBDEF_SetServiceName(s_ptr,tune->service_name->str_ptr);
               }
            }
         }

         AP_CI_PRINT(("%s: SID 0x%x s_ptr=%p", __FUNCTION__, tune->service_id, s_ptr));

         STB_SIReleaseStringDesc(tune->service_name);

         if ((s_ptr != NULL) && (tune->event_desc != NULL))
         {
            if (s_ptr->now_event != NULL)
            {
               /* Free the existing now event */
               DBDEF_DeleteEventList(s_ptr->now_event);
            }

            if ((s_ptr->now_event = STB_AppGetMemory(sizeof(ADB_EVENT_REC))) != NULL)
            {
               memset(s_ptr->now_event, 0, sizeof(ADB_EVENT_REC));
               s_ptr->now_event->desc_list_head = tune->event_desc;
               s_ptr->now_event->desc_list_tail = tune->event_desc;
            }
         }

         DBDEF_ReleaseAccess();

         if (s_ptr != NULL)
         {
            tune_data.type = CIPLUS_TUNE_TYPE_CI_SERVICE;
            tune_data.flags = flags;
            tune_data.u.service.s_ptr = s_ptr;

            if (tune->pmt == NULL)
            {
               /* The PMT hasn't been provided so needs to be obtained from the broadcast */
               tune_data.u.service.reqd_si = APP_SI_MODE_CIPLUS_UPDATE;
               tune_data.u.service.pmt = NULL;
            }
            else
            {
               /* The PMT has been provided so shouldn't be obtained from the broadcast */
               tune_data.u.service.reqd_si = APP_SI_MODE_CIPLUS_NO_PAT_PMT;

               /* Copy the PMT data.
                * This is a bit nasty, but the PMT data is allocated here and needs to freed by
                * when the event is handled, or sometime after. */
               pmt_size = ((tune->pmt[1] & 0x0f) << 8) + tune->pmt[2];
               if (pmt_size > 0)
               {
                  if ((tune_data.u.service.pmt = STB_AppGetMemory(pmt_size)) != NULL)
                  {
                     memcpy(tune_data.u.service.pmt, tune->pmt, pmt_size);
                  }
               }
            }

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_TUNE, &tune_data, sizeof(tune_data));
         }
         else
         {
            ACI_TuneReply(INVALID_RES_ID, module, CIP_TUNER_SERVICE_NOT_FOUND);
         }
      }
      else
      {
         /* Just tuning to a transport */
         tune_data.type = CIPLUS_TUNE_TYPE_TRANSPORT;
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_TUNE, &tune_data, sizeof(tune_data));
      }
   }
   FUNCTION_FINISH(ACI_TuneToDelSysDesc);
}

/**
 * @brief   Ask the host control module to restore replaced PIDs and to close
 *          the session with the host control resource
 * @param   module specifies module required
 * @return  TRUE if operation successful, FALSE otherwise
 */
BOOLEAN ACI_AskRelease(U32BIT module)
{
   BOOLEAN retval;

   FUNCTION_START(ACI_AskRelease);

   retval = ACI_OpAskRelease(module);
   if (!retval)
   {
      retval = ACI_HcAskRelease(module);
   }

   FUNCTION_FINISH(ACI_AskRelease);

   return(retval);
}

/**
 * @brief   This function is called by the host to send the status of the tune
 *          operation to the module.
 * @param   path - decode path used for tuning
 * @param   module - host control module
 * @param   status - tune operation status
 */
void ACI_TuneReply(U8BIT path, U32BIT module, E_CIP_TUNER_STATUS status)
{
   FUNCTION_START(ACI_TuneReply);
   if (!ACI_HcTuneReply(module, status))
   {
      ACI_OpTuneReply(path, module, status);
   }
   FUNCTION_FINISH(ACI_TuneReply);
}

/**
 * @brief   Use the CI module with the given path
 * @param   path - decode path
 * @param   module - CI module
 */
void ACI_UseCiModuleOnPath(U8BIT path, U32BIT module)
{
   void *service;
   U16BIT service_id;
   U8BIT slot_id;
   slot_id = STB_GetCIHostControlSlotId(module);
   if (slot_id != INVALID_RES_ID &&
      STB_DPUseCISlotWithPath(path, slot_id))
   {
      service = STB_DPGetTunedService(path);
      if (service != NULL)
      {
         service_id = ADB_GetServiceId(service);
      }
      else
      {
         service_id = 0xffff;
      }
      /* Set the recording mode based on whether the slot is being used on the live path */
      if (STB_DPIsLivePath(path) || STB_DPIsLivePath(STB_DPIsCISlotInUse(0, slot_id, path)))
      {
         AP_CI_PRINT(("%s(%u): slot %u, record mode is WATCH for service %u",
                        __FUNCTION__, path, slot_id, service_id));
         STB_CiCcSetRecordOperatingMode(slot_id, STB_CI_MODE_WATCH_AND_BUFFER, service_id);
      }
      else
      {
         AP_CI_PRINT(("%s(%u): slot %u, record mode UNATTENDED for service %u",
                        __FUNCTION__, path, slot_id, service_id));
         STB_CiCcSetRecordOperatingMode(slot_id, STB_CI_MODE_UNATTENDED_RECORDING, service_id);
      }
      /* Apply any CI+ keys */
      STB_CiKeysApply(path, slot_id);
   }
}

/**
 * @brief   Converts the pin value to ASCII and sends it to the CAM
 *          An STB_EVENT_CI_PIN_STATUS event will be sent to notify
 *          the validity, or otherwise, of the pin.
 * @param   slot_id - slot Id
 * @param   pin - pin value
 * @return  TRUE if the pin was sent, FALSE otherwise
 */
BOOLEAN ACI_SendPinToCam(U8BIT slot_id, U32BIT pin)
{
   U8BIT pin_str[16];
   BOOLEAN retval;

   FUNCTION_START(ACI_SendPinToCam);

   /* Convert the pin value to a string */
   sprintf((char *)pin_str, "%lu", (unsigned long)pin);

   /* Send the pin value to the CAM */
   retval = STB_CiCcSendPin(slot_id, pin_str);

   FUNCTION_FINISH(ACI_SendPinToCam);

   return(retval);
}

/**
 * @brief   Checks whether path is owned by CI module
 * @param   module specifies module (host control or operator)
 * @return  TRUE if path owned by module, FALSE otherwise
 */
BOOLEAN ACI_PathOwnedByModule(U8BIT path, U32BIT module)
{
   BOOLEAN owned;
   void *owner_data;
   U32BIT data_size;
   if ((STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS)))
   {
      owner_data = STB_DPGetOwnerData(path, &data_size);
      if (owner_data != NULL && module == *(U32BIT *)owner_data)
      {
         owned = TRUE;
      }
      else
      {
         owned = FALSE;
      }
   }
   else
   {
      owned = FALSE;
   }
   return owned;
}

/**
 * @brief   Register callback function to notify start of tuning process due to CI
 *          tune request
 * @param   cb_func Callback function - NULL disables the call.
 */
void ACI_RegisterStartTuneNotifyCallback(F_NotifyStartTuning cb_func)
{
   NotifyStartTuning = cb_func;
}
