/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CI - internal functions
 * @file    ap_ci_int.h
 * @date    December 2016
 * @author  Adam Sturtridge
 */

#ifndef AP_CI_INT_H
#define AP_CI_INT_H

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "stbsitab.h"
#include "ap_ci.h"

/*---Constant and macro definitions for public use-----------------------------*/

/*---Enumerations for public use-----------------------------------------------*/

typedef enum e_cip_tuner_status
{
   CIP_TUNER_LOCKED, /* STB_CI_TUNE_OK */
   CIP_TUNER_UNSUPPORTED_SYSTEM,
   CIP_TUNER_NOTLOCKED,
   CIP_TUNER_BUSY,
   CIP_TUNER_BAD_PARAMETER,
   CIP_TUNER_SERVICE_NOT_FOUND,
   CIP_TUNER_UNDEFINED_ERROR
} E_CIP_TUNER_STATUS;

/*---Global type defs for public use-------------------------------------------*/

typedef struct
{
   U32BIT module;
   U8BIT reply;
   U8BIT path;
} S_CIP_RELEASE_REPLY;

typedef struct adb_event_desc * pADB_EVENT_DESC;

typedef struct ci_tune_del_sys_desc
{
   SI_DELIVERY_SYS_DESC_TYPE type;
   SI_DELIVERY_SYS_DESC *desc;
   U16BIT service_id;
   U8BIT *pmt;
   U8BIT service_type;
   SI_STRING_DESC *service_name;
   pADB_EVENT_DESC event_desc;
} S_CIP_TUNE_DEL_SYS_DESC;

/*---Global Function prototypes for public use---------------------------------*/

/**
 * @brief    Initialise CI Host Control support
 */
void ACI_Initialise(void);

/**
 * @brief   Checks whether the given timer is associated with any of the CI+ profiles
 *          and starts the operator search if possible
 * @param   timer_handle timer handle
 * @return  TRUE if the timer is associated with a profile, FALSE otherwise
 */
BOOLEAN ACI_HandlePrivateTimer(U32BIT timer_handle);

/**
 * @brief   Ensures the TS is routed securely for CI+, either by setting the TS
 *          to pass through if a CI slot contains a CI+ CAM or bypass it if it doesn't.
 */
BOOLEAN ACI_SetSecureRouting(U8BIT path);

/**
 * @brief   Re-evaluate current state follwing a change related to usage rules
 * @param   path decode path
 */
void ACI_UsageRulesStatusChanged(U8BIT path);

/**
 * @brief  Handle PMT change
 * @param  pmt the new PMT
 */
void ACI_ProgramMapTableChanged(U8BIT *pmt);

/**
 * @brief   The given decode path is only trusted if it doesn't include a CI slot
 *          or the CI slot contains a CI+ CAM.
 * @param   path decode path
 * @return  TRUE if the decode path is trusted, FALSE otherwise
 */
BOOLEAN ACI_IsTrustedPath(U8BIT path);

/**
 * @brief   Use the CI module with the given path
 * @param   path - decode path
 * @param   module - CI module
 */
void ACI_UseCiModuleOnPath(U8BIT path, U32BIT module);

/**
 * @brief   Checks whether path is owned by CI module
 * @param   module specifies module (host control or operator)
 * @return  TRUE if path owned by module, FALSE otherwise
 */
BOOLEAN ACI_PathOwnedByModule(U8BIT path, U32BIT module);

/**
 * @brief   Schedule a tune to a transport
 * @param   module module id
 * @param   t_ptr target transport pointer
 */
void ACI_TuneToTransport(U32BIT module, void *t_ptr);

/**
 * @brief   Schedule a tune to a service
 * @param   module module id
 * @param   s_ptr target service pointer
 * @param   flags flags to pass to presentation engine
 */
void ACI_TuneToService(U32BIT module, void *s_ptr, E_CIP_TUNE_FLAGS flags);

/**
 * @brief   Tune to a service/transport defined by a delivery system descriptor,
 *          possibly with PMT data, if it's a service.
 * @param   module - module id
 * @param   tune - tuning parameters
 * @param   flags flags to pass to presentation engine
 */
void ACI_TuneToDelSysDesc(U32BIT module, S_CIP_TUNE_DEL_SYS_DESC *tune, E_CIP_TUNE_FLAGS flags);

/**
 * @brief   Ask the host control module to restore replaced PIDs and to close
 *          the session with the host control resource
 * @param   module specifies module required
 * @return  TRUE if operation successful, FALSE otherwise
 */
BOOLEAN ACI_AskRelease(U32BIT module);


/**
 * @brief   This function is called by the host to send the status of the tune
 *          operation to the module.
 * @param   path decode path used for tuning
 * @param   module host control module
 * @param   status tune operation status
 */
void ACI_TuneReply(U8BIT path, U32BIT module, E_CIP_TUNER_STATUS status);

#endif /* AP_CI_INT_H */

/******************************************************************************
** End of file
******************************************************************************/
