/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CI Host Control functions
 * @file    ap_cihc.c
 * @date    December 2016
 * @author  Adam Sturtridge
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"

#include "stberc.h"
#include "stbheap.h"

#include "ap_dbdef.h"
#include "ap_cntrl.h"

#include "stbci.h"
#include "stbcicc.h"
#include "ap_ci_int.h"
#include "ap_ci_support.h"
#include "ap_cihc.h"

//---constant definitions for this file----------------------------------------

#define DEBUG_HC
#ifdef DEBUG_HC
#define DBG_HC(x, ...) STB_SPDebugWrite( "%s:%d " x, __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
#define DBG_HC(x, ...)
#endif

#define DBG_ERR(x, ...) STB_SPDebugWrite( "ERROR %s:%d" x, __FUNCTION__, __LINE__, ##__VA_ARGS__);

#define DTAG_DEL_SYS_DVBT     0x5a
#define DTAG_DEL_SYS_DVBT2    0x04
#define DTAG_DEL_SYS_DVBC     0x44
#define DTAG_DEL_SYS_DVBC2    0x0d
#define DTAG_DEL_SYS_DVBS     0x43
#define DTAG_DEL_SYS_DVBS2    0x79
#define DTAG_EXT_DESC         0x7f

//---local typedefs, structs, enumerations for this file--------------------------------------------


typedef struct s_hc_status
{
   struct s_hc_status *next;
   U32BIT module;
   BOOLEAN tune_started;
   BOOLEAN ask_release;
   BOOLEAN disabled;
} S_HC_STATUS;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static S_HC_STATUS *cihc_list = NULL;
static void *cihc_mutex;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static S_HC_STATUS *FindHcState(U32BIT module);
static void* FindService(U16BIT net_id, U16BIT onet_id, U16BIT ts_id, U16BIT sid);
static void* FindTransport(U16BIT net_id, U16BIT onet_id, U16BIT ts_id);

//----------------------------------------------------------------------------
//--- Global function for this file ------------------------------------------
//----------------------------------------------------------------------------

/**
 * @brief    Initialise CI Host Control support
 */
void ACI_HcInitialise(void)
{
   cihc_mutex = STB_OSCreateMutex();
}

/**
 * @brief   This function is used by the CI stack to notify the host that a session
 *          with the host countrol resource has been opened by the module.
 *          This signals to the host that the module intends to request that the
 *          host tunes to other transport streams / services.
 *          Following this notification, the host must call the function
 *          STB_CIAskRelease to request that the module closes the session.
 *          The function STB_CIAskReleaseReply will be called to let the host know
 *          whether the control over the tuner has been released or not.
 * @param   module host control module
 *
 */
void ACI_HcNotifyHostControlSession(U32BIT module)
{
   S_HC_STATUS *hc_state;
   U8BIT path;
   S_ACTL_OWNER_INFO owner_info;

   FUNCTION_START(ACI_HcNotifyHostControlSession);

   if (FindHcState(module) != NULL)
   {
      DBG_ERR("(%u) HC module already exists", module)
   }
   else
   {
      hc_state = STB_GetMemory(sizeof(S_HC_STATUS));
      if (hc_state == NULL)
      {
         DBG_ERR("(%u) Memory alloc failed", module)
      }
      else
      {
         hc_state->module = module;
         hc_state->tune_started = FALSE;
         hc_state->ask_release = FALSE;
         hc_state->next = cihc_list;
         cihc_list = hc_state;

         if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
         {
            /* Tune the live path to the service requested by CI+ */
            owner_info.owner = RES_OWNER_CIPLUS;
            owner_info.data = &module;
            owner_info.data_size = sizeof(module);

            if (ACTL_AcquirePathOwnership(path, &owner_info))
            {
               /* Ownership of the live path has been acquired */
               DBG_HC("(%u): Acquired ownership of path %u", module, path)
            }
            else
            {
               /* Ownership of the live path has been denied */
               DBG_HC("(%u): Couldn't acquire ownership of path %u", module, path)
            }
         }
         else
         {
            /* No live path available, which means the CI stack will be able to take ownership
             * of the tuner if it issues a tune request, so ownership can be deferred */
            DBG_HC("(%u): No live path", module)
         }
      }
   }

   FUNCTION_FINISH(ACI_HcNotifyHostControlSession);
}

/**
 * @brief   This function is used by the CI stack to notify the host that a session
 *          with the host control resource has been closed by the module.
 *          This signals to the host that it has controls over the tuner and it
 *          does not need to call STB_CIAskRelease before using it.
 *          Following this call, any PIDs that have been replaced by STB_CIReplace
 *          should be restored, if possible.
 * @param   module - host control module
 */
void ACI_HcNotifyHostControlSessionClosed(U32BIT module)
{
   S_HC_STATUS *hc_state;
   S_HC_STATUS **phc_state;

   FUNCTION_START(ACI_HcNotifyHostControlSessionClosed);

   phc_state= &cihc_list;
   hc_state = cihc_list;
   while (hc_state != NULL)
   {
      if (hc_state->module == module)
      {
         U8BIT path;
         for (path = 0; path != STB_DPGetNumPaths(); path++)
         {
            if (ACI_PathOwnedByModule(path, module))
            {
               /* Release ownership of this path */
               DBG_HC("(%u): Releasing ownership of path %u (slot_id=%u)", module, path, STB_DPGetPathCISlot(path))
               ACTL_ReleasePathOwnership(path, RES_OWNER_CIPLUS);
               if (hc_state->ask_release)
               {
                  S_CIP_RELEASE_REPLY reply_data;
                  /* The host has asked the module to release host control, but the module hasn't
                   * replied and is now closing the session, so respond to the release request first */
                  reply_data.module = module;
                  reply_data.reply = 0;
                  reply_data.path = path;
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RELEASE_REPLY, &reply_data, sizeof(reply_data));
               }
            }
         }

         *phc_state = hc_state->next;
         STB_FreeMemory(hc_state);
         break;
      }
      phc_state = &(hc_state->next);
      hc_state = hc_state->next;
   }

   FUNCTION_FINISH(ACI_HcNotifyHostControlSessionClosed);
}


/**
 * @brief   Ask the module to restore replaced PIDs and to close the session
 *          with the host control resource
 * @param   module specifies module required
 * @return  TRUE if operation successful, FALSE otherwise
 */
BOOLEAN ACI_HcAskRelease(U32BIT module)
{
   S_HC_STATUS *hc_state;
   BOOLEAN retval;

   FUNCTION_START(ACI_HcAskRelease);

   DBG_HC("(%u)", module)

   hc_state = FindHcState(module);
   if (hc_state != NULL)
   {
      if (!hc_state->ask_release)
      {
         hc_state->ask_release = TRUE;
         retval = STB_CIAskRelease(module);
      }
      else
      {
         DBG_HC("(%u): already asked to release HC session", module)
         retval = TRUE;
      }
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACI_HcAskRelease);

   return(retval);
}

/**
 * @brief   This function is called by the CI+ stack to send the reply of a
 *          release request to the host.
 *          This function is called in response to a call to STB_CIAskRelease.
 *          The host must not assume that it has control over the tuner until
 *          this function is called.
 * @param   module specifies module required
 * @param   release_reply the reply to the release request
 */
void ACI_HcAskReleaseReply(U32BIT module, U8BIT release_reply)
{
   S_HC_STATUS *hc_state;
   S_CIP_RELEASE_REPLY reply_data;
   U8BIT path;

   FUNCTION_START(ACI_HcAskReleaseReply);

   DBG_HC("(%u, reply=%u)", module, release_reply)

   hc_state = FindHcState(module);
   if (hc_state != NULL)
   {
      if (!hc_state->ask_release)
      {
         DBG_HC("(%u): Host hasn't asked for release", module)
      }

      for (path = 0; path != STB_DPGetNumPaths(); path++)
      {
         if (ACI_PathOwnedByModule(path, module))
         {
            reply_data.path = path;
            reply_data.module = module;
            reply_data.reply = release_reply;

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RELEASE_REPLY, &reply_data, sizeof(reply_data));
         }
      }
      hc_state->ask_release = FALSE;
   }

   FUNCTION_FINISH(ACI_HcAskReleaseReply);
}

/**
 * @brief    Handle Tune request from the CAM
 * @param    module Host control module
 * @param    nid network ID
 * @param    onid original network ID
 * @param    tsid transport stream ID
 * @param    sid service ID
 */
void ACI_HcTune(U32BIT module, U16BIT nid, U16BIT onid, U16BIT tsid, U16BIT sid)
{
   S_HC_STATUS *hc_state;
   void *target;

   FUNCTION_START(ACI_HcTune);

   DBG_HC("(%u, nid=%u, onid=%u, tid=%u, sid=%u)", module, nid, onid, tsid, sid)

   hc_state = FindHcState(module);
   if (hc_state == NULL)
   {
      DBG_ERR("(%u): host control module not recognised",module)
      STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
   }
   else
   {
      if (hc_state->tune_started)
      {
         /* Tuner is already in use */
         DBG_HC("(%u): Tuner is already tuning", module)
         STB_CITuneReply(module, STB_CI_TUNE_TUNER_BUSY);
      }
      else
      {
         if (sid == 0)
         {
            target = FindTransport(nid, onid, tsid);
            if (target == NULL)
            {
               DBG_HC("(%u): Failed to find transport", module)
               STB_CITuneReply(module, STB_CI_TUNE_SERVICE_NOT_FOUND);
            }
            else
            {
               hc_state->tune_started = TRUE;
               ACI_TuneToTransport(module, target);
            }
         }
         else
         {
            target = FindService(nid, onid, tsid, sid);
            if (target == NULL)
            {
               DBG_HC("(%u): Failed to find service", module)
               STB_CITuneReply(module, STB_CI_TUNE_SERVICE_NOT_FOUND);
            }
            else
            {
               hc_state->tune_started = TRUE;
               ACI_TuneToService(module, target, CIP_NORMAL_TUNE);
            }
         }
      }
   }

   FUNCTION_FINISH(ACI_HcTune);
}

/**
 * @brief   This function is used by the CI+ stack to request that the host tunes
 *          to a transport stream using the given parameters.
 *          If the service_id is zero, the host should not report a PMT using
 *          STB_CIUpdatePmt().
 *          If the PMT is provided, it should be used when selecting elementary
 *          streams on the service. The PMT is not provided if service_id is zero.
 *          Following the tune operation (whether successful or not), the host
 *          must call the function STB_CITuneReply().
 * @param   module host control module
 * @param   service_id the service to tune to
 * @param   desc_loop_len length of descriptor loop in bytes
 * @param   desc_loop the descriptor loop
 * @param   pmt the PMT to use (or NULL)
 * @param   flags flags to pass to presentation engine
 */
void ACI_HcTuneBroadcastRequest(U32BIT module, U16BIT service_id,
   U16BIT desc_loop_len, U8BIT *desc_loop, U8BIT *pmt, E_CIP_TUNE_FLAGS flags)
{
   S_HC_STATUS *hc_state;
   U8BIT dtag;
   U8BIT dlen;
   S_CIP_TUNE_DEL_SYS_DESC tune_del_sys;
   BOOLEAN tune_valid;
   SI_STRING_DESC *provider;

   FUNCTION_START(ACI_HcTuneBroadcastRequest);

   DBG_HC("(%u, service_id=%u, desc_loop_len=%u, pmt=%p)", module, service_id, desc_loop_len, pmt)

   hc_state = FindHcState(module);
   if (hc_state != NULL)
   {
      if (!hc_state->tune_started)
      {
         if ((desc_loop_len > 0) && (desc_loop != NULL))
         {
            tune_valid = TRUE;

            if (pmt != NULL)
            {
               /* Check that the PMT is for the requested service ... */
               if ((service_id == 0) || (service_id != ((pmt[3] << 8) + pmt[4])) ||
                  /* ... and that the CICAM has authenticated */
                  !STB_CiCcAuthenticated(STB_GetCIHostControlSlotId(module)))
               {
                  STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
                  tune_valid = FALSE;
               }
            }

            if (tune_valid)
            {
               memset(&tune_del_sys, 0, sizeof(tune_del_sys));
               while ((desc_loop_len > 0) && tune_valid)
               {
                  dtag = desc_loop[0];

                  /* Descriptor length is +2 to allow for the tag and length bytes */
                  dlen = desc_loop[1] + 2;

                  switch (dtag)
                  {
                     case DTAG_DEL_SYS_DVBT:  /* Terrestrial delivery descriptor */
                     case DTAG_DEL_SYS_DVBS:  /* Satellite delivery descriptor */
                     case DTAG_DEL_SYS_DVBC:  /* Cable delivery descriptor */
                     {
                        DBG_HC("%s delivery descriptor", (dtag == DTAG_DEL_SYS_DVBT) ? "DTAG_DEL_SYS_DVBT" :
                           (dtag == DTAG_DEL_SYS_DVBS) ? "DTAG_DEL_SYS_DVBS" :
                           (dtag == DTAG_DEL_SYS_DVBC) ? "DTAG_DEL_SYS_DVBC" : "Unknown")
                        if (tune_del_sys.desc == NULL)
                        {
                           if (!STB_SIParseDelSysDesc(desc_loop, &tune_del_sys.type,
                                  &tune_del_sys.desc))
                           {
                              DBG_HC("could not parse delivery descriptor")
                              STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
                              tune_valid = FALSE;
                           }
                        }
                        else
                        {
                           /* Already have a delivery descriptor, so skip this one */
                           DBG_HC("(%u): Skipping additional delivery desc, dtag=0x%02x", module, dtag)
                        }
                        break;
                     }

                     case 0x48:  /* Service descriptor */
                        DBG_HC("(%u): Service desc", module)
                        if (STB_SIParseServiceDescriptor(desc_loop, &tune_del_sys.service_type, &provider,
                               &tune_del_sys.service_name))
                        {
                           /* Provider string isn't used so free it */
                           if (provider != NULL)
                           {
                              STB_SIReleaseStringDesc(provider);
                           }
                        }
                        break;

                     case 0x4d:  /* Short event descriptor */
                        DBG_HC("(%u): Short event desc", module)
                        tune_del_sys.event_desc = (ADB_EVENT_DESC *)STB_AppGetMemory(sizeof(ADB_EVENT_DESC) + dlen);
                        if (tune_del_sys.event_desc != NULL)
                        {
                           tune_del_sys.event_desc->desc_data = (U8BIT *)(tune_del_sys.event_desc + 1);
                           memcpy(tune_del_sys.event_desc->desc_data, desc_loop, dlen);
                           tune_del_sys.event_desc->next = NULL;
                        }
                        //STB_SIParseShortEventDescriptor(desc_loop, &tune_del_sys.event_desc);
                        break;

                     case 0x50:  /* Component descriptor */
                        DBG_HC("(%u): Component desc", module)
                        break;

                     case 0x55:  /* Parental rating descriptor */
                        DBG_HC("(%u): Parental rating desc", module)
                        break;

                     case 0x54:  /* Content descriptor */
                        DBG_HC("(%u): Content desc", module)
                        break;

                     default:
                        /* Skip the unhandled descriptor */
                        DBG_HC("Skipping unhandled descriptor 0x%02x, len %u bytes", dtag, dlen);
                        break;
                  }

                  desc_loop += dlen;
                  desc_loop_len -= dlen;
               }

               if (tune_valid)
               {
                  hc_state->tune_started = TRUE;

                  tune_del_sys.service_id = service_id;
                  tune_del_sys.pmt = pmt;

                  ACI_TuneToDelSysDesc(module, &tune_del_sys, flags);
               }
            }
         }
         else
         {
            DBG_ERR("(%u): descriptor loop invalid (%p,%u)",module, desc_loop, desc_loop_len)
            STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
         }
      }
      else
      {
         DBG_HC("(%u): Tuner is already tuning", module)
         STB_CITuneReply(module, STB_CI_TUNE_TUNER_BUSY);
      }
   }
   else
   {
      DBG_ERR("(%u): host control module not recognised",module)
      STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
   }

   FUNCTION_FINISH(ACI_HcTuneBroadcastRequest);
}

/**
 * @brief   Handle tune to service request from the CAM
 * @param   module Host control module
 * @param   s_ptr service
 * @param   flags flags to pass to presentation engine
 */
void ACI_HcTuneService(U32BIT module, void *s_ptr, E_CIP_TUNE_FLAGS flags)
{
   S_HC_STATUS *hc_state;

   FUNCTION_START(ACI_HcTuneService);

   DBG_HC("(%u, lcn=%u, flgs=%x)", module, ADB_GetServiceLcn(s_ptr), flags)

   hc_state = FindHcState(module);
   if (hc_state == NULL)
   {
      DBG_ERR("(%u): host control module not recognised",module)
      STB_CITuneReply(module, STB_CI_TUNE_BAD_PARAMETER);
   }
   else
   {
      if (hc_state->tune_started)
      {
         /* Tuner is already in use */
         DBG_HC("(%u): Tuner is already tuning", module)
         STB_CITuneReply(module, STB_CI_TUNE_TUNER_BUSY);
      }
      else
      {
         hc_state->tune_started = TRUE;
         ACI_TuneToService(module, s_ptr, flags);
      }
   }

   FUNCTION_FINISH(ACI_HcTuneService);
}

/**
 * @brief   Send status of tune operation to the CAM host control module
 * @param   module host control module
 * @param   status tune operation status
 * @return  TRUE if valid module, FALSE otherwise
 */
BOOLEAN ACI_HcTuneReply(U32BIT module, E_STB_CI_TUNE_STATUS status)
{
   S_HC_STATUS *hc_state;
   BOOLEAN retval;

   FUNCTION_START(ACI_HcTuneReply);

   DBG_HC("(mod=%u, status=%u)", module, status)

   hc_state = FindHcState(module);
   if (hc_state != NULL)
   {
      retval = TRUE;
      if (hc_state->tune_started)
      {
         hc_state->tune_started = FALSE;
         DBG_HC("(%u, status=%s)", module,
            (status == STB_CI_TUNE_OK) ? "CIP_TUNER_LOCKED" :
            (status == STB_CI_TUNE_TUNER_NOT_LOCKING) ? "CIP_TUNER_NOTLOCKED" :
            (status == STB_CI_TUNE_TUNER_BUSY) ? "CIP_TUNER_BUSY" :
            (status == STB_CI_TUNE_UNSUPPORTED_SYSTEM) ? "CIP_TUNER_UNSUPPORTED_SYSTEM" :
            (status == STB_CI_TUNE_BAD_PARAMETER) ? "CIP_TUNER_BAD_PARAM" :
            (status == STB_CI_TUNE_SERVICE_NOT_FOUND) ? "CIP_TUNER_SERVICE_NOT_FOUND" :
            (status == STB_CI_TUNE_UNDEFINED_ERROR) ? "CIP_TUNER_UNDEFINED_ERROR" : "unknown (returning UNDEFINED_ERROR)")
         (void)STB_CITuneReply(module, status);
      }
      else
      {
         DBG_HC("(%u): already asked to release HC session", module)
      }
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACI_HcTuneReply);

   return(retval);
}

//----------------------------------------------------------------------------
//--- Local function for this file ------------------------------------------
//----------------------------------------------------------------------------

/**
 * @brief   Find host control status info
 * @param   module host control module
 * @return  Pointer to host control status
 */
static S_HC_STATUS *FindHcState(U32BIT module)
{
   S_HC_STATUS *hc_state;
   STB_OSMutexLock(cihc_mutex);
   hc_state = cihc_list;
   while (hc_state != NULL)
   {
      if (module == hc_state->module)
      {
         if (hc_state->disabled)
         {
            hc_state = NULL;
         }
         break;
      }
      hc_state = hc_state->next;
   }
   STB_OSMutexUnlock(cihc_mutex);
   return hc_state;
}

/**
 * @brief   Find service given DVB locator
 * @param   net_id network id
 * @param   onet_id original network id
 * @param   ts_id transport id
 * @param   sid service id
 * @return  Pointer to service record
 */
static void* FindService(U16BIT net_id, U16BIT onet_id, U16BIT ts_id, U16BIT sid)
{
   ADB_SERVICE_REC *service;

   FUNCTION_START(FindService);

   DBDEF_RequestAccess();
   service = DBDEF_GetNextServiceRec(NULL);
   while (service != NULL)
   {
      /* Network id 0 is used as a wildcard */
      if ((service->serv_id == sid) &&
          (service->transport->tran_id == ts_id) &&
          (service->transport->orig_net_id == onet_id) &&
          ((net_id == 0) || (service->transport->network->net_id == net_id)))
      {
         break;
      }
      service = DBDEF_GetNextServiceRec(service);
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(FindService);

   return service;
}

/**
 * @brief   Find transport given DVB locator
 * @param   net_id network id
 * @param   onet_id original network id
 * @param   ts_id transport id
 * @return  Pointer to transport record
 */
static void* FindTransport(U16BIT net_id, U16BIT onet_id, U16BIT ts_id)
{
   ADB_TRANSPORT_REC *transport;

   FUNCTION_START(FindTransport);

   DBDEF_RequestAccess();
   transport = DBDEF_GetNextTransportRec(NULL);
   while (transport != NULL)
   {
      /* Network id 0 is used as a wildcard */
      if ((transport->tran_id == ts_id) &&
          (transport->orig_net_id == onet_id) &&
          ((net_id == 0) || (transport->network->net_id == net_id)))
      {
         break;
      }
      transport = DBDEF_GetNextTransportRec(transport);
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(FindTransport);

   return transport;
}

