/*******************************************************************************
 * Copyright  2016 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level CI Operator Profile functions
 * @file    ap_ciop.c
 * @date    December 2016
 * @author  Adam Sturtridge
 */

//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"

#include "stberc.h"
#include "stbheap.h"
#include "stbuni.h"

#include "dba.h"

#include "app.h"
#include "ap_dbacc.h"
#include "ap_dbdef.h"
#include "ap_ci.h"
#include "ap_state.h"

#include "stbci.h"
#include "ci_appmmi.h"
#include "ap_ci_int.h"
#include "ap_ci_support.h"
#include "ap_ciop.h"

//---constant definitions for this file----------------------------------------

#define DEBUG_OP
#ifdef DEBUG_OP
#define DBG_OP(x, ...) STB_SPDebugWrite( "%s:%d " x, __FUNCTION__, __LINE__, ##__VA_ARGS__);
#else
#define DBG_OP(x, ...)
#endif

#define CIP_SERVICE_TYPE_TV         0x00000001   /* MPEG2 TV */
#define CIP_SERVICE_TYPE_RADIO      0x00000002   /* MPEG1 Layer-II radio */
#define CIP_SERVICE_TYPE_TELETEXT   0x00000004   /* Teletext */
#define CIP_SERVICE_TYPE_AVC_RADIO  0x00000008   /* AVC radio */
#define CIP_SERVICE_TYPE_DATA       0x00000010   /* Data */
#define CIP_SERVICE_TYPE_HD_TV      0x00000020   /* MPEG2 HD TV */
#define CIP_SERVICE_TYPE_AVC_SD_TV  0x00000040   /* AVC SD TV */
#define CIP_SERVICE_TYPE_AVC_HD_TV  0x00000080   /* AVC HD TV */

#define CIP_DELIVERY_TYPE_DVBT      0x0001
#define CIP_DELIVERY_TYPE_DVBT2     0x0002
#define CIP_DELIVERY_TYPE_DVBC      0x0004
#define CIP_DELIVERY_TYPE_DVBC2     0x0008
#define CIP_DELIVERY_TYPE_DVBS      0x0010
#define CIP_DELIVERY_TYPE_DVBS2     0x0020

#define CIP_APP_TYPE_MHEG           0x0001
#define CIP_APP_TYPE_HBBTV          0x0002
#define CIP_APP_TYPE_OIPF           0x0004

#define INVALID_MODULE              0xFFFFFFFF

#define CIP_NUM_FREE_LCNS           2

//---local typedefs, structs, enumerations for this file--------------------------------------------

typedef enum
{
   CIP_OPERATOR_SEARCH_DEFERRED,
   CIP_OPERATOR_SEARCH_IMMEDIATE,
   CIP_OPERATOR_SEARCH_SCHEDULED
} E_CIP_OPERATOR_SEARCH_REQUEST_TYPE;

typedef struct ci_operator_nit
{
   U16BIT cicam_onet_id;
   U32BIT cicam_identifier;
   U32BIT lang_code;
   U8BIT profile_name_length;
   U8BIT *profile_name;
   SI_NIT_TABLE *table;
} S_CIP_OPERATOR_NIT;

typedef struct ci_host_info
{
   BOOLEAN standby;
   U8BIT num_service_types;
   U32BIT service_types;
   U16BIT delivery_types;
   U16BIT app_types;
} S_CIP_HOST_INFO;

typedef struct ci_operator_search
{
   E_CIP_OPERATOR_SEARCH_REQUEST_TYPE refresh_request;
   U16BIT refresh_date;
   U8BIT refresh_hour;
   U8BIT refresh_min;
   /**
    * @brief   CICAM ID used when refres_request = CIP_OPERATOR_SEARCH_SCHEDULED to associate a timer
    *          to the requesting CICAM
    */
   U32BIT cicam_id;
} S_CIP_OPERATOR_SEARCH;

typedef struct s_op_status
{
   struct s_op_status *next;
   U32BIT module;
   U8BIT slot_id;
   U8BIT op_desc_num;
   U16BIT host_delivery_types;
   U16BIT op_desc_loop_len;
   U16BIT op_current_loc_len;
   U8BIT *op_desc_loop;
   U8BIT *op_current_location;
   S_STB_CI_OPERATOR_STATUS op_status;
   S_STB_CI_OPERATOR_INFO op_info;
   BOOLEAN tune_started;
   BOOLEAN op_status_valid;
   BOOLEAN op_info_valid;
   BOOLEAN op_nit_received;
   BOOLEAN op_search_started;
} S_OP_STATUS;

//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static S_OP_STATUS *ciop_list = NULL;
static void *ciop_mutex;
static U32BIT op_module;
static U32BIT default_und_lang = ADB_LANG_CODE_UNDEF;

//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static S_OP_STATUS *FindOpState(U32BIT module);
static BOOLEAN StartOperatorSearch(S_OP_STATUS *op_state);
static BOOLEAN PrepareSearchInfo(S_CIP_HOST_INFO *host_info, S_STB_CI_SEARCH_START_INFO *search_info);
static void PerformOperatorTune(S_OP_STATUS *op_state);
static U32BIT ConvertLangCode(U8BIT *data);
static U8BIT GetCharacterCodeTable(S_STB_CI_OPERATOR_INFO *op_info);

static void ProcessNit(U32BIT module, S_CIP_OPERATOR_NIT *nit);
static void RequestHostInfo(S_CIP_HOST_INFO *host_info);
static BOOLEAN RequestOperatorSearch(U32BIT module, S_CIP_OPERATOR_SEARCH *op_search);

//----------------------------------------------------------------------------
//--- Global function for this file ------------------------------------------
//----------------------------------------------------------------------------

/**
 * @brief    Initialise CI Operator Profile support
 */
void ACI_OpInitialise(void)
{
   FUNCTION_START(ACI_OpInitialise);
   ciop_mutex = STB_OSCreateMutex();
   FUNCTION_FINISH(ACI_OpInitialise);
}

/**
 * @brief   Process CAM removal from slot for CA support
 * @param   slot_id CI slot identifier
 */
void ACI_OpSlotRemove(U8BIT slot_id)
{
   S_OP_STATUS *state;
   S_OP_STATUS **pstate;

   FUNCTION_START(ACI_OpSlotRemove);
   DBG_OP("(slot_id=%u)", slot_id)
   pstate = &ciop_list;
   STB_OSMutexLock(ciop_mutex);
   state = ciop_list;
   while (state != NULL)
   {
      if (state->slot_id == slot_id)
      {
         ADB_MarkCIModuleNotPresent(state->module);
         *pstate = state->next;
         STB_FreeMemory(state);
         state = *pstate;
      }
      else
      {
         pstate = &(state->next);
         state = state->next;
      }
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpSlotRemove);
}

/**
 * @brief   Called by the host to request changing to an operator profile and
 *          request the operator status to be sent
 * @param   module - operator profile module
 * @return  TRUE if request succeeded, FALSE otherwise
 */
BOOLEAN ACI_RequestOperatorStatus(U32BIT module)
{
   BOOLEAN retval;
   S_OP_STATUS *op_state;

   FUNCTION_START(ACI_RequestOperatorStatus);

   DBG_OP("(%u)", module)

   if (op_module == INVALID_MODULE)
   {
      retval = STB_CIRequestOperatorStatus(module);
      if (retval)
      {
         /* This is the module that's in operator profile */
         op_module = module;
         STB_OSMutexLock(ciop_mutex);
         op_state = FindOpState(module);
         if (op_state != NULL)
         {
            DBG_OP("(%u): Setting character table 0x%x", module,
               op_state->op_info.character_code_table)
            STB_SetDefaultAsciiTable(GetCharacterCodeTable(&(op_state->op_info)));

            if (((op_state->op_info.iso_639_language_code[0] != 'u') ||
               (op_state->op_info.iso_639_language_code[1] != 'n') ||
               (op_state->op_info.iso_639_language_code[2] != 'd')) &&
               ((op_state->op_info.iso_639_language_code[0] != 'q') ||
               (op_state->op_info.iso_639_language_code[1] != 'a') ||
               (op_state->op_info.iso_639_language_code[2] != 'a')))
            {
               default_und_lang = ACFG_GetUndefinedLanguageBehaviour();
               ACFG_SetUndefinedLanguageBehaviour(
                  ConvertLangCode(op_state->op_info.iso_639_language_code));
            }
         }
         STB_OSMutexUnlock(ciop_mutex);
      }
      else
      {
         DBG_OP("STB_CIRequestOperatorStatus(%u) failed", module)
      }
   }
   else
   {
      /* Previous operator module hasn't been exited */
      DBG_OP("Previous operator module hasn't been exited")
      retval = FALSE;
   }

   FUNCTION_FINISH(ACI_RequestOperatorStatus);

   return(retval);
}

/**
 * @brief   Request the current operator module to exit operator profile
 * @return  TRUE if request succeeded, FALSE otherwise
 */
BOOLEAN ACI_OperatorExit(void)
{
   BOOLEAN retval;

   FUNCTION_START(ACI_OperatorExit);

   DBG_OP("")
   if (op_module != INVALID_MODULE)
   {
      retval = STB_CISendOperatorExit(op_module);
      if (!retval)
      {
         DBG_OP("STB_CISendOperatorExit(%u) failed", op_module)
      }
      op_module = INVALID_MODULE;

      DBG_OP("Setting character table 0x%x (default)", 0)
      STB_SetDefaultAsciiTable(0);

      ACFG_SetUndefinedLanguageBehaviour(default_und_lang);
   }
   else
   {
      DBG_OP("no profile to exit")
      retval = FALSE;
   }

   FUNCTION_FINISH(ACI_OperatorExit);

   return(retval);
}

/*!**************************************************************************
 * @brief   Called by the app to start an operator profile search that has been requested
 * @param   module - operator module
 * @return  TRUE if the search is started, FALSE otherwise
 ****************************************************************************/
BOOLEAN ACI_StartOperatorSearch(U32BIT module)
{
   BOOLEAN retval;
   S_OP_STATUS *op_state;

   FUNCTION_START(ACI_StartOperatorSearch);

   DBG_OP("(%u):", module)

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      retval = StartOperatorSearch(op_state);
   }
   else
   {
      retval = FALSE;
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_StartOperatorSearch);

   return(retval);
}

/**
 * @brief   Returns the Operator Profile module associated with a CICAM ID.
 * @param   cicam_id CICAM ID
 * @param   module Pointer to the Operator Profile module
 * @return  TRUE if the module could be found, FALSE otherwise
 */
BOOLEAN ACI_FindOperatorProfileModule(U32BIT cicam_id, U32BIT *module)
{
   S_OP_STATUS *op_state;
   BOOLEAN retval = FALSE;

   FUNCTION_START(ACI_FindOperatorProfileModule);

   STB_OSMutexLock(ciop_mutex);
   op_state = ciop_list;
   while (op_state != NULL)
   {
      if (op_state->op_info_valid &&
         op_state->op_info.cicam_identifier == cicam_id)
      {
         *module = op_state->module;
         retval = TRUE;
         break;
      }
      op_state = op_state->next;
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_FindOperatorProfileModule);

   return retval;
}

/**
 * @brief   This function is called by the CI+ stack to deliver the operator
 *          status to the host.
 *          This should be called if the CICAM supports the Operator Profile.
 *          This provides the module to be used for other Operator Profile
 *          related calls.
 * @param   module operator profile module
 * @param   status operator status
 */
void ACI_OpNotifyOperatorStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status)
{
   S_OP_STATUS *op_state;
   S_CIP_OPERATOR_SEARCH op_search;
   BOOLEAN valid;

   FUNCTION_START(ACI_OpNotifyOperatorStatus);

   DBG_OP("(%u): slot_id=%u", module, STB_GetCIOperatorProfileSlotId(module))
   DBG_OP("   info_version:       %u", status->info_version)
   DBG_OP("   nit_version:        %u", status->nit_version)
   DBG_OP("   profile_type:       %u", status->profile_type)
   DBG_OP("   initialised_flag:   %u", status->initialised_flag)
   DBG_OP("   entitlement_change: %u", status->entitlement_change_flag)
   DBG_OP("   entitlement_valid:  %u", status->entitlement_valid_flag)
   DBG_OP("   refresh_request:    %u", status->refresh_request_flag)
   DBG_OP("   del system hint:    0x%02x", status->delivery_system_hint)
   DBG_OP("   error:              %u", status->error_flag)

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state == NULL)
   {
      op_state = STB_GetMemory(sizeof(S_OP_STATUS));
      if (op_state != NULL)
      {
         memset(op_state,0,sizeof(S_OP_STATUS));
         op_state->module = module;
         op_state->slot_id = STB_GetCIOperatorProfileSlotId(module);
         op_state->next = ciop_list;
         ciop_list = op_state;
      }
   }
   if (op_state != NULL)
   {
      if (!op_state->op_status_valid || (status->initialised_flag == 0) ||
          (op_state->op_status.info_version != status->info_version))
      {
         if ((status->initialised_flag != 0) && op_state->op_status_valid &&
             (status->nit_version != op_state->op_status.nit_version))
         {
            /* An updated NIT is available so indicate that a new one needs to be requested */
            op_state->op_nit_received = FALSE;
         }

         /* Request the operator info because this is the first time the status has been received,
          * or the profile hasn't been initialised yet, or the info has changed */
         DBG_OP("(%u): Requesting operator info", module)
         STB_CIRequestOperatorInfo(module);
      }
      else
      {
         /* The CAM defines its own private network */
         if ((status->initialised_flag == 0) ||
            (status->refresh_request_flag != STB_CI_REFRESH_NOT_REQUIRED))
         {
            /* A search is required to setup the private network.
             * Check that the search can proceed */
            memset(&op_search, 0, sizeof(op_search));

            valid = TRUE;

            if ((status->initialised_flag == 0) ||
                (status->refresh_request_flag == STB_CI_REFRESH_REQUIRED_IMMEDIATE))
            {
               op_search.refresh_request = CIP_OPERATOR_SEARCH_IMMEDIATE;
            }
            else if (status->refresh_request_flag == STB_CI_REFRESH_REQUIRED_SCHEDULED)
            {
               if (status->refresh_request_date.mjd != 0)
               {
                  op_search.refresh_request = CIP_OPERATOR_SEARCH_SCHEDULED;
                  op_search.refresh_date = status->refresh_request_date.mjd;
                  op_search.refresh_hour = status->refresh_request_time.hour;
                  op_search.refresh_min = status->refresh_request_time.minute;
                  op_search.cicam_id = op_state->op_info.cicam_identifier;
               }
               else
               {
                  /* Date isn't valid so don't proceed with the request */
                  valid = FALSE;
               }
            }
            else
            {
               op_search.refresh_request = CIP_OPERATOR_SEARCH_DEFERRED;
            }

            if (valid)
            {
               if (RequestOperatorSearch(module,&op_search))
               {
                  /* The search can be started immediately */
                  StartOperatorSearch(op_state);
               }
            }
         }
         else if ((op_state->op_status_valid &&
                   (op_state->op_status.nit_version != status->nit_version)) ||
                  (status->entitlement_change_flag &&
                   (status->refresh_request_flag == STB_CI_REFRESH_NOT_REQUIRED)))
         {
            /* Profile initialisation is complete and a CICAM NIT is available */
            DBG_OP("(%u): Requesting operator NIT", module)
            op_state->op_nit_received = FALSE;
            STB_CIRequestOperatorNit(module);
         }
      }

      /* Save the status */
      memcpy(&op_state->op_status, status, sizeof(S_STB_CI_OPERATOR_STATUS));
      op_state->op_status_valid = TRUE;
   }
   else
   {
      DBG_OP("(%u): unrecognised operator module", module)
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpNotifyOperatorStatus);
}

/**
 * @brief   This function is called by the CI+ stack to deliver the operator
 *          information to the host.
 * @param   module operator profile module
 * @param   info_version operator information version
 * @param   info operator information or NULL (if not valid)
 */
void ACI_OpNotifyOperatorInfo(U32BIT module, U8BIT info_version,
   S_STB_CI_OPERATOR_INFO *info)
{
   S_OP_STATUS *op_state;
   S_CIP_OPERATOR_SEARCH op_search;
   BOOLEAN valid;

   FUNCTION_START(ACI_OpNotifyOperatorInfo);

   DBG_OP("(%u, version=%u)", module, info_version)
   if (info != NULL)
   {
      DBG_OP("   cicam_onet_id:    0x%04x", info->cicam_original_network_id)
      DBG_OP("   cicam_identifier: 0x%08x", info->cicam_identifier)
      DBG_OP("   char_code_table:  %u", info->character_code_table)
      DBG_OP("   encoding_type_id: 0x%02x", info->encoding_type_id)
      DBG_OP("   iso lang_code:    %c%c%c", info->iso_639_language_code[0], info->iso_639_language_code[1],
              info->iso_639_language_code[2])
      DBG_OP("   eit_present_following_usage: %u", info->eit_present_following_usage)
      DBG_OP("   eit_schedule_usage: %u", info->eit_schedule_usage)
      DBG_OP("   profile name len: %u", info->profile_name_length)
      if (info->profile_name != NULL)
      {
         DBG_OP("   profile_name:    \"%.*s\"", info->profile_name_length, info->profile_name)
      }
   }

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      if ((op_state->op_status.initialised_flag == 0) ||
           (op_state->op_status.refresh_request_flag != STB_CI_REFRESH_NOT_REQUIRED))
      {
         /* A scan is required to setup the private network, check if it can be started now */
         memset(&op_search, 0, sizeof(op_search));

         valid = TRUE;

         if ((op_state->op_status.initialised_flag == 0) ||
             (op_state->op_status.refresh_request_flag == STB_CI_REFRESH_REQUIRED_IMMEDIATE))
         {
            op_search.refresh_request = CIP_OPERATOR_SEARCH_IMMEDIATE;
         }
         else if (op_state->op_status.refresh_request_flag == STB_CI_REFRESH_REQUIRED_SCHEDULED)
         {
            if (op_state->op_status.refresh_request_date.mjd != 0)
            {
               op_search.refresh_request = CIP_OPERATOR_SEARCH_SCHEDULED;
               op_search.refresh_date = op_state->op_status.refresh_request_date.mjd;
               op_search.refresh_hour = op_state->op_status.refresh_request_time.hour;
               op_search.refresh_min = op_state->op_status.refresh_request_time.minute;
               op_search.cicam_id = info->cicam_identifier;
            }
            else
            {
               /* Date isn't valid so don't proceed with the request */
               valid = FALSE;
            }
         }
         else
         {
            op_search.refresh_request = CIP_OPERATOR_SEARCH_DEFERRED;
         }

         if (valid)
         {
            if (RequestOperatorSearch(module,&op_search))
            {
               /* OK to start the search */
               StartOperatorSearch(op_state);
            }
         }
      }

      if ((op_state->op_status.profile_type == 1) &&
         (op_state->op_status.refresh_request_flag != STB_CI_REFRESH_REQUIRED_IMMEDIATE) &&
         (op_state->op_status.initialised_flag == 1) && !op_state->op_nit_received)
      {
         /* If the CICAM is ready to report a profile, but we don't have the NIT yet, request it */
         DBG_OP("(%u): No NIT, or entitlement changed, requesting operator NIT", module)
         STB_CIRequestOperatorNit(module);
      }

      if ((op_module != INVALID_MODULE) && op_state->op_info_valid &&
         (op_state->op_info.character_code_table != info->character_code_table))
      {
         /* The character code table for the active profile has changed */
         DBG_OP("(%u): Setting character table 0x%x", module, info->character_code_table)
         STB_SetDefaultAsciiTable(GetCharacterCodeTable(info));
      }

      /* Free any existing profile name */
      if (op_state->op_info_valid && (op_state->op_info.profile_name != NULL))
      {
         STB_FreeMemory(op_state->op_info.profile_name);
         op_state->op_info.profile_name = NULL;
         op_state->op_info.profile_name_length = 0;
      }

      if (info != NULL)
      {
         /* Save the operator info, including the profile name */
         memcpy(&op_state->op_info, info, sizeof(S_STB_CI_OPERATOR_INFO));

         if (info->profile_name != NULL)
         {
            if (info->profile_name_length > 0)
            {
               if ((op_state->op_info.profile_name = STB_GetMemory(info->profile_name_length)) != NULL)
               {
                  memcpy(op_state->op_info.profile_name, info->profile_name, info->profile_name_length);
               }
               else
               {
                  op_state->op_info.profile_name_length = 0;
               }
            }
            else
            {
               op_state->op_info.profile_name = NULL;
            }
         }

         if (!op_state->op_info_valid)
         {
            /* This should be the first time info has been received, so inform the DVB that
             * this module is now available */
            DBG_OP("(%u): Inform DVB that CAM is available for profile onet=0x%04x, cam_id=0x%08x",
                 module, info->cicam_original_network_id, info->cicam_identifier)

            ADB_MarkCIModulePresent(info->cicam_original_network_id, info->cicam_identifier);
         }

         op_state->op_info_valid = TRUE;
      }
      else
      {
         op_state->op_info_valid = FALSE;
      }

      /* Update the version of the info now held */
      op_state->op_status.info_version = info_version;

   }
   else
   {
      DBG_OP("(%u): unrecognised operator module", module)
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpNotifyOperatorInfo);
}

/**
 * @brief   This function is called by the CI+ stack to request that the host
 *          performs the tune request as described in the provided descriptors.
 *          The descriptor loop contains system delivery descriptors which the
 *          host must try in order until either an error condition is raised,
 *          the tune operation is successful or the descriptor loop is exhausted.
 *          The operation of the host should follow the pseudo code in section 14.7.5.11.
 *          When the tune attempt is done, the host must call the function
 *          ACI_OpSendOperatorTuneStatus with the results of the tune attempt.
 *          This function may be called during a profile search sequence.
 * @param   module operator profile module
 * @param   desc_loop_len length of descriptor loop in bytes
 * @param   desc_loop the descriptor loop
 */
void ACI_OpNotifyOperatorTune(U32BIT module, U16BIT desc_loop_len, U8BIT *desc_loop)
{
   S_OP_STATUS *op_state;

   FUNCTION_START(ACI_OpNotifyOperatorTune);

   DBG_OP("(%u, desc_loop_len=%u)", module, desc_loop_len)
   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      if (op_state->op_search_started)
      {
         /* Tuning operations need to be performed until all descriptors have been exhausted
          * or until a tuning operation is sucessful, so copy the descriptor data so it can
          * be processed as needed */
         if ((op_state->op_desc_loop = (U8BIT *)STB_GetMemory(desc_loop_len)) != NULL)
         {
            memcpy(op_state->op_desc_loop, desc_loop, desc_loop_len);
            op_state->op_desc_loop_len = desc_loop_len;
            op_state->op_desc_num = 0;
            /* Now process the descriptor loop to perform tune operations */
            PerformOperatorTune(op_state);
         }
      }
      else
      {
         DBG_OP("(%u): Received operator tune request but search hasn't been started", module)
      }
   }
   else
   {
      DBG_OP("(%u): unrecognised operator module", module)
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpNotifyOperatorTune);
}

/**
 * @brief   This function is called by the CI+ stack to deliver the operator search
 *          status to the host.
 *          This function will be called at the end of a profile search sequence.
 * @param   module operator profile module
 * @param   status operator status
 */
void ACI_OpNotifyOperatorSearchStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status)
{
   S_OP_STATUS *op_state;

   FUNCTION_START(ACI_OpNotifyOperatorSearchStatus);

   DBG_OP("(%u)", module)
   DBG_OP("   info_version:       %u", status->info_version)
   DBG_OP("   nit_version:        %u", status->nit_version)
   DBG_OP("   profile_type:       %u", status->profile_type)
   DBG_OP("   initialised_flag:   %u", status->initialised_flag)
   DBG_OP("   entitlement_change: %u", status->entitlement_change_flag)
   DBG_OP("   entitlement_valid:  %u", status->entitlement_valid_flag)
   DBG_OP("   refresh_request:    %u", status->refresh_request_flag)
   DBG_OP("   del system hint:    0x%02x", status->delivery_system_hint)
   DBG_OP("   error:              %u", status->error_flag)

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      /* Operator search has completed */
      //op_state->tuner_status = CI_TUNER_UNUSED;

      if (status->error_flag != STB_CI_OPERATOR_CANCELLED)
      {
         /* Any current NIT is no longer valid */
         op_state->op_nit_received = FALSE;
      }

      /* Operator profile is only valid if there wasn't an error during the search */
      if (status->error_flag == STB_CI_OPERATOR_NO_ERROR)
      {
         if (!op_state->op_info_valid || (op_state->op_status.info_version != status->info_version))
         {
            /* Need to request updated info */
            DBG_OP("(%u): Requesting operator info", module)
            STB_CIRequestOperatorInfo(module);
         }
         else
         {
            /* Info hasn't changed, so just request the CICAM NIT */
            DBG_OP("(%u): Requesting operator NIT", module)
            STB_CIRequestOperatorNit(module);
         }
      }

      /* Save the updated status */
      memcpy(&op_state->op_status, status, sizeof(S_STB_CI_OPERATOR_STATUS));
      op_state->op_status_valid = TRUE;

      /* Send an event to the app to indicate that the search has finished */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_OPERATOR_SEARCH_FINISHED, NULL, 0);
   }
   else
   {
      DBG_OP("(%u): unrecognised operator module", module)
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpNotifyOperatorSearchStatus);
}

/**
 * @brief   This function is called by the CI+ stack to deliver the operator NIT
 *          to the host.
 *          The CICAM delivers zero or more NIT sections to the host. Each section
 *          is a complete NIT section including the CRC-32 field, and the sections
 *          are provided without any padding between them.
 *          The NIT section data must be copied if it used after this function returns
 * @param   module - operator profile module
 * @param   nit_loop_length - length of NIT loop in bytes (may be 0)
 * @param   nit_sections - NIT section(s), if any
 */
void ACI_OpNotifyOperatorNit(U32BIT module, U16BIT nit_loop_length,
   U8BIT *nit_sections)
{
   S_OP_STATUS *op_state;
   SI_TABLE_RECORD table_rec;
   SI_SECTION_RECORD *sect, *s;
   SI_NIT_TABLE *nit_table;
   S_CIP_OPERATOR_NIT nit;
   U16BIT l, section_size;
   U8BIT *ptr;

   FUNCTION_START(ACI_OpNotifyOperatorNit);

   DBG_OP("(%u, nit_size=%u)", module, nit_loop_length)

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      if (nit_loop_length != 0)
      {
         l = 0;
         sect = NULL;
         memset(&table_rec, 0, sizeof(SI_TABLE_RECORD));
         while (l < nit_loop_length)
         {
            ptr = nit_sections + l;
            section_size = ((ptr[1] & 0xf) << 8) + ptr[2] + 3;
            l += section_size;
            s = (SI_SECTION_RECORD *)STB_GetMemory(sizeof(SI_SECTION_RECORD) + section_size);
            if (s != NULL)
            {
               s->next = sect;
               sect = s;
               sect->sect_num = ptr[6];
               sect->data_len = section_size;
               memcpy(&sect->data_start, ptr, section_size);
               table_rec.num_sect++;
            }
         }

         /* Populate a table_rec so the NIT can be parsed using existing functions */
         if (sect != NULL)
         {
            table_rec.path = INVALID_RES_ID;
            table_rec.tid = nit_sections[0];
            table_rec.version = (nit_sections[5] >> 1) & 0x1f;
            table_rec.xtid = (nit_sections[3] << 8) + nit_sections[4];
            table_rec.section_list = sect;

            if ((nit_table = STB_SIParseNitTable(&table_rec)) != NULL)
            {
               /* Pass the parsed table to be processed */
               nit.table = nit_table;
               nit.cicam_onet_id = op_state->op_info.cicam_original_network_id;
               nit.cicam_identifier = op_state->op_info.cicam_identifier;
               nit.lang_code = ConvertLangCode(op_state->op_info.iso_639_language_code);
               nit.profile_name_length = op_state->op_info.profile_name_length;
               nit.profile_name = op_state->op_info.profile_name;

               ProcessNit(module,&nit);

               if (op_state->op_status.entitlement_change_flag)
               {
                  /* Acknowledge that the change has been handled */
                  DBG_OP("(%u): Acknowledging entitlement change", module)
                  STB_CISendEntitlementAck(module);
               }
            }

            /* Can now free the copied section data */
            while (sect != NULL)
            {
               s = sect->next;
               STB_FreeMemory(sect);
               sect = s;
            }
         }
         else
         {
            DBG_OP("(%u): Failed to allocate memory to parse the NIT", module)
         }
      }
   }
   else
   {
      DBG_OP("(%u): unrecognised operator module", module)
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpNotifyOperatorNit);
}

/**
 * @brief   Ask the opertaor module to restore replaced PIDs
 * @param   module specifies module required
 * @return  TRUE if operation successful, FALSE otherwise
 */
BOOLEAN ACI_OpAskRelease(U32BIT module)
{
   S_OP_STATUS *op_state;
   BOOLEAN retval;
   U8BIT path;
   S_CIP_RELEASE_REPLY reply_data;

   FUNCTION_START(ACI_OpAskRelease);

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   STB_OSMutexUnlock(ciop_mutex);
   if (op_state == NULL)
   {
      retval = FALSE;
   }
   else
   {
      retval = TRUE;
      DBG_OP("(%u)", module)
      /* The CICAM does not need to be asked in this case */
      for (path = 0; path != STB_DPGetNumPaths(); path++)
      {
         if (ACI_PathOwnedByModule(path, module))
         {
            reply_data.path = path;
            reply_data.module = module;
            reply_data.reply = 0;
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_RELEASE_REPLY, &reply_data, sizeof(reply_data));
         }
      }
   }

   FUNCTION_FINISH(ACI_OpAskRelease);

   return(retval);
}

/**
 * @brief   This function is called by the host to send the status of the tune
 *          operation to the module.
 * @param   path - decode path used for tuning
 * @param   module - host control module
 * @param   status - tune operation status
 * @return  TRUE if operation succeeded, FALSE otherwise
 */
BOOLEAN ACI_OpTuneReply(U8BIT path, U32BIT module, E_STB_CI_TUNE_STATUS status)
{
   S_OP_STATUS *op_state;
   S_STB_CI_OPERATOR_TUNE_STATUS op_status;
   U8BIT tuner;
   BOOLEAN retval;

   FUNCTION_START(ACI_OpTuneReply);

   DBG_OP("(%u, status=%u)", module, status)
   retval = FALSE;

   STB_OSMutexLock(ciop_mutex);
   op_state = FindOpState(module);
   if (op_state != NULL)
   {
      if (op_state->tune_started)
      {
         op_state->tune_started = FALSE;

         if (status == STB_CI_TUNE_OK)
         {
            /* Tuned successfully, pass the result to the CI+ stack */
            op_status.descriptor_number = op_state->op_desc_num;
            op_status.status = STB_CI_OPERATOR_TUNE_OK;
            op_status.desc_loop = op_state->op_current_location;
            op_status.desc_loop_len = op_state->op_current_loc_len;

            if (path != INVALID_RES_ID)
            {
               if ((tuner = STB_DPGetPathTuner(path)) != INVALID_RES_ID)
               {
                  op_status.signal_strength = STB_TuneGetSignalStrength(tuner);
                  op_status.signal_quality = STB_TuneGetDataIntegrity(tuner);
               }
               else
               {
                  DBG_OP("Path %u doesn't have a tuner!", path)
                  op_status.signal_strength = 0;
                  op_status.signal_quality = 0;
               }
            }
            else
            {
               DBG_OP("Tuner locked but invalid decode path")
               op_status.signal_strength = 0;
               op_status.signal_quality = 0;
            }

            retval = STB_CISendOperatorTuneStatus(op_state->module, &op_status);

            if (op_state->op_current_location != NULL)
            {
               STB_FreeMemory(op_state->op_current_location);
               op_state->op_current_location = NULL;
               op_state->op_current_loc_len = 0;
            }
         }
         else
         {
            /* Tuning didn't succeed, so continue processing any remaining delivery descriptors */
            PerformOperatorTune(op_state);
         }
      }
      else
      {
         DBG_OP(" The tune was not started by module %u", module)
      }
   }
   STB_OSMutexUnlock(ciop_mutex);

   FUNCTION_FINISH(ACI_OpTuneReply);

   return(retval);
}

//----------------------------------------------------------------------------
//--- Local function for this file ------------------------------------------
//----------------------------------------------------------------------------

static S_OP_STATUS *FindOpState(U32BIT module)
{
   S_OP_STATUS *op_state;
   op_state = ciop_list;
   while (op_state != NULL && module != op_state->module)
   {
      op_state = op_state->next;
   }
   return op_state;
}

/*!**************************************************************************
 * @brief   Starts and operator search
 * @param   op_state - slot state info
 ****************************************************************************/
static BOOLEAN StartOperatorSearch(S_OP_STATUS *op_state)
{
   BOOLEAN retval;
   S_CIP_HOST_INFO host_info;
   S_STB_CI_SEARCH_START_INFO search_info;

   FUNCTION_START(StartOperatorSearch);

   retval = FALSE;

   /* A scan is required to setup the private network. Get scan info from the host */
   RequestHostInfo(&host_info);

   if (PrepareSearchInfo(&host_info, &search_info))
   {
      /* Save the delivery types supported by the host so they can be checked when tuning */
      op_state->host_delivery_types = host_info.delivery_types;

      if (STB_CIRequestOperatorSearchStart(op_state->module, &search_info))
      {
         DBG_OP("Starting operator search to setup network")

         /* Tuner is now under control of operator profile until the search completes */
         //op_state->tuner_status = CI_TUNER_OP;

         op_state->op_nit_received = FALSE;
         op_state->op_search_started = TRUE;
         op_state->op_desc_loop_len = 0;
         op_state->op_desc_loop = NULL;

         retval = TRUE;
      }
      else
      {
         DBG_OP("Failed to start operator search")
      }

      if (search_info.service_type != NULL)
      {
         STB_FreeMemory(search_info.service_type);
      }
      if (search_info.delivery_capability != NULL)
      {
         STB_FreeMemory(search_info.delivery_capability);
      }
      if (search_info.application_capability != NULL)
      {
         STB_FreeMemory(search_info.application_capability);
      }
   }
   else
   {
      DBG_OP("PrepareSearchInfo failed")
   }

   FUNCTION_FINISH(StartOperatorSearch);

   return(retval);
}

/*!**************************************************************************
 * @brief   Sets up the operator start search info to be passed to the CI+ stack
 *          from info provided by the host.
 * @param   host_info - host info
 * @param   search_info - structure to be setup before being passed to the CI+ stack.
 *                        Memory will be allocated for the arrays in this structure
 * @return  TRUE if the structure is setup successfully, FALSE otherwise
 ****************************************************************************/
static BOOLEAN PrepareSearchInfo(S_CIP_HOST_INFO *host_info, S_STB_CI_SEARCH_START_INFO *search_info)
{
   U16BIT counter;
   BOOLEAN retval;

   FUNCTION_START(PrepareSearchInfo);

   retval = TRUE;

   search_info->unattended_flag = host_info->standby;

   /* Setup the supported service types */
   if ((search_info->service_type = STB_GetMemory(host_info->num_service_types)) != NULL)
   {
      search_info->service_type_loop_length = host_info->num_service_types;

      counter = 0;
      if ((host_info->service_types & CIP_SERVICE_TYPE_TV) != 0)
      {
         search_info->service_type[counter] = 0x01;   /* MPEG2 TV */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_RADIO) != 0)
      {
         search_info->service_type[counter] = 0x02;      /* MPEG1 Layer-II radio */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_TELETEXT) != 0)
      {
         search_info->service_type[counter] = 0x03;      /* Teletext */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_AVC_RADIO) != 0)
      {
         search_info->service_type[counter] = 0x0a;      /* AVC radio */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_DATA) != 0)
      {
         search_info->service_type[counter] = 0x0c;      /* Data */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_HD_TV) != 0)
      {
         search_info->service_type[counter] = 0x11;      /* MPEG2 HD TV */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_AVC_SD_TV) != 0)
      {
         search_info->service_type[counter] = 0x16;      /* AVC SD TV */
         counter++;
      }
      if ((host_info->service_types & CIP_SERVICE_TYPE_AVC_HD_TV) != 0)
      {
         search_info->service_type[counter] = 0x19;      /* AVC HD TV */
      }
   }
   else
   {
      search_info->service_type_loop_length = 0;
      retval = FALSE;
   }

   /* Setup the supported delivery capabilities */
   counter = 0;
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBT) != 0)
   {
      counter += 1;
   }
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBT2) != 0)
   {
      counter += 2;
   }
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBC) != 0)
   {
      counter += 1;
   }
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBC2) != 0)
   {
      counter += 2;
   }
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBS) != 0)
   {
      counter += 1;
   }
   if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBS2) != 0)
   {
      counter += 1;
   }

   if ((search_info->delivery_capability = STB_GetMemory(counter)) != NULL)
   {
      search_info->delivery_capability_loop_length = counter;

      counter = 0;
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBT) != 0)
      {
         search_info->delivery_capability[counter] = 0x5a;
         counter += 1;
      }
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBT2) != 0)
      {
         search_info->delivery_capability[counter] = 0x7f;
         search_info->delivery_capability[counter + 1] = 0x04;
         counter += 2;
      }
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBC) != 0)
      {
         search_info->delivery_capability[counter] = 0x44;
         counter += 1;
      }
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBC2) != 0)
      {
         search_info->delivery_capability[counter] = 0x7f;
         search_info->delivery_capability[counter + 1] = 0x0d;
         counter += 2;
      }
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBS) != 0)
      {
         search_info->delivery_capability[counter] = 0x43;
         counter += 1;
      }
      if ((host_info->delivery_types & CIP_DELIVERY_TYPE_DVBS2) != 0)
      {
         search_info->delivery_capability[counter] = 0x79;
      }
   }
   else
   {
      search_info->delivery_capability_loop_length = 0;
      retval = FALSE;
   }

   /* Setup the supported application capabilities */
   counter = 0;
   if ((host_info->app_types & CIP_APP_TYPE_MHEG) != 0)
   {
      counter += 2;
   }
   if ((host_info->app_types & CIP_APP_TYPE_HBBTV) != 0)
   {
      counter += 2;
   }
   if ((host_info->app_types & CIP_APP_TYPE_OIPF) != 0)
   {
      counter += 2;
   }

   if (counter > 0)
   {
      if ((search_info->application_capability = STB_GetMemory(counter)) != NULL)
      {
         search_info->application_capability_loop_length = counter;

         counter = 0;
         if ((host_info->app_types & CIP_APP_TYPE_MHEG) != 0)
         {
            search_info->application_capability[counter] = 0x01;
            search_info->application_capability[counter + 1] = 0x06;
            counter += 2;
         }
         if ((host_info->app_types & CIP_APP_TYPE_HBBTV) != 0)
         {
            search_info->application_capability[counter] = 0x01;
            search_info->application_capability[counter + 1] = 0x23;
            counter += 2;
         }
         if ((host_info->app_types & CIP_APP_TYPE_OIPF) != 0)
         {
            search_info->application_capability[counter] = 0x01;
            search_info->application_capability[counter + 1] = 0x50;
         }
      }
      else
      {
         search_info->application_capability_loop_length = 0;
         retval = FALSE;
      }
   }
   else
   {
      search_info->application_capability_loop_length = 0;
      search_info->application_capability = NULL;
   }

   FUNCTION_FINISH(PrepareSearchInfo);

   return(retval);
}

static BOOLEAN ValidateFrequency(SI_DELIVERY_SYS_DESC_TYPE del_type, U32BIT frequency)
{
   BOOLEAN valid, done;
   U8BIT tuner, n;
   E_STB_TUNE_SYSTEM_TYPE type;
   U32BIT tnr_min_freq;
   U32BIT tnr_max_freq;

   FUNCTION_START(ValidateFrequency);

   valid = FALSE;
   done = FALSE;
   /* Find the first tuner supporting the type */
   n = STB_HWGetTunerPaths();
   for (tuner = 0; tuner < n; tuner++)
   {
      type = STB_TuneGetSupportedSystemType(tuner);
      switch (type)
      {
         case TUNE_SYSTEM_TYPE_DVBT:
         case TUNE_SYSTEM_TYPE_DVBT2:
         {
            if (del_type == SI_DEL_SYS_DESC_TYPE_TERR)
            {
               done = TRUE;
            }
            break;
         }
         case TUNE_SYSTEM_TYPE_DVBS:
         case TUNE_SYSTEM_TYPE_DVBS2:
         {
            if (del_type == SI_DEL_SYS_DESC_TYPE_SAT)
            {
               done = TRUE;
            }
            break;
         }
         case TUNE_SYSTEM_TYPE_DVBC:
         {
            if (del_type == SI_DEL_SYS_DESC_TYPE_CABLE)
            {
               done = TRUE;
            }
            break;
         }
         default:
         {
            break;
         }
      }
      if (done)
      {
         break;
      }
   }
   if (done)
   {
      tnr_min_freq = STB_TuneGetMinTunerFreqKHz(tuner);
      tnr_max_freq = STB_TuneGetMaxTunerFreqKHz(tuner);
      DBG_OP("min=%u, max=%u", tnr_min_freq, tnr_max_freq)
      if ((frequency > tnr_min_freq) && (frequency < tnr_max_freq))
      {
         valid = TRUE;
      }
   }

   DBG_OP("frequency=%u del_type=%d valid=%s", frequency, del_type,
      valid ? "TRUE" : "FALSE")

   FUNCTION_FINISH(ValidateFrequency);

   return valid;
}

/**
 * @brief   Processes the operator delivery system descriptors, passing status back to
 *          the CI+ stack, or starts the next tuning operation.
 * @param   op_state - slot state info
 */
static void PerformOperatorTune(S_OP_STATUS *op_state)
{
   U8BIT *end_ptr;
   U8BIT *desc_ptr;
   U8BIT desc_num;
   U8BIT desc_len;
   U8BIT *current_location_info;
   U8BIT current_location_len;
   S_STB_CI_OPERATOR_TUNE_STATUS tune_status;
   SI_DELIVERY_SYS_DESC_TYPE del_type;
   SI_DELIVERY_SYS_DESC *desc;
   BOOLEAN send_status;
   S_CIP_TUNE_DEL_SYS_DESC tune_del_sys;
   U32BIT frequency;

   FUNCTION_START(PerformOperatorTune);

   if (op_state->op_current_location != NULL)
   {
      STB_FreeMemory(op_state->op_current_location);
      op_state->op_current_location = NULL;
      op_state->op_current_loc_len = 0;
   }

   desc_ptr = op_state->op_desc_loop;
   end_ptr = desc_ptr + op_state->op_desc_loop_len;

   /* Find the start of the next descriptor to be used */
   for (desc_num = 0; (desc_num < op_state->op_desc_num) && (desc_ptr < end_ptr); )
   {
      desc_len = *(desc_ptr + 1);
      desc_ptr += (desc_len + 2);
      /* s2_satellite_delivery_descriptor should always be present together a
         satellite_delivery_descriptor and have to be dealt with it */
      if (*desc_ptr != 0x79)
      {
         desc_num++;
      }
   }

   if (desc_ptr < end_ptr)
   {
      send_status = FALSE;

      /* Iterate through the descriptors */
      while (desc_ptr < end_ptr)
      {
         desc_len = *(desc_ptr + 1);

         if (*desc_ptr == 0x43)
         {
            /* A satellite_delivery_descriptor may be followed by an s2_satellite_delivery_descriptor,
               DVBCore ignores the latter (for now), but the reply to the CICAM must include both */
            if ((desc_ptr + desc_len + 2 < end_ptr) && (*(desc_ptr + desc_len + 2) == 0x79))
            {
               desc_len += *(desc_ptr + desc_len + 3) + 2;
            }
         }

         /* Save this descriptor for passing back to the CI+ stack */
         if ((current_location_info = (U8BIT *)STB_GetMemory(desc_len + 2)) != NULL)
         {
            memcpy(current_location_info, desc_ptr, desc_len + 2);
            current_location_len = desc_len + 2;

            /* Find or add a transport for this delivery descriptor */

            /* Find any descriptors, immediately following the current one,
             * that would result in tuning to the same TS */
            desc_ptr += (desc_len + 2);

            if (desc_ptr == end_ptr)
            {
               /* This is the last descriptor, the descriptor_number in the reply must be 0xFF */
               op_state->op_desc_num = 0xFF;
            }
            else
            {
               /* This is not the last descriptor, the descriptor_number in the reply must be the
                  next unprocessed descriptor number */
               op_state->op_desc_num++;
            }

            /* Check whether the descriptor to be used for tuning can be used and is valid */
            desc = NULL;
            if (STB_SIParseDelSysDesc(current_location_info, &del_type, &desc))
            {
               /* Check that the delivery type is supported */
               switch (del_type)
               {
                  case SI_DEL_SYS_DESC_TYPE_TERR:
                     if (desc->terr.is_t2)
                     {
                        if ((op_state->host_delivery_types & CIP_DELIVERY_TYPE_DVBT2) == 0)
                        {
                           /* DVB-T2 isn't supported */
                           tune_status.status = STB_CI_OPERATOR_TUNE_NOT_SUPPORTED;
                           send_status = TRUE;
                        }
                        else
                        {
                           if ((desc->terr.u.t2.num_cells > 0) &&
                              (desc->terr.u.t2.cell[0].num_freqs > 0))
                           {
                              frequency = desc->terr.u.t2.cell[0].freq_hz[0] / 1000; // Convert to kHz
                           }
                           else
                           {
                              frequency = 0;
                           }
                        }
                     }
                     else
                     {
                        if ((op_state->host_delivery_types & CIP_DELIVERY_TYPE_DVBT) == 0)
                        {
                           /* DVB-T isn't supported */
                           tune_status.status = STB_CI_OPERATOR_TUNE_NOT_SUPPORTED;
                           send_status = TRUE;
                        }
                        else
                        {
                           frequency = desc->terr.u.t1.freq_hz / 1000; // Convert to kHz
                        }
                     }
                     break;

                  case SI_DEL_SYS_DESC_TYPE_CABLE:
                     if ((op_state->host_delivery_types & CIP_DELIVERY_TYPE_DVBC) == 0)
                     {
                        /* DVB-C isn't supported */
                        tune_status.status = STB_CI_OPERATOR_TUNE_NOT_SUPPORTED;
                        send_status = TRUE;
                     }
                     else
                     {
                        frequency = desc->cable.freq_hz / 1000; // Convert to kHz
                     }
                     break;

                  case SI_DEL_SYS_DESC_TYPE_SAT:
                     if ((op_state->host_delivery_types & CIP_DELIVERY_TYPE_DVBS) == 0)
                     {
                        /* DVB-S isn't supported */
                        tune_status.status = STB_CI_OPERATOR_TUNE_NOT_SUPPORTED;
                        send_status = TRUE;
                     }
                     else
                     {
                        /* freq_hz seems to contain MHz */
                        frequency = desc->sat.freq_hz * 1000; // Convert to kHz
                     }
                     break;
               }
               if (!send_status)
               {
                  /* Signal type is supported, now check that the frequency is valid */
                  send_status = !ValidateFrequency(del_type, frequency);
                  if (send_status)
                  {
                     tune_status.status = STB_CI_OPERATOR_TUNE_INVALID;
                  }
               }

               /* Status will be sent if an error has been detected */
               if (send_status)
               {
                  STB_SIReleaseDelSysDesc(desc, del_type);
                  break;
               }

               /* Try to tune to the TS defined by the current location descriptor.
                * It's the responsibility of the callback function to free the delivery descriptor */
               op_state->tune_started = TRUE;

               op_state->op_current_location = current_location_info;
               op_state->op_current_loc_len = current_location_len;

               memset(&tune_del_sys, 0, sizeof(tune_del_sys));
               tune_del_sys.type = del_type;
               tune_del_sys.desc = desc;
               ACI_TuneToDelSysDesc(op_state->module, &tune_del_sys, CIP_NORMAL_TUNE);
               break;
            }
            else
            {
               /* Delivery descriptor is invalid */
               tune_status.status = STB_CI_OPERATOR_TUNE_INVALID;
               send_status = TRUE;
               break;
            }
         }
      }

      if (send_status)
      {
         tune_status.descriptor_number = op_state->op_desc_num;
         tune_status.signal_strength = 0;
         tune_status.signal_quality = 0;
         tune_status.desc_loop = current_location_info;
         tune_status.desc_loop_len = current_location_len;

         STB_CISendOperatorTuneStatus(op_state->module, &tune_status);

         /* Now free the descriptor data sent to the CI+ stack */
         STB_FreeMemory(current_location_info);
      }
   }
   else
   {
      /* No more descriptors to tune to. The tuning operation has failed */
      tune_status.descriptor_number = 0xff;
      tune_status.signal_strength = 0;
      tune_status.signal_quality = 0;
      tune_status.status = STB_CI_OPERATOR_TUNE_NO_SIGNAL;
      tune_status.desc_loop_len = 0;
      tune_status.desc_loop = NULL;

      STB_CISendOperatorTuneStatus(op_state->module, &tune_status);

      /* Free any saved data */
      if (op_state->op_desc_loop != NULL)
      {
         STB_FreeMemory(op_state->op_desc_loop);
         op_state->op_desc_loop = NULL;
         op_state->op_desc_loop_len = 0;
      }
   }

   FUNCTION_FINISH(PerformOperatorTune);
}

static U32BIT ConvertLangCode(U8BIT *data)
{
   U32BIT lcode;
   U8BIT code_char;
   U16BIT i;

   FUNCTION_START(ConvertLangCode);

   lcode = 0;
   for (i = 0; i < 3; i++)
   {
      code_char = *data;
      data++;
      if ((code_char >= 'A') && (code_char <= 'Z'))
      {
         code_char += 0x20; // convert to lower case
      }
      lcode = (lcode << 8) | code_char;
   }

   FUNCTION_FINISH(ConvertLangCode);

   return(lcode);
}

/**
 * @brief   Returns the ASCII table to pass to STB_SetDefaultAsciiTable based on the values
 *          contained in the Operator Info. character_code_table = 0x1f Currently not supported.
 *          As stbuni.c does not define a function that does this, this function is implemented
 *          copying and pasting pieces of code in STB_ConvertStringToUnicode.
 * @param   op_info Operator Info structure
 * @return  Index to pass to STB_SetDefaultAsciiTable
 */
static U8BIT GetCharacterCodeTable(S_STB_CI_OPERATOR_INFO *op_info)
{
   U8BIT table;

   FUNCTION_START(GetCharacterCodeTable);

   /* Default */
   table = 0;

   if ((op_info->character_code_table >= 1) && (op_info->character_code_table <= 11))
   {
      // one of the Latin tables specified in the DVB SI specification from ISO 8859.
      // 1 => 8859-5, 2 => 8859-6, 3 => 8859-7, 4 => 8859-8, 5 => 8859-9
      // 6 => 8859-10, 7 => 8859-11, 8 => 8859-12, 9 => 8859-13, 10 => 8859-14, 11 => 8859-15
      table = op_info->character_code_table + 4;
   }
   else if (op_info->character_code_table == 0x10)
   {
      // next 2 bytes indicate the table from ISO 8859
      table = (op_info->second_byte_value << 8) | (op_info->third_byte_value);
   }

   FUNCTION_FINISH(GetCharacterCodeTable);

   return table;
}

/**
 * @brief   Adds a new linkage descriptor entry into the destination linked list
 * @param   source_desc_entry Pointer to the new linkage descriptor entry to be added into the
 *          destination linked list
 * @param   destination_list_head Pointer to the destination linked list head pointer
 * @param   destination_last_entry Pointer to the last entry pointer in the destination linked list
 * @param   destination_num_linkage_desc Pointer to the number of entries in the destination linked
 *          list
 */
static void CopyLinkageDesc(SI_LINKAGE_DESC_ENTRY *source_desc_entry,
   SI_LINKAGE_DESC_ENTRY **destination_list_head, SI_LINKAGE_DESC_ENTRY **destination_last_entry,
   U16BIT *destination_num_linkage_desc)
{
   SI_LINKAGE_DESC_ENTRY *desc_ptr;

   FUNCTION_START(CopyLinkageDesc);

   desc_ptr = (SI_LINKAGE_DESC_ENTRY *)STB_GetMemory(sizeof(SI_LINKAGE_DESC_ENTRY) + source_desc_entry->data_length);
   if (desc_ptr != NULL)
   {
      desc_ptr->next = NULL;
      desc_ptr->tran_id = source_desc_entry->tran_id;
      desc_ptr->orig_net_id = source_desc_entry->orig_net_id;
      desc_ptr->serv_id = source_desc_entry->serv_id;
      desc_ptr->link_type = source_desc_entry->link_type;
      desc_ptr->data_length = source_desc_entry->data_length;

      if (source_desc_entry->data_length != 0)
      {
         memcpy(&(desc_ptr->data), &(source_desc_entry->data), source_desc_entry->data_length);
      }

      // now add descriptor to linked list
      if (*destination_last_entry == NULL)
      {
         // first entry in the list
         *destination_list_head = desc_ptr;
      }
      else
      {
         // not the first entry
         (*destination_last_entry)->next = desc_ptr;
      }
      *destination_last_entry = desc_ptr;
      (*destination_num_linkage_desc)++;

      DBG_OP("Linkage desc: tid=0x%04x, onid=0x%04x, sid=0x%04x, type=%d, dlen=%d",
                   desc_ptr->tran_id, desc_ptr->orig_net_id, desc_ptr->serv_id,
                   desc_ptr->link_type, desc_ptr->data_length)
   }

   FUNCTION_FINISH(CopyLinkageDesc);
}

/**
 * @brief   Deletes a linkage descriptor linked list
 * @param   list_ptr First entry in the linked list
 */
static void DeleteLinkageDescriptorArray(SI_LINKAGE_DESC_ENTRY *list_ptr)
{
   SI_LINKAGE_DESC_ENTRY *desc_ptr, *tmp_ptr;

   FUNCTION_START(DeleteLinkageDescriptorArray);

   desc_ptr = list_ptr;
   while (desc_ptr != NULL)
   {
      tmp_ptr = desc_ptr->next;
      STB_FreeMemory(desc_ptr);
      desc_ptr = tmp_ptr;
   }

   FUNCTION_FINISH(DeleteLinkageDescriptorArray);
}

/**
 * @brief   Search for service with allocated LCN
 * @param   lcn to search for
 */
static ADB_SERVICE_REC *GetServiceFromLcn(U16BIT lcn)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(GetServiceFromLcn);
   s_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s_ptr != NULL)
   {
      if (s_ptr->allocated_lcn == lcn)
      {
         break;
      }
      s_ptr = DBDEF_GetNextServiceRec(s_ptr);
   }
   FUNCTION_FINISH(GetServiceFromLcn);

   return s_ptr;
}

/**
 * @brief   Search for free (unallocated) LCN's - one above, one below starting lcn (if poss)
 * @param   start_lcn starting value for search
 * @param   free_lcns array of free lcns
 * @return  number of free lcns found
 */
static U8BIT GetFreeLcns(U16BIT start_lcn, U16BIT *free_lcns)
{
   ADB_SERVICE_REC *s_ptr;
   U16BIT num_services, nx, lcn;
   U16BIT *existing_lcns;
   U8BIT num_found;

   FUNCTION_START(GetFreeLcns);
   if (start_lcn == 0)
   {
      start_lcn = 1;
   }
   num_found = 0;
   num_services = DBDEF_GetNumServices();
   if (num_services != 0 && free_lcns != NULL)
   {
      existing_lcns = STB_AppGetMemory(num_services * sizeof(U16BIT));
      if (existing_lcns != NULL)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         for (nx = 0; nx != num_services && s_ptr != NULL; nx++)
         {
            existing_lcns[nx] = s_ptr->allocated_lcn;
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }
         lcn = start_lcn;
         while (--lcn != 0)
         {
            for (nx = 0; nx != num_services; nx++)
            {
               if (existing_lcns[nx] == lcn)
               {
                  break;
               }
            }
            if (nx == num_services)
            {
               /* lcn is not used */
               free_lcns[num_found] = lcn;
               num_found++;
               break;
            }
         }
         lcn = start_lcn;
         while (++lcn <= 9999)
         {
            for (nx = 0; nx != num_services; nx++)
            {
               if (existing_lcns[nx] == lcn)
               {
                  break;
               }
            }
            if (nx == num_services)
            {
               /* this lcn is not used */
               free_lcns[num_found] = lcn;
               num_found++;
               if (num_found == CIP_NUM_FREE_LCNS)
               {
                  break;
               }
            }
         }
         STB_AppFreeMemory(existing_lcns);
      }
   }
   FUNCTION_FINISH(GetFreeLcns);
   return num_found;
}

/**
 * @brief   Create a 'virtual' event record to facilitate display of Event Information on Virtual Channel
 * @param   name_str name of Virtual channel (to be used as event name)
 * @param   event_info event information to go in event deascriptor
 * @param   event_info_len length of event information
 * @return  pointer to the event record
 */
static ADB_EVENT_REC *CreateVirtualEventRec(SI_STRING_DESC *name_str, U8BIT *event_info, U8BIT event_info_len)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_DESC *event_desc;
   U8BIT *data;
   U32BIT lang;
   U8BIT dlen;

   e_ptr = STB_AppGetMemory(sizeof(ADB_EVENT_REC));
   if (e_ptr != NULL)
   {
      memset(e_ptr, 0, sizeof(ADB_EVENT_REC));

      dlen = 5 + event_info_len;
      if (name_str != NULL)
      {
         dlen += name_str->nbytes;
      }
      event_desc = (ADB_EVENT_DESC *)STB_AppGetMemory(sizeof(ADB_EVENT_DESC) + dlen + 2);
      if (event_desc != NULL)
      {
         /* set lang to UND */
         lang = ACFG_GetUndefinedLanguageBehaviour();
         data = (U8BIT *)(event_desc + 1);
         event_desc->desc_data = data;
         data[0] = SHORT_EVENT_DTAG;
         data[1] = dlen;
         data[2] = (lang >> 16) & 0xFF;
         data[3] = (lang >> 8) & 0xFF;
         data[4] = lang & 0xFF;
         data += 5;
         if (name_str != NULL)
         {
            /* fill 'event name' with name of Virtual Channel */
            dlen = name_str->nbytes;
            *data = dlen;
            data++;
            memcpy(data, name_str->str_ptr, dlen);
            data += dlen;
         }
         else
         {
            *data = 0;
            data++;
         }
         /* add event description with the event information that
          * "may be used to populate the EPG information field". See TS 103 205, section 15.3.1  */
         dlen = event_info_len;
         data++;
         memcpy(data, event_info, dlen);
         e_ptr->desc_list_head = event_desc;
         e_ptr->desc_list_tail = event_desc;
         /* start and end  times are not specified: so set start to now with duration of 24 hours */
         e_ptr->start = STB_GCNowDHMSLocal();
         e_ptr->duration = DHMS_CREATE(0, 23, 59, 59);
      }
      else
      {
         STB_AppFreeMemory(e_ptr);
         e_ptr = NULL;
      }
   }
   return e_ptr;
}

/**
 * @brief   Add Virtual Channel for the specified CI-CAM network
 *          In event of a problem this returns details of an issue with
 *          information provided by Ci-CAM
 * @param   network Network record
 * @param   v_chan Parsed Virtual Channel descriptor fields
 * @return  pointer to NIT management service item
 */
static S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA *
AddVirtualChannel(ADB_NETWORK_REC *network, SI_CIPLUS_VIRTUAL_CHANNEL *v_chan)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_TRANSPORT_REC* t_ptr;
   S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA *s_item;

   FUNCTION_START(AddVirtualChannel);
   if (!STB_CIIsMmiEngineRegistered(v_chan->app_domain_id))
   {
      s_item = STB_AppGetMemory(sizeof(S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA));
      if (s_item != NULL)
      {
         s_item->channel_issue_type = STB_CI_OPERATOR_NIT_DOMAIN_NOT_SUPPORTED;
         s_item->logical_channel_number = v_chan->lcn;
         if (v_chan->name_str != NULL)
         {
            s_item->length_service_name = v_chan->name_str->nbytes;
            s_item->service_name = v_chan->name_str->str_ptr;
         }
         else
         {
            s_item->length_service_name = 0;
            s_item->service_name = NULL;
         }
         s_item->number_free_lcn = 0;
         s_item->free_lcn = NULL;
      }
   }
   else
   {
      s_ptr = GetServiceFromLcn(v_chan->lcn);
      if (s_ptr != NULL)
      {
         /* LCN Collision ! */
         s_item = STB_AppGetMemory(sizeof(S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA));
         if (s_item != NULL)
         {
            s_item->channel_issue_type = STB_CI_OPERATOR_NIT_LCN_COLLISION;
            s_item->logical_channel_number = v_chan->lcn;
            if (v_chan->name_str != NULL)
            {
               s_item->length_service_name = v_chan->name_str->nbytes;
               s_item->service_name = v_chan->name_str->str_ptr;
            }
            else
            {
               s_item->length_service_name = 0;
               s_item->service_name = NULL;
            }
            s_item->free_lcn = (U16BIT *)STB_AppGetMemory(CIP_NUM_FREE_LCNS * sizeof(U16BIT));
            s_item->number_free_lcn = GetFreeLcns(v_chan->lcn, s_item->free_lcn);
         }
      }
      else
      {
         s_item = NULL; /* set to NULL, because Virtual channel is supported and LCN is okay */

         /* create 'virtual' transport rec to link virtual channel to the network rec */
         t_ptr = DBDEF_AddVirtualTransportRec(network);
         if (t_ptr != NULL)
         {
            if ((s_ptr = DBDEF_AddServiceRec(0xFFFF, t_ptr)) != NULL)
            {
               s_ptr->new_service = TRUE;
               s_ptr->serv_lcn = v_chan->lcn;
               s_ptr->allocated_lcn = v_chan->lcn;
               DBDEF_SetServiceType(s_ptr, ADB_SERVICE_TYPE_VIRTUAL);
               if (v_chan->name_str != NULL)
               {
                  DBDEF_SetServiceName(s_ptr, v_chan->name_str->str_ptr);
                  DBDEF_SetServiceShortName(s_ptr, v_chan->name_str->str_ptr);
               }
               if (v_chan->provider_str != NULL)
               {
                  DBDEF_SetServiceProviderName(s_ptr, v_chan->provider_str->str_ptr);
               }
               if (v_chan->event_info_len != 0)
               {
                  s_ptr->now_event = CreateVirtualEventRec(v_chan->name_str, v_chan->event_info, v_chan->event_info_len);
               }
            }
            else
            {
               DBG_OP("failed to create Virtual channel");
               DBDEF_DeleteTransportRec(t_ptr);
            }
         }
         else
         {
            DBG_OP("failed to create Virtual transport");
         }
      }
   }
   FUNCTION_FINISH(AddVirtualChannel);
   return s_item;
}


/**
 * @brief   Process an NIT sent from a CI+ CAM to create or update an operator profile
 * @param   module - operator module
 * @param   nit - structure containing the NIT and other operator profile related info
 */
static void ProcessNit(U32BIT module, S_CIP_OPERATOR_NIT *nit)
{
   ADB_NETWORK_REC *n_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_TRANSPORT_REC *next_t_ptr;
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *next_s_ptr;
   BOOLEAN changed;
   SI_NIT_TRANSPORT_ENTRY *nit_entry;
   SI_CIPLUS_SERVICE *serv_entry;
   SI_STRING_DESC *str_desc;
   U8BIT *tmp_str;
   U16BIT profile_name_bytes;
   ADB_FAVLIST_REC *fav_list = NULL;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_ptr;
   S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA *serv_items;
   U8BIT serv_item_count;

   FUNCTION_START(ProcessNit);

   DBDEF_RequestAccess();

   /* Save the current profile type so it can be restored at the end */
   DBDEF_PushCIPlusProfile(nit->cicam_onet_id, nit->cicam_identifier);

   /* Check whether there's already a network for this CICAM */
   if ((n_ptr = DBDEF_FindNetworkForCIPlusProfile(nit->cicam_onet_id, nit->cicam_identifier)) == NULL)
   {
      /* Add a new network for this CAM and set its profile info */
      n_ptr = DBDEF_AddNetworkRec(nit->table->net_id, NULL);
      DBG_OP("(%lu): Added network %p", module, n_ptr)

      n_ptr->profile_type = ADB_PROFILE_TYPE_CIPLUS;
      DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_PROFILE_TYPE, (U32BIT)n_ptr->profile_type);

      n_ptr->cicam_onet_id = nit->cicam_onet_id;
      DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_ORIG_NET_ID, n_ptr->cicam_onet_id);

      n_ptr->cicam_identifier = nit->cicam_identifier;
      DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_PROFILE_CAM_ID, n_ptr->cicam_identifier);
   }

   /* Now process the NIT if there's a network record */
   if (n_ptr != NULL)
   {
      if (n_ptr->profile_name != NULL)
      {
         profile_name_bytes = n_ptr->profile_name->nbytes;
      }
      else
      {
         profile_name_bytes = 0;
      }
      if ((profile_name_bytes != nit->profile_name_length + 2) ||
         (memcmp(n_ptr->profile_name->str_ptr, nit->profile_name, nit->profile_name_length) != 0))
      {
         if (n_ptr->profile_name != NULL)
         {
            DBDEF_ReleaseString(n_ptr->profile_name);
         }

         /* Profile string isn't null terminated, so null terminate it before saving it. If
            profile_name_length=0 the resulting string will be empty */
         if ((tmp_str = (U8BIT *)STB_AppGetMemory(nit->profile_name_length + 2)) != NULL)
         {
            memset(tmp_str, 0, nit->profile_name_length + 2);
            if (nit->profile_name_length > 0)
            {
               memcpy(tmp_str, nit->profile_name, nit->profile_name_length);
            }

            n_ptr->profile_name = DBDEF_MakeString(nit->lang_code, tmp_str, nit->profile_name_length + 2);

            STB_AppFreeMemory(tmp_str);
         }
      }

      if (n_ptr->fav_list_id != 0)
      {
         fav_list = DBDEF_FindFavouriteList(n_ptr->fav_list_id);
         if (fav_list != NULL)
         {
            DBDEF_DeleteAllServicesFromFavouriteList(fav_list);
         }
      }

      if (fav_list == NULL)
      {
         fav_list = DBDEF_AddFavouriteList(0, n_ptr->profile_name->str_ptr, n_ptr->cicam_identifier, -1);
         if (fav_list != NULL)
         {
            n_ptr->fav_list_id = fav_list->list_id;
            DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_FAVLIST_ID, n_ptr->fav_list_id);
         }
      }

      n_ptr->module_present = TRUE;
      n_ptr->module = module;

      DBG_OP("   NIT v%u", nit->table->version)

      /* Save version in network record */
      n_ptr->nit_version = nit->table->version;
      DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_VERSION, n_ptr->nit_version);
      n_ptr->nit_version_changed = FALSE;

      /* Update the network name */
      changed = TRUE;
      if ((n_ptr->name_str != NULL) && (nit->table->name_str != NULL))
      {
         str_desc = nit->table->name_str;
         if (str_desc->nbytes == n_ptr->name_str->nbytes)
         {
            if (memcmp(n_ptr->name_str->str_ptr, str_desc->str_ptr, n_ptr->name_str->nbytes) == 0)
            {
               changed = FALSE;
            }
         }
      }

      if (changed)
      {
         /* Release old name */
         if (n_ptr->name_str != NULL)
         {
            STB_AppFreeMemory(n_ptr->name_str);
            n_ptr->name_str = NULL;
         }

         /* Add new name */
         if (nit->table->name_str != NULL)
         {
            str_desc = nit->table->name_str;
            n_ptr->name_str = DBDEF_MakeString(nit->lang_code, str_desc->str_ptr, str_desc->nbytes);

            DBG_OP("   Network name \"%s\"", str_desc->str_ptr)
         }
      }

      serv_item_count = 0;
      if (nit->table->ciplus_virtual_channel != NULL)
      {
         serv_items = AddVirtualChannel(n_ptr, nit->table->ciplus_virtual_channel);
         if (serv_items != NULL)
         {
            serv_item_count++;
         }
      }
      else
      {
         serv_items = NULL;
      }

      /* Mark any existing transports in this network as unavailable so they can be deleted
       * if they're not found while parsing the transport list */
      t_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t_ptr != NULL)
      {
         if (t_ptr->network == n_ptr)
         {
            t_ptr->available = FALSE;
         }
         t_ptr = DBDEF_GetNextTransportRec(t_ptr);
      }

      /* Process each transport */
      nit_entry = nit->table->transport_list;
      while (nit_entry != NULL)
      {
         t_ptr = DBDEF_FindTransportRecByIds(NULL, nit->table->net_id, nit_entry->orig_net_id, nit_entry->tran_id);
         if (t_ptr != NULL)
         {
            if ((t_ptr->network == NULL) || (t_ptr->network->profile_type == ADB_PROFILE_TYPE_BROADCAST))
            {
               /* Do not re-use transport records from the broadcast prfile for a CI+ profile */
               t_ptr = NULL;
            }
         }

         if ( t_ptr != NULL)
         {
            DBA_SetRecordParent(t_ptr->dba_rec, n_ptr->dba_rec);
            DBA_SaveRecord(t_ptr->dba_rec);
         }
         else
         {
            /* A delivery descriptor is required to add the transport */
            if (nit_entry->del_sys_desc != NULL)
            {
               /* Add the new transport */
               switch (nit_entry->del_sys_desc_type)
               {
                  case SI_DEL_SYS_DESC_TYPE_TERR:
                  {
                     if (nit_entry->del_sys_desc->terr.is_t2)
                     {
                        if ((nit_entry->del_sys_desc->terr.u.t2.num_cells > 0) &&
                            (nit_entry->del_sys_desc->terr.u.t2.cell[0].num_freqs > 0))
                        {
                           t_ptr = DBDEF_AddTerrestrialTransportRec( nit_entry->del_sys_desc->terr.u.t2.cell[0].freq_hz[0],
                                 nit_entry->del_sys_desc->terr.u.t2.plp_id, n_ptr);

                           DBG_OP("Add DVB-T2 transport %p, freq %lu Hz", t_ptr, nit_entry->del_sys_desc->terr.u.t2.cell[0].freq_hz[0])
                        }
                     }
                     else
                     {
                        t_ptr = DBDEF_AddTerrestrialTransportRec(nit_entry->del_sys_desc->terr.u.t1.freq_hz, 0, n_ptr);

                        DBG_OP("Add DVB-T transport %p, freq %lu Hz", t_ptr,nit_entry->del_sys_desc->terr.u.t1.freq_hz)
                     }
                     break;
                  }

                  case SI_DEL_SYS_DESC_TYPE_CABLE:
                  {
                     t_ptr = DBDEF_AddCableTransportRec(nit_entry->del_sys_desc->cable.freq_hz,
                           nit_entry->del_sys_desc->cable.symbol_rate, n_ptr);
                     DBG_OP("Add DVB-C transport %p, freq %lu Hz, symbol rate %u Kbits", t_ptr,
                          nit_entry->del_sys_desc->cable.freq_hz,
                          nit_entry->del_sys_desc->cable.symbol_rate)
                     break;
                  }

                  case SI_DEL_SYS_DESC_TYPE_SAT:
                  {
                     t_ptr = DBDEF_AddSatTransportRec(nit_entry->del_sys_desc->sat.freq_hz,
                           nit_entry->del_sys_desc->sat.sym_rate,
                           nit_entry->del_sys_desc->sat.polarity,
                           nit_entry->del_sys_desc->sat.dvb_s2,
                           nit_entry->del_sys_desc->sat.modulation, n_ptr);
                     DBG_OP("Add DVB-%s transport %p, %lu Hz, symrate %u, polarity %u, modulation %u",
                          (nit_entry->del_sys_desc->sat.dvb_s2 ? "S2" : "S"),
                          t_ptr, nit_entry->del_sys_desc->sat.freq_hz,
                          nit_entry->del_sys_desc->sat.sym_rate,
                          nit_entry->del_sys_desc->sat.polarity,
                          nit_entry->del_sys_desc->sat.modulation)
                     break;
                  }
               }

               if (t_ptr != NULL)
               {
                  DBDEF_SetTransportOrigNetworkId(t_ptr, nit_entry->orig_net_id);
                  DBDEF_SetTransportTransportId(t_ptr, nit_entry->tran_id);
               }
            }
         }

         if (t_ptr != NULL)
         {
            t_ptr->available = TRUE;

            DBG_OP("Transport ids: %u/%u", t_ptr->orig_net_id, t_ptr->tran_id)

            /* Mark any services on this transport as unavailable
             * so they can be deleted later if not found */
            s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr);
            while (s_ptr != NULL)
            {
               s_ptr->unavailable = TRUE;
               s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);
            }

            serv_entry = nit_entry->ciplus_service_list;
            while (serv_entry != NULL)
            {
               s_ptr = DBDEF_FindServiceRecByIds(NULL, nit->table->net_id,
                        nit_entry->orig_net_id, nit_entry->tran_id, serv_entry->id);
               if (s_ptr != NULL)
               {
                  if ((s_ptr->transport->network->profile_type == ADB_PROFILE_TYPE_BROADCAST) ||
                     (s_ptr->allocated_lcn != serv_entry->lcn))
                  {
                     /* Do not re-use service records from the broadcast profile for a CI+ profile.
                        Also, CI+ services can have the same SID, so if the LCN is different, it's
                        a different service */
                     s_ptr = NULL;
                  }
               }

               if (s_ptr == NULL)
               {
                  next_s_ptr = GetServiceFromLcn(serv_entry->lcn);
                  if (next_s_ptr != NULL)
                  {
                     /* LCN Collision ! */
                     serv_items = STB_AppResizeMemory(serv_items,
                        (serv_item_count+1) * sizeof(S_STB_CI_OPERATOR_NIT_MANAGEMENT_SERVICE_ITEM_DATA));
                     if (serv_items != NULL)
                     {
                        serv_items[serv_item_count].channel_issue_type = STB_CI_OPERATOR_NIT_LCN_COLLISION;
                        serv_items[serv_item_count].logical_channel_number = serv_entry->lcn;
                        if (serv_entry->name_str != NULL)
                        {
                           serv_items[serv_item_count].length_service_name = serv_entry->name_str->nbytes;
                           serv_items[serv_item_count].service_name = serv_entry->name_str->str_ptr;
                        }
                        else
                        {
                           serv_items[serv_item_count].length_service_name = 0;
                           serv_items[serv_item_count].service_name = NULL;
                        }
                        serv_items[serv_item_count].free_lcn = (U16BIT *)STB_AppGetMemory(CIP_NUM_FREE_LCNS * sizeof(U16BIT));
                        serv_items[serv_item_count].number_free_lcn = GetFreeLcns(serv_entry->lcn, serv_items[serv_item_count].free_lcn);
                        serv_item_count++;
                     }
                  }
                  else
                  {
                     /* Add the new service */
                     if ((s_ptr = DBDEF_AddServiceRec(serv_entry->id, t_ptr)) != NULL)
                     {
                        s_ptr->new_service = TRUE;
                     }
                  }
               }
               else
               {
                  s_ptr->new_service = FALSE;
               }

               if (s_ptr != NULL)
               {
                  s_ptr->found = TRUE;

                  /* Update the service type */
                  if (s_ptr->new_service || (s_ptr->serv_type != serv_entry->type))
                  {
                     s_ptr->serv_type = serv_entry->type;
                  }
                  if (s_ptr->new_service || (s_ptr->serv_lcn != serv_entry->lcn))
                  {
                     s_ptr->serv_lcn = serv_entry->lcn;
                     s_ptr->allocated_lcn = serv_entry->lcn;
                  }
                  s_ptr->hidden = !serv_entry->visible;
                  DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                  s_ptr->selectable = serv_entry->selectable;
                  DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                  /* Update the service's name */
                  changed = TRUE;
                  if ((s_ptr->name_str != NULL) && (serv_entry->name_str != NULL))
                  {
                     str_desc = serv_entry->name_str;
                     if (str_desc->nbytes == s_ptr->name_str->nbytes)
                     {
                        if (memcmp(s_ptr->name_str->str_ptr, str_desc->str_ptr, str_desc->nbytes) == 0)
                        {
                           changed = FALSE;
                        }
                     }
                  }

                  if (changed)
                  {
                     /* Free the old name */
                     if (s_ptr->name_str != NULL)
                     {
                        DBDEF_ReleaseString(s_ptr->name_str);
                        s_ptr->name_str = NULL;
                     }
                     /* Set the new name */
                     if (serv_entry->name_str != NULL)
                     {
                        str_desc = serv_entry->name_str;
                        s_ptr->name_str = DBDEF_MakeString(nit->lang_code, str_desc->str_ptr, str_desc->nbytes);
                     }
                  }

                  /* Update the provider */
                  if (s_ptr->provider_str != NULL)
                  {
                     DBDEF_ReleaseString(s_ptr->provider_str);
                     s_ptr->provider_str = NULL;
                  }

                  if (serv_entry->provider_str != NULL)
                  {
                     str_desc = serv_entry->provider_str;
                     s_ptr->provider_str = DBDEF_MakeString(nit->lang_code, str_desc->str_ptr, str_desc->nbytes);
                  }

                  s_ptr->unavailable = FALSE;
#ifdef CI_DEBUG
                  if (s_ptr->new_service)
                  {
                     DBG_OP("Added service %p, \"%s\", LCN=%u", s_ptr, s_ptr->name_str->str_ptr,s_ptr->serv_lcn)
                  }
                  else
                  {
                     DBG_OP("Updated service %p, \"%s\", LCN=%u", s_ptr, s_ptr->name_str->str_ptr,s_ptr->serv_lcn)
                  }
#endif
               }

               serv_entry = serv_entry->next;
            }

#if 0
            /* Now delete any services that are no longer present */
            s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr);
            while (s_ptr != NULL)
            {
               next_s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);

               if (s_ptr->unavailable)
               {
                  /* Send an event to inform the app that this service is being deleted */
                  DBG_OP("Delete unavailable service %p, nm ptr %p, LCN=%u", s_ptr, s_ptr->name_str, s_ptr->serv_lcn)
                  if (s_ptr->name_str != NULL)
                  {
                     DBG_OP("   name \"%s\"", s_ptr->name_str->str_ptr)
                  }

                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_DELETED,
                     &s_ptr, sizeof(s_ptr));

                  DBDEF_DeleteServiceRec(s_ptr);
               }

               s_ptr = next_s_ptr;
            }
#endif
         }

         nit_entry = nit_entry->next;
      }

      /* Delete the linkage array for repopulation */
      DeleteLinkageDescriptorArray(n_ptr->linkage_desc_list);
      n_ptr->last_linkage_entry = NULL;
      n_ptr->num_linkage_entries = 0;
      n_ptr->linkage_desc_list = NULL;

      /* Copy all network linkage descripters  */
      linkage_desc_ptr = nit->table->linkage_desc_list;
      while (linkage_desc_ptr != NULL)
      {
         CopyLinkageDesc(linkage_desc_ptr,
            &n_ptr->linkage_desc_list,
            &n_ptr->last_linkage_entry,
            &n_ptr->num_linkage_entries);

         linkage_desc_ptr = linkage_desc_ptr->next;
      }

#if 0
      /* Check the transports for any that are no longer available */
      t_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t_ptr != NULL)
      {
         next_t_ptr = DBDEF_GetNextTransportRec(t_ptr);

         if (!((ADB_TRANSPORT_REC *)t_ptr)->available)
         {
            DBG_OP("Delete unavailable transport %p", t_ptr)
            DBDEF_DeleteTransportRec((ADB_TRANSPORT_REC *)t_ptr);
         }

         t_ptr = next_t_ptr;
      }
#endif

      if (fav_list != NULL)
      {
         t_ptr = NULL;
         while ((t_ptr = DBDEF_GetNextTransportRec(t_ptr)) != NULL)
         {
            if (t_ptr->network == n_ptr)
            {
               s_ptr = NULL;
               while ((s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr)) != NULL)
               {
                  DBDEF_AddServiceToFavouriteList(fav_list, s_ptr, -1);
               }
            }
         }

         DBA_LockDatabase();
         DBA_SaveDatabase();
         DBA_UnlockDatabase();
      }
      if (serv_items != NULL)
      {
         /* There was an error allocating service - report to CAM */
         STB_CISendOperatorNITManagement(module, STB_CI_OPERATOR_NIT_ERROR_NIT_ERROR, serv_item_count, serv_items);
         while (serv_item_count != 0)
         {
            serv_item_count--;
            if (serv_items[serv_item_count].free_lcn != NULL)
            {
               STB_AppFreeMemory(serv_items[serv_item_count].free_lcn);
            }
         }
         STB_AppFreeMemory(serv_items);
      }
      else
      {
         STB_CISendOperatorNITManagement(module, STB_CI_OPERATOR_NIT_ERROR_NO_ISSUE, 0, NULL);
      }
   }

   DBDEF_PopProfile();
   DBDEF_ReleaseAccess();
   /* Free the table */
   STB_SIReleaseNitTable(nit->table);
   FUNCTION_FINISH(ProcessNit);
}


/**
 * @brief   Fills the provided structure with host related info, such as which
 *          service types, delivery systems and app types are supported
 * @param   host_info - structure that will be filled in
 */
static void RequestHostInfo(S_CIP_HOST_INFO *host_info)
{
   U8BIT i, num_tuners;
   E_STB_TUNE_SYSTEM_TYPE system;

   FUNCTION_START(RequestHostInfo);

   host_info->standby = ASTE_InStandby();

   /* Setup the supported service types */
   host_info->num_service_types = 8;
   host_info->service_types = CIP_SERVICE_TYPE_TV | CIP_SERVICE_TYPE_RADIO |
      CIP_SERVICE_TYPE_TELETEXT | CIP_SERVICE_TYPE_AVC_RADIO | CIP_SERVICE_TYPE_DATA |
      CIP_SERVICE_TYPE_HD_TV | CIP_SERVICE_TYPE_AVC_SD_TV | CIP_SERVICE_TYPE_AVC_HD_TV;

   /* Setup the supported delivery types */
   host_info->delivery_types = 0;

   num_tuners = STB_HWGetTunerPaths();
   for (i = 0; i < num_tuners; i++)
   {
      system = STB_TuneGetSupportedSystemType(i);
      switch (system)
      {
         case TUNE_SYSTEM_TYPE_DVBT2:
            host_info->delivery_types |= CIP_DELIVERY_TYPE_DVBT2;
            /* Intentionally falling through to the next case */
         case TUNE_SYSTEM_TYPE_DVBT:
            host_info->delivery_types |= CIP_DELIVERY_TYPE_DVBT;
            break;
         case TUNE_SYSTEM_TYPE_DVBS2:
            host_info->delivery_types |= CIP_DELIVERY_TYPE_DVBS2;
            /* Intentionally falling through to the next case */
         case TUNE_SYSTEM_TYPE_DVBS:
            host_info->delivery_types |= CIP_DELIVERY_TYPE_DVBS;
            break;
         case TUNE_SYSTEM_TYPE_DVBC:
            host_info->delivery_types |= CIP_DELIVERY_TYPE_DVBC;
            break;
         default:
            break;
      }
   }

   /* Setup the supported application capabilities */
   host_info->app_types = 0;
#ifdef INTEGRATE_MHEG5
   host_info->app_types |= CIP_APP_TYPE_MHEG;
#endif
#ifdef HBBTV_BUILD
   host_info->app_types |= CIP_APP_TYPE_HBBTV;
#endif

   FUNCTION_FINISH(RequestHostInfo);
}

/**
 * @brief   Returns a boolean value indicating whether an operator search can be started.
 *          This is based on what mode (yes/no/ask) the app is in. If the mode is ask,
 *          then the app will be informed of the request.
 * @param   module - operator module making the request
 * @param   op_search - structure containing info related to the search
 * @return  TRUE if the search can proceed immediately, FALSE otherwise
 */
static BOOLEAN RequestOperatorSearch(U32BIT module, S_CIP_OPERATOR_SEARCH *op_search)
{
   BOOLEAN retval;
   U16BIT date;
   U8BIT hours, mins, secs;
   void *profile;

   FUNCTION_START(RequestOperatorSearch);
   retval = FALSE;
   STB_GCGetGMTDateTime(&date, &hours, &mins, &secs);
   if ((op_search->refresh_request == CIP_OPERATOR_SEARCH_SCHEDULED) &&
       STB_GCIsFutureDateTime(op_search->refresh_date, op_search->refresh_hour, op_search->refresh_min, 0))
   {
      /* Set the operator search date/time for the module */
      ACI_ScheduleOperatorSearch(op_search->cicam_id, op_search->refresh_date, op_search->refresh_hour,
         op_search->refresh_min);

      /* If there is a profile associated to this module, update it with the search request */
      if ((profile = ADB_FindProfileForModule(module)) != NULL)
      {
         ADB_SetProfileSearchRequired(profile, TRUE, op_search->refresh_date,
            op_search->refresh_hour, op_search->refresh_min);
      }
   }
   else
   {
      switch (ACI_GetOperatorSearchMode())
      {
         case CIP_START_OPERATOR_SEARCH_NO:
         {
            /* Search can't be started, so inform the app that a search is required */
            ACI_OperatorSearchRequired(module, TRUE);
            break;
         }

         case CIP_START_OPERATOR_SEARCH_YES:
         {
            /* Search can be started */
            retval = TRUE;
            break;
         }

         case CIP_START_OPERATOR_SEARCH_ASK:
         {
            /* Set flag to indicate a search is required */
            ACI_OperatorSearchRequired(module, TRUE);

            if (op_search->refresh_request == CIP_OPERATOR_SEARCH_IMMEDIATE)
            {
               /* This is an urgent request, so ask the user immediately */
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_REQUEST_OPERATOR_SEARCH,
                  &module, sizeof(module));
            }
            break;
         }
      }
   }

   FUNCTION_FINISH(RequestOperatorSearch);
   return(retval);
}

/****************************************************************************
** End of file
*****************************************************************************/
