/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application control functions
 *
 * @file    ap_cntrl.c
 * @date    04/09/2003
 */

// debug macros
#define DEBUG_TUNING
#define DEBUG_DECODING
#define DEBUG_AV_OUTPUT
//#define DEBUG_PRINTING_ENABLED
/*#define DEBUG_STANDBY*/
/*#define DEBUG_SERVICE_SEARCH*/
/*#define DEBUG_TOT_SEARCH*/
/*#define DEBUG_SSU_SEARCH*/
/*#define DEBUG_EIT_SEARCH*/
/*#define DEBUG_STARTUP_SEARCH*/
/*#define DEBUG_SUBTITLES*/

/*#define DEBUG_MHEG*/

//#define DEBUG_MEMORY_USAGE

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <string.h>
#include <stdio.h>

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"
#include "stbhwini.h"
#include "stbhwfp.h"
#include "stbhwav.h"
#include "stbheap.h"
#include "stbgc.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbllist.h"
#include "stbhwdmx.h"
#include "stbdpc.h"
#include "stbdsapi.h"
#include "stbpes.h"
#include "stbpvr.h"
#include "stberc.h"
#include "stbvtc.h"

#ifdef INCLUDE_OTA_SSU
#include "stbota.h"
#endif

#include "stbver.h"
#include "stbebutt.h"

#ifdef INTEGRATE_HBBTV
#include "hbbtv_api.h"
#endif

#include "app.h"
#include "ap_cfg.h"
#include "app_nvm.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"
#include "ap_si.h"
#include "ap_cntrl.h"
#include "ap_state.h"

#include "ap_pvr.h"

#ifdef COMMON_INTERFACE
#include "ap_ci_int.h"
#include "stbcicc.h"
#include "ap_ci.h"
#include "ap_ciop.h"
#include "stbci.h"
#endif
#include "ca_glue.h"
#include "ap_ca.h"

#include "dba.h"

#ifdef DEBUG_STANDBY
#define DBG_STDBY(X)    STB_SPDebugWrite X
#else
#define DBG_STDBY(X)
#endif

#ifdef DEBUG_SSU_SEARCH
#define DBG_SSU(X)    STB_SPDebugWrite X
#else
#define DBG_SSU(X)
#endif

#ifdef DEBUG_TOT_SEARCH
#define TOT_DBG(X)      STB_SPDebugWrite X
#else
#define TOT_DBG(X)
#endif

#ifdef DEBUG_EIT_SEARCH
#define EIT_DBG(X)      STB_SPDebugWrite X
#else
#define EIT_DBG(X)
#endif

#ifdef DEBUG_SUBTITLES
#define SUBT_DBG(X)      STB_SPDebugWrite X
#else
#define SUBT_DBG(X)
#endif

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#endif
#define FUNCTION_START(X)  STB_SPDebugWrite(">>> %s\n", # X)

#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#endif
#define FUNCTION_FINISH(X)  STB_SPDebugWrite("<<< %s\n", # X)
#endif

#ifdef PVR_LOG
extern FILE *pvr_log;
extern void LogDateTime();
#endif

//---constant definitions for this file-------------------------------------------------------------


//---local typedefs, structs, enumerations for this file--------------------------------------------

typedef enum
{
   EXTERNAL_EVENT,
   INTERNAL_EVENT
} E_ACTL_EVENT_TYPE;

typedef enum
{
   INT_EVENT_INITIALISE,
   INT_EVENT_ENABLE,
   INT_EVENT_DISABLE,
   INT_EVENT_UPDATE,
   INT_EVENT_RF_CHAN_TUNE_REQD,
   INT_EVENT_TPTR_TUNE_REQD,
   INT_EVENT_USERDEFINED_TUNE_REQD,
   INT_EVENT_TUNE_OFF,
   INT_EVENT_RESTART_DECODING,
   INT_EVENT_RESTART_AUDIO,
   INT_EVENT_RESTART_SUBTITLES,
   INT_EVENT_SCRAMBLE_CHANGE,
   INT_EVENT_RELEASE_DECODE_LOCK,
   INT_EVENT_STANDBY_ON,
   INT_EVENT_STANDBY_OFF,
   INT_EVENT_STANDBY_VCR_ACTIVE,
   INT_EVENT_ALT_AV_ALLOWED,
   INT_EVENT_ALT_AV_NOT_ALLOWED,
   INT_EVENT_START_MHEG_TUNE,
   INT_EVENT_ANALOG_TV_ON,
   INT_EVENT_ANALOG_TV_OFF,
   INT_EVENT_ANALOG_VIDEO_ALLOWED,
   INT_EVENT_ANALOG_VIDEO_NOT_ALLOWED
} E_ACTL_INTERNAL_EVENT;

typedef enum
{
   TUNED_STATUS_NO_SIGNAL,
   TUNED_STATUS_BAD_SIGNAL,
   TUNED_STATUS_GOOD_SIGNAL
} E_ACTL_TUNED_STATUS;

typedef enum
{
   TSTATE_OFF,
   TSTATE_TUNING,
   TSTATE_NOT_TUNED,
   TSTATE_TUNED,
   TSTATE_ANALOG
} E_ACTL_TUNING_STATE;

// the order of these states is important - the following code uses >= comparisons to check
// for a number of states in one go. Do not change without checking the code.
typedef enum
{
   DSTATE_DISABLED,
   DSTATE_OFF,
   DSTATE_STARTING_BAD_SIGNAL,
   DSTATE_STARTING,
   DSTATE_LOCKED,
   DSTATE_DECODING,
   DSTATE_PAUSED_SIGNAL,
   DSTATE_PAUSED_USER,
   DSTATE_PAUSED_SIGNAL_AND_USER
} E_ACTL_DECODING_STATE;


typedef struct
{
   const E_STB_AV_VIDEO_FORMAT mode;
   U16BIT width;
   U16BIT height;
} S_VIDEO_FORMAT_DESC;


typedef enum
{
   SEARCH_TYPE_NOSEARCH,
   SEARCH_TYPE_SERVICE_FREQ,
   SEARCH_TYPE_SERVICE_NETWORK,
   SEARCH_TYPE_STARTUP,
   SEARCH_TYPE_MANUAL_FREQ,
   SEARCH_TYPE_MANUAL_NETWORK,
   SEARCH_TYPE_SSU,
   SEARCH_TYPE_SSU_DOWNLOAD,
   SEARCH_TYPE_OTA_UPDATE,
   SEARCH_TYPE_TOT,
   SEARCH_TYPE_EIT_PF,
   SEARCH_TYPE_EIT_SCHED,
   SEARCH_TYPE_EIT_PF_SCHED
} E_SEARCH_TYPE;


#ifdef INCLUDE_OTA_SSU
typedef struct
{
   BOOLEAN ota_started;
   U8BIT ota_path;
   void *ota_location;
   U16BIT ota_onet_id;
   U16BIT ota_tran_id;
   U16BIT ota_serv_id;
   F_SSU_VERSION_CALLBACK version_cb;
} S_OTA_SETTINGS;
#endif

typedef struct
{
   U8BIT version;
   BOOLEAN valid;  /* the following setting data are valid */
   U8BIT path;
   U8BIT action_type;
   U8BIT processing_order;
   U8BIT private_data_byte[256];
   //
   U16BIT ssu_data_pid;
   ADB_TRANSPORT_REC * transport;
   //
   U8BIT jsonString[512];
} S_UNT_SETTINGS;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

static BOOLEAN initialised = FALSE;

#ifdef INCLUDE_OTA_SSU
static S_OTA_SETTINGS ota_settings;
#endif

static S_UNT_SETTINGS unt_settings = { 0xFF, FALSE, INVALID_RES_ID, 0, 0, {0}, 0x1FFF, NULL, {0} };

// required state
static U16BIT rf_chan_id_required;
static void **transport_required;
static E_STB_DP_SIGNAL_TYPE signal_type_required;
static BOOLEAN tuner_relock_required;
static void **service_required;
static E_APP_SI_MODE *si_required;
static BOOLEAN decode_lock_override_required;

#ifdef COMMON_INTERFACE
static void *tuneto_live_service;
static void *tuneto_live_transport;
static E_ACTL_SI_SRCH_REQD tuneto_live_si;
static BOOLEAN tuneto_relock;
static BOOLEAN tuneto_for_recording;
static S_ACTL_OWNER_INFO *tuneto_owner_info;
static U32BIT ci_ai_module;
#endif

static S_MANUAL_TUNING_PARAMS *user_tuning_params;
static E_ACTL_TUNING_STATE *tuning_state;
static E_ACTL_DECODING_STATE *decoding_state;
static void **cntrl_mutex;

// status information
static void **current_satellite;
static ADB_SERVICE_REC **current_service;
static void **current_transport;
static E_APP_SI_MODE *current_si;
static BOOLEAN current_tuner_relock;
static E_ACTL_AV_MODE av_mode;
static BOOLEAN audio_muted;
static U8BIT audio_volume;
static U8BIT ad_volume;
static S8BIT volume_scaling = 0;
static BOOLEAN ci_ui_request;
static BOOLEAN *decoding_started;
static BOOLEAN *decoding_locked;
static S32BIT *now_event_id;

static BOOLEAN dvb_subtitles;
static BOOLEAN subtitles_started;
static BOOLEAN subtitles_running;
static BOOLEAN subtitles_suppressed;

static BOOLEAN external_control = FALSE;
static F_ServiceGetSubtitlePid GetSubtitlePidFunc = ADB_ServiceGetSubtitlePid;

static E_ACTL_TUNED_STATUS *tuned_status;
static BOOLEAN audio_desc_on;

//pause flag
static BOOLEAN *decode_paused_flag;

static U8BIT standby_path = INVALID_RES_ID;

static U8BIT monitor_path = INVALID_RES_ID;

/* Values used during searches */
static E_SEARCH_TYPE current_search_type = SEARCH_TYPE_NOSEARCH;
static E_SEARCH_TYPE required_search_type = SEARCH_TYPE_NOSEARCH;
static U8BIT search_path = INVALID_RES_ID;
static U16BIT search_list_id;
static E_STB_DP_SIGNAL_TYPE current_search_tuner_type = SIGNAL_NONE;

/* Service search */
static E_STB_TUNE_THIERARCHY terr_hierarchy;
static U8BIT t2_plp_id;

/* Startup search */
static U16BIT search_num_transports;
static void **search_transport_list = NULL;

/* EIT search */
static U32BIT eit_search_end_timer = INVALID_TIMER_HANDLE;
static U32BIT standby_wakeup_timer = INVALID_TIMER_HANDLE;
#ifdef COMMON_INTERFACE
static U32DHMS start_standby_time;
#endif
static U16BIT standby_grace_timeout = 0;
static U32BIT standby_grace_timer = INVALID_TIMER_HANDLE;

static BOOLEAN hdmi_connected = FALSE;

// can't make this const due to VIDEO_FORMAT_AUTO
static S_VIDEO_FORMAT_DESC app_video_formats[] = {
   {VIDEO_FORMAT_2160P50UHD, 3840, 2160},
   {VIDEO_FORMAT_2160P60UHD, 3840, 2160},
   {VIDEO_FORMAT_2160P25UHD, 3840, 2160},
   {VIDEO_FORMAT_2160P30UHD, 3840, 2160},
   {VIDEO_FORMAT_2160P24UHD, 3840, 2160},
   {VIDEO_FORMAT_1080P60HD, 1920, 1080},
   {VIDEO_FORMAT_1080P50HD, 1920, 1080},
   {VIDEO_FORMAT_1080P30HD, 1920, 1080},
   {VIDEO_FORMAT_1080I50HD, 1920, 1080},
   {VIDEO_FORMAT_1080P25HD, 1920, 1080},
   {VIDEO_FORMAT_1080IHD, 1920, 1080},
   {VIDEO_FORMAT_720P60HD, 1280, 720},
   {VIDEO_FORMAT_720P50HD, 1280, 720},
   {VIDEO_FORMAT_576PHD, 720, 576},
   {VIDEO_FORMAT_576IHD, 720, 576},
   {VIDEO_FORMAT_ORIGINAL, 0, 0},
   {VIDEO_FORMAT_AUTO, 0, 0}
};

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)

static U8BIT AcquireLivePath(void *s_ptr, S_ACTL_OWNER_INFO *owner_info,
   BOOLEAN override_parental_lock, void *t_ptr, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on);
static U8BIT AcquireRecordingPath(void *s_ptr, S_ACTL_OWNER_INFO *owner_info);
#ifdef COMMON_INTERFACE
static S_ACTL_OWNER_INFO* CopyOwnerInfo(S_ACTL_OWNER_INFO *owner_info);
static void AskRelease(U8BIT path, void *s_ptr, S_ACTL_OWNER_INFO *owner_info,
   BOOLEAN override_parental_lock, void *t_ptr, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on);
static void CIPowerDownReadyCallback(void);
#endif

static void ControlTuning(U8BIT path, E_ACTL_EVENT_TYPE type, U32BIT event);
static void ControlDecoding(U8BIT path, E_ACTL_EVENT_TYPE type, U32BIT event);
static void ControlAvOutput(E_ACTL_EVENT_TYPE type, U32BIT event);
static void AVSetAudioVolumes(BOOLEAN save_to_nvm);

static void SetupForTuning(U8BIT path, U32BIT event);
static void StartTune(U8BIT path, S_MANUAL_TUNING_PARAMS *tuning_params);
static void SetupForDecoding(U8BIT path);
static void StartSiProcess(U8BIT path);

static U8BIT AcquirePathForTransport(void *t_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info);
static U8BIT AcquirePathForService(void *s_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info);

static void ContinueServiceSearch(U8BIT path, U32BIT event);
static U16BIT NextSearchId(void **tlist, U16BIT num_transports);
static void ReleasePath(U8BIT path);
static void EndSearch(U8BIT path);
static void ContinueTransportSearch(U8BIT path);
static void FinishManualSearch(U8BIT path, U32BIT event);
static void FinishSearch(U8BIT path);
static BOOLEAN StartEitSearch(void);
static void ContinueEitSearch(U8BIT path);
static BOOLEAN EitSearchNeededNow(void);

static U16BIT GetNumSatSearchIds(void);
static void MapIdToSatTuneParams(U16BIT search_id, S_MANUAL_TUNING_PARAMS *tuning_params);

static BOOLEAN EnterStandby(void **recording_service);
static void GetNextWakeupTime(U32DHMS *wakeup_date_time, BOOLEAN *wakeup_for_search,
   E_STB_DP_SIGNAL_TYPE *search_tuner_type, BOOLEAN *wakeup_for_recording, U16BIT *onet_id,
   U16BIT *trans_id, U16BIT *service_id);
static BOOLEAN StartSearch(E_STB_DP_SIGNAL_TYPE tuner_type);
static U32DHMS GetDateTimeNow(void);
static U32DHMS CalculateWakeupTime(U32DHMS date_time);

#if defined(DEBUG_TUNING) || defined(DEBUG_DECODING) || defined(DEBUG_AV_OUTPUT) || defined(DEBUG_MHEG)
static U8BIT* GetEventDebugString(E_ACTL_EVENT_TYPE type, U32BIT event);
#endif

static void FinishEitSearch(U8BIT path);
static BOOLEAN GetNextEitSearchTime(U32DHMS *next);

static void UpdateTransportParameters(U8BIT path);
static U32BIT CreateTimer(U32DHMS date_time);

static void nitTableUpdate(U8BIT path, SI_NIT_TABLE *nit_table, SI_TABLE_RECORD *table_rec);
static void untTableUpdate(U8BIT path, SI_UNT_TABLE *unt_table, SI_TABLE_RECORD *table_rec);

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Control system initialisation
 */
void ACTL_InitialiseAppControl(void)
{
   U8BIT num_paths;
   U8BIT path;

   FUNCTION_START(ACTL_InitialiseAppControl);

   if (!initialised)
   {
      num_paths = STB_DPGetNumPaths();

      user_tuning_params = STB_AppGetMemory(sizeof(S_MANUAL_TUNING_PARAMS) * num_paths);
      tuning_state = STB_AppGetMemory(sizeof(E_ACTL_TUNING_STATE) * num_paths);
      tuned_status = STB_AppGetMemory(sizeof(E_ACTL_TUNED_STATUS) * num_paths);
      decoding_state = STB_AppGetMemory(sizeof(E_ACTL_DECODING_STATE) * num_paths);
      decoding_started = STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
      decoding_locked = STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
      decode_paused_flag = STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
      cntrl_mutex = STB_AppGetMemory(sizeof(void *) * num_paths);
      now_event_id = STB_AppGetMemory(sizeof(S32BIT) * num_paths);

      current_satellite = STB_AppGetMemory(sizeof(void *) * num_paths);
      current_transport = STB_AppGetMemory(sizeof(void *) * num_paths);
      current_service = STB_AppGetMemory(sizeof(ADB_SERVICE_REC *) * num_paths);
      current_si = STB_AppGetMemory(sizeof(E_APP_SI_MODE) * num_paths);

      transport_required = STB_AppGetMemory(sizeof(void *) * num_paths);
      service_required = STB_AppGetMemory(sizeof(ADB_SERVICE_REC *) * num_paths);
      si_required = STB_AppGetMemory(sizeof(E_APP_SI_MODE) * num_paths);

      dvb_subtitles = FALSE;
      subtitles_started = FALSE;
      subtitles_running = FALSE;
      subtitles_suppressed = FALSE;

      // do av control first because it turns on front panel led
      ControlAvOutput(INTERNAL_EVENT, INT_EVENT_INITIALISE);

      for (path = 0; path < num_paths; path++)
      {
         cntrl_mutex[path] = STB_OSCreateMutex();
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_INITIALISE);
      }

      for (path = 0; path < num_paths; path++)
      {
         ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_INITIALISE);
      }

      ACTL_EnableCiModule();

      ASTE_Initialise();

      GetSubtitlePidFunc = ADB_ServiceGetSubtitlePid;
      
      ASI_SetUpdateUntFunction((F_UntTableUpdate)untTableUpdate);

      initialised = TRUE;
   }

   FUNCTION_FINISH(ACTL_InitialiseAppControl);
}

/**
 * @brief   Actions external events
 * @param   event the event to be handled
 */
void ACTL_ActionEvent(U32BIT event, void *event_data)
{
   U8BIT path;
   U32BIT handle;
   U32DHMS search_time;
#ifdef INCLUDE_OTA_SSU
   U16BIT onet_id, tran_id, serv_id;
   void *t_ptr;
#endif

   FUNCTION_START(ACTL_ActionEvent);

   path = INVALID_RES_ID;

   switch (event)
   {
      // tuning events
      case STB_EVENT_TUNE_SIGNAL_DATA_BAD:
      case STB_EVENT_TUNE_SIGNAL_DATA_OK:
      case STB_EVENT_TUNE_LOCKED:
      {
         /* The path passed in these events is actually the tuner number so
          * this needs to be converted back to the associated path, but a tuner
          * can be associated with multiple paths so handle the event for all. */
         if (event_data != NULL)
         {
            path = INVALID_RES_ID;
            while ((path = STB_DPPathForTuner(path, *(U8BIT *)event_data)) != INVALID_RES_ID)
            {
               ControlTuning(path, EXTERNAL_EVENT, event);

#ifdef INCLUDE_OTA_SSU
               if ((current_search_type == SEARCH_TYPE_SSU) ||
                  (current_search_type == SEARCH_TYPE_OTA_UPDATE))
               {
                  DBG_SSU(("%s: Tuner locked, OTA started = %u", __FUNCTION__, ota_settings.ota_started));
                  if (!ota_settings.ota_started && (ota_settings.ota_tran_id != ADB_INVALID_DVB_ID))
                  {
                     DBG_SSU(("%s: Tuner locked, looking for OTA image on 0x%04x/%04x/%04x",
                        __FUNCTION__, ota_settings.ota_onet_id, ota_settings.ota_tran_id,
                        ota_settings.ota_serv_id));

                     if (STB_OTAStartLoader(path, ota_settings.ota_onet_id, ota_settings.ota_tran_id,
                        ota_settings.ota_serv_id, ADB_INVALID_DVB_ID, ota_settings.version_cb))
                     {
                        ota_settings.ota_started = TRUE;
                     }
#ifdef DEBUG_SSU_SEARCH
                     else
                     {
                        DBG_SSU(("%s: Tuner locked but failed to start OTA", __FUNCTION__));
                     }
#endif
                  }
               }
               else if  (current_search_type == SEARCH_TYPE_SSU_DOWNLOAD) {
                  if (STB_EVENT_TUNE_LOCKED == event) {
                      ACTL_StartSSUDownload(path);
                  }
               }
#endif /* INCLUDE_OTA_SSU */
            }
         }

         if (current_search_type != SEARCH_TYPE_NOSEARCH)
         {
            /* Send event to allow the UI to update */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
         }

         if (((current_search_type == SEARCH_TYPE_EIT_PF) ||
            (current_search_type == SEARCH_TYPE_EIT_SCHED) ||
            (current_search_type == SEARCH_TYPE_EIT_PF_SCHED)) && (ACFG_GetEitUpdateTime() != 0))
         {
            /* All the EIT events should be collected within a certain time (depending on the country),
               set a timer to stop the EIT search */
            EIT_DBG(("ACTL_ActionEvent: EIT search timeout after %u mins", ACFG_GetEitUpdateTime()));
            search_time = STB_GCCalculateDHMS(STB_GCNowDHMSGmt(),
                  DHMS_CREATE(0, 0, ACFG_GetEitUpdateTime(), 0), CALC_ADD);
            eit_search_end_timer = CreateTimer(search_time);
         }
         break;
      }

      case STB_EVENT_TUNE_NOTLOCKED:
      {
         /* The path passed in these events is actually the tuner number so
          * this needs to be converted back to the associated path, but a tuner
          * can be associated with multiple paths so handle the event for all. */
         if (event_data != NULL)
         {
            path = INVALID_RES_ID;
            while ((path = STB_DPPathForTuner(path, *(U8BIT *)event_data)) != INVALID_RES_ID)
            {
               switch (current_search_type)
               {
                  case SEARCH_TYPE_SERVICE_FREQ:
                  case SEARCH_TYPE_SERVICE_NETWORK:
                  case SEARCH_TYPE_MANUAL_NETWORK:
                     ContinueServiceSearch(path, event);
                     break;

#ifdef INCLUDE_OTA_SSU
                  case SEARCH_TYPE_SSU:
                     if (ota_settings.ota_started)
                     {
                        DBG_SSU(("%s: Tuner not locked, stopping OTA download", __FUNCTION__));
                        STB_OTAStopLoader();
                        ota_settings.ota_started = FALSE;

                        if (ota_settings.ota_path != INVALID_RES_ID)
                        {
                           DBG_SSU(("%s: Releasing path %u used for OTA", __FUNCTION__,
                              ota_settings.ota_path));

                           ACTL_TuneOff(ota_settings.ota_path);
                           STB_DPReleasePath(ota_settings.ota_path, RES_OWNER_NONE);
                           ota_settings.ota_path = INVALID_RES_ID;
                        }
                     }
                     ContinueTransportSearch(path);
                     break;

                  case SEARCH_TYPE_OTA_UPDATE:
                     if (ota_settings.ota_started)
                     {
                        DBG_SSU(("%s: Tuner not locked, stopping OTA download", __FUNCTION__));
                        STB_OTAStopLoader();
                        ota_settings.ota_started = FALSE;

                        if (ota_settings.ota_path != INVALID_RES_ID)
                        {
                           DBG_SSU(("%s: Releasing path %u used for OTA", __FUNCTION__,
                              ota_settings.ota_path));

                           ACTL_TuneOff(ota_settings.ota_path);
                           STB_DPReleasePath(ota_settings.ota_path, RES_OWNER_NONE);
                           ota_settings.ota_path = INVALID_RES_ID;
                        }
                     }
                     current_search_type = SEARCH_TYPE_NOSEARCH;
                     break;
#endif /* INCLUDE_OTA_SSU */

                  case SEARCH_TYPE_STARTUP:
                  case SEARCH_TYPE_TOT:
                     ContinueTransportSearch(path);
                     break;

                  case SEARCH_TYPE_EIT_PF:
                  case SEARCH_TYPE_EIT_SCHED:
                  case SEARCH_TYPE_EIT_PF_SCHED:
                     if (path == search_path)
                     {
                        ContinueEitSearch(path);
                     }
                     break;

                  case SEARCH_TYPE_MANUAL_FREQ:
                     FinishManualSearch(path, event);
                     ControlTuning(path, EXTERNAL_EVENT, event);
                     break;

                  default:
                     ControlTuning(path, EXTERNAL_EVENT, event);
                     break;
               }
            }
         }
         break;
      }

      case STB_EVENT_SEARCH_FAIL:
      case STB_EVENT_SEARCH_SUCCESS:
      {
         switch (current_search_type)
         {
            case SEARCH_TYPE_SERVICE_FREQ:
            case SEARCH_TYPE_SERVICE_NETWORK:
            case SEARCH_TYPE_MANUAL_NETWORK:
               if (search_path != INVALID_RES_ID)
               {
                  ContinueServiceSearch(search_path, event);
               }
               break;

            case SEARCH_TYPE_TOT:
               if (search_path != INVALID_RES_ID)
               {
                  if (event == STB_EVENT_SEARCH_SUCCESS)
                  {
                     /* Once we have the first TOT, no need to continue */
                     FinishSearch(search_path);
                  }
                  else
                  {
                     ContinueTransportSearch(search_path);
                  }
               }
               break;

            case SEARCH_TYPE_STARTUP:
               if (search_path != INVALID_RES_ID)
               {
                  ContinueTransportSearch(search_path);
               }
               break;

            case SEARCH_TYPE_MANUAL_FREQ:
               if (search_path != INVALID_RES_ID)
               {
                  if ((signal_type_required == SIGNAL_COFDM) &&
                      (ACTL_GetTerRfChanType(search_list_id) == TERR_TYPE_DVBT2))
                  {
                     /* T2 may have multiple PLPs, so need to continue the search */
                     ContinueServiceSearch(search_path, event);
                  }
                  else if ((signal_type_required == SIGNAL_QAM) &&
                     (SEARCH_TYPE_MANUAL_NETWORK == required_search_type))
                  {
#if 1
                     FinishManualSearch(search_path, event);
                     search_path = INVALID_RES_ID;
                     break;
#else
                     // clean up the empty transports (from BAT, ... etc.)
                     ADB_TRANSPORT_REC *t1_ptr = DBDEF_GetNextTransportRec(NULL);
                     while (t1_ptr != NULL) {
                         ADB_TRANSPORT_REC *t2_ptr = DBDEF_GetNextTransportRec(t1_ptr);
                         if (0 == t1_ptr->frequency) {
                             DBDEF_DeleteTransportRec(t1_ptr);
                         }
                         t1_ptr = t2_ptr;
                     }
                     // begin to search NIT/user's transport list
                     ContinueServiceSearch(search_path, event);
#endif
                  }
                  else
                  {
                     FinishManualSearch(search_path, event);
                     search_path = INVALID_RES_ID;
                  }
               }
               break;

            case SEARCH_TYPE_EIT_PF:
            case SEARCH_TYPE_EIT_SCHED:
            case SEARCH_TYPE_EIT_PF_SCHED:
               if (search_path != INVALID_RES_ID)
               {
                  /* Finished, continue on the next transport */
                  ContinueEitSearch(search_path);
               }
               break;

#ifdef INCLUDE_OTA_SSU
            case SEARCH_TYPE_SSU:
               if (event == STB_EVENT_SEARCH_SUCCESS)
               {
                  ota_settings.ota_location = ASI_GetNextOTALocation(ADB_GetTunedNetwork(search_path),
                     NULL, &onet_id, &tran_id, &serv_id);

                  if ((ota_settings.ota_location != NULL) &&
                     ((t_ptr = ADB_GetTransportFromIds(ADB_INVALID_DVB_ID, onet_id, tran_id)) != NULL))
                  {
                     DBG_SSU(("%s(%u): Looking for OTA image on 0x%04x/%04x/%04x", __FUNCTION__,
                        search_path, onet_id, tran_id, serv_id));

                     if (!ACTL_StartOTAUpdate(search_path, t_ptr, serv_id, ota_settings.version_cb))
                     {
                        DBG_SSU(("%s: Failed to start OTA, continuing with search", __FUNCTION__));
                        ContinueTransportSearch(search_path);
                     }
                  }
                  else
                  {
                     #ifdef INCLUDE_HWACOM_SSU
                     if (TRUE == unt_settings.valid) {
                        // notification has been sent, just stop searching
                        // the upper layer has to start the SSU OTA download and update
                     }
                     else                     
                     #endif
                     {
                     DBG_SSU(("%s: No OTA available, continuing with search", __FUNCTION__));
                     ContinueTransportSearch(search_path);
                     }
                  }
               }
               else
               {
                  DBG_SSU(("%s: No OTA available, continuing with search", __FUNCTION__));
                  ContinueTransportSearch(search_path);
               }
               break;
#endif

            default:
               break;
         }
         break;
      }

#ifdef INCLUDE_OTA_SSU
      case STB_EVENT_OTA_SW_UPGRADE_DOWNLOADING:
      {
         break;
      }
      
      case STB_EVENT_OTA_SW_UPGRADE_FOUND:
      {
         /* Nothing to be done, just pass the event on to the app */
         STB_OTAContinueDownload(TRUE);
         break;
      }

      case STB_EVENT_OTA_SW_UPGRADE_NOTFOUND:
      case STB_EVENT_OTA_SW_UPGRADE_ERROR:
      {
         if ((current_search_type == SEARCH_TYPE_SSU) || (current_search_type == SEARCH_TYPE_OTA_UPDATE))
         {
            DBG_SSU(("%s: Failed to find an OTA update", __FUNCTION__));
            ACTL_StopOTAUpdate();

            if ((current_search_type == SEARCH_TYPE_SSU) && (search_path != INVALID_RES_ID))
            {
               /* Try the next OTA location on this transport */
               ota_settings.ota_location = ASI_GetNextOTALocation(ADB_GetTunedNetwork(search_path),
                  ota_settings.ota_location, &onet_id, &tran_id, &serv_id);

               if ((ota_settings.ota_location != NULL) &&
                  ((t_ptr = ADB_GetTransportFromIds(ADB_INVALID_DVB_ID, onet_id, tran_id)) != NULL))
               {
                  DBG_SSU(("%s(%u): Looking for OTA image on 0x%04x/%04x/%04x", __FUNCTION__,
                     search_path, onet_id, tran_id, serv_id));

                  if (!ACTL_StartOTAUpdate(search_path, t_ptr, serv_id, ota_settings.version_cb))
                  {
                     DBG_SSU(("%s: Failed to start OTA, continuing with search", __FUNCTION__));
                     ContinueTransportSearch(search_path);
                  }
               }
               else
               {
                  DBG_SSU(("%s: Continuing SSU search", __FUNCTION__));
                  ContinueTransportSearch(search_path);
               }
            }
         }
         break;
      }
#endif

      // decoding events
      case STB_EVENT_AUDIO_DECODE_STARTED:
      {
         /* The path passed in these events is the decoder number
          * so this needs to be converted back to the associated path */
         if (event_data != NULL)
         {
            path = STB_DPPathForAudioDecoder(*(U8BIT *)event_data);
         }

         if (path != INVALID_RES_ID)
         {
            ControlDecoding(path, EXTERNAL_EVENT, event);
         }
         break;
      }

      case STB_EVENT_VIDEO_DECODE_STARTED:
      case STB_EVENT_DECODE_LOCKED:
      {
         /* The path passed in these events is the decoder number
          * so this needs to be converted back to the associated path */
         if (event_data != NULL)
         {
            path = STB_DPPathForVideoDecoder(*(U8BIT *)event_data);
         }

         if (path != INVALID_RES_ID)
         {
            ControlDecoding(path, EXTERNAL_EVENT, event);
         }
         break;
      }

#ifdef COMMON_INTERFACE
      case STB_EVENT_CI_APP_INFO:
      {
         ci_ai_module = *((U32BIT*)event_data);
         break;
      }
#endif

      case APP_EVENT_SERVICE_STREAMS_CHANGED:
      {
         void * event_serv = *(void **)event_data;
         /* The service may be used by more than one path, so need to check them all */
         for (path = 0; path < STB_DPGetNumPaths(); path++)
         {
            if (ADB_GetTunedService(path) == event_serv)
            {
               if (STB_DPIsDecodingPath(path))
               {
                  /* PIDs for the decoding have changed */
               }
               else if (STB_DPIsMonitoringPath(path))
               {
                  /* PIDs for the monitoring have changed */
                  if (search_path != path) {
                     SetupForDecoding(path);
                  }
                  else {
                  }
               }
               else if (STB_DPIsRecordingPath(path))
               {
                  /* PIDs for the recording have changed */
                  APVR_PidsUpdated(path);
               }
            }
         }
         break;
      }

      case APP_EVENT_SERVICE_AUDIO_PID_UPDATE:
      case APP_EVENT_SERVICE_VIDEO_PID_UPDATE:
      case APP_EVENT_SERVICE_SUBTITLE_UPDATE:
      case APP_EVENT_SERVICE_SCRAMBLE_CHANGE:
      case APP_EVENT_SERVICE_RUNNING:
      {
         /* All these events contain the service they apply to in the event data,
          * but this service may be valid for more than one path so they all need to be checked */
         for (path = 0; path < STB_DPGetNumPaths(); path++)
         {
            if (ADB_GetTunedService(path) == event_data)
            {
               STB_OSMutexLock(cntrl_mutex[path]);

               /* Initialise variables for playback */
               if ((event_data == APVR_GetPlaybackService()) && (path != INVALID_RES_ID))
               {
                  service_required[path] = (void *)APVR_GetPlaybackService();
                  current_service[path] = service_required[path];
                  tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL;
               }

               if (STB_DPIsDecodingPath(path))
               {
                  ControlDecoding(path, EXTERNAL_EVENT, event);
               }
               if (STB_DPIsRecordingPath(path) && !STB_DPIsRecording(path, &handle))
               {
                  /* The recording can only be started if the PIDs for the service are known */
                  if ((ADB_GetServiceAudioPid(event_data) != 0) ||
                      (ADB_GetServiceVideoPid(event_data) != 0))
                  {
                     ATMR_CheckRecordStatus(TRUE, event_data);
                  }
                  else
                  {
                     /* Recording can't be started with no PIDs */
                     ATMR_RecordingFailed(path);
                     STB_DPReleasePath(path, RES_OWNER_DVB);

                     /* Send event to UI to inform a recording has failed due to no data */
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);
                  }
               }

               STB_OSMutexUnlock(cntrl_mutex[path]);
            }
         }
         break;
      }

      case APP_EVENT_SERVICE_VIDEO_CODEC_CHANGED:
      {
         if (!external_control)
         {
            path = STB_DPGetPathForService(event_data);
            if (STB_DPIsDecodingPath(path))
            {
               STB_DPSetVideoCodec(path, ADB_GetVideoCodecFromStream(ADB_GetServiceVideoType(event_data)));
               ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_DECODING);
            }
         }
         break;
      }

      case APP_EVENT_SERVICE_AUDIO_CODEC_CHANGED:
      {
         if (!external_control)
         {
            path = STB_DPGetPathForService(event_data);
            if (STB_DPIsDecodingPath(path))
            {
               STB_DPSetAudioCodec(path, ADB_GetAudioCodecFromStream(ADB_GetServiceAudioType(event_data)));
               ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_AUDIO);
            }
         }
         break;
      }

      case APP_EVENT_SERVICE_NOT_RUNNING:
      {
         path = *(U8BIT *)event_data;
         if (STB_DPIsDecodingPath(path))
         {
            ControlDecoding(path, EXTERNAL_EVENT, event);
         }
         break;
      }

      default:
      {
         break;
      }
   }

   FUNCTION_FINISH(ACTL_ActionEvent);
}

/**
 * @brief   Handles all the private timer events
 * @param   timer_handle timer handle
 * @return  TRUE if the timer event has been handled, FALSE otherwise
 */
BOOLEAN ACTL_HandlePrivateTimerEvent(U32BIT timer_handle)
{
   BOOLEAN retval = FALSE;
   U32DHMS wakeup;
   U16BIT onet_id, trans_id, service_id;
   void *s_ptr;

   FUNCTION_START(ACTL_HandlePrivateTimerEvent);

   if (ATMR_GetType(timer_handle) == TIMER_TYPE_PRIVATE)
   {
      if (timer_handle == standby_wakeup_timer)
      {
         ATMR_DeleteTimer(standby_wakeup_timer);
         standby_wakeup_timer = INVALID_TIMER_HANDLE;

         /* Check what's to be started by this timer */
         if (APP_NvmRead(STANDBY_STATE_NVM) == STDBY_WAKE_FOR_RECORDING)
         {
            /* Find the details for the first wakeup timer to be started */
            if (ATMR_GetFirstWakeupTime(&wakeup, &onet_id, &trans_id, &service_id) != INVALID_TIMER_HANDLE)
            {
               if ((s_ptr = ADB_FindServiceByIds(onet_id, trans_id, service_id)) != NULL)
               {
                  if (standby_path != INVALID_RES_ID)
                  {
                     /* Release the decode path that was acquired when entering standby */
                     STB_DPReleasePath(standby_path, RES_OWNER_NONE);
                     standby_path = INVALID_RES_ID;
                  }

                  if ((standby_path = AcquirePathForService(s_ptr, FALSE, FALSE, NULL)) != INVALID_RES_ID)
                  {
                     ACTL_TuneToService(standby_path, NULL, s_ptr, FALSE, TRUE);
                  }
               }
               else
               {
                  /* Couldn't find the service to be recorded so go back into standby */
                  ACTL_EnterStandby();
               }
            }
            else
            {
               /* No recording timer found, go back into standby */
               ACTL_EnterStandby();
            }
         }
         else
         {
            if (!StartSearch(signal_type_required))
            {
               /* No search started so re-enter standby */
               ACTL_EnterStandby();
            }
         }

         retval = TRUE;
      }
      else if (timer_handle == standby_grace_timer)
      {
         ATMR_DeleteTimer(standby_grace_timer);
         standby_grace_timer = INVALID_TIMER_HANDLE;

         DBG_STDBY(("Standby grace time expired"));

         if ((BOOLEAN)APP_NvmRead(STANDBY_POWERSAVE_NVM))
         {
            /* Put box into low-power standby mode */
            DBG_STDBY(("Entering low power standby"));
            STB_HWSetStandbyState(HW_STANDBY_LOWPOWER);
         }
         else
         {
            /* Nothing is being started so go into the lowest active standby state */
            DBG_STDBY(("Entering passive standby"));
            STB_HWSetStandbyState(HW_STANDBY_PASSIVE);
         }
      }
      else
      {
         if (timer_handle == eit_search_end_timer)
         {
            ATMR_DeleteTimer(eit_search_end_timer);
            eit_search_end_timer = INVALID_TIMER_HANDLE;

            if ((current_search_type == SEARCH_TYPE_EIT_PF) ||
               (current_search_type == SEARCH_TYPE_EIT_SCHED) ||
               (current_search_type == SEARCH_TYPE_EIT_PF_SCHED))
            {
               if (search_path != INVALID_RES_ID)
               {
                  ContinueEitSearch(search_path);
               }
            }

            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(ACTL_HandlePrivateTimerEvent);

   return retval;
}

/**
 * @brief   Returns the current satellite being used by the given decode path
 * @param   path decode path
 * @return  Pointer to satellite
 */
void* ACTL_GetCurrentSatellite(U8BIT path)
{
   void *sat_ptr;

   FUNCTION_START(ACTL_GetCurrentSatellite);

   if (path != INVALID_RES_ID)
   {
      sat_ptr = current_satellite[path];
   }
   else
   {
      sat_ptr = NULL;
   }

   FUNCTION_FINISH(ACTL_GetCurrentSatellite);

   return(sat_ptr);
}

/**
 * @brief   Returns the number of entries in the rf channel table
 * @param   tuner_type tuner type
 * @return  number of entries in the channel table for the present country
 */
U16BIT ACTL_GetNumRfChanArrayEntries(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   ACFG_ANA_RF_CHANNEL_DATA *ana_channel_table;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetNumRfChanArrayEntries);

   num_rf_chans = 0;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_ANALOG:
         ACFG_GetAnaRfChannelTable(country_code, &ana_channel_table, &num_rf_chans);
         break;
      case SIGNAL_COFDM:
         ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans);
         break;
      case SIGNAL_QAM:
         ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans);
         break;
      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetNumRfChanArrayEntries);
   return(num_rf_chans);
}

/**
 * @brief   Returns a pointer to the channel name
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  pointer to the const name string
 */
U8BIT* ACTL_GetRfChanName(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id)
{
   U8BIT *ret_val;
   ACFG_ANA_RF_CHANNEL_DATA *ana_channel_table;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetRfChanName);

   ret_val = NULL;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_ANALOG:
         if (ACFG_GetAnaRfChannelTable(country_code, &ana_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = ana_channel_table[id].name;
         }
         break;
      case SIGNAL_COFDM:
         if (ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = ter_channel_table[id].name;
         }
         break;
      case SIGNAL_QAM:
         if (ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = cab_channel_table[id].name;
         }
         break;
      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetRfChanName);
   return(ret_val);
}

/**
 * @brief   Returns the channel symbol rate
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  symbol rate of selected id
 */
U16BIT ACTL_GetRfSymbolRate(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id)
{
   U16BIT ret_val;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetRfSymbolRate);

   ret_val = 0;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_QAM:
         if (ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = cab_channel_table[id].symbol_rate;
         }
         break;

      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetRfSymbolRate);
   return(ret_val);
}

/**
 * @brief   Returns the modulation mode
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  Modulation mode (E_STB_DP_TMODE or E_STB_DP_CMODE depending on the tuner type
 */
U8BIT ACTL_GetRfModulation(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id)
{
   U8BIT ret_val;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetRfModulation);

   ret_val = 0;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_QAM:
         if (ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = cab_channel_table[id].mode;
         }
         break;

      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetRfModulation);
   return(ret_val);
}

/**
 * @brief   Returns a pointer to the channel name
 * @param   tuner_type tuner type
 * @param   id index in the channel table for the current country
 * @return  pointer to the const name string
 */
U32BIT ACTL_GetRfChanFreqHz(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT id)
{
   U32BIT ret_val;
   ACFG_ANA_RF_CHANNEL_DATA *ana_channel_table;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetRfChanFreqHz);

   ret_val = 0;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_ANALOG:
         if (ACFG_GetAnaRfChannelTable(country_code, &ana_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = ana_channel_table[id].freq_hz;
         }
         break;
      case SIGNAL_COFDM:
         if (ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = ter_channel_table[id].freq_hz;
         }
         break;
      case SIGNAL_QAM:
         if (ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans) &&
             (id < num_rf_chans))
         {
            ret_val = cab_channel_table[id].freq_hz;
         }
         break;
      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetRfChanFreqHz);
   return(ret_val);
}

/**
 * @brief   Returns the rf name appropriate to the frequency specified
 * @param   tuner_type tuner type
 * @param   freq_hz frequency in hz
 * @return  pointer to the const name string, or NULL if not found
 */
U8BIT* ACTL_GetRfNameFromFreq(E_STB_DP_SIGNAL_TYPE tuner_type, U32BIT freq_hz)
{
   U8BIT *ret_val;
   ACFG_ANA_RF_CHANNEL_DATA *ana_channel_table;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT i, num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_GetRfNameFromFreq);

   ret_val = NULL;
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_ANALOG:
         if (ACFG_GetAnaRfChannelTable(country_code, &ana_channel_table, &num_rf_chans))
         {
            for (i = 0; i < num_rf_chans; i++)
            {
               if (ana_channel_table[i].freq_hz == freq_hz)
               {
                  ret_val = ana_channel_table[i].name;
                  break;
               }
            }
         }
         break;
      case SIGNAL_COFDM:
         if (ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans))
         {
            for (i = 0; i < num_rf_chans; i++)
            {
               if (ter_channel_table[i].freq_hz == freq_hz)
               {
                  ret_val = ter_channel_table[i].name;
                  break;
               }
            }
         }
         break;
      case SIGNAL_QAM:
         if (ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans))
         {
            for (i = 0; i < num_rf_chans; i++)
            {
               if (cab_channel_table[i].freq_hz == freq_hz)
               {
                  ret_val = cab_channel_table[i].name;
                  break;
               }
            }
         }
         break;
      default:
         break;
   }

   FUNCTION_FINISH(ACTL_GetRfNameFromFreq);
   return(ret_val);
}

/**
 * @brief   Returns the terrestrial type (T or T2) for the given channel id
 * @param   id index in the terrestrial channel table for the current country
 * @param   terrestrial type for the channel
 */
E_STB_DP_TTYPE ACTL_GetTerRfChanType(U16BIT id)
{
   E_STB_DP_TTYPE ret_val;
   BOOLEAN success;
   ACFG_TER_RF_CHANNEL_DATA *rf_channel_table;
   U16BIT num_rf_chans;

   FUNCTION_START(ACTL_GetTerRfChanType);

   ret_val = TERR_TYPE_UNKNOWN;

   success = ACFG_GetTerRfChannelTable(ACFG_GetCountry(), &rf_channel_table, &num_rf_chans);
   if ((success == TRUE) && (id < num_rf_chans))
   {
      ret_val = rf_channel_table[id].type;
   }

   FUNCTION_FINISH(ACTL_GetTerRfChanType);

   return(ret_val);
}

/**
 * @brief   Returns the current signal strength
 * @param   path decode path
 * @return  the signal strength as percentage of maximum (0-100)
 */
U8BIT ACTL_GetSignalStrength(U8BIT path)
{
    U8BIT tuner;
    U8BIT strength = 0;

    FUNCTION_START(ACTL_GetSignalStrength);

    tuner = STB_DPGetPathTuner(path);
    if (tuner != INVALID_RES_ID) {
        strength = STB_TuneGetSignalStrength(tuner);
    }

    FUNCTION_FINISH(ACTL_GetSignalStrength);

    return strength;
}

/**
 * @brief   Returns the Bit Error Rate
 * @param   path decode path
 * @return  bit_error_ratio (BER) in 1.0E-7(i.e. 10^-7) unit
 */
U32BIT ACTL_GetSignalBitErrorRate(U8BIT path)
{
    U8BIT tuner;
    U32BIT bit_error_rate = 0;

    FUNCTION_START(ACTL_GetSignalBitErrorRate);

    tuner = STB_DPGetPathTuner(path);
    if (tuner != INVALID_RES_ID) {
        bit_error_rate = STB_TuneGetBer(tuner);
    }

    FUNCTION_FINISH(ACTL_GetSignalBitErrorRate);
 
    return bit_error_rate;
}

/**
 * @brief   Returns the Signal Noise Ratio
 * @param   path decode path
 * @return  the unit of snr is 0.1dB
 */
U32BIT ACTL_GetSignalNoiseRatio(U8BIT path)
{
    U8BIT tuner;
    U32BIT signal_noise_ratio = 0;

    FUNCTION_START(ACTL_GetSignalNoiseRatio);

    tuner = STB_DPGetPathTuner(path);
    if (tuner != INVALID_RES_ID) {
        signal_noise_ratio = STB_TuneGetSnr(tuner);
    }

    FUNCTION_FINISH(ACTL_GetSignalNoiseRatio);
  
    return signal_noise_ratio;
}

/**
 * @brief   Tunes to the specified rf channel array entry for DVB-T and DVB-C
 * @param   path decode path
 * @param   id index in the current channel table (for the current signal type and country)
 * @param   reqd_si the type of SI handling required
 * @param   relock_on if FALSE auto relock will be turned off before tuning, otherwise it will be on
 * @return  TRUE if id valid, FALSE otherwise
 */
BOOLEAN ACTL_TuneToRfChanArrayEntry(U8BIT path, U16BIT id, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on)
{
   BOOLEAN ret_val;
   BOOLEAN success;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   U32BIT country_code;

   FUNCTION_START(ACTL_TuneToRfChanArrayEntry);

   ret_val = FALSE;
   country_code = ACFG_GetCountry();

   switch (signal_type_required)
   {
      case SIGNAL_COFDM:
         success = ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans);
         break;
      case SIGNAL_QAM:
         success = ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans);
         break;
      default:
         num_rf_chans = 0;
         success = FALSE;
         break;
   }

   ADB_SetTunedTransport(path, NULL);

   if ((success == TRUE) && (id < num_rf_chans))
   {
      STB_OSMutexLock(cntrl_mutex[path]);
      ret_val = TRUE;
      rf_chan_id_required = id;
      transport_required[path] = NULL;
      tuner_relock_required = relock_on;
      service_required[path] = NULL;
      decode_lock_override_required = FALSE;
      switch (reqd_si)
      {
         case ACTL_SI_CHANNEL_SEARCH:        {si_required[path] = APP_SI_MODE_CHANNEL_SEARCH; break; }
         case ACTL_SI_STARTUP_SEARCH:        {si_required[path] = APP_SI_MODE_STARTUP_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SEARCH:       {si_required[path] = APP_SI_MODE_EVENT_PF_SEARCH; break; }
         case ACTL_SI_EVENT_SCHED_SEARCH:    {si_required[path] = APP_SI_MODE_EVENT_SCHED_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SCHED_SEARCH: {si_required[path] = APP_SI_MODE_EVENT_PF_SCHED_SEARCH; break; }
         case ACTL_SI_DVB_SSU_SEARCH:        {si_required[path] = APP_SI_MODE_DVB_SSU_SEARCH; break; }
         case ACTL_SI_USER_DEFINED:          {si_required[path] = APP_SI_MODE_USER_DEFINED; break; }
         default:                            {si_required[path] = APP_SI_MODE_NO_SI; break; }
      }
      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_RF_CHAN_TUNE_REQD);
      STB_OSMutexUnlock(cntrl_mutex[path]);
   }
   else
   {
      ret_val = FALSE;
      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
   }

   FUNCTION_FINISH(ACTL_TuneToRfChanArrayEntry);

   return(ret_val);
}

/**
 * @brief   Tunes to the specified rf channel array entry in analogue mode
 * @param   path decode path
 * @brief   id index in the analogue channel table for the current country
 * @return  TRUE if id valid, FALSE otherwise
 */
BOOLEAN ACTL_TuneToRfChanArrayAnalogEntry(U8BIT path, U16BIT id)
{
   BOOLEAN ret_val;
   BOOLEAN success;
   ACFG_ANA_RF_CHANNEL_DATA *rf_channel_table;
   U16BIT num_rf_chans;

   FUNCTION_START(ACTL_TuneToRfChanArrayAnalogEntry);

   ret_val = FALSE;
   success = ACFG_GetAnaRfChannelTable(ACFG_GetCountry(), &rf_channel_table, &num_rf_chans);

   if ((success == TRUE) && (id < num_rf_chans))
   {
      STB_OSMutexLock(cntrl_mutex[path]);
      ret_val = TRUE;
      rf_chan_id_required = id;
      signal_type_required = SIGNAL_ANALOG;
      transport_required[path] = NULL;
      tuner_relock_required = FALSE;
      service_required[path] = NULL;
      decode_lock_override_required = FALSE;
      si_required[path] = APP_SI_MODE_NO_SI;

      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_RF_CHAN_TUNE_REQD);
      STB_OSMutexUnlock(cntrl_mutex[path]);
   }
   else
   {
      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
   }

   ADB_SetTunedTransport(path, NULL);

   FUNCTION_FINISH(ACTL_TuneToRfChanArrayAnalogEntry);
   return(ret_val);
}

/**
 * @brief   Tunes to the given set of tuning parameters
 * @param   path decode path
 * @param   tuning_params tuning parameters to be used
 * @param   reqd_si SI search mode to be used if tuning is successful
 * @param   relock_on defines whether the tuner is to attempt to relock if lock is lost
 * @return  TRUE if tuning is started, FALSE otherwise
 */
BOOLEAN ACTL_TuneToUserDefinedParams(U8BIT path, S_MANUAL_TUNING_PARAMS *tuning_params,
   E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on)
{
   BOOLEAN ret_val;

   FUNCTION_START(ACTL_TuneToUserDefinedParams);

   ret_val = FALSE;

   if ((path != INVALID_RES_ID) && (tuning_params != NULL))
   {
      STB_OSMutexLock(cntrl_mutex[path]);

      ADB_SetTunedTransport(path, NULL);

      transport_required[path] = NULL;
      tuner_relock_required = relock_on;
      service_required[path] = NULL;
      decode_lock_override_required = FALSE;

      switch (reqd_si)
      {
         case ACTL_SI_CHANNEL_SEARCH:        {si_required[path] = APP_SI_MODE_CHANNEL_SEARCH; break; }
         case ACTL_SI_CHANNEL_SEARCH_NO_NIT: {si_required[path] = APP_SI_MODE_CHANNEL_SEARCH_NO_NIT; break; }
         case ACTL_SI_STARTUP_SEARCH:        {si_required[path] = APP_SI_MODE_STARTUP_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SEARCH:       {si_required[path] = APP_SI_MODE_EVENT_PF_SEARCH; break; }
         case ACTL_SI_EVENT_SCHED_SEARCH:    {si_required[path] = APP_SI_MODE_EVENT_SCHED_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SCHED_SEARCH: {si_required[path] = APP_SI_MODE_EVENT_PF_SCHED_SEARCH; break; }
         case ACTL_SI_DVB_SSU_SEARCH:        {si_required[path] = APP_SI_MODE_DVB_SSU_SEARCH; break; }
         case ACTL_SI_USER_DEFINED:          {si_required[path] = APP_SI_MODE_UPDATE; break; }
         default:                            {si_required[path] = APP_SI_MODE_NO_SI; break; }
      }

      signal_type_required = STB_DPGetSignalType(path);

      /* Save the tuning params */
      memcpy(&user_tuning_params[path], tuning_params, sizeof(S_MANUAL_TUNING_PARAMS));

      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_USERDEFINED_TUNE_REQD);

      STB_OSMutexUnlock(cntrl_mutex[path]);

      ret_val = TRUE;
   }

   FUNCTION_FINISH(ACTL_TuneToUserDefinedParams);

   return(ret_val);
}

/**
 * @brief   Tunes to the given transport and sets the type of SI monitoring that
 *          will be started when the tuning completes.
 * @param   path decode path, use INVALID_RES_ID to allow the DVB to acquire an appropriate path
 * @param   owner_info owner info for the path, can be NULL (used by CI+)
 * @param   t_ptr transport to tune to
 * @param   reqd_si type of SI monitoring to be started when tuned
 * @param   relock_on TRUE if the tuner is to attempt to relock if the signal is lost
 * @return  decode path that is being used to tune to the transport.
 *          Returning INVALID_RES_ID means the path couldn't be acquired immediately and
 *          doesn't mean that it has failed
 */
U8BIT ACTL_TuneToTransport(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *t_ptr,
   E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on)
{
   FUNCTION_START(ACTL_TuneToTransport);

   if (path == INVALID_RES_ID)
   {
      /* Try to acquire a live path */
      path = AcquireLivePath(NULL, owner_info, FALSE, t_ptr, reqd_si, relock_on);
   }

   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);

      transport_required[path] = t_ptr;
      tuner_relock_required = relock_on;
      service_required[path] = NULL;
      decode_lock_override_required = FALSE;

      switch (reqd_si)
      {
         case ACTL_SI_CHANNEL_SEARCH:        {si_required[path] = APP_SI_MODE_CHANNEL_SEARCH; break; }
         case ACTL_SI_CHANNEL_SEARCH_NO_NIT: {si_required[path] = APP_SI_MODE_CHANNEL_SEARCH_NO_NIT; break; }
         case ACTL_SI_STARTUP_SEARCH:        {si_required[path] = APP_SI_MODE_STARTUP_SEARCH; break; }
         case ACTL_SI_TOT_SEARCH:            {si_required[path] = APP_SI_MODE_TOT_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SEARCH:       {si_required[path] = APP_SI_MODE_EVENT_PF_SEARCH; break; }
         case ACTL_SI_EVENT_SCHED_SEARCH:    {si_required[path] = APP_SI_MODE_EVENT_SCHED_SEARCH; break; }
         case ACTL_SI_EVENT_PF_SCHED_SEARCH: {si_required[path] = APP_SI_MODE_EVENT_PF_SCHED_SEARCH; break; }
         case ACTL_SI_DVB_SSU_SEARCH:        {si_required[path] = APP_SI_MODE_DVB_SSU_SEARCH; break; }
         case ACTL_SI_USER_DEFINED:          {si_required[path] = APP_SI_MODE_USER_DEFINED; break; }
         default:                            {si_required[path] = APP_SI_MODE_NO_SI; break; }
      }

      ADB_SetTunedTransport(path, t_ptr);

      if (t_ptr != NULL)
      {
         STB_SPDebugWrite("Transport frequency: %uHz", ((ADB_TRANSPORT_REC *)t_ptr)->frequency);
         signal_type_required = ADB_GetTransportSignalType(t_ptr);
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TPTR_TUNE_REQD);
      }
      else
      {
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
      }

      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_TuneToTransport);

   return(path);
}

/**
 * @brief   Starts the process of tuning to the specified service. If the service is to be
 *          tuned on the live path, then path should be passed as INVALID_RES_ID to ensure
 *          that any CI+ resource handling is performed.
 * @param   path decode path, INVALID_RES_ID for the live path
 * @param   owner_info owner module requesting the tune, can be NULL (used by CI+)
 * @param   s_ptr required service
 * @param   override_lock TRUE if parental lock settings are to be ignored.
 * @param   for_live FALSE for recording, TRUE otherwise
 * @return  decode path that is being used to tune to the service.
 *          Returning INVALID_RES_ID means the path couldn't be acquired immediately and
 *          doesn't mean that it has failed
 */
U8BIT ACTL_TuneToService(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *s_ptr,
   BOOLEAN override_lock, BOOLEAN for_live)
{
   BOOLEAN slot_acquired;
   U32BIT handle;
#if defined(EIT_EPG_4DAY_ACT) || defined(EIT_EPG_8DAY_ACT)
   void **slist;
   U16BIT num_services;
   U16BIT i;
#endif
#if defined(INTEGRATE_HBBTV)
   U16BIT onet_id, ts_id, serv_id;
#endif

   FUNCTION_START(ACTL_TuneToService);

#ifdef DEBUG_TUNING
   STB_SPDebugWrite("ACTL_TuneToService(path=%d, owner_info=%p, service=0x%x, for_live=%u)",
      path, owner_info, s_ptr, for_live);
#endif

   if (path == INVALID_RES_ID)
   {
      if (for_live)
      {
         /* Try to acquire a live path */
         path = AcquireLivePath(s_ptr, owner_info, override_lock, NULL, APP_SI_MODE_UPDATE, FALSE);
      }
      else
      {
         /* Try to acquire a recording path */
         path = AcquireRecordingPath(s_ptr, owner_info);
         // FIXME: reserve path 'zero' for playback (for realtek)
         if (0 == path) {
            path = AcquireRecordingPath(s_ptr, owner_info);
            STB_DPReleasePath(0, RES_OWNER_DVB);
         }
      }
   }

   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);

      service_required[path] = s_ptr;
      transport_required[path] = ADB_GetServiceTransportPtr(s_ptr);
      tuner_relock_required = TRUE;
      decode_lock_override_required = override_lock;

      if (s_ptr != NULL)
      {
         if (ADB_GetServiceType(s_ptr) == ADB_SERVICE_TYPE_ANALOG)
         {
            si_required[path] = APP_SI_MODE_NO_SI;
            signal_type_required = SIGNAL_ANALOG;

            if ((APP_NvmRead(PARENTAL_LOCK_NVM) != PARENTAL_LOCK_OFF) &&
                (decode_lock_override_required == FALSE) &&
                (ADB_GetServiceLockedFlag(s_ptr) == TRUE))
            {
               ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_LOCKED, NULL, 0);
            }
            else
            {
               ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TPTR_TUNE_REQD);
            }
         }
#ifdef COMMON_INTERFACE
         else if (ADB_GetServiceType(s_ptr) == ADB_SERVICE_TYPE_VIRTUAL)
         {
            /* When on Virtual Channel, CAM is not expected to receive any TS data. So tune off previous channel */
            service_required[path] = NULL;
            transport_required[path] = NULL;
            current_service[path] = NULL;
            ADB_SetTunedService(path, NULL);
            ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);

            /* Inform CI-CAM that user has selected the 'Virtual Channel' */
            STB_EnterCICamChannel(ci_ai_module);
         }
#endif
         else
         {
#if defined(EIT_EPG_4DAY_ACT) || defined(EIT_EPG_8DAY_ACT)
            if (STB_DPIsLivePath(path) &&
                (current_transport[path] != NULL) &&
                (transport_required[path] != current_transport[path]) &&
                (s_ptr != current_service[path]))
            {
               /* Get the list of services on this transport so that all event data can be deleted */
               ADB_GetTransportServiceList(current_transport[path], &slist, &num_services);

               for (i = 0; i < num_services; i++)
               {
                  ADB_DeleteServiceEvents(slist[i], FALSE);
               }

               ADB_ReleaseServiceList(slist, num_services);
            }
#endif

            si_required[path] = APP_SI_MODE_UPDATE;
            slot_acquired = FALSE;

#ifdef COMMON_INTERFACE
            if (ADB_GetServiceSDTReceived(s_ptr))
            {
               /* Check whether the SDT needs to be reacquired for this service due to the
                * retention limit of the CI protection descriptor expiring */
               if (ADB_HasCIProtectionExpired(s_ptr))
               {
                  /* The SDT for this service needs to be reacquired to update
                   * the CI protection descriptor */
                  si_required[path] = APP_SI_MODE_CIPLUS_UPDATE;
               }
               else
               {
                  /* Only acquire a CI slot if the path doesn't have one yet */
                  if (STB_DPGetPathCISlot(path) == INVALID_RES_ID)
                  {
                     slot_acquired = ACI_AcquireCISlot(path, s_ptr);
                  }
               }
               if (!slot_acquired)
               {
                  /* Acquire a CA slot if needed */
                  ACA_AcquireCADescrambler(path, s_ptr);
               }
            }
            else
            {
               /* The SDT needs to be acquired for the service before a CAM can be acquired */
               si_required[path] = APP_SI_MODE_CIPLUS_UPDATE;
            }
#else
            if (!slot_acquired)
            {
               /* Acquire a CA slot if needed */
               ACA_AcquireCADescrambler(path, s_ptr);
            }
#endif

            signal_type_required = ADB_GetTransportSignalType(transport_required[path]);

            /* Is tuning required? */
            if (ADB_GetTunedTransport(path) != transport_required[path])
            {
               current_service[path] = (ADB_SERVICE_REC *)s_ptr;
               ADB_SetTunedService(path, s_ptr);
               ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TPTR_TUNE_REQD);
            }
            else
            {
               tuning_state[path] = TSTATE_TUNED;
               current_service[path] = (ADB_SERVICE_REC *)s_ptr;
               ADB_SetTunedService(path, s_ptr);

               /* Start SI processing */
               StartSiProcess(path);

               if (STB_DPIsDecodingPath(path))
               {
                  ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_DECODING);
#if defined(INTEGRATE_HBBTV)
                  if (STB_DPIsLivePath(path))
                  {
                     /* Inform the presentation engine that the channel change succeeded */
                     ADB_GetServiceIds(s_ptr, &onet_id, &ts_id, &serv_id);
                     HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                        HBBTV_CHANNEL_CHANGE_SUCCEEDED);
                  }
#endif
               }

               if (STB_DPIsRecordingPath(path) && !STB_DPIsRecording(path, &handle))
               {
                  /* The recording can only be started here if the PIDs for the service
                   * are known, otherwise it will have to start when the PMT for the service
                   * has been received. Only the audio PID is checked as this should be
                   * available for both radio and TV services */
                  if (ADB_GetServiceAudioPid(ATMR_GetRecordService(path)) != 0)
                  {
                     ATMR_CheckRecordStatus(TRUE, s_ptr);
                  }
               }
            }
         }
      }
      else
      {
         current_service[path] = NULL;
         ADB_SetTunedService(path, s_ptr);
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
      }

      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_TuneToService);

   return(path);
}

/**
 * @brief   Starts the process of tuning to a given transport or service that's
 *          defined by the given delivery system descriptor. If the DSD defines
 *          a transport that doesn't currently exist, then one will be created,
 *          and if a service_id is given and a service with this ID doesn't exist
 *          on the transport, then one will be created, but will be marked as hidden
 *          and non-selectable so it won't appear to a user.
 * @param   path decode path to be tuned, must be valid
 * @param   dsd_type delivery system descriptor type (T, C or S)
 * @param   dsd delivery system descriptor
 * @param   service_id pass as 0 if just tuning to a transport
 * @param   reqd_si SI processing mode that should be selected when tuned.
 *          If tuning to a service the normal UPDATE mode will be used
 *          and this setting will be ignored.
 * @return  TRUE if the tune operation is started, FALSE otherwise
 */
BOOLEAN ACTL_TuneUsingDSD(U8BIT path, SI_DELIVERY_SYS_DESC_TYPE dsd_type,
   SI_DELIVERY_SYS_DESC *dsd, U16BIT service_id, E_ACTL_SI_SRCH_REQD reqd_si)
{
   BOOLEAN retval;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ACTL_TuneUsingDSD);

   retval = FALSE;

   if (path != INVALID_RES_ID)
   {
      DBDEF_RequestAccess();

      switch (dsd_type)
      {
         case SI_DEL_SYS_DESC_TYPE_TERR:
         {
            if ((t_ptr = DBDEF_FindTerrestrialTransportRec(dsd->terr.u.t1.freq_hz, 0)) == NULL)
            {
               t_ptr = DBDEF_AddTerrestrialTransportRec(dsd->terr.u.t1.freq_hz, 0, NULL);
            }
            break;
         }

         case SI_DEL_SYS_DESC_TYPE_CABLE:
         {
            if ((t_ptr = DBDEF_FindCableTransportRec(dsd->cable.freq_hz,
                       dsd->cable.symbol_rate)) == NULL)
            {
               t_ptr = DBDEF_AddCableTransportRec(dsd->cable.freq_hz, dsd->cable.symbol_rate, NULL);
            }
            break;
         }

         case SI_DEL_SYS_DESC_TYPE_SAT:
         {
            if ((t_ptr = DBDEF_FindSatTransportRec(dsd->sat.freq_hz, dsd->sat.sym_rate,
                       dsd->sat.polarity, dsd->sat.dvb_s2, dsd->sat.modulation, NULL)) == NULL)
            {
               t_ptr = DBDEF_AddSatTransportRec(dsd->sat.freq_hz, dsd->sat.sym_rate,
                     dsd->sat.polarity, dsd->sat.dvb_s2, dsd->sat.modulation, NULL);
            }
            break;
         }

         default:
         {
            t_ptr = NULL;
            break;
         }
      }

      DBDEF_ReleaseAccess();

      if (t_ptr != NULL)
      {
         if (service_id != 0)
         {
            DBDEF_RequestAccess();

            if ((s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                       t_ptr->tran_id, service_id)) == NULL)
            {
               /* Create a service record */
               if ((s_ptr = DBDEF_AddServiceRec(service_id, t_ptr)) != NULL)
               {
                  /* Make the service hidden and non-selectable so it isn't available to the user */
                  s_ptr->hidden = TRUE;
                  DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                  s_ptr->selectable = FALSE;
                  DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                  s_ptr->serv_type = ADB_SERVICE_TYPE_DSD;
                  DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

                  ADB_SaveDatabase();
               }
            }

            DBDEF_ReleaseAccess();

            if (s_ptr != NULL)
            {
               ACTL_TuneToService(path, NULL, s_ptr, FALSE, TRUE);
               retval = TRUE;
            }
         }
         else
         {
            ACTL_TuneToTransport(path, NULL, t_ptr, reqd_si, TRUE);
            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(ACTL_TuneUsingDSD);

   return(retval);
}

/**
 * @brief   Returns whether the given path is currently tuned
 * @param   path decode path
 * @return  TRUE if tuned, FALSE otherwise
 */
BOOLEAN ACTL_IsTuned(U8BIT path)
{
   BOOLEAN tuned;

   FUNCTION_START(ACTL_IsTuned);

   tuned = FALSE;

   if ((path != INVALID_RES_ID) && (tuning_state[path] == TSTATE_TUNED))
   {
      tuned = TRUE;
   }

   FUNCTION_FINISH(ACTL_IsTuned);

   return(tuned);
}

/**
 * @brief   Starts the SI to acquire the PMT and fill the ip service in
 */
void ACTL_SetupPlayback(void)
{
   U8BIT path;

   path = STB_DPGetPlaybackPath();
   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);

      current_si[path] = APP_SI_MODE_NO_SI;
      si_required[path] = APP_SI_MODE_STARTUP_SEARCH;
      service_required[path] = NULL;
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_ENABLE);
      StartSiProcess(path);
      tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL; /* Pretend we have a good signal */
      /* The state machine inside ControlDecoding needs the followuing disable command to work properly
       * before to actually start decoding */
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);

      STB_OSMutexUnlock(cntrl_mutex[path]);
   }
}

/**
 * @brief   Start streaming the specified service
 * @param   s_ptr the pointer to the service structure containing all the information
 */
void ACTL_StartPlayback(void *s_ptr)
{
   U8BIT path;
   FUNCTION_START(ACTL_StartPlayback);

   //DBG_STR("ACTL_StartPlayback(service=0x%x)\n", s_ptr);

   path = STB_DPGetPlaybackPath();
   if (path != INVALID_RES_ID)
   {
      service_required[path] = s_ptr;
      transport_required[path] = NULL;
      if (s_ptr != NULL)
      {
         si_required[path] = APP_SI_MODE_UPDATE;
         signal_type_required = SIGNAL_COFDM;
         current_service[path] = (ADB_SERVICE_REC *)s_ptr;
         STB_DPStopSI(path); /* We don't need to get the SI updates */
      }
      else
      {
         current_service[path] = NULL;
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
      }
   }

   FUNCTION_FINISH(ACTL_StartPlayback);
}

/**
 * @brief   Sets up and starts decoding for the given service.
 *          This is used for PVR playback, but could also be used to start decoding
 *          for other non-broadcast services.
 * @param   path decode path
 * @param   s_ptr service
 */
void ACTL_StartDecoding(U8BIT path, void *s_ptr)
{
   FUNCTION_START(ACTL_StartDecoding);

   if ((path != INVALID_RES_ID) && STB_DPIsDecodingPath(path))
   {
      STB_OSMutexLock(cntrl_mutex[path]);
      current_service[path] = s_ptr;
      service_required[path] = s_ptr;
      tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL;
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_ENABLE);
      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_StartDecoding);
}

/**
 * @brief   Stops tuning on the given path
 * @param   path decode path
 */
void ACTL_TuneOff(U8BIT path)
{
   FUNCTION_START(ACTL_TuneOff);

   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);
      transport_required[path] = NULL;
      service_required[path] = NULL;
      ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
      current_service[path] = NULL;
      current_transport[path] = NULL;
      current_satellite[path] = NULL;
      ADB_SetTunedTransport(path, NULL);
      ADB_SetTunedService(path, NULL);
      now_event_id[path] = -1;
      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_TuneOff);
}

/**
 * @brief   Stops decoding on the given path
 * @param   path decode path
 */
void ACTL_DecodeOff(U8BIT path)
{
   FUNCTION_START(ACTL_DecodeOff);

   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);
      service_required[path] = NULL;
      if (STB_DPIsDecodingPath(path))
      {
         ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
      }
      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_DecodeOff);
}

/**
 * @brief   Releases the lock on a channel after decoding has been blocked due
 *          to the service being parental locked, after which decoding will be started.
 */
void ACTL_ReleaseChannelLock(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_ReleaseChannelLock);

   path = ACTL_GetActivePath();
   if (path != INVALID_RES_ID)
   {
      if (signal_type_required == SIGNAL_ANALOG)
      {
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TPTR_TUNE_REQD);
      }
      else
      {
         ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RELEASE_DECODE_LOCK);
      }
   }

   FUNCTION_FINISH(ACTL_ReleaseChannelLock);
}

/**
 * @brief   Stops and restarts audio decoding on the live path. This may be required
 *          due to a change in language preferences, or some other audio setting.
 */
void ACTL_ReTuneAudio(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_ReTuneAudio);

   path = ACTL_GetActivePath();

   if ((path != INVALID_RES_ID) && STB_DPIsDecodingPath(path))
   {
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_AUDIO);
   }

   FUNCTION_FINISH(ACTL_ReTuneAudio);
}

/**
 * @brief   Stops and restarts subtitle decoding on the live path. This may be required
 *          due to a change in language preferences, or some other setting.
 */
void ACTL_ReTuneSubtitles(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_ReTuneSubtitles);

   path = ACTL_GetActivePath();
   if (path != INVALID_RES_ID)
   {
      SUBT_DBG(("ACTL_ReTuneSubtitles: calling ControlDecoding with INT_EVENT_RESTART_SUBTITLES"));
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_SUBTITLES);
   }

   FUNCTION_FINISH(ACTL_ReTuneSubtitles);
}

/**
 * @brief   Returns whether decoding has been started on the given path
 * @param   path decode path
 * @return  TRUE if decoding has started, FALSE otherwise
 */
BOOLEAN ACTL_HasDecodingStarted(U8BIT path)
{
   BOOLEAN started;
   FUNCTION_START(ACTL_HasDecodingStarted);
   if (path == INVALID_RES_ID)
   {
      started = FALSE;
   }
   else
   {
      started = decoding_started[path];
   }
   FUNCTION_FINISH(ACTL_HasDecodingStarted);
   return started;
}

/**
 * @brief   Returns whether decoding is locked, due to parental locking, on the given path
 * @param   path decode path
 * @return  TRUE if decoding is locked, FALSE otherwise
 */
BOOLEAN ACTL_IsDecodingLocked(U8BIT path)
{
   BOOLEAN locked;
   FUNCTION_START(ACTL_IsDecodingLocked);
   if (path == INVALID_RES_ID)
   {
      locked = FALSE;
   }
   else
   {
      locked = decoding_locked[path];
   }
   FUNCTION_FINISH(ACTL_IsDecodingLocked);
   return locked;
}

/**
 * @brief   Reports the standby state to the A/V output controller
 * @param   state TRUE for standby is on, FALSE for standby is off
 */
void ACTL_SetStandbyState(BOOLEAN state)
{
   E_ACTL_INTERNAL_EVENT event;

   FUNCTION_START(ACTL_SetStandbyState);

   if (state)
   {
      event = INT_EVENT_STANDBY_ON;
      ASTE_EnterStandby(FALSE);
   }
   else
   {
      event = INT_EVENT_STANDBY_OFF;
      ASTE_LeaveStandby();
   }

   ControlAvOutput(INTERNAL_EVENT, event);

   FUNCTION_FINISH(ACTL_SetStandbyState);
}

/**
 * @brief   Reports standby state to the a/v output control state machine
 */
void ACTL_SetStandbyVCRActive(void)
{
   FUNCTION_START(ACTL_SetStandbyVCRActive);

   ControlAvOutput(INTERNAL_EVENT, INT_EVENT_STANDBY_VCR_ACTIVE);

   FUNCTION_FINISH(ACTL_SetStandbyVCRActive);
}

#if 0
/**
 *

 *
 * @brief   Reports scart input allowed/not allowed to the a/v output control state machine
 *
 * @param   state - TRUE for scart input allowed, FALSE for not allowed
 *

 *
 */
void ACTL_AllowAltAVSources(BOOLEAN state)
{
   E_ACTL_INTERNAL_EVENT event;

   FUNCTION_START(ACTL_AllowAltAVSources);

   if (state == TRUE)
   {
      event = INT_EVENT_ALT_AV_ALLOWED;
   }
   else
   {
      event = INT_EVENT_ALT_AV_NOT_ALLOWED;
   }
   ControlAvOutput(INTERNAL_EVENT, event);

   FUNCTION_FINISH(ACTL_AllowAltAVSources);
}

#endif

/**
 * @brief   Sets the video window to the size specified. Coordinates are relative
 *          to the screen resolution
 * @param   win_x window X position
 * @param   win_y window Y position
 * @param   win_width window width (0 for full screen)
 * @param   win_height window height (0 for full screen)
 */
void ACTL_SetVideoWindow(S16BIT win_x, S16BIT win_y, U16BIT win_width, U16BIT win_height)
{
   S_RECTANGLE output;

   FUNCTION_START(ACTL_SetVideoWindow);

   if (win_width == 0 || win_height == 0)
   {
      STB_VTSetVideoOutput(NULL);
   }
   else
   {
      output.left = win_x;
      output.top = win_y;
      output.width = win_width;
      output.height = win_height;
      STB_VTSetVideoOutput(&output);
   }

   FUNCTION_FINISH(ACTL_SetVideoWindow);
}

/**
 * @brief   Used to set the TV aspect ratio
 * @param   aspect_ratio TV aspect ratio, e.g. 4:3
 */
void ACTL_SetTVAspectRatio(E_STB_AV_ASPECT_RATIO aspect_ratio)
{
   E_STB_AV_VIDEO_FORMAT actual_mode;
   U16BIT width, height;

   FUNCTION_START(ACTL_SetTVAspectRatio);

   /* Save the setting */
   APP_NvmSave(ASPECT_RATIO_NVM, aspect_ratio, TRUE);

   actual_mode = ACTL_GetActualVideoMode(&width, &height);

   STB_AVSetTVType(0, aspect_ratio, actual_mode);

   FUNCTION_FINISH(ACTL_SetTVAspectRatio);
}

/**
 * @brief   Returns the current TV aspect ratio
 * @return  aspect ratio
 */
E_STB_AV_ASPECT_RATIO ACTL_GetTVAspectRatio(void)
{
   FUNCTION_START(ACTL_GetTVAspectRatio);
   FUNCTION_FINISH(ACTL_GetTVAspectRatio);
   return((E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM));
}

/**
 * @brief   Used to set the TV aspect mode which defines how the video will be
 *          displayed based on the aspect ratio of the TV and video, along with
 *          some other factors.
 * @param   aspect_mode TV aspect mode, e.g. letterbox
 */
void ACTL_SetTVAspectMode(E_STB_AV_ASPECT_MODE aspect_mode)
{
   FUNCTION_START(ACTL_SetTVAspectMode);

   /* Save the setting */
   APP_NvmSave(ASPECT_MODE_NVM, aspect_mode, TRUE);

   STB_VTSetVideoAlignmentPref(aspect_mode);

   FUNCTION_FINISH(ACTL_SetTVAspectMode);
}

/**
 * @brief   Returns the current TV aspect mode
 * @return  aspect mode
 */
E_STB_AV_ASPECT_MODE ACTL_GetTVAspectMode(void)
{
   FUNCTION_START(ACTL_GetTVAspectMode);
   FUNCTION_FINISH(ACTL_GetTVAspectMode);
   return((E_STB_AV_ASPECT_MODE)APP_NvmRead(ASPECT_MODE_NVM));
}

/**
 * @brief   Used to set the aspect conversion applied to the video based on the
 *          TV aspect ratio and aspect mode to be applied. The given settings will
 *          also be saved.
 * @param   aspect_ratio TV aspect ratio, e.g. 4:3
 * @param   aspect_mode TV aspect mode, e.g. letterbox
 */
void ACTL_SetTVAspectConversion(E_STB_AV_ASPECT_RATIO aspect_ratio, E_STB_AV_ASPECT_MODE aspect_mode)
{
   E_STB_AV_VIDEO_FORMAT actual_mode;
   U16BIT width, height;

   FUNCTION_START(ACTL_SetTVAspectConversion);

   /* Save the settings */
   APP_NvmSave(ASPECT_RATIO_NVM, aspect_ratio, FALSE);
   APP_NvmSave(ASPECT_MODE_NVM, aspect_mode, TRUE);

   actual_mode = ACTL_GetActualVideoMode(&width, &height);

   STB_AVSetTVType(0, aspect_ratio, actual_mode);

   STB_VTSetVideoAlignmentPref(aspect_mode);

   FUNCTION_FINISH(ACTL_SetTVAspectConversion);
}

/**
 * @brief   Returns av_mode
 * @return  Scart AV status
 */
E_ACTL_AV_MODE ACTL_GetAvModeStatus(void)
{
   FUNCTION_START(ACTL_GetAvModeStatus);
   FUNCTION_FINISH(ACTL_GetAvModeStatus);
   return(av_mode);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   Starts the process of tuning to the specified service provided by CI+.
 *          The same rules apply as when tuning to a normal service using ACTL_TuneToService.
 * @param   path decode path, INVALID_RES_ID for the live path
 * @param   owner_info owner module requesting the tune, can be NULL
 * @param   s_ptr required service
 * @param   si_mode SI search mode to be started on successful tuning
 */
void ACTL_TuneToCIService(U8BIT path, S_ACTL_OWNER_INFO *owner_info, void *s_ptr, E_APP_SI_MODE si_mode)
{
   FUNCTION_START(ACTL_TuneToCIService);

#ifdef DEBUG_TUNING
   STB_SPDebugWrite("ACTL_TuneToCIService(path=%d, owner_info=%p, service=0x%x, si_mode=%u)",
      path, owner_info, s_ptr, si_mode);
#endif

   if (path == INVALID_RES_ID)
   {
      /* Try to acquire a live path */
      path = AcquireLivePath(s_ptr, owner_info, FALSE, NULL, si_mode, FALSE);
   }

   if (path != INVALID_RES_ID)
   {
      STB_OSMutexLock(cntrl_mutex[path]);

      service_required[path] = s_ptr;
      transport_required[path] = ADB_GetServiceTransportPtr(s_ptr);
      tuner_relock_required = TRUE;
      decode_lock_override_required = FALSE;

      if (s_ptr != NULL)
      {
         si_required[path] = si_mode;
         signal_type_required = ADB_GetTransportSignalType(transport_required[path]);

         /* If tuning to the current service then just start decoding, otherwise tune */
         if (current_service[path] != s_ptr)
         {
            current_service[path] = (ADB_SERVICE_REC *)s_ptr;
            ADB_SetTunedService(path, s_ptr);
            ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TPTR_TUNE_REQD);
         }
         else
         {
            /* Start SI processing */
            StartSiProcess(path);

            if (STB_DPIsDecodingPath(path))
            {
               ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_DECODING);
            }
         }
      }
      else
      {
         current_service[path] = NULL;
         ADB_SetTunedService(path, NULL);
         ControlTuning(path, INTERNAL_EVENT, INT_EVENT_TUNE_OFF);
      }

      STB_OSMutexUnlock(cntrl_mutex[path]);
   }

   FUNCTION_FINISH(ACTL_TuneToCIService);
}

/**
 * @brief   Called when a reply is received from the CI+ stack in response to the request
 *          for it to release control of the live path and tuner. If it has released the
 *          path, then tuning can continue as requested.
 * @param   path Path associated with the response
 * @param   continue_tuning TRUE if tuning should continue, FALSE otherwise
 */
void ACTL_ContinueCIPLUSTune(U8BIT path, BOOLEAN continue_tuning)
{
   FUNCTION_START(ACTL_ContinueCIPLUSTune);

   if (continue_tuning)
   {
      if (path != INVALID_RES_ID)
      {
         if (!STB_DPIsRecordingPath(path))
         {
            STB_DPReleasePath(path, RES_OWNER_CIPLUS);
            path = INVALID_RES_ID;
         }

         if (tuneto_for_recording)
         {
            path = AcquireRecordingPath(tuneto_live_service, tuneto_owner_info);
         }

         if (tuneto_live_service != NULL)
         {
            ACTL_TuneToService(path, tuneto_owner_info, tuneto_live_service,
               decode_lock_override_required, TRUE);
         }
         else if (tuneto_live_transport != NULL)
         {
            ACTL_TuneToTransport(path, tuneto_owner_info, tuneto_live_transport,
               tuneto_live_si, tuneto_relock);
         }
      }
   }

   /* Reset variables now that a reply has been received */
   tuneto_live_service = NULL;
   tuneto_live_transport = NULL;

   if (tuneto_owner_info != NULL)
   {
      /* Release the copy of the owner info */
      if (tuneto_owner_info->data != NULL)
      {
         STB_AppFreeMemory(tuneto_owner_info->data);
      }

      STB_AppFreeMemory(tuneto_owner_info);
      tuneto_owner_info = NULL;
   }

   FUNCTION_FINISH(ACTL_ContinueCIPLUSTune);
}

#endif

/**
 * @brief   Enables CI module
 */
void ACTL_EnableCiModule(void)
{
   FUNCTION_START(ACTL_EnableCiModule);
   #ifdef COMMON_INTERFACE
   STB_GCSetCIStandby(FALSE);
   #endif
   FUNCTION_FINISH(ACTL_EnableCiModule);
}

/**
 * @brief   Disables CI module
 */
void ACTL_DisableCiModule(void)
{
   FUNCTION_START(ACTL_DisableCiModule);
   #ifdef COMMON_INTERFACE
   STB_GCSetCIStandby(TRUE);
   #endif
   FUNCTION_FINISH(ACTL_DisableCiModule);
}

/**
 * @brief   Returns state of ci_ui_required flag
 * @return  TRUE if CI OPEN event has been received
 */
BOOLEAN ACTL_IsCiUiRequired(void)
{
   FUNCTION_START(ACTL_IsCiUiRequired);
   FUNCTION_FINISH(ACTL_IsCiUiRequired);
   return(ci_ui_request);
}

/**
 * @brief   Gets user paused flag value
 * @brief   path decode path
 * @return  TRUE if the subtitles are in paused mode, FALSE otherwise
 */
BOOLEAN ACTL_GetDecodePausedState(U8BIT path)
{
   FUNCTION_START(ACTL_GetDecodePausedState);
   FUNCTION_FINISH(ACTL_GetDecodePausedState);
   return(decode_paused_flag[path]);
}

/**
 *

 *
 * @brief   sends string to be displayed via pixelworks if it has changed (Must not send
 *                message to PixelWorks if there is no need - it is too slow)
 *
 * @param   U8BIT* str_ptr
 *

 *
 */
void ACTL_SetAnalogChanIdString(U8BIT *str_ptr)
{
#if 0
   U16BIT string_len;

   FUNCTION_START(ACTL_SetAnalogChanIdString);

   analog_string_changed = FALSE;

   if (str_ptr == NULL)
   {
      // no string required - if string was valid then it has changed
      if (valid_analog_string == TRUE)
      {
         valid_analog_string = FALSE;
         analog_string_changed = TRUE;
      }
   }
   else
   {
      // we do want to display a string - but has it changed?
      if (valid_analog_string == FALSE)
      {
         analog_string_changed = TRUE;
      }
      else
      {
         if (strncmp((char *)str_ptr, (char *)analog_string_buffer, (sizeof(analog_string_buffer) - 1)) != 0)
         {
            analog_string_changed = TRUE;
         }
      }

      if (analog_string_changed == TRUE)
      {
         // string changed - copy into string buffer...
         // terminate string if larger than analog buffer
         string_len = strlen((char *)str_ptr);
         if (string_len >= sizeof(analog_string_buffer))
         {
            string_len = sizeof(analog_string_buffer) - 1;
         }
         memcpy(analog_string_buffer, str_ptr, string_len);
         //replace last char with null terminator
         analog_string_buffer[string_len] = 0;
         valid_analog_string = TRUE;
      }
   }


   if (analog_string_changed == TRUE)
   {
      ControlStatusIndication(STB_DPGetLivePath());
   }

   FUNCTION_FINISH(ACTL_SetAnalogChanIdString);
#else
   USE_UNWANTED_PARAM(str_ptr);
#endif
}

/**
 * @brief   Enables or disables analog video display
 * @param   allow_analog_video if TRUE analog video is enabled, otherwise disabled
 */
void ACTL_AllowAnalogVideo(BOOLEAN allow_analog_video)
{
   E_ACTL_INTERNAL_EVENT event;

   FUNCTION_START(ACTL_AllowAnalogVideo);

   if (allow_analog_video == TRUE)
   {
      event = INT_EVENT_ANALOG_VIDEO_ALLOWED;
   }
   else
   {
      event = INT_EVENT_ANALOG_VIDEO_NOT_ALLOWED;
   }
   ControlAvOutput(INTERNAL_EVENT, event);

   FUNCTION_FINISH(ACTL_AllowAnalogVideo);
}

static void * GetStream(void *service, ADB_STREAM_LIST_TYPE stream_list_type, U16BIT pid);
static void * GetStream(void *service, ADB_STREAM_LIST_TYPE stream_list_type, U16BIT pid)
{
   void *stream = NULL;
   void **list;
   U16BIT length = 0;
   ADB_GetStreamList(service, stream_list_type, &list, &length);
   for (U16BIT i = 0; i < length; i++)
   {
      if (ADB_GetStreamPID(list[i]) == pid)
      {
         stream = list[i];
         list[i] = list[--length]; // Return value to be released with ReleaseStream
         break;
      }
   }
   ADB_ReleaseStreamList(list, length);

   return stream;
}

/**
 * @brief   the locked service
 */
void * g_locked_service = NULL;

/**
 * @brief   set up the locked service for the next tuning
 * @param   the parameters of the locked service
 */
void ACTL_SetLockedService(S_ACTL_SERVICE_PARAMS * serviceParams)
{
   if (NULL == serviceParams) {
      g_locked_service = NULL;
      return;
   }

   //if (SI_DEL_SYS_DESC_TYPE_CABLE == signal_type) {
      // EN 300 468
      // Table 34: Modulation scheme for cable
      E_STB_DP_CMODE cmode[] = {
         MODE_QAM_AUTO,
         MODE_QAM_16,
         MODE_QAM_32,
         MODE_QAM_64,
         MODE_QAM_128,
         MODE_QAM_256
      };
      U32BIT freq = serviceParams->frequency;
      E_STB_DP_CMODE mode = cmode[serviceParams->modulation];
      U32BIT symbol_rate = serviceParams->symbol_rate;

      DBDEF_RequestAccess();
      ADB_TRANSPORT_REC * t_ptr = DBDEF_FindCableTransportRec(freq, SYMBOL_RATE_AUTO);
      if (NULL == t_ptr) {
         // generate transport records
         t_ptr = DBDEF_AddCableTransportRec(freq, symbol_rate, NULL);
         if (NULL != t_ptr) 
             DBDEF_SetCableTransportMode(t_ptr, mode);
      }
      DBDEF_ReleaseAccess();

      if (NULL != t_ptr) {
         // find the service
         ADB_SERVICE_REC * s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr);
         while (s_ptr != NULL)
         {
            BOOLEAN found = TRUE;
            for (int i=0; i < serviceParams->componentNum; i++) {
               S_ACTL_COMPONENT * pComponent = &serviceParams->components[i];
               void * stream = GetStream(s_ptr, ADB_STREAM_LIST_ALL, pComponent->pid);
               if (NULL == stream) {
                  found = FALSE;
                  break;
               }            
            }
            if (TRUE == found) {
               break;
            }

            s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);
         }
         
         if (NULL == s_ptr) {
            DBDEF_RequestAccess();
            if ((s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                       t_ptr->tran_id, 0 /*ADB_INVALID_DVB_ID*/)) == NULL)
            {
               /* Create a service record */
               s_ptr = DBDEF_AddServiceRec(0, t_ptr);
            }
            
            if (s_ptr != NULL)
            {
               /* Make the service hidden and non-selectable so it isn't available to the user */
               s_ptr->hidden = TRUE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

               s_ptr->selectable = FALSE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

               s_ptr->serv_type = ADB_SERVICE_TYPE_TV;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);
               
               s_ptr->pcr_pid = serviceParams->pcr_pid;
               for (int i=0; i < serviceParams->componentNum; i++) {
                  S_ACTL_COMPONENT * pComponent = &serviceParams->components[i];
                  switch (pComponent->type) {
                     case 2:  // video
                        s_ptr->video_pid = pComponent->pid;
                        s_ptr->video_ecm_pid = pComponent->ecm_pid;
                        break;
                     
                     case 3: // audio
                        s_ptr->audio_pid = pComponent->pid;
                        s_ptr->audio_ecm_pid = pComponent->ecm_pid;
                        break;
                  }            
               }
               s_ptr->pmt_received = TRUE;

               ADB_SaveDatabase();
            }
            DBDEF_ReleaseAccess();
         }

         g_locked_service = s_ptr;
      }
   //}  // if (SI_DEL_SYS_DESC_TYPE_CABLE == signal_type)
}

/**
 * @brief   reset the locked service
 * @param   none
 */
void * ACTL_GetLockedService(void)
{
   return g_locked_service;
}

/**
 * @brief   Returns whether audio description has been turned on
 * @return  TRUE if AD is on
 */
BOOLEAN ACTL_IsAudioDescriptionOn(void)
{
   FUNCTION_START(ACTL_IsAudioDescriptionOn);
   FUNCTION_FINISH(ACTL_IsAudioDescriptionOn);
   return(audio_desc_on);
}

/**
 * @brief   Starts decoding an audio description stream, if available, on the given path.
 *          If AD isn't currently available it will be marked as enabled and started
 *          when it becomes available.
 * @param   path decode path
 * @return  TRUE if AD is turned on, FALSE otherwise
 */
BOOLEAN ACTL_StartAudioDescription(U8BIT path)
{
   void *s_ptr;
   U16BIT pid;
   BOOLEAN broadcast_mix;
   E_STB_DP_AUDIO_MODE mode;
   ADB_STREAM_TYPE type;

   FUNCTION_START(ACTL_StartAudioDescription);

   if (path != INVALID_RES_ID)
   {
      if ((s_ptr = ADB_GetTunedService(path)) != NULL)
      {
         audio_desc_on = TRUE;
         STB_DPSetADEnabled(path, TRUE);

         DBDEF_RequestAccess();
         pid = DBDEF_GetReqdADPid(s_ptr, &mode, &type, &broadcast_mix);
         DBDEF_ReleaseAccess();

         if ((pid != 0) && (pid != 0xffff))
         {
            if (broadcast_mix)
            {
               /* Need to change the main audio PID */
               STB_DPSetAudioPID(path, pid);
               STB_DPSetEcmPIDs(
                  path,
                  ((ADB_SERVICE_REC *)s_ptr)->ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->video_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->audio_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->ttext_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->data_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->ad_ecm_pid
               );
 
               ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_AUDIO);
            }
            else
            {
               STB_DPSetADPID(path, pid);
               STB_DPSetEcmPIDs(
                  path,
                  ((ADB_SERVICE_REC *)s_ptr)->ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->video_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->audio_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->ttext_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->data_ecm_pid,
                  ((ADB_SERVICE_REC *)s_ptr)->ad_ecm_pid
               );
               STB_DPSetADMode(path, mode);
               STB_DPSetADCodec(path, ADB_GetAudioCodecFromStream(type));
               ACTL_SetADVolume((U8BIT)APP_NvmRead(AD_VOLUME_NVM));
               STB_DPStartADDecoding(path);
            }
         }
      }
   }

   FUNCTION_FINISH(ACTL_StartAudioDescription);

   return(audio_desc_on);
}

/**
 * @brief   Stops AD decoding
 * @param   path decode path
 */
void ACTL_StopAudioDescription(U8BIT path)
{
   FUNCTION_START(ACTL_StopAudioDescription);

   if (path != INVALID_RES_ID)
   {
      audio_desc_on = FALSE;
      STB_DPSetADEnabled(path, FALSE);
      STB_DPStopADDecoding(path);

      /* If AD is broadcast mix then main audio decoding needs to be changed */
      ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_AUDIO);
   }

   FUNCTION_FINISH(ACTL_StopAudioDescription);
}

/**
 * @brief   Sets the AD volume
 * @param   volume AD volume as a percentage
 */
void ACTL_SetADVolume(U8BIT volume)
{
   FUNCTION_START(ACTL_SetADVolume);

   ad_volume = volume;

   AVSetAudioVolumes(TRUE);

   FUNCTION_FINISH(ACTL_SetADVolume);
}

/**
 * @brief   Reads the saved video format and returns the best mode available if it's
 *          set to AUTO or is invalid for the currently connected TV.
 * @param   width returns the new video width if saved mode isn't valid, can be NULL
 * @param   height returns the new video height if saved mode isn't valid, can be NULL
 * @return  current video mode, or best video mode available
 */
E_STB_AV_VIDEO_FORMAT ACTL_GetActualVideoMode(U16BIT *width, U16BIT *height)
{
   E_STB_AV_VIDEO_FORMAT current_mode;
   E_STB_AV_VIDEO_FORMAT *supported_modes;
   U8BIT num_modes, num_app_modes;
   U8BIT i, j;
   BOOLEAN valid_mode;

   FUNCTION_START(ACTL_GetActualVideoMode);

   current_mode = (E_STB_AV_VIDEO_FORMAT)APP_NvmRead(HDMI_RESOLUTION_NVM);
   num_modes = STB_AVGetHDMISupportedModes(&supported_modes);
   if (num_modes > 0)
   {
      DBGPRINT("current mode %d", current_mode);
      /* Check whether the user selected mode is supported */
      valid_mode = FALSE;
      /* The list of supported modes passed back by platform can include
       * the AUTO - HW layer will know if the TV gives 'native' mode */
      for (i = 0; i != num_modes; i++)
      {
         DBGPRINT("supported %d", supported_modes[i]);
         if (supported_modes[i] == current_mode)
         {
            valid_mode = TRUE;
            hdmi_connected = TRUE;
            break;
         }
      }

      /* Select the best mode that is supported. It's assumed that the app modes
       * are ordered from best to worst */
      num_app_modes = sizeof(app_video_formats) / sizeof(app_video_formats[0]);

      for (j = 0; j != num_app_modes; j++)
      {
         if (app_video_formats[j].mode == VIDEO_FORMAT_AUTO)
         {
            STB_AVGetHDMINativeResolution(&(app_video_formats[j].width),
               &(app_video_formats[j].height));
         }
         if (!valid_mode)
         {
            DBGPRINT("try mode %d", app_video_formats[j].mode);
            for (i = 0; i != num_modes; i++)
            {
               if (app_video_formats[j].mode == supported_modes[i])
               {
                  DBGPRINT("select mode %d", supported_modes[i]);
                  /* This is the best mode */
                  current_mode = supported_modes[i];

               #ifdef DEBUG_AV_OUTPUT
                  STB_SPDebugWrite("ACTL_GetActualVideoMode: Best HDMI res is %ux%u",
                     app_video_formats[j].width, app_video_formats[j].height);
               #endif
                  valid_mode = TRUE;
                  break;
               }
            }
         }
         if (valid_mode && app_video_formats[j].mode == current_mode)
         {
            if (width != NULL)
            {
               *width = app_video_formats[j].width;
            }

            if (height != NULL)
            {
               *height = app_video_formats[j].height;
            }
            break;
         }
      }
   }
   else
   {
      /* Select the last app mode */
      num_app_modes = sizeof(app_video_formats) / sizeof(app_video_formats[0]);
      current_mode = app_video_formats[num_app_modes - 1].mode;
      *width = app_video_formats[num_app_modes - 1].width;
      *height = app_video_formats[num_app_modes - 1].height;
   }

   FUNCTION_FINISH(ACTL_GetActualVideoMode);

   return(current_mode);
}

/**
 * @brief   Update video mode sets MHEG resolution as well as platform
 *          Also, stops subtitles during the operation.
 *          If HDMI resolution mode is not supported, it chooses the best
 *          one available.
 * @param   aspect TV aspect ratio, e.g. 4:3 or 16:9
 * @param   force forces resolution update when TRUE
 */
void ACTL_UpdateVideoMode(E_STB_AV_ASPECT_RATIO aspect, BOOLEAN force)
{
   E_STB_AV_VIDEO_FORMAT current_mode;
   E_STB_AV_VIDEO_FORMAT actual_mode;
   U16BIT width, height;
   BOOLEAN subt_displayed;

   FUNCTION_START(ACTL_UpdateVideoMode);

   current_mode = (E_STB_AV_VIDEO_FORMAT)APP_NvmRead(HDMI_RESOLUTION_NVM);
   actual_mode = ACTL_GetActualVideoMode(&width, &height);
   if (actual_mode != current_mode || force)
   {
      DBGPRINT("mode=%d actual=%d width=%d,height=%d", current_mode, actual_mode, width, height);

      if (current_mode != VIDEO_FORMAT_AUTO && current_mode != actual_mode && ACTL_IsHDMIConnected())
      {
         /* Override the invalid user setting with the one selected */
         APP_NvmSave(HDMI_RESOLUTION_NVM, actual_mode, TRUE);
      }

      /* If necessary, stop subtitles before changing tv format */
      subt_displayed = ACTL_AreSubtitlesDisplayed();

      if (ACTL_AreSubtitlesStarted())
      {
         ACTL_StopSubtitles();
      }

      STB_AVSetTVType(0, aspect, actual_mode);

      /* If necessary re-start subtitles */
      if (subt_displayed)
      {
         ACTL_StartSubtitles();
      }
   }
   FUNCTION_FINISH(ACTL_UpdateVideoMode);
}

/**
 * @brief   Checks that the selected HDMI resolution mode is supported and, if not,
 *          chooses the best one available. This function is called by the event task when
 *          the HDMI connected event is received.
 */
void ACTL_HDMIConnected(void)
{
#ifdef COMMON_INTERFACE
   U8BIT path;
#endif

   FUNCTION_START(ACTL_HDMIConnected);

   DBGPRINT("%d", hdmi_connected);

   ACTL_UpdateVideoMode((E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), TRUE);

#ifdef COMMON_INTERFACE
   if (initialised && ((path = STB_DPGetLivePath()) != INVALID_RES_ID))
   {
      /* HDMI is reconnected, URI must be re-applied (HDCP may or may not be active). */
      ACI_UsageRulesStatusChanged(path);
   }
#endif

   hdmi_connected = TRUE;

   FUNCTION_FINISH(ACTL_HDMIConnected);
}

/**
 * @brief   Sets flag to indicate HDMI is now disconnected.
 *          This function is called by the event task when the HDMI disconnected event is received.
 */
void ACTL_HDMIDisconnected(void)
{
   FUNCTION_START(ACTL_HDMIDisconnected);
   DBGPRINT("%d", hdmi_connected);
   hdmi_connected = FALSE;
   FUNCTION_FINISH(ACTL_HDMIDisconnected);
}

/**
 * @brief   Returns whether the HDMI is connected or not.
 * @return  TRUE if the HDMI is connected, FALSE otherwise
 */
BOOLEAN ACTL_IsHDMIConnected(void)
{
   FUNCTION_START(ACTL_IsHDMIConnected);
   DBGPRINT("%d", hdmi_connected);
   FUNCTION_FINISH(ACTL_IsHDMIConnected);
   return(hdmi_connected);
}

/**
 * @brief   Returns an array of valid HDMI resolutions and the index of the
 *          currently selected format. The final entry in each array will be
 *          the auto video format.
 * @param   video_formats pointer to an array of video formats.
 * @param   current_index used to return index of currently selected video
 *          format
 * @return  Size of returned arrays, will always be at least 1
 */
U16BIT ACTL_GetHDMIResolutions(E_STB_AV_VIDEO_FORMAT **video_formats, U16BIT *current_index)
{
   E_STB_AV_VIDEO_FORMAT current_mode;
   E_STB_AV_VIDEO_FORMAT *supported_modes;
   U8BIT num_modes, num_app_modes;
   U16BIT num_valid_modes;
   U8BIT i, j;

   FUNCTION_START(ACTL_GetHDMIResolutionStrings);

   num_valid_modes = 0;

   current_mode = (E_STB_AV_VIDEO_FORMAT)APP_NvmRead(HDMI_RESOLUTION_NVM);
   num_modes = STB_AVGetHDMISupportedModes(&supported_modes);

   /* Find the number of modes supported by the app and stb */
   num_app_modes = sizeof(app_video_formats) / sizeof(app_video_formats[0]);

   for (j = 0; j != num_app_modes; j++)
   {
      for (i = 0; i != num_modes; i++)
      {
         if (app_video_formats[j].mode == supported_modes[i])
         {
            num_valid_modes++;
            break;
         }
      }
   }

   num_valid_modes++;  /* Adding AUTO */
   *video_formats = (E_STB_AV_VIDEO_FORMAT *)STB_AppGetMemory(num_valid_modes * sizeof(E_STB_AV_VIDEO_FORMAT));

   num_valid_modes = 0;

   for (j = 0; j != num_app_modes; j++)
   {
      for (i = 0; i != num_modes; i++)
      {
         if (app_video_formats[j].mode == supported_modes[i])
         {
            (*video_formats)[num_valid_modes] = app_video_formats[j].mode;

            if (current_mode == supported_modes[i])
            {
               *current_index = num_valid_modes;
            }
            num_valid_modes++;
            break;
         }
      }
   }

   /* Add AUTO */
   (*video_formats)[num_valid_modes] = VIDEO_FORMAT_AUTO;
   if (current_mode == VIDEO_FORMAT_AUTO)
   {
      *current_index = num_valid_modes;
   }
   num_valid_modes++;

   FUNCTION_FINISH(ACTL_GetHDMIResolutionStrings);

   return(num_valid_modes);
}

/**
 * @brief   Enables or disables parental control. This enables or disables
 *          locking on a per channel basis. If an age has previously been set,
 *          this will override the setting, effectively turning off age related
 *          blockign of channels.
 * @param   enabled TRUE if parental control is to be enabled
 */
void ACTL_SetParentalControl(BOOLEAN enabled)
{
   U8BIT setting;

   FUNCTION_START(ACTL_SetParentalControl);

   setting = (U8BIT)APP_NvmRead(PARENTAL_LOCK_NVM);

   if (enabled)
   {
      if (setting != PARENTAL_LOCK_ON)
      {
         APP_NvmSave(PARENTAL_LOCK_NVM, PARENTAL_LOCK_ON, TRUE);
      }
   }
   else
   {
      if (setting != PARENTAL_LOCK_OFF)
      {
         APP_NvmSave(PARENTAL_LOCK_NVM, PARENTAL_LOCK_OFF, TRUE);
      }
   }

   FUNCTION_FINISH(ACTL_SetParentalControl);
}

/**
 * @brief   Returns whether parental control is enabled. This will also return
 *          TRUE if parental control has been set to a valid age.
 * @return  TRUE if parental control is enabled, FALSE otherwise
 */
BOOLEAN ACTL_ParentalControlEnabled(void)
{
   BOOLEAN enabled;

   FUNCTION_START(ACTL_ParentalControlEnabled);

   if (APP_NvmRead(PARENTAL_LOCK_NVM) != PARENTAL_LOCK_OFF)
   {
      enabled = TRUE;
   }
   else
   {
      enabled = FALSE;
   }

   FUNCTION_FINISH(ACTL_ParentalControlEnabled);

   return(enabled);
}

/**
 * @brief   Sets the age (valid values 4-18) at which parental control will be
 *          will be applied. If the age is invalid, no change will be made to
 *          the current setting.
 * @param   age DVB SI age to be set
 */
void ACTL_SetParentalControlAge(U8BIT age)
{
   U8BIT setting;

   FUNCTION_START(ACTL_SetParentalControlAge);

   if ((age >= 4) && (age <= 18))
   {
      setting = (U8BIT)APP_NvmRead(PARENTAL_LOCK_NVM);
      if (age != setting)
      {
         APP_NvmSave(PARENTAL_LOCK_NVM, age, TRUE);
      }
   }

   FUNCTION_FINISH(ACTL_SetParentalControlAge);
}

/**
 * @brief   Returns the current age set for parental control. 0 will be returned
 *          if parental control is disabled or no age is set.
 * @return  age in the range 4-18, or 0
 */
U8BIT ACTL_GetParentalControlAge(void)
{
   U8BIT age;

   FUNCTION_START(ACTL_GetParentalControlAge);

   age = (U8BIT)APP_NvmRead(PARENTAL_LOCK_NVM);
   if ((age < 4) || (age > 18))
   {
      age = 0;
   }

   FUNCTION_FINISH(ACTL_GetParentalControlAge);

   return(age);
}

/**
 * @brief   Checks the parental control for the current event on the
 *          given service to determine whether decoding should be locked
 * @param   path decode path
 * @param   s_ptr service
 */
void ACTL_ApplyParentalControl(U8BIT path, void *s_ptr)
{
   void *now_event;
   U8BIT parental_lock;
   U8BIT age;
   BOOLEAN restart_decoding;

   FUNCTION_START(ACTL_ApplyParentalControl);

   parental_lock = (U8BIT)APP_NvmRead(PARENTAL_LOCK_NVM);
   if (parental_lock == PARENTAL_LOCK_OFF)
   {
      /* No parental control, so ensure decoding isn't locked */
      STB_DPSetLockEnable(path, FALSE);
   }
   else
   {
      restart_decoding = FALSE;

      /* Check whether the current service is locked before checking whether age related parental
       * control should be applied */
      if (ADB_GetServiceLockedFlag(s_ptr))
      {
         if (!STB_DPGetLockEnable(path) && (decoding_state[path] != DSTATE_STARTING) &&
            (decoding_state[path] != DSTATE_DECODING))
         {
            /* Service needs to be locked */
            STB_DPSetLockEnable(path, TRUE);

            if (!STB_DPGetLockMode(path))
            {
               STB_DPSetLockMode(path, TRUE);
            }

            restart_decoding = TRUE;
         }
      }
      else
      {
         /* Check whether parental control applies to the programme now being broadcast,
          * or the service being viewed */
         ADB_GetNowNextEvents(s_ptr, &now_event, NULL);
         if (now_event != NULL)
         {
            /* If the now event hasn't changed then don't need to reapply parental control */
            if (ADB_GetEventId(now_event) != now_event_id[path])
            {
               now_event_id[path] = ADB_GetEventId(now_event);

               /* Generate age part of ETSI 300468 */
               age = ADB_GetEventParentalAge(now_event);
               if ((age >= 0x01) && (age <= 0x0f))
               {
                  age += 3;
               }

               /* Parental locking needs to be applied according to the set age */
               if (age >= parental_lock)
               {
                  /* Parental locking is enabled */
                  STB_DPSetLockEnable(path, TRUE);

                  /* Decoding needs to be locked */
                  if (!STB_DPGetLockMode(path))
                  {
                     STB_DPSetLockMode(path, TRUE);
#ifdef INTEGRATE_HBBTV
                     HBBTV_NotifyParentalRatingChange(TRUE);
#endif
                  }

                  restart_decoding = TRUE;
               }
               else
               {
                  /* Decoding no longer needs to be locked */
                  STB_DPSetLockEnable(path, FALSE);

                  if (STB_DPGetLockMode(path))
                  {
                     STB_DPSetLockMode(path, FALSE);
#ifdef INTEGRATE_HBBTV
                     HBBTV_NotifyParentalRatingChange(FALSE);
#endif
                     restart_decoding = TRUE;
                  }
               }
            }

            ADB_ReleaseEventData(now_event);
         }
         else
         {
            /* Without the now event, the service should be unlocked */
            STB_DPSetLockEnable(path, FALSE);

            if (STB_DPGetLockMode(path))
            {
               STB_DPSetLockMode(path, FALSE);
               restart_decoding = TRUE;
            }
         }
      }

      if (restart_decoding)
      {
         if (decoding_state[path] >= DSTATE_STARTING_BAD_SIGNAL)
         {
            /* Stop and start decoding to force the change to locked mode */
            STB_DPStopDecoding(path);
            STB_DPStartDecoding(path);
         }
      }
   }

   FUNCTION_FINISH(ACTL_ApplyParentalControl);
}

/**
 * @brief   Checks content protection requirements for the current event on the
 *          given service to determine whether HDCP has to be used. The HDMI output
 *          will be enabled/disabled accordingly.
 * @param   s_ptr service
 */
void ACTL_ApplyHDCP(void *s_ptr)
{
   void *now_event;
   U8BIT level;

   FUNCTION_START(ACTL_ApplyHDCP);

   if (ACFG_IsNordigCountry())
   {
#ifdef COMMON_INTERFACE
      /* If CI+ indicates that HDCP is required then it will have set the state of the HDMI output,
       * in which case nothing should be done here to override it. */
      if (!STB_CiCcIsHDCPRequired(ADB_GetServiceId(s_ptr)))
#endif
      {
         /* Use the content protection level to determine whether HDCP is required or not */
         level = ADB_GetServiceContentProtectionLevel(s_ptr);

         switch (level)
         {
            case 0x03:
               /* Protection is mandatory regardless of the content */
               if (STB_AVIsHDCPAuthenticated())
               {
                  /* Turn the HDMI output on */
                  STB_AVEnableHDMIDecoding();
               }
               else
               {
                  /* Turn the HDMI output off */
                  STB_AVDisableHDMIDecoding();
               }
               break;

            case 0x02:
               /* Protection is mandatory for HD or above content, but not for SD (<= 576 lines).
                * As HDCP is currently always on, treat this the same as case 0x03 */
               if (ADB_GetServiceIsHd(s_ptr))
               {
                  if (STB_AVIsHDCPAuthenticated())
                  {
                     /* Turn the HDMI output on */
                     STB_AVEnableHDMIDecoding();
                  }
                  else
                  {
                     /* Turn the HDMI output off */
                     STB_AVDisableHDMIDecoding();
                  }
               }
               else
               {
                  /* Turn the HDMI output on */
                  STB_AVEnableHDMIDecoding();
               }
               break;

            case 0x00:
               /* Protection shall be turned off! Always enable the output */
               STB_AVEnableHDMIDecoding();
               break;

            case 0x01:
            default:
               /* Protection isn't required and can be on or off, so enable output */
               STB_AVEnableHDMIDecoding();
               break;
         }
      }
   }
   else
   {
      /* Check that the programme now being broadcast is allowed to be shown on an HD output */
      ADB_GetNowNextEvents(s_ptr, &now_event, NULL);
      if (now_event != NULL)
      {
#ifdef COMMON_INTERFACE
         /* If CI+ indicates that HDCP is required then it will have set the state of the HDMI output,
          * in which case nothing should be done here to override it. */
         if (!STB_CiCcIsHDCPRequired(ADB_GetServiceId(s_ptr)))
#endif
         {
            /* Check whether the current event should only be output with
             * content protection enabled */
            if (!ADB_GetEventDoNotScramble(now_event))
            {
               if (!STB_AVIsHDCPAuthenticated())
               {
                  /* Turn the HDMI output off */
                  STB_AVDisableHDMIDecoding();
               }
               else
               {
                  /* Turn the HDMI output on */
                  STB_AVEnableHDMIDecoding();
               }
            }
            else
            {
               /* Make sure the HDMI output is on */
               STB_AVEnableHDMIDecoding();
            }
         }

         ADB_ReleaseEventData(now_event);
      }
      else
      {
#ifdef COMMON_INTERFACE
         /* If CI+ indicates that HDCP is required then it will have set the state of the HDMI output,
          * in which case nothing should be done here to override it. */
         if (!STB_CiCcIsHDCPRequired(ADB_GetServiceId(s_ptr)))
#endif
         {
            /* Base protection on the service */
            if (!ADB_GetServiceDoNotScramble(s_ptr))
            {
               if (!STB_AVIsHDCPAuthenticated())
               {
                  /* Turn the HDMI output off */
                  STB_AVDisableHDMIDecoding();
               }
               else
               {
                  /* Turn the HDMI output on */
                  STB_AVEnableHDMIDecoding();
               }
            }
            else
            {
               /* Make sure the HDMI output is on */
               STB_AVEnableHDMIDecoding();
            }
         }
      }
   }

   FUNCTION_FINISH(ACTL_ApplyHDCP);
}

/**
 * @brief   Stops subtitles being displayed if another OSD requires them to be
            hidden, such as CI+
 * @return  TRUE if suppressed, FALSE otherwise
 */
BOOLEAN ACTL_SuppressSubtitles(BOOLEAN suppress)
{
   FUNCTION_START(ACTL_SuppressSubtitles);
   subtitles_suppressed = suppress;
   if(suppress == FALSE && subtitles_running)
   {
      ACTL_StartSubtitles();
   }
   FUNCTION_FINISH(ACTL_SuppressSubtitles);
   return(subtitles_suppressed);
}

/**
 * @brief   Starts subtitle processing and display if the current service has
 *          valid subtitle data. If DVB subtitles aren't available, teletext will
 *          be used if available.
 * @return  TRUE if started, FALSE otherwise
 */
BOOLEAN ACTL_StartSubtitles(void)
{
   ADB_SUBT_INFO subt_info;
   U16BIT pid;
   U8BIT path;

   FUNCTION_START(ACTL_StartSubtitles);

   path = ACTL_GetActivePath();
   if (!subtitles_suppressed && path != INVALID_RES_ID)
   {
      pid = GetSubtitlePidFunc(current_service[path], &subt_info);
      if (pid != 0)
      {
         SUBT_DBG(("ACTL_StartSubtitles: pid=%x dvb_subtitles=%d", pid,subt_info.is_dvb_subt));
         dvb_subtitles = subt_info.is_dvb_subt;
         if (dvb_subtitles)
         {
            SUBT_DBG(("ACTL_StartSubtitles: calling STB_SUBStart(%u,%u)",subt_info.u.subt.cpage,subt_info.u.subt.apage));
            STB_SUBStart(path, pid, subt_info.u.subt.cpage, subt_info.u.subt.apage);
            STB_SUBEnable(path, TRUE);
         }
         else
         {
            //STB_EBUTT_SetCacheMethod(EBUTT_CACHING_METHOD_PREVIOUS_NEXT);
            STB_EBUTT_SetCacheMethod(EBUTT_CACHING_METHOD_NAVIGATION_TREE);

            SUBT_DBG(("ACTL_StartSubtitles: calling STB_EBUTT_Start(%u,%u)", subt_info.u.ttxt.magazine,subt_info.u.ttxt.page));
            STB_EBUTT_Start(ACTL_GetActivePath(), pid, subt_info.u.ttxt.magazine, subt_info.u.ttxt.page);
            STB_EBUTT_Show(EBUTT_CHARACTER_SET_DESIGNATION_LATIN_DEFAULT, FALSE, FALSE);
         }
      }
   }

   subtitles_running = TRUE;
   subtitles_started = TRUE;

   FUNCTION_FINISH(ACTL_StartSubtitles);

   return(subtitles_started);
}

/**
 * @brief   Disables the display of subtitles but processing continues
 */
void ACTL_PauseSubtitles(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_PauseSubtitles);

   if (subtitles_started && subtitles_running)
   {
      if (dvb_subtitles)
      {
         path = ACTL_GetActivePath();
         SUBT_DBG(("ACTL_PauseSubtitles: calling STB_SUBEnable(FALSE)"));
         STB_SUBEnable(path, FALSE);
      }
      else
      {
         SUBT_DBG(("ACTL_PauseSubtitles: calling STB_EBUTT_Hide"));
         STB_EBUTT_Hide();
      }
      subtitles_running = FALSE;
   }

   FUNCTION_FINISH(ACTL_PauseSubtitles);
}

/**
 * @brief   Resumes the display of subtitles after they've been paused
 */
void ACTL_ResumeSubtitles(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_ResumeSubtitles);

   if (subtitles_started && !subtitles_running)
   {
      path = ACTL_GetActivePath();

      if (!subtitles_suppressed && ADB_ServiceHasSubtitles(current_service[path], &dvb_subtitles))
      {
         if (dvb_subtitles)
         {
            SUBT_DBG(("ACTL_ResumeSubtitles calling STB_SUBEnable(TRUE)"));
            STB_SUBEnable(path, TRUE);
         }
         else
         {
            STB_EBUTT_Show(EBUTT_CHARACTER_SET_DESIGNATION_LATIN_DEFAULT, FALSE, FALSE);
         }
      }

      subtitles_running = TRUE;
   }

   FUNCTION_FINISH(ACTL_ResumeSubtitles);
}

/**
 * @brief   Stops subtitles from being displayed and processed
 */
void ACTL_StopSubtitles(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_StopSubtitles);

   if (subtitles_started)
   {
      path = ACTL_GetActivePath();

      if (dvb_subtitles)
      {
         if (subtitles_running)
         {
            SUBT_DBG(("ACTL_StopSubtitles: calling STB_SUBEnable(FALSE)"));
            STB_SUBEnable(path, FALSE);
         }

         SUBT_DBG(("ACTL_StopSubtitles: calling STB_SUBStop"));
         STB_SUBStop(path);
      }
      else
      {
         SUBT_DBG(("ACTL_StopSubtitles:: calling STB_EBUTT_Hide"));
         STB_EBUTT_Hide();
         STB_EBUTT_Stop(path, TRUE);
      }
      subtitles_running = FALSE;
      subtitles_started = FALSE;
   }

   FUNCTION_FINISH(ACTL_StopSubtitles);
}

/**
 * @brief   Returns whether subtitles are being displayed
 */
BOOLEAN ACTL_AreSubtitlesDisplayed(void)
{
   FUNCTION_START(ACTL_AreSubtitlesDisplayed);
   FUNCTION_FINISH(ACTL_AreSubtitlesDisplayed);
   return(subtitles_running);
}

/**
 * @brief   Returns whether subtitles have been started, even if they aren't being displayed
 */
BOOLEAN ACTL_AreSubtitlesStarted(void)
{
   FUNCTION_START(ACTL_AreSubtitlesStarted);
   FUNCTION_FINISH(ACTL_AreSubtitlesStarted);
   return(subtitles_started);
}

/**
 * @brief   Returns whether trailer booking is available based on whether it's enabled
 *          and the number of links currently available on the given service
 * @param   serv_ptr service on which trailer booking is being queried
 * @return  TRUE if trailer booking is available, FALSE otherwise
 */
BOOLEAN ACTL_IsTrailerBookingAvailable(void *serv_ptr)
{
   BOOLEAN available;
   U8BIT num_links;

   FUNCTION_START(ACTL_IsTrailerBookingAvailable);

   available = FALSE;

   num_links = ADB_GetServiceNumRCTLinks(serv_ptr);
   if (num_links > 0)
   {
      available = TRUE;
   }

   FUNCTION_FINISH(ACTL_IsTrailerBookingAvailable);

   return(available);
}

/**
 * @brief   Returns the active path (live, playback, etc...), i.e. the one using the decoders
 * @return  active path
 */
U8BIT ACTL_GetActivePath(void)
{
   U8BIT path;

   FUNCTION_START(ACTL_GetActivePath);

   path = STB_DPGetPlaybackPath();
   if (path == INVALID_RES_ID)
   {
      path = STB_DPGetLivePath();
   }

   FUNCTION_FINISH(ACTL_GetActivePath);

   return path;
}

/**
 * @brief   Returns the search path
 * @return  search path
 */
U8BIT ACTL_GetSearchPath(void)
{
   return search_path;
}

/**
 * @brief   Entry point for starting a service search for a full retune or to update
 *          the existing service lineup.
 * @param   tuner_type tuner type that defines the search to be performed
 * @param   type type of service search to be performed
 * @return  TRUE if search starts successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartServiceSearch(E_STB_DP_SIGNAL_TYPE tuner_type, E_ACTL_SEARCH_TYPE type)
{
   BOOLEAN retval;
   S_MANUAL_TUNING_PARAMS tune_params;

   FUNCTION_START(ACTL_StartServiceSearch);

   retval = FALSE;

#ifdef DEBUG_SERVICE_SEARCH
   STB_SPDebugWrite("%s(%u, %u)", __FUNCTION__, tuner_type, type);
#endif

   /* Path to be used for the search doesn't require decoders */
   search_path = STB_DPAcquireTunerPath(tuner_type, NULL, NULL, RES_OWNER_DVB, DP_PRIORITY_LOW,
      FALSE, FALSE);
   if (search_path != INVALID_RES_ID)
   {
      /* Set the state in NVM */
      APP_NvmSave(SEARCHING_STATE_NVM, (U32BIT)TRUE, TRUE);

      ACTL_DisableCiModule();

      search_list_id = 0;
      signal_type_required = tuner_type;

      switch (tuner_type)
      {
         case SIGNAL_COFDM:
            terr_hierarchy = TUNE_THIERARCHY_NONE;
            t2_plp_id = 0;
            STB_DPSetTerrPLP(search_path, t2_plp_id);

            /* Map search type for searches supported by DVB-T */
            switch (type)
            {
               case ACTL_FREQ_SEARCH:
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  break;

               case ACTL_NETWORK_SEARCH:
                  /* Network search starts off as a freq search until an NIT is found */
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_NETWORK;
                  break;

               default:
                  current_search_type = SEARCH_TYPE_NOSEARCH;
                  break;
            }

            if (current_search_type != SEARCH_TYPE_NOSEARCH)
            {
               /* Start the search */
               retval = ACTL_TuneToRfChanArrayEntry(search_path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
            }
            break;
         case SIGNAL_QAM:
            /* Map search type for searches supported by DVB-C */
            switch (type)
            {
               case ACTL_FREQ_SEARCH:
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  break;

               case ACTL_NETWORK_SEARCH:
                  /* Network search starts off as a freq search until an NIT is found */
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_NETWORK;
                  break;

               default:
                  current_search_type = SEARCH_TYPE_NOSEARCH;
                  break;
            }

            if (current_search_type != SEARCH_TYPE_NOSEARCH)
            {
               /* Start the search */
               retval = ACTL_TuneToRfChanArrayEntry(search_path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
            }
            break;
         case SIGNAL_ANALOG:
            /* Only support frequency search for analog */
            current_search_type = SEARCH_TYPE_SERVICE_FREQ;
            required_search_type = SEARCH_TYPE_SERVICE_FREQ;

            /* Start the search */
            retval = ACTL_TuneToRfChanArrayAnalogEntry(search_path, search_list_id);
            break;
         case SIGNAL_QPSK:
            /* Map search type for searches supported by DVB-S */
            switch (type)
            {
               case ACTL_FREQ_SEARCH:
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  break;

               case ACTL_NETWORK_SEARCH:
                  /* Network search starts off as a freq search until a NIT is found */
                  current_search_type = SEARCH_TYPE_SERVICE_FREQ;
                  required_search_type = SEARCH_TYPE_SERVICE_NETWORK;
                  break;

               default:
                  current_search_type = SEARCH_TYPE_NOSEARCH;
                  break;
            }

            if (current_search_type != SEARCH_TYPE_NOSEARCH)
            {
               /* Setup the tuning params */
               MapIdToSatTuneParams(search_list_id, &tune_params);

               /* Start the search */
               retval = ACTL_TuneToUserDefinedParams(search_path, &tune_params, ACTL_SI_CHANNEL_SEARCH, FALSE);
            }
            break;
         default:
            signal_type_required = SIGNAL_NONE;
            break;
      }

      if (retval)
      {
         if (ASTE_InStandby())
         {
            DBG_STDBY(("Starting service search"));
            ASTE_StartServiceSearch();

            /* Save the date the search is being performed and the standby state */
            APP_NvmSave(LAST_CHAN_SRCH_NVM, DHMS_DATE(STB_GCNowDHMSLocal()), FALSE);

            DBG_STDBY(("Setting STANDBY_STATE_NVM=STDBY_UPDATE_SEARCH"));
            APP_NvmSave(STANDBY_STATE_NVM, STDBY_UPDATE_SEARCH, TRUE);
         }
         else
         {
            /* Send event to update the UI */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
         }
      }
      else
      {
         current_search_type = SEARCH_TYPE_NOSEARCH;
         required_search_type = SEARCH_TYPE_NOSEARCH;
         EndSearch(search_path);
         search_path = INVALID_RES_ID;
      }
   }

   FUNCTION_FINISH(ACTL_StartServiceSearch);

   return(retval);
}

/**
 * @brief   When the search has completed, this function should be called to see
 *          whether the target region UI should be presented. This is only needed
 *          for DVB-T/T2 searches.
 * @return  TRUE if target region UI is required, FALSE otherwise
 */
BOOLEAN ACTL_IsTargetRegionRequired(void)
{
   BOOLEAN retval;
   U32BIT *country_codes;

   FUNCTION_START(ACTL_IsTargetRegionRequired);

   if (ADB_GetNetworkTargetCountries(&country_codes) > 0)
   {
      STB_AppFreeMemory(country_codes);
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ACTL_IsTargetRegionRequired);

   return(retval);
}

/**
 * @brief   Function to stop the service search before it completes
 */
void ACTL_StopServiceSearch(void)
{
   FUNCTION_START(ACTL_StopServiceSearch);

   EndSearch(search_path);
   search_path = INVALID_RES_ID;

   FUNCTION_FINISH(ACTL_StopServiceSearch);
}

/**
 * @brief   Function to start the startup search when booting from cold. This
 *          search checks the validity of the services contained in the database.
 *          It's only valid for DVB-T/T2 and DVB-C.
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartStartupSearch(void)
{
   BOOLEAN retval;
   void **tlist;
   U16BIT num_transports;

   FUNCTION_START(ACTL_StartStartupSearch);

   retval = FALSE;

   ADB_GetTransportList(&tlist, &num_transports);

   if (num_transports > 0)
   {
      search_list_id = 0;
      search_num_transports = num_transports;
      current_search_type = SEARCH_TYPE_STARTUP;

      search_path = STB_DPAcquireTunerPath(ADB_GetTransportSignalType(tlist[search_list_id]), NULL, NULL,
         RES_OWNER_DVB, DP_PRIORITY_LOW, FALSE, FALSE);
      if (search_path != INVALID_RES_ID)
      {
         ACTL_TuneToTransport(search_path, NULL, tlist[search_list_id], ACTL_SI_STARTUP_SEARCH, FALSE);
         retval = TRUE;
      }

      ADB_ReleaseTransportList(tlist, num_transports);
   }

   FUNCTION_FINISH(ACTL_StartStartupSearch);

   return(retval);
}

#ifdef INCLUDE_OTA_SSU
/**
 * @brief   Function to start a search to see if there's an SSU available
 * @param   tuner_type restrict SSU search to transports of the given type;
 *                     SIGNAL_NONE means all available transports will be used.
 * @param   version_callback callback function that will be called if an OTA is found that should
 *                           check whether the update is for this system and the version is later
 *                           than the current version
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartSSUSearch(E_STB_DP_SIGNAL_TYPE tuner_type, F_SSU_VERSION_CALLBACK version_callback)
{
   BOOLEAN retval;
   void **tlist;
   U16BIT num_transports;

   FUNCTION_START(ACTL_StartSSUSearch);

   DBG_SSU(("ACTL_StartSSUSearch"));

   /* Start search */
   ADB_GetTransportListForTunerType(tuner_type, &tlist, &num_transports);

   retval = FALSE;

   if (num_transports > 0)
   {
      search_list_id = 0;
      search_num_transports = num_transports;

      search_path = STB_DPAcquireTunerPath(ADB_GetTransportSignalType(tlist[0]), NULL, NULL,
         RES_OWNER_DVB, DP_PRIORITY_LOW, FALSE, FALSE);
      if (search_path != INVALID_RES_ID)
      {
         /* DVB SSU upgrade system - tune to the first transport to look for the SSU info required */
         current_search_type = SEARCH_TYPE_SSU;
         current_search_tuner_type = tuner_type;

         ota_settings.ota_location = NULL;
         ota_settings.ota_onet_id = ADB_INVALID_DVB_ID;
         ota_settings.ota_tran_id = ADB_INVALID_DVB_ID;
         ota_settings.ota_serv_id = ADB_INVALID_DVB_ID;
         ota_settings.version_cb = version_callback;

         ACTL_TuneToTransport(search_path, NULL, tlist[0], ACTL_SI_DVB_SSU_SEARCH, FALSE);

         if (ASTE_InStandby())
         {
            DBG_SSU(("Starting OTA SSU search"));
            ASTE_StartSSUSearch();

            /* Save the date the search is being performed. This is set even if it doesn't start
             * successfully to prevent it constantly retrying */
            APP_NvmSave(OTA_LAST_UPDATE_SRCH_NVM, DHMS_DATE(STB_GCNowDHMSLocal()), TRUE);

            DBG_STDBY(("Setting STANDBY_STATE_NVM=STDBY_UPDATE_SEARCH"));
            APP_NvmSave(STANDBY_STATE_NVM, STDBY_UPDATE_SEARCH, FALSE);
         }

         retval = TRUE;
      }

      ADB_ReleaseTransportList(tlist, num_transports);
   }

   FUNCTION_FINISH(ACTL_StartSSUSearch);

   return(retval);
}

/**
 * @brief   Function to prepare a SSU data download
 *
 * @return  TRUE OK, FALSE otherwise
 */
BOOLEAN ACTL_PrepareSSUDownload(void)
{
    BOOLEAN retval = FALSE;

    FUNCTION_START(ACTL_PrepareSSUDownload);

    DBG_SSU(("ACTL_PrepareSSUDownload"));

    U8BIT path = unt_settings.path;
    U16BIT ssu_data_pid = unt_settings.ssu_data_pid;
    ADB_TRANSPORT_REC * transport = unt_settings.transport;
    S_ACTL_OWNER_INFO *owner_info = NULL;

    DBG_SSU(("%s(path=%u, ts=%p, pid=0x%04x)", __FUNCTION__, path, transport, ssu_data_pid));
    
    path = AcquirePathForTransport(transport, FALSE, TRUE, owner_info);
    unt_settings.path = path;
    if (path != INVALID_RES_ID)
    {
        current_search_type = SEARCH_TYPE_SSU_DOWNLOAD;
        if (ADB_GetTunedTransport(path) != transport)
        {
            /* The path will be retuned so stop it first */
            ACTL_TuneOff(path);
        }

        unt_settings.path = ACTL_TuneToTransport(path, NULL, transport, ACTL_SI_NO_SEARCH, TRUE);
        if (unt_settings.path != INVALID_RES_ID)
        {
            DBG_SSU(("%s: Tuning to transport 0x%04x/%04x, path=%u", __FUNCTION__,
                    ADB_GetTransportOriginalNetworkId(transport),
                    ADB_GetTransportTid(transport), unt_settings.path));

            retval = TRUE;
       }
       else
       {
            DBG_SSU(("%s: Failed to tune to transport 0x(%04x/%04x)", __FUNCTION__,
                ADB_GetTransportOriginalNetworkId(transport), ADB_GetTransportTid(transport)));

            current_search_type = SEARCH_TYPE_NOSEARCH;
       }
   }
   else {
       DBG_SSU(("%s: Failed to acquire a decode-path for transport 0x(%04x/%04x)", __FUNCTION__,
           ADB_GetTransportOriginalNetworkId(transport), ADB_GetTransportTid(transport)));
   }

   FUNCTION_FINISH(ACTL_PrepareSSUDownload);

   return(retval);
}

/**
 * @brief   Function to start a SSU data download
 *
 * @return  TRUE if the download is started, FALSE otherwise
 */
BOOLEAN ACTL_StartSSUDownload(U8BIT path)
{
    BOOLEAN retval = FALSE;

    FUNCTION_START(ACTL_StartSSUDownload);

    if (INVALID_RES_ID == path)
        goto end;

    if (path != unt_settings.path)
        goto end;

    DBG_SSU(("ACTL_StartSSUDownload"));

    U16BIT ssu_data_pid = unt_settings.ssu_data_pid;
    ADB_TRANSPORT_REC * transport = unt_settings.transport;

    /* Already tuned to the correct transport so the search for the OTA can be started */

#if 0
    retval = STB_OTAStartLoader(path, ADB_GetTransportOriginalNetworkId(transport),
        ADB_GetTransportTid(transport), 0, ssu_data_pid, NULL);
#else
    /* OTA via SSU service ID: 0xFFFF */
    retval = STB_OTAStartLoader(path, ADB_GetTransportOriginalNetworkId(transport),
        ADB_GetTransportTid(transport), 0xFFFF, DVB_INVALID_ID, NULL);
#endif
    if (!retval && (current_search_type == SEARCH_TYPE_SSU_DOWNLOAD))
    {
        current_search_type = SEARCH_TYPE_NOSEARCH;
    }

    DBG_SSU(("%s: Already tuned to transport, OTA search %s", __FUNCTION__,
        (retval ? "started" : "failed")));

end:
    FUNCTION_FINISH(ACTL_StartSSUDownload);

    return(retval);
}

/**
 * @brief   Function to get the SSU data download progress
 *
 * @return  Download progress as a percentage [0 - 100]
 */
U8BIT ACTL_GetSSUDownloadProgress(void)
{
    FUNCTION_START(ACTL_GetSSUDownloadProgress);
    FUNCTION_FINISH(ACTL_GetSSUDownloadProgress);

    return STB_OTAGetProgress();
}

/**
 * @brief   Function to stop a SSU data download
 *
 * @return  TRUE if the download is stoped, FALSE otherwise
 */
BOOLEAN ACTL_StopSSUDownload(void)
{
    BOOLEAN retval = FALSE;

    FUNCTION_START(ACTL_StopSSUDownload);

    DBG_SSU(("ACTL_StopSSUDownload"));

    U8BIT path = unt_settings.path;
    U16BIT ssu_data_pid = unt_settings.ssu_data_pid;
    ADB_TRANSPORT_REC * transport = unt_settings.transport;
    
    if (INVALID_RES_ID != path) {
        STB_OTAStopLoader();

        if (NULL != transport) {
            if (ADB_GetTunedTransport(path) == transport) {
                ACTL_TuneOff(path);
            }
        }
        ReleasePath(path);
        unt_settings.path = INVALID_RES_ID;            
    }

    if (SEARCH_TYPE_SSU_DOWNLOAD == current_search_type) {
        current_search_type = SEARCH_TYPE_NOSEARCH;
    }
            
    FUNCTION_FINISH(ACTL_StopSSUDownload);

    return(retval);
}

/**
 * @brief   Function to get a SSU update information
 *
 * @return  JSON string
 */
const U8BIT * ACTL_GetSSUInfo(void)
{
    if (unt_settings.valid)
       return (const U8BIT *)unt_settings.jsonString;

    return NULL;
}

#endif /* INCLUDE_OTA_SSU */

/**
 * @brief   Function to start a TOT search which is required to set the system clock
 *          when starting from power off if a real-time clock isn't available.
 * @param   tuner_type restrict TOT search to transports of the given type;
 *                     SIGNAL_NONE means all available transports will be used.
 * @return  TRUE if the search is started, FALSE otherwise
 */
BOOLEAN ACTL_StartTotSearch(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   BOOLEAN retval;
   void **tlist;
   U16BIT num_transports;

   FUNCTION_START(ACTL_StartTotSearch);

   retval = FALSE;

   TOT_DBG(("ACTL_StartTotSearch(tuner_type=%s)",
      ((tuner_type == SIGNAL_COFDM) ? "DVB-T" :
       (tuner_type == SIGNAL_QPSK) ? "DVB-S" :
       (tuner_type == SIGNAL_QAM) ? "DVB-C" : "All")));

   ADB_GetTransportListForTunerType(tuner_type, &tlist, &num_transports);

   if (num_transports > 0)
   {
      search_list_id = 0;
      search_num_transports = num_transports;
      required_search_type = SEARCH_TYPE_TOT;
      current_search_type = SEARCH_TYPE_TOT;
      current_search_tuner_type = tuner_type;

      search_path = STB_DPAcquireTunerPath(ADB_GetTransportSignalType(tlist[search_list_id]), NULL, NULL,
         RES_OWNER_DVB, DP_PRIORITY_LOW, FALSE, FALSE);
      if (search_path != INVALID_RES_ID)
      {
         TOT_DBG(("ACTL_StartTotSearch: Tuning, num_transports=%u", search_num_transports));
         ACTL_TuneToTransport(search_path, NULL, tlist[search_list_id], ACTL_SI_TOT_SEARCH, FALSE);
         retval = TRUE;
      }

      ADB_ReleaseTransportList(tlist, num_transports);
   }
#ifdef DEBUG_TOT_SEARCH
   else
   {
      TOT_DBG(("ACTL_StartTotSearch: num_transports = 0"));
   }
#endif

   if (retval == FALSE)
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
   }

   FUNCTION_FINISH(ACTL_StartTotSearch);

   return(retval);
}

/**
 * @brief   Function to stop the TOT search before it completes
 */
void ACTL_StopTotSearch(void)
{
   FUNCTION_START(ACTL_StopTotSearch);

   if (search_path != INVALID_RES_ID)
   {
      FinishSearch(search_path);
   }
#ifdef DEBUG_TOT_SEARCH
   else
   {
      TOT_DBG(("ACTL_StopTotSearch: invalid path"));
   }
#endif

   FUNCTION_FINISH(ACTL_StopTotSearch);
}

/**
 * @brief   Entry point for starting an EIT search
 * @param   tuner_type limit the search to transports of this tuner type;
 *                     pass as SIGNAL_NONE for all transports
 * @param   search_type event search type to be performed, p/f only, sched only, or both
 * @return  TRUE if search starts successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartEitSearch(E_STB_DP_SIGNAL_TYPE tuner_type, E_ACTL_EVENT_SEARCH search_type)
{
   BOOLEAN retval;
   void **transport_list;
   U16BIT num_transports;
   void **network_list;
   U16BIT num_networks;
   U16BIT i, j;
   void *t_ptr;

   FUNCTION_START(ACTL_StartEitSearch);

   retval = FALSE;

   ADB_GetTransportListForTunerType(tuner_type, &transport_list, &num_transports);
   if ((transport_list != NULL) && (num_transports != 0))
   {
      search_num_transports = num_transports;

      /* Some networks may have a transport that contains full SI data, in which case other
       * transports on that network can be removed */
      ADB_GetNetworkList(&network_list, &num_networks);
      if ((network_list != NULL) && (num_networks != 0))
      {
         for (i = 0; i < num_networks; i++)
         {
            t_ptr = ADB_GetFullSITransport(network_list[i]);
            if (t_ptr != NULL)
            {
               /* This network has a transport with full SI data, so remove other transports
                * from the list that are in this network */
               for (j = 0; j < num_transports; j++)
               {
                  if ((transport_list[j] != t_ptr) &&
                     (ADB_GetTransportNetworkPtr(transport_list[j]) == network_list[i]))
                  {
                     /* This transport is in the network so doesn't need to be checked */
                     transport_list[j] = NULL;
                     search_num_transports--;
                  }
               }
            }
         }

         ADB_ReleaseNetworkList(network_list);
      }

      if (search_num_transports != 0)
      {
         /* Allocate and copy the transports to be searched */
         search_transport_list = (void **)STB_AppGetMemory(search_num_transports * sizeof(void *));
         if (search_transport_list != NULL)
         {
            for (i = 0, j = 0; i < num_transports; i++)
            {
               if (transport_list[i] != NULL)
               {
                  search_transport_list[j] = transport_list[i];
                  j++;
               }
            }

            ADB_ReleaseTransportList(transport_list, num_transports);

            /* Now ready to start the search */
            EIT_DBG(("%s: Starting EIT %s search on %u transports", __FUNCTION__,
               ((search_type == ACTL_EVENT_SEARCH_PF) ? "p/f" :
                (search_type == ACTL_EVENT_SEARCH_SCHED) ? "sched" : "p/f+sched"),
               search_num_transports));

            search_list_id = 0;

            switch (search_type)
            {
               case ACTL_EVENT_SEARCH_PF:
                  current_search_type = SEARCH_TYPE_EIT_PF;
                  break;
               case ACTL_EVENT_SEARCH_SCHED:
                  current_search_type = SEARCH_TYPE_EIT_SCHED;
                  break;
               case ACTL_EVENT_SEARCH_PF_SCHED:
                  current_search_type = SEARCH_TYPE_EIT_PF_SCHED;
                  break;
            }

            retval = StartEitSearch();
            if (!retval)
            {
               current_search_type = SEARCH_TYPE_NOSEARCH;
            }
         }
         else
         {
            EIT_DBG(("%s: Failed to allocate memory for transports to be searched", __FUNCTION__));
            ADB_ReleaseTransportList(transport_list, num_transports);
            search_num_transports = 0;
         }
      }
      else
      {
         EIT_DBG(("%s: No transports left after removing network transports!", __FUNCTION__));
         ADB_ReleaseTransportList(transport_list, num_transports);
      }
   }
   else
   {
      EIT_DBG(("%s: Failed to find any transports for tuner type %u", __FUNCTION__, tuner_type));
   }

   FUNCTION_FINISH(ACTL_StartEitSearch);

   return(retval);
}

/**
 * @brief   Function to stop the EIT search before it completes
 */
void ACTL_StopEitSearch(void)
{
   FUNCTION_START(ACTL_StopEitSearch);

   if (search_path != INVALID_RES_ID)
   {
      ACTL_TuneOff(search_path);
      STB_DPReleasePath(search_path, RES_OWNER_DVB);
      search_path = INVALID_RES_ID;
   }
#ifdef DEBUG_EIT_SEARCH
   else
   {
      EIT_DBG(("ACTL_StopEitSearch: invalid path"));
   }
#endif

   FUNCTION_FINISH(ACTL_StopEitSearch);
}

/**
 * @brief   Returns TRUE if current search has finished.
 *          This works for any of the available types of searches.
 * @return  TRUE if search has finished, FALSE otherwise
 */
BOOLEAN ACTL_IsSearchComplete(void)
{
   FUNCTION_START(ACTL_IsSearchComplete);
   FUNCTION_FINISH(ACTL_IsSearchComplete);
   return(current_search_type == SEARCH_TYPE_NOSEARCH);
}

/**
 * @brief   Returns search progress as a percentage.
 *          This works for any of the available types of searches, but the
 *          scope for providing progress on anything other than the service
 *          searches is fairly limited.
 * @return  Percent complete, 0 if not started
 */
U8BIT ACTL_GetSearchProgress(void)
{
   U8BIT progress;
   U16BIT num;
#if defined(DEBUG_SERVICE_SEARCH) || defined(DEBUG_TOT_SEARCH) || defined(DEBUG_SSU_SEARCH) || \
   defined(DEBUG_STARTUP_SEARCH)
   static U8BIT last_progress = 0;
#endif


   FUNCTION_START(ACTL_GetSearchProgress);

   switch (current_search_type)
   {
      case SEARCH_TYPE_SERVICE_FREQ:
         if (signal_type_required == SIGNAL_QPSK)
         {
            if ((num = GetNumSatSearchIds()) == 0)
            {
               progress = 0;
            }
            else
            {
               progress = (search_list_id * 100) / num;
            }
#ifdef DEBUG_SERVICE_SEARCH
            if (progress != last_progress)
            {
               STB_SPDebugWrite("Progress: %u of %u, %u%%", search_list_id, num, progress);
               last_progress = progress;
            }
#endif
         }
         else
         {
            if ((num = ACTL_GetNumRfChanArrayEntries(signal_type_required)) == 0)
            {
               progress = 0;
            }
            else
            {
               progress = (search_list_id * 100) / num;
            }
#ifdef DEBUG_SERVICE_SEARCH
            if (progress != last_progress)
            {
               STB_SPDebugWrite("Progress: %u of %u, %u%%", search_list_id, num, progress);
               last_progress = progress;
            }
#endif
         }
         break;

      case SEARCH_TYPE_SERVICE_NETWORK:
      case SEARCH_TYPE_MANUAL_NETWORK:
         if ((num = ADB_GetNumTransports()) == 0)
         {
            progress = 0;
         }
         else
         {
            progress = (search_list_id * 100) / num;
         }
#ifdef DEBUG_SERVICE_SEARCH
         if (progress != last_progress)
         {
            STB_SPDebugWrite("Progress: %u of %u, %u%%", search_list_id, num, progress);
            last_progress = progress;
         }
#endif
         break;

      case SEARCH_TYPE_TOT:
      case SEARCH_TYPE_STARTUP:
      case SEARCH_TYPE_SSU:
         if (search_num_transports == 0)
         {
            progress = 0;
         }
         else
         {
            progress = (search_list_id * 100) / search_num_transports;
         }
#if defined(DEBUG_TOT_SEARCH) || defined(DEBUG_SSU_SEARCH) || defined(DEBUG_STARTUP_SEARCH)
         if (progress != last_progress)
         {
            STB_SPDebugWrite("Progress: %u of %u, %u%%", search_list_id, search_num_transports, progress);
            last_progress = progress;
         }
#endif
         break;

      default:
         progress = 0;
         break;
   }

   FUNCTION_FINISH(ACTL_GetSearchProgress);

   return(progress);
}

U8BIT ACTL_GetServiceSearchPath()
{
    return search_path;
}

/**
 * @brief   Start a service search on, or just tune to, a transport, using chan_id
 *          as an index into the country's tuning table.
 * @param   tuner_type tuner type to be used for the search
 * @param   chan_id index into the country's channel array to tune to
 * @param   start_search TRUE if a service search is to be started, otherwise just tune
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN ACTL_StartManualSearchById(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT chan_id, BOOLEAN start_search)
{
   BOOLEAN retval;
   U16BIT num_chans;
   E_ACTL_SI_SRCH_REQD si_type;

   FUNCTION_START(ACTL_StartManualSearchById);

   retval = FALSE;

   num_chans = ACTL_GetNumRfChanArrayEntries(tuner_type);
   if (chan_id < num_chans)
   {
      if (search_path == INVALID_RES_ID)
      {
         /* Path to be used for the search doesn't require decoders */
         search_path = STB_DPAcquireTunerPath(tuner_type, NULL, NULL, RES_OWNER_DVB, DP_PRIORITY_LOW,
            FALSE, FALSE);
      }
      if (search_path != INVALID_RES_ID)
      {
         if (start_search)
         {
            si_type = ACTL_SI_CHANNEL_SEARCH;
            current_search_type = SEARCH_TYPE_MANUAL_FREQ;
            required_search_type = SEARCH_TYPE_MANUAL_FREQ;
            if (tuner_type == SIGNAL_COFDM)
            {
               terr_hierarchy = TUNE_THIERARCHY_NONE;
               t2_plp_id = 0;
               STB_DPSetTerrPLP(search_path, t2_plp_id);
            }
         }
         else
         {
            si_type = ACTL_SI_NO_SEARCH;
            current_search_type = SEARCH_TYPE_NOSEARCH;
            required_search_type = SEARCH_TYPE_NOSEARCH;
         }

         signal_type_required = tuner_type;
         search_list_id = chan_id;

         retval = ACTL_TuneToRfChanArrayEntry(search_path, chan_id, si_type, FALSE);
         if (!retval)
         {
            current_search_type = SEARCH_TYPE_NOSEARCH;
            required_search_type = SEARCH_TYPE_NOSEARCH;

            STB_DPReleasePath(search_path, RES_OWNER_DVB);
            search_path = INVALID_RES_ID;
         }
      }
   }

   FUNCTION_FINISH(ACTL_StartManualSearchById);

   return(retval);
}

/**
 * @brief   Start a service search on, or just tune to, the transport defined
 *          by the given tuning parameters
 * @param   tuner_type tuner type to be used for the search
 * @param   tuning_params user defined tuning parameters
 * @param   type type of service search to be performed
 * @return  TRUE if the search is started successfully, FALSE otherwise
 */
BOOLEAN ACTL_StartManualSearch(E_STB_DP_SIGNAL_TYPE tuner_type,
   S_MANUAL_TUNING_PARAMS *tuning_params, E_ACTL_SEARCH_TYPE type)
{
   BOOLEAN retval;

   FUNCTION_START(ACTL_StartManualSearch);

   retval = FALSE;

#ifdef DEBUG_SERVICE_SEARCH
   STB_SPDebugWrite("%s(%u, %u)", __FUNCTION__, tuner_type, type);
#endif

   if (tuning_params != NULL)
   {
      /* Path to be used for the search doesn't require decoders */
      search_path = STB_DPAcquireTunerPath(tuner_type, NULL, NULL, RES_OWNER_DVB, DP_PRIORITY_LOW,
         FALSE, FALSE);
      if (search_path != INVALID_RES_ID)
      {
         ACTL_DisableCiModule();

         switch (tuner_type)
         {
            case SIGNAL_COFDM:
               terr_hierarchy = TUNE_THIERARCHY_NONE;
               t2_plp_id = tuning_params->u.terr.plp_id;
               break;
            case SIGNAL_QAM:
               break;
            case SIGNAL_ANALOG:
               break;
            case SIGNAL_QPSK:
               break;
            default:
               tuner_type = SIGNAL_NONE;
               break;
         }

         signal_type_required = tuner_type;

         if (signal_type_required != SIGNAL_NONE)
         {
            if (type == ACTL_FREQ_SEARCH)
            {
               current_search_type = SEARCH_TYPE_MANUAL_FREQ;
               required_search_type = SEARCH_TYPE_MANUAL_FREQ;
               //
               ASI_ResetAppSiModeFlag(search_path);
               ASI_SetAppSiModeFlag(search_path, ASI_MODE_FLAG_IGNORE_NIT_TRANSPORT, TRUE);
               ASI_SetUpdateNitFunction(nitTableUpdate);
            }
            else
            {
               current_search_type = SEARCH_TYPE_MANUAL_FREQ;
               required_search_type = SEARCH_TYPE_MANUAL_NETWORK;
               //
               ASI_ResetAppSiModeFlag(search_path);
               ASI_SetUpdateNitFunction(NULL);
            }

            /* Start the search */
            search_list_id = 0;
            retval = ACTL_TuneToUserDefinedParams(search_path, tuning_params, (required_search_type == SEARCH_TYPE_MANUAL_NETWORK) ? ACTL_SI_CHANNEL_SEARCH:ACTL_SI_CHANNEL_SEARCH_NO_NIT, FALSE);

            if (retval)
            {
               if (ASTE_InStandby())
               {
                  DBG_STDBY(("Starting manual service search"));
                  ASTE_StartServiceSearch();

                  /* Save the date the search is being performed and the standby state */
                  APP_NvmSave(LAST_CHAN_SRCH_NVM, DHMS_DATE(STB_GCNowDHMSLocal()), FALSE);

                  DBG_STDBY(("Setting STANDBY_STATE_NVM=STDBY_UPDATE_SEARCH"));
                  APP_NvmSave(STANDBY_STATE_NVM, STDBY_UPDATE_SEARCH, TRUE);
               }
               else
               {
                  /* Send event to update the UI */
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
               }
            }
            else
            {
               current_search_type = SEARCH_TYPE_NOSEARCH;
               required_search_type = SEARCH_TYPE_NOSEARCH;
               EndSearch(search_path);
               search_path = INVALID_RES_ID;
            }
         }
      }
   }

   FUNCTION_FINISH(ACTL_StartManualSearch);

   return(retval);
}

/**
 * @brief   Function to be called when a manual service search has completed,
 *          or is being stopped.
 */
void ACTL_FinishManualSearch(void)
{
   FUNCTION_START(ACTL_FinishManualSearch);

   if (search_path != INVALID_RES_ID)
   {
      ACTL_TuneOff(search_path);
      STB_DPReleasePath(search_path, RES_OWNER_DVB);
      search_path = INVALID_RES_ID;
   }

#ifdef INTEGRATE_HBBTV
   HBBTV_NotifyServiceListChange();
#endif

   APP_NvmSave(FIRST_BOOT_NVM, (U32BIT)FALSE, TRUE);

   required_search_type = SEARCH_TYPE_NOSEARCH;
   current_search_type = SEARCH_TYPE_NOSEARCH;

   /* Re-enable the CI module */
   ACTL_EnableCiModule();

   if (ASTE_InStandby())
   {
      ASTE_SearchComplete();
   }

   FUNCTION_FINISH(ACTL_FinishManualSearch);
}

/**
 * @brief   Start the signal detection, or just tune to, the transport defined
 *          by the given tuning parameters
 * @param   tuner_type tuner type to be used for the search
 * @param   tuning_params user defined tuning parameters
 * @return  TRUE successful, FALSE otherwise
 */
BOOLEAN ACTL_StartSignalDetection(E_STB_DP_SIGNAL_TYPE tuner_type, S_MANUAL_TUNING_PARAMS *tuning_params)
{
   BOOLEAN retval;

   FUNCTION_START(ACTL_StartSignalDetection);

   retval = FALSE;

   if (tuning_params != NULL)
   {
      /* Path to be used for the search doesn't require decoders */
      monitor_path = STB_DPAcquireTunerPath(tuner_type, NULL, NULL, RES_OWNER_DVB, DP_PRIORITY_LOW,
         FALSE, FALSE);
      if (0 == monitor_path) {
         monitor_path = STB_DPAcquireTunerPath(
            tuner_type, NULL, NULL,
            RES_OWNER_DVB, DP_PRIORITY_LOW,
            FALSE, FALSE
         );
         STB_DPReleasePath(0, RES_OWNER_DVB);
      }
      
      if (monitor_path != INVALID_RES_ID)
      {
         ACTL_DisableCiModule();

         switch (tuner_type)
         {
            case SIGNAL_COFDM:
               terr_hierarchy = TUNE_THIERARCHY_NONE;
               t2_plp_id = tuning_params->u.terr.plp_id;
               break;
            case SIGNAL_QAM:
               break;
            case SIGNAL_ANALOG:
               break;
            case SIGNAL_QPSK:
               break;
            default:
               tuner_type = SIGNAL_NONE;
               break;
         }

         signal_type_required = tuner_type;
         if (signal_type_required != SIGNAL_NONE)
         {
            current_search_type = SEARCH_TYPE_NOSEARCH;
            required_search_type = SEARCH_TYPE_NOSEARCH;
            //
            ASI_ResetAppSiModeFlag(monitor_path);
            ASI_SetUpdateNitFunction(NULL);
            /* tune */
            retval = ACTL_TuneToUserDefinedParams(monitor_path, tuning_params, ACTL_SI_USER_DEFINED, FALSE);
            if (TRUE == retval)
            {
               if (ASTE_InStandby())
               {
                  DBG_STDBY(("Starting manual service search"));
               }
               else
               {
                  /* Send event to update the UI */
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
               }
            }
            else
            {
               ACTL_StopSignalDetection();
            }
         }
      }
   }

   FUNCTION_FINISH(ACTL_StartSignalDetection);

   return(retval);
}

/**
 * @brief   Function to be called to stop the signal detection.
 */
void ACTL_StopSignalDetection(void)
{
   FUNCTION_START(ACTL_StopSignalDetection);

   if (monitor_path != INVALID_RES_ID)
   {
      ACTL_TuneOff(monitor_path);
      STB_DPReleasePath(monitor_path, RES_OWNER_DVB);
      monitor_path = INVALID_RES_ID;
   }
   required_search_type = SEARCH_TYPE_NOSEARCH;
   current_search_type = SEARCH_TYPE_NOSEARCH;

   /* Re-enable the CI module */
   ACTL_EnableCiModule();

   if (ASTE_InStandby())
   {
      ASTE_SearchComplete();
   }

   FUNCTION_FINISH(ACTL_StopSignalDetection);
}

#ifdef INCLUDE_OTA_SSU
/**
 * @brief   Starts looking for an OTA update on the given transport. If no decode path is provided,
 *          or it is and it isn't tuned or is tuned to a different transport, then it will be tuned
 *          to the given transport before it starts looking for an OTA update.
 * @param   path decode path to be used to tune to the given transport. If INVALID_RES_ID is used
 *               then the live path will be used or a decode path will be acquired
 * @param   transport transport to tune to to check whether an OTA is available
 * @param   serv_id ID of the service on teh transport that contains the OTA
 * @param   version_callback callback function that will be called if an OTA is found that should
 *                           check whether the update is for this system and the version is later
 *                           than the current version
 * @return  TRUE if the search is started, which may mean a tune request has been made, FALSE otherwise
 */
BOOLEAN ACTL_StartOTAUpdate(U8BIT path, void *transport, U16BIT serv_id,
   F_SSU_VERSION_CALLBACK version_callback)
{
   BOOLEAN retval;
   U8BIT ota_path;

   FUNCTION_START(ACTL_StartOTAUpdate);

   retval = FALSE;

   ota_settings.ota_started = FALSE;
   ota_settings.ota_path = INVALID_RES_ID;

   DBG_SSU(("%s(path=%u, ts=%p, serv=0x%04x, cb=%p)", __FUNCTION__, path, transport, serv_id,
      version_callback));

   if (path == INVALID_RES_ID)
   {
      path = STB_DPGetLivePath();
   }

   if ((path == INVALID_RES_ID) || (ADB_GetTunedTransport(path) != transport))
   {
      /* Need to tune to the given transport */
      ota_settings.ota_onet_id = ADB_GetTransportOriginalNetworkId(transport);
      ota_settings.ota_tran_id = ADB_GetTransportTid(transport);
      ota_settings.ota_serv_id = serv_id;
      ota_settings.version_cb = version_callback;

      if (current_search_type == SEARCH_TYPE_NOSEARCH)
      {
         current_search_type = SEARCH_TYPE_OTA_UPDATE;
      }

      if (path != INVALID_RES_ID)
      {
         /* The path will be retuned so stop it first */
         ACTL_TuneOff(path);
      }

      ota_path = ACTL_TuneToTransport(path, NULL, transport, ACTL_SI_NO_SEARCH, TRUE);
      if (ota_path != INVALID_RES_ID)
      {
         if (ota_path != path)
         {
            /* Only save the decode path if it's been acquired specifically for the OTA */
            ota_settings.ota_path = ota_path;
         }

         DBG_SSU(("%s: Tuning to transport 0x%04x/%04x, path=%u", __FUNCTION__,
            ADB_GetTransportOriginalNetworkId(transport), ADB_GetTransportTid(transport), ota_path));

         retval = TRUE;
      }
      else
      {
         DBG_SSU(("%s: Failed to tune to transport 0x%04x/%04x", __FUNCTION__,
            ADB_GetTransportOriginalNetworkId(transport), ADB_GetTransportTid(transport)));

         if (current_search_type == SEARCH_TYPE_OTA_UPDATE)
         {
            current_search_type = SEARCH_TYPE_NOSEARCH;
         }
      }
   }
   else
   {
      /* Already tuned to the correct transport so the search for the OTA can be started */
      if (current_search_type == SEARCH_TYPE_NOSEARCH)
      {
         current_search_type = SEARCH_TYPE_OTA_UPDATE;
      }

      retval = STB_OTAStartLoader(path, ADB_GetTransportOriginalNetworkId(transport),
         ADB_GetTransportTid(transport), serv_id, ADB_INVALID_DVB_ID, version_callback);
      if (!retval && (current_search_type == SEARCH_TYPE_OTA_UPDATE))
      {
         current_search_type = SEARCH_TYPE_NOSEARCH;
      }

      DBG_SSU(("%s: Already tuned to transport, OTA search %s", __FUNCTION__,
         (retval ? "started" : "failed")));

      ota_settings.ota_started = retval;
   }

   FUNCTION_FINISH(ACTL_StartOTAUpdate);

   return(retval);
}

/**
 * @brief   Once an OTA update has been found or not then this function must be called
 *          to continue or terminate the update.
 * @param   do_update TRUE if the update is to be downloaded and installed, FALSE otherwise
 */
void ACTL_ContinueOTAUpdate(BOOLEAN do_update)
{
   FUNCTION_START(ACTL_ContinueOTAUpdate);

   DBG_SSU(("ACTL_ContinueOTAUpdate(%s)", do_update ? "TRUE" : "FALSE"));

   //ASI_RefuseSSU(!do_update);
   STB_OTAContinueDownload(do_update);

   FUNCTION_FINISH(ACTL_ContinueOTAUpdate);
}

/**
 * @brief   Stops an OTA update
 */
void ACTL_StopOTAUpdate(void)
{
   FUNCTION_START(ACTL_StopOTAUpdate);

   DBG_SSU(("ACTL_StopOTAUpdate"));

   STB_OTAStopLoader();

   if (current_search_type == SEARCH_TYPE_OTA_UPDATE)
   {
      current_search_type = SEARCH_TYPE_NOSEARCH;
   }

   ota_settings.ota_onet_id = ADB_INVALID_DVB_ID;
   ota_settings.ota_tran_id = ADB_INVALID_DVB_ID;
   ota_settings.ota_serv_id = ADB_INVALID_DVB_ID;
   ota_settings.ota_started = FALSE;

   FUNCTION_FINISH(ACTL_StopOTAUpdate);
}
#endif /* INCLUDE_OTA_SSU */

/**
 * @brief   Sets the MHEG5 audio volume adjust to be applied
 * @param   control TRUE if MHEG has control of Audio/Video
 */
void ACTL_SetMhegAVControl(BOOLEAN control)
{
   FUNCTION_START(ACTL_SetMhegAVControl);
   external_control = control;
   FUNCTION_FINISH(ACTL_SetMhegAVControl);
}

/**
 * @brief   Sets function retrieve Subtitle PID and info. Allows external MHEG5 module to
 *          override the default function ADB_ServiceGetSubtitlePid.
 * @param   func Function to get subtitle info, NULL if use default
 */
void ACTL_SetMhegSubtitlePidFunc(F_ServiceGetSubtitlePid func)
{
   if (func == NULL)
   {
      GetSubtitlePidFunc = ADB_ServiceGetSubtitlePid;
   }
   else
   {
      GetSubtitlePidFunc = func;
   }
}

/**
 * @brief   Sets the audio volume scaling to be applied
 * @param   Adjusting the volume from -100% to 100%, i.e. adjusting it to silent or double.
 */
void ACTL_SetVolumeAdjustment(S8BIT scaling)
{
   FUNCTION_START(ACTL_SetVolumeAdjustment);

   volume_scaling = scaling;
   if(volume_scaling < -100)
     volume_scaling = -100;
   if(volume_scaling > 100)
     volume_scaling = 100;

   /* update output volume, given new scaling */
   AVSetAudioVolumes(FALSE);

   FUNCTION_FINISH(ACTL_SetVolumeAdjustment);
}

/**
 * @brief   Returns the current audio volume
 * @return  The current volume on a scale of 0-100.
 */
U8BIT ACTL_GetVolume(void)
{
   FUNCTION_START(ACTL_GetVolume);
   FUNCTION_FINISH(ACTL_GetVolume);
   return(audio_volume);
}

/**
 * @brief   Sets the main audio volume and returns the new volume
 * @param   volume unsigned value giving volume to be applied
 * @return  The new volume on a scale of 0-100.
 */
U8BIT ACTL_SetVolume(U8BIT volume)
{
   S8BIT volume_change;
   U8BIT new_volume;

   FUNCTION_START(ACTL_SetVolume);

   volume_change = (S8BIT)volume - audio_volume;
   if (volume_change != 0)
   {
      new_volume = ACTL_ChangeVolume(volume_change);
   }
   else
   {
      new_volume = audio_volume;
   }

   FUNCTION_FINISH(ACTL_SetVolume);

   return(new_volume);
}

/**
 * @brief   Changes the main audio volume and returns the new volume
 * @param   volume_change signed value giving change in volume to be applied
 * @return  The new volume on a scale of 0-100.
 */
U8BIT ACTL_ChangeVolume(S8BIT volume_change)
{
   FUNCTION_START(ACTL_ChangeVolume);

   if (volume_change != 0)
   {
      /* Increase volume */
      if (audio_muted)
      {
         audio_muted = FALSE;
         AVSetAudioVolumes(FALSE);
      }
      else
      {
         if (volume_change > 0)
         {
            audio_volume += volume_change;
            if (audio_volume > 100)
            {
               audio_volume = 100;
            }
         }
         else
         {
            if (audio_volume < -volume_change)
            {
               audio_volume = 0;
            }
            else
            {
               audio_volume += volume_change;
            }
         }

         AVSetAudioVolumes(TRUE);
      }
   }

   FUNCTION_FINISH(ACTL_ChangeVolume);

   return(audio_volume);
}

/**
 * @brief   Sets the audio mute state
 * @param   mute TRUE to set mute, FALSE otherwise
 */
void ACTL_SetMute(BOOLEAN mute)
{
   FUNCTION_START(ACTL_SetMute);

   if (mute)
   {
      audio_muted = TRUE;
   }
   else
   {
      audio_muted = FALSE;
   }
   AVSetAudioVolumes(FALSE);

   FUNCTION_FINISH(ACTL_SetMute);
}

/**
 * @brief   Toggles the current mute state and returns the new mute setting
 * @return  The new mute setting, TRUE = muted
 */
BOOLEAN ACTL_ToggleMute(void)
{
   FUNCTION_START(ACTL_ToggleMute);

   if (audio_muted)
   {
      audio_muted = FALSE;
   }
   else
   {
      audio_muted = TRUE;
   }
   AVSetAudioVolumes(FALSE);

   FUNCTION_FINISH(ACTL_ToggleMute);

   return(audio_muted);
}

/**
 * @brief   Returns the muted state of the audio
 * @return  TRUE if muted, FALSE otherwise
 */
BOOLEAN ACTL_IsMuted(void)
{
   FUNCTION_START(ACTL_IsMuted);
   FUNCTION_FINISH(ACTL_IsMuted);
   return(audio_muted);
}

/**
 * @brief   Acquires a decode path suitable for tuning to the given transport
 * @param   t_ptr the transport the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be acquired with the path, FALSE otherwise
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the acquired path, should be NULL for live TV (used by CI+)
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT ACTL_AcquirePathForTransport(void *t_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info)
{
   U8BIT path;

   FUNCTION_START(ACTL_AcquirePathForService);

   path = AcquirePathForTransport(t_ptr, with_decoders, for_recording, owner_info);

   FUNCTION_FINISH(ACTL_AcquirePathForService);

   return(path);
}

/**
 * @brief   Acquires a decode path suitable for tuning to the given service
 * @param   s_ptr the service the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be acquired with the path, FALSE otherwise
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the acquired path, should be NULL for live TV (used by CI+)
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT ACTL_AcquirePathForService(void *s_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info)
{
   U8BIT path;

   FUNCTION_START(ACTL_AcquirePathForService);

   path = AcquirePathForService(s_ptr, with_decoders, for_recording, owner_info);

   FUNCTION_FINISH(ACTL_AcquirePathForService);

   return(path);
}

/**
 * @brief   Finds the path being used to view the given service and releases it
 * @param   s_ptr service pointer
 * @param   owner owner requesting the path to be released
 */
void ACTL_ReleaseLivePathForService(void *s_ptr, E_STB_DP_RES_OWNER owner)
{
   U8BIT path;

   FUNCTION_START(ACTL_ReleaseLivePathForService);

   for (path = 0; path < STB_DPGetNumPaths(); path++)
   {
      if (STB_DPIsLivePath(path) && (STB_DPGetTunedService(path) == s_ptr))
      {
         ACTL_TuneOff(path);
         STB_DPReleasePath(path, owner);
         break;
      }
   }

   FUNCTION_FINISH(ACTL_ReleaseLivePathForService);
}

/**
 * @brief   Releases the decode path and all resources no longer needed.
 *          The path may not be released if the path is owned by a module
 * @param   path decode path
 * @return  TRUE if the path is released, FALSE otherwise
 */
BOOLEAN ACTL_ReleasePath(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(ACTL_ReleasePath);

#ifdef COMMON_INTERFACE
   if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
   {
      void *owner_data;
      U32BIT data_size;
      if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
      {
         retval = ACI_AskRelease(*(U32BIT *)owner_data);
      }
      else
      {
         retval = FALSE;
      }
   }
   else
#endif
   {
      retval = STB_DPReleasePath(path, RES_OWNER_NONE);
   }

   FUNCTION_FINISH(ACTL_ReleasePath);

   return(retval);
}

/**
 * @brief   Attempts to take ownership of the given path (used by CI+)
 * @param   path decode path to be owned
 * @param   owner_info owner info for the path
 * @return  TRUE if the ownership is set, FALSE otherwise
 */
BOOLEAN ACTL_AcquirePathOwnership(U8BIT path, S_ACTL_OWNER_INFO *owner_info)
{
   BOOLEAN retval;

   FUNCTION_START(ACTL_AcquirePathOwnership);

   retval = FALSE;

   if (STB_DPIsOwnedBy(path, owner_info->owner))
   {
      /* Path is already owned by the module */
      retval = TRUE;
   }
   else if (STB_DPIsOwnedBy(path, RES_OWNER_NONE))
   {
      /* Ownership of the path can be taken, so set the owner and save any owner data */
      STB_DPSetOwner(path, owner_info->owner);
      STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);
      retval = TRUE;
   }

   FUNCTION_FINISH(ACTL_AcquirePathOwnership);

   return(retval);
}

/**
 * @brief   Releases ownership of the path, and frees any associated data,
 *          if the given owner is the path's owner (used by CI+)
 * @param   path decode path
 * @param   owner owner releasing ownership
 * @return  TRUE if the ownership is released or the path isn't owned, FALSE otherwise
 */
BOOLEAN ACTL_ReleasePathOwnership(U8BIT path, E_STB_DP_RES_OWNER owner)
{
   BOOLEAN retval;

   FUNCTION_START(ACTL_ReleasePathOwnership);

   retval = FALSE;

   if (STB_DPIsOwnedBy(path, RES_OWNER_NONE))
   {
      /* Path doesn't have an owner */
      retval = TRUE;
   }
   else if (STB_DPIsOwnedBy(path, owner))
   {
      /* The module does own the path so is allowed to release its ownership of it */
      STB_DPSetOwner(path, RES_OWNER_NONE);
      STB_DPSetOwnerData(path, NULL, 0);
      retval = TRUE;
   }

   FUNCTION_FINISH(ACTL_ReleasePathOwnership);

   return(retval);
}

/**
 * @brief   Checks whether there's a tuner available to view the given service.
 *          This takes into account whether tuners are being used for recording
 *          or have been acquired by CI+.
 * @param   s_ptr service being checked
 * @return  TRUE if service can be viewed, FALSE otherwise
 */
BOOLEAN ACTL_CanServiceBeViewed(void *s_ptr)
{
   void *t_ptr;
   BOOLEAN viewable;

   FUNCTION_START(ACTL_CanServiceBeViewed);

   viewable = FALSE;

   t_ptr = ADB_GetServiceTransportPtr(s_ptr);
   if (t_ptr != NULL)
   {
      viewable = STB_DPCanTuneTo(ADB_GetTransportSignalType(t_ptr), s_ptr, t_ptr);
   }

   FUNCTION_FINISH(ACTL_CanServiceBeViewed);

   return(viewable);
}

/**
 * @brief   Sets a grace period during which a system will enter active standby before going
 *          into passive or low power standby. This allows a system to be brought back out of
 *          standby quickly for a period of time.
 * @param   num_seconds grace period in seconds
 */
void ACTL_SetStandbyGracePeriod(U16BIT num_seconds)
{
   FUNCTION_START(ACTL_SetStandbyGracePeriod);
   standby_grace_timeout = num_seconds;
   FUNCTION_FINISH(ACTL_SetStandbyGracePeriod);
}

/**
 * @brief   Puts DVB into standby mode. It will continue to monitor SI for recordings,
 *          SSU updates, etc., unless it goes into low power standby
 */
void ACTL_EnterStandby(void)
{
   U8BIT path, num_paths;
   void *s_ptr;
   U32BIT recording_handle;
   U8BIT hours, mins, secs;
   U16BIT grace_timeout;
   U32DHMS enter_standby_time;

   FUNCTION_START(ACTL_EnterStandby);

   DBG_STDBY(("%s", __FUNCTION__));

   ACTL_StopSubtitles();

   if (APVR_IsTimeshiftStarted())
   {
      /* Hide video before entering standby */
      STB_AVBlankVideo(0, TRUE);

      /* Stop timeshift before entering standby */
      APVR_StopPauseRecord(FALSE);
   }

   /* Turn the live path off if it isn't being used for recording */
   if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
   {
      /* Save the current live service LCN to nvm */
      if ((s_ptr = ADB_GetTunedService(path)) != NULL)
      {
         DBG_STDBY(("ACTL_EnterStandby: saving service to NVM... LCN=%d", ADB_GetServiceLcn(s_ptr)));
         APP_NvmSave(LIVE_SERVICE_LCN_NVM, ADB_GetServiceLcn(s_ptr), TRUE);
      }

      if (!STB_DPIsRecordingPath(path))
      {
         ACTL_TuneOff(path);
      }

      STB_DPReleasePath(path, RES_OWNER_NONE);
   }

   /* This function can be called many times without ACTL_LeaveStandby being called, such as
    * when a recording completes whilst in standby, so release the original path that was
    * acquired to monitor the SI for that recording as another one will be acquired if
    * another recording is due to start soon */
   if ((standby_path != INVALID_RES_ID) && !APVR_GetRecordingHandle(standby_path, &recording_handle))
   {
      /* Release the decode path that was acquired when entering standby */
      STB_DPReleasePath(standby_path, RES_OWNER_NONE);
   }

   standby_path = INVALID_RES_ID;

   ACTL_SetStandbyState(TRUE);

   if (APVR_IsRecordingInProgress())
   {
      DBG_STDBY(("Recording in progress, entering active standby"));
      APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_RECORDING, TRUE);
      ASTE_EnterStandby(TRUE);
      STB_HWSetStandbyState(HW_STANDBY_ACTIVE);
   }
   else
   {
      ASTE_EnterStandby(FALSE);

      /* Check whether standby should be entered; it may be time to start a background search,
       * or a recording may be starting soon */
      if (EnterStandby(&s_ptr))
      {
         /* Save database before going into standby as this may result in a power off */
         ADB_SaveDatabase();

         /* Save the EIT data, if supported by the database being used */
         ADB_SaveEventSchedule(SIGNAL_NONE, ACTL_GetCurrentSatellite(STB_DPGetLivePath()));

         /* Turn off all tuners and release all paths, whatever they're being used for */
         num_paths = STB_DPGetNumPaths();

         for (path = 0; path < num_paths; path++)
         {
            ACTL_TuneOff(path);
         }

         STB_DPReleaseAllPaths();

#ifdef COMMON_INTERFACE
         hours = standby_grace_timeout / 3600;
         mins = (standby_grace_timeout % 3600) / 60;
         secs = standby_grace_timeout % 60;
         start_standby_time = STB_GCCalculateDHMS(STB_GCNowDHMSGmt(),
            STB_GCCreateDHMS(0, hours, mins, secs), CALC_ADD);

         if (STB_CIStartPowerDown(CIPowerDownReadyCallback))
         {
            /* according to CI+ 1.4, must have at least 30 seconds */
            grace_timeout = (standby_grace_timeout > 30)? standby_grace_timeout : 30;
         }
         else
         {
            grace_timeout = standby_grace_timeout;
         }
#else
         grace_timeout = standby_grace_timeout;
#endif
         if (grace_timeout != 0)
         {
            /* Create a private timer for the standby grace period, which is defined in seconds,
             * and enter the active standby state until the grace period ends */
            hours = grace_timeout / 3600;
            mins = (grace_timeout % 3600) / 60;
            secs = grace_timeout % 60;

            DBG_STDBY(("Starting standby grace period of %02u:%02u:%02u", hours, mins, secs));

            enter_standby_time = STB_GCCalculateDHMS(STB_GCNowDHMSGmt(),
               STB_GCCreateDHMS(0, hours, mins, secs), CALC_ADD);

            standby_grace_timer = CreateTimer(enter_standby_time);

            STB_HWSetStandbyState(HW_STANDBY_ACTIVE);
         }
         else
         {
            if ((BOOLEAN)APP_NvmRead(STANDBY_POWERSAVE_NVM))
            {
               /* Put box into low-power standby mode */
               DBG_STDBY(("Entering low power standby"));
               STB_HWSetStandbyState(HW_STANDBY_LOWPOWER);
            }
            else
            {
               /* Nothing is being started so go into the lowest active standby state */
               DBG_STDBY(("Entering passive standby"));
               STB_HWSetStandbyState(HW_STANDBY_PASSIVE);
            }
         }
      }
      else if (s_ptr != NULL)
      {
         /* A service to be recorded has been returned, so tune to it to start monitoring the SI */
         if ((standby_path = AcquirePathForService(s_ptr, FALSE, FALSE, NULL)) != INVALID_RES_ID)
         {
            ACTL_TuneToService(standby_path, NULL, s_ptr, FALSE, TRUE);
         }

         /* Entering standby but just with the outputs turned off */
         DBG_STDBY(("Recording due to start soon, entering active standby"));
         APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_WAKE_FOR_RECORDING, TRUE);
         STB_HWSetStandbyState(HW_STANDBY_ACTIVE);
      }
      else
      {
         /* A background search has been requested so just go into active standby */
         DBG_STDBY(("Background search due to start, entering active standby"));
         APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_WAKE_FOR_SEARCH, TRUE);
         STB_HWSetStandbyState(HW_STANDBY_ACTIVE);
      }
   }

   FUNCTION_FINISH(ACTL_EnterStandby);
}

/**
 * @brief   Brings the DVB out of standby mode
 * @param   s_ptr service to be tuned to for live viewing. If NULL then the last
 *          service viewed will be restored
 * @param   tune_to_service if s_ptr is NULL then this boolean is used to define
 *          whether to tune to the last service viewed
 */
void ACTL_LeaveStandby(void *s_ptr, BOOLEAN tune_to_service)
{
   U16BIT lcn;
   U16BIT num_services;
   void **slist;
   U8BIT path;
   U32BIT recording_handle;

   FUNCTION_START(ACTL_LeaveStandby);

   DBG_STDBY(("%s(%p, tune_to_service=%u)", __FUNCTION__, s_ptr, tune_to_service));

   ASTE_LeaveStandby();

   /* May have to do something if searches are being performed when leaving standby */
   if (required_search_type != SEARCH_TYPE_NOSEARCH)
   {
      switch (required_search_type)
      {
         case SEARCH_TYPE_SERVICE_FREQ:
         case SEARCH_TYPE_SERVICE_NETWORK:
            /* Stop the search but save the results so far */
            ACTL_StopServiceSearch();
            ADB_FinaliseDatabaseAfterSearch(TRUE, signal_type_required, NULL, FALSE, FALSE, FALSE);
            break;

         case SEARCH_TYPE_EIT_PF:
         case SEARCH_TYPE_EIT_SCHED:
         case SEARCH_TYPE_EIT_PF_SCHED:
            ACTL_StopEitSearch();
            break;

         case SEARCH_TYPE_SSU:
            /* The SSU search can't be stopped */
            break;

         default:
            break;
      }
   }

   STB_HWSetStandbyState(HW_STANDBY_ON);

   ACTL_SetStandbyState(FALSE);

   STB_PVRSetStandbyState(FALSE);

   /* The screen aspect and/or resolution may have been changed whilst DVBCore was in standby,
    * so read the current settings and ensure the stack knows about any changes */
   if (ACTL_IsHDMIConnected())
   {
      ACTL_UpdateVideoMode((E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), TRUE);
   }

   /* Volume may have been changed while in standby, so re-read the settings so they can be applied
    * when tuning to the first service */
   audio_volume = (U8BIT)APP_NvmRead(VOLUME_NVM);
   ad_volume = (U8BIT)APP_NvmRead(AD_VOLUME_NVM);

   DBG_STDBY(("Setting STANDBY_STATE_NVM=STDBY_POWER_ON"));
   APP_NvmSave(STANDBY_STATE_NVM, STDBY_POWER_ON, TRUE);

   ACTL_EnableCiModule();

   if (standby_wakeup_timer != INVALID_TIMER_HANDLE)
   {
      ATMR_DeleteTimer(standby_wakeup_timer);
      standby_wakeup_timer = INVALID_TIMER_HANDLE;
   }

   if (standby_grace_timer != INVALID_TIMER_HANDLE)
   {
      ATMR_DeleteTimer(standby_grace_timer);
      standby_grace_timer = INVALID_TIMER_HANDLE;
   }

   /* Can only tune to a service if there isn't a search still in progress */
   if (required_search_type == SEARCH_TYPE_NOSEARCH)
   {
      if ((s_ptr == NULL) && tune_to_service)
      {
         /* Restore the last service that was being viewed */
         lcn = (U16BIT)APP_NvmRead(LIVE_SERVICE_LCN_NVM);
         DBG_STDBY(("ACTL_LeaveStandby: LCN in NVM=%u", lcn));
         if ((s_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, lcn, FALSE)) == NULL)
         {
            /* Use the first available TV service */
            ADB_GetServiceList(ADB_SERVICE_LIST_TV_DATA, &slist, &num_services);
            if ((slist != NULL) && (num_services > 0))
            {
               s_ptr = slist[0];
#ifdef DEBUG_STANDBY
               if (s_ptr != NULL)
               {
                  DBG_STDBY(("ACTL_LeaveStandby: LCN not found, use first TV service, serv_id %u",
                     ADB_GetServiceId(s_ptr)));
               }
#endif

               ADB_ReleaseServiceList(slist, num_services);
            }

            if (s_ptr == NULL)
            {
               /* Use the first available service, whatever it is! */
               ADB_GetServiceList(ADB_SERVICE_LIST_ALL, &slist, &num_services);
               if ((slist != NULL) && (num_services > 0))
               {
                  s_ptr = slist[0];
#ifdef DEBUG_STANDBY
                  if (s_ptr != NULL)
                  {
                     DBG_STDBY(("ACTL_LeaveStandby: LCN not found, use first service, serv_id %u",
                        ADB_GetServiceId(s_ptr)));
                  }
#endif
                  ADB_ReleaseServiceList(slist, num_services);
               }
            }
         }
#ifdef DEBUG_STANDBY
         else
         {
            DBG_STDBY(("ACTL_LeaveStandby: LCN %u found, serv_id %u", lcn, ADB_GetServiceId(s_ptr)));
         }
#endif
      }

      if ((standby_path != INVALID_RES_ID) && !APVR_GetRecordingHandle(standby_path, &recording_handle))
      {
         /* Release the decode path that was acquired when entering standby */
         STB_DPReleasePath(standby_path, RES_OWNER_NONE);
      }

      standby_path = INVALID_RES_ID;

      if (s_ptr != NULL)
      {
         if (!ACTL_CanServiceBeViewed(s_ptr))
         {
            /* The service can't be viewed so use the service of the live path, if it exists */
            path = STB_DPGetLivePath();
            if (path != INVALID_RES_ID)
            {
               s_ptr = ADB_GetTunedService(path);
#ifdef DEBUG_STANDBY
               if (s_ptr != NULL)
               {
                  DBG_STDBY(("%s: can't tune to service so tuning to live service %u", __FUNCTION__,
                     ADB_GetServiceId(s_ptr)));
               }
#endif
            }
            else
            {
               /* There's no live path so tune to the first path that's tuned to a service */
               for (path = 0, s_ptr = NULL; (path < STB_DPGetNumPaths()) && (s_ptr == NULL); path++)
               {
                  s_ptr = ADB_GetTunedService(path);
               }
#ifdef DEBUG_STANDBY
               if (s_ptr != NULL)
               {
                  DBG_STDBY(("%s: can't tune to service so tuning to service %u", __FUNCTION__,
                     ADB_GetServiceId(s_ptr)));
               }
#endif
            }
         }

         if (s_ptr != NULL)
         {
            ACTL_TuneToService(INVALID_RES_ID, NULL, s_ptr, FALSE, TRUE);
         }
#ifdef DEBUG_STANDBY
         else
         {
            DBG_STDBY(("%s: No service to tune to!", __FUNCTION__));
         }
#endif
      }
   }

   FUNCTION_FINISH(ACTL_LeaveStandby);
}

/**
 * @brief   Checks whether the service tuned to on the live path has changed
 *          and informs the UI if it has.
 * @return  TRUE if the service has changed, FALSE otherwise
 */
BOOLEAN ACTL_CheckLiveServiceChange(void)
{
   BOOLEAN retval;
   U8BIT live_path;
   void *tuned_service;

   FUNCTION_START(ACTL_CheckLiveServiceChange);

   retval = FALSE;

   if (((live_path = STB_DPGetLivePath()) != INVALID_RES_ID) && (current_service[live_path] != NULL))
   {
      tuned_service = ADB_GetTunedService(live_path);

      if (tuned_service != current_service[live_path])
      {
         /* The live service has been changed, inform the UI */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED, &tuned_service, sizeof(void *));

         /* Update the current service and transport */
         current_service[live_path] = tuned_service;
         current_transport[live_path] = ADB_GetTunedTransport(live_path);

         retval = TRUE;
      }
   }

   FUNCTION_FINISH(ACTL_CheckLiveServiceChange);

   return(retval);
}

/**
 * @brief   Sets the current profile. (CI+ only)
 *          If the profile is being set to CI+ profile and the necessary CAM isn't
 *          present, then setting the profile will fail.
 * @param   profile profile to be set
 * @return  TRUE if the profile is set successfully, FALSE otherwise
 */
BOOLEAN ACTL_SetActiveProfile(void *profile)
{
   BOOLEAN retval;
   ADB_PROFILE_TYPE type;
#ifdef COMMON_INTERFACE
   U16BIT cicam_onet_id;
   U32BIT cicam_id;
#endif

   FUNCTION_START(ACTL_SetActiveProfile);

   retval = FALSE;

   if (profile != NULL)
   {
#ifdef COMMON_INTERFACE
      if (DBDEF_GetCurrentProfileType() == ADB_PROFILE_TYPE_CIPLUS)
      {
         /* Changing from the current CI+ profile, so need to exit it */
         ACI_OperatorExit();
      }
#endif

      type = ADB_GetProfileType(profile);
      if (type == ADB_PROFILE_TYPE_BROADCAST)
      {
         DBDEF_SelectBroadcastProfile();
         retval = TRUE;
      }
#ifdef COMMON_INTERFACE
      else if (type == ADB_PROFILE_TYPE_CIPLUS)
      {
         if (ADB_GetProfileModulePresent(profile))
         {
            /* Send the request to enter the CI+ operator profile */
            if (ACI_RequestOperatorStatus(ADB_GetProfileModule(profile)))
            {
               ADB_GetProfileCAMInfo(profile, &cicam_onet_id, &cicam_id);
               DBDEF_SelectCIPlusProfile(cicam_onet_id, cicam_id);
               retval = TRUE;
            }
         }
      }
#endif
   }

   FUNCTION_FINISH(ACTL_SetActiveProfile);

   return(retval);
}

/**
 * @brief   Setup default audio decoding params on current tuned service
 *          Can be used by external module (e.g. MHEG5)
 * @param   path decode path to be owned
 * @param   s_ptr service pointer
 * @return  E_ACTL_DECODE_CHANGE - indicate whether been change of decode params
 */
E_ACTL_DECODE_CHANGE ACTL_SetupAudioDecoding(U8BIT path, void *s_ptr)
{
   E_ACTL_DECODE_CHANGE change;
   E_STB_DP_AUDIO_MODE mode;
   E_STB_DP_AUDIO_CODEC codec;
   ADB_STREAM_TYPE type;
   U16BIT pid;
   BOOLEAN broadcast_mix;

   FUNCTION_START(ACTL_SetupAudioDecoding);

   change = ACTL_NO_CHANGE;
   DBDEF_RequestAccess();
   if (!ACTL_IsAudioDescriptionOn())
   {
      pid = DBDEF_GetReqdAudioPid(s_ptr, &mode, &type);
   }
   else
   {
      pid = DBDEF_GetReqdADPid(s_ptr, &mode, &type, &broadcast_mix);
      if (broadcast_mix)
      {
         STB_DPSetADPID(path, 0);
      }
      else
      {
         codec = ADB_GetAudioCodecFromStream(type);
         if (pid != STB_DPGetADPID(path) || mode != STB_DPGetADMode(path) || codec != STB_DPGetADCodec(path))
         {
            change |= ACTL_ADESC_CHANGE;
         }
         STB_DPSetADPID(path, pid);
         STB_DPSetADMode(path, mode);
         STB_DPSetADCodec(path, ADB_GetAudioCodecFromStream(type));
         pid = DBDEF_GetReqdAudioPid(s_ptr, &mode, &type);
      }
   }
   //((ADB_SERVICE_REC *)s_ptr)->audio_pid = pid; /* save current audio pid */
   DBDEF_ReleaseAccess();

   codec = ADB_GetAudioCodecFromStream(type);
   if (pid != STB_DPGetAudioPID(path) || mode != STB_DPGetAudioMode(path) || codec != STB_DPGetAudioCodec(path))
   {
      change |= ACTL_AUDIO_CHANGE;
   }
   STB_DPSetAudioPID(path, pid);
   STB_DPSetEcmPIDs(
      path,
      ((ADB_SERVICE_REC *)s_ptr)->ecm_pid,
      ((ADB_SERVICE_REC *)s_ptr)->video_ecm_pid,
      ((ADB_SERVICE_REC *)s_ptr)->audio_ecm_pid,
      ((ADB_SERVICE_REC *)s_ptr)->ttext_ecm_pid,
      ((ADB_SERVICE_REC *)s_ptr)->data_ecm_pid,
      ((ADB_SERVICE_REC *)s_ptr)->ad_ecm_pid
   );
   
   STB_DPSetAudioMode(path, mode);
   STB_DPSetAudioCodec(path, codec);

   STB_DPSetPCRPID(path, ADB_GetServicePCRPid(s_ptr));

   FUNCTION_FINISH(ACTL_SetupAudioDecoding);
   return change;
}

BOOLEAN ACTL_SwitchLivePath(U8BIT fromPath, U8BIT toPath)
{
   if ((NULL == current_service[toPath]) || (FALSE == current_service[toPath]->pmt_received)) {
      return FALSE;
   }

   STB_DPSwitchLivePath(fromPath, toPath);

#if 0
   // ACTL exchange path variables
   void ** pp;
   void * p;
   
   pp = transport_required[fromPath];
   transport_required[fromPath] = transport_required[toPath];
   transport_required[toPath] = pp;

   pp = service_required[fromPath];
   service_required[fromPath] = service_required[toPath];
   service_required[toPath] = pp;
 
   p = decoding_state[fromPath];
   decoding_state[fromPath] = decoding_state[toPath];
   decoding_state[toPath] = p;   

   pp = cntrl_mutex[fromPath];
   cntrl_mutex[fromPath] = cntrl_mutex[toPath];
   cntrl_mutex[toPath] = pp;

   pp = current_service[fromPath];
   current_service[fromPath] = current_service[toPath];
   current_service[toPath] = pp;

   pp = current_transport[fromPath];
   current_transport[fromPath] = current_transport[toPath];
   current_transport[toPath] = pp;

   p = current_si[fromPath];
   current_si[fromPath] = current_si[toPath];
   current_si[toPath] = p;
#endif

   E_ACTL_DECODING_STATE state = decoding_state[fromPath];
   decoding_state[fromPath] = decoding_state[toPath];
   decoding_state[toPath] = state;
   
   BOOLEAN dl = decoding_locked[fromPath];
   decoding_locked[fromPath] = decoding_locked[toPath];
   decoding_locked[toPath] = dl;

   BOOLEAN ds = decoding_started[fromPath];
   decoding_started[fromPath] = decoding_started[toPath];
   decoding_started[toPath] = ds;

   SetupForDecoding(toPath);
   STB_DPStartDecoding(toPath | (1<<7));  // set the preloading bit
   
   return TRUE;
}

//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------

/**
 *

 *
 * @brief   Used to convert event type and id into a string to be used in debug messages
 *
 * @param   type  - type of event i.e. INTERNAL_EVENT or EXTERNAL_EVENT
 * @param   event - the event to be actioned
 *
 * @return   pointer to the appropriate string
 *
 */
#if defined(DEBUG_TUNING) || defined(DEBUG_DECODING) || defined(DEBUG_AV_OUTPUT) || defined(DEBUG_MHEG)
static U8BIT* GetEventDebugString(E_ACTL_EVENT_TYPE type, U32BIT event)
{
   U8BIT *string;

   if (type == INTERNAL_EVENT)
   {
      switch (event)
      {
         case INT_EVENT_INITIALISE:                {string = (U8BIT *)"initialise"; break; }
         case INT_EVENT_ENABLE:                    {string = (U8BIT *)"enable"; break; }
         case INT_EVENT_DISABLE:                   {string = (U8BIT *)"disable"; break; }
         case INT_EVENT_UPDATE:                    {string = (U8BIT *)"update"; break; }
         case INT_EVENT_RF_CHAN_TUNE_REQD:         {string = (U8BIT *)"rf tune"; break; }
         case INT_EVENT_TPTR_TUNE_REQD:            {string = (U8BIT *)"tp tune"; break; }
         case INT_EVENT_USERDEFINED_TUNE_REQD:     {string = (U8BIT *)"user tune"; break; }
         case INT_EVENT_TUNE_OFF:                  {string = (U8BIT *)"tune off"; break; }
         case INT_EVENT_RESTART_DECODING:          {string = (U8BIT *)"restart decode"; break; }
         case INT_EVENT_RESTART_AUDIO:             {string = (U8BIT *)"restart audio"; break; }
         case INT_EVENT_RESTART_SUBTITLES:         {string = (U8BIT *)"restart subtitles"; break; }
         case INT_EVENT_RELEASE_DECODE_LOCK:       {string = (U8BIT *)"release decode lock"; break; }
         case INT_EVENT_STANDBY_ON:                {string = (U8BIT *)"standby on"; break; }
         case INT_EVENT_STANDBY_OFF:               {string = (U8BIT *)"standby off"; break; }
         case INT_EVENT_STANDBY_VCR_ACTIVE:        {string = (U8BIT *)"standby vcr active"; break; }
         case INT_EVENT_ALT_AV_ALLOWED:            {string = (U8BIT *)"scart allowed"; break; }
         case INT_EVENT_ALT_AV_NOT_ALLOWED:        {string = (U8BIT *)"scart not allowed"; break; }
         case INT_EVENT_ANALOG_TV_ON:              {string = (U8BIT *)"analogue tv on"; break; }
         case INT_EVENT_ANALOG_TV_OFF:             {string = (U8BIT *)"analogue tv off"; break; }
         case INT_EVENT_ANALOG_VIDEO_ALLOWED:      {string = (U8BIT *)"analogue video allowed"; break; }
         case INT_EVENT_ANALOG_VIDEO_NOT_ALLOWED:  {string = (U8BIT *)"analogue video not allowed"; break; }
         default:                                  {string = (U8BIT *)"unknown internal event"; break; }
      }
   }
   else  // i.e. external event
   {
      switch (event)
      {
         case STB_EVENT_TUNE_LOCKED:            {string = (U8BIT *)"TUNE LOCKED"; break; }
         case STB_EVENT_TUNE_NOTLOCKED:         {string = (U8BIT *)"TUNE NOT LOCKED"; break; }
         case STB_EVENT_TUNE_SIGNAL_DATA_BAD:   {string = (U8BIT *)"SIGNAL BAD"; break; }
         case STB_EVENT_TUNE_SIGNAL_DATA_OK:    {string = (U8BIT *)"SIGNAL OK"; break; }
         case STB_EVENT_AUDIO_DECODE_STARTED:   {string = (U8BIT *)"AUDIO DECODE STARTED"; break; }
         case STB_EVENT_VIDEO_DECODE_STARTED:   {string = (U8BIT *)"VIDEO DECODE STARTED"; break; }
         case STB_EVENT_AUDIO_DECODE_STOPPED:   {string = (U8BIT *)"AUDIO DECODE STOPPED"; break; }
         case STB_EVENT_VIDEO_DECODE_STOPPED:   {string = (U8BIT *)"VIDEO DECODE STOPPED"; break; }
         case STB_EVENT_DECODE_LOCKED:          {string = (U8BIT *)"DECODE LOCKED"; break; }
#if 0
         case STB_EVENT_SCART_DISCONNECTED:     {string = (U8BIT *)"SCART OFF"; break; }
         case STB_EVENT_SCART_FORCE:            {string = (U8BIT *)"SCART ON"; break; }
#endif
         case STB_EVENT_CI_OPEN_MODULE:         {string = (U8BIT *)"CI UI ON"; break; }
         case STB_EVENT_CI_CLOSE_MODULE:        {string = (U8BIT *)"CI UI OFF"; break; }
         case APP_EVENT_SERVICE_NOT_RUNNING:    {string = (U8BIT *)"SERV NOT RUNNING"; break; }
         case APP_EVENT_SERVICE_RUNNING:        {string = (U8BIT *)"SERV RUNNING"; break; }
         case APP_EVENT_SERVICE_AUDIO_PID_UPDATE: {string = (U8BIT *)"AUDIO PID CHANGE"; break; }
         case APP_EVENT_SERVICE_VIDEO_PID_UPDATE: {string = (U8BIT *)"VIDEO PID CHANGE"; break; }
         case APP_EVENT_SERVICE_SUBTITLE_UPDATE: {string = (U8BIT *)"SUBTITLE CHANGE"; break; }
         case APP_EVENT_SERVICE_SCRAMBLE_CHANGE: {string = (U8BIT *)"SERV SCRAMBLE CHANGE"; break; }
         default:                               {string = (U8BIT *)"unknown external event"; break; }
      }
   }
   return(string);
}

#endif

#ifdef COMMON_INTERFACE
/**
 * @brief   Sends a request to the CICAM to release the path and prepares the internal status for
 *          the reply
 * @param   path Path owned by the CICAM
 * @param   s_ptr Service being tuned to, can be NULL if t_ptr isn't NULL
 * @param   owner Module that will own the path when it's acquired
 * @param   override_parental_lock Should parental lock be overridden when tuning to a service
 * @param   t_ptr Transport being tuned to, can be NULL if s_ptr isn't NULL
 * @param   reqd_si Type of SI search to be started when tuning to a transport
 * @param   relock_on Should tuner relock be applied when tuning to a transport
 */
static void AskRelease(U8BIT path, void *s_ptr, S_ACTL_OWNER_INFO *owner_info,
   BOOLEAN override_parental_lock, void *t_ptr, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on)
{
   void *owner_data;
   U32BIT data_size;

   /* Save the transport to be tuned to if the live path is eventually released */
   tuneto_live_service = s_ptr;
   decode_lock_override_required = override_parental_lock;
   tuneto_live_transport = t_ptr;
   tuneto_live_si = reqd_si;
   tuneto_relock = relock_on;
   tuneto_for_recording = FALSE;
   tuneto_owner_info = CopyOwnerInfo(owner_info);

   /* Request the path/tuner to be released */
   if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
   {
      ACI_AskRelease(*(U32BIT *)owner_data);
   }
   else
   {
      DBGPRINT("Need to ask for path %u to be released by CI+, but there's no owner data!", path);
   }
}

#endif /*COMMON_INTERFACE*/

/*!**************************************************************************
 * @brief   Handles the releasing and acquiring of a live path when tuning to a service or transport.
 *          The path value returned from here may be INVALID_RES_ID if a request has to be made to
 *          release the existing live path.
 * @param   s_ptr - service being tuned to, can be NULL if t_ptr isn't NULL
 * @param   owner - module that will own the path when it's acquired
 * @param   override_parental_lock - should parental lock be overridden when tuning to a service
 * @param   t_ptr - transport being tuned to, can be NULL if s_ptr isn't NULL
 * @param   reqd_si - type of SI search to be started when tuning to a transport
 * @param   relock_on - should tuner relock be applied when tuning to a transport
 * @return  live path, if acquired, or INVALID_RES_ID if a path can't be acquired or
 *          the existing path has been requested to be released.
 ****************************************************************************/
static U8BIT AcquireLivePath(void *s_ptr, S_ACTL_OWNER_INFO *owner_info,
   BOOLEAN override_parental_lock, void *t_ptr, E_ACTL_SI_SRCH_REQD reqd_si, BOOLEAN relock_on)
{
   E_STB_DP_SIGNAL_TYPE signal_type;
   U8BIT path;
   E_STB_DP_RES_OWNER owner;
#ifdef COMMON_INTERFACE
   BOOLEAN ask_release;
#endif

   FUNCTION_START(AcquireLivePath);

#ifndef COMMON_INTERFACE
   USE_UNWANTED_PARAM(override_parental_lock);
   USE_UNWANTED_PARAM(reqd_si);
   USE_UNWANTED_PARAM(relock_on);
#endif

   if (owner_info != NULL)
   {
      owner = owner_info->owner;
   }
   else
   {
      owner = RES_OWNER_NONE;
   }

   if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
   {
      /* Existing live path that needs to be released */
      if (STB_DPIsOwnedBy(path, owner) || STB_DPIsOwnedBy(path, RES_OWNER_NONE))
      {
         /* Live path has no owner or is owned by the module wanting to tune,
          * so release the existing live path and acquire a new one */
         if (current_service[path] != NULL)
         {
            ACTL_ReleaseLivePathForService(current_service[path], owner);
         }
         else
         {
            STB_DPReleasePath(path, owner);
         }

         if (s_ptr != NULL)
         {
            path = AcquirePathForService(s_ptr, TRUE, FALSE, owner_info);
         }
         else if (t_ptr != NULL)
         {
            signal_type = ADB_GetTransportSignalType(t_ptr);
            path = STB_DPAcquireTunerPath(signal_type, NULL, t_ptr, owner, DP_PRIORITY_LOW, FALSE, FALSE);

            if ((path != INVALID_RES_ID) && (owner_info != NULL))
            {
               /* Save any owner data with the new path */
               STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);

#ifdef COMMON_INTERFACE
               if (owner == RES_OWNER_CIPLUS)
               {
                  /* The path needs to include the CI slot of the module making the tuning request */
                  ACI_UseCiModuleOnPath(path, *(U32BIT *)owner_info->data);
               }
#endif
            }
         }

#ifdef COMMON_INTERFACE
         if ((path == INVALID_RES_ID) && (owner == RES_OWNER_CIPLUS))
         {
            ACI_TuneReply(path, *(U32BIT *)owner_info->data, CIP_TUNER_BUSY);
         }
#endif
      }
      else
      {
         /* Live path isn't owned by the module wanting to tune, so need to ask
          * for the resource to be released */
#ifdef COMMON_INTERFACE
         if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
         {
            AskRelease(path, s_ptr, owner_info, override_parental_lock, t_ptr, reqd_si, relock_on);

            /* Clear the path so tuning doesn't start now */
            path = INVALID_RES_ID;
         }
         else
#endif
         {
            STB_SPDebugWrite("[%s:%d]: Live path needs to be released but has an owner which isn't CI+\n",
               __FUNCTION__, __LINE__);
         }
      }
   }
   else
   {
      /* There's no current live path, so can just acquire one */
      if (s_ptr != NULL)
      {
         path = AcquirePathForService(s_ptr, TRUE, FALSE, owner_info);
      }
      else if (t_ptr != NULL)
      {
         /* If there is a (non-live) path in use by the requesting owner, release it like for the
            live path case */
         for (path = 0; path < STB_DPGetNumPaths(); path++)
         {
            if (STB_DPIsOwnedBy(path, owner) || STB_DPIsOwnedBy(path, RES_OWNER_NONE))
            {
               STB_DPReleasePath(path, owner);
               break;
            }
         }

         signal_type = ADB_GetTransportSignalType(t_ptr);
         path = STB_DPAcquireTunerPath(signal_type, NULL, t_ptr, owner, DP_PRIORITY_LOW, FALSE, FALSE);

         if ((path != INVALID_RES_ID) && (owner_info != NULL))
         {
            /* Save any owner data with the new path */
            STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);

#ifdef COMMON_INTERFACE
            if (owner == RES_OWNER_CIPLUS)
            {
               /* The path needs to include the CI slot of the module making the tuning request */
               ACI_UseCiModuleOnPath(path, *(U32BIT *)owner_info->data);
            }
#endif
         }
      }

#ifdef COMMON_INTERFACE
      if (path == INVALID_RES_ID)
      {
         ask_release = FALSE;
         for (path = 0; path < STB_DPGetNumPaths(); path++)
         {
            /* Live path isn't owned by the module wanting to tune, so need to ask
             * for the resource to be released */
            if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
            {
               ask_release = TRUE;
               AskRelease(path, s_ptr, owner_info, override_parental_lock, t_ptr, reqd_si,
                  relock_on);

               /* Clear the path so tuning doesn't start now */
               path = INVALID_RES_ID;
               break;
            }
            else

            {
               STB_SPDebugWrite("[%s:%d]: Live path needs to be released but has an owner which isn't CI+\n",
                  __FUNCTION__, __LINE__);
            }
         }

         if (!ask_release && (owner == RES_OWNER_CIPLUS))
         {
            /* If CI+ was trying to tune, report the failure */
            path = INVALID_RES_ID;
            ACI_TuneReply(path, *(U32BIT *)owner_info->data, CIP_TUNER_BUSY);
         }
      }
#endif
   }

   FUNCTION_FINISH(AcquireLivePath);

   return(path);
}

static U8BIT AcquireRecordingPath(void *s_ptr, S_ACTL_OWNER_INFO *owner_info)
{
   void *t_ptr;
   E_STB_DP_SIGNAL_TYPE sig_type;
   U8BIT path;
   E_STB_DP_RES_OWNER owner;
   BOOLEAN slot_acquired;
#ifdef COMMON_INTERFACE
   void *owner_data;
   U32BIT data_size;
#endif
   U8BIT *pmt_data;
   U16BIT data_len;
   U16BIT *ca_ids;
   U16BIT num_ca_ids;

   FUNCTION_START(AcquireRecordingPath);

   path = INVALID_RES_ID;

   if (owner_info != NULL)
   {
      owner = owner_info->owner;
   }
   else
   {
      owner = RES_OWNER_NONE;
   }

   /* Get the transport for the service to find out what tuner type is required */
   t_ptr = ADB_GetServiceTransportPtr(s_ptr);
   if (t_ptr != NULL)
   {
      sig_type = ADB_GetTransportSignalType(t_ptr);

      slot_acquired = FALSE;

      path = STB_DPAcquireTunerPath(sig_type, s_ptr, t_ptr, owner, DP_PRIORITY_HIGH, FALSE, TRUE);
      if (path != INVALID_RES_ID)
      {
         if (owner_info != NULL)
         {
            /* Save any owner data with the new path */
            STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);
         }

#ifdef COMMON_INTERFACE
         slot_acquired = ACI_AcquireCISlotForRecording(path, s_ptr);
#endif
         if (!slot_acquired)
         {
            if ((pmt_data = ADB_GetServicePMTData(s_ptr, &data_len)) != NULL)
            {
               /* Get the CA system IDs from the PMT */
               if ((num_ca_ids = STB_SIGetPmtCaIdDescArray(pmt_data, &ca_ids)) != 0)
               {
                  /* Check whether the CA system needs to be involved in the recording */
                  if (STB_CADescramblerRequiredForRecording(ca_ids, num_ca_ids))
                  {
                     ACA_AcquireCADescrambler(path, s_ptr);
                  }

                  STB_SIReleaseCaIdDescArray(ca_ids, num_ca_ids);
               }
            }
         }

         if (STB_DPIsLivePath(path) && (s_ptr != current_service[path]))
         {
            /* The live service has been changed, inform the UI */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED, &s_ptr, sizeof(void *));
         }
      }
      else
      {
         /* No path available for recording. The live path would have been taken if it was
          * available, but it hasn't, so it is probably owned by another module */
         if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
         {
#ifdef COMMON_INTERFACE
            if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
            {
               /* Save the service to be tuned to if the live path is eventually released */
               tuneto_live_service = s_ptr;
               decode_lock_override_required = FALSE;
               tuneto_live_transport = NULL;
               tuneto_for_recording = TRUE;
               tuneto_owner_info = CopyOwnerInfo(owner_info);

               /* Request the path/tuner to be released */
               if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
               {
                  ACI_AskRelease(*(U32BIT *)owner_data);
               }
               else
               {
                  STB_SPDebugWrite("[%s:%d]: Need to ask for path %u to be released by CI+, but there's no owner data!",
                     __FUNCTION__, __LINE__, path);
               }

               /* Clear the path so tuning doesn't start now */
               path = INVALID_RES_ID;
            }
            else
#endif
            {
               path = INVALID_RES_ID;
               STB_SPDebugWrite("[%s:%d]: Live path needs to be released but has an owner which isn't CI+\n",
                  __FUNCTION__, __LINE__);
            }
         }
         else
         {
            STB_SPDebugWrite("[%s:%d]: No live path but can't acquire a recording path!\n", __FUNCTION__, __LINE__);
         }
      }
   }

   FUNCTION_FINISH(AcquireRecordingPath);

   return(path);
}

#ifdef COMMON_INTERFACE
static S_ACTL_OWNER_INFO* CopyOwnerInfo(S_ACTL_OWNER_INFO *owner_info)
{
   S_ACTL_OWNER_INFO *copy;

   FUNCTION_START(CopyOwnerInfo);

   if (owner_info != NULL)
   {
      if ((copy = STB_AppGetMemory(sizeof(S_ACTL_OWNER_INFO))) != NULL)
      {
         copy->owner = owner_info->owner;
         copy->data = NULL;
         copy->data_size = 0;

         if ((owner_info->data != NULL) && (owner_info->data_size > 0))
         {
            if ((copy->data = STB_AppGetMemory(owner_info->data_size)) != NULL)
            {
               memcpy(copy->data, owner_info->data, owner_info->data_size);
               copy->data_size = owner_info->data_size;
            }
         }
      }
   }
   else
   {
      copy = NULL;
   }

   FUNCTION_FINISH(CopyOwnerInfo);

   return(copy);
}

#endif

/**
 *

 *
 * @brief   Called from ControlTuning to start tuning
 *
 * @param   path - decode path
 * @param   event - the tune request to be actioned i.e. INT_EVENT_RF_CHAN_TUNE_REQD,
 *                        INT_EVENT_ANALOG_RF_TUNE_REQD, or INT_EVENT_TPTR_TUNE_REQD
 *

 *
 */
static void SetupForTuning(U8BIT path, U32BIT event)
{
   U32BIT country_code;
   ACFG_ANA_RF_CHANNEL_DATA *ana_channel_table;
   ACFG_TER_RF_CHANNEL_DATA *ter_channel_table;
   ACFG_CAB_RF_CHANNEL_DATA *cab_channel_table;
   U16BIT num_rf_chans;
   S_MANUAL_TUNING_PARAMS tuning_params;
   U8BIT num_freqs;
   U32BIT *additional_freqs;
   BOOLEAN success;

   FUNCTION_START(SetupForTuning);

   if (event == INT_EVENT_RF_CHAN_TUNE_REQD)
   {
      country_code = ACFG_GetCountry();

      switch (signal_type_required)
      {
         case SIGNAL_ANALOG:
            success = ACFG_GetAnaRfChannelTable(country_code, &ana_channel_table, &num_rf_chans);
            break;
         case SIGNAL_COFDM:
            success = ACFG_GetTerRfChannelTable(country_code, &ter_channel_table, &num_rf_chans);
            break;
         case SIGNAL_QAM:
            success = ACFG_GetCabRfChannelTable(country_code, &cab_channel_table, &num_rf_chans);
            break;
         default:
            num_rf_chans = 0;
            success = FALSE;
            break;
      }

      if (success && (rf_chan_id_required < num_rf_chans))
      {
         switch (signal_type_required)
         {
            case SIGNAL_ANALOG:
               tuning_params.freq = ana_channel_table[rf_chan_id_required].freq_hz;
               tuning_params.u.ana.offset = 0;
               tuning_params.u.ana.vtype = ANLG_VIDEO_PAL_I;
               break;
            case SIGNAL_COFDM:
               tuning_params.freq = ter_channel_table[rf_chan_id_required].freq_hz;
               tuning_params.u.terr.type = ter_channel_table[rf_chan_id_required].type;
               tuning_params.u.terr.mode = ter_channel_table[rf_chan_id_required].mode;
               tuning_params.u.terr.bwidth = ter_channel_table[rf_chan_id_required].bwidth;

               /* When searching, the PLP will already have been set, so get the set value */
               tuning_params.u.terr.plp_id = STB_DPGetTerrPLP(path);
               break;
            case SIGNAL_QAM:
               tuning_params.freq = cab_channel_table[rf_chan_id_required].freq_hz;
               tuning_params.u.cab.mode = cab_channel_table[rf_chan_id_required].mode;
               tuning_params.u.cab.symbol_rate = cab_channel_table[rf_chan_id_required].symbol_rate;
               break;
            default:
               break;
         }
      }
      else
      {
         // can't access rf chan table - set frequency to 0 to force a not locked event when
         // STB_DPStartTune() is called
         tuning_params.freq = 0;
      }
   }
   else if (event == INT_EVENT_USERDEFINED_TUNE_REQD)
   {
      /* Copy the saved values */
      memcpy(&tuning_params, &user_tuning_params[path], sizeof(S_MANUAL_TUNING_PARAMS));
   }
   else
   {
      switch (signal_type_required)
      {
         case SIGNAL_ANALOG:
            ADB_GetTransportAnalogTuningParams(transport_required[path], &tuning_params.freq,
               &tuning_params.u.ana.offset, &tuning_params.u.ana.vtype);
            break;
         case SIGNAL_COFDM:
            ADB_GetTransportTerrestrialTuningParams(transport_required[path], &tuning_params.u.terr.type,
               &tuning_params.freq, &tuning_params.u.terr.mode, &tuning_params.u.terr.bwidth,
               &tuning_params.u.terr.plp_id);
            break;
         case SIGNAL_QAM:
            ADB_GetTransportCableTuningParams(transport_required[path], &tuning_params.freq,
               &tuning_params.u.cab.mode, &tuning_params.u.cab.symbol_rate);
               if (MODE_QAM_AUTO == tuning_params.u.cab.mode)
                   tuning_params.u.cab.mode = MODE_QAM_256;
            break;
         case SIGNAL_QPSK:
            tuning_params.u.sat.satellite = ADB_GetTransportSatellite(transport_required[path]);
            ADB_GetTransportSatTuningParams(transport_required[path], &tuning_params.freq,
               &tuning_params.u.sat.polarity, &tuning_params.u.sat.symbol_rate,
               &tuning_params.u.sat.fec, &tuning_params.u.sat.dvb_s2, &tuning_params.u.sat.modulation);
            break;
         default:
            tuning_params.freq = 0;
            break;
      }

      /* Set any additional frequencies that could be used for this transport so these can be tried
       * if the tuner fails to lock using the main frequency */
      DBDEF_GetTransportAdditionalFrequencies(transport_required[path], &num_freqs, &additional_freqs);
      STB_DPSetAdditionalFrequencies(path, num_freqs, additional_freqs);
   }

   StartTune(path, &tuning_params);

   FUNCTION_FINISH(SetupForTuning);
}

/*!**************************************************************************
 * @brief   Internal function to setup the tuning parameters to the decode path
 *          and start tuning based on the signal_type_required variable.
 * @param   path - decode path to be tuned
 * @param   tuning_params - structure defining the tuning params
 ****************************************************************************/
static void StartTune(U8BIT path, S_MANUAL_TUNING_PARAMS *tuning_params)
{
   void *lnb_ptr, *band;
   ADB_LNB_SETTINGS lnb_settings;
   static S_STB_DP_LNB_BAND *sat_lnb_bands = NULL;
   U8BIT n_bands, i;

   FUNCTION_START(StartTune);

   if (sat_lnb_bands != NULL)
   {
      STB_AppFreeMemory(sat_lnb_bands);
      sat_lnb_bands = NULL;
   }

   // setup tuning parameters...
   STB_DPSetSignalType(path, signal_type_required);
   STB_DPSetFrequency(path, tuning_params->freq);

   switch (signal_type_required)
   {
      case SIGNAL_ANALOG:
         STB_DPSetAnalogFreqOff(path, tuning_params->u.ana.offset);
         STB_DPSetAnalogVideoType(path, tuning_params->u.ana.vtype);
         current_satellite[path] = NULL;
         break;
      case SIGNAL_COFDM:
         STB_DPSetTerrType(path, tuning_params->u.terr.type);
         STB_DPSetTerrPLP(path, tuning_params->u.terr.plp_id);
         STB_DPSetTerrMode(path, tuning_params->u.terr.mode);
         STB_DPSetTerrBandwidth(path, tuning_params->u.terr.bwidth);
         current_satellite[path] = NULL;
         break;
      case SIGNAL_QAM:
         STB_DPSetCableMode(path, tuning_params->u.cab.mode);
         STB_DPSetSymbolRate(path, tuning_params->u.cab.symbol_rate);
         current_satellite[path] = NULL;
         break;
      case SIGNAL_QPSK:
         STB_DPSetPolarity(path, tuning_params->u.sat.polarity);
         STB_DPSetSymbolRate(path, tuning_params->u.sat.symbol_rate);
         STB_DPSetFEC(path, tuning_params->u.sat.fec);
         STB_DPSetDVBS2(path, tuning_params->u.sat.dvb_s2);
         STB_DPSetModulation(path, tuning_params->u.sat.modulation);

         /* Set the current satellite */
         current_satellite[path] = tuning_params->u.sat.satellite;

         /* Get the LNB associated with the given satellite */
         if ((lnb_ptr = ADB_GetSatelliteLNB(tuning_params->u.sat.satellite)) != NULL)
         {
            if (ADB_GetLNBSettings(lnb_ptr, &lnb_settings))
            {
               STB_DPSetLNBType(path, lnb_settings.type);
               STB_DPSetLNBPower(path, lnb_settings.power);
               STB_DPSetLNB12v(path, lnb_settings.is_12v);
               STB_DPSetLNB22k(path, lnb_settings.is_22k);

               // DiSEqC Support
               STB_DPSetDISEQCPosition(path, lnb_settings.is_diseqc_posn);
               STB_DPSetDISEQCCSwitch(path, lnb_settings.c_switch);
               STB_DPSetDISEQCUSwitch(path, lnb_settings.u_switch);
               STB_DPSetDISEQCTone(path, lnb_settings.diseqc_tone);
               STB_DPSetPulsePosition(path, lnb_settings.is_pulse_posn);
               STB_DPSetDISEQCSMATV(path, lnb_settings.is_smatv);
               STB_DPSetDISEQCRepeats(path, lnb_settings.diseqc_repeats);
               STB_DPSetUnicableFrequency(path, lnb_settings.unicable_if);
               STB_DPSetUnicableChannel(path, lnb_settings.unicable_chan);

               /* Set the bands only if it's a user defined LNB */
               if (lnb_settings.type == LNB_TYPE_USER_DEFINED)
               {
                  band = ADB_GetNextLNBBand(NULL);
                  n_bands = 0;
                  while (band != NULL)
                  {
                     if (ADB_GetLNBBandLNB(band) == lnb_ptr)
                     {
                        n_bands++;
                     }
                     band = ADB_GetNextLNBBand(band);
                  }

                  if (n_bands > 0)
                  {
                     sat_lnb_bands = STB_AppGetMemory(n_bands * sizeof(S_STB_DP_LNB_BAND));
                     if (sat_lnb_bands != NULL)
                     {
                        band = ADB_GetNextLNBBand(NULL);
                        i = 0;
                        while (band != NULL)
                        {
                           if (ADB_GetLNBBandLNB(band) == lnb_ptr)
                           {
                              ADB_GetLNBBandParameters(band, &(sat_lnb_bands[i++]));
                           }
                           if (i >= n_bands)
                           {
                              break;
                           }
                           band = ADB_GetNextLNBBand(band);
                        }

                        STB_DPSetUserDefinedLNBBands(path, n_bands, sat_lnb_bands);
                     }
                  }
#ifdef DEBUG_TUNING
                  else
                  {
                     STB_SPDebugWrite("StartTune: no bands defined for this user defined LNB");
                  }
#endif
               }
            }
         }
         break;
      default:
         break;
   }

   STB_DPSetTuneRelock(path, tuner_relock_required);
   STB_DPStartTune(path);

   current_transport[path] = transport_required[path];
   current_tuner_relock = tuner_relock_required;
   current_si[path] = APP_SI_MODE_NO_SI;

   FUNCTION_FINISH(StartTune);
}

/**
 *

 *
 * @brief   Called from ControlDecoding to set system ready to start decoding
 *
 * @param   path - decode path
 *

 *
 */
static void SetupForDecoding(U8BIT path)
{
   E_STB_AV_ASPECT_MODE aspect_mode;
   void **stream_list;
   U16BIT num_streams;
   U16BIT video_pid;
   U16BIT subt_pid;
   ADB_STREAM_TYPE stream_type;
   U16BIT i;
   U8BIT audio_decoder;
   E_STB_DIGITAL_AUDIO_TYPE audio_type;
   E_STB_DP_AUDIO_CODEC audio_codec;
   ADB_SUBT_INFO subt_info;

   FUNCTION_START(SetupForDecoding);

   // setup pids
   video_pid = ADB_GetServiceVideoPid(service_required[path]);

   if (!external_control)
   {
      STB_DPSetPCRPID(path, ADB_GetServicePCRPid(service_required[path]));
      STB_DPSetVideoPID(path, video_pid);
      ADB_SERVICE_REC * s_ptr = service_required[path];
      if (NULL != s_ptr) {
         if (TRUE == s_ptr->pmt_received) {
            STB_DPSetEcmPIDs(
               path,
               (ADB_SERVICE_REC *)s_ptr->ecm_pid,
               (ADB_SERVICE_REC *)s_ptr->video_ecm_pid,
               (ADB_SERVICE_REC *)s_ptr->audio_ecm_pid,
               (ADB_SERVICE_REC *)s_ptr->ttext_ecm_pid,
               0,
               (ADB_SERVICE_REC *)s_ptr->ad_ecm_pid
            );

            STB_DPSetDecodePIDs(
               path,
               (ADB_SERVICE_REC *)s_ptr->pcr_pid,
               (ADB_SERVICE_REC *)s_ptr->video_pid,
               (ADB_SERVICE_REC *)s_ptr->audio_pid,
               (ADB_SERVICE_REC *)s_ptr->ttext_pid,
               0,
               (ADB_SERVICE_REC *)s_ptr->ad_pid
            );
         }
      }

      /* Setup AV decoding according to the type of stream */
      ADB_GetStreamList(service_required[path], ADB_VIDEO_LIST_STREAM, &stream_list, &num_streams);
      if ((num_streams > 0) && (stream_list != NULL))
      {
         /* Find the video stream being used */
         for (i = 0; i < num_streams; i++)
         {
            if (ADB_GetStreamPID(stream_list[i]) == video_pid)
            {
               stream_type = ADB_GetStreamType(stream_list[i]);
               STB_DPSetVideoCodec(path, ADB_GetVideoCodecFromStream(stream_type));
            }
         }

         ADB_ReleaseStreamList(stream_list, num_streams);
      }

      ACTL_SetupAudioDecoding(path, service_required[path]);
   }

   if (STB_DPIsDecodingPath(path))
   {
      /* Ensure the audio output is setup correctly */
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      audio_type = (E_STB_DIGITAL_AUDIO_TYPE)APP_NvmRead(SPDIF_OUTPUT_NVM);
      audio_codec = STB_DPGetAudioCodec(path);
      if ((audio_type == DIGITAL_AUDIO_COMPRESSED) && ((audio_codec == AUDIO_CODEC_MP2) ||
         (audio_codec == AUDIO_CODEC_MP3)))
      {
         audio_type = DIGITAL_AUDIO_PCM;
      }
      STB_AVSetSpdifMode(audio_decoder, audio_type);
      STB_AVSetAudioDelay(audio_decoder, (U16BIT)APP_NvmRead(LIP_SYNC_ADJUSTMENT_NVM));
      audio_type = (E_STB_DIGITAL_AUDIO_TYPE)APP_NvmRead(HDMI_AUDIO_OUTPUT_NVM);
      if ((audio_type == DIGITAL_AUDIO_COMPRESSED) && ((audio_codec == AUDIO_CODEC_MP2) ||
         (audio_codec == AUDIO_CODEC_MP3)))
      {
         audio_type = DIGITAL_AUDIO_PCM;
      }
      STB_AVSetHDMIAudioMode(audio_decoder, audio_type);

      subt_pid = GetSubtitlePidFunc(current_service[path], &subt_info);
      if (subt_pid != 0)
      {
         STB_ChangePesCollectionPID(path, subt_pid);
      }

      // reset aspect conversion to default
      aspect_mode = (E_STB_AV_ASPECT_MODE)APP_NvmRead(ASPECT_MODE_NVM);

      STB_VTSetVideoAlignmentPref(aspect_mode);

      AVSetAudioVolumes(FALSE);

      ACTL_ApplyParentalControl(path, service_required[path]);
      ACTL_ApplyHDCP(service_required[path]);
   }

   FUNCTION_FINISH(SetupForDecoding);
}

/**
 *

 *
 * @brief   Called from ControlTuning to start si gathering
 *
 * @param   path - decode path
 *

 *
 */
static void StartSiProcess(U8BIT path)
{
   FUNCTION_START(StartSiProcess);

   if (si_required[path] == APP_SI_MODE_NO_SI)
   {
      STB_DPStopSI(path);
   }
   else
   {
      ASI_SetAppSiMode(path, si_required[path]);

      if (si_required[path] == APP_SI_MODE_UPDATE)
      {
         STB_DPSetSearchMode(path, FALSE);
      }
      else if (si_required[path] == APP_SI_MODE_USER_DEFINED)
      {
         STB_DPSetSearchMode(path, FALSE);
      }
      else
      {
         STB_DPSetSearchMode(path, TRUE);
      }
      STB_DPStartSI(path);
   }
   current_si[path] = si_required[path];

   FUNCTION_FINISH(StartSiProcess);
}

/**
 *

 *
 * @brief   Implementation of Control Tuning state transition diagram. (See LCDTV spec)
 *                Actions events associated with tuning.
 *
 * @param   path - decode path
 * @param   type  - type of event i.e. INTERNAL_EVENT or EXTERNAL_EVENT
 * @param   event - the event to be actioned
 *

 *
 */
static void ControlTuning(U8BIT path, E_ACTL_EVENT_TYPE type, U32BIT event)
{
   U32BIT handle;
   void *s_ptr;
   E_STB_DP_TUNE_STATUS tune_status;
#ifdef COMMON_INTERFACE
   void *owner_data;
   U32BIT data_size;
   U8BIT slot_id;
#endif
#if defined(INTEGRATE_HBBTV)
   U16BIT onet_id, ts_id, serv_id;
#endif

   FUNCTION_START(ControlTuning);

   STB_OSMutexLock(cntrl_mutex[path]);

   if ((type == INTERNAL_EVENT) && (event == INT_EVENT_INITIALISE))
   {
      // initialisation...
      memset(&user_tuning_params[path], 0, sizeof(S_MANUAL_TUNING_PARAMS));
      tuning_state[path] = TSTATE_OFF;
      tuned_status[path] = TUNED_STATUS_NO_SIGNAL;

      transport_required[path] = NULL;
      service_required[path] = NULL;
      si_required[path] = APP_SI_MODE_NO_SI;

      current_satellite[path] = NULL;
      current_service[path] = NULL;
      current_transport[path] = NULL;
      current_si[path] = APP_SI_MODE_NO_SI;

      current_tuner_relock = FALSE;
      #ifdef DEBUG_TUNING
      STB_SPDebugWrite("   new tune state(%d): Off", path);
      #endif
   }
   else
   {
      #ifdef DEBUG_TUNING
      {
         U8BIT *state_string;
         U8BIT *event_string;

         switch (tuning_state[path])
         {
            case TSTATE_OFF:        {state_string = (U8BIT *)"Off"; break; }
            case TSTATE_TUNING:     {state_string = (U8BIT *)"Tuning"; break; }
            case TSTATE_NOT_TUNED:  {state_string = (U8BIT *)"Not tuned"; break; }
            case TSTATE_TUNED:      {state_string = (U8BIT *)"Tuned"; break; }
            case TSTATE_ANALOG:     {state_string = (U8BIT *)"Analogue"; break; }
            default:                {state_string = (U8BIT *)"Unknown state - error"; break; }
         }
         event_string = GetEventDebugString(type, event);
         STB_SPDebugWrite("ControlTuning(%d): %s, %s", path, state_string, event_string);
      }
      #endif

      switch (tuning_state[path])
      {
         case TSTATE_OFF:
         {
            if ((type == INTERNAL_EVENT) &&
                ((event == INT_EVENT_RF_CHAN_TUNE_REQD) ||
                 (event == INT_EVENT_TPTR_TUNE_REQD) ||
                 (event == INT_EVENT_USERDEFINED_TUNE_REQD)))
            {
               if (signal_type_required == SIGNAL_ANALOG)
               {
                  #ifdef DEBUG_TUNING
                  STB_SPDebugWrite("   new tune state(%d): Analogue", path);
                  #endif
                  tuning_state[path] = TSTATE_ANALOG;
                  SetupForTuning(path, event);

                  if (service_required[path] != NULL)
                  {
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_STARTED, NULL, 0);
                     ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_ON);
                  }
               }
               else
               {
                  #ifdef DEBUG_TUNING
                  STB_SPDebugWrite("   new tune state(%d): Tuning", path);
                  #endif
                  tuning_state[path] = TSTATE_TUNING;
                  SetupForTuning(path, event);
               }
            }
            break;
         }

         case TSTATE_TUNING:
         {
            if (type == INTERNAL_EVENT)
            {
               if (event == INT_EVENT_TUNE_OFF)
               {
                  #ifdef DEBUG_TUNING
                  STB_SPDebugWrite("   new tune state(%d): Off", path);
                  #endif
                  // stop tuning
                  STB_DPTuneOff(path);
                  tuning_state[path] = TSTATE_OFF;
               }
               else if ((event == INT_EVENT_RF_CHAN_TUNE_REQD) ||
                        (event == INT_EVENT_TPTR_TUNE_REQD) || (event == INT_EVENT_USERDEFINED_TUNE_REQD))
               {
                  if (signal_type_required == SIGNAL_ANALOG)
                  {
                     #ifdef DEBUG_TUNING
                     STB_SPDebugWrite("   new tune state(%d): Analogue", path);
                     #endif
                     tuning_state[path] = TSTATE_ANALOG;
                     if (service_required[path] != NULL)
                     {
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_STARTED, NULL, 0);
                        ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_ON);
                     }
                  }
                  SetupForTuning(path, event);
#if defined(INTEGRATE_HBBTV)
                  if (STB_DPIsLivePath(path))
                  {
                     /* Inform the presentation engine that the channel change was interrupted */
                     if (service_required[path] != NULL)
                     {
                        ADB_GetServiceIds(service_required[path], &onet_id, &ts_id, &serv_id);
                     }
                     else
                     {
                        onet_id = ADB_INVALID_DVB_ID;
                        ts_id = ADB_INVALID_DVB_ID;
                        serv_id = ADB_INVALID_DVB_ID;
                     }
                     HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                        HBBTV_CHANNEL_CHANGE_INTERRUPTED);
                  }
#endif
               }
            }
            else   // i.e. external event
            {
               switch (event)
               {
                  case STB_EVENT_TUNE_NOTLOCKED:
                  {
                     /* This event may be related to a previous tune request, so check the midware
                      * tuner state and if it's still waiting to tune or has been stopped then
                      * this event is for a previous tune request and should be ignored */
                     tune_status = STB_DPGetTuneStatus(path);
                     if (tune_status == TUNE_NO_LOCK)
                     {
                        // tried to tune but no signal - need to enable decode control so that it can
                        // control status indication
                        #ifdef DEBUG_TUNING
                        STB_SPDebugWrite("   new tune state(%d): Not tuned", path);
                        #endif
                        /* Decoding should only be performed on the live path */
                        if (STB_DPIsDecodingPath(path))
                        {
                           ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_ENABLE);
                        }

                        tuning_state[path] = TSTATE_NOT_TUNED;

#ifdef COMMON_INTERFACE
                        /* If the tuner is being used by CI+ then it needs to be notified
                         * of the tuning status */
                        if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
                        {
                           if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
                           {
                              ACI_TuneReply(path, *(U32BIT *)owner_data, CIP_TUNER_NOTLOCKED);
                           }
                        }
#endif
                        if (STB_DPIsRecordingPath(path))
                        {
                           if (STB_DPIsRecording(path, &handle))
                           {
#ifdef COMMON_INTERFACE
                              if ((slot_id = STB_DPGetPathCISlot(path)) != INVALID_RES_ID)
                              {
                                 STB_CiCcSendRecordStop(slot_id);
                              }
#endif
                              /* Stop the recording and release the path */
                              APVR_StopRecording(handle);
                           }
                           else
                           {
                              /* Recording not yet started */
                              ATMR_RecordingFailed(path);
                              STB_DPReleasePath(path, RES_OWNER_DVB);

                              /* Send event to UI to inform a recording has failed */
                              STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                                 EV_PVR_RECORDING_FAILED, NULL, 0);
                           }
                        }
#if defined(INTEGRATE_HBBTV)
                        if (STB_DPIsLivePath(path))
                        {
                           /* Inform the presentation engine that the channel change failed */
                           if (service_required[path] != NULL)
                           {
                              ADB_GetServiceIds(service_required[path], &onet_id, &ts_id, &serv_id);
                           }
                           else
                           {
                              onet_id = ADB_INVALID_DVB_ID;
                              ts_id = ADB_INVALID_DVB_ID;
                              serv_id = ADB_INVALID_DVB_ID;
                           }
                           HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                              HBBTV_CHANNEL_CHANGE_NO_SIGNAL);
                        }
#endif
                     }
                     break;
                  }
                  case STB_EVENT_TUNE_LOCKED:
                  {
                     /* This event may be related to a previous tune request, so check the midware
                      * tuner state and if it's still waiting to tune or has been stopped then
                      * this event is for a previous tune request and should be ignored */
                     tune_status = STB_DPGetTuneStatus(path);
                     if (tune_status == TUNE_LOCKED)
                     {
                        // tuned - enable decode control and start si
                        #ifdef DEBUG_TUNING
                        STB_SPDebugWrite("   new tune state(%d): Tuned", path);
                        #endif
                        tuning_state[path] = TSTATE_TUNED;
                        tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL;

                        UpdateTransportParameters(path);

                        /* Start SI processing */
                        StartSiProcess(path);

                        if (STB_DPIsLivePath(path) && (current_service[path] != NULL) &&
                            current_service[path]->pmt_received)
                        {
                           /* The PMT has already been received for the service being tuned to
                            * so the UI should be informed of the change of service now */
                           STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED,
                              &current_service[path], sizeof(void *));
#if defined(INTEGRATE_HBBTV)
                           /* Inform the presentation engine that the channel change succeeded */
                           ADB_GetServiceIds(current_service[path], &onet_id, &ts_id, &serv_id);
                           HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                              HBBTV_CHANNEL_CHANGE_SUCCEEDED);
#endif
                        }

#ifdef COMMON_INTERFACE
                        /* If the path is owned by CI+, and we're tuning to a service or transport and
                         * no SI collection is to be started or the PMT for the service has already
                         * been received, then CI+ needs to be notified of the result of the tune.
                         * If tuning to a service and SI collection will be performed and the PMT
                         * hasn't yet been received, then CI+ will be notified when the event is
                         * received indicating completion */
                        if (STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS) &&
                            ((si_required[path] == APP_SI_MODE_NO_SI) ||
                             ((current_service[path] != NULL) && current_service[path]->pmt_received)))
                        {
                           if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
                           {
                              ACI_TuneReply(path, *(U32BIT *)owner_data, CIP_TUNER_LOCKED);
                           }
                        }
#endif

                        if (STB_DPIsDecodingPath(path))
                        {
                           if ((current_service[path] != NULL) && (TRUE == current_service[path]->pmt_received)) {
                              /* Start decoding. If this is also a recording path then
                               * recording will be started after the decoding has started */
                              ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_ENABLE);
                           }
                        }
                        else if (STB_DPIsMonitoringPath(path)) {
                           if (FALSE == STB_DPGetSearchMode(path)) {
                              ADB_SERVICE_REC * service_ptr = current_service[path];
                              if ((NULL != service_ptr) && (TRUE == service_ptr->pmt_received)) {
                                 STB_DPSetEcmPIDs(path, service_ptr->ecm_pid, service_ptr->video_ecm_pid, service_ptr->audio_ecm_pid,
                                    service_ptr->ttext_ecm_pid, 0, service_ptr->ad_ecm_pid);

                                 STB_DPSetDecodePIDs(path, service_ptr->pcr_pid, service_ptr->video_pid, service_ptr->audio_pid,
                                    service_ptr->ttext_pid, 0, service_ptr->ad_pid);
                              }
                           }
                        }

                        if (STB_DPIsRecordingPath(path) && !STB_DPIsRecording(path, &handle))
                        {
                           /* The recording can only be started here if the PIDs for the service
                            * are known, otherwise it will have to start when the PMT for the service
                            * has been received. */
                           s_ptr = ATMR_GetRecordService(path);
                           if ((ADB_GetServiceAudioPid(s_ptr) != 0) ||
                              (ADB_GetServiceVideoPid(s_ptr) != 0))
                           {
                              ATMR_CheckRecordStatus(TRUE, s_ptr);
                           }
                        }
                     }
                     break;
                  }
               }
            }
            break;
         }

         case TSTATE_NOT_TUNED:
         {
            if (type == INTERNAL_EVENT)
            {
               switch (event)
               {
                  case INT_EVENT_TUNE_OFF:
                  {
                     // stop tuning
                     #ifdef DEBUG_TUNING
                     STB_SPDebugWrite("   new tune state(%d): Off", path);
                     #endif
                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
                     }
                     STB_DPTuneOff(path);
                     tuning_state[path] = TSTATE_OFF;
                     break;
                  }
                  case INT_EVENT_RF_CHAN_TUNE_REQD:
                  case INT_EVENT_TPTR_TUNE_REQD:
                  case INT_EVENT_USERDEFINED_TUNE_REQD:
                  {
                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
                     }
                     if (signal_type_required == SIGNAL_ANALOG)
                     {
                        #ifdef DEBUG_TUNING
                        STB_SPDebugWrite("   new tune state(%d): Analogue", path);
                        #endif
                        tuning_state[path] = TSTATE_ANALOG;
                        if (service_required[path] != NULL)
                        {
                           STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_STARTED, NULL, 0);
                           ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_ON);
                        }
                     }
                     else
                     {
                        #ifdef DEBUG_TUNING
                        STB_SPDebugWrite("   new tune state(%d): Tuning", path);
                        #endif
                        tuning_state[path] = TSTATE_TUNING;
                     }
                     SetupForTuning(path, event);
                     break;
                  }
               }
            }
            else   // i.e. external event
            {
               if (event == STB_EVENT_TUNE_LOCKED)
               {
                  /* If the tuner is no longer locked then this event is no longer valid */
                  tune_status = STB_DPGetTuneStatus(path);
                  if (tune_status == TUNE_LOCKED)
                  {
                     // tuned - enable decode control and start si
                     #ifdef DEBUG_TUNING
                     STB_SPDebugWrite("   new tune state(%d): Tuned", path);
                     #endif
                     tuning_state[path] = TSTATE_TUNED;
                     tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL;

                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_DECODING);
                     }

                     /* Start SI */
                     StartSiProcess(path);

                     if (STB_DPIsLivePath(path) && (current_service[path] != NULL) &&
                         current_service[path]->pmt_received)
                     {
                        /* The PMT has already been received for the service being tuned to
                         * so the UI should be informed of the change of service now */
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED,
                           &current_service[path], sizeof(void *));
#if defined(INTEGRATE_HBBTV)
                        /* Inform the presentation engine that the channel change succeeded */
                        ADB_GetServiceIds(current_service[path], &onet_id, &ts_id, &serv_id);
                        HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                           HBBTV_CHANNEL_CHANGE_SUCCEEDED);
#endif
                     }
                  }
               }
            }
            break;
         }

         case TSTATE_TUNED:
         {
            if (type == INTERNAL_EVENT)
            {
               if ((event == INT_EVENT_TUNE_OFF) || (event == INT_EVENT_RF_CHAN_TUNE_REQD) ||
                   (event == INT_EVENT_TPTR_TUNE_REQD) || (event == INT_EVENT_USERDEFINED_TUNE_REQD))
               {
                  if (STB_DPIsLivePath(path))
                  {
                     // check if pmt has been reported (this must be read before StopSI is done
                     // because that clears the flag
                     ASI_PmtReported(path);

                     // stop SI ready for re-tune
                     STB_DPStopSI(path);
                  }
                  if (STB_DPIsMonitoringPath(path))
                  {
                     // check if pmt has been reported (this must be read before StopSI is done
                     // because that clears the flag
                     ASI_PmtReported(path);

                     // stop SI ready for re-tune
                     STB_DPStopSI(path);
                  }
                  switch (event)
                  {
                     case INT_EVENT_TUNE_OFF:
                     {
                        // stop tuning
                        #ifdef DEBUG_TUNING
                        STB_SPDebugWrite("   new tune state(%d): Off", path);
                        #endif
                        tuning_state[path] = TSTATE_OFF;
                        tuned_status[path] = TUNED_STATUS_NO_SIGNAL;
                        if (STB_DPIsDecodingPath(path))
                        {
                           ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
                        }
                        STB_DPTuneOff(path);
                        break;
                     }
                     case INT_EVENT_RF_CHAN_TUNE_REQD:
                     case INT_EVENT_USERDEFINED_TUNE_REQD:
                     {
                        tuned_status[path] = TUNED_STATUS_NO_SIGNAL;
                        if (STB_DPIsDecodingPath(path))
                        {
                           ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
                        }
                        if (signal_type_required == SIGNAL_ANALOG)
                        {
                           #ifdef DEBUG_TUNING
                           STB_SPDebugWrite("   new tune state(%d): Analogue", path);
                           #endif
                           tuning_state[path] = TSTATE_ANALOG;
                        }
                        else
                        {
                           #ifdef DEBUG_TUNING
                           STB_SPDebugWrite("   new tune state(%d): Tuning", path);
                           #endif
                           tuning_state[path] = TSTATE_TUNING;
                        }
                        SetupForTuning(path, event);
                        break;
                     }
                     case INT_EVENT_TPTR_TUNE_REQD:
                     {
                        if ((transport_required[path] != current_transport[path]) ||
                            (tuner_relock_required != current_tuner_relock) ||
                            (si_required[path] != current_si[path]))
                        {
                           // something has changed - re-tune
                           tuned_status[path] = TUNED_STATUS_NO_SIGNAL;
                           if (STB_DPIsDecodingPath(path))
                           {
                              ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_DISABLE);
                           }
                           if (signal_type_required == SIGNAL_ANALOG)
                           {
                              #ifdef DEBUG_TUNING
                              STB_SPDebugWrite("   new tune state(%d): Analogue", path);
                              #endif
                              tuning_state[path] = TSTATE_ANALOG;
                              if (service_required[path] != NULL)
                              {
                                 STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_STARTED, NULL, 0);
                                 ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_ON);
                              }
                           }
                           else
                           {
                              #ifdef DEBUG_TUNING
                              STB_SPDebugWrite("   new tune state(%d): Tuning", path);
                              #endif
                              tuning_state[path] = TSTATE_TUNING;
                           }
                           SetupForTuning(path, event);
                        }
                        else
                        {
                           // already tuned to the correct transport - just restart decoding to the
                           // new service and restart si.
                           if (STB_DPIsDecodingPath(path))
                           {
                              ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_RESTART_DECODING);
                           }

                           StartSiProcess(path);
                        }
                        break;
                     }
                  }
               }
            }
            else   // i.e. external event
            {
               switch (event)
               {
                  case STB_EVENT_TUNE_LOCKED:
                  case STB_EVENT_TUNE_SIGNAL_DATA_OK:
                  {
                     tuned_status[path] = TUNED_STATUS_GOOD_SIGNAL;
                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_UPDATE);
                     }
                     else if (STB_DPIsMonitoringPath(path)) {
                        if (FALSE == STB_DPGetSearchMode(path)) {
                           ADB_SERVICE_REC * service_ptr = current_service[path];
                           if ((NULL != service_ptr) && (TRUE == service_ptr->pmt_received)) {
                              STB_DPSetEcmPIDs(path, service_ptr->ecm_pid, service_ptr->video_ecm_pid, service_ptr->audio_ecm_pid,
                                 service_ptr->ttext_ecm_pid, 0, service_ptr->ad_ecm_pid);

                              STB_DPSetDecodePIDs(path, service_ptr->pcr_pid, service_ptr->video_pid, service_ptr->audio_pid,
                                 service_ptr->ttext_pid, 0, service_ptr->ad_pid);
                           }
                        }
                     }
                     break;
                  }
                  case STB_EVENT_TUNE_NOTLOCKED:
                  {
                     tuned_status[path] = TUNED_STATUS_NO_SIGNAL;
                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_UPDATE);
                     }
                     break;
                  }
                  case STB_EVENT_TUNE_SIGNAL_DATA_BAD:
                  {
                     tuned_status[path] = TUNED_STATUS_BAD_SIGNAL;
                     if (STB_DPIsDecodingPath(path))
                     {
                        ControlDecoding(path, INTERNAL_EVENT, INT_EVENT_UPDATE);
                     }
                     break;
                  }
               }
            }
            break;
         }

         case TSTATE_ANALOG:
         {
            // analogue signal...
            if (type == INTERNAL_EVENT)
            {
               if (event == INT_EVENT_TUNE_OFF)
               {
                  #ifdef DEBUG_TUNING
                  STB_SPDebugWrite("   new tune state(%d): Off", path);
                  #endif
                  // stop tuning
                  STB_DPTuneOff(path);
                  tuning_state[path] = TSTATE_OFF;
                  ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_OFF);
               }
               else if ((event == INT_EVENT_RF_CHAN_TUNE_REQD) ||
                        (event == INT_EVENT_TPTR_TUNE_REQD) || (event == INT_EVENT_USERDEFINED_TUNE_REQD))
               {
                  if (signal_type_required == SIGNAL_ANALOG)
                  {
                     if (service_required[path] != NULL)
                     {
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ANALOG_STARTED, NULL, 0);
                        ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_ON);
                     }
                  }
                  else
                  {
                     #ifdef DEBUG_TUNING
                     STB_SPDebugWrite("   new tune state(%d): Tuning", path);
                     #endif
                     tuning_state[path] = TSTATE_TUNING;
                     ControlAvOutput(INTERNAL_EVENT, INT_EVENT_ANALOG_TV_OFF);
                  }
                  SetupForTuning(path, event);
               }
            }
            break;
         }
      }
   }

   STB_OSMutexUnlock(cntrl_mutex[path]);

   FUNCTION_FINISH(ControlTuning);
}

/**
 *

 *
 * @brief   Implementation of Control Decoding state transition diagram. (See LCDTV spec)
 *                Actions events associated with a/v decoding and decoding status information.
 *
 * @param   path - decode path
 * @param   type  - type of event i.e. INTERNAL_EVENT or EXTERNAL_EVENT
 * @param   event - the event to be actioned
 *

 *
 */
static void ControlDecoding(U8BIT path, E_ACTL_EVENT_TYPE type, U32BIT event)
{
   E_STB_DP_DECODE_STATUS video_status;
   E_STB_DP_DECODE_STATUS audio_status;
#if defined(INTEGRATE_HBBTV)
   U16BIT onet_id, ts_id, serv_id;
#endif

   FUNCTION_START(ControlDecoding);

   ASSERT(path < STB_DPGetNumPaths());

   STB_OSMutexLock(cntrl_mutex[path]);

   /* Treat service not running and pid update events from application SI
    * as a request to restart decoding */
   if (type == EXTERNAL_EVENT)
   {
      if (event == APP_EVENT_SERVICE_NOT_RUNNING)
      {
         type = INTERNAL_EVENT;
         event = INT_EVENT_DISABLE;
      }
      else if ((event == APP_EVENT_SERVICE_VIDEO_PID_UPDATE) ||
               (event == APP_EVENT_SERVICE_RUNNING))
      {
         type = INTERNAL_EVENT;
         if (decoding_state[path] == DSTATE_DISABLED)
         {
            event = INT_EVENT_ENABLE;
         }
         else
         {
            event = INT_EVENT_RESTART_DECODING;
         }
      }
      else if (event == APP_EVENT_SERVICE_SUBTITLE_UPDATE)
      {
         type = INTERNAL_EVENT;
         event = INT_EVENT_RESTART_SUBTITLES;
      }
      else if (event == APP_EVENT_SERVICE_SCRAMBLE_CHANGE)
      {
         type = INTERNAL_EVENT;
         if (decoding_state[path] == DSTATE_DISABLED)
         {
            event = INT_EVENT_ENABLE;
         }
         else
         {
            event = INT_EVENT_SCRAMBLE_CHANGE;
         }
      }
   }
   else if (type == INTERNAL_EVENT)
   {
      /* Decoding can't be restarted if the decoder is disabled, so change the event */
      if ((event == INT_EVENT_RESTART_DECODING) && (decoding_state[path] == DSTATE_DISABLED))
      {
         event = INT_EVENT_ENABLE;
      }
   }

   if ((type == INTERNAL_EVENT) && (event == INT_EVENT_INITIALISE))
   {
      // initialisation...
      decoding_state[path] = DSTATE_DISABLED;
      decoding_started[path] = FALSE;
      decoding_locked[path] = FALSE;
      now_event_id[path] = -1;
      #ifdef DEBUG_DECODING
      STB_SPDebugWrite("   new decode state(%d): Disabled", path);
      #endif
   }
   else
   {
      #ifdef DEBUG_DECODING
      {
         U8BIT *state_string;
         U8BIT *event_string;

         switch (decoding_state[path])
         {
            case DSTATE_DISABLED:               {state_string = (U8BIT *)"Disabled"; break; }
            case DSTATE_OFF:                    {state_string = (U8BIT *)"Off"; break; }
            case DSTATE_STARTING_BAD_SIGNAL:    {state_string = (U8BIT *)"Starting bad"; break; }
            case DSTATE_STARTING:               {state_string = (U8BIT *)"Starting"; break; }
            case DSTATE_LOCKED:                 {state_string = (U8BIT *)"Locked"; break; }
            case DSTATE_DECODING:               {state_string = (U8BIT *)"Decoding"; break; }
            case DSTATE_PAUSED_SIGNAL:          {state_string = (U8BIT *)"Signal pause"; break; }
            case DSTATE_PAUSED_USER:            {state_string = (U8BIT *)"User pause"; break; }
            case DSTATE_PAUSED_SIGNAL_AND_USER: {state_string = (U8BIT *)"Signal/user pause"; break; }
            default:                            {state_string = (U8BIT *)"Unknown decode state - error"; break; }
         }
         event_string = GetEventDebugString(type, event);
         STB_SPDebugWrite("ControlDecoding(%d): %s, %s", path, state_string, event_string);
      }
      #endif

      // first, code common to all states...
      if ((type == INTERNAL_EVENT) &&
          ((event == INT_EVENT_DISABLE) ||
           (event == INT_EVENT_RESTART_DECODING) ||
           (event == INT_EVENT_SCRAMBLE_CHANGE)))
      {
         if (decoding_state[path] != DSTATE_DISABLED)
         {
            if (decoding_state[path] >= DSTATE_DECODING)   // i.e. decoding or any of the paused states
            {
               if (STB_DPIsLivePath(path))
               {
                  /* If MHEG-5 or other external is enabled, it will deal with PID updates */
                  if (!external_control)
                  {
                     /*STB_AVBlankVideo(0, TRUE);*/
                  }

                  if (event != INT_EVENT_RESTART_DECODING)
                  {
                     /* Ensure subtitles are stopped */
                     ACTL_StopSubtitles();
                  }
               }
            }

            if ((decoding_state[path] == DSTATE_STARTING) || (decoding_state[path] == DSTATE_DECODING))
            {
               /* Decoding needs to be stopped if the service being viewed has changed from
                * FTA to scrambled so that the video is no longer shown on screen, but MHEG
                * could be controlling the video and it may be viewing video from another
                * service that isn't scrambled. In this situation, MHEG needs to be told that
                * there's been an update to the current service so that it can decide to stop
                * decoding if it's using the service's video, but if it isn't then decoding
                * should be stopped here.
                * As it stands, this change will probably cause problem(s) with the MHEG test
                * suite, so the ability to notify MHEG about changes to the service needs to be
                * added soon.
                * There's an equivalent change to start decoding below which will also need to
                * be changed. */
               if (!external_control ||
                   ((event == INT_EVENT_SCRAMBLE_CHANGE) &&
                    ADB_GetServiceScrambledFlag(service_required[path])))
               {
                  STB_DPStopDecoding(path);
               }
            }

            if ((event == INT_EVENT_DISABLE) ||
                ((event == INT_EVENT_RESTART_DECODING) && (service_required[path] == NULL)) ||
                (((event == INT_EVENT_RESTART_DECODING) || (event == INT_EVENT_SCRAMBLE_CHANGE)) &&
                 (service_required[path] != NULL) &&
                 (ADB_GetServiceNotRunningFlag(service_required[path]) == TRUE)))
            {
               if (decoding_state[path] >= DSTATE_STARTING)   // i.e. starting, locked, decoding or any of the paused states
               {
                  // clear pids to ensure no decoding takes place by accident
                  if (!external_control)
                  {
                     STB_DPSetEcmPIDs(path, 0, 0, 0, 0, 0, 0);
                     STB_DPSetDecodePIDs(path, 0, 0, 0, 0, 0, 0);
                  }
               }
               if (event == INT_EVENT_DISABLE)
               {
                  #ifdef DEBUG_DECODING
                  STB_SPDebugWrite("   new decode state(%d): Disabled", path);
                  #endif
                  decoding_state[path] = DSTATE_DISABLED;
               }
               else
               {
                  #ifdef DEBUG_DECODING
                  STB_SPDebugWrite("   new decode state(%d): Off", path);
                  #endif
                  decoding_state[path] = DSTATE_OFF;
               }
            }
            else   // i.e. INT_EVENT_RESTART_DECODING and service_required != NULL
            {
               // setup system ready to start decoding...
               SetupForDecoding(path);

               if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
               {
                  if (STB_DPIsDecodingPath(path))
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Starting", path);
                     #endif
                     /* An MHEG app could be running when a service changes from scrambled to FTA
                      * in which case starting decoding here could cause problems, but see above
                      * when decoding is stopped for a fuller description */
                     if (!external_control || (event == INT_EVENT_SCRAMBLE_CHANGE))
                     {
                        /* If external control is enabled, external (eg MHEG) will deal with PID updates */
                        STB_DPStartDecoding(path);
                     }
                  }

                  decoding_state[path] = DSTATE_STARTING;
               }
               else
               {
                  #ifdef DEBUG_DECODING
                  STB_SPDebugWrite("   new decode state(%d): Starting bad", path);
                  #endif
                  decoding_state[path] = DSTATE_STARTING_BAD_SIGNAL;
               }
            }
         }
      }
      else if (((type == INTERNAL_EVENT) && (event == INT_EVENT_RESTART_AUDIO)) ||
               (event == APP_EVENT_SERVICE_AUDIO_PID_UPDATE))
      {
         // if audio pid has already been set in decode path need to update it. Then, if decoding
         // has already started need to restart the audio decoding

         /* An MHEG app could be running when a service changes from scrambled to FTA
          * in which case starting decoding here could cause problems, but see above
          * when decoding is stopped for a fuller description */
         if (!external_control)
         {
            if (decoding_state[path] > DSTATE_OFF)
            {
               E_ACTL_DECODE_CHANGE changed;

               changed = ACTL_SetupAudioDecoding(path, service_required[path]);

               /* No need to restart audio decoding if the PID hasn't changed */
               if ((event != APP_EVENT_SERVICE_AUDIO_PID_UPDATE) || changed)
               {
                  /* Audio PID has changed so need to restart audio decoding */
                  if ((decoding_state[path] == DSTATE_STARTING) ||
                      (decoding_state[path] == DSTATE_DECODING))
                  {
                     /* After stop func, start func will apply changes setup in ACTL_SetupAudioDecoding
                      * Note: stop and start of AD does nothing, unless it was enabled and has valid pid */
                     STB_DPStopADDecoding(path);
                     STB_DPStopAudioDecoding(path);
                     STB_DPStartAudioDecoding(path);
                     STB_DPStartADDecoding(path);
                  }
               }
               #ifdef DEBUG_DECODING
               else
               {
                  STB_SPDebugWrite("   no audio PID change, not restarting audio decoding(%u)", path);
               }
               #endif
            }
         }
      }
      else if ((type == INTERNAL_EVENT) && (event == INT_EVENT_RESTART_SUBTITLES))
      {
         // subbtitles are running in decoding and paused states - so if in one of those states
         // stop and restart subtitles
         if ((decoding_state[path] == DSTATE_STARTING) || (decoding_state[path] >= DSTATE_DECODING))
         {
            /* Restart the subtitles, it will select either DVB subtitle or teletext subtitle */
            if (ACTL_AreSubtitlesDisplayed())
            {
               /* Restart subtitles and display them, if available */
               ACTL_StopSubtitles();
               ACTL_StartSubtitles();
            }
            else if (ACTL_AreSubtitlesStarted())
            {
               /* Need to start and then pause subtitles to stop them being immediately shown */
               ACTL_StopSubtitles();
               ACTL_StartSubtitles();
               ACTL_PauseSubtitles();
            }
         } else {
	    STB_SPDebugWrite("sheng test subtitle pass decoding check, ControlDecoding()::decoding_state[path] path: (%u), state:%d", path, decoding_state[path]);
		/* Test Solve Johnson report problem, line 8/19  subtitle not work or enable not normal,
		Add special cast, 解決有時切換到不能播的頻道時，subtitle 開啟會停留在上次的選項，且切換語言無效 */
	    if (decoding_state[path]  == DSTATE_DISABLED) {
		STB_SPDebugWrite("sheng test subtitle pass decoding checkControlDecoding()::sheng type == DSTATE_DISABLED go workrun test");
		if (ACTL_AreSubtitlesDisplayed())
		{
		   STB_SPDebugWrite("sheng test subtitle pass decoding checkControlDecoding()::ACTL_AreSubtitlesDisplayed()");
                  ACTL_StopSubtitles();
                  ACTL_StartSubtitles();
		} else if (ACTL_AreSubtitlesStarted()) {
		  STB_SPDebugWrite("sheng test subtitle pass decoding checkControlDecoding()::ACTL_AreSubtitlesStarted()");
		  ACTL_StopSubtitles();
		  ACTL_StartSubtitles();
		  ACTL_PauseSubtitles();
		} else {
		   STB_SPDebugWrite("sheng test subtitle pass decoding checksubtitle ControlDecoding()::sheng no AreSubtitlesDisplayed and no ACTL_AreSubtitlesStarted()");
		}
	     }
	 }
      }
      else
      {
         // switch on state variable for code relevant to particular states
         switch (decoding_state[path])
         {
            case DSTATE_DISABLED:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_ENABLE))
               {
                  if (service_required[path] != NULL)
                  {
                     SetupForDecoding(path);
                     if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
                     {
                        if (STB_DPIsDecodingPath(path))
                        {
                           #ifdef DEBUG_DECODING
                           STB_SPDebugWrite("   new decode state(%d): Starting", path);
                           #endif
                           STB_DPStartDecoding(path);
                        }

                        decoding_state[path] = DSTATE_STARTING;
                     }
                     else
                     {
                        #ifdef DEBUG_DECODING
                        STB_SPDebugWrite("   new decode state(%d): Starting bad", path);
                        #endif
                        decoding_state[path] = DSTATE_STARTING_BAD_SIGNAL;
                     }
                  }
               }
               break;
            }

            case DSTATE_OFF:
            {
               // DISABLE and RESTART DECODING events handled in general code above - nothing else
               // to do in this state
               break;
            }

            case DSTATE_STARTING_BAD_SIGNAL:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
                  {
                     if (STB_DPIsLivePath(path))
                     {
                     #ifdef DEBUG_DECODING
                        STB_SPDebugWrite("   new decode state(%d): Starting", path);
                     #endif
                        STB_DPStartDecoding(path);
                     }

                     decoding_state[path] = DSTATE_STARTING;
                  }
               }
               break;
            }

            case DSTATE_STARTING:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] != TUNED_STATUS_GOOD_SIGNAL)
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Starting bad", path);
                     #endif
                     STB_DPStopDecoding(path);
                     decoding_state[path] = DSTATE_STARTING_BAD_SIGNAL;
                  }
               }
               else if ((type == EXTERNAL_EVENT) && (event == STB_EVENT_DECODE_LOCKED))
               {
                  #ifdef DEBUG_DECODING
                  STB_SPDebugWrite("   new decode state(%d): Locked", path);
                  #endif
                  decoding_state[path] = DSTATE_LOCKED;
                  if (path == STB_DPGetPlaybackPath())
                  {
                     STB_AVBlankVideo(0, TRUE);
                     APVR_PausePlay();
                  }
#if defined(INTEGRATE_HBBTV)
                  if (STB_DPIsLivePath(path))
                  {
                     /* Inform the presentation engine that the channel is locked */
                     if (service_required[path] != NULL)
                     {
                        ADB_GetServiceIds(service_required[path], &onet_id, &ts_id, &serv_id);
                     }
                     else
                     {
                        onet_id = ADB_INVALID_DVB_ID;
                        ts_id = ADB_INVALID_DVB_ID;
                        serv_id = ADB_INVALID_DVB_ID;
                     }
                     HBBTV_NotifyChannelChangeStatus(onet_id, ts_id, serv_id,
                        HBBTV_CHANNEL_CHANGE_PARENTAL_LOCKED);
                  }
#endif
               }
               else if ((type == EXTERNAL_EVENT) &&
                        ((event == STB_EVENT_VIDEO_DECODE_STARTED) || (event == STB_EVENT_AUDIO_DECODE_STARTED)))
               {
                  video_status = STB_DPGetVideoStatus(path);
                  audio_status = STB_DPGetAudioStatus(path);

                  if ((video_status == DECODE_RUNNING) || (audio_status == DECODE_RUNNING))
                  {
                     if (video_status == DECODE_RUNNING)
                     {
                        STB_AVBlankVideo(0, FALSE);
#ifdef COMMON_INTERFACE
                        /* Re-evaluate usage rules */
                        ACI_UsageRulesStatusChanged(path);
#endif
                     }

                     if ((video_status == DECODE_RUNNING) ||
                         ((audio_status == DECODE_RUNNING) && (STB_DPGetVideoPID(path) == 0)))
                     {
                        #ifdef DEBUG_DECODING
                        STB_SPDebugWrite("   new decode state(%d): Decoding", path);
                        #endif
                        decoding_state[path] = DSTATE_DECODING;
                     }
                  }
               }
               break;
            }

            case DSTATE_LOCKED:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_RELEASE_DECODE_LOCK))
               {
                  STB_DPSetLockEnable(path, FALSE);
                  if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
                  {
                     if (path == STB_DPGetPlaybackPath())
                     {
                        STB_AVBlankVideo(0, FALSE);
                     }
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Starting", path);
                     #endif
                     STB_DPStartDecoding(path);
                     decoding_state[path] = DSTATE_STARTING;
                  }
                  else
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Starting bad", path);
                     #endif
                     decoding_state[path] = DSTATE_STARTING_BAD_SIGNAL;
                  }
               }
               break;
            }

            case DSTATE_DECODING:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] != TUNED_STATUS_GOOD_SIGNAL)
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Signal pause", path);
                     #endif
                     if (STB_DPIsLivePath(path))
                     {
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_DECODE_PAUSED, &path, sizeof(U8BIT));
                     }
                     decoding_state[path] = DSTATE_PAUSED_SIGNAL;
                     STB_DPStopDecoding(path);
                  }
               }
               else if ((type == EXTERNAL_EVENT) && (event == STB_EVENT_VIDEO_DECODE_STARTED))
               {
                  video_status = STB_DPGetVideoStatus(path);

                  //the video unblanking belows applies to both live & playback path
                  if ((video_status == DECODE_RUNNING) && STB_DPIsDecodingPath(path))
                  {
                     STB_AVBlankVideo(0, FALSE);
#ifdef COMMON_INTERFACE
                     /* Re-evaluate usage rules */
                     ACI_UsageRulesStatusChanged(path);
#endif
                  }
               }
               break;
            }

            case DSTATE_PAUSED_SIGNAL:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
                  {
                     if (STB_DPIsLivePath(path))
                     {
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_DECODE_RESUMED, &path, sizeof(U8BIT));
                        #ifdef DEBUG_DECODING
                        STB_SPDebugWrite("   new decode state(%d): Decoding", path);
                        #endif
                        STB_DPStartDecoding(path);
                     }

                     decoding_state[path] = DSTATE_DECODING;
                  }
               }
               break;
            }

            case DSTATE_PAUSED_USER:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] != TUNED_STATUS_GOOD_SIGNAL)
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): Signal/user pause", path);
                     #endif
                     decoding_state[path] = DSTATE_PAUSED_SIGNAL_AND_USER;
                  }
               }
               break;
            }

            case DSTATE_PAUSED_SIGNAL_AND_USER:
            {
               if ((type == INTERNAL_EVENT) && (event == INT_EVENT_UPDATE))
               {
                  if (tuned_status[path] == TUNED_STATUS_GOOD_SIGNAL)
                  {
                     #ifdef DEBUG_DECODING
                     STB_SPDebugWrite("   new decode state(%d): User pause", path);
                     #endif
                     decoding_state[path] = DSTATE_PAUSED_USER;
                  }
               }
               break;
            }
         }
      }

      // update decoding_started flag
      decoding_started[path] = (decoding_state[path] >= DSTATE_DECODING);

      // update decoding_locked flag
      decoding_locked[path] = (decoding_state[path] == DSTATE_LOCKED);

      //update decoding paused flag
      decode_paused_flag[path] = (decoding_state[path] >= DSTATE_PAUSED_SIGNAL);
   }

   STB_OSMutexUnlock(cntrl_mutex[path]);

   FUNCTION_FINISH(ControlDecoding);
}

/**
 *

 *
 * @brief   Implementation of Control A/V output state transition diagram. (See LCDTV spec)
 *                Actions events associated with selection of a/v source.
 *
 * @param   type  - type of event i.e. INTERNAL_EVENT or EXTERNAL_EVENT
 * @param   event - the event to be actioned
 *

 *
 */
static void ControlAvOutput(E_ACTL_EVENT_TYPE type, U32BIT event)
{
   U8BIT path;
   BOOLEAN scart_ip_changed_to_on;
   BOOLEAN scart_ip_changed_to_off;
   E_STB_AV_VIDEO_FORMAT actual_mode;
   U16BIT width, height;

   static BOOLEAN allow_alt_av_sources;
   static BOOLEAN scart_ip_present;

   FUNCTION_START(ControlAvOutput);

   #ifdef DEBUG_AV_OUTPUT
   {
      U8BIT *state_string;
      U8BIT *event_string;

      if ((type == INTERNAL_EVENT) && (event == INT_EVENT_INITIALISE))
      {
         state_string = "Uninitialised";
      }
      else
      {
         switch (av_mode)
         {
            case ACTL_STANDBY_MODE:       {state_string = "Standby"; break; }
            case ACTL_STANDBY_SCART_MODE: {state_string = "Standby, vcr i/p"; break; }
            case ACTL_TV_MODE:            {state_string = "Tv mode"; break; }
            case ACTL_SCART_MODE:         {state_string = "Vcr i/p mode"; break; }
         }
      }
      event_string = GetEventDebugString(type, event);
      STB_SPDebugWrite("ACTL_AvOutput(%s): %s", state_string, event_string);
   }
   #endif

   if ((type == INTERNAL_EVENT) && (event == INT_EVENT_INITIALISE))
   {
      // tell app SI task about change in standby state
      ASI_SetStandbyState(FALSE);

      // initialisation...
      av_mode = ACTL_TV_MODE;
      allow_alt_av_sources = FALSE;
      scart_ip_present = FALSE;
      audio_muted = FALSE;
      audio_volume = (U8BIT)APP_NvmRead(VOLUME_NVM);
      ad_volume = (U8BIT)APP_NvmRead(AD_VOLUME_NVM);

      if (ACTL_IsHDMIConnected())
      {
         ACTL_UpdateVideoMode((E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), TRUE);
      }

      #ifdef INCLUDE_UHF_MODULATOR
      STB_AVSetUhfModulatorChannel((U8BIT)APP_NvmRead(RF_CHANNEL_NUM_NVM));
      #endif

      /* Set SPDIF and lip-sync settings */
      path = ACTL_GetActivePath();
      if (path != INVALID_RES_ID)
      {
         STB_AVSetSpdifMode(STB_DPGetPathAudioDecoder(path), (E_STB_DIGITAL_AUDIO_TYPE)APP_NvmRead(SPDIF_OUTPUT_NVM));
         STB_AVSetAudioDelay(STB_DPGetPathAudioDecoder(path), (U16BIT)APP_NvmRead(LIP_SYNC_ADJUSTMENT_NVM));
         STB_AVSetHDMIAudioMode(STB_DPGetPathAudioDecoder(path), (E_STB_DIGITAL_AUDIO_TYPE)APP_NvmRead(HDMI_AUDIO_OUTPUT_NVM));
      }

      STB_AVSetAudioMute(0, FALSE);

      AVSetAudioVolumes(FALSE);

      #ifdef DEBUG_AV_OUTPUT
      STB_SPDebugWrite("   new state: TV mode");
      #endif
   }
   else if ((type == INTERNAL_EVENT) && (event == INT_EVENT_ALT_AV_ALLOWED))
   {
      allow_alt_av_sources = TRUE;
   }
   else if ((type == INTERNAL_EVENT) && (event == INT_EVENT_ALT_AV_NOT_ALLOWED))
   {
      allow_alt_av_sources = FALSE;
   }
   else
   {
      scart_ip_changed_to_on = FALSE;
      scart_ip_changed_to_off = FALSE;
#if 0
      if ((type == EXTERNAL_EVENT) && (event == STB_EVENT_SCART_FORCE))
      {
         scart_ip_changed_to_on = TRUE;
         scart_ip_present = TRUE;
      }
      else if ((type == EXTERNAL_EVENT) && (event == STB_EVENT_SCART_DISCONNECTED))
      {
         scart_ip_changed_to_off = TRUE;
         scart_ip_present = FALSE;
      }
#endif

      switch (av_mode)
      {
         case ACTL_STANDBY_MODE:
         {
            if ((type == INTERNAL_EVENT) && (event == INT_EVENT_STANDBY_OFF))
            {
               // tell app SI task about change in standby state
               ASI_SetStandbyState(FALSE);
               STB_AVSetHDMIStandby(FALSE);

               AVSetAudioVolumes(FALSE);

               // changing to TV mode
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Tv mode");
               #endif
               STB_AVSetAVOutput(TRUE);
               STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, (E_STB_AV_SOURCES)APP_NvmRead(TV_SCART_TYPE_NVM), 0);
               DBGPRINT("");
               actual_mode = ACTL_GetActualVideoMode(&width, &height);
               STB_AVSetTVType(0, (E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), actual_mode);
               STB_AVSetAVOutputSource(AV_OUTPUT_VCR_SCART, AV_SOURCE_ENCODER_COMPOSITE, 0);
               av_mode = ACTL_TV_MODE;
            }
            else if (scart_ip_present == TRUE)
            {
               // change to standby/vcr ip mode and allow input from vcr scart
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Standby, vcr i/p");
               #endif
               STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, AV_SOURCE_VCR_COMPOSITE, 0);
               av_mode = ACTL_STANDBY_SCART_MODE;
            }
            break;
         }
         case ACTL_STANDBY_SCART_MODE:
         {
            if ((type == INTERNAL_EVENT) && (event == INT_EVENT_STANDBY_OFF))
            {
               // tell app SI task about change in standby state
               ASI_SetStandbyState(FALSE);
               STB_AVSetHDMIStandby(FALSE);

               AVSetAudioVolumes(FALSE);

               // changing to TV mode
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Tv mode");
               #endif
               STB_AVSetAVOutput(TRUE);
               STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, (E_STB_AV_SOURCES)APP_NvmRead(TV_SCART_TYPE_NVM), 0);
               DBGPRINT("");
               STB_AVSetTVType(0, (E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), VIDEO_FORMAT_PALI);
               STB_AVSetAVOutputSource(AV_OUTPUT_VCR_SCART, AV_SOURCE_ENCODER_COMPOSITE, 0);
               av_mode = ACTL_TV_MODE;
            }
            else if (scart_ip_present == FALSE)
            {
               // change to standby mode and blank video output
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Standby");
               #endif
               STB_AVSetAVOutput(FALSE);
               STB_AVSetHDMIStandby(TRUE);
               av_mode = ACTL_STANDBY_MODE;
            }
            break;
         }
         case ACTL_TV_MODE:
         {
            if ((type == INTERNAL_EVENT) && (event == INT_EVENT_STANDBY_ON))
            {
               if (scart_ip_present == TRUE)
               {
                  // changing to standby/vcr ip mode
                  #ifdef DEBUG_AV_OUTPUT
                  STB_SPDebugWrite("   new state: Standby, vcr i/p");
                  #endif
                  STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, AV_SOURCE_VCR_COMPOSITE, 0);
                  av_mode = ACTL_STANDBY_SCART_MODE;
               }
               else
               {
                  // changing to normal standby mode
                  #ifdef DEBUG_AV_OUTPUT
                  STB_SPDebugWrite("   new state: Standby");
                  #endif
                  STB_AVSetAVOutput(FALSE);
                  STB_AVSetHDMIStandby(TRUE);
                  av_mode = ACTL_STANDBY_MODE;
               }
               // tell app SI task about change in standby state
               ASI_SetStandbyState(TRUE);
            }
            else if ((allow_alt_av_sources == TRUE) && (scart_ip_changed_to_on == TRUE))
            {
               // change to vcr ip mode
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Vcr i/p mode");
               #endif
               STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, AV_SOURCE_VCR_COMPOSITE, 0);
               av_mode = ACTL_SCART_MODE;
            }
            break;
         }
         case ACTL_SCART_MODE:
         {
            if ((type == INTERNAL_EVENT) &&
               ((event == INT_EVENT_STANDBY_ON) || (event == INT_EVENT_STANDBY_VCR_ACTIVE)))
            {
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Standby, vcr i/p");
               #endif
               // tell app SI task about change in standby state
               if (event == INT_EVENT_STANDBY_VCR_ACTIVE)
               {
                  STB_AVSetAudioMute(0, TRUE);
                  STB_AVSetAVOutputSource(AV_OUTPUT_VCR_SCART, AV_SOURCE_ENCODER_COMPOSITE, 0);
               }
               else
               {
                  ASI_SetStandbyState(TRUE);
               }
               av_mode = ACTL_STANDBY_SCART_MODE;
            }
            else if (scart_ip_changed_to_off == TRUE)
            {
               // change to tv mode
               #ifdef DEBUG_AV_OUTPUT
               STB_SPDebugWrite("   new state: Tv mode");
               #endif
               STB_AVSetAVOutputSource(AV_OUTPUT_TV_SCART, (E_STB_AV_SOURCES)APP_NvmRead(TV_SCART_TYPE_NVM), 0);
               DBGPRINT("");
               STB_AVSetTVType(0, (E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM), VIDEO_FORMAT_PALI);
               STB_AVSetAVOutputSource(AV_OUTPUT_VCR_SCART, AV_SOURCE_ENCODER_COMPOSITE, 0);
               av_mode = ACTL_TV_MODE;
            }
            break;
         }
         default:
            break;
      }
   }

   FUNCTION_FINISH(ControlAvOutput);
}

/**
 *

 *
 * @brief   Sets the audio volume and optionally saves the setting to NVM
 *
 * @param   save_to_nvm - TRUE if the setting should be saved to NVM
 *

 *
 */
static void AVSetAudioVolumes(BOOLEAN save_to_nvm)
{
   U8BIT path;

   path = ACTL_GetActivePath();
   if ((path == INVALID_RES_ID) || !STB_DPIsDecodingPath(path))
   {
      path = STB_DPGetPlaybackPath();
   }

   if (path != INVALID_RES_ID)
   {
      S16BIT scaled_vol = 0;
      S16BIT ad_scaled_vol = 0;

      if(audio_muted == FALSE)
      {
         scaled_vol = audio_volume + (volume_scaling * audio_volume)/100;
         if(scaled_vol < 0)
           scaled_vol = 0;
         if(scaled_vol > 100)
           scaled_vol = 100;

         ad_scaled_vol = ad_volume + (volume_scaling * ad_volume)/100;
         if(ad_scaled_vol < 0)
           ad_scaled_vol = 0;
         if(ad_scaled_vol > 100)
           ad_scaled_vol = 100;
      }

      STB_AVSetAudioVolume(STB_DPGetPathAudioDecoder(path), scaled_vol);
      STB_AVSetADVolume(STB_DPGetPathAudioDecoder(path), ad_scaled_vol);
   }

   if (save_to_nvm)
   {
      /* Save the unscaled volume levels whether the actual volume was able to be set or not */
      APP_NvmSave(VOLUME_NVM, audio_volume, TRUE);
      APP_NvmSave(AD_VOLUME_NVM, ad_volume, TRUE);
   }
}

/*!**************************************************************************
 * @brief   Acquires a decode path for live signal for the given transport
 * @param   t_ptr the transport the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be allocated
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the decode path
 * @return  ID of path or INVALID_RES_ID on failure
 ****************************************************************************/
static U8BIT AcquirePathForTransport(void *t_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info)
{
   E_STB_DP_SIGNAL_TYPE sig_type;
   U8BIT path;
   E_STB_DP_RES_OWNER owner;

   FUNCTION_START(AcquirePathForTransport);

   path = INVALID_RES_ID;

   /* find out what tuner's required */
   if (t_ptr != NULL)
   {
      sig_type = ADB_GetTransportSignalType(t_ptr);

      if (owner_info != NULL)
      {
         owner = owner_info->owner;
      }
      else
      {
         owner = RES_OWNER_NONE;
      }

      path = STB_DPAcquireTunerPath(sig_type, NULL, t_ptr, owner, DP_PRIORITY_LOW, with_decoders,
         for_recording);
      if ((path != INVALID_RES_ID) && (owner_info != NULL))
      {
         STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);
      }
   }

   FUNCTION_FINISH(AcquirePathForTransport);

   return(path);
}

/*!**************************************************************************
 * @brief   Acquires a decode path for live signal for the given service
 * @param   s_ptr the service the decode path will be used for
 * @param   with_decoders TRUE if decoders are to be allocated
 * @param   for_recording TRUE if the path will be used for recording
 * @param   owner_info owner of the decode path
 * @return  ID of path or INVALID_RES_ID on failure
 ****************************************************************************/
static U8BIT AcquirePathForService(void *s_ptr, BOOLEAN with_decoders, BOOLEAN for_recording,
   S_ACTL_OWNER_INFO *owner_info)
{
   void *t_ptr;
   E_STB_DP_SIGNAL_TYPE sig_type;
   U8BIT path;
   E_STB_DP_RES_OWNER owner;

   FUNCTION_START(AcquirePathForService);

   path = INVALID_RES_ID;

   /* Get the transport for the service to find out what tuner's required */
   t_ptr = ADB_GetServiceTransportPtr(s_ptr);
   if (t_ptr != NULL)
   {
      sig_type = ADB_GetTransportSignalType(t_ptr);

      if (owner_info != NULL)
      {
         owner = owner_info->owner;
      }
      else
      {
         owner = RES_OWNER_NONE;
      }

      path = STB_DPAcquireTunerPath(sig_type, s_ptr, t_ptr, owner, DP_PRIORITY_LOW, with_decoders,
         for_recording);
      if ((path != INVALID_RES_ID) && (owner_info != NULL))
      {
         STB_DPSetOwnerData(path, owner_info->data, owner_info->data_size);
      }
   }

   FUNCTION_FINISH(AcquirePathForService);

   return(path);
}

/*!**************************************************************************
 * @brief   Function to handle events received during a service search
 * @param   path - decode path being used for the search
 * @param   event - event that caused this function to be called
 ****************************************************************************/
static void ContinueServiceSearch(U8BIT path, U32BIT event)
{
   void **tlist;
   U16BIT num_transports;
   void *t_ptr;
   S_MANUAL_TUNING_PARAMS tune_params;

   FUNCTION_START(ContinueServiceSearch);

   if ((required_search_type == SEARCH_TYPE_SERVICE_NETWORK) ||
       (required_search_type == SEARCH_TYPE_MANUAL_NETWORK))
   {
      ADB_GetTransportList(&tlist, &num_transports);
   }
   else
   {
      tlist = NULL;
      num_transports = 0;
   }

   if (event == STB_EVENT_TUNE_NOTLOCKED)
   {
      /* Report no signal to the SI task */
      if (((t_ptr = ADB_GetTunedTransport(path)) == NULL) && (tlist != NULL))
      {
         t_ptr = tlist[search_list_id];
      }

      if (t_ptr != NULL)
      {
         ADB_ReportNoSignalDuringSearch(t_ptr);
      }
      if (signal_type_required == SIGNAL_COFDM)
      {
         if (ACTL_GetTerRfChanType(search_list_id) == TERR_TYPE_DVBT2)
         {
            /* Handle T2 PLP tuning */
            if ((terr_hierarchy != TUNE_THIERARCHY_NONE) && (t2_plp_id > 1))
            {
               /* Try tuning to the next PLP on this same frequency */
               t2_plp_id--;
               STB_DPSetTerrPLP(path, t2_plp_id);
            }
            else
            {
               /* Finished tuning to each possible PLP on this frequency,
                * so move to the next channel */
               search_list_id = NextSearchId(tlist, num_transports);
               terr_hierarchy = TUNE_THIERARCHY_NONE;
            }
         }
         else
         {
            /* DVB-T channel, so nothing else to do; move to next channel */
            terr_hierarchy = TUNE_THIERARCHY_NONE;
            t2_plp_id = 0;
            STB_DPSetTerrPLP(path, t2_plp_id);
            search_list_id = NextSearchId(tlist, num_transports);
         }
      }
      else
      {
         /* Move to next channel */
         search_list_id = NextSearchId(tlist, num_transports);
      }

      if (current_search_type == SEARCH_TYPE_SERVICE_FREQ)
      {
         if (((signal_type_required == SIGNAL_QPSK) && (search_list_id < GetNumSatSearchIds())) ||
             (search_list_id < ACTL_GetNumRfChanArrayEntries(signal_type_required)))
         {
            switch (signal_type_required)
            {
               case SIGNAL_ANALOG:
                  /* Not finished yet - tune to next channel */
                  ACTL_TuneToRfChanArrayAnalogEntry(path, search_list_id);
                  break;
               case SIGNAL_COFDM:
                  /* Not finished yet - tune to next channel or PLP */
                  ACTL_TuneToRfChanArrayEntry(path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               case SIGNAL_QAM:
                  /* Not finished yet - tune to next channel */
                  ACTL_TuneToRfChanArrayEntry(path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               case SIGNAL_QPSK:
                  /* Not finished yet - map the search list id to the next set of tuning params */
                  MapIdToSatTuneParams(search_list_id, &tune_params);
                  ACTL_TuneToUserDefinedParams(path, &tune_params, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               default:
                  break;
            }
         }
         else
         {
            /* No more channels so search has finished */
            EndSearch(path);
         }
      }
      else if ((current_search_type == SEARCH_TYPE_SERVICE_NETWORK) ||
               (current_search_type == SEARCH_TYPE_MANUAL_NETWORK))
      {
#ifdef DEBUG_SERVICE_SEARCH
         STB_SPDebugWrite("Tuning to transport %u of %u", search_list_id, num_transports);
#endif
         if ((search_list_id < num_transports) && tlist)
         {
            ACTL_TuneToTransport(path, NULL, tlist[search_list_id], ACTL_SI_CHANNEL_SEARCH_NO_NIT, FALSE);
         }
         else
         {
            /* No more transports to search */
            EndSearch(path);
         }
      }

      if (!ASTE_InStandby())
      {
         /* Send event to update the UI */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
      }
   }
   else if ((event == STB_EVENT_SEARCH_SUCCESS) || (event == STB_EVENT_SEARCH_FAIL))
   {
      t_ptr = ADB_GetTunedTransport(path);
      if (t_ptr != NULL)
      {
       if ((signal_type_required == SIGNAL_QAM) &&
           (SEARCH_TYPE_MANUAL_FREQ == current_search_type) &&
           (SEARCH_TYPE_MANUAL_NETWORK == required_search_type))
       {
         // the search flag is controlled by nit_update callback function    
       }
       else {
         /* Indicate that this transport has now been used during the search */
         ADB_SetTransportSearchFlag(t_ptr, TRUE);
       }
      }
      ACTL_TuneOff(path);
      if ((signal_type_required == SIGNAL_COFDM) &&
          (ACTL_GetTerRfChanType(search_list_id) == TERR_TYPE_DVBT2))
      {
         if (terr_hierarchy == TUNE_THIERARCHY_NONE)
         {
            /* Query the hierarchy to see how many PLPs are on this frequency and
             * try tuning to each one, starting at the maximum value */
            terr_hierarchy = STB_TuneGetActualTerrHierarchy(STB_DPGetPathTuner(path));
            if ((terr_hierarchy != TUNE_THIERARCHY_NONE) &&
                (terr_hierarchy > TUNE_THIERARCHY_NONE))
            {
               /* The hierarchy value returned is the maximum PLP id to be used */
               t2_plp_id = (U8BIT)terr_hierarchy;
               STB_DPSetTerrPLP(path, t2_plp_id);
            }
            else
            {
               if (current_search_type == SEARCH_TYPE_MANUAL_FREQ)
               {
                  /* Search has finished */
                  FinishManualSearch(path, event);
               }
               else
               {
                  /* Just a single PLP on this frequency so move to next channel */
                  search_list_id = NextSearchId(tlist, num_transports);
                  terr_hierarchy = TUNE_THIERARCHY_NONE;
               }
            }
         }
         else if (t2_plp_id > 1)
         {
            t2_plp_id--;
            STB_DPSetTerrPLP(path, t2_plp_id);
         }
         else
         {
            /* Finished tuning to each possible PLP on this frequency */
            if (current_search_type == SEARCH_TYPE_MANUAL_FREQ)
            {
               /* No more PLPs, so search has completed */
               FinishManualSearch(path, event);
            }
            else
            {
               /* Move to the next channel */
               search_list_id = NextSearchId(tlist, num_transports);
               terr_hierarchy = TUNE_THIERARCHY_NONE;
               t2_plp_id = 0;
               STB_DPSetTerrPLP(path, t2_plp_id);
            }
         }
      }
      else
      {
         /* Nothing else to do; move to next channel */
         search_list_id = NextSearchId(tlist, num_transports);
         // 'current_search_type' may be changed in NextSearchId()
      }

      if ((current_search_type == SEARCH_TYPE_SERVICE_FREQ) ||
          (current_search_type == SEARCH_TYPE_MANUAL_FREQ))
      {
         if (((signal_type_required == SIGNAL_QPSK) && (search_list_id < GetNumSatSearchIds())) ||
             (search_list_id < ACTL_GetNumRfChanArrayEntries(signal_type_required)))
         {
            switch (signal_type_required)
            {
               case SIGNAL_ANALOG:
                  /* Not finished yet - tune to next channel */
                  ACTL_TuneToRfChanArrayAnalogEntry(path, search_list_id);
                  break;
               case SIGNAL_COFDM:
                  /* Not finished yet - tune to next channel or PLP */
                  ACTL_TuneToRfChanArrayEntry(path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               case SIGNAL_QAM:
                  /* Not finished yet - tune to next channel */
                  ACTL_TuneToRfChanArrayEntry(path, search_list_id, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               case SIGNAL_QPSK:
                  /* Not finished yet - map the search list id to the next set of tuning params */
                  MapIdToSatTuneParams(search_list_id, &tune_params);
                  ACTL_TuneToUserDefinedParams(path, &tune_params, ACTL_SI_CHANNEL_SEARCH, FALSE);
                  break;
               default:
                  break;
            }
         }
         else
         {
            /* No more channels so search has finished */
            EndSearch(path);
         }
      }
      else if ((current_search_type == SEARCH_TYPE_SERVICE_NETWORK) ||
               (current_search_type == SEARCH_TYPE_MANUAL_NETWORK))
      {
#ifdef DEBUG_SERVICE_SEARCH
         STB_SPDebugWrite("Tuning to transport %u of %u", search_list_id, num_transports);
#endif
         if (search_list_id < num_transports)
         {
            ACTL_TuneToTransport(path, NULL, tlist[search_list_id], ACTL_SI_CHANNEL_SEARCH_NO_NIT, FALSE);
         }
         else
         {
            /* No more transports to search */
            EndSearch(path);
         }
      }

      if (!ASTE_InStandby())
      {
         /* Send event to update the UI */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
      }
   }

   if (tlist != NULL)
   {
      ADB_ReleaseTransportList(tlist, num_transports);
   }

   FUNCTION_FINISH(ContinueServiceSearch);
}

/*!**************************************************************************
 * @brief   Checks for the switch from frequency search to network search and returns
 *          the next search id as appropriate.
 * @return  Next search id that should be used
 ****************************************************************************/
static U16BIT NextSearchId(void **tlist, U16BIT num_transports)
{
   U16BIT next_id;

   FUNCTION_START(NextSearchId);

   next_id = search_list_id;

   if ((required_search_type == SEARCH_TYPE_SERVICE_NETWORK) &&
       (current_search_type == SEARCH_TYPE_SERVICE_FREQ) && tlist)
   {
      /* Check to see if any transports have been added via delivery descriptors. If they have,
       * they won't have been searched yet */
      for (next_id = 0; next_id < num_transports; )
      {
         if (!ADB_GetTransportSearchFlag(tlist[next_id]))
         {
            /* Found a transport that hasn't been searched yet,
             * so take this as the switch to network search mode */
            current_search_type = SEARCH_TYPE_SERVICE_NETWORK;
            break;
         }
         else
         {
            next_id++;
         }
      }

      if (current_search_type != SEARCH_TYPE_SERVICE_NETWORK)
      {
         next_id = search_list_id + 1;
      }
   }
   else if ((required_search_type == SEARCH_TYPE_MANUAL_NETWORK) &&
       (current_search_type == SEARCH_TYPE_MANUAL_FREQ) && tlist)
   {
      /* Check to see if any transports have been added via delivery descriptors. If they have,
       * they won't have been searched yet */
      for (next_id = 0; next_id < num_transports; )
      {
         if (!ADB_GetTransportSearchFlag(tlist[next_id]))
         {
            /* Found a transport that hasn't been searched yet,
             * so take this as the switch to network search mode */
            current_search_type = SEARCH_TYPE_MANUAL_NETWORK;
            break;
         }
         else
         {
            next_id++;
         }
      }
   }
   else
   {
      if (((current_search_type == SEARCH_TYPE_SERVICE_NETWORK) ||
          (current_search_type == SEARCH_TYPE_MANUAL_NETWORK)) && tlist)
      {
         /* Find the next transport that hasn't been searched */
         for (; (next_id < num_transports) && ADB_GetTransportSearchFlag(tlist[next_id]); next_id++)
            ;
      }
      else
      {
         next_id++;
      }
   }

   FUNCTION_FINISH(NextSearchId);

   return(next_id);
}

static U16BIT GetNumSatSearchIds(void)
{
   U32BIT country_code;
   U16BIT max_scan_ids;

   FUNCTION_START(GetNumSatSearchIds);

   country_code = ACFG_GetCountry();

   max_scan_ids = (ACFG_GetMaxSatelliteScanFreq(country_code) -
                   ACFG_GetMinSatelliteScanFreq(country_code)) / ACFG_GetSatelliteScanFreqInc(country_code) + 1;
   max_scan_ids *= ACFG_GetSatelliteScanNumSymbolRates(country_code);
   max_scan_ids *= 2; /* For horizontal and vertical */
   if (ACFG_GetSatelliteScanDvbS2(country_code))
   {
      max_scan_ids *= 2;
   }

   FUNCTION_FINISH(GetNumSatSearchIds);

   return(max_scan_ids);
}

static void MapIdToSatTuneParams(U16BIT search_id, S_MANUAL_TUNING_PARAMS *tuning_params)
{
   U32BIT country_code;
   U16BIT num_tunes_per_freq;
   U16BIT num_tunes_per_srate;
   U8BIT srate_id;
   U16BIT *srates;

   FUNCTION_START(MapIdToSatTuneParams);

   country_code = ACFG_GetCountry();

   /* Tune to horizontal and vertical for each symbol rate */
   num_tunes_per_srate = 2;
   if (ACFG_GetSatelliteScanDvbS2(country_code))
   {
      num_tunes_per_srate *= 2;
   }

   num_tunes_per_freq = num_tunes_per_srate * ACFG_GetSatelliteScanNumSymbolRates(country_code);

   tuning_params->freq = ACFG_GetMinSatelliteScanFreq(country_code) + (search_id / num_tunes_per_freq) *
      ACFG_GetSatelliteScanFreqInc(country_code);

   srate_id = (search_id % num_tunes_per_freq);
   srates = ACFG_GetSatelliteScanSymbolRates(country_code);

   tuning_params->u.sat.symbol_rate = srates[srate_id / num_tunes_per_srate];

   if ((srate_id % 2) == 0)
   {
      tuning_params->u.sat.polarity = POLARITY_HORIZONTAL;
   }
   else
   {
      tuning_params->u.sat.polarity = POLARITY_VERTICAL;
   }

   if (ACFG_GetSatelliteScanDvbS2(country_code))
   {
      if (((srate_id % 4) / 2) == 0)
      {
         tuning_params->u.sat.dvb_s2 = FALSE;
      }
      else
      {
         tuning_params->u.sat.dvb_s2 = TRUE;
      }
   }
   else
   {
      tuning_params->u.sat.dvb_s2 = FALSE;
   }

   tuning_params->u.sat.fec = FEC_AUTOMATIC;
   tuning_params->u.sat.modulation = MOD_QPSK;
   tuning_params->u.sat.satellite = ADB_GetNextSatellite(NULL);

#ifdef DEBUG_SERVICE_SEARCH
   STB_SPDebugWrite("%s(%u): f=%u, sr=%u, p=%u, s2=%u", __FUNCTION__, search_id, tuning_params->freq,
      tuning_params->u.sat.symbol_rate, tuning_params->u.sat.polarity, tuning_params->u.sat.dvb_s2);
#endif

   FUNCTION_FINISH(MapIdToSatTuneParams);
}

static void ReleasePath(U8BIT path)
{
   FUNCTION_START(ReleasePath);

   /* Release the path being used */
   STB_DPReleasePath(path, RES_OWNER_DVB);
   if (search_path == path)
   {
      search_path = INVALID_RES_ID;
   }

   FUNCTION_FINISH(ReleasePath);
}

/*!**************************************************************************
 * @brief   Called at the end of a service search to finalise the database and
 *          determine whether a target region needs to be chosen to finish the search.
 ****************************************************************************/
static void EndSearch(U8BIT path)
{
   FUNCTION_START(EndSearch);

   /* Stop SI processing and release the path being used for the search */
   if (path != INVALID_RES_ID)
   {
      STB_DPStopSI(path);
      ReleasePath(path);

      ACTL_EnableCiModule();

      /* Ensure first boot is false and set searching state to false */
      APP_NvmSave(FIRST_BOOT_NVM, (U32BIT)FALSE, FALSE);
      APP_NvmSave(SEARCHING_STATE_NVM, (U32BIT)FALSE, TRUE);

      current_search_type = SEARCH_TYPE_NOSEARCH;
      required_search_type = SEARCH_TYPE_NOSEARCH;
      
      ASI_ResetAppSiModeFlag(path);

      if (ASTE_InStandby())
      {
         /* Was performing a background search in standby so re-enter standby
          * to see if anything else needs to be done */
         ASTE_SearchComplete();

         /* Although events aren't sent for each transport during a service search when in standby,
          * do send one now the search has finished as it was the app that started the search
          * and so needs to be informed that it's now finished */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
      }
   }

   FUNCTION_FINISH(EndSearch);
}

/*!**************************************************************************
 * @brief   Function to handle events received during a transport search (e.g. startup, SSU)
 * @param   path - decode path being used for the search
 ****************************************************************************/
static void ContinueTransportSearch(U8BIT path)
{
   void **tlist;
   U16BIT num_transports;
   BOOLEAN search_complete;

   FUNCTION_START(ContinueTransportSearch);

   search_complete = FALSE;
   search_list_id++;

   if (search_list_id < search_num_transports)
   {
      ADB_GetTransportListForTunerType(current_search_tuner_type, &tlist, &num_transports);

      /* Release the current path and acquire one suitable for the next transport */
      STB_DPReleasePath(path, RES_OWNER_DVB);

      do
      {
         path = STB_DPAcquireTunerPath(ADB_GetTransportSignalType(tlist[search_list_id]),
               NULL, NULL, RES_OWNER_DVB, DP_PRIORITY_LOW, FALSE, FALSE);

         if (path == INVALID_RES_ID)
         {
            /* No tuner exists for this type of transport!
             * Try the next one */
            search_list_id++;
         }
      }
      while ((path == INVALID_RES_ID) && (search_list_id < search_num_transports));

      if (path != INVALID_RES_ID)
      {
         /* Next transport */
         switch (current_search_type)
         {
            case SEARCH_TYPE_STARTUP:
               ACTL_TuneToTransport(path, NULL, tlist[search_list_id], ACTL_SI_STARTUP_SEARCH, FALSE);
               break;
            case SEARCH_TYPE_TOT:
               ACTL_TuneToTransport(path, NULL, tlist[search_list_id], ACTL_SI_TOT_SEARCH, FALSE);
               break;
            case SEARCH_TYPE_SSU:
               ACTL_TuneToTransport(path, NULL, tlist[search_list_id], ACTL_SI_DVB_SSU_SEARCH, FALSE);
               break;

            default:
               /* Unknown search type so finish search */
               current_search_type = SEARCH_TYPE_NOSEARCH;
               required_search_type = SEARCH_TYPE_NOSEARCH;
               current_search_tuner_type = SIGNAL_NONE;
               break;
         }
      }
      else
      {
         /* Search has completed */
         current_search_type = SEARCH_TYPE_NOSEARCH;
         required_search_type = SEARCH_TYPE_NOSEARCH;
         current_search_tuner_type = SIGNAL_NONE;
      }

      ADB_ReleaseTransportList(tlist, num_transports);
   }
   else
   {
      /* Finished */
      required_search_type = SEARCH_TYPE_NOSEARCH;

      /* Release the path being used */
      ReleasePath(path);

      if (current_search_type == SEARCH_TYPE_SSU)
      {
         /* SSU search in standby will have been started by the app, so it needs to decide whether
          * to go back into standby */
         current_search_type = SEARCH_TYPE_NOSEARCH;
         current_search_tuner_type = SIGNAL_NONE;

         if (ASTE_InStandby())
         {
            ASTE_SearchComplete();

            search_complete = TRUE;
         }
      }
      else
      {
         current_search_type = SEARCH_TYPE_NOSEARCH;
         current_search_tuner_type = SIGNAL_NONE;

         if (ASTE_InStandby())
         {
            /* Was performing a background search in standby so re-enter standby
             * to see if anything else needs to be done */
            ASTE_SearchComplete();
            ACTL_EnterStandby();

            search_complete = TRUE;
         }
      }
   }

   /* A startup search will be done when in standby so the event wants to be sent otherwise
    * the UI won't know when the search completes */
   if (!ASTE_InStandby() || (ASTE_InStandby() && search_complete))
   {
      /* Send event to update the UI */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);
   }

   FUNCTION_FINISH(ContinueTransportSearch);
}

static BOOLEAN StartEitSearch(void)
{
   BOOLEAN search_started;
   void *t_ptr;
   E_ACTL_SI_SRCH_REQD si_search;

   FUNCTION_START(StartEitSearch);

   for (search_started = FALSE; (search_list_id < search_num_transports) && !search_started; search_list_id++)
   {
      t_ptr = search_transport_list[search_list_id];

      search_path = STB_DPAcquireTunerPath(ADB_GetTransportSignalType(t_ptr), NULL, t_ptr,
         RES_OWNER_DVB, DP_PRIORITY_LOW, FALSE, FALSE);
      if (search_path != INVALID_RES_ID)
      {
         EIT_DBG(("%s: Tuning to transport %u of %u, freq %lu", __FUNCTION__,
            search_list_id, search_num_transports, ADB_GetTransportFreqHz(t_ptr)));

         switch (current_search_type)
         {
            case SEARCH_TYPE_EIT_PF:
               si_search = ACTL_SI_EVENT_PF_SEARCH;
               break;
            case SEARCH_TYPE_EIT_SCHED:
               si_search = ACTL_SI_EVENT_SCHED_SEARCH;
               break;
            case SEARCH_TYPE_EIT_PF_SCHED:
               si_search = ACTL_SI_EVENT_PF_SCHED_SEARCH;
               break;
            case SEARCH_TYPE_NOSEARCH:
            default:
               si_search = ACTL_SI_NO_SEARCH;
               break;
         }

         if (si_search != ACTL_SI_NO_SEARCH)
         {
            ACTL_TuneToTransport(search_path, NULL, t_ptr, si_search, TRUE);
            search_started = TRUE;
         }
      }
   }

   FUNCTION_FINISH(StartEitSearch);

   return(search_started);
}

static void ContinueEitSearch(U8BIT path)
{
   BOOLEAN search_started;

   FUNCTION_START(ContinueEitSearch);

   if (path != INVALID_RES_ID)
   {
      ACTL_TuneOff(path);
      ReleasePath(path);
      path = INVALID_RES_ID;
   }

   search_started = StartEitSearch();
   if (!search_started)
   {
      FinishEitSearch(path);
      if (ASTE_InStandby())
      {
         /* Was performing an EIT search in standby so re-enter standby
          * to see if anything else needs to be done */
         ACTL_EnterStandby();
      }
   }

   FUNCTION_FINISH(ContinueEitSearch);
}

/*!**************************************************************************
 * @brief   Function to handle events received during a manual search
 * @param   path - decode path being used for the search
 * @param   event - event that caused this function to be called
 ****************************************************************************/
static void FinishManualSearch(U8BIT path, U32BIT event)
{
   void *t_ptr;

   FUNCTION_START(FinishManualSearch);

   if (event == STB_EVENT_TUNE_NOTLOCKED)
   {
      if ((t_ptr = ADB_GetTunedTransport(path)) != NULL)
      {
         ADB_ReportNoSignalDuringSearch(t_ptr);
      }

      ACTL_TuneOff(path);

      /* Search finished */
      current_search_type = SEARCH_TYPE_NOSEARCH;
      required_search_type = SEARCH_TYPE_NOSEARCH;

      ReleasePath(path);
   }
   else if ((event == STB_EVENT_SEARCH_SUCCESS) || (event == STB_EVENT_SEARCH_FAIL))
   {
      if ((t_ptr = ADB_GetTunedTransport(path)) != NULL)
      {
         ADB_SetTransportSearchFlag(t_ptr, TRUE);
      }

      ACTL_TuneOff(path);

      /* Search finished */
      current_search_type = SEARCH_TYPE_NOSEARCH;
      required_search_type = SEARCH_TYPE_NOSEARCH;

      ReleasePath(path);
   }

   /* Send event to update the UI */
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);

   FUNCTION_FINISH(FinishManualSearch);
}

/*!**************************************************************************
 * @brief   Stops a search, tunes off and releases the path
 * @param   path - decode path being used for the search
 ****************************************************************************/
static void FinishSearch(U8BIT path)
{
   FUNCTION_START(FinishSearch);

   current_search_type = SEARCH_TYPE_NOSEARCH;
   current_search_tuner_type = SIGNAL_NONE;
   required_search_type = SEARCH_TYPE_NOSEARCH;

   ACTL_TuneOff(path);
#if defined(DEBUG_TOT_SEARCH) || defined(DEBUG_EIT_SEARCH)
   STB_SPDebugWrite("FinishSearch stopping SI");
#endif

   if (path != INVALID_RES_ID)
   {
      STB_DPStopSI(path);
      ReleasePath(path);
   }

   /* Send event to update the UI */
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_UI, EV_TYPE_UPDATE, NULL, 0);

   FUNCTION_FINISH(FinishSearch);
}

/**
 * @brief   Returns the date/time an EIT search is required in local time. This may be now.
 * @param   next Pointer to a the variable where the next search date/time (in local time) will be
 *          stored. The value contained in this variable has meaning only if the function returns
 *          TRUE.
 * @return  TRUE if a valid time is returned, FALSE if no EIT search is needed.
 */
static BOOLEAN GetNextEitSearchTime(U32DHMS *next)
{
   U32DHMS last_eit_srch;
   U16BIT n_of_searches;
   U8BIT ohour, omin;
   U16BIT time_mins;
   U32DHMS next_srch_gmt;
   BOOLEAN srch_needed;

   last_eit_srch = (U32DHMS)APP_NvmRead(LAST_EIT_UPDATE_NVM);

   if (last_eit_srch == 0)
   {
      EIT_DBG(("GetNextEitSearchTime: never searched before"));
      /* Never searched before, start counting from now */
      last_eit_srch = GetDateTimeNow();
      APP_NvmSave(LAST_EIT_UPDATE_NVM, last_eit_srch, TRUE);
   }

   n_of_searches = ACFG_GetEitSearchesPerDay();
   if (n_of_searches > 0)
   {
      srch_needed = TRUE;

      time_mins = ((24 * 60)-1) / n_of_searches; /* time to wait in minutes */
      ohour = time_mins / 60;
      omin = time_mins % 60;

      EIT_DBG(("GetNextEitSearchTime: last day=%d time=%02u:%02u n_of_searches=%d, time to wait h=%d m=%d",
               DHMS_DATE(last_eit_srch), DHMS_HOUR(last_eit_srch), DHMS_MINS(last_eit_srch),
               n_of_searches, ohour, omin));

      *next = STB_GCCalculateDHMS(last_eit_srch, DHMS_CREATE(0, ohour, omin, 0), CALC_ADD);

      /* EIT search time is local, so convert to GMT to see if it's in the future */
      next_srch_gmt = STB_GCConvertDHMS(*next, CONV_GMT);

      if (!STB_GCIsFutureDateTime(DHMS_DATE(next_srch_gmt), DHMS_HOUR(next_srch_gmt),
             DHMS_MINS(next_srch_gmt), 0))
      {
         /* Last search was missed, so a search is needed now */
         *next = STB_GCNowDHMSLocal();
      }
   }
   else
   {
      srch_needed = FALSE;
   }

   return(srch_needed);
}

static BOOLEAN EitSearchNeededNow(void)
{
   U32DHMS next_eit_srch;
   U32DHMS now_local;
   BOOLEAN retval = FALSE;

   if (GetNextEitSearchTime(&next_eit_srch))
   {
      now_local = STB_GCNowDHMSLocal();

      if (next_eit_srch <= now_local)
      {
         /* Search needs to be start immediately */
         retval = TRUE;
      }
   }

   return retval;
}

/*!**************************************************************************
 * @brief
 * @return
 ****************************************************************************/
static BOOLEAN EnterStandby(void **recording_service)
{
   BOOLEAN enter_standby;
   U32DHMS now_local;
   U8BIT day, wday, month;
   U16BIT year;
   BOOLEAN wakeup_for_search;
   U32DHMS wakeup_date_time;
   U32BIT time_diff;
   BOOLEAN wakeup_for_recording;
   U16BIT onet_id, trans_id, service_id;
   BOOLEAN standby_set;
   E_STB_DP_SIGNAL_TYPE tuner_type;

   FUNCTION_START(EnterStandby);

   enter_standby = FALSE;
   *recording_service = NULL;

   standby_set = FALSE;

   while (!standby_set)
   {
      standby_set = TRUE;

      now_local = GetDateTimeNow();
      if (DHMS_DATE(now_local) == 0)
      {
         /* Entering standby before date/time has been set, so it isn't possible to work out
          * what should be done, so all we can do is go into standby */
         enter_standby = TRUE;
         break;
      }

      GetNextWakeupTime(&wakeup_date_time, &wakeup_for_search, &tuner_type,
         &wakeup_for_recording, &onet_id, &trans_id, &service_id);

      DBG_STDBY(("EnterStandby: time now=%u@%02u:%02u, wakeup=%u@%02u:%02u",
                 DHMS_DATE(now_local), DHMS_HOUR(now_local), DHMS_MINS(now_local),
                 DHMS_DATE(wakeup_date_time), DHMS_HOUR(wakeup_date_time), DHMS_MINS(wakeup_date_time)));

      /* Is the current date/time within the wakeup time? */
      if ((!wakeup_for_search && !wakeup_for_recording) || (wakeup_date_time > now_local))
      {
         /* No search to be started now, so go into standby */
         enter_standby = TRUE;

         STB_PVRSetStandbyState(TRUE);
         ACTL_DisableCiModule();

         /* Save state to indicate the reason for the next wakeup */
         if (wakeup_for_recording)
         {
            DBG_STDBY(("%s: Setting wakeup for recording", __FUNCTION__));
            APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_WAKE_FOR_RECORDING, TRUE);
         }
         else if (wakeup_for_search)
         {
            DBG_STDBY(("%s: Setting wakeup for search", __FUNCTION__));
            APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_WAKE_FOR_SEARCH, TRUE);
         }
         else
         {
            DBG_STDBY(("%s: No wakeup needed", __FUNCTION__));
            APP_NvmSave(STANDBY_STATE_NVM, (U32BIT)STDBY_STANDBY, TRUE);
         }

         /* Update the RTC in the front panel with the current local time */
         now_local = GetDateTimeNow();
         STB_GCGetDateInfo(DHMS_DATE(now_local), DHMS_HOUR(now_local), DHMS_MINS(now_local),
            &day, &wday, &month, &year);

         STB_FPSetClock(DHMS_DATE(now_local), year, month, day, DHMS_DATE(now_local),
            DHMS_DATE(now_local), DHMS_DATE(now_local));

#ifdef DEBUG_STANDBY
         if (wakeup_for_search || wakeup_for_recording)
         {
            STB_GCGetDateInfo(DHMS_DATE(wakeup_date_time), DHMS_HOUR(wakeup_date_time),
               DHMS_MINS(wakeup_date_time), &day, &wday, &month, &year);
            DBG_STDBY(("%s: Wakeup date/time= %02u/%02u/%u @ %02u:%02u", __FUNCTION__,
                       day, month, year, DHMS_HOUR(wakeup_date_time), DHMS_MINS(wakeup_date_time)));
         }
#endif

         if ((BOOLEAN)APP_NvmRead(STANDBY_POWERSAVE_NVM))
         {
            /* Powersave mode is enabled */
            if (wakeup_for_search || wakeup_for_recording)
            {
               /* Calculate the difference between the current time and the wakeup time in minutes */
               time_diff = (DHMS_DATE(wakeup_date_time) - DHMS_DATE(now_local)) * 24 * 60 +
                  (DHMS_HOUR(wakeup_date_time) - DHMS_HOUR(now_local)) * 60 +
                  (DHMS_MINS(wakeup_date_time) - DHMS_MINS(now_local));

               /* If the wakeup time is very soon, or any time up to when the grace standby period
                * will expire, then don't bother going into passive or low power standby */
               if ((time_diff < 2) || (time_diff <= (U32BIT)(standby_grace_timeout + 59) / 60))
               {
                  enter_standby = FALSE;

                  /* Create a private timer set for the wakeup time */
                  standby_wakeup_timer = CreateTimer(STB_GCConvertDHMS(wakeup_date_time, CONV_GMT));
               }
               else
               {
                  /* Set the wakeup time */
                  STB_GCGetDateInfo(DHMS_DATE(wakeup_date_time), DHMS_HOUR(wakeup_date_time),
                     DHMS_MINS(wakeup_date_time), &day, &wday, &month, &year);

                  STB_HWSetWakeUpTime(DHMS_DATE(wakeup_date_time), year, month, day,
                     DHMS_HOUR(wakeup_date_time), DHMS_MINS(wakeup_date_time), time_diff);
               }
            }
         }
         else
         {
            if (wakeup_for_search || wakeup_for_recording)
            {
               /* Create a private timer set for the wakeup time */
               standby_wakeup_timer = CreateTimer(STB_GCConvertDHMS(wakeup_date_time, CONV_GMT));
            }
         }
      }
      else
      {
         enter_standby = FALSE;

         if (wakeup_for_recording)
         {
            /* Not going into standby because recording will be started soon.
             * Return the service to be recorded */
            *recording_service = ADB_FindServiceByIds(onet_id, trans_id, service_id);
            DBG_STDBY(("%s: Recording to be started", __FUNCTION__));
         }
         else if (wakeup_for_search)
         {
            DBG_STDBY(("%s: Search being started for tuner_type %u", __FUNCTION__, tuner_type));

            if (!StartSearch(tuner_type))
            {
               /* Should have started a search but the search failed,
                * so will need to work out what should be started instead */
               standby_set = FALSE;
            }
         }
      }
   }

   FUNCTION_FINISH(EnterStandby);

   return(enter_standby);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   This function is called as result of call to STB_CIStartPowerDown() and
 *          confirms that for all CAMs it is now OK to power down OK
 */
static void CIPowerDownReadyCallback(void)
{
   FUNCTION_START(CIPowerDownReadyCallback);

   if (start_standby_time < STB_GCNowDHMSGmt() &&
      standby_grace_timer != INVALID_TIMER_HANDLE)
   {
      ATMR_DeleteTimer(standby_grace_timer);
      standby_grace_timer = INVALID_TIMER_HANDLE;

      DBG_STDBY(("Standby grace time with CI expired"));

      if ((BOOLEAN)APP_NvmRead(STANDBY_POWERSAVE_NVM))
      {
         /* Put box into low-power standby mode */
         DBG_STDBY(("Entering low power standby"));
         STB_HWSetStandbyState(HW_STANDBY_LOWPOWER);
      }
      else
      {
         /* Nothing is being started so go into the lowest active standby state */
         DBG_STDBY(("Entering passive standby"));
         STB_HWSetStandbyState(HW_STANDBY_PASSIVE);
      }
   }

   FUNCTION_FINISH(CIPowerDownReadyCallback);
}

#endif /*COMMON_INTERFACE*/

/**
 * @brief   Works out the next time the DVB should wakeup, taking into account any background
 *          searches and recordings if PVR is enabled.
 * @param   wakeup_date_time pointer to return the date/time the board should wakeup
 * @param   wakeup_for_search pointer to return whether the board should wakeup to
 *                            perform a background search
 * @param   search_tuner_type pointer to return the tuner type that a service search needs
 *                            to be started on
 * @param   wakeup_for_recording pointer to return whether the board should wakeup to
 *                               perform a recording
 * @param   onet_id pointer to the original network id of the service on which the next
 *                  recording is to be performed
 * @param   trans_id pointer to the transport id of the service on which the next
 *                   recording is to be performed
 * @param   service_id pointer to the service id of the service on which the next
 *                     recording is to be performed
 */
static void GetNextWakeupTime(U32DHMS *wakeup_date_time, BOOLEAN *wakeup_for_search,
   E_STB_DP_SIGNAL_TYPE *search_tuner_type, BOOLEAN *wakeup_for_recording, U16BIT *onet_id,
   U16BIT *trans_id, U16BIT *service_id)
{
   U32DHMS now_local;
   U16BIT search_date;
   U8BIT search_hours, search_mins;
   U16BIT time_mins;
   U16BIT last_service_srch;
   U16BIT background_search_start;
   U16BIT background_search_end;
   BOOLEAN service_search_enabled;
   BOOLEAN ssu_search_enabled;
   BOOLEAN eit_search_enabled;
   U16BIT last_update_srch;
   U8BIT hours, mins, secs;
   U16BIT date;
   U32BIT timer_handle;
   U32DHMS recording_wakeup;
   U32DHMS now_offset;
#ifdef COMMON_INTERFACE
   U32BIT module;
   U32BIT time_diff;
#endif

   FUNCTION_START(GetNextWakeupTime);

   /* Read from NVM when updates were last performed */
   last_service_srch = (U16BIT)APP_NvmRead(LAST_CHAN_SRCH_NVM);
   last_update_srch = (U16BIT)APP_NvmRead(OTA_LAST_UPDATE_SRCH_NVM);

   now_local = GetDateTimeNow();

   /* The number of minutes since midnight */
   time_mins = DHMS_HOUR(now_local) * 60 + DHMS_MINS(now_local);

   DBG_STDBY(("%s: today=%u time=%02u:%02u", __FUNCTION__, DHMS_DATE(now_local),
              DHMS_HOUR(now_local), DHMS_MINS(now_local)));
   DBG_STDBY(("  last service search=%u", last_service_srch));
   DBG_STDBY(("  last SSU search=%u", last_update_srch));

   /* Work out whether something should be started now or when the next operation needs to be performed */
   *wakeup_for_search = FALSE;
   *wakeup_for_recording = FALSE;

   service_search_enabled = ACFG_GetBackgroundServiceSearch();
   ssu_search_enabled = ACFG_GetBackgroundSSUSearch();

   ACFG_GetBackgroundSearchTime(&background_search_start, &background_search_end);

   DBG_STDBY(("%s: background_search_start=%d background_search_end=%d", __FUNCTION__,
              background_search_start, background_search_end));

   /* First calculate when the box should wakeup to do the next service, SSU or EIT search */
   if ((((last_service_srch == DHMS_DATE(now_local)) || !service_search_enabled) &&
        ((last_update_srch == DHMS_DATE(now_local)) || (APP_NvmRead(OTA_TYPE_NVM) != OTA_TYPE_AUTO) ||
         !ssu_search_enabled)) || (time_mins >= background_search_end))
   {
      /* Next time to search is tomorrow */
      search_date = DHMS_DATE(now_local) + 1;
   }
   else
   {
      /* Search can be done today */
      search_date = DHMS_DATE(now_local);
   }

   search_hours = background_search_start / 60;
   search_mins = background_search_start - (search_hours * 60);

   if (APP_NvmRead(STANDBY_STATE_NVM) == STDBY_POWER_ON)
   {
      /* If we're entering stand-by from POWER_ON state, reset the last EIT search time,
         as during power on the EITs have been collected */
      DBG_STDBY(("%s: resetting last EIT search to today", __FUNCTION__));
      APP_NvmSave(LAST_EIT_UPDATE_NVM, now_local, FALSE);
   }

#ifdef FREEVIEW_PLAYBACK
   eit_search_enabled = TRUE;
#else
   eit_search_enabled = FALSE;
#endif

   /* Calculate when the box should wakeup for the next alarm or recording */
   timer_handle = ATMR_GetFirstWakeupTime(&recording_wakeup, onet_id, trans_id, service_id);
   if (timer_handle != INVALID_TIMER_HANDLE)
   {
      *wakeup_for_recording = TRUE;
   }
   else
   {
      *wakeup_for_recording = FALSE;
   }

   if (*wakeup_for_recording)
   {
      DBG_STDBY(("%s: Next wakeup on service 0x%04x/0x%04x/0x%04x on %u @ %02u:%02u",
                 __FUNCTION__, *onet_id, *trans_id, *service_id, DHMS_DATE(recording_wakeup),
                 DHMS_HOUR(recording_wakeup), DHMS_MINS(recording_wakeup)));

      if (ATMR_GetType(timer_handle) == TIMER_TYPE_PVR_RECORD)
      {
         /* Convert the wakeup time to local time and subtract 15 minutes to
          * ensure any updates to the event are detected */
         recording_wakeup = CalculateWakeupTime(recording_wakeup);
      }

      /* Work out which comes first, the recording or the search */
      if ((!service_search_enabled && !ssu_search_enabled && !eit_search_enabled) ||
          (recording_wakeup <= DHMS_CREATE(search_date, search_hours, search_mins, 0)) ||
          (recording_wakeup <= now_local))
      {
         /* Need to wakeup for recording first */
         *wakeup_date_time = recording_wakeup;
      }
      else
      {
         /* Search time is earlier, but make sure it isn't too close to the recording */
         STB_GCCalculateDateTime(search_date, search_hours, search_mins, 0, 0, 30, 0,
            &date, &hours, &mins, &secs, CALC_ADD);

         if (recording_wakeup <= DHMS_CREATE(date, hours, mins, secs))
         {
            /* There may not be time to do the search, so wakeup for recording first */
            *wakeup_date_time = recording_wakeup;
         }
         else
         {
            *search_tuner_type = SIGNAL_NONE;

            if (ACFG_IsNordigCountry())
            {
               if (ACFG_IsNordigService(SIGNAL_COFDM) && ASI_CheckForServiceChange(SIGNAL_COFDM))
               {
                  DBG_STDBY(("%s: DVB-T service line-up may have changed", __FUNCTION__));
                  *search_tuner_type = SIGNAL_COFDM;
               }
               else if (ACFG_IsNordigService(SIGNAL_QPSK) && ASI_CheckForServiceChange(SIGNAL_QPSK))
               {
                  DBG_STDBY(("%s: DVB-S service line-up may have changed", __FUNCTION__));
                  *search_tuner_type = SIGNAL_QPSK;
               }
               else if (ACFG_IsNordigService(SIGNAL_QAM) && ASI_CheckForServiceChange(SIGNAL_QAM))
               {
                  DBG_STDBY(("%s: DVB-C service line-up may have changed", __FUNCTION__));
                  *search_tuner_type = SIGNAL_QAM;
               }

               if (*search_tuner_type == SIGNAL_NONE)
               {
                  /* No search needed so next wakeup is for the recording */
                  *wakeup_date_time = recording_wakeup;
               }
               else
               {
                  /* There's enough time to do the search, so set its wakeup time 2 minutes from
                   * now so it doesn't start as soon as standby is entered, allowing time for
                   * a user to exit standby */
                  now_offset = STB_GCCalculateDHMS(now_local, DHMS_CREATE(0, 0, 2, 0), CALC_ADD);
                  search_date = DHMS_DATE(now_offset);
                  search_hours = DHMS_HOUR(now_offset);;
                  search_mins = DHMS_MINS(now_offset);

                  *wakeup_date_time = DHMS_CREATE(search_date, search_hours, search_mins, 0);
                  *wakeup_for_recording = FALSE;
                  *wakeup_for_search = TRUE;
               }
            }
            else
            {
               *wakeup_date_time = recording_wakeup;
            }
         }
      }
   }
   else
   {
      *search_tuner_type = SIGNAL_NONE;

      if (service_search_enabled && ACFG_IsNordigCountry())
      {
         if (ACFG_IsNordigService(SIGNAL_COFDM) && ASI_CheckForServiceChange(SIGNAL_COFDM))
         {
            DBG_STDBY(("%s: DVB-T service line-up may have changed", __FUNCTION__));
            *search_tuner_type = SIGNAL_COFDM;
         }
         else if (ACFG_IsNordigService(SIGNAL_QPSK) && ASI_CheckForServiceChange(SIGNAL_QPSK))
         {
            DBG_STDBY(("%s: DVB-S service line-up may have changed", __FUNCTION__));
            *search_tuner_type = SIGNAL_QPSK;
         }
         else if (ACFG_IsNordigService(SIGNAL_QAM) && ASI_CheckForServiceChange(SIGNAL_QAM))
         {
            DBG_STDBY(("%s: DVB-C service line-up may have changed", __FUNCTION__));
            *search_tuner_type = SIGNAL_QAM;
         }
      }

      if (((service_search_enabled || ssu_search_enabled) && (*search_tuner_type != SIGNAL_NONE))
         || eit_search_enabled)
      {
         /* Wakeup for the next update search so set its wakeup time 2 minutes from
          * now so it doesn't start as soon as standby is entered, allowing time for
          * a user to exit standby */
         now_offset = STB_GCCalculateDHMS(now_local, DHMS_CREATE(0, 0, 2, 0), CALC_ADD);
         search_date = DHMS_DATE(now_offset);
         search_hours = DHMS_HOUR(now_offset);;
         search_mins = DHMS_MINS(now_offset);

         *wakeup_date_time = DHMS_CREATE(search_date, search_hours, search_mins, 0);
         *wakeup_for_search = TRUE;
      }
      else
      {
         /* No reason to wakeup, just down to the user */
         *wakeup_date_time = DHMS_CREATE(0, 0, 0, 0);
         *wakeup_for_search = FALSE;
      }
   }

#ifdef COMMON_INTERFACE
   /* Calculate the difference between the current time and the wakeup time in minutes */
   time_diff = (DHMS_DATE(*wakeup_date_time) - DHMS_DATE(now_local)) * 24 * 60 +
               (DHMS_HOUR(*wakeup_date_time) - DHMS_HOUR(now_local)) * 60 +
               (DHMS_MINS(*wakeup_date_time) - DHMS_MINS(now_local));

   /* Allow 30 minutes before any other search or recording needs to be started
    * before checking if an operator profile search needs to be started */
   if (!(*wakeup_for_search) || (time_diff > 30))
   {
      /* Check whether an operator search needs to be started immediately.
       * This will be true if a CAM has requested an update search and it isn't urgent
       * or the user rejected the request to run it at the time. In either case, it
       * may now be possible to run it. */
      if (ACI_IsOperatorSearchRequired() || ACI_GetFirstOperatorSearchModule(&module))
      {
         /* Set the date/time so the search starts now */
         *wakeup_date_time = now_local;
         *wakeup_for_search = TRUE;
      }
      else
      {
         /* Check if there are any profiles that need to be updated at a scheduled time */
         if (ACI_GetFirstScheduledOperatorSearch(&module, &search_date, &search_hours, &search_mins))
         {
            /* Allow 30 minutes to complete the operator search */
            STB_GCCalculateDateTime(search_date, search_hours, search_mins, 0, 0, 30, 0,
               &date, &hours, &mins, &secs, CALC_ADD);

            if ((*wakeup_date_time == 0) ||
                (DHMS_CREATE(date, hours, mins, 0) <= *wakeup_date_time))
            {
               /* The operator search can be started first */
               *wakeup_date_time = DHMS_CREATE(search_date, search_hours, search_mins, 0);
               *wakeup_for_search = TRUE;
            }
         }
      }
   }
   else
   {
      /* Check if there are any profiles that need to be updated at a scheduled time */
      if (ACI_GetFirstScheduledOperatorSearch(&module, &search_date, &search_hours, &search_mins))
      {
         /* Allow 30 minutes to complete the operator search */
         STB_GCCalculateDateTime(search_date, search_hours, search_mins, 0, 0, 30, 0,
            &date, &hours, &mins, &secs, CALC_ADD);

         if ((*wakeup_date_time == 0) ||
             (DHMS_CREATE(date, hours, mins, 0) <= *wakeup_date_time))
         {
            /* The operator search can be started first */
            *wakeup_date_time = DHMS_CREATE(search_date, search_hours, search_mins, 0);
            *wakeup_for_search = TRUE;
         }
      }
   }
#endif

   DBG_STDBY(("%s: wakeup on %u@%02u:%02u", __FUNCTION__, DHMS_DATE(*wakeup_date_time),
              DHMS_HOUR(*wakeup_date_time), DHMS_MINS(*wakeup_date_time)));

   FUNCTION_FINISH(GetNextWakeupTime);
}

/*!**************************************************************************
 * @brief   Checks whether a service or SSU search should be started, based on the
 *          current date and time of day, and starts the search if required.
 * @return  TRUE if search is started, FALSE otherwise
 ****************************************************************************/
static BOOLEAN StartSearch(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   BOOLEAN search_started;
   U32DHMS now_local;
   U16BIT time_mins;
   U16BIT last_service_srch;
   U16BIT background_search_start;
   U16BIT background_search_end;
   U16BIT last_update_srch;
   #ifdef DEBUG_STANDBY
   U32DHMS last_eit_srch;
   #endif
   #ifdef COMMON_INTERFACE
   U32BIT module;
   U16BIT search_date;
   U8BIT search_hours, search_mins;
   U8BIT gmt_hours, gmt_mins;
   #endif

   FUNCTION_START(StartSearch);

   search_started = FALSE;

   if (tuner_type != SIGNAL_NONE)
   {
      /* Check whether a service update search should be started now */
      now_local = GetDateTimeNow();

      /* Number of minutes since midnight */
      time_mins = DHMS_HOUR(now_local) * 60 + DHMS_MINS(now_local);

      ACFG_GetBackgroundSearchTime(&background_search_start, &background_search_end);

      /* Read from NVM when updates were last performed */
      last_service_srch = (U16BIT)APP_NvmRead(LAST_CHAN_SRCH_NVM);
      last_update_srch = (U16BIT)APP_NvmRead(OTA_LAST_UPDATE_SRCH_NVM);
#ifdef DEBUG_STANDBY
      last_eit_srch = (U32DHMS)APP_NvmRead(LAST_EIT_UPDATE_NVM);
#endif
      DBG_STDBY(("%s: today=%u", __FUNCTION__, DHMS_DATE(now_local)));
      DBG_STDBY(("  last service search=%u, enabled=%u", last_service_srch,
                 ACFG_GetBackgroundServiceSearch()));
      DBG_STDBY(("  last SSU search=%u, enabled=%u", last_update_srch, ACFG_GetBackgroundSSUSearch()));
      DBG_STDBY(("  last EIT update=%u@%02u:%02u, needed=%u", DHMS_DATE(last_eit_srch),
                 DHMS_HOUR(last_eit_srch), DHMS_MINS(last_eit_srch), EitSearchNeededNow()));

      if ((ACFG_IsNordigCountry() && ACFG_IsNordigService(tuner_type) && ASI_CheckForServiceChange(tuner_type)) ||
         (ACFG_GetBackgroundServiceSearch() && (last_service_srch != DHMS_DATE(now_local)) &&
          ((background_search_start <= time_mins) && (time_mins < background_search_end))))
      {
         /* Inform the app that a service search needs to be started */
         DBG_STDBY(("Sending request to start service search"));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_START_SERVICE_SEARCH,
            &tuner_type, sizeof(E_STB_DP_SIGNAL_TYPE));
         search_started = TRUE;
      }

      if (!search_started && ACFG_GetBackgroundSSUSearch() &&
          (APP_NvmRead(OTA_TYPE_NVM) == OTA_TYPE_AUTO) &&
          (last_update_srch != DHMS_DATE(now_local)) &&
          ((background_search_start <= time_mins) && (time_mins < background_search_end)))
      {
         /* An SSU search needs to be performed, so inform the app so it can start it */
         DBG_STDBY(("Sending request to start an SSU search"));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_START_SSU_SEARCH, NULL, 0);
         search_started = TRUE;
      }

      if (!search_started && EitSearchNeededNow())
      {
         /* Start EIT update search */

         /* Save the date the search is being performed. This is set even if it doesn't start
          * successfully to prevent it constantly retrying */
         APP_NvmSave(LAST_EIT_UPDATE_NVM, now_local, TRUE);

         if (ACTL_StartEitSearch(SIGNAL_NONE, ACTL_EVENT_SEARCH_PF_SCHED))
         {
            if (ASTE_InStandby())
            {
               DBG_STDBY(("Setting STANDBY_STATE_NVM=STDBY_UPDATE_SEARCH"));
               APP_NvmSave(STANDBY_STATE_NVM, STDBY_UPDATE_SEARCH, FALSE);
            }

            search_started = TRUE;
         }
      }

#ifdef COMMON_INTERFACE
      if (!search_started)
      {
         if (ACI_IsOperatorSearchRequired())
         {
            DBG_STDBY(("Starting immediate CI+ operator profile search"));

            /* An operator search needs to be started immediately */
            if (ACI_StartOperatorSearchForModule(ACI_GetOperatorSearchModule()))
            {
               search_started = TRUE;
            }
         }
         else if (ACI_GetFirstOperatorSearchModule(&module))
         {
            DBG_STDBY(("Starting CI+ operator profile search for module %u", module));

            /* A module has requested an operator search but it
             * hasn't been run yet, so start it now */
            if (ACI_StartOperatorSearchForModule(module))
            {
               search_started = TRUE;
            }
         }
         else
         {
            /* Check whether it's time to start a scheduled operator search */
            if (ACI_GetFirstScheduledOperatorSearch(&module, &search_date, &search_hours, &search_mins))
            {
               gmt_hours = time_mins / 60;
               gmt_mins = time_mins % 60;

               if (STB_GCCompareDateTime(DHMS_DATE(now_local), gmt_hours, gmt_mins, 0,
                      search_date, search_hours, search_mins, 0, COMP_1GE2))
               {
                  DBG_STDBY(("Starting scheduled CI+ operator profile search for module %u", module));

                  /* The operator search for this module needs to be started */
                  if (ACI_StartOperatorSearchForModule(module))
                  {
                     search_started = TRUE;
                  }
               }
            }
         }
      }
#endif
   }

   FUNCTION_FINISH(StartSearch);

   return(search_started);
}

/**
 * @brief   Gets the current local date & time
 * @return  current local time
 */
static U32DHMS GetDateTimeNow(void)
{
   U32DHMS local_time;

   FUNCTION_START(GetDateTimeNow);

   local_time = STB_GCNowDHMSLocal();

   FUNCTION_FINISH(GetDateTimeNow);

   return(local_time);
}

/**
 * @brief   Calculates the date & time when the box should wakeup from standby by converting
 *          the given date/time to local time and subtracting 15 minutes
 * @param   date_time - date to be converted
 * @return  date/time to wake up
 */
static U32DHMS CalculateWakeupTime(U32DHMS date_time)
{
   U32DHMS wakeup;

   FUNCTION_START(CalculateWakeupTime);

   wakeup = STB_GCConvertDHMS(date_time, CONV_LOCAL);
   wakeup = STB_GCCalculateDHMS(wakeup, DHMS_CREATE(0, 0, 15, 0), CALC_SUB);

   FUNCTION_FINISH(CalculateWakeupTime);

   return(wakeup);
}

static void FinishEitSearch(U8BIT path)
{
   EIT_DBG(("FinishEitSearch(%u)", path));

   if (eit_search_end_timer != INVALID_TIMER_HANDLE)
   {
      ATMR_DeleteTimer(eit_search_end_timer);
      eit_search_end_timer = INVALID_TIMER_HANDLE;
   }

   FinishSearch(path);

   ADB_ReleaseTransportList(search_transport_list, search_num_transports);
   search_transport_list = NULL;
   search_num_transports = 0;
}

static void UpdateTransportParameters(U8BIT path)
{
   U8BIT tuner;
   ADB_TRANSPORT_REC *t_ptr;
   E_STB_DP_TMODE tmode;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TTYPE terr_type;
   E_STB_DP_CMODE cmode;
   U32BIT srate;

   FUNCTION_START(UpdateTransportParameters);

   tuner = STB_DPGetPathTuner(path);
   if (tuner != INVALID_RES_ID)
   {
      if ((t_ptr = current_transport[path]) == NULL)
      {
         t_ptr = ADB_GetTunedTransport(path);
      }

      if (t_ptr != NULL)
      {
         DBDEF_RequestAccess();

         switch (t_ptr->sig_type)
         {
            case SIGNAL_COFDM:
            {
               t_ptr->u.terr.freq_offset = STB_TuneGetActualTerrFreqOffset(tuner);
               t_ptr->u.terr.constellation = STB_TuneGetActualTerrConstellation(tuner);
               t_ptr->u.terr.hierarchy = STB_TuneGetActualTerrHierarchy(tuner);
               t_ptr->u.terr.lp_code_rate = STB_TuneGetActualTerrLpCodeRate(tuner);
               t_ptr->u.terr.hp_code_rate = STB_TuneGetActualTerrHpCodeRate(tuner);
               t_ptr->u.terr.guard_int = STB_TuneGetActualTerrGuardInt(tuner);
               switch (STB_TuneGetActualTerrMode(tuner))
               {
                  case TUNE_MODE_COFDM_1K:   {tmode = MODE_COFDM_1K; break; }
                  case TUNE_MODE_COFDM_2K:   {tmode = MODE_COFDM_2K; break; }
                  case TUNE_MODE_COFDM_4K:   {tmode = MODE_COFDM_4K; break; }
                  case TUNE_MODE_COFDM_8K:   {tmode = MODE_COFDM_8K; break; }
                  case TUNE_MODE_COFDM_16K:  {tmode = MODE_COFDM_16K; break; }
                  case TUNE_MODE_COFDM_32K:  {tmode = MODE_COFDM_32K; break; }
                  default:                   {tmode = MODE_COFDM_UNDEFINED; break; }
               }
               if (t_ptr->u.terr.tmode != tmode)
               {
                  t_ptr->u.terr.tmode = tmode;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_MODE, tmode);
               }

               switch (STB_TuneGetActualTerrBwidth(tuner))
               {
                  case TUNE_TBWIDTH_8MHZ:   {bwidth = TBWIDTH_8MHZ; break; }
                  case TUNE_TBWIDTH_7MHZ:   {bwidth = TBWIDTH_7MHZ; break; }
                  case TUNE_TBWIDTH_6MHZ:   {bwidth = TBWIDTH_6MHZ; break; }
                  default:                  {bwidth = 0; break; }
               }
               if (t_ptr->u.terr.bwidth != bwidth)
               {
                  t_ptr->u.terr.bwidth = bwidth;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_BWIDTH, bwidth);
               }

               switch (STB_TuneGetSystemType(tuner))
               {
                  case TUNE_SYSTEM_TYPE_DVBT2: {terr_type = TERR_TYPE_DVBT2; break; }
                  case TUNE_SYSTEM_TYPE_DVBT:  {terr_type = TERR_TYPE_DVBT;  break; }
                  default:                   {terr_type = TERR_TYPE_UNKNOWN; break; }
               }

               if (t_ptr->u.terr.terr_type != terr_type)
               {
                  t_ptr->u.terr.terr_type = terr_type;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_TERR_TYPE, terr_type);
               }
               break;
            }
            case SIGNAL_QAM:
            {
               /* Symbol rate is stored in Ksym/sec */
               if ((srate = STB_TuneGetActualSymbolRate(tuner)) != SYMBOL_RATE_AUTO)
               {
                  srate = srate / 1000;

                  if (t_ptr->u.cab.symbol_rate != srate)
                  {
                     t_ptr->u.cab.symbol_rate = srate;
                     DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, srate);
                  }
               }

               switch (STB_TuneGetActualCableMode(tuner))
               {
                  case TUNE_MODE_QAM_4:      {cmode = MODE_QAM_4; break; }
                  case TUNE_MODE_QAM_8:      {cmode = MODE_QAM_8; break; }
                  case TUNE_MODE_QAM_16:     {cmode = MODE_QAM_16; break; }
                  case TUNE_MODE_QAM_32:     {cmode = MODE_QAM_32; break; }
                  case TUNE_MODE_QAM_64:     {cmode = MODE_QAM_64; break; }
                  case TUNE_MODE_QAM_128:    {cmode = MODE_QAM_128; break; }
                  case TUNE_MODE_QAM_256:    {cmode = MODE_QAM_256; break; }
                  default:                   {cmode = MODE_QAM_AUTO; break; }
               }
               if (t_ptr->u.cab.cmode != cmode)
               {
                  t_ptr->u.cab.cmode = cmode;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_CTRAN_MODE, cmode);
               }
               break;
            }
            case SIGNAL_QPSK:
            {
               /* Symbol rate is stored in Ksym/sec */
               if ((srate = STB_TuneGetActualSymbolRate(tuner)) != SYMBOL_RATE_AUTO)
               {
                  srate = srate / 1000;

                  if (t_ptr->u.sat.symbol_rate != srate)
                  {
                     t_ptr->u.sat.symbol_rate = (U16BIT)srate;
                     DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, srate);
                  }
               }
               break;
            }
            default:
            {
               break;
            }
         }

         DBDEF_ReleaseAccess();

         ADB_SaveDatabase();
      }
   }

   FUNCTION_FINISH(UpdateTransportParameters);
}

static U32BIT CreateTimer(U32DHMS date_time)
{
   U32BIT handle;
   S_TIMER_INFO timer_info;

   FUNCTION_START(CreateTimer);

   /* Initialise the timer info, set the date/time the timer is to trigger and create the timer */
   ATMR_InitialiseTimer(&timer_info, TIMER_TYPE_PRIVATE, NULL, NULL);

   timer_info.start_time = date_time;
   timer_info.ram_only = TRUE;

   handle = ATMR_AddTimer(&timer_info);

   FUNCTION_FINISH(CreateTimer);

   return(handle);
}

static void nitTableUpdate(U8BIT path, SI_NIT_TABLE *nit_table, SI_TABLE_RECORD *table_rec)
{
    if (NULL == nit_table)
        return;

    E_STB_DP_SIGNAL_TYPE signal_type;
    signal_type = STB_DPGetSignalType(path);

    if (SIGNAL_QAM == signal_type) {
        if (SEARCH_TYPE_MANUAL_FREQ == current_search_type) {
            BOOLEAN ignore_nit_transports;            
            ignore_nit_transports = ASI_GetAppSiModeFlag(path, ASI_MODE_FLAG_IGNORE_NIT_TRANSPORT);

            //if (TRUE == ignore_nit_transports) {
                U32BIT freqBegin = user_tuning_params[path].freq;
                U32BIT freqEnd = user_tuning_params[path].end_freq;
                E_STB_DP_CMODE mode = user_tuning_params[path].u.cab.mode;
                U16BIT symbol_rate = user_tuning_params[path].u.cab.symbol_rate;
                U32BIT bwidth;
                U32BIT freq;

                switch (user_tuning_params[path].u.cab.bwidth) {
                   case TBWIDTH_8MHZ:
                        bwidth = 8000000;
                        break;
                   case TBWIDTH_7MHZ:
                        bwidth = 7000000;
                        break;
                   case TBWIDTH_6MHZ:
                   default:
                        bwidth = 6000000;
                        break;
                }

                DBDEF_RequestAccess();

                ADB_NETWORK_REC * n_ptr;
                n_ptr = DBDEF_FindNetworkRec(nit_table->net_id, NULL);
                if (NULL != n_ptr) {
                    ADB_TRANSPORT_REC * t_ptr = (ADB_TRANSPORT_REC *)current_transport[path];
                    if (NULL != t_ptr)
                        t_ptr->searched = TRUE;

                    freq = freqBegin;
                    while (freq <= freqEnd) {
                        // generate transport records
                        t_ptr = DBDEF_FindCableTransportRec(freq, SYMBOL_RATE_AUTO);
                        if (NULL == t_ptr) {
                            t_ptr = DBDEF_AddCableTransportRec(freq, symbol_rate, n_ptr);
                            if (NULL != t_ptr)
                                DBDEF_SetCableTransportMode(t_ptr, mode);
                        }
                        else {
                            if (t_ptr == current_transport[path]) {
                                t_ptr->searched = FALSE;
                            }    
                        }
                        freq += bwidth;
                    }
                }

                DBDEF_ReleaseAccess();
            //}  // if (TRUE == ignore_nit_transports)
            ASI_SetUpdateNitFunction(NULL);
        }  // if (SEARCH_TYPE_MANUAL_FREQ == current_search_type)
    }  // if (SIGNAL_QAM == signal_type)
}

static void untTableUpdate(U8BIT path, SI_UNT_TABLE *unt_table, SI_TABLE_RECORD *table_rec)
{
    S_UNT_SETTINGS settings;
    SI_SSU_LOCATION_DESC *ssu_location_desc;
    SI_DELIVERY_SYS_DESC_TYPE signal_type;
    SI_DELIVERY_SYS_DESC *del_sys_desc;
    ADB_TRANSPORT_REC * t_ptr = NULL;

    if (NULL == unt_table)
        goto end;

    if (unt_table->version == unt_settings.version) {
        goto same;
    }

    // new UNT
    unt_settings.version = unt_table->version;
    unt_settings.valid = FALSE;
    // SSU location descriptor
    ssu_location_desc = unt_table->ssu_location_desc;
    if (NULL == ssu_location_desc)
        goto end;

    if (0x000A != ssu_location_desc->data_broadcast_id)
        goto end;

    if (NULL == ssu_location_desc->private_data_byte)
        goto end;

    // delivery system descriptor
    del_sys_desc = unt_table->del_sys_desc;
    if (NULL == del_sys_desc)
        goto end;

    signal_type = unt_table->del_sys_desc_type;
    if (SI_DEL_SYS_DESC_TYPE_CABLE == signal_type) {
        U32BIT freq = del_sys_desc->cable.freq_hz;
        E_STB_DP_CMODE mode = del_sys_desc->cable.modulation;
        U16BIT symbol_rate = del_sys_desc->cable.symbol_rate;
        DBDEF_RequestAccess();
        // generate transport records
        t_ptr = DBDEF_FindCableTransportRec(freq, SYMBOL_RATE_AUTO);
        if (NULL == t_ptr) {
            t_ptr = DBDEF_AddCableTransportRec(freq, symbol_rate, NULL);
            if (NULL != t_ptr)
                DBDEF_SetCableTransportMode(t_ptr, mode);
        }
        DBDEF_ReleaseAccess();
    }  // if (SI_DEL_SYS_DESC_TYPE_CABLE == signal_type)

    if (NULL == t_ptr)
        goto end;

    unt_settings.action_type = unt_table->action_type;
    unt_settings.processing_order = unt_table->processing_order;
    //
    memcpy (unt_settings.private_data_byte, ssu_location_desc->private_data_byte, ssu_location_desc->private_data_length);
    unt_settings.private_data_byte[ssu_location_desc->private_data_length] = 0;
    unt_settings.ssu_data_pid = ssu_location_desc->association_tag;
    //
    unt_settings.transport = t_ptr;

    snprintf (
        (char *)unt_settings.jsonString,
        512,
        "{\"onid\":%hu, \"tsid\":%hu, \"sid\":%hu, \"data_pid\":%hu, \"info\":%s}",
        t_ptr->orig_net_id, t_ptr->tran_id, 0,
        unt_settings.ssu_data_pid, unt_settings.private_data_byte
    );
    unt_settings.valid = TRUE;
            
same:            
    STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SSU_UPDATE_DETECTED,
          unt_settings.private_data_byte, (ssu_location_desc->private_data_length + 1));

end:
    return;
}

//**************************************************************************************************
// End of File
//**************************************************************************************************
