/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application database access functions
 * @file    ap_dbacc.c
 * @date    24/03/2003
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <string.h>
#include <stdio.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"
#include "stbheap.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbuni.h"
#include "stbgc.h"
#include "stberc.h"

#include "stbllist.h"

#include "app.h"
#include "app_nvm.h"
#include "ap_cfg.h"
#include "ap_cfdat.h"

#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"
#include "ap_cntrl.h"

#include "dba.h"

//---constant definitions for this file-------------------------------------------------------------
#define CI_PROTECTION_DESC_TIME_LIMIT     (7 * 24 * 60 * 60)   /* 7 days in seconds */

#define ADB_FAV_SERVICE_LIST_TYPE   (ADB_SERVICE_LIST_FAV_GROUP_A | ADB_SERVICE_LIST_FAV_GROUP_B | \
                                     ADB_SERVICE_LIST_FAV_GROUP_C | ADB_SERVICE_LIST_FAV_GROUP_D)

#define FULL_SI_LINKAGE_TYPE               0x04
#define EPG_SERVICE_LINKAGE_TYPE           0x02
#define NORDIG_SIMULCAST_TYPE              0x82

//#define DEBUG_DBACC

#ifdef DEBUG_DBACC
#define DBG_DBACC(X)    STB_SPDebugWrite X
#else
#define DBG_DBACC(X)
#endif

//---local typedefs, structs, enumerations for this file--------------------------------------------
typedef struct s_component
{
   U8BIT tag;
   U8BIT type;    /* component type as defined in the EITp's component descriptor */
   U8BIT content; /* stream content as defined in the EITp's component descriptor */
} S_COMPONENT;

typedef struct s_stream
{
   U32BIT lang_code;
   ADB_AUDIO_TYPE audio_type;
   ADB_SUBTITLE_TYPE subt_type;
   ADB_STREAM_TYPE stream_type;
   E_STB_DP_AUDIO_MODE audio_mode;
   U8BIT ttext_type;
   U8BIT ttext_magazine;
   U8BIT ttext_page;
   U16BIT composition_page;
   U16BIT ancillary_page;
   S_COMPONENT *tag_array_ptr;
   U8BIT num_tag_entries;
   U16BIT pid;
   BOOLEAN in_use;
   BOOLEAN has_ca_descriptor;
   U8BIT component_type; /* component_type as defined in the EITp's component descriptor */
} S_STREAM;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)


//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)

static BOOLEAN EventAtTime(ADB_EVENT_REC *event_ptr, U32DHMS time);
static U8BIT* CopyString(ADB_STRING *str_desc, BOOLEAN to_unicode);
static U8BIT* CopySIString(SI_STRING_DESC *str_desc);
static U8BIT* ExtractShortName(ADB_STRING *str_desc);

//transports
static U8BIT* GetTransportName(ADB_TRANSPORT_REC *t_ptr);
static U16BIT GetTransportTid(ADB_TRANSPORT_REC *t_ptr);
static U8BIT GetTransportTunedStrength(ADB_TRANSPORT_REC *t_ptr);
static U8BIT GetTransportTunedQuality(ADB_TRANSPORT_REC *t_ptr);
static U16BIT GetTransportOriginalNetworkId(ADB_TRANSPORT_REC *t_ptr);
static U32BIT GetTransportFreq(ADB_TRANSPORT_REC *t_ptr);

static U8BIT GetTransportBwidth(ADB_TRANSPORT_REC *t_ptr);
static S8BIT GetTransportOffset(ADB_TRANSPORT_REC *t_ptr);
static U8BIT* GetTransportConstellationString(ADB_TRANSPORT_REC *t_ptr);
static U8BIT* GetTransportHierarchyString(ADB_TRANSPORT_REC *t_ptr);

//services
static BOOLEAN CheckServiceInListType(ADB_SERVICE_REC *s_ptr, U32BIT list_type, BOOLEAN inc_hidden,
   BOOLEAN ignore_selectable);

static void GetServiceList(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr,
   BOOLEAN show_hidden, BOOLEAN ignore_selectable);

static U8BIT* GetServiceNameByLangAndPrefId(ADB_SERVICE_REC *s_ptr, U8BIT lang, U8BIT pref_name_id, BOOLEAN short_name);

static U16BIT GetServiceLcn(ADB_SERVICE_REC *s_ptr);
static U16BIT GetServiceId(ADB_SERVICE_REC *s_ptr);

static BOOLEAN GetServiceUnavailFlag(void *s_ptr);
static BOOLEAN GetServiceNewFlag(void *s_ptr);

// network
static U8BIT* GetNetworkName(ADB_NETWORK_REC *n_ptr, BOOLEAN short_name);
static U8BIT* GetNetworkNameByLang(ADB_NETWORK_REC *n_ptr, U8BIT lang, BOOLEAN short_name);
static U8BIT* GetDefaultNetworkName(ADB_NETWORK_REC *n_ptr, BOOLEAN short_name);
static U16BIT GetNetworkId(ADB_NETWORK_REC *n_ptr);

//events
static S_EVENT* CopyEventRec(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr);
static BOOLEAN FindDoNotScramble(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr);

static U8BIT* GetEventCridType(void *serv_ptr, void *event_ptr, U8BIT crid_type, U8BIT index);
static U8BIT NumberOfCridsOfType(S_EVENT *event_ptr, U8BIT crid_type);

static BOOLEAN IsStreamOfType(ADB_STREAM_REC *stream_ptr, ADB_STREAM_LIST_TYPE stream_list_type);
static BOOLEAN HasNordigSimulcastService(void *serv_ptr);

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Initialises database access
 */
void ADB_Initialise(void)
{
   FUNCTION_START(ADB_Initialise);

   DBDEF_Initialise();

   DBDEF_RequestAccess();
   DBDEF_LoadDatabase(NULL);
   DBDEF_SortServicesByLcn();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_Initialise);
}

/**
 * @brief   Sets up the database for a search
 * @param   tuner_type - type of tuner that will be used for the search
 * @param   satellite - satellite on which search is being performed, NULL if not relevant
 * @param   retune - TRUE if a retune is to be performed, FALSE otherwise
 * @param   manual_search - TRUE for manual search, FALSE otherwise
 */
void ADB_PrepareDatabaseForSearch(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN retune, BOOLEAN manual_search)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;
   ADB_SATELLITE_REC *sat_ptr;
   ADB_BAT_VERSION_REC *ver_rec;

   FUNCTION_START(ADB_PrepareDatabaseForSearch);

   DBDEF_RequestAccess();

   /* Backup the database in preparation for the search so it can be
    * restored if the search is cancelled */
   DBA_LockDatabase();
   DBA_BackupDatabase(NULL);
   DBA_UnlockDatabase();

   if (retune)
   {
      DBDEF_DeleteRecordsForTunerType(tuner_type, satellite);
   }
   else
   {
      /* Prepare services for the search */
      s_ptr = DBDEF_GetNextServiceRec(NULL);
      while (s_ptr != NULL)
      {
         if (DBDEF_ServiceForTunerType(s_ptr, tuner_type, satellite))
         {
            /* Clear new flag */
            s_ptr->new_service = FALSE;

            if (!manual_search)
            {
               /* Set the service as unavailable, which will be updated
                * again during the search if the service is found */
               s_ptr->unavailable = TRUE;
            }
         }

         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }

      /* Prepare transports for the search */
      t_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t_ptr != NULL)
      {
         if (DBDEF_TransportForTunerType(t_ptr, tuner_type, satellite))
         {
            /* Clear version history on this transport */
            DBDEF_ClearTableVersionHistory(t_ptr);

            /* Set the flag to show the transport hasn't been searched yet */
            t_ptr->searched = FALSE;

            if (!manual_search)
            {
               /* Set the flag to show the transport hasn't been found yet */
               t_ptr->available = FALSE;
            }
         }

         t_ptr = DBDEF_GetNextTransportRec(t_ptr);
      }
   }

   if (satellite != NULL)
   {
      /* Clear the BAT versions to prevent them being ignored when the scan is performed */
      sat_ptr = (ADB_SATELLITE_REC *)satellite;
      ver_rec = sat_ptr->bat_version_list;
      while (ver_rec != NULL)
      {
         sat_ptr->bat_version_list = ver_rec->next;
         STB_AppFreeMemory(ver_rec);
         ver_rec = sat_ptr->bat_version_list;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_PrepareDatabaseForSearch);
}

/**
 * @brief   Completes the database setup after a search
 * @param   save_changes - TRUE if the updated database is to be kept, FALSE otherwise
 * @param   tuner_type - type of tuner that was used for the search
 * @param   satellite - satellite on which search was performed, NULL if not relevant
 * @param   search_completed TRUE if the search was completed, FALSE if terminated before completion
 * @param   clear_new_flags TRUE if all new flags are to be cleared (e.g. after a full search rather
 *          than an update search)
 * @param   manual_search TRUE for manual searches, FALSE otherwise
 */
void ADB_FinaliseDatabaseAfterSearch(BOOLEAN save_changes, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN search_completed, BOOLEAN clear_new_flags, BOOLEAN manual_search)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_FinaliseDatabaseAfterSearch);

   if (save_changes)
   {
      DBDEF_RequestAccess();

      if (clear_new_flags)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s_ptr != NULL)
         {
            s_ptr->new_service = FALSE;

            if (s_ptr->unavailable)
            {
               /* If the service is unavailable but it's parent transport wasn't searched during
                * the scan, then just set it back to being available because we don't know
                * whether it's available or not */
               if ((t_ptr = s_ptr->transport) != NULL)
               {
                  if (!t_ptr->searched)
                  {
                     s_ptr->unavailable = FALSE;
                  }
               }
               else
               {
                  /* No transport! */
                  s_ptr->unavailable = FALSE;
               }
            }

            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }
      }

      DBDEF_AllocateLcns(tuner_type, TRUE);
      DBDEF_SortServicesByLcn();
      DBDEF_TidyDatabaseAfterSearch(tuner_type, satellite, search_completed, manual_search);

      DBDEF_ReleaseAccess();

      ADB_SaveDatabase();
   }
   else
   {
      /* Restore previous database state */
      DBDEF_RequestAccess();
      DBA_LockDatabase();

      if (DBA_CanRestoreDatabase())
      {
         DBDEF_DeleteAllRecords();

         DBA_RestoreDatabase();
         DBA_SaveDatabase();
         DBDEF_LoadDatabase(NULL);
         DBDEF_SortServicesByLcn();
      }

      DBA_UnlockDatabase();
      DBDEF_ReleaseAccess();
   }

   DBA_LockDatabase();
   DBA_EraseBackupDatabase();
   DBA_UnlockDatabase();

   ADB_ReleaseDatabaseSearchData();

   FUNCTION_FINISH(ADB_FinaliseDatabaseAfterSearch);
}

/**
 * @brief   Checks through all services after a service search and determines whether any of them
 *          require the same LCN. This function assumes that ADB_FinaliseDatabaseAfterSearch has
 *          already been called.
 * @return  TRUE if duplicates are found, FALSE otherwise
 */
BOOLEAN ADB_AreLcnsDuplicated(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   BOOLEAN result;

   FUNCTION_START(ADB_AreLcnsDuplicated);

   DBDEF_RequestAccess();

   /* Allocate LCNs to each service to determine if there are any duplicates,
    * but don't store the result */
   result = DBDEF_AllocateLcns(tuner_type, FALSE);

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AreLcnsDuplicated);

   return(result);
}

/**
 * @brief   Allocates LCNs to services after a service search. This function assumes that
 *          ADB_FinaliseDatabaseAfterSearch has already been called.
 */
void ADB_AllocateLcns(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   FUNCTION_START(ADB_AllocateLcns);

   DBDEF_RequestAccess();

   DBDEF_AllocateLcns(tuner_type, TRUE);
   DBDEF_SortServicesByLcn();

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AllocateLcns);
}

/**
 * @brief   Frees all data that's only required for service search. This should be called after
 *          the search is complete and LCNs have been assigned, etc.
 */
void ADB_ReleaseDatabaseSearchData(void)
{
   ADB_NETWORK_REC *nptr;
   ADB_TRANSPORT_REC *tptr;
   ADB_SERVICE_REC *sptr;

   FUNCTION_START(ADB_ReleaseDatabaseSearchData);

   DBDEF_RequestAccess();

   nptr = DBDEF_GetNextNetworkRec(NULL);
   while (nptr != NULL)
   {
      if (nptr->target_region_name_list != NULL)
      {
         STB_SIReleaseTargetRegionNameList(nptr->target_region_name_list);
         nptr->target_region_name_list = NULL;
      }

      if (nptr->target_region_list != NULL)
      {
         STB_SIReleaseTargetRegionList(nptr->target_region_list);
         nptr->target_region_list = NULL;
      }

      nptr = DBDEF_GetNextNetworkRec(nptr);
   }

   tptr = DBDEF_GetNextTransportRec(NULL);
   while (tptr != NULL)
   {
      if ((tptr->sig_type == SIGNAL_COFDM) && (tptr->u.terr.target_region_list != NULL))
      {
         STB_SIReleaseTargetRegionList(tptr->u.terr.target_region_list);
         tptr->u.terr.target_region_list = NULL;
      }

      tptr = DBDEF_GetNextTransportRec(tptr);
   }

   sptr = DBDEF_GetNextServiceRec(NULL);
   while (sptr != NULL)
   {
      if (sptr->target_region_list != NULL)
      {
         STB_SIReleaseTargetRegionList(sptr->target_region_list);
         sptr->target_region_list = NULL;
      }

      if (sptr->hd_lcn_desc != NULL)
      {
         STB_FreeMemory(sptr->hd_lcn_desc);
         sptr->hd_lcn_desc = NULL;
      }

      sptr = DBDEF_GetNextServiceRec(sptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_ReleaseDatabaseSearchData);
}

/**
 * @brief   Clears the service database and resets everything to default values
 */
void ADB_ResetDatabase(void)
{
   U8BIT path;

   FUNCTION_START(ADB_ResetDatabase);

   // Reset of database means that must stop tuning/decoding,
   // otherwise ap_si.c will try to access deleted records
   path = STB_DPGetLivePath();
   if (path != INVALID_RES_ID)
   {
      ACTL_TuneOff(path);
   }

   DBDEF_RequestAccess();
   DBA_LockDatabase();
   DBDEF_DeleteAllRecords();
   DBA_ClearDatabase();
   DBA_UnlockDatabase();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_ResetDatabase);
}

/**
 * @brief   Saves the database to non-volatile memory
 */
void ADB_SaveDatabase(void)
{
   FUNCTION_START(ADB_SaveDatabase);

   DBDEF_RequestAccess();
   DBA_LockDatabase();
   DBA_SaveDatabase();
   DBA_UnlockDatabase();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SaveDatabase);
}

/**
 * @brief   Saves the event schedule for all services for the given type of tuner
 *          to the database. Note: the database has to support this operation.
 * @param   tuner_type tuner type that the services must be on, SIGNAL_NONE will save the
 *                     schedule data for all services
 * @param   satellite if tuner_type is SIGNAL_QPSK then only events for services on
 *                    the given satellite will be saved. If NULL, events for services
 *                    on all satellites will be saved.
 */
void ADB_SaveEventSchedule(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_SaveEventSchedule);

   DBDEF_RequestAccess();

   s_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s_ptr != NULL)
   {
      if ((tuner_type == SIGNAL_NONE) || DBDEF_ServiceForTunerType(s_ptr, tuner_type, satellite))
      {
         DBDEF_SaveServiceEventSchedule(s_ptr);
      }
      s_ptr = DBDEF_GetNextServiceRec(s_ptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SaveEventSchedule);
}

/**
 * @brief   Deletes all networks, transports and services related to the given type of tuner
 * @param   tuner_type tuner type
 * @param   satellite if tuner_type is SIGNAL_QPSK then only services on the given satellite
 *          will be deleted. If NULL, services on all satellite will be deleted
 */
void ADB_DeleteServices(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   FUNCTION_START(ADB_DeleteServices);

   DBDEF_RequestAccess();
   DBDEF_DeleteRecordsForTunerType(tuner_type, satellite);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_DeleteServices);
}

#ifdef DEBUG_PRINT_DATABASE
/**
 * @brief   Outputs the contents of the service database to the console in human readable form
 */
void ADB_PrintDatabase(void)
{
   FUNCTION_START(ADB_PrintDatabase);

   DBDEF_RequestAccess();
   DBDEF_PrintAllRecords();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_PrintDatabase);
}

#endif

/**
 * @brief   Frees the memory used by any of the name lists (e.g. networks, services, etc)
 * @param   name_list name list to be freed
 * @param   num_names number of items in the name list
 */
void ADB_ReleaseNameList(U8BIT **name_list, U16BIT num_names)
{
   U16BIT i;

   FUNCTION_START(ADB_ReleaseNameList);

   if (name_list != NULL)
   {
      for (i = 0; i < num_names; i++)
      {
         STB_ReleaseUnicodeString(name_list[i]);
      }

      STB_AppFreeMemory(name_list);
   }

   FUNCTION_FINISH(ADB_ReleaseNameList);
}

/**
 * @brief   Returns the number of LNBs in the database
 * @return  Number of LNBs
 */
U16BIT ADB_GetNumLNBs(void)
{
   U16BIT num;

   FUNCTION_START(ADB_GetNumLNBs);

   DBDEF_RequestAccess();
   num = DBDEF_GetNumLNBs();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumLNBs);

   return(num);
}

/**
 * @brief   Creates a new LNB in the database with the given settings
 * @param   settings settings to be given to the new LNB
 * @return  pointer to the new LNB, NULL if creation fails
 */
void* ADB_AddLNB(ADB_LNB_SETTINGS *settings)
{
   ADB_LNB_REC *lnb;
   ADB_STRING *name;
   U8BIT *name_str;

   FUNCTION_START(ADB_AddLNB);

   DBDEF_RequestAccess();

   name_str = settings->name;
   name = DBDEF_MakeString(0, name_str, STB_GetNumBytesInString(name_str));
   if ((lnb = DBDEF_AddLNB(settings->type, name)) != NULL)
   {
      DBDEF_SetLNBPower(lnb, settings->power);
      DBDEF_SetLNBDiSEqCTone(lnb, settings->diseqc_tone);
      DBDEF_SetLNBCSwitch(lnb, settings->c_switch);
      DBDEF_SetLNB22k(lnb, settings->is_22k);
      DBDEF_SetLNB12V(lnb, settings->is_12v);
      DBDEF_SetLNBPulsePosition(lnb, settings->is_pulse_posn);
      DBDEF_SetLNBDiSEqCPosition(lnb, settings->is_diseqc_posn);
      DBDEF_SetLNBSmatv(lnb, settings->is_smatv);
      DBDEF_SetLNBRepeats(lnb, settings->diseqc_repeats);
      DBDEF_SetLNBUSwitch(lnb, settings->u_switch);
      DBDEF_SetLNBUnicable(lnb, settings->unicable_if, settings->unicable_chan);
   }
   else if (name != NULL)
   {
      DBDEF_ReleaseString(name);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddLNB);

   return(lnb);
}

/**
 * @brief   Returns the next LNB from the database
 * @param   lnb_ptr current LNB pointer, NULL to get the first LNB
 * @return  pointer to LNB
 */
void* ADB_GetNextLNB(void *lnb_ptr)
{
   void *next_lnb;

   FUNCTION_START(ADB_GetNextLNB);

   DBDEF_RequestAccess();
   next_lnb = DBDEF_GetNextLNBRec(lnb_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNextLNB);

   return(next_lnb);
}

/**
 * @brief   Returns the current settings for the given LNB
 * @param   lnb_ptr LNB
 * @param   settings structure in which the values are returned
 * @return  TRUE if the LNB is valid and values are returned, FALSE otherwise
 */
BOOLEAN ADB_GetLNBSettings(void *lnb_ptr, ADB_LNB_SETTINGS *settings)
{
   BOOLEAN retval;
   ADB_LNB_REC *lnb;

   FUNCTION_START(ADB_GetLNBSettings);

   if (lnb_ptr != NULL)
   {
      DBDEF_RequestAccess();

      lnb = (ADB_LNB_REC *)lnb_ptr;

      settings->type = lnb->type;
      settings->power = lnb->power;
      settings->is_12v = lnb->is_12v;
      settings->is_22k = lnb->is_22k;
      settings->is_pulse_posn = lnb->is_pulse_posn;
      settings->is_diseqc_posn = lnb->is_diseqc_posn;
      settings->diseqc_tone = lnb->diseqc_tone;
      settings->c_switch = lnb->c_switch;
      settings->u_switch = lnb->u_switch;
      settings->is_smatv = lnb->is_smatv;
      settings->diseqc_repeats = lnb->diseqc_repeats;
      settings->unicable_if = lnb->unicable_if;
      settings->unicable_chan = lnb->unicable_chan;
      settings->name = lnb->name->str_ptr;

      DBDEF_ReleaseAccess();

      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ADB_GetLNBSettings);

   return(retval);
}

/**
 * @brief   Returns the number of LNB bands in the database
 * @return  Number of LNB bands
 */
U16BIT ADB_GetNumLNBBands(void)
{
   U16BIT num;

   FUNCTION_START(ADB_GetNumLNBBands);

   DBDEF_RequestAccess();
   num = DBDEF_GetNumLNBBands();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumLNBBands);

   return num;
}

/**
 * @brief   Creates a new LNB band in the database with the given settings
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @param   lnb_ptr pointer to an LNB already present in the database
 * @return  pointer to new LNB band
 */
void* ADB_AddLNBBand(S_STB_DP_LNB_BAND *band_parameters, void *lnb_ptr)
{
   ADB_LNB_BAND_REC *band_rec;

   FUNCTION_START(ADB_AddLNBBand);

   DBDEF_RequestAccess();

   band_rec = DBDEF_AddLNBBandRec(band_parameters, lnb_ptr);

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddLNBBand);

   return band_rec;
}

/**
 * @brief   Returns the next LNB band from the database
 * @param   band_ptr current LNB band pointer, NULL to get the first LNB band
 * @return  pointer to LNB band
 */
void* ADB_GetNextLNBBand(void *band_ptr)
{
   void *next_band;

   FUNCTION_START(ADB_GetNextLNBBand);

   DBDEF_RequestAccess();
   next_band = DBDEF_GetNextLNBBandRec(band_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNextLNBBand);

   return(next_band);
}

/**
 * @brief   Returns the LNB associated with the band
 * @param   band_ptr LNB band
 * @return  pointer to LNB
 */
void* ADB_GetLNBBandLNB(void *band_ptr)
{
   void *lnb_ptr;

   FUNCTION_START(ADB_GetLNBBandLNB);

   DBDEF_RequestAccess();

   if (band_ptr != NULL)
   {
      lnb_ptr = ((ADB_LNB_BAND_REC *)band_ptr)->lnb;
   }
   else
   {
      lnb_ptr = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetLNBBandLNB);

   return(lnb_ptr);
}

/**
 * @brief   Returns the structure describing the band and the expected LNB behaviour in it.
 * @param   band_ptr LNB band
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @return  TRUE if the band has been found, FALSE otherwise
 */
BOOLEAN ADB_GetLNBBandParameters(void *band_ptr, S_STB_DP_LNB_BAND *params)
{
   BOOLEAN success;

   FUNCTION_START(ADB_GetLNBBandParameters);

   if (band_ptr != NULL)
   {
      success = TRUE;

      DBDEF_RequestAccess();

      memcpy(params, &(((ADB_LNB_BAND_REC *)band_ptr)->band_params), sizeof(S_STB_DP_LNB_BAND));

      DBDEF_ReleaseAccess();
   }
   else
   {
      success = FALSE;
   }

   FUNCTION_FINISH(ADB_GetLNBBandParameters);

   return success;
}

/**
 * @brief   Deletes LNB bands associated with a given LNB
 * @param   lnb_ptr a pointer to an LNB record, LNB bands associated with this LNB are deleted
 */
void ADB_DeleteAssociatedLNBBands(void *lnb_ptr)
{
   ADB_LNB_BAND_REC *band_ptr = NULL;
   ADB_LNB_BAND_REC *next_band_ptr;

   FUNCTION_START(ADB_DeleteAssociatedLNBBands);

   DBDEF_RequestAccess();

   band_ptr = DBDEF_GetNextLNBBandRec(band_ptr);
   while (band_ptr != NULL)
   {
      next_band_ptr = DBDEF_GetNextLNBBandRec(band_ptr);

      if (band_ptr->lnb == lnb_ptr)
      {
         DBDEF_DeleteLNBBandRec(band_ptr);
      }

      band_ptr = next_band_ptr;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_DeleteAssociatedLNBBands);
}

/**
 * @brief   Returns the number of satellites in the database
 * @return  Number of satellites
 */
U16BIT ADB_GetNumSatellites(void)
{
   U16BIT num;

   FUNCTION_START(ADB_GetNumSatellites);

   DBDEF_RequestAccess();
   num = DBDEF_GetNumSatellites();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumSatellites);

   return(num);
}

/**
 * @brief   Creates a new satellite in the database with the given settings
 * @param   name_str name to be given to the satellite, must be a DVB format string
 * @param   dish_pos
 * @param   long_pos longitudinal position of the satellite in 1/10ths degree (e.g. 19.2 would be
 *          192)
 * @param   east_west whether the position of defined from east (TRUE) or west (FALSE)
 * @param   lnb_ptr pointer to an LNB already present in the database
 * @return  pointer to new satellite
 */
void* ADB_AddSatellite(U8BIT *name_str, U16BIT dish_pos, U16BIT long_pos, BOOLEAN east_west,
   void *lnb_ptr)
{
   ADB_SATELLITE_REC *sat_ptr;
   ADB_STRING *name;

   FUNCTION_START(ADB_AddSatellite);

   DBDEF_RequestAccess();

   name = DBDEF_MakeString(0, name_str, STB_GetNumBytesInString(name_str));
   sat_ptr = DBDEF_AddSatelliteRec(name, dish_pos, long_pos, east_west, lnb_ptr);

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddSatellite);
   return(sat_ptr);
}

/**
 * @brief   Returns the next satellite from the database
 * @param   sat_ptr current satellite pointer, NULL to get the first satellite
 * @return  pointer to satellite
 */
void* ADB_GetNextSatellite(void *sat_ptr)
{
   void *next_sat;

   FUNCTION_START(ADB_GetNextSatellite);

   DBDEF_RequestAccess();
   next_sat = DBDEF_GetNextSatelliteRec(sat_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNextSatellite);

   return(next_sat);
}

/**
 * @brief   Returns the pointer to the name of the satellite
 * @param   sat_ptr satellite
 * @return  pointer to satellite name
 */
U8BIT* ADB_GetSatelliteName(void *sat_ptr)
{
   ADB_SATELLITE_REC *sat = (ADB_SATELLITE_REC *)sat_ptr;
   U8BIT *name;

   FUNCTION_START(ADB_GetSatelliteName);

   DBDEF_RequestAccess();

   if ((sat_ptr != NULL) && (sat->name != NULL))
   {
      name = sat->name->str_ptr;
   }
   else
   {
      name = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetSatelliteName);

   return(name);
}

/**
 * @brief   Returns the LNB associated with the given satellite
 * @param   sat_ptr satellite
 * @return  pointer to LNB
 */
void* ADB_GetSatelliteLNB(void *sat_ptr)
{
   void *lnb_ptr;

   FUNCTION_START(ADB_GetSatelliteLNB);

   DBDEF_RequestAccess();

   if (sat_ptr != NULL)
   {
      lnb_ptr = ((ADB_SATELLITE_REC *)sat_ptr)->lnb;
   }
   else
   {
      lnb_ptr = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetSatelliteLNB);

   return(lnb_ptr);
}

/**
 * @brief   Returns the longitudinal position of the given satellite in 1/10ths degree
 * @param   sat_ptr satellite
 * @return  satellite position in 1/10ths degree
 */
U16BIT ADB_GetSatelliteLongitude(void *sat_ptr)
{
   S16BIT long_pos;

   FUNCTION_START(ADB_GetSatelliteLongitude);

   if (sat_ptr != NULL)
   {
      DBDEF_RequestAccess();
      long_pos = ((ADB_SATELLITE_REC *)sat_ptr)->long_pos;
      DBDEF_ReleaseAccess();
      if (long_pos < 0)
      {
         long_pos *= -1;
      }
   }
   else
   {
      long_pos = 0;
   }

   FUNCTION_FINISH(ADB_GetSatelliteLongitude);

   return (U16BIT)long_pos;
}

/**
 * @brief   Returns the position direction of the given satellite
 * @param   sat_ptr satellite
 * @return  TRUE=east, FALSE=west
 */
BOOLEAN ADB_GetSatelliteDirection(void *sat_ptr)
{
   BOOLEAN east_west;

   FUNCTION_START(ADB_GetSatelliteDirection);

   if (sat_ptr != NULL)
   {
      DBDEF_RequestAccess();
      east_west = ((ADB_SATELLITE_REC *)sat_ptr)->east_west;
      DBDEF_ReleaseAccess();
   }
   else
   {
      east_west = FALSE;
   }

   FUNCTION_FINISH(ADB_GetSatelliteDirection);

   return(east_west);
}

/**
 * @brief   Returns the number of network records in the database
 * @return  Number of networks
 */
U16BIT ADB_GetNumNetworks(void)
{
   U16BIT num_networks;

   FUNCTION_START(ADB_GetNumNetworks);

   DBDEF_RequestAccess();
   num_networks = DBDEF_GetNumNetworks();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumNetworks);

   return(num_networks);
}

/**
 * @brief   Allocates and returns a list of the network records in the database.
 * @param   nlist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no networks
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no networks
 */
void ADB_GetNetworkList(void ***nlist_ptr, U16BIT *num_entries_ptr)
{
   void **nlist;
   U16BIT num_entries;
   ADB_NETWORK_REC *n_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetNetworkList);

   DBDEF_RequestAccess();

   if ((num_entries = DBDEF_GetNumNetworks()) > 0)
   {
      nlist = STB_AppGetMemory(num_entries * sizeof(void *));
      if (nlist != NULL)
      {
         n_ptr = DBDEF_GetNextNetworkRec(NULL);
         for (i = 0; ((i < num_entries) && (n_ptr != NULL)); i++)
         {
            nlist[i] = (void *)n_ptr;
            n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
         }
      }
      else
      {
         num_entries = 0;
      }

      *nlist_ptr = nlist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();
   FUNCTION_FINISH(ADB_GetNetworkList);
}

/**
 * @brief   Frees a network list returned by ADB_GetNetworkList
 * @param   nlist network list to be freed
 */
void ADB_ReleaseNetworkList(void **nlist)
{
   FUNCTION_START(ADB_ReleaseNetworkList);

   if (nlist != NULL)
   {
      STB_AppFreeMemory(nlist);
   }

   FUNCTION_FINISH(ADB_ReleaseNetworkList);
}

/**
 * @brief   Returns a list of names, in UTF-8 format, corresponding to the given network list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   nlist the array of network records
 * @param   num_entries number of items in the list
 * @param   short_names TRUE if the short network names should be returned
 * @return  pointer to the list of name strings
 */
U8BIT** ADB_GetNetworkListNames(void **nlist, U16BIT num_entries, BOOLEAN short_names)
{
   U8BIT **names_array;
   U16BIT i;

   FUNCTION_START(ADB_GetNetworkListNames);

   if (nlist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            names_array[i] = GetNetworkName((ADB_NETWORK_REC *)(nlist[i]), short_names);
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetNetworkListNames);
   return(names_array);
}

/**
 * @brief   Allocates and returns an array of network ids for the given networks
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   nlist the array of network records
 * @param   num_entries number of items in the list
 * @return  pointer to the network id array, NULL if no networks are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetNetworkListIds(void **nlist, U16BIT num_entries)
{
   U32BIT *nid_array;
   U16BIT i;

   FUNCTION_START(ADB_GetNetworkListIds);

   if (nlist != NULL)
   {
      DBDEF_RequestAccess();
      nid_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (nid_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            nid_array[i] = (U32BIT)GetNetworkId((ADB_NETWORK_REC *)nlist[i]);
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      nid_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetNetworkListIds);

   return(nid_array);
}

/**
 * @brief   Returns the name, in UTF-8 format, of the given network.
 *          The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   n_ptr network
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetNetworkName(void *n_ptr)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetNetworkName);

   DBDEF_RequestAccess();
   name_str = GetNetworkName((ADB_NETWORK_REC *)n_ptr, FALSE);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkName);

   return(name_str);
}

/**
 * @brief   Returns the name in the language defined by lang, in UTF-8 format, of the
 *          given network. The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   n_ptr network
 * @param   lang language id
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetNetworkNameByLang(void *n_ptr, U8BIT lang)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetNetworkNameByLang);

   DBDEF_RequestAccess();
   name_str = GetNetworkNameByLang((ADB_NETWORK_REC *)n_ptr, lang, FALSE);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkNameByLang);

   return(name_str);
}

/**
 * @brief   Returns the network id of the given network
 * @param   n_ptr network
 * @return  network id, 0 if network isn't found
 */
U16BIT ADB_GetNetworkId(void *n_ptr)
{
   U16BIT nid_value;

   FUNCTION_START(ADB_GetNetworkId);

   DBDEF_RequestAccess();
   nid_value = GetNetworkId((ADB_NETWORK_REC *)n_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkId);

   return(nid_value);
}

/**
 * @brief   Returns a combined array of country codes for all discovered networks
 * @param   code_array pointer to an array that will be allocated in UI temp memory
 * @return  Number of codes returned in the array
 */
U8BIT ADB_GetNetworkTargetCountries(U32BIT **code_array)
{
   U8BIT num_countries;
   U8BIT i;
   ADB_NETWORK_REC *nptr;
   SI_NIT_TARGET_REGION_NAME_DESC *tptr;
   U32BIT *new_array;
   BOOLEAN code_found;

   FUNCTION_START(ADB_GetNetworkTargetCountries);

   DBDEF_RequestAccess();

   num_countries = 0;
   *code_array = NULL;

   nptr = DBDEF_GetNextNetworkRec(NULL);
   while (nptr != NULL)
   {
      tptr = nptr->target_region_name_list;
      while (tptr != NULL)
      {
         code_found = FALSE;

         /* Check whether this country code is already in the array */
         if (*code_array != NULL)
         {
            for (i = 0; (i < num_countries) && !code_found; i++)
            {
               if ((*code_array)[i] == tptr->country_code)
               {
                  code_found = TRUE;
               }
            }
         }

         if (!code_found)
         {
            new_array = (U32BIT *)STB_AppGetMemory((num_countries + 1) * sizeof(U32BIT));
            if (new_array != NULL)
            {
               memcpy(new_array, *code_array, num_countries * sizeof(U32BIT));
               STB_AppFreeMemory(*code_array);
            }

            *code_array = new_array;

            if (*code_array != NULL)
            {
               (*code_array)[num_countries] = tptr->country_code;
               num_countries++;
            }
         }

         tptr = tptr->next;
      }

      nptr = DBDEF_GetNextNetworkRec(nptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkTargetCountries);

   return(num_countries);
}

/**
 * @brief   Returns an array of primary region codes and names for the given country
 * @param   country_code code of country
 * @param   code_array pointer to an array of primary region codes that will be
 *          allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *          of the primary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkPrimaryTargetRegions(U32BIT country_code, U8BIT **code_array, U8BIT ***name_array)
{
   U16BIT num_regions;
   U8BIT i, j;
   ADB_NETWORK_REC *nptr;
   SI_NIT_TARGET_REGION_NAME_DESC *tptr;
   U8BIT *new_codes;
   U8BIT **new_names;
   BOOLEAN code_found;

   FUNCTION_START(ADB_GetNetworkPrimaryTargetRegions);

   DBDEF_RequestAccess();

   num_regions = 0;
   *code_array = NULL;
   *name_array = NULL;

   nptr = DBDEF_GetNextNetworkRec(NULL);
   while (nptr != NULL)
   {
      tptr = nptr->target_region_name_list;
      while (tptr != NULL)
      {
         if ((tptr->country_code == country_code) && (tptr->num_names > 0) &&
             (tptr->name_array != NULL))
         {
            /* Check each primary region defined for this country and add it to the array of
             * regions if it's new */
            for (i = 0; i < tptr->num_names; i++)
            {
               /* Does this region name define a primary region? */
               if (tptr->name_array[i].region_depth == 1)
               {
                  code_found = FALSE;

                  /* This is a name for a primary region, check whether it's already in the code array */
                  if (*code_array != NULL)
                  {
                     for (j = 0; (j < num_regions) && !code_found; j++)
                     {
                        if ((*code_array)[j] == tptr->name_array[i].primary_region_code)
                        {
                           code_found = TRUE;
                        }
                     }
                  }

                  if (!code_found)
                  {
                     new_codes = (U8BIT *)STB_AppGetMemory((num_regions + 1) * sizeof(U8BIT));
                     new_names = (U8BIT **)STB_AppGetMemory((num_regions + 1) * sizeof(U8BIT *));
                     if ((new_codes != NULL) && (new_names != NULL))
                     {
                        memcpy(new_codes, *code_array, num_regions * sizeof(U8BIT));
                        STB_AppFreeMemory(*code_array);

                        memcpy(new_names, *name_array, num_regions * sizeof(U8BIT *));
                        STB_AppFreeMemory(*name_array);
                     }

                     *code_array = new_codes;
                     *name_array = (void *)new_names;

                     if ((*code_array != NULL) && (*name_array != NULL))
                     {
                        (*code_array)[num_regions] = tptr->name_array[i].primary_region_code;
                        (*name_array)[num_regions] = CopySIString(tptr->name_array[i].region_name);
                        num_regions++;
                     }
                  }
               }
            }
         }

         tptr = tptr->next;
      }

      nptr = DBDEF_GetNextNetworkRec(nptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkPrimaryTargetRegions);

   return(num_regions);
}

/**
 * @brief   Returns an array of secondary region codes and names for the given country
 *          and primary region.
 * @param   country_code code of country
 * @param   primary_region primary region code
 * @param   code_array pointer to an array of secondary region codes that will be
 *          allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *          of the secondary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkSecondaryTargetRegions(U32BIT country_code, U8BIT primary_region,
   U8BIT **code_array, U8BIT ***name_array)
{
   U16BIT num_regions;
   U8BIT i, j;
   ADB_NETWORK_REC *nptr;
   SI_NIT_TARGET_REGION_NAME_DESC *tptr;
   U8BIT *new_codes;
   U8BIT **new_names;
   BOOLEAN code_found;

   FUNCTION_START(ADB_GetNetworkSecondaryTargetRegions);

   DBDEF_RequestAccess();

   num_regions = 0;
   *code_array = NULL;
   *name_array = NULL;

   nptr = DBDEF_GetNextNetworkRec(NULL);
   while (nptr != NULL)
   {
      tptr = nptr->target_region_name_list;
      while (tptr != NULL)
      {
         if ((tptr->country_code == country_code) && (tptr->num_names > 0) &&
             (tptr->name_array != NULL))
         {
            /* Check each secondary region defined for this country and primary and add
             * it to the array of regions if it's new */
            for (i = 0; i < tptr->num_names; i++)
            {
               /* Does this region name define a secondary region? */
               if ((tptr->name_array[i].region_depth == 2) &&
                   (tptr->name_array[i].primary_region_code = primary_region))
               {
                  code_found = FALSE;

                  /* This is a name for a secondary region, check whether it's already in the code array */
                  if (*code_array != NULL)
                  {
                     for (j = 0; (j < num_regions) && !code_found; j++)
                     {
                        if ((*code_array)[j] == tptr->name_array[i].secondary_region_code)
                        {
                           code_found = TRUE;
                        }
                     }
                  }

                  if (!code_found)
                  {
                     new_codes = (U8BIT *)STB_AppGetMemory((num_regions + 1) * sizeof(U8BIT));
                     new_names = (U8BIT **)STB_AppGetMemory((num_regions + 1) * sizeof(U8BIT *));
                     if ((new_codes != NULL) && (new_names != NULL))
                     {
                        memcpy(new_codes, *code_array, num_regions * sizeof(U8BIT));
                        STB_AppFreeMemory(*code_array);

                        memcpy(new_names, *name_array, num_regions * sizeof(U8BIT *));
                        STB_AppFreeMemory(*name_array);
                     }

                     *code_array = new_codes;
                     *name_array = (void *)new_names;

                     if ((*code_array != NULL) && (*name_array != NULL))
                     {
                        (*code_array)[num_regions] = tptr->name_array[i].secondary_region_code;
                        (*name_array)[num_regions] = CopySIString(tptr->name_array[i].region_name);
                        num_regions++;
                     }
                  }
               }
            }
         }

         tptr = tptr->next;
      }

      nptr = DBDEF_GetNextNetworkRec(nptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkSecondaryTargetRegions);

   return(num_regions);
}

/**
 * @brief   Returns an array of tertiary region codes and names for the given country,
 *          primary region and secondary region.
 * @param   country_code code of country
 * @param   primary_region primary region code
 * @param   secondary_region secondary region code
 * @param   code_array pointer to an array of tertiary region codes that will be
 *                       allocated in UI temp memory.
 * @param   name_array pointer to an array of strings that represent the names
 *                       of the tertiary regions.
 * @return  Number of codes & names returned in the arrays
 */
U16BIT ADB_GetNetworkTertiaryTargetRegions(U32BIT country_code, U8BIT primary_region,
   U8BIT secondary_region, U16BIT **code_array, U8BIT ***name_array)
{
   U16BIT num_regions;
   U8BIT i, j;
   ADB_NETWORK_REC *nptr;
   SI_NIT_TARGET_REGION_NAME_DESC *tptr;
   U16BIT *new_codes;
   U8BIT **new_names;
   BOOLEAN code_found;

   FUNCTION_START(ADB_GetNetworkTertiaryTargetRegions);

   DBDEF_RequestAccess();

   num_regions = 0;
   *code_array = NULL;
   *name_array = NULL;

   nptr = DBDEF_GetNextNetworkRec(NULL);
   while (nptr != NULL)
   {
      tptr = nptr->target_region_name_list;
      while (tptr != NULL)
      {
         if ((tptr->country_code == country_code) && (tptr->num_names > 0) &&
             (tptr->name_array != NULL))
         {
            /* Check each secondary region defined for this country and primary and add
             * it to the array of regions if it's new */
            for (i = 0; i < tptr->num_names; i++)
            {
               /* Does this region name define a tertiary region? */
               if ((tptr->name_array[i].region_depth == 3) &&
                   (tptr->name_array[i].primary_region_code = primary_region) &&
                   (tptr->name_array[i].secondary_region_code = secondary_region))
               {
                  code_found = FALSE;

                  /* This is a name for a tertiary region, check whether it's already in the code array */
                  if (*code_array != NULL)
                  {
                     for (j = 0; (j < num_regions) && !code_found; j++)
                     {
                        if ((*code_array)[j] == tptr->name_array[i].tertiary_region_code)
                        {
                           code_found = TRUE;
                        }
                     }
                  }

                  if (!code_found)
                  {
                     new_codes = (U16BIT *)STB_AppGetMemory((num_regions + 1) * sizeof(U16BIT));
                     new_names = (U8BIT **)STB_AppGetMemory((num_regions + 1) * sizeof(U8BIT *));
                     if ((new_codes != NULL) && (new_names != NULL))
                     {
                        memcpy(new_codes, *code_array, num_regions * sizeof(U16BIT));
                        STB_AppFreeMemory(*code_array);

                        memcpy(new_names, *name_array, num_regions * sizeof(U8BIT *));
                        STB_AppFreeMemory(*name_array);
                     }

                     *code_array = new_codes;
                     *name_array = (void *)new_names;

                     if ((*code_array != NULL) && (*name_array != NULL))
                     {
                        (*code_array)[num_regions] = tptr->name_array[i].tertiary_region_code;
                        (*name_array)[num_regions] = CopySIString(tptr->name_array[i].region_name);
                        num_regions++;
                     }
                  }
               }
            }
         }

         tptr = tptr->next;
      }

      nptr = DBDEF_GetNextNetworkRec(nptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkTertiaryTargetRegions);

   return(num_regions);
}

/**
 * @brief   Returns the network currently tuned to on the given decode path
 * @param   path decode path
 * @return  network pointer, or NULL if invalid path of not tuned
 */
void* ADB_GetTunedNetwork(U8BIT path)
{
   void *n_ptr;

   FUNCTION_START(ADB_GetTunedNetwork);

   DBDEF_RequestAccess();
   n_ptr = (void *)DBDEF_GetTunedNetwork(path);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTunedNetwork);

   return(n_ptr);
}

/**
 * @brief   Allocates and returns a list of all transport records in the database.
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *                    be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *                          value will be 0 on return if there are no transports
 */
void ADB_GetTransportList(void ***tlist_ptr, U16BIT *num_entries_ptr)
{
   FUNCTION_START(ADB_GetTransportList);

   ADB_GetTransportListForTunerType(SIGNAL_NONE, tlist_ptr, num_entries_ptr);

   FUNCTION_FINISH(ADB_GetTransportList);
}

/**
 * @brief
 * @brief   Allocates and returns a list of all transport records in the database
 *          for the given type of tuner.
 * @param   tuner_type transports of this tuner type will be returned; if SIGNAL_NONE is given
 *                     then all transports will be returned
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *                    be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *                          value will be 0 on return if there are no transports
 */
void ADB_GetTransportListForTunerType(E_STB_DP_SIGNAL_TYPE tuner_type, void ***tlist_ptr,
   U16BIT *num_entries_ptr)
{
   void **tlist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListForTunerType);

   *tlist_ptr = NULL;
   *num_entries_ptr = 0;

   DBDEF_RequestAccess();

   if ((max_no_entries = DBDEF_GetNumTransports()) > 0)
   {
      tlist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      num_entries = 0;
      if (tlist != NULL)
      {
         t_ptr = DBDEF_GetNextTransportRec(NULL);
         for (i = 0; ((i < max_no_entries) && (t_ptr != NULL)); i++)
         {
            if ((tuner_type == SIGNAL_NONE) || (t_ptr->sig_type == tuner_type))
            {
               tlist[num_entries] = (void *)t_ptr;
               num_entries++;
            }

            t_ptr = DBDEF_GetNextTransportRec(t_ptr);
         }
      }

      *tlist_ptr = tlist;
      *num_entries_ptr = num_entries;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportListForTunerType);
}

/**
 * @brief   Allocates and returns a list of all transport records in the database
 *          for the given network.
 * @param   n_ptr network pointer
 * @param   tlist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no transports
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no transports
 */
void ADB_GetNetworkTransportList(void *n_ptr, void ***tlist_ptr, U16BIT *num_entries_ptr)
{
   void **tlist;
   U16BIT max_no_entries;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT i;
   U16BIT num_entries;

   FUNCTION_START(ADB_GetNetworkTransportList);

   DBDEF_RequestAccess();

   num_entries = 0;

   if ((max_no_entries = DBDEF_GetNumTransports()) > 0)
   {
      tlist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (tlist != NULL)
      {
         t_ptr = DBDEF_GetNextTransportRec(NULL);
         for (i = 0; ((i < max_no_entries) && (t_ptr != NULL)); i++)
         {
            /* if transport belongs to the current network */
            if ((t_ptr->network != NULL) && (t_ptr->network == n_ptr))
            {
               tlist[num_entries] = (void *)t_ptr;
               num_entries++;
            }
            t_ptr = DBDEF_GetNextTransportRec(t_ptr);
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(tlist);
            tlist = NULL;
         }
      }

      *tlist_ptr = tlist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkTransportList);
}

/**
 * @brief   Returns the number of transports in the database
 * @return  Number of transports
 */
U16BIT ADB_GetNumTransports(void)
{
   U16BIT num_transports;

   FUNCTION_START(ADB_GetNumTransports);

   DBDEF_RequestAccess();
   num_transports = DBDEF_GetNumTransports();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumTransports);

   return(num_transports);
}

/**
 * @brief   Frees a transport list returned from one of the ADB_GetTransportList functions
 * @param   tlist transport list to be freed
 * @param   num_transports number of transports in the list
 */
void ADB_ReleaseTransportList(void **tlist, U16BIT num_transports)
{
   FUNCTION_START(ADB_ReleaseTransportList);

   USE_UNWANTED_PARAM(num_transports);

   if (tlist != NULL)
   {
      STB_AppFreeMemory(tlist);
   }

   FUNCTION_FINISH(ADB_ReleaseTransportList);
}

/**
 * @brief   Returns a list of names, in UTF-8 format, corresponding to the given transport list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the list of name strings
 */
U8BIT** ADB_GetTransportListNames(void **tlist, U16BIT num_entries)
{
   U8BIT **names_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListNames);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            names_array[i] = GetTransportName((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListNames);
   return(names_array);
}

/**
 * @brief   Allocates and returns an array of transport ids for the given transports.
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the transport id array, NULL if no transports are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetTransportListTids(void **tlist, U16BIT num_entries)
{
   U32BIT *tid_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListTids);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      tid_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (tid_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            tid_array[i] = (U32BIT)GetTransportTid((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      tid_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListTids);

   return(tid_array);
}

/**
 * @brief   Allocates and returns an array of percentage values representing the
 *          signal strength when each transport in the given list was tuned to.
 *          The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the signal strength array, NULL if no transports are provided
 *          or memory allocation fails
 */
U32BIT* ADB_GetTransportListTunedStrengths(void **tlist, U16BIT num_entries)
{
   U32BIT *tuned_signal_strength_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListTunedStrengths);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      tuned_signal_strength_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (tuned_signal_strength_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            tuned_signal_strength_array[i] = (U32BIT)GetTransportTunedStrength((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      tuned_signal_strength_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListTunedStrengths);

   return(tuned_signal_strength_array);
}

/**
 * @brief   Allocates and returns an array of original network ids for each transport
 *          in the given list was tuned to. The returned array should be freed using
 *          STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the array of original network ids, NULL if no transports are
 *          provided or memory allocation fails
 */
U32BIT* ADB_GetTransportListOriginalNetworkIds(void **tlist, U16BIT num_entries)
{
   U32BIT *orig_network_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListOriginalNetworkIds);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      orig_network_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (orig_network_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            orig_network_array[i] = (U32BIT)GetTransportOriginalNetworkId((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      orig_network_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListOriginalNetworkIds);

   return(orig_network_array);
}

/**
 * @brief   Allocates and returns an array of tuning frequencies for each transport
 *          in the given list. The returned array should be freed using STB_AppFreeMemory.
 * @param   tlist the array of transport records
 * @param   num_entries number of items in the list
 * @return  pointer to the array of frequencies, NULL if no transports are
 *          provided or memory allocation fails
 */
U32BIT* ADB_GetTransportListFreqs(void **tlist, U16BIT num_entries)
{
   U32BIT *freq_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListFreqs);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      freq_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (freq_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            freq_array[i] = (U32BIT)GetTransportFreq((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      freq_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListFreqs);

   return(freq_array);
}

/* The next four functions are NEVER used! */

/**
 *

 *
 * @brief   Returns the transport bwidth_array
 *
 * @param   transport pointer
 *
 * @return   bwidth_array
 *
 */
U32BIT* ADB_GetTransportListBwidths(void **tlist, U16BIT num_entries)
{
   U32BIT *bwidth_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListBwidths);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      bwidth_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (bwidth_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            bwidth_array[i] = (U32BIT)GetTransportBwidth((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      bwidth_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListBwidths);

   return(bwidth_array);
}

/**
 *

 *
 * @brief   Returns the transport offset array
 *
 * @param   transport pointer
 *
 * @return   offset_array
 *
 */
U32BIT* ADB_GetTransportListOffsets(void **tlist, U16BIT num_entries)
{
   U32BIT *offset_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListOffsets);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      offset_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (offset_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            offset_array[i] = (U32BIT)GetTransportOffset((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      offset_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListOffsets);

   return(offset_array);
}

/**
 *

 *
 * @brief   Returns the transport constellation string array
 *
 * @param   transport pointer
 *
 * @return   constellation_array
 *
 */
U8BIT** ADB_GetTransportListConstellationStrings(void **tlist, U16BIT num_entries)
{
   U8BIT **constellation_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListConstellationStrings);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      constellation_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (constellation_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            constellation_array[i] = GetTransportConstellationString((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      constellation_array = NULL;
   }
   FUNCTION_FINISH(ADB_GetTransportListConstellationStrings);

   return(constellation_array);
}

/**
 *

 *
 * @brief   Returns the transport hierarchy string array
 *
 * @param   transport pointer
 *
 * @return   hierarchy_array
 *
 */
U8BIT** ADB_GetTransportListHierarchyStrings(void **tlist, U16BIT num_entries)
{
   U8BIT **hierarchy_array;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportListHierarchyStrings);

   if (tlist != NULL)
   {
      DBDEF_RequestAccess();
      hierarchy_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (hierarchy_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            hierarchy_array[i] = GetTransportHierarchyString((ADB_TRANSPORT_REC *)(tlist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      hierarchy_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetTransportListHierarchyStrings);

   return(hierarchy_array);
}

/**
 * @brief   Returns whether a transport has been used during a service scan operation
 * @param   t_ptr transport
 * @return  TRUE if the transport has been used, FALSE otherwise
 */
BOOLEAN ADB_GetTransportSearchFlag(void *t_ptr)
{
   BOOLEAN searched;

   FUNCTION_START(ADB_GetTransportSearchFlag);

   searched = FALSE;

   if (t_ptr != NULL)
   {
      DBDEF_RequestAccess();
      searched = ((ADB_TRANSPORT_REC *)t_ptr)->searched;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportSearchFlag);

   return(searched);
}

/**
 * @brief   Sets whether a transport has been used during a service scan operation
 * @param   t_ptr transport
 * @param   state set the flag to TRUE or FALSE
 */
void ADB_SetTransportSearchFlag(void *t_ptr, BOOLEAN state)
{
   FUNCTION_START(ADB_SetTransportSearchFlag);

   if (t_ptr != NULL)
   {
      DBDEF_RequestAccess();
      ((ADB_TRANSPORT_REC *)t_ptr)->searched = state;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetTransportSearchFlag);
}

/**
 * @brief   Set the signal level to 0 and searched flag to TRUE for the given transport
 * @param   t_ptr pointer to the transport
 */
void ADB_ReportNoSignalDuringSearch(void *t_ptr)
{
   ADB_TRANSPORT_REC *trans_ptr = (ADB_TRANSPORT_REC *)t_ptr;

   FUNCTION_START(ADB_ReportNoSignalDuringSearch);

   if (t_ptr != NULL)
   {
      DBDEF_RequestAccess();

      trans_ptr->searched = TRUE;
      trans_ptr->signal_level_at_search = 0;

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_ReportNoSignalDuringSearch);
}

/**
 * @brief   Returns the signal type for the given transport
 * @param   t_ptr transport
 * @return  signal type
 */
E_STB_DP_SIGNAL_TYPE ADB_GetTransportSignalType(void *t_ptr)
{
   E_STB_DP_SIGNAL_TYPE stype;

   FUNCTION_START(ADB_GetTransportSignalType);

   DBDEF_RequestAccess();
   if (t_ptr != NULL)
   {
      stype = ((ADB_TRANSPORT_REC *)t_ptr)->sig_type;
   }
   else
   {
      stype = SIGNAL_NONE;
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportSignalType);

   return(stype);
}

/**
 * @brief   Returns the name, in UTF-8 format, of the given transport.
 *          The returned name should be freed using STB_ReleaseUnicodeString.
 * @param   t_ptr transport
 * @return  UTF-8 format string, NULL on failure
 */
U8BIT* ADB_GetTransportName(void *t_ptr)
{
   U8BIT *name_str_ptr;

   FUNCTION_START(ADB_GetTransportName);

   DBDEF_RequestAccess();
   name_str_ptr = GetTransportName((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportName);

   return(name_str_ptr);
}

/**
 * @brief   Returns the transport id of the given transport
 * @param   t_ptr transport
 * @return  transport id, 0 if transport isn't found
 */
U16BIT ADB_GetTransportTid(void *t_ptr)
{
   U16BIT tid_value;

   FUNCTION_START(ADB_GetTransportTid);

   DBDEF_RequestAccess();
   tid_value = GetTransportTid((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportTid);

   return(tid_value);
}

/**
 * @brief   Returns the signal strength, as a percentage, of the tuned transport
 * @param   t_ptr cable transport
 * @return  signal strength as a percentage
 */
U8BIT ADB_GetTransportTunedStrength(void *t_ptr)
{
   U8BIT tuned_signal_strength;

   FUNCTION_START(ADB_GetTransportTunedStrength);

   DBDEF_RequestAccess();
   tuned_signal_strength = GetTransportTunedStrength((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportTunedStrength);

   return(tuned_signal_strength);
}

/**
 * @brief   Returns the signal quality, as a percentage, of the tuned transport
 * @param   t_ptr cable transport
 * @return  signal quality as a percentage
 */
U8BIT ADB_GetTransportTunedQuality(void *t_ptr)
{
   U8BIT tuned_signal_quality;

   FUNCTION_START(ADB_GetTransportTunedQuality);

   DBDEF_RequestAccess();
   tuned_signal_quality = GetTransportTunedQuality((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportTunedQuality);

   return(tuned_signal_quality);
}

/**
 * @brief   Returns the original network id of the given transport
 * @param   t_ptr transport
 * @return  original network id, 0 if transport isn't found
 */
U16BIT ADB_GetTransportOriginalNetworkId(void *t_ptr)
{
   U16BIT orig_network_id;

   FUNCTION_START(ADB_GetTransportOriginalNetworkId);

   DBDEF_RequestAccess();
   orig_network_id = GetTransportOriginalNetworkId((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportOriginalNetworkId);

   return(orig_network_id);
}

/**
 * @brief   Returns the network of the given transport
 * @param   t_ptr transport
 * @return  pointer to the network, possibly NULL no NIT was found for the transport
 */
void* ADB_GetTransportNetworkPtr(void *t_ptr)
{
   void *n_ptr;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportNetworkPtr);

   n_ptr = NULL;
   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      n_ptr = trans->network;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportNetworkPtr);

   return(n_ptr);
}

/**
 * @brief   Returns the frequency, in Hz for DVB-T/T2 and DVB-C, and MHz for DVB-S/S2,
 *          of the given transport
 * @param   t_ptr transport
 * @return  frequency, 0 if transport not found
 */
U32BIT ADB_GetTransportFreqHz(void *t_ptr)
{
   U32BIT tp_freq;

   FUNCTION_START(ADB_GetTransportFreqHz);

   DBDEF_RequestAccess();
   tp_freq = GetTransportFreq((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportFreqHz);

   return(tp_freq);
}

/**
 * @brief   Returns the bandwidth value of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  bandwidth, return value is undefined if transport isn't DVB-T/T2
 */
U8BIT ADB_GetTransportBwidth(void *t_ptr)
{
   U8BIT bwidth;

   FUNCTION_START(ADB_GetTransportBwidth);

   DBDEF_RequestAccess();
   bwidth = GetTransportBwidth((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportBwidth);

   return(bwidth);
}

/**
 * @brief   Returns the terrestrial mode of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  mode, returns MODE_COFDM_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_DP_TMODE ADB_GetTransportTerrMode(void *t_ptr)
{
   E_STB_DP_TMODE mode;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportTerrMode);

   mode = MODE_COFDM_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         mode = trans->u.terr.tmode;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportTerrMode);

   return(mode);
}

/**
 * @brief   Returns the tuning offset of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  bandwidth, return value is undefined if transport isn't DVB-T/T2
 */
S8BIT ADB_GetTransportOffset(void *t_ptr)
{
   S8BIT offset;

   FUNCTION_START(ADB_GetTransportOffset);

   DBDEF_RequestAccess();
   offset = GetTransportOffset((ADB_TRANSPORT_REC *)t_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportOffset);

   return(offset);
}

/**
 * @brief   Returns the constellation of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCONST_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCONST ADB_GetTransportConstellation(void *t_ptr)
{
   E_STB_TUNE_TCONST constellation;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportConstellation);

   constellation = TUNE_TCONST_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         constellation = trans->u.terr.constellation;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportConstellation);

   return(constellation);
}

/**
 * @brief   Returns the hierarchy of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  hierarchy, returns TUNE_THIERARCHY_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_THIERARCHY ADB_GetTransportHierarchy(void *t_ptr)
{
   E_STB_TUNE_THIERARCHY hierarchy;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportHierarchy);

   hierarchy = TUNE_THIERARCHY_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         hierarchy = trans->u.terr.hierarchy;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportHierarchy);

   return(hierarchy);
}

/**
 * @brief   Returns the LP coderate of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCODERATE_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCODERATE ADB_GetTransportLpCodeRate(void *t_ptr)
{
   E_STB_TUNE_TCODERATE coderate;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportLpCodeRate);

   coderate = TUNE_TCODERATE_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         coderate = trans->u.terr.lp_code_rate;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportLpCodeRate);

   return(coderate);
}

/**
 * @brief   Returns the HP coderate of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  constellation, returns TUNE_TCODERATE_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TCODERATE ADB_GetTransportHpCodeRate(void *t_ptr)
{
   E_STB_TUNE_TCODERATE coderate;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportHpCodeRate);

   coderate = TUNE_TCODERATE_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         coderate = trans->u.terr.hp_code_rate;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportHpCodeRate);

   return(coderate);
}

/**
 * @brief   Returns the guard interval of the given DVB-T/T2 transport
 * @param   t_ptr transport
 * @return  guard interval, returns TUNE_TGUARDINT_UNDEFINED if transport isn't DVB-T/T2
 */
E_STB_TUNE_TGUARDINT ADB_GetTransportGuardInt(void *t_ptr)
{
   E_STB_TUNE_TGUARDINT guard_int;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportGuardInt);

   guard_int = TUNE_TGUARDINT_UNDEFINED;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_COFDM)
      {
         guard_int = trans->u.terr.guard_int;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportGuardInt);

   return(guard_int);
}

/**
 * @brief   Returns the QAM mode of the given cable transport
 * @param   t_ptr cable transport
 * @return  QAM mode, returns MODE_QAM_AUTO if transport isn't DVB-C
 */
E_STB_DP_CMODE ADB_GetTransportCableMode(void *t_ptr)
{
   E_STB_DP_CMODE mode;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportCableMode);

   mode = MODE_QAM_AUTO;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_QAM)
      {
         mode = trans->u.cab.cmode;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportCableMode);

   return(mode);
}

/**
 * @brief   Returns the symbol rate of the given cable transport
 * @param   t_ptr cable transport
 * @return  symbol rate, returns 0 if transport isn't DVB-C
 */
U16BIT ADB_GetTransportCableSymbolRate(void *t_ptr)
{
   U16BIT sym_rate;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportCableSymbolRate);

   sym_rate = 0;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_QAM)
      {
         sym_rate = (U16BIT)trans->u.cab.symbol_rate;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportCableSymbolRate);

   return(sym_rate);
}

/**
 * @brief   Returns the symbol rate of the given satellite transponder
 * @param   t_ptr satellite transponder
 * @return  symbol rate, 0 if transport isn't DVB-S
 */
U16BIT ADB_GetTransportSatelliteSymbolRate(void *t_ptr)
{
   U16BIT sym_rate;
   ADB_TRANSPORT_REC *trans;

   FUNCTION_START(ADB_GetTransportSatelliteSymbolRate);

   sym_rate = 0;

   if (t_ptr != NULL)
   {
      trans = (ADB_TRANSPORT_REC *)t_ptr;

      DBDEF_RequestAccess();
      if (trans->sig_type == SIGNAL_QPSK)
      {
         sym_rate = (U16BIT)trans->u.sat.symbol_rate;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetTransportSatelliteSymbolRate);

   return(sym_rate);
}

/**
 * @brief   Returns the settings to tune to the given DVB-T/T2 transport
 * @param   t_ptr transport to get the tuning parameters for
 * @param   terr_type pointer to return the terrestrial type, TERR_TYPE_UNKNOWN on error
 * @param   freq_hz pointer to return the tuning frequency in Hz, 0 on error
 * @param   mode pointer to return the terrestrial mode, MODE_COFDM_UNDEFINED on error
 * @param   bwidth pointer to return the bandwidth
 * @param   plp_id pointer to return the PLP value for DVB-T2, will be 0 for DVB-T
 */
void ADB_GetTransportTerrestrialTuningParams(void *t_ptr, E_STB_DP_TTYPE *terr_type, U32BIT *freq_hz,
   E_STB_DP_TMODE *mode, E_STB_DP_TBWIDTH *bwidth, U8BIT *plp_id)
{
   ADB_TRANSPORT_REC *trans_rec;

   FUNCTION_START(ADB_GetTransportTerrestrialTuningParams);

   DBDEF_RequestAccess();

   if (t_ptr != NULL)
   {
      trans_rec = (ADB_TRANSPORT_REC *)t_ptr;

      *terr_type = trans_rec->u.terr.terr_type;
      *freq_hz = trans_rec->frequency;
      *mode = trans_rec->u.terr.tmode;
      *bwidth = trans_rec->u.terr.bwidth;

      if (trans_rec->u.terr.terr_type == TERR_TYPE_DVBT2)
      {
         *plp_id = trans_rec->u.terr.plp_id;
      }
      else
      {
         *plp_id = 0;
      }
   }
   else
   {
      *terr_type = TERR_TYPE_UNKNOWN;
      *freq_hz = 0;
      *mode = MODE_COFDM_UNDEFINED;
      *bwidth = TBWIDTH_8MHZ;
      *plp_id = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportTerrestrialTuningParams);
}

/**
 * @brief   Returns the settings to tune to the given terrestrial analog transport
 * @param   t_ptr transport to get the tuning parameters for
 * @param   freq_hz pointer to return the tuning frequency in Hz, 0 on error
 * @param   offset pointer to return offset value, 0 on error
 * @param   vtype pointer to return the video type
 */
void ADB_GetTransportAnalogTuningParams(void *t_ptr, U32BIT *freq_hz, S8BIT *offset,
   E_STB_DP_ANALOG_VIDEO_TYPE *vtype)
{
   FUNCTION_START(ADB_GetTransportAnalogTuningParams);

   DBDEF_RequestAccess();

   if (t_ptr != NULL)
   {
      *freq_hz = ((ADB_TRANSPORT_REC *)t_ptr)->frequency;
      *offset = ((ADB_TRANSPORT_REC *)t_ptr)->u.anal.freq_offset;
      *vtype = ((ADB_TRANSPORT_REC *)t_ptr)->u.anal.vtype;
   }
   else
   {
      *freq_hz = 0;
      *offset = 0;
      *vtype = ANLG_VIDEO_PAL_I;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportAnalogTuningParams);
}

/**
 * @brief   Returns the parameters needed to tune a cable tuner to a transport
 * @param   t_ptr transport to tune to
 * @param   freq_hz pointer to the returned freq of the transport
 * @param   mode pointer to the returned cable mode of the transport
 * @param   symrate pointer to the returned symbol rate of the transport
 */
void ADB_GetTransportCableTuningParams(void *t_ptr, U32BIT *freq_hz, E_STB_DP_CMODE *mode,
   U16BIT *symrate)
{
   ADB_TRANSPORT_REC *trans_rec;

   FUNCTION_START(ADB_GetTransportCableTuningParams);

   DBDEF_RequestAccess();

   if (t_ptr != NULL)
   {
      trans_rec = (ADB_TRANSPORT_REC *)t_ptr;

      *freq_hz = trans_rec->frequency;
      *mode = trans_rec->u.cab.cmode;
      *symrate = (U16BIT)trans_rec->u.cab.symbol_rate;
   }
   else
   {
      *freq_hz = 0;
      *mode = MODE_QAM_AUTO;
      *symrate = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportCableTuningParams);
}

/**
 * @brief   Returns the parameters needed to tune a satellite tuner to a transport
 * @param   t_ptr transport to tune to
 * @param   freq_hz pointer to the returned freq of the transport
 * @param   polarity pointer to the returned polarity of the transport
 * @param   symrate pointer to the returned symbol rate of the transport
 * @param   fec pointer to the returned FEC of the transport
 * @param   dvb_s2 returned as TRUE if a DVB_S2 transport
 * @param   modulation pointer to returned modulation used by the transport
 */
void ADB_GetTransportSatTuningParams(void *t_ptr, U32BIT *freq_hz, E_STB_DP_POLARITY *polarity,
   U16BIT *symrate, E_STB_DP_FEC *fec, BOOLEAN *dvb_s2, E_STB_DP_MODULATION *modulation)
{
   ADB_TRANSPORT_REC *trans_rec;

   FUNCTION_START(ADB_GetTransportSatTuningParams);

   DBDEF_RequestAccess();

   if (t_ptr != NULL)
   {
      trans_rec = (ADB_TRANSPORT_REC *)t_ptr;

      *freq_hz = trans_rec->frequency;
      *polarity = trans_rec->u.sat.polarity;
      *symrate = trans_rec->u.sat.symbol_rate;
      *fec = trans_rec->u.sat.fec_code;
      *dvb_s2 = trans_rec->u.sat.dvb_s2;
      *modulation = trans_rec->u.sat.modulation;
   }
   else
   {
      *freq_hz = 0;
      *polarity = POLARITY_HORIZONTAL;
      *symrate = 0;
      *fec = FEC_AUTOMATIC;
      *dvb_s2 = FALSE;
      *modulation = MOD_QPSK;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportSatTuningParams);
}

/**
 * @brief   Returns the parent satellite for the given transport
 * @param   t_ptr transport
 * @return  pointer to satellite
 */
void* ADB_GetTransportSatellite(void *t_ptr)
{
   ADB_NETWORK_REC *n_ptr;
   void *sat_ptr;

   FUNCTION_START(ADB_GetTransportSatellite);

   DBDEF_RequestAccess();

   sat_ptr = NULL;

   if (t_ptr != NULL)
   {
      if ((n_ptr = ((ADB_TRANSPORT_REC *)t_ptr)->network) != NULL)
      {
         sat_ptr = n_ptr->satellite;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportSatellite);

   return(sat_ptr);
}

/**
 * @brief   Finds the transport with the given ids
 * @param   net_id network id, can be ADB_INVALID_DVB_ID
 * @param   onet_id original network id
 * @param   tran_id transport id
 * @return  transport, or NULL if not found
 */
void* ADB_GetTransportFromIds(U16BIT net_id, U16BIT onet_id, U16BIT tran_id)
{
   void *t_ptr;

   FUNCTION_START(ADB_GetTransportFromIds);

   DBDEF_RequestAccess();
   t_ptr = DBDEF_FindTransportRecByIds(NULL, net_id, onet_id, tran_id);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportFromIds);
   return(t_ptr);
}

/**
 * @brief   Sets the given transport as the one tuned to on the given decode path.
 *          The transport's network is also set as the tuned network, if valid
 * @param   path decode path
 * @param   t_ptr transport
 */
void ADB_SetTunedTransport(U8BIT path, void *t_ptr)
{
   FUNCTION_START(ADB_SetTunedTransport);

   DBDEF_RequestAccess();

   DBDEF_SetTunedTransport(path, t_ptr);

   if (t_ptr != NULL)
   {
      DBDEF_SetTunedNetwork(path, ((ADB_TRANSPORT_REC *)t_ptr)->network);
   }
   else
   {
      DBDEF_SetTunedNetwork(path, NULL);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetTunedTransport);
}

/**
 * @brief   Returns the transport that's tuned to on the given decode path
 * @param   path decode path
 * @return  transport, NULL if not tuned
 */
void* ADB_GetTunedTransport(U8BIT path)
{
   void *t_ptr;

   FUNCTION_START(ADB_GetTunedTransport);

   DBDEF_RequestAccess();
   t_ptr = (void *)DBDEF_GetTunedTransport(path);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTunedTransport);

   return(t_ptr);
}

#if 0
/**
 * @brief   Creates a new analog service in the database
 * @return  TRUE on success, FALSE otherwise
 */
BOOLEAN ADB_AddAnalogService(void)
{
   BOOLEAN success;

   FUNCTION_START(ADB_AddAnalogService);

   DBDEF_RequestAccess();
   success = DBDEF_AddAnalogService();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddAnalogService);
   return(success);
}

#endif

/**
 * @brief   Add service record defined by the given delivery system descriptor.
 *          If the DSD defines a transport that doesn't currently exist, then one
 *          will be created, and if a service_id is given and a service with this ID
 *          doesn't exist on the transport, then one will be created, but will be
 *          marked as hidden and non-selectable so it won't appear to a user.
 * @param   data Sequence of bytes representing a system delivery descriptor
 *          as defined in EN300468 section 6.2.13
 * @param   service_id service ID
 * @return  service, or NULL if service not found or added
 */
void *ADB_AddServiceUsingDsd(U8BIT *data, U16BIT service_id)
{
   SI_DELIVERY_SYS_DESC_TYPE dsd_type;
   SI_DELIVERY_SYS_DESC *dsd;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr = NULL;

   FUNCTION_START(ADB_AddServiceUsingDsd);

   if (service_id != 0 && service_id != ADB_INVALID_DVB_ID &&
      STB_SIParseDelSysDesc(data, &dsd_type, &dsd))
   {
      DBDEF_RequestAccess();

      switch (dsd_type)
      {
         case SI_DEL_SYS_DESC_TYPE_TERR:
         {
            if (dsd->terr.is_t2)
            {
               if ((dsd->terr.u.t2.num_cells > 0) &&
                   (dsd->terr.u.t2.cell[0].num_freqs > 0))
               {
                  if ((t_ptr = DBDEF_FindTerrestrialTransportRec(dsd->terr.u.t2.cell[0].freq_hz[0], dsd->terr.u.t2.plp_id)) == NULL)
                  {
                     t_ptr = DBDEF_AddTerrestrialTransportRec(dsd->terr.u.t2.cell[0].freq_hz[0], dsd->terr.u.t2.plp_id, NULL);
                  }
               }
            }
            else
            {
               if ((t_ptr = DBDEF_FindTerrestrialTransportRec(dsd->terr.u.t1.freq_hz, 0)) == NULL)
               {
                  t_ptr = DBDEF_AddTerrestrialTransportRec(dsd->terr.u.t1.freq_hz, 0, NULL);
               }
            }
            break;
         }

         case SI_DEL_SYS_DESC_TYPE_CABLE:
         {
            if ((t_ptr = DBDEF_FindCableTransportRec(dsd->cable.freq_hz,
                       dsd->cable.symbol_rate)) == NULL)
            {
               t_ptr = DBDEF_AddCableTransportRec(dsd->cable.freq_hz, dsd->cable.symbol_rate, NULL);
            }
            break;
         }

         case SI_DEL_SYS_DESC_TYPE_SAT:
         {
            if ((t_ptr = DBDEF_FindSatTransportRec(dsd->sat.freq_hz, dsd->sat.sym_rate,
                       dsd->sat.polarity, dsd->sat.dvb_s2, dsd->sat.modulation, NULL)) == NULL)
            {
               t_ptr = DBDEF_AddSatTransportRec(dsd->sat.freq_hz, dsd->sat.sym_rate,
                     dsd->sat.polarity, dsd->sat.dvb_s2, dsd->sat.modulation, NULL);
            }
            break;
         }

         default:
         {
            t_ptr = NULL;
            break;
         }
      }
      STB_SIReleaseDelSysDesc(dsd, dsd_type);

      if (t_ptr != NULL)
      {
         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                       t_ptr->tran_id, service_id);
         if (s_ptr == NULL)
         {
            /* Create a service record */
            s_ptr = DBDEF_AddServiceRec(service_id, t_ptr);
            if (s_ptr != NULL)
            {
               /* Make the service hidden and non-selectable so it isn't available to the user */
               s_ptr->hidden = TRUE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

               s_ptr->selectable = FALSE;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

               s_ptr->serv_type = ADB_SERVICE_TYPE_DSD;
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

               ADB_SaveDatabase();
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_AddServiceUsingDsd);

   return((void *)s_ptr);
}

/**
 * @brief   Returns the total number of services in the database
 * @return  number of services
 */
U16BIT ADB_GetNumServices(void)
{
   U16BIT num_serv;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_GetNumServices);

   DBDEF_RequestAccess();

   s_ptr = DBDEF_GetNextServiceRec(NULL);
   for (num_serv = 0; s_ptr != NULL; s_ptr = DBDEF_GetNextServiceRec(s_ptr))
   {
      if (s_ptr->found && DBDEF_ServiceInProfile(s_ptr))
      {
         num_serv++;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumServices);

   return(num_serv);
}

/**
 * @brief   Returns the number of services in the database that would be returned
 *          with the given list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   inc_hidden TRUE if hidden services should be included in the returned count
 * @return  number of services
 */
U16BIT ADB_GetNumServicesInList(U32BIT list_type, BOOLEAN inc_hidden)
{
   U16BIT num_serv;
   ADB_SERVICE_REC *s_ptr;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_GetNumServicesInList);

   num_serv = 0;

   DBDEF_RequestAccess();

   if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
   {
      if ((fav_list = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
      {
         num_serv = DBDEF_GetNumServicesInFavouriteList(fav_list);
      }
   }
   else
   {
      s_ptr = DBDEF_GetNextServiceRec(NULL);
      while (s_ptr != NULL)
      {
         if (CheckServiceInListType(s_ptr, list_type, inc_hidden, FALSE) == TRUE)
         {
            num_serv++;
         }
         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumServicesInList);

   return(num_serv);
}

/**
 * @brief   Allocates and returns a list of all services in the database for the
 *          given list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetServiceList(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr)
{
   FUNCTION_START(ADB_GetServiceList);

   GetServiceList(list_type, slist_ptr, num_entries_ptr, FALSE, FALSE);

   FUNCTION_FINISH(ADB_GetServiceList);
}

/**
 * @brief   Allocates and returns a list of all services in the database for the
 *          given list type, including hidden services
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 * @param   ignore_selectable defines whether the returned service list should include services
 *          that are signalled as being unselectable. Use TRUE to ignore a service's selectable flag
 */
void ADB_GetServiceListIncludingHidden(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr,
   BOOLEAN ignore_selectable)
{
   FUNCTION_START(ADB_GetServiceListIncludingHidden);

   GetServiceList(list_type, slist_ptr, num_entries_ptr, TRUE, ignore_selectable);

   FUNCTION_FINISH(ADB_GetServiceListIncludingHidden);
}

/**
 * @brief   Returns the number of services marked as deleted with the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @return  number of services marked as deleted
 */
U16BIT ADB_GetNumDeletedServices(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   U16BIT num_services;

   FUNCTION_START(ADB_GetNumDeletedServices);

   DBDEF_RequestAccess();
   num_services = DBDEF_GetNumDeletedServices(tuner_type);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumDeletedServices);

   return(num_services);
}

/**
 * @brief   Returns a list of services that have been marked as 'deleted' for the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @param   slist_ptr pointer to an array that will be allocated, value will be NULL on return if
 *                    there are no services. The list must be freed using ADB_ReleaseServiceList
 * @return  number of services in the returned list
 */
U16BIT ADB_GetDeletedServiceList(E_STB_DP_SIGNAL_TYPE tuner_type, void ***slist_ptr)
{
   U16BIT num_services;

   FUNCTION_START(ADB_GetDeletedServiceList)

   DBDEF_RequestAccess();
   num_services = DBDEF_GetDeletedServiceList(tuner_type, slist_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetDeletedServiceList)

   return(num_services);
}

/**
 * @brief   Allocates and returns a list of all services in the database on the given transport
 * @param   t_ptr transport to be queried
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetTransportServiceList(void *t_ptr, void ***slist_ptr, U16BIT *num_entries_ptr)
{
   void **slist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetTransportServiceList);

   DBDEF_RequestAccess();

   num_entries = 0;

   if ((max_no_entries = DBDEF_GetNumServices()) > 0)
   {
      slist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (slist != NULL)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         for (i = 0; ((i < max_no_entries) && (s_ptr != NULL)); i++)
         {
            if (s_ptr->transport == t_ptr)
            {
               slist[num_entries] = (void *)s_ptr;
               num_entries++;
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(slist);
            slist = NULL;
         }
      }

      *slist_ptr = slist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTransportServiceList);
}

/**
 * @brief   Allocates and returns a list of all services in the database on all
 *          transports on the given network
 * @param   n_ptr network to be queried
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetNetworkServiceList(void *n_ptr, void ***slist_ptr, U16BIT *num_entries_ptr)
{
   void **slist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetNetworkServiceList);

   DBDEF_RequestAccess();

   num_entries = 0;

   if ((max_no_entries = DBDEF_GetNumServices()) > 0)
   {
      slist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (slist != NULL)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         for (i = 0; ((i < max_no_entries) && (s_ptr != NULL)); i++)
         {
            t_ptr = s_ptr->transport;
            if (t_ptr != NULL)
            {
               if (t_ptr->network == n_ptr)
               {
                  slist[num_entries] = (void *)s_ptr;
                  num_entries++;
               }
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(slist);
            slist = NULL;
         }
      }

      *slist_ptr = slist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNetworkServiceList);
}

/**
 * @brief   Allocates and returns a list of all services that are marked as being unavailable.
 *          A service becomes unavailable when it disappears from the SDT.
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetUnavailableServiceList(void ***slist_ptr, U16BIT *num_entries_ptr)
{
   void **slist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetUnavailableServiceList);

   DBDEF_RequestAccess();

   num_entries = 0;

   if ((max_no_entries = DBDEF_GetNumServices()) > 0)
   {
      slist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (slist != NULL)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         for (i = 0; ((i < max_no_entries) && (s_ptr != NULL)); i++)
         {
            if (s_ptr->unavailable == TRUE)
            {
               slist[num_entries] = (void *)s_ptr;
               num_entries++;
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(slist);
            slist = NULL;
         }
      }

      *slist_ptr = slist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetUnavailableServiceList);
}

/**
 * @brief   Allocates and returns a list of all services that are marked as locked
 *          for parental control purposes
 * @param   slist_ptr pointer to an array that will be allocated, value will
 *          be NULL on return if there are no services
 * @param   num_entries_ptr pointer to the number of items in the returned array,
 *          value will be 0 on return if there are no services
 */
void ADB_GetLockedServiceList(void ***slist_ptr, U16BIT *num_entries_ptr)
{
   void **slist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetLockedServiceList);

   DBDEF_RequestAccess();

   num_entries = 0;

   if ((max_no_entries = DBDEF_GetNumServices()) > 0)
   {
      slist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (slist != NULL)
      {
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         for (i = 0; ((i < max_no_entries) && (s_ptr != NULL)); i++)
         {
            if (s_ptr->locked)
            {
               slist[num_entries] = (void *)s_ptr;
               num_entries++;
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(slist);
            slist = NULL;
         }
      }

      *slist_ptr = slist;
   }

   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetLockedServiceList);
}

/**
 * @brief   Frees a list of services returned from one of the functions that
 *          returns a service list, such as ADB_GetServiceList
 * @param   slist service list to be freed
 * @param   num_servs number of services in the list
 */
void ADB_ReleaseServiceList(void **slist, U16BIT num_servs)
{
   FUNCTION_START(ADB_ReleaseServiceList);

   USE_UNWANTED_PARAM(num_servs);

   if (slist != NULL)
   {
      STB_AppFreeMemory(slist);
   }

   FUNCTION_FINISH(ADB_ReleaseServiceList);
}

/**
 * @brief   Creates a subset of locked services from the supplied service list. Memory is allocated
 *          as TempMemory so will be released when the calling screen closes
 * @param   slist pointer to an array of service record pointers
 * @param   num_entries number of entries in the list
 * @param   locked_slist_ptr address for the return of the locked slist
 * @param   locked_num_entries_ptr address for the return of the number of entries in the locked list
 */
void ADB_GetServiceListLockedSubset(void **slist, U16BIT num_entries,
   void ***locked_slist_ptr, U16BIT *locked_num_entries_ptr)
{
   void **new_slist;
   U16BIT new_num_entries;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListLockedSubset);

   DBDEF_RequestAccess();

   if (slist != NULL)
   {
      new_slist = STB_AppGetMemory(num_entries * sizeof(void *));
      new_num_entries = 0;
      if (new_slist != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            s_ptr = slist[i];
            if (s_ptr != NULL)
            {
               if (s_ptr->locked)
               {
                  new_slist[new_num_entries] = (void *)s_ptr;
                  new_num_entries++;
               }
            }
         }

         // if there are no locked entries free the new list and return NULL
         if (new_num_entries == 0)
         {
            STB_AppFreeMemory(new_slist);
            new_slist = NULL;
         }
      }

      *locked_slist_ptr = new_slist;
      *locked_num_entries_ptr = new_num_entries;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceListLockedSubset);
}

/**
 * @brief   Sorts the given list of services into alphabetical order. Case is significant
 * @param   slist service list to be sorted
 * @param   num_entries number of services in the list
 * @param   short_name TRUE if the short name for a service should be used
 */
void ADB_SortServiceListAlphabetically(void **slist, U16BIT num_entries, BOOLEAN short_name)
{
   U16BIT i, j;
   U8BIT *name1;
   U8BIT *name2;
   void *tmp;

   FUNCTION_START(ADB_SortServiceListAlphabetically);

   if (slist != NULL)
   {
      for (i = 0; i < num_entries - 1; i++)
      {
         for (j = i + 1; j < num_entries; j++)
         {
            if (short_name)
            {
               name1 = ADB_GetServiceShortName(slist[i], FALSE);
               name2 = ADB_GetServiceShortName(slist[j], FALSE);
            }
            else
            {
               name1 = ADB_GetServiceFullName(slist[i], FALSE);
               name2 = ADB_GetServiceFullName(slist[j], FALSE);
            }

            if (STB_CompareUnicodeStrings(name1, name2, TRUE, FALSE) > 0)
            {
               /* Swap items */
               tmp = slist[i];
               slist[i] = slist[j];
               slist[j] = tmp;
            }
         }
      }
   }

   FUNCTION_FINISH(ADB_SortServiceListAlphabetically);
}

/**
 * @brief   Checks whether the given service is in the current service list
 * @param   serv_ptr service to be checked
 * @return  TRUE if the service is found, FALSE otherwise
 */
BOOLEAN ADB_IsValidService(void *serv_ptr)
{
   BOOLEAN retval;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_IsValidService);

   retval = FALSE;

   DBDEF_RequestAccess();
   s_ptr = DBDEF_GetNextServiceRec(NULL);
   while (!retval && (s_ptr != NULL))
   {
      if (s_ptr == serv_ptr)
      {
         retval = TRUE;
      }
      else
      {
         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_IsValidService);

   return(retval);
}

/**
 * @brief   Returns the signal type for transport of given service. It also indicates
 *          the mode - i.e. whether it is T2, S2 or C2
 * @param   s_ptr service
 * @param   is_sig2 return signal mode, TRUE means T2, S2 or C2 (depending on signal type)
 * @return  signal type
 */
E_STB_DP_SIGNAL_TYPE ADB_GetServiceSignalType(void *s_ptr, BOOLEAN *is_sig2)
{
   E_STB_DP_SIGNAL_TYPE result;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ADB_GetServiceSignalType);

   result = SIGNAL_NONE;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      t_ptr = ((ADB_SERVICE_REC *)s_ptr)->transport;
      if (t_ptr != NULL)
      {
         result = t_ptr->sig_type;
         switch (t_ptr->sig_type)
         {
            case SIGNAL_QPSK:
            {
               *is_sig2 = t_ptr->u.sat.dvb_s2;
               break;
            }
            case SIGNAL_COFDM:
            {
               *is_sig2 = (t_ptr->u.terr.terr_type == TERR_TYPE_DVBT2)? TRUE : FALSE;
               break;
            }
            case SIGNAL_QAM:
            {
               *is_sig2 = FALSE;
               break;
            }
            default:
            {
               *is_sig2 = FALSE;
            }
         }
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceSignalType);

   return result;
}

/**
 * @brief   Returns the next service, in LCN order, for the given list type, starting
 *          from the given service. If serv is NULL then the first service is returned.
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   serv starting service, NULL to get the first service
 * @return  next service, or NULL if no more services
 */
void* ADB_GetNextServiceInList(U32BIT list_type, void *serv)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_GetNextServiceInList);

   DBDEF_RequestAccess();

   s_ptr = NULL;
   fav_list = NULL;
   fav_serv = NULL;

   // find next service and check it is in the required list]
   if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
   {
      if ((fav_list = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
      {
         /* The service may be NULL, in which case the favourite service will also be NULL
          * and this is ok because it means to get the first service in the list */
         fav_serv = DBDEF_FindServiceInFavouriteList(fav_list, serv);

         if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
      }
   }
   else
   {
      s_ptr = DBDEF_GetNextServiceRec(serv);
   }

   while (s_ptr != NULL)
   {
      if (CheckServiceInListType(s_ptr, list_type, FALSE, FALSE) == TRUE)
      {
         if (ACFG_IsNordigCountry())
         {
            if (HasNordigSimulcastService(s_ptr) == FALSE)
            {
               break;
            }
         }
         else
         {
            break;
         }
      }

      if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
      {
         if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
         else
         {
            s_ptr = NULL;
         }
      }
      else
      {
         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNextServiceInList);

   return((void *)s_ptr);
}

/**
 * @brief   Returns the previous service, in LCN order, for the given list type, starting
 *          from the given service. If serv is NULL then the last service is returned.
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   serv starting service, NULL to get the last service
 * @return  previous service, or NULL if no more services
 */
void* ADB_GetPrevServiceInList(U32BIT list_type, void *serv)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_GetPrevServiceInList);

   DBDEF_RequestAccess();

   s_ptr = NULL;
   fav_list = NULL;
   fav_serv = NULL;

   // find next service and check it is in the required list
   if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
   {
      if ((fav_list = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
      {
         /* The service may be NULL, in which case the favourite service will also be NULL
          * and this is ok because it means to get the first service in the list */
         fav_serv = DBDEF_FindServiceInFavouriteList(fav_list, serv);

         if ((fav_serv = DBDEF_GetPrevServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
      }
   }
   else
   {
      s_ptr = DBDEF_GetPrevServiceRec(serv);
   }

   while (s_ptr != NULL)
   {
      if (CheckServiceInListType(s_ptr, list_type, FALSE, FALSE) == TRUE)
      {
         if (ACFG_IsNordigCountry())
         {
            if (HasNordigSimulcastService(s_ptr) == FALSE)
            {
               break;
            }
         }
         else
         {
            break;
         }
      }

      if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
      {
         if ((fav_serv = DBDEF_GetPrevServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
         else
         {
            s_ptr = NULL;
         }
      }
      else
      {
         s_ptr = DBDEF_GetPrevServiceRec(s_ptr);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetPrevServiceInList);

   return((void *)s_ptr);
}

/**
 * @brief   Returns the service matching the given LCN and list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   lcn locgical channel number
 * @param   show_unselectable used by MHEG to allow it to find services that
 *          would normally not be selectable by a user
 * @return  service, or NULL if not found
 */
void* ADB_FindServiceByLcn(U32BIT list_type, U16BIT lcn, BOOLEAN show_unselectable)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_FindServiceByLcn);

   DBDEF_RequestAccess();

   s_ptr = NULL;
   fav_list = NULL;
   fav_serv = NULL;

   // find required lcn
   if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
   {
      if ((fav_list = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
      {
         if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, NULL)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
      }
   }
   else
   {
      s_ptr = DBDEF_GetNextServiceRec(NULL);
   }

   while (s_ptr != NULL)
   {
      if (ACFG_IsNordigCountry())
      {
         /* for Nordig, we need to check the service type as well */
         if ((s_ptr->allocated_lcn == lcn) &&
             ((list_type == ADB_SERVICE_LIST_ALL) || (list_type == s_ptr->serv_type) ||
              ((list_type & ADB_FAV_SERVICE_LIST_TYPE) != 0) ||
              ((list_type == ADB_SERVICE_LIST_TV) && ((s_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
                                                      (s_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
                                                      (s_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
                                                      (s_ptr->serv_type == ADB_SERVICE_TYPE_UHD_TV))) ||
              ((list_type == ADB_SERVICE_LIST_RADIO) && (s_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO))))
         {
            if (HasNordigSimulcastService(s_ptr) == FALSE)
            {
               break;
            }
         }
      }
      else if (s_ptr->allocated_lcn == lcn)
      {
         break;
      }

      if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
      {
         if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
         {
            s_ptr = fav_serv->serv_ptr;
         }
         else
         {
            s_ptr = NULL;
         }
      }
      else
      {
         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }
   }

   // check service matching reqd lcn is in the required list
   if (CheckServiceInListType(s_ptr, list_type, TRUE, show_unselectable) == FALSE)
   {
      s_ptr = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_FindServiceByLcn);

   return((void *)s_ptr);
}

/**
 * @brief   Returns the service matching the given, or preceding, LCN and list type
 * @param   list_type defines the types of services to be included,
 *          multiple types can be combined by ORing the values
 * @param   lcn logical channel number
 * @return  service, or NULL if not found
 */
void* ADB_FindServiceByNearestLcn(U32BIT list_type, U16BIT lcn)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *last_valid_s_ptr;

   FUNCTION_START(ADB_FindServiceByNearestLcn);

   last_valid_s_ptr = NULL;

   DBDEF_RequestAccess();

   s_ptr = DBDEF_GetPrevServiceRec(NULL);
   while (s_ptr != NULL)
   {
      if (CheckServiceInListType(s_ptr, list_type, FALSE, FALSE) == TRUE)
      {
         last_valid_s_ptr = s_ptr;
         if (s_ptr->allocated_lcn <= lcn)
         {
            break; //break out of the while loop
         }
      }

      s_ptr = DBDEF_GetPrevServiceRec(s_ptr);
   }

   DBDEF_ReleaseAccess();

   if (s_ptr == NULL)
   {
      s_ptr = last_valid_s_ptr;
   }

   FUNCTION_FINISH(ADB_FindServiceByNearestLcn);

   return((void *)s_ptr);
}

/**
 * @brief   Returns a pointer to the service matching the given IDs.
 * @param   onet_id original network id
 * @param   tid transport id
 * @param   sid service id
 * @return  service pointer, or NULL if not found
 */
void* ADB_FindServiceByIds(U16BIT onet_id, U16BIT tid, U16BIT sid)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_FindServiceByIds);

   DBDEF_RequestAccess();

   s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, onet_id, tid, sid);

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_FindServiceByIds);

   return((void *)s_ptr);
}

/**
 * @brief   Returns the index into the given list of services of the service with the given LCN
 * @param   lcn logical channel number to search for
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindLcnInServiceList(U16BIT lcn, void **slist_ptr, U16BIT num_entries)
{
   U16BIT i;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_FindLcnInServiceList);

   if ((slist_ptr == NULL) || (num_entries == 0) || (lcn == 0))
   {
      i = num_entries;
   }
   else
   {
      DBDEF_RequestAccess();
      for (i = 0; i < num_entries; i++)
      {
         s_ptr = (ADB_SERVICE_REC *)slist_ptr[i];
         if (s_ptr->allocated_lcn == lcn)
         {
            break;
         }
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_FindLcnInServiceList);

   return(i);
}

/**
 * @brief   Returns the index in the given list of services of the service matching
 *          the given, or preceding, LCN
 * @param   lcn logical channel number
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindNearestLcnInServiceList(U16BIT lcn, void **slist_ptr, U16BIT num_entries)
{
   U16BIT i;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_FindNearestLcnInServiceList);

   if ((slist_ptr == NULL) || (num_entries == 0) || (lcn == 0))
   {
      i = 0;
   }
   else
   {
      DBDEF_RequestAccess();
      for (i = num_entries - 1; i > 0; i--)
      {
         s_ptr = (ADB_SERVICE_REC *)(*(slist_ptr + i));
         if (s_ptr->allocated_lcn <= lcn)
         {
            break;
         }
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_FindNearestLcnInServiceList);

   return(i);
}

/**
 * @brief   Returns the index in the given list of services of the given service
 * @param   s_ptr service to search for
 * @param   slist_ptr list of services to be searched
 * @param   num_entries number of services in the list
 * @return  index of the service, or num_entries if service isn't found
 */
U16BIT ADB_FindServiceInList(void *s_ptr, void **slist_ptr, U16BIT num_entries)
{
   U16BIT i;

   FUNCTION_START(ADB_FindServiceInList);

   if ((slist_ptr == NULL) || (num_entries == 0) || (s_ptr == NULL))
   {
      i = num_entries;
   }
   else
   {
      for (i = 0; i < num_entries; i++)
      {
         if (s_ptr == slist_ptr[i])
         {
            break;
         }
      }
   }

   FUNCTION_FINISH(ADB_FindServiceInList);

   return(i);
}

/**
 * @brief   Allocates and returns a list of the fullnames for all the services in the given service
 *          list. The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   use_pref_name TRUE if the preferred names are to be returned.
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListFullNames(void **slist, U16BIT num_entries, BOOLEAN use_pref_names)
{
   ADB_STRING *name_str;
   U16BIT nchar;
   U8BIT **names_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListFullNames);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            name_str = DBDEF_GetServiceName((ADB_SERVICE_REC *)slist[i], FALSE, use_pref_names);
            if (name_str != NULL)
            {
               names_array[i] = STB_ConvertStringToUTF8(name_str->str_ptr, &nchar, FALSE,
                     name_str->lang_code);
            }
            else
            {
               names_array[i] = NULL;
            }
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListFullNames);

   return(names_array);
}

/**
 * @brief   Allocates and returns a list of the short names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   use_pref_name TRUE if the preferred names are to be returned.
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListShortNames(void **slist, U16BIT num_entries, BOOLEAN use_pref_names)
{
   ADB_STRING *name_str;
   U8BIT **names_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListShortNames);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            name_str = DBDEF_GetServiceName((ADB_SERVICE_REC *)slist[i], TRUE, use_pref_names);
            if (name_str != NULL)
            {
               names_array[i] = ExtractShortName(name_str);
            }
            else
            {
               names_array[i] = NULL;
            }
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListShortNames);

   return(names_array);
}

/**
 * @brief   Allocates and returns an array of the logical channel numbers that
 *          have been assigned to all the services in the given service list.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of LCNs, NULL if list is empty or memory allocation fails
 */
U16BIT* ADB_GetServiceListLcns(void **slist, U16BIT num_entries)
{
   U16BIT *lcn_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListLcns);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      lcn_array = STB_AppGetMemory(num_entries * sizeof(U16BIT));
      if (lcn_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            lcn_array[i] = GetServiceLcn((ADB_SERVICE_REC *)(slist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      lcn_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListLcns);

   return(lcn_array);
}

/**
 * @brief   Allocates and returns an array of the logical channel numbers that
 *          each of the services in the given service list originally requested.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of LCNs, NULL if list is empty or memory allocation fails
 */
U16BIT* ADB_GetServiceListOrigLcns(void **slist, U16BIT num_entries)
{
   U16BIT *lcn_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListOrigLcns);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      lcn_array = STB_AppGetMemory(num_entries * sizeof(U16BIT));
      if (lcn_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            lcn_array[i] = ((ADB_SERVICE_REC *)slist[i])->serv_lcn;
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      lcn_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListOrigLcns);

   return(lcn_array);
}

#if 0
/**
 *

 *
 * @brief   Returns a list of SID's corresponding to the given list of service records.
 *                Memory is allocated as TempMemory so will be released when the calling screen
 *                closes
 *
 * @param   slist       - pointer to an array of service record pointers
 * @param   num_entries - number of entries in the list
 *
 * @return   pointer to the name string array
 *
 */
U32BIT* ADB_GetServiceListSIDs(void **slist, U16BIT num_entries)
{
   U32BIT *sid_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListSIDs);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      sid_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (sid_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            sid_array[i] = (U32BIT)GetServiceId((ADB_SERVICE_REC *)(slist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      sid_array = NULL;
   }
   FUNCTION_FINISH(ADB_GetServiceListSIDs);
   return(sid_array);
}

#endif

/**
 * @brief   Allocates and returns an array of the unavailable flags for each of
 *          the services in the given service list. A value of TRUE means the
 *          service is unavailable, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListUnavailableFlag(void **slist, U16BIT num_entries)
{
   U32BIT *unavail_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListUnavailableFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      unavail_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (unavail_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            unavail_array[i] = (U32BIT)GetServiceUnavailFlag((ADB_SERVICE_REC *)(slist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      unavail_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListUnavailableFlag);

   return(unavail_array);
}

/**
 * @brief   Allocates and returns an array of the 'new' flags for each of the
 *          services in the given service list. A service is marked as 'new' when
 *          it's added to the service database following an update service scan.
 *          A value of TRUE means the service is new, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListNewFlag(void **slist, U16BIT num_entries)
{
   U32BIT *new_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListNewFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      new_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (new_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            new_array[i] = (U32BIT)GetServiceNewFlag((ADB_SERVICE_REC *)(slist[i]));
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      new_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListNewFlag);

   return(new_array);
}

/**
 * @brief   Allocates and returns an array of the 'locked' flags for each of the
 *          services in the given service list. A service is marked as 'locked'
 *          due to setup of parental control.
 *          A value of TRUE means the service is locked, FALSE otherwise.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListLockedFlag(void **slist, U16BIT num_entries)
{
   U32BIT *lock_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListlockedFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      lock_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (lock_array != NULL)
      {
         for (i = 0; i != num_entries; i++)
         {
            lock_array[i] = (U32BIT)((ADB_SERVICE_REC *)slist[i])->locked;
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      lock_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListlockedFlag);

   return(lock_array);
}

/**
 * @brief   Allocates and returns an array of the scrambled flags for each of the
 *          services in the given service list. A value of TRUE means the service
 *          is signalled as scrambled and FALSE means it's free-to-air.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListScrambledFlag(void **slist, U16BIT num_entries)
{
   U32BIT *scrambled_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListScrambledFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      scrambled_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (scrambled_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            scrambled_array[i] = (U32BIT)(((ADB_SERVICE_REC *)slist[i])->scrambled);
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      scrambled_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListScrambledFlag);

   return(scrambled_array);
}

/**
 * @brief   Allocates and returns an array indicating whether each service in the
 *          given service list is an HD service. A value of TRUE means the service
 *          is signalled as being HD and FALSE means it's SD.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListHdFlag(void **slist, U16BIT num_entries)
{
   U32BIT *hd_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListHdFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      hd_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (hd_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            if ((((ADB_SERVICE_REC *)slist[i])->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
                (((ADB_SERVICE_REC *)slist[i])->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
                (((ADB_SERVICE_REC *)slist[i])->serv_type == ADB_SERVICE_TYPE_UHD_TV))
            {
               hd_array[i] = 1;
            }
            else
            {
               hd_array[i] = 0;
            }
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      hd_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListHdFlag);

   return(hd_array);
}

/**
 * @brief   Sets the locked state of all services in the given service list
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @param   locked TRUE if the services are to be locked, or FALSE to unlock
 */
void ADB_SetServiceListLockedFlag(void **slist, U16BIT num_entries, BOOLEAN locked)
{
   U16BIT i;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_SetServiceListLockedFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();

      for (i = 0; i != num_entries; i++)
      {
         s_ptr = (ADB_SERVICE_REC *)slist[i];

         s_ptr->locked = locked;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LOCKED, locked);
         DBA_SaveRecord(s_ptr->dba_rec);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceListLockedFlag);
}

/**
 * @brief   Allocates and returns an array indicating whether each service in the
 *          given service list is signalled as being hidden or not. A value of TRUE
 *          means the service is hidden and FALSE means it's visible.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListHiddenFlag(void **slist, U16BIT num_entries)
{
   U32BIT *hidden_array;
   U16BIT i;

   FUNCTION_START(ADB_GetServiceListHiddenFlag);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      hidden_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (hidden_array != NULL)
      {
         for (i = 0; i != num_entries; i++)
         {
            hidden_array[i] = (U32BIT)(((ADB_SERVICE_REC *)slist[i])->hidden);
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      hidden_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListHiddenFlag);

   return(hidden_array);
}

/**
 * @brief   Allocates and returns a list of network names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListNetworkNames(void **slist, U16BIT num_entries)
{
   U8BIT **names_array;
   U16BIT i;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ADB_GetServiceListNetworkNames);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            t_ptr = ((ADB_SERVICE_REC *)slist[i])->transport;
            if (t_ptr == NULL || t_ptr->network == NULL)
            {
               names_array[i] = NULL;
            }
            else
            {
               names_array[i] = GetNetworkName(t_ptr->network, FALSE);
            }
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListNetworkNames);

   return(names_array);
}

/**
 * @brief   Allocates and returns a list of transport names for all the services
 *          in the given service list.
 *          The returned list should be freed using ADB_ReleaseNameList.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of service names as UTF-8 strings
 */
U8BIT** ADB_GetServiceListTransportNames(void **slist, U16BIT num_entries)
{
   U8BIT **names_array;
   U16BIT i;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ADB_GetServiceListTransportNames);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      names_array = STB_AppGetMemory(num_entries * sizeof(U8BIT *));
      if (names_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            t_ptr = ((ADB_SERVICE_REC *)slist[i])->transport;

            names_array[i] = GetTransportName(t_ptr);
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      names_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListTransportNames);

   return(names_array);
}

/**
 * @brief   Allocates and returns an array of the signal strengths of the transports
 *          containing the given list of services.
 *          The returned list should be freed using STB_AppFreeMemory.
 * @param   slist list of services
 * @param   num_entries number of services in the list
 * @return  allocated array of flags, NULL if list is empty or memory allocation fails
 */
U32BIT* ADB_GetServiceListTransportStrengths(void **slist, U16BIT num_entries)
{
   U32BIT *tp_strength_array;
   U16BIT i;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ADB_GetServiceListTransportStrengths);

   if (slist != NULL)
   {
      DBDEF_RequestAccess();
      tp_strength_array = STB_AppGetMemory(num_entries * sizeof(U32BIT));
      if (tp_strength_array != NULL)
      {
         for (i = 0; i < num_entries; i++)
         {
            t_ptr = ((ADB_SERVICE_REC *)slist[i])->transport;
            if (t_ptr != NULL)
            {
               tp_strength_array[i] = (U32BIT)GetTransportTunedStrength(t_ptr);
            }
            else
            {
               tp_strength_array[i] = 0xff;
            }
         }
      }
      DBDEF_ReleaseAccess();
   }
   else
   {
      tp_strength_array = NULL;
   }

   FUNCTION_FINISH(ADB_GetServiceListTransportStrengths);

   return(tp_strength_array);
}

/**
 * @brief   Returns the full name of the given service as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   use_pref_name TRUE if the preferred name should be returned
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceFullName(void *s_ptr, BOOLEAN use_pref_name)
{
   ADB_STRING *name_str;
   U16BIT nchar;
   U8BIT *retval;

   FUNCTION_START(ADB_GetServiceFullName);

   retval = NULL;

   DBDEF_RequestAccess();

   if ((name_str = DBDEF_GetServiceName((ADB_SERVICE_REC *)s_ptr, FALSE, use_pref_name)) != NULL)
   {
      retval = STB_ConvertStringToUTF8(name_str->str_ptr, &nchar, FALSE, name_str->lang_code);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceFullName);

   return(retval);
}

/**
 * @brief   Returns the short name of the given service as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   use_pref_name TRUE if the preferred name should be returned
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceShortName(void *s_ptr, BOOLEAN use_pref_name)
{
   ADB_STRING *name_str;
   U8BIT *retval;

   FUNCTION_START(ADB_GetServiceShortName);

   retval = NULL;

   DBDEF_RequestAccess();

   if ((name_str = DBDEF_GetServiceName((ADB_SERVICE_REC *)s_ptr, TRUE, use_pref_name)) != NULL)
   {
      retval = ExtractShortName(name_str);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceShortName);

   return(retval);
}

/**
 * @brief   Returns the full name of the given service as a UTF-8 string using
 *          the given language and preferred name id, rather than the defaults.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language id
 * @param   pref_name_id preferred name id
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceFullNameByLangAndPrefId(void *s_ptr, U8BIT lang, U8BIT pref_name_id)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetServiceFullNameByLangAndPrefId);

   DBDEF_RequestAccess();
   name_str = GetServiceNameByLangAndPrefId((ADB_SERVICE_REC *)s_ptr, lang, pref_name_id, FALSE);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceFullNameByLangAndPrefId);

   return(name_str);
}

/**
 * @brief   Returns the short name of the given service as a UTF-8 string using
 *          the given language and preferred name id, rather than the defaults.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language id
 * @param   pref_name_id preferred name id
 * @return  pointer to UTF-8 string, or NULL
 */
U8BIT* ADB_GetServiceShortNameByLangAndPrefId(void *s_ptr, U8BIT lang, U8BIT pref_name_id)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetServiceShortNameByLangAndPrefId);

   DBDEF_RequestAccess();
   name_str = GetServiceNameByLangAndPrefId((ADB_SERVICE_REC *)s_ptr, lang, pref_name_id, TRUE);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceShortNameByLangAndPrefId);

   return(name_str);
}

/**
 * @brief   Returns the logical channel number assigned to the given service
 * @param   s_ptr service
 * @return  logical channel number
 */
U16BIT ADB_GetServiceLcn(void *s_ptr)
{
   U16BIT lcn_value;

   FUNCTION_START(ADB_GetServiceLcn);

   DBDEF_RequestAccess();
   lcn_value = GetServiceLcn((ADB_SERVICE_REC *)s_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceLcn);

   return(lcn_value);
}

/**
 * @brief   Returns the signalled service id of the given service
 * @param   s_ptr service
 * @return  service id
 */
U16BIT ADB_GetServiceId(void *s_ptr)
{
   U16BIT sid_value;

   FUNCTION_START(ADB_GetServiceId);

   DBDEF_RequestAccess();
   sid_value = GetServiceId((ADB_SERVICE_REC *)s_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceId);

   return(sid_value);
}

/**
 * @brief   Returns the original network id, transport id and service id for the given service
 * @param   s_ptr service handle
 * @param   onet_id pointer for the return of the original network id
 * @param   trans_id pointer for the return of the transport id
 * @param   serv_id pointer for the return of the service id
 */
void ADB_GetServiceIds(void *s_ptr, U16BIT *onet_id, U16BIT *trans_id, U16BIT *serv_id)
{
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ADB_GetServiceIds);

   DBDEF_RequestAccess();

   *serv_id = ((ADB_SERVICE_REC *)s_ptr)->serv_id;

   t_ptr = ((ADB_SERVICE_REC *)s_ptr)->transport;
   if (t_ptr != NULL)
   {
      *trans_id = t_ptr->tran_id;
      *onet_id = t_ptr->orig_net_id;
   }
   else
   {
      *trans_id = ADB_INVALID_DVB_ID;
      *onet_id = ADB_INVALID_DVB_ID;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceIds);
}

/**
 * @brief   Sets whether the EIT schedule events for this service are held in memory.
 *          All services hold schedule events in memory by default
 * @param   s_ptr service
 * @param   state FALSE to disable, TRUE to enable
 */
void ADB_SetServiceScheduleState(void *s_ptr, BOOLEAN state)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetServiceScheduleState);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      serv_ptr->sched_disabled = !state;
      DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_SCHED_DISABLED, serv_ptr->sched_disabled);

      if (serv_ptr->sched_disabled && (serv_ptr->event_schedule != NULL))
      {
         DBDEF_DeleteEventList(serv_ptr->event_schedule);
         serv_ptr->event_schedule = NULL;
         serv_ptr->num_events_in_schedule = 0;
         ASI_NotifyEitSchedUpdate(s_ptr, 0, APP_SI_EIT_JOURNAL_TYPE_CLEAR);
      }

      DBA_SaveRecord(serv_ptr->dba_rec);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceScheduleState);
}

/**
 * @brief   Returns whether the EIT schedule events are being held in memory for the service
 * @param   s_ptr service
 * @return  TRUE if data is being held, FALSE otherwise
 */
BOOLEAN ADB_GetServiceScheduleState(void *s_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(ADB_GetServiceScheduleState);

   retval = FALSE;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      retval = !((ADB_SERVICE_REC *)s_ptr)->sched_disabled;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceScheduleState);

   return(retval);
}

/**
 * @brief   Sets whether the now/next EIT events for this service are held in memory.
 *          All services save now/next events by default
 * @param   s_ptr service
 * @param   state FALSE to disable, TRUE to enable
 */
void ADB_SetServiceNowNextState(void *s_ptr, BOOLEAN state)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetServiceNowNextState);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      serv_ptr->now_next_disabled = !state;
      DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_NOWNEXT_DISABLED,
         serv_ptr->now_next_disabled);

      if (serv_ptr->now_next_disabled)
      {
         if (serv_ptr->now_event != NULL)
         {
            DBDEF_DeleteEventList(serv_ptr->now_event);
            serv_ptr->now_event = NULL;
         }

         if (serv_ptr->next_event != NULL)
         {
            DBDEF_DeleteEventList(serv_ptr->next_event);
            serv_ptr->next_event = NULL;
         }
      }

      DBA_SaveRecord(serv_ptr->dba_rec);

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceNowNextState);
}

/**
 * @brief   Returns whether the EIT now/next data is being held in memory for the service
 * @param   s_ptr service
 * @return  TRUE if data is being held, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNowNextState(void *s_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(ADB_GetServiceNowNextState);

   retval = FALSE;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      retval = !((ADB_SERVICE_REC *)s_ptr)->now_next_disabled;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceNowNextState);

   return(retval);
}

/**
 * @brief   Returns the logical channel number originally requested by the given service
 * @param   s_ptr service
 * @return  logical channel number
 */
U16BIT ADB_GetServiceOrigLcn(void *s_ptr)
{
   U16BIT orig_lcn;

   FUNCTION_START(ADB_GetServiceOrigLcn);

   orig_lcn = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      orig_lcn = ((ADB_SERVICE_REC *)s_ptr)->serv_lcn;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceOrigLcn);
   return(orig_lcn);
}

/**
 * @brief   Returns the PCR PID for the given service
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServicePCRPid(void *s_ptr)
{
   U16BIT pcr_pid;

   FUNCTION_START(ADB_GetServicePCRPid);

   pcr_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      pcr_pid = ((ADB_SERVICE_REC *)s_ptr)->pcr_pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServicePCRPid);
   return(pcr_pid);
}

/**
 * @brief   Returns the audio PID for the given service. This will be the one chosen
 *          based on the settings for language, etc, or the one explicitly set (e.g.
 *          due to user selection).
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceAudioPid(void *s_ptr)
{
   U16BIT a_pid;

   FUNCTION_START(ADB_GetServiceAudioPid);

   a_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      a_pid = ((ADB_SERVICE_REC *)s_ptr)->audio_pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceAudioPid);
   return(a_pid);
}

/**
 * @brief   Returns the audio type being used by the given service
 * @param   s_ptr service
 * @return  Audio type, defaults to ADB_AUDIO_STREAM (MPEG1)
 */
ADB_STREAM_TYPE ADB_GetServiceAudioType(void *s_ptr)
{
   ADB_STREAM_TYPE audio_type;

   FUNCTION_START(ADB_GetServiceAudioType);

   audio_type = ADB_AUDIO_STREAM;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      audio_type = ((ADB_SERVICE_REC *)s_ptr)->audio_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceAudioType);
   return(audio_type);
}

/**
 * @brief   Returns the PID being used for DVB subtitles and/or teletext for the
 *          given service. This will be the one chosen based on the settings for
 *          language, etc, or the one explicitly set (e.g. due to user selection).
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceTextPid(void *s_ptr)
{
   U16BIT t_pid;

   FUNCTION_START(ADB_GetServiceTextPid);

   t_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      t_pid = ((ADB_SERVICE_REC *)s_ptr)->ttext_pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceTextPid);
   return(t_pid);
}

/**
 * @brief   Returns whether the service is signalled as an HD service
 * @param   s_ptr service
 * @return  TRUE if it can be determined that the service is HD, FALSE otherwise
 */
BOOLEAN ADB_GetServiceIsHd(void *s_ptr)
{
   BOOLEAN is_hd;
   ADB_SERVICE_TYPE serv_type;

   FUNCTION_START(ADB_GetServiceIsHd);

   is_hd = FALSE;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      serv_type = ((ADB_SERVICE_REC *)s_ptr)->serv_type;

      if ((serv_type == ADB_SERVICE_TYPE_HD_TV) ||
          (serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
          (serv_type == ADB_SERVICE_TYPE_UHD_TV))
      {
         is_hd = TRUE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceIsHd);

   return(is_hd);
}

/**
 * @brief   Returns the audio PID to be used for the given service based on language
 *          settings, output format, etc.
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredAudioPid(void *s_ptr)
{
   U16BIT a_pid;
   E_STB_DP_AUDIO_MODE audio_mode;
   ADB_STREAM_TYPE audio_type;

   FUNCTION_START(ADB_GetRequiredAudioPid);

   a_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      a_pid = DBDEF_GetReqdAudioPid(s_ptr, &audio_mode, &audio_type);
      ((ADB_SERVICE_REC *)s_ptr)->audio_pid = a_pid;
      ADB_STREAM_REC * stream_rec = DBDEF_FindStreamRecById(s_ptr,  a_pid);
      if (NULL != stream_rec) {
         ((ADB_SERVICE_REC *)s_ptr)->audio_ecm_pid = stream_rec->ecm_pid;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetRequiredAudioPid);
   return(a_pid);
}

/**
 * @brief   Returns the PID to be used for audio description for the given service
 *          based on language settings, output format, etc.
 * @param   s_ptr service
 * @param   broadcast_mix returned as TRUE if the selected AD is broadcast mix
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredADPid(void *s_ptr, BOOLEAN *broadcast_mix)
{
   U16BIT ad_pid;
   E_STB_DP_AUDIO_MODE ad_mode;
   ADB_STREAM_TYPE ad_type;

   FUNCTION_START(ADB_GetRequiredADPid);

   ad_pid = 0;
   *broadcast_mix = FALSE;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      ad_pid = DBDEF_GetReqdADPid(s_ptr, &ad_mode, &ad_type, broadcast_mix);
      ((ADB_SERVICE_REC *)s_ptr)->ad_pid = ad_pid;

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetRequiredADPid);
   return(ad_pid);
}

/**
 * @brief   Returns the mode to be used for audio for the given service based on
 *          language settings, output format, etc.
 * @param   s_ptr service
 * @return  mode, default is AUDIO_STEREO
 */
E_STB_DP_AUDIO_MODE ADB_GetRequiredAudioMode(void *s_ptr)
{
   E_STB_DP_AUDIO_MODE audio_mode;
   ADB_STREAM_TYPE audio_type;

   FUNCTION_START(ADB_GetRequiredAudioMode);

   audio_mode = AUDIO_STEREO;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_GetReqdAudioPid(s_ptr, &audio_mode, &audio_type);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetRequiredAudioMode);
   return(audio_mode);
}

/**
 * @brief   Returns the PID to be used for teletext for the given service based
 *          on language settings
 * @param   s_ptr service
 * @param   for_subtitles find the teletext service marked as being subtitles
 * @param   magazine returns the magazine part of the teletext page number
 * @param   page returns the page number part of the teletext page number
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetRequiredTtextPid(void *s_ptr, BOOLEAN for_subtitles, U8BIT *magazine, U8BIT *page)
{
   U16BIT t_pid;

   FUNCTION_START(ADB_GetRequiredTtextPid);

   t_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_GetReqdTtextPid(s_ptr, for_subtitles, &t_pid, magazine, page);
      ((ADB_SERVICE_REC *)s_ptr)->ttext_pid = t_pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetRequiredTtextPid);
   return(t_pid);
}

/**
 * @brief   Returns the PID to be used for DVB subtitles for the given service based
 *          on language settings
 * @param   s_ptr service
 * @param   cpage_ptr returns the composition page number
 * @param   apage_ptr returns the ancilliary page number
 * @return  PID value, 0 if not found
 */
U16BIT ADB_GetRequiredSubtitleParams(void *s_ptr, U16BIT *cpage_ptr, U16BIT *apage_ptr)
{
   U16BIT pid;
   U16BIT cpage;
   U16BIT apage;

   FUNCTION_START(ADB_GetRequiredSubtitleParams);

   pid = 0;
   cpage = 0;
   apage = 0;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      DBDEF_GetReqdSubtitleParams(s_ptr, &pid, &cpage, &apage);
      ((ADB_SERVICE_REC *)s_ptr)->subtitle_pid = pid;
      ((ADB_SERVICE_REC *)s_ptr)->subtitle_cpage = cpage;
      ((ADB_SERVICE_REC *)s_ptr)->subtitle_apage = apage;

      DBDEF_ReleaseAccess();
   }

   *cpage_ptr = cpage;
   *apage_ptr = apage;

   FUNCTION_FINISH(ADB_GetRequiredSubtitleParams);

   return(pid);
}

/**
 * @brief   Returns the video PID for the given service.
 * @param   s_ptr service
 * @return  PID value, or 0 if not known or invalid service
 */
U16BIT ADB_GetServiceVideoPid(void *s_ptr)
{
   U16BIT v_pid;

   FUNCTION_START(ADB_GetServiceVideoPid);

   v_pid = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      v_pid = ((ADB_SERVICE_REC *)s_ptr)->video_pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceVideoPid);
   return(v_pid);
}

/**
 * @brief   Returns the video type being used by the given service
 * @param   s_ptr service
 * @return  Video type, defaults to ADB_VIDEO_STREAM (MPEG2)
 */
ADB_STREAM_TYPE ADB_GetServiceVideoType(void *s_ptr)
{
   ADB_STREAM_TYPE video_type;

   FUNCTION_START(ADB_GetServiceVideoType);

   video_type = ADB_VIDEO_STREAM;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      video_type = ((ADB_SERVICE_REC *)s_ptr)->video_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceVideoType);
   return(video_type);
}

/**
 * @brief   Returns the signalled type of the given service
 * @param   s_ptr service
 * @return  service type, returns 0 if unknown
 */
ADB_SERVICE_TYPE ADB_GetServiceType(void *s_ptr)
{
   U16BIT s_type;

   FUNCTION_START(ADB_GetServiceType);

   s_type = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      s_type = ((ADB_SERVICE_REC *)s_ptr)->serv_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceType);
   return(s_type);
}

/**
 * @brief   Returns the service group mask of the given service
 * @param   s_ptr service
 * @return  service group mask
 */
U32BIT ADB_GetServiceGroupMask(void *s_ptr)
{
   U32BIT group_mask;

   FUNCTION_START(ADB_GetServiceGroupMask);

   group_mask = 0;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      group_mask = ((ADB_SERVICE_REC *)s_ptr)->group_mask;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceGroupMask);
   return(group_mask);
}

/**
 * @brief   Returns the provider name, as a UTF-8 string, using the default language,
 *          of the given service.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @return  UTF-8 provider name, or NULL
 */
U8BIT* ADB_GetServiceProviderName(void *s_ptr)
{
   ADB_STRING *name_str;
   U16BIT nchar;
   U8BIT *retval;

   FUNCTION_START(ADB_GetServiceProviderName);

   retval = NULL;

   DBDEF_RequestAccess();

   if ((name_str = DBDEF_GetServiceProviderName(s_ptr)) != NULL)
   {
      retval = STB_ConvertStringToUTF8(name_str->str_ptr, &nchar, FALSE, name_str->lang_code);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceProviderName);

   return(retval);
}

/**
 * @brief   Returns the provider name, as a UTF-8 string, using the given language,
 *          of the given service.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   s_ptr service
 * @param   lang language identifier, one of the ACFG_DB_LANG_ values defined in ap_cfg.h
 * @return  UTF-8 provider name, or NULL
 */
U8BIT* ADB_GetServiceProviderNameByLang(void *s_ptr, U8BIT lang)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetServiceProviderNameByLang);

   name_str = NULL;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      name_str = CopyString(((ADB_SERVICE_REC *)s_ptr)->provider_array[lang], TRUE);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceProviderNameByLang);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the default service provider name of the specified service
 *
 * @param   service pointer
 *
 * @return   provider name
 *
 */
U8BIT* ADB_GetDefaultServiceProviderName(void *s_ptr)
{
   U8BIT *name_str;

   FUNCTION_START(ADB_GetDefaultServiceProviderName);

   name_str = NULL;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      name_str = CopyString(((ADB_SERVICE_REC *)s_ptr)->provider_str, TRUE);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetDefaultServiceProviderName);
   return(name_str);
}

/**
 * @brief   Returns the status of the unavailable flag of the given service.
 *          This flag indicates whether a service is no longer signalled in the SDT.
 * @param   s_ptr service
 * @return  TRUE if the service is unavailable, FALSE otherwise
 */
BOOLEAN ADB_GetServiceUnavailableFlag(void *s_ptr)
{
   BOOLEAN unavailable;

   FUNCTION_START(ADB_GetServiceUnavailableFlag);

   unavailable = FALSE;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      unavailable = ((ADB_SERVICE_REC *)s_ptr)->unavailable;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceUnavailableFlag);
   return(unavailable);
}

/**
 * @brief   Returns the running status of the given service as reported by the SDT
 * @param   s_ptr service
 * @return  running status will be 0 if service isn't valid or SDT hasn't been parsed yet
 */
U8BIT ADB_GetServiceRunningStatus(void *s_ptr)
{
   U8BIT status;

   FUNCTION_START(ADB_GetServiceRunningStatus);

   status = 0;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      status = ((ADB_SERVICE_REC *)s_ptr)->running_status;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceRunningStatus);

   return(status);
}

/**
 * @brief   Returns the status of the 'not running' flag of the given service.
 *          This flag indicates whether a service is signalled as not running
 *          in the SDT, or is removed from the PAT.
 * @param   s_ptr service
 * @return  TRUE if the service is not running, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNotRunningFlag(void *s_ptr)
{
   BOOLEAN not_running;

   FUNCTION_START(ADB_GetServiceNotRunningFlag);

   not_running = FALSE;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      not_running = ((ADB_SERVICE_REC *)s_ptr)->not_running;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceNotRunningFlag);
   return(not_running);
}

/**
 * @brief   Returns the status of the 'scrambled' flag of the given service.
 *          This flag is set depending on the scrambled state of all the streams
 *          that makeup the service being marked as free-to-air.
 * @param   s_ptr service
 * @return  TRUE if the service is scrambled, FALSE otherwise
 */
BOOLEAN ADB_GetServiceScrambledFlag(void *s_ptr)
{
   BOOLEAN serv_scrambled;

   FUNCTION_START(ADB_GetServiceScrambledFlag);

   serv_scrambled = FALSE;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      serv_scrambled = ((ADB_SERVICE_REC *)s_ptr)->scrambled;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceScrambledFlag);
   return(serv_scrambled);
}

/**
 * @brief   Used to query whether the given service has a CA descriptor
 * @param   s_ptr service
 * @return  TRUE if the service has a CA descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetServiceHasCaDesc(void *s_ptr)
{
   BOOLEAN has_ca_desc;

   FUNCTION_START(ADB_GetServiceHasCaDesc);

   has_ca_desc = FALSE;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      has_ca_desc = ((ADB_SERVICE_REC *)s_ptr)->has_ca_descriptor;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceHasCaDesc);
   return(has_ca_desc);
}

/**
 * @brief   Returns the do_not_scramble flag for the service which is based on
 *          the presence of an FTA descriptor. The value returned will be found
 *          by looking backwards from the service to the transport, etc.
 * @param   s_ptr pointer to the service
 * @return  FALSE if the service is to be protected, TRUE otherwise
 */
BOOLEAN ADB_GetServiceDoNotScramble(void *s_ptr)
{
   BOOLEAN do_not_scramble;

   FUNCTION_START(ADB_GetServiceDoNotScramble);

   do_not_scramble = TRUE;

   if (s_ptr != NULL)
   {
      do_not_scramble = FindDoNotScramble(NULL, s_ptr);
   }

   FUNCTION_FINISH(ADB_GetServiceDoNotScramble);

   return(do_not_scramble);
}

/**
 * @brief   Returns the content protection level value for the service that will have been
 *          set by the Nordig content protection descriptor in the PMT
 * @param   s_ptr service
 * @return  Value of the content protection level
 */
U8BIT ADB_GetServiceContentProtectionLevel(void *s_ptr)
{
   U8BIT level;

   FUNCTION_START(ADB_GetServiceContentProtectionLevel);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      level = ((ADB_SERVICE_REC *)s_ptr)->content_protection_level;
      DBDEF_ReleaseAccess();
   }
   else
   {
      /* The default level according to Nordig v2.5.1 is 0x01,
       * which means content protection isn't required */
      level = 0x01;
   }

   FUNCTION_FINISH(ADB_GetServiceContentProtectionLevel);

   return(level);
}

/**
 * @brief   Returns the status of the 'new' flag of the given service.
 *          This flag indicates whether a service has been added to the service
 *          lineup following a service scan.
 * @param   s_ptr service
 * @return  TRUE if the service is new, FALSE otherwise
 */
BOOLEAN ADB_GetServiceNewFlag(void *s_ptr)
{
   BOOLEAN new_service = FALSE;

   FUNCTION_START(ADB_GetServiceNewFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      new_service = ((ADB_SERVICE_REC *)s_ptr)->new_service;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceNewFlag);
   return(new_service);
}

/**
 * @brief   Returns the status of the 'locked' flag of the given service, which
 *          is used by the parental control.
 * @param   s_ptr service
 * @return  TRUE if the service is locked, FALSE otherwise
 */
BOOLEAN ADB_GetServiceLockedFlag(void *s_ptr)
{
   BOOLEAN locked = FALSE;

   FUNCTION_START(ADB_GetServiceLockedFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      locked = ((ADB_SERVICE_REC *)s_ptr)->locked;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceLockedFlag);
   return(locked);
}

/**
 * @brief   Changes the current state of the given service from locked to unlocked,
 *          or vice versa.
 * @param   s_ptr service
 */
void ADB_ToggleServiceLockedFlag(void *s_ptr)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;
   BOOLEAN locked;

   FUNCTION_START(ADB_ToggleServiceLockedFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      locked = serv_ptr->locked;
      if (locked)
      {
         serv_ptr->locked = FALSE;
      }
      else
      {
         serv_ptr->locked = TRUE;
      }

      DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_LOCKED, serv_ptr->locked);
      DBA_SaveRecord(serv_ptr->dba_rec);

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_ToggleServiceLockedFlag);
}

/**
 * @brief   Locks or unlocks the given service.
 * @param   s_ptr service
 * @param   locked TRUE to lock the service, FALSE to unlock
 */
void ADB_SetServiceLockedFlag(void *s_ptr, BOOLEAN locked)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetServiceLockedFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      serv_ptr->locked = locked;
      DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_LOCKED, serv_ptr->locked);
      DBA_SaveRecord(serv_ptr->dba_rec);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceLockedFlag);
}

/**
 * @brief   Returns a value indicating whether the given service is a Freesat service
 * @param   s_ptr service handle
 * @return  TRUE if service is a Freesat service, FALSE otherwise
 */
BOOLEAN ADB_IsFreesatService(void *s_ptr)
{
   BOOLEAN retval;
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_IsFreesatService);

   retval = FALSE;

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      if (serv_ptr->freesat_id != INVALID_FREESAT_SERV_ID)
      {
         retval = TRUE;
      }
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_IsFreesatService);

   return(retval);
}

/**
 * @brief   Returns Freesat service ID of the given service
 * @param   s_ptr service handle
 * @return  Freesat ID of the service, INVALID_FREESAT_SERV_ID if the service isn't a Freesat service
 */
U16BIT ADB_GetFreesatServiceId(void *s_ptr)
{
   U16BIT freesat_id;
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_GetFreesatServiceId);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      freesat_id = serv_ptr->freesat_id;
      DBDEF_ReleaseAccess();
   }
   else
   {
      freesat_id = INVALID_FREESAT_SERV_ID;
   }

   FUNCTION_FINISH(ADB_GetFreesatServiceId);

   return(freesat_id);
}

#if 0
/**
 *

 *
 * @brief   Sets locked flags of the specified service list according to settings of the tag array.
 *                1: Only the locked flags corresponding to non-zero values in the tag array will be set
 *                2: If the tag array is NULL, then all locked flags in the service list will be set
 *
 * @param   tag_array   - array of tag values
 * @param   slist       - service list pointer
 * @param   num_entries - service list size
 * @param   locked      - required locked flag setting
 *

 *
 */
void ADB_SetServiceListTaggedLockedFlag(U32BIT *tag_array, void **slist, U16BIT num_entries, BOOLEAN locked)
{
   U16BIT i;

   FUNCTION_START(ADB_SetServiceListTaggedLockedFlag);

   if ((num_entries > 0) && (slist != NULL))
   {
      if (tag_array == NULL)
      {
         //Set all flags in list
         ADB_SetServiceListLockedFlag(slist, num_entries, locked);
      }
      else
      {
         DBDEF_RequestAccess();
         //Only set flags associated with non-zero tag array values
         for (i = 0; i != num_entries; i++)
         {
            if (tag_array[i] > 0)
            {
               if (locked)
               {
                  DBA_ServFlgLock(slist[i]);
               }
               else
               {
                  DBA_ServFlgUnlock(slist[i]);
               }
               DBA_CommitService(slist[i]);
            }
         }
         DBDEF_ReleaseAccess();
      }
   }
   FUNCTION_FINISH(ADB_SetServiceListTaggedLockedFlag);
}

#endif

/**
 * @brief   Returns a pointer to the service's parent transport record
 * @param   s_ptr pointer to the service
 * @return  transport, or NULL
 */
void* ADB_GetServiceTransportPtr(void *s_ptr)
{
   void *t_ptr;

   FUNCTION_START(ADB_GetServiceTransportPtr);

   t_ptr = NULL;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      t_ptr = ((ADB_SERVICE_REC *)s_ptr)->transport;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceTransportPtr);
   return(t_ptr);
}

/**
 * @brief   Sets the tuned service for the given path, together with the associated
 *          tuned transport and network. The 'new' flag for the service is aso cleared.
 * @param   path decode path
 * @param   s_ptr service
 */
void ADB_SetTunedService(U8BIT path, void *s_ptr)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetTunedService);

   DBDEF_RequestAccess();

   DBDEF_SetTunedService(path, s_ptr);

   if (s_ptr != NULL)
   {
      // clear new flag
      serv_ptr->new_service = FALSE;

      // set tuned transport and network or satellite
      t_ptr = serv_ptr->transport;
      DBDEF_SetTunedTransport(path, t_ptr);

      if (t_ptr != NULL)
      {
         DBDEF_SetTunedNetwork(path, t_ptr->network);
      }
      else
      {
         DBDEF_SetTunedNetwork(path, NULL);
      }
   }
   else
   {
      DBDEF_SetTunedTransport(path, NULL);
      DBDEF_SetTunedNetwork(path, NULL);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetTunedService);
}

/**
 * @brief   Returns the tuned service for the given decode path.
 * @param   path decode path
 * @return  tuned service, or NULL
 */
void* ADB_GetTunedService(U8BIT path)
{
   void *s_ptr;

   FUNCTION_START(ADB_GetTunedService);

   DBDEF_RequestAccess();
   s_ptr = DBDEF_GetTunedService(path);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetTunedService);

   return(s_ptr);
}

/**
 * @brief   Returns a copy of the default authority CRID string for the given service.
 *          This string will be returned as broadcast and should be freed using STB_AppFreMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to a copy of the CRID string, or NULL
 */
U8BIT* ADB_GetServiceDefaultAuthority(void *serv_ptr)
{
   U8BIT *crid_str;

   FUNCTION_START(ADB_GetServiceDefaultAuthority);

   DBDEF_RequestAccess();
   crid_str = DBDEF_GetServiceDefaultAuthority(serv_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceDefaultAuthority);

   return(crid_str);
}

/**
 * @brief   Adds the given RCT link info to the end of the list of existing RCT links
 *          already defined for the given service
 * @param   serv_ptr service to which link is to be added
 * @param   link_ptr link to be added
 */
void ADB_ServiceAddRCTLink(void *serv_ptr, void *link_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_RCT_LINK_INFO *l_ptr;
   ADB_RCT_LINK_INFO *curr_ptr;

   FUNCTION_START(ADB_ServiceAddRCTLink);

   ASSERT(serv_ptr != NULL);
   ASSERT(link_ptr != NULL);

   DBDEF_RequestAccess();

   s_ptr = (ADB_SERVICE_REC *)serv_ptr;
   l_ptr = (ADB_RCT_LINK_INFO *)link_ptr;

   l_ptr->next = NULL;

   if (s_ptr->rct_link_list == NULL)
   {
      s_ptr->rct_link_list = link_ptr;
   }
   else
   {
      /* Add the link to the end of the existing list */
      for (curr_ptr = s_ptr->rct_link_list; curr_ptr->next != NULL; )
      {
         curr_ptr = curr_ptr->next;
      }

      curr_ptr->next = link_ptr;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_ServiceAddRCTLink);
}

/**
 * @brief   Frees all RCT link info for the given service
 * @param   serv_ptr service from which links are to be cleared
 */
void ADB_ServiceReleaseRCTLinks(void *serv_ptr)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_ServiceReleaseRCTLinks);

   ASSERT(serv_ptr != NULL);

   DBDEF_RequestAccess();

   s_ptr = (ADB_SERVICE_REC *)serv_ptr;
   if (s_ptr->rct_link_list != NULL)
   {
      DBDEF_DeleteRCTLinks(s_ptr->rct_link_list);
      s_ptr->rct_link_list = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_ServiceReleaseRCTLinks);
}

/**
 * @brief   Returns the number of RCT links for the given service
 * @param   serv_ptr service from which links are to be cleared
 * @return  Number of RCT links
 */
U8BIT ADB_GetServiceNumRCTLinks(void *serv_ptr)
{
   U8BIT num_links;
   ADB_RCT_LINK_INFO *l_ptr;

   FUNCTION_START(ADB_GetServiceNumRCTLinks);

   DBDEF_RequestAccess();

   num_links = 0;

   if (serv_ptr != NULL)
   {
      for (l_ptr = ((ADB_SERVICE_REC *)serv_ptr)->rct_link_list; l_ptr != NULL; l_ptr = l_ptr->next)
      {
         num_links++;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceNumRCTLinks);

   return(num_links);
}

/**
 * @brief   Returns a copy of the RCT links for the given service.
 *          The returned list should be freed using ADB_ReleaseRCTLinks.
 * @param   serv_ptr service from which links are to be copied
 * @param   num_links pointer to return the number of links
 * @return  Copy of links, or NULL
 */
void* ADB_GetServiceRCTLinks(void *serv_ptr, U8BIT *num_links)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_RCT_LINK_INFO *links;
   ADB_RCT_LINK_INFO *l_ptr;
   ADB_RCT_LINK_INFO *new_ptr;
   ADB_RCT_LINK_INFO *last_link;
   U8BIT i;

   FUNCTION_START(ADB_GetServiceRCTLinks);

   DBDEF_RequestAccess();

   links = NULL;
   *num_links = 0;
   s_ptr = (ADB_SERVICE_REC *)serv_ptr;

   if ((s_ptr != NULL) && (s_ptr->rct_link_list != NULL))
   {
      last_link = NULL;

      for (l_ptr = s_ptr->rct_link_list; l_ptr != NULL; l_ptr = l_ptr->next)
      {
         new_ptr = (ADB_RCT_LINK_INFO *)STB_AppGetMemory(sizeof(ADB_RCT_LINK_INFO));
         if (new_ptr != NULL)
         {
            memset(new_ptr, 0, sizeof(ADB_RCT_LINK_INFO));

            new_ptr->is_group_trailer = l_ptr->is_group_trailer;

            if (l_ptr->uri_string != NULL)
            {
               new_ptr->uri_string = (U8BIT *)STB_AppGetMemory(strlen((char *)l_ptr->uri_string) + 1);
               if (new_ptr->uri_string != NULL)
               {
                  strncpy((char *)new_ptr->uri_string, (char *)l_ptr->uri_string, strlen((char *)l_ptr->uri_string) + 1);

                  new_ptr->can_use_default_icon = l_ptr->can_use_default_icon;
                  new_ptr->icon_id = l_ptr->icon_id;

                  for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
                  {
                     if (l_ptr->promo_text[i] != NULL)
                     {
                        new_ptr->promo_text[i] = DBDEF_CopyString(l_ptr->promo_text[i]);
                     }
                  }

                  if (l_ptr->event_name != NULL)
                  {
                     new_ptr->event_name = DBDEF_CopyString(l_ptr->event_name);
                  }
               }
            }

            if (links == NULL)
            {
               links = new_ptr;
               last_link = new_ptr;
            }
            else
            {
               last_link->next = new_ptr;
               last_link = new_ptr;
            }

            (*num_links)++;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceRCTLinks);

   return(links);
}

/**
 * @brief   Returns the RCT link following the given link
 * @param   link_ptr pointer to an RCT link
 * @return  next RCT link, or NULL
 */
void* ADB_GetNextRCTLink(void *link_ptr)
{
   void *next_link;

   FUNCTION_START(ADB_GetNextRCTLink);

   next_link = NULL;

   if (link_ptr != NULL)
   {
      next_link = ((ADB_RCT_LINK_INFO *)link_ptr)->next;
   }

   FUNCTION_FINISH(ADB_GetNextRCTLink);

   return(next_link);
}

/**
 * @brief   Frees the given list of RCT links
 * @param   links links to be freed
 */
void ADB_ReleaseRCTLinks(void *links)
{
   FUNCTION_START(ADB_ReleaseRCTLinks);

   DBDEF_DeleteRCTLinks(links);

   FUNCTION_FINISH(ADB_ReleaseRCTLinks);
}

/**
 * @brief   Returns the RCT link's promotional text based on the default language
 * @param   link pointer to link
 * @return  Pointer to promotional link string
 */
U8BIT* ADB_GetRCTLinkPromoText(void *link_ptr)
{
   ADB_RCT_LINK_INFO *link;
   U8BIT *text;
   U8BIT *lang_ids;
   U16BIT i;
   U16BIT nchars;

   FUNCTION_START(ADB_GetRCTLinkPromoText);

   text = NULL;

   if (link_ptr != NULL)
   {
      link = (ADB_RCT_LINK_INFO *)link_ptr;

      lang_ids = DBDEF_GetTextLang();

      if (lang_ids != NULL)
      {
         for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
              (lang_ids[i] != ACFG_INVALID_DB_LANG) && (text == NULL); i++)
         {
            text = STB_ConvertStringToUTF8(link->promo_text[lang_ids[i]]->str_ptr, &nchars, FALSE,
                  link->promo_text[lang_ids[i]]->lang_code);
         }
      }

      if (text == NULL)
      {
         /* Required language is not available - look for first non-null language */
         for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
         {
            text = link->promo_text[i]->str_ptr;
            if (text != NULL)
            {
               text = STB_ConvertStringToUTF8(link->promo_text[i]->str_ptr, &nchars, FALSE,
                     link->promo_text[i]->lang_code);
               break;
            }
         }
      }
   }

   FUNCTION_FINISH(ADB_GetRCTLinkPromoText);

   return(text);
}

/**
 * @brief   Returns whether the given link is for a group trailer
 * @param   link_ptr pointer to an RCT link
 * @return  Link's is_group_trailer flag
 */
BOOLEAN ADB_IsRCTLinkGroupTrailer(void *link_ptr)
{
   BOOLEAN is_group_trailer;

   FUNCTION_START(ADB_IsRCTLinkGroupTrailer);

   if (link_ptr != NULL)
   {
      is_group_trailer = ((ADB_RCT_LINK_INFO *)link_ptr)->is_group_trailer;
   }
   else
   {
      is_group_trailer = FALSE;
   }

   FUNCTION_FINISH(ADB_IsRCTLinkGroupTrailer);

   return(is_group_trailer);
}

/**
 * @brief   Returns the name of the given RCT link
 * @param   link_ptr pointer to an RCT link
 * @return  pointer to the link's name - NOTE: this shouldn't be changed or freed!
 */
U8BIT* ADB_GetRCTLinkName(void *link_ptr)
{
   U8BIT *name;
   ADB_RCT_LINK_INFO *lptr;
   U16BIT nchars;

   FUNCTION_START(ADB_GetRCTLinkName);

   if (link_ptr != NULL)
   {
      lptr = (ADB_RCT_LINK_INFO *)link_ptr;

      name = STB_ConvertStringToUTF8(lptr->event_name->str_ptr, &nchars, FALSE,
            lptr->event_name->lang_code);
   }
   else
   {
      name = NULL;
   }

   FUNCTION_FINISH(ADB_GetRCTLinkName);

   return(name);
}

/**
 * @brief   Returns the uri string of the given RCT link
 * @param   link_ptr pointer to an RCT link
 * @return  pointer to the link's uri - NOTE: this shouldn't be changed or freed!
 */
U8BIT* ADB_GetRCTLinkUriString(void *link_ptr)
{
   U8BIT *uri_string;

   FUNCTION_START(ADB_GetRCTLinkUriString);

   if (link_ptr != NULL)
   {
      uri_string = ((ADB_RCT_LINK_INFO *)link_ptr)->uri_string;
   }
   else
   {
      uri_string = NULL;
   }

   FUNCTION_FINISH(ADB_GetRCTLinkUriString);

   return(uri_string);
}

/**
 * @brief   Adds the given image icon to the end of the service's icon list.
 *          The icon id is checked and if it matches an icon already in the list
 *          then the new icon replaces the existing one.
 * @param   serv_ptr service to which link is to be added
 * @param   icon_ptr icon to be added
 * @return  TRUE if the icon is added, FALSE otherwise
 */
BOOLEAN ADB_ServiceAddImageIcon(void *serv_ptr, void *icon_ptr)
{
   BOOLEAN icon_added;
   ADB_SERVICE_REC *s_ptr;
   ADB_IMAGE_ICON *i_ptr;
   ADB_IMAGE_ICON *curr_ptr;
   ADB_IMAGE_ICON *prev_ptr;

   FUNCTION_START(ADB_ServiceAddImageIcon);

   ASSERT(serv_ptr != NULL);
   ASSERT(icon_ptr != NULL);

   DBDEF_RequestAccess();

   s_ptr = (ADB_SERVICE_REC *)serv_ptr;
   i_ptr = (ADB_IMAGE_ICON *)icon_ptr;

   /* Check whether the icon id is already in the list */
   for (curr_ptr = s_ptr->icon_list, prev_ptr = NULL; (curr_ptr != NULL) && (i_ptr->icon_id != curr_ptr->icon_id); )
   {
      prev_ptr = curr_ptr;
      curr_ptr = curr_ptr->next;
   }

   if (curr_ptr == NULL)
   {
      i_ptr->next = NULL;

      if (s_ptr->icon_list == NULL)
      {
         /* First icon in the list */
         s_ptr->icon_list = i_ptr;
      }
      else
      {
         /* Add new icon to the end of the list */
         prev_ptr->next = i_ptr;
      }

      icon_added = TRUE;
   }
   else
   {
      /* Icon with this id already exists. Only replace it if it can't be determined that it's
       * the same image */
      if ((i_ptr->icon_type != curr_ptr->icon_type) ||
          (i_ptr->transport_mode != curr_ptr->transport_mode) ||
          ((i_ptr->transport_mode == ICON_TRANS_URL) &&
           (STB_CompareStringsIgnoreCase(i_ptr->icon_url, curr_ptr->icon_url) != 0)))
      {
         i_ptr->next = curr_ptr->next;

         if (prev_ptr == NULL)
         {
            s_ptr->icon_list = i_ptr;
         }
         else
         {
            prev_ptr->next = i_ptr;
         }

         /* Set next pointer to NULL as only this icon should be deleted */
         curr_ptr->next = NULL;
         DBDEF_DeleteImageIcons(curr_ptr);

         icon_added = TRUE;
      }
      else
      {
         /* Icons are the same so ignore the new one */
         DBDEF_DeleteImageIcons(i_ptr);

         icon_added = FALSE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_ServiceAddImageIcon);

   return(icon_added);
}

/**
 * @brief   Checks all the RCT links for the given service to determine whether any
 *          of them specify that the default icon can be used.
 * @param   serv_ptr service containing the RCT links
 * @return  TRUE if default icon can be used, FALSE otherwise
 */
BOOLEAN ADB_ServiceRCTCanUseDefaultIcon(void *serv_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_RCT_LINK_INFO *link;
   BOOLEAN can_use_default;

   FUNCTION_START(ADB_ServiceRCTCanUseDefaultIcon);

   can_use_default = FALSE;

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      if (s_ptr->rct_link_list != NULL)
      {
         for (link = s_ptr->rct_link_list; (link != NULL) && !can_use_default; link = link->next)
         {
            if (link->can_use_default_icon)
            {
               can_use_default = TRUE;
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_ServiceRCTCanUseDefaultIcon);

   return(can_use_default);
}

/**
 * @brief   Searches all the RCT links for a service to see if any of them define an
 *          icon to be used, and if one is found then all the data required to display
 *          the icon is returned.
 * @param   serv_ptr service
 * @param   icon_data address of pointer to icon data. This data shouldn't be freed.
 * @param   data_size pointer to variable containing amount of data
 * @param   pos_valid pointer to boolean indicating whether returned x,y values are valid
 * @param   x_pos pointer to returned X position to display the icon
 * @param   y_pos pointer to returned Y position to display the icon
 * @param   width pointer to returned width of the icon
 * @param   height pointer to returned height of the icon
 * @param   coord_system pointer to coord system of returned pos and size values
 * @return  TRUE if icon data is returned, FALSE otherwise
 */
BOOLEAN ADB_ServiceGetRCTIcon(void *serv_ptr, U8BIT **icon_data, U32BIT *data_size, BOOLEAN *pos_valid,
   U16BIT *x_pos, U16BIT *y_pos, U16BIT *width, U16BIT *height, E_ICON_COORD_SYSTEM *coord_system)
{
   BOOLEAN icon_available;
   ADB_SERVICE_REC *s_ptr;
   ADB_RCT_LINK_INFO *link;
   U8BIT icon_id;
   ADB_IMAGE_ICON *icon_ptr;

   FUNCTION_START(ADB_ServiceGetRCTIcon);

   icon_available = FALSE;

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      if (s_ptr->rct_link_list != NULL)
      {
         icon_id = 0;

         for (link = s_ptr->rct_link_list; (link != NULL) && (icon_id == 0); link = link->next)
         {
            if ((link->icon_id > 0) && (link->icon_id <= 7))
            {
               if (icon_id == 0)
               {
                  icon_id = link->icon_id;
               }
            }
         }

         if (icon_id != 0)
         {
            /* Find the referenced icon */
            for (icon_ptr = s_ptr->icon_list; icon_ptr != NULL; icon_ptr = icon_ptr->next)
            {
               if (icon_ptr->icon_id == icon_id)
               {
                  /* Check that the data is available */
                  if (icon_ptr->icon_data != NULL)
                  {
                     *icon_data = icon_ptr->icon_data;
                     *data_size = icon_ptr->data_len;

                     *pos_valid = icon_ptr->position_defined;
                     if (icon_ptr->position_defined)
                     {
                        *x_pos = icon_ptr->x_pos;
                        *y_pos = icon_ptr->y_pos;
                     }

                     *width = icon_ptr->width;
                     *height = icon_ptr->height;
                     *coord_system = icon_ptr->coord_system;

                     icon_available = TRUE;
                  }
                  break;
               }
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_ServiceGetRCTIcon);

   return(icon_available);
}

/**
 * @brief   Deletes the given service from the database
 * @param   s_ptr service to be deleted
 */
void ADB_DeleteServiceRec(void *s_ptr)
{
   FUNCTION_START(ADB_DeleteServiceRec);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_DeleteServiceRec(s_ptr);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_DeleteServiceRec);
}

/**
 * @brief   Searches for a replacement service of the given type for the given service
 * @param   s_ptr service that a replacement is required for
 * @param   alt_serv_type type of replacement service to look for (e.g. CA replacement)
 * @return  alternative service found, or NULL
 */
void* ADB_GetAlternativeService(void *s_ptr, U8BIT alt_serv_type)
{
   void *alt_serv;
   ADB_ALT_SERV_REC *alt_serv_rec;

   FUNCTION_START(ADB_GetAlternativeService);

   alt_serv = NULL;
   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      // is there an alternate service?
      alt_serv_rec = ((ADB_SERVICE_REC *)s_ptr)->alt_serv_list;
      while (alt_serv_rec != NULL)
      {
         if (alt_serv_type == alt_serv_rec->link_type)
         {
            alt_serv = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, alt_serv_rec->onet_id,
                  alt_serv_rec->tran_id, alt_serv_rec->serv_id);
            if (alt_serv != NULL)
            {
               break;
            }
         }
         alt_serv_rec = alt_serv_rec->next;
      }
      DBDEF_ReleaseAccess();
   }
   FUNCTION_FINISH(ADB_GetAlternativeService);
   return(alt_serv);
}

/**
 * @brief   Determines subtitle PID and whether it is DVB or teletext, for the given service
 * @param   serv_ptr service
 * @param   subt_info pointer to ADB_SUBT_INFO for extra info on subtitles
 * @return  PID to collect subtitles data, zero if none available
 */
U16BIT ADB_ServiceGetSubtitlePid(void *serv_ptr, ADB_SUBT_INFO *subt_info)
{
   U16BIT pid, text_pid;
   U16BIT cpage, apage;
   U8BIT magazine, page;
   E_STREAM_MATCH_TYPE dvb_match_type;
   E_STREAM_MATCH_TYPE ttx_match_type;

   FUNCTION_START(ADB_ServiceGetSubtitlePid);

   pid = 0;
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();
      dvb_match_type = DBDEF_GetReqdSubtitleParams(serv_ptr, &pid, &cpage, &apage);
      ttx_match_type = DBDEF_GetReqdTtextPid(serv_ptr, TRUE, &text_pid, &magazine, &page);
      DBDEF_ReleaseAccess();

      if (dvb_match_type != STREAM_MATCH_NONE && dvb_match_type <= ttx_match_type)
      {
         subt_info->is_dvb_subt = TRUE;
         subt_info->u.subt.cpage = cpage;
         subt_info->u.subt.apage = apage;
      }
      else if (ttx_match_type != STREAM_MATCH_NONE)
      {
         pid = text_pid;
         subt_info->is_dvb_subt = FALSE;
         subt_info->u.ttxt.magazine = magazine;
         subt_info->u.ttxt.page = page;
      }
   }

   FUNCTION_FINISH(ADB_ServiceGetSubtitlePid);

   return(pid);
}

/**
 * @brief   Determines whether the given service has subtitles, DVB or teletext
 * @param   serv_ptr service
 * @param   dvb_subs returned as TRUE if the subtitles are DVB, FALSE for teletext
 * @return  TRUE if subtitles are available
 */
BOOLEAN ADB_ServiceHasSubtitles(void *serv_ptr, BOOLEAN *dvb_subs)
{
   BOOLEAN has_subs;
   U16BIT dvb_pid;
   U16BIT cpage, apage;
   E_STREAM_MATCH_TYPE dvb_match_type;
   E_STREAM_MATCH_TYPE ttx_match_type;
   U16BIT text_pid;
   U8BIT magazine, page;

   FUNCTION_START(ADB_ServiceHasSubtitles);

   has_subs = FALSE;

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      /* Try for DVB subs first */
      dvb_match_type = DBDEF_GetReqdSubtitleParams(serv_ptr, &dvb_pid, &cpage, &apage);

      ttx_match_type = DBDEF_GetReqdTtextPid(serv_ptr, TRUE, &text_pid, &magazine, &page);

      if (dvb_match_type != STREAM_MATCH_NONE)
      {
         has_subs = TRUE;

         if (dvb_match_type <= ttx_match_type)
         {
            *dvb_subs = TRUE;
         }
         else
         {
            *dvb_subs = FALSE;
         }
      }
      else if (ttx_match_type != STREAM_MATCH_NONE)
      {
         has_subs = TRUE;
         *dvb_subs = FALSE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_ServiceHasSubtitles);

   return(has_subs);
}

/**
 * @brief   Returns the number of streams of the given type for the given service
 * @param   serv_ptr service
 * @param   stream_list_type type of stream
 * @return  Number of streams
 */
U16BIT ADB_GetNumStreams(void *serv_ptr, ADB_STREAM_LIST_TYPE stream_list_type)
{
   U16BIT num_streams;
   ADB_STREAM_REC *stream_ptr;

   FUNCTION_START(ADB_GetNumStreams);

   DBDEF_RequestAccess();

   num_streams = 0;
   stream_ptr = ((ADB_SERVICE_REC *)serv_ptr)->stream_list;
   while (stream_ptr != NULL)
   {
      if (IsStreamOfType(stream_ptr, stream_list_type))
      {
         num_streams++;
      }

      stream_ptr = stream_ptr->next;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumStreams);

   return(num_streams);
}

/**
 * @brief   Allocates and returns an array of the streams of the given type and
 *          for the given service.
 *          The returned array should be freed using ADB_ReleaseStreamList.
 * @param   serv_ptr service
 * @param   stream_list_type type of streams to be returned
 * @param   streamlist_ptr pointer to array that will be allocated
 * @param   num_entries_ptr pointer to number of items returned in the array
 */
void ADB_GetStreamList(void *serv_ptr, ADB_STREAM_LIST_TYPE stream_list_type, void ***streamlist_ptr,
   U16BIT *num_entries_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   U16BIT no_entries;
   U16BIT max_entries;
   ADB_STREAM_REC *stream_ptr;
   U16BIT i, j;
   S_STREAM *temp_stream;
   S_STREAM **temp_array;
   S_STREAM *sptr;
   U8BIT tag_index;
   ADB_EVENT_DESC *event_desc;
   U8BIT component_tag;

   FUNCTION_START(ADB_GetStreamList);

   DBDEF_RequestAccess();

   no_entries = 0;
   max_entries = 0;
   temp_array = NULL;

   if (serv_ptr != NULL)
   {
      s_ptr = (ADB_SERVICE_REC *)serv_ptr;
      stream_ptr = s_ptr->stream_list;
      while (stream_ptr != NULL)
      {
         max_entries++;
         stream_ptr = stream_ptr->next;
      }

      if (max_entries > 0)
      {
         temp_array = STB_AppGetMemory(max_entries * sizeof(S_STREAM *));
         if (temp_array != NULL)
         {
            stream_ptr = s_ptr->stream_list;
            for (i = 0; ((i < max_entries) && (stream_ptr != NULL)); i++)
            {
               if (IsStreamOfType(stream_ptr, stream_list_type))
               {
                  temp_stream = STB_AppGetMemory(sizeof(S_STREAM));
                  if (temp_stream != NULL)
                  {
                     temp_stream->num_tag_entries = 0;
                     temp_stream->tag_array_ptr = STB_AppGetMemory(stream_ptr->num_tags * sizeof(S_COMPONENT));
                     if ((temp_stream->tag_array_ptr != NULL) && (stream_ptr->tag_array != NULL))
                     {
                        for (tag_index = 0; tag_index < stream_ptr->num_tags; tag_index++)
                        {
                           temp_stream->tag_array_ptr[tag_index].tag = stream_ptr->tag_array[tag_index];
                        }
                        temp_stream->num_tag_entries = stream_ptr->num_tags;
                     }
                     temp_stream->pid = stream_ptr->pid;
                     temp_stream->stream_type = stream_ptr->type;
                     temp_stream->in_use = stream_ptr->in_use;
                     temp_stream->has_ca_descriptor = stream_ptr->has_ca_descriptor;
#if 0
                     temp_stream->component_type = stream_ptr->component_type;
#else
                     /* Find the component type from the now event */
                     if ((s_ptr->now_event != NULL) && (temp_stream->tag_array_ptr != NULL))
                     {
                        event_desc = s_ptr->now_event->desc_list_head;

                        while ((event_desc = DBDEF_FindEventDescriptor(event_desc, COMPONENT_DTAG, 0)) != NULL)
                        {
                           component_tag = event_desc->desc_data[4];

                           /* See if this stream has this component tag */
                           for (tag_index = 0; tag_index != temp_stream->num_tag_entries; tag_index++)
                           {
                              if (temp_stream->tag_array_ptr[tag_index].tag == component_tag)
                              {
                                 temp_stream->component_type = event_desc->desc_data[3];
                                 temp_stream->tag_array_ptr[tag_index].content = event_desc->desc_data[2]; /* stream_content_ext (bits 4..7) and stream_content (bits 0.3) */
                                 temp_stream->tag_array_ptr[tag_index].type = event_desc->desc_data[3];
                                 break;
                              }
                           }

                           event_desc = event_desc->next;
                        }
                     }
#endif

                     if ((temp_stream->stream_type == ADB_AUDIO_STREAM) ||
                         (temp_stream->stream_type == ADB_AAC_AUDIO_STREAM) ||
                         (temp_stream->stream_type == ADB_HEAAC_AUDIO_STREAM) ||
                         (temp_stream->stream_type == ADB_AC3_AUDIO_STREAM) ||
                         (temp_stream->stream_type == ADB_EAC3_AUDIO_STREAM)
                         )
                     {
                        /* For dual mono audio, a stream is available twice, with the same lang,
                         * pid and type, so check for this and don't include it again unless all
                         * streams have been asked for */
                        if (stream_list_type != ADB_STREAM_LIST_ALL)
                        {
                           for (j = 0; j < no_entries; j++)
                           {
                              sptr = temp_array[j];
                              if ((sptr->pid == stream_ptr->pid) &&
                                  (sptr->lang_code == stream_ptr->data.audio.lang_code))
                              {
                                 /* This stream is already represented in the array being returned,
                                  * so this one can be ignored */
                                 STB_AppFreeMemory(temp_stream);
                                 temp_stream = NULL;
                                 break;
                              }
                           }
                        }

                        if (temp_stream != NULL)
                        {
                           temp_stream->lang_code = stream_ptr->data.audio.lang_code;
                           temp_stream->audio_type = stream_ptr->data.audio.type;
                           temp_stream->audio_mode = stream_ptr->data.audio.mode;
                        }
                     }
                     else if (temp_stream->stream_type == ADB_SUBTITLE_STREAM)
                     {
                        temp_stream->lang_code = stream_ptr->data.subtitle.lang_code;
                        temp_stream->subt_type = stream_ptr->data.subtitle.type;
                        temp_stream->composition_page = stream_ptr->data.subtitle.composition_page;
                        temp_stream->ancillary_page = stream_ptr->data.subtitle.ancillary_page;
                     }
                     else if (temp_stream->stream_type == ADB_TTEXT_STREAM)
                     {
                        temp_stream->lang_code = stream_ptr->data.ttext.lang_code;
                        temp_stream->ttext_type = stream_ptr->data.ttext.type;
                        temp_stream->ttext_magazine = stream_ptr->data.ttext.magazine;
                        temp_stream->ttext_page = stream_ptr->data.ttext.page;
                     }

                     if (temp_stream != NULL)
                     {
                        temp_array[no_entries] = temp_stream;
                        no_entries++;
                     }
                  }
               }
               stream_ptr = stream_ptr->next;
            }

            if (no_entries == 0)
            {
               STB_AppFreeMemory(temp_array);
               temp_array = NULL;
            }
         }
      }
   }

   *num_entries_ptr = no_entries;
   *streamlist_ptr = (void **)temp_array;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetStreamList);
}

/**
 * @brief   Frees the memory allocated for a stream list using ADB_GetStreamList
 * @param   streamlist_ptr stream list to be freed
 * @param   num_entries number of items in the stream list
 */
void ADB_ReleaseStreamList(void **streamlist_ptr, U16BIT num_entries)
{
   U16BIT i;
   S_STREAM *stream;

   FUNCTION_START(ADB_ReleaseStreamList);

   if (streamlist_ptr != NULL)
   {
      for (i = 0; i < num_entries; i++)
      {
         stream = streamlist_ptr[i];
         if (stream->tag_array_ptr != NULL)
         {
            STB_AppFreeMemory(stream->tag_array_ptr);
         }

         STB_AppFreeMemory(streamlist_ptr[i]);
      }

      STB_AppFreeMemory(streamlist_ptr);
   }

   FUNCTION_FINISH(ADB_ReleaseStreamList);
}

/**
 * @brief   Returns the type of the given stream
 * @param   stream_ptr stream
 * @return  stream type
 */
ADB_STREAM_TYPE ADB_GetStreamType(void *stream_ptr)
{
   ADB_STREAM_TYPE stream_type;
   FUNCTION_START(ADB_GetStreamType);

   stream_type = 0;

   if (stream_ptr != NULL)
   {
      stream_type = ((S_STREAM *)stream_ptr)->stream_type;
   }

   FUNCTION_FINISH(ADB_GetStreamType);
   return(stream_type);
}

/**
 * @brief   Returns the number of tag values defined for the given stream
 * @param   stream_ptr stream
 * @return  number of tag values
 */
U8BIT ADB_GetStreamNumTags(void *stream_ptr)
{
   U8BIT num;

   FUNCTION_START(ADB_GetStreamNumTags);

   num = 0;

   if (stream_ptr != NULL)
   {
      num = ((S_STREAM *)stream_ptr)->num_tag_entries;
   }

   FUNCTION_FINISH(ADB_GetStreamNumTags);

   return(num);
}

/**
 * @brief   Returns the requested tag value of the given stream
 * @param   stream_ptr stream
 * @param   index tag value to return, must be < ADB_GetStreamNumTags
 * @return  tag value
 */
U8BIT ADB_GetStreamTag(void *stream_ptr, U8BIT index)
{
   U8BIT tag;
   S_STREAM *stream;

   FUNCTION_START(ADB_GetStreamTag);

   tag = 0;

   if (stream_ptr != NULL)
   {
      stream = (S_STREAM *)stream_ptr;

      if ((index < stream->num_tag_entries) && (stream->tag_array_ptr != NULL))
      {
         tag = stream->tag_array_ptr[index].tag;
      }
   }

   FUNCTION_FINISH(ADB_GetStreamTag);

   return(tag);
}

/**
 * @brief   Returns the PID for the given stream
 * @param   stream_ptr stream
 * @return  PID
 */
U16BIT ADB_GetStreamPID(void *stream_ptr)
{
   U16BIT pid;
   FUNCTION_START(ADB_GetStreamPID);

   pid = 0;
   if (stream_ptr != NULL)
   {
      pid = ((S_STREAM *)stream_ptr)->pid;
   }

   FUNCTION_FINISH(ADB_GetStreamPID);
   return(pid);
}

/**
 * @brief   Returns whether the given stream is marked as being 'in use', which
 *          means it will be the stream being decoded for video, audio, etc.
 * @param   stream_ptr stream
 * @return  TRUE if stream is in use, FALSE otherwise
 */
BOOLEAN ADB_GetStreamInUse(void *stream_ptr)
{
   BOOLEAN in_use;
   FUNCTION_START(ADB_GetStreamInUse);

   in_use = FALSE;
   if (stream_ptr != NULL)
   {
      in_use = ((S_STREAM *)stream_ptr)->in_use;
   }

   FUNCTION_FINISH(ADB_GetStreamInUse);
   return(in_use);
}

/**
 * @brief   Returns whether the given stream has a CA descriptor
 * @param   stream_ptr stream
 * @return  TRUE if stream has a CA descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetStreamHasCaDesc(void *stream_ptr)
{
   BOOLEAN has_ca_desc;
   FUNCTION_START(ADB_GetStreamHasCaDesc);

   has_ca_desc = FALSE;
   if (stream_ptr != NULL)
   {
      has_ca_desc = ((S_STREAM *)stream_ptr)->has_ca_descriptor;
   }

   FUNCTION_FINISH(ADB_GetStreamHasCaDesc);
   return(has_ca_desc);
}

/**
 * @brief   Returns the language code associated with the given audio stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetAudioStreamLangCode(void *stream_ptr)
{
   U32BIT lang_code;
   FUNCTION_START(ADB_GetAudioStreamLangCode);

   lang_code = 0;
   if (stream_ptr != NULL)
   {
      lang_code = ((S_STREAM *)stream_ptr)->lang_code;
   }

   FUNCTION_FINISH(ADB_GetAudioStreamLangCode);
   return(lang_code);
}

/**
 * @brief   Returns the audio type of the given stream
 * @param   stream_ptr stream
 * @return  audio type
 */
ADB_AUDIO_TYPE ADB_GetAudioStreamType(void *stream_ptr)
{
   ADB_AUDIO_TYPE type;
   FUNCTION_START(ADB_GetAudioStreamType);

   type = 0;
   if (stream_ptr != NULL)
   {
      type = ((S_STREAM *)stream_ptr)->audio_type;
   }

   FUNCTION_FINISH(ADB_GetAudioStreamType);
   return(type);
}

/**
 * @brief   Returns the audio mode of the given stream
 * @param   stream_ptr stream
 * @return  audio mode
 */
E_STB_DP_AUDIO_MODE ADB_GetAudioStreamMode(void *stream_ptr)
{
   E_STB_DP_AUDIO_MODE mode = AUDIO_UNDEF;

   FUNCTION_START(ADB_GetAudioStreamMode);

   if (stream_ptr != NULL)
   {
      mode = ((S_STREAM *)stream_ptr)->audio_mode;
   }

   FUNCTION_FINISH(ADB_GetAudioStreamMode);

   return mode;
}

/**
 * @brief   Returns the 3 char language code of the given teletext stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetTtextStreamLangCode(void *stream_ptr)
{
   U32BIT lang_code;
   FUNCTION_START(ADB_GetTtextStreamLangCode);

   lang_code = 0;
   if (stream_ptr != NULL)
   {
      lang_code = ((S_STREAM *)stream_ptr)->lang_code;
   }

   FUNCTION_FINISH(ADB_GetTtextStreamLangCode);
   return(lang_code);
}

/**
 * @brief   Returns the type, as defined in the PMT, of the given teletext stream
 * @param   stream_ptr stream
 * @return  teletext stream type (ADB_TELETEXT_TYPE)
 */
U8BIT ADB_GetTtextStreamType(void *stream_ptr)
{
   U8BIT type;
   FUNCTION_START(ADB_GetTtextStream);

   type = 0;
   if (stream_ptr != NULL)
   {
      type = ((S_STREAM *)stream_ptr)->ttext_type;
   }

   FUNCTION_FINISH(ADB_GetTtextStreamType);
   return(type);
}

/**
 * @brief   Returns the magazine value of the given teletext stream
 * @param   stream_ptr stream
 * @return  magazine value
 */
U8BIT ADB_GetTtextStreamMagazine(void *stream_ptr)
{
   U8BIT magazine;
   FUNCTION_START(ADB_GetTtextStreamMagazine);

   magazine = 0;
   if (stream_ptr != NULL)
   {
      magazine = ((S_STREAM *)stream_ptr)->ttext_magazine;
   }

   FUNCTION_FINISH(ADB_GetTtextStreamMagazine);
   return(magazine);
}

/**
 * @brief   Returns the page value of the given teletext stream
 * @param   stream_ptr stream
 * @return  page value
 */
U8BIT ADB_GetTtextStreamPage(void *stream_ptr)
{
   U8BIT page;
   FUNCTION_START(ADB_GetTtextStreamPage);

   page = 0;
   if (stream_ptr != NULL)
   {
      page = ((S_STREAM *)stream_ptr)->ttext_page;
   }

   FUNCTION_FINISH(ADB_GetTtextStreamPage);
   return(page);
}

/**
 * @brief   Returns the 3 char language code of the given subtitle stream
 * @param   stream_ptr stream
 * @return  3 char language code encoded as a 32-bit number
 */
U32BIT ADB_GetSubtitleStreamLangCode(void *stream_ptr)
{
   U32BIT lang_code;
   FUNCTION_START(ADB_GetSubtitleStreamLangCode);

   lang_code = 0;
   if (stream_ptr != NULL)
   {
      lang_code = ((S_STREAM *)stream_ptr)->lang_code;
   }

   FUNCTION_FINISH(ADB_GetSubtitleStreamLangCode);
   return(lang_code);
}

/**
 * @brief   Returns the composition page value of the given subtitle stream
 * @param   stream_ptr stream
 * @return  composition page value
 */
U16BIT ADB_GetSubtitleStreamCompositionPage(void *stream_ptr)
{
   U16BIT comp_page;
   FUNCTION_START(ADB_GetSubtitleStreamCompositionPage);

   comp_page = 0;
   if (stream_ptr != NULL)
   {
      comp_page = ((S_STREAM *)stream_ptr)->composition_page;
   }

   FUNCTION_FINISH(ADB_GetSubtitleStreamCompositionPage);
   return(comp_page);
}

/**
 * @brief   Returns the ancillary page value of the given subtitle stream
 * @param   stream_ptr stream
 * @return  ancillary page value
 */
U16BIT ADB_GetSubtitleStreamAncillaryPage(void *stream_ptr)
{
   U16BIT anc_page;
   FUNCTION_START(ADB_GetSubtitleStreamAncillaryPage);

   anc_page = 0;
   if (stream_ptr != NULL)
   {
      anc_page = ((S_STREAM *)stream_ptr)->ancillary_page;
   }

   FUNCTION_FINISH(ADB_GetSubtitleStreamAncillaryPage);
   return(anc_page);
}

/**
 * @brief   Returns the component type as defined in the EITp/f's component descriptor
 *          for the given stream
 * @param   stream_ptr stream
 * @return  component type
 */
U8BIT ADB_GetStreamComponentType(void *stream_ptr)
{
   U8BIT component_type = 0;

   FUNCTION_START(ADB_GetStreamComponentType);

   if (stream_ptr != NULL)
   {
      component_type = ((S_STREAM *)stream_ptr)->component_type;
   }

   FUNCTION_FINISH(ADB_GetStreamComponentType);

   return component_type;
}

/**
 * @brief   For the given stream and index into component tag array for stream,
 *          return stream content and component type as defined in the EITp/f's component descriptor
 *          See EN300468, sec 6.2.8 and table 26 for meaning of return values.
 * @param   stream_ptr stream
 * @param   index tag value to return, must be < ADB_GetStreamNumTags
 * @return  bits 15..12 - stream_content_ext, bits 11..8 - stream_content, bits 7..0 - component type
 */
U16BIT ADB_GetStreamTagContentType(void *stream_ptr, U8BIT index)
{
   U16BIT content_type;
   S_STREAM *stream;

   FUNCTION_START(ADB_GetStreamTagContentType);

   content_type = 0;

   if (stream_ptr != NULL)
   {
      stream = (S_STREAM *)stream_ptr;

      if ((index < stream->num_tag_entries) && (stream->tag_array_ptr != NULL))
      {
         content_type = (U16BIT)(stream->tag_array_ptr[index].content << 8) | (U16BIT)stream->tag_array_ptr[index].type;
      }
   }

   FUNCTION_FINISH(ADB_GetStreamTagContentType);

   return(content_type);
}

/**
 * @brief   Returns the type, as defined in the PMT, of the given subtitle stream
 * @param   stream_ptr stream
 * @return  subtitle stream type
 */
ADB_SUBTITLE_TYPE ADB_GetSubtitleStreamType(void *stream_ptr)
{
   ADB_SUBTITLE_TYPE type;
   FUNCTION_START(ADB_GetSubtitleStreamType);

   type = 0;
   if (stream_ptr != NULL)
   {
      type = ((S_STREAM *)stream_ptr)->subt_type;
   }

   FUNCTION_FINISH(ADB_GetSubtitleStreamType);
   return(type);
}

/**
 * @brief   Sets the primary audio language to be used
 * @param   country_code current country code
 * @param   db_lang_id language id
 */
void ADB_SetAudioLang(U32BIT country_code, U8BIT db_lang_id)
{
   U8BIT *lang_ids;

   FUNCTION_START(ADB_SetAudioLang);

   lang_ids = ACFG_GetDbLangId(country_code, db_lang_id);
   if (lang_ids != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetAudioLang(lang_ids);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetAudioLang);
}

/**
 * @brief   Sets the secondary audio language to be used. This will be selected
 *          if the primary audio language isn't available
 * @param   country_code current country code
 * @param   lang_id language id
 */
void ADB_SetSecondaryAudioLang(U32BIT country_code, U8BIT lang_id)
{
   U8BIT *lang_ids;

   FUNCTION_START(ADB_SetSecondaryAudioLang);

   lang_ids = ACFG_GetDbLangId(country_code, lang_id);
   if (lang_ids != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetSecondaryAudioLang(lang_ids);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetSecondaryAudioLang);
}

/**
 * @brief   Sets the primary subtitle and teletext language to be used
 * @param   country_code current country id
 * @param   lang_id language id
 */
void ADB_SetTextLang(U32BIT country_code, U8BIT lang_id)
{
   U8BIT *lang_ids;

   FUNCTION_START(ADB_SetTextLang);

   lang_ids = ACFG_GetDbLangId(country_code, lang_id);
   if (lang_ids != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetTextLang(lang_ids);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetTextLang);
}

/**
 * @brief   Sets the secondary subtitle and teletext language to be used. This
 *          will be used if the primary language isn't available
 * @param   country_code current country code
 * @param   lang_id language id
 */
void ADB_SetSecondaryTextLang(U32BIT country_code, U8BIT lang_id)
{
   U8BIT *lang_ids;

   FUNCTION_START(ADB_SetSecondaryTextLang);

   lang_ids = ACFG_GetDbLangId(country_code, lang_id);
   if (lang_ids != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetSecondaryTextLang(lang_ids);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetSecondaryTextLang);
}

/**
 * @brief   Makes copies of the now and/or next events (EITp/f) for the given service.
 *          The returned events should be freed using ADB_ReleaseEventData.
 * @param   serv_ptr get the now/next events for this service
 * @param   now_event address in which to return a pointer to the now event
 *          This value can be NULL if the now event isn't to be returned
 * @param   next_event address in which to return a pointer to the next event
 *          This value can be NULL if the now event isn't to be returned
 */
void ADB_GetNowNextEvents(void *serv_ptr, void **v_now_event, void **v_next_event)
{
   FUNCTION_START(ADB_GetNowNextEvents);

   if (v_now_event != NULL)
   {
      *v_now_event = NULL;
   }
   if (v_next_event != NULL)
   {
      *v_next_event = NULL;
   }

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      /* The FTA content management descriptor is only "valid" for the now
       * event, even if it is included in others for info purposes, but searching
       * for it in each event could take quite a while, so just do it for the now event */
      if (v_now_event != NULL)
      {
         *v_now_event = (void *)CopyEventRec(((ADB_SERVICE_REC *)serv_ptr)->now_event, (ADB_SERVICE_REC *)serv_ptr);
      }

      if (v_next_event != NULL)
      {
         *v_next_event = (void *)CopyEventRec(((ADB_SERVICE_REC *)serv_ptr)->next_event, (ADB_SERVICE_REC *)serv_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetNowNextEvents);
}

/**
 * @brief   Allocates and returns an array containing copies of events for the given
 *          service from the service's EIT schedule.
 *          The returned array should be freed using ADB_ReleaseEventList.
 * @param   include_old_events if TRUE, all events in the services schedule will be returned,
 *          but if FALSE then the first two events in the returned list
 *          will be the now/next events, followed by the remaining schedule
 * @param   serv_ptr service to get events for
 * @param   elist_ptr address in which to return a pointer to the alloctaed array of events
 * @param   num_entries_ptr address in which to return the number of events in the list
 */
void ADB_GetEventSchedule(BOOLEAN include_old_events, void *serv_ptr, void ***elist_ptr,
   U16BIT *num_entries_ptr)
{
   void **elist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *event_ptr;
   S_EVENT *tmp_event;
   U32DHMS end_time;

   FUNCTION_START(ADB_GetEventSchedule);

   // initialise return values
   num_entries = 0;
   elist = NULL;

   // Check service ptr is not null
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      max_no_entries = 0;

      if (!include_old_events)
      {
         /* Use now/next as the first 2 events if they're present */
         if (s_ptr->now_event != NULL)
         {
            max_no_entries++;
         }

         if (s_ptr->next_event != NULL)
         {
            max_no_entries++;
         }
      }

      max_no_entries += s_ptr->num_events_in_schedule;

      if (max_no_entries > 0)
      {
         elist = STB_AppGetMemory(max_no_entries * sizeof(void *));
         if (elist != NULL)
         {
            e_ptr = s_ptr->event_schedule;

            if (!include_old_events)
            {
               /* Start list with now/next followed by events from the schedule */
               tmp_event = NULL;
               event_ptr = NULL;

               if (s_ptr->now_event != NULL)
               {
                  tmp_event = CopyEventRec(s_ptr->now_event, s_ptr);
                  if (tmp_event != NULL)
                  {
                     event_ptr = s_ptr->now_event;
                     elist[num_entries] = tmp_event;
                     num_entries++;
                  }
               }

               if (s_ptr->next_event != NULL)
               {
                  tmp_event = CopyEventRec(s_ptr->next_event, s_ptr);
                  if (tmp_event != NULL)
                  {
                     event_ptr = s_ptr->next_event;
                     elist[num_entries] = tmp_event;
                     num_entries++;
                  }
               }

               if (event_ptr != NULL)
               {
                  /* Get the end time of the last event (either now or next) */
                  end_time = STB_GCCalculateDHMS(event_ptr->start, event_ptr->duration, CALC_ADD);

                  /* Find the first event in the schedule that starts after the 'next' event ends */
                  while (e_ptr != NULL)
                  {
                     if (end_time <= e_ptr->start)
                     {
                        break;
                     }
                     e_ptr = e_ptr->next;
                  }
               }
            }

            // go through event schedule copying each event in turn
            for (; (num_entries != max_no_entries) && (e_ptr != NULL); )
            {
               tmp_event = CopyEventRec(e_ptr, s_ptr);
               if (tmp_event != NULL)
               {
                  elist[num_entries] = tmp_event;
                  num_entries++;
               }
               e_ptr = e_ptr->next;
            }
         }
         else
         {
            num_entries = 0;
         }
      }

      DBDEF_ReleaseAccess();
   }

   *elist_ptr = elist;
   *num_entries_ptr = num_entries;

   FUNCTION_FINISH(ADB_GetEventSchedule);
}

/**
 * @brief   Creates a copy of the event schedule for the given service, but only includes
 *          events of the given genre type
 * @param   genre event genre to be matched
 * @param   include_old_events if FALSE, the first events returned will be the now/next events,
 *          if they match the genre, otherwise events that may have expired
 *          will also be included
 * @param   serv_ptr service
 * @param   elist_ptr returned with an allocated array of events
 * @param   num_entries_ptr returned with the number of entries in the returned array
 */
void ADB_GetEventScheduleByGenre(ADB_EVENT_CONTENT genre, BOOLEAN include_old_events, void *serv_ptr,
   void ***elist_ptr, U16BIT *num_entries_ptr)
{
   void **elist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *event_ptr;
   S_EVENT *tmp_event;
   U32DHMS end_time;

   FUNCTION_START(ADB_GetEventScheduleByGenre);

   // initailise return values
   num_entries = 0;
   elist = NULL;

   // Check service ptr is not null
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      max_no_entries = 0;

      if (!include_old_events)
      {
         /* Use now/next as the first 2 events if they're present */
         if (s_ptr->now_event != NULL)
         {
            max_no_entries++;
         }

         if (s_ptr->next_event != NULL)
         {
            max_no_entries++;
         }
      }

      max_no_entries += s_ptr->num_events_in_schedule;

      if (max_no_entries > 0)
      {
         elist = STB_AppGetMemory(max_no_entries * sizeof(void *));
         if (elist != NULL)
         {
            e_ptr = s_ptr->event_schedule;

            if (!include_old_events)
            {
               /* Start list with now/next followed by events from the schedule */
               tmp_event = NULL;
               event_ptr = NULL;

               if (s_ptr->now_event != NULL)
               {
                  if (DBDEF_GetEventGenre(s_ptr->now_event, s_ptr) == genre)
                  {
                     /* The FTA content management descriptor is only "valid" for the now
                      * event, even if it is included in others for info purposes, but searching
                      * for it in each event could take quite a while, so just do it for the now event */
                     tmp_event = CopyEventRec(s_ptr->now_event, s_ptr);
                     if (tmp_event != NULL)
                     {
                        event_ptr = s_ptr->now_event;
                        elist[num_entries] = tmp_event;
                        num_entries++;
                     }
                  }
               }

               if (s_ptr->next_event != NULL)
               {
                  if (DBDEF_GetEventGenre(s_ptr->next_event, s_ptr) == genre)
                  {
                     tmp_event = CopyEventRec(s_ptr->next_event, s_ptr);
                     if (tmp_event != NULL)
                     {
                        event_ptr = s_ptr->next_event;
                        elist[num_entries] = tmp_event;
                        num_entries++;
                     }
                  }
               }

               if (event_ptr != NULL)
               {
                  /* Get the end time of the last event (either now or next) */
                  end_time = STB_GCCalculateDHMS(event_ptr->start, event_ptr->duration, CALC_ADD);

                  /* Find the first event in the schedule that starts after the 'next' event ends */
                  while (e_ptr != NULL)
                  {
                     if (end_time <= e_ptr->start)
                     {
                        break;
                     }

                     e_ptr = e_ptr->next;
                  }
               }
            }

            // go through event schedule copying each event in turn
            for (; (num_entries < max_no_entries) && (e_ptr != NULL); )
            {
               if (DBDEF_GetEventGenre(e_ptr, s_ptr) == genre)
               {
                  tmp_event = CopyEventRec(e_ptr, s_ptr);
                  if (tmp_event != NULL)
                  {
                     elist[num_entries] = tmp_event;
                     num_entries++;
                  }
               }

               e_ptr = e_ptr->next;
            }
         }
         else
         {
            num_entries = 0;
         }
      }

      DBDEF_ReleaseAccess();
   }

   *elist_ptr = elist;
   *num_entries_ptr = num_entries;

   FUNCTION_FINISH(ADB_GetEventScheduleByGenre);
}

/**
 * @brief   Frees any memory allocated for the given event and the event itself
 * @param   event_ptr event to be freed
 */
void ADB_ReleaseEventData(void *event_ptr)
{
   FUNCTION_START(ADB_ReleaseEventData);

   if (event_ptr != NULL)
   {
      STB_AppFreeMemory(event_ptr);
   }

   FUNCTION_FINISH(ADB_ReleaseEventData);
}

/**
 * @brief   Frees all teh events in the given list and all associated memory for those events
 * @param   elist event list to be freed
 * @param   num_entries number of events in the list
 */
void ADB_ReleaseEventList(void **elist, U16BIT num_entries)
{
   U16BIT i;

   FUNCTION_START(ADB_ReleaseEventList);

   if (elist != NULL)
   {
      for (i = 0; i < num_entries; i++)
      {
         ADB_ReleaseEventData(elist[i]);
      }

      // release the event list itself
      STB_AppFreeMemory(elist);
   }

   FUNCTION_FINISH(ADB_ReleaseEventList);
}

/**
 * @brief   Deletes all events and event related data for the given service
 * @param   serv_ptr service for the event
 * @param   delete_now_next specifies whether the now/next events should also be deleted
 */
void ADB_DeleteServiceEvents(void *serv_ptr, BOOLEAN delete_now_next)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ADB_DeleteServiceEvents);

   if (serv_ptr != NULL)
   {
      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      DBDEF_RequestAccess();

      if (s_ptr->event_schedule != NULL)
      {
         DBDEF_DeleteEventList(s_ptr->event_schedule);
         s_ptr->event_schedule = NULL;
         s_ptr->num_events_in_schedule = 0;
         ASI_NotifyEitSchedUpdate(s_ptr, 0, APP_SI_EIT_JOURNAL_TYPE_CLEAR);
      }

      if (delete_now_next)
      {
         if (s_ptr->now_event != NULL)
         {
            DBDEF_DeleteEventList(s_ptr->now_event);
            s_ptr->now_event = NULL;
         }

         if (s_ptr->next_event != NULL)
         {
            DBDEF_DeleteEventList(s_ptr->next_event);
            s_ptr->next_event = NULL;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_DeleteServiceEvents);
}

/**
 * @brief   Deletes all events and event related data for all services
 * @param   delete_now_next specifies whether the now/next events should also be deleted
 */
void ADB_DeleteAllServiceEvents(BOOLEAN delete_now_next)
{
   U16BIT num_services;
   U16BIT index;
   void **slist;

   FUNCTION_START(ADB_DeleteAllServiceEvents);

   ADB_GetServiceList(ADB_SERVICE_LIST_ALL, &slist, &num_services);

   if ((num_services > 0) && (slist != NULL))
   {
      for (index = 0; index < num_services; index++)
      {
         if (slist[index] != NULL)
         {
            ADB_DeleteServiceEvents(slist[index], delete_now_next);
         }
      }

      STB_AppFreeMemory(slist);
   }

   FUNCTION_FINISH(ADB_DeleteAllServiceEvents);
}

/**
 * @brief   Returns a copy of the event with the given event ID on the given service
 * @param   serv_ptr service for the event
 * @param   event_id ID of the event to be returned
 * @return  pointer to a copy of the event, or NULL if the event isn't found
 */
void* ADB_GetEvent(void *serv_ptr, U16BIT event_id)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   void *event_ptr = NULL;

   FUNCTION_START(ADB_GetEvent);

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;
      e_ptr = DBDEF_FindScheduleEventById(s_ptr, event_id);
      if (e_ptr != NULL)
      {
         event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEvent);

   return(event_ptr);
}

/**
 * @brief   Returns a copy of the raw SI descriptor data with the given descriptor
 *          tag id and, optionally, extended descriptor tag id, for the event with
 *          the given event id.
 *          The data must be freed using ADB_ReleaseEventSIDescriptorData.
 * @param   serv_ptr service for the event
 * @param   event_id id of the event on the service
 * @param   dtag_id descriptor tag id
 * @param   ext_dtag_id extended descriptor tag id, or negative if not to be used
 * @param   desc_len number of bytes in the returned descriptor data
 * @return  pointer to the descriptor data, or NULL if event or descriptor isn't found
 */
U8BIT* ADB_GetEventSIDescriptorData(void *serv_ptr, U16BIT event_id, U8BIT dtag_id,
   S16BIT ext_dtag_id, U16BIT *desc_len)
{
   U8BIT *desc_data;
#ifdef INTEGRATE_HBBTV
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_DESC *event_desc;
   U16BIT num_bytes;
   U8BIT *desc_ptr;
#endif

   FUNCTION_START(ADB_GetEventSIDescriptorData);

   desc_data = NULL;
   *desc_len = 0;

#ifdef INTEGRATE_HBBTV
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;
      e_ptr = DBDEF_FindScheduleEventById(s_ptr, event_id);
      if (e_ptr != NULL)
      {
         /* Search for the requested descriptor to get the number of bytes to be returned */
         num_bytes = 0;
         event_desc = e_ptr->desc_list_head;

         while ((event_desc = DBDEF_FindEventDescriptor(event_desc, dtag_id, 0)) != NULL)
         {
            if ((ext_dtag_id < 0) ||
               ((event_desc->desc_data[0] == EXTENSION_DTAG) && (event_desc->desc_data[2] == ext_dtag_id)))
            {
               /* This descriptor needs to be copied
                * +2 for the dtag and length */
               num_bytes += event_desc->desc_data[1] + 2;
            }

            event_desc = event_desc->next;
         }

         /* Check there's data to be returned */
         if (num_bytes > 0)
         {
            if ((desc_data = STB_AppGetMemory(num_bytes)) != NULL)
            {
               *desc_len = num_bytes;
               desc_ptr = desc_data;

               event_desc = e_ptr->desc_list_head;

               while ((event_desc = DBDEF_FindEventDescriptor(event_desc, dtag_id, 0)) != NULL)
               {
                  if ((ext_dtag_id < 0) ||
                     ((event_desc->desc_data[0] == EXTENSION_DTAG) && (event_desc->desc_data[2] == ext_dtag_id)))
                  {
                     /* This descriptor needs to be copied */
                     memcpy(desc_ptr, &event_desc->desc_data[0], event_desc->desc_data[1] + 2);
                     desc_ptr += event_desc->desc_data[1] + 2;
                  }

                  event_desc = event_desc->next;
               }
            }
         }
      }

      DBDEF_ReleaseAccess();
   }
#else
   USE_UNWANTED_PARAM(serv_ptr);
   USE_UNWANTED_PARAM(event_id);
   USE_UNWANTED_PARAM(dtag_id);
   USE_UNWANTED_PARAM(ext_dtag_id);
#endif

   FUNCTION_FINISH(ADB_GetEventSIDescriptorData);

   return(desc_data);
}

/**
 * @brief   Frees the data returned by ADB_GetEventSIDescriptorData
 * @param   desc_data data to be freed
 * @param   desc_len number of bytes
 */
void ADB_ReleaseEventSIDescriptorData(U8BIT *desc_data, U16BIT desc_len)
{
   FUNCTION_START(ADB_ReleaseEventSIDescriptorData);

   if ((desc_data != NULL) && (desc_len > 0))
   {
      STB_AppFreeMemory(desc_data);
   }

   FUNCTION_FINISH(ADB_ReleaseEventSIDescriptorData);
}

/**
 * @brief   Returns a copy of the event preceeding the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   time Date/time from which to start searching
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_EarlierEvent(void *serv_ptr, U32DHMS time)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *prev_ptr;
   void *event_ptr = NULL;

   FUNCTION_START(ADB_GetEarlierEvent);

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      if (s_ptr->num_events_in_schedule != 0)
      {
         /* Find the first schedule event with a date/time >= the given time */
         prev_ptr = NULL;
         for (e_ptr = s_ptr->event_schedule; e_ptr != NULL; e_ptr = e_ptr->next)
         {
            if (e_ptr->start >= time)
            {
               if (prev_ptr != NULL)
               {
                  /* The previous event is the earlier event to be returned */
                  event_ptr = (void *)CopyEventRec(prev_ptr, s_ptr);
               }
               break;
            }

            prev_ptr = e_ptr;
         }
      }
      else
      {
         /* There is no schedule yet so just check the now/next events */
         if (s_ptr->now_event != NULL)
         {
            e_ptr = s_ptr->now_event;

            if (e_ptr->start < time)
            {
               event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
            }
         }
         else if (s_ptr->next_event != NULL)
         {
            e_ptr = s_ptr->next_event;

            if (e_ptr->start < time)
            {
               event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEarlierEvent);

   return(event_ptr);
}

/**
 * @brief   Returns a copy of the event preceeding the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   date MJD date code
 * @param   hour hour, 0-23
 * @param   min minute, 0-59
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_GetEarlierEvent(void *serv_ptr, U16BIT date, U8BIT hour, U8BIT min)
{
   return ADB_EarlierEvent(serv_ptr, DHMS_CREATE(date, hour, min, 0));
}

/**
 * @brief   Returns a copy of the event following the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   time Date/time from which to start searching
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_LaterEvent(void *serv_ptr, U32DHMS time)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   void *event_ptr = NULL;

   FUNCTION_START(ADB_GetLaterEvent);

   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      if (s_ptr->num_events_in_schedule != 0)
      {
         /* Iterate through the event schedule to find the first event with a
          * start date/time later than the given date/time */
         for (e_ptr = s_ptr->event_schedule; e_ptr != NULL; e_ptr = e_ptr->next)
         {
            if (e_ptr->start >= time)
            {
               event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
               break;
            }
         }
      }
      else
      {
         /* There is no schedule yet so just check the now/next events */
         if (s_ptr->now_event != NULL)
         {
            e_ptr = s_ptr->now_event;

            if (e_ptr->start >= time)
            {
               event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
            }
         }
         else if (s_ptr->next_event != NULL)
         {
            e_ptr = s_ptr->next_event;

            if (e_ptr->start >= time)
            {
               event_ptr = (void *)CopyEventRec(e_ptr, s_ptr);
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetLaterEvent);

   return(event_ptr);
}

/**
 * @brief   Returns a copy of the event following the given date/time on the given service
 * @param   serv_ptr service for the event
 * @param   date MJD date code
 * @param   hour hour, 0-23
 * @param   min minute, 0-59
 * @return  pointer to a copy of the event, or NULL if an event isn't found
 */
void* ADB_GetLaterEvent(void *serv_ptr, U16BIT date, U8BIT hour, U8BIT min)
{
   return ADB_LaterEvent(serv_ptr, DHMS_CREATE(date, hour, min, 0));
}

/**
 * @brief   Finds the event scheduled to be broadcast at the specified time on the specified service
 * @param   time date/time when the event is scheduled for broadcast
 * @param   serv_ptr handle of the service on which the event is scheduled for broadcast
 * @return  event handler, NULL if not found. The event data must be released with ADB_ReleaseEventData.
 */
void* ADB_FindEventFromTime(U32DHMS time, void *serv_ptr)
{
   void *retval;
   U32DHMS end_time;
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *event_ptr;

   FUNCTION_START(ADB_FindEventFromTime);

   // initialise return values
   retval = NULL;

   // Check service ptr is not null
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      end_time = 0;

      /* Use now/next as the first 2 events if they're present */
      event_ptr = s_ptr->now_event;
      if (event_ptr != NULL)
      {
         /* Get the end time of the "now" event */
         end_time = STB_GCCalculateDHMS(event_ptr->start, event_ptr->duration, CALC_ADD);

         if( EventAtTime(event_ptr, time) )
         {
            retval = CopyEventRec(event_ptr, s_ptr);
         }
      }

      if (retval == NULL)
      {
         event_ptr = s_ptr->next_event;
         if (event_ptr != NULL)
         {
            /* Get the end time of the "next" event */
            end_time = STB_GCCalculateDHMS(event_ptr->start, event_ptr->duration, CALC_ADD);

            if( EventAtTime(event_ptr, time) )
            {
               retval = CopyEventRec(event_ptr, s_ptr);
            }
         }
      }

      DBDEF_ReleaseAccess();

      if (retval == NULL)
      {
         if(end_time <= time)
         {
            retval = ADB_FindEITScheduleEventFromTime(time, serv_ptr);
         }
      }
   }

   FUNCTION_FINISH(ADB_FindEventFromTime);

   return retval;
}

void* ADB_FindEITScheduleEventFromTime(U32DHMS time, void *serv_ptr)
{
   void *retval;
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;

   FUNCTION_START(ADB_FindEITScheduleEventFromTime);

   // initialise return values
   retval = NULL;

   // Check service ptr is not null
   if (serv_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)serv_ptr;

      if (s_ptr->num_events_in_schedule != 0)
      {
         // go through event schedule checking each event in turn
         for(e_ptr = s_ptr->event_schedule; e_ptr != NULL; e_ptr = e_ptr->next)
         {
            if ( EventAtTime(e_ptr, time) )
            {
               retval = CopyEventRec(e_ptr, s_ptr);
               break;
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_FindEITScheduleEventFromTime);

   return retval;
}

/**
 * @brief   Returns a value representing the date and time of the start of the given event
 * @param   event_ptr event
 * @return  date/time
 */
U32DHMS ADB_GetEventStartDateTime(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U32DHMS start;

   FUNCTION_START(ADB_GetEventStartDateTime);

   start = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         start = e_ptr->start;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventStartDateTime);

   return(start);
}

/**
 * @brief   Returns the MJD date code for the start of the given event
 * @param   event_ptr event
 * @return  date code
 */
U16BIT ADB_GetEventStartDate(void *event_ptr)
{
   return DHMS_DATE(ADB_GetEventStartDateTime(event_ptr));
}

/**
 * @brief   Returns the hour (0-23) value for the start of the given event
 * @param   event_ptr event
 * @return  hour value (0-23)
 */
U8BIT ADB_GetEventStartHour(void *event_ptr)
{
   return DHMS_HOUR(ADB_GetEventStartDateTime(event_ptr));
}

/**
 * @brief   Returns the minute (0-59) value for the start of the given event
 * @param   event_ptr event
 * @return  minute value (0-59)
 */
U8BIT ADB_GetEventStartMin(void *event_ptr)
{
   return DHMS_MINS(ADB_GetEventStartDateTime(event_ptr));
}

/**
 * @brief   Returns the seconds (0-59) value for the start of the given event
 * @param   event_ptr event
 * @return  seconds value (0-59)
 */
U8BIT ADB_GetEventStartSecs(void *event_ptr)
{
   return DHMS_SECS(ADB_GetEventStartDateTime(event_ptr));
}

/**
 * @brief   Returns a value representing the duration of the given event
 * @param   event_ptr event
 * @return  duration
 */
U32DHMS ADB_GetEventDuration(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U32DHMS duration;

   FUNCTION_START(ADB_GetEventDuration);

   duration = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         duration = e_ptr->duration;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventDuration);

   return(duration);
}

/**
 * @brief   Returns the hour value for the duration of the given event
 * @param   event_ptr event
 * @return  hour value
 */
U8BIT ADB_GetEventDurationHour(void *event_ptr)
{
   return DHMS_HOUR(ADB_GetEventDuration(event_ptr));
}

/**
 * @brief   Returns the minute value (0-59) for the duration of the given event
 * @param   event_ptr event
 * @return  minute value (0-59)
 */
U8BIT ADB_GetEventDurationMin(void *event_ptr)
{
   return DHMS_MINS(ADB_GetEventDuration(event_ptr));
}

/**
 * @brief   Returns the seconds value (0-59) for the duration of the given event
 * @param   event_ptr event
 * @return  seconds value (0-59)
 */
U8BIT ADB_GetEventDurationSecs(void *event_ptr)
{
   return DHMS_SECS(ADB_GetEventDuration(event_ptr));
}

/**
 * @brief   Returns a value representing the date and time of the end of the given event
 * @param   event_ptr event
 * @return  date/time
 */
U32DHMS ADB_GetEventEndDateTime(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U32DHMS end;

   FUNCTION_START(ADB_GetEventEndDateTime);

   end = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         end = STB_GCCalculateDHMS(e_ptr->start, e_ptr->duration, CALC_ADD);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventEndDateTime);

   return(end);
}

/**
 * @brief   Returns whether DVB subtitles are signalled as being available for the given event
 * @param   event_ptr event
 * @return  TRUE if subtitles are available, FALSE otherwise
 */
BOOLEAN ADB_GetEventSubtitlesAvailFlag(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   BOOLEAN available;

   FUNCTION_START(ADB_GetEventSubtitlesAvailFlag);

   available = FALSE;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         available = DBDEF_GetEventSubtitlesAvailFlag(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventSubtitlesAvailFlag);

   return(available);
}

/**
 * @brief   Returns the level 1 value of the first content identifier for the given event
 * @param   event_ptr event
 * @return  event content value, ADB_EVENT_CONTENT_UNCLASSIFIED if not available
 */
ADB_EVENT_CONTENT ADB_GetEventContentDesc(void *event_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_CONTENT genre;

   FUNCTION_START(ADB_GetEventContentDesc);

   genre = ADB_EVENT_CONTENT_UNCLASSIFIED;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      s_ptr = (ADB_SERVICE_REC *)((S_EVENT *)event_ptr)->serv_ptr;

      e_ptr = DBDEF_FindScheduleEventById(s_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         genre = DBDEF_GetEventGenre(e_ptr, s_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventContentDesc);

   return(genre);
}

/**
 * @brief   Returns the level 1 and 2 values for all content identifiers for the given event
 * @param   event_ptr event
 * @param   p_len address to return the number of bytes in the returned data
 * @return  pointer to the content data
 */
U8BIT* ADB_GetEventContentData(void *event_ptr, U8BIT *p_len)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *retval;

   FUNCTION_START(ADB_GetEventContentData);

   retval = NULL;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         retval = DBDEF_GetEventContentData(e_ptr, p_len);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventContentData);

   return(retval);
}

/**
 * @brief   Returns the parental age value for the given event
 * @param   event_ptr event
 * @return  parental age value, 0 for invalid event
 */
U8BIT ADB_GetEventParentalAge(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT age;

   FUNCTION_START(ADB_GetEventParentalAge);

   age = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         age = DBDEF_GetEventParentalAge(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventParentalAge);

   return(age);
}

/**
 * @brief   Returns the name of the event as a UTF-8 string in the currently selected language.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventName(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *name_str;

   FUNCTION_START(ADB_GetEventName);

   name_str = NULL;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         name_str = DBDEF_GetEventName(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventName);

   return(name_str);
}

/**
 * @brief   Returns the short event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventDescription(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *desc_str;

   FUNCTION_START(ADB_GetEventDescription);

   desc_str = NULL;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         desc_str = DBDEF_GetEventDescription(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventDescription);

   return(desc_str);
}

/**
 * @brief   Checks whether the given event has any extended event descriptors
 * @param   event_ptr event
 * @return  TRUE if the event has at least one extended event descriptor, FALSE otherwise
 */
BOOLEAN ADB_GetEventHasExtendedDescription(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   BOOLEAN retval;

   FUNCTION_START(ADB_GetEventHasExtendedDescription);

   retval = FALSE;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         if (DBDEF_FindEventDescriptor(e_ptr->desc_list_head, EXTENDED_EVENT_DTAG, 0) != NULL)
         {
            retval = TRUE;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventHasExtendedDescription);

   return(retval);
}

/**
 * @brief   Returns the extended event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   event_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* ADB_GetEventExtendedDescription(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *desc_str;

   FUNCTION_START(ADB_GetEventExtendedDescription);

   desc_str = NULL;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         desc_str = DBDEF_GetEventExtendedDescription(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventExtendedDescription);

   return(desc_str);
}

/**
 * @brief   Returns the items of extended event descriptor as item descriptor and item itself.
 * @param   event_ptr event
 * @param   num_items_ptr address in which to return the number of items
 * @return  ADB_EVENT_ITEMIZED_INFO pointer to the list of items, NULL on error
 */
ADB_EVENT_ITEMIZED_INFO* ADB_GetEventItemizedDescription(void *event_ptr, U16BIT *num_items_ptr)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_ITEMIZED_INFO *event_itemized_info;

   FUNCTION_START(ADB_GetEventItemizedDescription);

   event_itemized_info = NULL;
   *num_items_ptr = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         event_itemized_info = DBDEF_GetEventItemizedDescription(e_ptr, num_items_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventItemizedDescription);

   return(event_itemized_info);
}

/**
 * @brief   Frees the memory used by event itemized data
 * @param   event_itemized_info pointer of info
 * @param   num_of_items number of items to release
 */
void ADB_ReleaseEventItemizedInfo(ADB_EVENT_ITEMIZED_INFO *event_itemized_info, U16BIT num_of_items)
{
   U16BIT i;

   FUNCTION_START(ADB_ReleaseEventItemizedInfo);

   if (event_itemized_info != NULL)
   {
      for (i = 0; i < num_of_items; i++)
      {
         STB_ReleaseUnicodeString(event_itemized_info[i].item_description);
         STB_ReleaseUnicodeString(event_itemized_info[i].item);
      }
      STB_AppFreeMemory(event_itemized_info);
   }
   FUNCTION_FINISH(ADB_ReleaseEventItemizedInfo);
}

/**
 * @brief   Returns the guidance text for an event, either from the event itself
 *          or the event's service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   event_ptr event, can be NULL, in which case any guidance defined by the
 *                    service will be returned
 * @param   serv_ptr service, can be NULL to just get guidance text from the event
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* ADB_GetEventGuidance(void *event_ptr, void *serv_ptr, U8BIT *type, U8BIT *mode)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *text_str;

   FUNCTION_START(ADB_GetEventGuidance);

   text_str = NULL;

   DBDEF_RequestAccess();
   if (event_ptr != NULL)
   {
      e_ptr = DBDEF_FindScheduleEventById(serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         text_str = DBDEF_GetEventGuidance(e_ptr, serv_ptr, type, mode);
      }
   }
   else
   {
      text_str = DBDEF_GetServiceGuidanceData(serv_ptr, type, mode);
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetEventGuidance);

   return(text_str);
}

/**
 * @brief   Returns the event id for the given event
 * @param   event_ptr event
 * @return  event id
 */
U16BIT ADB_GetEventId(void *event_ptr)
{
   U16BIT event_id = 0;

   FUNCTION_START(ADB_GetEventId);

   if (event_ptr != NULL)
   {
      event_id = ((S_EVENT *)event_ptr)->event_id;
   }

   FUNCTION_FINISH(ADB_GetEventId);

   return(event_id);
}

/**
 * @brief   Returns whether audio description is signalled as being available for the given event
 * @param   event_ptr event
 * @return  TRUE if available, FALSE otherwise
 */
BOOLEAN ADB_GetEventAudioDescriptionFlag(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   BOOLEAN audio_desc;

   FUNCTION_START(ADB_GetEventAudioDescriptionFlag);

   audio_desc = FALSE;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         audio_desc = DBDEF_GetEventAudioDescriptionFlag(e_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventAudioDescriptionFlag);

   return(audio_desc);
}

/**
 * @brief   Returns whether the event is signalled with none of its streams being CA scrambled
 * @param   event_ptr pointer to the event to check
 * @return  TRUE if the event all streams are free-to-air, FALSE otherwise
 */
BOOLEAN ADB_GetEventFreeToAir(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   BOOLEAN fta;

   FUNCTION_START(ADB_GetEventFreeToAir);

   fta = FALSE;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         fta = e_ptr->free_to_air;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventFreeToAir);

   return(fta);
}

/**
 * @brief   Returns the do_not_scramble flag for the event, which will have been
 *          found by looking backwards in the event, SDT, transport and NIT, until
 *          found or not.
 * @param   event_ptr pointer to the event to check
 * @return  FALSE if the event is to be protected, TRUE otherwise
 */
BOOLEAN ADB_GetEventDoNotScramble(void *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   BOOLEAN do_not_scramble;

   FUNCTION_START(ADB_GetEventDoNotScramble);

   do_not_scramble = TRUE;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         do_not_scramble = FindDoNotScramble(e_ptr, ((S_EVENT *)event_ptr)->serv_ptr);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventDoNotScramble);

   return(do_not_scramble);
}

/**
 * @brief   Returns whether there's an HD event linked to the SD event and returns
 *          the info for it. The DVB info is checked to ensure it relates to a known
 *          service.
 * @param   event_ptr pointer to the event to check
 * @param   verify_event TRUE, if the schedule should be checked for the target event
 * @param   only_simulcast TRUE, if only an HD simulcast event should be returned
 * @param   hd_serv_ptr pointer to return the service for the HD event, if found.
 *          This can be NULL if not required.
 * @param   hd_event_ptr pointer to return the HD event, if found. Note that this will only be
 *          filled in if verify_event is TRUE. This can be NULL if not required.
 * @return  TRUE if an event is found, FALSE otherwise
 */
BOOLEAN ADB_GetEventHDLinkageInfo(void *event_ptr, BOOLEAN verify_event, BOOLEAN only_simulcast,
   void **hd_serv_ptr, void **hd_event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *hd_event;
   ADB_SERVICE_REC *hd_service;
   BOOLEAN event_found;

   FUNCTION_START(ADB_GetEventHDLinkageInfo);

   event_found = FALSE;

   if (hd_serv_ptr != NULL)
   {
      *hd_serv_ptr = NULL;
   }

   if (hd_event_ptr != NULL)
   {
      *hd_event_ptr = NULL;
   }

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         event_found = DBDEF_GetEventHDLinkageInfo(e_ptr, verify_event, only_simulcast, &hd_service, &hd_event);

         if (event_found && (hd_event_ptr != NULL))
         {
            *hd_event_ptr = (void *)CopyEventRec(hd_event, hd_service);
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventHDLinkageInfo);

   return(event_found);
}

/**
 * @brief   Checks whether the two events are the same.
 *          Not all fields are checked, just date, time, duration and event ID
 * @param   event1_ptr pointer to the first of the events to check
 * @param   event2_ptr pointer to the second of the events to check
 * @return  TRUE if the two events are the same, FALSE otherwise
 */
BOOLEAN ADB_IsSameEvent(void *event1_ptr, void *event2_ptr)
{
   ADB_EVENT_REC *e1_ptr;
   ADB_EVENT_REC *e2_ptr;
   BOOLEAN same;

   FUNCTION_START(ADB_IsSameEvent);

   same = FALSE;

   if ((event1_ptr != NULL) && (event2_ptr != NULL))
   {
      DBDEF_RequestAccess();

      e1_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event1_ptr)->serv_ptr, ((S_EVENT *)event1_ptr)->event_id);
      if (e1_ptr != NULL)
      {
         e2_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event2_ptr)->serv_ptr,
            ((S_EVENT *)event2_ptr)->event_id);
         if (e2_ptr != NULL)
         {
            if ((e1_ptr->start == e2_ptr->start) &&
                (e1_ptr->duration == e2_ptr->duration) && (e1_ptr->event_id == e2_ptr->event_id))
            {
               same = TRUE;
            }
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_IsSameEvent);

   return(same);
}

/**
 * @brief   Returns the full programme CRID of the given event (including IMI).
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a programme CRID
 */
U8BIT* ADB_GetEventFullProgrammeCrid(void *serv_ptr, void *event_ptr)
{
   U8BIT *crid_str;

   FUNCTION_START(ADB_GetEventFullProgrammeCrid);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, UK_PROGRAMME_CRID, 0);
   }
   else
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, TVA_PROGRAMME_CRID, 0);
   }

   FUNCTION_FINISH(ADB_GetEventFullProgrammeCrid);

   return(crid_str);
}

/**
 * @brief   Returns TRUE if the given CRID represents a split event (i.e. it
 *          contains an Instance Metadata Identifier (IMI)).
 * @param   crid CRID string
 * @return  TRUE if an IMI is found
 */
BOOLEAN ADB_IsSplitProgrammeCrid(U8BIT *crid)
{
   U16BIT crid_len;
   U8BIT *crid_ptr;
   BOOLEAN imi_found;

   FUNCTION_START(ADB_IsSplitProgrammeCrid);

   imi_found = FALSE;

   if (crid != NULL)
   {
      crid_len = strlen((char *)crid);
      if (crid_len > 0)
      {
         /* Search backwards through the string looking for a '#' char.
          * The search is terminated by finding a '/' or the start of the string */
         crid_ptr = crid + crid_len - 1;
         while ((crid_ptr != crid) && !imi_found)
         {
            if (*crid_ptr == '#')
            {
               imi_found = TRUE;
            }
            else if (*crid_ptr == '/')
            {
               /* No IMI found in the last section of the CRID string */
               crid_ptr = crid;
            }
            else
            {
               crid_ptr--;
            }
         }
      }
   }

   FUNCTION_FINISH(ADB_IsSplitProgrammeCrid);

   return(imi_found);
}

/**
 * @brief   Returns the programme CRID of the given event (excluding IMI)
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a programme CRID
 */
U8BIT* ADB_GetEventProgrammeCrid(void *serv_ptr, void *event_ptr)
{
   U8BIT *crid_str;
   U16BIT crid_len;
   U8BIT *crid_ptr;
   BOOLEAN imi_found;

   FUNCTION_START(ADB_GetEventProgrammeCrid);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, UK_PROGRAMME_CRID, 0);
   }
   else
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, TVA_PROGRAMME_CRID, 0);
   }

   if ((crid_str != NULL) && ADB_IsSplitProgrammeCrid(crid_str))
   {
      /* Search backwards through the string looking for a '#' char.
       * The search is terminated by finding a '/' or the start of the string */
      crid_len = strlen((char *)crid_str);
      crid_ptr = crid_str + crid_len - 1;
      imi_found = FALSE;

      while ((crid_ptr != crid_str) && !imi_found)
      {
         if (*crid_ptr == '#')
         {
            /* Terminate the crid string here */
            *crid_ptr = '\0';
            imi_found = TRUE;
         }
         else if (*crid_ptr == '/')
         {
            /* No IMI found in the last section of the CRID string */
            crid_ptr = crid_str;
         }
         else
         {
            crid_ptr--;
         }
      }
   }

   FUNCTION_FINISH(ADB_GetEventProgrammeCrid);

   return(crid_str);
}

/**
 * @brief   Returns the full CRID for the given CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_str CRID string
 * @return  pointer to a string containing the CRID name
 */
U8BIT* ADB_GetFullCrid(void *serv_ptr, U8BIT *event_str)
{
   U8BIT *crid_str = NULL;

   FUNCTION_START(ADB_GetFullCrid);

   ASSERT(serv_ptr != NULL);
   ASSERT(event_str != NULL);

   if ((serv_ptr != NULL) && (event_str != NULL))
   {
      DBDEF_RequestAccess();
      crid_str = DBDEF_GetFullCrid(serv_ptr, event_str);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetFullCrid);

   return(crid_str);
}

/**
 * @brief   Returns the full series CRID of the given event
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   index 0 based index of series CRID to be returned
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a series CRID
 */
U8BIT* ADB_GetEventSeriesCrid(U8BIT index, void *serv_ptr, void *event_ptr)
{
   U8BIT *crid_str;

   FUNCTION_START(ADB_GetEventSeriesCrid);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, UK_SERIES_CRID, index);
   }
   else
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, TVA_SERIES_CRID, index);
   }

   FUNCTION_FINISH(ADB_GetEventSeriesCrid);

   return(crid_str);
}

/**
 * @brief   Returns the episode name of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  TRUE: valid series info, FALSE: invalid series info
 */
BOOLEAN ADB_GetEventSeriesInfo(void *event_ptr, U8BIT **episode_name, S32BIT *episode_number, U8BIT *content_status, BOOLEAN *last_episode)
{
   BOOLEAN ret = FALSE;
   ADB_EVENT_REC *e_ptr;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(DBDEF_GetEventSeriesInfo);
   if (event_ptr == NULL)
       goto end;

   s_ptr = ((S_EVENT *)event_ptr)->serv_ptr;
   if (s_ptr == NULL)
       goto end;

   DBDEF_RequestAccess();

   e_ptr = DBDEF_FindScheduleEventById(s_ptr, ((S_EVENT *)event_ptr)->event_id);
   if (NULL != e_ptr)
   {
      ret = DBDEF_GetEventSeriesInfo(e_ptr, episode_name, episode_number, content_status, last_episode);
      ADB_TRANSPORT_REC *t_ptr = s_ptr->transport;
      if ((TRUE == ret) && (NULL != t_ptr))
      {
         if (1 == t_ptr->orig_net_id)
         {
            // FIXME: for Hsinchu CATV series linking descriptor
            *episode_number = (*episode_number) / 2;
         }
      }
   }

   DBDEF_ReleaseAccess();

end:
   FUNCTION_FINISH(DBDEF_GetEventSeriesInfo);

   return(ret);
}

/**
 * @brief   Returns the number of series CRIDs for the given event
 * @param   event_ptr pointer to the event
 * @return  number of series CRIDs
 */
U8BIT ADB_GetEventNumSeriesCrids(void *event_ptr)
{
   U8BIT num_crids;

   FUNCTION_START(ADB_GetEventNumSeriesCrids);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      num_crids = NumberOfCridsOfType((S_EVENT *)event_ptr, UK_SERIES_CRID);
   }
   else
   {
      num_crids = NumberOfCridsOfType((S_EVENT *)event_ptr, TVA_SERIES_CRID);
   }

   FUNCTION_FINISH(ADB_GetEventNumSeriesCrids);

   return(num_crids);
}

/**
 * @brief   Returns the number of recommendation CRIDs for the given event
 * @param   event_ptr pointer to the event
 * @return  number of recommendation CRIDs
 */
U8BIT ADB_GetEventNumRecommendationCrids(void *event_ptr)
{
   U8BIT num_crids;

   FUNCTION_START(ADB_GetEventNumRecommendationCrids);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      num_crids = NumberOfCridsOfType((S_EVENT *)event_ptr, UK_RECOMMENDATION_CRID);
   }
   else
   {
      num_crids = NumberOfCridsOfType((S_EVENT *)event_ptr, TVA_RECOMMENDATION_CRID);
   }

   FUNCTION_FINISH(ADB_GetEventNumRecommendationCrids);

   return(num_crids);
}

/**
 * @brief   Returns the full recommendation CRID of the given event
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   index 0 based index of recommendation CRID to be returned
 * @param   serv_ptr service for the event
 * @param   event_ptr event for which the CRID is to be returned
 * @return  pointer to a string containing the CRID name, or NULL if there isn't a CRID
 */
U8BIT* ADB_GetEventRecommendationCrid(U8BIT index, void *serv_ptr, void *event_ptr)
{
   U8BIT *crid_str;

   FUNCTION_START(ADB_GetEventRecommendationCrid);

   if (ACFG_GetCountry() == COUNTRY_CODE_UK)
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, UK_RECOMMENDATION_CRID, index);
   }
   else
   {
      crid_str = GetEventCridType(serv_ptr, event_ptr, TVA_RECOMMENDATION_CRID, index);
   }

   FUNCTION_FINISH(ADB_GetEventRecommendationCrid);

   return(crid_str);
}

/**
 * @brief   Retrieves a list of components associated with the specified event, as described by
 *          component descriptors in the EIT.
 * @param   event_ptr Pointer to the event
 * @param   component_list Pointer to the returned list of components. The list must be freed by
 *          calling STB_AppFreeMemory, provided the returned number of components is greater than 0.
 * @return  Number of components associated with the specified event.
 */
U8BIT ADB_GetEventComponentList(void *event_ptr, ADB_EVENT_COMPONENT_INFO **component_list)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT num_components;

   FUNCTION_START(ADB_GetEventComponentList);

   num_components = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         num_components = DBDEF_GetEventComponentList(e_ptr, component_list);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetEventComponentList);

   return(num_components);
}

/**
 * @brief   Returns a copy of the authority part of the CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   crid CRID string
 * @return  authority string
 */
U8BIT* ADB_GetAuthorityFromCrid(U8BIT *crid)
{
   U8BIT *auth_str;
   U16BIT auth_str_len;
   U16BIT i;

   FUNCTION_START(ADB_GetAuthorityFromCrid);

   ASSERT(crid != NULL);

   auth_str_len = strlen((char *)crid);

   for (i = 0; i < auth_str_len; i++)
   {
      if (crid[i] == '/')
      {
         auth_str_len = i + 1;
         break;
      }
   }

   auth_str = (U8BIT *)STB_AppGetMemory(auth_str_len);
   if (auth_str != NULL)
   {
      memcpy(auth_str, crid, auth_str_len);
      auth_str[auth_str_len - 1] = '\0';
   }

   FUNCTION_FINISH(ADB_GetAuthorityFromCrid);

   return(auth_str);
}

/**
 * @brief   Finds an alternative event for the given programme CRID and returns
 *          a pointer to a copy of the event and a pointer to the service
 * @param   prog_crid programme CRID to search for
 * @param   original_event_ptr pointer to the event for which an alternative is being sought, can be NULL
 * @param   serv_ptr pointer to return the service record if an alternate event is found
 * @return  pointer to a copy of the event, or NULL
 */
void* ADB_FindEventFromCrid(U8BIT *prog_crid, void *original_event_ptr, void **serv_ptr)
{
   void **slist;
   void **elist;
   void *event_ptr;
   U16BIT num_services;
   U16BIT num_events;
   U8BIT *event_prog_crid;
   U16BIT si, ei;
   U32DHMS now_time;
   BOOLEAN event_found;
   void *found_event_ptr;
   U16BIT onet_id, tran_id, serv_id;
   U16BIT event_id;

   FUNCTION_START(ADB_FindEventFromCrid);

   found_event_ptr = NULL;

   ADB_GetServiceList(ADB_SERVICE_LIST_ALL, &slist, &num_services);

   if ((num_services > 0) && (slist != NULL))
   {
      event_found = FALSE;

      for (si = 0; !event_found && (si < num_services); si++)
      {
         ADB_GetEventSchedule(FALSE, slist[si], &elist, &num_events);

         if ((num_events > 0) && (elist != NULL))
         {
            /* Get the current date/time to check for old events */
            now_time = STB_GCNowDHMSGmt();

            for (ei = 0; !event_found && (ei < num_events); ei++)
            {
               event_ptr = elist[ei];

               /* Check that the event isn't already in the past */
               if (ADB_GetEventStartDateTime(event_ptr) >= now_time)
               {
                  event_prog_crid = ADB_GetEventProgrammeCrid(slist[si], event_ptr);
                  if (event_prog_crid != NULL)
                  {
                     if (STB_CompareStringsIgnoreCase(event_prog_crid, prog_crid) == 0)
                     {
                        /* The programme CRIDs match so this must be the same programme content */
                        if ((original_event_ptr != NULL) && ADB_IsSameEvent(original_event_ptr, event_ptr))
                        {
                           /* This is the same event so it can't be used */
                           event_ptr = NULL;
                        }

                        if (event_ptr != NULL)
                        {
                           /* Check that this event isn't already set to be recorded */
                           ADB_GetServiceIds(slist[si], &onet_id, &tran_id, &serv_id);

                           event_id = ADB_GetEventId(event_ptr);

                           if (ATMR_FindTimerFromEvent(onet_id, tran_id, serv_id, event_id) != INVALID_TIMER_HANDLE)
                           {
                              /* An altenate event has been found so return a copy of it */
                              event_found = TRUE;
                              *serv_ptr = slist[si];
                              found_event_ptr = ADB_GetEvent(slist[si], event_id);
                           }
                        }
                     }

                     STB_AppFreeMemory(event_prog_crid);
                  }
               }
            }

            ADB_ReleaseEventList(elist, num_events);
         }
      }

      STB_AppFreeMemory(slist);
   }

   FUNCTION_FINISH(ADB_FindEventFromCrid);

   return(found_event_ptr);
}

/**
 * @brief   Sets the hidden status of the given service, which means the service
 *          would not be presented to the user in any list of available services.
 * @param   s_ptr service
 * @param   hidden TRUE to hide the service, FALSE to make it visible
 */
void ADB_SetServiceHiddenFlag(void *s_ptr, BOOLEAN hidden)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetServiceHiddenFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      serv_ptr->hidden = hidden;
      DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, serv_ptr->hidden);
      DBA_SaveRecord(serv_ptr->dba_rec);

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceHiddenFlag);
}

/**
 * @brief   Returns the hidden status of the given service. A hidden service
 *          should not be present to the user in any list of available services.
 * @param   s_ptr service
 * @return  TRUE if the service is hidden or doesn't exist, FALSE if it's visible
 */
BOOLEAN ADB_GetServiceHiddenFlag(void *s_ptr)
{
   BOOLEAN hidden = TRUE;

   FUNCTION_START(ADB_GetServiceHiddenFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      hidden = ((ADB_SERVICE_REC *)s_ptr)->hidden;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceHiddenFlag);

   return(hidden);
}

/**
 * @brief   Returns the selectable status of the given service
 * @param   s_ptr service
 * @return  TRUE if the service is valid and is selectable, FALSE otherwise
 */
BOOLEAN ADB_GetServiceSelectableFlag(void *s_ptr)
{
   BOOLEAN selectable = FALSE;

   FUNCTION_START(ADB_GetServiceSelectableFlag);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      selectable = ((ADB_SERVICE_REC *)s_ptr)->selectable;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceSelectableFlag);

   return(selectable);
}

/**
 * @brief   Returns whether the LCN assigned to the given service can be changed
 * @param   s_ptr service
 * @return  TRUE if the service is valid and its LCN can be edited, FALSE otherwise
 */
BOOLEAN ADB_GetServiceLcnEditable(void *s_ptr)
{
   BOOLEAN lcn_editable = FALSE;

   FUNCTION_START(ADB_GetServiceLcnEditable);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      lcn_editable = ((ADB_SERVICE_REC *)s_ptr)->lcn_editable;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceLcnEditable);

   return(lcn_editable);
}

/**
 * @brief   Sets the 'deleted' flag for the given service as a deleted service. If it's marked
 *          as deleted then it won't be included in any of the normal service lists
 * @param   s_ptr service to be changed
 * @param   deleted TRUE if the service is to be marked as deleted, FALSE to make it available
 */
void ADB_SetServiceDeletedFlag(void *s_ptr, BOOLEAN deleted)
{
   FUNCTION_START(ADB_SetServiceDeletedFlag);

   DBDEF_RequestAccess();
   DBDEF_SetServiceDeletedFlag(s_ptr, deleted);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetServiceDeletedFlag);
}

/**
 * @brief   Sets the assigned logical channel number of the given service.
 *          This will override any LCN assigned when performing a service scan.
 * @param   s_ptr service
 * @param   lcn logical channel number
 */
void ADB_SetServiceLcn(void *s_ptr, U16BIT lcn)
{
   ADB_SERVICE_REC *serv_ptr = (ADB_SERVICE_REC *)s_ptr;

   FUNCTION_START(ADB_SetServiceLcn);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (serv_ptr->allocated_lcn != lcn)
      {
         serv_ptr->allocated_lcn = lcn;
         DBA_SetFieldValue(serv_ptr->dba_rec, DBA_FIELD_SERV_LCN, serv_ptr->allocated_lcn);
         DBA_SaveRecord(serv_ptr->dba_rec);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceLcn);
}

/**
 * @brief   Returns the favourite group value for the given service. This is an 8-bit
 *          value that can define whether the service is in any of upto 8 favourite
 *          groups. The value is a bit flag, so 1 means it's in group 1, 2 means it's
 *          in group 2, 4 means it's in group 3, 7 means it's in groups 1, 2 and 3, etc.
 * @param   s_ptr service
 * @return  favourite group bit value
 */
U8BIT ADB_GetServiceFavGroups(void *s_ptr)
{
   U8BIT groups = 0;

   FUNCTION_START(ADB_GetServiceFavGroups);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      groups = ((ADB_SERVICE_REC *)s_ptr)->fav_groups;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetServiceFavGroups);

   return(groups);
}

/**
 * @brief   Sets the favourite group value for the given service. This is an 8-bit
 *          value that can define whether the service is in any of upto 8 favourite
 *          groups. The value is a bit flag, so 1 means it's in group 1, 2 means it's
 *          in group 2, 4 means it's in group 3, 7 means it's in groups 1, 2 and 3, etc.
 * @param   s_ptr service
 * @param   fav_groups bit value, where a 1 means the service is in a group and 0 that it isn't
 */
void ADB_SetServiceFavGroups(void *s_ptr, U8BIT groups)
{
   FUNCTION_START(ADB_SetServiceFavGroups);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetServiceFavGroups(s_ptr, groups);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetServiceFavGroups);
}

/**
 * @brief   Explicitly sets or clears the stream that will be used for audio on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and types, but if FALSE then the audio stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code audio language to be used for the selected stream if 'valid' is TRUE
 * @param   audio_type audio type to be used for the selected stream if 'valid' is TRUE
 * @param   stream_type stream type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdAudioStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code,
   ADB_AUDIO_TYPE audio_type, ADB_STREAM_TYPE stream_type)
{
   FUNCTION_START(ADB_SetReqdAudioStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_valid = valid;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_lang_code = lang_code;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_type = audio_type;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_stream_type = stream_type;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_pid = 0x1FFF;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetReqdAudioStreamSettings);
}

/**
 * @brief   Explicitly sets or clears the stream that will be used for audio on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given PID, but if FALSE then the audio stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   pid PID to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdAudioStreamPID(void *s_ptr, BOOLEAN valid, U16BIT pid)
{
   FUNCTION_START(ADB_SetReqdAudioStreamPID);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_valid = valid;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_pid = pid;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetReqdAudioStreamPID);
}

/**
 * @brief   Returns the settings defined by ADB_SetReqdAudioStreamSettings that will be
 *          used for audio on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set audio language code
 * @param   audio_type returns the set audio type
 * @param   stream_type returns the set stream type
 */
void ADB_GetReqdAudioStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code,
   ADB_AUDIO_TYPE *audio_type, ADB_STREAM_TYPE *stream_type)
{
   FUNCTION_START(ADB_GetReqdAudioStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      *valid = ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_valid;
      *lang_code = ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_lang_code;
      *audio_type = ((ADB_SERVICE_REC *)s_ptr)->reqd_audio_type;
      *stream_type = ((ADB_SERVICE_REC *)s_ptr)->reqd_stream_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetReqdAudioStreamSettings);
}

/**
 * @brief   Explicitly sets or clears the stream that will be used for subtitles on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and type, but if FALSE then the stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code language to be used for the selected stream if 'valid' is TRUE
 * @param   subt_type subtitle type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdSubtitleStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code, ADB_SUBTITLE_TYPE subtitle_type)
{
   FUNCTION_START(ADB_SetReqdSubtitleStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_valid = valid;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_lang_code = lang_code;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_type = subtitle_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetReqdSubtitleStreamSettings);
}

/**
 * @brief   Returns the settings defined by ADB_SetReqdSubtitleStreamSettings that will be
 *          used for subtitles on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set language code
 * @param   subt_type returns the set subtitle type
 */
void ADB_GetReqdSubtitleStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code, ADB_SUBTITLE_TYPE *subtitle_type)
{
   FUNCTION_START(ADB_GetReqdSubtitleStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      *valid = ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_valid;
      *lang_code = ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_lang_code;
      *subtitle_type = ((ADB_SERVICE_REC *)s_ptr)->reqd_subtitle_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetReqdSubtitleStreamSettings);
}

/**
 * @brief   Explicitly sets or clears the stream that will be used for teletext on the
 *          given service. If 'valid' is TRUE then the stream is set according to the
 *          given language and type, but if FALSE then the stream selected for
 *          the service will be based on the default settings.
 * @param   s_ptr service
 * @param   valid TRUE to set the stream using given values, FALSE to use defaults
 * @param   lang_code language to be used for the selected stream if 'valid' is TRUE
 * @param   ttext_type teletext type to be used for the selected stream if 'valid' is TRUE
 */
void ADB_SetReqdTeletextStreamSettings(void *s_ptr, BOOLEAN valid, U32BIT lang_code, ADB_TELETEXT_TYPE ttext_type)
{
   FUNCTION_START(ADB_SetReqdTeletextStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_valid = valid;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_lang_code = lang_code;
      ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_type = ttext_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetReqdTeletextStreamSettings);
}

/**
 * @brief   Returns the settings defined by ADB_SetReqdTeletextStreamSettings that will be
 *          used for teletext on the given service
 * @param   s_ptr service
 * @param   valid return for the valid flag state
 * @param   lang_code returns the set language code
 * @param   ttext_type returns the set teletext type
 */
void ADB_GetReqdTeletextStreamSettings(void *s_ptr, BOOLEAN *valid, U32BIT *lang_code, ADB_TELETEXT_TYPE *ttext_type)
{
   FUNCTION_START(ADB_GetReqdTeletextStreamSettings);

   if (s_ptr != NULL)
   {
      DBDEF_RequestAccess();
      *valid = ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_valid;
      *lang_code = ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_lang_code;
      *ttext_type = ((ADB_SERVICE_REC *)s_ptr)->reqd_ttext_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetReqdTeletextStreamSettings);
}

#if 0
/**
 * @brief   Updates analog service names - ASSUMES NORMAL ASCII CODED
 * @param   s_ptr service to be changed
 * @param   new_name pointer to new name
 */
void ADB_SetAnalogServiceName(void *s_ptr, U8BIT *new_name)
{
   U8BIT new_len;

   FUNCTION_START(ADB_SetAnalogServiceName);

   DBDEF_RequestAccess();
   if (new_name != NULL)
   {
      new_len = strlen((char *)new_name) + 1;
   }
   else
   {
      new_len = 0;
   }
   DBDEF_SetAnalogServiceName((ADB_SERVICE_REC *)s_ptr, new_name, new_len);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetAnalogServiceName);
}

#endif

/**
 * @brief   Sort all services in the database according to their LCNs
 */
void ADB_SortServicesByLcn(void)
{
   FUNCTION_START(ADB_SortServicesByLcn);

   DBDEF_RequestAccess();
   DBDEF_SortServicesByLcn();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SortServicesByLcn);
}

/**
 * @brief   Adds a CRID record to the database; use ADB_SaveCridRecord to store it.
 *          If a CRID record with the given string already exists then a new record won't be
 *          created and the handle of the existing record will be returned.
 * @param   crid CRID string to be added
 * @param   series TRUE if this is a series CRID
 * @param   recommended TRUE if this is a recommended programme CRID
 * @return  CRID record handle, NULL if creation fails
 */
void* ADB_AddCridRecord(U8BIT *crid, BOOLEAN series, BOOLEAN recommended)
{
   void *crid_rec;

   FUNCTION_START(ADB_AddCridRecord);

   crid_rec = ADB_FindCridRecord(crid);
   if (crid_rec == NULL)
   {
      DBDEF_RequestAccess();
      crid_rec = DBDEF_AddCridRecord(crid, series, recommended);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_AddCridRecord);

   return(crid_rec);
}

/**
 * @brief   Saves a CRID record to non-volatile storage
 * @param   c_ptr pointer to CRID record
 */
void ADB_SaveCridRecord(void *c_ptr)
{
   FUNCTION_START(ADB_SaveCridRecord);

   DBDEF_RequestAccess();

   if (DBDEF_IsValidCridRecord((ADB_CRID_REC *)c_ptr))
   {
      DBA_SaveRecord(((ADB_CRID_REC *)c_ptr)->dba_rec);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SaveCridRecord);
}

/**
 * @brief   Searches for and returns the CRID record with the given CRID string
 * @param   crid_str pointer to CRID string
 * @return  pointer to the CRID record if one is found, or NULL
 */
void* ADB_FindCridRecord(U8BIT *crid_str)
{
   ADB_CRID_REC *c_ptr;

   FUNCTION_START(ADB_FindCridRecord);

   DBDEF_RequestAccess();

   c_ptr = DBDEF_GetNextCridRecord(NULL);

   while (c_ptr != NULL)
   {
      if ((c_ptr->crid_str != NULL) && (c_ptr->crid_str->str_ptr != NULL))
      {
         if (STB_CompareStringsIgnoreCase(c_ptr->crid_str->str_ptr, crid_str) == 0)
         {
            break;
         }
      }

      c_ptr = DBDEF_GetNextCridRecord(c_ptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_FindCridRecord);

   return(c_ptr);
}

/**
 * @brief   Sets the date and time fields on the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   datetime date and time to be saved in the record
 */
void ADB_SetCridDateTime(void *c_ptr, U32DHMS datetime)
{
   FUNCTION_START(ADB_SetCridDateTime);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         DBDEF_SetCridDateTime(c_ptr, datetime);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetCridDateTime);
}

/**
 * @brief   Sets the service ID on the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   serv_id service id
 */
void ADB_SetCridService(void *c_ptr, U16BIT serv_id)
{
   FUNCTION_START(ADB_SetCridService);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         DBDEF_SetCridService(c_ptr, serv_id);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetCridService);
}

/**
 * @brief   Sets the programme name field of the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   prog_name pointer to name string, the name is copied
 */
void ADB_SetCridProgrammeName(void *c_ptr, U8BIT *prog_name)
{
   FUNCTION_START(ADB_SetCridProgrammeName);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         DBDEF_SetCridProgrammeName(c_ptr, prog_name);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetCridProgrammeName);
}

/**
 * @brief   Returns the programme name field of the given CRID record.
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to the programme name
 */
U8BIT* ADB_GetCridProgrammeName(void *c_ptr)
{
   U8BIT *prog_name;

   FUNCTION_START(ADB_GetCridProgrammeName);

   prog_name = NULL;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         prog_name = CopyString(((ADB_CRID_REC *)c_ptr)->name_str, TRUE);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridProgrammeName);

   return(prog_name);
}

/**
 * @brief   Updates the time the CRID was last seen in the EIT
 * @param   c_ptr pointer to CRID record
 */
void ADB_UpdateCridEitDate(void *c_ptr)
{
   FUNCTION_START(ADB_UpdateEitDate);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_UpdateCridEitDate(c_ptr);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_UpdateEitDate);
}

/**
 * @brief   Returns TRUE if the CRID record given represents a programme (split event)
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a programme
 */
BOOLEAN ADB_IsProgrammeCrid(void *c_ptr)
{
   BOOLEAN result;
   ADB_CRID_REC *crid_ptr = (ADB_CRID_REC *)c_ptr;

   FUNCTION_START(ADB_IsProgrammeCrid);

   result = FALSE;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         if (!crid_ptr->series_flag && !crid_ptr->recommended_flag)
         {
            result = TRUE;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_IsProgrammeCrid);

   return(result);
}

/**
 * @brief   Returns TRUE if the CRID record given represents a series
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a series
 */
BOOLEAN ADB_IsSeriesCrid(void *c_ptr)
{
   BOOLEAN result;

   FUNCTION_START(ADB_IsSeriesCrid);

   result = FALSE;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         result = ((ADB_CRID_REC *)c_ptr)->series_flag;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_IsSeriesCrid);

   return(result);
}

/**
 * @brief   Returns TRUE if the CRID record given represents a recommendation
 * @param   c_ptr pointer to CRID record
 * @return  TRUE if the CRID record is for a recommended event/series
 */
BOOLEAN ADB_IsRecommendationCrid(void *c_ptr)
{
   BOOLEAN result;

   FUNCTION_START(ADB_IsRecommendationCrid);

   result = FALSE;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         result = ((ADB_CRID_REC *)c_ptr)->recommended_flag;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_IsRecommendationCrid);

   return(result);
}

/**
 * @brief   Sets the do not delete flag for the given CRID
 * @param   c_ptr pointer to CRID record
 * @param   do_not_delete value to set the flag to
 */
void ADB_SetCridDoNotDelete(void *c_ptr, BOOLEAN do_not_delete)
{
   FUNCTION_START(ADB_SetCridDoNotDelete);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_SetCridDoNotDelete(c_ptr, do_not_delete);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetCridDoNotDelete);
}

/**
 * @brief   Returns the value of the do not delete flag for the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  value of the flag, FALSE if the CRID record isn't valid
 */
BOOLEAN ADB_GetCridDoNotDelete(void *c_ptr)
{
   BOOLEAN result;

   FUNCTION_START(ADB_GetCridDoNotDelete);

   result = FALSE;
   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         result = ((ADB_CRID_REC *)c_ptr)->do_not_delete;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridDoNotDelete);

   return(result);
}

/**
 * @brief   Returns the CRID string from the given CRID record.
 *          The returned value should not be freed.
 * @param   c_ptr pointer to CRID record
 * @return  pointer to the CRID string
 */
U8BIT* ADB_GetCridString(void *c_ptr)
{
   U8BIT *crid_str;

   FUNCTION_START(ADB_GetCridString);

   crid_str = NULL;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr) && (((ADB_CRID_REC *)c_ptr)->crid_str != NULL))
      {
         crid_str = ((ADB_CRID_REC *)c_ptr)->crid_str->str_ptr;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridString);

   return(crid_str);
}

/**
 * @brief   Returns a value representing the date & time held in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  date/time value
 */
U32DHMS ADB_GetCridDateTime(void *c_ptr)
{
   U32DHMS time;

   FUNCTION_START(ADB_GetCridDateTime);

   time = 0;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         time = ((ADB_CRID_REC *)c_ptr)->date_time;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridDateTime);

   return time;
}

/**
 * @brief   Returns the service ID held in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  service id, or ADB_INVALID_DVB_ID if not valid
 */
U16BIT ADB_GetCridService(void *c_ptr)
{
   U16BIT serv_id;

   FUNCTION_START(ADB_GetCridService);

   serv_id = ADB_INVALID_DVB_ID;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         serv_id = ((ADB_CRID_REC *)c_ptr)->serv_id;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridService);

   return(serv_id);
}

/**
 * @brief   Returns the EIT update date contained in the given CRID record
 * @param   c_ptr pointer to CRID record
 * @return  GMT date from the record
 */
U16BIT ADB_GetCridEitDate(void *c_ptr)
{
   U16BIT date;

   FUNCTION_START(ADB_GetCridEitDate);

   date = 0;

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();

      if (DBDEF_IsValidCridRecord(c_ptr))
      {
         date = ((ADB_CRID_REC *)c_ptr)->eit_date;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetCridEitDate);

   return(date);
}

/**
 * @brief   Deletes the given CRID record from the database
 * @param   c_ptr pointer to CRID record
 */
void ADB_DeleteCridRecord(void *c_ptr)
{
   FUNCTION_START(ADB_DeleteCridRecord);

   if (c_ptr != NULL)
   {
      DBDEF_RequestAccess();
      DBDEF_DeleteCridRecord(c_ptr);
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_DeleteCridRecord);
}

/**
 * @brief   Creates an array of CRID record pointers to pass back to the caller
 *          with the number of entries in the array. The returned array should be
 *          freed using ADB_ReleaseCridRecordList.
 * @param   clist_ptr address for the return of the CRID array
 * @param   num_entries_ptr address for the return of the number of entries
 * @param   inc_series_crids TRUE if returned list is to include any series CRIDs
 * @param   inc_rec_crids TRUE if returned list is to include any recommendation CRIDs
 * @param   inc_prog_crids TRUE if returned list is to include any programme CRIDs
 */
void ADB_GetCridRecordList(void ***clist_ptr, U16BIT *num_entries_ptr, BOOLEAN inc_series_crids,
   BOOLEAN inc_rec_crids, BOOLEAN inc_prog_crids)
{
   void **clist;
   U16BIT num_entries, num_crids;
   ADB_CRID_REC *c_ptr;
   U16BIT i, j;

   FUNCTION_START(ADB_GetCridRecordList);

   DBDEF_RequestAccess();

   /* Count the number of required CRIDs */
   num_entries = DBDEF_GetNumCridRecords();
   num_crids = 0;
   clist = NULL;

   c_ptr = DBDEF_GetNextCridRecord(NULL);

   for (i = 0; (i < num_entries) && (c_ptr != NULL); i++)
   {
      if (c_ptr->series_flag)
      {
         if (inc_series_crids)
         {
            num_crids++;
         }
      }
      else if (c_ptr->recommended_flag)
      {
         if (inc_rec_crids)
         {
            num_crids++;
         }
      }
      else
      {
         if (inc_prog_crids)
         {
            num_crids++;
         }
      }

      c_ptr = DBDEF_GetNextCridRecord(c_ptr);
   }

   if (num_crids > 0)
   {
      clist = STB_AppGetMemory(num_crids * sizeof(void *));

      if (clist != NULL)
      {
         c_ptr = DBDEF_GetNextCridRecord(NULL);

         for (i = 0, j = 0; (i != num_entries) && (c_ptr != NULL); i++)
         {
            if (c_ptr->series_flag)
            {
               if (inc_series_crids)
               {
                  clist[j] = (void *)c_ptr;
                  j++;
               }
            }
            else if (c_ptr->recommended_flag)
            {
               if (inc_rec_crids)
               {
                  clist[j] = (void *)c_ptr;
                  j++;
               }
            }
            else
            {
               if (inc_prog_crids)
               {
                  clist[j] = (void *)c_ptr;
                  j++;
               }
            }

            c_ptr = DBDEF_GetNextCridRecord(c_ptr);
         }
      }
      else
      {
         num_crids = 0;
      }
   }

   *clist_ptr = clist;
   *num_entries_ptr = num_crids;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetCridRecordList);
}

/**
 * @brief   Frees memory allocated for a CRID list due to a call to ADB_GetCridRecordList
 * @param   crid_list crid array
 * @param   num_entries number of items in the array
 */
void ADB_ReleaseCridRecordList(void **crid_list, U16BIT num_entries)
{
   FUNCTION_START(ADB_ReleaseCridRecordList);

   USE_UNWANTED_PARAM(num_entries);

   if (crid_list != NULL)
   {
      STB_AppFreeMemory(crid_list);
   }

   FUNCTION_FINISH(ADB_ReleaseCridRecordList);
}

/**
 * @brief   Returns a transport that is signalled as containing a complete set of
 *          SI data via an NIT linkage descripter
 * @param   n_ptr network pointer
 * @return  pointer to transport
 */
void* ADB_GetFullSITransport(void *n_ptr)
{
   ADB_NETWORK_REC *net_ptr;
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   void *tran_rec;

   FUNCTION_START(ADB_GetFullSITransport);

   DBDEF_RequestAccess();

   tran_rec = NULL;

   net_ptr = (ADB_NETWORK_REC *)n_ptr;
   if (net_ptr != NULL)
   {
      desc_ptr = net_ptr->linkage_desc_list;
      while ((desc_ptr != NULL) && (tran_rec == NULL))
      {
         if (desc_ptr->link_type == FULL_SI_LINKAGE_TYPE)
         {
            tran_rec = (void *)DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                  desc_ptr->orig_net_id, desc_ptr->tran_id);
         }
         desc_ptr = desc_ptr->next;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetFullSITransport);

   return(tran_rec);
}

/**
 * @brief   Returns a service that is signalled as containing EPG data via an NIT linkage descriptor
 * @param   n_ptr Network pointer
 * @return  pointer to service
 */
void* ADB_GetEPGService(void *n_ptr)
{
   ADB_NETWORK_REC *net_ptr;
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   void *epg_service;

   FUNCTION_START(ADB_GetEPGService);

   DBDEF_RequestAccess();

   epg_service = NULL;

   net_ptr = (ADB_NETWORK_REC *)n_ptr;
   if (net_ptr != NULL)
   {
      desc_ptr = net_ptr->linkage_desc_list;
      while ((desc_ptr != NULL) && (epg_service == NULL))
      {
         if (desc_ptr->link_type == EPG_SERVICE_LINKAGE_TYPE)
         {
            epg_service = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
               desc_ptr->orig_net_id, desc_ptr->tran_id, desc_ptr->serv_id);
         }
         desc_ptr = desc_ptr->next;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetEPGService);

   return epg_service;
}

/**
 * @brief   Returns the number of favourite lists
 * @return  number of favourite lists
 */
U16BIT ADB_GetNumFavouriteLists(void)
{
   U16BIT num_favs;

   FUNCTION_START(ADB_GetNumFavouriteLists);

   DBDEF_RequestAccess();
   num_favs = DBDEF_GetNumFavouriteLists();
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetNumFavouriteLists);

   return(num_favs);
}

/**
 * @brief   Creates a new favourite list and adds it to the existing lists, if any.
 * @param   name name to be given to the list
 * @param   user_data user defined value to be stored in the list
 * @param   index 0 based index where the new list should be inserted in any existing list.
 *          0 = at start, negative = at end, any other value = position
 * @param   list_id used to get the unique id assigned to the list, can be NULL
 * @return  TRUE if the list is created successfully, FALSE otherwise
 */
BOOLEAN ADB_AddFavouriteList(U8BIT *name, U32BIT user_data, S16BIT index, U8BIT *list_id)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_AddFavouriteList);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_AddFavouriteList(0, name, user_data, index)) != NULL)
   {
      if (list_id != NULL)
      {
         /* Return the id assigned to the list */
         *list_id = fav_list->list_id;
      }

      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddFavouriteList);

   return(retval);
}

/**
 * @brief   Returns the list id of the favourite list defined by the given index
 * @param   index list index
 * @return  list id, 0 if list isn't found
 */
U8BIT ADB_GetFavouriteListIdByIndex(U16BIT index)
{
   ADB_FAVLIST_REC *fl_ptr;
   U8BIT list_id;

   FUNCTION_START(ADB_GetFavouriteListIdByIndex);

   DBDEF_RequestAccess();

   fl_ptr = DBDEF_GetNextFavouriteList(NULL);
   while ((fl_ptr != NULL) && (index > 0))
   {
      fl_ptr = DBDEF_GetNextFavouriteList(fl_ptr);
      index--;
   }

   if (fl_ptr != NULL)
   {
      list_id = fl_ptr->list_id;
   }
   else
   {
      list_id = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetFavouriteListIdByIndex);

   return(list_id);
}

/**
 * @brief   Returns the list id of the favourite list with the given user data
 * @param   user_data user data value
 * @return  list id, 0 if list isn't found
 */
U8BIT ADB_GetFavouriteListByUserData(U32BIT user_data)
{
   ADB_FAVLIST_REC *fl_ptr;
   U8BIT list_id;

   FUNCTION_START(ADB_GetFavouriteListByUserData);

   DBDEF_RequestAccess();

   fl_ptr = DBDEF_GetNextFavouriteList(NULL);
   while ((fl_ptr != NULL) && (fl_ptr->user_data != user_data))
   {
      fl_ptr = DBDEF_GetNextFavouriteList(fl_ptr);
   }

   if (fl_ptr != NULL)
   {
      list_id = fl_ptr->list_id;
   }
   else
   {
      list_id = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetFavouriteListByUserData);

   return(list_id);
}

/**
 * @brief   Returns the name of the given favourite list, as a Unicode string
 * @param   list_id list id
 * @return  name of the favourite list as a unicode string, which must be
 *          released using STB_ReleaseUnicodeString
 */
U8BIT* ADB_GetFavouriteListName(U8BIT list_id)
{
   U8BIT *name;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_GetFavouriteListName);

   DBDEF_RequestAccess();

   name = NULL;

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      name = CopyString(fav_list->name, TRUE);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetFavouriteListName);

   return(name);
}

/**
 * @brief   Set the name of the given favourite list
 * @param   list_id list id
 * @param   name of the favourite list
 * @return  TRUE if the name is set successfully, FALSE otherwise
 */
BOOLEAN ADB_SetFavouriteListName(U8BIT list_id, U8BIT *name)
{
   ADB_FAVLIST_REC *fav_list;
   BOOLEAN success = TRUE;

   FUNCTION_START(ADB_SetFavouriteListName);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
       DBDEF_SetFavouriteListName(fav_list, name);
   }
   else
   {
       success = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetFavouriteListName);

   return(success);
}

/**
 * @brief   Returns the user data of the given favourite list
 * @param   list_id list id
 * @return  user data value, will be 0 if list not found
 */
U32BIT ADB_GetFavouriteListUserData(U8BIT list_id)
{
   U32BIT user_data;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_GetFavouriteListUserData);

   DBDEF_RequestAccess();

   user_data = 0;

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      user_data = fav_list->user_data;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetFavouriteListUserData);

   return(user_data);
}

/**
 * @brief   Set the user data of the given favourite list
 * @param   list_id list id
 * @param   user_data value
 */
void ADB_SetFavouriteListUserData(U8BIT list_id, U32BIT user_data)
{
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_SetFavouriteListUserData);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      DBDEF_SetFavouriteListUserData(fav_list, user_data);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetFavouriteListUserData);
}

/**
 * @brief   Change the order of the favourite lists by moving the given list to
 *          the given position
 * @param   list_id favourite list to be moved
 * @param   index position to move the list to, negative value will move the list to the end
 * @return  TRUE if the list is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListTo(U8BIT list_id, S16BIT index)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_MoveFavouriteListTo);

   retval = FALSE;

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      DBDEF_MoveFavouriteListTo(fav_list, index);
      retval = TRUE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_MoveFavouriteListTo);

   return(retval);
}

/**
 * @brief   Deletes the favourite list with the given list id
 * @param   list_id id of the favourite list to be deleted
 */
void ADB_DeleteFavouriteList(U8BIT list_id)
{
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_DeleteFavouriteList);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      DBDEF_DeleteFavouriteList(fav_list);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_DeleteFavouriteList);
}

/**
 * @brief   Adds the given service to the favourite list defined by list_id,
 *          with the service being optionally added at a given position
 * @param   list_id favourite list id
 * @param   serv_ptr service to be added
 * @param   index position to add the service in the list,
 *          0 = at start, negative = at end, any other value is the position
 * @return  TRUE if the service is successfully added, FALSE otherwise
 */
BOOLEAN ADB_AddServiceToFavouriteList(U8BIT list_id, void *serv_ptr, S16BIT index)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_AddServiceToFavouriteList);

   retval = FALSE;

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      /* Add the service to this list */
      if (DBDEF_AddServiceToFavouriteList(fav_list, (ADB_SERVICE_REC *)serv_ptr, index) != NULL)
      {
         retval = TRUE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_AddServiceToFavouriteList);

   return(retval);
}

/**
 * @brief   Change the order of the services in the given favourite lists by
 *          moving it to the given position
 * @param   list_id favourite list to be moved
 * @param   serv_ptr service to be moved
 * @param   index position in the list to move the service to,
 *          negative value will move service to the end of the list
 * @return  TRUE if the service is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListServiceTo(U8BIT list_id, void *serv_ptr, S16BIT index)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_MoveFavouriteListServiceTo);

   retval = FALSE;

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      if ((fav_serv = DBDEF_FindServiceInFavouriteList(fav_list, serv_ptr)) != NULL)
      {
         DBDEF_MoveFavouriteListServiceTo(fav_list, fav_serv, index);
         retval = TRUE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_MoveFavouriteListServiceTo);

   return(retval);
}

/**
 * @brief   Change the order of the services in the given favourite lists by
 *          moving it to the given position
 * @param   list_id favourite list to be moved
 * @param   current_index index of the service to be moved
 * @param   new_index position in the list to move the service to,
 *          negative value will move service to the end of the list
 * @return  TRUE if the service is moved successfully, FALSE otherwise
 */
BOOLEAN ADB_MoveFavouriteListServiceFromTo(U8BIT list_id, S16BIT current_index, S16BIT new_index)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_MoveFavouriteListServiceFromTo);

   retval = FALSE;

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, NULL);
      while ((fav_serv != NULL) && (current_index > 0))
      {
         fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv);
         current_index--;
      }

      if (fav_serv != NULL)
      {
         DBDEF_MoveFavouriteListServiceTo(fav_list, fav_serv, new_index);
         retval = TRUE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_MoveFavouriteListServiceFromTo);

   return(retval);
}

/**
 * @brief   Removes the given service from the favourite list defined by the list id
 * @param   list_id favourite list id
 * @param   serv_ptr service to be removed from the list
 */
void ADB_DeleteServiceFromFavouriteList(U8BIT list_id, void *serv_ptr)
{
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ADB_DeleteServiceFromFavouriteList);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      /* Delete the service from this list */
      if ((fav_serv = DBDEF_FindServiceInFavouriteList(fav_list, serv_ptr)) != NULL)
      {
         DBDEF_DeleteServiceFromFavouriteList(fav_list, fav_serv);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_DeleteServiceFromFavouriteList);
}

/**
 * @brief   Removes all services from the favourite list defined by list_id
 * @param   list_id favourite list id
 */
void ADB_DeleteAllServicesFromFavouriteList(U8BIT list_id)
{
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_DeleteAllServicesFromFavouriteList);

   DBDEF_RequestAccess();

   if ((fav_list = DBDEF_FindFavouriteList(list_id)) != NULL)
   {
      DBDEF_DeleteAllServicesFromFavouriteList(fav_list);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_DeleteAllServicesFromFavouriteList);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   Returns whether the SDT has been received for the given service
 * @param   s_ptr service pointer
 * @return  TRUE if the SDT has been received
 */
BOOLEAN ADB_GetServiceSDTReceived(void *s_ptr)
{
   BOOLEAN sdt_received;

   FUNCTION_START(ADB_GetServiceSDTReceived);

   DBDEF_RequestAccess();

   if (s_ptr != NULL)
   {
      sdt_received = ((ADB_SERVICE_REC *)s_ptr)->sdt_received;
   }
   else
   {
      sdt_received = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceSDTReceived);

   return(sdt_received);
}

/**
 * @brief   Returns the current CI protection descriptor for the given service
 * @param   s_ptr service pointer
 * @return  Protection descriptor data or NULL if not available
 */
U8BIT* ADB_GetServiceCIProtectionDesc(void *s_ptr)
{
   U8BIT *ci_data;

   FUNCTION_START(ADB_GetServiceCIProtectionDesc);

   DBDEF_RequestAccess();

   if (s_ptr != NULL)
   {
      ci_data = ((ADB_SERVICE_REC *)s_ptr)->ci_protection_desc;
   }
   else
   {
      ci_data = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServiceCIProtectionDesc);

   return(ci_data);
}

/**
 * @brief   Checks whether the Ci protection desc needs to be updated for the given service
 * @param   s_ptr service pointer
 * @return  TRUE if the descriptor has expired and needs to be updated
 */
BOOLEAN ADB_HasCIProtectionExpired(void *s_ptr)
{
   BOOLEAN expired;
   U32BIT time_diff;

   FUNCTION_START(ADB_HasCIProtectionExpired);

   DBDEF_RequestAccess();

   expired = FALSE;

   if (s_ptr != NULL)
   {
      if (((ADB_SERVICE_REC *)s_ptr)->ci_protection_desc != NULL)
      {
         time_diff = STB_OSGetClockRTC() - ((ADB_SERVICE_REC *)s_ptr)->ci_prot_last_update;
         if (time_diff >= CI_PROTECTION_DESC_TIME_LIMIT)
         {
            expired = TRUE;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_HasCIProtectionExpired);

   return(expired);
}

#endif /* COMMON_INTERFACE */

/**
 * @brief   Returns the current PMT data for the given service
 * @param   s_ptr service pointer
 * @param   data_len pointer to return the size of the PMT data, only valid
 *          return value isn't NULL
 * @return  PMT data or NULL if not available
 */
U8BIT* ADB_GetServicePMTData(void *s_ptr, U16BIT *data_len)
{
   U8BIT *pmt_data;
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(ADB_GetServicePMTData);

   DBDEF_RequestAccess();

   if (s_ptr != NULL)
   {
      serv_ptr = (ADB_SERVICE_REC *)s_ptr;
      pmt_data = serv_ptr->pmt_data;
      *data_len = serv_ptr->pmt_data_len;
   }
   else
   {
      pmt_data = NULL;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServicePMTData);

   return(pmt_data);
}

/**
 * @brief   Returns the current PMT PID for the given service
 * @param   s_ptr service pointer
 * @return  PMT PID
 */
U16BIT ADB_GetServicePmtPid(void *s_ptr)
{
   U16BIT pmt_pid = 0;

   FUNCTION_START(ADB_GetServicePmtPid);

   DBDEF_RequestAccess();
   pmt_pid = DBDEF_GetServicePmtPid((ADB_SERVICE_REC *)s_ptr);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetServicePmtPid);

   return pmt_pid;
}

/**
 * @brief   Returns the current AIT PID for the given service
 * @param   s_ptr service pointer
 * @return  AIT PID; zero PID means AIT table not present
 */
U16BIT ADB_GetServiceAitPid(void *s_ptr)
{
   U16BIT ait_pid = 0;

   FUNCTION_START(ADB_GetServiceAitPid);

#ifdef INTEGRATE_HBBTV
   DBDEF_RequestAccess();
   ait_pid = ((ADB_SERVICE_REC *)s_ptr)->ait_pid;
   DBDEF_ReleaseAccess();
#else
   USE_UNWANTED_PARAM(s_ptr);
#endif

   FUNCTION_FINISH(ADB_GetServiceAitPid);

   return ait_pid;
}

/**
 * @brief   Returns an array of available profiles. The array will be allocated
 *          within the function and should be released using ADB_ReleaseProfileList.
 *          This is a CI+ feature.
 * @param   profile_list pointer to the returned array of profiles.
 *                         Value isn't valid if 0 is returned
 * @param   active_profile pointer to return the index of the currently active profile
 * @return  Number of profiles
 */
U16BIT ADB_GetProfileList(void ***profile_list, U16BIT *active_profile)
{
   U16BIT num_profiles;

   FUNCTION_START(ADB_GetProfileList);

   DBDEF_RequestAccess();
   num_profiles = DBDEF_GetProfileList(profile_list, active_profile);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_GetProfileList);

   return(num_profiles);
}

/**
 * @brief   Frees a profile list acquired using ADB_GetProfileList
 * @param   profile_list pointer to the array of profiles.
 * @param   num_profiles number of profiles in the array
 */
void ADB_ReleaseProfileList(void **profile_list, U16BIT num_profiles)
{
   FUNCTION_START(ADB_ReleaseProfileList);

   DBDEF_ReleaseProfileList(profile_list, num_profiles);

   FUNCTION_FINISH(ADB_ReleaseProfileList);
}

/**
 * @brief   Returns the type of the given profile
 * @param   profile profile handle
 * @return  profile type
 */
ADB_PROFILE_TYPE ADB_GetProfileType(void *profile)
{
   ADB_PROFILE_TYPE type = ADB_PROFILE_TYPE_BROADCAST;

   FUNCTION_START(ADB_GetProfileType);

   if (profile != NULL)
   {
      DBDEF_RequestAccess();
      type = ((ADB_NETWORK_REC *)profile)->profile_type;
      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileType);

   return(type);
}

/**
 * @brief   Returns the default list type for the current profile. This function returns
 *          ADB_SERVICE_LIST_ALL if no CI+ profile is active, otherwise the favourite list
 *          containing the profile channels.
 * @return  Default service list type fot the current profile.
 */
ADB_SERVICE_LIST_TYPE ADB_GetServiceListTypeForCurrentProfile(void)
{
   ADB_SERVICE_LIST_TYPE list_type;
#ifdef COMMON_INTERFACE
   ADB_NETWORK_REC *n_ptr;
#endif

   FUNCTION_START(ADB_GetServiceListTypeForCurrentProfile);

   list_type = ADB_SERVICE_LIST_ALL;

#ifdef COMMON_INTERFACE
   DBDEF_RequestAccess();
   n_ptr = DBDEF_GetCurrentCIPlusProfile();
   if (n_ptr != NULL)
   {
      if (n_ptr->fav_list_id != 0)
      {
         list_type = ADB_LIST_TYPE_FROM_FAVLIST(n_ptr->fav_list_id);
      }
   }
   DBDEF_ReleaseAccess();
#endif

   FUNCTION_FINISH(ADB_GetServiceListTypeForCurrentProfile);

   return list_type;
}

#ifdef COMMON_INTERFACE
/**
 * @brief   Finds the profile for the given CI+ operator module
 * @param   module operator module
 * @return  Profile handle, NULL if not found
 */
void* ADB_FindProfileForModule(U32BIT module)
{
   void *profile;

   FUNCTION_START(ADB_FindProfileForModule);

   DBDEF_RequestAccess();
   profile = DBDEF_FindNetworkForCIPlusModule(module);
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_FindProfileForModule);

   return(profile);
}

/**
 * @brief   Deletes all records relating to the given profile from the database
 * @param   profile profile handle to be deleted
 * @return  TRUE if the profile is deleted, FALSE otherwise
 */
BOOLEAN ADB_DeleteProfile(void *profile)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;
   ADB_TRANSPORT_REC *t_ptr, *next_t_ptr;
   ADB_FAVLIST_REC *fav;

   FUNCTION_START(ADB_DeleteProfile);

   retval = FALSE;

   if (profile != NULL)
   {
      n_ptr = (ADB_NETWORK_REC *)profile;

      DBDEF_RequestAccess();

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         /* Can't delete the active profile */
         if (!DBDEF_IsActiveProfile(n_ptr))
         {
            DBDEF_PushCIPlusProfile(n_ptr->cicam_onet_id, n_ptr->cicam_identifier);

            DBA_LockDatabase();

            /* First remove the channel list */
            if (n_ptr->fav_list_id != 0)
            {
               fav = DBDEF_FindFavouriteList(n_ptr->fav_list_id);
               if (fav != NULL)
               {
                  DBDEF_DeleteFavouriteList(fav);
               }
            }

            t_ptr = DBDEF_GetNextTransportRec(NULL);
            while (t_ptr != NULL)
            {
               next_t_ptr = DBDEF_GetNextTransportRec(t_ptr);
               if (t_ptr->network == n_ptr)
               {
                  DBDEF_DeleteTransportRec(t_ptr);
               }

               t_ptr = next_t_ptr;
            }

            DBDEF_DeleteNetworkRec(n_ptr);

            DBA_SaveDatabase();
            DBA_UnlockDatabase();

            DBDEF_PopProfile();

            retval = TRUE;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_DeleteProfile);

   return(retval);
}

/**
 * @brief   Returns the name of the given profile as a unicode string.
 *          The returned string must be freed using STB_ReleaseUnicodeString.
 * @param   profile profile handle
 * @return  profile name, can be NULL
 */
U8BIT* ADB_GetProfileName(void *profile)
{
   U8BIT *name;

   FUNCTION_START(ADB_GetProfileName);

   if (profile != NULL)
   {
      DBDEF_RequestAccess();
      name = CopyString(((ADB_NETWORK_REC *)profile)->profile_name, TRUE);
      DBDEF_ReleaseAccess();
   }
   else
   {
      name = NULL;
   }

   FUNCTION_FINISH(ADB_GetProfileName);

   return(name);
}

/**
 * @brief   Mark CI module present for network specified by
 *          original network ID and CAM ID
 * @param   cicam_onet_id returned original network ID
 * @param   cicam_id returned CAM identifier
 */
void ADB_MarkCIModulePresent(U16BIT cicam_onet_id, U32BIT cicam_id)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_MarkCIModulePresent);

   DBDEF_RequestAccess();
   /* Mark the module as present if a network record exists for it */
   if ((n_ptr = DBDEF_FindNetworkForCIPlusProfile(cicam_onet_id, cicam_id)) != NULL)
   {
      n_ptr->module_present = TRUE;
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_MarkCIModulePresent);
}

/**
 * @brief   Finds the profile for the given CI+ operator module,
 *          and marks it not present
 * @param   module operator module
 */
void ADB_MarkCIModuleNotPresent(U32BIT module)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_MarkCIModuleNotPresent);

   DBDEF_RequestAccess();
   n_ptr = DBDEF_FindNetworkForCIPlusModule(module);
   if (n_ptr != NULL)
   {
      n_ptr->module_present = FALSE;
   }
   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_MarkCIModuleNotPresent);
}

/**
 * @brief   Returns whether the CI+ module for the given profile is available
 * @param   profile profile handle
 * @return  TRUE if present, FALSE otherwise
 */
BOOLEAN ADB_GetProfileModulePresent(void *profile)
{
   BOOLEAN present;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_GetProfileModulePresent);

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         present = n_ptr->module_present;
      }
      else
      {
         present = TRUE;
      }

      DBDEF_ReleaseAccess();
   }
   else
   {
      present = FALSE;
   }

   FUNCTION_FINISH(ADB_GetProfileModulePresent);

   return(present);
}

/**
 * @brief   Returns the operator module handle of the given profile
 * @param   profile profile handle
 * @return  module handle, or 0 if not available
 */
U32BIT ADB_GetProfileModule(void *profile)
{
   U32BIT module;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_GetProfileModule);

   module = 0;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         module = n_ptr->module;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileModule);

   return(module);
}

/**
 * @brief   Returns the original network ID and CAM ID of the given profile
 * @param   profile profile handle
 * @param   cicam_onet_id returned original network ID
 * @param   cicam_id returned CAM identifier
 * @return  TRUE if the values are returned, FALSE otherwise
 */
BOOLEAN ADB_GetProfileCAMInfo(void *profile, U16BIT *cicam_onet_id, U32BIT *cicam_id)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_GetProfileCAMInfo);

   retval = FALSE;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         *cicam_onet_id = n_ptr->cicam_onet_id;
         *cicam_id = n_ptr->cicam_identifier;
         retval = TRUE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileCAMInfo);

   return(retval);
}

/**
 * @brief   Set the flag indicating whether this operator profile needs to be updated with a scan
 * @param   profile profile handle
 * @param   search_required flag to be set
 * @param   date UTC MJD date code, if a scheduled search is required, 0 if not
 * @param   hour UTC hour, if a scheduled search is required, 0 if not
 * @param   min UTC minute, if a scheduled search is required, 0 if not
 * @return  TRUE if successful, FALSE if the profile is not valid
 */
BOOLEAN ADB_SetProfileSearchRequired(void *profile, BOOLEAN search_required, U16BIT date,
   U8BIT hour, U8BIT min)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_SetProfileSearchRequired);

   retval = FALSE;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         n_ptr->op_search_required = search_required;
         DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_OPERATOR_SEARCH, (U32BIT)search_required);

         n_ptr->op_search_date = date;
         DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_OP_SEARCH_DATE, (U32BIT)n_ptr->op_search_date);

         n_ptr->op_search_time = (hour * 60) + min;
         DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_OP_SEARCH_TIME, (U32BIT)n_ptr->op_search_time);

         retval = TRUE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_SetProfileSearchRequired);

   return(retval);
}

/**
 * @brief   Returns whether the given operator profile requires a scan in order to be updated
 * @param   profile profile handle
 * @return  TRUE if a scan is required, FALSE otherwise
 */
BOOLEAN ADB_GetProfileSearchRequired(void *profile)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_GetProfileSearchRequired);

   retval = FALSE;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         /* Only return TRUE if it isn't a scheduled search, or is a scheduled
          * search and the date/time has passed */
         if (n_ptr->op_search_required && ((n_ptr->op_search_date == 0) ||
                                           !STB_GCIsFutureDateTime(n_ptr->op_search_date, n_ptr->op_search_time / 60, n_ptr->op_search_time % 60, 0)))
         {
            retval = TRUE;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileSearchRequired);

   return(retval);
}

/**
 * @brief   Returns the scheduled operator search date/time for the given profile
 * @param   profile profile handle
 * @param   date pointer to return the date code as MJD
 * @param   hours pointer to return the search hour in GMT
 * @param   mins pointer to return the search minute in GMT
 * @return  TRUE if the profile is valid and a scheduled date/time is returned, FALSE otherwise
 */
BOOLEAN ADB_GetProfileSearchDateTime(void *profile, U16BIT *date, U8BIT *hours, U8BIT *mins)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ADB_GetProfileSearchDateTime);

   retval = FALSE;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if ((n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS) && n_ptr->op_search_required &&
          (n_ptr->op_search_date != 0))
      {
         *date = n_ptr->op_search_date;
         *hours = n_ptr->op_search_time / 60;
         *mins = n_ptr->op_search_time % 60;
         retval = TRUE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileSearchDateTime);

   return(retval);
}

/**
 * @brief   Sets the timer handle being used for the scheduled profile search
 * @param   profile profile handle
 * @param   timer_handle timer handle
 */
void ADB_SetProfileSearchTimer(U32BIT cicam_id, U32BIT timer_handle)
{
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(ADB_SetProfileSearchTimer);

   DBDEF_RequestAccess();

   t = DBDEF_FindCicamTimer(cicam_id);
   if (t != NULL)
   {
      DBDEF_DeleteCicamTimerRec(t);
   }

   if (timer_handle != INVALID_TIMER_HANDLE)
   {
      DBDEF_AddCicamTimer(cicam_id, timer_handle);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ADB_SetProfileSearchTimer);
}

/**
 * @brief   Returns the operator search timer handle of the given profile
 * @param   profile profile handle
 * @return  timer handle, or INVALID_TIMER_HANDLE if not available
 */
U32BIT ADB_GetProfileSearchTimer(void *profile)
{
   U32BIT timer_handle;
   ADB_NETWORK_REC *n_ptr;
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(ADB_GetProfileSearchTimer);

   timer_handle = INVALID_TIMER_HANDLE;

   if (profile != NULL)
   {
      DBDEF_RequestAccess();

      n_ptr = (ADB_NETWORK_REC *)profile;

      if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
      {
         t = DBDEF_FindCicamTimer(n_ptr->cicam_identifier);
         if (t != NULL)
         {
            timer_handle  = t->timer_handle;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_GetProfileSearchTimer);

   return(timer_handle);
}

/**
 * @brief   Deletes the CICAM timer record with the specified timer handle
 * @param   timer_handle Timer handle to be searched for
 */
void ADB_DeleteCicamTimerbyHandle(U32BIT timer_handle)
{
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(ADB_DeleteCicamTimerbyHandle);

   if (timer_handle != INVALID_TIMER_HANDLE)
   {
      DBDEF_RequestAccess();

      t = DBDEF_FindCicamTimerByHandle(timer_handle);
      if (t != NULL)
      {
         DBDEF_DeleteCicamTimerRec(t);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ADB_DeleteCicamTimerbyHandle);
}

#endif /* COMMON_INTERFACE */


/*!**************************************************************************
 * @brief   Returns the full CRID of the type given for the given event
 * @param   serv_ptr service for the event
 * @param   event_ptr event CRID to be returned
 * @param   crid_type CRID type to be returned
 * @param   index the index of the CRID to be returned (only valid for series CRIDs)
 * @return  pointer to a string containing the CRID name,
 *          or NULL if the event doesn't have the requested CRID type
 ****************************************************************************/
static U8BIT* GetEventCridType(void *serv_ptr, void *event_ptr, U8BIT crid_type, U8BIT index)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT *crid_str;

   FUNCTION_START(GetEventCridType);

   ASSERT(serv_ptr != NULL);
   ASSERT(event_ptr != NULL);

   crid_str = NULL;

   if ((serv_ptr != NULL) && (event_ptr != NULL))
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById((ADB_SERVICE_REC *)serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         crid_str = DBDEF_GetEventCrid(serv_ptr, e_ptr, crid_type, index);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(GetEventCridType);

   return(crid_str);
}

/**
 * @brief   Returns the number of CRIDs of the given type for the event
 * @param   event_ptr counting the CRIDs for this event
 * @param   crid_type CRID type to be returned
 * @return  number of CRIDs found
 */
static U8BIT NumberOfCridsOfType(S_EVENT *event_ptr, U8BIT crid_type)
{
   ADB_EVENT_REC *e_ptr;
   U8BIT crids_found;

   FUNCTION_START(NumberOfCridsOfType);

   ASSERT(event_ptr != NULL);

   crids_found = 0;

   if (event_ptr != NULL)
   {
      DBDEF_RequestAccess();

      e_ptr = DBDEF_FindScheduleEventById(((S_EVENT *)event_ptr)->serv_ptr, ((S_EVENT *)event_ptr)->event_id);
      if (e_ptr != NULL)
      {
         crids_found = DBDEF_NumberOfCridsOfType(e_ptr, crid_type);
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(NumberOfCridsOfType);

   return(crids_found);
}

/*!**************************************************************************
 * @brief   Checks whether the given stream is of the given type
 * @param   stream_ptr stream to be checked
 * @param   stream_list_type type of stream
 * @return  TRUE if the stream type matches the given type, FALSE otherwise
 ****************************************************************************/
static BOOLEAN IsStreamOfType(ADB_STREAM_REC *stream_ptr, ADB_STREAM_LIST_TYPE stream_list_type)
{
   BOOLEAN retval;

   FUNCTION_START(IsStreamOfType);

   retval = FALSE;

   if ((stream_list_type == ADB_STREAM_LIST_ALL) ||
       (((stream_list_type & ADB_VIDEO_LIST_STREAM) != 0) &&
        ((stream_ptr->type == ADB_VIDEO_STREAM) ||
         (stream_ptr->type == ADB_H264_VIDEO_STREAM) ||
         (stream_ptr->type == ADB_H265_VIDEO_STREAM)
        )) ||
       (((stream_list_type & ADB_AUDIO_LIST_STREAM) != 0) &&
        ((stream_ptr->type == ADB_AUDIO_STREAM) ||
         (stream_ptr->type == ADB_AAC_AUDIO_STREAM) ||
         (stream_ptr->type == ADB_HEAAC_AUDIO_STREAM) ||
         (stream_ptr->type == ADB_AC3_AUDIO_STREAM) ||
         (stream_ptr->type == ADB_EAC3_AUDIO_STREAM)
        )) ||
       (((stream_list_type & ADB_TTEXT_LIST_STREAM) != 0) && (stream_ptr->type == ADB_TTEXT_STREAM)) ||
       (((stream_list_type & ADB_TTEXT_SUBT_LIST_STREAM) != 0) && (stream_ptr->type == ADB_TTEXT_STREAM) &&
        ((stream_ptr->data.ttext.type == ADB_TELETEXT_TYPE_SUBTITLE) ||
         (stream_ptr->data.ttext.type == ADB_TELETEXT_TYPE_SUBTITLE_HARD_HEARING))) ||
       (((stream_list_type & ADB_SUBTITLE_LIST_STREAM) != 0) && (stream_ptr->type == ADB_SUBTITLE_STREAM)) ||
       (((stream_list_type & ADB_DATA_LIST_STREAM) != 0) && (stream_ptr->type == ADB_DATA_STREAM)))
   {
      retval = TRUE;
   }

   FUNCTION_FINISH(IsStreamOfType);

   return(retval);
}

/**
 * @brief   Derive audio codec from stream
 * @param   audio_stream_type from which to derive the audio codec
 * @return  audio codec that matches audio_stream_type, AUDIO_CODEC_AUTO if no match
 */
E_STB_DP_AUDIO_CODEC ADB_GetAudioCodecFromStream(ADB_STREAM_TYPE audio_stream_type)
{
   switch (audio_stream_type)
   {
      case ADB_AAC_AUDIO_STREAM:
         return AUDIO_CODEC_AAC;

      case ADB_HEAAC_AUDIO_STREAM:
         return AUDIO_CODEC_HEAAC;

      case ADB_AC3_AUDIO_STREAM:
         return AUDIO_CODEC_AC3;

      case ADB_EAC3_AUDIO_STREAM:
         return AUDIO_CODEC_EAC3;

      case ADB_AUDIO_STREAM:
         return AUDIO_CODEC_MP2;

      default:
         break;
   }
   return AUDIO_CODEC_AUTO;
}

/**
 * @brief   Derive video codec from stream
 * @param   video_stream_type from which to derive the video codec
 * @return  video codec that matches video_stream_type, VIDEO_CODEC_AUTO if no match
 */
E_STB_DP_VIDEO_CODEC ADB_GetVideoCodecFromStream(ADB_STREAM_TYPE video_stream_type)
{
   switch (video_stream_type)
   {
      case ADB_H264_VIDEO_STREAM:
         return VIDEO_CODEC_H264;

      case ADB_VIDEO_STREAM:
         return VIDEO_CODEC_MPEG2;

      case ADB_H265_VIDEO_STREAM:
        return VIDEO_CODEC_H265;

      default:
         break;
   }
   return VIDEO_CODEC_AUTO;
}

/**
 * @brief   Returns the service pointer of event
 * @param   event_ptr event pointer
 * @return  service pointer of event
 */
void* ADB_GetEventService(void *event_ptr)
{
   void *service_ptr = NULL;

   FUNCTION_START(ADB_GetEventService);

   if (event_ptr != NULL)
   {
      service_ptr = ((S_EVENT *)event_ptr)->serv_ptr;
   }

   FUNCTION_FINISH(ADB_GetEventService);

   return service_ptr;
}

//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------

/**
 *
 * @brief   Determine if a given event spans a specific time
 *
 * @param   event_ptr - pointer to an ADB_EVENT_REC structure
 * @param   time - time to check the event against
 *
 * @return   TRUE if the time occurs during the event, FALSE otherwise
 *
 */
static BOOLEAN EventAtTime(ADB_EVENT_REC *event_ptr, U32DHMS time)
{
   BOOLEAN value = FALSE;
   U32DHMS start_time, end_time;

   start_time = event_ptr->start;
   if (start_time <= time)
   {
      end_time = STB_GCCalculateDHMS(event_ptr->start, event_ptr->duration, CALC_ADD);
      if (end_time > time)
      {
         value = TRUE;
      }
   }

   return value;
}

/**
 * @brief   Copies the string from an ADB_STRING structure into temp memory
 * @param   str_desc - pointer to an ADB_STRING structure
 * @return   pointer to the copied string
 */
static U8BIT* CopyString(ADB_STRING *str_desc, BOOLEAN to_unicode)
{
   U16BIT str_len;
   U8BIT *copied_string;

   FUNCTION_START(CopyString);

   copied_string = NULL;
   if (str_desc != NULL)
   {
      if (to_unicode)
      {
         copied_string = STB_ConvertStringToUTF8(str_desc->str_ptr, &str_len, FALSE,
               str_desc->lang_code);
      }
      else
      {
         str_len = (U16BIT)str_desc->nbytes;
         copied_string = (U8BIT *)STB_AppGetMemory(str_len);
         if (copied_string != NULL)
         {
            memcpy(copied_string, str_desc->str_ptr, str_len);
         }
      }
   }

   FUNCTION_FINISH(CopyString);
   return(copied_string);
}

static U8BIT* CopySIString(SI_STRING_DESC *str_desc)
{
   U32BIT str_len;
   U8BIT *copied_string;

   FUNCTION_START(CopySIString);

   copied_string = NULL;
   if (str_desc != NULL)
   {
      str_len = str_desc->nbytes;
      copied_string = (U8BIT *)STB_AppGetMemory(str_len);
      if (copied_string != NULL)
      {
         memcpy(copied_string, str_desc->str_ptr, str_len);
      }
   }

   FUNCTION_FINISH(CopySIString);
   return(copied_string);
}

/**
 *

 *
 * @brief   Extracts the short name, defined by control codes 0x86 and 0x87, from the full
 *                name. If there are no control codes the short name will be the same as
 *                the full name
 *
 * @param   str_desc - pointer to full name string
 *
 * @return   pointer to the short name string in temp memory
 *
 */
static U8BIT* ExtractShortName(ADB_STRING *str_desc)
{
   U8BIT *ucode_name;
   U16BIT full_name_len;
   BOOLEAN reverse;
   U16BIT short_name_len;
   BOOLEAN copy_state;
   U16BIT i;
   U16BIT ucode_char;
   U16BIT wr_id;
   U16BIT rd_id;
   U8BIT *short_name;
   U32BIT num_bytes;

   FUNCTION_START(ExtractShortName);

   short_name = NULL;
   if (str_desc != NULL)
   {
      ucode_name = STB_ConvertStringToUnicode(str_desc->str_ptr, &reverse, &full_name_len, FALSE,
            str_desc->lang_code);

      if (ucode_name != NULL)
      {
         // control codes 0x0086/0xe086 and 0x0087/0xe087 are used to identify short name - 86 means
         // following characters are in the short name, 87 means following characters are excluded from
         // short name
         //
         // so, search along the unicode string for control codes shifting characters up to beginning

         wr_id = 1;       // skip unicode identifier code in unicode_name[0]
         rd_id = 1;
         copy_state = FALSE;
         short_name_len = 0;
         for (i = 0; i < full_name_len; i++)
         {
            ucode_char = (ucode_name[rd_id] << 8) | ucode_name[rd_id + 1];
            if ((ucode_char == 0x0086) || (ucode_char == 0xe086))
            {
               copy_state = TRUE;
            }
            else if ((ucode_char == 0x0087) || (ucode_char == 0xe087))
            {
               copy_state = FALSE;
            }
            else
            {
               if (copy_state == TRUE)
               {
                  ucode_name[wr_id] = ucode_name[rd_id];
                  ucode_name[wr_id + 1] = ucode_name[rd_id + 1];
                  wr_id += 2;
                  short_name_len++;
               }
            }
            rd_id += 2;
         }

         if (short_name_len != 0)
         {
            // control characters were found and the name has been shortened - insert a new null
            // terminator at the appropriate place
            ucode_name[wr_id] = 0x00;
            ucode_name[wr_id + 1] = 0x00;
         }

         short_name = STB_ConvertUTF16toUTF8(ucode_name, &num_bytes);

         // release the unicode string storage
         STB_ReleaseUnicodeString(ucode_name);
      }
   }

   FUNCTION_FINISH(ExtractShortName);
   return(short_name);
}

/**
 *

 *
 * @brief   Returns the name for the specified transport.Memory is allocated as TempMemory so
 *                will be released when the calling screen closes
 *
 * @param   t_ptr - transport pointer
 *
 * @return   pointer to the name string
 *
 */
static U8BIT* GetTransportName(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT *name_str;
   ADB_STRING str;

   FUNCTION_START(GetTransportName);

   name_str = NULL;

   if (t_ptr != NULL)
   {
      str.lang_code = 0;
      str.nbytes = 0;
      str.str_ptr = ACTL_GetRfNameFromFreq(t_ptr->sig_type, t_ptr->frequency);
      name_str = CopyString(&str, TRUE);
   }

   FUNCTION_FINISH(GetTransportName);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the transport id of the specified transport
 *
 * @param   transport pointer
 *
 * @return   tid_value
 *
 */
static U16BIT GetTransportTid(ADB_TRANSPORT_REC *t_ptr)
{
   U16BIT tid_value;

   FUNCTION_START(GetTransportTid);

   tid_value = 0;
   if (t_ptr != NULL)
   {
      tid_value = t_ptr->tran_id;
   }

   FUNCTION_FINISH(GetTransportTid);
   return(tid_value);
}

/**
 *

 *
 * @brief   Returns the tuned signal strength of the specified transport
 *
 * @param   transport pointer
 *
 * @return   tuned signal strength as a percentage
 *
 */
static U8BIT GetTransportTunedStrength(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT tuned_signal_strength;

   FUNCTION_START(GetTransportTunedStrength);

   tuned_signal_strength = 0;
   if (t_ptr != NULL)
   {
      if (t_ptr->signal_level_at_search != BAD_SIGNAL_STATUS)
      {
         tuned_signal_strength = GET_SIGNAL_STRENGTH(t_ptr->signal_level_at_search);
      }
   }

   FUNCTION_FINISH(GetTransportTunedStrength);
   return(tuned_signal_strength);
}

/**
 *

 *
 * @brief   Returns the tuned signal quality of the specified transport
 *
 * @param   transport pointer
 *
 * @return   tuned signal quality as a percentage
 *
 */
static U8BIT GetTransportTunedQuality(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT tuned_signal_quality;

   FUNCTION_START(GetTransportTunedQuality);

   tuned_signal_quality = 0;
   if (t_ptr != NULL)
   {
      if (t_ptr->signal_level_at_search != BAD_SIGNAL_STATUS)
      {
         tuned_signal_quality = GET_SIGNAL_QUALITY(t_ptr->signal_level_at_search);
      }
   }

   FUNCTION_FINISH(GetTransportTunedQuality);
   return(tuned_signal_quality);
}

/**
 *

 *
 * @brief   Returns the original nid of the specified transport
 *
 * @param   transport pointer
 *
 * @return   orig_nid
 *
 */
static U16BIT GetTransportOriginalNetworkId(ADB_TRANSPORT_REC *t_ptr)
{
   U16BIT orig_nid;

   FUNCTION_START(GetTransportListOriginalNetworkId);

   orig_nid = 0;
   if (t_ptr != NULL)
   {
      orig_nid = t_ptr->orig_net_id;
   }

   FUNCTION_FINISH(GetTransportOriginalNetworkId);
   return(orig_nid);
}

/**
 *

 *
 * @brief   Returns the freq of the specified transport
 *
 * @param   transport pointer
 *
 * @return   tp_freq
 *
 */
static U32BIT GetTransportFreq(ADB_TRANSPORT_REC *t_ptr)
{
   U32BIT tp_freq;

   FUNCTION_START(GetTransportFreq);

   tp_freq = 0;
   if (t_ptr != NULL)
   {
      tp_freq = (t_ptr->frequency);
   }

   FUNCTION_FINISH(GetTransportFreq);
   return(tp_freq);
}

/**
 *

 *
 * @brief   Returns the freq of the specified transport
 *
 * @param   transport pointer
 *
 * @return   bwidth
 *
 */
static U8BIT GetTransportBwidth(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT bwidth_no;

   FUNCTION_START(GetTransportBwidth);

   bwidth_no = 0;
   if (t_ptr != NULL)
   {
      switch (t_ptr->u.terr.bwidth)
      {
         case TBWIDTH_8MHZ:   {bwidth_no = 8; break; }
         case TBWIDTH_7MHZ:   {bwidth_no = 7; break; }
         case TBWIDTH_6MHZ:   {bwidth_no = 6; break; }
         case TBWIDTH_5MHZ:   {bwidth_no = 5; break; }
         case TBWIDTH_10MHZ:  {bwidth_no = 10; break; }
         case TBWIDTH_UNDEFINED:  {bwidth_no = 0; break; }
      }
   }

   FUNCTION_FINISH(GetTransportBwidth);
   return(bwidth_no);
}

/**
 *

 *
 * @brief   Returns the offset of the specified transport
 *
 * @param   transport pointer
 *
 * @return   offset
 *
 */
static S8BIT GetTransportOffset(ADB_TRANSPORT_REC *t_ptr)
{
   S8BIT offset;

   FUNCTION_START(GetTransportOffset);

   offset = 0;
   if (t_ptr != NULL)
   {
      offset = t_ptr->u.terr.freq_offset;
   }

   FUNCTION_FINISH(GetTransportOffset);

   return(offset);
}

/**
 *

 *
 * @brief   Returns the constellation of the specified transport
 *
 * @param   transport pointer
 *
 * @return   constallation
 *
 */
static U8BIT* GetTransportConstellationString(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT *temp_str;
   U8BIT *trans_qam_string_ptr;
   U32BIT nbytes;

   FUNCTION_START(GetTransportConstellationString);

   temp_str = NULL;
   if (t_ptr != NULL)
   {
      switch (t_ptr->u.terr.constellation)
      {
         case TUNE_TCONST_QPSK:  {temp_str = (U8BIT *)"QPSK QAM"; break; }
         case TUNE_TCONST_QAM16: {temp_str = (U8BIT *)"16 QAM"; break; }
         case TUNE_TCONST_QAM64: {temp_str = (U8BIT *)"64 QAM"; break; }
         default: break;
      }
   }

   trans_qam_string_ptr = NULL;
   if (temp_str != NULL)
   {
      nbytes = STB_GetNumBytesInString(temp_str);
      trans_qam_string_ptr = STB_AppGetMemory(sizeof(U8BIT) * nbytes);
      if (trans_qam_string_ptr != NULL)
      {
         memcpy(trans_qam_string_ptr, temp_str, nbytes);
      }
   }

   FUNCTION_FINISH(GetTransportConstellationString);
   return(trans_qam_string_ptr);
}

/**
 *

 *
 * @brief   Returns the heirarchy of the specified transport
 *
 * @param   transport pointer
 *
 * @return   hierarchy
 *
 */
static U8BIT* GetTransportHierarchyString(ADB_TRANSPORT_REC *t_ptr)
{
   U8BIT *temp_str;
   U8BIT *hierarchy_str;
   U8BIT nbytes;

   FUNCTION_START(GetTransportHierarchyString);

   temp_str = NULL;
   if (t_ptr != NULL)
   {
      switch (t_ptr->u.terr.hierarchy)
      {
         case TUNE_THIERARCHY_NONE: {temp_str = (U8BIT *)"None"; break; }
         case TUNE_THIERARCHY_1:    {temp_str = (U8BIT *)"1"; break; }
         case TUNE_THIERARCHY_2:    {temp_str = (U8BIT *)"2"; break; }
         case TUNE_THIERARCHY_4:    {temp_str = (U8BIT *)"4"; break; }
         default: break;
      }
   }

   hierarchy_str = NULL;
   if (temp_str != NULL)
   {
      nbytes = (U8BIT)STB_GetNumBytesInString(temp_str);
      hierarchy_str = STB_AppGetMemory(sizeof(U8BIT) * nbytes);
      if (hierarchy_str != NULL)
      {
         memcpy(hierarchy_str, temp_str, nbytes);
      }
   }

   FUNCTION_FINISH(GetTransportHierarchyString);
   return(hierarchy_str);
}

/**
 *

 *
 * @brief   Checks teh supplied service matches the specified list. Note that list type can
 *                be an OR'ing or more than one type.
 *
 * @param   s_ptr      - service to be checked
 * @param   list_type  - defines the type of service records required in the list
 * @param   inc_hidden - TRUE to include hidden services in list
 * @param   ignore_selectable - TRUE to include non-selectable services in list
 *
 * @return   TRUE if service is in the specified list, FALSE otherwise
 *
 */
static BOOLEAN CheckServiceInListType(ADB_SERVICE_REC *s_ptr, U32BIT list_type, BOOLEAN inc_hidden,
   BOOLEAN ignore_selectable)
{
   BOOLEAN retval;
   ADB_FAVLIST_REC *fl_ptr;

   FUNCTION_START(CheckServiceInListType);

   retval = FALSE;

   if (s_ptr != NULL)
   {
      if (!s_ptr->hidden || ignore_selectable || (inc_hidden && s_ptr->selectable))
      {
         if ((list_type == ADB_SERVICE_LIST_ALL) ||
             (((list_type & ADB_SERVICE_LIST_TV) != 0) &&
              ((s_ptr->serv_type == ADB_SERVICE_TYPE_TV) ||
               (s_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
               (s_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
               (s_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
               (s_ptr->serv_type == ADB_SERVICE_TYPE_UHD_TV)
              )) ||
             (((list_type & ADB_SERVICE_LIST_RADIO) != 0) &&
              ((s_ptr->serv_type == ADB_SERVICE_TYPE_RADIO) ||
               (s_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO)
              )) ||
             (((list_type & ADB_SERVICE_LIST_DATA) != 0) && (s_ptr->serv_type == ADB_SERVICE_TYPE_DATA)) ||
             (((list_type & ADB_SERVICE_LIST_FAV_GROUP_A) != 0) && ((s_ptr->fav_groups & FAV_GROUP_A) != 0)) ||
             (((list_type & ADB_SERVICE_LIST_FAV_GROUP_B) != 0) && ((s_ptr->fav_groups & FAV_GROUP_B) != 0)) ||
             (((list_type & ADB_SERVICE_LIST_FAV_GROUP_C) != 0) && ((s_ptr->fav_groups & FAV_GROUP_C) != 0)) ||
             (((list_type & ADB_SERVICE_LIST_FAV_GROUP_D) != 0) && ((s_ptr->fav_groups & FAV_GROUP_D) != 0)) ||
             (((list_type & ADB_SERVICE_LIST_ANALOG) != 0) && ((s_ptr->serv_type == ADB_SERVICE_TYPE_ANALOG) != 0)) ||
             ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0))
         {
            if ((list_type & ADB_SERVICE_LIST_FREESAT) == 0 ||
                (s_ptr->freesat_id != INVALID_FREESAT_SERV_ID &&
                 ((s_ptr->allocated_lcn >= 100 && s_ptr->allocated_lcn <= 999) ||
                  (s_ptr->allocated_lcn >= 1100 && s_ptr->allocated_lcn <= 1999))))
            {
               if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
               {
                  /* Check that the service is in the given favourite list */
                  if ((fl_ptr = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
                  {
                     if (DBDEF_FindServiceInFavouriteList(fl_ptr, s_ptr) != NULL)
                     {
                        /* Service is in the given favourite list */
                        retval = TRUE;
                     }
                  }
               }
               else
               {
                  retval = TRUE;
               }
            }
         }
      }
   }

   FUNCTION_FINISH(CheckServiceInListType);

   return(retval);
}

/**
 *

 *
 * @brief   Creates a list of service record pointers and passes back to the caller, with
 *                number of entries in the list. Memory needs to be freed when finished with.
 *
 * @param   list_type       - defines the type of service records required in the list
 * @param   slist_ptr       - address for the return of the slist
 * @param   num_entries_ptr - address for the return of the number of entries in the list
 *

 *
 */
static void GetServiceList(U32BIT list_type, void ***slist_ptr, U16BIT *num_entries_ptr,
   BOOLEAN show_hidden, BOOLEAN ignore_selectable)
{
   void **slist;
   U16BIT max_no_entries;
   U16BIT num_entries;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;
   ADB_FAVLIST_REC *fav_list;
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(GetServiceList);

   DBDEF_RequestAccess();

   slist = NULL;
   fav_list = NULL;
   fav_serv = NULL;
   num_entries = 0;
   max_no_entries = 0;

   /* Getting services for a favourite list by iterating through all the services
    * ignores the ordering that's associated with a favourite list, so favourite list
    * services have to be acquired directly from the favourite list */
   if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
   {
      if ((fav_list = DBDEF_FindFavouriteList(ADB_FAVLIST_FROM_LIST_TYPE(list_type))) != NULL)
      {
         max_no_entries = DBDEF_GetNumServicesInFavouriteList(fav_list);
      }
   }
   else
   {
      max_no_entries = DBDEF_GetNumServices();
   }

   if (max_no_entries > 0)
   {
      slist = STB_AppGetMemory(max_no_entries * sizeof(void *));
      if (slist != NULL)
      {
         if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
         {
            if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, NULL)) != NULL)
            {
               s_ptr = fav_serv->serv_ptr;
            }
            else
            {
               s_ptr = NULL;
            }
         }
         else
         {
            s_ptr = DBDEF_GetNextServiceRec(NULL);
         }

         for (i = 0; ((i < max_no_entries) && (s_ptr != NULL)); i++)
         {
            if (DBDEF_ServiceInProfile(s_ptr) /*&&
               (CheckServiceInListType(s_ptr, list_type, show_hidden, ignore_selectable) == TRUE)*/)
            {
               if (ACFG_IsNordigCountry())
               {
                  if (!HasNordigSimulcastService(s_ptr))
                  {
                     slist[num_entries] = (void *)s_ptr;
                     num_entries++;
                  }
               }
               else
               {
                  slist[num_entries] = (void *)s_ptr;
                  num_entries++;
               }
            }

            if ((list_type & ADB_SERVICE_LIST_FAV_LIST) != 0)
            {
               if ((fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv)) != NULL)
               {
                  s_ptr = fav_serv->serv_ptr;
               }
               else
               {
                  s_ptr = NULL;
               }
            }
            else
            {
               s_ptr = DBDEF_GetNextServiceRec(s_ptr);
            }
         }

         if (num_entries == 0)
         {
            STB_AppFreeMemory(slist);
            slist = NULL;
         }
      }
   }

   *slist_ptr = slist;
   *num_entries_ptr = num_entries;

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(GetServiceList);
}

/**
 *

 *
 * @brief   Returns the name for the specified service, language and preferred name id.
 *                Memory is allocated as TempMemory so will be released when the calling screen
 *                closes
 *
 * @param   s_ptr        - service pointer
 * @param   lang         - required language id
 * @param   pref_name_id - required preferred name id
 * @param   short_name   - TRUE if short name is required, otherwise full name is returned
 *
 * @return   pointer to the name string
 *
 */
static U8BIT* GetServiceNameByLangAndPrefId(ADB_SERVICE_REC *s_ptr, U8BIT lang, U8BIT pref_name_id,
   BOOLEAN short_name)
{
   U8BIT *name_str;

   FUNCTION_START(GetServiceNameByLangAndPrefId);

   name_str = NULL;
   if (s_ptr != NULL)
   {
      if (short_name == TRUE)
      {
         name_str = ExtractShortName(s_ptr->name_array[lang][pref_name_id]);
      }
      else
      {
         name_str = CopyString(s_ptr->name_array[lang][pref_name_id], TRUE);
      }
   }
   FUNCTION_FINISH(GetServiceNameByLangAndPrefId);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the lcn for the specified service.Memory is allocated as TempMemory so
 *                will be released when the calling screen closes
 *
 * @param   s_ptr - service pointer
 *
 * @return   lcn
 *
 */
static U16BIT GetServiceLcn(ADB_SERVICE_REC *s_ptr)
{
   U16BIT lcn_value;

   FUNCTION_START(GetServiceLcn);

   lcn_value = 0;
   if (s_ptr != NULL)
   {
      lcn_value = s_ptr->allocated_lcn;
   }
   FUNCTION_FINISH(GetServiceLcn);

   return(lcn_value);
}

/**
 *

 *
 * @brief   Returns the service id of the specified service
 *
 * @param   service pointer
 *
 * @return   sid_value
 *
 */
static U16BIT GetServiceId(ADB_SERVICE_REC *s_ptr)
{
   U16BIT sid_value;

   FUNCTION_START(GetServiceId);

   sid_value = 0;
   if (s_ptr != NULL)
   {
      sid_value = s_ptr->serv_id;
   }
   FUNCTION_FINISH(GetServiceId);

   return(sid_value);
}

/**
 *

 *
 * @brief   Returns the unavailable status of the specified service
 *
 * @param   service pointer
 *
 * @return   unavailable
 *
 */
static BOOLEAN GetServiceUnavailFlag(void *s_ptr)
{
   BOOLEAN unavailable;

   FUNCTION_START(GetServiceUnavailFlag);

   unavailable = FALSE;
   if (s_ptr != NULL)
   {
      unavailable = ((ADB_SERVICE_REC *)s_ptr)->unavailable;
   }

   FUNCTION_FINISH(GetServiceUnavailFlag);
   return(unavailable);
}

/**
 *

 *
 * @brief   Returns the new flag of the specified service
 *
 * @param   service pointer
 *
 * @return   new
 *
 */
static BOOLEAN GetServiceNewFlag(void *s_ptr)
{
   BOOLEAN new_service;

   FUNCTION_START(GetServiceNewFlag);

   new_service = FALSE;
   if (s_ptr != NULL)
   {
      new_service = ((ADB_SERVICE_REC *)s_ptr)->new_service;
   }

   FUNCTION_FINISH(GetServiceNewFlag);
   return(new_service);
}

/**
 *

 *
 * @brief   Returns the name for the specified network.Memory is allocated as TempMemory so
 *                will be released when the calling screen closes
 *
 * @param   n_ptr - network pointer
 *
 * @return   pointer to the name string
 *
 */
static U8BIT* GetNetworkName(ADB_NETWORK_REC *n_ptr, BOOLEAN short_name)
{
   U8BIT *name_str;
   U8BIT *lang_ids;
   U8BIT i;

   FUNCTION_START(GetNetworkName);
   name_str = NULL;

   if (n_ptr != NULL)
   {
      // check if string is defined, if not use default name
      lang_ids = DBDEF_GetTextLang();

      if (lang_ids != NULL)
      {
         for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
              (lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
         {
            if (n_ptr->name_array[lang_ids[i]] != NULL)
            {
               name_str = GetNetworkNameByLang(n_ptr, lang_ids[i], short_name);
            }
         }
      }

      if (name_str == NULL)
      {
         // use the default name
         name_str = GetDefaultNetworkName(n_ptr, short_name);
      }
   }
   FUNCTION_FINISH(GetNetworkName);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the name for the specified network and language. Memory is allocated as
 *                TempMemory so will be released when the calling screen closes
 *
 * @param   n_ptr - network pointer
 * @param   lang  - required language id
 *
 * @return   pointer to the name string
 *
 */
static U8BIT* GetNetworkNameByLang(ADB_NETWORK_REC *n_ptr, U8BIT lang, BOOLEAN short_name)
{
   U8BIT *name_str;

   FUNCTION_START(GetNetworkNameByLang);

   name_str = NULL;
   if (n_ptr != NULL)
   {
      if (short_name)
      {
         name_str = ExtractShortName(n_ptr->name_array[lang]);
      }
      else
      {
         name_str = CopyString(n_ptr->name_array[lang], TRUE);
      }
   }
   FUNCTION_FINISH(GetNetworkNameByLang);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the default name for the specified network.
 *                Memory is allocated as TempMemory so will be released when the calling screen
 *                closes
 *
 * @param   n_ptr - network pointer
 *
 * @return   pointer to the name string
 *
 */
static U8BIT* GetDefaultNetworkName(ADB_NETWORK_REC *n_ptr, BOOLEAN short_name)
{
   U8BIT *name_str;

   FUNCTION_START(GetDefaultNetworkName);

   name_str = NULL;

   if (n_ptr != NULL)
   {
      if (short_name)
      {
         name_str = ExtractShortName(n_ptr->name_str);
      }
      else
      {
         name_str = CopyString(n_ptr->name_str, TRUE);
      }
   }
   FUNCTION_FINISH(GetDefaultNetworkName);
   return(name_str);
}

/**
 *

 *
 * @brief   Returns the network id of the specified network
 *
 * @param   network pointer
 *
 * @return   nid_value
 *
 */
static U16BIT GetNetworkId(ADB_NETWORK_REC *n_ptr)
{
   U16BIT nid_value;

   FUNCTION_START(GetNetworkId);

   nid_value = 0;
   if (n_ptr != NULL)
   {
      nid_value = n_ptr->net_id;
   }
   FUNCTION_FINISH(GetNetworkId);

   return(nid_value);
}

/**
 *

 *
 * @brief   Makes a copy of the specified ADB_EVENT_REC into a temporary S_EVENT, for use in
 *                the application. The copy is necessary in case the SI task updates the events
 *                while the UI is displaying the events. So grab a snapshot here, then it does not
 *                matter if the database record is deleted and re-created.
 *
 * @param   e_ptr - pointer to the specified ADB_EVENT_REC
 * @param   s_ptr - the service of the event
 *
 * @return   the pointer to the new S_EVENT structure
 *
 */
static S_EVENT* CopyEventRec(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr)
{
   S_EVENT *tmp_event;

   FUNCTION_START(CopyEventRec);

   tmp_event = NULL;

   if (e_ptr != NULL)
   {
      tmp_event = STB_AppGetMemory(sizeof(S_EVENT));
      if (tmp_event != NULL)
      {
         memset(tmp_event, 0, sizeof(S_EVENT));

         tmp_event->serv_ptr = (void *)s_ptr;
         tmp_event->event_id = e_ptr->event_id;
      }
   }

   FUNCTION_FINISH(CopyEventRec);

   return(tmp_event);
}

static BOOLEAN FindDoNotScramble(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr)
{
   BOOLEAN has_desc;
   BOOLEAN do_not_scramble;

   FUNCTION_START(FindDoNotScramble);

   has_desc = FALSE;
   do_not_scramble = TRUE;

   if (e_ptr != NULL)
   {
      if (e_ptr->has_content_management_desc)
      {
         has_desc = TRUE;
         do_not_scramble = e_ptr->do_not_scramble;
      }
   }

   if (!has_desc)
   {
      /* Check the service for the flag */
      if (s_ptr->has_fta_desc)
      {
         do_not_scramble = s_ptr->do_not_scramble;
      }
      else
      {
         /* Check the service's transport record */
         if (s_ptr->transport != NULL)
         {
            if (s_ptr->transport->has_fta_desc)
            {
               do_not_scramble = s_ptr->transport->do_not_scramble;
            }
            else
            {
               /* Check the network record */
               if (s_ptr->transport->network != NULL)
               {
                  if (s_ptr->transport->network->has_fta_desc)
                  {
                     do_not_scramble = s_ptr->transport->network->do_not_scramble;
                  }
                  else
                  {
                     /* Check the satellite */
                     if (s_ptr->transport->network->satellite != NULL)
                     {
                        if (s_ptr->transport->network->satellite->has_fta_desc)
                        {
                           do_not_scramble = s_ptr->transport->network->satellite->do_not_scramble;
                        }
                     }
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(FindDoNotScramble);

   return(do_not_scramble);
}

/*!**************************************************************************
 * @brief   This checks for if a service has a Nordig Simulcast in its Linkage
 * @param   serv_ptr service pointer to check for simulcast service
 * @return  BOOLEAN TRUE - Simulcast service, FALSE - Non-Simulcast service
 ****************************************************************************/
static BOOLEAN HasNordigSimulcastService(void *serv_ptr)
{
   ADB_SERVICE_REC *temp_serv_ptr;
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   BOOLEAN found;

   FUNCTION_START(HasNordigSimulcastService);

   found = FALSE;

   temp_serv_ptr = (ADB_SERVICE_REC *)serv_ptr;
   if (temp_serv_ptr != NULL)
   {
      desc_ptr = temp_serv_ptr->linkage_desc_list;
      while ((desc_ptr != NULL) && !found)
      {
         if (desc_ptr->link_type == NORDIG_SIMULCAST_TYPE)
         {
            if (DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, desc_ptr->orig_net_id,
                   desc_ptr->tran_id, desc_ptr->serv_id) != NULL)
            {
               found = TRUE;
            }
         }
         desc_ptr = desc_ptr->next;
      }
   }

   FUNCTION_FINISH(HasNordigSimulcastService);

   return(found);
}

//**************************************************************************************************
// End of File
//**************************************************************************************************
