/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application database control
 * @file    ap_dbdef.c
 * @date    March 2003
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <string.h>
#include <stdio.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwav.h"
#include "stbhwos.h"

#include "stbheap.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbuni.h"
#include "stbgc.h"

#include "stbllist.h"
#include "dba.h"

#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"
#include "ap_cfdat.h"

#include "ap_cntrl.h"
#include "app_nvm.h"



//---constant definitions for this file-------------------------------------------------------------
#define NID_PRIVATE_BOUNDARY     0xFF01
#define ONID_PRIVATE_BOUNDARY    0xFF00

#ifdef COUNTRY_FINLAND
#define FINLAND_ONID             0x20F6
#endif

#ifdef COUNTRY_IRELAND
#define IRELAND_ONID             0x2174
#endif

#define PRIVATE_NETWORK_ID0      0xff01   /* See TS 101 162, section 5.6.2 */

#define TIMER_HANDLE_BASE        INVALID_TIMER_HANDLE + 1
#define LAST_TIMER_HANDLE        0xffffffff

#define PRIV_DATA_SPEC_DTAG      0x5f

#define UK_DTT_PRIVATE_DATA_CODE    0x0000233a
#define FREESAT_PRIVATE_DATA_CODE   0x46534154

//#define DEBUG_DBDEF

#ifdef DEBUG_DBDEF
#define DBG_DBDEF(X)    STB_SPDebugWrite X
#else
#define DBG_DBDEF(X)
#endif


//---local typedefs, structs, enumerations for this file--------------------------------------------

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

static void *db_access_mutex;

CREATE_LINK_LIST_HEADER(lnb_rec_list);
CREATE_LINK_LIST_HEADER(satellite_rec_list);
CREATE_LINK_LIST_HEADER(lnb_band_rec_list);
CREATE_LINK_LIST_HEADER(network_rec_list);
CREATE_LINK_LIST_HEADER(transport_rec_list);
CREATE_LINK_LIST_HEADER(service_rec_list);
CREATE_LINK_LIST_HEADER(crid_rec_list);
CREATE_LINK_LIST_HEADER(favlist_list);
CREATE_LINK_LIST_HEADER(timer_list);
#ifdef COMMON_INTERFACE
CREATE_LINK_LIST_HEADER(cicam_timer_list);
#endif

static U8BIT num_paths;

static ADB_NETWORK_REC **tuned_network = NULL;

static U32BIT last_timer_handle;

/* Profile related info */
static ADB_PROFILE_TYPE current_profile;
static ADB_PROFILE_TYPE pushed_profile;
#ifdef COMMON_INTERFACE
static U16BIT cicam_onet_id;
static U32BIT cicam_identifier;
static U16BIT pushed_cicam_onet_id;
static U32BIT pushed_cicam_identifier;
#endif

static U16BIT num_analog_channels;        // used for allocating the next analogue channel num/name
static ADB_NETWORK_REC *analog_network;

static U8BIT *text_langs;
static U8BIT *audio_langs;
static U8BIT *second_audio_langs;
static U8BIT *second_text_langs;

static const ADB_STREAM_TYPE audio_priority_table[] = {
   ADB_AUDIO_STREAM,
   ADB_AC3_AUDIO_STREAM,
   ADB_EAC3_AUDIO_STREAM,
   ADB_AAC_AUDIO_STREAM,
   ADB_HEAAC_AUDIO_STREAM
};


//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static void DeleteLNBRec(ADB_LNB_REC *lnb_ptr);
static void DeleteSatelliteRec(ADB_SATELLITE_REC *n_ptr);
static void DeleteNetworkRec(ADB_NETWORK_REC *n_ptr);

#ifdef COUNTRY_UK
static BOOLEAN ApplyTargetRegionRules(ADB_SERVICE_REC *s1_ptr, ADB_SERVICE_REC *s2_ptr,
   U16BIT *next_allocated_lcn, BOOLEAN *lcn_assigned);
static U8BIT GetTargetRegionMatchDepth(ADB_SERVICE_REC *s_ptr, U8BIT match_depth,
   U32BIT country_code, U8BIT primary_region, U8BIT secondary_region, U16BIT tertiary_region);
#endif /* COUNTRY_UK */

static BOOLEAN UserPrefsMatch(ADB_TELETEXT_TYPE ttext_type, E_SUBTITLE_TYPE user_pref);

static U16BIT GetReqdAudioPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *audio_mode,
   ADB_STREAM_TYPE *audio_type, U8BIT *db_lang_ids, BOOLEAN *primary_found, ADB_STREAM_REC **selected_stream);
static U8BIT AudioStreamPriority(ADB_STREAM_TYPE stream_type);
static E_STREAM_MATCH_TYPE GetReqdSubtitleParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr, U16BIT *cpage_ptr,
   U16BIT *apage_ptr, U8BIT *db_lang_ids, BOOLEAN *primary_found, ADB_STREAM_REC **selected_stream);
static E_STREAM_MATCH_TYPE GetReqdTtextParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr, BOOLEAN for_subtitles,
   U8BIT *magazine, U8BIT *page, U8BIT *db_lang_ids, BOOLEAN *primary_found, ADB_STREAM_REC **selected_stream);
static U16BIT GetReqdADPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *ad_mode, U8BIT *db_lang_ids,
   ADB_STREAM_REC **selected_stream, BOOLEAN *broadcast_mix);
#if defined(COUNTRY_FINLAND) || defined(COUNTRY_IRELAND)
static U16BIT FindLargestLcnNumber(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT onet_id);
#endif

static S16BIT LcnSortCompareFunc(LINK_LIST_PTR_BLK **blk_1, LINK_LIST_PTR_BLK **blk_2);
static S16BIT FavServiceSortCompare(LINK_LIST_PTR_BLK **blk1, LINK_LIST_PTR_BLK **blk2);

#if 0
static void InsertEventInSchedule(ADB_SERVICE_REC *serv_ptr, ADB_EVENT_REC *event_ptr);
#endif

static S16BIT TimerSortDateTime(LINK_LIST_PTR_BLK **timer1, LINK_LIST_PTR_BLK **timer2);
static S16BIT TimerSortName(LINK_LIST_PTR_BLK **timer1, LINK_LIST_PTR_BLK **timer2);
static BOOLEAN IsVideoStreamType(ADB_STREAM_TYPE type);

static U8BIT* ReadLanguageCode(U8BIT *dptr, U32BIT *lang_code);
static U8BIT* CopyString(ADB_STRING *str_desc, BOOLEAN to_unicode);

#ifdef DEBUG_PRINT_DATABASE
static void PrintLNBRec(ADB_LNB_REC *lnb_ptr);
static void PrintSatelliteRec(ADB_SATELLITE_REC *sat_ptr);
static void PrintNetworkRec(ADB_NETWORK_REC *n_ptr);
static void PrintTransportRec(ADB_TRANSPORT_REC *t_ptr);
static void PrintServiceRec(ADB_SERVICE_REC *s_ptr);
#endif


//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Initialises the database, preparing for it to be accessed
 */
void DBDEF_Initialise(void)
{
   U8BIT i;

   FUNCTION_START(DBDEF_Initialise);

   /* Initialise the database */
   DBA_Initialise();

   if (tuned_network == NULL)
   {
      num_paths = STB_DPGetNumPaths();

      tuned_network = (ADB_NETWORK_REC **)STB_AppGetMemory(sizeof(ADB_NETWORK_REC *) * num_paths);

      if (tuned_network != NULL)
      {
         for (i = 0; i < num_paths; i++)
         {
            tuned_network[i] = NULL;
         }
      }
   }

   last_timer_handle = INVALID_TIMER_HANDLE;

   db_access_mutex = STB_OSCreateMutex();

   FUNCTION_FINISH(DBDEF_Initialise);
}

/**
 * @brief   Requests access to the app's database
 */
void DBDEF_RequestAccess(void)
{
   FUNCTION_START(DBDEF_RequestAccess);
   STB_OSMutexLock(db_access_mutex);
   FUNCTION_FINISH(DBDEF_RequestAccess);
}

/**
 * @brief   Releases access to the app's database
 */
void DBDEF_ReleaseAccess(void)
{
   FUNCTION_START(DBDEF_ReleaseAccess);
   STB_OSMutexUnlock(db_access_mutex);
   FUNCTION_FINISH(DBDEF_ReleaseAccess);
}

/**
 * @brief   Loads the service database from non-volatile storage and creates the RAM version
 *          of this data that will be used by the DVB stack
 * @param   db_pathname pathname of the database file to be loaded, may be NULL or ignored
 *                        if not relevant for the type of storage being used.
 * @return  TRUE if the database is loaded, FALSE otherwise
 */
BOOLEAN DBDEF_LoadDatabase(U8BIT *db_pathname)
{
   void *dba_rec;
   void *parent_rec;
   ADB_LNB_REC *lnb_ptr;
   ADB_SATELLITE_REC *sat_ptr;
   ADB_LNB_BAND_REC *band_ptr;
   ADB_NETWORK_REC *n_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;
   ADB_CRID_REC *c_ptr;
   ADB_FAVLIST_REC *fl_ptr;
   ADB_FAVSERV_REC *fs_ptr;
   U32BIT tmp32bit;
   U8BIT *string;
   U16BIT num_bytes;
   U16BIT fl_index, fs_index;
   ADB_TIMER_REC *timer;
#ifdef COMMON_INTERFACE
   ADB_CICAM_TIMER_REC *cicam_timer;
#endif
#if 0
   ADB_EVENT_REC *e_ptr;
   U8BIT *data;
   U16BIT data_len;
   U32BIT lang_code;
   U16BIT index;
#endif
   BOOLEAN retval;

   FUNCTION_START(DBDEF_LoadDatabase);

   retval = DBA_LoadDatabase(db_pathname);
   if (retval)
   {
      /* Read records from database to create DVB runtime database */

      /* Read all LNB records */
      dba_rec = DBA_FindRecord(DBA_RECORD_LNB, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database LNB record */
         lnb_ptr = STB_AppGetMemory(sizeof(ADB_LNB_REC));
         if (lnb_ptr != NULL)
         {
            memset(lnb_ptr, 0, sizeof(ADB_LNB_REC));
            STB_LLAddBlockToEnd(&lnb_rec_list, (LINK_LIST_PTR_BLK *)lnb_ptr);

            lnb_ptr->dba_rec = dba_rec;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_TYPE, &tmp32bit);
            switch ((E_STB_DP_LNB_TYPE)tmp32bit)
            {
               case LNB_TYPE_SINGLE:
               case LNB_TYPE_UNIVERSAL:
               case LNB_TYPE_UNICABLE:
               case LNB_TYPE_USER_DEFINED:
               {
                  lnb_ptr->type = (E_STB_DP_LNB_TYPE)tmp32bit;
                  break;
               }
               default:
               {
                  /* Invalid setting, so force to default! */
                  lnb_ptr->type = LNB_TYPE_SINGLE;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_TYPE, (U32BIT)LNB_TYPE_SINGLE);
                  break;
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_POWER, &tmp32bit);
            switch ((E_STB_DP_LNB_POWER)tmp32bit)
            {
               case LNB_POWER_OFF:
               case LNB_POWER_ON:
               case LNB_POWER_AUTO:
                  lnb_ptr->power = (E_STB_DP_LNB_POWER)tmp32bit;
                  break;
               default:
                  /* Use default setting */
                  lnb_ptr->power = LNB_POWER_AUTO;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_POWER, (U32BIT)LNB_POWER_AUTO);
                  break;
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_22K, &tmp32bit);
            lnb_ptr->is_22k = (tmp32bit ? TRUE : FALSE);
            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_12V, &tmp32bit);
            lnb_ptr->is_12v = (tmp32bit ? TRUE : FALSE);
            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_PULSEPOSN, &tmp32bit);
            lnb_ptr->is_pulse_posn = (tmp32bit ? TRUE : FALSE);
            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISPOSN, &tmp32bit);
            lnb_ptr->is_diseqc_posn = (tmp32bit ? TRUE : FALSE);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISTONE, &tmp32bit);
            switch ((E_STB_DP_DISEQC_TONE)tmp32bit)
            {
               case DISEQC_TONE_OFF:
               case DISEQC_TONE_A:
               case DISEQC_TONE_B:
               {
                  lnb_ptr->diseqc_tone = (E_STB_DP_DISEQC_TONE)tmp32bit;
                  break;
               }
               default:
               {
                  /* Invalid setting, so force to default! */
                  lnb_ptr->diseqc_tone = DISEQC_TONE_OFF;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_DISTONE,
                     (U32BIT)lnb_ptr->diseqc_tone);
                  break;
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISCSWITCH, &tmp32bit);
            switch ((E_STB_DP_DISEQC_CSWITCH)tmp32bit)
            {
               case DISEQC_CSWITCH_OFF:
               case DISEQC_CSWITCH_A:
               case DISEQC_CSWITCH_B:
               case DISEQC_CSWITCH_C:
               case DISEQC_CSWITCH_D:
               {
                  lnb_ptr->c_switch = (E_STB_DP_DISEQC_CSWITCH)tmp32bit;
                  break;
               }
               default:
               {
                  /* Invalid setting, so force to default! */
                  lnb_ptr->c_switch = DISEQC_CSWITCH_OFF;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_DISCSWITCH,
                     (U32BIT)lnb_ptr->c_switch);
                  break;
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISUSWITCH, &tmp32bit);
            if (tmp32bit <= 16L)
            {
               lnb_ptr->u_switch = (U8BIT)tmp32bit;
            }
            else
            {
               /* Invalid setting, so force to default! */
               lnb_ptr->u_switch = 0;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_DISUSWITCH,
                  (U32BIT)lnb_ptr->u_switch);
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISSMATV, &tmp32bit);
            lnb_ptr->is_smatv = (tmp32bit ? TRUE : FALSE);
            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_DISREPEAT, &tmp32bit);
            if (tmp32bit <= 3L)
            {
               lnb_ptr->diseqc_repeats = (U8BIT)tmp32bit;
            }
            else
            {
               /* Invalid setting, so force to default! */
               lnb_ptr->diseqc_repeats = 0;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_DISREPEAT,
                  (U32BIT)lnb_ptr->diseqc_repeats);
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_UNICABLEFREQ, &tmp32bit);
            lnb_ptr->unicable_if = tmp32bit;
            DBA_GetFieldValue(dba_rec, DBA_FIELD_LNB_UNICABLECHAN, &tmp32bit);
            if (tmp32bit <= 7L)
            {
               lnb_ptr->unicable_chan = (U8BIT)tmp32bit;
            }
            else
            {
               lnb_ptr->unicable_chan = 0;
            }

            if (DBA_GetFieldString(dba_rec, DBA_FIELD_LNB_NAME, &string, &num_bytes) &&
               (num_bytes > 0))
            {
               lnb_ptr->name = DBDEF_MakeString(0, string, num_bytes);
            }
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_LNB, NULL, dba_rec);
      }

      /* Read all LNB band records */
      dba_rec = DBA_FindRecord(DBA_RECORD_LNB_BAND,  NULL, NULL);
      while (dba_rec != NULL)
      {
         band_ptr = STB_AppGetMemory(sizeof(ADB_LNB_BAND_REC));
         if (band_ptr != NULL)
         {
            memset(band_ptr, 0, sizeof(ADB_LNB_BAND_REC));

            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find LNB record matching parent LNB for this satellite */
               lnb_ptr = (ADB_LNB_REC *)STB_LLGetFirstBlock(&lnb_rec_list);
               while (lnb_ptr != NULL)
               {
                  if (lnb_ptr->dba_rec == parent_rec)
                  {
                     band_ptr->lnb = lnb_ptr;
                     break;
                  }
                  lnb_ptr = (ADB_LNB_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lnb_ptr);
               }

               if (band_ptr->lnb != NULL)
               {
                  /* Add this LNB band to the list */
                  STB_LLAddBlockToEnd(&lnb_band_rec_list, (LINK_LIST_PTR_BLK *)band_ptr);

                  band_ptr->dba_rec = dba_rec;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_POLARITY, &tmp32bit);
                  band_ptr->band_params.polarity = (E_STB_DP_POLARITY)tmp32bit;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_MIN_FREQUENCY, &tmp32bit);
                  band_ptr->band_params.min_freq = (U16BIT)tmp32bit;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_MAX_FREQUENCY, &tmp32bit);
                  band_ptr->band_params.max_freq = (U16BIT)tmp32bit;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_LOCAL_OSC_FREQUENCY, &tmp32bit);
                  band_ptr->band_params.local_oscillator_frequency = (U16BIT)tmp32bit;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_LNB_VOLTAGE, &tmp32bit);
                  band_ptr->band_params.lnb_voltage = (E_STB_TUNE_LNB_VOLTAGE)tmp32bit;

                  DBA_GetFieldValue(dba_rec, DBA_FIELD_BAND_22_KHZ, &tmp32bit);
                  band_ptr->band_params.tone_22k = (BOOLEAN)tmp32bit;
               }
               else
               {
                  STB_AppFreeMemory(band_ptr);
               }
            }
         }
         dba_rec = DBA_FindRecord(DBA_RECORD_LNB_BAND, NULL, dba_rec);
      }

      /* Read all satellite records */
      dba_rec = DBA_FindRecord(DBA_RECORD_SATELLITE, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database satellite record */
         sat_ptr = STB_AppGetMemory(sizeof(ADB_SATELLITE_REC));
         if (sat_ptr != NULL)
         {
            memset(sat_ptr, 0, sizeof(ADB_SATELLITE_REC));

            /* Find the LNB for this satellite */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find LNB record matching parent LNB for this satellite */
               lnb_ptr = (ADB_LNB_REC *)STB_LLGetFirstBlock(&lnb_rec_list);
               while (lnb_ptr != NULL)
               {
                  if (lnb_ptr->dba_rec == parent_rec)
                  {
                     sat_ptr->lnb = lnb_ptr;
                     break;
                  }
                  lnb_ptr = (ADB_LNB_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lnb_ptr);
               }
            }

            if (sat_ptr->lnb != NULL)
            {
               /* Add this satellite to the list */
               STB_LLAddBlockToEnd(&satellite_rec_list, (LINK_LIST_PTR_BLK *)sat_ptr);

               sat_ptr->dba_rec = dba_rec;

               DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes);
               sat_ptr->name = DBDEF_MakeString(0, string, num_bytes);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_SAT_DISH, &tmp32bit);
               sat_ptr->dish_pos = (U16BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_SAT_LONGWE, &tmp32bit);
               sat_ptr->east_west = (U16BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_SAT_LONGPOS, &tmp32bit);
               if (tmp32bit <= 1800L)
               {
                  sat_ptr->long_pos = (U16BIT)tmp32bit;
               }
               else
               {
                  /* Invalid setting, so force to default! */
                  sat_ptr->long_pos = (U8BIT)0;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_SAT_LONGPOS, (U32BIT)sat_ptr->long_pos);
               }
            }
            else
            {
               STB_AppFreeMemory(sat_ptr);
            }
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_SATELLITE, NULL, dba_rec);
      }

      /* Read all network records */
      dba_rec = DBA_FindRecord(DBA_RECORD_NETWORK, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database network record */
         n_ptr = STB_AppGetMemory(sizeof(ADB_NETWORK_REC));
         if (n_ptr != NULL)
         {
            memset(n_ptr, 0, sizeof(ADB_NETWORK_REC));
            STB_LLAddBlockToEnd(&network_rec_list, (LINK_LIST_PTR_BLK *)n_ptr);

            n_ptr->dba_rec = dba_rec;

            /* Network may have a parent satellite record */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find satellite record matching parent satellite for this network */
               sat_ptr = (ADB_SATELLITE_REC *)STB_LLGetFirstBlock(&satellite_rec_list);
               while (sat_ptr != NULL)
               {
                  if (sat_ptr->dba_rec == parent_rec)
                  {
                     n_ptr->satellite = sat_ptr;
                     break;
                  }
                  sat_ptr = (ADB_SATELLITE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)sat_ptr);
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_NET_ID, &tmp32bit);
            n_ptr->net_id = tmp32bit;

            DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes);
            n_ptr->name_str = DBDEF_MakeString(0, string, num_bytes);

            /* Initialise version change FALSE as should only be TRUE when there is a version change */
            n_ptr->nit_version_changed = FALSE;

            /* Get the nit version */
            DBA_GetFieldValue(dba_rec, DBA_FIELD_VERSION, &tmp32bit);
            n_ptr->nit_version = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_PROFILE_TYPE, &tmp32bit);
            n_ptr->profile_type = tmp32bit;

#ifdef COMMON_INTERFACE
            DBA_GetFieldValue(n_ptr->dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
            n_ptr->cicam_onet_id = tmp32bit;

            DBA_GetFieldValue(n_ptr->dba_rec, DBA_FIELD_PROFILE_CAM_ID, &tmp32bit);
            n_ptr->cicam_identifier = tmp32bit;
#endif
         }
         dba_rec = DBA_FindRecord(DBA_RECORD_NETWORK, NULL, dba_rec);
      }

      /* Read all terrestrial transport records */
      analog_network = NULL;
      dba_rec = DBA_FindRecord(DBA_RECORD_TERR_TRANSPORT, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database transport record */
         t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
         if (t_ptr != NULL)
         {
            memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));
            STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

            t_ptr->dba_rec = dba_rec;

            /* Make link to parent network record */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find network record matching parent network for this transport */
               n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
               while (n_ptr != NULL)
               {
                  if (n_ptr->dba_rec == parent_rec)
                  {
                     t_ptr->network = n_ptr;
                     break;
                  }
                  n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
               }
            }

            // read back frequency and check if analogue or digital (top bit set for analogue)
            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, &tmp32bit);
            if ((tmp32bit & 0x80000000L) != 0)
            {
               /* Analogue terrestrial transport */
               t_ptr->frequency = (tmp32bit & 0x7fffffffL);

               t_ptr->sig_type = SIGNAL_ANALOG;
               t_ptr->tran_id = 0xffff;
               t_ptr->signal_level_at_search = BAD_SIGNAL_STATUS;

               if (analog_network == NULL)
               {
                  analog_network = t_ptr->network;
               }

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
               t_ptr->u.anal.vtype = (E_STB_DP_ANALOG_VIDEO_TYPE)tmp32bit;
#if 0
               DBA_GetFieldValue(dba_rec, DBA_FIELD_TTRAN_FREQ_OFFSET, &tmp32bit);
               t_ptr->u.anal.freq_offset = (S8BIT)tmp32bit;
#endif
            }
            else
            {
               /* Digital terrestrial transport */
               t_ptr->frequency = tmp32bit;
               t_ptr->sig_type = SIGNAL_COFDM;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TTRAN_BWIDTH, &tmp32bit);
               t_ptr->u.terr.bwidth = (E_STB_DP_TBWIDTH)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TTRAN_MODE, &tmp32bit);
               t_ptr->u.terr.tmode = (E_STB_DP_TMODE)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
               t_ptr->tran_id = tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
               t_ptr->orig_net_id = tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_STRENGTH, &tmp32bit);
               t_ptr->signal_level_at_search = (U8BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_QUALITY, &tmp32bit);
               t_ptr->signal_level_at_search = GET_SIGNAL_STATUS(tmp32bit, t_ptr->signal_level_at_search);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TTRAN_TERR_TYPE, &tmp32bit);
               t_ptr->u.terr.terr_type = (E_STB_DP_TTYPE)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TTRAN_PLP_ID, &tmp32bit);
               t_ptr->u.terr.plp_id = (U8BIT)tmp32bit;

               /* Initialise tuning parameters not stored in nvm and requiring non-zero state */
               t_ptr->u.terr.constellation = TUNE_TCONST_UNDEFINED;
               t_ptr->u.terr.hierarchy = TUNE_THIERARCHY_UNDEFINED;
               t_ptr->u.terr.lp_code_rate = TUNE_TCODERATE_UNDEFINED;
               t_ptr->u.terr.hp_code_rate = TUNE_TCODERATE_UNDEFINED;
               t_ptr->u.terr.guard_int = TUNE_TGUARDINT_UNDEFINED;
            }

            /* Initialise version change FALSE as should only be TRUE when there is a version change */
            t_ptr->sdt_version_changed = FALSE;

            /* Get the sdt version */
            DBA_GetFieldValue(dba_rec, DBA_FIELD_VERSION, &tmp32bit);
            t_ptr->sdt_version = (U8BIT)tmp32bit;
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_TERR_TRANSPORT, NULL, dba_rec);
      }

      /* Read all cable transport records */
      dba_rec = DBA_FindRecord(DBA_RECORD_CAB_TRANSPORT, NULL, NULL);
      while (dba_rec != NULL)
      {
         // make app database transport record
         t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
         if (t_ptr != NULL)
         {
            memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));
            STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

            t_ptr->dba_rec = dba_rec;

            /* Make link to parent network record */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find network record matching parent network for this transport */
               n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
               while (n_ptr != NULL)
               {
                  if (n_ptr->dba_rec == parent_rec)
                  {
                     t_ptr->network = n_ptr;
                     break;
                  }
                  n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
               }
            }

            t_ptr->sig_type = SIGNAL_QAM;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, &tmp32bit);
            t_ptr->frequency = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_CTRAN_MODE, &tmp32bit);
            t_ptr->u.cab.cmode = (E_STB_DP_TMODE)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SRATE, &tmp32bit);
            t_ptr->u.cab.symbol_rate = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
            t_ptr->tran_id = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
            t_ptr->orig_net_id = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_STRENGTH, &tmp32bit);
            t_ptr->signal_level_at_search = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_QUALITY, &tmp32bit);
            t_ptr->signal_level_at_search = GET_SIGNAL_STATUS(tmp32bit, t_ptr->signal_level_at_search);

            /* Initialise version change FALSE as should only be TRUE when there is a version change */
            t_ptr->sdt_version_changed = FALSE;

            /* Get the sdt version */
            DBA_GetFieldValue(dba_rec, DBA_FIELD_VERSION, &tmp32bit);
            t_ptr->sdt_version = (U8BIT)tmp32bit;
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_CAB_TRANSPORT, NULL, dba_rec);
      }

      /* Read all satellite transport records */
      dba_rec = DBA_FindRecord(DBA_RECORD_SAT_TRANSPORT, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database transport record */
         t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
         if (t_ptr != NULL)
         {
            memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));
            STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

            t_ptr->dba_rec = dba_rec;

            /* Make link to parent network */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               // find network record matching parent network for this transport
               n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
               while (n_ptr != NULL)
               {
                  if (n_ptr->dba_rec == parent_rec)
                  {
                     t_ptr->network = n_ptr;
                     break;
                  }
                  n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
               }
            }
            else
            {
               DBG_DBDEF("Parent rec not found");
            }

            t_ptr->sig_type = SIGNAL_QPSK;

            /* Set the transponder id */
            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
            t_ptr->tran_id = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
            t_ptr->orig_net_id = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, &tmp32bit);
            t_ptr->frequency = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_STRAN_POL, &tmp32bit);
            switch ((E_STB_DP_POLARITY)tmp32bit)
            {
               case POLARITY_HORIZONTAL:
               case POLARITY_VERTICAL:
               case POLARITY_LEFT:
               case POLARITY_RIGHT:
               {
                  t_ptr->u.sat.polarity = (E_STB_DP_POLARITY)tmp32bit;
                  break;
               }
               default:
               {
                  /* Invalid setting, so force to default! */
                  t_ptr->u.sat.polarity = POLARITY_HORIZONTAL;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_STRAN_POL,
                     (U32BIT)t_ptr->u.sat.polarity);
                  break;
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SRATE, &tmp32bit);
            t_ptr->u.sat.symbol_rate = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_STRAN_FEC, &tmp32bit);
            switch ((E_STB_DP_FEC)tmp32bit)
            {
               case FEC_AUTOMATIC:
               case FEC_1_2:
               case FEC_2_3:
               case FEC_3_4:
               case FEC_5_6:
               case FEC_7_8:
               case FEC_1_4:
               case FEC_1_3:
               case FEC_2_5:
               case FEC_8_9:
               case FEC_9_10:
               case FEC_3_5:
               case FEC_4_5:
               {
                  t_ptr->u.sat.fec_code = (E_STB_DP_FEC)tmp32bit;
                  break;
               }
               default:
               {
                  /* Invalid setting, so force to default! */
                  t_ptr->u.sat.fec_code = FEC_AUTOMATIC;
                  DBA_SetFieldValue(dba_rec, DBA_FIELD_STRAN_FEC,
                     (U32BIT)t_ptr->u.sat.fec_code);
                  break;
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_STRAN_DVBS2, &tmp32bit);
            t_ptr->u.sat.dvb_s2 = ((tmp32bit != 0) ? TRUE : FALSE);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_STRAN_MODULATION, &tmp32bit);
            t_ptr->u.sat.modulation = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_STRENGTH, &tmp32bit);
            t_ptr->signal_level_at_search = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TRAN_SIGNAL_QUALITY, &tmp32bit);
            t_ptr->signal_level_at_search = GET_SIGNAL_STATUS(tmp32bit, t_ptr->signal_level_at_search);

            /* Get the sdt version */
            DBA_GetFieldValue(dba_rec, DBA_FIELD_VERSION, &tmp32bit);
            t_ptr->sdt_version = (U8BIT)tmp32bit;
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_SAT_TRANSPORT, NULL, dba_rec);
      }

      /* Read service records */
      num_analog_channels = 0;
      dba_rec = DBA_FindRecord(DBA_RECORD_SERVICE, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database service record */
         s_ptr = STB_AppGetMemory(sizeof(ADB_SERVICE_REC));
         if (s_ptr != NULL)
         {
            memset(s_ptr, 0, sizeof(ADB_SERVICE_REC));
            STB_LLAddBlockToEnd(&service_rec_list, (LINK_LIST_PTR_BLK *)s_ptr);

            s_ptr->dba_rec = dba_rec;

            /* Make link to parent transport */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               // find transport record matching parent transport for this service
               t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
               while (t_ptr != NULL)
               {
                  if (t_ptr->dba_rec == parent_rec)
                  {
                     s_ptr->transport = t_ptr;
                     break;
                  }
                  t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)t_ptr);
               }
            }

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_ID, &tmp32bit);
            s_ptr->serv_id = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_LCN, &tmp32bit);
            s_ptr->allocated_lcn = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_REQ_LCN, &tmp32bit);
            s_ptr->serv_lcn = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_TYPE, &tmp32bit);
            s_ptr->serv_type = tmp32bit;

            if (s_ptr->serv_type == ADB_SERVICE_TYPE_ANALOG)
            {
               num_analog_channels++;
            }

            DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes);
            s_ptr->name_str = DBDEF_MakeString(0, string, num_bytes);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_FAV_GROUPS, &tmp32bit);
            s_ptr->fav_groups = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_LOCKED, &tmp32bit);
            s_ptr->locked = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_HIDDEN, &tmp32bit);
            s_ptr->hidden = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_SELECTABLE, &tmp32bit);
            s_ptr->selectable = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_SCHED_DISABLED, &tmp32bit);
            s_ptr->sched_disabled = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_NOWNEXT_DISABLED, &tmp32bit);
            s_ptr->now_next_disabled = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_FREESAT_ID, &tmp32bit);
            s_ptr->freesat_id = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_REGION_ID, &tmp32bit);
            s_ptr->region_id = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_LCN_EDITABLE, &tmp32bit);
            s_ptr->lcn_editable = (BOOLEAN)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERV_DELETED, &tmp32bit);
            s_ptr->deleted = (BOOLEAN)tmp32bit;

            /* Mark all service records as found */
            s_ptr->found = TRUE;
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_SERVICE, NULL, dba_rec);
      }

      /* Read CRID records */
      dba_rec = DBA_FindRecord(DBA_RECORD_CRID, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Make app database crid record */
         c_ptr = STB_AppGetMemory(sizeof(ADB_CRID_REC));
         if (c_ptr != NULL)
         {
            memset(c_ptr, 0, sizeof(ADB_CRID_REC));
            STB_LLAddBlockToEnd(&crid_rec_list, (LINK_LIST_PTR_BLK *)c_ptr);

            c_ptr->dba_rec = dba_rec;

            DBA_GetFieldString(dba_rec, DBA_FIELD_TIMER_CRID, &string, &num_bytes);
            c_ptr->crid_str = DBDEF_MakeString(0, string, num_bytes);

            DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes);
            c_ptr->name_str = DBDEF_MakeString(0, string, num_bytes);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_CRID_SERIES, &tmp32bit);
            c_ptr->series_flag = (tmp32bit != 0);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_CRID_RECOMMENDED, &tmp32bit);
            c_ptr->recommended_flag = (tmp32bit != 0);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_CRID_DO_NOT_DELETE, &tmp32bit);
            c_ptr->do_not_delete = (tmp32bit != 0);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_CRID_EIT_DATE, &tmp32bit);
            c_ptr->eit_date = (U16BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_STARTTIME, &tmp32bit);
            c_ptr->date_time = tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_SERVICE_ID, &tmp32bit);
            c_ptr->serv_id = (U16BIT)tmp32bit;
         }
         dba_rec = DBA_FindRecord(DBA_RECORD_CRID, NULL, dba_rec);
      }

      /* Read favourite lists */
      dba_rec = DBA_FindRecord(DBA_RECORD_FAV_LIST, NULL, NULL);
      while (dba_rec != NULL)
      {
         /* Create an app database record for this list */
         fl_ptr = STB_AppGetMemory(sizeof(ADB_FAVLIST_REC));
         if (fl_ptr != NULL)
         {
            memset(fl_ptr, 0, sizeof(ADB_FAVLIST_REC));
            STB_LLAddBlockToEnd(&favlist_list, (LINK_LIST_PTR_BLK *)fl_ptr);

            fl_ptr->dba_rec = dba_rec;

            /* Init the list of services in this favourite list */
            STB_LLInitialiseHeader(&fl_ptr->serv_list);

            DBA_GetFieldValue(dba_rec, DBA_FIELD_FAVLIST_ID, &tmp32bit);
            fl_ptr->list_id = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_FAVLIST_INDEX, &tmp32bit);
            fl_ptr->index = (U8BIT)tmp32bit;

            DBA_GetFieldValue(dba_rec, DBA_FIELD_FAVLIST_USER_DATA, &tmp32bit);
            fl_ptr->user_data = tmp32bit;

            DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes);
            fl_ptr->name = DBDEF_MakeString(0, string, num_bytes);
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_FAV_LIST, NULL, dba_rec);
      }

      /* Read the services in all favourite lists */
      dba_rec = DBA_FindRecord(DBA_RECORD_FAV_SERV, NULL, NULL);
      while (dba_rec != NULL)
      {
         fs_ptr = STB_AppGetMemory(sizeof(ADB_FAVSERV_REC));
         if (fs_ptr != NULL)
         {
            memset(fs_ptr, 0, sizeof(ADB_FAVSERV_REC));

            /* Create the link to the actual service */
            parent_rec = DBA_GetRecordParent(dba_rec);
            if (parent_rec != NULL)
            {
               /* Find service record matching parent service */
               s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
               while (s_ptr != NULL)
               {
                  if (s_ptr->dba_rec == parent_rec)
                  {
                     fs_ptr->serv_ptr = s_ptr;
                     break;
                  }
                  s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
               }
            }

            /* If the service hasn't been found then there's no point reading the remaining fields */
            if (fs_ptr->serv_ptr != NULL)
            {
               fs_ptr->dba_rec = dba_rec;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_FAVLIST_ID, &tmp32bit);
               fs_ptr->list_id = (U8BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_FAVLIST_INDEX, &tmp32bit);
               fs_ptr->index = (U16BIT)tmp32bit;

               /* Find the favourite list this service is in */
               fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
               while (fl_ptr != NULL)
               {
                  if (fl_ptr->list_id == fs_ptr->list_id)
                  {
                     /* Found the favourite list, add the service to it.
                      * The list will be sorted into the correct order at the end */
                     STB_LLAddBlockToEnd(&fl_ptr->serv_list, (LINK_LIST_PTR_BLK *)fs_ptr);
                     break;
                  }

                  fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
               }

               if (fl_ptr == NULL)
               {
                  /* The favourite list the service is in wasn't found,
                   * so no point keeping the service */
                  STB_AppFreeMemory(fs_ptr);
               }
            }
            else
            {
               STB_AppFreeMemory(fs_ptr);
            }
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_FAV_SERV, NULL, dba_rec);
      }

      /* Now sort the services in each favourite list into the correct order
       * as defined by the service's index.
       * The index of each list is also set to ensure they're consecutive */
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
      for (fl_index = 0; fl_ptr != NULL; fl_index++)
      {
         if (fl_ptr->index != fl_index)
         {
            fl_ptr->index = fl_index;
            DBA_SetFieldValue(fl_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)fl_index);
         }

         STB_LLSort(&fl_ptr->serv_list, FavServiceSortCompare);

         /* Now set the index of the services in this list to ensure they're consecutive */
         fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fl_ptr->serv_list);
         for (fs_index = 0; fs_ptr != NULL; fs_index++)
         {
            if (fs_ptr->index != fs_index)
            {
               fs_ptr->index = fs_index;
               DBA_SetFieldValue(fl_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)fs_index);
            }

            fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
         }

         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
      }

      /* Read timers */
      dba_rec = DBA_FindRecord(DBA_RECORD_TIMER, NULL, NULL);
      while (dba_rec != NULL)
      {
         DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_TYPE, &tmp32bit);

         if ((tmp32bit == TIMER_TYPE_ALARM) || (tmp32bit == TIMER_TYPE_SLEEP) ||
             (tmp32bit == TIMER_TYPE_PVR_RECORD) || (tmp32bit == TIMER_TYPE_PRIVATE))
         {
            /* Make app database service record */
            timer = (ADB_TIMER_REC *)STB_AppGetMemory(sizeof(ADB_TIMER_REC));
            if (timer != NULL)
            {
               memset(timer, 0, sizeof(ADB_TIMER_REC));
               STB_LLAddBlockToEnd(&timer_list, (LINK_LIST_PTR_BLK *)timer);

               timer->dba_rec = dba_rec;
               timer->type = (E_TIMER_TYPE)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_HANDLE, &tmp32bit);
               timer->handle = tmp32bit;

               /* Set the base for the next timer handle so it's greater than
                * the biggest timer handle currently in use */
               if (timer->handle > last_timer_handle)
               {
                  last_timer_handle = timer->handle;
               }

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_STARTTIME, &tmp32bit);
               timer->start_time = tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_FREQUENCY, &tmp32bit);
               timer->frequency = (E_TIMER_FREQ)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_MISSED, &tmp32bit);
               if (tmp32bit == 1)
               {
                  timer->missed = TRUE;
               }
               else
               {
                  timer->missed = FALSE;
               }

               if (DBA_GetFieldString(dba_rec, DBA_FIELD_REC_NAME, &string, &num_bytes) &&
                  (num_bytes > 0))
               {
                  memcpy(timer->name, &string[0], num_bytes);
               }

               switch (timer->type)
               {
                  case TIMER_TYPE_ALARM:
                  {
                     DBA_GetFieldValue(dba_rec, DBA_FIELD_SERVICE_ID, &tmp32bit);
                     timer->u.alarm.service_id = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
                     timer->u.alarm.transport_id = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
                     timer->u.alarm.orig_net_id = (U16BIT)tmp32bit;

                     if (timer->u.alarm.service_id != ADB_INVALID_DVB_ID)
                     {
                        timer->u.alarm.change_service = TRUE;
                     }
                     else
                     {
                        timer->u.alarm.change_service = FALSE;
                     }

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_RAMPVOLUME, &tmp32bit);
                     if (tmp32bit == 1)
                     {
                        timer->u.alarm.ramp_volume = TRUE;
                     }
                     else
                     {
                        timer->u.alarm.ramp_volume = FALSE;
                     }
                     break;
                  }

                  case TIMER_TYPE_PVR_RECORD:
                  {
                     /* Initialise path to INVALID_RES_ID, the correct path will be assigned when
                      * the recording starts*/
                     timer->u.record.path = INVALID_RES_ID;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_DURATION, &tmp32bit);
                     timer->u.record.duration = tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_EVENT_TRIGGERED, &tmp32bit);
                     if (tmp32bit == 1)
                     {
                        timer->u.record.event_triggered = TRUE;

                        DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_EVENTID, &tmp32bit);
                        timer->u.record.event_id = (U16BIT)tmp32bit;
                     }
                     else
                     {
                        timer->u.record.event_triggered = FALSE;
                     }

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_SERVICE_ID, &tmp32bit);
                     timer->u.record.service_id = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TRANSPORT_ID, &tmp32bit);
                     timer->u.record.transport_id = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_ORIG_NET_ID, &tmp32bit);
                     timer->u.record.orig_net_id = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_DISKID, &tmp32bit);
                     timer->u.record.disk_id = (U16BIT)tmp32bit;

                     if (DBA_GetFieldString(dba_rec, DBA_FIELD_TIMER_CRID, &string, &num_bytes) &&
                         (num_bytes > 0))
                     {
                        memcpy(timer->u.record.prog_crid, &string[0], num_bytes);
                     }

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_CRID_RECOMMENDED, &tmp32bit);
                     if (tmp32bit == 1)
                     {
                        timer->u.record.recommendation = TRUE;
                     }
                     else
                     {
                        timer->u.record.recommendation = FALSE;
                     }

                     if (DBA_GetFieldString(dba_rec, DBA_FIELD_TIMER_OTHERCRID, &string, &num_bytes) &&
                         (num_bytes > 0))
                     {
                        memcpy(timer->u.record.other_crid, &string[0], num_bytes);
                     }

                     if (DBA_GetFieldString(dba_rec, DBA_FIELD_TIMER_ADDITIONAL_INFO, &string, &num_bytes) &&
                         (num_bytes > 0))
                     {
                        memcpy(timer->u.record.additional_info, string, num_bytes);
                     }

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_START_PADDING, &tmp32bit);
                     timer->u.record.start_padding = tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_END_PADDING, &tmp32bit);
                     timer->u.record.end_padding = tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_NOTIFY_TIME, &tmp32bit);
                     timer->u.record.notify_time = (U16BIT)tmp32bit;

                     DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_DO_NOT_DELETE, &tmp32bit);
                     if (tmp32bit == 1)
                     {
                        timer->u.record.do_not_delete = TRUE;
                     }
                     else
                     {
                        timer->u.record.do_not_delete = FALSE;
                     }
                     break;
                  }

                  default:
                     break;
               }
            }
         }

         dba_rec = DBA_FindRecord(DBA_RECORD_TIMER, NULL, dba_rec);
      }

#ifdef COMMON_INTERFACE
      /* Read CICAM timers */
      dba_rec = DBA_FindRecord(DBA_RECORD_CICAM_TIMER, NULL, NULL);
      while (dba_rec != NULL)
      {
         cicam_timer = (ADB_CICAM_TIMER_REC *)STB_AppGetMemory(sizeof(ADB_CICAM_TIMER_REC));
         DBA_GetFieldValue(dba_rec, DBA_FIELD_TIMER_HANDLE, &tmp32bit);
         cicam_timer->timer_handle = tmp32bit;
         DBA_GetFieldValue(dba_rec, DBA_FIELD_PROFILE_CAM_ID, &tmp32bit);
         cicam_timer->cicam_identifier = tmp32bit;
         STB_LLAddBlockToEnd(&cicam_timer_list, (LINK_LIST_PTR_BLK *)cicam_timer);

         dba_rec = DBA_FindRecord(DBA_RECORD_CICAM_TIMER, NULL, dba_rec);
      }
#endif

#if 0
      /* Read stored events */
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
      while (s_ptr != NULL)
      {
         dba_rec = DBA_FindRecord(DBA_RECORD_EVENT, s_ptr->dba_rec, NULL);
         while (dba_rec != NULL)
         {
            e_ptr = (ADB_EVENT_REC *)STB_AppGetMemory(sizeof(ADB_EVENT_REC));
            if (e_ptr != NULL)
            {
               memset(e_ptr, 0, sizeof(ADB_EVENT_REC));

               e_ptr->dba_rec = dba_rec;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_ID, &tmp32bit);
               e_ptr->event_id = (U16BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_STARTTIME, &tmp32bit);
               e_ptr->start = (U32DHMS)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_DURATION, &tmp32bit);
               e_ptr->duration = (U32DHMS)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_VERSION, &tmp32bit);
               e_ptr->version = (U8BIT)tmp32bit;
#if 0
               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_AGE_RATING, &tmp32bit);
               e_ptr->parental_age_rating = (U8BIT)tmp32bit;

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_SCRAMBLED, &tmp32bit);
               e_ptr->scrambled = (tmp32bit == 0 ? FALSE : TRUE);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_SUBTITLES, &tmp32bit);
               e_ptr->subtitles_avail = (tmp32bit == 0 ? FALSE : TRUE);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_AUDIO_DESC, &tmp32bit);
               e_ptr->audio_desc = (tmp32bit == 0 ? FALSE : TRUE);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_FREE_TO_AIR, &tmp32bit);
               e_ptr->has_fta_desc = (tmp32bit == 0 ? FALSE : TRUE);

               DBA_GetFieldValue(dba_rec, DBA_FIELD_EVENT_DO_NOT_SCRAMBLE, &tmp32bit);
               e_ptr->do_not_scramble = (tmp32bit == 0 ? FALSE : TRUE);

               for (index = 0; index < ACFG_NUM_DB_LANGUAGES; index++)
               {
                  if ((lang_code = ACFG_ConvertLangIdToCode(index)) != 0)
                  {
                     if (e_ptr->name_array[index] != NULL)
                     {
                        if (DBA_GetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_NAME,
                               lang_code, &string, &num_bytes))
                        {
                           e_ptr->name_array[index] = DBDEF_MakeString(lang_code, string, num_bytes);
                        }
                     }

                     if (e_ptr->desc_array[index] != NULL)
                     {
                        if (DBA_GetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_DESCRIPTION,
                               lang_code, &string, &num_bytes))
                        {
                           e_ptr->desc_array[index] = DBDEF_MakeString(lang_code, string, num_bytes);
                        }
                     }

                     if (e_ptr->extended_info[index].event_text != NULL)
                     {
                        if (DBA_GetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_EXTENDED_DESC,
                               lang_code, &string, &num_bytes))
                        {
                           e_ptr->extended_info[index].event_text = DBDEF_MakeString(lang_code, string, num_bytes);
                        }
                     }

                     if (e_ptr->guidance[index] != NULL)
                     {
                        if (DBA_GetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_GUIDANCE,
                               lang_code, &string, &num_bytes))
                        {
                           e_ptr->guidance[index] = DBDEF_MakeString(lang_code, string, num_bytes);
                        }
                     }
                  }
               }

               if (DBA_GetFieldData(dba_rec, DBA_FIELD_EVENT_CONTENT_DATA, &data, &data_len) &&
                   (data_len != 0))
               {
                  if ((e_ptr->content_data = STB_AppGetMemory(data_len)) != NULL)
                  {
                     memcpy(e_ptr->content_data, data, data_len);
                  }
               }
#endif
               /* Find where the event should be added into the service's schedule */
               InsertEventInSchedule(s_ptr, e_ptr);
            }

            dba_rec = DBA_FindRecord(DBA_RECORD_EVENT, s_ptr->dba_rec, dba_rec);
         }

         s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
      }
#endif
   }

   FUNCTION_FINISH(DBDEF_LoadDatabase);

   return(retval);
}

/**
 *

 *
 * @brief   Deletes all records in the database
 *

 *

 *
 */
void DBDEF_DeleteAllRecords(void)
{
   LINK_LIST_PTR_BLK *tmp_ptr;
   LINK_LIST_PTR_BLK *next_ptr;

   FUNCTION_START(DBDEF_DeleteAllRecords);

   /* Delete service records */
   tmp_ptr = STB_LLGetFirstBlock(&service_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteServiceRec((ADB_SERVICE_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete transport records */
   tmp_ptr = STB_LLGetFirstBlock(&transport_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteTransportRec((ADB_TRANSPORT_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete network records */
   tmp_ptr = STB_LLGetFirstBlock(&network_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DeleteNetworkRec((ADB_NETWORK_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete satellite records */
   tmp_ptr = STB_LLGetFirstBlock(&satellite_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DeleteSatelliteRec((ADB_SATELLITE_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete LNB band records */
   tmp_ptr = STB_LLGetFirstBlock(&lnb_band_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteLNBBandRec((ADB_LNB_BAND_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete LNB records */
   tmp_ptr = STB_LLGetFirstBlock(&lnb_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DeleteLNBRec((ADB_LNB_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete favourite list records */
   tmp_ptr = STB_LLGetFirstBlock(&favlist_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteFavouriteList((ADB_FAVLIST_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete crid records */
   tmp_ptr = STB_LLGetFirstBlock(&crid_rec_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteCridRecord((ADB_CRID_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Delete timer records */
   tmp_ptr = STB_LLGetFirstBlock(&timer_list);
   while (tmp_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(tmp_ptr);
      DBDEF_DeleteTimerRec((ADB_TIMER_REC *)tmp_ptr);
      tmp_ptr = next_ptr;
   }

   /* Clear analog details */
   analog_network = NULL;
   num_analog_channels = 0;

   FUNCTION_FINISH(DBDEF_DeleteAllRecords);
}

/*!**************************************************************************
 * @brief   Deletes all network, transport and services records for the given tuner type
 * @param   tuner_type tuner type
 * @param   satellite only used when tuner_type == SIGNAL_QPSK to specify the satellite
 *                      from which services should be deleted. If tuner_type is SIGNAL_QPSK
 *                      and satellite is NULL then all DVB-S services will be deleted.
 ****************************************************************************/
void DBDEF_DeleteRecordsForTunerType(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_NETWORK_REC *n_ptr;
   void *next_ptr;
   BOOLEAN delete_network;

   FUNCTION_START(DBDEF_DeleteRecordsForTunerType);

   /* Delete service records */
   s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   while (s_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(&s_ptr->ptrs);

      if (DBDEF_ServiceForTunerType(s_ptr, tuner_type, satellite))
      {
         DBDEF_DeleteServiceRec(s_ptr);
      }

      s_ptr = (ADB_SERVICE_REC *)next_ptr;
   }

   /* Delete transport records */
   t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   while (t_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(&t_ptr->ptrs);

      if (DBDEF_TransportForTunerType(t_ptr, tuner_type, satellite))
      {
         DBDEF_DeleteTransportRec(t_ptr);
      }

      t_ptr = (ADB_TRANSPORT_REC *)next_ptr;
   }

   /* Delete network records */
   n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
   while (n_ptr != NULL)
   {
      next_ptr = STB_LLGetNextBlock(&n_ptr->ptrs);

      if (DBDEF_NetworkInProfile(n_ptr))
      {
         /* See if this network is still referenced by any transports */
         delete_network = TRUE;
         t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
         while ((t_ptr != NULL) && delete_network)
         {
            if (t_ptr->network == n_ptr)
            {
               /* Transport for this network found, so it shouldn't be deleted */
               delete_network = FALSE;
            }

            t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock(&t_ptr->ptrs);
         }

         if (delete_network)
         {
            DeleteNetworkRec(n_ptr);
         }
      }

      n_ptr = (ADB_NETWORK_REC *)next_ptr;
   }

   if (tuner_type == SIGNAL_ANALOG)
   {
      analog_network = NULL;
      num_analog_channels = 0;
   }

   FUNCTION_FINISH(DBDEF_DeleteRecordsForTunerType);
}

#ifdef DEBUG_PRINT_DATABASE
/**
 *

 *
 * @brief   Prints all records in the database
 *

 *

 *
 */
void DBDEF_PrintAllRecords(void)
{
   void *list_ptr;

   FUNCTION_START(DBDEF_PrintAllRecords);

   STB_SPDebugWrite("Database entries:");

   // print LNB records
   STB_SPDebugWrite("\tLNB records:");
   list_ptr = DBA_FirstLnb();
   while (list_ptr != NULL)
   {
      STB_SPDebugWrite("\t0x%x", list_ptr);
      PrintLNBRec((ADB_LNB_REC *)list_ptr);
      list_ptr = DBA_FindNext(list_ptr);
   }

   // print satellite records
   STB_SPDebugWrite("\tSatellite records:");
   list_ptr = DBA_FirstSatellite();
   while (list_ptr != NULL)
   {
      STB_SPDebugWrite("\t0x%x", list_ptr);
      PrintSatelliteRec((ADB_SATELLITE_REC *)list_ptr);
      list_ptr = DBA_FindNext(list_ptr);
   }

   // print network records
   STB_SPDebugWrite("\tNetwork records:");
   list_ptr = DBA_FindFirst(DBA_REC_NETWORK);
   while (list_ptr != NULL)
   {
      STB_SPDebugWrite("\t0x%x", list_ptr);
      PrintNetworkRec((ADB_NETWORK_REC *)list_ptr);
      list_ptr = DBA_FindNext(list_ptr);
   }

   // print transport records
   STB_SPDebugWrite("\tTransport records:");
   list_ptr = DBA_FindFirst(DBA_REC_TRANSPORT);
   while (list_ptr != NULL)
   {
      STB_SPDebugWrite("\t0x%x", list_ptr);
      PrintTransportRec((ADB_TRANSPORT_REC *)list_ptr);
      list_ptr = DBA_FindNext(list_ptr);
   }

   // delete service records
   STB_SPDebugWrite("\tService records:");
   list_ptr = DBA_FindFirst(DBA_REC_SERVICE);
   while (list_ptr != NULL)
   {
      STB_SPDebugWrite("\t0x%x", list_ptr);
      PrintServiceRec((ADB_SERVICE_REC *)list_ptr);
      list_ptr = DBA_FindNext(list_ptr);
   }

   STB_SPDebugWrite("End of database");

   FUNCTION_FINISH(DBDEF_PrintAllRecords);
}

#endif

/**
 *

 *
 * @brief   Deletes specified service record
 *
 * @param   s_ptr - pointer to service record
 *

 *
 */
void DBDEF_DeleteServiceRec(ADB_SERVICE_REC *s_ptr)
{
   U8BIT i, j;
   ADB_FAVLIST_REC *fl_ptr;
   ADB_FAVSERV_REC *fs_ptr;
   ADB_SERVICE_REC *s2_ptr;
   BOOLEAN events_shared;

   FUNCTION_START(DBDEF_DeleteServiceRec);

   if (s_ptr != NULL)
   {
      /* Remove the service from all favourite lists */
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
      while (fl_ptr != NULL)
      {
         if ((fs_ptr = DBDEF_FindServiceInFavouriteList(fl_ptr, s_ptr)) != NULL)
         {
            DBDEF_DeleteServiceFromFavouriteList(fl_ptr, fs_ptr);
         }

         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
      }

      STB_LLRemoveBlock(&s_ptr->ptrs);

      for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
      {
         if (s_ptr->provider_array[i] != NULL)
         {
            STB_AppFreeMemory(s_ptr->provider_array[i]);
         }
         for (j = 0; j < ADB_NUM_SERV_NAME_IDS; j++)
         {
            if (s_ptr->name_array[i][j] != NULL)
            {
               STB_AppFreeMemory(s_ptr->name_array[i][j]);
            }
         }
         if (s_ptr->guidance[i].text != NULL)
         {
            DBDEF_ReleaseString(s_ptr->guidance[i].text);
         }
      }

      if (s_ptr->provider_str != NULL)
      {
         STB_AppFreeMemory(s_ptr->provider_str);
      }

      DBDEF_DeleteStreamList(s_ptr->stream_list);
      DBDEF_DeleteAltServList(s_ptr->alt_serv_list);

      /* The now, next and schedule events may be shared with another service,
       * so first check there isn't another service using this schedule list */
      events_shared = FALSE;
      s2_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
      while (s2_ptr != NULL)
      {
         if ((s2_ptr != s_ptr) &&
             (((s2_ptr->event_schedule != NULL) && (s2_ptr->event_schedule == s_ptr->event_schedule)) ||
              ((s2_ptr->now_event != NULL) && (s2_ptr->now_event == s_ptr->now_event)) ||
              ((s2_ptr->next_event != NULL) && (s2_ptr->next_event == s_ptr->next_event))))
         {
            events_shared = TRUE;
            break;
         }

         s2_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s2_ptr);
      }

      if (!events_shared)
      {
         DBDEF_DeleteEventList(s_ptr->now_event);
         DBDEF_DeleteEventList(s_ptr->next_event);
         DBDEF_DeleteEventList(s_ptr->event_schedule);
      }

      DBDEF_DeleteRCTLinks(s_ptr->rct_link_list);
      DBDEF_DeleteImageIcons(s_ptr->icon_list);

      if (s_ptr->serv_type == ADB_SERVICE_TYPE_ANALOG)
      {
         num_analog_channels--;
      }
#ifdef COMMON_INTERFACE
      if (s_ptr->serv_type == ADB_SERVICE_TYPE_VIRTUAL)
      {
         /* delete associated virtual transport rec */
         DBDEF_DeleteTransportRec(s_ptr->transport);
      }
#endif

      DBA_DestroyRecord(s_ptr->dba_rec);

      STB_AppFreeMemory(s_ptr);
   }

   FUNCTION_FINISH(DBDEF_DeleteServiceRec);
}

/*!**************************************************************************
 * @brief   Returns the number of stream records in the given list
 * @param   slist list of stream recs
 * @return  Number of stream recs
 ****************************************************************************/
U16BIT DBDEF_NumStreamsInList(ADB_STREAM_REC *slist)
{
   U16BIT num;

   FUNCTION_START(DBDEF_NumStreamsInList);

   for (num = 0; slist != NULL; num++)
   {
      slist = slist->next;
   }

   FUNCTION_FINISH(DBDEF_NumStreamsInList);

   return(num);
}

/*!**************************************************************************
 * @brief   Creates a copy of a stream list, allocating new memory as required
 * @param   slist list of stream recs to be copied
 * @return  copied stream list
 ****************************************************************************/
ADB_STREAM_REC* DBDEF_CopyStreamList(ADB_STREAM_REC *slist)
{
   ADB_STREAM_REC *new_list;
   ADB_STREAM_REC *slptr;
   ADB_STREAM_REC *srec;

   FUNCTION_START(DBDEF_CopyStreamList);

   new_list = NULL;

   if (slist != NULL)
   {
      slptr = slist;
      while (slptr != NULL)
      {
         if ((srec = STB_AppGetMemory(sizeof(ADB_STREAM_REC))) != NULL)
         {
            memcpy(srec, slptr, sizeof(ADB_STREAM_REC));

            if (srec->num_tags > 0)
            {
               if ((srec->tag_array = STB_AppGetMemory(srec->num_tags)) != NULL)
               {
                  memcpy(srec->tag_array, slptr->tag_array, srec->num_tags);
               }
               else
               {
                  srec->num_tags = 0;
               }
            }
            else
            {
               srec->tag_array = NULL;
            }

            /* Prepend the new stream rec to the list */
            srec->next = new_list;
            new_list = srec;
         }

         slptr = slptr->next;
      }
   }

   FUNCTION_FINISH(DBDEF_CopyStreamList);

   return(new_list);
}

/**
 *

 *
 * @brief   Deletes all records in a service stream list
 *
 * @param   slist - the first stream record in the stream list
 *

 *
 */
void DBDEF_DeleteStreamList(ADB_STREAM_REC *slist)
{
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *next_rec;

   stream_rec = slist;
   while (stream_rec != NULL)
   {
      next_rec = stream_rec->next;

      if (stream_rec->tag_array != NULL)
      {
         STB_AppFreeMemory(stream_rec->tag_array);
      }

      STB_AppFreeMemory(stream_rec);
      stream_rec = next_rec;
   }
}

/**
 * @brief   Find an event for a service from its event_id
 * @param   s_ptr service to be searched
 * @param   event_id ID of the event to be found
 * @return  pointer to the ADB_EVENT_REC, or NULL if not found
 */
ADB_EVENT_REC* DBDEF_FindScheduleEventById(ADB_SERVICE_REC *s_ptr, U16BIT event_id)
{
   ADB_EVENT_REC *e_ptr;

   FUNCTION_START(DBDEF_FindScheduleEventById);

   if ((s_ptr->now_event != NULL) && (s_ptr->now_event->event_id == event_id))
   {
      e_ptr = s_ptr->now_event;
   }
   else if ((s_ptr->next_event != NULL) && (s_ptr->next_event->event_id == event_id))
   {
      e_ptr = s_ptr->next_event;
   }
   else
   {
      e_ptr = s_ptr->event_schedule;
      while ((e_ptr != NULL) && (e_ptr->event_id != event_id))
      {
         e_ptr = e_ptr->next;
      }
   }

   FUNCTION_FINISH(DBDEF_FindScheduleEventById);

   return(e_ptr);
}

/**
 * @brief   Searches a descriptor list for the first descriptor with the given descriptor tag.
 * @param   start_desc descriptor where the search should start
 * @param   desc_tag descriptor tag to be found
 * @param   private_data_specifier the value of the private data specifier that must precede the
 *                                 descriptor being searched for. Use 0 if it isn't required.
 * @return  pointer to the descriptor, or NULL if not found
 */
ADB_EVENT_DESC* DBDEF_FindEventDescriptor(ADB_EVENT_DESC *start_desc, U8BIT desc_tag,
   U32BIT private_data_specifier)
{
   ADB_EVENT_DESC *desc_ptr;
   U32BIT priv_data_code;

   FUNCTION_START(DBDEF_FindEventDescriptor);

   priv_data_code = 0;
   desc_ptr = start_desc;

   while (desc_ptr != NULL)
   {
      if (private_data_specifier == 0)
      {
         if ((desc_ptr->desc_data[0] == desc_tag))
         {
            /* Found the descriptor */
            break;
         }
      }
      else
      {
         /* Descriptor to be found must be preceded by the given private data specifier,
          * so check for any private data specifier descriptors so the value preceding
          * the descriptor to be found can be checked */
         if (desc_ptr->desc_data[0] == PRIV_DATA_SPEC_DTAG)
         {
            priv_data_code = (desc_ptr->desc_data[2] << 24) | (desc_ptr->desc_data[3] << 16) |
               (desc_ptr->desc_data[4] << 8) | desc_ptr->desc_data[5];
         }
         else if ((desc_ptr->desc_data[0] == desc_tag) && (priv_data_code == private_data_specifier))
         {
            /* Found the descriptor */
            break;
         }
      }

      desc_ptr = desc_ptr->next;
   }

   FUNCTION_FINISH(DBDEF_FindEventDescriptor);

   return(desc_ptr);
}

/**
 * @brief   Returns the name of the event as a UTF-8 string in the currently selected language.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventName(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc, *und_event_desc;
   U8BIT *text_lang_ids;
   U32BIT lang_code, und_lang_code;
   U8BIT lang_id, i, und_lang_id;
   U8BIT *dptr;
   SI_STRING_DESC *temp_str;
   U16BIT str_len;
   U8BIT *name_str;

   FUNCTION_START(DBDEF_GetEventName);

   ASSERT(e_ptr != NULL);

   name_str = NULL;

   text_lang_ids = DBDEF_GetTextLang();
   if (text_lang_ids != NULL)
   {
      event_desc = e_ptr->desc_list_head;

      und_lang_id = ACFG_INVALID_DB_LANG;
      und_lang_code = ACFG_GetUndefinedLanguageBehaviour();
      if (und_lang_code != ADB_LANG_CODE_UNDEF)
      {
         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(und_lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
               ((text_lang_ids[i] != ACFG_INVALID_DB_LANG)); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  und_lang_id = lang_id;
                  break;
               }
            }
         }
      }

      und_event_desc = NULL;
      /* Search for a short event descriptor with a language code included in the text langs */
      while ((name_str == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, SHORT_EVENT_DTAG, 0)) != NULL))
      {
         dptr = ReadLanguageCode(&event_desc->desc_data[2], &lang_code);

         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            for (i = 0; (i != ACFG_MAX_DB_LANG_CODES) &&
               (text_lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  /* Required language found, read the event name */
                  dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
                  if (temp_str != NULL)
                  {
                     name_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, lang_code);
                     STB_SIReleaseStringDesc(temp_str);
                  }
               }
            }
         }
         if ((name_str == NULL) && (und_lang_id != ACFG_INVALID_DB_LANG) && (und_event_desc == NULL))
         {
            /* If this descriptor is good enough for the undefined language behaviour, save it */
            if ((lang_code == ADB_LANG_CODE_UNDEF) || (lang_code == ADB_LANG_CODE_QAA))
            {
               und_event_desc = event_desc;
            }
         }
         event_desc = event_desc->next;
      }

      if ((name_str == NULL) && (und_event_desc != NULL))
      {
         /* No string has been found, apply the undefined language behaviour */
         dptr = ReadLanguageCode(&und_event_desc->desc_data[2], &lang_code);
         dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
         if (temp_str != NULL)
         {
            name_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, und_lang_code);
            STB_SIReleaseStringDesc(temp_str);
         }
      }
   }

   if (name_str == NULL)
   {
      /* No text langs defined, or failed to find an event descriptor with a required lang,
       * so return the first event name */
      event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, SHORT_EVENT_DTAG, 0);
      if (event_desc != NULL)
      {
         dptr = ReadLanguageCode(&event_desc->desc_data[2], &lang_code);
         dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
         if (temp_str != NULL)
         {
            name_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, lang_code);
            STB_SIReleaseStringDesc(temp_str);
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventName);

   return(name_str);
}

/**
 * @brief   Returns the short event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventDescription(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc, *und_event_desc;
   U8BIT *text_lang_ids;
   U32BIT lang_code, und_lang_code;
   U8BIT lang_id, i, und_lang_id;
   U8BIT *dptr;
   SI_STRING_DESC *temp_str;
   U16BIT str_len;
   U8BIT *desc_str;

   FUNCTION_START(DBDEF_GetEventDescription);

   ASSERT(e_ptr != NULL);

   desc_str = NULL;

   text_lang_ids = DBDEF_GetTextLang();
   if (text_lang_ids != NULL)
   {
      /* Search for a short event descriptor with a language code included in the text langs */
      event_desc = e_ptr->desc_list_head;

      und_lang_id = ACFG_INVALID_DB_LANG;
      und_lang_code = ACFG_GetUndefinedLanguageBehaviour();
      if (und_lang_code != ADB_LANG_CODE_UNDEF)
      {
         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(und_lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
               ((text_lang_ids[i] != ACFG_INVALID_DB_LANG)); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  und_lang_id = lang_id;
                  break;
               }
            }
         }
      }

      und_event_desc = NULL;
      while ((desc_str == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, SHORT_EVENT_DTAG, 0)) != NULL))
      {
         dptr = ReadLanguageCode(&event_desc->desc_data[2], &lang_code);

         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            /* Skip the event name */
            dptr += dptr[0] + 1;

            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
               (text_lang_ids[i] != ACFG_INVALID_DB_LANG) && (desc_str == NULL); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  /* Required language found, read the event description */
                  dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
                  if (temp_str != NULL)
                  {
                     desc_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, lang_code);
                     STB_SIReleaseStringDesc(temp_str);
                  }
               }
            }
         }
         if ((desc_str == NULL) && (und_lang_id != ACFG_INVALID_DB_LANG) && (und_event_desc == NULL))
         {
            /* If this descriptor is good enough for the undefined language behaviour, save it */
            if ((lang_code == ADB_LANG_CODE_UNDEF) || (lang_code == ADB_LANG_CODE_QAA))
            {
               und_event_desc = event_desc;
            }
         }
         event_desc = event_desc->next;
      }

      if ((desc_str == NULL) && (und_event_desc != NULL))
      {
         /* No string has been found, apply the undefined language behaviour */
         dptr = ReadLanguageCode(&und_event_desc->desc_data[2], &lang_code);
         dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
         if (temp_str != NULL)
         {
            desc_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, und_lang_code);
            STB_SIReleaseStringDesc(temp_str);
         }
      }
   }

   if (desc_str == NULL)
   {
      /* No text langs defined, or failed to find an event descriptor with a required lang,
       * so return the first event name */
      event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, SHORT_EVENT_DTAG, 0);
      if (event_desc != NULL)
      {
         dptr = ReadLanguageCode(&event_desc->desc_data[2], &lang_code);

         /* Skip the event name */
         dptr += dptr[0] + 1;

         dptr = STB_SIReadString(dptr[0], dptr + 1, &temp_str);
         if (temp_str != NULL)
         {
            desc_str = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, lang_code);
            STB_SIReleaseStringDesc(temp_str);
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventDescription);

   return(desc_str);
}

/**
 * @brief   Returns the extended event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventExtendedDescription(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc, *und_event_desc;
   ADB_EVENT_DESC *selected_event_desc;
   U8BIT *text_lang_ids;
   U32BIT lang_code;
   U32BIT selected_lang_code, und_lang_code;
   U8BIT lang_id, i, und_lang_id;
   U8BIT *dptr;
   U8BIT *end_ptr;
   U8BIT desc_num;
   SI_STRING_DESC *si_string;
   U16BIT str_len;
   U8BIT *temp_str;
   U8BIT *temp2_str;
   U8BIT *desc_str;

   FUNCTION_START(DBDEF_GetEventExtendedDescription);

   ASSERT(e_ptr != NULL);

   desc_str = NULL;

   selected_event_desc = NULL;

   text_lang_ids = DBDEF_GetTextLang();
   if (text_lang_ids != NULL)
   {
      event_desc = e_ptr->desc_list_head;

      und_lang_id = ACFG_INVALID_DB_LANG;
      und_lang_code = ACFG_GetUndefinedLanguageBehaviour();
      if (und_lang_code != ADB_LANG_CODE_UNDEF)
      {
         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(und_lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
               ((text_lang_ids[i] != ACFG_INVALID_DB_LANG)); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  und_lang_id = lang_id;
                  break;
                }
            }
         }
      }

      und_event_desc = NULL;
      /* Search for an extended event descriptor with a language code
       * included in the text langs and a descriptor number of 0 */
      while ((selected_event_desc == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL))
      {
         /* Check the descriptor number is 0 */
         if ((event_desc->desc_data[2] >> 4) == 0)
         {
            dptr = ReadLanguageCode(&event_desc->desc_data[3], &lang_code);

            /* See if this lang is in the list of valid langs */
            lang_id = ACFG_ConvertLangCodeToId(lang_code);
            if (lang_id != ACFG_INVALID_DB_LANG)
            {
               for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                  (text_lang_ids[i] != ACFG_INVALID_DB_LANG) && (selected_event_desc == NULL); i++)
               {
                  if (lang_id == text_lang_ids[i])
                  {
                     /* Required language found, read the event description */
                     selected_event_desc = event_desc;
                  }
               }
            }
            if ((selected_event_desc == NULL) && (und_lang_id != ACFG_INVALID_DB_LANG) &&
               (und_event_desc == NULL))
            {
               /* If this descriptor is good enough for the undefined language behaviour,
                  save it */
               if ((lang_code == ADB_LANG_CODE_UNDEF) || (lang_code == ADB_LANG_CODE_QAA))
               {
                  und_event_desc = event_desc;
               }
            }
         }
         event_desc = event_desc->next;
      }
      if ((selected_event_desc == NULL) && (und_event_desc != NULL))
      {
         selected_event_desc = und_event_desc;
      }
   }

   if (selected_event_desc == NULL)
   {
      /* No text langs defined, or failed to find an event descriptor with a required lang,
       * so use the first extended event descriptor with a descriptor number of 0 */
      event_desc = e_ptr->desc_list_head;
      while ((selected_event_desc == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL))
      {
         /* Check the descriptor number is 0 */
         if ((event_desc->desc_data[2] >> 4) == 0)
         {
            selected_event_desc = event_desc;
         }

         event_desc = event_desc->next;
      }
   }

   if (selected_event_desc != NULL)
   {
      /* Read all extended event descriptors with consecutive descriptor numbers and the
       * same language, to build up the extended descriptor string */
      ReadLanguageCode(&selected_event_desc->desc_data[3], &selected_lang_code);
      desc_num = 0;

      event_desc = selected_event_desc;
      while (event_desc != NULL)
      {
         end_ptr = &event_desc->desc_data[2] + event_desc->desc_data[1];

         /* Skip past the items to the non-itemised text */
         dptr = &event_desc->desc_data[6] + event_desc->desc_data[6] + 1;

         /* Check the lengths used are valid. Some have been found that are defined such
          * that the end of the text extends beyond the end of the descriptor! */
         if (dptr + dptr[0] < end_ptr)
         {
            /* Read the description string */
            temp_str = NULL;

            dptr = STB_SIReadString(dptr[0], dptr + 1, &si_string);
            if (si_string != NULL)
            {
               temp_str = STB_ConvertStringToUTF8(si_string->str_ptr, &str_len, FALSE,
                  selected_lang_code);
               STB_SIReleaseStringDesc(si_string);
            }

            if ((temp_str != NULL) && (desc_str != NULL))
            {
               temp2_str = STB_ConcatUnicodeStrings(desc_str, temp_str);
               STB_ReleaseUnicodeString(desc_str);
               STB_ReleaseUnicodeString(temp_str);
               desc_str = temp2_str;
            }
            else if ((temp_str != NULL) && (desc_str == NULL))
            {
               desc_str = temp_str;
            }

            /* If this is the last descriptor then the string is complete */
            if ((event_desc->desc_data[2] & 0x0f) == desc_num)
            {
               /* No more descriptors need to be read */
               event_desc = NULL;
            }
            else
            {
               /* Find the next extended event descriptor for this language */
               desc_num++;

               event_desc = event_desc->next;
               while ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL)
               {
                  if ((event_desc->desc_data[2] >> 4) == desc_num)
                  {
                     ReadLanguageCode(&event_desc->desc_data[3], &lang_code);
                     if (lang_code == selected_lang_code)
                     {
                        /* Next descriptor found */
                        break;
                     }
                  }

                  event_desc = event_desc->next;
               }
            }
         }
         else
         {
            /* Invalid descriptor, so can't append any more */
            event_desc = NULL;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventExtendedDescription);

   return(desc_str);
}

/**
 * @brief   Returns the items of extended event descriptor as item descriptor and item itself.
 * @param   e_ptr event
 * @param   num_items_ptr address in which to return the number of items
 * @return  ADB_EVENT_ITEMIZED_INFO pointer to the list of items, NULL on error
 */
ADB_EVENT_ITEMIZED_INFO* DBDEF_GetEventItemizedDescription(ADB_EVENT_REC *e_ptr, U16BIT *num_items_ptr)
{
   ADB_EVENT_DESC *event_desc, *und_event_desc;
   ADB_EVENT_DESC *selected_event_desc;
   U8BIT *text_lang_ids;
   U32BIT lang_code;
   U32BIT selected_lang_code, und_lang_code;
   U8BIT lang_id, i, und_lang_id;
   U8BIT *end_ptr;
   U8BIT desc_num;
   U16BIT str_len;
   U8BIT *item_dptr;
   U8BIT *item_end_ptr;
   U8BIT length_of_items;
   U16BIT index;
   ADB_EVENT_ITEMIZED_INFO *event_itemized_info;

   FUNCTION_START(ADB_GetEventItemizedDescription);

   ASSERT(e_ptr != NULL);

   event_itemized_info = NULL;
   index = 0;
   selected_event_desc = NULL;

   text_lang_ids = DBDEF_GetTextLang();
   if (text_lang_ids != NULL)
   {
      event_desc = e_ptr->desc_list_head;

      und_lang_id = ACFG_INVALID_DB_LANG;
      und_lang_code = ACFG_GetUndefinedLanguageBehaviour();
      if (und_lang_code != ADB_LANG_CODE_UNDEF)
      {
         /* See if this lang is in the list of valid langs */
         lang_id = ACFG_ConvertLangCodeToId(und_lang_code);
         if (lang_id != ACFG_INVALID_DB_LANG)
         {
            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
               ((text_lang_ids[i] != ACFG_INVALID_DB_LANG)); i++)
            {
               if (lang_id == text_lang_ids[i])
               {
                  und_lang_id = lang_id;
                  break;
               }
            }
         }
      }
      und_event_desc = NULL;

      /* Search for an extended event descriptor with a language code
       * included in the text langs and a descriptor number of 0 */
      while ((selected_event_desc == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL))
      {
         /* Check the descriptor number is 0 */
         if ((event_desc->desc_data[2] >> 4) == 0)
         {
            ReadLanguageCode(&event_desc->desc_data[3], &lang_code);

            /* See if this lang is in the list of valid langs */
            lang_id = ACFG_ConvertLangCodeToId(lang_code);
            if (lang_id != ACFG_INVALID_DB_LANG)
            {
               for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                  (text_lang_ids[i] != ACFG_INVALID_DB_LANG) && (selected_event_desc == NULL); i++)
               {
                  if (lang_id == text_lang_ids[i])
                  {
                     /* Required language found, read the event description */
                     selected_event_desc = event_desc;
                  }
               }
            }
            if ((selected_event_desc == NULL) && (und_lang_id != ACFG_INVALID_DB_LANG) &&
               (und_event_desc == NULL))
            {
               /* If this descriptor is good enough for the undefined language behaviour,
                  save it */
               if ((lang_code == ADB_LANG_CODE_UNDEF) || (lang_code == ADB_LANG_CODE_QAA))
               {
                  und_event_desc = event_desc;
               }
            }
         }
         event_desc = event_desc->next;
      }
      if ((selected_event_desc == NULL) && (und_event_desc != NULL))
      {
         selected_event_desc = und_event_desc;
      }
   }

   if (selected_event_desc == NULL)
   {
      /* No text langs defined, or failed to find an event descriptor with a required lang,
       * so use the first extended event descriptor with a descriptor number of 0 */
      event_desc = e_ptr->desc_list_head;
      while ((selected_event_desc == NULL) &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL))
      {
         /* Check the descriptor number is 0 */
         if ((event_desc->desc_data[2] >> 4) == 0)
         {
            selected_event_desc = event_desc;
         }
         event_desc = event_desc->next;
      }
   }

   if (selected_event_desc != NULL)
   {
      /* Read all extended event descriptors with consecutive descriptor numbers and the
       * same language, to build up the extended descriptor string */
      ReadLanguageCode(&selected_event_desc->desc_data[3], &selected_lang_code);
      desc_num = 0;

      event_desc = selected_event_desc;
      while (event_desc != NULL)
      {
         end_ptr = &event_desc->desc_data[2] + event_desc->desc_data[1];

         /* Get the length and address of items from event descriptor */
         item_dptr = &event_desc->desc_data[6];
         length_of_items = event_desc->desc_data[6];
         item_end_ptr = item_dptr + length_of_items;

         /* Get the item descripton data and item data from event descriptor*/
         /* Check the lengths used are valid.*/
         if ((length_of_items != 0) && (item_dptr != NULL) && (item_end_ptr < end_ptr))
         {
            SI_STRING_DESC *item_string;
            U8BIT *temp_item_str;

            item_dptr++;
            while(item_dptr < item_end_ptr)
            {
               if (index == 0 )
               {
                  event_itemized_info = STB_AppGetMemory(sizeof(ADB_EVENT_ITEMIZED_INFO));
               }
               else
               {
                  event_itemized_info = STB_AppResizeMemory(event_itemized_info,
                     (index + 1) * sizeof(ADB_EVENT_ITEMIZED_INFO));
               }

               if (event_itemized_info != NULL)
               {
                  item_dptr = STB_SIReadString(item_dptr[0], item_dptr + 1, &item_string);

                  if (item_string != NULL)
                  {
                     temp_item_str = STB_ConvertStringToUTF8(item_string->str_ptr,
                                                            &str_len,
                                                            FALSE,
                                                            selected_lang_code);
                     event_itemized_info[index].item_description = temp_item_str;
                     STB_SIReleaseStringDesc(item_string);
                  }

                  item_dptr = STB_SIReadString(item_dptr[0], item_dptr + 1, &item_string);
                  if (item_string != NULL)
                  {
                     temp_item_str = STB_ConvertStringToUTF8(item_string->str_ptr,
                                                            &str_len,
                                                            FALSE,
                                                            selected_lang_code);
                     event_itemized_info[index].item = temp_item_str;
                     STB_SIReleaseStringDesc(item_string);
                  }
                  index++;
               }
            }

            // If this is the last descriptor then the string is complete
            if ((event_desc->desc_data[2] & 0x0f) == desc_num)
            {
               //No more descriptors need to be read
               event_desc = NULL;
            }
            else
            {
               //Find the next extended event descriptor for this language
               desc_num++;

               event_desc = event_desc->next;
               while ((event_desc = DBDEF_FindEventDescriptor(event_desc, EXTENDED_EVENT_DTAG, 0)) != NULL)
               {
                  if ((event_desc->desc_data[2] >> 4) == desc_num)
                  {
                     ReadLanguageCode(&event_desc->desc_data[3], &lang_code);
                     if (lang_code == selected_lang_code)
                     {
                        //Next descriptor found
                        break;
                      }
                  }
                  event_desc = event_desc->next;
               }
            }
         }
         else
         {
            /* Invalid descriptor, so can't append any more */
            event_desc = NULL;
         }
      }//while (event_desc != NULL)
   }

   *num_items_ptr = index;

   FUNCTION_FINISH(DBDEF_GetEventItemizedDescription);

   return(event_itemized_info);
}

/**
 * @brief   Returns the guidance text for an event, either from the event itself
 *          or the event's service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   e_ptr event, can be NULL, in which case any guidance defined by the
 *                    service will be returned
 * @param   s_ptr service, can be NULL to just get guidance text from the event
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetEventGuidance(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr, U8BIT *type, U8BIT *mode)
{
   ADB_EVENT_DESC *event_desc;
   U32BIT country_code;
   U8BIT *dptr;
   U8BIT dlen;
   U8BIT *text_lang_ids;
   U32BIT lang_code;
   U8BIT lang_id;
   SI_STRING_DESC *si_string;
   U16BIT str_len;
   U8BIT *text_str;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT i;

   FUNCTION_START(DBDEF_GetEventGuidance);

   text_str = NULL;

   if (e_ptr != NULL)
   {
      /* UK DTT guidance descriptor */
      country_code = ACFG_GetCountry();
      if ((country_code == COUNTRY_CODE_UK) && ((t_ptr = s_ptr->transport) != NULL))
      {
         if (t_ptr->sig_type == SIGNAL_COFDM)
         {
            text_lang_ids = DBDEF_GetTextLang();
            if (text_lang_ids != NULL)
            {
               /* Search for a guidance descriptor with a language code included in the text langs */
               event_desc = e_ptr->desc_list_head;

               while ((text_str == NULL) &&
                  ((event_desc = DBDEF_FindEventDescriptor(event_desc, USER_DEFINED_DTAG_0x89,
                  UK_DTT_PRIVATE_DATA_CODE)) != NULL))
               {
                  dlen = event_desc->desc_data[1];
                  dptr = &event_desc->desc_data[2];

                  *type = *dptr & 0x03;
                  if ((*type == 0) || (*type == 1))
                  {
                     dptr++;
                     dlen--;

                     if (*type == 1)
                     {
                        /* Mode is a 1-bit value */
                        *mode = *dptr & 0x01;
                        dptr++;
                        dlen--;
                     }
                     else
                     {
                        *mode = 0;
                     }

                     dptr = ReadLanguageCode(dptr, &lang_code);
                     dlen -= 3;

                     lang_id = ACFG_ConvertLangCodeToId(lang_code);
                     if (lang_id != ACFG_INVALID_DB_LANG)
                     {
                        for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                           (text_lang_ids[i] != ACFG_INVALID_DB_LANG) && (text_str == NULL); i++)
                        {
                           if (lang_id == text_lang_ids[i])
                           {
                              STB_SIReadString(dlen, dptr, &si_string);
                              if (si_string != NULL)
                              {
                                 text_str = STB_ConvertStringToUTF8(si_string->str_ptr, &str_len,
                                    FALSE, lang_code);
                                 STB_SIReleaseStringDesc(si_string);
                              }
                           }
                        }
                     }
                  }

                  event_desc = event_desc->next;
               }
            }

            if (text_str == NULL)
            {
               /* No text langs defined, or failed to find a guidance descriptor with a desired
                * lang, so just return the first one available */
               if ((event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head,
                  USER_DEFINED_DTAG_0x89, UK_DTT_PRIVATE_DATA_CODE)) != NULL)
               {
                  dlen = event_desc->desc_data[1];
                  dptr = &event_desc->desc_data[2];

                  *type = *dptr & 0x03;
                  if ((*type == 0) || (*type == 1))
                  {
                     dptr++;
                     dlen--;

                     if (*type == 1)
                     {
                        /* Mode is a 1-bit value */
                        *mode = *dptr & 0x01;
                        dptr++;
                        dlen--;
                     }
                     else
                     {
                        *mode = 0;
                     }

                     dptr = ReadLanguageCode(dptr, &lang_code);
                     dlen -= 3;

                     STB_SIReadString(dlen, dptr, &si_string);
                     if (si_string != NULL)
                     {
                        text_str = STB_ConvertStringToUTF8(si_string->str_ptr, &str_len,
                           FALSE, lang_code);
                        STB_SIReleaseStringDesc(si_string);
                     }
                  }
               }
            }
         }
      }
   }
   if (text_str == NULL)
   {
      /* No event guidance, so see if the service has guidance defined */
      text_str = DBDEF_GetServiceGuidanceData(s_ptr, type, mode);
   }
   FUNCTION_FINISH(DBDEF_GetEventGuidance);

   return(text_str);
}

/**
 * @brief   Returns the guidance text for the service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   s_ptr service
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetServiceGuidanceData(ADB_SERVICE_REC *s_ptr, U8BIT *type, U8BIT *mode)
{
   U8BIT *text_str;
   U8BIT *lang_ids;
   U16BIT i;

   FUNCTION_START(DBDEF_GetServiceGuidance);

   text_str = NULL;

   if (s_ptr != NULL)
   {
      if ((lang_ids = DBDEF_GetTextLang()) != NULL)
      {
         for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) && (lang_ids[i] != ACFG_INVALID_DB_LANG) &&
              (text_str == NULL); i++)
         {
            if ((s_ptr->guidance[lang_ids[i]].text != NULL) &&
               ((s_ptr->guidance[lang_ids[i]].type == 0) || (s_ptr->guidance[lang_ids[i]].type == 1)))
            {
               *type = s_ptr->guidance[lang_ids[i]].type;
               *mode = s_ptr->guidance[lang_ids[i]].mode;
               text_str = CopyString(s_ptr->guidance[lang_ids[i]].text, TRUE);
            }
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetServiceGuidanceData);

   return(text_str);
}

/**
 * @brief   Returns the guidance text for the service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   s_ptr service
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetServiceGuidance(ADB_SERVICE_REC *s_ptr)
{
   U8BIT *text_str;
   U8BIT gtype;
   U8BIT gmode;

   FUNCTION_START(DBDEF_GetServiceGuidance);

   text_str = DBDEF_GetServiceGuidanceData(s_ptr, &gtype, &gmode);

   FUNCTION_FINISH(DBDEF_GetServiceGuidance);

   return(text_str);
}

/**
 * @brief   Returns the parental age value for the given event
 * @param   e_ptr event
 * @return  parental age value, 0 for invalid event
 */
U8BIT DBDEF_GetEventParentalAge(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc;
   U32BIT country_code;
   U32BIT lang_code;
   U8BIT dlen;
   U8BIT *dptr;
   U8BIT age;

   FUNCTION_START(DBDEF_GetEventParentalAge);

   age = 0;

   if ((event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, PARENTAL_RATING_DTAG, 0)) != NULL)
   {
      dlen = event_desc->desc_data[1];
      if (dlen >= 4)
      {
         country_code = ACFG_GetCountry();

         dptr = &event_desc->desc_data[2];
         while (dlen >= 4)
         {
            dptr = ReadLanguageCode(dptr, &lang_code);
            
            // for Taiwan DVB-T
            if (lang_code == COUNTRY_CODE_TAIWAN_NUM)
               lang_code = COUNTRY_CODE_TAIWAN;

            if (lang_code == country_code)
            {
               age = *dptr;
               break;
            }

            dptr++;
            dlen -= 4;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventParentalAge);

   return(age);
}

/**
 * @brief   Returns whether audio description is signalled as being available for the given event
 * @param   e_ptr event
 * @return  TRUE if available, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventAudioDescriptionFlag(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT stream_content;
   U8BIT component_type;
   BOOLEAN audio_desc;

   FUNCTION_START(DBDEF_GetEventAudioDescriptionFlag);

   audio_desc = FALSE;

   if (e_ptr != NULL)
   {
      event_desc = e_ptr->desc_list_head;

      while ((event_desc = DBDEF_FindEventDescriptor(event_desc, COMPONENT_DTAG, 0)) != NULL)
      {
         stream_content = event_desc->desc_data[2] & 0x0f;
         component_type = event_desc->desc_data[3];

         if ((stream_content == 0x02) &&
            ((component_type == 0x40) || (component_type == 0x42) ||
             (component_type == 0x47) || (component_type == 0x48)))
         {
            audio_desc = TRUE;
            break;
         }
         else if ((stream_content == 0x06) &&
            ((component_type == 0x40) || (component_type == 0x42) ||
             (component_type == 0x44) || (component_type == 0x46) ||
             ((component_type >= 0x47) && (component_type <= 0x4a))))
         {
            audio_desc = TRUE;
            break;
         }

         event_desc = event_desc->next;
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventAudioDescriptionFlag);

   return(audio_desc);
}

/**
 * @brief   Returns whether there's an HD event linked to the SD event and returns
 *          the info for it. The DVB info is checked to ensure it relates to a known
 *          service.
 * @param   e_ptr pointer to the event to check
 * @param   verify_event TRUE, if the schedule should be checked for the target event
 * @param   only_simulcast TRUE, if only an HD simulcast event should be returned
 * @param   hd_serv_ptr pointer to return the service for the HD event, if found.
 *          This can be NULL if not required.
 * @param   hd_event_ptr pointer to return the HD event, if found. Note that this will only be
 *          filled in if verify_event is TRUE. This can be NULL if not required.
 * @return  TRUE if an event is found, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventHDLinkageInfo(ADB_EVENT_REC *e_ptr, BOOLEAN verify_event, BOOLEAN only_simulcast,
   ADB_SERVICE_REC **hd_serv_ptr, ADB_EVENT_REC **hd_event_ptr)
{
   ADB_EVENT_DESC *event_desc;
   ADB_EVENT_DESC *desc_ptr;
   BOOLEAN event_found;
   U8BIT link_type;
   U8BIT *dptr;
   U16BIT onet_id;
   U16BIT trans_id;
   U16BIT serv_id;
   ADB_SERVICE_REC *s_ptr = NULL;
   U16BIT event_id;
   U8BIT target_id_type;
   U16BIT target_onet_id;
   U16BIT target_trans_id;
   U16BIT target_serv_id;
   ADB_EVENT_REC *target_event;
   U8BIT *end_ptr;
   U8BIT *info_ptr;
   U8BIT dlen;

   FUNCTION_START(DBDEF_GetEventHDLinkageInfo);

   event_found = FALSE;

   if (hd_serv_ptr != NULL)
   {
      *hd_serv_ptr = NULL;
   }

   if (hd_event_ptr != NULL)
   {
      *hd_event_ptr = NULL;
   }

   if (e_ptr != NULL)
   {
      /* There may be multiple links so find the first one that points to a valid service
       * with a valid event */
      event_desc = e_ptr->desc_list_head;

      while (!event_found &&
         ((event_desc = DBDEF_FindEventDescriptor(event_desc, LINKAGE_DTAG, 0)) != NULL))
      {
         /* Just interested in event or extended event linkage descriptors */
         link_type = event_desc->desc_data[8];

         if ((link_type == 0x0D) || (link_type == 0x0E))
         {
            dptr = &event_desc->desc_data[2];

            trans_id = (dptr[0] << 8) | dptr[1];
            onet_id = (dptr[2] << 8) | dptr[3];
            serv_id = (dptr[4] << 8) | dptr[5];

            if (link_type == 0x0D)
            {
               /* Event linkage, check the linked service is known */
               s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, onet_id, trans_id, serv_id);
               if (s_ptr != NULL)
               {
                  if (verify_event)
                  {
                     /* Event linkage */
                     event_id = (dptr[7] << 8) | dptr[8];

                     target_event = DBDEF_FindScheduleEventById(s_ptr, event_id);
                     if (target_event != NULL)
                     {
                        if ((only_simulcast && ((dptr[9] & 0x40) != 0)) || !only_simulcast)
                        {
                           event_found = TRUE;

                           if (hd_serv_ptr != NULL)
                           {
                              *hd_serv_ptr = s_ptr;
                           }
                           if (hd_event_ptr != NULL)
                           {
                              *hd_event_ptr = target_event;
                           }
                        }
                     }
                  }
                  else
                  {
                     if ((only_simulcast && ((dptr[9] & 0x40) != 0)) || !only_simulcast)
                     {
                        event_found = TRUE;

                        if (hd_serv_ptr != NULL)
                        {
                           *hd_serv_ptr = s_ptr;
                        }
                     }
                  }
               }
            }
            else
            {
               /* Extended event linkage contains a number of linked events, so need
                * to check each one to find the first one that's valid */
               end_ptr = &dptr[8] + dptr[7];
               dptr += 8;

               while (!event_found && (dptr < end_ptr))
               {
                  /* Work out the length of the data included in this
                   * extended info block. Minimum length is 3 bytes */
                  dlen = 3;

                  target_id_type = (dptr[2] & 0x0c) >> 2;

                  if (target_id_type == 3)
                  {
                     /* Length of user defined ID */
                     dlen += 2;
                  }
                  else
                  {
                     if (target_id_type == 1)
                     {
                        /* Length of transport stream ID */
                        dlen += 2;
                     }
                     if ((dptr[2] & 0x02) != 0)
                     {
                        /* Info includes the target original network ID */
                        dlen += 2;
                     }
                     if ((dptr[2] & 0x01) != 0)
                     {
                        /* Info includes the target service ID */
                        dlen += 2;
                     }
                  }

                  /* Check link_type to see if this is a link to an HD event */
                  if (((dptr[2] & 0x30) >> 4) == 1)
                  {
                     if ((only_simulcast && ((dptr[2] & 0x40) != 0)) || !only_simulcast)
                     {
                        event_id = (dptr[0] << 8) | dptr[1];

                        /* Use target_id_type and additional flags to determine how to
                         * find the target service */
                        if (target_id_type == 3)
                        {
                           /* The user defined type is profiled by the DTG for use on UK DTT,
                            * so check whether this instance of the descriptor is preceded
                            * by the UK DTT private data specifier */
                           if (ACFG_GetCountry() == COUNTRY_CODE_UK)
                           {
                              desc_ptr = e_ptr->desc_list_head;

                              while (((desc_ptr = DBDEF_FindEventDescriptor(desc_ptr, LINKAGE_DTAG,
                                 UK_DTT_PRIVATE_DATA_CODE)) != NULL) && (desc_ptr != event_desc))
                              {
                                 desc_ptr = desc_ptr->next;
                              }

                              if (desc_ptr != NULL)
                              {
                                 /* This is a UK extended event descriptor in which the user
                                  * defined value defines the LCN of the service linked to */
                                 target_serv_id = (dptr[3] << 8) | dptr[4];
                                 s_ptr = DBDEF_FindServiceRecByLcn(target_serv_id, NULL, TRUE);
                              }
                           }
                        }
                        else
                        {
                           info_ptr = &dptr[3];

                           target_onet_id = onet_id;
                           target_trans_id = trans_id;
                           target_serv_id = serv_id;

                           if (target_id_type == 1)
                           {
                              /* Read the target transport ID */
                              target_trans_id = (info_ptr[0] << 8) | info_ptr[1];
                              info_ptr += 2;
                           }
                           if ((dptr[2] & 0x02) != 0)
                           {
                              /* Info includes the target original network ID */
                              target_onet_id = (info_ptr[0] << 8) | info_ptr[1];
                              info_ptr += 2;
                           }
                           if ((dptr[2] & 0x01) != 0)
                           {
                              /* Info includes the target service ID */
                              target_serv_id = (info_ptr[0] << 8) | info_ptr[1];
                           }

                           s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                              target_onet_id, target_trans_id, target_serv_id);
                        }

                        if (s_ptr != NULL)
                        {
                           if (verify_event)
                           {
                              /* Now check the target service to see if the event exists */
                              target_event = DBDEF_FindScheduleEventById(s_ptr, event_id);
                              if (target_event != NULL)
                              {
                                 event_found = TRUE;

                                 if (hd_serv_ptr != NULL)
                                 {
                                    *hd_serv_ptr = s_ptr;
                                 }
                                 if (hd_event_ptr != NULL)
                                 {
                                    *hd_event_ptr = target_event;
                                 }
                              }
                           }
                           else
                           {
                              event_found = TRUE;

                              if (*hd_serv_ptr != NULL)
                              {
                                 *hd_serv_ptr = s_ptr;
                              }
                           }
                        }
                     }
                  }

                  dptr += dlen;
               }
            }
         }

         event_desc = event_desc->next;
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventHDLinkageInfo);

   return(event_found);
}

/**
 * @brief   Retrieves a list of components associated with the specified event, as described by
 *          component descriptors in the EIT.
 * @param   e_ptr Pointer to the event
 * @param   component_list Pointer to the returned list of components. The list must be freed by
 *          calling STB_AppFreeMemory, provided the returned number of components is greater than 0.
 * @return  Number of components associated with the specified event.
 */
U8BIT DBDEF_GetEventComponentList(ADB_EVENT_REC *e_ptr, ADB_EVENT_COMPONENT_INFO **component_list)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT num_components;
   U8BIT index;
   U32BIT size;

   FUNCTION_START(DBDEF_GetEventComponentList);

   num_components = 0;

   if (e_ptr != NULL)
   {
      /* Count the number of components */
      event_desc = e_ptr->desc_list_head;
      while (event_desc != NULL)
      {
         event_desc = DBDEF_FindEventDescriptor(event_desc, COMPONENT_DTAG, 0);
         if (event_desc != NULL)
         {
            num_components++;
            event_desc = event_desc->next;
         }
      }

      if (num_components != 0)
      {
         size = num_components * sizeof(ADB_EVENT_COMPONENT_INFO);
         *component_list = STB_AppGetMemory(size);
         if (*component_list != NULL)
         {
            /* Copy the component info into the array to be returned */
            event_desc = e_ptr->desc_list_head;
            index = 0;
            while (event_desc != NULL)
            {
               event_desc = DBDEF_FindEventDescriptor(event_desc, COMPONENT_DTAG, 0);
               if (event_desc != NULL)
               {
                  (*component_list)[index].stream_content = event_desc->desc_data[2] & 0x0f;
                  (*component_list)[index].component_type = event_desc->desc_data[3];
                  (*component_list)[index].component_tag = event_desc->desc_data[4];
                  memcpy((*component_list)[index].language_code, &event_desc->desc_data[5], 3);
                  index++;
                  event_desc = event_desc->next;
               }
            }
         }
         else
         {
            num_components = 0;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventComponentList);

   return(num_components);
}

/**
 * @brief   Returns whether DVB subtitles are signalled as being available for the given event
 * @param   e_ptr event
 * @return  TRUE if subtitles are available, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventSubtitlesAvailFlag(ADB_EVENT_REC *e_ptr)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT component_type;
   BOOLEAN available;

   FUNCTION_START(DBDEF_GetEventSubtitlesAvailFlag);

   available = FALSE;

   if (e_ptr != NULL)
   {
      event_desc = e_ptr->desc_list_head;

      while ((event_desc = DBDEF_FindEventDescriptor(event_desc, COMPONENT_DTAG, 0)) != NULL)
      {
         if ((event_desc->desc_data[2] & 0x0f) == 0x03)
         {
            component_type = event_desc->desc_data[3];

            if (((0x10 <= component_type) && (component_type <= 0x14)) ||
                ((0x20 <= component_type) && (component_type <= 0x24)))
            {
               available = TRUE;
               break;
            }
         }

         event_desc = event_desc->next;
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventSubtitlesAvailFlag);

   return(available);
}

/**
 * @brief   Returns the level 1 and 2 values for all content identifiers for the given event
 * @param   event_ptr event
 * @param   p_len address to return the number of bytes in the returned data
 * @return  pointer to the content data
 */
U8BIT* DBDEF_GetEventContentData(ADB_EVENT_REC *e_ptr, U8BIT *p_len)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT *retval;
   U8BIT len;

   FUNCTION_START(DBDEF_GetEventContentData);

   retval = NULL;

   if (e_ptr != NULL)
   {
      if ((event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, CONTENT_DTAG, 0)) != NULL)
      {
         len = event_desc->desc_data[1];
         if (len != 0)
         {
            retval = &event_desc->desc_data[2];
            *p_len = len;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventContentData);

   return(retval);
}

/**
 * @brief   Returns the genre of an event
 * @param   e_ptr event
 * @param   s_ptr service containing the event
 * @return  event content value, ADB_EVENT_CONTENT_UNCLASSIFIED if not available
 */
ADB_EVENT_CONTENT DBDEF_GetEventGenre(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_EVENT_DESC *event_desc;
   U8BIT content_value;
   U32BIT country_code;
   ADB_EVENT_CONTENT genre;
   U8BIT *conversion_array;

   FUNCTION_START(DBDEF_GetEventGenre);

   genre = ADB_EVENT_CONTENT_UNCLASSIFIED;

   if (e_ptr != NULL)
   {
      if ((event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, CONTENT_DTAG, 0)) != NULL)
      {
         if (event_desc->desc_data[1] != 0)
         {
            /* Only level_1 from the first content descriptor is currently used */
            country_code = ACFG_GetCountry();
            conversion_array = ACFG_GetEventContentTypes(country_code);

            content_value = event_desc->desc_data[2] >> 4;
            if (NULL != conversion_array) {
               genre = (ADB_EVENT_CONTENT)conversion_array[content_value];
            }

            if (content_value == 0x0f)
            {
               if ((country_code == COUNTRY_CODE_UK) && ((t_ptr = s_ptr->transport) != NULL))
               {
                  /* Special case processing for UK: 0xF is defined as DRAMA
                   * only if one of the UK private data specifiers is present,
                   * otherwise it should be unclassified */
                  if (t_ptr->sig_type == SIGNAL_COFDM)
                  {
                     /* Search for the content descriptor again with the UK DTT
                      * private data specifier */
                     if (DBDEF_FindEventDescriptor(e_ptr->desc_list_head, CONTENT_DTAG,
                        UK_DTT_PRIVATE_DATA_CODE) == NULL)
                     {
                        /* The content should be unclassified */
                        genre = ADB_EVENT_CONTENT_UNCLASSIFIED;
                     }
                  }
                  else if (t_ptr->sig_type == SIGNAL_QPSK)
                  {
                     /* Search for the content descriptor again with the UK Freesat
                      * private data specifier */
                     if (DBDEF_FindEventDescriptor(e_ptr->desc_list_head, CONTENT_DTAG,
                        FREESAT_PRIVATE_DATA_CODE) == NULL)
                     {
                        /* The content should be unclassified */
                        genre = ADB_EVENT_CONTENT_UNCLASSIFIED;
                     }
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventGenre);

   return(genre);
}

/**
 * @brief   Returns the full CRID of the type given for the given event
 * @param   serv_ptr service for the event
 * @param   event_ptr event CRID to be returned
 * @param   crid_type CRID type to be returned
 * @param   index the index of the CRID to be returned (only valid for series CRIDs)
 * @return  pointer to a string containing the CRID name,
 *          or NULL if the event doesn't have the requested CRID type
 */
U8BIT* DBDEF_GetEventCrid(ADB_SERVICE_REC *serv_ptr, ADB_EVENT_REC *e_ptr, U8BIT crid_type, U8BIT index)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT crids_found;
   U8BIT *dptr;
   U8BIT *end_ptr;
   U8BIT type;
   SI_STRING_DESC *si_string;
   U8BIT *crid_str;

   FUNCTION_START(DBDEF_GetEventCrid);

   /* Each event can have multiple content identifier descriptors, so need to search for
    * the index'th one of the given type */
   crid_str = NULL;
   crids_found = 0;
   event_desc = e_ptr->desc_list_head;

   while ((crid_str == NULL) &&
      ((event_desc = DBDEF_FindEventDescriptor(event_desc, CONT_ID_DESC_DTAG, 0)) != NULL))
   {
      dptr = &event_desc->desc_data[2];
      end_ptr = dptr + event_desc->desc_data[1];

      while ((crid_str == NULL) && (dptr < end_ptr))
      {
         /* Currently only support location 0 */
         if ((dptr[0] & CRID_LOCATION_MASK) == CRID_LOCATION_0)
         {
            type = (dptr[0] & CRID_TYPE_MASK) >> CRID_TYPE_SHIFT;

            if (type == crid_type)
            {
               if ((type == UK_PROGRAMME_CRID) || (type == TVA_PROGRAMME_CRID) ||
                  (index == crids_found))
               {
                  STB_SIReadString(dptr[1], dptr + 2, &si_string);
                  if (si_string != NULL)
                  {
                     crid_str = DBDEF_GetFullCrid(serv_ptr, si_string->str_ptr);
                  }
               }

               crids_found++;
            }

            dptr += dptr[1] + 2;
         }
         else if ((dptr[0] & CRID_LOCATION_MASK) == CRID_LOCATION_1)
         {
            dptr += 3;
         }
         else
         {
            dptr += 2;
         }
      }

      event_desc = event_desc->next;
   }

   FUNCTION_FINISH(DBDEF_GetEventCrid);

   return(crid_str);
}

/**
 * @brief   Returns the episode name of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
BOOLEAN DBDEF_GetEventSeriesInfo(ADB_EVENT_REC *e_ptr, U8BIT **episode_name, S32BIT *episode_number, U8BIT *content_status, BOOLEAN *last_episode)
{
   ADB_EVENT_DESC *event_desc;
   BOOLEAN ret = FALSE;

   FUNCTION_START(DBDEF_GetEventSeriesInfo);

   ASSERT(e_ptr != NULL);

   if (e_ptr != NULL)
   {
      if ((event_desc = DBDEF_FindEventDescriptor(e_ptr->desc_list_head, 0xD0 /*USER_DEFINED_DTAG_0xD0*/, 0)) != NULL)
      {
         U8BIT dlen;
         U8BIT *dptr;

         dlen = event_desc->desc_data[1];
         dptr = &event_desc->desc_data[2];
         
         if (dlen > 2)
         {
            SI_STRING_DESC *temp_str;
            U16BIT str_len;

            dptr = STB_SIReadString(dlen - 2, dptr, &temp_str);
            *episode_name = STB_ConvertStringToUTF8(temp_str->str_ptr, &str_len, FALSE, 0);
            STB_SIReleaseStringDesc(temp_str);
            *episode_number = ((uint16_t)dptr[0] << 5) + (dptr[1] >> 3);
            *content_status = (dptr[1] >> 1) & 0x03;
            *last_episode = (dptr[1] & 0x01) ? TRUE : FALSE;
            ret = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetEventSeriesInfo);

   return(ret);
}

/**
 * @brief   Returns the number of CRIDs of the given type for the event
 * @param   event_ptr counting the CRIDs for this event
 * @param   crid_type CRID type to be returned
 * @return  number of CRIDs found
 */
U8BIT DBDEF_NumberOfCridsOfType(ADB_EVENT_REC *e_ptr, U8BIT crid_type)
{
   ADB_EVENT_DESC *event_desc;
   U8BIT crids_found;
   U8BIT *dptr;
   U8BIT *end_ptr;
   U8BIT type;

   FUNCTION_START(DBDEF_NumberOfCridsOfType);

   crids_found = 0;

   if (e_ptr != NULL)
   {
      /* Each event can have multiple content identifier descriptors */
      event_desc = e_ptr->desc_list_head;

      while ((event_desc = DBDEF_FindEventDescriptor(event_desc, CONT_ID_DESC_DTAG, 0)) != NULL)
      {
         dptr = &event_desc->desc_data[2];
         end_ptr = dptr + event_desc->desc_data[1];

         while (dptr < end_ptr)
         {
            /* Currently only support location 0 */
            if ((dptr[0] & CRID_LOCATION_MASK) == CRID_LOCATION_0)
            {
               type = (dptr[0] & CRID_TYPE_MASK) >> CRID_TYPE_SHIFT;

               if (type == crid_type)
               {
                  crids_found++;
               }

               dptr += dptr[1] + 2;
            }
            else if ((dptr[0] & CRID_LOCATION_MASK) == CRID_LOCATION_1)
            {
               dptr += 3;
            }
            else
            {
               dptr += 2;
            }
         }

         event_desc = event_desc->next;
      }
   }

   FUNCTION_FINISH(DBDEF_NumberOfCridsOfType);

   return(crids_found);
}

/**
 * @brief   Returns the full CRID for the given CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_str CRID string
 * @return  pointer to a string containing the CRID name
 */
U8BIT* DBDEF_GetFullCrid(ADB_SERVICE_REC *serv_ptr, U8BIT *event_str)
{
   U8BIT *authority_str;
   U8BIT *crid_str = NULL;
   U16BIT str_len;

   FUNCTION_START(DBDEF_GetFullCrid);

   if ((serv_ptr != NULL) && (event_str != NULL))
   {
      if (*event_str == '/')
      {
         /* The CRID doesn't include the authority so get it from the service */
         authority_str = DBDEF_GetServiceDefaultAuthority(serv_ptr);

         if (authority_str != NULL)
         {
            str_len = strlen((char *)authority_str) + strlen((char *)event_str) + 1;
            crid_str = STB_AppGetMemory(str_len);
            if (crid_str != NULL)
            {
               strncpy((char *)crid_str, (char *)authority_str, str_len);
               strncat((char *)crid_str, (char *)event_str, str_len);
            }

            STB_AppFreeMemory(authority_str);
         }
      }
      else
      {
         /* The CRID includes the authority as well */
         str_len = strlen((char *)event_str) + 1;
         crid_str = STB_AppGetMemory(str_len);
         if (crid_str != NULL)
         {
            strncpy((char *)crid_str, (char *)event_str, str_len);
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetFullCrid);

   return(crid_str);
}

/**
 * @brief   Returns a copy of the default authority CRID string for the given service.
 *          This string will be returned as broadcast and should be freed using STB_AppFreMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to a copy of the CRID string, or NULL
 */
U8BIT* DBDEF_GetServiceDefaultAuthority(ADB_SERVICE_REC *s_ptr)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_NETWORK_REC *n_ptr;
   U8BIT *crid_str;

   FUNCTION_START(DBDEF_GetServiceDefaultAuthority);

   crid_str = NULL;

   if (s_ptr != NULL)
   {
      /* The search order for the default authority CRID is defined as service,
       * transport and finally network */
      if (s_ptr->def_authority != NULL)
      {
         crid_str = CopyString(s_ptr->def_authority, FALSE);
      }
      else
      {
         /* No default authority defined in the service so try the transport */
         t_ptr = s_ptr->transport;

         if (t_ptr != NULL)
         {
            if (t_ptr->def_authority != NULL)
            {
               crid_str = CopyString(t_ptr->def_authority, FALSE);
            }
            else
            {
               /* No default authority defined in the transport so try the network */
               n_ptr = t_ptr->network;
               if (n_ptr != NULL)
               {
                  if (n_ptr->def_authority != NULL)
                  {
                     crid_str = CopyString(n_ptr->def_authority, FALSE);
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetServiceDefaultAuthority);

   return(crid_str);
}

/**
 * @brief   Deletes all events in the given list
 * @param   elist list of events to be deleted
 */
void DBDEF_DeleteEventList(ADB_EVENT_REC *elist)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *next_ptr;
   ADB_EVENT_DESC *next;

   FUNCTION_START(DBDEF_DeleteEventList);

   e_ptr = elist;
   while (e_ptr != NULL)
   {
      next_ptr = e_ptr->next;

      while (e_ptr->desc_list_head != NULL)
      {
         next = e_ptr->desc_list_head->next;
         STB_AppFreeMemory(e_ptr->desc_list_head);
         e_ptr->desc_list_head = next;
      }

      e_ptr->desc_list_tail = NULL;

      DBA_DestroyRecord(e_ptr->dba_rec);

      STB_AppFreeMemory(e_ptr);

      e_ptr = next_ptr;
   }

   FUNCTION_FINISH(DBDEF_DeleteEventList);
}

/**
 *

 *
 * @brief   Deletes all records in a service alternate service list
 *
 * @param   aslist - the first alt serv record in the list
 *

 *
 */
void DBDEF_DeleteAltServList(ADB_ALT_SERV_REC *aslist)
{
   ADB_ALT_SERV_REC *alt_serv_rec;
   ADB_ALT_SERV_REC *next_rec;

   alt_serv_rec = aslist;
   while (alt_serv_rec != NULL)
   {
      next_rec = alt_serv_rec->next;
      STB_AppFreeMemory(alt_serv_rec);
      alt_serv_rec = next_rec;
   }
}

/*!**************************************************************************
 * @brief   Frees the given list of RCT links
 * @param   links links to be freed
 ****************************************************************************/
void DBDEF_DeleteRCTLinks(ADB_RCT_LINK_INFO *links)
{
   ADB_RCT_LINK_INFO *l_ptr;
   ADB_RCT_LINK_INFO *next_ptr;
   U8BIT i;

   FUNCTION_START(ADB_ReleaseRCTLinks);

   for (l_ptr = links; l_ptr != NULL; l_ptr = next_ptr)
   {
      next_ptr = l_ptr->next;

      if (l_ptr->uri_string != NULL)
      {
         STB_AppFreeMemory(l_ptr->uri_string);
      }

      DBDEF_ReleaseString(l_ptr->event_name);

      for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
      {
         DBDEF_ReleaseString(l_ptr->promo_text[i]);
      }

      STB_AppFreeMemory(l_ptr);
   }

   FUNCTION_FINISH(ADB_ReleaseRCTLinks);
}

/*!**************************************************************************
 * @brief   Frees given list of image icons and any associated memory
 * @param   icon_list icon list to be freed
 ****************************************************************************/
void DBDEF_DeleteImageIcons(ADB_IMAGE_ICON *icon_list)
{
   ADB_IMAGE_ICON *icon_ptr;
   ADB_IMAGE_ICON *next_ptr;

   FUNCTION_START(DBDEF_DeleteImageIcons);

   for (icon_ptr = icon_list; icon_ptr != NULL; icon_ptr = next_ptr)
   {
      next_ptr = icon_ptr->next;

      if (icon_ptr->destroy_func != NULL)
      {
         STB_SPDebugWrite("destroy file=%s", icon_ptr->icon_url);
         icon_ptr->destroy_func( icon_ptr->dsm_handle );
      }

      if (icon_ptr->icon_url != NULL)
      {
         STB_AppFreeMemory(icon_ptr->icon_url);
      }

      if (icon_ptr->icon_data != NULL)
      {
         STB_AppFreeMemory(icon_ptr->icon_data);
      }

      STB_AppFreeMemory(icon_ptr);
   }

   FUNCTION_FINISH(DBDEF_DeleteImageIcons);
}

/**
 * @brief   Creates an ADB_STRING, copying the given data into it. If the string passed in
 *          is NULL or the number of bytes is 0, then no string will be created
 * @param   lang_code ISO 639-2 3 char language code
 * @param   str_ptr pointer to string data to be copied into the ADB_STRING
 * @param   nbytes number of bytes of string data
 * @return  pointer to new ADB_STRING, or NULL
 */
ADB_STRING* DBDEF_MakeString(U32BIT lang_code, U8BIT *str_ptr, U16BIT nbytes)
{
   ADB_STRING *string;

   FUNCTION_START(DBDEF_MakeString);

   if ((str_ptr == NULL) || (nbytes == 0))
   {
      string = NULL;
   }
   else
   {
      string = STB_AppGetMemory(sizeof(ADB_STRING) + nbytes);
      if (string != NULL)
      {
         string->lang_code = lang_code;
         string->nbytes = nbytes;
         string->str_ptr = ((U8BIT *)string) + sizeof(ADB_STRING);
         memcpy(string->str_ptr, str_ptr, nbytes);
      }
   }

   FUNCTION_FINISH(DBDEF_MakeString);

   return(string);
}

/**
 * @brief   Creates a copy of the given ADB_STRING
 * @param   src_str string to be copied
 * @return  new ADB_STRING
 */
ADB_STRING* DBDEF_CopyString(ADB_STRING *src_str)
{
   ADB_STRING *string;

   FUNCTION_START(DBDEF_CopyString);

   string = NULL;

   if (src_str != NULL)
   {
      string = DBDEF_MakeString(src_str->lang_code, src_str->str_ptr, src_str->nbytes);
   }

   FUNCTION_FINISH(DBDEF_CopyString);

   return(string);
}

/**
 * @brief   Concatenates an SI_STRING_DESC string to the end of an ADB_STRING string,
 *          removing any terminating '\0' chars from the first string, and returns
 *          a new string.
 * @param   str1 string 1
 * @param   str2 string 2
 * @return  new ADB_STRING containing the contents of both strings
 */
ADB_STRING* DBDEF_ConcatSIString(ADB_STRING *str1, SI_STRING_DESC *str2)
{
   U32BIT nbytes;
   ADB_STRING *string;

   FUNCTION_START(DBDEF_ConcatSIString);

   /* Any trailing '\0's need to be excluded from the copy to the new string */
   nbytes = str1->nbytes;
   while (str1->str_ptr[nbytes - 1] == '\0')
   {
      nbytes--;
   }

   /* Create the new string and copy the data from both strings into it */
   string = STB_AppGetMemory(sizeof(ADB_STRING) + nbytes + str2->nbytes);
   if (string != NULL)
   {
      string->lang_code = str1->lang_code;
      string->nbytes = nbytes + str2->nbytes;
      string->str_ptr = ((U8BIT *)string) + sizeof(ADB_STRING);

      memcpy(string->str_ptr, str1->str_ptr, nbytes);
      memcpy(string->str_ptr + nbytes, str2->str_ptr, str2->nbytes);
   }

   FUNCTION_FINISH(DBDEF_ConcatSIString);

   return(string);
}

/**
 * @brief   Releases an ADB_STRING
 * @param   string string to be freed
 */
void DBDEF_ReleaseString(ADB_STRING *string)
{
   FUNCTION_START(DBDEF_ReleaseString);

   if (string != NULL)
   {
      STB_AppFreeMemory(string);
   }

   FUNCTION_FINISH(DBDEF_ReleaseString);
}

/**
 * @brief   Returns the number of LNB records in the database
 * @return  number of LNB records
 */
U16BIT DBDEF_GetNumLNBs(void)
{
   FUNCTION_START(DBDEF_GetNumLNBs);
   FUNCTION_FINISH(DBDEF_GetNumLNBs);
   return(STB_LLGetNumBlocks(&lnb_rec_list));
}

/**
 * @brief   Add an LNB record to the database
 * @param   type single, universal, unicable or used defined
 * @param   name LNB name, NULL for no name
 * @return  pointer to the new LNB record or NULL if not created
 */
ADB_LNB_REC* DBDEF_AddLNB(E_STB_DP_LNB_TYPE type, ADB_STRING *name)
{
   ADB_LNB_REC *lnb_ptr;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddLNB);

   lnb_ptr = STB_AppGetMemory(sizeof(ADB_LNB_REC));
   if (lnb_ptr != NULL)
   {
      memset(lnb_ptr, 0, sizeof(ADB_LNB_REC));

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_LNB, NULL);
      if (dba_rec != NULL)
      {
         lnb_ptr->dba_rec = dba_rec;

         /* Fill in details in record */
         lnb_ptr->type = type;
         DBA_SetFieldValue(dba_rec, DBA_FIELD_LNB_TYPE, (U32BIT)lnb_ptr->type);

         if (name != NULL)
         {
            lnb_ptr->name = name;
            DBA_SetFieldString(dba_rec, DBA_FIELD_LNB_NAME, name->str_ptr, name->nbytes);
         }

         STB_LLAddBlockToEnd(&lnb_rec_list, (LINK_LIST_PTR_BLK *)lnb_ptr);
      }
      else
      {
         STB_AppFreeMemory(lnb_ptr);
         lnb_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddLNB);

   return(lnb_ptr);
}

/**
 * @brief   Sets the LNB power setting
 * @param   lnb pointer to LNB record
 * @param   power power setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBPower(ADB_LNB_REC *lnb, E_STB_DP_LNB_POWER power)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBPower);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->power = power;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_POWER, (U32BIT)power);
   }

   FUNCTION_FINISH(DBDEF_SetLNBPower);

   return(retval);
}

/**
 * @brief   Sets the LNB 22k setting
 * @param   lnb pointer to LNB record
 * @param   is_22k 22k setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNB22k(ADB_LNB_REC *lnb, BOOLEAN is_22k)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNB22k);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->is_22k = is_22k;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_22K, (U32BIT)is_22k);
   }

   FUNCTION_FINISH(DBDEF_SetLNB22k);

   return(retval);
}

/**
 * @brief   Sets the LNB 12V setting
 * @param   lnb pointer to LNB record
 * @param   is_12v 12V setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNB12V(ADB_LNB_REC *lnb, BOOLEAN is_12v)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNB12V);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->is_12v = is_12v;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_12V, (U32BIT)is_12v);
   }

   FUNCTION_FINISH(DBDEF_SetLNB12V);

   return(retval);
}

/**
 * @brief   Sets the LNB pulse position setting
 * @param   lnb pointer to LNB record
 * @param   is_pulse_posn pulse position setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBPulsePosition(ADB_LNB_REC *lnb, BOOLEAN is_pulse_posn)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBPulsePosition);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->is_pulse_posn = is_pulse_posn;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_PULSEPOSN, (U32BIT)is_pulse_posn);
   }

   FUNCTION_FINISH(DBDEF_SetLNBPulsePosition);

   return(retval);
}

/**
 * @brief   Sets the LNB DiSEqC position setting
 * @param   lnb pointer to LNB record
 * @param   is_diseqc_posn diseqc position setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBDiSEqCPosition(ADB_LNB_REC *lnb, BOOLEAN is_diseqc_posn)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBDiSEqcPosition);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->is_diseqc_posn = is_diseqc_posn;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISPOSN, (U32BIT)is_diseqc_posn);
   }

   FUNCTION_FINISH(DBDEF_SetLNBDiSEqcPosition);

   return(retval);
}

/**
 * @brief   Sets the LNB DiSEqC tone setting
 * @param   lnb pointer to LNB record
 * @param   diseqc_tone diseqc tone setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBDiSEqCTone(ADB_LNB_REC *lnb, E_STB_DP_DISEQC_TONE diseqc_tone)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBDiSEqCTone);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->diseqc_tone = diseqc_tone;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISTONE, (U32BIT)diseqc_tone);
   }

   FUNCTION_FINISH(DBDEF_SetLNBDiSEqCTone);

   return(retval);
}

/**
 * @brief   Sets the LNB committed switch setting
 * @param   lnb pointer to LNB record
 * @param   cswitch switch setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBCSwitch(ADB_LNB_REC *lnb, E_STB_DP_DISEQC_CSWITCH cswitch)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBCSwitch);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->c_switch = cswitch;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISCSWITCH, (U32BIT)cswitch);
   }

   FUNCTION_FINISH(DBDEF_SetLNBCSwitch);

   return(retval);
}

/**
 * @brief   Sets the LNB uncommitted switch setting
 * @param   lnb pointer to LNB record
 * @param   uswitch switch setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBUSwitch(ADB_LNB_REC *lnb, U8BIT uswitch)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBUSwitch);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->u_switch = uswitch;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISUSWITCH, (U32BIT)uswitch);
   }

   FUNCTION_FINISH(DBDEF_SetLNBUSwitch);

   return(retval);
}

/**
 * @brief   Sets the LNB SMATV setting
 * @param   lnb pointer to LNB record
 * @param   is_smatv SMATV setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBSmatv(ADB_LNB_REC *lnb, BOOLEAN is_smatv)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBSmatv);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->is_smatv = is_smatv;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISSMATV, (U32BIT)is_smatv);
   }

   FUNCTION_FINISH(DBDEF_SetLNBSmatv);

   return(retval);
}

/**
 * @brief   Sets the LNB message repeat setting in the range 0 to 3
 * @param   lnb pointer to LNB record
 * @param   repeats repeat setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBRepeats(ADB_LNB_REC *lnb, U8BIT repeats)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBRepeats);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->diseqc_repeats = repeats;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_DISREPEAT, (U32BIT)repeats);
   }

   FUNCTION_FINISH(DBDEF_SetLNBRepeats);

   return(retval);
}

/**
 * @brief   Sets the LNB Unicable settings
 * @param   lnb pointer to LNB record
 * @param   inter_freq intermediate frequency setting
 * @param   chan channel setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBUnicable(ADB_LNB_REC *lnb, U32BIT inter_freq, U8BIT chan)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_SetLNBUnicable);

   retval = FALSE;

   if (lnb != NULL)
   {
      lnb->unicable_if = inter_freq;
      DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_UNICABLEFREQ, inter_freq);
      lnb->unicable_chan = chan;
      retval = DBA_SetFieldValue(lnb->dba_rec, DBA_FIELD_LNB_UNICABLECHAN, (U32BIT)chan);
   }

   FUNCTION_FINISH(DBDEF_SetLNBUnicable);

   return(retval);
}

/**
 * @brief   Returns the next LNB record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   lnb_ptr LNB record pointer, NULL if first record is required
 * @return  pointer to next LNB record
 */
ADB_LNB_REC* DBDEF_GetNextLNBRec(ADB_LNB_REC *lnb_ptr)
{
   ADB_LNB_REC *ret_ptr;

   FUNCTION_START(DBDEF_GetNextLNBRec);

   if (lnb_ptr == NULL)
   {
      ret_ptr = (ADB_LNB_REC *)STB_LLGetFirstBlock(&lnb_rec_list);
   }
   else
   {
      ret_ptr = (ADB_LNB_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lnb_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextLNBRec);

   return(ret_ptr);
}

/**
 * @brief   Returns the number of satellite records in the database
 * @return  number of satellite records
 */
U16BIT DBDEF_GetNumSatellites(void)
{
   FUNCTION_START(DBDEF_GetNumSatellites);
   FUNCTION_FINISH(DBDEF_GetNumSatellites);
   return(STB_LLGetNumBlocks(&satellite_rec_list));
}

/**
 * @brief   Add a satellite record to the database
 * @param   name_str satellite name
 * @param   dish_pos
 * @param   long_pos position in 1/10ths of a degree
 * @param   east_west TRUE=east, FALSE=west
 * @param   associated_lnb LNB record the satellite record is associated with
 * @return  pointer to the new satellite record or NULL if not created
 */
ADB_SATELLITE_REC* DBDEF_AddSatelliteRec(ADB_STRING *name_str, U16BIT dish_pos, U16BIT long_pos,
   BOOLEAN east_west, ADB_LNB_REC *associated_lnb)
{
   ADB_LNB_REC *lnb_ptr;
   ADB_SATELLITE_REC *sat_ptr;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddSatelliteRec);

   sat_ptr = NULL;

   if (associated_lnb != NULL)
   {
      // validate associated LNB by checking if is already in database
      lnb_ptr = (ADB_LNB_REC *)STB_LLGetFirstBlock(&lnb_rec_list);
      while (lnb_ptr != NULL)
      {
         if (lnb_ptr == associated_lnb)
         {
            break;
         }
         lnb_ptr = (ADB_LNB_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lnb_ptr);
      }

      if (lnb_ptr != NULL)
      {
         sat_ptr = STB_AppGetMemory(sizeof(ADB_SATELLITE_REC));
         if (sat_ptr != NULL)
         {
            memset(sat_ptr, 0, sizeof(ADB_SATELLITE_REC));

            sat_ptr->lnb = lnb_ptr;

            /* Add nvm record */
            dba_rec = DBA_CreateRecord(DBA_RECORD_SATELLITE, lnb_ptr->dba_rec);
            if (dba_rec != NULL)
            {
               sat_ptr->dba_rec = dba_rec;

               if (name_str != NULL)
               {
                  sat_ptr->name = name_str;
                  DBA_SetFieldString(dba_rec, DBA_FIELD_REC_NAME, name_str->str_ptr,
                     name_str->nbytes);
               }

               sat_ptr->dish_pos = dish_pos;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_SAT_DISH, sat_ptr->dish_pos);

               if (long_pos <= 1800)
               {
                  sat_ptr->long_pos = long_pos;
               }
               else
               {
                  sat_ptr->long_pos = 0;
               }
               DBA_SetFieldValue(dba_rec, DBA_FIELD_SAT_LONGPOS, sat_ptr->long_pos);

               if (east_west)
               {
                  sat_ptr->east_west = 1;
               }
               else
               {
                  sat_ptr->east_west = 0;
               }
               DBA_SetFieldValue(dba_rec, DBA_FIELD_SAT_LONGWE, sat_ptr->east_west);

               STB_LLAddBlockToEnd(&satellite_rec_list, (LINK_LIST_PTR_BLK *)sat_ptr);
            }
            else
            {
               STB_AppFreeMemory(sat_ptr);
               sat_ptr = NULL;
            }
         }
      }
   }

   FUNCTION_FINISH(DBDEF_AddSatelliteRec);

   return(sat_ptr);
}

/**
 * @brief   Returns the next satellite record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   sat_ptr satellite record pointer, NULL if first record is required
 * @return  pointer to next satellite record
 */
ADB_SATELLITE_REC* DBDEF_GetNextSatelliteRec(ADB_SATELLITE_REC *sat_ptr)
{
   ADB_SATELLITE_REC *ret_ptr;

   FUNCTION_START(DBDEF_GetNextSatelliteRec);

   if (sat_ptr == NULL)
   {
      ret_ptr = (ADB_SATELLITE_REC *)STB_LLGetFirstBlock(&satellite_rec_list);
   }
   else
   {
      ret_ptr = (ADB_SATELLITE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)sat_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextSatelliteRec);

   return(ret_ptr);
}

/**
 * @brief   Returns the number of LNB band records in the database
 * @return  number of LNB band records
 */
U16BIT DBDEF_GetNumLNBBands(void)
{
   FUNCTION_START(DBDEF_GetNumLNBBands);
   FUNCTION_FINISH(DBDEF_GetNumLNBBands);
   return(STB_LLGetNumBlocks(&lnb_band_rec_list));
}

/**
 * @brief   Add an LNB band record to the database
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @param   associated_lnb LNB record the band record is associated with.
 * @return  pointer to the new LNB band record or NULL if not created
 */
ADB_LNB_BAND_REC* DBDEF_AddLNBBandRec(S_STB_DP_LNB_BAND *band_parameters, ADB_LNB_REC *associated_lnb)
{
   ADB_LNB_REC *lnb_ptr;
   ADB_LNB_BAND_REC *band_ptr;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddLNBBandRec);

   band_ptr = NULL;

   if (associated_lnb != NULL)
   {
      // validate associated LNB by checking if is already in database
      lnb_ptr = (ADB_LNB_REC *)STB_LLGetFirstBlock(&lnb_rec_list);
      while (lnb_ptr != NULL)
      {
         if (lnb_ptr == associated_lnb)
         {
            break;
         }
         lnb_ptr = (ADB_LNB_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)lnb_ptr);
      }

      if (lnb_ptr != NULL)
      {
         band_ptr = STB_AppGetMemory(sizeof(ADB_LNB_BAND_REC));
         if (band_ptr != NULL)
         {
            memset(band_ptr, 0, sizeof(ADB_LNB_BAND_REC));

            band_ptr->lnb = lnb_ptr;

            /* Add nvm record */
            dba_rec = DBA_CreateRecord(DBA_RECORD_LNB_BAND, lnb_ptr->dba_rec);
            if (dba_rec != NULL)
            {
               band_ptr->dba_rec = dba_rec;

               band_ptr->band_params.polarity = band_parameters->polarity;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_POLARITY, band_ptr->band_params.polarity);

               band_ptr->band_params.min_freq = band_parameters->min_freq;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_MIN_FREQUENCY, band_ptr->band_params.min_freq);

               band_ptr->band_params.max_freq = band_parameters->max_freq;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_MAX_FREQUENCY, band_ptr->band_params.max_freq);

               band_ptr->band_params.local_oscillator_frequency =
                  band_parameters->local_oscillator_frequency;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_LOCAL_OSC_FREQUENCY,
                  band_ptr->band_params.local_oscillator_frequency);

               band_ptr->band_params.lnb_voltage = band_parameters->lnb_voltage;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_LNB_VOLTAGE,
                  band_ptr->band_params.lnb_voltage);

               band_ptr->band_params.tone_22k = band_parameters->tone_22k;
               DBA_SetFieldValue(dba_rec, DBA_FIELD_BAND_22_KHZ, band_ptr->band_params.tone_22k);

               STB_LLAddBlockToEnd(&lnb_band_rec_list, (LINK_LIST_PTR_BLK *)band_ptr);
            }
            else
            {
               STB_AppFreeMemory(band_ptr);
               band_ptr = NULL;
            }
         }
      }
   }

   FUNCTION_FINISH(DBDEF_AddLNBBandRec);

   return band_ptr;
}

/**
 * @brief   Returns the next LNB band record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   band_ptr LNB band record pointer, NULL if first record is required
 * @return  pointer to next LNB band record
 */
ADB_LNB_BAND_REC* DBDEF_GetNextLNBBandRec(ADB_LNB_BAND_REC *band_ptr)
{
   ADB_LNB_BAND_REC *ret_ptr;

   FUNCTION_START(DBDEF_GetNextLNBBandRec);

   if (band_ptr == NULL)
   {
      ret_ptr = (ADB_LNB_BAND_REC *)STB_LLGetFirstBlock(&lnb_band_rec_list);
   }
   else
   {
      ret_ptr = (ADB_LNB_BAND_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)band_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextLNBBandRec);

   return(ret_ptr);
}

/**
 * @brief   Deletes an LNB band record from the database
 * @param   band_ptr pointer to record to be deleted
 */
void DBDEF_DeleteLNBBandRec(ADB_LNB_BAND_REC *band_ptr)
{
   FUNCTION_START(DBDEF_DeleteLNBBandRec);

   if (band_ptr != NULL)
   {
      STB_LLRemoveBlock(&band_ptr->ptrs);

      DBA_DestroyRecord(band_ptr->dba_rec);

      STB_AppFreeMemory(band_ptr);
   }

   FUNCTION_FINISH(DBDEF_DeleteLNBBandRec);
}

/**
 * @brief   Returns the number of networks in ther service database
 * @return  number of transports
 */
U16BIT DBDEF_GetNumNetworks(void)
{
   FUNCTION_START(DBDEF_GetNumNetworks);
   FUNCTION_FINISH(DBDEF_GetNumNetworks);
   return(STB_LLGetNumBlocks(&network_rec_list));
}

/**
 * @brief   Adds a new network record to the database with the given network ID
 * @param   net_id network ID
 * @param   satellite satellite that contains the network, can be NULL
 * @return  pointer to new network record
 */
ADB_NETWORK_REC* DBDEF_AddNetworkRec(U16BIT net_id, ADB_SATELLITE_REC *satellite)
{
   ADB_NETWORK_REC *n_ptr;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddNetworkRec);

   n_ptr = STB_AppGetMemory(sizeof(ADB_NETWORK_REC));
   if (n_ptr != NULL)
   {
      memset(n_ptr, 0, sizeof(ADB_NETWORK_REC));

      /* Create database record */
      if (satellite != NULL)
      {
         n_ptr->satellite = satellite;
         dba_rec = DBA_CreateRecord(DBA_RECORD_NETWORK, satellite->dba_rec);
      }
      else
      {
         dba_rec = DBA_CreateRecord(DBA_RECORD_NETWORK, NULL);
      }

      if (dba_rec != NULL)
      {
         n_ptr->dba_rec = dba_rec;

         /* Fill in record details */
         n_ptr->net_id = net_id;
         DBA_SetFieldValue(dba_rec, DBA_FIELD_NET_ID, net_id);

         n_ptr->profile_type = ADB_PROFILE_TYPE_BROADCAST;
         DBA_SetFieldValue(dba_rec, DBA_FIELD_PROFILE_TYPE, (U32BIT)n_ptr->profile_type);

         STB_LLAddBlockToEnd(&network_rec_list, (LINK_LIST_PTR_BLK *)n_ptr);
      }
      else
      {
         STB_AppFreeMemory(n_ptr);
         n_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddNetworkRec);

   return(n_ptr);
}

/**
 * @brief   Deletes specified network record
 * @param   n_ptr Pointer to network record
 */
void DBDEF_DeleteNetworkRec(ADB_NETWORK_REC *n_ptr)
{
   FUNCTION_START(DBDEF_DeleteNetworkRec);
   DeleteNetworkRec(n_ptr);
   FUNCTION_FINISH(DBDEF_DeleteNetworkRec);
}

/**
 * @brief   Set or change the name of the given network
 * @param   n_ptr pointer to network record to be updated
 * @param   name name to be given to the network, can be any format, or NULL to clear the network name
 */
void DBDEF_SetNetworkName(ADB_NETWORK_REC *n_ptr, U8BIT *name)
{
   BOOLEAN changed;
   U16BIT new_name_len;

   FUNCTION_START(DBDEF_SetNetworkName);

   changed = TRUE;
   new_name_len = 0;

   if (name != NULL)
   {
      new_name_len = STB_GetNumBytesInString(name);
   }

   if (n_ptr->name_str != NULL)
   {
      if (new_name_len == n_ptr->name_str->nbytes)
      {
         /* String lengths are the same, check the bytes */
         if (memcmp(name, n_ptr->name_str->str_ptr, new_name_len) == 0)
         {
            /* Names are the same */
            changed = FALSE;
         }
      }
   }

   if (changed)
   {
      /* Free the current name */
      if (n_ptr->name_str != NULL)
      {
         DBDEF_ReleaseString(n_ptr->name_str);
         n_ptr->name_str = NULL;
      }

      if (name != NULL)
      {
         n_ptr->name_str = DBDEF_MakeString(0, name, new_name_len);
      }

      if (n_ptr->name_str != NULL)
      {
         DBA_SetFieldString(n_ptr->dba_rec, DBA_FIELD_REC_NAME, n_ptr->name_str->str_ptr,
            n_ptr->name_str->nbytes);
      }
      else
      {
         DBA_SetFieldString(n_ptr->dba_rec, DBA_FIELD_REC_NAME, NULL, 0);
      }

      DBA_SaveRecord(n_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetNetworkName);
}

/**
 * @brief   Finds the network with the given network ID
 * @param   net_id network ID to be found
 * @param   satellite satellite the network is on, can be NULL
 * @return  pointer to network record, or NULL if not found
 */
ADB_NETWORK_REC* DBDEF_FindNetworkRec(U16BIT net_id, ADB_SATELLITE_REC *satellite)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_FindNetworkRec);

   n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
   while (n_ptr != NULL)
   {
      if ((n_ptr->net_id == net_id) && DBDEF_NetworkInProfile(n_ptr) &&
         ((satellite == NULL) || (n_ptr->satellite == satellite)))
      {
         break;
      }
      n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindNetworkRec);

   return(n_ptr);
}

/**
 * @brief   Returns the network following the one given. If the argument is NULL
 *          then the first network will be returned.
 * @param   n_ptr network record, NULL if first network is required
 * @return  pointer to next network, NULL if no more networks
 */
ADB_NETWORK_REC* DBDEF_GetNextNetworkRec(ADB_NETWORK_REC *n_ptr)
{
   ADB_NETWORK_REC *net_ptr;

   FUNCTION_START(DBDEF_GetNextNetworkRec);

   if (n_ptr == NULL)
   {
      net_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
   }
   else
   {
      net_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
   }

   while ((net_ptr != NULL) && !DBDEF_NetworkInProfile(net_ptr))
   {
      net_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)net_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextNetworkRec);

   return(net_ptr);
}

/**
 * @brief   Returns the number of transports in ther service database
 * @return  number of transports
 */
U16BIT DBDEF_GetNumTransports(void)
{
   FUNCTION_START(DBDEF_GetNumTransports);
   FUNCTION_FINISH(DBDEF_GetNumTransports);
   return(STB_LLGetNumBlocks(&transport_rec_list));
}

/**
 * @brief   Returns the transport following the one given. If the argument is NULL
 *          then the first transport will be returned.
 * @param   t_ptr transport record, NULL if first transport is required
 * @return  pointer to next transport, NULL if no more transports
 */
ADB_TRANSPORT_REC* DBDEF_GetNextTransportRec(ADB_TRANSPORT_REC *t_ptr)
{
   ADB_TRANSPORT_REC *trans_ptr;

   FUNCTION_START(DBDEF_GetNextTransportRec);

   if (t_ptr == NULL)
   {
      trans_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   }
   else
   {
      trans_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)t_ptr);
   }

   while ((trans_ptr != NULL) && !DBDEF_TransportInProfile(trans_ptr))
   {
      trans_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)trans_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextTransportRec);

   return(trans_ptr);
}

/**
 * @brief   Adds a terrestrial transport record with the given frequency and PLP id
 * @param   freq_hz frequency in Hz
 * @param   plp_id PLP id for T2, pass 0 for T
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddTerrestrialTransportRec(U32BIT freq_hz, U8BIT plp_id,
   ADB_NETWORK_REC *network)
{
   ADB_TRANSPORT_REC *t_ptr;
   void *dba_rec;
   void *dba_network;

   FUNCTION_START(DBDEF_AddTerrestrialTransportRec);

   t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
   if (t_ptr != NULL)
   {
      memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));

      t_ptr->network = network;

      /* Check if top bit is set - if so this is an analog frequency */
      if ((freq_hz & 0x80000000L) != 0)
      {
         /* Analog */
         t_ptr->sig_type = SIGNAL_ANALOG;
         t_ptr->frequency = (freq_hz & 0x7fffffffL);
      }
      else
      {
         t_ptr->sig_type = SIGNAL_COFDM;
         t_ptr->frequency = freq_hz;

         t_ptr->u.terr.plp_id = plp_id;

         /* Initialise tuning parameters requiring non-zero state */
         t_ptr->u.terr.constellation = TUNE_TCONST_UNDEFINED;
         t_ptr->u.terr.hierarchy = TUNE_THIERARCHY_UNDEFINED;
         t_ptr->u.terr.lp_code_rate = TUNE_TCODERATE_UNDEFINED;
         t_ptr->u.terr.hp_code_rate = TUNE_TCODERATE_UNDEFINED;
         t_ptr->u.terr.guard_int = TUNE_TGUARDINT_UNDEFINED;
      }

      if (network != NULL)
      {
         dba_network = network->dba_rec;
      }
      else
      {
         dba_network = NULL;
      }

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_TERR_TRANSPORT, dba_network);
      if (dba_rec != NULL)
      {
         t_ptr->dba_rec = dba_rec;

         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, freq_hz); // incl top bit set for analog
         DBA_SetFieldValue(dba_rec, DBA_FIELD_TTRAN_PLP_ID, plp_id);

         STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

         DBA_SaveRecord(t_ptr->dba_rec);
      }
      else
      {
         STB_AppFreeMemory(t_ptr);
         t_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddTerrestrialTransportRec);

   return(t_ptr);
}

/**
 * @brief   Find the terrestrial transport record in the database matching the given params
 * @param   freq_hz frequency in Hz
 * @param   plp_id PLP id for T2, pass 0 to T
 * @return  pointer to the transport record or NULL if not found
 */
ADB_TRANSPORT_REC* DBDEF_FindTerrestrialTransportRec(U32BIT freq_hz, U8BIT plp_id)
{
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(DBDEF_FindTerrestrialTransportRec);

   t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   while (t_ptr != NULL)
   {
      if ((t_ptr->sig_type == SIGNAL_COFDM) &&
          (t_ptr->frequency == freq_hz) &&
          (t_ptr->u.terr.plp_id == plp_id))
      {
         break;
      }
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock(&t_ptr->ptrs);
   }

   FUNCTION_FINISH(DBDEF_FindTerrestrialTransportRec);

   return(t_ptr);
}

/**
 * @brief   Adds a cable transport record with the given frequency and symbol rate
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddCableTransportRec(U32BIT freq_hz, U32BIT symbol_rate, ADB_NETWORK_REC *network)
{
   ADB_TRANSPORT_REC *t_ptr;
   void *dba_rec;
   void *dba_network;

   FUNCTION_START(DBDEF_AddCableTransportRec);

   t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
   if (t_ptr != NULL)
   {
      memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));

      t_ptr->network = network;
      t_ptr->sig_type = SIGNAL_QAM;
      t_ptr->frequency = freq_hz;
      t_ptr->u.cab.symbol_rate = symbol_rate;

      if (network != NULL)
      {
         dba_network = network->dba_rec;
      }
      else
      {
         dba_network = NULL;
      }

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_CAB_TRANSPORT, dba_network);
      if (dba_rec != NULL)
      {
         t_ptr->dba_rec = dba_rec;

         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, freq_hz);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_SRATE, symbol_rate);

         STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

         DBA_SaveRecord(t_ptr->dba_rec);
      }
      else
      {
         STB_AppFreeMemory(t_ptr);
         t_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddCableTransportRec);

   return(t_ptr);
}

/**
 * @brief   Find the cable transport record in the database matching the given params
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @return  pointer to the transport record or NULL if not found
 */
ADB_TRANSPORT_REC* DBDEF_FindCableTransportRec(U32BIT freq_hz, U32BIT symbol_rate)
{
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(DBDEF_FindCableTransportRec);

   t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   while (t_ptr != NULL)
   {
      if ((t_ptr->sig_type == SIGNAL_QAM) && (t_ptr->frequency == freq_hz) &&
          (symbol_rate == SYMBOL_RATE_AUTO ||
           (t_ptr->u.cab.symbol_rate == SYMBOL_RATE_AUTO) ||
           (t_ptr->u.cab.symbol_rate == symbol_rate)))
      {
         break;
      }
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)t_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindCableTransportRec);

   return(t_ptr);
}

void DBDEF_SetCableTransportSymbolRate(ADB_TRANSPORT_REC *t_ptr, U16BIT symbol_rate)
{
   FUNCTION_START(DBDEF_SetCableTransportSymbolRate);

   if (t_ptr->u.cab.symbol_rate != symbol_rate)
   {
      t_ptr->u.cab.symbol_rate = symbol_rate;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, t_ptr->u.cab.symbol_rate);
      DBA_SaveRecord(t_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetCableTransportSymbolRate);
}

void DBDEF_SetCableTransportMode(ADB_TRANSPORT_REC *t_ptr, E_STB_DP_CMODE cmode)
{
   FUNCTION_START(DBDEF_SetCableTransportMode);

   if (t_ptr->u.cab.cmode != cmode)
   {
      t_ptr->u.cab.cmode = cmode;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_CTRAN_MODE, t_ptr->u.cab.cmode);
      DBA_SaveRecord(t_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetCableTransportMode);
}

/**
 * @brief   Adds a satellite transport record with the given frequency, symbol rate and polarity
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   polarity polarity
 * @param   dvb_s2 TRUE for DVB-S2
 * @param   modulation modulation type
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddSatTransportRec(U32BIT freq_hz, U16BIT symbol_rate,
   E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation,
   ADB_NETWORK_REC *network)
{
   ADB_TRANSPORT_REC *t_ptr;
   void *dba_rec;
   void *dba_network;

   FUNCTION_START(DBDEF_AddSatTransportRec);

   t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
   if (t_ptr != NULL)
   {
      memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));

      t_ptr->network = network;
      t_ptr->sig_type = SIGNAL_QPSK;
      t_ptr->frequency = freq_hz;
      t_ptr->u.sat.symbol_rate = symbol_rate;
      t_ptr->u.sat.polarity = polarity;
      t_ptr->u.sat.dvb_s2 = dvb_s2;
      t_ptr->u.sat.modulation = modulation;

      if (network != NULL)
      {
         dba_network = network->dba_rec;
      }
      else
      {
         dba_network = NULL;
      }

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_SAT_TRANSPORT, dba_network);
      if (dba_rec != NULL)
      {
         t_ptr->dba_rec = dba_rec;

         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, freq_hz);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_SRATE, symbol_rate);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_STRAN_POL, polarity);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_STRAN_DVBS2, dvb_s2);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_STRAN_MODULATION, modulation);

         STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

         DBA_SaveRecord(t_ptr->dba_rec);
      }
      else
      {
         STB_AppFreeMemory(t_ptr);
         t_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddSatTransportRec);

   return(t_ptr);
}

/**
 * @brief   Find the satellite transport record in the database matching the given params
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   dvb_s2 TRUE if DVB-S2
 * @param   modulation modulation type
 * @return  pointer to the transport record or NULL if not found
 */
ADB_TRANSPORT_REC* DBDEF_FindSatTransportRec(U32BIT freq_hz, U16BIT symbol_rate,
   E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation, void *satellite)
{
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(DBDEF_FindSatTransportRec);

   t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   while (t_ptr != NULL)
   {
      if (t_ptr->sig_type == SIGNAL_QPSK)
      {
         if ((t_ptr->frequency == freq_hz) && (t_ptr->u.sat.symbol_rate == symbol_rate) &&
             (t_ptr->u.sat.polarity == polarity) && (t_ptr->u.sat.dvb_s2 == dvb_s2) &&
             (t_ptr->u.sat.modulation == modulation))
         {
            if ((satellite == NULL) ||
                ((t_ptr->network != NULL) &&
                 (t_ptr->network->satellite != NULL) &&
                 (t_ptr->network->satellite == satellite)))
            {
               break;
            }
         }
      }
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)t_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindSatTransportRec);

   return(t_ptr);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   Adds a virtual transport record, for 'Virtual Channel' service
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddVirtualTransportRec(ADB_NETWORK_REC *network)
{
   ADB_TRANSPORT_REC *t_ptr;
   void *dba_rec;
   void *dba_network;

   FUNCTION_START(DBDEF_AddVirtualTransportRec);

   t_ptr = STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC));
   if (t_ptr != NULL)
   {
      memset(t_ptr, 0, sizeof(ADB_TRANSPORT_REC));

      t_ptr->network = network;

      t_ptr->sig_type = SIGNAL_NONE;

      if (network != NULL)
      {
         dba_network = network->dba_rec;
      }
      else
      {
         dba_network = NULL;
      }

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_TERR_TRANSPORT, dba_network);
      if (dba_rec != NULL)
      {
         t_ptr->dba_rec = dba_rec;

         DBA_SetFieldValue(dba_rec, DBA_FIELD_TRAN_FREQ, 0);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_TTRAN_PLP_ID, 0);

         STB_LLAddBlockToEnd(&transport_rec_list, (LINK_LIST_PTR_BLK *)t_ptr);

         DBA_SaveRecord(t_ptr->dba_rec);
      }
      else
      {
         STB_AppFreeMemory(t_ptr);
         t_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddVirtualTransportRec);

   return(t_ptr);
}
#endif /*COMMON_INTERFACE*/

/**
 * @brief   Sets the tuning parameters for an existing satellite transport
 * @param   t_ptr satellite transport record
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   polarity polarity
 * @param   dvb_s2 TRUE for DVB-S2
 * @param   modulation modulation type
 * @param   network network the transport is on, can be NULL
 */
void DBDEF_SetSatTransportTuningParams(ADB_TRANSPORT_REC *t_ptr, U32BIT freq_hz,
   U16BIT symbol_rate, E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation,
   ADB_NETWORK_REC *network)
{
   FUNCTION_START(DBDEF_SetSatTransportTuningParams);

   if (t_ptr != NULL)
   {
      if ((network != NULL) && (t_ptr->network != network))
      {
         t_ptr->network = network;
         DBA_SetRecordParent(t_ptr->dba_rec, network->dba_rec);
      }

      t_ptr->frequency = freq_hz;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_FREQ, freq_hz);

      t_ptr->u.sat.symbol_rate = symbol_rate;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, symbol_rate);

      t_ptr->u.sat.polarity = polarity;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_STRAN_POL, polarity);

      t_ptr->u.sat.dvb_s2 = dvb_s2;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_STRAN_DVBS2, dvb_s2);

      t_ptr->u.sat.modulation = modulation;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_STRAN_MODULATION, modulation);

      DBA_SaveRecord(t_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetSatTransportTuningParams);
}

/**
 * @brief   Find a transport record matching the given set of IDs, starting from the given transport
 * @param   transp start search from this transport, use NULL to start from the first transport
 * @param   net_id network ID, use ADB_INVALID_DVB_ID to ignore this value
 * @param   onet_id original network ID, use ADB_INVALID_DVB_ID to ignore this value
 * @param   tran_id transport ID, use ADB_INVALID_DVB_ID to ignore this value
 * @return  pointer to transport record, NULL if none found
 */
ADB_TRANSPORT_REC* DBDEF_FindTransportRecByIds(ADB_TRANSPORT_REC *transp, U16BIT net_id,
   U16BIT onet_id, U16BIT tran_id)
{
   ADB_TRANSPORT_REC *t_ptr;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_FindTransportRecByIds);

   if (transp == NULL)
   {
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetFirstBlock(&transport_rec_list);
   }
   else
   {
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)transp);
   }

   while (t_ptr != NULL)
   {
      if (DBDEF_TransportInProfile(t_ptr))
      {
         if (((tran_id == ADB_INVALID_DVB_ID) || (t_ptr->tran_id == tran_id)) &&
             ((onet_id == ADB_INVALID_DVB_ID) || (t_ptr->orig_net_id == onet_id)))
         {
            n_ptr = t_ptr->network;

            if ((net_id == ADB_INVALID_DVB_ID) || (n_ptr == NULL))
            {
               /* Found it */
               break;
            }
            else if (n_ptr != NULL)
            {
               if (n_ptr->net_id == net_id)
               {
                  /* Found it */
                  break;
               }
            }
         }
      }
      t_ptr = (ADB_TRANSPORT_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)t_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindTransportRecByIds);

   return(t_ptr);
}

/**
 * @brief   Sets the transport ID of the given transport
 * @param   t_ptr transport record the ID will be set for
 * @param   tran_id transport ID to be set
 */
void DBDEF_SetTransportTransportId(ADB_TRANSPORT_REC *t_ptr, U16BIT tran_id)
{
   FUNCTION_START(DBDEF_SetTransportTransportId);

   if (t_ptr->tran_id != tran_id)
   {
      t_ptr->tran_id = tran_id;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRANSPORT_ID, t_ptr->tran_id);
      DBA_SaveRecord(t_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetTransportTransportId);
}

/**
 * @brief   Sets the original network ID of the given transport
 * @param   t_ptr transport record the ID will be set for
 * @param   orig_net_id original network ID to be set
 */
void DBDEF_SetTransportOrigNetworkId(ADB_TRANSPORT_REC *t_ptr, U16BIT orig_net_id)
{
   FUNCTION_START(DBDEF_SetTransportOrigNetworkId);

   if (t_ptr->orig_net_id != orig_net_id)
   {
      t_ptr->orig_net_id = orig_net_id;
      DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_ORIG_NET_ID, t_ptr->orig_net_id);
      DBA_SaveRecord(t_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetTransportOrigNetworkId);
}

/**
 * @brief   Returns the additional frequencies associated with a transport.
 *          The returned array should not be modified or freed.
 * @param   t_ptr transport being queried
 * @param   num_freqs pointer to a value that returns the number of additional frequencies in the array
 * @param   freqs pointer to an array that contains the additional frequencies
 */
void DBDEF_GetTransportAdditionalFrequencies(ADB_TRANSPORT_REC *t_ptr, U8BIT *num_freqs, U32BIT **freqs)
{
   FUNCTION_START(DBDEF_GetTransportAdditionalFrequencies);

   if (t_ptr != NULL)
   {
      *num_freqs = t_ptr->num_additional_frequencies;
      *freqs = t_ptr->additional_frequencies;
   }
   else
   {
      *num_freqs = 0;
      *freqs = NULL;
   }

   FUNCTION_FINISH(DBDEF_GetTransportAdditionalFrequencies);
}

/**
 * @brief   Deletes the given transport from the service database, deleting any service
 *          records that it's the parent of beforehand.
 * @param   t_ptr pointer to transport to be deleted
 */
void DBDEF_DeleteTransportRec(ADB_TRANSPORT_REC *t_ptr)
{
   LINK_LIST_PTR_BLK *s_ptr;
   LINK_LIST_PTR_BLK *next_s_ptr;

   FUNCTION_START(DBDEF_DeleteTransportRec);

   /* Delete any service records that have this transport as their parent */
   s_ptr = STB_LLGetFirstBlock(&service_rec_list);
   while (s_ptr != NULL)
   {
      next_s_ptr = STB_LLGetNextBlock(s_ptr);

      if (((ADB_SERVICE_REC *)s_ptr)->transport == t_ptr)
      {
         DBDEF_DeleteServiceRec((ADB_SERVICE_REC *)s_ptr);
      }

      s_ptr = next_s_ptr;
   }

   STB_LLRemoveBlock(&t_ptr->ptrs);

   DBDEF_ClearTableVersionHistory(t_ptr);

   DBA_DestroyRecord(t_ptr->dba_rec);

   if (t_ptr->additional_frequencies != NULL)
   {
      STB_AppFreeMemory(t_ptr->additional_frequencies);
   }

   STB_AppFreeMemory(t_ptr);

   FUNCTION_FINISH(DBDEF_DeleteTransportRec);
}

/**
 *

 *
 * @brief   Clears the version histories stored in the transport records
 *

 *

 *
 */
void DBDEF_ClearTableVersionHistory(ADB_TRANSPORT_REC *t_ptr)
{
   ADB_PMT_VERSION_REC *pmt_ver_rec;
   void *tmp_ptr;

   FUNCTION_START(DBDEF_ClearTableVersionHistory);

   if (t_ptr != NULL)
   {
      // free pmt version list
      pmt_ver_rec = t_ptr->pmt_version_list;
      while (pmt_ver_rec != NULL)
      {
         tmp_ptr = (void *)pmt_ver_rec;
         pmt_ver_rec = pmt_ver_rec->next;
         STB_AppFreeMemory(tmp_ptr);
      }
      t_ptr->pmt_version_list = NULL;
   }

   FUNCTION_FINISH(DBDEF_ClearTableVersionHistory);
}

/**
 * @brief   Returns the total number of services
 * @return  total number of services
 */
U16BIT DBDEF_GetNumServices(void)
{
   U16BIT num_services;
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetNumServices);

   serv_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   for (num_services = 0; serv_ptr != NULL; )
   {
      if (!serv_ptr->deleted)
      {
         num_services++;
      }

      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNumServices);

   return(num_services);
}

/**
 * @brief   Returns the service after the one given. If NULL is passed then the first
 *          service in the list is returned.
 * @param   s_ptr pointer to service, NULL if first service is to be returned
 * @return  pointer to service, or NULL if no more services
 */
ADB_SERVICE_REC* DBDEF_GetNextServiceRec(ADB_SERVICE_REC *s_ptr)
{
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetNextServiceRec);

   if (s_ptr == NULL)
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   }
   else
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   while ((serv_ptr != NULL) && serv_ptr->deleted)
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextServiceRec);

   return(serv_ptr);
}

/**
 * @brief   Returns the service before the one given. If NULL is passed then the last
 *          service in the list is returned.
 * @param   s_ptr pointer to service, NULL if last service is to be returned
 * @return  pointer to service, or NULL if no more services
 */
ADB_SERVICE_REC* DBDEF_GetPrevServiceRec(ADB_SERVICE_REC *s_ptr)
{
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetPrevServiceRec);

   if (s_ptr == NULL)
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetLastBlock(&service_rec_list);
   }
   else
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetPrevBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   while ((serv_ptr != NULL) && serv_ptr->deleted)
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetPrevBlock((LINK_LIST_PTR_BLK *)serv_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetPrevServiceRec);

   return(serv_ptr);
}

/**
 * @brief   Adds a new service record to the service database with the given service ID
 *          and parent transport
 * @param   serv_id service ID for the new service
 * @param   t_ptr parent transport for the new service
 * @return  pointer to the new service, or NULL if the creation fails
 */
ADB_SERVICE_REC* DBDEF_AddServiceRec(U16BIT serv_id, ADB_TRANSPORT_REC *t_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   void *trans_dba_rec;
   void *serv_dba_rec;

   FUNCTION_START(DBDEF_AddServiceRec);

   s_ptr = STB_AppGetMemory(sizeof(ADB_SERVICE_REC));
   if (s_ptr != NULL)
   {
      memset(s_ptr, 0, sizeof(ADB_SERVICE_REC));
      s_ptr->reqd_audio_pid = 0x1FFF;

      trans_dba_rec = NULL;
      if (t_ptr != NULL)
      {
         trans_dba_rec = t_ptr->dba_rec;
      }

      /* Add service record to database */
      serv_dba_rec = DBA_CreateRecord(DBA_RECORD_SERVICE, trans_dba_rec);
      if (serv_dba_rec != NULL)
      {
         s_ptr->dba_rec = serv_dba_rec;
         s_ptr->transport = t_ptr;

         s_ptr->serv_id = serv_id;
         DBA_SetFieldValue(serv_dba_rec, DBA_FIELD_SERV_ID, (U32BIT)serv_id);

         s_ptr->freesat_id = INVALID_FREESAT_SERV_ID;
         DBA_SetFieldValue(serv_dba_rec, DBA_FIELD_SERV_FREESAT_ID, (U32BIT)s_ptr->freesat_id);

         s_ptr->lcn_editable = TRUE;
         DBA_SetFieldValue(serv_dba_rec, DBA_FIELD_SERV_LCN_EDITABLE, (U32BIT)s_ptr->lcn_editable);

         /* Default all services to being visible and selectable */
         s_ptr->hidden = FALSE;
         DBA_SetFieldValue(serv_dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

         s_ptr->selectable = TRUE;
         DBA_SetFieldValue(serv_dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

         STB_LLAddBlockToEnd(&service_rec_list, (LINK_LIST_PTR_BLK *)s_ptr);
      }
      else
      {
         /* Not enough NVM or RAM to store this service */
         STB_AppFreeMemory(s_ptr);
         s_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddServiceRec);

   return(s_ptr);
}

/**
 * @brief   Creates a copy of the given service, copying the service's attributes,
 *          e.g. service name, scrambled, locked, hidden, etc, but not the runtime specific
 *          data such as the stream list or event schedule data.
 * @param   orig_serv service to be copied
 * @return  new copied service, NULL on failure
 */
ADB_SERVICE_REC* DBDEF_CopyServiceRec(ADB_SERVICE_REC *orig_serv)
{
   ADB_SERVICE_REC *s_ptr;
   U8BIT i, j;

   FUNCTION_START(DBDEF_CopyServiceRec);

   s_ptr = DBDEF_AddServiceRec(orig_serv->serv_id, orig_serv->transport);
   if (s_ptr != NULL)
   {
      s_ptr->serv_type = orig_serv->serv_type;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

      if (s_ptr->serv_type == ADB_SERVICE_TYPE_ANALOG)
      {
         num_analog_channels++;
      }

      s_ptr->serv_lcn = orig_serv->serv_lcn;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);

      s_ptr->allocated_lcn = orig_serv->allocated_lcn;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN, s_ptr->allocated_lcn);

      if ((s_ptr->name_str = DBDEF_CopyString(orig_serv->name_str)) != NULL)
      {
         DBA_SetFieldString(s_ptr->dba_rec, DBA_FIELD_REC_NAME, s_ptr->name_str->str_ptr,
            s_ptr->name_str->nbytes);
      }

      s_ptr->fav_groups = orig_serv->fav_groups;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_FAV_GROUPS, s_ptr->fav_groups);

      s_ptr->locked = orig_serv->locked;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LOCKED, s_ptr->locked);

      s_ptr->hidden = orig_serv->hidden;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

      s_ptr->selectable = orig_serv->selectable;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

      s_ptr->lcn_editable = orig_serv->lcn_editable;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN_EDITABLE, s_ptr->lcn_editable);

      s_ptr->sched_disabled = orig_serv->sched_disabled;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SCHED_DISABLED, s_ptr->sched_disabled);

      s_ptr->now_next_disabled = orig_serv->now_next_disabled;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_NOWNEXT_DISABLED, s_ptr->now_next_disabled);

      s_ptr->freesat_id = orig_serv->freesat_id;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_FREESAT_ID, s_ptr->freesat_id);

      s_ptr->region_id = orig_serv->region_id;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REGION_ID, s_ptr->region_id);

      s_ptr->deleted = orig_serv->deleted;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_DELETED, s_ptr->deleted);

      s_ptr->provider_str = DBDEF_CopyString(orig_serv->provider_str);
      s_ptr->short_name_str = DBDEF_CopyString(orig_serv->short_name_str);

      for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
      {
         for (j = 0; j < ADB_NUM_SERV_NAME_IDS; j++)
         {
            s_ptr->name_array[i][j] = DBDEF_CopyString(orig_serv->name_array[i][j]);
         }

         s_ptr->short_name_array[i] = DBDEF_CopyString(orig_serv->short_name_array[i]);
         s_ptr->provider_array[i] = DBDEF_CopyString(orig_serv->provider_array[i]);
         s_ptr->guidance[i].text = DBDEF_CopyString(orig_serv->guidance[i].text);
         s_ptr->guidance[i].type = orig_serv->guidance[i].type;
         s_ptr->guidance[i].mode = orig_serv->guidance[i].mode;
      }

      s_ptr->scrambled = orig_serv->scrambled;

      s_ptr->eit_now_next_avail = orig_serv->eit_now_next_avail;
      s_ptr->eit_sched_avail = orig_serv->eit_sched_avail;

      s_ptr->running_status = orig_serv->running_status;
      s_ptr->not_running = orig_serv->not_running;
      s_ptr->unavailable = orig_serv->unavailable;
      s_ptr->has_fta_desc = orig_serv->has_fta_desc;
      s_ptr->do_not_scramble = orig_serv->do_not_scramble;

      s_ptr->def_authority = DBDEF_CopyString(orig_serv->def_authority);

      s_ptr->stream_list = DBDEF_CopyStreamList(orig_serv->stream_list);

      s_ptr->pcr_pid = orig_serv->pcr_pid;
      s_ptr->video_pid = orig_serv->video_pid;
      s_ptr->audio_pid = orig_serv->audio_pid;
      s_ptr->ad_pid = orig_serv->ad_pid;
      s_ptr->ttext_pid = orig_serv->ttext_pid;
      s_ptr->ttext_mag = orig_serv->ttext_mag;
      s_ptr->ttext_page = orig_serv->ttext_page;
      s_ptr->subtitle_pid = orig_serv->subtitle_pid;
      s_ptr->subtitle_cpage = orig_serv->subtitle_cpage;
      s_ptr->subtitle_apage = orig_serv->subtitle_apage;
      s_ptr->video_type = orig_serv->video_type;
      s_ptr->audio_type = orig_serv->audio_type;

      DBA_SaveRecord(s_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_CopyServiceRec);

   return(s_ptr);
}

/**
 * @brief   Set or change the name of a service
 * @param   s_ptr service
 * @param   name name to be given to the service, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceName(ADB_SERVICE_REC *s_ptr, U8BIT *name)
{
   BOOLEAN changed;
   U16BIT new_name_len;

   FUNCTION_START(DBDEF_SetServiceName);

   changed = TRUE;
   new_name_len = 0;

   if (name != NULL)
   {
      new_name_len = STB_GetNumBytesInString(name);
   }

   if (s_ptr->name_str != NULL)
   {
      if (new_name_len == s_ptr->name_str->nbytes)
      {
         /* String lengths are the same, check the bytes */
         if (memcmp(name, s_ptr->name_str->str_ptr, new_name_len) == 0)
         {
            /* Names are the same */
            changed = FALSE;
         }
      }
   }

   if (changed)
   {
      /* Free the current name */
      if (s_ptr->name_str != NULL)
      {
         DBDEF_ReleaseString(s_ptr->name_str);
         s_ptr->name_str = NULL;
      }

      if (name != NULL)
      {
         s_ptr->name_str = DBDEF_MakeString(0, name, new_name_len);
      }

      if (s_ptr->name_str != NULL)
      {
         DBA_SetFieldString(s_ptr->dba_rec, DBA_FIELD_REC_NAME, s_ptr->name_str->str_ptr,
            s_ptr->name_str->nbytes);
      }
      else
      {
         DBA_SetFieldString(s_ptr->dba_rec, DBA_FIELD_REC_NAME, NULL, 0);
      }

      DBA_SaveRecord(s_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetServiceName);

   return(changed);
}

/**
 * @brief   Set or change the short name of a service
 * @param   s_ptr service
 * @param   name new short name of the service, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceShortName(ADB_SERVICE_REC *s_ptr, U8BIT *name)
{
   BOOLEAN changed;
   U16BIT new_name_len;

   FUNCTION_START(DBDEF_SetServiceShortName);

   changed = TRUE;
   new_name_len = 0;

   if (name != NULL)
   {
      new_name_len = STB_GetNumBytesInString(name);
   }

   if (s_ptr->short_name_str != NULL)
   {
      if (new_name_len == s_ptr->short_name_str->nbytes)
      {
         /* String lengths are the same, check the bytes */
         if (memcmp(name, s_ptr->short_name_str->str_ptr, new_name_len) == 0)
         {
            /* Names are the same */
            changed = FALSE;
         }
      }
   }
   else if (name == NULL)
   {
      /* Names are both NULL */
      changed = FALSE;
   }

   if (changed)
   {
      /* Free the current name */
      if (s_ptr->short_name_str != NULL)
      {
         DBDEF_ReleaseString(s_ptr->short_name_str);
         s_ptr->short_name_str = NULL;
      }

      if (name != NULL)
      {
         s_ptr->short_name_str = DBDEF_MakeString(0, name, new_name_len);
      }
   }

   FUNCTION_FINISH(DBDEF_SetServiceShortName);

   return(changed);
}

/**
 * @brief   Set or change the name of a service's provider
 * @param   s_ptr service
 * @param   name name to be given to the service's provider, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceProviderName(ADB_SERVICE_REC *s_ptr, U8BIT *name)
{
   BOOLEAN changed;
   U16BIT new_name_len;

   FUNCTION_START(DBDEF_SetServiceProviderName);

   changed = TRUE;
   new_name_len = 0;

   if (name != NULL)
   {
      new_name_len = STB_GetNumBytesInString(name);
   }

   if (s_ptr->provider_str != NULL)
   {
      if (new_name_len == s_ptr->provider_str->nbytes)
      {
         /* String lengths are the same, check the bytes */
         if (memcmp(name, s_ptr->provider_str->str_ptr, new_name_len) == 0)
         {
            /* Names are the same */
            changed = FALSE;
         }
      }
   }
   else if (name == NULL)
   {
      /* Names are both NULL */
      changed = FALSE;
   }

   if (changed)
   {
      /* Free the current name */
      if (s_ptr->provider_str != NULL)
      {
         DBDEF_ReleaseString(s_ptr->provider_str);
         s_ptr->provider_str = NULL;
      }

      if (name != NULL)
      {
         s_ptr->provider_str = DBDEF_MakeString(0, name, new_name_len);
      }
   }

   FUNCTION_FINISH(DBDEF_SetServiceProviderName);

   return(changed);
}

/**
 * @brief   Sets the service group mask for the given service record
 * @param   s_ptr service
 * @param   group_mask the CNS channel group mask
 * @return  TRUE if the group mask is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceGroupMask(ADB_SERVICE_REC *s_ptr, U32BIT group_mask)
{
   BOOLEAN changed;

   FUNCTION_START(DBDEF_SetServiceGroupMask);

   changed = FALSE;

   if (s_ptr->group_mask != group_mask)
   {
      s_ptr->group_mask = group_mask;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_GROUP_MASK, s_ptr->group_mask);
      DBA_SaveRecord(s_ptr->dba_rec);
      changed = TRUE;
   }

   FUNCTION_FINISH(DBDEF_SetServiceGroupMask);

   return(changed);
}

/**
 * @brief   Sets the service type for the given service record
 * @param   s_ptr service
 * @param   serv_type type of service
 * @return  TRUE if the type is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceType(ADB_SERVICE_REC *s_ptr, ADB_SERVICE_TYPE serv_type)
{
   BOOLEAN changed;

   FUNCTION_START(DBDEF_SetServiceType);

   changed = FALSE;

   if (s_ptr->serv_type != serv_type)
   {
      s_ptr->serv_type = serv_type;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);
      DBA_SaveRecord(s_ptr->dba_rec);
      changed = TRUE;
   }

   FUNCTION_FINISH(DBDEF_SetServiceType);

   return(changed);
}

/**
 * @brief   Saves the event schedule of a service to the service database
 * @param   s_ptr event data to be saved for this service
 */
void DBDEF_SaveServiceEventSchedule(ADB_SERVICE_REC *s_ptr)
{
#if 0
   ADB_EVENT_REC *e_ptr;
   U16BIT i;
#endif

   FUNCTION_START(DBDEF_SaveServiceEventSchedule);

#if 0
   /* Create a database record for each event and save the current values for the record */
   e_ptr = s_ptr->event_schedule;
   while (e_ptr != NULL)
   {
      if (e_ptr->dba_rec == NULL)
      {
         e_ptr->dba_rec = DBA_CreateRecord(DBA_RECORD_EVENT, s_ptr->dba_rec);
      }

      if (e_ptr->dba_rec != NULL)
      {
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_ID, e_ptr->event_id);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_STARTTIME, e_ptr->start);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_DURATION, e_ptr->duration);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_VERSION, e_ptr->version);
#if 0
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_AGE_RATING, e_ptr->parental_age_rating);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_SCRAMBLED, e_ptr->scrambled);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_SUBTITLES, e_ptr->subtitles_avail);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_AUDIO_DESC, e_ptr->audio_desc);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_FREE_TO_AIR, e_ptr->has_fta_desc);
         DBA_SetFieldValue(e_ptr->dba_rec, DBA_FIELD_EVENT_DO_NOT_SCRAMBLE, e_ptr->do_not_scramble);

         for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
         {
            if (e_ptr->name_array[i] != NULL)
            {
               DBA_SetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_NAME,
                  e_ptr->name_array[i]->lang_code, e_ptr->name_array[i]->str_ptr,
                  e_ptr->name_array[i]->nbytes);
            }

            if (e_ptr->desc_array[i] != NULL)
            {
               DBA_SetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_DESCRIPTION,
                  e_ptr->desc_array[i]->lang_code, e_ptr->desc_array[i]->str_ptr,
                  e_ptr->desc_array[i]->nbytes);
            }

            if (e_ptr->extended_info[i].event_text != NULL)
            {
               DBA_SetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_EXTENDED_DESC,
                  e_ptr->extended_info[i].event_text->lang_code,
                  e_ptr->extended_info[i].event_text->str_ptr,
                  e_ptr->extended_info[i].event_text->nbytes);
            }

            if (e_ptr->guidance[i] != NULL)
            {
               DBA_SetFieldLangString(e_ptr->dba_rec, DBA_FIELD_EVENT_GUIDANCE,
                  e_ptr->guidance[i]->lang_code, e_ptr->guidance[i]->str_ptr,
                  e_ptr->guidance[i]->nbytes);
            }
         }

         if ((e_ptr->content_data != NULL) && (e_ptr->content_len > 0))
         {
            DBA_SetFieldData(e_ptr->dba_rec, DBA_FIELD_EVENT_CONTENT_DATA, e_ptr->content_data,
               e_ptr->content_len);
         }
#endif
         DBA_SaveRecord(e_ptr->dba_rec);
      }

      e_ptr = e_ptr->next;
   }
#else
   USE_UNWANTED_PARAM(s_ptr);
#endif

   FUNCTION_FINISH(DBDEF_SaveServiceEventSchedule);
}

/**
 * @brief   Find the next service following the given service that's on the given transport
 * @param   s_ptr find the service after this one, use NULL to get the first service
 * @param   t_ptr transport the service is on
 * @return  pointer to the service found, or NULL if not found
 */
ADB_SERVICE_REC* DBDEF_GetNextServiceOnTransport(ADB_SERVICE_REC *s_ptr, ADB_TRANSPORT_REC *t_ptr)
{
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetNextServiceOnTransport);

   if (s_ptr == NULL)
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   }
   else
   {
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   /* Look for service with required transport */
   while (serv_ptr != NULL)
   {
      if ((serv_ptr->transport == t_ptr) && !serv_ptr->deleted)
      {
         break;
      }
      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextServiceOnTransport);

   return(serv_ptr);
}

/**
 * @brief   Search for the service with the given service ID on the given transport
 * @param   serv_id service ID
 * @param   t_ptr transport to be searched
 * @return  pointer to the service, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRec(U16BIT serv_id, ADB_TRANSPORT_REC *t_ptr)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(DBDEF_FindServiceRec);

   s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   while (s_ptr != NULL)
   {
      if ((s_ptr->serv_id == serv_id) && (s_ptr->transport == t_ptr))
      {
         break;
      }

      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindServiceRec);

   return(s_ptr);
}

/**
 * @brief   Search for a service with the given IDs
 * @param   servp start searching from this service, NULL to start from first service
 * @param   net_id network ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   onet_id original network ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   tran_id transport ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   serv_id service ID, use ADB_INVALID_DVB_ID if not to be used
 * @return  pointer to the next service found, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByIds(ADB_SERVICE_REC *servp, U16BIT net_id, U16BIT onet_id,
   U16BIT tran_id, U16BIT serv_id)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_FindServiceRecByIds);

   if (servp == NULL)
   {
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   }
   else
   {
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)servp);
   }

   while (s_ptr != NULL)
   {
      if ((serv_id == ADB_INVALID_DVB_ID) || (s_ptr->serv_id == serv_id))
      {
         t_ptr = s_ptr->transport;
         if (t_ptr != NULL)
         {
            if (DBDEF_TransportInProfile(t_ptr))
            {
               if (((tran_id == ADB_INVALID_DVB_ID) || (t_ptr->tran_id == tran_id)) &&
                   ((onet_id == ADB_INVALID_DVB_ID) || (t_ptr->orig_net_id == onet_id)))
               {
                  if (net_id == ADB_INVALID_DVB_ID)
                  {
                     /* Don't care about the network ID, so use this service rec */
                     break;
                  }
                  else
                  {
                     n_ptr = t_ptr->network;
                     if (n_ptr != NULL)
                     {
                        if (n_ptr->net_id == net_id)
                        {
                           /* Found it */
                           break;
                        }
                     }
                  }
               }
            }
         }
      }

      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindServiceRecByIds);

   return(s_ptr);
}

/**
 * @brief   Find the service with the given LCN, and optionally on the given transport
 * @param   lcn LCN
 * @param   t_ptr transport the service must be on, use NULL for any transport
 * @param   allocated_lcn TRUE if the search should be based on the LCN a service has been
 *                        allocated rather than the one it requested (its service LCN)
 * @return  pointer to service if found, NULL otherwise
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByLcn(U16BIT lcn, ADB_TRANSPORT_REC *t_ptr, BOOLEAN allocated_lcn)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(DBDEF_FindServiceRecByLcn);

   s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   while (s_ptr != NULL)
   {
      if ((allocated_lcn && (s_ptr->allocated_lcn == lcn)) ||
          (!allocated_lcn && (s_ptr->serv_lcn == lcn)))
      {
         if (DBDEF_ServiceInProfile(s_ptr))
         {
            if ((t_ptr == NULL) || (t_ptr == s_ptr->transport))
            {
               break;
            }
         }
      }
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindServiceRecByLcn);

   return(s_ptr);
}

/**
 * @brief   Search for a service with the given Freesat ID
 * @param   servp start searching from this service, NULL to start from first service
 * @param   freesat_id Freesat service ID
 * @return  pointer to the next service found, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByFreesatId(ADB_SERVICE_REC *servp, U16BIT freesat_id)
{
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(DBDEF_FindServiceRecByFreesatId);

   if (servp == NULL)
   {
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   }
   else
   {
      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)servp);
   }

   while (s_ptr != NULL)
   {
      if (s_ptr->freesat_id == freesat_id)
      {
         break;
      }

      s_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)s_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindServiceRecByFreesatId);

   return(s_ptr);
}

/**
 * @brief   Search for a stream with the given PID
 * @param   servp searching from this service
 * @param   pid stream PID, use ADB_INVALID_DVB_ID if not to be used
 * @return  stream record found, or NULL if no service found
 */
ADB_STREAM_REC* DBDEF_FindStreamRecById(ADB_SERVICE_REC *servp, U16BIT pid)
{
   ADB_STREAM_REC * stream_rec = NULL;

   FUNCTION_START(DBDEF_FindStreamRecById);

   if (servp != NULL)
   {
      stream_rec = servp->stream_list;
      while (stream_rec != NULL)
      {
         if (pid == stream_rec->pid)
            break;

         stream_rec = stream_rec->next;
      }
   }

   FUNCTION_FINISH(DBDEF_FindStreamRecById);

   return stream_rec;
}

/**
 *

 *
 * @brief   Gets the appropriate audio pid - looks first for the pid matching exactly the
 *                required audio settings in the database, then for the pid matching the default
 *                language, then the first audip pid in the stream list.
 *                If no pid found the returned pid value will be 0
 *
 * @param   s_ptr - service record ptr
 * @param   audio_mode - pointer to the returned channel setting (mono, stereo, left, right)
 * @param   audio_type - pointer to the returned stream type (codec)
 *
 * @return   required audio pid
 *
 */
U16BIT DBDEF_GetReqdAudioPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *audio_mode, ADB_STREAM_TYPE *audio_type)
{
   U8BIT *db_lang_ids;
   U16BIT a_pid;
   BOOLEAN found_primary;
   ADB_STREAM_REC *selected_stream;

   FUNCTION_START(DBDEF_GetReqdAudioPid);

   found_primary = FALSE;

   /* Get the first primary language */
   db_lang_ids = DBDEF_GetAudioLang();
   a_pid = GetReqdAudioPid(s_ptr, audio_mode, audio_type, db_lang_ids, &found_primary, &selected_stream);

   /* Check for a secondary language if the first is not found and secondary is valid */
   if (found_primary)
   {
      selected_stream->in_use = TRUE;
   }
   else
   {
      if ((db_lang_ids = DBDEF_GetSecondaryAudioLang()) != NULL)
      {
         a_pid = GetReqdAudioPid(s_ptr, audio_mode, audio_type, db_lang_ids, &found_primary, &selected_stream);
         if (selected_stream != NULL)
         {
            selected_stream->in_use = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetReqdAudioPid);

   return(a_pid);
}

/**
 *

 *
 * @brief   Gets the appropriate audio description pid - looks first for the pid matching exactly the
 *                required audio settings in the database, then for the pid matching the default
 *                language, then the first audip pid in the stream list.
 *                If no pid found the returned pid value will be 0
 *
 * @param   s_ptr - service record ptr
 * @param   ad_mode - pointer to the returned channel setting (mono)
 * @param   ad_type - pointer to the returned stream type (codec)
 * @param   broadcast_mix - pointer to the returned boolean indicator of broadcast mix audio
 *
 * @return   required audio pid
 *
 */
U16BIT DBDEF_GetReqdADPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *ad_mode, ADB_STREAM_TYPE *ad_type, BOOLEAN *broadcast_mix)
{
   U16BIT ad_pid;
   U8BIT *db_lang_ids;
   ADB_STREAM_REC *selected_stream;

   FUNCTION_START(DBDEF_GetReqdADPid);

   /* Get the first primary language */
   db_lang_ids = DBDEF_GetAudioLang();
   ad_pid = GetReqdADPid(s_ptr, ad_mode, db_lang_ids, &selected_stream, broadcast_mix);
   if (selected_stream != NULL)
   {
      selected_stream->in_use = TRUE;
      *ad_type = selected_stream->type;
   }
   else
   {
      /* Check for a secondary language */
      if ((db_lang_ids = DBDEF_GetSecondaryAudioLang()) != NULL)
      {
         ad_pid = GetReqdADPid(s_ptr, ad_mode, db_lang_ids, &selected_stream, broadcast_mix);
         if (selected_stream != NULL)
         {
            selected_stream->in_use = TRUE;
            *ad_type = selected_stream->type;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetReqdADPid);

   return(ad_pid);
}

/**
 *

 *
 * @brief   Gets the appropriate teletext pid - looks first for the params matching exactly the
 *                required teletext settings in the database, then for the pid
 *                matching the default language, then the first teletext stream in the stream list.
 *                If no teletext found the returned pid will be 0
 *
 * @param   s_ptr - service record ptr
 * @param   for_subtitles - TRUE if subtitle stream is to be selected
 * @param   magazine - returns the magazine defined in the stream
 * @param   page - returns the defined page in the stream
 *
 * @return   The required PID value
 *
 */
E_STREAM_MATCH_TYPE DBDEF_GetReqdTtextPid(ADB_SERVICE_REC *s_ptr, BOOLEAN for_subtitles, U16BIT *pid_ptr,
   U8BIT *magazine, U8BIT *page)
{
   U8BIT *db_lang_ids;
   BOOLEAN found_primary;
   E_STREAM_MATCH_TYPE match_type;
   ADB_STREAM_REC *selected_stream;

   FUNCTION_START(DBDEF_GetReqdTtextPid);

   found_primary = FALSE;

   db_lang_ids = DBDEF_GetTextLang();

   match_type = GetReqdTtextParams(s_ptr, pid_ptr, for_subtitles, magazine, page, db_lang_ids,
         &found_primary, &selected_stream);

   if (found_primary)
   {
      selected_stream->in_use = TRUE;
   }
   else
   {
      /* Check for a secondary language */
      if ((db_lang_ids = DBDEF_GetSecondaryTextLang()) != NULL)
      {
         match_type = GetReqdTtextParams(s_ptr, pid_ptr, for_subtitles, magazine, page, db_lang_ids,
               &found_primary, &selected_stream);
         if (selected_stream != NULL)
         {
            selected_stream->in_use = TRUE;
         }
      }
   }

   if ((match_type == STREAM_MATCH_NOT_ZERO) && ACFG_IsNordigCountry())
   {
      /* set match_type to STREAM_MATCH_NONE as Nordig test will not show the non-zero DVB subtitle pid
         if it doesn't match with the subtitle lang */
      match_type = STREAM_MATCH_NONE;
      if (selected_stream != NULL)
      {
         selected_stream->in_use = FALSE;
      }
   }

   FUNCTION_FINISH(DBDEF_GetReqdTtextPid);

   return(match_type);
}

/**
 *

 *
 * @brief   Gets the appropriate subtitle pid and page ids - looks first for the params
 *                matching exactly the required subtitle settings in the database, then for the pid
 *                matching the default language, then the first subtitle stream in the stream list.
 *                If no subtitles found the returned pid will be 0
 *
 * @param   s_ptr     - service record ptr
 * @param   pid_ptr   - pointer for the return of the pid
 * @param   cpage_ptr - pointer for the return of the composition page
 * @param   apage_ptr - pointer for the return of the ancilliary page
 *

 *
 */
E_STREAM_MATCH_TYPE DBDEF_GetReqdSubtitleParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr,
   U16BIT *cpage_ptr, U16BIT *apage_ptr)
{
   U8BIT *db_lang_ids;
   BOOLEAN found_primary;
   E_STREAM_MATCH_TYPE match_type;
   ADB_STREAM_REC *selected_stream;

   FUNCTION_START(DBDEF_GetReqdSubtitleParams);

   found_primary = FALSE;

   db_lang_ids = DBDEF_GetTextLang();
   match_type = GetReqdSubtitleParams(s_ptr, pid_ptr, cpage_ptr, apage_ptr, db_lang_ids,
         &found_primary, &selected_stream);

   if (found_primary)
   {
      /* Mark the selected stream as in use */
      selected_stream->in_use = TRUE;
   }
   else
   {
      /* Check for a secondary language */
      if ((db_lang_ids = DBDEF_GetSecondaryTextLang()) != NULL)
      {
         match_type = GetReqdSubtitleParams(s_ptr, pid_ptr, cpage_ptr, apage_ptr, db_lang_ids,
               &found_primary, &selected_stream);
         if (selected_stream != NULL)
         {
            selected_stream->in_use = TRUE;
         }
      }
   }

   if ((match_type == STREAM_MATCH_NOT_ZERO) && ACFG_IsNordigCountry())
   {
      /* set match_type to STREAM_MATCH_NONE as Nordig test will not show the non-zero DVB subtitle pid
         if it doesn't match with the subtitle lang */
      match_type = STREAM_MATCH_NONE;
      if (selected_stream != NULL)
      {
         selected_stream->in_use = FALSE;
      }
   }

   FUNCTION_FINISH(DBDEF_GetReqdSubtitleParams);

   return(match_type);
}

/*!**************************************************************************
 * @brief   Returns the video pid and type that should be used for the given service
 *          from the list of video streams available. The "best one" is based on HD
 *          being better than SD and uses the priorities from the Nordig spec.
 * @param   s_ptr service
 * @param   video_type pointer used to return the type of the video selected
 * @return  Selected video PID
 ****************************************************************************/
U16BIT DBDEF_GetReqdVideoPid(ADB_SERVICE_REC *s_ptr, ADB_STREAM_TYPE *video_type)
{
   U16BIT video_pid;
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *best_stream;

   FUNCTION_START(DBDEF_GetReqdVideoPid);

   video_pid = 0;
   *video_type = ADB_VIDEO_STREAM;

   if (s_ptr != NULL)
   {
      best_stream = NULL;

      stream_rec = s_ptr->stream_list;
      while (stream_rec != NULL)
      {
         if (IsVideoStreamType(stream_rec->type))
         {
            /* Reset the in_use flag in case the selection changes */
            stream_rec->in_use = FALSE;

            if ((best_stream == NULL || stream_rec->type > best_stream->type))
            {
               /* Don't know how to differentiate between H265 UHD, H265 HD, H265 SD, H264 HD and H264 SD at the moment,
                * so always select this stream because it will be better than an MPEG2 stream */
               best_stream = stream_rec;
            }
         }

         stream_rec = stream_rec->next;
      }

      if (best_stream != NULL)
      {
         video_pid = best_stream->pid;
         *video_type = best_stream->type;
         best_stream->in_use = TRUE;
      }
   }

   FUNCTION_FINISH(DBDEF_GetReqdVideoPid);

   return(video_pid);
}

/**
 * @brief   Returns a pointer to the service name, taking into account perferred
 *          names and short names, if available
 * @param   s_ptr service
 * @param   short_name TRUE if short name is to be returned, if available
 * @param   pref_name TRUE if preferred name is to be returned, if available
 * @return  pointer to the service name
 */
ADB_STRING* DBDEF_GetServiceName(ADB_SERVICE_REC *s_ptr, BOOLEAN short_name, BOOLEAN pref_name)
{
   ADB_EVENT_DESC *event_desc;
   ADB_STRING *name_str;
   U8BIT *lang_ids;
   U8BIT pref_name_id;
   U8BIT i;

   FUNCTION_START(DBDEF_GetServiceName);

   name_str = NULL;

   if (s_ptr != NULL)
   {
      if (pref_name)
      {
         pref_name_id = 0;

         if (ACFG_GetCountry() == COUNTRY_CODE_AUSTRALIA)
         {
            /* Check to see if there's a preferred name ID descriptor included in the
             * now event. Although this is a user defined descriptor, it isn't protected
             * by a private data specifier */
            if ((s_ptr->now_event != NULL) &&
               ((event_desc = DBDEF_FindEventDescriptor(s_ptr->now_event->desc_list_head,
                  USER_DEFINED_DTAG_0x85, 0)) != NULL))
            {
               if (event_desc->desc_data[1] == 1)
               {
                  pref_name_id = event_desc->desc_data[2];
               }
            }
         }

         // find required language and preferred name id
         lang_ids = DBDEF_GetTextLang();

         if (lang_ids != NULL)
         {
            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                 (lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
            {
               name_str = s_ptr->name_array[lang_ids[i]][pref_name_id];
            }

            for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                 (lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
            {
               name_str = s_ptr->name_array[lang_ids[i]][0];
            }
         }
      }

      if (name_str == NULL)
      {
         if (short_name)
         {
            if (s_ptr->short_name_str != NULL)
            {
               name_str = s_ptr->short_name_str;
            }

            if (name_str == NULL)
            {
               /* Try to find a multilingual short name */
               lang_ids = DBDEF_GetTextLang();

               if (lang_ids != NULL)
               {
                  for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
                       (lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
                  {
                     name_str = s_ptr->short_name_array[lang_ids[i]];
                  }
               }
            }
         }

         if (name_str == NULL)
         {
            /* Default to the full service name */
            name_str = s_ptr->name_str;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_GetServiceName);

   return(name_str);
}

/**
 * @brief   Returns a pointer to the service provider name
 * @param   s_ptr service
 * @return  pointer to the service provider name
 */
ADB_STRING* DBDEF_GetServiceProviderName(ADB_SERVICE_REC *s_ptr)
{
   ADB_STRING *name_str;
   U8BIT *lang_ids;
   U8BIT i;

   FUNCTION_START(DBDEF_GetServiceProviderName);

   name_str = NULL;

   if (s_ptr != NULL)
   {
      lang_ids = DBDEF_GetTextLang();

      if (lang_ids != NULL)
      {
         for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) &&
              (lang_ids[i] != ACFG_INVALID_DB_LANG) && (name_str == NULL); i++)
         {
            name_str = s_ptr->provider_array[lang_ids[i]];
         }
      }

      if (name_str == NULL)
      {
         name_str = s_ptr->provider_str;
      }
   }

   FUNCTION_FINISH(DBDEF_GetServiceProviderName);

   return(name_str);
}

/**
 * @brief   Returns the service group mask
 * @param   s_ptr service
 * @return  the service group mask
 */
U32BIT DBDEF_GetServiceGroupMask(ADB_SERVICE_REC *s_ptr)
{
   U32BIT group_mask = 0;

   FUNCTION_START(DBDEF_GetServiceGroupMask);

   if (NULL != s_ptr)
       group_mask = s_ptr->group_mask;

   FUNCTION_FINISH(DBDEF_GetServiceGroupMask);
 
   return group_mask;
}

/**
 * @brief   Marks whether a service is deleted or not
 * @param   s_ptr service
 * @param   deleted TRUE to mark the service as deleted, FALSE as available
 */
void DBDEF_SetServiceDeletedFlag(ADB_SERVICE_REC *s_ptr, BOOLEAN deleted)
{
   FUNCTION_START(DBDEF_SetServiceDeletedFlag);

   s_ptr->deleted = deleted;
   DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_DELETED, s_ptr->deleted);

   FUNCTION_FINISH(DBDEF_SetServiceDeletedFlag);
}

/**
 * @brief   Returns the number of services that are marked as deleted for the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @return  number of services
 */
U16BIT DBDEF_GetNumDeletedServices(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   U16BIT num_services;
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetNumDeletedServices);

   serv_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
   for (num_services = 0; serv_ptr != NULL; )
   {
      if ((tuner_type == SIGNAL_NONE) ||
         ((serv_ptr->transport != NULL) && (serv_ptr->transport->sig_type == tuner_type)))
      {
         if (serv_ptr->deleted)
         {
            num_services++;
         }
      }

      serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNumDeletedServices);

   return(num_services);
}

/**
 * @brief
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @param   slist_ptr pointer to an array that will be allocated, value will be NULL on return if
 *                    there are no services. The list must be freed using STB_AppFreeMemory
 * @return  number of services in the returned list
 */
U16BIT DBDEF_GetDeletedServiceList(E_STB_DP_SIGNAL_TYPE tuner_type, void ***slist_ptr)
{
   U16BIT num_services;
   ADB_SERVICE_REC *serv_ptr;

   FUNCTION_START(DBDEF_GetDeletedServiceList);

   *slist_ptr = NULL;

   if ((num_services = DBDEF_GetNumDeletedServices(tuner_type)) != 0)
   {
      *slist_ptr = STB_AppGetMemory(num_services * sizeof(void *));
      if (*slist_ptr != NULL)
      {
         serv_ptr = (ADB_SERVICE_REC *)STB_LLGetFirstBlock(&service_rec_list);
         for (num_services = 0; serv_ptr != NULL; )
         {
            if ((tuner_type == SIGNAL_NONE) ||
               ((serv_ptr->transport != NULL) && (serv_ptr->transport->sig_type == tuner_type)))
            {
               if (serv_ptr->deleted)
               {
                  (*slist_ptr)[num_services] = serv_ptr;
                  num_services++;
               }
            }

            serv_ptr = (ADB_SERVICE_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv_ptr);
         }
      }
      else
      {
         num_services = 0;
      }
   }

   FUNCTION_FINISH(DBDEF_GetDeletedServiceList);

   return(num_services);
}

/**
 *

 *
 * @brief   Adds an analogue service, transport and, if necessary, network to the database
 *                based on the current tuning parameters in the decode path
 *

 *
 * @return   TRUE if new service record successfully created
 *
 */
BOOLEAN DBDEF_AddAnalogService(void)
{
   BOOLEAN retval;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SERVICE_REC *s_ptr;
   U32BIT freq_hz;
   U8BIT temp_buff[10];
   U16BIT chan_num;
   U8BIT name_len;

   FUNCTION_START(DBDEF_AddAnalogService);

   retval = FALSE;

   // if analogue network does not exist create it
   if (analog_network == NULL)
   {
      /* Add a new private network for use by the analog services */
      analog_network = DBDEF_FindOrAddPrivateNetwork(NULL);
   }

   if (analog_network != NULL)
   {
      /* Add transport to network - top bit set defines analog transport */
      freq_hz = STB_DPGetFrequency(0);
      t_ptr = DBDEF_AddTerrestrialTransportRec((freq_hz | 0x80000000L), 0, analog_network);
      if (t_ptr != NULL)
      {
         /* Setup rest of transport record */
         t_ptr->sig_type = SIGNAL_ANALOG;
         t_ptr->u.anal.freq_offset = STB_DPGetAnalogFreqOff(0);
         t_ptr->u.anal.vtype = STB_DPGetAnalogVideoType(0);
         t_ptr->tran_id = 0xffff;
         t_ptr->signal_level_at_search = BAD_SIGNAL_STATUS;

         // now add the new analogue service
         s_ptr = DBDEF_AddServiceRec(0, t_ptr);
         if (s_ptr != NULL)
         {
            s_ptr->serv_type = ADB_SERVICE_TYPE_ANALOG;
            DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

            // allocate channel number and name
            num_analog_channels++;
            chan_num = 900 + num_analog_channels;
            s_ptr->allocated_lcn = chan_num;
            DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN, s_ptr->allocated_lcn);

            name_len = snprintf((char *)temp_buff, 10, "%02d", num_analog_channels);
            s_ptr->name_str = DBDEF_MakeString(0, temp_buff, (name_len + 1));
            DBA_SetFieldString(s_ptr->dba_rec, DBA_FIELD_REC_NAME, s_ptr->name_str->str_ptr,
               s_ptr->name_str->nbytes);

            DBA_SaveRecord(s_ptr->dba_rec);

            // indicate success
            retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_AddAnalogService);
   return(retval);
}

/**
 *

 *
 * @brief   Updates analog service names - ASSUMES NORMAL ASCII CODED
 *
 * @param   s_ptr    - service to be changed
 * @param   new_name - pointer to new name
 * @param   new_len  - number of bytes in new name (incl null terminator)
 *

 *
 */
void DBDEF_SetAnalogServiceName(ADB_SERVICE_REC *s_ptr, U8BIT *new_name, U8BIT new_len)
{
   S32BIT tmp_s32bit;
   BOOLEAN changed;

   FUNCTION_START(DBDEF_SetAnalogServiceName);

   if (s_ptr != NULL)
   {
      // check if changed
      changed = TRUE;
      if ((s_ptr->name_str != NULL) && (new_name != NULL))
      {
         if (s_ptr->name_str->nbytes == new_len)
         {
            tmp_s32bit = memcmp(s_ptr->name_str->str_ptr, new_name, new_len);
            if (tmp_s32bit == 0)
            {
               changed = FALSE;
            }
         }
      }
      if (changed == TRUE)
      {
         // release old name
         if (s_ptr->name_str != NULL)
         {
            STB_AppFreeMemory(s_ptr->name_str);
            s_ptr->name_str = NULL;
         }
         // add new name
         if (new_name != NULL)
         {
            s_ptr->name_str = DBDEF_MakeString(0, new_name, new_len);
            DBA_SetFieldString(s_ptr->dba_rec, DBA_FIELD_REC_NAME, s_ptr->name_str->str_ptr,
               s_ptr->name_str->nbytes);
            DBA_SaveRecord(s_ptr->dba_rec);
         }
      }
   }

   FUNCTION_FINISH(DBDEF_SetAnalogServiceName);
}

/**
 *

 *
 * @brief   Updates the pmt pid
 *
 * @param   s_ptr    - service to be changed
 * @param   pmt_pid  - PID of the PMT associated with the service
 *

 *
 */
void DBDEF_SetServicePmtPid(ADB_SERVICE_REC *s_ptr, U16BIT pmt_pid)
{
   FUNCTION_START(DBDEF_SetServicePmtPid);
   if (s_ptr != NULL)
   {
      s_ptr->pmt_pid = pmt_pid;
   }
   FUNCTION_FINISH(DBDEF_SetServicePmtPid);
}

/**
 *

 *
 * @brief   gets the PMT PID associated with the service
 *
 * @param   s_ptr pointer to the service
 *
 * @return   PMT PID
 *
 */
U16BIT DBDEF_GetServicePmtPid(ADB_SERVICE_REC *s_ptr)
{
   U16BIT pmt_pid = 0;

   FUNCTION_START(DBDEF_GetServicePmtPid);

   if (s_ptr != NULL)
   {
      pmt_pid = s_ptr->pmt_pid;
   }

   FUNCTION_FINISH(DBDEF_GetServicePmtPid);

   return pmt_pid;
}

/**
 * @brief   Sets the favourite groups for a service
 * @param   s_ptr service
 * @param   groups bitmask containing the favourite groups the service is to be in
 */
void DBDEF_SetServiceFavGroups(ADB_SERVICE_REC *s_ptr, U8BIT groups)
{
   FUNCTION_START(DBDEF_SetServiceFavGroups);

   if (s_ptr != NULL)
   {
      s_ptr->fav_groups = groups;
      DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_FAV_GROUPS, s_ptr->fav_groups);
   }

   FUNCTION_FINISH(DBDEF_SetServiceFavGroups);
}

/**
 * @brief   Sort the full service list into ascending logical channel number order
 */
void DBDEF_SortServicesByLcn(void)
{
   FUNCTION_START(DBDEF_SortServicesByLcn);
   STB_LLSort(&service_rec_list, LcnSortCompareFunc);
   FUNCTION_FINISH(DBDEF_SortServicesByLcn);
}

/**
 *

 *
 * @brief   allocates lcns - expects allocated lcn for all services to be 0
 *
 * @param   assign_lcns - if set to FALSE the LCNs assigned to the services as a result
 *                              of running this function will be reset at the end.
 * @param   use_target_region - TRUE if the target region info is to be used when services
 *                                    with duplicate LCNs are encountered.
 *
 * @return   BOOLEAN - TRUE if more than one service requires the same LCN
 *
 */
BOOLEAN DBDEF_AllocateLcns(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN assign_lcns)
{
   U32BIT country_code;
   AllocLcnFunc func_ptr;
   BOOLEAN duplicate_lcn_found;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(DBDEF_AllocateLcns);

   /* Call the appropriate function for the country and type of tuner */
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_COFDM:
         func_ptr = ACFG_GetTerrestrialLcnFunction(country_code);
         break;

      case SIGNAL_QAM:
         func_ptr = ACFG_GetCableLcnFunction(country_code);
         break;

      case SIGNAL_QPSK:
         func_ptr = ACFG_GetSatelliteLcnFunction(country_code);
         break;

      default:
         func_ptr = NULL;
         break;
   }

   if (func_ptr != NULL)
   {
      duplicate_lcn_found = func_ptr(tuner_type);

      if (assign_lcns)
      {
         /* Now go through all services looking for allocated LCNs that have changed */
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s_ptr != NULL)
         {
            if (s_ptr->allocated_lcn != s_ptr->old_allocated_lcn)
            {
               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN, s_ptr->allocated_lcn);
               DBA_SaveRecord(s_ptr->dba_rec);
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }
      }
      else
      {
         /* Clear the LCN assigned to each service */
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s_ptr != NULL)
         {
            s_ptr->allocated_lcn = 0;
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }
      }
   }
   else
   {
      duplicate_lcn_found = FALSE;
   }

   FUNCTION_FINISH(DBDEF_AllocateLcns);

   return(duplicate_lcn_found);
}

BOOLEAN DBDEF_AllocateLcnsDefault(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT first_unallocated_lcn;
   U16BIT next_allocated_lcn;
   U16BIT last_allocated_lcn;
   U16BIT s1_strength;
   U16BIT s2_strength;
   BOOLEAN duplicate_lcn_found;

   FUNCTION_START(DBDEF_AllocateLcnsDefault);

   // determine next available special number and save last allocated number
   first_unallocated_lcn = ACFG_GetFirstUnallocatedLcn(ACFG_GetCountry(), tuner_type);
   next_allocated_lcn = first_unallocated_lcn;
   last_allocated_lcn = ACFG_GetLastUnallocatedLcn(ACFG_GetCountry(), tuner_type);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      s1_ptr->old_allocated_lcn = s1_ptr->allocated_lcn;
      if ((s1_ptr->allocated_lcn >= next_allocated_lcn) && (s1_ptr->allocated_lcn < last_allocated_lcn))
      {
         next_allocated_lcn = s1_ptr->allocated_lcn + 1;
      }
      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   duplicate_lcn_found = FALSE;

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      // if allocated_lcn has not already been set
      if (s1_ptr->serv_lcn == 0)
      {
         /* Only allocate an lcn if one hasn't already been allocated */
         if (s1_ptr->allocated_lcn == 0)
         {
            // invalid lcn - allocate one
            s1_ptr->allocated_lcn = next_allocated_lcn;
            next_allocated_lcn++;
         }
      }
      else if ((s1_ptr->allocated_lcn == 0) || (s1_ptr->allocated_lcn >= first_unallocated_lcn))
      {
         // valid lcn supplied - check for duplicates....

         //initialise allocated lcn for this service to the broadcast value - it may be changed
         // later
         s1_ptr->allocated_lcn = s1_ptr->serv_lcn;

         // find strength of this service
         s1_strength = 0;

         t_ptr = s1_ptr->transport;
         if (t_ptr != NULL)
         {
            s1_strength = t_ptr->signal_level_at_search;
         }

         // for all services in the list check for the same lcn number.
         // If a duplicate is found check the comparative strengths of the signal. The strongest
         // keeps the proper number, the weakest has a number allocated from the range defined
         // for this country onwards.
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            if ((s1_ptr->serv_lcn == s2_ptr->serv_lcn) && (s1_ptr != s2_ptr))
            {
               duplicate_lcn_found = TRUE;

               /* Decide which service should be assigned its desired LCN based on signal strength */
               s2_strength = 0;
               t_ptr = s2_ptr->transport;
               if (t_ptr != NULL)
               {
                  s2_strength = t_ptr->signal_level_at_search;
               }

               if (s1_strength < s2_strength)
               {
                  // s1 is weaker - change s1 lcn and break out of this loop, no need to look at
                  // more s2 services
                  s1_ptr->allocated_lcn = next_allocated_lcn;
                  next_allocated_lcn++;
                  break;
               }
               else
               {
                  // s1 is stronger, but has s2 already been allocated the proper lcn
                  if (s2_ptr->allocated_lcn == s2_ptr->serv_lcn)
                  {
                     if (s2_ptr->unavailable == FALSE)
                     {
                        // s2 has the lcn - for the moment leave it there and give s1 an allocated
                        // lcn. Later we will check whether to transfer the lcn and the favourites
                        // etc
                        s1_ptr->allocated_lcn = next_allocated_lcn;
                        next_allocated_lcn++;
                        break;
                     }
                     else
                     {
                        // s2 is unavailable so make it relinquish its lcn and allocate it a new
                        // one
                        s2_ptr->allocated_lcn = next_allocated_lcn;
                        next_allocated_lcn++;
                     }
                  }
               }
            }
            s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
         }
      }
      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   FUNCTION_FINISH(DBDEF_AllocateLcnsDefault);

   return(duplicate_lcn_found);
}

#ifdef COUNTRY_UK
BOOLEAN DBDEF_AllocateLcnsUK(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   ADB_SERVICE_REC *s1_ptr, *next_s1_ptr;
   ADB_SERVICE_REC *s2_ptr, *next_s2_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT next_allocated_lcn;
   U16BIT first_unallocated_lcn, reqd_lcn;
   U16BIT s1_strength;
   U16BIT s2_strength;
   BOOLEAN duplicate_lcn_found;
   BOOLEAN lcn_assigned, visible;
   S16BIT onid1, onid2;

   FUNCTION_START(DBDEF_AllocateLcnsUK);

   /* Get the first and next LCN to be allocated to a service that isn't assigned one */
   first_unallocated_lcn = ACFG_GetFirstUnallocatedLcn(ACFG_GetCountry(), tuner_type);
   next_allocated_lcn = first_unallocated_lcn;
   duplicate_lcn_found = FALSE;

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      next_s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      // if allocated_lcn has not already been set
      if (s1_ptr->serv_lcn == 0)
      {
         /* Only allocate an lcn if one hasn't already been allocated, or the service is using
          * an LCN in the unallocated range */
         if ((s1_ptr->allocated_lcn == 0) || (s1_ptr->allocated_lcn >= first_unallocated_lcn))
         {
            s1_ptr->allocated_lcn = next_allocated_lcn;
            next_allocated_lcn++;
         }
      }
      else if ((s1_ptr->allocated_lcn == 0) || (s1_ptr->allocated_lcn >= first_unallocated_lcn))
      {
         // valid lcn supplied - check for duplicates....

         //initialise allocated lcn for this service to the broadcast value - it may be changed
         // later
         s1_ptr->allocated_lcn = s1_ptr->serv_lcn;

         // find strength of this service
         s1_strength = 0;

         t_ptr = s1_ptr->transport;
         if (t_ptr != NULL)
         {
            s1_strength = t_ptr->signal_level_at_search;
         }

         // for all services in the list check for the same lcn number.
         // If a duplicate is found check the comparative strengths of the signal. The strongest
         // keeps the proper number, the weakest has a number allocated from the range defined
         // for this country onwards.
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            next_s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
            if ((s1_ptr->serv_lcn == s2_ptr->serv_lcn) && (s1_ptr != s2_ptr))
            {
               duplicate_lcn_found = TRUE;
               lcn_assigned = FALSE;

               if (tuner_type == SIGNAL_COFDM)
               {
                  if (!ApplyTargetRegionRules(s1_ptr, s2_ptr, &next_allocated_lcn, &lcn_assigned))
                  {
                     break;
                  }

                  if (lcn_assigned == TRUE)
                  {
                     onid1 = -1;
                     onid2 = -1;
                     if (s1_ptr->transport != NULL)
                     {
                        onid1 = s1_ptr->transport->orig_net_id;
                     }
                     if (s2_ptr->transport != NULL)
                     {
                        onid2 = s2_ptr->transport->orig_net_id;
                     }

                     if ((s1_ptr->serv_id == s2_ptr->serv_id) && (onid1 == onid2) && (onid1 != -1))
                     {
                        /* Services are identical, delete the service outside the user's region */
                        if (s1_ptr->allocated_lcn >= first_unallocated_lcn)
                        {
                           DBDEF_DeleteServiceRec(s1_ptr);
                        }
                        else if (s2_ptr->allocated_lcn >= first_unallocated_lcn)
                        {
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                     }
                  }
               }

               if (!lcn_assigned)
               {
                  /* Decide which service should be assigned its desired LCN based on signal strength */
                  s2_strength = 0;
                  t_ptr = s2_ptr->transport;
                  if (t_ptr != NULL)
                  {
                     s2_strength = t_ptr->signal_level_at_search;
                  }

                  if (s1_strength < s2_strength)
                  {
                     // s1 is weaker - change s1 lcn and break out of this loop, no need to look at
                     // more s2 services
                     s1_ptr->allocated_lcn = next_allocated_lcn;
                     next_allocated_lcn++;
                     break;
                  }
                  else
                  {
                     // s1 is stronger, but has s2 already been allocated the proper lcn
                     if (s2_ptr->allocated_lcn == s2_ptr->serv_lcn)
                     {
                        if (s2_ptr->unavailable == FALSE)
                        {
                           // s2 has the lcn - for the moment leave it there and give s1 an allocated
                           // lcn. Later we will check whether to transfer the lcn and the favourites
                           // etc
                           s1_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                           break;
                        }
                        else
                        {
                           // s2 is unavailable so make it relinquish its lcn and allocate it a new
                           // one
                           s2_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                        }
                     }
                  }
               }
            }
            s2_ptr = next_s2_ptr;
         }
      }
      s1_ptr = next_s1_ptr;
   }

   /* Now apply the HD LCN descriptor, if present and applicable */
   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      if ((s1_ptr->allocated_lcn < first_unallocated_lcn) && (s1_ptr->hd_lcn_desc != NULL))
      {
         reqd_lcn = s1_ptr->hd_lcn_desc->serv_lcn;
         if (reqd_lcn != s1_ptr->serv_lcn)
         {
            s2_ptr = DBDEF_GetNextServiceRec(NULL);
            while (s2_ptr != NULL)
            {
               if (reqd_lcn == s2_ptr->serv_lcn)
               {
                  /* This service is being replaced in the LCN lineup. Give this service
                   * the LCN originally assigned to the one replacing it. */
                  s2_ptr->old_allocated_lcn = s2_ptr->allocated_lcn;
                  s2_ptr->allocated_lcn = s1_ptr->allocated_lcn;
                  break;
               }
               s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
            }

            s1_ptr->old_allocated_lcn = s1_ptr->allocated_lcn;
            s1_ptr->allocated_lcn = reqd_lcn;
         }

         /* See if the service's attributes should be changed */
         visible = !s1_ptr->hidden;
         if (s1_ptr->hd_lcn_desc->visible != visible)
         {
            if (s1_ptr->hd_lcn_desc->visible)
            {
               s1_ptr->hidden = FALSE;
               DBA_SetFieldValue(s1_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, FALSE);
            }
            else
            {
               s1_ptr->hidden = TRUE;
               DBA_SetFieldValue(s1_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, TRUE);
            }
         }

         /* The description for the HD simulcast LCN descriptor in the DBook states that
          * whether the service is visible or hidden it is always selectable */
         s1_ptr->selectable = TRUE;
         DBA_SetFieldValue(s1_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, TRUE);

         DBA_SaveRecord(s1_ptr->dba_rec);
      }
      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   FUNCTION_FINISH(DBDEF_AllocateLcnsUK);

   return(duplicate_lcn_found);
}

#endif /* COUNTRY_UK */

#ifdef COUNTRY_FINLAND
BOOLEAN DBDEF_AllocateLcnsFinland(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT next_allocated_lcn;
   U16BIT last_allocated_lcn;
   U16BIT s1_strength;
   U16BIT s2_strength;
   BOOLEAN duplicate_lcn_found;
   BOOLEAN lcn_assigned;

   FUNCTION_START(DBDEF_AllocateLcnsFinland);

   /* to find the first special lcn number, for Nordig, it sets to the number after the largest lcn of
      the services transmitted by its own country's orignal network */
   next_allocated_lcn = FindLargestLcnNumber(tuner_type, FINLAND_ONID) + 1;
   last_allocated_lcn = ACFG_GetLastUnallocatedLcn(ACFG_GetCountry(), tuner_type);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      s1_ptr->old_allocated_lcn = s1_ptr->allocated_lcn;

      /* if the channels got from other original network which is outside the country, reset the lcn
         so that the lcn will not keep increasing on manual tune or update  */
      if (s1_ptr->transport->orig_net_id != FINLAND_ONID)
      {
         s1_ptr->allocated_lcn = 0;
      }
      if ((s1_ptr->allocated_lcn >= next_allocated_lcn) && (s1_ptr->allocated_lcn < last_allocated_lcn))
      {
         next_allocated_lcn = s1_ptr->allocated_lcn + 1;
      }
      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   duplicate_lcn_found = FALSE;

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      // if allocated_lcn has not already been set
      if (s1_ptr->serv_lcn == 0)
      {
         /* Allocate the next available LCN to the service */
         s1_ptr->allocated_lcn = next_allocated_lcn;
         next_allocated_lcn++;
      }
      else if (s1_ptr->allocated_lcn == 0)
      {
         // valid lcn supplied - check for duplicates....

         //initialise allocated lcn for this service to the broadcast value - it may be changed
         // later
         s1_ptr->allocated_lcn = s1_ptr->serv_lcn;

         // find strength of this service
         s1_strength = 0;
         t_ptr = s1_ptr->transport;

         if (t_ptr != NULL)
         {
            if (tuner_type == SIGNAL_COFDM)
            {
               /* If the orginal network id is not for Finland when the country was set to Finland
                * it means that the service we got is from another country's network.
                * Set the lcn of the service to next allocated lcn */
               if (t_ptr->orig_net_id != FINLAND_ONID)
               {
                  s1_ptr->allocated_lcn = next_allocated_lcn;
                  next_allocated_lcn++;
               }
            }

            s1_strength = t_ptr->signal_level_at_search;
         }

         // for all services in the list check for the same lcn number.
         // If a duplicate is found check the comparative strengths of the signal. The strongest
         // keeps the proper number, the weakest has a number allocated from the range defined
         // for this country onwards.
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            if ((s1_ptr->serv_lcn == s2_ptr->serv_lcn) && (s1_ptr != s2_ptr))
            {
               /* Two services from different networks are allowed to have the same LCN */
               if ((s1_ptr->transport->orig_net_id) == (s2_ptr->transport->orig_net_id))
               {
                  duplicate_lcn_found = TRUE;
                  lcn_assigned = FALSE;

                  /* Check for a HD service on service 2 */
                  /* for Nordig, if the lcn number of two services are identical and the service type is different,
                     then each service should get its signaled lcn. */
                  switch (s2_ptr->serv_type)
                  {
                     case ADB_SERVICE_TYPE_AVC_SD_TV:
                     case ADB_SERVICE_TYPE_HD_TV:
                     case ADB_SERVICE_TYPE_MPEG2_HD:
                     case ADB_SERVICE_TYPE_UHD_TV:
                     {
                        s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                        /* If there is and service 1 is an SD then allocate service 1 at next allocated lcn */
                        if (s1_ptr->serv_type == ADB_SERVICE_TYPE_TV)
                        {
                           s1_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                           lcn_assigned = TRUE;
                        }
                        else if ((s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_RADIO) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_DATA))
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           lcn_assigned = TRUE;
                        }
                        break;
                     }
                     case ADB_SERVICE_TYPE_AVC_RADIO:
                     {
                        s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                        if (s1_ptr->serv_type == ADB_SERVICE_TYPE_RADIO)
                        {
                           s1_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                           lcn_assigned = TRUE;
                        }
                        else if (s1_ptr->serv_type == ADB_SERVICE_TYPE_DATA)
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           lcn_assigned = TRUE;
                        }
                        break;
                     }
                     case ADB_SERVICE_TYPE_TV:
                     {
                        if ((s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_RADIO) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_DATA))
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                           lcn_assigned = TRUE;
                        }
                        else if ((s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_UHD_TV))
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           s2_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                           lcn_assigned = TRUE;
                        }
                        break;
                     }
                     case ADB_SERVICE_TYPE_RADIO:
                     {
                        if ((s1_ptr->serv_type == ADB_SERVICE_TYPE_TV) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_DATA) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
                            (s1_ptr->serv_type == ADB_SERVICE_TYPE_UHD_TV))
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                           lcn_assigned = TRUE;
                        }
                        else if (s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO)
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           s2_ptr->allocated_lcn = next_allocated_lcn;
                           next_allocated_lcn++;
                           lcn_assigned = TRUE;
                        }
                        break;
                     }
                     default:
                     {
                        if (s1_ptr->serv_type != ADB_SERVICE_TYPE_DATA)
                        {
                           s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
                           s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                           lcn_assigned = TRUE;
                        }
                        break;
                     }
                  }

                  if (!lcn_assigned)
                  {
                     /* Decide which service should be assigned its desired LCN based on signal strength */
                     s2_strength = 0;
                     t_ptr = s2_ptr->transport;
                     if (t_ptr != NULL)
                     {
                        s2_strength = t_ptr->signal_level_at_search;
                     }

                     if (s1_strength < s2_strength)
                     {
                        // s1 is weaker - change s1 lcn and break out of this loop, no need to look at
                        // more s2 services
                        s1_ptr->allocated_lcn = next_allocated_lcn;
                        next_allocated_lcn++;
                        break;
                     }
                     else
                     {
                        // s1 is stronger, but has s2 already been allocated the proper lcn
                        if (s2_ptr->allocated_lcn == s2_ptr->serv_lcn)
                        {
                           if (s2_ptr->unavailable == FALSE)
                           {
                              // s2 has the lcn - for the moment leave it there and give s1 an allocated
                              // lcn. Later we will check whether to transfer the lcn and the favourites
                              // etc
                              s1_ptr->allocated_lcn = next_allocated_lcn;
                              next_allocated_lcn++;
                              break;
                           }
                           else
                           {
                              // s2 is unavailable so make it relinquish its lcn and allocate it a new
                              // one
                              s2_ptr->allocated_lcn = next_allocated_lcn;
                              next_allocated_lcn++;
                           }
                        }
                     }
                  }
               }
            }
            s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   FUNCTION_FINISH(DBDEF_AllocateLcnsFinland);

   return(duplicate_lcn_found);
}

#endif /* COUNTRY_FINLAND */

#ifdef COUNTRY_IRELAND
BOOLEAN DBDEF_AllocateLcnsIreland(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT next_allocated_lcn;
   U16BIT last_allocated_lcn;
   U16BIT s1_strength;
   U16BIT s2_strength;
   BOOLEAN duplicate_lcn_found;
   BOOLEAN lcn_assigned;

   FUNCTION_START(DBDEF_AllocateLcnsIreland);

   /* Find the current largest LCN assigned to services on the Irish network, which will then be
    * used as the first LCN to assign to services without an LCN or are on a different network */
   next_allocated_lcn = FindLargestLcnNumber(tuner_type, IRELAND_ONID) + 1;
   last_allocated_lcn = ACFG_GetLastUnallocatedLcn(ACFG_GetCountry(), tuner_type);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      if ((s1_ptr->transport != NULL) && (s1_ptr->transport->sig_type == tuner_type))
      {
         s1_ptr->old_allocated_lcn = s1_ptr->allocated_lcn;

         /* If the service isn't from the Irish network then reset its allocated LCN
          * so it can be reassigned */
         if (s1_ptr->transport->orig_net_id != IRELAND_ONID)
         {
            s1_ptr->allocated_lcn = 0;
         }

         if ((s1_ptr->allocated_lcn >= next_allocated_lcn) && (s1_ptr->allocated_lcn < last_allocated_lcn))
         {
            next_allocated_lcn = s1_ptr->allocated_lcn + 1;
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   /* The Irish services must be assigned their requested LCNs, with services from other networks,
    * which in this case will be the UK, being assigned LCNs above the largest used Irish LCN, so
    * all Irish services are assigned LCNs first. */
   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      t_ptr = s1_ptr->transport;
      if ((t_ptr != NULL) && (t_ptr->sig_type == tuner_type))
      {
         if ((s1_ptr->allocated_lcn == 0) && (s1_ptr->selectable || !s1_ptr->hidden))
         {
            s1_strength = 0;

            if (t_ptr->orig_net_id == IRELAND_ONID)
            {
               /* Set the LCN it wants, but this may be changed */
               if (s1_ptr->serv_lcn != 0)
               {
                  s1_ptr->allocated_lcn = s1_ptr->serv_lcn;
               }
               else
               {
                  s1_ptr->allocated_lcn = next_allocated_lcn;
                  next_allocated_lcn++;
               }

               s1_strength = t_ptr->signal_level_at_search;

               /* Check for duplicate LCNs for Irish services, and if one is found then the assigned
                * LCN is based on the service type followed by the signal strength of the two services */
               s2_ptr = DBDEF_GetNextServiceRec(s1_ptr);
               while (s2_ptr != NULL)
               {
                  t_ptr = s2_ptr->transport;

                  if ((t_ptr != NULL) && (t_ptr->sig_type == tuner_type))
                  {
                     if ((s1_ptr->serv_lcn == s2_ptr->serv_lcn) &&
                        (t_ptr->orig_net_id == IRELAND_ONID) &&
                        (s1_ptr->transport->orig_net_id) == (t_ptr->orig_net_id))
                     {
                        duplicate_lcn_found = TRUE;
                        lcn_assigned = FALSE;

                        /* For LCN clashes, advanced codec services take priority over SD quality services */
                        switch (s2_ptr->serv_type)
                        {
                           case ADB_SERVICE_TYPE_AVC_SD_TV:
                           case ADB_SERVICE_TYPE_HD_TV:
                           case ADB_SERVICE_TYPE_MPEG2_HD:
                           case ADB_SERVICE_TYPE_UHD_TV:
                           {
                              /* If the first service is also an advanced codec service then the
                               * service that gets the LCN will be based on signal strength */
                              if ((s1_ptr->serv_type != ADB_SERVICE_TYPE_AVC_SD_TV) &&
                                 (s1_ptr->serv_type != ADB_SERVICE_TYPE_HD_TV) &&
                                 (s1_ptr->serv_type != ADB_SERVICE_TYPE_MPEG2_HD) &&
                                 (s1_ptr->serv_type != ADB_SERVICE_TYPE_UHD_TV))
                              {
                                 /* Advanced codec service takes the LCN,
                                  * so assign a new one to the first service */
                                 s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                                 s1_ptr->allocated_lcn = next_allocated_lcn;
                                 next_allocated_lcn++;
                                 lcn_assigned = TRUE;
                              }
                              break;
                           }
                           case ADB_SERVICE_TYPE_AVC_RADIO:
                           {
                              if (s1_ptr->serv_type != ADB_SERVICE_TYPE_AVC_RADIO)
                              {
                                 s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                                 s1_ptr->allocated_lcn = next_allocated_lcn;
                                 next_allocated_lcn++;
                                 lcn_assigned = TRUE;
                              }
                              break;
                           }
                           case ADB_SERVICE_TYPE_TV:
                           {
                              if ((s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD) ||
                                 (s1_ptr->serv_type == ADB_SERVICE_TYPE_UHD_TV))
                              {
                                 /* First service keeps the LCN because it's an advanced codec service */
                                 s2_ptr->allocated_lcn = next_allocated_lcn;
                                 next_allocated_lcn++;
                                 lcn_assigned = TRUE;
                              }
                              break;
                           }
                           case ADB_SERVICE_TYPE_RADIO:
                           {
                              if (s1_ptr->serv_type == ADB_SERVICE_TYPE_AVC_RADIO)
                              {
                                 /* First service keeps the LCN because it's an advanced codec service */
                                 s2_ptr->allocated_lcn = next_allocated_lcn;
                                 next_allocated_lcn++;
                                 lcn_assigned = TRUE;
                              }
                              break;
                           }
                           default:
                           {
                              break;
                           }
                        }

                        if (!lcn_assigned)
                        {
                           /* Decide which service should be assigned its desired LCN based on signal strength */
                           s2_strength = t_ptr->signal_level_at_search;

                           if (s1_strength < s2_strength)
                           {
                              /* s1 is weaker so assign the LCN to s2 */
                              s2_ptr->allocated_lcn = s2_ptr->serv_lcn;
                              s1_ptr->allocated_lcn = next_allocated_lcn;
                              next_allocated_lcn++;
                           }
                           else
                           {
                              /* s1 is stronger so assign an LCN to s2 */
                              s2_ptr->allocated_lcn = next_allocated_lcn;
                              next_allocated_lcn++;
                           }
                        }
                     }
                  }

                  s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
               }
            }
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   /* The next allocated LCN will be the next one to be used, but this will now be used as an
    * offset, so set it to the highest assigned LCN, which is one less than the current value */
   if (next_allocated_lcn != 0)
   {
      next_allocated_lcn--;
   }

   /* Now assigned LCNs to all non-Irish services. It is recommended to keep the relative LCN
    * order of services on other networks, so all services that have requested an LCN are assigned
    * first, followed by all those that haven't, which ensures that these services end up at the
    * end of the list */
   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      t_ptr = s1_ptr->transport;
      if ((t_ptr != NULL) && (t_ptr->sig_type == tuner_type))
      {
         if ((s1_ptr->serv_lcn != 0) && (s1_ptr->allocated_lcn == 0))
         {
            if (t_ptr->orig_net_id != IRELAND_ONID)
            {
               /* Check that this LCN hasn't already been assigned to a service */
               if (DBDEF_FindServiceRecByLcn(s1_ptr->serv_lcn + next_allocated_lcn, NULL, TRUE) == NULL)
               {
                  s1_ptr->allocated_lcn = s1_ptr->serv_lcn + next_allocated_lcn;

                  s1_strength = t_ptr->signal_level_at_search;

                  /* Check for duplicate LCNs and if one is found then check the signal strength of the two
                   * services. The strongest keeps the LCN and the weakest is allocated another LCN */
                  s2_ptr = DBDEF_GetNextServiceRec(s1_ptr);
                  while (s2_ptr != NULL)
                  {
                     if ((s2_ptr->serv_lcn != 0) && (s1_ptr->serv_lcn == s2_ptr->serv_lcn) &&
                        (s2_ptr->selectable || !s2_ptr->hidden) &&
                        (s2_ptr->transport != NULL) &&
                        (s1_ptr->transport->orig_net_id) == (s2_ptr->transport->orig_net_id))
                     {
                        duplicate_lcn_found = TRUE;

                        /* Decide which service should be assigned its desired LCN based on signal strength */
                        s2_strength = s2_ptr->transport->signal_level_at_search;

                        if (s1_strength < s2_strength)
                        {
                           /* s2 has a stronger signal so assign it the LCN and clear the LCN from s1
                            * so it will be assigned an LCN later */
                           s2_ptr->allocated_lcn = s1_ptr->allocated_lcn;
                           s1_ptr->allocated_lcn = 0;
                        }
                     }

                     s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
                  }
               }
            }
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   /* Find the largest allocated LCN, which will be used for all remaining services */
   next_allocated_lcn = 0;

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      if ((s1_ptr->transport != NULL) && (s1_ptr->transport->sig_type == tuner_type))
      {
         if (s1_ptr->allocated_lcn >= next_allocated_lcn)
         {
            next_allocated_lcn = s1_ptr->allocated_lcn + 1;
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   /* Now assign LCNs to all services that haven't already been assigned one */
   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      if ((s1_ptr->transport != NULL) && (s1_ptr->transport->sig_type == tuner_type))
      {
         /* If allocated_lcn has not already been set */
         if ((s1_ptr->allocated_lcn == 0) && (s1_ptr->selectable || !s1_ptr->hidden))
         {
            /* Allocate the next available LCN to the service */
            s1_ptr->allocated_lcn = next_allocated_lcn;
            next_allocated_lcn++;
         }
      }

      s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
   }

   FUNCTION_FINISH(DBDEF_AllocateLcnsIreland);

   return(duplicate_lcn_found);
}
#endif /* COUNTRY_IRELAND */

/**
 *

 *
 * @brief   sets the currently tuned network
 *
 * @param   t_ptr - pointer to the network
 *

 *
 */
void DBDEF_SetTunedNetwork(U8BIT path, ADB_NETWORK_REC *n_ptr)
{
   FUNCTION_START(DBDEF_SetTunedNetwork);
   ASSERT(path < num_paths);
   tuned_network[path] = n_ptr;
   FUNCTION_FINISH(DBDEF_SetTunedNetwork);
}

/**
 *

 *
 * @brief   gets the currently tuned network
 *

 *
 * @return   pointer to the network
 *
 */
ADB_NETWORK_REC* DBDEF_GetTunedNetwork(U8BIT path)
{
   FUNCTION_START(DBDEF_GetTunedNetwork);
   ASSERT(path < num_paths);
   FUNCTION_FINISH(DBDEF_GetTunedNetwork);
   return(tuned_network[path]);
}

/**
 *

 *
 * @brief   sets the currently tuned transport
 *
 * @param   t_ptr - pointer to the transport
 *

 *
 */
void DBDEF_SetTunedTransport(U8BIT path, ADB_TRANSPORT_REC *t_ptr)
{
   FUNCTION_START(DBDEF_SetTunedTransport);
   ASSERT(path < num_paths);
   STB_DPSetTunedTransport(path, t_ptr);
   FUNCTION_FINISH(DBDEF_SetTunedTransport);
}

/**
 *

 *
 * @brief   gets the currently tuned transport
 *

 *
 * @return   pointer to the transport
 *
 */
ADB_TRANSPORT_REC* DBDEF_GetTunedTransport(U8BIT path)
{
   FUNCTION_START(DBDEF_GetTunedTransport);
   ASSERT(path < num_paths);
   FUNCTION_FINISH(DBDEF_GetTunedTransport);
   return(STB_DPGetTunedTransport(path));
}

/*!**************************************************************************
 * @brief   Sets the currently tuned service
 * @param   path decode path
 * @param   s_ptr service tuned to
 ****************************************************************************/
void DBDEF_SetTunedService(U8BIT path, ADB_SERVICE_REC *s_ptr)
{
   FUNCTION_START(DBDEF_SetTunedService);

   ASSERT(path < num_paths);
   if (path < num_paths)
   {
      STB_DPSetTunedService(path, s_ptr);
   }

   FUNCTION_FINISH(DBDEF_SetTunedService);
}

/*!**************************************************************************
 * @brief   Returns the currently tuned service on the given path
 * @param   path decode path
 * @return  service pointer
 ****************************************************************************/
ADB_SERVICE_REC* DBDEF_GetTunedService(U8BIT path)
{
   void *s_ptr;

   FUNCTION_START(DBDEF_GetTunedService);

   ASSERT(path < num_paths);

   s_ptr = NULL;

   if (path < num_paths)
   {
      s_ptr = STB_DPGetTunedService(path);
   }

   FUNCTION_FINISH(DBDEF_GetTunedService);

   return(s_ptr);
}

/**
 *

 *
 * @brief   sets current text lang
 *
 * @param   lang_ids - array of required language ids
 *

 *
 */
void DBDEF_SetTextLang(U8BIT *lang_ids)
{
   FUNCTION_START(DBDEF_SetTextLang);
   text_langs = lang_ids;
   FUNCTION_FINISH(DBDEF_SetTextLang);
}

/**
 *

 *
 * @brief   returns array of current text langs
 *

 *
 * @return   lang ids
 *
 */
U8BIT* DBDEF_GetTextLang(void)
{
   FUNCTION_START(DBDEF_GetTextLang);
   FUNCTION_FINISH(DBDEF_GetTextLang);
   return(text_langs);
}

/**
 *

 *
 * @brief   sets secondary text lang
 *
 * @param   lang_ids - array of required language ids
 *

 *
 */
void DBDEF_SetSecondaryTextLang(U8BIT *lang_ids)
{
   FUNCTION_START(DBDEF_SetSecondaryTextLang);
   second_text_langs = lang_ids;
   FUNCTION_FINISH(DBDEF_SetSecondaryTextLang);
}

/**
 *

 *
 * @brief   returns array of secondary text langs
 *

 *
 * @return   lang ids
 *
 */
U8BIT* DBDEF_GetSecondaryTextLang(void)
{
   FUNCTION_START(DBDEF_GetSecondaryTextLang);
   FUNCTION_FINISH(DBDEF_GetSecondaryTextLang);
   return(second_text_langs);
}

/**
 *

 *
 * @brief   sets current audio lang
 *
 * @param   lang_id - id of required language
 *

 *
 */
void DBDEF_SetAudioLang(U8BIT *lang_ids)
{
   FUNCTION_START(DBDEF_SetAudioLang);
   audio_langs = lang_ids;
   FUNCTION_FINISH(DBDEF_SetAudioLang);
}

/**
 *

 *
 * @brief   sets the secondary audio lang
 *
 * @param   lang_ids - ids of required language
 *

 *
 */
void DBDEF_SetSecondaryAudioLang(U8BIT *lang_ids)
{
   FUNCTION_START(DBDEF_SetSecondaryAudioLang);
   second_audio_langs = lang_ids;
   FUNCTION_FINISH(DBDEF_SetSecondaryAudioLang);
}

/**
 *

 *
 * @brief   returns current audio lang
 *

 *
 * @return   lang id
 *
 */
U8BIT* DBDEF_GetAudioLang(void)
{
   FUNCTION_START(DBDEF_GetAudioLang);
   FUNCTION_FINISH(DBDEF_GetAudioLang);
   return(audio_langs);
}

/**
 *

 *
 * @brief   returns current secondary audio lang
 *

 *
 * @return   lang id
 *
 */
U8BIT* DBDEF_GetSecondaryAudioLang(void)
{
   FUNCTION_START(DBDEF_GetSecondaryAudioLang);
   FUNCTION_FINISH(DBDEF_GetSecondaryAudioLang);
   return(second_audio_langs);
}

/**
 * @brief   Calls any country and tuner type specific function to tidy up the database following
 *          a search, such as removing duplicate services, etc.
 * @param   tuner_type tuner type used for the search
 * @param   satellite satellite search was performed on for SIGNAL_QPSK, ignored otherwise
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseAfterSearch(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN search_completed, BOOLEAN manual)
{
   U32BIT country_code;
   DBTidyFunc func_ptr;
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *next_s_ptr;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(DBDEF_TidyDatabaseAfterSearch);

   /* Call the appropriate function for the country and type of tuner */
   country_code = ACFG_GetCountry();

   switch (tuner_type)
   {
      case SIGNAL_COFDM:
         func_ptr = ACFG_GetTerrestrialDBTidyFunction(country_code);
         break;

      case SIGNAL_QAM:
         func_ptr = ACFG_GetCableDBTidyFunction(country_code);
         break;

      case SIGNAL_QPSK:
         func_ptr = ACFG_GetSatelliteDBTidyFunction(country_code);
         break;

      default:
         func_ptr = NULL;
         break;
   }

   if (func_ptr != NULL)
   {
      func_ptr(tuner_type, search_completed, manual);
   }

   /* Remove any services that were added (e.g. as a result of processing a BAT), but
    * weren't actually found due to the transport not being found or weren't in an SDT */
   s_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s_ptr != NULL)
   {
      next_s_ptr = DBDEF_GetNextServiceRec(s_ptr);

      if (!s_ptr->found)
      {
         DBDEF_DeleteServiceRec(s_ptr);
      }

      s_ptr = next_s_ptr;
   }

   DBDEF_RemoveEmptyTransports(tuner_type, satellite);

   if (!search_completed)
   {
      /* Mark all transports as available */
      t_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t_ptr != NULL)
      {
         t_ptr->available = TRUE;
         t_ptr = DBDEF_GetNextTransportRec(t_ptr);
      }
   }

   FUNCTION_FINISH(DBDEF_TidyDatabaseAfterSearch);
}

/**
 * @brief   Iterates through all the services following a search search to see whether any need
 *          to be deleted based on the rules for that country
 * @param   tuner_type tuner type used for the search
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseNordig(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_SERVICE_REC *next1_ptr;
   ADB_SERVICE_REC *next2_ptr;
   ADB_TRANSPORT_REC *t1_ptr;
   ADB_TRANSPORT_REC *t2_ptr;
   BOOLEAN serv_removed;

   FUNCTION_START(DBDEF_TidyDatabaseNordig);
   USE_UNWANTED_PARAM(tuner_type);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      serv_removed = FALSE;
      next1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      if ((s1_ptr->transport != NULL) && (s1_ptr->transport->network != NULL))
      {
         if (DBDEF_NetworkInProfile(s1_ptr->transport->network) &&
             ((s1_ptr->transport->network->net_id >= NID_PRIVATE_BOUNDARY) ||
              (s1_ptr->transport->orig_net_id >= ONID_PRIVATE_BOUNDARY)))
         {
            DBDEF_DeleteServiceRec(s1_ptr);
            serv_removed = TRUE;
         }
      }

      if (!serv_removed && s1_ptr->unavailable)
      {
         DBDEF_DeleteServiceRec(s1_ptr);
         serv_removed = TRUE;
      }

      /* Remove services contained by the transport which haven't been found during the search */
      if (!serv_removed && !manual && (s1_ptr->transport != NULL) &&
            (s1_ptr->transport->available == FALSE))
      {
         DBDEF_DeleteServiceRec(s1_ptr);
      }

      s1_ptr = next1_ptr;
   }

   /* Remove deuplicate services if they are not matching lcns */
   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      t1_ptr = s1_ptr->transport;
      next1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      if ((t1_ptr != NULL) && DBDEF_TransportInProfile(t1_ptr))
      {
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            next2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
            if ((s1_ptr != s2_ptr) && (s1_ptr->serv_id == s2_ptr->serv_id))
            {
               t2_ptr = s2_ptr->transport;
               if ((t2_ptr != NULL) && DBDEF_TransportInProfile(t2_ptr))
               {
                  if ((t1_ptr->orig_net_id == t2_ptr->orig_net_id) &&
                      (t1_ptr->tran_id == t2_ptr->tran_id))
                  {
                     if (manual == TRUE)
                     {
                        if (s1_ptr->new_service == TRUE)
                        {
                           if (s2_ptr == next1_ptr)
                           {
                              next1_ptr = DBDEF_GetNextServiceRec(next1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else if (s2_ptr->new_service == TRUE)
                        {
                           DBDEF_DeleteServiceRec(s1_ptr);
                           break;
                        }
                     }
                     else
                     {
                        /* Services are the same; choose the strongest one */
                        if (t1_ptr->signal_level_at_search >= t2_ptr->signal_level_at_search)
                        {
                           if (s2_ptr->serv_lcn == s1_ptr->serv_lcn)
                           {
                              if (s2_ptr == next1_ptr)
                              {
                                 next1_ptr = DBDEF_GetNextServiceRec(next1_ptr);
                              }
                              /* Delete the second service */
                              DBDEF_DeleteServiceRec(s2_ptr);
                           }
                        }
                        else
                        {
                           if (s2_ptr->serv_lcn == s1_ptr->serv_lcn)
                           {
                              /* Delete the first service */
                              DBDEF_DeleteServiceRec(s1_ptr);
                           }
                           break;
                        }
                     }
                  }
               }
            }
            s2_ptr = next2_ptr;
         }
      }
      s1_ptr = next1_ptr;
   }

   if (search_completed)
   {
      /* Remove any transports that haven't been found during the search */
      t1_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t1_ptr != NULL)
      {
         t2_ptr = DBDEF_GetNextTransportRec(t1_ptr);

         if (!t1_ptr->available)
         {
            DBDEF_DeleteTransportRec(t1_ptr);
         }

         t1_ptr = t2_ptr;
      }
   }

   FUNCTION_FINISH(DBDEF_TidyDatabaseNordig);
}

/**
 * @brief   Iterates through all the services following a search search to see whether any need
 *          to be deleted based on the rules for that country
 * @param   tuner_type tuner type used for the search
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseUK(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *next_s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_SERVICE_REC *next_s2_ptr;
   ADB_TRANSPORT_REC *t1_ptr;
   ADB_TRANSPORT_REC *t2_ptr;

   FUNCTION_START(DBDEF_TidyDatabaseUK);
   USE_UNWANTED_PARAM(tuner_type);
   USE_UNWANTED_PARAM(search_completed);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      next_s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      t1_ptr = s1_ptr->transport;
      if ((t1_ptr != NULL) && DBDEF_TransportInProfile(t1_ptr))
      {
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            next_s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
            if ((s1_ptr != s2_ptr) && (s1_ptr->serv_id == s2_ptr->serv_id))
            {
               t2_ptr = s2_ptr->transport;
               if ((t2_ptr != NULL) && DBDEF_TransportInProfile(t2_ptr))
               {
                  if (t1_ptr->orig_net_id == t2_ptr->orig_net_id)
                  {
                     if (manual == TRUE)
                     {
                        if (s1_ptr->new_service == TRUE)
                        {
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else if (s2_ptr->new_service == TRUE)
                        {
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                     else
                     {
                        /* Services are the same; choose the strongest one */
                        if (t1_ptr->signal_level_at_search >= t2_ptr->signal_level_at_search)
                        {
                           /* Delete the second service */
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else
                        {
                           /* Delete the first service */
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                  }
               }
            }

            s2_ptr = next_s2_ptr;
         }
      }
      s1_ptr = next_s1_ptr;
   }

   FUNCTION_FINISH(DBDEF_TidyDatabaseUK);
}

/**
 * @brief   Iterates through all the services following a search search to see whether any need
 *          to be deleted based on the rules for that country
 * @param   tuner_type tuner type used for the search
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseSatUK(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *next_s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_SERVICE_REC *next_s2_ptr;
   ADB_TRANSPORT_REC *t1_ptr;
   ADB_TRANSPORT_REC *t2_ptr;

   FUNCTION_START(DBDEF_TidyDatabaseSatUK);
   USE_UNWANTED_PARAM(tuner_type);
   USE_UNWANTED_PARAM(search_completed);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      next_s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      t1_ptr = s1_ptr->transport;
      if ((t1_ptr != NULL) && DBDEF_TransportInProfile(t1_ptr))
      {
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            next_s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);

            /* Freesat allows identical services, so only want to delete a service if neither
             * of them are Freesat services */
            if ((s1_ptr != s2_ptr) && (s1_ptr->serv_id == s2_ptr->serv_id) &&
                (s1_ptr->freesat_id == INVALID_FREESAT_SERV_ID) &&
                (s2_ptr->freesat_id == INVALID_FREESAT_SERV_ID))
            {
               t2_ptr = s2_ptr->transport;
               if ((t2_ptr != NULL) && DBDEF_TransportInProfile(t2_ptr))
               {
                  if ((t1_ptr->orig_net_id == t2_ptr->orig_net_id) &&
                      (t1_ptr->tran_id == t2_ptr->tran_id))
                  {
                     if (manual)
                     {
                        if (s1_ptr->new_service)
                        {
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else if (s2_ptr->new_service)
                        {
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                     else
                     {
                        /* Services are the same; choose the strongest one */
                        if (t1_ptr->signal_level_at_search >= t2_ptr->signal_level_at_search)
                        {
                           /* Delete the second service */
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else
                        {
                           /* Delete the first service */
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                  }
               }
            }

            s2_ptr = next_s2_ptr;
         }
      }
      s1_ptr = next_s1_ptr;
   }

   FUNCTION_FINISH(DBDEF_TidyDatabaseSatUK);
}

/**
 * @brief   Iterates through all the services following a search search to see whether any need
 *          to be deleted based on the rules for that country
 * @param   tuner_type tuner type used for the search
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseDefault(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual)
{
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *next_s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_SERVICE_REC *next_s2_ptr;
   ADB_TRANSPORT_REC *t1_ptr;
   ADB_TRANSPORT_REC *t2_ptr;

   FUNCTION_START(DBDEF_TidyDatabaseDefault);
   USE_UNWANTED_PARAM(tuner_type);
   USE_UNWANTED_PARAM(search_completed);

   s1_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s1_ptr != NULL)
   {
      next_s1_ptr = DBDEF_GetNextServiceRec(s1_ptr);
      t1_ptr = s1_ptr->transport;
      if ((t1_ptr != NULL) && DBDEF_TransportInProfile(t1_ptr))
      {
         s2_ptr = DBDEF_GetNextServiceRec(NULL);
         while (s2_ptr != NULL)
         {
            next_s2_ptr = DBDEF_GetNextServiceRec(s2_ptr);
            if ((s1_ptr != s2_ptr) && (s1_ptr->serv_id == s2_ptr->serv_id))
            {
               t2_ptr = s2_ptr->transport;
               if ((t2_ptr != NULL) && DBDEF_TransportInProfile(t2_ptr))
               {
                  if ((t1_ptr->orig_net_id == t2_ptr->orig_net_id) &&
                      (t1_ptr->tran_id == t2_ptr->tran_id))
                  {
                     if (manual)
                     {
                        if (s1_ptr->new_service)
                        {
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else if (s2_ptr->new_service)
                        {
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                     else
                     {
                        /* Services are the same; choose the strongest one */
                        if (t1_ptr->signal_level_at_search >= t2_ptr->signal_level_at_search)
                        {
                           /* Delete the second service */
                           if (s2_ptr == next_s1_ptr)
                           {
                              next_s1_ptr = DBDEF_GetNextServiceRec(next_s1_ptr);
                           }
                           DBDEF_DeleteServiceRec(s2_ptr);
                        }
                        else
                        {
                           /* Delete the first service */
                           DBDEF_DeleteServiceRec(s1_ptr);
                           s1_ptr = NULL;
                           break;
                        }
                     }
                  }
               }
            }

            s2_ptr = next_s2_ptr;
         }
      }
      s1_ptr = next_s1_ptr;
   }

   FUNCTION_FINISH(DBDEF_TidyDatabaseDefault);
}

/**
 * @brief   Delete all transport records that don't have any services.
 * @param   tuner_type - empty transports of this signal type will be deleted
 * @param   satellite - if the signal type is SIGNAL_QPSK then this parameter specifies
 *                      the satellite the transports must be on. Ignored if tuner type
 *                      isn't SIGNAL_QPSK, and can be NULL to indicate all satellites.
 */
void DBDEF_RemoveEmptyTransports(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   ADB_TRANSPORT_REC *t_ptr, *next;
   ADB_SERVICE_REC *s_ptr;
   BOOLEAN transport_empty;

   FUNCTION_START(DBDEF_RemoveEmptyTransports);

   t_ptr = DBDEF_GetNextTransportRec(NULL);
   while (t_ptr != NULL)
   {
      transport_empty = TRUE;
      next = DBDEF_GetNextTransportRec(t_ptr);

      if (DBDEF_TransportForTunerType(t_ptr, tuner_type, satellite))
      {
         /* Check whether any services still reference this transport */
         s_ptr = DBDEF_GetNextServiceRec(NULL);
         while ((s_ptr != NULL) && transport_empty)
         {
            if (s_ptr->transport == t_ptr)
            {
               /* Found a service whose parent is this transport */
               transport_empty = FALSE;
            }
            s_ptr = DBDEF_GetNextServiceRec(s_ptr);
         }

         if (transport_empty)
         {
            /* No services refer to this transport, so it can be deleted */
            DBDEF_DeleteTransportRec(t_ptr);
         }
      }

      t_ptr = next;
   }

   FUNCTION_FINISH(DBDEF_RemoveEmptyTransports);
}

/**
 * @brief   Creates a CRID record and adds it to the database
 * @param   crid CRID string
 * @param   series TRUE if this is a series CRID
 * @param   recommended TRUE if this is a recommended programme CRID
 * @return  pointer to the created CRID record
 */
ADB_CRID_REC* DBDEF_AddCridRecord(U8BIT *crid, BOOLEAN series, BOOLEAN recommended)
{
   ADB_CRID_REC *c_ptr;
   U16BIT crid_len;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddCridRecord);

   c_ptr = STB_AppGetMemory(sizeof(ADB_CRID_REC));
   if (c_ptr != NULL)
   {
      memset(c_ptr, 0, sizeof(ADB_CRID_REC));

      crid_len = STB_GetNumBytesInString(crid);
      c_ptr->crid_str = DBDEF_MakeString(0, crid, crid_len);
      c_ptr->series_flag = series;
      c_ptr->recommended_flag = recommended;

      STB_LLAddBlockToEnd(&crid_rec_list, (LINK_LIST_PTR_BLK *)c_ptr);

      /* Add database record */
      dba_rec = DBA_CreateRecord(DBA_RECORD_CRID, NULL);
      if (dba_rec != NULL)
      {
         c_ptr->dba_rec = dba_rec;

         // fill in details in record
         DBA_SetFieldString(dba_rec, DBA_FIELD_TIMER_CRID, crid, crid_len);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_CRID_SERIES, (U32BIT)series);
         DBA_SetFieldValue(dba_rec, DBA_FIELD_CRID_RECOMMENDED, (U32BIT)recommended);

         /* Set the current date as the last time the record was seen in the EIT */
         c_ptr->eit_date = STB_GCGetGMTDate();
         DBA_SetFieldValue(dba_rec, DBA_FIELD_CRID_EIT_DATE, c_ptr->eit_date);
      }
      else
      {
         STB_AppFreeMemory(c_ptr);
         c_ptr = NULL;
      }
   }

   FUNCTION_FINISH(DBDEF_AddCridRecord);

   return(c_ptr);
}

/**
 * @brief   Sets the date and time fields in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   date_time date & time to set in the crid
 */
void DBDEF_SetCridDateTime(ADB_CRID_REC *c_ptr, U32DHMS date_time)
{
   FUNCTION_START(DBDEF_SetCridDateTime);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      c_ptr->date_time = date_time;
      DBA_SetFieldValue(c_ptr->dba_rec, DBA_FIELD_TIMER_STARTTIME, date_time);
   }

   FUNCTION_FINISH(DBDEF_SetCridDateTime);
}

/**
 * @brief   Sets the service ID in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   serv_id service ID
 */
void DBDEF_SetCridService(ADB_CRID_REC *c_ptr, U16BIT serv_id)
{
   FUNCTION_START(DBDEF_SetCridService);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      c_ptr->serv_id = serv_id;
      DBA_SetFieldValue(c_ptr->dba_rec, DBA_FIELD_SERVICE_ID, (U32BIT)serv_id);
   }

   FUNCTION_FINISH(DBDEF_SetCridService);
}

/**
 * @brief   Sets the programme name field of the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   prog_name name string to save in the CRID record
 */
void DBDEF_SetCridProgrammeName(ADB_CRID_REC *c_ptr, U8BIT *prog_name)
{
   U16BIT name_len;

   FUNCTION_START(DBDEF_SetCridProgrammeName);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      name_len = STB_GetNumBytesInString(prog_name);
      c_ptr->name_str = DBDEF_MakeString(0, prog_name, name_len);

      DBA_SetFieldString(c_ptr->dba_rec, DBA_FIELD_REC_NAME, prog_name, name_len);
   }

   FUNCTION_FINISH(DBDEF_SetCridProgrammeName);
}

/**
 * @brief   Sets the do not delete flag in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   do_not_delete new value of the flag
 */
void DBDEF_SetCridDoNotDelete(ADB_CRID_REC *c_ptr, BOOLEAN do_not_delete)
{
   FUNCTION_START(DBDEF_SetCridDoNotDelete);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      c_ptr->do_not_delete = do_not_delete;
      DBA_SetFieldValue(c_ptr->dba_rec, DBA_FIELD_CRID_DO_NOT_DELETE, (U32BIT)do_not_delete);
   }

   FUNCTION_FINISH(DBDEF_SetCridDoNotDelete);
}

/**
 * @brief   Updates the stored EIT date of this CRID with the current GMT date
 * @param   c_ptr pointer to CRID record
 */
void DBDEF_UpdateCridEitDate(ADB_CRID_REC *c_ptr)
{
   FUNCTION_START(DBDEF_UpdateCridEitDate);

   ASSERT(c_ptr != NULL);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      c_ptr->eit_date = STB_GCGetGMTDate();
      DBA_SetFieldValue(c_ptr->dba_rec, DBA_FIELD_CRID_EIT_DATE, c_ptr->eit_date);
   }

   FUNCTION_FINISH(DBDEF_UpdateCridEitDate);
}

/**
 * @brief   Deletes the given CRID record from the database
 * @param   c_ptr pointer to CRID record to be deleted
 */
void DBDEF_DeleteCridRecord(ADB_CRID_REC *c_ptr)
{
   FUNCTION_START(DBDEF_DeleteCridRecord);

   if (DBDEF_IsValidCridRecord(c_ptr))
   {
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)c_ptr);

      if (c_ptr->name_str != NULL)
      {
         STB_AppFreeMemory(c_ptr->name_str);
      }

      if (c_ptr->crid_str != NULL)
      {
         STB_AppFreeMemory(c_ptr->crid_str);
      }

      DBA_DestroyRecord(c_ptr->dba_rec);

      STB_AppFreeMemory(c_ptr);
   }

   FUNCTION_FINISH(DBDEF_DeleteCridRecord);
}

/**
 * @brief   Returns the number of CRID records in the database
 * @return  number of records
 */
U16BIT DBDEF_GetNumCridRecords(void)
{
   FUNCTION_START(DBDEF_GetNumCridRecords);
   FUNCTION_FINISH(DBDEF_GetNumCridRecords);
   return(STB_LLGetNumBlocks(&crid_rec_list));
}

/**
 * @brief   Returns the next CRID record after the one specified. If the record
 *          specified is NULL then the first record is returned.
 * @param   c_ptr CRID record pointer, NULL if first record is required
 * @return  pointer to the next record after the one specified
 */
ADB_CRID_REC* DBDEF_GetNextCridRecord(ADB_CRID_REC *c_ptr)
{
   ADB_CRID_REC *ret_ptr;

   FUNCTION_START(DBDEF_GetNextCridRecord);

   if (c_ptr == NULL)
   {
      ret_ptr = (ADB_CRID_REC *)STB_LLGetFirstBlock(&crid_rec_list);
   }
   else
   {
      ret_ptr = (ADB_CRID_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)c_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextCridRecord);

   return(ret_ptr);
}

/**
 * @brief   Checks whether the given crid record is in the list of valid crid records
 * @param   c_ptr crid record to look for
 * @return  TRUE if crid record is valid, FALSE otherwise
 */
BOOLEAN DBDEF_IsValidCridRecord(ADB_CRID_REC *c_ptr)
{
   BOOLEAN valid;
   void *next;

   FUNCTION_START(DBDEF_IsValidCridRecord);

   valid = FALSE;

   if (c_ptr != NULL)
   {
      next = STB_LLGetFirstBlock(&crid_rec_list);
      while ((next != NULL) && !valid)
      {
         if (next == c_ptr)
         {
            valid = TRUE;
         }
         else
         {
            next = STB_LLGetNextBlock(next);
         }
      }
   }

   FUNCTION_FINISH(DBDEF_IsValidCridRecord);

   return(valid);
}

/**
 * @brief   Returns the number of favourite lists
 * @return  number of favourite lists
 */
U16BIT DBDEF_GetNumFavouriteLists(void)
{
   FUNCTION_START(DBDEF_GetNumFavouriteLists);
   FUNCTION_FINISH(DBDEF_GetNumFavouriteLists);
   return(STB_LLGetNumBlocks(&favlist_list));
}

/**
 * @brief   Creates a new favourite list and adds it to the list of favourite lists.
 *          Creation of the new list will fail if a list_id can't be found for it,
 *          or memory allocation fails.
 * @param   list_id id to be given to the list, should be passed as 0 to have an id assigned.
 *                  If an id is supplied and a list already exists with that id
 *                  then no list will be created.
 * @param   name name to be given to the list, call be NULL
 * @param   user_data user defined value to be stored in the list,
 * @param   index where to add the new list into the existing lists,
 *                     0 = at start, negative = at end, else 0 based index
 * @return  pointer to new favourite list, or NULL on failure
 */
ADB_FAVLIST_REC* DBDEF_AddFavouriteList(U8BIT list_id, U8BIT *name, U32BIT user_data, S16BIT index)
{
   BOOLEAN id_used;
   ADB_FAVLIST_REC *fl_ptr;
   ADB_FAVLIST_REC *fav_list;
   void *dba_rec;
   U16BIT name_len;

   FUNCTION_START(DBDEF_AddFavouriteList);

   fav_list = NULL;

   if (list_id == 0)
   {
      /* Find an available id for the new list. 0 is an invalid list id */
      for (list_id = 1; list_id != 0; list_id++)
      {
         id_used = FALSE;

         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
         while ((fl_ptr != NULL) && !id_used)
         {
            if (fl_ptr->list_id == list_id)
            {
               /* This id is used, try another one */
               id_used = TRUE;
            }
            else
            {
               fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
            }
         }

         if (!id_used)
         {
            /* Usable id has been found */
            break;
         }
      }
   }
   else
   {
      /* Check the id isn't already being used */
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
      while (fl_ptr != NULL)
      {
         if (fl_ptr->list_id == list_id)
         {
            /* This id is used */
            break;
         }

         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
      }

      if (fl_ptr != NULL)
      {
         /* The given list id can't be used */
         list_id = 0;
      }
   }

   if (list_id != 0)
   {
      /* Id available, create a new fav list */
      if ((fav_list = STB_AppGetMemory(sizeof(ADB_FAVLIST_REC))) != NULL)
      {
         memset(fav_list, 0, sizeof(ADB_FAVLIST_REC));

         dba_rec = DBA_CreateRecord(DBA_RECORD_FAV_LIST, NULL);
         if (dba_rec != NULL)
         {
            fav_list->dba_rec = dba_rec;

            fav_list->list_id = list_id;
            DBA_SetFieldValue(dba_rec, DBA_FIELD_FAVLIST_ID, (U32BIT)list_id);

            fav_list->user_data = user_data;
            DBA_SetFieldValue(dba_rec, DBA_FIELD_FAVLIST_USER_DATA, user_data);

            if (name == NULL)
            {
               fav_list->name = NULL;
               DBA_SetFieldString(dba_rec, DBA_FIELD_REC_NAME, NULL, 0);
            }
            else
            {
               name_len = STB_GetNumBytesInString(name);
               fav_list->name = DBDEF_MakeString(0, name, name_len);
               DBA_SetFieldString(dba_rec, DBA_FIELD_REC_NAME, name, name_len);
            }

            STB_LLInitialiseHeader(&fav_list->serv_list);

            if (index == 0)
            {
               /* Add to start of list */
               STB_LLAddBlockToStart(&favlist_list, (LINK_LIST_PTR_BLK *)fav_list);
            }
            else if ((index < 0) || (index >= STB_LLGetNumBlocks(&favlist_list)))
            {
               /* Add to end of list */
               STB_LLAddBlockToEnd(&favlist_list, (LINK_LIST_PTR_BLK *)fav_list);
            }
            else
            {
               /* Inserting new list at a given position */
               fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
               while (fl_ptr != NULL)
               {
                  if (fl_ptr->index >= index)
                  {
                     /* Need to insert before this item */
                     STB_LLAddBlockBefore((LINK_LIST_PTR_BLK *)fl_ptr, (LINK_LIST_PTR_BLK *)fav_list);
                     break;
                  }
                  fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
               }
            }

            /* Reassign index of each list according to the new order */
            fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
            for (index = 0; fl_ptr != NULL; index++)
            {
               if (fl_ptr->index != index)
               {
                  fl_ptr->index = index;
                  DBA_SetFieldValue(fl_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
               }

               fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
            }
         }
         else
         {
            STB_AppFreeMemory(fav_list);
            fav_list = NULL;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_AddFavouriteList);

   return(fav_list);
}

/**
 * @brief   Returns the favourite list following the given item
 * @param   fav_list return the item after this one, if NULL, returns the first item
 * @return  next favourite list, or NULL if no more
 */
ADB_FAVLIST_REC* DBDEF_GetNextFavouriteList(ADB_FAVLIST_REC *fav_list)
{
   ADB_FAVLIST_REC *next;

   FUNCTION_START(DBDEF_GetNextFavouriteList);

   if (fav_list == NULL)
   {
      next = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
   }
   else
   {
      next = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fav_list);
   }

   FUNCTION_FINISH(DBDEF_GetNextFavouriteList);

   return(next);
}

/**
 * @brief   Return the favourite list with the given list id
 * @param   list_id favourite list id to find
 * @return  pointer to favourite list record, or NULL if not found
 */
ADB_FAVLIST_REC* DBDEF_FindFavouriteList(U8BIT list_id)
{
   ADB_FAVLIST_REC *fl_ptr;

   FUNCTION_START(DBDEF_FindFavouriteList);

   fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
   while (fl_ptr != NULL)
   {
      if (fl_ptr->list_id == list_id)
      {
         break;
      }
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindFavouriteList);

   return(fl_ptr);
}

/**
 * @brief   Saves the given user data with a favourite list
 * @param   fav_list favourite list
 * @param   user_data data to be stored with the list
 */
void DBDEF_SetFavouriteListUserData(ADB_FAVLIST_REC *fav_list, U32BIT user_data)
{
   FUNCTION_START(DBDEF_SetFavouriteListUserData);

   if (fav_list != NULL)
   {
      fav_list->user_data = user_data;
      DBA_SetFieldValue(fav_list->dba_rec, DBA_FIELD_FAVLIST_USER_DATA, user_data);
   }

   FUNCTION_FINISH(DBDEF_SetFavouriteListUserData);
}

/**
 * @brief   Changes the order of the favourite lists by moving the given list
 *          to the given position
 * @param   fav_list favourite list to be moved
 * @param   index position to move the list to
 * @return  TRUE if successful, FALSE otherwise
 */
void DBDEF_MoveFavouriteListTo(ADB_FAVLIST_REC *fav_list, S16BIT index)
{
   ADB_FAVLIST_REC *fl_ptr;

   FUNCTION_START(DBDEF_MoveFavouriteListTo);

   if (fav_list->index != index)
   {
      /* Remove the block from it's current position in the list */
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)fav_list);

      if (index == 0)
      {
         STB_LLAddBlockToStart(&favlist_list, (LINK_LIST_PTR_BLK *)fav_list);
      }
      else if ((index < 0) || (index >= STB_LLGetNumBlocks(&favlist_list)))
      {
         STB_LLAddBlockToEnd(&favlist_list, (LINK_LIST_PTR_BLK *)fav_list);
      }
      else
      {
         /* Find position to insert the list */
         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
         while (fl_ptr != NULL)
         {
            if (fl_ptr->index >= index)
            {
               /* Need to insert before this item */
               STB_LLAddBlockBefore((LINK_LIST_PTR_BLK *)fl_ptr, (LINK_LIST_PTR_BLK *)fav_list);
               break;
            }
            fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
         }
      }

      /* Reassign index of each list according to the new order */
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
      for (index = 0; fl_ptr != NULL; index++)
      {
         if (fl_ptr->index != index)
         {
            fl_ptr->index = index;
            DBA_SetFieldValue(fl_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
         }

         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
      }
   }

   FUNCTION_FINISH(DBDEF_MoveFavouriteListTo);
}

/**
 * @brief   Deletes the given favourite list
 * @param   fav_list favourite list to be deleted
 */
void DBDEF_DeleteFavouriteList(ADB_FAVLIST_REC *fav_list)
{
   ADB_FAVSERV_REC *fs_ptr;
   ADB_FAVLIST_REC *fl_ptr;
   U16BIT index;

   FUNCTION_START(DBDEF_DeleteFavouriteList);

   if (fav_list != NULL)
   {
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)fav_list);

      /* Delete all services from this favourite list */
      fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
      while (fs_ptr != NULL)
      {
         DBDEF_DeleteServiceFromFavouriteList(fav_list, fs_ptr);
         fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
      }

      if (fav_list->name != NULL)
      {
         STB_AppFreeMemory(fav_list->name);
      }

      DBA_DestroyRecord(fav_list->dba_rec);

      STB_AppFreeMemory(fav_list);

      /* Reassign list indexes to reflect updated positions */
      fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetFirstBlock(&favlist_list);
      for (index = 0; fl_ptr != NULL; index++)
      {
         if (fl_ptr->index != index)
         {
            fl_ptr->index = index;
            DBA_SetFieldValue(fl_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
         }
         fl_ptr = (ADB_FAVLIST_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fl_ptr);
      }
   }

   FUNCTION_FINISH(DBDEF_DeleteFavouriteList);
}

/**
 * @brief   Returns the number of services in the given favourite list
 * @param   fav_list favourite list
 * @return  number of services
 */
U16BIT DBDEF_GetNumServicesInFavouriteList(ADB_FAVLIST_REC *fav_list)
{
   FUNCTION_START(DBDEF_GetNumServicesInFavouriteList);
   FUNCTION_FINISH(DBDEF_GetNumServicesInFavouriteList);
   return(STB_LLGetNumBlocks(&fav_list->serv_list));
}

/**
 * @brief   Adds a new service to the given favourite list at the given position
 * @param   fav_list list service is to be added to
 * @param   serv_ptr service to be added to the list
 * @param   index position to add the service in the list,
 *                  0 = at start, negative = at end, any other value is the position
 * @return  new favourite service record, or NULL on failure
 */
ADB_FAVSERV_REC* DBDEF_AddServiceToFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_SERVICE_REC *serv_ptr, S16BIT index)
{
   ADB_FAVSERV_REC *fs_ptr;
   ADB_FAVSERV_REC *fav_serv;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddServiceToFavouriteList);

   fav_serv = NULL;

   /* Check that the service isn't already in the list */
   fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
   while (fs_ptr != NULL)
   {
      if (fs_ptr->serv_ptr == serv_ptr)
      {
         /* Service already in this list */
         break;
      }
      fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
   }

   if (fs_ptr == NULL)
   {
      /* Service not in list, add it */
      if ((fav_serv = STB_AppGetMemory(sizeof(ADB_FAVSERV_REC))) != NULL)
      {
         memset(fav_serv, 0, sizeof(ADB_FAVSERV_REC));

         dba_rec = DBA_CreateRecord(DBA_RECORD_FAV_SERV, serv_ptr->dba_rec);
         if (dba_rec != NULL)
         {
            fav_serv->dba_rec = dba_rec;
            fav_serv->serv_ptr = serv_ptr;

            fav_serv->list_id = fav_list->list_id;
            DBA_SetFieldValue(dba_rec, DBA_FIELD_FAVLIST_ID, (U32BIT)fav_serv->list_id);

            if (index == 0)
            {
               /* Add to start of list */
               STB_LLAddBlockToStart(&fav_list->serv_list, (LINK_LIST_PTR_BLK *)fav_serv);
            }
            else if ((index < 0) || (index >= STB_LLGetNumBlocks(&fav_list->serv_list)))
            {
               /* Add to end of list */
               STB_LLAddBlockToEnd(&fav_list->serv_list, (LINK_LIST_PTR_BLK *)fav_serv);
            }
            else
            {
               /* Insert into given position */
               fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
               while (fs_ptr != NULL)
               {
                  if (fs_ptr->index >= index)
                  {
                     /* Insert before this item */
                     STB_LLAddBlockBefore((LINK_LIST_PTR_BLK *)fs_ptr, (LINK_LIST_PTR_BLK *)fav_serv);
                     break;
                  }
                  fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
               }
            }

            /* Reassign index of each service according to the new order */
            fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
            for (index = 0; fs_ptr != NULL; index++)
            {
               if (fs_ptr->index != index)
               {
                  fs_ptr->index = index;
                  DBA_SetFieldValue(fs_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
               }

               fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
            }
         }
         else
         {
            STB_AppFreeMemory(fav_serv);
            fav_serv = NULL;
         }
      }
   }

   FUNCTION_FINISH(DBDEF_AddServiceToFavouriteList);

   return(fav_serv);
}

/**
 * @brief   Returns the ADB_FAVSERV_REC from the given favourite list for the given service
 * @param   fav_list favourite list to be searched
 * @param   serv_ptr service to be searched for
 * @return  pointer to service record, or NULL if not found
 */
ADB_FAVSERV_REC* DBDEF_FindServiceInFavouriteList(ADB_FAVLIST_REC *fav_list, void *serv_ptr)
{
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(DBDEF_FindServiceInFavouriteList);

   fav_serv = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
   while (fav_serv != NULL)
   {
      if (fav_serv->serv_ptr == serv_ptr)
      {
         break;
      }

      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fav_serv);
   }

   FUNCTION_FINISH(DBDEF_FindServiceInFavouriteList);

   return(fav_serv);
}

/**
 * @brief   Returns the next favourite list service record
 * @param   fav_list favourite list
 * @param   fav_serv current service, or NULL if first service to be returned
 * @return  pointer to service record, or NULL if no more services
 */
ADB_FAVSERV_REC* DBDEF_GetNextServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_FAVSERV_REC *fav_serv)
{
   FUNCTION_START(DBDEF_GetNextServiceFromFavouriteList);

   if (fav_serv == NULL)
   {
      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
   }
   else
   {
      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fav_serv);
   }

   FUNCTION_FINISH(DBDEF_GetNextServiceFromFavouriteList);

   return(fav_serv);
}

/**
 * @brief   Returns the previous favourite list service record
 * @param   fav_list favourite list
 * @param   fav_serv current service, or NULL if last service to be returned
 * @return  pointer to service record, or NULL if no more services
 */
ADB_FAVSERV_REC* DBDEF_GetPrevServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_FAVSERV_REC *fav_serv)
{
   FUNCTION_START(DBDEF_GetPrevServiceFromFavouriteList);

   if (fav_serv == NULL)
   {
      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetLastBlock(&fav_list->serv_list);
   }
   else
   {
      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetPrevBlock((LINK_LIST_PTR_BLK *)fav_serv);
   }

   FUNCTION_FINISH(DBDEF_GetPrevServiceFromFavouriteList);

   return(fav_serv);
}

/**
 * @brief   Changes the order of the services in the favourite list by moving
 *          the given service to the given position
 * @param   fav_list favourite list
 * @param   fav_serv service to be moved
 * @param   index position to move the service to
 * @return  TRUE if successful, FALSE otherwise
 */
void DBDEF_MoveFavouriteListServiceTo(ADB_FAVLIST_REC *fav_list, ADB_FAVSERV_REC *fav_serv,
   S16BIT index)
{
   ADB_FAVSERV_REC *fs_ptr;

   FUNCTION_START(DBDEF_MoveFavouriteListServiceTo);

   if (fav_serv->index != index)
   {
      /* Remove the block from it's current position in the list */
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)fav_serv);

      if (index == 0)
      {
         STB_LLAddBlockToStart(&fav_list->serv_list, (LINK_LIST_PTR_BLK *)fav_serv);
      }
      else if ((index < 0) || (index >= STB_LLGetNumBlocks(&fav_list->serv_list)))
      {
         STB_LLAddBlockToEnd(&fav_list->serv_list, (LINK_LIST_PTR_BLK *)fav_serv);
      }
      else
      {
         /* Find position to insert the list */
         fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
         while (fs_ptr != NULL)
         {
            if (fs_ptr->index >= index)
            {
               /* Need to insert before this item */
               STB_LLAddBlockBefore((LINK_LIST_PTR_BLK *)fs_ptr, (LINK_LIST_PTR_BLK *)fav_serv);
               break;
            }
            fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
         }
      }

      /* Reassign index of each service according to the new order */
      fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
      for (index = 0; fs_ptr != NULL; index++)
      {
         if (fs_ptr->index != index)
         {
            fs_ptr->index = index;
            DBA_SetFieldValue(fs_ptr->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
         }

         fs_ptr = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fs_ptr);
      }
   }

   FUNCTION_FINISH(DBDEF_MoveFavouriteListServiceTo);
}

/**
 * @brief   Delete the given service from the given favourite list
 * @param   fav_list favourite list to delete service from
 * @param   fav_serv service to be deleted from the favourite list
 */
void DBDEF_DeleteServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list, ADB_FAVSERV_REC *fav_serv)
{
   U16BIT index;

   FUNCTION_START(DBDEF_DeleteServiceFromFavouriteList);

   if ((fav_list != NULL) && (fav_serv != NULL))
   {
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)fav_serv);

      DBA_DestroyRecord(fav_serv->dba_rec);

      STB_AppFreeMemory(fav_serv);

      /* Reassign list indexes for all the services remaining in the list */
      fav_serv = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
      for (index = 0; fav_serv != NULL; index++)
      {
         if (fav_serv->index != index)
         {
            fav_serv->index = index;
            DBA_SetFieldValue(fav_serv->dba_rec, DBA_FIELD_FAVLIST_INDEX, (U32BIT)index);
         }
         fav_serv = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)fav_serv);
      }
   }

   FUNCTION_FINISH(DBDEF_DeleteServiceFromFavouriteList);
}

/**
 * @brief   Delete the all services from the given favourite list
 * @param   fav_list favourite list
 */
void DBDEF_DeleteAllServicesFromFavouriteList(ADB_FAVLIST_REC *fav_list)
{
   ADB_FAVSERV_REC *serv;
   ADB_FAVSERV_REC *next;

   FUNCTION_START(DBDEF_DeleteAllServicesFromFavouriteList);

   if (fav_list != NULL)
   {
      serv = (ADB_FAVSERV_REC *)STB_LLGetFirstBlock(&fav_list->serv_list);
      while (serv != NULL)
      {
         next = (ADB_FAVSERV_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)serv);

         STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)serv);

         DBA_DestroyRecord(serv->dba_rec);

         STB_AppFreeMemory(serv);

         serv = next;
      }
   }

   FUNCTION_FINISH(DBDEF_DeleteAllServicesFromFavouriteList);
}

/**
 * @brief   Set or change the name of the given favourite list
 * @param   f_ptr pointer to favourite record to be updated
 * @param   name of the favourite list
 * @return  void
 */
void DBDEF_SetFavouriteListName(ADB_FAVLIST_REC *f_ptr, U8BIT *name)
{
   BOOLEAN changed;
   U16BIT new_name_len;

   FUNCTION_START(DBDEF_SetFavouriteListName);

   changed = TRUE;
   new_name_len = 0;

   if (name != NULL)
   {
      new_name_len = STB_GetNumBytesInString(name);
   }

   if (f_ptr->name != NULL)
   {
      if (new_name_len == f_ptr->name->nbytes)
      {
         /* String lengths are the same, check the bytes */
         if (memcmp(name, f_ptr->name->str_ptr, new_name_len) == 0)
         {
            /* Names are the same */
            changed = FALSE;
         }
      }
   }

   if (changed)
   {
      /* Free the current name */
      if (f_ptr->name != NULL)
      {
         DBDEF_ReleaseString(f_ptr->name);
         f_ptr->name = NULL;
      }

      if (name != NULL)
      {
         f_ptr->name = DBDEF_MakeString(0, name, new_name_len);
      }

      if (f_ptr->name != NULL)
      {
         DBA_SetFieldString(f_ptr->dba_rec, DBA_FIELD_REC_NAME, f_ptr->name->str_ptr,
            f_ptr->name->nbytes);
      }
      else
      {
         DBA_SetFieldString(f_ptr->dba_rec, DBA_FIELD_REC_NAME, NULL, 0);
      }

      DBA_SaveRecord(f_ptr->dba_rec);
   }

   FUNCTION_FINISH(DBDEF_SetFavouriteListName);
}


/**
 * @brief   Checks whether the given service is a terrestrial, cable or satellite service, and
 *          optionally check whether it's on a particular satellite
 * @param   s_ptr service to be checked
 * @param   tuner_type type of tuner to check
 * @param   satellite if tuner type is SIGNAL_QPSK then check whether the service is from this
 *                      satellite. This can be NULL if the satellite isn't important and is ignored
 *                      for a non-satellite tuner.
 * @return  TRUE if the service is of the given type, FALSE otherwise
 */
BOOLEAN DBDEF_ServiceForTunerType(ADB_SERVICE_REC *s_ptr, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_ServiceForTunerType);

   if ((s_ptr != NULL) && (s_ptr->transport != NULL))
   {
      retval = DBDEF_TransportForTunerType(s_ptr->transport, tuner_type, satellite);
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(DBDEF_ServiceForTunerType);

   return(retval);
}

/**
 * @brief   Checks whether the given transport is a terrestrial, cable or satellite transport, and
 *          optionally check whether it's on a particular satellite
 * @param   t_ptr transport to be checked
 * @param   tuner_type type of tuner to check
 * @param   satellite if tuner type is SIGNAL_QPSK then check whether the transport is from this
 *                      satellite. This can be NULL if the satellite isn't important and is ignored
 *                      for a non-satellite tuner.
 * @return  TRUE if the transport is of the given type, FALSE otherwise
 */
BOOLEAN DBDEF_TransportForTunerType(ADB_TRANSPORT_REC *t_ptr, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_TransportForTunerType);

   if ((t_ptr != NULL) && (t_ptr->sig_type == tuner_type) &&
       ((satellite == NULL) || (tuner_type != SIGNAL_QPSK) ||
        ((tuner_type == SIGNAL_QPSK) && (t_ptr->network != NULL) &&
         (t_ptr->network->satellite == satellite))))
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(DBDEF_TransportForTunerType);

   return(retval);
}

/**
 * @brief   Creates a new timer record in the database, assigning it a unique handle
 * @param   store_in_nvm save the timer in non-volatile memory
 * @return  pointer to the new timer, or NULL on failure
 */
ADB_TIMER_REC* DBDEF_AddTimerRec(BOOLEAN store_in_nvm)
{
   ADB_TIMER_REC *timer;
   void *dba_rec;
   U32BIT handle;

   FUNCTION_START(DBDEF_AddTimerRec);

   timer = (ADB_TIMER_REC *)STB_AppGetMemory(sizeof(ADB_TIMER_REC));
   if (timer != NULL)
   {
      memset(timer, 0, sizeof(ADB_TIMER_REC));

      /* Find next unused handle */
      handle = last_timer_handle + 1;
      if (handle == LAST_TIMER_HANDLE)
      {
         handle = TIMER_HANDLE_BASE;
      }

      while (DBDEF_FindTimerRec(handle) != NULL)
      {
         handle++;
         if (handle == LAST_TIMER_HANDLE)
         {
            handle = TIMER_HANDLE_BASE;
         }
      }

      last_timer_handle = handle;

      timer->handle = handle;

      if (store_in_nvm)
      {
         /* Create the database record */
         dba_rec = DBA_CreateRecord(DBA_RECORD_TIMER, NULL);
         if (dba_rec != NULL)
         {
            timer->dba_rec = dba_rec;

            DBA_SetFieldValue(dba_rec, DBA_FIELD_TIMER_HANDLE, timer->handle);
         }
         else
         {
            STB_AppFreeMemory(timer);
            timer = NULL;
         }
      }

      if (timer != NULL)
      {
         STB_LLAddBlockToEnd(&timer_list, (LINK_LIST_PTR_BLK *)timer);
      }
   }

   FUNCTION_FINISH(DBDEF_AddTimerRec);

   return(timer);
}

/**
 * @brief   Returns the timer record with the given timer handle
 * @param   handle timer handle
 * @return  pointer to the timer record, or NULL if not found
 */
ADB_TIMER_REC* DBDEF_FindTimerRec(U32BIT handle)
{
   ADB_TIMER_REC *timer;

   FUNCTION_START(DBDEF_FindTimerRec);

   timer = DBDEF_GetNextTimerRec(NULL);
   while ((timer != NULL) && (timer->handle != handle))
   {
      timer = DBDEF_GetNextTimerRec(timer);
   }

   FUNCTION_FINISH(DBDEF_FindTimerRec);

   return(timer);
}

/**
 * @brief   Returns the next timer record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   timer_ptr timer record pointer, NULL if first record is required
 * @return  pointer to next timer record
 */
ADB_TIMER_REC* DBDEF_GetNextTimerRec(ADB_TIMER_REC *timer_ptr)
{
   ADB_TIMER_REC *ret_ptr;

   FUNCTION_START(DBDEF_GetNextTimerRec);

   if (timer_ptr == NULL)
   {
      ret_ptr = (ADB_TIMER_REC *)STB_LLGetFirstBlock(&timer_list);
   }
   else
   {
      ret_ptr = (ADB_TIMER_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)timer_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextTimerRec);

   return(ret_ptr);
}

/**
 * @brief   Sorts timer list into date/time or alphabetical order
 * @param   date_time_order TRUE to sort into date/time order, FALSE for alphabetical
 */
void DBDEF_SortTimers(BOOLEAN date_time_order)
{
   FUNCTION_START(DBDEF_SortTimers);

   if (date_time_order)
   {
      STB_LLSort(&timer_list, TimerSortDateTime);
   }
   else
   {
      STB_LLSort(&timer_list, TimerSortName);
   }

   FUNCTION_FINISH(DBDEF_SortTimers);
}

/**
 * @brief   Deletes the given timer from the database
 * @param   timer timer to be deleted
 */
void DBDEF_DeleteTimerRec(ADB_TIMER_REC *timer)
{
   FUNCTION_START(DBDEF_DeleteTimerRec);

   if (timer != NULL)
   {
      STB_LLRemoveBlock((LINK_LIST_PTR_BLK *)timer);

      if (timer->dba_rec != NULL)
      {
         DBA_DestroyRecord(timer->dba_rec);
      }

      STB_AppFreeMemory(timer);
   }

   FUNCTION_FINISH(DBDEF_DeleteTimerRec);
}

/**
 * @brief   Returns the number of network profiles
 */
U16BIT DBDEF_GetNumProfiles(void)
{
   U16BIT num_profiles;
   ADB_NETWORK_REC *n_ptr;
   BOOLEAN broadcast_found;

   FUNCTION_START(DBDEF_GetNumProfiles);

   num_profiles = 0;
   broadcast_found = FALSE;

   n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
   while (n_ptr != NULL)
   {
      if (n_ptr->profile_type == ADB_PROFILE_TYPE_BROADCAST)
      {
         /* Only count the broadcast profile once */
         if (!broadcast_found)
         {
            num_profiles++;
            broadcast_found = TRUE;
         }
      }
      else
      {
         num_profiles++;
      }

      n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNumProfiles);

   return(num_profiles);
}

/**
 * @brief   Gets a list of the available network profiles
 * @param   profile_list address of pointer to an array that will be allocated by this
 *                       function that will contain the profiles
 * @param   active_profile pointer to return the index of the currently active profile
 * @return  number of profiles in the returned array
 */
U16BIT DBDEF_GetProfileList(void ***profile_list, U16BIT *active_profile)
{
   U16BIT num_profiles;
   ADB_NETWORK_REC *n_ptr;
   BOOLEAN broadcast_found;

   FUNCTION_START(DBDEF_GetProfileList);

   num_profiles = DBDEF_GetNumProfiles();
   if (num_profiles > 0)
   {
      *profile_list = (void **)STB_AppGetMemory(num_profiles * sizeof(void *));
      if (*profile_list != NULL)
      {
         broadcast_found = FALSE;
         num_profiles = 0;

         n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
         while (n_ptr != NULL)
         {
            if (n_ptr->profile_type == ADB_PROFILE_TYPE_BROADCAST)
            {
               /* Only count the broadcast profile once */
               if (!broadcast_found)
               {
                  (*profile_list)[num_profiles] = (void *)n_ptr;

                  if (current_profile == ADB_PROFILE_TYPE_BROADCAST)
                  {
                     /* Return as the current profile */
                     *active_profile = num_profiles;
                  }

                  num_profiles++;
                  broadcast_found = TRUE;
               }
            }
            else
            {
               (*profile_list)[num_profiles] = (void *)n_ptr;

            #ifdef COMMON_INTERFACE
               if ((current_profile == n_ptr->profile_type) &&
                   (cicam_onet_id == n_ptr->cicam_onet_id) &&
                   (cicam_identifier == n_ptr->cicam_identifier))
               {
                  /* Return as the current profile */
                  *active_profile = num_profiles;
               }
            #endif /*COMMON_INTERFACE*/

               num_profiles++;
            }

            n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
         }
      }
      else
      {
         num_profiles = 0;
      }
   }

   FUNCTION_FINISH(DBDEF_GetProfileList);

   return(num_profiles);
}

/**
 * @brief   Frees a profile list returned by DBDEF_GetProfileList
 * @param   profile_list profile list to be freed
 * @param   num_profiles number of profiles in the list
 */
void DBDEF_ReleaseProfileList(void **profile_list, U16BIT num_profiles)
{
   FUNCTION_START(DBDEF_ReleaseProfileList);
   USE_UNWANTED_PARAM(num_profiles);

   if (profile_list != NULL)
   {
      STB_AppFreeMemory(profile_list);
   }

   FUNCTION_FINISH(DBDEF_ReleaseProfileList);
}

/**
 * @brief   Is the given profile the currently active profile?
 * @param   profile profile
 * @return  TRUE if the given profile is the currently active profile, FALSE otherwise
 */
BOOLEAN DBDEF_IsActiveProfile(ADB_NETWORK_REC *profile)
{
   BOOLEAN retval;
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_IsActiveProfile);

   retval = FALSE;

   n_ptr = (ADB_NETWORK_REC *)STB_LLGetFirstBlock(&network_rec_list);
   while (n_ptr != NULL)
   {
      if (n_ptr == profile)
      {
         if (n_ptr->profile_type == current_profile)
         {
            if ((current_profile == ADB_PROFILE_TYPE_BROADCAST)
         #ifdef COMMON_INTERFACE
                || ((current_profile == ADB_PROFILE_TYPE_CIPLUS) &&
                    (n_ptr->cicam_onet_id == cicam_onet_id) && (n_ptr->cicam_identifier == cicam_identifier))
         #endif /*COMMON_INTERFACE*/
                )
            {
               retval = TRUE;
            }
         }
         break;
      }

      n_ptr = (ADB_NETWORK_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)n_ptr);
   }

   FUNCTION_FINISH(DBDEF_IsActiveProfile);

   return(retval);
}

/**
 * @brief   Returns the current profile type
 * @return  Current profile type in use
 */
ADB_PROFILE_TYPE DBDEF_GetCurrentProfileType(void)
{
   FUNCTION_START(DBDEF_GetCurrentProfileType);
   FUNCTION_FINISH(DBDEF_GetCurrentProfileType);
   return(current_profile);
}

/**
 * @brief   Sets the broadcast profile type for for all network, transport
 *          and service record accesses
 */
void DBDEF_SelectBroadcastProfile(void)
{
   FUNCTION_START(DBDEF_SelectBroadcastProfile);
   current_profile = ADB_PROFILE_TYPE_BROADCAST;
   FUNCTION_FINISH(DBDEF_SelectBroadcastProfile);
}

/**
 * @brief   Saves the current profile and any related data so that it can restored
 *          using DBDEF_PopProfile(), and sets the broadcast profile type for for
 *          all network, transport and service record accesses
 */
void DBDEF_PushBroadcastProfile(void)
{
   FUNCTION_START(DBDEF_PushBroadcastProfile);

   pushed_profile = current_profile;
#ifdef COMMON_INTERFACE
   if (current_profile == ADB_PROFILE_TYPE_CIPLUS)
   {
      pushed_cicam_onet_id = cicam_onet_id;
      pushed_cicam_identifier = cicam_identifier;
   }
#endif
   current_profile = ADB_PROFILE_TYPE_BROADCAST;

   FUNCTION_FINISH(DBDEF_PushBroadcastProfile);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   Saves the current profile and any related data so that it can be restored
 *          using DBDEF_PopProfile(), and sets the current CI+ profile type for for all
 *          network, transport and service record accesses
 * @param   onet_id original network id of the CI+ operator profile
 * @param   cicam_id CICAM identifier of the CI+ operator profile
 */
void DBDEF_PushCIPlusProfile(U16BIT onet_id, U32BIT cicam_id)
{
   FUNCTION_START(DBDEF_SelectCIPlusProfile);

   /* Save the current profile type */
   pushed_profile = current_profile;

   if (current_profile == ADB_PROFILE_TYPE_CIPLUS)
   {
      pushed_cicam_onet_id = cicam_onet_id;
      pushed_cicam_identifier = cicam_identifier;
   }

   current_profile = ADB_PROFILE_TYPE_CIPLUS;
   cicam_onet_id = onet_id;
   cicam_identifier = cicam_id;

   FUNCTION_FINISH(DBDEF_SelectCIPlusProfile);
}

/**
 * @brief   Sets the CI+ profile type for for all network, transport
 *          and service record accesses
 * @param   onet_id original network id of the CI+ operator profile
 * @param   cicam_id CICAM identifier of the CI+ operator profile
 */
void DBDEF_SelectCIPlusProfile(U16BIT onet_id, U32BIT cicam_id)
{
   FUNCTION_START(DBDEF_SelectCIPlusProfile);

   current_profile = ADB_PROFILE_TYPE_CIPLUS;
   cicam_onet_id = onet_id;
   cicam_identifier = cicam_id;

   FUNCTION_FINISH(DBDEF_SelectCIPlusProfile);
}

#endif

/**
 * @brief   Restores a previously pushed profile
 */
void DBDEF_PopProfile(void)
{
   FUNCTION_START(DBDEF_PopProfile);

   current_profile = pushed_profile;

#ifdef COMMON_INTERFACE
   if (pushed_profile == ADB_PROFILE_TYPE_CIPLUS)
   {
      cicam_onet_id = pushed_cicam_onet_id;
      cicam_identifier = pushed_cicam_identifier;
   }
#endif

   FUNCTION_FINISH(DBDEF_PopProfile);
}

/**
 * @brief   Checks whether the given service is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   s_ptr service to be checked
 * @return  TRUE if the service isn't part of a network or is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_ServiceInProfile(ADB_SERVICE_REC *s_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_ServiceInProfile);

   retval = TRUE;

   if ((s_ptr->transport != NULL) && (s_ptr->transport->network != NULL))
   {
      if (s_ptr->transport->network->profile_type != current_profile)
      {
         retval = FALSE;
      }
#ifdef COMMON_INTERFACE
      else if ((current_profile == ADB_PROFILE_TYPE_CIPLUS) &&
               ((s_ptr->transport->network->cicam_onet_id != cicam_onet_id) ||
                (s_ptr->transport->network->cicam_identifier != cicam_identifier)))
      {
         /* Profile is CI+, but for a different CAM */
         retval = FALSE;
      }
#endif
   }

   FUNCTION_FINISH(DBDEF_ServiceInProfile);

   return(retval);
}

/**
 * @brief   Checks whether the given transport is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   t_ptr transport to be checked
 * @return  TRUE if the transport isn't part of a network or is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_TransportInProfile(ADB_TRANSPORT_REC *t_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_TransportInProfile);

   retval = TRUE;

   if (t_ptr->network != NULL)
   {
      if (t_ptr->network->profile_type != current_profile)
      {
         retval = FALSE;
      }
#ifdef COMMON_INTERFACE
      else if ((current_profile == ADB_PROFILE_TYPE_CIPLUS) &&
               ((t_ptr->network->cicam_onet_id != cicam_onet_id) ||
                (t_ptr->network->cicam_identifier != cicam_identifier)))
      {
         /* Profile is CI+, but for a different CAM */
         retval = FALSE;
      }
#endif
   }

   FUNCTION_FINISH(DBDEF_TransportInProfile);

   return(retval);
}

/**
 * @brief   Checks whether the given network is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   n_ptr network to be checked
 * @return  TRUE if the network is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_NetworkInProfile(ADB_NETWORK_REC *n_ptr)
{
   BOOLEAN retval;

   FUNCTION_START(DBDEF_NetworkInProfile);

   retval = TRUE;

   if (n_ptr->profile_type != current_profile)
   {
      retval = FALSE;
   }
#ifdef COMMON_INTERFACE
   else if ((current_profile == ADB_PROFILE_TYPE_CIPLUS) &&
            ((n_ptr->cicam_onet_id != cicam_onet_id) ||
             (n_ptr->cicam_identifier != cicam_identifier)))
   {
      /* Profile is CI+, but for a different CAM */
      retval = FALSE;
   }
#endif

   FUNCTION_FINISH(DBDEF_NetworkInProfile);

   return(retval);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Finds the CI+ profile network record
 * @param   cicam_onet_id CI+ private network id
 * @param   cicam_identifier CI+ CAM identifier
 * @return  Pointer to the network record or NULL if not found
 ****************************************************************************/
ADB_NETWORK_REC* DBDEF_FindNetworkForCIPlusProfile(U16BIT cicam_onet_id, U32BIT cicam_identifier)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_FindNetworkForCIPlusProfile);

   n_ptr = DBDEF_GetNextNetworkRec(NULL);
   while (n_ptr != NULL)
   {
      if ((n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS) && (n_ptr->cicam_onet_id == cicam_onet_id) &&
          (n_ptr->cicam_identifier == cicam_identifier))
      {
         break;
      }

      n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindNetworkForCIPlusProfile);

   return(n_ptr);
}

/*!**************************************************************************
 * @brief   Finds the CI+ profile network record for the given module.
 *          The module must be present for the record to be found
 * @param   module CI+ operator module
 * @return  Pointer to the network record or NULL if not found
 ****************************************************************************/
ADB_NETWORK_REC* DBDEF_FindNetworkForCIPlusModule(U32BIT module)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(DBDEF_FindNetworkForCIPlusModule);

   n_ptr = DBDEF_GetNextNetworkRec(NULL);
   while (n_ptr != NULL)
   {
      if ((n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS) && n_ptr->module_present &&
          (n_ptr->module == module))
      {
         break;
      }

      n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
   }

   FUNCTION_FINISH(DBDEF_FindNetworkForCIPlusModule);

   return(n_ptr);
}

/**
 * @brief   Reuturns the currently selected CI+ profile.
 * @return  Network record of the selected CI+ profile or NULL if the current profile is broadcast
 */
ADB_NETWORK_REC *DBDEF_GetCurrentCIPlusProfile(void)
{
   ADB_NETWORK_REC *n_ptr = NULL;

   FUNCTION_START(DBDEF_GetCurrentCIPlusProfile);

   if (current_profile == ADB_PROFILE_TYPE_CIPLUS)
   {
      n_ptr = DBDEF_FindNetworkForCIPlusProfile(cicam_onet_id, cicam_identifier);
   }

   FUNCTION_FINISH(DBDEF_GetCurrentCIPlusProfile);

   return n_ptr;
}

/**
 * @brief   Add CICAM timer record to the database
 * @param   cicam_id CICAM ID
 * @param   timer_handle Timer handle
 * @return  pointer to the new record or NULL if not created
 */
ADB_CICAM_TIMER_REC* DBDEF_AddCicamTimer(U32BIT cicam_id, U32BIT timer_handle)
{
   ADB_CICAM_TIMER_REC *t;
   void *dba_rec;

   FUNCTION_START(DBDEF_AddCicamTimer);

   DBA_LockDatabase();
   dba_rec = DBA_CreateRecord(DBA_RECORD_CICAM_TIMER, NULL);
   if (dba_rec != NULL)
   {
      t = (ADB_CICAM_TIMER_REC *)STB_AppGetMemory(sizeof(ADB_CICAM_TIMER_REC));
      if (t != NULL)
      {
         t->dba_rec = dba_rec;
         t->cicam_identifier = cicam_id;
         DBA_SetFieldValue(dba_rec, DBA_FIELD_TIMER_HANDLE, t->cicam_identifier);
         t->timer_handle = timer_handle;
         DBA_SetFieldValue(dba_rec, DBA_FIELD_PROFILE_CAM_ID, t->timer_handle);

         STB_LLAddBlockToEnd(&cicam_timer_list, (LINK_LIST_PTR_BLK *)t);
         DBA_SaveRecord(dba_rec);
         DBA_SaveDatabase();
      }
      else
      {
         DBA_DestroyRecord(dba_rec);
      }
   }
   else
   {
      t = NULL;
   }
   DBA_UnlockDatabase();

   FUNCTION_FINISH(DBDEF_AddCicamTimer);

   return t;
}

/**
 * @brief   Deletes the given cicam timer from the service database.
 * @param   cicam_timer_ptr pointer to cicam timer to be deleted
 */
void DBDEF_DeleteCicamTimerRec(ADB_CICAM_TIMER_REC *cicam_timer_ptr)
{
   FUNCTION_START(DBDEF_DeleteCicamTimerRec);

   STB_LLRemoveBlock(&cicam_timer_ptr->ptrs);
   DBA_LockDatabase();
   DBA_DestroyRecord(cicam_timer_ptr->dba_rec);
   DBA_SaveDatabase();
   DBA_UnlockDatabase();
   STB_AppFreeMemory(cicam_timer_ptr);

   FUNCTION_FINISH(DBDEF_DeleteCicamTimerRec);
}

/**
 * @brief   Returns the CICAM timer following the one given. If the argument is NULL
 *          then the first CICAM timer will be returned.
 * @param   cicam_timer_ptr CICAM timer record, NULL if first CICAM timer is required
 * @return  pointer to next CICAM timer, NULL if no more CICAM timers
 */
ADB_CICAM_TIMER_REC* DBDEF_GetNextCicamTimerRec(ADB_CICAM_TIMER_REC *cicam_timer_ptr)
{
   ADB_CICAM_TIMER_REC *t_ptr;

   FUNCTION_START(DBDEF_GetNextCicamTimerRec);

   if (cicam_timer_ptr == NULL)
   {
      t_ptr = (ADB_CICAM_TIMER_REC *)STB_LLGetFirstBlock(&cicam_timer_list);
   }
   else
   {
      t_ptr = (ADB_CICAM_TIMER_REC *)STB_LLGetNextBlock((LINK_LIST_PTR_BLK *)cicam_timer_ptr);
   }

   FUNCTION_FINISH(DBDEF_GetNextCicamTimerRec);

   return(t_ptr);
}

/**
 * @brief   Returns the CICAM timer associated with a CICAM ID
 * @param   cicam_id CICAM ID
 * @return  CICAM timer record or NULL if none is found
 */
ADB_CICAM_TIMER_REC* DBDEF_FindCicamTimer(U32BIT cicam_id)
{
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(DBDEF_FindCicamTimer);

   t = DBDEF_GetNextCicamTimerRec(NULL);
   while (t != NULL)
   {
      if (t->cicam_identifier == cicam_id)
      {
         break;
      }
      t = DBDEF_GetNextCicamTimerRec(t);
   }

   FUNCTION_FINISH(DBDEF_FindCicamTimer)

   return t;
}

/**
 * @brief   Returns the CICAM timer associated with the specified timer handle
 * @param   handle Timer handle
 * @return  CICAM timer record or NULL if none is found
 */
ADB_CICAM_TIMER_REC* DBDEF_FindCicamTimerByHandle(U32BIT handle)
{
   ADB_CICAM_TIMER_REC *t;

   FUNCTION_START(DBDEF_FindCicamTimerByHandle);

   t = DBDEF_GetNextCicamTimerRec(NULL);
   while (t != NULL)
   {
      if (t->timer_handle == handle)
      {
         break;
      }
      t = DBDEF_GetNextCicamTimerRec(t);
   }

   FUNCTION_FINISH(DBDEF_FindCicamTimerByHandle)

   return t;
}

#endif /*COMMON_INTERFACE*/

/**
 * @brief   Find or add a private network, assigning an unused private network ID.
 * @param   satellite optional satellite, can be NULL
 * @return  pointer to existing or new network record
 */
ADB_NETWORK_REC* DBDEF_FindOrAddPrivateNetwork(void *satellite)
{
   ADB_NETWORK_REC *n_ptr;
   U16BIT net_id;

   FUNCTION_START(DBDEF_FindOrAddPrivateNetwork);

   n_ptr = NULL;
   net_id = PRIVATE_NETWORK_ID0;

   /* Check to see if there's a private network for the given satellite */
   while ((n_ptr = DBDEF_FindNetworkRec(net_id, NULL)) != NULL)
   {
      if (n_ptr->satellite == satellite)
      {
         /* Found a private network for the given satellite */
         break;
      }

      /* Try the next private network id */
      net_id++;
   }

   if (n_ptr == NULL)
   {
      /* No network found so need to add one */
      if ((n_ptr = DBDEF_AddNetworkRec(net_id, satellite)) != NULL)
      {
         DBA_SaveRecord(n_ptr->dba_rec);
      }
   }

   FUNCTION_FINISH(DBDEF_FindOrAddPrivateNetwork);

   return(n_ptr);
}

//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------

/*!**************************************************************************
 * @brief   Deletes an LNB record from the RAM database
 * @param   lnb_ptr pointer to record to be deleted
 ****************************************************************************/
static void DeleteLNBRec(ADB_LNB_REC *lnb_ptr)
{
   FUNCTION_START(DeleteLNBRec);

   STB_LLRemoveBlock(&lnb_ptr->ptrs);

   if (lnb_ptr->name != NULL)
   {
      DBDEF_ReleaseString(lnb_ptr->name);
   }

   DBA_DestroyRecord(lnb_ptr->dba_rec);

   STB_AppFreeMemory(lnb_ptr);

   FUNCTION_FINISH(DeleteLNBRec);
}

/*!**************************************************************************
 * @brief   Deletes a satellite record from the RAM database
 * @param   sat_ptr pointer to record to be deleted
 ****************************************************************************/
static void DeleteSatelliteRec(ADB_SATELLITE_REC *sat_ptr)
{
   ADB_BAT_VERSION_REC *ver_rec;

   FUNCTION_START(DeleteSatelliteRec);

   STB_LLRemoveBlock(&sat_ptr->ptrs);

   if (sat_ptr->name != NULL)
   {
      DBDEF_ReleaseString(sat_ptr->name);
   }

   if (sat_ptr->def_authority != NULL)
   {
      DBDEF_ReleaseString(sat_ptr->def_authority);
   }

   ver_rec = sat_ptr->bat_version_list;
   while (ver_rec != NULL)
   {
      sat_ptr->bat_version_list = ver_rec->next;
      STB_AppFreeMemory(ver_rec);
      ver_rec = sat_ptr->bat_version_list;
   }

   DBA_DestroyRecord(sat_ptr->dba_rec);

   STB_AppFreeMemory(sat_ptr);

   FUNCTION_FINISH(DeleteSatelliteRec);
}

/**
 *

 *
 * @brief   Deletes specified network record
 *
 * @param   n_ptr - pointer to network record
 *

 *
 */
static void DeleteNetworkRec(ADB_NETWORK_REC *n_ptr)
{
   U8BIT i;

   FUNCTION_START(DeleteNetworkRec);

   STB_LLRemoveBlock(&n_ptr->ptrs);

   for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
   {
      if (n_ptr->name_array[i] != NULL)
      {
         STB_AppFreeMemory(n_ptr->name_array[i]);
      }
   }

   if (n_ptr->name_str != NULL)
   {
      STB_AppFreeMemory(n_ptr->name_str);
   }

   DBA_DestroyRecord(n_ptr->dba_rec);

   STB_AppFreeMemory(n_ptr);

   FUNCTION_FINISH(DeleteNetworkRec);
}

#ifdef DEBUG_PRINT_DATABASE
/*!**************************************************************************
 * @brief   Print details of the spcified LNB record
 * @param   lnb_ptr pointer to LNB record
 ****************************************************************************/
static void PrintLNBRec(ADB_LNB_REC *lnb_ptr)
{
   static const char *diseqc_cswitch[] = { "Off", "1", "2", "3", "4" };
   static const char *diseqc_tone_burst[] = { "Off", "A", "B" };

   FUNCTION_START(PrintLNBRec);

   STB_SPDebugWrite("\t\tType: %s", (lnb_ptr->type == LNB_TYPE_SINGLE ? "Single" : "Universal"));
   STB_SPDebugWrite("\t\tFreq lo: %d", lnb_ptr->freq_low);
   if (lnb_ptr->type == LNB_TYPE_UNIVERSAL)
   {
      STB_SPDebugWrite("\t\tFreq hi: %d", lnb_ptr->freq_high);
   }

   if (lnb_ptr->is_pulse_posn)
   {
      STB_SPDebugWrite("\t\tLNB motor: pulse");
   }
   else
   {
      if (lnb_ptr->is_diseqc_posn)
      {
         STB_SPDebugWrite("\t\tLNB motor: DiSEqC 1.2");
      }
      else
      {
         STB_SPDebugWrite("\t\tLNB motor: none");
      }
   }

   STB_SPDebugWrite("\t\tDiSEqC 1.0 (c-switch): %s", diseqc_cswitch[lnb_ptr->c_switch]);
   if (lnb_ptr->u_switch)
   {
      STB_SPDebugWrite("\t\tDiSEqC 1.1 (u-switch): %d", lnb_ptr->u_switch);
   }
   else
   {
      STB_SPDebugWrite("\t\tDiSEqC 1.1 (u-switch): None");
   }

   STB_SPDebugWrite("\t\tDiSEqC repeats: %d", lnb_ptr->diseqc_repeats);
   STB_SPDebugWrite("\t\tDiSEqC tone burst: %s", diseqc_tone_burst[lnb_ptr->diseqc_tone]);
   STB_SPDebugWrite("\t\t12v: %d", (lnb_ptr->is_12v ? "On" : "Off"));
   if (lnb_ptr->type == LNB_TYPE_SINGLE)
   {
      STB_SPDebugWrite("\t\t22k: %d", (lnb_ptr->is_22k ? "On" : "Off"));
   }

   STB_SPDebugWrite("\t\tSMATV %d", (lnb_ptr->is_smatv ? "On" : "Off"));

   FUNCTION_FINISH(PrintLNBRec);
}

/*!**************************************************************************
 * @brief   Print details of the spcified satellite record
 * @param   sat_ptr pointer to satellite record
 ****************************************************************************/
static void PrintSatelliteRec(ADB_SATELLITE_REC *sat_ptr)
{
   S16BIT long_pos;
   char ewc;
   FUNCTION_START(PrintSatelliteRec);

   if (sat_ptr->name.str_ptr != NULL)
   {
      STB_SPDebugWrite("\t\tSatellite name: \"%s\"", sat_ptr->name.str_ptr);
   }
   long_pos = sat_ptr->long_pos;
   if (long_pos < 0)
   {
      long_pos *= -1;
      ewc = 'W';
   }
   else
   {
      ewc = 'E';
   }
   STB_SPDebugWrite("\t\tPosition: %d.%01d %c", long_pos / 10, long_pos % 10, ewc );

   FUNCTION_FINISH(PrintSatelliteRec);
}

/**
 *

 *
 * @brief   Prints details of the specified network record
 *
 * @param   n_ptr - pointer to network record
 *

 *
 */
static void PrintNetworkRec(ADB_NETWORK_REC *n_ptr)
{
   U8BIT i;

   FUNCTION_START(PrintNetworkRec);

   STB_SPDebugWrite("\t\tNetwork id: %d", n_ptr->net_id);

   if (n_ptr->name_str != NULL)
   {
      STB_SPDebugWrite("\t\tNetwork name: \"%s\"", n_ptr->name_str->str_ptr);
   }

   STB_SPDebugWrite("\t\tDB rec: 0x%x", n_ptr->stb_db_rec);
#if 0
   if (n_ptr->stb_db_rec != NULL)
   {
      STB_PrintRecord(n_ptr->stb_db_rec);
   }
#endif

   for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
   {
      if ((n_ptr->name_array[i] != NULL) &&
          (n_ptr->name_array[i]->nbytes > 0))
      {
         STB_SPDebugWrite("\t\tName %d: %s", i, n_ptr->name_array[i]->str_ptr);
      }
   }

   STB_SPDebugWrite("\t\tProfile type: %u", n_ptr->profile_type);

   STB_SPDebugWrite("\t\tSatellite 0x%x", n_ptr->satellite);

   FUNCTION_FINISH(PrintNetworkRec);
}

#endif /* DEBUG_PRINT_DATABASE */

#ifdef DEBUG_PRINT_DATABASE
/**
 *

 *
 * @brief   Prints details of the specified transport record
 *
 * @param   t_ptr - pointer to transport record
 *

 *
 */
static void PrintTransportRec(ADB_TRANSPORT_REC *t_ptr)
{
   FUNCTION_START(PrintTransportRec);

   if (t_ptr->name_str != NULL)
   {
      STB_SPDebugWrite("\t\tTransport name: \"%s\"", ACTL_GetRfNameFromFreq(t_ptr->sig_type, t_ptr->freq_hz));
   }

   STB_SPDebugWrite("\t\tTransport id: %d", t_ptr->tran_id);
   STB_SPDebugWrite("\t\tOriginal network id: %d", t_ptr->orig_net_id);
   STB_SPDebugWrite("\t\tParent network: 0x%x", t_ptr->network);

   STB_SPDebugWrite("\t\tDB rec: 0x%x", t_ptr->stb_db_rec);
#if 0
   if (t_ptr->stb_db_rec != NULL)
   {
      STB_PrintRecord(t_ptr->stb_db_rec);
   }
#endif

   STB_SPDebugWrite("\t\tFreq: %dHz", t_ptr->freq_hz);
   STB_SPDebugWrite("\t\tPAT version: %d", t_ptr->pat_version);
   STB_SPDebugWrite("\t\tSDT version: %d", t_ptr->sdt_version);
   STB_SPDebugWrite("\t\tPMT version list: 0x%x", t_ptr->pmt_version_list);
//   STB_SPDebugWrite("\t\tEIT version list: 0x%x", t_ptr->eit_version_list);

   FUNCTION_FINISH(PrintTransportRec);
}

#endif

#ifdef DEBUG_PRINT_DATABASE
/**
 *

 *
 * @brief   Prints details of the specified service record
 *
 * @param   s_ptr - pointer to service record
 *

 *
 */
static void PrintServiceRec(ADB_SERVICE_REC *s_ptr)
{
   FUNCTION_START(PrintServiceRec);

   if (s_ptr->name_str != NULL)
   {
      STB_SPDebugWrite("\t\tService name: \"%s\"", s_ptr->name_str->str_ptr);
   }

   STB_SPDebugWrite("\t\tService id: %d", s_ptr->serv_id);

   if (s_ptr->provider_str != NULL)
   {
      STB_SPDebugWrite("\t\tProvider: \"%s\"", s_ptr->provider_str->str_ptr);
   }

   STB_SPDebugWrite("\t\tDB rec: 0x%x", s_ptr->stb_db_rec);
#if 0
   if (s_ptr->stb_db_rec != NULL)
   {
      STB_PrintRecord(s_ptr->stb_db_rec);
   }
#endif

   STB_SPDebugWrite("\t\tService LCN: %d", s_ptr->serv_lcn);
   STB_SPDebugWrite("\t\tAllocated LCN: %d", s_ptr->allocated_lcn);
   STB_SPDebugWrite("\t\tScrambled: %s", (s_ptr->scrambled ? "Yes" : "No"));
   STB_SPDebugWrite("\t\tParent transport: 0x%x", s_ptr->transport);
   STB_SPDebugWrite("\t\tNow/next available: %s", (s_ptr->eit_now_next_avail ? "Yes" : "No"));
   STB_SPDebugWrite("\t\tSchedule available: %s", (s_ptr->eit_sched_avail ? "Yes" : "No"));
   STB_SPDebugWrite("\t\tDeleted: %u", s_ptr->deleted);

   STB_SPDebugWrite("\t\tPCR PID: %d", s_ptr->pcr_pid);
   STB_SPDebugWrite("\t\tVideo PID: %d", s_ptr->video_pid);
   STB_SPDebugWrite("\t\tAudio PID: %d", s_ptr->audio_pid);
   STB_SPDebugWrite("\t\tSubtitle PID: %d", s_ptr->subtitle_pid);
   STB_SPDebugWrite("\t\tTeletext PID: %d", s_ptr->ttext_pid);
   STB_SPDebugWrite("\t\tTeletext lang code: 0x%x", s_ptr->reqd_ttext_lang_code);

   STB_SPDebugWrite("\t\tRunning status: %u", s_ptr->running_status);
   STB_SPDebugWrite("\t\tService running: %s", (s_ptr->not_running ? "No" : "Yes"));

   FUNCTION_FINISH(PrintServiceRec);
}

#endif

/*!**************************************************************************
 * @brief    Gets a required audio pid
 * @param    s_ptr service record ptr
 * @param    audio_mode pointer to the returned channel setting (mono, stereo, left, right)
 * @param    audio_type pointer to the type of audio
 * @param    db_lang_ids Language ids to be matching
 * @param    primary_found If it found the required language
 * @return   U16BIT - Found Pid
 ****************************************************************************/
static U16BIT GetReqdAudioPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *audio_mode,
   ADB_STREAM_TYPE *audio_type, U8BIT *db_lang_ids, BOOLEAN *primary_found, ADB_STREAM_REC **selected_stream)
{
   U16BIT audio_pid;
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *nzero_pid_stream;
   ADB_STREAM_REC *lang_code_match_stream;
   ADB_STREAM_REC *qaa_match_stream;
   ADB_STREAM_REC *audio_type_match_stream;
   ADB_STREAM_REC *default_match_stream;
   ADB_STREAM_REC *exact_match_stream;
   U32BIT def_lang_code;
   ADB_AUDIO_TYPE def_audio_type;
   BOOLEAN prefer_hd_stream;
   U8BIT stream_priority;
   U8BIT def_stream_priority;
   U8BIT lang_stream_priority;
   U8BIT qaa_stream_priority;
   U8BIT type_stream_priority;
   U8BIT nzero_stream_priority;
   U8BIT lang_lang_priority;
   U8BIT def_lang_priority;
   U8BIT i;

   FUNCTION_START(GetReqdAudioPid);

   audio_pid = 0;
   *audio_mode = AUDIO_STEREO;
   *audio_type = ADB_AUDIO_STREAM;
   *primary_found = FALSE;
   *selected_stream = NULL;

   if (s_ptr != NULL)
   {
      nzero_pid_stream = NULL;
      lang_code_match_stream = NULL;
      qaa_match_stream = NULL;
      audio_type_match_stream = NULL;
      default_match_stream = NULL;
      exact_match_stream = NULL;
      def_stream_priority = 0;
      lang_stream_priority = 0;
      qaa_stream_priority = 0;
      type_stream_priority = 0;
      nzero_stream_priority = 0;
      lang_lang_priority = ACFG_MAX_DB_LANG_CODES;
      def_lang_priority = ACFG_MAX_DB_LANG_CODES;

      def_audio_type = (ADB_AUDIO_TYPE)APP_NvmRead(AUDIO_TYPE_NVM);
      prefer_hd_stream = APP_NvmRead(PREFER_HD_AUDIO_NVM);

      stream_rec = s_ptr->stream_list;
      while (stream_rec != NULL)
      {
         if ((stream_rec->type == ADB_AUDIO_STREAM) ||
             (stream_rec->type == ADB_AAC_AUDIO_STREAM) ||
             (stream_rec->type == ADB_HEAAC_AUDIO_STREAM) ||
             (stream_rec->type == ADB_AC3_AUDIO_STREAM) ||
             (stream_rec->type == ADB_EAC3_AUDIO_STREAM)
             )
         {
            // ensure in use flag is cleared
            stream_rec->in_use = FALSE;

            if (stream_rec->pid != 0)
            {
               // if the audio has been required by PID and it matches, consider this an exact match
               if ((s_ptr->reqd_audio_pid < 0x1FFF) && (s_ptr->reqd_audio_pid == stream_rec->pid))
               {
                  exact_match_stream = stream_rec;
                  *primary_found = TRUE;
               }
               // is this audio stream an exact match for the reqd audio settings?
               else if ((s_ptr->reqd_audio_valid == TRUE) &&
                      (exact_match_stream == NULL) &&
                   (stream_rec->data.audio.lang_code == s_ptr->reqd_audio_lang_code) &&
                   (stream_rec->data.audio.type == s_ptr->reqd_audio_type) &&
                   (stream_rec->type == s_ptr->reqd_stream_type))
               {
                  // yes - found it!
                  exact_match_stream = stream_rec;
                  *primary_found = TRUE;
               }
               else
               {
                  stream_priority = AudioStreamPriority(stream_rec->type);

                  if (db_lang_ids != NULL)
                  {
                     // if not an exact match, check if this stream matches the default lang and type
                     for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) && (db_lang_ids[i] != ACFG_INVALID_DB_LANG); i++)
                     {
                        def_lang_code = ACFG_ConvertLangIdToCode(db_lang_ids[i]);

                        if ((def_lang_code == stream_rec->data.audio.lang_code) &&
                            ((def_audio_type == ADB_AUDIO_TYPE_UNDEFINED) ||
                             (def_audio_type == ADB_AUDIO_TYPE_CLEAN_EFFECTS)) &&
                            (def_audio_type == stream_rec->data.audio.type))
                        {
                           if ((default_match_stream == NULL) || (i < def_lang_priority))
                           {
                              /* No stream selected yet so use this one */
                              default_match_stream = stream_rec;
                              def_stream_priority = stream_priority;
                              def_lang_priority = i;
                              *primary_found = TRUE;
                           }
                           else
                           {
                              /* Check whether this stream uses a better encoding
                               * than the currently selected one */
                              if (!prefer_hd_stream)
                              {
                                 /* The minimum match needs to be found if the output is PCM */
                                 if (stream_priority < def_stream_priority)
                                 {
                                    /* This stream is a better match so use it instead */
                                    default_match_stream = stream_rec;
                                    def_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == def_stream_priority)
                                 {
                                    /* Both streams use the same codec so a non-multichannel one
                                     * should be selected, if available */
                                    if ((default_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       default_match_stream = stream_rec;
                                       def_stream_priority = stream_priority;
                                    }
                                 }
                              }
                              else
                              {
                                 if (stream_priority > def_stream_priority)
                                 {
                                    /* This stream has better audio encoding so select it instead */
                                    default_match_stream = stream_rec;
                                    def_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == def_stream_priority)
                                 {
                                    /* Both streams use the same codec so a multichannel one
                                     * should be selected, if available */
                                    if ((default_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       default_match_stream = stream_rec;
                                       def_stream_priority = stream_priority;
                                    }
                                 }
                              }
                           }
                        }

                        /* Does it match the default language, but not if it's an AD stream?
                         * In the D-Book, "und" is defined as being assumed to be English, in
                         * which case it should be chosen above "qaa". */
                        if (((def_lang_code == stream_rec->data.audio.lang_code) ||
                             (stream_rec->data.audio.lang_code == ADB_LANG_CODE_UNDEF)) &&
                            (stream_rec->data.audio.type != ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED))
                        {
                           if ((lang_code_match_stream == NULL) || (i < lang_lang_priority))
                           {
                              /* No stream selected yet so use this one */
                              lang_code_match_stream = stream_rec;
                              lang_stream_priority = stream_priority;
                              lang_lang_priority = i;
                           }
                           else
                           {
                              /* Check whether this stream uses a better encoding
                               * than the currently selected one */
                              if (!prefer_hd_stream)
                              {
                                 /* The minimum match needs to be found if the output is PCM */
                                 if (stream_priority < lang_stream_priority)
                                 {
                                    /* This stream is a better match so use it instead */
                                    lang_code_match_stream = stream_rec;
                                    lang_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == lang_stream_priority)
                                 {
                                    /* Both streams use the same codec so a non-multichannel one
                                     * should be selected, if available */
                                    if ((lang_code_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       lang_code_match_stream = stream_rec;
                                       lang_stream_priority = stream_priority;
                                    }
                                 }
                              }
                              else
                              {
                                 if (stream_priority > lang_stream_priority)
                                 {
                                    /* This stream has better audio encoding so select it instead */
                                    lang_code_match_stream = stream_rec;
                                    lang_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == lang_stream_priority)
                                 {
                                    /* Both streams use the same codec so a multichannel one
                                     * should be selected, if available */
                                    if ((lang_code_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       lang_code_match_stream = stream_rec;
                                       lang_stream_priority = stream_priority;
                                    }
                                 }
                              }
                           }
                        }

                        /* Is it the Original Language Soundtrack? */
                        if ((stream_rec->data.audio.lang_code == ADB_LANG_CODE_QAA) &&
                            (stream_rec->data.audio.type != ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED))
                        {
                           if (qaa_match_stream == NULL)
                           {
                              qaa_match_stream = stream_rec;
                              qaa_stream_priority = stream_priority;
                           }
                           else
                           {
                              /* Check whether this stream uses a better encoding
                               * than the currently selected one */
                              if (!prefer_hd_stream)
                              {
                                 /* The minimum match needs to be found if the output is PCM */
                                 if (stream_priority < qaa_stream_priority)
                                 {
                                    /* This stream is a better match so use it instead */
                                    qaa_match_stream = stream_rec;
                                    qaa_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == qaa_stream_priority)
                                 {
                                    /* Both streams use the same codec so a non-multichannel one
                                     * should be selected, if available */
                                    if ((qaa_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       qaa_match_stream = stream_rec;
                                       qaa_stream_priority = stream_priority;
                                    }
                                 }
                              }
                              else
                              {
                                 if (stream_priority > qaa_stream_priority)
                                 {
                                    /* This stream has better audio encoding so select it instead */
                                    qaa_match_stream = stream_rec;
                                    qaa_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == qaa_stream_priority)
                                 {
                                    /* Both streams use the same codec so a multichannel one
                                     * should be selected, if available */
                                    if ((qaa_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       qaa_match_stream = stream_rec;
                                       qaa_stream_priority = stream_priority;
                                    }
                                 }
                              }
                           }
                        }
                     }
                  }

                  // does it match the default type only
                  if (((def_audio_type == ADB_AUDIO_TYPE_UNDEFINED) ||
                       (def_audio_type == ADB_AUDIO_TYPE_CLEAN_EFFECTS)) &&
                      (def_audio_type == stream_rec->data.audio.type))
                  {
                     if (audio_type_match_stream == NULL)
                     {
                        audio_type_match_stream = stream_rec;
                        type_stream_priority = stream_priority;
                     }
                     else
                     {
                        /* Check whether this stream uses a better encoding
                         * than the currently selected one */
                        if (!prefer_hd_stream)
                        {
                           /* The minimum match needs to be found if the output is PCM */
                           if (stream_priority < type_stream_priority)
                           {
                              /* This stream is a better match so use it instead */
                              audio_type_match_stream = stream_rec;
                              type_stream_priority = stream_priority;
                           }
                           else if (stream_priority == type_stream_priority)
                           {
                              /* Both streams use the same codec so a non-multichannel one
                               * should be selected, if available */
                              if ((audio_type_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                  (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                              {
                                 /* This stream is a better match so use it instead */
                                 audio_type_match_stream = stream_rec;
                                 type_stream_priority = stream_priority;
                              }
                           }
                        }
                        else
                        {
                           if (stream_priority > type_stream_priority)
                           {
                              /* This stream has better audio encoding so select it instead */
                              audio_type_match_stream = stream_rec;
                              type_stream_priority = stream_priority;
                           }
                           else if (stream_priority == type_stream_priority)
                           {
                              /* Both streams use the same codec so a multichannel one
                               * should be selected, if available */
                              if ((audio_type_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                  (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                              {
                                 /* This stream is a better match so use it instead */
                                 audio_type_match_stream = stream_rec;
                                 type_stream_priority = stream_priority;
                              }
                           }
                        }
                     }
                  }

                  // is this the first non-zero pid
                  if (nzero_pid_stream == NULL)
                  {
                     nzero_pid_stream = stream_rec;
                     nzero_stream_priority = stream_priority;
                  }
                  else
                  {
                     /* Check whether this stream uses a better encoding
                      * than the currently selected one */
                     if (!prefer_hd_stream)
                     {
                        /* The minimum match needs to be found if the output is PCM */
                        if (stream_priority < nzero_stream_priority)
                        {
                           /* This stream is a better match so use it instead */
                           nzero_pid_stream = stream_rec;
                           nzero_stream_priority = stream_priority;
                        }
                        else if (stream_priority == nzero_stream_priority)
                        {
                           /* Both streams use the same codec so a non-multichannel one
                            * should be selected, if available */
                           if ((nzero_pid_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                               (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                           {
                              /* This stream is a better match so use it instead */
                              nzero_pid_stream = stream_rec;
                              nzero_stream_priority = stream_priority;
                           }
                        }
                     }
                     else
                     {
                        if (stream_priority > nzero_stream_priority)
                        {
                           /* This stream has better audio encoding so select it instead */
                           nzero_pid_stream = stream_rec;
                           nzero_stream_priority = stream_priority;
                        }
                        else if (stream_priority == nzero_stream_priority)
                        {
                           /* Both streams use the same codec so a multichannel one
                            * should be selected, if available */
                           if ((nzero_pid_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                               (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                           {
                              /* This stream is a better match so use it instead */
                              nzero_pid_stream = stream_rec;
                              nzero_stream_priority = stream_priority;
                           }
                        }
                     }
                  }
               }
            }
         }
         stream_rec = stream_rec->next;
      }

      // choose the stream which provides the best match to the requirements
      stream_rec = exact_match_stream;
      if (stream_rec == NULL)
      {
         stream_rec = default_match_stream;
         if (stream_rec == NULL)
         {
            stream_rec = lang_code_match_stream;
            if (stream_rec == NULL)
            {
               stream_rec = qaa_match_stream;
               if (stream_rec == NULL)
               {
                  stream_rec = audio_type_match_stream;
                  if (stream_rec == NULL)
                  {
                     stream_rec = nzero_pid_stream;
                  }
               }
            }
         }
      }

      // if a stream has been found mark it in use
      if (stream_rec != NULL)
      {
         audio_pid = stream_rec->pid;
         *audio_mode = stream_rec->data.audio.mode;
         *audio_type = stream_rec->type;
         *selected_stream = stream_rec;
      }
   }

   FUNCTION_FINISH(GetReqdAudioPid);

   return(audio_pid);
}

/*!**************************************************************************
 * @brief   Return the audio priority of the given stream type based on a fixed table
 * @param   stream_type audio stream type
 * @return  priority value - higher is better
 ****************************************************************************/
static U8BIT AudioStreamPriority(ADB_STREAM_TYPE stream_type)
{
   S8BIT priority;
   U16BIT num_entries;
   U8BIT i;

   FUNCTION_START(AudioStreamPriority);

   priority = 0;
   num_entries = sizeof(audio_priority_table) / sizeof(audio_priority_table[0]);

   for (i = 0; i < num_entries; i++)
   {
      if (stream_type == audio_priority_table[i])
      {
         priority = i;
         break;
      }
   }

   FUNCTION_FINISH(AudioStreamPriority);

   return(priority);
}

/*!**************************************************************************
 * @brief    Gets a required subtitle pid
 * @param    s_ptr service record ptr
 * @param    pid_ptr   - pointer for the return of the pid
 * @param    cpage_ptr pointer for the return of the composition page
 * @param    apage_ptr pointer for the return of the ancilliary page
 * @param    db_lang_ids Language ids to be matching
 * @param    primary_found If it found the required language
 * @return   U16BIT - Found Pid
 ****************************************************************************/
static E_STREAM_MATCH_TYPE GetReqdSubtitleParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr,
   U16BIT *cpage_ptr, U16BIT *apage_ptr, U8BIT *db_lang_ids, BOOLEAN *primary_found,
   ADB_STREAM_REC **selected_stream)
{
   E_STREAM_MATCH_TYPE match_type;
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *nzero_pid_stream;
   ADB_STREAM_REC *undef_match_stream;
   ADB_STREAM_REC *default_match_stream;
   ADB_STREAM_REC *aspect_match_stream;
   ADB_STREAM_REC *exact_match_stream;
   U32BIT def_lang_code;
   E_STB_AV_ASPECT_RATIO aspect_ratio;
   U8BIT i;
   E_SUBTITLE_TYPE subtitle_type;

   FUNCTION_START(GetReqdSubtitleParams);

   *primary_found = FALSE;
   *selected_stream = NULL;

   if (s_ptr != NULL)
   {
      nzero_pid_stream = NULL;
      undef_match_stream = NULL;
      default_match_stream = NULL;
      aspect_match_stream = NULL;
      exact_match_stream = NULL;

      aspect_ratio = (E_STB_AV_ASPECT_RATIO)APP_NvmRead(ASPECT_RATIO_NVM);
      subtitle_type = (E_SUBTITLE_TYPE)APP_NvmRead(SUBTITLE_TYPE_NVM);

      stream_rec = s_ptr->stream_list;
      while (stream_rec != NULL)
      {
         if (stream_rec->type == ADB_SUBTITLE_STREAM)
         {
            // ensure in use flag is cleared
            stream_rec->in_use = FALSE;

            if (stream_rec->pid != 0)
            {
               // is this stream an exact match for the reqd settings?
               if ((s_ptr->reqd_subtitle_valid == TRUE) &&
                   (exact_match_stream == NULL) &&
                   (stream_rec->data.subtitle.lang_code == s_ptr->reqd_subtitle_lang_code) &&
                   (stream_rec->data.subtitle.type == s_ptr->reqd_subtitle_type))
               {
                  // yes - found it!
                  exact_match_stream = stream_rec;
                  *primary_found = TRUE;
               }
               else
               {
                  if (db_lang_ids != NULL)
                  {
                     /* Not an exact match so check whether this stream matches the
                      * default lang and aspect ratio */
                     for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) && (db_lang_ids[i] != ACFG_INVALID_DB_LANG); i++)
                     {
                        def_lang_code = ACFG_ConvertLangIdToCode(db_lang_ids[i]);

                        if (subtitle_type == SUBTITLE_NORMAL)
                        {
                           if ((aspect_match_stream == NULL) &&
                               (def_lang_code == stream_rec->data.subtitle.lang_code) &&
                               (((aspect_ratio == ASPECT_RATIO_4_3) &&
                                 (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_4_3)) ||
                                ((aspect_ratio == ASPECT_RATIO_16_9) &&
                                 (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_16_9))))
                           {
                              aspect_match_stream = stream_rec;
                              *primary_found = TRUE;
                           }
                        }
                        else
                        {
                           if ((aspect_match_stream == NULL) &&
                               (def_lang_code == stream_rec->data.subtitle.lang_code) &&
                               (((aspect_ratio == ASPECT_RATIO_4_3) &&
                                 (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3)) ||
                                ((aspect_ratio == ASPECT_RATIO_16_9) &&
                                 (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9))))
                           {
                              aspect_match_stream = stream_rec;
                              *primary_found = TRUE;
                           }
                        }

                        // if not an exact match, check if this stream matches the default lang
                        if (subtitle_type == SUBTITLE_NORMAL)
                        {
                           if ((default_match_stream == NULL) &&
                               (def_lang_code == stream_rec->data.subtitle.lang_code) &&
                               ((stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_4_3) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_16_9) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HD)))
                           {
                              default_match_stream = stream_rec;
                              *primary_found = TRUE;
                           }
                        }
                        else
                        {
                           /* If both DVB and HoH subtitle present, it will select HoH subtitle
                            * as it matches with the user setting, if there is only DVB subtitle
                            * available, we still need to show the DVB subtitle even though the
                            * user setting is HoH */
                           if ((default_match_stream == NULL) && (def_lang_code == stream_rec->data.subtitle.lang_code) &&
                               ((stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_4_3) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_16_9) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HD)))
                           {
                              default_match_stream = stream_rec;
                              *primary_found = TRUE;
                           }
                           if ((def_lang_code == stream_rec->data.subtitle.lang_code) &&
                               ((stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9) ||
                                (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_HD)))
                           {
                              default_match_stream = stream_rec;
                              *primary_found = TRUE;
                           }
                        }
                     }
                  }

                  // does it match the default type only
                  if ((undef_match_stream == NULL) &&
                      ((stream_rec->data.subtitle.lang_code == ADB_LANG_CODE_UNDEF) ||
                       (stream_rec->data.subtitle.lang_code == ADB_LANG_CODE_QAA)))
                  {
                     undef_match_stream = stream_rec;
                  }

                  // is this the first non-zero pid
                  if (nzero_pid_stream == NULL)
                  {
                     nzero_pid_stream = stream_rec;
                  }
               }
            }
         }
         stream_rec = stream_rec->next;
      }

      // choose the stream which provides the best match to the requirements
      stream_rec = exact_match_stream;
      match_type = STREAM_MATCH_EXACT;
      if (stream_rec == NULL)
      {
         stream_rec = aspect_match_stream;

         match_type = STREAM_MATCH_ASPECT;
         if (stream_rec == NULL)
         {
            stream_rec = default_match_stream;
            match_type = STREAM_MATCH_LANG;
            if (stream_rec == NULL)
            {
               stream_rec = undef_match_stream;
               match_type = STREAM_MATCH_TYPE;
               if (stream_rec == NULL)
               {
                  stream_rec = nzero_pid_stream;
                  match_type = STREAM_MATCH_NOT_ZERO;

                  if ((stream_rec != NULL) && ACFG_IsNordigCountry())
                  {
                     /* match_type should be set to STREAM_MATCH_NONE if HoH(Hard of Hearing) was
                      * disabled in order to disable the HoH subtitle even if there is an HoH
                      * subtitle present */
                     if ((subtitle_type == SUBTITLE_NORMAL) &&
                        ((stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING) ||
                         (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3) ||
                         (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9)))
                     {
                        /* set stream_rec to NULL in order to set match_type = STREAM_MATCH_NONE as Nordig spec wouldn't show
                           subtitle if above case not matched */
                        stream_rec = NULL;
                     }
                  }
               }
            }
         }
      }

      // if a stream has been found, return details
      if ((stream_rec != NULL) &&
         ((stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_4_3) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_16_9) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_221_1) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HD) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_4_3) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_16_9) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_221_1) ||
          (stream_rec->data.subtitle.type == ADB_SUBTITLE_TYPE_DVB_HARD_HEARING_HD)))
      {
         *pid_ptr = stream_rec->pid;
         *cpage_ptr = stream_rec->data.subtitle.composition_page;
         *apage_ptr = stream_rec->data.subtitle.ancillary_page;
         *selected_stream = stream_rec;
      }
      else
      {
         *pid_ptr = 0;
         *cpage_ptr = 0;
         *apage_ptr = 0;
         match_type = STREAM_MATCH_NONE;
      }
   }
   else
   {
      *pid_ptr = 0;
      *cpage_ptr = 0;
      *apage_ptr = 0;
      match_type = STREAM_MATCH_NONE;
   }

   FUNCTION_FINISH(GetReqdSubtitleParams);

   return(match_type);
}

/**
 *

 *
 * @brief   Gets the appropriate teletext pid - looks first for the params matching exactly the
 *                required teletext settings in the database, then for the pid
 *                matching the default language, then the first teletext stream in the stream list.
 *                If no teletext found the returned pid will be 0
 *
 * @param   s_ptr - service record ptr
 * @param   pid_ptr - pointer for the return of the pid
 * @param   for_subtitles - TRUE if subtitle stream is to be selected
 * @param   magazine - returns the magazine defined in the stream
 * @param   page - returns the defined page in the stream
 * @param   db_lang_ids - Language ids to be matching
 * @param   primary_found - If it found the required language
 *
 * @return   The required PID value
 *
 */
static E_STREAM_MATCH_TYPE GetReqdTtextParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr, BOOLEAN for_subtitles,
   U8BIT *magazine, U8BIT *page, U8BIT *db_lang_ids, BOOLEAN *primary_found, ADB_STREAM_REC **selected_stream)
{
   E_STREAM_MATCH_TYPE match_type;
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *nzero_pid_stream;
   ADB_STREAM_REC *default_match_stream;
   ADB_STREAM_REC *exact_match_stream;
   ADB_STREAM_REC *normal_subt_nzero_pid_stream;
   ADB_STREAM_REC *normal_subt_default_match_stream;
   U32BIT def_lang_code;
   U8BIT i;
   E_SUBTITLE_TYPE subtitle_type;

   FUNCTION_START(GetReqdTtextParams);

   match_type = STREAM_MATCH_NONE;
   *primary_found = FALSE;
   *selected_stream = NULL;

   subtitle_type = (E_SUBTITLE_TYPE)APP_NvmRead(SUBTITLE_TYPE_NVM);

   if (s_ptr != NULL)
   {
      nzero_pid_stream = NULL;
      default_match_stream = NULL;
      exact_match_stream = NULL;
      normal_subt_nzero_pid_stream = NULL;
      normal_subt_default_match_stream = NULL;

      stream_rec = s_ptr->stream_list;
      while (stream_rec != NULL)
      {
         if (stream_rec->type == ADB_TTEXT_STREAM)
         {
            // ensure in use flag is cleared
            stream_rec->in_use = FALSE;

            if (stream_rec->pid != 0)
            {
               // is this stream an exact match for the reqd settings?
               if ((s_ptr->reqd_ttext_valid == TRUE) && (exact_match_stream == NULL) &&
                   (stream_rec->data.ttext.lang_code == s_ptr->reqd_ttext_lang_code) &&
                   (stream_rec->data.ttext.type == s_ptr->reqd_ttext_type))
               {
                  // yes - found it!
                  exact_match_stream = stream_rec;
                  *primary_found = TRUE;
               }
               else
               {
                  if (db_lang_ids != NULL)
                  {
                     // if not an exact match, check if this stream matches the default lang
                     for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) && (db_lang_ids[i] != ACFG_INVALID_DB_LANG); i++)
                     {
                        def_lang_code = ACFG_ConvertLangIdToCode(db_lang_ids[i]);

                        if ((default_match_stream == NULL) &&
                            (def_lang_code == stream_rec->data.ttext.lang_code) &&
                            ((for_subtitles && (UserPrefsMatch(stream_rec->data.ttext.type, subtitle_type) == TRUE)) ||
                             (!for_subtitles && (stream_rec->data.ttext.type != ADB_TELETEXT_TYPE_SUBTITLE) &&
                              (stream_rec->data.ttext.type != ADB_TELETEXT_TYPE_SUBTITLE_HARD_HEARING) &&
                              (stream_rec->data.ttext.type == ADB_TELETEXT_TYPE_INITIAL))))
                        {
                           default_match_stream = stream_rec;
                           *primary_found = TRUE;
                        }
                        else if ((normal_subt_default_match_stream == NULL) &&
                                 (def_lang_code == stream_rec->data.ttext.lang_code) &&
                                 (for_subtitles && (stream_rec->data.ttext.type == ADB_TELETEXT_TYPE_SUBTITLE) &&
                                 (subtitle_type == SUBTITLE_HARD_OF_HEARING)))
                        {
                           normal_subt_default_match_stream = stream_rec;
                           *primary_found = TRUE;
                        }
                     }
                  }

                  // is this the first non-zero pid
                  if ((nzero_pid_stream == NULL) &&
                      ((for_subtitles && (UserPrefsMatch(stream_rec->data.ttext.type, subtitle_type) == TRUE)) ||
                       (!for_subtitles && (stream_rec->data.ttext.type != ADB_TELETEXT_TYPE_SUBTITLE))))
                  {
                     nzero_pid_stream = stream_rec;
                  }
                  else if ((normal_subt_nzero_pid_stream == NULL) &&
                           (for_subtitles && (stream_rec->data.ttext.type == ADB_TELETEXT_TYPE_SUBTITLE) &&
                           (subtitle_type == SUBTITLE_HARD_OF_HEARING)))
                  {
                     normal_subt_nzero_pid_stream = stream_rec;
                  }
               }
            }
         }
         stream_rec = stream_rec->next;
      }

      // choose the stream which provides the best match to the requirements
      stream_rec = exact_match_stream;
      match_type = STREAM_MATCH_EXACT;
      if (stream_rec == NULL)
      {
         stream_rec = default_match_stream;
         match_type = STREAM_MATCH_LANG;
         if (stream_rec == NULL)
         {
            stream_rec = normal_subt_default_match_stream;
            match_type = STREAM_MATCH_LANG;
            if (stream_rec == NULL)
            {
               stream_rec = nzero_pid_stream;
               match_type = STREAM_MATCH_NOT_ZERO;
               if (stream_rec == NULL)
               {
                  stream_rec = normal_subt_nzero_pid_stream;
                  match_type = STREAM_MATCH_NOT_ZERO;

                  if ((stream_rec != NULL) && ACFG_IsNordigCountry())
                  {
                     /* set stream_rec to NULL in order to set match_type = STREAM_MATCH_NONE as
                      * Nordig spec wouldn't show teletext if the above case is not matched */
                     stream_rec = NULL;
                  }
               }
               else if (ACFG_IsNordigCountry())
               {
                  /* set stream_rec to NULL in order to set match_type = STREAM_MATCH_NONE as
                   * Nordig spec wouldn't show teletext if the above cae is not matched */
                  stream_rec = NULL;
               }
            }
         }
      }

      // if a stream has been found mark it in use and return details
      if (stream_rec != NULL)
      {
         *pid_ptr = stream_rec->pid;
         *magazine = stream_rec->data.ttext.magazine;
         *page = stream_rec->data.ttext.page;
         *selected_stream = stream_rec;
      }
      else
      {
         *pid_ptr = 0;
         *magazine = 0;
         *page = 0;
      }
   }
   else
   {
      *pid_ptr = 0;
      *magazine = 0;
      *page = 0;
   }

   FUNCTION_FINISH(GetReqdTtextParams);

   return(match_type);
}

/**
 * @brief   Gets a required AD pid
 * @param   s_ptr service record ptr
 * @param   audio_mode pointer to the returned channel setting (mono, stereo, left, right)
 * @param   db_lang_ids language ids used for matching
 * @param   selected_stream returns with the pointer to the selected stream, NULL if not found
 * @param   broadcast_mix returned as TRUE if returns stream is broadcast mix AD
 * @return  AD PID
 */
static U16BIT GetReqdADPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *ad_mode, U8BIT *db_lang_ids,
   ADB_STREAM_REC **selected_stream, BOOLEAN *broadcast_mix)
{
   U16BIT ad_pid;
   ADB_STREAM_REC *stream_rec;
   ADB_STREAM_REC *lang_code_match_stream;
   ADB_STREAM_REC *exact_match_stream;
   ADB_STREAM_REC *nar_match_stream;
   U32BIT def_lang_code;
   BOOLEAN prefer_hd_stream;
   U8BIT stream_priority;
   U8BIT lang_stream_priority;
   U8BIT nar_stream_priority;
   U8BIT i;

   FUNCTION_START(GetReqdADPid);

   ad_pid = 0;
   *ad_mode = AUDIO_MONO;
   *selected_stream = NULL;
   *broadcast_mix = FALSE;

   if (s_ptr != NULL)
   {
      lang_code_match_stream = NULL;
      exact_match_stream = NULL;
      nar_match_stream = NULL;

      prefer_hd_stream = APP_NvmRead(PREFER_HD_AUDIO_NVM);
      db_lang_ids = DBDEF_GetAudioLang();

      stream_rec = s_ptr->stream_list;
      while (stream_rec != NULL)
      {
         if ((stream_rec->type == ADB_AUDIO_STREAM) ||
             (stream_rec->type == ADB_AAC_AUDIO_STREAM) ||
             (stream_rec->type == ADB_HEAAC_AUDIO_STREAM) ||
             (stream_rec->type == ADB_AC3_AUDIO_STREAM) ||
             (stream_rec->type == ADB_EAC3_AUDIO_STREAM)
             )
         {
            // ensure in use flag is cleared
            stream_rec->in_use = FALSE;

            if (stream_rec->pid != 0)
            {
               // is this audio stream an exact match for the reqd audio settings?
               if ((s_ptr->reqd_audio_valid == TRUE) &&
                   (exact_match_stream == NULL) &&
                   (stream_rec->data.audio.lang_code == s_ptr->reqd_audio_lang_code) &&
                   (stream_rec->data.audio.type == ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED))
               {
                  // yes - found it!
                  exact_match_stream = stream_rec;
               }
               else
               {
                  stream_priority = AudioStreamPriority(stream_rec->type);
                  if (db_lang_ids != NULL)
                  {
                     /* If not an exact match, does it match the default language */
                     for (i = 0; (i < ACFG_MAX_DB_LANG_CODES) && (db_lang_ids[i] != ACFG_INVALID_DB_LANG); i++)
                     {
                        def_lang_code = ACFG_ConvertLangIdToCode(db_lang_ids[i]);

                        if (((def_lang_code == stream_rec->data.audio.lang_code) ||
                             (stream_rec->data.audio.lang_code == ADB_LANG_CODE_UNDEF)) &&
                            (stream_rec->data.audio.type == ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED))
                        {
                           if (lang_code_match_stream == NULL)
                           {
                              /* No stream selected yet so use this one */
                              lang_code_match_stream = stream_rec;
                              lang_stream_priority = stream_priority;
                           }
                           else
                           {
                              /* Check whether this stream uses a better encoding
                               * than the currently selected one */
                              if (!prefer_hd_stream)
                              {
                                 /* The minimum match needs to be found if the output is PCM */
                                 if (stream_priority < lang_stream_priority)
                                 {
                                    /* This stream is a better match so use it instead */
                                    lang_code_match_stream = stream_rec;
                                    lang_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == lang_stream_priority)
                                 {
                                    /* Both streams use the same codec so a non-multichannel one
                                     * should be selected, if available */
                                    if ((lang_code_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       lang_code_match_stream = stream_rec;
                                       lang_stream_priority = stream_priority;
                                    }
                                 }
                              }
                              else
                              {
                                 if (stream_priority > lang_stream_priority)
                                 {
                                    /* This stream has better audio encoding so select it instead */
                                    lang_code_match_stream = stream_rec;
                                    lang_stream_priority = stream_priority;
                                 }
                                 else if (stream_priority == lang_stream_priority)
                                 {
                                    /* Both streams use the same codec so a multichannel one
                                     * should be selected, if available */
                                    if ((lang_code_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                        (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                                    {
                                       /* This stream is a better match so use it instead */
                                       lang_code_match_stream = stream_rec;
                                       lang_stream_priority = stream_priority;
                                    }
                                 }
                              }
                           }
                        }
                     }
                  }

                  if (stream_rec->data.audio.lang_code == ADB_LANG_CODE_NAR)
                  {
                     if (nar_match_stream == NULL)
                     {
                        nar_match_stream = stream_rec;
                        nar_stream_priority = stream_priority;
                     }
                     else
                     {
                        /* Check whether this stream uses a better encoding
                         * than the currently selected one */
                        if (!prefer_hd_stream)
                        {
                           /* The minimum match needs to be found if the output is PCM */
                           if (stream_priority < nar_stream_priority)
                           {
                              /* This stream is a better match so use it instead */
                              nar_match_stream = stream_rec;
                              nar_stream_priority = stream_priority;
                           }
                           else if (stream_priority == nar_stream_priority)
                           {
                              /* Both streams use the same codec so a non-multichannel one
                               * should be selected, if available */
                              if ((nar_match_stream->data.audio.mode == AUDIO_MULTICHANNEL) &&
                                  (stream_rec->data.audio.mode != AUDIO_MULTICHANNEL))
                              {
                                 /* This stream is a better match so use it instead */
                                 nar_match_stream = stream_rec;
                                 nar_stream_priority = stream_priority;
                              }
                           }
                        }
                        else
                        {
                           if (stream_priority > nar_stream_priority)
                           {
                              /* This stream has better audio encoding so select it instead */
                              nar_match_stream = stream_rec;
                              nar_stream_priority = stream_priority;
                           }
                           else if (stream_priority == nar_stream_priority)
                           {
                              /* Both streams use the same codec so a multichannel one
                               * should be selected, if available */
                              if ((nar_match_stream->data.audio.mode != AUDIO_MULTICHANNEL) &&
                                  (stream_rec->data.audio.mode == AUDIO_MULTICHANNEL))
                              {
                                 /* This stream is a better match so use it instead */
                                 nar_match_stream = stream_rec;
                                 nar_stream_priority = stream_priority;
                              }
                           }
                        }
                     }
                  }
               }
            }
         }
         stream_rec = stream_rec->next;
      }

      // choose the stream which provides the best match to the requirements
      stream_rec = exact_match_stream;
      if (stream_rec == NULL)
      {
         stream_rec = lang_code_match_stream;
         if (stream_rec == NULL)
         {
            stream_rec = nar_match_stream;
            if (stream_rec != NULL)
            {
               *broadcast_mix = TRUE;
            }
         }
      }

      // if a stream has been found mark it in use
      if (stream_rec != NULL)
      {
         ad_pid = stream_rec->pid;
         *ad_mode = stream_rec->data.audio.mode;
         *selected_stream = stream_rec;
      }
   }

   FUNCTION_FINISH(GetReqdADPid);

   return(ad_pid);
}

#if defined(COUNTRY_FINLAND) || defined(COUNTRY_IRELAND)
/*!**************************************************************************
 * @brief   Returns the largest LCN from all the available services
 * @param   tuner_type only use services found on this tuner type.
 *                       Use SIGNAL_NONE for all services
 * @param   onet_id the original network ID to be matched when checking.
 *                    Use ADB_INVALID_DVB_ID for all services
 * @return  Largest LCN
 ****************************************************************************/
static U16BIT FindLargestLcnNumber(E_STB_DP_SIGNAL_TYPE tuner_type, U16BIT onet_id)
{
   ADB_SERVICE_REC *s_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   U16BIT largest_lcn = 0;

   FUNCTION_START(FindLargestLcnNumber);

   s_ptr = DBDEF_GetNextServiceRec(NULL);
   while (s_ptr != NULL)
   {
      /* Can only check the service's signal type and orig net id if it has a transport */
      t_ptr = s_ptr->transport;
      if ((tuner_type == SIGNAL_NONE) || (onet_id == ADB_INVALID_DVB_ID) || (t_ptr != NULL))
      {
         if ((tuner_type == SIGNAL_NONE) || (t_ptr->sig_type == tuner_type))
         {
            if ((onet_id == ADB_INVALID_DVB_ID) || (onet_id == t_ptr->orig_net_id))
            {
               if (s_ptr->serv_lcn > largest_lcn)
               {
                  largest_lcn = s_ptr->serv_lcn;
               }
            }
         }
      }

      s_ptr = DBDEF_GetNextServiceRec(s_ptr);
   }

   FUNCTION_FINISH(FindLargestLcnNumber);

   return largest_lcn;
}

#endif

#ifdef COUNTRY_UK
/*!**************************************************************************
 * @brief   Implements the rules that should be applied when target regions are available
 *          and 2 services require the same LCN.
 * @param   s1_ptr pointer to service whose LCN is being set
 * @param   s2_ptr pointer to service that requires the same LCN
 * @param   next_allocated_lcn pointer to value of next LCN that should be assigned
 *                               in the 800+ range. The value is incremented for returning.
 * @param   lcn_assigned returned value indicating whether an LCN has been assigned to either service
 * @return  TRUE if searching should continue.
 ****************************************************************************/
static BOOLEAN ApplyTargetRegionRules(ADB_SERVICE_REC *s1_ptr, ADB_SERVICE_REC *s2_ptr,
   U16BIT *next_allocated_lcn, BOOLEAN *lcn_assigned)
{
   BOOLEAN continue_search;
   U8BIT device_region_depth;
   U8BIT s1_region_depth, s2_region_depth;
   U32BIT device_country_code;
   U8BIT device_primary_region;
   U8BIT device_secondary_region;
   U16BIT device_tertiary_region;

   FUNCTION_START(ApplyTargetRegionRules);

   continue_search = TRUE;
   *lcn_assigned = FALSE;

   device_region_depth = (U8BIT)APP_NvmRead(TARGET_REGION_DEPTH_NVM);
   if (device_region_depth > 0)
   {
      device_primary_region = 0;
      device_secondary_region = 0;
      device_tertiary_region = 0;

      device_country_code = (U32BIT)APP_NvmRead(TARGET_REGION_COUNTRY_NVM);

      if (device_region_depth > 1)
      {
         device_primary_region = (U8BIT)APP_NvmRead(TARGET_REGION_PRIMARY_NVM);
         if (device_region_depth > 2)
         {
            device_secondary_region = (U8BIT)APP_NvmRead(TARGET_REGION_SECONDARY_NVM);
            if (device_region_depth > 3)
            {
               device_tertiary_region = (U8BIT)APP_NvmRead(TARGET_REGION_TERTIARY_NVM);
            }
         }
      }

      /* Now find the service that best matches the target region depth defined for the device */
      s1_region_depth = GetTargetRegionMatchDepth(s1_ptr, device_region_depth, device_country_code,
            device_primary_region, device_secondary_region, device_tertiary_region);

      s2_region_depth = GetTargetRegionMatchDepth(s2_ptr, device_region_depth, device_country_code,
            device_primary_region, device_secondary_region, device_tertiary_region);

      if (s1_region_depth < s2_region_depth)
      {
         /* First service doesn't get the LCN it wants, so give it the next available LCN */
         s1_ptr->allocated_lcn = *next_allocated_lcn;
         (*next_allocated_lcn)++;
         *lcn_assigned = TRUE;
         continue_search = FALSE;
      }
      else if (s1_region_depth > s2_region_depth)
      {
         /* s1 should get the LCN it wants */
         if (s2_ptr->allocated_lcn == s2_ptr->serv_lcn)
         {
            if (s2_ptr->unavailable == FALSE)
            {
               /* s2 has the lcn - for the moment leave it there and give s1 an allocated lcn.
                * Later we will check whether to transfer the lcn and the favourites */
               s1_ptr->allocated_lcn = *next_allocated_lcn;
               (*next_allocated_lcn)++;
               continue_search = FALSE;
            }
            else
            {
               /* s2 is unavailable so make it relinquish its lcn and allocate it a new one */
               s2_ptr->allocated_lcn = *next_allocated_lcn;
               (*next_allocated_lcn)++;
            }
         }

         *lcn_assigned = TRUE;
      }
   }

   FUNCTION_FINISH(ApplyTargetRegionRules);

   return(continue_search);
}

/*!**************************************************************************
 * @brief   Returns the number of target regions that match the given service.
 * @param   s_ptr pointer to service to be checked
 * @param   match_depth number of codes given that are valid and available for checking.
 *                        1 = only match country code
 *                        2 = match to primary region level
 *                        3 = match to secondary region level
 *                        4 = match to tertiary region level
 * @param   country_code     - region codes to be matched against
 * @param   primary_region
 * @param   secondary_region
 * @param   tertiary_region
 * @return  The number of matches
 ****************************************************************************/
static U8BIT GetTargetRegionMatchDepth(ADB_SERVICE_REC *s_ptr, U8BIT match_depth,
   U32BIT country_code, U8BIT primary_region, U8BIT secondary_region, U16BIT tertiary_region)
{
   SI_TARGET_REGION_DESC *tptr;
   SI_TARGET_REGION *rptr;
   U8BIT best_depth;

   tptr = s_ptr->target_region_list;
   if (tptr == NULL)
   {
      if ((s_ptr->transport != NULL) &&
          (s_ptr->transport->sig_type == SIGNAL_COFDM))
      {
         tptr = s_ptr->transport->u.terr.target_region_list;
         if (tptr == NULL)
         {
            if (s_ptr->transport->network != NULL)
            {
               tptr = s_ptr->transport->network->target_region_list;
            }
         }
      }
   }

   best_depth = 0;

   while (tptr != NULL)
   {
      if (tptr->country_code == country_code)
      {
         if (best_depth < 1)
         {
            best_depth = 1;
         }

         if (match_depth > 1)
         {
            rptr = tptr->region_list;
            while (rptr != NULL)
            {
               if ((rptr->region_depth >= 1) && (rptr->primary_region_code == primary_region))
               {
                  if (best_depth < 2)
                  {
                     best_depth = 2;
                  }

                  if (match_depth > 2)
                  {
                     if ((rptr->region_depth >= 2) && (rptr->secondary_region_code == secondary_region))
                     {
                        if (best_depth < 3)
                        {
                           best_depth = 3;
                        }

                        if (match_depth > 3)
                        {
                           if ((rptr->region_depth >= 3) && (rptr->tertiary_region_code == tertiary_region))
                           {
                              if (best_depth < 4)
                              {
                                 best_depth = 4;
                              }
                           }
                        }
                     }
                  }
               }

               rptr = rptr->next;
            }
         }
      }

      tptr = tptr->next;
   }

   return(best_depth);
}

#endif /* COUNTRY_UK */

/*!**************************************************************************
 * @brief   Checks the teletext subtitle type against the subtitle type
 *          user preference
 * @param   ADB_TELETEXT_TYPE ttext_type teletex type in the stream
 * @param   E_SUBTITLE_TYPE user_pref user preference
 * @return  TRUE if the teletext type is compatible with the user preference
 ****************************************************************************/
static BOOLEAN UserPrefsMatch(ADB_TELETEXT_TYPE ttext_type, E_SUBTITLE_TYPE user_pref)
{
   BOOLEAN retval = FALSE;

   if (((ttext_type == ADB_TELETEXT_TYPE_SUBTITLE) && (user_pref == SUBTITLE_NORMAL)) ||
       ((ttext_type == ADB_TELETEXT_TYPE_SUBTITLE_HARD_HEARING) && (user_pref == SUBTITLE_HARD_OF_HEARING)))
   {
      retval = TRUE;
   }

   return retval;
}

/**
 * @brief   Function used by STB_LLSort to sort services into ascending LCN order
 * @param   blk1 pointer to service record
 * @param   blk2 pointer to service record
 * @return  -1 if lcn of blk1 is < lcn of blk2, +1 if lcn of blk1 is > blk2, 0 otherwise
 */
static S16BIT LcnSortCompareFunc(LINK_LIST_PTR_BLK **blk_1, LINK_LIST_PTR_BLK **blk_2)
{
   S16BIT ret_val;
   ADB_SERVICE_REC *s1_ptr;
   ADB_SERVICE_REC *s2_ptr;
   S16BIT lcn_1;
   S16BIT lcn_2;

   FUNCTION_START(LcnSortCompareFunc);

   ASSERT(blk_1 != NULL);
   ASSERT(blk_2 != NULL);

   s1_ptr = (ADB_SERVICE_REC *)*blk_1;
   s2_ptr = (ADB_SERVICE_REC *)*blk_2;
   ret_val = 0;
   if ((s1_ptr != NULL) && (s2_ptr != NULL))
   {
      lcn_1 = (S16BIT)s1_ptr->allocated_lcn;
      lcn_2 = (S16BIT)s2_ptr->allocated_lcn;
      ret_val = lcn_1 - lcn_2;
   }

   FUNCTION_FINISH(LcnSortCompareFunc);

   return(ret_val);
}

/**
 * @brief   Function used by STB_LLSort to compare services in a favourite list
 *          according to their index values
 * @param   blk1 pointer to favourite service record
 * @param   blk2 pointer to favourite service record
 * @return  -1 if index of blk1 is < index of blk2, +1 if index of blk1 is > blk2, 0 otherwise
 */
static S16BIT FavServiceSortCompare(LINK_LIST_PTR_BLK **blk1, LINK_LIST_PTR_BLK **blk2)
{
   S16BIT retval;
   ADB_FAVSERV_REC *fs1;
   ADB_FAVSERV_REC *fs2;

   FUNCTION_START(FavServiceSortCompare);

   retval = 0;

   fs1 = (ADB_FAVSERV_REC *)*blk1;
   fs2 = (ADB_FAVSERV_REC *)*blk2;

   if ((fs1 != NULL) && (fs2 != NULL))
   {
      if (fs1->index < fs2->index)
      {
         retval = -1;
      }
      else if (fs1->index > fs2->index)
      {
         retval = 1;
      }
   }

   FUNCTION_FINISH(FavServiceSortCompare);

   return(retval);
}

#if 0
/**
 * @brief   Insert and event into a services event schedule in the correct position based on time
 * @param   serv_ptr service of the event
 * @param   event_ptr event to be inserted
 */
static void InsertEventInSchedule(ADB_SERVICE_REC *serv_ptr, ADB_EVENT_REC *event_ptr)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC **add_ptr;
   U32DHMS end_date_time;

   FUNCTION_START(InsertEventInSchedule);

   /* Skip any events that end before the one being added starts */
   add_ptr = &(serv_ptr->event_schedule);

   if (serv_ptr->event_schedule != NULL)
   {
      e_ptr = serv_ptr->event_schedule;
      end_date_time = STB_GCCalculateDHMS(e_ptr->start, e_ptr->duration, CALC_ADD);

      while ((e_ptr != NULL) && (end_date_time < event_ptr->start))
      {
         add_ptr = &(e_ptr->next);
         if ((e_ptr = e_ptr->next) != NULL)
         {
            end_date_time = STB_GCCalculateDHMS(e_ptr->start, e_ptr->duration, CALC_ADD);
         }
      }
   }

   event_ptr->next = *add_ptr;
   *add_ptr = event_ptr;

   serv_ptr->num_events_in_schedule++;

   FUNCTION_FINISH(InsertEventInSchedule);
}
#endif

/**
 * @brief   Linked list callback function to sort timers into ascending date/time order
 * @param   timer1 address of pointer to first timer
 * @param   timer2 address of pointer to second timer
 * @return  -1 if timer1 starts before timer2, +1 if timer1 starts after timer2,
 *          and 0 if the start times are equal
 */
static S16BIT TimerSortDateTime(LINK_LIST_PTR_BLK **timer1, LINK_LIST_PTR_BLK **timer2)
{
   S16BIT retval;
   ADB_TIMER_REC *t1_ptr;
   ADB_TIMER_REC *t2_ptr;

   FUNCTION_START(TimerSortDateTime);

   ASSERT(timer1 != NULL);
   ASSERT(timer2 != NULL);

   t1_ptr = (ADB_TIMER_REC *)*timer1;
   t2_ptr = (ADB_TIMER_REC *)*timer2;
   retval = 0;

   if ((t1_ptr != NULL) && (t2_ptr != NULL))
   {
      if (t1_ptr->start_time > t2_ptr->start_time)
      {
         retval = 1;
      }
      else
      {
         if (t1_ptr->start_time < t2_ptr->start_time)
         {
            retval = -1;
         }
         else
         {
            /* Items are equal */
            retval = 0;
         }
      }
   }

   FUNCTION_FINISH(TimerSortDateTime);

   return(retval);
}

/**
 * @brief   Linked list callback function to sort timers into ascending name order
 * @param   timer1 address of pointer to first timer
 * @param   timer2 address of pointer to second timer
 * @return  -1 if name of timer1 is less than name of timer2, +1 if timer1 is greater than timer2,
 *          and 0 if the names are the same
 */
static S16BIT TimerSortName(LINK_LIST_PTR_BLK **timer1, LINK_LIST_PTR_BLK **timer2)
{
   S16BIT retval;
   ADB_TIMER_REC *t1_ptr;
   ADB_TIMER_REC *t2_ptr;
   U8BIT *name1;
   U8BIT *name2;
   BOOLEAN reverse;
   U16BIT num_chars;

   FUNCTION_START(TimerSortName);

   ASSERT(timer1 != NULL);
   ASSERT(timer2 != NULL);

   t1_ptr = (ADB_TIMER_REC *)*timer1;
   t2_ptr = (ADB_TIMER_REC *)*timer2;
   retval = 0;

   if ((t1_ptr != NULL) && (t2_ptr != NULL))
   {
      name1 = STB_ConvertStringToUnicode(t1_ptr->name, &reverse, &num_chars, FALSE, 0);
      if (name1 != NULL)
      {
         name2 = STB_ConvertStringToUnicode(t2_ptr->name, &reverse, &num_chars, FALSE, 0);
         if (name2 != NULL)
         {
            retval = STB_CompareUnicodeStrings(name1, name2, TRUE, TRUE);

            STB_ReleaseUnicodeString(name2);
         }

         STB_ReleaseUnicodeString(name1);
      }
   }

   FUNCTION_FINISH(TimerSortName);

   return(retval);
}

/**
 * @brief   Returns TRUE if the specified stream type is a video type
 * @param   type Stream type
 * @return  TRUE if the stream type is one of the video types
 */
static BOOLEAN IsVideoStreamType(ADB_STREAM_TYPE type)
{
   BOOLEAN is_video;

   FUNCTION_START(IsVideoStreamType);

   if ((type == ADB_VIDEO_STREAM) || (type == ADB_H264_VIDEO_STREAM) ||
      (type == ADB_H265_VIDEO_STREAM))
   {
      is_video = TRUE;
   }
   else
   {
      is_video = FALSE;
   }

   FUNCTION_FINISH(IsVideoStreamType);

   return is_video;
}

static U8BIT* ReadLanguageCode(U8BIT *dptr, U32BIT *lang_code)
{
   U32BIT lcode;
   U8BIT i;
   U8BIT code_char;

   FUNCTION_START(ReadLanguageCode);

   lcode = 0;
   for (i = 0; i < 3; i++)
   {
      code_char = *dptr;
      dptr++;
      if ((code_char >= 'A') && (code_char <= 'Z'))
      {
         code_char += 0x20; // convert to lower case
      }
      lcode = (lcode << 8) | code_char;
   }
   *lang_code = lcode;

   FUNCTION_FINISH(ReadLanguageCode);

   return(dptr);
}

static U8BIT* CopyString(ADB_STRING *str_desc, BOOLEAN to_unicode)
{
   U16BIT str_len;
   U8BIT *copied_string;

   FUNCTION_START(CopyString);

   copied_string = NULL;
   if (str_desc != NULL)
   {
      if (to_unicode)
      {
         copied_string = STB_ConvertStringToUTF8(str_desc->str_ptr, &str_len, FALSE,
               str_desc->lang_code);
      }
      else
      {
         str_len = (U16BIT)str_desc->nbytes;
         copied_string = (U8BIT *)STB_AppGetMemory(str_len);
         if (copied_string != NULL)
         {
            memcpy(copied_string, str_desc->str_ptr, str_len);
         }
      }
   }

   FUNCTION_FINISH(CopyString);

   return(copied_string);
}

