/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application database control
 * @file    ap_dbdef.h
 * @date    March 2003
 */

#ifndef _AP_DBDEF_H
#define _AP_DBDEF_H

#include "techtype.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbllist.h"
#include "ap_cfg.h"
#include "ap_tmr.h"
#include "ap_dbacc.h"

//---Constant and macro definitions for public use-----------------------------

#define ADB_LANG_CODE_UNDEF            0x756e64
#define ADB_LANG_CODE_QAA              0x716161
#define ADB_LANG_CODE_NAR              0x6e6172

/* DVB SI descriptor tags */
#define LINKAGE_DTAG                   0x4a
#define SHORT_EVENT_DTAG               0x4d
#define EXTENDED_EVENT_DTAG            0x4e
#define COMPONENT_DTAG                 0x50
#define CONTENT_DTAG                   0x54
#define PARENTAL_RATING_DTAG           0x55
#define PRIVATE_DATA_SPEC_DTAG         0x5f
#define CONT_ID_DESC_DTAG              0x76
#define FTA_CONTENT_DESC_DTAG          0x7e
#define EXTENSION_DTAG                 0x7f
#define USER_DEFINED_DTAG_0x85         0x85
#define USER_DEFINED_DTAG_0x89         0x89

#define LINK_TYPE_EVENT                0x0d
#define LINK_TYPE_EXT_EVENT            0x0e

#define INVALID_FREESAT_SERV_ID        0x8000

/* Content identifier descriptor CRID types */
#define TVA_PROGRAMME_CRID             0x01
#define TVA_SERIES_CRID                0x02
#define TVA_RECOMMENDATION_CRID        0x03

#define UK_PROGRAMME_CRID              0x31
#define UK_SERIES_CRID                 0x32
#define UK_RECOMMENDATION_CRID         0x33

//---Enumerations for public use-----------------------------------------------
typedef enum
{
   STREAM_MATCH_EXACT,
   STREAM_MATCH_ASPECT,
   STREAM_MATCH_LANG,
   STREAM_MATCH_TYPE,
   STREAM_MATCH_NOT_ZERO,
   STREAM_MATCH_NONE
} E_STREAM_MATCH_TYPE;

//---Global type defs for public use-------------------------------------------

// Database structures - strings
typedef struct adb_string
{
   U32BIT lang_code;
   U16BIT nbytes;
   U8BIT *str_ptr;
} ADB_STRING;


//--------------------------------------------------
// Database structures - satellite records
typedef struct adb_lnb_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   E_STB_DP_LNB_TYPE type;
   E_STB_DP_LNB_POWER power;
   BOOLEAN is_22k;
   BOOLEAN is_12v;
   BOOLEAN is_pulse_posn;
   BOOLEAN is_diseqc_posn;
   E_STB_DP_DISEQC_TONE diseqc_tone;
   E_STB_DP_DISEQC_CSWITCH c_switch;
   U8BIT u_switch;            // 0 to 16 inclusive
   BOOLEAN is_smatv;
   U8BIT diseqc_repeats;      // 0 to 3 inclusive
   U32BIT unicable_if;
   U8BIT unicable_chan;
   ADB_STRING *name;
} ADB_LNB_REC;

typedef struct adb_bat_version_rec
{
   struct adb_bat_version_rec *next;
   U16BIT bouquet_id;
   U8BIT version;
} ADB_BAT_VERSION_REC;

typedef struct adb_satellite_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   ADB_LNB_REC *lnb;
   ADB_STRING *name;
   U16BIT dish_pos;
   U16BIT long_pos;           // 0 to 1800 inclusive
   BOOLEAN east_west;         // 0=West,1=East

   BOOLEAN has_fta_desc;
   BOOLEAN do_not_scramble;
   ADB_STRING *def_authority;

   ADB_BAT_VERSION_REC *bat_version_list;
} ADB_SATELLITE_REC;

typedef struct adb_lnb_band_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   ADB_LNB_REC *lnb;
   S_STB_DP_LNB_BAND band_params;
} ADB_LNB_BAND_REC;


//--------------------------------------------------
// Database structures - network records
typedef struct adb_network_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   U16BIT net_id;
   U8BIT nit_version;
   BOOLEAN nit_version_changed;

   ADB_STRING *name_str;
   ADB_STRING *name_array[ACFG_NUM_DB_LANGUAGES];
   BOOLEAN has_fta_desc;
   BOOLEAN do_not_scramble;
   SI_NIT_TARGET_REGION_NAME_DESC *target_region_name_list;
   SI_TARGET_REGION_DESC *target_region_list;
   SI_LINKAGE_DESC_ENTRY *last_linkage_entry;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_list;
   U16BIT num_linkage_entries;
   ADB_STRING *def_authority;
   ADB_SATELLITE_REC *satellite;
   ADB_PROFILE_TYPE profile_type;

#ifdef COMMON_INTERFACE
   BOOLEAN module_present;
   U32BIT module;
   U16BIT cicam_onet_id;
   U32BIT cicam_identifier;
   BOOLEAN op_search_required;
   U16BIT op_search_date;
   U16BIT op_search_time;
   ADB_STRING *profile_name;
   /**
    * @brief   CI+ profile channel list
    */
   U8BIT fav_list_id;
#endif
} ADB_NETWORK_REC;


//--------------------------------------------------
// Database structures - transport records
typedef struct adb_pmt_version_rec
{
   struct adb_pmt_version_rec *next;
   U16BIT serv_id;
   U8BIT version;
} ADB_PMT_VERSION_REC;

typedef struct adb_transport_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   U16BIT tran_id;
   U16BIT orig_net_id;
   U32BIT frequency;
   ADB_NETWORK_REC *network;
   E_STB_DP_SIGNAL_TYPE sig_type;
   U16BIT signal_level_at_search;

   U8BIT num_additional_frequencies;
   U32BIT *additional_frequencies;

   /* The following union is accessed based on sig_type */
   union
   {
      struct _analog_trans_rec
      {
         S8BIT freq_offset;
         E_STB_DP_ANALOG_VIDEO_TYPE vtype;
      } anal;
      struct _terr_trans_rec
      {
         E_STB_DP_TBWIDTH bwidth;
         E_STB_DP_TMODE tmode;
         E_STB_DP_TTYPE terr_type;
         U8BIT plp_id;
         S8BIT freq_offset;
         E_STB_TUNE_TCONST constellation;
         E_STB_TUNE_THIERARCHY hierarchy;
         E_STB_TUNE_TCODERATE lp_code_rate;
         E_STB_TUNE_TCODERATE hp_code_rate;
         E_STB_TUNE_TGUARDINT guard_int;
         SI_TARGET_REGION_DESC *target_region_list;
      } terr;
      struct _cable_trans_rec
      {
         E_STB_DP_CMODE cmode;
         U16BIT symbol_rate;
      } cab;
      struct _sat_trans_rec
      {
         U16BIT symbol_rate;
         E_STB_DP_POLARITY polarity;
         E_STB_DP_FEC fec_code;
         E_STB_DP_FEC_TYPE fec_type;
         BOOLEAN dvb_s2;
         E_STB_DP_MODULATION modulation;
      } sat;
   } u;

   U8BIT pat_version;
   BOOLEAN sdt_received;
   BOOLEAN sdt_version_changed;
   U8BIT sdt_version;
   ADB_PMT_VERSION_REC *pmt_version_list;
   BOOLEAN has_fta_desc;
   BOOLEAN do_not_scramble;
   ADB_STRING *def_authority;
   BOOLEAN searched;
   BOOLEAN available;
} ADB_TRANSPORT_REC;

//--------------------------------------------------------
// Database structures - service, event and stream records

typedef struct adb_audio_stream_data
{
   U32BIT lang_code;
   ADB_AUDIO_TYPE type;
   E_STB_DP_AUDIO_MODE mode;
} ADB_AUDIO_STREAM_DATA;

typedef struct adb_subtitle_stream_data
{
   U32BIT lang_code;
   ADB_SUBTITLE_TYPE type;
   U16BIT composition_page;
   U16BIT ancillary_page;
} ADB_SUBTITLE_STREAM_DATA;

typedef struct adb_ttext_stream_data
{
   U32BIT lang_code;
   U8BIT type;
   U8BIT magazine;
   U8BIT page;
} ADB_TTEXT_STREAM_DATA;

typedef union adb_stream_data
{
   ADB_AUDIO_STREAM_DATA audio;
   ADB_SUBTITLE_STREAM_DATA subtitle;
   ADB_TTEXT_STREAM_DATA ttext;
} ADB_STREAM_DATA;

typedef struct adb_stream_rec
{
   struct adb_stream_rec *next;
   ADB_STREAM_TYPE type;
   U8BIT *tag_array;
   U8BIT num_tags;
   U16BIT pid;
   BOOLEAN in_use;
   ADB_STREAM_DATA data;
   BOOLEAN has_ca_descriptor;
   U16BIT ecm_pid;
} ADB_STREAM_REC;

typedef struct adb_event_desc
{
   U8BIT *desc_data;
   struct adb_event_desc *next;
} ADB_EVENT_DESC;

typedef struct adb_event_rec
{
   struct adb_event_rec *next;
   void *dba_rec;

   U16BIT event_id;
   U32DHMS start;
   U32DHMS duration;
   U8BIT running_status;
   BOOLEAN free_to_air;
   U8BIT version;

   /* Keep values parsed from the content management descriptor that are used by DVBCore */
   BOOLEAN has_content_management_desc;
   BOOLEAN do_not_scramble;

   ADB_EVENT_DESC *desc_list_head;
   ADB_EVENT_DESC *desc_list_tail;
} ADB_EVENT_REC;

typedef struct adb_alt_serv_rec
{
   struct adb_alt_serv_rec *next;
   U16BIT onet_id;
   U16BIT tran_id;
   U16BIT serv_id;
   U8BIT link_type;
} ADB_ALT_SERV_REC;

typedef enum
{
   ICON_TYPE_PNG,
   ICON_TYPE_JPEG
} E_ICON_TYPE;

typedef struct adb_icon_image
{
   struct adb_icon_image *next;
   U8BIT icon_id;
   BOOLEAN position_defined;
   E_ICON_COORD_SYSTEM coord_system;
   U16BIT x_pos;
   U16BIT y_pos;
   U16BIT width;
   U16BIT height;
   E_ICON_TYPE icon_type;
   E_ICON_TRANSPORT_MODE transport_mode;
   U8BIT *icon_url;
   U32BIT data_len;
   U8BIT *icon_data;
   void (*destroy_func)(void *);
   void *dsm_handle;
} ADB_IMAGE_ICON;

typedef struct adb_rct_link_info
{
   struct adb_rct_link_info *next;
   BOOLEAN is_group_trailer;
   U8BIT *uri_string;
   BOOLEAN can_use_default_icon;
   U8BIT icon_id;
   ADB_STRING *promo_text[ACFG_NUM_DB_LANGUAGES];
   ADB_STRING *event_name;
} ADB_RCT_LINK_INFO;

typedef struct
{
   U8BIT type;
   U8BIT mode;
   ADB_STRING *text;
} ADB_GUIDANCE_DESC;

typedef struct adb_service_rec
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   ADB_TRANSPORT_REC *transport;
   ADB_SERVICE_TYPE serv_type;
   U16BIT serv_id;
   U16BIT serv_lcn;
   U16BIT allocated_lcn;
   U16BIT old_allocated_lcn;
   ADB_STRING *name_str;
   ADB_STRING *provider_str;
   ADB_STRING *short_name_str;
   ADB_STRING *short_name_array[ACFG_NUM_DB_LANGUAGES];
   ADB_STRING *name_array[ACFG_NUM_DB_LANGUAGES][ADB_NUM_SERV_NAME_IDS];
   ADB_STRING *provider_array[ACFG_NUM_DB_LANGUAGES];
   ADB_GUIDANCE_DESC guidance[ACFG_NUM_DB_LANGUAGES];
   ADB_STREAM_REC *stream_list;
   ADB_ALT_SERV_REC *alt_serv_list;
   U8BIT fav_groups;
   BOOLEAN deleted;

   BOOLEAN sched_disabled;
   BOOLEAN now_next_disabled;

   BOOLEAN eit_now_next_avail;
   ADB_EVENT_REC *now_event;
   ADB_EVENT_REC *next_event;

   BOOLEAN eit_sched_avail;
   U16BIT num_events_in_schedule;
   ADB_EVENT_REC *event_schedule;

   BOOLEAN pmt_received;

   U16BIT pcr_pid;
   U16BIT video_pid;
   U16BIT audio_pid;
   U16BIT ad_pid;
   U16BIT ttext_pid;
   U8BIT ttext_mag;
   U8BIT ttext_page;
   U16BIT subtitle_pid;
   U16BIT subtitle_cpage;
   U16BIT subtitle_apage;
   ADB_STREAM_TYPE video_type;
   ADB_STREAM_TYPE audio_type;

   BOOLEAN reqd_audio_valid;
   U32BIT reqd_audio_lang_code;
   ADB_AUDIO_TYPE reqd_audio_type;
   ADB_STREAM_TYPE reqd_stream_type;
   U16BIT reqd_audio_pid;
   BOOLEAN reqd_subtitle_valid;
   U32BIT reqd_subtitle_lang_code;
   ADB_SUBTITLE_TYPE reqd_subtitle_type;
   BOOLEAN reqd_ttext_valid;
   U32BIT reqd_ttext_lang_code;
   ADB_TELETEXT_TYPE reqd_ttext_type;

   BOOLEAN found;             /* Used during service search */
   U8BIT running_status;
   BOOLEAN not_running;
   BOOLEAN unavailable;
   BOOLEAN new_service;
   BOOLEAN locked;
   BOOLEAN hidden;
   BOOLEAN selectable;
   BOOLEAN lcn_editable;
   BOOLEAN scrambled;

   BOOLEAN has_fta_desc;
   BOOLEAN do_not_scramble;
   U8BIT content_protection_level;

   SI_TARGET_REGION_DESC *target_region_list;

   ADB_STRING *def_authority;

   U16BIT rct_pid;
   ADB_RCT_LINK_INFO *rct_link_list;
   ADB_IMAGE_ICON *icon_list;

#ifdef COMMON_INTERFACE
   BOOLEAN sdt_received;
   U8BIT *ci_protection_desc;
   U32BIT ci_prot_last_update;
#endif
   U8BIT *pmt_data;
   U16BIT pmt_data_len;
   U16BIT pmt_pid;
   SI_LINKAGE_DESC_ENTRY *last_linkage_entry;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_list;
   U16BIT num_linkage_entries;
   SI_LCN_DESC *hd_lcn_desc;

   BOOLEAN has_ca_descriptor;

   /* Freesat specific values */
   U16BIT freesat_id;
   U16BIT region_id;

   U16BIT bat_pid;
   U16BIT eit_pf_pid;
   U16BIT eit_pf_plus_pid;
   U16BIT eit_sched_pid;
   U16BIT nit_pid;
   U16BIT sdt_pid;
   U16BIT tdt_pid;
   U16BIT tot_pid;
#ifdef INTEGRATE_HBBTV
   U16BIT ait_pid;
#endif
   U16BIT unt_pid;
   U32BIT group_mask;  // the CNS channel group mask (ref. CNS SDT code)

   U16BIT ecm_pid;
   U16BIT video_ecm_pid;
   U16BIT audio_ecm_pid;
   U16BIT ad_ecm_pid;
   U16BIT ttext_ecm_pid;
   U16BIT data_ecm_pid;
} ADB_SERVICE_REC;


//--------------------------------------------------------
// Database structures - crid records

typedef struct adb_crid_record
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   U16BIT eit_date;           /* The date the CRID was last seen in the EITs */
   BOOLEAN series_flag;       /* The CRID represents a series */
   BOOLEAN recommended_flag;  /* The CRID represents a recommended programme or series */
   BOOLEAN do_not_delete;     /* TRUE if any recordings created from this CRID are to be locked */
   U32DHMS date_time;         /* Date to be used when searching for a split event */
   U16BIT serv_id;            /* Service to be used when searching for a split event */
   ADB_STRING *name_str;      /* Programme name for series/recommendations */
   ADB_STRING *crid_str;      /* CRID string */
} ADB_CRID_REC;


//--------------------------------------------------------
// Database structures - favourite lists

typedef struct adb_favserv_rec
{
   LINK_LIST_PTR_BLK ptrs;       /* prev/next service in favourite list */
   void *dba_rec;

   ADB_SERVICE_REC *serv_ptr;
   U8BIT list_id;                /* Id of the fav list this service is in */
   U16BIT index;                 /* Defines the position of this service in the list */
} ADB_FAVSERV_REC;

typedef struct adb_favlist_rec
{
   LINK_LIST_PTR_BLK ptrs;       /* List of favourite lists */
   void *dba_rec;

   U8BIT list_id;                /* Unique id for a list */
   U8BIT index;                  /* Defines the position of this list in the favourite lists */
   U32BIT user_data;             /* User defined value stored with the list */
   ADB_STRING *name;
   LINK_LIST_HEADER serv_list;   /* List of service in this favourite list */
} ADB_FAVLIST_REC;

//--------------------------------------------------------
// Database structures - timers

typedef struct
{
   BOOLEAN change_service;
   U16BIT service_id;
   U16BIT transport_id;
   U16BIT orig_net_id;
   BOOLEAN ramp_volume;
} ADB_ALARM_REC;

typedef struct
{
   U32DHMS duration;
   BOOLEAN event_triggered;
   U16BIT event_id;
   U16BIT service_id;
   U16BIT transport_id;
   U16BIT orig_net_id;
   U8BIT prog_crid[TMR_PVR_CRID_LEN_MAX];
   U8BIT other_crid[TMR_PVR_CRID_LEN_MAX];
   U16BIT disk_id;
   BOOLEAN recommendation;
   U16BIT notify_time;
   BOOLEAN notified;
   U8BIT path;
   U32BIT recording_handle;
   U8BIT additional_info[TMR_PVR_ADDINFO_LEN_MAX];
   BOOLEAN has_start_padding;
   S32BIT start_padding;
   BOOLEAN has_end_padding;
   S32BIT end_padding;
   BOOLEAN do_not_delete;
   BOOLEAN programme_started;
   BOOLEAN programme_finished;
} ADB_PVR_RECORD_REC;

typedef struct
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   U32BIT handle;
   E_TIMER_TYPE type;
   U32DHMS start_time;
   U8BIT name[TMR_MAX_NAME_LENGTH];
   E_TIMER_FREQ frequency;

   BOOLEAN expired;
   BOOLEAN starting;
   BOOLEAN started;
   BOOLEAN selected;
   BOOLEAN missed;
   union
   {
      ADB_ALARM_REC alarm;
      ADB_PVR_RECORD_REC record;
   } u;
} ADB_TIMER_REC;

#ifdef COMMON_INTERFACE
typedef struct
{
   LINK_LIST_PTR_BLK ptrs;
   void *dba_rec;

   U32BIT cicam_identifier;
   U32BIT timer_handle;
} ADB_CICAM_TIMER_REC;
#endif

/* Internal structure used to pass events to an application.
 * The application handles these as void* handles */
typedef struct s_event
{
   void *serv_ptr;
   U16BIT event_id;
} S_EVENT;


//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Initialises the database, preparing for it to be accessed
 */
void DBDEF_Initialise(void);

/**
 * @brief   Requests access to the app's database
 */
void DBDEF_RequestAccess(void);

/**
 * @brief   Releases access to the app's database
 */
void DBDEF_ReleaseAccess(void);

BOOLEAN DBDEF_LoadDatabase(U8BIT *db_pathname);
void DBDEF_DeleteAllRecords(void);
BOOLEAN DBDEF_SaveDatabase(void);

void DBDEF_DeleteRecordsForTunerType(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

void DBDEF_DeleteServiceRec(ADB_SERVICE_REC *s_ptr);
U16BIT DBDEF_NumStreamsInList(ADB_STREAM_REC *slist);
ADB_STREAM_REC* DBDEF_CopyStreamList(ADB_STREAM_REC *slist);
void DBDEF_DeleteStreamList(ADB_STREAM_REC *slist);
void DBDEF_DeleteAltServList(ADB_ALT_SERV_REC *aslist);

void DBDEF_DeleteRCTLinks(ADB_RCT_LINK_INFO *links);
void DBDEF_DeleteImageIcons(ADB_IMAGE_ICON *icon_list);

/**
 * @brief   Sort the full service list into ascending logical channel number order
 */
void DBDEF_SortServicesByLcn(void);

BOOLEAN DBDEF_AllocateLcns(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN assign_lcns);
BOOLEAN DBDEF_AllocateLcnsDefault(E_STB_DP_SIGNAL_TYPE tuner_type);
#ifdef COUNTRY_UK
BOOLEAN DBDEF_AllocateLcnsUK(E_STB_DP_SIGNAL_TYPE tuner_type);
#endif
#ifdef COUNTRY_FINLAND
BOOLEAN DBDEF_AllocateLcnsFinland(E_STB_DP_SIGNAL_TYPE tuner_type);
#endif
#ifdef COUNTRY_IRELAND
BOOLEAN DBDEF_AllocateLcnsIreland(E_STB_DP_SIGNAL_TYPE tuner_type);
#endif

/**
 * @brief   Returns the number of LNB records in the database
 * @return  number of LNB records
 */
U16BIT DBDEF_GetNumLNBs(void);

/**
 * @brief   Add an LNB record to the database
 * @param   type single, universal, unicable or used defined
 * @param   name LNB name, NULL for no name
 * @return  pointer to the new LNB record or NULL if not created
 */
ADB_LNB_REC* DBDEF_AddLNB(E_STB_DP_LNB_TYPE type, ADB_STRING *name);

/**
 * @brief   Sets the LNB power setting
 * @param   lnb pointer to LNB record
 * @param   power power setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBPower(ADB_LNB_REC *lnb, E_STB_DP_LNB_POWER power);

/**
 * @brief   Sets the LNB 22k setting
 * @param   lnb pointer to LNB record
 * @param   is_22k 22k setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNB22k(ADB_LNB_REC *lnb, BOOLEAN is_22k);

/**
 * @brief   Sets the LNB 12V setting
 * @param   lnb pointer to LNB record
 * @param   is_12v 12V setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNB12V(ADB_LNB_REC *lnb, BOOLEAN is_12v);

/**
 * @brief   Sets the LNB pulse position setting
 * @param   lnb pointer to LNB record
 * @param   is_pulse_posn pulse position setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBPulsePosition(ADB_LNB_REC *lnb, BOOLEAN is_pulse_posn);

/**
 * @brief   Sets the LNB DiSEqC position setting
 * @param   lnb pointer to LNB record
 * @param   is_diseqc_posn diseqc position setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBDiSEqCPosition(ADB_LNB_REC *lnb, BOOLEAN is_diseqc_posn);

/**
 * @brief   Sets the LNB DiSEqC tone setting
 * @param   lnb pointer to LNB record
 * @param   diseqc_tone diseqc tone setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBDiSEqCTone(ADB_LNB_REC *lnb, E_STB_DP_DISEQC_TONE diseqc_tone);

/**
 * @brief   Sets the LNB committed switch setting
 * @param   lnb pointer to LNB record
 * @param   cswitch switch setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBCSwitch(ADB_LNB_REC *lnb, E_STB_DP_DISEQC_CSWITCH cswitch);

/**
 * @brief   Sets the LNB uncommitted switch setting
 * @param   lnb pointer to LNB record
 * @param   uswitch switch setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBUSwitch(ADB_LNB_REC *lnb, U8BIT uswitch);

/**
 * @brief   Sets the LNB SMATV setting
 * @param   lnb pointer to LNB record
 * @param   is_smatv SMATV setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBSmatv(ADB_LNB_REC *lnb, BOOLEAN is_smatv);

/**
 * @brief   Sets the LNB message repeat setting in the range 0 to 3
 * @param   lnb pointer to LNB record
 * @param   repeats repeat setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBRepeats(ADB_LNB_REC *lnb, U8BIT repeats);

/**
 * @brief   Sets the LNB Unicable settings
 * @param   lnb pointer to LNB record
 * @param   inter_freq intermediate frequency setting
 * @param   chan channel setting
 * @return  TRUE if value set successfully, FALSE otherwise
 */
BOOLEAN DBDEF_SetLNBUnicable(ADB_LNB_REC *lnb, U32BIT inter_freq, U8BIT chan);

/**
 * @brief   Returns the next LNB record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   lnb_ptr LNB record pointer, NULL if first record is required
 * @return  pointer to next LNB record
 */
ADB_LNB_REC* DBDEF_GetNextLNBRec(ADB_LNB_REC *lnb_ptr);

/**
 * @brief   Returns the number of satellite records in the database
 * @return  number of satellite records
 */
U16BIT DBDEF_GetNumSatellites(void);

/**
 * @brief   Add a satellite record to the database
 * @param   name_str satellite name
 * @param   dish_pos
 * @param   long_pos position in 1/10ths of a degree
 * @param   east_west TRUE=east, FALSE=west
 * @param   associated_lnb LNB record the satellite record is associated with
 * @return  pointer to the new satellite record or NULL if not created
 */
ADB_SATELLITE_REC* DBDEF_AddSatelliteRec(ADB_STRING *name_str, U16BIT dish_pos, U16BIT long_pos,
   BOOLEAN east_west, ADB_LNB_REC *associated_lnb);

/**
 * @brief   Returns the next satellite record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   sat_ptr satellite record pointer, NULL if first record is required
 * @return  pointer to next satellite record
 */
ADB_SATELLITE_REC* DBDEF_GetNextSatelliteRec(ADB_SATELLITE_REC *sat_ptr);

/**
 * @brief   Returns the number of LNB band records in the database
 * @return  number of LNB band records
 */
U16BIT DBDEF_GetNumLNBBands(void);

/**
 * @brief   Add an LNB band record to the database
 * @param   band_parameters Structure describing the band and the expected LNB behaviour.
 * @param   associated_lnb LNB record the band record is associated with.
 * @return  pointer to the new LNB band record or NULL if not created
 */
ADB_LNB_BAND_REC* DBDEF_AddLNBBandRec(S_STB_DP_LNB_BAND *band_parameters, ADB_LNB_REC *associated_lnb);

/**
 * @brief   Returns the next LNB band record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   band_ptr LNB band record pointer, NULL if first record is required
 * @return  pointer to next LNB band record
 */
ADB_LNB_BAND_REC* DBDEF_GetNextLNBBandRec(ADB_LNB_BAND_REC *band_ptr);

/**
 * @brief   Deletes an LNB band record from the database
 * @param   band_ptr pointer to record to be deleted
 */
void DBDEF_DeleteLNBBandRec(ADB_LNB_BAND_REC *band_ptr);

/**
 * @brief   Returns the number of networks in ther service database
 * @return  number of transports
 */
U16BIT DBDEF_GetNumNetworks(void);

/**
 * @brief   Adds a new network record to the database with the given network ID
 * @param   net_id network ID
 * @param   satellite satellite that contains the network, can be NULL
 * @return  pointer to new network record
 */
ADB_NETWORK_REC* DBDEF_AddNetworkRec(U16BIT net_id, ADB_SATELLITE_REC *satellite);

/**
 * @brief   Deletes specified network record
 * @param   n_ptr Pointer to network record
 */
void DBDEF_DeleteNetworkRec(ADB_NETWORK_REC *n_ptr);

/**
 * @brief   Set or change the name of the given network
 * @param   n_ptr pointer to network record to be updated
 * @param   name name to be given to the network, can be any format, or NULL to clear the network name
 */
void DBDEF_SetNetworkName(ADB_NETWORK_REC *n_ptr, U8BIT *name);

/**
 * @brief   Finds the network with the given network ID
 * @param   net_id network ID to be found
 * @param   satellite satellite the network is on, can be NULL
 * @return  pointer to network record, or NULL if not found
 */
ADB_NETWORK_REC* DBDEF_FindNetworkRec(U16BIT net_id, ADB_SATELLITE_REC *satellite);

/**
 * @brief   Returns the network following the one given. If the argument is NULL
 *          then the first network will be returned.
 * @param   n_ptr network record, NULL if first network is required
 * @return  pointer to next network, NULL if no more networks
 */
ADB_NETWORK_REC* DBDEF_GetNextNetworkRec(ADB_NETWORK_REC *n_ptr);

/**
 * @brief   Returns the number of transports in ther service database
 * @return  number of transports
 */
U16BIT DBDEF_GetNumTransports(void);

/**
 * @brief   Returns the transport following the one given. If the argument is NULL
 *          then the first transport will be returned.
 * @param   t_ptr transport record, NULL if first transport is required
 * @return  pointer to next transport, NULL if no more transports
 */
ADB_TRANSPORT_REC* DBDEF_GetNextTransportRec(ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Adds a terrestrial transport record with the given frequency and PLP id
 * @param   freq_hz frequency in Hz
 * @param   plp_id PLP id for T2, pass 0 for T
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddTerrestrialTransportRec(U32BIT freq_hz, U8BIT plp_id,
   ADB_NETWORK_REC *network);
ADB_TRANSPORT_REC* DBDEF_FindTerrestrialTransportRec(U32BIT freq_hz, U8BIT plp_id);

/**
 * @brief   Adds a cable transport record with the given frequency and symbol rate
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddCableTransportRec(U32BIT freq_hz, U32BIT symbol_rate,
   ADB_NETWORK_REC *network);
ADB_TRANSPORT_REC* DBDEF_FindCableTransportRec(U32BIT freq_hz, U32BIT symbol_rate);

void DBDEF_SetCableTransportSymbolRate(ADB_TRANSPORT_REC *t_ptr, U16BIT symbol_rate);
void DBDEF_SetCableTransportMode(ADB_TRANSPORT_REC *t_ptr, E_STB_DP_CMODE cmode);

/**
 * @brief   Adds a satellite transport record with the given frequency, symbol rate and polarity
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   polarity polarity
 * @param   dvb_s2 TRUE for DVB-S2
 * @param   modulation modulation type
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddSatTransportRec(U32BIT freq_hz, U16BIT symbol_rate,
   E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation,
   ADB_NETWORK_REC *network);

/**
 * @brief   Find the satellite transport record in the database matching the given params
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   dvb_s2 TRUE if DVB-S2
 * @param   modulation modulation type
 * @return  pointer to the transport record or NULL if not found
 */
ADB_TRANSPORT_REC* DBDEF_FindSatTransportRec(U32BIT freq_hz, U16BIT symbol_rate,
   E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation, void *satellite);

#ifdef COMMON_INTERFACE
/**
 * @brief   Adds a virtual transport record, for 'Virtual Channel' service
 * @param   network network the transport is on
 * @return  pointer to the new transport record or NULL if not created
 */
ADB_TRANSPORT_REC* DBDEF_AddVirtualTransportRec(ADB_NETWORK_REC *network);
#endif

/**
 * @brief   Sets the tuning parameters for an existing satellite transport
 * @param   t_ptr satellite transport record
 * @param   freq_hz frequency in Hz
 * @param   symbol_rate symbol rate
 * @param   polarity polarity
 * @param   dvb_s2 TRUE for DVB-S2
 * @param   modulation modulation type
 * @param   network network the transport is on, can be NULL
 */
void DBDEF_SetSatTransportTuningParams(ADB_TRANSPORT_REC *t_ptr, U32BIT freq_hz,
   U16BIT symbol_rate, E_STB_DP_POLARITY polarity, BOOLEAN dvb_s2, E_STB_DP_MODULATION modulation,
   ADB_NETWORK_REC *network);

/**
 * @brief   Find a transport record matching the given set of IDs, starting from the given transport
 * @param   transp start search from this transport, use NULL to start from the first transport
 * @param   net_id network ID, use ADB_INVALID_DVB_ID to ignore this value
 * @param   onet_id original network ID, use ADB_INVALID_DVB_ID to ignore this value
 * @param   tran_id transport ID, use ADB_INVALID_DVB_ID to ignore this value
 * @return  pointer to transport record, NULL if none found
 */
ADB_TRANSPORT_REC* DBDEF_FindTransportRecByIds(ADB_TRANSPORT_REC *transp, U16BIT net_id,
   U16BIT onet_id, U16BIT tran_id);

/**
 * @brief   Sets the transport ID of the given transport
 * @param   t_ptr transport record the ID will be set for
 * @param   tran_id transport ID to be set
 */
void DBDEF_SetTransportTransportId(ADB_TRANSPORT_REC *t_ptr, U16BIT tran_id);

/**
 * @brief   Sets the original network ID of the given transport
 * @param   t_ptr transport record the ID will be set for
 * @param   orig_net_id original network ID to be set
 */
void DBDEF_SetTransportOrigNetworkId(ADB_TRANSPORT_REC *t_ptr, U16BIT orig_net_id);

/**
 * @brief   Returns the additional frequencies associated with a transport.
 *          The returned array should not be modified or freed.
 * @param   t_ptr transport being queried
 * @param   num_freqs pointer to a value that returns the number of additional frequencies in the array
 * @param   freqs pointer to an array that contains the additional frequencies
 */
void DBDEF_GetTransportAdditionalFrequencies(ADB_TRANSPORT_REC *t_ptr, U8BIT *num_freqs, U32BIT **freqs);

/**
 * @brief   Deletes the given transport from the service database
 * @param   t_ptr pointer to transport to be deleted
 */
void DBDEF_DeleteTransportRec(ADB_TRANSPORT_REC *t_ptr);

void DBDEF_ClearTableVersionHistory(ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Returns the total number of services
 * @return  total number of services
 */
U16BIT DBDEF_GetNumServices(void);

/**
 * @brief   Returns the service after the one given. If NULL is passed then the first
 *          service in the list is returned.
 * @param   s_ptr pointer to service, NULL if first service is to be returned
 * @return  pointer to service, or NULL if no more services
 */
ADB_SERVICE_REC* DBDEF_GetNextServiceRec(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Returns the service before the one given. If NULL is passed then the last
 *          service in the list is returned.
 * @param   s_ptr pointer to service, NULL if last service is to be returned
 * @return  pointer to service, or NULL if no more services
 */
ADB_SERVICE_REC* DBDEF_GetPrevServiceRec(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Adds a new service record to the service database with the given service ID
 *          and parent transport
 * @param   serv_id service ID for the new service
 * @param   t_ptr parent transport for the new service
 * @return  pointer to the new service, or NULL if the creation fails
 */
ADB_SERVICE_REC* DBDEF_AddServiceRec(U16BIT serv_id, ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Creates a copy of the given service, copying the service's attributes,
 *          e.g. service name, scrambled, locked, hidde, etc, but not the runtime specific
 *          data such as the stream list or event schedule data.
 * @param   orig_serv service to be copied
 * @return  new copied service, NULL on failure
 */
ADB_SERVICE_REC* DBDEF_CopyServiceRec(ADB_SERVICE_REC *orig_serv);

/**
 * @brief   Set or change the name of a service
 * @param   s_ptr service
 * @param   name name to be given to the service, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceName(ADB_SERVICE_REC *s_ptr, U8BIT *name);

/**
 * @brief   Set or change the short name of a service
 * @param   s_ptr service
 * @param   name new short name of the service, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceShortName(ADB_SERVICE_REC *s_ptr, U8BIT *name);

/**
 * @brief   Set or change the name of a service's provider
 * @param   s_ptr service
 * @param   name name to be given to the service's provider, can be any format,
 *               or NULL to clear the current name
 * @return  TRUE if the name is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceProviderName(ADB_SERVICE_REC *s_ptr, U8BIT *name);

/**
 * @brief   Sets the service group mask for the given service record
 * @param   s_ptr service
 * @param   group_mask the CNS channel group mask
 * @return  TRUE if the group mask is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceGroupMask(ADB_SERVICE_REC *s_ptr, U32BIT group_mask);

/**
 * @brief   Sets the service type for the given service record
 * @param   s_ptr service
 * @param   serv_type type of service
 * @return  TRUE if the type is changed, FALSE otherwise
 */
BOOLEAN DBDEF_SetServiceType(ADB_SERVICE_REC *s_ptr, ADB_SERVICE_TYPE serv_type);

/**
 * @brief   Saves the event schedule of a service to the service database
 * @param   s_ptr event data to be saved for this service
 */
void DBDEF_SaveServiceEventSchedule(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Find the next service following the given service that's on the given transport
 * @param   s_ptr find the service after this one, use NULL to get the first service
 * @param   t_ptr transport the service is on
 * @return  pointer to the service found, or NULL if not found
 */
ADB_SERVICE_REC* DBDEF_GetNextServiceOnTransport(ADB_SERVICE_REC *s_ptr, ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Search for the service with the given service ID on the given transport
 * @param   serv_id service ID
 * @param   t_ptr transport to be searched
 * @return  pointer to the service, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRec(U16BIT serv_id, ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Search for a service with the given IDs
 * @param   servp start searching from this service, NULL to start from first service
 * @param   net_id network ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   onet_id original network ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   tran_id transport ID, use ADB_INVALID_DVB_ID if not to be used
 * @param   serv_id service ID, use ADB_INVALID_DVB_ID if not to be used
 * @return  pointer to the next service found, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByIds(ADB_SERVICE_REC *servp, U16BIT net_id, U16BIT onet_id,
   U16BIT tran_id, U16BIT serv_id);

/**
 * @brief   Find the service with the given LCN, and optionally on the given transport
 * @param   lcn LCN
 * @param   t_ptr transport the service must be on, use NULL for any transport
 * @param   allocated_lcn TRUE if the search should be based on the LCN a service has been
 *                        allocated rather than the one it requested (its service LCN)
 * @return  pointer to service if found, NULL otherwise
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByLcn(U16BIT lcn, ADB_TRANSPORT_REC *t_ptr, BOOLEAN allocated_lcn);

/**
 * @brief   Search for a service with the given Freesat ID
 * @param   servp start searching from this service, NULL to start from first service
 * @param   freesat_id Freesat service ID
 * @return  pointer to the next service found, or NULL if no service found
 */
ADB_SERVICE_REC* DBDEF_FindServiceRecByFreesatId(ADB_SERVICE_REC *servp, U16BIT freesat_id);

/**
 * @brief   Search for a stream with the given PID
 * @param   servp searching from this service
 * @param   pid stream PID, use ADB_INVALID_DVB_ID if not to be used
 * @return  stream record found, or NULL if no service found
 */
ADB_STREAM_REC* DBDEF_FindStreamRecById(ADB_SERVICE_REC *servp, U16BIT pid);

U16BIT DBDEF_GetReqdAudioPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *audio_mode, ADB_STREAM_TYPE *audio_type);
U16BIT DBDEF_GetReqdADPid(ADB_SERVICE_REC *s_ptr, E_STB_DP_AUDIO_MODE *ad_mode, ADB_STREAM_TYPE *ad_type,
   BOOLEAN *broadcast_mix);
E_STREAM_MATCH_TYPE DBDEF_GetReqdTtextPid(ADB_SERVICE_REC *s_ptr, BOOLEAN for_subtitles, U16BIT *pid_ptr,
   U8BIT *magazine, U8BIT *page);
E_STREAM_MATCH_TYPE DBDEF_GetReqdSubtitleParams(ADB_SERVICE_REC *s_ptr, U16BIT *pid_ptr,
   U16BIT *cpage_ptr, U16BIT *apage_ptr);
U16BIT DBDEF_GetReqdVideoPid(ADB_SERVICE_REC *s_ptr, ADB_STREAM_TYPE *video_type);

/**
 * @brief   Returns a pointer to the service name, taking into account perferred
 *          names and short names, if available
 * @param   s_ptr service
 * @param   short_name TRUE if short name is to be returned, if available
 * @param   pref_name TRUE if preferred name is to be returned, if available
 * @return  pointer to the service name
 */
ADB_STRING* DBDEF_GetServiceName(ADB_SERVICE_REC *s_ptr, BOOLEAN short_name, BOOLEAN pref_name);

/**
 * @brief   Returns a pointer to the service provider name
 * @param   s_ptr service
 * @return  pointer to the service provider name
 */
ADB_STRING* DBDEF_GetServiceProviderName(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Returns the service group mask
 * @param   s_ptr service
 * @return  the service group mask
 */
U32BIT DBDEF_GetServiceGroupMask(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Marks whether a service is deleted or not
 * @param   s_ptr service
 * @param   deleted TRUE to mark the service as deleted, FALSE as available
 */
void DBDEF_SetServiceDeletedFlag(ADB_SERVICE_REC *s_ptr, BOOLEAN deleted);

/**
 * @brief   Returns the number of services that are marked as deleted for the given tuner type
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @return  number of services
 */
U16BIT DBDEF_GetNumDeletedServices(E_STB_DP_SIGNAL_TYPE tuner_type);

/**
 * @brief
 * @param   tuner_type type of tuner that will be used for the search, if SIGNAL_NONE is specified
 *                     then all services marked as deleted will be returned
 * @param   slist_ptr pointer to an array that will be allocated, value will be NULL on return if
 *                    there are no services. The list must be freed using STB_AppFreeMemory
 * @return  number of services in the returned list
 */
U16BIT DBDEF_GetDeletedServiceList(E_STB_DP_SIGNAL_TYPE tuner_type, void ***slist_ptr);

BOOLEAN DBDEF_AddAnalogService(void);
void DBDEF_SetAnalogServiceName(ADB_SERVICE_REC *s_ptr, U8BIT *new_name, U8BIT new_len);
void DBDEF_SetServicePmtPid(ADB_SERVICE_REC *s_ptr, U16BIT pmt_pid);
U16BIT DBDEF_GetServicePmtPid(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Sets the favourite groups for a service
 * @param   s_ptr service
 * @param   groups bitmask containing the favourite groups the service is to be in
 */
void DBDEF_SetServiceFavGroups(ADB_SERVICE_REC *s_ptr, U8BIT groups);

/**
 * @brief   Find an event for a service from its event_id
 * @param   s_ptr service to be searched
 * @param   event_id ID of the event to be found
 * @return  pointer to the ADB_EVENT_REC, or NULL if not found
 */
ADB_EVENT_REC* DBDEF_FindScheduleEventById(ADB_SERVICE_REC *s_ptr, U16BIT event_id);

/**
 * @brief   Searches a descriptor list for the first descriptor with the given descriptor tag.
 * @param   start_desc descriptor where the search should start
 * @param   desc_tag descriptor tag to be found
 * @param   private_data_specifier the value of the private data specifier that must precede the
 *                                 descriptor being searched for. Use 0 if it isn't required.
 * @return  pointer to the descriptor, or NULL if not found
 */
ADB_EVENT_DESC* DBDEF_FindEventDescriptor(ADB_EVENT_DESC *start_desc, U8BIT desc_tag,
   U32BIT private_data_specifier);

/**
 * @brief   Returns the name of the event as a UTF-8 string in the currently selected language.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventName(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns the short event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventDescription(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns the extended event description text of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
U8BIT* DBDEF_GetEventExtendedDescription(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns the items of extended event descriptor as item descriptor and item itself.
 * @param   e_ptr event
 * @param   num_items_ptr address in which to return the number of items
 * @return  ADB_EVENT_ITEMIZED_INFO pointer to the list of items, NULL on error
 */
ADB_EVENT_ITEMIZED_INFO* DBDEF_GetEventItemizedDescription(ADB_EVENT_REC *e_ptr, U16BIT *num_items_ptr);

/**
 * @brief   Returns the guidance text for an event, either from the event itself
 *          or the event's service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   e_ptr event, can be NULL, in which case any guidance defined by the
 *                    service will be returned
 * @param   s_ptr service, can be NULL to just get guidance text from the event
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetEventGuidance(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr, U8BIT *type, U8BIT *mode);

/**
 * @brief   Returns the guidance text for the service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   s_ptr service
 * @param   type pointer to value to return the guidance type, valid if guidance text is returned
 * @param   mode pointer to value to return the guidance mode, valid if guidance text is returned
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetServiceGuidanceData(ADB_SERVICE_REC *s_ptr, U8BIT *type, U8BIT *mode);

/**
 * @brief   Returns the guidance text for the service, as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString
 * @param   s_ptr service
 * @return  pointer to UTF-8 guidance string, NULL if there isn't any guidance text
 */
U8BIT* DBDEF_GetServiceGuidance(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Returns the parental age value for the given event
 * @param   e_ptr event
 * @return  parental age value, 0 for invalid event
 */
U8BIT DBDEF_GetEventParentalAge(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns whether audio description is signalled as being available for the given event
 * @param   e_ptr event
 * @return  TRUE if available, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventAudioDescriptionFlag(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns whether there's an HD event linked to the SD event and returns
 *          the info for it. The DVB info is checked to ensure it relates to a known
 *          service.
 * @param   e_ptr pointer to the event to check
 * @param   verify_event TRUE, if the schedule should be checked for the target event
 * @param   only_simulcast TRUE, if only an HD simulcast event should be returned
 * @param   hd_serv_ptr pointer to return the service for the HD event, if found.
 *          This can be NULL if not required.
 * @param   hd_event_ptr pointer to return the HD event, if found. Note that this will only be
 *          filled in if verify_event is TRUE. This can be NULL if not required.
 * @return  TRUE if an event is found, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventHDLinkageInfo(ADB_EVENT_REC *e_ptr, BOOLEAN verify_event, BOOLEAN only_simulcast,
   ADB_SERVICE_REC **hd_serv_ptr, ADB_EVENT_REC **hd_event_ptr);

/**
 * @brief   Retrieves a list of components associated with the specified event, as described by
 *          component descriptors in the EIT.
 * @param   e_ptr Pointer to the event
 * @param   component_list Pointer to the returned list of components. The list must be freed by
 *          calling STB_AppFreeMemory, provided the returned number of components is greater than 0.
 * @return  Number of components associated with the specified event.
 */
U8BIT DBDEF_GetEventComponentList(ADB_EVENT_REC *e_ptr, ADB_EVENT_COMPONENT_INFO **component_list);

/**
 * @brief   Returns whether DVB subtitles are signalled as being available for the given event
 * @param   e_ptr event
 * @return  TRUE if subtitles are available, FALSE otherwise
 */
BOOLEAN DBDEF_GetEventSubtitlesAvailFlag(ADB_EVENT_REC *e_ptr);

/**
 * @brief   Returns the genre of an event
 * @param   e_ptr event
 * @param   s_ptr service containing the event
 * @return  event content value, ADB_EVENT_CONTENT_UNCLASSIFIED if not available
 */
ADB_EVENT_CONTENT DBDEF_GetEventGenre(ADB_EVENT_REC *e_ptr, ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Returns the level 1 and 2 values for all content identifiers for the given event
 * @param   e_ptr event
 * @param   p_len address to return the number of bytes in the returned data
 * @return  pointer to the content data
 */
U8BIT* DBDEF_GetEventContentData(ADB_EVENT_REC *e_ptr, U8BIT *p_len);

/**
 * @brief   Returns the full CRID of the type given for the given event
 * @param   serv_ptr service for the event
 * @param   e_ptr event CRID to be returned
 * @param   crid_type CRID type to be returned
 *                      (CRID_TYPE_PROGRAMME, CRID_TYPE_SERIES, or CRID_TYPE_RECOMMENDATION)
 * @param   index the index of the CRID to be returned (only valid for series CRIDs)
 * @return  pointer to a string containing the CRID name,
 *          or NULL if the event doesn't have the requested CRID type
 */
U8BIT* DBDEF_GetEventCrid(ADB_SERVICE_REC *serv_ptr, ADB_EVENT_REC *e_ptr, U8BIT crid_type, U8BIT index);

/**
 * @brief   Returns the episode name of the event as a UTF-8 string.
 *          The returned string should be freed using STB_ReleaseUnicodeString.
 * @param   e_ptr event
 * @return  pointer to UTF-8 string, NULL on error
 */
BOOLEAN DBDEF_GetEventSeriesInfo(ADB_EVENT_REC *e_ptr, U8BIT **episode_name, S32BIT *episode_number, U8BIT *content_status, BOOLEAN *last_episode);

/**
 * @brief   Returns the number of CRIDs of the given type for the event
 * @param   e_ptr counting the CRIDs for this event
 * @param   crid_type CRID type to be returned
 *                      (CRID_TYPE_PROGRAMME, CRID_TYPE_SERIES, or CRID_TYPE_RECOMMENDATION)
 * @return  number of CRIDs found
 */
U8BIT DBDEF_NumberOfCridsOfType(ADB_EVENT_REC *e_ptr, U8BIT crid_type);

/**
 * @brief   Returns the full CRID for the given CRID string
 *          The returned string should be freed using STB_AppFreeMemory
 * @param   serv_ptr service for the event
 * @param   event_str CRID string
 * @return  pointer to a string containing the CRID name
 */
U8BIT* DBDEF_GetFullCrid(ADB_SERVICE_REC *serv_ptr, U8BIT *event_str);

/**
 * @brief   Returns a copy of the default authority CRID string for the given service.
 *          This string will be returned as broadcast and should be freed using STB_AppFreMemory
 * @param   c_ptr pointer to CRID record
 * @return  pointer to a copy of the CRID string, or NULL
 */
U8BIT* DBDEF_GetServiceDefaultAuthority(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Deletes all events in the given list
 * @param   elist list of events to be deleted
 */
void DBDEF_DeleteEventList(ADB_EVENT_REC *elist);

void DBDEF_SetTunedNetwork(U8BIT path, ADB_NETWORK_REC *n_ptr);
ADB_NETWORK_REC* DBDEF_GetTunedNetwork(U8BIT path);

void DBDEF_SetTunedTransport(U8BIT path, ADB_TRANSPORT_REC *t_ptr);
ADB_TRANSPORT_REC* DBDEF_GetTunedTransport(U8BIT path);
void DBDEF_SetTunedService(U8BIT path, ADB_SERVICE_REC *s_ptr);
ADB_SERVICE_REC* DBDEF_GetTunedService(U8BIT path);

void DBDEF_SetTextLang(U8BIT *lang_ids);
U8BIT* DBDEF_GetTextLang(void);
void DBDEF_SetSecondaryTextLang(U8BIT *lang_ids);
U8BIT* DBDEF_GetSecondaryTextLang(void);
void DBDEF_SetAudioLang(U8BIT *lang_ids);
U8BIT* DBDEF_GetAudioLang(void);
U8BIT* DBDEF_GetSecondaryAudioLang(void);
void DBDEF_SetSecondaryAudioLang(U8BIT *lang_ids);

/**
 * @brief   Calls any country and tuner type specific function to tidy up the database following
 *          a search, such as removing duplicate services, etc.
 * @param   tuner_type tuner type used for the search
 * @param   satellite satellite search was performed on for SIGNAL_QPSK, ignored otherwise
 * @param   search_completed TRUE if the search completed, FALSE if terminated prematurely
 * @param   manual TRUE if a manual search was performed, FALSE otherwise
 */
void DBDEF_TidyDatabaseAfterSearch(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite,
   BOOLEAN search_completed, BOOLEAN manual);

void DBDEF_TidyDatabaseNordig(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual);
void DBDEF_TidyDatabaseUK(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual);
void DBDEF_TidyDatabaseSatUK(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual);
void DBDEF_TidyDatabaseDefault(E_STB_DP_SIGNAL_TYPE tuner_type, BOOLEAN search_completed, BOOLEAN manual);

/**
 * @brief   Delete all transport records that don't have any services.
 * @param   tuner_type - empty transports of this signal type will be deleted
 * @param   satellite - if the signal type is SIGNAL_QPSK then this parameter specifies
 *                      the satellite the transports must be on. Ignored if tuner type
 *                      isn't SIGNAL_QPSK, and can be NULL to indicate all satellites.
 */
void DBDEF_RemoveEmptyTransports(E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

void DBDEF_DeleteLinkageDescripterArray(SI_LINKAGE_DESC_ENTRY *list_ptr);

/**
 * @brief   Creates a CRID record and adds it to the database
 * @param   crid CRID string
 * @param   series TRUE if this is a series CRID
 * @param   recommended TRUE if this is a recommended programme CRID
 * @return  pointer to the created CRID record
 */
ADB_CRID_REC* DBDEF_AddCridRecord(U8BIT *crid, BOOLEAN series, BOOLEAN recommended);

/**
 * @brief   Sets the date and time fields in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   date_time date & time to set in the crid
 */
void DBDEF_SetCridDateTime(ADB_CRID_REC *c_ptr, U32DHMS date_time);

/**
 * @brief   Sets the service ID in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   serv_id service ID
 */
void DBDEF_SetCridService(ADB_CRID_REC *c_ptr, U16BIT serv_id);

/**
 * @brief   Sets the programme name field of the given CRID record
 * @param   c_ptr pointer to CRID record
 * @param   prog_name name string to save in the CRID record
 */
void DBDEF_SetCridProgrammeName(ADB_CRID_REC *c_ptr, U8BIT *prog_name);

/**
 * @brief   Sets the do not delete flag in the crid record
 * @param   c_ptr pointer to CRID record
 * @param   do_not_delete new value of the flag
 */
void DBDEF_SetCridDoNotDelete(ADB_CRID_REC *c_ptr, BOOLEAN do_not_delete);

/**
 * @brief   Updates the stored EIT date of this CRID with the current GMT date
 * @param   c_ptr pointer to CRID record
 */
void DBDEF_UpdateCridEitDate(ADB_CRID_REC *c_ptr);

/**
 * @brief   Deletes the given CRID record from the database
 * @param   c_ptr pointer to CRID record to be deleted
 */
void DBDEF_DeleteCridRecord(ADB_CRID_REC *c_ptr);

/**
 * @brief   Returns the number of CRID records in the database
 * @return  number of records
 */
U16BIT DBDEF_GetNumCridRecords(void);

/**
 * @brief   Returns the next CRID record after the one specified. If the record
 *          specified is NULL then the first record is returned.
 * @param   c_ptr CRID record pointer, NULL if first record is required
 * @return  pointer to the next record after the one specified
 */
ADB_CRID_REC* DBDEF_GetNextCridRecord(ADB_CRID_REC *c_ptr);

/**
 * @brief   Checks whether the given crid record is in the list of valid crid records
 * @param   c_ptr crid record to look for
 * @return  TRUE if crid record is valid, FALSE otherwise
 */
BOOLEAN DBDEF_IsValidCridRecord(ADB_CRID_REC *c_ptr);

/**
 * @brief   Returns the number of favourite lists
 * @return  number of favourite lists
 */
U16BIT DBDEF_GetNumFavouriteLists(void);

/**
 * @brief   Creates a new favourite list and adds it to the list of favourite lists.
 *          Creation of the new list will fail if a list_id can't be found for it,
 *          or memory allocation fails.
 * @param   list_id id to be given to the list, should be passed as 0 to have an id assigned.
 *                  If an id is supplied and a list already exists with that id
 *                  then no list will be created.
 * @param   name name to be given to the list, call be NULL
 * @param   user_data user defined valueto be stored in the list,
 * @param   index where to add the new list into the existing lists,
 *                     0 = at start, negative = at end, else 0 based index
 * @return  pointer to new favourite list, or NULL on failure
 */
ADB_FAVLIST_REC* DBDEF_AddFavouriteList(U8BIT list_id, U8BIT *name, U32BIT user_data, S16BIT index);

/**
 * @brief   Returns the favourite list following the given item
 * @param   fav_list return the item after this one, if NULL, returns the first item
 * @return  next favourite list, or NULL if no more
 */
ADB_FAVLIST_REC* DBDEF_GetNextFavouriteList(ADB_FAVLIST_REC *fav_list);

/**
 * @brief   Return the favourite list with the given list id
 * @param   list_id favourite list id to find
 * @return  pointer to favourite list record, or NULL if not found
 */
ADB_FAVLIST_REC* DBDEF_FindFavouriteList(U8BIT list_id);

/**
 * @brief   Saves the given user data with a favourite list
 * @param   fav_list favourite list
 * @param   user_data data to be stored with the list
 */
void DBDEF_SetFavouriteListUserData(ADB_FAVLIST_REC *fav_list, U32BIT user_data);

/**
 * @brief   Changes the order of the favourite lists by moving the given list
 *          to the given position
 * @param   fav_list favourite list to be moved
 * @param   index position to move the list to
 * @return  TRUE if successful, FALSE otherwise
 */
void DBDEF_MoveFavouriteListTo(ADB_FAVLIST_REC *fav_list, S16BIT index);

/**
 * @brief   Deletes the given favourite list
 * @param   fav_list favourite list to be deleted
 */
void DBDEF_DeleteFavouriteList(ADB_FAVLIST_REC *fav_list);

/**
 * @brief   Returns the number of services in the given favourite list
 * @param   fav_list favourite list
 * @return  number of services
 */
U16BIT DBDEF_GetNumServicesInFavouriteList(ADB_FAVLIST_REC *fav_list);

/**
 * @brief   Adds a new service to the given favourite list at the given position
 * @param   fav_list list service is to be added to
 * @param   serv_ptr service to be added to the list
 * @param   index position to add the service in the list,
 *                  0 = at start, negative = at end, any other value is the position
 * @return  new favourite service record, or NULL on failure
 */
ADB_FAVSERV_REC* DBDEF_AddServiceToFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_SERVICE_REC *serv_ptr, S16BIT index);

/**
 * @brief   Returns the ADB_FAVSERV_REC from the given favourite list for the given service
 * @param   fav_list favourite list to be searched
 * @param   serv_ptr service to be searched for
 * @return  pointer to service record, or NULL if not found
 */
ADB_FAVSERV_REC* DBDEF_FindServiceInFavouriteList(ADB_FAVLIST_REC *fav_list, void *serv_ptr);

/**
 * @brief   Returns the next favourite list service record
 * @param   fav_list favourite list
 * @param   fav_serv current service, or NULL if first service to be returned
 * @return  pointer to service record, or NULL if no more services
 */
ADB_FAVSERV_REC* DBDEF_GetNextServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_FAVSERV_REC *fav_serv);

/**
 * @brief   Returns the previous favourite list service record
 * @param   fav_list favourite list
 * @param   fav_serv current service, or NULL if last service to be returned
 * @return  pointer to service record, or NULL if no more services
 */
ADB_FAVSERV_REC* DBDEF_GetPrevServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list,
   ADB_FAVSERV_REC *fav_serv);

/**
 * @brief   Changes the order of the services in the favourite list by moving
 *          the given service to the given position
 * @param   fav_list favourite list
 * @param   fav_serv service to be moved
 * @param   index position to move the service to
 * @return  TRUE if successful, FALSE otherwise
 */
void DBDEF_MoveFavouriteListServiceTo(ADB_FAVLIST_REC *fav_list, ADB_FAVSERV_REC *fav_serv,
   S16BIT index);

/**
 * @brief   Delete the given service from the given favourite list
 * @param   fav_list favourite list to delete service from
 * @param   fav_serv service to be deleted from the favourite list
 */
void DBDEF_DeleteServiceFromFavouriteList(ADB_FAVLIST_REC *fav_list, ADB_FAVSERV_REC *fav_serv);

/**
 * @brief   Delete the all services from the given favourite list
 * @param   fav_list favourite list
 */
void DBDEF_DeleteAllServicesFromFavouriteList(ADB_FAVLIST_REC *fav_list);

/**
 * @brief   Set or change the name of the given favourite list
 * @param   f_ptr pointer to favourite record to be updated
 * @param   name of the favourite list
 * @return  void
 */
void DBDEF_SetFavouriteListName(ADB_FAVLIST_REC *f_ptr, U8BIT *name);

/**
 * @brief   Checks whether the given service is a terrestrial, cable or satellite service, and
 *          optionally check whether it's on a particular satellite
 * @param   s_ptr service to be checked
 * @param   tuner_type type of tuner to check
 * @param   satellite if tuner type is SIGNAL_QPSK then check whether the service is from this
 *                      satellite. This can be NULL if the satellite isn't important and is ignored
 *                      for a non-satellite tuner.
 * @return  TRUE if the service is of the given type, FALSE otherwise
 */
BOOLEAN DBDEF_ServiceForTunerType(ADB_SERVICE_REC *s_ptr, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

/**
 * @brief   Checks whether the given transport is a terrestrial, cable or satellite transport, and
 *          optionally check whether it's on a particular satellite
 * @param   t_ptr transport to be checked
 * @param   tuner_type type of tuner to check
 * @param   satellite if tuner type is SIGNAL_QPSK then check whether the transport is from this
 *                      satellite. This can be NULL if the satellite isn't important and is ignored
 *                      for a non-satellite tuner.
 * @return  TRUE if the transport is of the given type, FALSE otherwise
 */
BOOLEAN DBDEF_TransportForTunerType(ADB_TRANSPORT_REC *t_ptr, E_STB_DP_SIGNAL_TYPE tuner_type, void *satellite);

/**
 * @brief   Creates a new timer record in the database, assigning it a unique handle
 * @param   store_in_nvm save the timer in non-volatile memory
 * @return  pointer to the new timer, or NULL on failure
 */
ADB_TIMER_REC* DBDEF_AddTimerRec(BOOLEAN store_in_nvm);

/**
 * @brief   Returns the timer record with the given timer handle
 * @param   handle timer handle
 * @return  pointer to the timer record, or NULL if not found
 */
ADB_TIMER_REC* DBDEF_FindTimerRec(U32BIT handle);

/**
 * @brief   Returns the next timer record after the one given. If the argument is
 *          NULL then the first record is returned.
 * @param   timer_ptr timer record pointer, NULL if first record is required
 * @return  pointer to next timer record
 */
ADB_TIMER_REC* DBDEF_GetNextTimerRec(ADB_TIMER_REC *timer_ptr);

/**
 * @brief   Sorts timer list into date/time or alphabetical order
 * @param   date_time_order TRUE to sort into date/time order, FALSE for alphabetical
 */
void DBDEF_SortTimers(BOOLEAN date_time_order);

/**
 * @brief   Deletes the given timer from the database
 * @param   timer timer to be deleted
 */
void DBDEF_DeleteTimerRec(ADB_TIMER_REC *timer);

/**
 * @brief   Returns the number of network profiles
 */
U16BIT DBDEF_GetNumProfiles(void);

/**
 * @brief   Gets a list of the available network profiles
 * @param   profile_list address of pointer to an array that will be allocated by this
 *                       function that will contain the profiles
 * @param   active_profile pointer to return the index of the currently active profile
 * @return  number of profiles in the returned array
 */
U16BIT DBDEF_GetProfileList(void ***profile_list, U16BIT *active_profile);

/**
 * @brief   Frees a profile list returned by DBDEF_GetProfileList
 * @param   profile_list profile list to be freed
 * @param   num_profiles number of profiles in the list
 */
void DBDEF_ReleaseProfileList(void **profile_list, U16BIT num_profiles);

/**
 * @brief   Is the given profile the currently active profile?
 * @param   profile profile
 * @return  TRUE if the given profile is the currently active profile, FALSE otherwise
 */
BOOLEAN DBDEF_IsActiveProfile(ADB_NETWORK_REC *profile);

/**
 * @brief   Returns the current profile type
 * @return  Current profile type in use
 */
ADB_PROFILE_TYPE DBDEF_GetCurrentProfileType(void);

/**
 * @brief   Sets the broadcast profile type for for all network, transport
 *          and service record accesses
 */
void DBDEF_SelectBroadcastProfile(void);

/**
 * @brief   Saves the current profile and any related data so that it can restored
 *          using DBDEF_PopProfile(), and sets the broadcast profile type for for
 *          all network, transport and service record accesses
 */
void DBDEF_PushBroadcastProfile(void);

#ifdef COMMON_INTERFACE
/**
 * @brief   Saves the current profile and any related data so that it can be restored
 *          using DBDEF_PopProfile(), and sets the current CI+ profile type for for all
 *          network, transport and service record accesses
 * @param   onet_id original network id of the CI+ operator profile
 * @param   cicam_id CICAM identifier of the CI+ operator profile
 */
void DBDEF_SelectCIPlusProfile(U16BIT onet_id, U32BIT cicam_id);

/**
 * @brief   Sets the CI+ profile type for for all network, transport
 *          and service record accesses
 * @param   onet_id original network id of the CI+ operator profile
 * @param   cicam_id CICAM identifier of the CI+ operator profile
 */
void DBDEF_PushCIPlusProfile(U16BIT onet_id, U32BIT cicam_id);
#endif

/**
 * @brief   Restores a previously pushed profile
 */
void DBDEF_PopProfile(void);

/**
 * @brief   Checks whether the given service is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   s_ptr service to be checked
 * @return  TRUE if the service isn't part of a network or is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_ServiceInProfile(ADB_SERVICE_REC *s_ptr);

/**
 * @brief   Checks whether the given transport is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   t_ptr transport to be checked
 * @return  TRUE if the transport isn't part of a network or is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_TransportInProfile(ADB_TRANSPORT_REC *t_ptr);

/**
 * @brief   Checks whether the given network is valid for the current profile.
 *          There may be multiple CI+ profiles and so data related to the CAM is also checked.
 * @param   n_ptr network to be checked
 * @return  TRUE if the network is in the current profile, FALSE otherwise
 */
BOOLEAN DBDEF_NetworkInProfile(ADB_NETWORK_REC *n_ptr);

#ifdef COMMON_INTERFACE
ADB_NETWORK_REC* DBDEF_FindNetworkForCIPlusProfile(U16BIT cicam_onet_id, U32BIT cicam_identifier);
ADB_NETWORK_REC* DBDEF_FindNetworkForCIPlusModule(U32BIT module);

/**
 * @brief   Reuturns the currently selected CI+ profile.
 * @return  Network record of the selected CI+ profile or NULL if the current profile is broadcast
 */
ADB_NETWORK_REC *DBDEF_GetCurrentCIPlusProfile(void);

/**
 * @brief   Add CICAM timer record to the database
 * @param   cicam_id CICAM ID
 * @param   timer_handle Timer handle
 * @return  pointer to the new record or NULL if not created
 */
ADB_CICAM_TIMER_REC* DBDEF_AddCicamTimer(U32BIT cicam_id, U32BIT timer_handle);

/**
 * @brief   Deletes the given cicam timer from the service database.
 * @param   cicam_timer_ptr pointer to cicam timer to be deleted
 */
void DBDEF_DeleteCicamTimerRec(ADB_CICAM_TIMER_REC *cicam_timer_ptr);

/**
 * @brief   Returns the CICAM timer following the one given. If the argument is NULL
 *          then the first CICAM timer will be returned.
 * @param   cicam_timer_ptr CICAM timer record, NULL if first CICAM timer is required
 * @return  pointer to next CICAM timer, NULL if no more CICAM timers
 */
ADB_CICAM_TIMER_REC* DBDEF_GetNextCicamTimerRec(ADB_CICAM_TIMER_REC *cicam_timer_ptr);

/**
 * @brief   Returns the CICAM timer associated with a CICAM ID
 * @param   cicam_id CICAM ID
 * @return  CICAM timer record or NULL if none is found
 */
ADB_CICAM_TIMER_REC* DBDEF_FindCicamTimer(U32BIT cicam_id);

/**
 * @brief   Returns the CICAM timer associated with the specified timer handle
 * @param   handle Timer handle
 * @return  CICAM timer record or NULL if none is found
 */
ADB_CICAM_TIMER_REC* DBDEF_FindCicamTimerByHandle(U32BIT handle);

#endif

/**
 * @brief   Find or add a private network, assigning an unused private network ID.
 * @param   satellite optional satellite, can be NULL
 * @return  pointer to existing or new network record
 */
ADB_NETWORK_REC* DBDEF_FindOrAddPrivateNetwork(void *satellite);

#ifdef DEBUG_PRINT_DATABASE
void DBDEF_PrintAllRecords(void);
#endif

/**
 * @brief   Creates an ADB_STRING, copying the given data into it. If the string passed in
 *          is NULL or the number of bytes is 0, then no string will be created
 * @param   lang_code ISO 639-2 3 char language code
 * @param   str_ptr pointer to string data to be copied into the ADB_STRING
 * @param   nbytes number of bytes of string data
 * @return  pointer to new ADB_STRING, or NULL
 */
ADB_STRING* DBDEF_MakeString(U32BIT lang_code, U8BIT *str_ptr, U16BIT nbytes);

/**
 * @brief   Creates a copy of the given ADB_STRING
 * @param   src_str string to be copied
 * @return  new ADB_STRING
 */
ADB_STRING* DBDEF_CopyString(ADB_STRING *src_str);

/**
 * @brief   Concatenates an SI_STRING_DESC string to the end of an ADB_STRING string,
 *          removing any terminating '\0' chars from the first string, and returns
 *          a new string.
 * @param   str1 string 1
 * @param   str2 string 2
 * @return  new ADB_STRING containing the contents of both strings
 */
ADB_STRING* DBDEF_ConcatSIString(ADB_STRING *str1, SI_STRING_DESC *str2);

/**
 * @brief   Releases an ADB_STRING
 * @param   string string to be freed
 */
void DBDEF_ReleaseString(ADB_STRING *string);

#endif //  _AP_DBDEF_H

//*****************************************************************************
// End of file
//*****************************************************************************

