/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2013 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Database XML import/export
 * @file    ap_dbxml.c
 * @date    November 2013
 * @author  Ocean Blue
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <string.h>
#include <stdio.h>

// third party header files
#include <libxml/xmlwriter.h>
#include <libxml/xmlreader.h>

// Ocean Blue Software header files
#include "techtype.h"
#include "dbgfuncs.h"

#include "stbhwos.h"
#include "stbuni.h"
#include "stbdpc.h"

#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"

#include "dba.h"

//---constant definitions for this file-------------------------------------------------------------
//#define DEBUG_XML
#ifdef DEBUG_XML
#define DBG_XML(X)      STB_SPDebugWrite X
#else
#define DBG_XML(X)
#endif


/* Definitions for XML Import / Export format */
#define XMLENCODING                 "utf-8"

#if 0
// xmltv for EPG
#define XMLTAG_TV "tv"
#define XMLTAG_CHANNEL "channel"
#define XMLTAG_DISPLAYNAME "display-name"
#define XMLTAG_URL "url"
#define XMLTAG_PROGRAMME "programme"
#define XMLTAG_TITLE "title"
#define XMLTAG_DESC "desc"
#define XMLTAG_SUBTITLES "subtitles"
#define XMLTAG_RATING "rating"
#define XMLTAG_VIDEO "video"
#define XMLTAG_QUALITY "quality"
#define XMLTAG_CATEGORY "category"
#define XMLTAG_AUDIO "audio"
#define XMLTAG_PRESENT "present"
#define XMLTAG_STEREO "stereo"
#define XMLTAG_VALUE "value"
#endif

/* XML tags used for service database */
#define XMLTAG_SERVICE_DB           (xmlChar *)"dvb_records"

/* Generic tags used for storing information */
#define XMLTAG_TYPE                 (xmlChar *)"type"
#define XMLTAG_NAME                 (xmlChar *)"name"
#define XMLTAG_VERSION              (xmlChar *)"version"
#define XMLTAG_NET_ID               (xmlChar *)"net_id"
#define XMLTAG_ORIG_NET_ID          (xmlChar *)"orig_net_id"
#define XMLTAG_TRAN_ID              (xmlChar *)"tran_id"
#define XMLTAG_SERV_ID              (xmlChar *)"serv_id"
#define XMLTAG_OFF                  (xmlChar *)"off"
#define XMLTAG_ON                   (xmlChar *)"on"
#define XMLTAG_AUTO                 (xmlChar *)"auto"
#define XMLTAG_0                    (xmlChar *)"0"
#define XMLTAG_1                    (xmlChar *)"1"
#define XMLTAG_2                    (xmlChar *)"2"
#define XMLTAG_4                    (xmlChar *)"4"
#define XMLTAG_5                    (xmlChar *)"5"
#define XMLTAG_6                    (xmlChar *)"6"
#define XMLTAG_7                    (xmlChar *)"7"
#define XMLTAG_8                    (xmlChar *)"8"
#define XMLTAG_10                   (xmlChar *)"10"
#define XMLTAG_16                   (xmlChar *)"16"
#define XMLTAG_32                   (xmlChar *)"32"
#define XMLTAG_64                   (xmlChar *)"64"
#define XMLTAG_128                  (xmlChar *)"128"
#define XMLTAG_A                    (xmlChar *)"a"
#define XMLTAG_B                    (xmlChar *)"b"
#define XMLTAG_C                    (xmlChar *)"c"
#define XMLTAG_D                    (xmlChar *)"d"

/* XML tags for LNBs */
#define XMLTAG_LNB                  (xmlChar *)"lnb"
#define XMLTAG_LNB_POWER            (xmlChar *)"power"
#define XMLTAG_LNB_IS22K            (xmlChar *)"is22k"
#define XMLTAG_LNB_IS12V            (xmlChar *)"is12v"
#define XMLTAG_LNB_ISPULSEPOS       (xmlChar *)"pulse_pos"
#define XMLTAG_LNB_ISDISEQCPOS      (xmlChar *)"diseqc_pos"
#define XMLTAG_LNB_DISEQC_TONE      (xmlChar *)"diseqc_tone"
#define XMLTAG_LNB_CSWITCH          (xmlChar *)"cswitch"
#define XMLTAG_LNB_USWITCH          (xmlChar *)"uswitch"
#define XMLTAG_LNB_ISSMATV          (xmlChar *)"issmatv"
#define XMLTAG_LNB_DISEQC_REPEATS   (xmlChar *)"repeats"
#define XMLTAG_LNB_UNICABLE_IF      (xmlChar *)"unicable_if"
#define XMLTAG_LNB_UNICABLE_CHAN    (xmlChar *)"unicable_chan"
#define XMLTAG_LNB_SINGLE           (xmlChar *)"single"
#define XMLTAG_LNB_UNIVERSAL        (xmlChar *)"universal"
#define XMLTAG_LNB_UNICABLE         (xmlChar *)"unicable"
#define XMLTAG_LNB_USER_DEFINED     (xmlChar *)"user_defined"

/* XML tags for LNB bands */
#define XMLTAG_LNB_BAND             (xmlChar *)"lnb_band"
#define XMLTAG_LNB_POLARITY         (xmlChar *)"polarity"
#define XMLTAG_LNB_BAND_MIN_FREQ    (xmlChar *)"min_frequency"
#define XMLTAG_LNB_BAND_MAX_FREQ    (xmlChar *)"max_frequency"
#define XMLTAG_LNB_BAND_LO_FREQ     (xmlChar *)"lo_frequency"
#define XMLTAG_LNB_BAND_VOLTAGE     (xmlChar *)"lnb_voltage"
#define XMLTAG_LNB_BAND_22KHZ       (xmlChar *)"22_khz"
#define XMLTAG_LNB_BAND_14V         (xmlChar *)"14V"
#define XMLTAG_LNB_BAND_18V         (xmlChar *)"18V"

/* XML tags for satellites */
#define XMLTAG_SATELLITE            (xmlChar *)"satellite"
#define XMLTAG_SAT_DISHPOS          (xmlChar *)"dish_pos"
#define XMLTAG_SAT_LONGPOS          (xmlChar *)"long_pos"
#define XMLTAG_SAT_EASTWEST         (xmlChar *)"east_west"

/* XML tags for networks */
#define XMLTAG_NETWORK              (xmlChar *)"network"
#define XMLTAG_PROFILETYPE          (xmlChar *)"profile_type"
#define XMLTAG_PROFILE_BROADCAST    (xmlChar *)"broadcast"
#define XMLTAG_PROFILE_CIPLUS       (xmlChar *)"ciplus"
#define XMLTAG_PROFILE_NAME         (xmlChar *)"profile_name"
#define XMLTAG_CAM_ONET             (xmlChar *)"CAM_onet"
#define XMLTAG_CAM_ID               (xmlChar *)"CAM_id"

/* XML tags for transports */
#define XMLTAG_SAT_TRANSPORT        (xmlChar *)"sat_transport"
#define XMLTAG_CAB_TRANSPORT        (xmlChar *)"cab_transport"
#define XMLTAG_TERR_TRANSPORT       (xmlChar *)"terr_transport"
#define XMLTAG_ANALOG_TRANSPORT     (xmlChar *)"analog_transport"
#define XMLTAG_TRAN_FREQUENCY       (xmlChar *)"freq"
#define XMLTAG_TRAN_SKEW            (xmlChar *)"skew"
#define XMLTAG_TRAN_SYMRATE         (xmlChar *)"symrate"
#define XMLTAG_TRAN_POLARITY        (xmlChar *)"polarity"
#define XMLTAG_TRAN_FECCODE         (xmlChar *)"fec_code"
#define XMLTAG_TRAN_FECTYPE         (xmlChar *)"fec_type"
#define XMLTAG_TRAN_MODULATION      (xmlChar *)"modulation"
#define XMLTAG_TRAN_DVBS2           (xmlChar *)"dvb_s2"
#define XMLTAG_TRAN_MODE            (xmlChar *)"mode"
#define XMLTAG_TRAN_BWIDTH          (xmlChar *)"bwidth"
#define XMLTAG_TRAN_TERRTYPE        (xmlChar *)"terr_type"
#define XMLTAG_TRAN_PLPID           (xmlChar *)"plp_id"
#define XMLTAG_TRAN_OFFSET          (xmlChar *)"offset"
#define XMLTAG_TRAN_CONSTELLATION   (xmlChar *)"const"
#define XMLTAG_TRAN_HIERARCHY       (xmlChar *)"hier"
#define XMLTAG_TRAN_LPCODE          (xmlChar *)"lp_code"
#define XMLTAG_TRAN_HPCODE          (xmlChar *)"hp_code"
#define XMLTAG_TRAN_GUARDINT        (xmlChar *)"guard_int"
#define XMLTAG_TRAN_VTYPE           (xmlChar *)"vtype"
#define XMLTAG_HORIZONTAL           (xmlChar *)"horizontal"
#define XMLTAG_VERTICAL             (xmlChar *)"vertical"
#define XMLTAG_LEFT                 (xmlChar *)"left"
#define XMLTAG_RIGHT                (xmlChar *)"right"
#define XMLTAG_1_2                  (xmlChar *)"1_2"
#define XMLTAG_1_3                  (xmlChar *)"1_3"
#define XMLTAG_1_4                  (xmlChar *)"1_4"
#define XMLTAG_1_8                  (xmlChar *)"1_8"
#define XMLTAG_1_16                 (xmlChar *)"1_16"
#define XMLTAG_1_32                 (xmlChar *)"1_32"
#define XMLTAG_1_128                (xmlChar *)"1_128"
#define XMLTAG_2_3                  (xmlChar *)"2_3"
#define XMLTAG_2_5                  (xmlChar *)"2_5"
#define XMLTAG_3_4                  (xmlChar *)"3_4"
#define XMLTAG_5_6                  (xmlChar *)"5_6"
#define XMLTAG_7_8                  (xmlChar *)"7_8"
#define XMLTAG_8_9                  (xmlChar *)"8_9"
#define XMLTAG_9_10                 (xmlChar *)"9_10"
#define XMLTAG_3_5                  (xmlChar *)"3_5"
#define XMLTAG_4_5                  (xmlChar *)"4_5"
#define XMLTAG_19_128               (xmlChar *)"19_128"
#define XMLTAG_19_256               (xmlChar *)"19_256"
#define XMLTAG_FEC_DVBS1            (xmlChar *)"dvbs1"
#define XMLTAG_FEC_DVBS2            (xmlChar *)"dvbs2"
#define XMLTAG_QPSK                 (xmlChar *)"qpsk"
#define XMLTAG_8PSK                 (xmlChar *)"8psk"
#define XMLTAG_16QAM                (xmlChar *)"16qam"
#define XMLTAG_QAM4                 (xmlChar *)"qam4"
#define XMLTAG_QAM8                 (xmlChar *)"qam8"
#define XMLTAG_QAM16                (xmlChar *)"qam16"
#define XMLTAG_QAM32                (xmlChar *)"qam32"
#define XMLTAG_QAM64                (xmlChar *)"qam64"
#define XMLTAG_QAM128               (xmlChar *)"qam128"
#define XMLTAG_QAM256               (xmlChar *)"qam256"
#define XMLTAG_DVBT                 (xmlChar *)"dvbt"
#define XMLTAG_DVBT2                (xmlChar *)"dvbt2"

/* XML tags for services */
#define XMLTAG_SERVICE              (xmlChar *)"service"
#define XMLTAG_SERV_LCN             (xmlChar *)"serv_lcn"
#define XMLTAG_SERV_ALLOCLCN        (xmlChar *)"alloc_lcn"
#define XMLTAG_SERV_FAVGROUPS       (xmlChar *)"fav_groups"
#define XMLTAG_SERV_HIDDEN          (xmlChar *)"hidden"
#define XMLTAG_SERV_SELECTABLE      (xmlChar *)"selectable"
#define XMLTAG_SERV_LOCKED          (xmlChar *)"locked"
#define XMLTAG_SERV_SCHED_DISABLED  (xmlChar *)"sched_disabled"
#define XMLTAG_SERV_PF_DISABLED     (xmlChar *)"pf_disabled"
#define XMLTAG_SERV_EDIT_LCN        (xmlChar *)"edit_lcn"
#define XMLTAG_SERV_FREESAT_ID      (xmlChar *)"freesat_id"
#define XMLTAG_SERV_REGION_ID       (xmlChar *)"region_id"
#define XMLTAG_TV                   (xmlChar *)"tv"
#define XMLTAG_RADIO                (xmlChar *)"radio"
#define XMLTAG_AVC_RADIO            (xmlChar *)"avc_radio"
#define XMLTAG_DATA                 (xmlChar *)"data"
#define XMLTAG_MPEG2_HD             (xmlChar *)"mpeg2_hd"
#define XMLTAG_AVC_SD_TV            (xmlChar *)"avc_sd_tv"
#define XMLTAG_HD_TV                (xmlChar *)"hd_tv"
#define XMLTAG_UHD_TV               (xmlChar *)"uhd_tv"
#define XMLTAG_ANALOG               (xmlChar *)"analog"

/* XML tags for favourite lists and services */
#define XMLTAG_FAVLIST              (xmlChar *)"favlist"
#define XMLTAG_FAVSERV              (xmlChar *)"favserv"
#define XMLTAG_ID                   (xmlChar *)"id"
#define XMLTAG_INDEX                (xmlChar *)"index"
#define XMLTAG_USERDATA             (xmlChar *)"userdata"

/* XML tags for CRID records */
#define XMLTAG_CRID                 (xmlChar *)"crid"
#define XMLTAG_EIT_DATE             (xmlChar *)"eit_date"
#define XMLTAG_SERIES               (xmlChar *)"series"
#define XMLTAG_RECOMMENDED          (xmlChar *)"recommended"
#define XMLTAG_DATETIME             (xmlChar *)"datetime"
#define XMLTAG_PROG_CRID            (xmlChar *)"prog_crid"

/* XML tags for timers */
#define XMLTAG_TIMER                (xmlChar *)"timer"
#define XMLTAG_HANDLE               (xmlChar *)"handle"
#define XMLTAG_TIMER_ALARM          (xmlChar *)"alarm"
#define XMLTAG_TIMER_SLEEP          (xmlChar *)"sleep"
#define XMLTAG_TIMER_PVR_RECORD     (xmlChar *)"pvr_rec"
#define XMLTAG_TIMER_PRIVATE        (xmlChar *)"private"
#define XMLTAG_FREQ                 (xmlChar *)"freq"
#define XMLTAG_TIMER_ONCE           (xmlChar *)"once"
#define XMLTAG_TIMER_WEEKLY         (xmlChar *)"weekly"
#define XMLTAG_TIMER_WEEKENDDAYS    (xmlChar *)"weekenddays"
#define XMLTAG_TIMER_WEEKDAYS       (xmlChar *)"weekdays"
#define XMLTAG_TIMER_DAILY          (xmlChar *)"daily"
#define XMLTAG_TIMER_HOURLY         (xmlChar *)"hourly"
#define XMLTAG_MISSED               (xmlChar *)"missed"
#define XMLTAG_CHANGESERVICE        (xmlChar *)"change"
#define XMLTAG_RAMPVOLUME           (xmlChar *)"volume"
#define XMLTAG_DURATION             (xmlChar *)"duration"
#define XMLTAG_TRIGGERED            (xmlChar *)"triggered"
#define XMLTAG_EVENT_ID             (xmlChar *)"event"
#define XMLTAG_DISK_ID              (xmlChar *)"disk"
#define XMLTAG_NOTIFY               (xmlChar *)"notify"
#define XMLTAG_START_PAD            (xmlChar *)"start_pad"
#define XMLTAG_END_PAD              (xmlChar *)"end_pad"
#define XMLTAG_OTHER_CRID           (xmlChar *)"other_crid"
#define XMLTAG_ADDITIONAL_INFO      (xmlChar *)"add_info"

#define UC_QUOTE           0x0022
#define UC_AMPERSAND       0x0026
#define UC_APOSTROPHE      0x0027
#define UC_LESS_THAN       0x003c
#define UC_GREATER_THAN    0x003e

//---local typedefs, structs, enumerations for this file--------------------------------------------

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static void ExportLNBRec(xmlTextWriterPtr writer, ADB_LNB_REC *lnb_ptr);
static void ExportLNBBandeRec(xmlTextWriterPtr writer, ADB_LNB_BAND_REC *band_ptr);
static void ExportSatelliteRec(xmlTextWriterPtr writer, ADB_SATELLITE_REC *sat_ptr);
static void ExportNetworkRec(xmlTextWriterPtr writer, ADB_NETWORK_REC *n_ptr);
static void ExportTransportRec(xmlTextWriterPtr writer, ADB_TRANSPORT_REC *t_ptr);
static void ExportServiceRec(xmlTextWriterPtr writer, ADB_SERVICE_REC *s_ptr);
static void ExportFavList(xmlTextWriterPtr writer, ADB_FAVLIST_REC *fav_list);
static void ExportCridRec(xmlTextWriterPtr writer, ADB_CRID_REC *crid_rec);
static void ExportTimerRec(xmlTextWriterPtr writer, ADB_TIMER_REC *timer);

static ADB_LNB_REC* ReadLNB(xmlTextReaderPtr reader);
static ADB_LNB_BAND_REC* ReadLNBBand(xmlTextReaderPtr reader, ADB_LNB_REC *lnb_ptr);
static ADB_SATELLITE_REC* ReadSatellite(xmlTextReaderPtr reader, ADB_LNB_REC *lnb_ptr);
static ADB_NETWORK_REC* ReadNetwork(xmlTextReaderPtr reader, ADB_SATELLITE_REC *sat_ptr);
static ADB_TRANSPORT_REC* ReadSatTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr);
static ADB_TRANSPORT_REC* ReadCableTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr);
static ADB_TRANSPORT_REC* ReadTerrTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr);
static ADB_TRANSPORT_REC* ReadAnalogTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr);
static ADB_SERVICE_REC* ReadService(xmlTextReaderPtr reader, ADB_TRANSPORT_REC *t_ptr);
static ADB_FAVLIST_REC* ReadFavList(xmlTextReaderPtr reader);
static ADB_FAVSERV_REC* ReadFavServ(xmlTextReaderPtr reader, ADB_FAVLIST_REC *fav_list);
static ADB_CRID_REC* ReadCridRec(xmlTextReaderPtr reader);
static U32BIT ReadTimerRec(xmlTextReaderPtr reader);

static void XMLWriteValueString(xmlTextWriterPtr writer, xmlChar *tag_str, ADB_STRING *adb_string);


BOOLEAN ADB_ImportDB(U8BIT *filename)
{
   xmlTextReaderPtr reader;
   BOOLEAN retval;
   xmlChar *node_name;
   xmlReaderTypes node_type;
   ADB_LNB_REC *lnb_ptr;
   ADB_SATELLITE_REC *sat_ptr;
   ADB_NETWORK_REC *n_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ADB_ImportDB);

   retval = FALSE;

   /* This macro checks the libxml2 version used when building and version used at runtime
    * are compatible */
   LIBXML_TEST_VERSION

   reader = xmlNewTextReaderFilename((const char *)filename);
   if (reader != NULL)
   {
      DBDEF_RequestAccess();

      lnb_ptr = NULL;
      sat_ptr = NULL;
      n_ptr = NULL;
      t_ptr = NULL;
      fav_list = NULL;

      while (xmlTextReaderRead(reader))
      {
         node_type = xmlTextReaderNodeType(reader);

         if ((node_type == XML_READER_TYPE_ELEMENT) || (node_type == XML_READER_TYPE_END_ELEMENT))
         {
            node_name = xmlTextReaderName(reader);

            if (node_name != NULL)
            {
               if (xmlStrEqual(node_name, (xmlChar *)XMLTAG_LNB) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     lnb_ptr = ReadLNB(reader);
                  }
                  else
                  {
                     lnb_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, (xmlChar *)XMLTAG_LNB_BAND) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     ReadLNBBand(reader, lnb_ptr);
                  }
               }
               else if (xmlStrEqual(node_name, (xmlChar *)XMLTAG_SATELLITE) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     sat_ptr = ReadSatellite(reader, lnb_ptr);
                  }
                  else
                  {
                     sat_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_NETWORK) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     n_ptr = ReadNetwork(reader, sat_ptr);
                  }
                  else
                  {
                     n_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_SAT_TRANSPORT) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     t_ptr = ReadSatTransport(reader, n_ptr);
                  }
                  else
                  {
                     t_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_CAB_TRANSPORT) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     t_ptr = ReadCableTransport(reader, n_ptr);
                  }
                  else
                  {
                     t_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_TERR_TRANSPORT) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     t_ptr = ReadTerrTransport(reader, n_ptr);
                  }
                  else
                  {
                     t_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_ANALOG_TRANSPORT) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     t_ptr = ReadAnalogTransport(reader, n_ptr);
                  }
                  else
                  {
                     t_ptr = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_SERVICE) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     ReadService(reader, t_ptr);
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_FAVLIST) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     fav_list = ReadFavList(reader);
                  }
                  else
                  {
                     fav_list = NULL;
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_FAVSERV) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     ReadFavServ(reader, fav_list);
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_CRID) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     ReadCridRec(reader);
                  }
               }
               else if (xmlStrEqual(node_name, XMLTAG_TIMER) != 0)
               {
                  if (node_type == XML_READER_TYPE_ELEMENT)
                  {
                     ReadTimerRec(reader);
                  }
               }

               xmlFree(node_name);
            }
         }
      }

      DBDEF_SortServicesByLcn();
      DBDEF_ReleaseAccess();

      xmlFreeTextReader(reader);

      ADB_SaveDatabase();

      retval = TRUE;
   }

   FUNCTION_FINISH(ADB_ImportDB);

   return(retval);
}

static ADB_LNB_REC* ReadLNB(xmlTextReaderPtr reader)
{
   xmlChar *name;
   xmlChar *value;
   E_STB_DP_LNB_TYPE type;
   E_STB_DP_LNB_POWER power;
   BOOLEAN is_22k;
   BOOLEAN is_12v;
   BOOLEAN pulse_pos;
   BOOLEAN diseqc_pos;
   E_STB_DP_DISEQC_TONE diseqc_tone;
   E_STB_DP_DISEQC_CSWITCH cswitch;
   U8BIT uswitch;
   BOOLEAN is_smatv;
   U8BIT repeats;
   U32BIT unicable_if;
   U8BIT unicable_chan;
   BOOLEAN lnb_found;
   ADB_LNB_REC *lnb_ptr;
   U8BIT *lnb_name;
   U16BIT name_len;
   ADB_STRING *adb_string;

   FUNCTION_START(ReadLNB);

   lnb_name = NULL;

   /* Read LNB settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_TYPE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_LNB_SINGLE) != 0)
               {
                  type = LNB_TYPE_SINGLE;
               }
               else if (xmlStrEqual(value, XMLTAG_LNB_UNIVERSAL) != 0)
               {
                  type = LNB_TYPE_UNIVERSAL;
               }
               else if (xmlStrEqual(value, XMLTAG_LNB_UNICABLE) != 0)
               {
                  type = LNB_TYPE_UNICABLE;
               }
               else if (xmlStrEqual(value, XMLTAG_LNB_USER_DEFINED) != 0)
               {
                  type = LNB_TYPE_USER_DEFINED;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  lnb_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_POWER) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_OFF) != 0)
               {
                  power = LNB_POWER_OFF;
               }
               else if (xmlStrEqual(value, XMLTAG_ON) != 0)
               {
                  power = LNB_POWER_ON;
               }
               else if (xmlStrEqual(value, XMLTAG_AUTO) != 0)
               {
                  power = LNB_POWER_AUTO;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_IS22K) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  is_22k = TRUE;
               }
               else
               {
                  is_22k = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_IS12V) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  is_12v = TRUE;
               }
               else
               {
                  is_12v = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_ISPULSEPOS) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  pulse_pos = TRUE;
               }
               else
               {
                  pulse_pos = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_ISDISEQCPOS) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  diseqc_pos = TRUE;
               }
               else
               {
                  diseqc_pos = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_DISEQC_TONE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_OFF) != 0)
               {
                  diseqc_tone = DISEQC_TONE_OFF;
               }
               else if (xmlStrEqual(value, XMLTAG_A) != 0)
               {
                  diseqc_tone = DISEQC_TONE_A;
               }
               else if (xmlStrEqual(value, XMLTAG_B) != 0)
               {
                  diseqc_tone = DISEQC_TONE_B;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_CSWITCH) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_OFF) != 0)
               {
                  cswitch = DISEQC_CSWITCH_OFF;
               }
               else if (xmlStrEqual(value, XMLTAG_A) != 0)
               {
                  cswitch = DISEQC_CSWITCH_A;
               }
               else if (xmlStrEqual(value, XMLTAG_B) != 0)
               {
                  cswitch = DISEQC_CSWITCH_B;
               }
               else if (xmlStrEqual(value, XMLTAG_C) != 0)
               {
                  cswitch = DISEQC_CSWITCH_C;
               }
               else if (xmlStrEqual(value, XMLTAG_D) != 0)
               {
                  cswitch = DISEQC_CSWITCH_D;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_USWITCH) != 0)
            {
               uswitch = (U8BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_ISSMATV) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  is_smatv = TRUE;
               }
               else
               {
                  is_smatv = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_DISEQC_REPEATS) != 0)
            {
               repeats = (U8BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_UNICABLE_IF) != 0)
            {
               unicable_if = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_UNICABLE_CHAN) != 0)
            {
               unicable_chan = (U8BIT)strtol((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* See if an LNB exists with the settings that have been read */
   lnb_found = FALSE;

   lnb_ptr = DBDEF_GetNextLNBRec(NULL);
   while (!lnb_found && (lnb_ptr != NULL))
   {
      if (type == lnb_ptr->type)
      {
         switch (type)
         {
            case LNB_TYPE_SINGLE:
            case LNB_TYPE_UNIVERSAL:
            case LNB_TYPE_USER_DEFINED:
            {
               if ((((lnb_name == NULL) && (lnb_ptr->name == NULL)) ||
                  (((lnb_name != NULL) && (lnb_ptr->name != NULL)) &&
                  (memcmp(lnb_name, lnb_ptr->name->str_ptr, lnb_ptr->name->nbytes) == 0))) &&
                  (lnb_ptr->is_22k == is_22k) && (lnb_ptr->is_12v == is_12v) &&
                  (lnb_ptr->power == power) && (lnb_ptr->is_pulse_posn == pulse_pos) &&
                  (lnb_ptr->is_diseqc_posn == diseqc_pos))
               {
                  /* LNBs are the same */
                  lnb_found = TRUE;
               }
               break;
            }

            case LNB_TYPE_UNICABLE:
            {
               if ((((lnb_name == NULL) && (lnb_ptr->name == NULL)) ||
                  (((lnb_name != NULL) && (lnb_ptr->name != NULL)) &&
                  (memcmp(lnb_name, lnb_ptr->name->str_ptr, lnb_ptr->name->nbytes) == 0))) &&
                  (lnb_ptr->unicable_if == unicable_if) && (lnb_ptr->unicable_chan == unicable_chan))
               {
                  /* LNBs are the same */
                  lnb_found = TRUE;
               }
               break;
            }

            default:
            {
               lnb_found = FALSE;
               break;
            }
         }
      }

      if (!lnb_found)
      {
         lnb_ptr = DBDEF_GetNextLNBRec(lnb_ptr);
      }
   }

   if (lnb_ptr == NULL)
   {
      /* No LNB found, so create a new one */
      if (lnb_name != NULL)
      {
         adb_string = DBDEF_MakeString(0, lnb_name, STB_GetNumBytesInString(lnb_name));
      }
      else
      {
         adb_string = NULL;
      }
      if ((lnb_ptr = DBDEF_AddLNB(type, adb_string)) != NULL)
      {
         DBDEF_SetLNBPower(lnb_ptr, power);
         DBDEF_SetLNBDiSEqCTone(lnb_ptr, diseqc_tone);
         DBDEF_SetLNBCSwitch(lnb_ptr, cswitch);
         DBDEF_SetLNB22k(lnb_ptr, is_22k);
         DBDEF_SetLNB12V(lnb_ptr, is_12v);
         DBDEF_SetLNBPulsePosition(lnb_ptr, pulse_pos);
         DBDEF_SetLNBDiSEqCPosition(lnb_ptr, diseqc_pos);
         DBDEF_SetLNBSmatv(lnb_ptr, is_smatv);
         DBDEF_SetLNBRepeats(lnb_ptr, repeats);
         DBDEF_SetLNBUSwitch(lnb_ptr, uswitch);
         DBDEF_SetLNBUnicable(lnb_ptr, unicable_if, unicable_chan);
      }
      else if (adb_string != NULL)
      {
         DBDEF_ReleaseString(adb_string);
      }
   }

   FUNCTION_FINISH(ReadLNB);

   return(lnb_ptr);
}

static ADB_LNB_BAND_REC* ReadLNBBand(xmlTextReaderPtr reader, ADB_LNB_REC *lnb_ptr)
{
   xmlChar *name;
   xmlChar *value;
   ADB_LNB_BAND_REC *band_ptr;
   S_STB_DP_LNB_BAND params;

   FUNCTION_START(ReadLNBBand);

   /* Read satellite settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_LNB_POLARITY) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_HORIZONTAL) != 0)
               {
                  params.polarity = POLARITY_HORIZONTAL;
               }
               else if (xmlStrEqual(value, XMLTAG_VERTICAL) != 0)
               {
                  params.polarity = POLARITY_VERTICAL;
               }
               else if (xmlStrEqual(value, XMLTAG_LEFT) != 0)
               {
                  params.polarity = POLARITY_LEFT;
               }
               else if (xmlStrEqual(value, XMLTAG_RIGHT) != 0)
               {
                  params.polarity = POLARITY_RIGHT;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_BAND_MIN_FREQ) != 0)
            {
               params.min_freq = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_BAND_MAX_FREQ) != 0)
            {
               params.max_freq = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_BAND_LO_FREQ) != 0)
            {
               params.local_oscillator_frequency = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_BAND_VOLTAGE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_OFF) != 0)
               {
                  params.lnb_voltage = LNB_VOLTAGE_OFF;
               }
               else if (xmlStrEqual(value, XMLTAG_LNB_BAND_14V) != 0)
               {
                  params.lnb_voltage = LNB_VOLTAGE_14V;
               }
               else if (xmlStrEqual(value, XMLTAG_LNB_BAND_18V) != 0)
               {
                  params.lnb_voltage = LNB_VOLTAGE_18V;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_LNB_BAND_22KHZ) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_OFF) != 0)
               {
                  params.tone_22k = FALSE;
               }
               else if (xmlStrEqual(value, XMLTAG_ON) != 0)
               {
                  params.tone_22k = FALSE;
               }
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* See if a band exists with the settings that have been read */
   band_ptr = DBDEF_GetNextLNBBandRec(NULL);
   while (band_ptr != NULL)
   {
      if ((band_ptr->lnb == lnb_ptr) && (band_ptr->band_params.polarity == params.polarity) &&
         (band_ptr->band_params.min_freq == params.min_freq) &&
         (band_ptr->band_params.max_freq == params.max_freq) &&
         (band_ptr->band_params.local_oscillator_frequency == params.local_oscillator_frequency) &&
         (band_ptr->band_params.lnb_voltage == params.lnb_voltage) &&
         (band_ptr->band_params.tone_22k == params.tone_22k))
      {
         /* Found a band with the same settings */
         break;
      }

      band_ptr = DBDEF_GetNextLNBBandRec(band_ptr);
   }

   if (band_ptr == NULL)
   {
      /* Create a new band record */
      band_ptr = DBDEF_AddLNBBandRec(&params, lnb_ptr);
   }

   FUNCTION_FINISH(ReadLNBBand);

   return band_ptr;
}

static ADB_SATELLITE_REC* ReadSatellite(xmlTextReaderPtr reader, ADB_LNB_REC *lnb_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *sat_name;
   U16BIT dish_pos;
   U16BIT long_pos;
   BOOLEAN east_west;
   ADB_STRING *adb_string;
   ADB_SATELLITE_REC *sat_ptr;

   FUNCTION_START(ReadSatellite);

   sat_name = NULL;

   /* Read satellite settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  sat_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SAT_DISHPOS) != 0)
            {
               dish_pos = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SAT_LONGPOS) != 0)
            {
               long_pos = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SAT_EASTWEST) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  east_west = TRUE;
               }
               else
               {
                  east_west = FALSE;
               }
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* See if a satellite exists with the settings that have been read */
   sat_ptr = DBDEF_GetNextSatelliteRec(NULL);
   while (sat_ptr != NULL)
   {
      if ((sat_ptr->dish_pos == dish_pos) && (sat_ptr->long_pos == long_pos) &&
         (sat_ptr->east_west == east_west))
      {
         /* Found a satellite with the same settings */
         break;
      }

      sat_ptr = DBDEF_GetNextSatelliteRec(sat_ptr);
   }

   if (sat_ptr == NULL)
   {
      /* Create a new satellite record */
      if (sat_name != NULL)
      {
         adb_string = DBDEF_MakeString(0, sat_name, STB_GetNumBytesInString(sat_name));
      }
      else
      {
         adb_string = NULL;
      }

      sat_ptr = DBDEF_AddSatelliteRec(adb_string, dish_pos, long_pos, east_west, lnb_ptr);
      if (sat_ptr == NULL)
      {
         DBDEF_ReleaseString(adb_string);
      }
   }

   if (sat_name != NULL)
   {
      xmlFree(sat_name);
   }

   FUNCTION_FINISH(ReadSatellite);

   return(sat_ptr);
}

static ADB_NETWORK_REC* ReadNetwork(xmlTextReaderPtr reader, ADB_SATELLITE_REC *sat_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *net_name;
   U16BIT net_id;
   ADB_NETWORK_REC *n_ptr;
#ifdef COMMON_INTERFACE
   ADB_PROFILE_TYPE profile_type;
   U8BIT *profile_name;
   U16BIT cam_onet_id;
   U32BIT cam_id;
#endif

   FUNCTION_START(ReadNetwork);

   n_ptr = NULL;
   net_id = 0;
   net_name = NULL;

#ifdef COMMON_INTERFACE
   profile_type = ADB_PROFILE_TYPE_BROADCAST;
   profile_name = NULL;
   cam_onet_id = 0;
   cam_id = 0;
#endif

   /* Read network settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  net_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_NET_ID) != 0)
            {
               net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
#ifdef COMMON_INTERFACE
            else if (xmlStrEqual(name, XMLTAG_PROFILETYPE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_PROFILE_BROADCAST) != 0)
               {
                  profile_type = ADB_PROFILE_TYPE_BROADCAST;
               }
               else if (xmlStrEqual(value, XMLTAG_PROFILE_CIPLUS) != 0)
               {
                  profile_type = ADB_PROFILE_TYPE_CIPLUS;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_PROFILE_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  profile_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_CAM_ONET) != 0)
            {
               cam_onet_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_CAM_ID) != 0)
            {
               cam_id = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
#endif
            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* 0 is a reserved network ID so shouldn't be used */
   if (net_id != 0)
   {
      /* See if this network exists */
      n_ptr = DBDEF_FindNetworkRec(net_id, sat_ptr);
      if (n_ptr == NULL)
      {
         /* Create a new network record */
         n_ptr = DBDEF_AddNetworkRec(net_id, sat_ptr);
         if (n_ptr != NULL)
         {
            DBDEF_SetNetworkName(n_ptr, net_name);

#ifdef COMMON_INTERFACE
            /* Set profile type and name */
            n_ptr->profile_type = profile_type;
            DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_PROFILE_TYPE, n_ptr->profile_type);

            n_ptr->cicam_onet_id = cam_onet_id;
            n_ptr->cicam_identifier = cam_id;
#endif
         }
      }
   }

   if (net_name != NULL)
   {
      xmlFree(net_name);
   }

#ifdef COMMON_INTERFACE
   if (profile_name != NULL)
   {
      xmlFree(profile_name);
   }
#endif

   FUNCTION_FINISH(ReadNetwork);

   return(n_ptr);
}

static ADB_TRANSPORT_REC* ReadSatTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U32BIT frequency;
   U16BIT sym_rate;
   E_STB_DP_POLARITY polarity;
   BOOLEAN dvb_s2;
   E_STB_DP_MODULATION modulation;
   U16BIT net_id;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ReadSatTransport);

   orig_net_id = ADB_INVALID_DVB_ID;
   tran_id = ADB_INVALID_DVB_ID;
   frequency = 0;
   sym_rate = 0;
   polarity = POLARITY_HORIZONTAL;
   dvb_s2 = FALSE;
   modulation = MOD_AUTO;

   /* Read transport settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_ORIG_NET_ID) != 0)
            {
               orig_net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_FREQUENCY) != 0)
            {
               frequency = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_SYMRATE) != 0)
            {
               sym_rate = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_POLARITY) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_HORIZONTAL) != 0)
               {
                  polarity = POLARITY_HORIZONTAL;
               }
               else if (xmlStrEqual(value, XMLTAG_VERTICAL) != 0)
               {
                  polarity = POLARITY_VERTICAL;
               }
               else if (xmlStrEqual(value, XMLTAG_LEFT) != 0)
               {
                  polarity = POLARITY_LEFT;
               }
               else if (xmlStrEqual(value, XMLTAG_RIGHT) != 0)
               {
                  polarity = POLARITY_RIGHT;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_MODULATION) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_AUTO) != 0)
               {
                  modulation = MOD_AUTO;
               }
               else if (xmlStrEqual(value, XMLTAG_QPSK) != 0)
               {
                  modulation = MOD_QPSK;
               }
               else if (xmlStrEqual(value, XMLTAG_8PSK) != 0)
               {
                  modulation = MOD_8PSK;
               }
               else if (xmlStrEqual(value, XMLTAG_16QAM) != 0)
               {
                  modulation = MOD_16QAM;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_DVBS2) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  dvb_s2 = TRUE;
               }
               else
               {
                  dvb_s2 = FALSE;
               }
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if (n_ptr != NULL)
   {
      net_id = n_ptr->net_id;
   }
   else
   {
      net_id = ADB_INVALID_DVB_ID;
   }

   t_ptr = DBDEF_FindTransportRecByIds(NULL, net_id, orig_net_id, tran_id);
   while (t_ptr != NULL)
   {
      if ((n_ptr == NULL) || (t_ptr->network == n_ptr))
      {
         /* No network, or transport is on the same network so this transport already exists */
         break;
      }

      /* Try the next transport */
      t_ptr = DBDEF_FindTransportRecByIds(t_ptr, net_id, orig_net_id, tran_id);
   }

   if (t_ptr == NULL)
   {
      /* Need to add a new transport */
      t_ptr = DBDEF_AddSatTransportRec(frequency, sym_rate, polarity, dvb_s2, modulation, n_ptr);
      if (t_ptr != NULL)
      {
         DBDEF_SetTransportTransportId(t_ptr, tran_id);
         DBDEF_SetTransportOrigNetworkId(t_ptr, orig_net_id);
      }
   }

   FUNCTION_FINISH(ReadSatTransport);

   return(t_ptr);
}

static ADB_TRANSPORT_REC* ReadCableTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U32BIT frequency;
   E_STB_DP_CMODE cmode;
   U16BIT sym_rate;
   U16BIT net_id;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ReadCableTransport);

   orig_net_id = ADB_INVALID_DVB_ID;
   tran_id = ADB_INVALID_DVB_ID;
   frequency = 0;
   sym_rate = 0;
   cmode = MODE_QAM_AUTO;

   /* Read transport settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_ORIG_NET_ID) != 0)
            {
               orig_net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_FREQUENCY) != 0)
            {
               frequency = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_SYMRATE) != 0)
            {
               sym_rate = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_MODE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_AUTO) != 0)
               {
                  cmode = MODE_QAM_AUTO;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM4) != 0)
               {
                  cmode = MODE_QAM_4;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM8) != 0)
               {
                  cmode = MODE_QAM_8;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM16) != 0)
               {
                  cmode = MODE_QAM_16;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM32) != 0)
               {
                  cmode = MODE_QAM_32;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM64) != 0)
               {
                  cmode = MODE_QAM_64;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM128) != 0)
               {
                  cmode = MODE_QAM_128;
               }
               else if (xmlStrEqual(value, XMLTAG_QAM256) != 0)
               {
                  cmode = MODE_QAM_256;
               }
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if (n_ptr != NULL)
   {
      net_id = n_ptr->net_id;
   }
   else
   {
      net_id = ADB_INVALID_DVB_ID;
   }

   t_ptr = DBDEF_FindTransportRecByIds(NULL, net_id, orig_net_id, tran_id);
   while (t_ptr != NULL)
   {
      if ((n_ptr == NULL) || (t_ptr->network == n_ptr))
      {
         /* No network, or transport is on the same network so this transport already exists */
         break;
      }

      /* Try the next transport */
      t_ptr = DBDEF_FindTransportRecByIds(t_ptr, net_id, orig_net_id, tran_id);
   }

   if (t_ptr == NULL)
   {
      /* Need to add a new transport */
      t_ptr = DBDEF_AddCableTransportRec(frequency, sym_rate, n_ptr);
      if (t_ptr != NULL)
      {
         DBDEF_SetCableTransportMode(t_ptr, cmode);
         DBDEF_SetTransportTransportId(t_ptr, tran_id);
         DBDEF_SetTransportOrigNetworkId(t_ptr, orig_net_id);
      }
   }

   FUNCTION_FINISH(ReadCableTransport);

   return(t_ptr);
}

static ADB_TRANSPORT_REC* ReadTerrTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr)
{
   xmlChar *name;
   xmlChar *value;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TMODE tmode;
   E_STB_DP_TTYPE terr_type;
   U8BIT plp_id;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U32BIT frequency;
   U16BIT net_id;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ReadTerrTransport);

   bwidth = TBWIDTH_UNDEFINED;
   tmode = MODE_COFDM_UNDEFINED;
   terr_type = TERR_TYPE_UNKNOWN;
   plp_id = 0;
   orig_net_id = ADB_INVALID_DVB_ID;
   tran_id = ADB_INVALID_DVB_ID;
   frequency = 0;

   /* Read transport settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_TRAN_BWIDTH) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_5) != 0)
               {
                  bwidth = TBWIDTH_5MHZ;
               }
               else if (xmlStrEqual(value, XMLTAG_6) != 0)
               {
                  bwidth = TBWIDTH_6MHZ;
               }
               else if (xmlStrEqual(value, XMLTAG_7) != 0)
               {
                  bwidth = TBWIDTH_7MHZ;
               }
               else if (xmlStrEqual(value, XMLTAG_8) != 0)
               {
                  bwidth = TBWIDTH_8MHZ;
               }
               else if (xmlStrEqual(value, XMLTAG_10) != 0)
               {
                  bwidth = TBWIDTH_10MHZ;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_MODE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  tmode = MODE_COFDM_1K;
               }
               else if (xmlStrEqual(value, XMLTAG_2) != 0)
               {
                  tmode = MODE_COFDM_2K;
               }
               else if (xmlStrEqual(value, XMLTAG_4) != 0)
               {
                  tmode = MODE_COFDM_4K;
               }
               else if (xmlStrEqual(value, XMLTAG_8) != 0)
               {
                  tmode = MODE_COFDM_8K;
               }
               else if (xmlStrEqual(value, XMLTAG_16) != 0)
               {
                  tmode = MODE_COFDM_16K;
               }
               else if (xmlStrEqual(value, XMLTAG_32) != 0)
               {
                  tmode = MODE_COFDM_32K;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_TERRTYPE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_DVBT) != 0)
               {
                  terr_type = TERR_TYPE_DVBT;
               }
               else if (xmlStrEqual(value, XMLTAG_DVBT2) != 0)
               {
                  terr_type = TERR_TYPE_DVBT2;
               }
            }
            if (xmlStrEqual(name, XMLTAG_TRAN_PLPID) != 0)
            {
               plp_id = (U8BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_ORIG_NET_ID) != 0)
            {
               orig_net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_FREQUENCY) != 0)
            {
               frequency = (U32BIT)strtoll((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if (n_ptr != NULL)
   {
      net_id = n_ptr->net_id;
   }
   else
   {
      net_id = ADB_INVALID_DVB_ID;
   }

   t_ptr = DBDEF_FindTransportRecByIds(NULL, net_id, orig_net_id, tran_id);
   while (t_ptr != NULL)
   {
      if ((n_ptr == NULL) || (t_ptr->network == n_ptr))
      {
         /* No network, or transport is on the same network so this transport already exists */
         break;
      }

      /* Try the next transport */
      t_ptr = DBDEF_FindTransportRecByIds(t_ptr, net_id, orig_net_id, tran_id);
   }

   if (t_ptr == NULL)
   {
      /* Need to add a new transport */
      t_ptr = DBDEF_AddTerrestrialTransportRec(frequency, plp_id, n_ptr);
      if (t_ptr != NULL)
      {
         DBDEF_SetTransportTransportId(t_ptr, tran_id);
         DBDEF_SetTransportOrigNetworkId(t_ptr, orig_net_id);

         t_ptr->u.terr.tmode = tmode;
         DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_MODE, tmode);

         t_ptr->u.terr.bwidth = bwidth;
         DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_BWIDTH, bwidth);

         t_ptr->u.terr.terr_type = terr_type;
         DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_TERR_TYPE, terr_type);

         DBA_SaveRecord(t_ptr->dba_rec);
      }
   }

   FUNCTION_FINISH(ReadTerrTransport);

   return(t_ptr);
}

static ADB_TRANSPORT_REC* ReadAnalogTransport(xmlTextReaderPtr reader, ADB_NETWORK_REC *n_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT tran_id;
   U32BIT frequency;
   U16BIT net_id;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ReadAnalogTransport);

   tran_id = 0;
   frequency = 0;

   /* Read transport settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_TRAN_FREQUENCY) != 0)
            {
               frequency = (U32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if (n_ptr != NULL)
   {
      net_id = n_ptr->net_id;
   }
   else
   {
      net_id = ADB_INVALID_DVB_ID;
   }

   t_ptr = DBDEF_FindTransportRecByIds(NULL, net_id, ADB_INVALID_DVB_ID, tran_id);
   while (t_ptr != NULL)
   {
      if ((n_ptr == NULL) || (t_ptr->network == n_ptr))
      {
         /* No network, or transport is on the same network so this transport already exists */
         break;
      }

      /* Try the next transport */
      t_ptr = DBDEF_FindTransportRecByIds(t_ptr, net_id, ADB_INVALID_DVB_ID, tran_id);
   }

   if (t_ptr == NULL)
   {
      /* Need to add a new transport */
      t_ptr = DBDEF_AddTerrestrialTransportRec(frequency, 0, n_ptr);
      if (t_ptr != NULL)
      {
         DBDEF_SetTransportTransportId(t_ptr, tran_id);
      }
   }

   FUNCTION_FINISH(ReadAnalogTransport);

   return(t_ptr);
}

static ADB_SERVICE_REC* ReadService(xmlTextReaderPtr reader, ADB_TRANSPORT_REC *t_ptr)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *serv_name;
   ADB_SERVICE_TYPE serv_type;
   U16BIT serv_id;
   U16BIT serv_lcn;
   U16BIT alloc_lcn;
   U8BIT favgroups;
   BOOLEAN hidden;
   BOOLEAN selectable;
   BOOLEAN locked;
   BOOLEAN sched_disabled;
   BOOLEAN pf_disabled;
   BOOLEAN edit_lcn;
   U16BIT freesat_id;
   U16BIT region_id;
   ADB_SERVICE_REC *s_ptr;

   FUNCTION_START(ReadService);

   serv_name = NULL;
   serv_type = ADB_SERVICE_TYPE_DATA;
   serv_id = 0;
   serv_lcn = 0;
   alloc_lcn = 0;
   favgroups = 0;
   hidden = FALSE;
   selectable = FALSE;
   locked = FALSE;
   sched_disabled = FALSE;
   pf_disabled = FALSE;
   edit_lcn = FALSE;
   freesat_id = INVALID_FREESAT_SERV_ID;
   region_id = 0;

   /* Read service settings */
   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  serv_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TYPE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_TV) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_TV;
               }
               else if (xmlStrEqual(value, XMLTAG_RADIO) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_RADIO;
               }
               else if (xmlStrEqual(value, XMLTAG_AVC_RADIO) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_AVC_RADIO;
               }
               else if (xmlStrEqual(value, XMLTAG_DATA) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_DATA;
               }
               else if (xmlStrEqual(value, XMLTAG_MPEG2_HD) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_MPEG2_HD;
               }
               else if (xmlStrEqual(value, XMLTAG_AVC_SD_TV) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_AVC_SD_TV;
               }
               else if (xmlStrEqual(value, XMLTAG_HD_TV) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_HD_TV;
               }
               else if (xmlStrEqual(value, XMLTAG_UHD_TV) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_UHD_TV;
               }
               else if (xmlStrEqual(value, XMLTAG_ANALOG) != 0)
               {
                  serv_type = ADB_SERVICE_TYPE_ANALOG;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ID) != 0)
            {
               serv_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_LCN) != 0)
            {
               serv_lcn = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ALLOCLCN) != 0)
            {
               alloc_lcn = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_FAVGROUPS) != 0)
            {
               favgroups = (U8BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_HIDDEN) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  hidden = TRUE;
               }
               else
               {
                  hidden = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_SELECTABLE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  selectable = TRUE;
               }
               else
               {
                  selectable = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_LOCKED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  locked = TRUE;
               }
               else
               {
                  locked = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_SCHED_DISABLED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  sched_disabled = TRUE;
               }
               else
               {
                  sched_disabled = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_PF_DISABLED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  pf_disabled = TRUE;
               }
               else
               {
                  pf_disabled = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_EDIT_LCN) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  edit_lcn = TRUE;
               }
               else
               {
                  edit_lcn = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_FREESAT_ID) != 0)
            {
               freesat_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_REGION_ID) != 0)
            {
               region_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* See if this service already exists */
   if (freesat_id != INVALID_FREESAT_SERV_ID)
   {
      /* There can be multiple services with the same Freesat ID, so check them all */
      s_ptr = DBDEF_FindServiceRecByFreesatId(NULL, freesat_id);
      while (s_ptr != NULL)
      {
         if (s_ptr->region_id == region_id)
         {
            /* This is the same service */
            break;
         }

         s_ptr = DBDEF_FindServiceRecByFreesatId(s_ptr, freesat_id);
      }
   }
   else if (t_ptr != NULL)
   {
      s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
         t_ptr->tran_id, serv_id);
      while (s_ptr != NULL)
      {
         if ((s_ptr->serv_lcn == serv_lcn) && (s_ptr->allocated_lcn == alloc_lcn))
         {
            /* This is the same service */
            break;
         }

         s_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
            t_ptr->tran_id, serv_id);
      }
   }
   else
   {
      s_ptr = DBDEF_FindServiceRecByLcn(serv_lcn, t_ptr, alloc_lcn);
   }

   if (s_ptr == NULL)
   {
      /* Need to create a new service */
      s_ptr = DBDEF_AddServiceRec(serv_id, t_ptr);
      if (s_ptr != NULL)
      {
         s_ptr->found = TRUE;

         if (serv_name != NULL)
         {
            DBDEF_SetServiceName(s_ptr, serv_name);
         }

         DBDEF_SetServiceType(s_ptr, serv_type);

         s_ptr->serv_lcn = serv_lcn;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);

         s_ptr->allocated_lcn = alloc_lcn;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN, s_ptr->allocated_lcn);

         s_ptr->fav_groups = favgroups;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_FAV_GROUPS, s_ptr->fav_groups);

         s_ptr->hidden = hidden;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

         s_ptr->selectable = selectable;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

         s_ptr->locked = locked;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LOCKED, s_ptr->locked);

         s_ptr->sched_disabled = sched_disabled;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SCHED_DISABLED, s_ptr->sched_disabled);

         s_ptr->now_next_disabled = pf_disabled;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_NOWNEXT_DISABLED, s_ptr->now_next_disabled);

         s_ptr->lcn_editable = edit_lcn;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN_EDITABLE, s_ptr->lcn_editable);

         s_ptr->freesat_id = freesat_id;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_FREESAT_ID, s_ptr->freesat_id);

         s_ptr->region_id = region_id;
         DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REGION_ID, s_ptr->region_id);

         DBA_SaveRecord(s_ptr->dba_rec);
      }
   }

   if (serv_name != NULL)
   {
      xmlFree(serv_name);
   }

   FUNCTION_FINISH(ReadService);

   return(s_ptr);
}

static ADB_FAVLIST_REC* ReadFavList(xmlTextReaderPtr reader)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *list_name;
   U8BIT list_id;
   S16BIT index;
   U32BIT user_data;
   ADB_FAVLIST_REC *fav_list;

   FUNCTION_START(ReadFavList);

   list_name = NULL;
   list_id = 0;
   index = -1;
   user_data = 0;

   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  list_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_ID) != 0)
            {
               list_id = (U8BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_INDEX) != 0)
            {
               index = (S16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_USERDATA) != 0)
            {
               user_data = (U32BIT)strtoll((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if (list_id != 0)
   {
      /* Check whether a list already exists with this id */
      fav_list = DBDEF_FindFavouriteList(list_id);
      if (fav_list != NULL)
      {
         if (fav_list->user_data != user_data)
         {
            /* The list being imported wants an ID that's already in use, so it will have to
             * be given a new one */
            fav_list = DBDEF_AddFavouriteList(0, list_name, user_data, index);
         }
      }
      else
      {
         /* Create a new favourite list with the given id */
         fav_list = DBDEF_AddFavouriteList(list_id, list_name, user_data, index);
      }
   }
   else
   {
      fav_list = NULL;
   }

   if (list_name != NULL)
   {
      xmlFree(list_name);
   }

   FUNCTION_FINISH(ReadFavList);

   return(fav_list);
}

static ADB_FAVSERV_REC* ReadFavServ(xmlTextReaderPtr reader, ADB_FAVLIST_REC *fav_list)
{
   xmlChar *name;
   xmlChar *value;
   S16BIT index;
   U16BIT lcn;
   U16BIT freesat_id;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U16BIT serv_id;
   ADB_FAVSERV_REC *fav_serv;
   void *s_ptr;

   FUNCTION_START(ReadFavServ);

   index = -1;
   lcn = 0;
   freesat_id = INVALID_FREESAT_SERV_ID;
   orig_net_id = ADB_INVALID_DVB_ID;
   tran_id = ADB_INVALID_DVB_ID;
   serv_id = ADB_INVALID_DVB_ID;

   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_INDEX) != 0)
            {
               index = (S16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ALLOCLCN) != 0)
            {
               lcn = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_FREESAT_ID) != 0)
            {
               freesat_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_ORIG_NET_ID) != 0)
            {
               orig_net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ID) != 0)
            {
               serv_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   /* Find the service */
   if (lcn != 0)
   {
      s_ptr = DBDEF_FindServiceRecByLcn(lcn, NULL, TRUE);
   }
   else
   {
      if (freesat_id != INVALID_FREESAT_SERV_ID)
      {
         s_ptr = DBDEF_FindServiceRecByFreesatId(NULL, freesat_id);
      }
      else
      {
         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, orig_net_id, tran_id, serv_id);
      }
   }

   if (s_ptr != NULL)
   {
      /* Check whether this service is already included in the favourite list */
      fav_serv = DBDEF_FindServiceInFavouriteList(fav_list, s_ptr);
      if (fav_serv == NULL)
      {
         fav_serv = DBDEF_AddServiceToFavouriteList(fav_list, s_ptr, index);
      }
   }
   else
   {
      fav_serv = NULL;
   }

   FUNCTION_FINISH(ReadFavServ);

   return(fav_serv);
}

static ADB_CRID_REC* ReadCridRec(xmlTextReaderPtr reader)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *prog_name;
   U8BIT *crid_str;
   BOOLEAN series;
   BOOLEAN recommended;
   U16BIT eit_date;
   U32DHMS date_time;
   U16BIT serv_id;
   ADB_CRID_REC *crid_rec;

   FUNCTION_START(ReadCridRec);

   prog_name = NULL;
   crid_str = NULL;
   eit_date = 0;
   series = FALSE;
   recommended = FALSE;
   date_time = 0;
   serv_id = ADB_INVALID_DVB_ID;

   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  prog_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_PROG_CRID) != 0)
            {
               /* Copy the crid string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  crid_str = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_EIT_DATE) != 0)
            {
               eit_date = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERIES) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  series = TRUE;
               }
               else
               {
                  series = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_RECOMMENDED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  recommended = TRUE;
               }
               else
               {
                  recommended = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_DATETIME) != 0)
            {
               date_time = (U32DHMS)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ID) != 0)
            {
               serv_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   if ((crid_str != NULL) && (serv_id != ADB_INVALID_DVB_ID))
   {
      crid_rec = DBDEF_AddCridRecord(crid_str, series, recommended);
      if (crid_rec != NULL)
      {
         DBDEF_SetCridDateTime(crid_rec, date_time);
         DBDEF_SetCridService(crid_rec, serv_id);

         if (prog_name != NULL)
         {
            DBDEF_SetCridProgrammeName(crid_rec, prog_name);
         }

         if (eit_date != 0)
         {
            crid_rec->eit_date = eit_date;
            DBA_SetFieldValue(crid_rec->dba_rec, DBA_FIELD_CRID_EIT_DATE, crid_rec->eit_date);
         }

         DBA_SaveRecord(crid_rec->dba_rec);
      }
   }
   else
   {
      crid_rec = NULL;
   }

   if (prog_name != NULL)
   {
      xmlFree(prog_name);
   }

   if (crid_str != NULL)
   {
      xmlFree(crid_str);
   }

   FUNCTION_FINISH(ReadCridRec);

   return(crid_rec);
}

static U32BIT ReadTimerRec(xmlTextReaderPtr reader)
{
   xmlChar *name;
   xmlChar *value;
   U16BIT name_len;
   U8BIT *timer_name;
   E_TIMER_TYPE timer_type;
   U32DHMS start_time;
   E_TIMER_FREQ timer_freq;
   BOOLEAN change_service;
   U16BIT orig_net_id;
   U16BIT tran_id;
   U16BIT serv_id;
   BOOLEAN ramp_volume;
   U32DHMS duration;
   BOOLEAN event_triggered;
   U16BIT event_id;
   U16BIT disk_id;
   BOOLEAN recommendation;
   U16BIT notify;
   S32BIT start_pad;
   S32BIT end_pad;
   U8BIT *prog_crid;
   U8BIT *other_crid;
   U8BIT *add_info;
   U32BIT timer_handle;
   ADB_TIMER_REC *timer;
   S_TIMER_INFO timer_info;

   FUNCTION_START(ReadTimerRec);

   timer_name = NULL;
   timer_type = TIMER_TYPE_NONE;
   start_time = 0;
   timer_freq = TIMER_FREQ_ONCE;
   change_service = FALSE;
   orig_net_id = ADB_INVALID_DVB_ID;
   tran_id = ADB_INVALID_DVB_ID;
   serv_id = ADB_INVALID_DVB_ID;
   ramp_volume = FALSE;
   duration = 0;
   event_triggered = FALSE;
   event_id = 0;
   disk_id = 0;
   recommendation = FALSE;
   notify = 0;
   start_pad = 0;
   end_pad = 0;
   prog_crid = NULL;
   other_crid = NULL;
   add_info = NULL;

   while (xmlTextReaderMoveToNextAttribute(reader) == 1)
   {
      name = xmlTextReaderName(reader);
      if (name != NULL)
      {
         value = xmlTextReaderValue(reader);
         if (value != NULL)
         {
            if (xmlStrEqual(name, XMLTAG_NAME) != 0)
            {
               /* Copy the name string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  timer_name = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_TYPE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_TIMER_ALARM) != 0)
               {
                  timer_type = TIMER_TYPE_ALARM;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_SLEEP) != 0)
               {
                  timer_type = TIMER_TYPE_ALARM;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_PVR_RECORD) != 0)
               {
                  timer_type = TIMER_TYPE_PVR_RECORD;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_PRIVATE) != 0)
               {
                  timer_type = TIMER_TYPE_PRIVATE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_DATETIME) != 0)
            {
               start_time = (U32DHMS)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_FREQ) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_TIMER_ONCE) != 0)
               {
                  timer_freq = TIMER_FREQ_ONCE;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_WEEKLY) != 0)
               {
                  timer_freq = TIMER_FREQ_WEEKLY;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_WEEKENDDAYS) != 0)
               {
                  timer_freq = TIMER_FREQ_WEEKENDDAYS;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_WEEKDAYS) != 0)
               {
                  timer_freq = TIMER_FREQ_WEEKDAYS;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_DAILY) != 0)
               {
                  timer_freq = TIMER_FREQ_DAILY;
               }
               else if (xmlStrEqual(value, XMLTAG_TIMER_HOURLY) != 0)
               {
                  timer_freq = TIMER_FREQ_HOURLY;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_CHANGESERVICE) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  change_service = TRUE;
               }
               else
               {
                  change_service = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_ORIG_NET_ID) != 0)
            {
               orig_net_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRAN_ID) != 0)
            {
               tran_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_SERV_ID) != 0)
            {
               serv_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_RAMPVOLUME) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  ramp_volume = TRUE;
               }
               else
               {
                  ramp_volume = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_DURATION) != 0)
            {
               duration = (U32DHMS)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_TRIGGERED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  event_triggered = TRUE;
               }
               else
               {
                  event_triggered = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_EVENT_ID) != 0)
            {
               event_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_DISK_ID) != 0)
            {
               disk_id = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_RECOMMENDED) != 0)
            {
               if (xmlStrEqual(value, XMLTAG_1) != 0)
               {
                  recommendation = TRUE;
               }
               else
               {
                  recommendation = FALSE;
               }
            }
            else if (xmlStrEqual(name, XMLTAG_NOTIFY) != 0)
            {
               notify = (U16BIT)strtol((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_START_PAD) != 0)
            {
               start_pad = (S32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_END_PAD) != 0)
            {
               end_pad = (S32BIT)strtoll((const char *)value, NULL, 0);
            }
            else if (xmlStrEqual(name, XMLTAG_PROG_CRID) != 0)
            {
               /* Copy the CRID string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  prog_crid = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_OTHER_CRID) != 0)
            {
               /* Copy the CRID string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  other_crid = xmlUTF8Strndup(value, name_len);
               }
            }
            else if (xmlStrEqual(name, XMLTAG_ADDITIONAL_INFO) != 0)
            {
               /* Copy the string */
               name_len = xmlUTF8Strlen(value);
               if (name_len > 0)
               {
                  add_info = xmlUTF8Strndup(value, name_len);
               }
            }

            xmlFree(value);
         }

         xmlFree(name);
      }
   }

   timer_handle = INVALID_TIMER_HANDLE;

   if (timer_type != TIMER_TYPE_NONE)
   {
      /* Check to see whether this timer already exists */
      if (timer_type == TIMER_TYPE_PVR_RECORD)
      {
         if (prog_crid != NULL)
         {
            timer_handle = ATMR_FindTimerFromCrid(prog_crid);
         }
         else if ((serv_id != ADB_INVALID_DVB_ID) && (tran_id != ADB_INVALID_DVB_ID))
         {
            if (event_triggered)
            {
               timer_handle = ATMR_FindTimerFromEvent(orig_net_id, tran_id, serv_id, event_id);
            }
         }
      }

      if (timer_handle == INVALID_TIMER_HANDLE)
      {
         /* Look for a timer with the same type, start time and frequency */
         timer = DBDEF_GetNextTimerRec(NULL);
         while (timer != NULL)
         {
            if ((timer->type == timer_type) && (timer->start_time == start_time) &&
               (timer->frequency == timer_freq))
            {
               if (timer_type == TIMER_TYPE_ALARM)
               {
                  if ((timer->u.alarm.service_id == serv_id) &&
                     (timer->u.alarm.transport_id == tran_id) &&
                     (timer->u.alarm.orig_net_id == orig_net_id))
                  {
                     /* Alarm timer found */
                     timer_handle = timer->handle;
                     break;
                  }
               }
               else if (timer_type == TIMER_TYPE_PVR_RECORD)
               {
                  if ((timer->u.record.duration == duration) &&
                     (timer->u.record.orig_net_id == orig_net_id) &&
                     (timer->u.record.transport_id == tran_id) &&
                     (timer->u.record.service_id == serv_id))
                  {
                     /* PVR timer found */
                     timer_handle = timer->handle;
                     break;
                  }
               }
               else
               {
                  /* Sleep or private timer found */
                  timer_handle = timer->handle;
                  break;
               }
            }
            timer = DBDEF_GetNextTimerRec(timer);
         }
      }

      if (timer_handle == INVALID_TIMER_HANDLE)
      {
         /* Timer needs to be created */
         memset(&timer_info, 0, sizeof(timer_info));

         timer_info.type = timer_type;

         if (timer_name != NULL)
         {
            memcpy(timer_info.name, timer_name, STB_GetNumBytesInString(timer_name));
         }

         timer_info.frequency = timer_freq;
         timer_info.start_time = start_time;

         if (timer_type == TIMER_TYPE_ALARM)
         {
            timer_info.u.alarm.change_service = change_service;
            timer_info.u.alarm.service_id = serv_id;
            timer_info.u.alarm.transport_id = tran_id;
            timer_info.u.alarm.orig_net_id = orig_net_id;
            timer_info.u.alarm.ramp_volume = ramp_volume;
         }
         else if (timer_type == TIMER_TYPE_PVR_RECORD)
         {
            timer_info.u.record.duration = duration;
            timer_info.u.record.event_triggered = event_triggered;
            timer_info.u.record.event_id = event_id;
            timer_info.u.record.service_id = serv_id;
            timer_info.u.record.transport_id = tran_id;
            timer_info.u.record.orig_net_id = orig_net_id;
            timer_info.u.record.disk_id = disk_id;
            timer_info.u.record.recommendation = recommendation;
            timer_info.u.record.notify_time = notify;

            if (prog_crid != NULL)
            {
               memcpy(timer_info.u.record.prog_crid, prog_crid, STB_GetNumBytesInString(prog_crid));
            }
            if (other_crid != NULL)
            {
               memcpy(timer_info.u.record.other_crid, other_crid, STB_GetNumBytesInString(other_crid));
            }
         }

         timer_handle = ATMR_AddTimer(&timer_info);
         if (timer_handle != INVALID_TIMER_HANDLE)
         {
            if (add_info != NULL)
            {
               ATMR_SetAdditionalInfo(timer_handle, add_info, STB_GetNumBytesInString(add_info));
            }
            if (start_pad != 0)
            {
               ATMR_SetStartPadding(timer_handle, start_pad);
            }
            if (end_pad != 0)
            {
               ATMR_SetEndPadding(timer_handle, end_pad);
            }
         }
      }
   }

   if (timer_name != NULL)
   {
      xmlFree(timer_name);
   }
   if (prog_crid != NULL)
   {
      xmlFree(prog_crid);
   }
   if (other_crid != NULL)
   {
      xmlFree(other_crid);
   }
   if (add_info != NULL)
   {
      xmlFree(add_info);
   }

   FUNCTION_FINISH(ReadTimerRec);

   return(timer_handle);
}

/**
 * @brief   Exports the database as an XML file
 * @param   filename pathname of the XML file to export the database to
 * @return  TRUE if successful, otherwise FALSE
 */
BOOLEAN ADB_ExportDB(U8BIT *filename)
{
   BOOLEAN success = FALSE;
   xmlTextWriterPtr writer;
   ADB_LNB_REC *lnb_ptr;
   ADB_NETWORK_REC *n_ptr;
   ADB_FAVLIST_REC *fav_list;
   ADB_CRID_REC *crid_rec;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ADB_ExportDB);

   /* This macro checks the libxml2 version used when building and version used at runtime
    * are compatible */
   LIBXML_TEST_VERSION

   writer = xmlNewTextWriterFilename((const char *)filename, 0);
   if (writer != NULL)
   {
      /* Set indentation to make output readable. This won't affect the format because
       * record settings are stored as attributes */
      xmlTextWriterSetIndent(writer, 1);

      if (xmlTextWriterStartDocument(writer, NULL, XMLENCODING, NULL) >= 0)
      {
         if (xmlTextWriterStartElement(writer, XMLTAG_SERVICE_DB) >= 0)
         {
            DBDEF_RequestAccess();

            /* Export LNB records */
            lnb_ptr = DBDEF_GetNextLNBRec(NULL);
            while (lnb_ptr != NULL)
            {
               ExportLNBRec(writer, lnb_ptr);
               lnb_ptr = DBDEF_GetNextLNBRec(lnb_ptr);
            }

            /* Export any network records that aren't related to a satellite */
            n_ptr = DBDEF_GetNextNetworkRec(NULL);
            while (n_ptr != NULL)
            {
               if (n_ptr->satellite == NULL)
               {
                  ExportNetworkRec(writer, n_ptr);
               }

               n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
            }

            /* Export all favourite lists */
            fav_list = DBDEF_GetNextFavouriteList(NULL);
            while (fav_list != NULL)
            {
               ExportFavList(writer, fav_list);
               fav_list = DBDEF_GetNextFavouriteList(fav_list);
            }

            /* Export all CRIDs */
            crid_rec = DBDEF_GetNextCridRecord(NULL);
            while (crid_rec != NULL)
            {
               ExportCridRec(writer, crid_rec);
               crid_rec = DBDEF_GetNextCridRecord(crid_rec);
            }

            /* Export all timers */
            timer = DBDEF_GetNextTimerRec(NULL);
            while (timer != NULL)
            {
               ExportTimerRec(writer, timer);
               timer = DBDEF_GetNextTimerRec(timer);
            }

            DBDEF_ReleaseAccess();

            xmlTextWriterEndElement(writer);
         }

         xmlTextWriterEndDocument(writer);
      }

      xmlFreeTextWriter(writer);

      success = TRUE;
   }

   FUNCTION_FINISH(ADB_ExportDB);

   return success;
}

#if 0
/**
 * @brief   Exports the current event schedule as an XML file
 * @param   filename The path and filename to export to
 * @param   utc_date_filter If non-zero, only events matching date will be exported
 * @param   type The service type(s) to export
 * @param   use_dvb_uri TRUE to use dvb:// uri for id, otherwise allocated lcn
 * @return  TRUE if successful, otherwise FALSE
 */
BOOLEAN ADB_ExportEPG( U8BIT *filename, U16BIT utc_date_filter, ADB_SERVICE_LIST_TYPE type, BOOLEAN use_dvb_uri)
{
   BOOLEAN success = FALSE;
   U16BIT i;

   ADB_SERVICE_REC *s_ptr = NULL;
   U8BIT *serv_name = NULL;
   ADB_EVENT_REC *e_ptr = NULL;
   ADB_STREAM_REC *stream_ptr = NULL;

   U16BIT start_year;
   U8BIT start_month;
   U8BIT start_day;
   U16BIT end_date;
   U16BIT end_year;
   U8BIT end_month;
   U8BIT end_day;
   U8BIT end_hour;
   U8BIT end_minute;
   U8BIT end_secs;
   U8BIT wday_notused;
   U8BIT lcode_string[4];
   BOOLEAN audio_present = FALSE;
   BOOLEAN audio_ac3 = FALSE;
   U8BIT chan_id[256];
   U16BIT onid = 0;
   U16BIT tid = 0;

   FILE *fd;

   U8BIT str[512];

   FUNCTION_START(ADB_ExportEPG);

   fd = fopen((const char *)filename, "w+");

   if (fd != NULL)
   {
      // Mark this file as UTF-16 Big Endian
      fwrite(utf16_bom, 1, 2, fd);

      sprintf((char *)str, "%s", XMLHEADER);
      XMLWriteString(fd, str, TRUE, FALSE);

      sprintf((char *)str, "generator-info-name=\"Sunrise\" generator-info-url=\"http://www.oceanbluesoftware.com\"");
      XMLOpenTag(fd, XMLTAG_TV, str, TRUE);

      DBDEF_RequestAccess();

      // First iterate the services and produce the service list at the head of the file
      s_ptr = DBDEF_GetNextServiceRec(NULL);
      while (s_ptr != NULL)
      {
         if (CheckServiceInListType(s_ptr, type, FALSE, FALSE))
         {
            serv_name = GetServiceName(s_ptr, TRUE, TRUE);

            if (use_dvb_uri == TRUE)
            {
               onid = 0;
               tid = 0;
               if (s_ptr->parent_transport != NULL)
               {
                  tid = s_ptr->parent_transport->tran_id;
                  onid = s_ptr->parent_transport->orig_net_id;
               }

               sprintf((char *)chan_id, "id=\"dvb://%x.%x.%x\"", onid, tid, s_ptr->serv_id);
            }
            else
            {
               sprintf((char *)chan_id, "id=\"%d\"", s_ptr->allocated_lcn);
            }

            XMLOpenTag(fd, XMLTAG_CHANNEL, chan_id, TRUE);

            if (s_ptr->def_authority != NULL)
            {
               XMLOpenTag(fd, XMLTAG_URL, NULL, FALSE);
               XMLWriteString(fd, s_ptr->def_authority->str_ptr, FALSE, FALSE);
               XMLCloseTag(fd, XMLTAG_URL);
            }

            XMLOpenTag(fd, XMLTAG_DISPLAYNAME, NULL, FALSE);
            XMLWriteString(fd, serv_name, FALSE, TRUE);
            XMLCloseTag(fd, XMLTAG_DISPLAYNAME);

            XMLCloseTag(fd, XMLTAG_CHANNEL);

            STB_AppFreeMemory(serv_name);
         }

         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }

      // Iterate again and pull out all the event data for each service
      s_ptr = DBDEF_GetNextServiceRec(NULL);
      while (s_ptr != NULL)
      {
         if (CheckServiceInListType(s_ptr, type, FALSE, FALSE))
         {
            // Determine if audio is present and also if Dolby Digital AC3
            // These flags will be used later when describing event audio
            stream_ptr = s_ptr->stream_list;
            audio_ac3 = FALSE;
            audio_present = FALSE;
            while (stream_ptr != NULL)
            {
               if ((stream_ptr->type == ADB_AUDIO_STREAM) || (stream_ptr->type == ADB_AAC_AUDIO_STREAM) ||
                   (stream_ptr->type == ADB_HEAAC_AUDIO_STREAM))
               {
                  audio_present = TRUE;
               }

               if ((stream_ptr->type == ADB_AC3_AUDIO_STREAM) || (stream_ptr->type == ADB_EAC3_AUDIO_STREAM))
               {
                  audio_present = TRUE;
                  audio_ac3 = TRUE;
                  break;
               }
               stream_ptr = stream_ptr->next;
            }

            e_ptr = s_ptr->event_schedule;

            while (e_ptr != NULL)
            {
               // Only export if caller isn't filtering or the date matches
               if ((utc_date_filter == 0) || (e_ptr->start_date == utc_date_filter))
               {
                  if (use_dvb_uri == TRUE)
                  {
                     onid = 0;
                     tid = 0;
                     if (s_ptr->parent_transport != NULL)
                     {
                        tid = s_ptr->parent_transport->tran_id;
                        onid = s_ptr->parent_transport->orig_net_id;
                     }
                     sprintf((char *)chan_id, "dvb://%x.%x.%x", onid, tid, s_ptr->serv_id);
                  }
                  else
                  {
                     sprintf((char *)chan_id, "%d", s_ptr->allocated_lcn);
                  }

                  // Work out the end time from the start time and duration
                  STB_GCCalculateDateTime(e_ptr->start_date, e_ptr->start_hrs, e_ptr->start_mins, e_ptr->start_secs,
                     e_ptr->duration_hrs, e_ptr->duration_mins, e_ptr->duration_secs,
                     &end_date, &end_hour, &end_minute, &end_secs, CALC_ADD);

                  // Get the start and end YYYYMMDD
                  STB_GCGetDateInfo(e_ptr->start_date, e_ptr->start_hrs, e_ptr->start_mins, &start_day, &wday_notused,
                     &start_month, &start_year);
                  STB_GCGetDateInfo(end_date, end_hour, end_minute, &end_day, &wday_notused, &end_month, &end_year);

                  sprintf((char *)str, "start=\"%04d%02d%02d%02d%02d\" stop=\"%04d%02d%02d%02d%02d\" channel=\"%s\"",
                     start_year, start_month, start_day, e_ptr->start_hrs, e_ptr->start_mins,
                     end_year, end_month, end_day, end_hour, end_minute, chan_id);

                  XMLOpenTag(fd, XMLTAG_PROGRAMME, str, TRUE);

                  // Export event title and descriptions in all available languages
                  for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
                  {
                     GetLangStringFromId(i, lcode_string);

                     if (e_ptr->name_array[i] != NULL)
                     {
                        sprintf((char *)str, "lang=\"%s\"", lcode_string );
                        XMLOpenTag(fd, XMLTAG_TITLE, str, FALSE);
                        XMLWriteString(fd, e_ptr->name_array[i]->str_ptr, FALSE, TRUE);
                        XMLCloseTag(fd, XMLTAG_TITLE);
                     }

                     if (e_ptr->desc_array[i] != NULL)
                     {
                        sprintf((char *)str, "lang=\"%s\"", lcode_string );
                        XMLOpenTag(fd, XMLTAG_DESC, str, FALSE);
                        XMLWriteString(fd, e_ptr->desc_array[i]->str_ptr, FALSE, TRUE);
                        XMLCloseTag(fd, XMLTAG_DESC);
                     }
                  }

                  // Category - This should really be output as a string, but the UI will want to
                  // present this in the local language so give the code and it can do as it wants
                  if (e_ptr->content_len > 0)
                  {
                     XMLWriteU32(fd, XMLTAG_CATEGORY, e_ptr->content_data[0]);
                  }

                  // Describe the video
                  XMLOpenTag(fd, XMLTAG_VIDEO, NULL, TRUE);
                  if ((s_ptr->serv_type == ADB_SERVICE_TYPE_TV) || (s_ptr->serv_type == ADB_SERVICE_TYPE_AVC_SD_TV) ||
                      (s_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) || (s_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD))
                  {
                     XMLOpenTag(fd, XMLTAG_PRESENT, NULL, FALSE);
                     sprintf((char *)str, "yes");
                     XMLWriteString(fd, str, FALSE, FALSE);
                     XMLCloseTag(fd, XMLTAG_PRESENT);

                     XMLOpenTag(fd, XMLTAG_QUALITY, NULL, FALSE);
                     if ((s_ptr->serv_type == ADB_SERVICE_TYPE_HD_TV) || (s_ptr->serv_type == ADB_SERVICE_TYPE_MPEG2_HD))
                     {
                        sprintf((char *)str, "HDTV");
                     }
                     else
                     {
                        sprintf((char *)str, "SDTV");
                     }
                     XMLWriteString(fd, str, FALSE, FALSE);
                     XMLCloseTag(fd, XMLTAG_QUALITY);
                  }
                  else
                  {
                     XMLOpenTag(fd, XMLTAG_PRESENT, NULL, FALSE);
                     sprintf((char *)str, "no");
                     XMLWriteString(fd, str, FALSE, FALSE);
                     XMLCloseTag(fd, XMLTAG_PRESENT);
                  }
                  XMLCloseTag(fd, XMLTAG_VIDEO);

                  // Describe the audio
                  XMLOpenTag(fd, XMLTAG_AUDIO, NULL, TRUE);
                  XMLWriteBool(fd, XMLTAG_PRESENT, audio_present);
                  if (audio_present)
                  {
                     XMLOpenTag(fd, XMLTAG_STEREO, NULL, FALSE);
                     if (audio_ac3)
                     {
                        sprintf((char *)str, "dolby digital");
                     }
                     else
                     {
                        sprintf((char *)str, "stereo");
                     }
                     XMLWriteString(fd, str, FALSE, FALSE);
                     XMLCloseTag(fd, XMLTAG_STEREO);
                  }
                  XMLCloseTag(fd, XMLTAG_AUDIO);

                  // Describe the age rating if non zero
                  if (e_ptr->parental_age_rating > 0)
                  {
                     XMLOpenTag(fd, XMLTAG_RATING, (U8BIT *)"system=\"age\"", TRUE);
                     XMLWriteU32(fd, XMLTAG_VALUE, e_ptr->parental_age_rating);
                     XMLCloseTag(fd, XMLTAG_RATING);
                  }

                  // Flag subtitles if present
                  if (e_ptr->subtitles_avail == TRUE)
                  {
                     // Teletext in xmltv just means user can enable/disable as opposed to
                     // a programme that is already subtitles by the broadcaster (can't remove)
                     sprintf((char *)str, "  <%s type=\"teletext\" />", XMLTAG_SUBTITLES);
                     XMLWriteString(fd, str, TRUE, FALSE);
                  }

                  // End description of this event
                  XMLCloseTag(fd, XMLTAG_PROGRAMME);
               }
               e_ptr = e_ptr->next;
            }
         }
         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }

      DBDEF_ReleaseAccess();

      // Close the XML and the file
      XMLCloseTag(fd, XMLTAG_TV);
      fclose(fd);

      success = TRUE;
   }

   FUNCTION_FINISH(ADB_ExportEPG);

   return success;
}

#endif

static void ExportLNBRec(xmlTextWriterPtr writer, ADB_LNB_REC *lnb_ptr)
{
   ADB_SATELLITE_REC *sat_ptr;
   ADB_LNB_BAND_REC *band_ptr;
   xmlChar *str_ptr;

   FUNCTION_START(ExportLNBRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_LNB) >= 0)
   {
      switch (lnb_ptr->type)
      {
         case LNB_TYPE_SINGLE:
            str_ptr = XMLTAG_LNB_SINGLE;
            break;
         case LNB_TYPE_UNIVERSAL:
            str_ptr = XMLTAG_LNB_UNIVERSAL;
            break;
         case LNB_TYPE_UNICABLE:
            str_ptr = XMLTAG_LNB_UNICABLE;
            break;
         case LNB_TYPE_USER_DEFINED:
            str_ptr = XMLTAG_LNB_USER_DEFINED;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_TYPE, str_ptr);
      }

      switch (lnb_ptr->power)
      {
         case LNB_POWER_OFF:
            str_ptr = XMLTAG_OFF;
            break;
         case LNB_POWER_ON:
            str_ptr = XMLTAG_ON;
            break;
         case LNB_POWER_AUTO:
            str_ptr = XMLTAG_AUTO;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_POWER, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_IS22K, "%u", lnb_ptr->is_22k);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_IS12V, "%u", lnb_ptr->is_12v);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_ISPULSEPOS, "%u", lnb_ptr->is_pulse_posn);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_ISDISEQCPOS, "%u", lnb_ptr->is_diseqc_posn);

      switch (lnb_ptr->diseqc_tone)
      {
         case DISEQC_TONE_OFF:
            str_ptr = XMLTAG_OFF;
            break;
         case DISEQC_TONE_A:
            str_ptr = XMLTAG_A;
            break;
         case DISEQC_TONE_B:
            str_ptr = XMLTAG_B;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_DISEQC_TONE, str_ptr);
      }

      switch (lnb_ptr->c_switch)
      {
         case DISEQC_CSWITCH_OFF:
            str_ptr = XMLTAG_OFF;
            break;
         case DISEQC_CSWITCH_A:
            str_ptr = XMLTAG_A;
            break;
         case DISEQC_CSWITCH_B:
            str_ptr = XMLTAG_B;
            break;
         case DISEQC_CSWITCH_C:
            str_ptr = XMLTAG_C;
            break;
         case DISEQC_CSWITCH_D:
            str_ptr = XMLTAG_D;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_CSWITCH, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_USWITCH, "%u", lnb_ptr->u_switch);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_ISSMATV, "%u", lnb_ptr->is_smatv);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_DISEQC_REPEATS, "%u", lnb_ptr->diseqc_repeats);

      if (lnb_ptr->type == LNB_TYPE_UNICABLE)
      {
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_UNICABLE_IF, "%lu", (unsigned long)lnb_ptr->unicable_if);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_UNICABLE_CHAN, "%u", lnb_ptr->unicable_chan);
      }

      if (lnb_ptr->name != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, lnb_ptr->name);
      }

      /* Export any band defined for this LNB */
      band_ptr = DBDEF_GetNextLNBBandRec(NULL);
      while (band_ptr != NULL)
      {
         if (band_ptr->lnb == lnb_ptr)
         {
            ExportLNBBandeRec(writer, band_ptr);
         }

         band_ptr = DBDEF_GetNextLNBBandRec(band_ptr);
      }

      /* Export any satellite records using this LNB */
      sat_ptr = DBDEF_GetNextSatelliteRec(NULL);
      while (sat_ptr != NULL)
      {
         if (sat_ptr->lnb == lnb_ptr)
         {
            ExportSatelliteRec(writer, sat_ptr);
         }

         sat_ptr = DBDEF_GetNextSatelliteRec(sat_ptr);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportLNBRec);
}

static void ExportLNBBandeRec(xmlTextWriterPtr writer, ADB_LNB_BAND_REC *band_ptr)
{
   xmlChar *str_ptr;

   FUNCTION_START(ExportLNBBandeRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_LNB_BAND) >= 0)
   {
      switch (band_ptr->band_params.polarity)
      {
         case POLARITY_HORIZONTAL:
            str_ptr = XMLTAG_HORIZONTAL;
            break;
         case POLARITY_VERTICAL:
            str_ptr = XMLTAG_VERTICAL;
            break;
         case POLARITY_LEFT:
            str_ptr = XMLTAG_LEFT;
            break;
         case POLARITY_RIGHT:
            str_ptr = XMLTAG_RIGHT;
            break;
         default:
            str_ptr = NULL;
            break;
      }
      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_POLARITY, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_BAND_MIN_FREQ, "%lu",
         (unsigned long)band_ptr->band_params.min_freq);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_BAND_MAX_FREQ, "%lu",
         (unsigned long)band_ptr->band_params.max_freq);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_LNB_BAND_LO_FREQ, "%lu",
         (unsigned long)band_ptr->band_params.local_oscillator_frequency);

      switch (band_ptr->band_params.lnb_voltage)
      {
         case LNB_VOLTAGE_OFF:
            str_ptr = XMLTAG_OFF;
            break;
         case LNB_VOLTAGE_14V:
            str_ptr = XMLTAG_LNB_BAND_14V;
            break;
         case LNB_VOLTAGE_18V:
            str_ptr = XMLTAG_LNB_BAND_18V;
            break;
         default:
            str_ptr = NULL;
      }
      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_BAND_VOLTAGE, str_ptr);
      }

      if (band_ptr->band_params.tone_22k)
      {
         str_ptr = XMLTAG_ON;
      }
      else
      {
         str_ptr = XMLTAG_OFF;
      }
      xmlTextWriterWriteAttribute(writer, XMLTAG_LNB_BAND_VOLTAGE, str_ptr);

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportLNBBandeRec);
}

static void ExportSatelliteRec(xmlTextWriterPtr writer, ADB_SATELLITE_REC *sat_ptr)
{
   ADB_NETWORK_REC *n_ptr;

   FUNCTION_START(ExportSatelliteRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_SATELLITE) >= 0)
   {
      if (sat_ptr->name != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, sat_ptr->name);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SAT_DISHPOS, "%u", sat_ptr->dish_pos);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SAT_LONGPOS, "%u", sat_ptr->long_pos);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SAT_EASTWEST, "%u", sat_ptr->east_west);

      /* Export any network records that are children of this satellite */
      n_ptr = DBDEF_GetNextNetworkRec(NULL);
      while (n_ptr != NULL)
      {
         if (n_ptr->satellite == sat_ptr)
         {
            ExportNetworkRec(writer, n_ptr);
         }

         n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportSatelliteRec);
}

static void ExportNetworkRec(xmlTextWriterPtr writer, ADB_NETWORK_REC *n_ptr)
{
   ADB_TRANSPORT_REC *t_ptr;
   xmlChar *str_ptr;

   FUNCTION_START(ExportNetworkRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_NETWORK) >= 0)
   {
      if (n_ptr->name_str != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, n_ptr->name_str);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_NET_ID, "%u", n_ptr->net_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_VERSION, "%u", n_ptr->nit_version);

      switch (n_ptr->profile_type)
      {
         case ADB_PROFILE_TYPE_BROADCAST:
            str_ptr = XMLTAG_PROFILE_BROADCAST;
            break;
         case ADB_PROFILE_TYPE_CIPLUS:
            str_ptr = XMLTAG_PROFILE_CIPLUS;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_PROFILETYPE, str_ptr);
      }

#ifdef COMMON_INTERFACE
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_CAM_ONET, "%u", n_ptr->cicam_onet_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_CAM_ID, "%u", n_ptr->cicam_identifier);

      if (n_ptr->profile_name != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_PROFILE_NAME, n_ptr->profile_name);
      }
#endif

      /* Export any transport records that are children of this network */
      t_ptr = DBDEF_GetNextTransportRec(NULL);
      while (t_ptr != NULL)
      {
         if (t_ptr->network == n_ptr)
         {
            ExportTransportRec(writer, t_ptr);
         }

         t_ptr = DBDEF_GetNextTransportRec(t_ptr);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportNetworkRec);
}

static void ExportTransportRec(xmlTextWriterPtr writer, ADB_TRANSPORT_REC *t_ptr)
{
   ADB_SERVICE_REC *s_ptr;
   xmlChar *str_ptr;
   U32BIT value;
   BOOLEAN element_started;

   FUNCTION_START(ExportTransportRec);

   element_started = FALSE;

   switch (t_ptr->sig_type)
   {
      case SIGNAL_QPSK:
      {
         if (xmlTextWriterStartElement(writer, XMLTAG_SAT_TRANSPORT) >= 0)
         {
            element_started = TRUE;

            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_SYMRATE, "%u", t_ptr->u.sat.symbol_rate);

            switch (t_ptr->u.sat.polarity)
            {
               case POLARITY_HORIZONTAL:
                  str_ptr = XMLTAG_HORIZONTAL;
                  break;
               case POLARITY_VERTICAL:
                  str_ptr = XMLTAG_VERTICAL;
                  break;
               case POLARITY_LEFT:
                  str_ptr = XMLTAG_LEFT;
                  break;
               case POLARITY_RIGHT:
                  str_ptr = XMLTAG_RIGHT;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_POLARITY, str_ptr);
            }

            switch (t_ptr->u.sat.fec_code)
            {
               case FEC_AUTOMATIC:
                  str_ptr = XMLTAG_AUTO;
                  break;
               case FEC_1_2:
                  str_ptr = XMLTAG_1_2;
                  break;
               case FEC_2_3:
                  str_ptr = XMLTAG_2_3;
                  break;
               case FEC_3_4:
                  str_ptr = XMLTAG_3_4;
                  break;
               case FEC_5_6:
                  str_ptr = XMLTAG_5_6;
                  break;
               case FEC_7_8:
                  str_ptr = XMLTAG_7_8;
                  break;
               case FEC_1_4:
                  str_ptr = XMLTAG_1_4;
                  break;
               case FEC_1_3:
                  str_ptr = XMLTAG_1_3;
                  break;
               case FEC_2_5:
                  str_ptr = XMLTAG_2_5;
                  break;
               case FEC_8_9:
                  str_ptr = XMLTAG_8_9;
                  break;
               case FEC_9_10:
                  str_ptr = XMLTAG_9_10;
                  break;
               case FEC_3_5:
                  str_ptr = XMLTAG_3_5;
                  break;
               case FEC_4_5:
                  str_ptr = XMLTAG_4_5;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_FECCODE, str_ptr);
            }

            switch (t_ptr->u.sat.fec_type)
            {
               case FEC_TYPE_AUTO:
                  str_ptr = XMLTAG_AUTO;
                  break;
               case FEC_TYPE_DVBS1:
                  str_ptr = XMLTAG_FEC_DVBS1;
                  break;
               case FEC_TYPE_DVBS2:
                  str_ptr = XMLTAG_FEC_DVBS2;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_FECTYPE, str_ptr);
            }

            switch (t_ptr->u.sat.modulation)
            {
               case MOD_AUTO:
                  str_ptr = XMLTAG_AUTO;
                  break;
               case MOD_QPSK:
                  str_ptr = XMLTAG_QPSK;
                  break;
               case MOD_8PSK:
                  str_ptr = XMLTAG_8PSK;
                  break;
               case MOD_16QAM:
                  str_ptr = XMLTAG_16QAM;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_MODULATION, str_ptr);
            }

            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_DVBS2, "%u", t_ptr->u.sat.dvb_s2);
         }
         break;
      }

      case SIGNAL_QAM:
      {
         if (xmlTextWriterStartElement(writer, XMLTAG_CAB_TRANSPORT) >= 0)
         {
            element_started = TRUE;

            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_SYMRATE, "%u", t_ptr->u.cab.symbol_rate);

            switch (t_ptr->u.cab.cmode)
            {
               case MODE_QAM_AUTO:
                  str_ptr = XMLTAG_AUTO;
                  break;
               case MODE_QAM_4:
                  str_ptr = XMLTAG_QAM4;
                  break;
               case MODE_QAM_8:
                  str_ptr = XMLTAG_QAM8;
                  break;
               case MODE_QAM_16:
                  str_ptr = XMLTAG_QAM16;
                  break;
               case MODE_QAM_32:
                  str_ptr = XMLTAG_QAM32;
                  break;
               case MODE_QAM_64:
                  str_ptr = XMLTAG_QAM64;
                  break;
               case MODE_QAM_128:
                  str_ptr = XMLTAG_QAM128;
                  break;
               case MODE_QAM_256:
                  str_ptr = XMLTAG_QAM256;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_MODE, str_ptr);
            }
         }
         break;
      }

      case SIGNAL_COFDM:
      {
         if (xmlTextWriterStartElement(writer, XMLTAG_TERR_TRANSPORT) >= 0)
         {
            element_started = TRUE;

            switch (t_ptr->u.terr.bwidth)
            {
               case TBWIDTH_5MHZ:
                  value = 5;
                  break;
               case TBWIDTH_6MHZ:
                  value = 6;
                  break;
               case TBWIDTH_7MHZ:
                  value = 7;
                  break;
               case TBWIDTH_8MHZ:
                  value = 8;
                  break;
               case TBWIDTH_10MHZ:
                  value = 10;
                  break;
               default:
                  value = 0;
                  break;
            }

            if (value != 0)
            {
               xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_BWIDTH, "%lu", (unsigned long)value);
            }

            switch (t_ptr->u.terr.tmode)
            {
               case MODE_COFDM_1K:
                  value = 1;
                  break;
               case MODE_COFDM_2K:
                  value = 2;
                  break;
               case MODE_COFDM_4K:
                  value = 4;
                  break;
               case MODE_COFDM_8K:
                  value = 8;
                  break;
               case MODE_COFDM_16K:
                  value = 16;
                  break;
               case MODE_COFDM_32K:
                  value = 32;
                  break;
               default:
                  value = 0;
                  break;
            }

            if (value != 0)
            {
               xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_MODE, "%lu", (unsigned long)value);
            }

            switch (t_ptr->u.terr.terr_type)
            {
               case TERR_TYPE_DVBT:
                  str_ptr = XMLTAG_DVBT;
                  break;
               case TERR_TYPE_DVBT2:
                  str_ptr = XMLTAG_DVBT2;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_TERRTYPE, str_ptr);
            }

            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_PLPID, "%u", t_ptr->u.terr.plp_id);
            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_OFFSET, "%u", t_ptr->u.terr.freq_offset);

            switch (t_ptr->u.terr.constellation)
            {
               case TUNE_TCONST_QPSK:
                  str_ptr = XMLTAG_QPSK;
                  break;
               case TUNE_TCONST_QAM16:
                  str_ptr = XMLTAG_QAM16;
                  break;
               case TUNE_TCONST_QAM64:
                  str_ptr = XMLTAG_QAM64;
                  break;
               case TUNE_TCONST_QAM128:
                  str_ptr = XMLTAG_QAM128;
                  break;
               case TUNE_TCONST_QAM256:
                  str_ptr = XMLTAG_QAM256;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_CONSTELLATION, str_ptr);
            }

            switch (t_ptr->u.terr.hierarchy)
            {
               case TUNE_THIERARCHY_NONE:
                  value = 0;
                  break;
               case TUNE_THIERARCHY_1:
                  value = 1;
                  break;
               case TUNE_THIERARCHY_2:
                  value = 2;
                  break;
               case TUNE_THIERARCHY_4:
                  value = 4;
                  break;
               case TUNE_THIERARCHY_8:
                  value = 8;
                  break;
               case TUNE_THIERARCHY_16:
                  value = 16;
                  break;
               case TUNE_THIERARCHY_32:
                  value = 32;
                  break;
               case TUNE_THIERARCHY_64:
                  value = 64;
                  break;
               case TUNE_THIERARCHY_128:
                  value = 128;
                  break;
               default:
                  value = 0xffffffff;
                  break;
            }

            if (value != 0xffffffff)
            {
               xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_HIERARCHY, "%lu", (unsigned long)value);
            }

            switch (t_ptr->u.terr.lp_code_rate)
            {
               case TUNE_TCODERATE_1_2:
                  str_ptr = XMLTAG_1_2;
                  break;
               case TUNE_TCODERATE_2_3:
                  str_ptr = XMLTAG_2_3;
                  break;
               case TUNE_TCODERATE_3_4:
                  str_ptr = XMLTAG_3_4;
                  break;
               case TUNE_TCODERATE_5_6:
                  str_ptr = XMLTAG_5_6;
                  break;
               case TUNE_TCODERATE_7_8:
                  str_ptr = XMLTAG_7_8;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_LPCODE, str_ptr);
            }

            switch (t_ptr->u.terr.hp_code_rate)
            {
               case TUNE_TCODERATE_1_2:
                  str_ptr = XMLTAG_1_2;
                  break;
               case TUNE_TCODERATE_2_3:
                  str_ptr = XMLTAG_2_3;
                  break;
               case TUNE_TCODERATE_3_4:
                  str_ptr = XMLTAG_3_4;
                  break;
               case TUNE_TCODERATE_5_6:
                  str_ptr = XMLTAG_5_6;
                  break;
               case TUNE_TCODERATE_7_8:
                  str_ptr = XMLTAG_7_8;
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_HPCODE, str_ptr);
            }

            switch (t_ptr->u.terr.guard_int)
            {
               case TUNE_TGUARDINT_1_32:
                  str_ptr = (xmlChar *)"1_32";
                  break;
               case TUNE_TGUARDINT_1_16:
                  str_ptr = (xmlChar *)"1_16";
                  break;
               case TUNE_TGUARDINT_1_8:
                  str_ptr = (xmlChar *)"1_8";
                  break;
               case TUNE_TGUARDINT_1_4:
                  str_ptr = (xmlChar *)"1_4";
                  break;
               case TUNE_TGUARDINT_1_128:
                  str_ptr = (xmlChar *)"1_128";
                  break;
               case TUNE_TGUARDINT_19_128:
                  str_ptr = (xmlChar *)"19_128";
                  break;
               case TUNE_TGUARDINT_19_256:
                  str_ptr = (xmlChar *)"19_256";
                  break;
               default:
                  str_ptr = NULL;
                  break;
            }

            if (str_ptr != NULL)
            {
               xmlTextWriterWriteAttribute(writer, XMLTAG_TRAN_GUARDINT, str_ptr);
            }
         }
         break;
      }

      case SIGNAL_ANALOG:
      {
         if (xmlTextWriterStartElement(writer, XMLTAG_TERR_TRANSPORT) >= 0)
         {
            element_started = TRUE;

            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_OFFSET, "%u", t_ptr->u.anal.freq_offset);
            xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_VTYPE, "%u", t_ptr->u.anal.vtype);
         }
         break;
      }

      default:
      {
         break;
      }
   }

   if (element_started)
   {
      /* Write fields common to all transports */
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_ORIG_NET_ID, "%u", t_ptr->orig_net_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_ID, "%u", t_ptr->tran_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_VERSION, "%u", t_ptr->sdt_version);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_FREQUENCY, "%lu", (unsigned long)t_ptr->frequency);

      /* Export any service records that are children of this transport */
      s_ptr = DBDEF_GetNextServiceRec(NULL);
      while (s_ptr != NULL)
      {
         if (s_ptr->transport == t_ptr)
         {
            ExportServiceRec(writer, s_ptr);
         }

         s_ptr = DBDEF_GetNextServiceRec(s_ptr);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportTransportRec);
}

static void ExportServiceRec(xmlTextWriterPtr writer, ADB_SERVICE_REC *s_ptr)
{
   xmlChar *str_ptr;

   FUNCTION_START(ExportServiceRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_SERVICE) >= 0)
   {
      if (s_ptr->name_str != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, s_ptr->name_str);
      }

      switch (s_ptr->serv_type)
      {
         case ADB_SERVICE_TYPE_TV:
            str_ptr = XMLTAG_TV;
            break;
         case ADB_SERVICE_TYPE_RADIO:
            str_ptr = XMLTAG_RADIO;
            break;
         case ADB_SERVICE_TYPE_AVC_RADIO:
            str_ptr = XMLTAG_AVC_RADIO;
            break;
         case ADB_SERVICE_TYPE_DATA:
            str_ptr = XMLTAG_DATA;
            break;
         case ADB_SERVICE_TYPE_MPEG2_HD:
            str_ptr = XMLTAG_MPEG2_HD;
            break;
         case ADB_SERVICE_TYPE_AVC_SD_TV:
            str_ptr = XMLTAG_AVC_SD_TV;
            break;
         case ADB_SERVICE_TYPE_HD_TV:
            str_ptr = XMLTAG_HD_TV;
            break;
         case ADB_SERVICE_TYPE_UHD_TV:
            str_ptr = XMLTAG_UHD_TV;
            break;
         case ADB_SERVICE_TYPE_ANALOG:
            str_ptr = XMLTAG_ANALOG;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_TYPE, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ID, "%u", s_ptr->serv_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_LCN, "%u", s_ptr->serv_lcn);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ALLOCLCN, "%u", s_ptr->allocated_lcn);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_FAVGROUPS, "%u", s_ptr->fav_groups);

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_HIDDEN, "%u", s_ptr->hidden);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_SELECTABLE, "%u", s_ptr->selectable);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_LOCKED, "%u", s_ptr->locked);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_SCHED_DISABLED, "%u", s_ptr->sched_disabled);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_PF_DISABLED, "%u", s_ptr->now_next_disabled);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_EDIT_LCN, "%u", s_ptr->lcn_editable);

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_FREESAT_ID, "%u", s_ptr->freesat_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_REGION_ID, "%u", s_ptr->region_id);

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportServiceRec);
}

static void ExportFavList(xmlTextWriterPtr writer, ADB_FAVLIST_REC *fav_list)
{
   ADB_FAVSERV_REC *fav_serv;

   FUNCTION_START(ExportFavList);

   if (xmlTextWriterStartElement(writer, XMLTAG_FAVLIST) >= 0)
   {
      if (fav_list->name != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, fav_list->name);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_ID, "%u", fav_list->list_id);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_INDEX, "%u", fav_list->index);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_USERDATA, "%lu", (unsigned long)fav_list->user_data);

      /* Export each service in this list */
      fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, NULL);
      while (fav_serv != NULL)
      {
         if (fav_serv->serv_ptr != NULL)
         {
            if (xmlTextWriterStartElement(writer, XMLTAG_FAVSERV) >= 0)
            {
               xmlTextWriterWriteFormatAttribute(writer, XMLTAG_INDEX, "%u", fav_serv->index);

               if (fav_serv->serv_ptr->allocated_lcn == 0)
               {
                  /* Service can't be identified by its LCN */
                  if (fav_serv->serv_ptr->freesat_id == INVALID_FREESAT_SERV_ID)
                  {
                     /* Need to use the DVB triplet */
                     xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ID, "%u",
                        fav_serv->serv_ptr->serv_id);

                     if (fav_serv->serv_ptr->transport != NULL)
                     {
                        xmlTextWriterWriteFormatAttribute(writer, XMLTAG_ORIG_NET_ID, "%u",
                           fav_serv->serv_ptr->transport->orig_net_id);
                        xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_ID, "%u",
                           fav_serv->serv_ptr->transport->tran_id);
                     }
                  }
                  else
                  {
                     /* Use the Freesat ID as this will uniquely identify the service */
                     xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_FREESAT_ID, "%u",
                        fav_serv->serv_ptr->freesat_id);
                  }
               }
               else
               {
                  /* The LCN assigned to the service should be enough to identify it */
                  xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ALLOCLCN, "%u",
                     fav_serv->serv_ptr->allocated_lcn);
               }

               xmlTextWriterEndElement(writer);
            }
         }

         fav_serv = DBDEF_GetNextServiceFromFavouriteList(fav_list, fav_serv);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportFavList);
}

static void ExportCridRec(xmlTextWriterPtr writer, ADB_CRID_REC *crid_rec)
{
   FUNCTION_START(ExportCridRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_CRID) >= 0)
   {
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_EIT_DATE, "%u", crid_rec->eit_date);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERIES, "%u", crid_rec->series_flag);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_RECOMMENDED, "%u", crid_rec->recommended_flag);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_DATETIME, "%lu", (unsigned long)crid_rec->date_time);
      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ID, "%u", crid_rec->serv_id);

      if (crid_rec->name_str != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_NAME, crid_rec->name_str);
      }

      if (crid_rec->crid_str != NULL)
      {
         XMLWriteValueString(writer, XMLTAG_PROG_CRID, crid_rec->crid_str);
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportCridRec);
}

static void ExportTimerRec(xmlTextWriterPtr writer, ADB_TIMER_REC *timer)
{
   xmlChar *str_ptr;

   FUNCTION_START(ExportTimerRec);

   if (xmlTextWriterStartElement(writer, XMLTAG_TIMER) >= 0)
   {
      /* String will include null terminator so check for more than 1 byte */
      if (STB_GetNumBytesInString(timer->name) > 1)
      {
         if (STB_IsUnicodeString(timer->name))
         {
            /* Skip the first byte which is an indicator byte */
            xmlTextWriterWriteAttribute(writer, XMLTAG_NAME, timer->name + 1);
         }
         else
         {
            xmlTextWriterWriteAttribute(writer, XMLTAG_NAME, timer->name);
         }
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_HANDLE, "%lu", (unsigned long)timer->handle);

      switch (timer->type)
      {
         case TIMER_TYPE_ALARM:
            str_ptr = XMLTAG_TIMER_ALARM;
            break;
         case TIMER_TYPE_SLEEP:
            str_ptr = XMLTAG_TIMER_SLEEP;
            break;
         case TIMER_TYPE_PVR_RECORD:
            str_ptr = XMLTAG_TIMER_PVR_RECORD;
            break;
         case TIMER_TYPE_PRIVATE:
            str_ptr = XMLTAG_TIMER_PRIVATE;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_TYPE, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_DATETIME, "%lu", (unsigned long)timer->start_time);

      switch (timer->frequency)
      {
         case TIMER_FREQ_ONCE:
            str_ptr = XMLTAG_TIMER_ONCE;
            break;
         case TIMER_FREQ_WEEKLY:
            str_ptr = XMLTAG_TIMER_WEEKLY;
            break;
         case TIMER_FREQ_WEEKENDDAYS:
            str_ptr = XMLTAG_TIMER_WEEKENDDAYS;
            break;
         case TIMER_FREQ_WEEKDAYS:
            str_ptr = XMLTAG_TIMER_WEEKDAYS;
            break;
         case TIMER_FREQ_DAILY:
            str_ptr = XMLTAG_TIMER_DAILY;
            break;
         case TIMER_FREQ_HOURLY:
            str_ptr = XMLTAG_TIMER_HOURLY;
            break;
         default:
            str_ptr = NULL;
            break;
      }

      if (str_ptr != NULL)
      {
         xmlTextWriterWriteAttribute(writer, XMLTAG_FREQ, str_ptr);
      }

      xmlTextWriterWriteFormatAttribute(writer, XMLTAG_MISSED, "%u", timer->missed);

      if (timer->type == TIMER_TYPE_ALARM)
      {
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_CHANGESERVICE, "%u",
            timer->u.alarm.change_service);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_ORIG_NET_ID, "%u",
            timer->u.alarm.orig_net_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_ID, "%u", timer->u.alarm.transport_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ID, "%u", timer->u.alarm.service_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_RAMPVOLUME, "%u",
            timer->u.alarm.ramp_volume);
      }
      else if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_DURATION, "%lu", (unsigned long)timer->u.record.duration);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRIGGERED, "%u",
            timer->u.record.event_triggered);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_EVENT_ID, "%u", timer->u.record.event_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_SERV_ID, "%u", timer->u.record.service_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_TRAN_ID, "%u", timer->u.record.transport_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_ORIG_NET_ID, "%u",
            timer->u.record.orig_net_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_DISK_ID, "%u", timer->u.record.disk_id);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_RECOMMENDED, "%u",
            timer->u.record.recommendation);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_NOTIFY, "%u", timer->u.record.notify_time);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_START_PAD, "%ld",
            (unsigned long)timer->u.record.start_padding);
         xmlTextWriterWriteFormatAttribute(writer, XMLTAG_END_PAD, "%ld",
            (unsigned long)timer->u.record.end_padding);

         /* String will include null terminator so check for more than 1 byte */
         if (STB_GetNumBytesInString(timer->u.record.prog_crid) > 1)
         {
            xmlTextWriterWriteAttribute(writer, XMLTAG_PROG_CRID, timer->u.record.prog_crid);
         }
         if (STB_GetNumBytesInString(timer->u.record.other_crid) > 1)
         {
            xmlTextWriterWriteAttribute(writer, XMLTAG_OTHER_CRID, timer->u.record.other_crid);
         }
         if (STB_GetNumBytesInString(timer->u.record.additional_info) > 1)
         {
            xmlTextWriterWriteAttribute(writer, XMLTAG_ADDITIONAL_INFO, timer->u.record.additional_info);
         }
      }

      xmlTextWriterEndElement(writer);
   }

   FUNCTION_FINISH(ExportTimerRec);
}

#if 0
/**
 * @brief   Exports the current event schedule as an XML file
 * @param   lang_id The language ID to convert
 * @param   str the caller's string, must be at least 4 bytes
 */
static void GetLangStringFromId(U8BIT lang_id, U8BIT *str)
{
   U32BIT lang_code;

   lang_code = ACFG_ConvertLangIdToCode(lang_id);

   str[0] = (U8BIT)((lang_code >> 16) & 0x000000ff);
   str[1] = (U8BIT)((lang_code >> 8) & 0x000000ff);
   str[2] = (U8BIT)((lang_code) & 0x000000ff);
   str[3] = '\0';
}

#endif

static void XMLWriteValueString(xmlTextWriterPtr writer, xmlChar *tag_str, ADB_STRING *adb_string)
{
   U8BIT *utf8_str;
   U16BIT nchar;

   utf8_str = STB_ConvertStringToUTF8(adb_string->str_ptr, &nchar, TRUE, adb_string->lang_code);
   if (utf8_str != NULL)
   {
      /* Skip the first char, which is a DVB UTF-8 indicator byte */
      xmlTextWriterWriteAttribute(writer, tag_str, utf8_str + 1);

      STB_ReleaseUnicodeString(utf8_str);
   }
}

