/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Functions to allow string based searching of EPG data
 *
 * @file    ap_epgsearch.c
 * @date    13/08/2011
 * @author  Ocean Blue
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <wchar.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //debug functions

//#include "stbhwdsk.h"
#include "stbhwos.h"
#include "stbgc.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbheap.h"    // STB_AppFreeMemory
#include "stbuni.h"     // STB_ConcatUnicodeStrings()
#include "stbllist.h"

#include "app_nvm.h"    // for APP_NvmRead
#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_epgsearch.h"

#include "dba.h"

//---constant definitions for this file-------------------------------------------------------------
//#define AGS_DEBUG
#ifdef AGS_DEBUG
#define AGS_DBG(x)      STB_SPDebugWrite x
#else
#define AGS_DBG(x)
#endif

//---local typedefs, structs, enumerations for this file--------------------------------------------

/* This structure is used for timer database filters */
typedef struct
{
   U8BIT *search_str;
   E_AGS_SEARCH_FIELDS search_fields;
   E_AGS_MATCH_TYPE match_type;
   E_AGS_SEARCH_DAYS search_days;
   U32BIT min_time;
   U32BIT max_time;
   E_AGS_MATCH_ACTION match_action;

   U16BIT num_entries;
   void **event_list;
   U16BIT max_services;
   void *service;
   U16BIT cur_service_idx;
   U16BIT cur_event_idx;
} S_EPG_SEARCH;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static BOOLEAN CheckMatch(U8BIT *search_string, U8BIT *compare_string, E_AGS_MATCH_TYPE type);


//---global function definitions-----------------------------------------------

/**
 * @brief   Opens an EPG Search
 * @param   search_str The search string to compare
 * @param   search_fields The EPG parts to match
 * @param   match_type The type of comparison to make on the string
 * @param   search_days Which days of the week to search
 * @param   search_times Time range in which to search (NULL for any)
 * @param   match_action The action to perform on a search match
 * @return  Handle for search, NULL if failed
 */
void* AGS_OpenSearch( U8BIT *search_str, E_AGS_SEARCH_FIELDS search_fields,
   E_AGS_MATCH_TYPE match_type, E_AGS_SEARCH_DAYS search_days, S_AGS_TIME_RANGE *search_times,
   E_AGS_MATCH_ACTION match_action)
{
   void *ags_handle;
   S_EPG_SEARCH *this_search;
   U16BIT nchar;

   FUNCTION_START(AGS_OpenSearch);

   AGS_DBG(("AGS_OpenSearch"));
   ags_handle = STB_AppGetMemory(sizeof(S_EPG_SEARCH));

   if (ags_handle != NULL)
   {
      this_search = (S_EPG_SEARCH *)ags_handle;
      this_search->max_services = ADB_GetNumServices();
      this_search->service = ADB_GetNextServiceInList(ADB_SERVICE_LIST_DIGITAL, NULL);
      if (this_search->service != NULL)
      {
         ADB_GetEventSchedule(FALSE, this_search->service, &this_search->event_list, &this_search->num_entries);
      }
      else
      {
         this_search->num_entries = 0;
      }
      this_search->cur_service_idx = 0;
      this_search->cur_event_idx = 0;

      // Copy search settings into this opened search's context
      // Ensure we work with a Unicode version of the search string
      this_search->search_str = STB_ConvertStringToUTF8(search_str, &nchar, TRUE, 0);
      this_search->search_fields = search_fields;
      this_search->match_type = match_type;
      this_search->search_days = search_days;
      if (search_times != NULL)
      {
         this_search->min_time = (search_times->min_hour * 60 * 60) + (search_times->min_minutes * 60) + search_times->min_seconds;
         this_search->max_time = (search_times->max_hour * 60 * 60) + (search_times->max_minutes * 60) + search_times->max_seconds;
      }
      else
      {
         // If caller didn't want time range restriction, just use whole day as range
         this_search->min_time = 0;
         this_search->max_time = (24 * 60 * 60);
      }
      // Not used at the moment, will be for saved searches run daily etc.
      this_search->match_action = match_action;
   }

   FUNCTION_FINISH(AGS_OpenSearch);

   return ags_handle;
}

/**
 * @brief   Gets the next matching result for an open search
 * @param   ags_handle The handle of the open search
 * @param   event_id Caller's event id var to be set with matching result
 * @param   serv_ptr Caller's service pointer to be set with matching result
 * @return  BOOLEAN, TRUE=Match Found, FALSE=No more matches
 */
BOOLEAN AGS_GetNextResult( void *ags_handle, U16BIT *event_id, void **serv_ptr )
{
   S_EPG_SEARCH *this_search = ags_handle;
   void *cur_event_ptr;
   void *cur_service;
   U8BIT *event_name = NULL;
   U8BIT *event_desc = NULL;
   U8BIT *event_ext_desc = NULL;
   U32DHMS utc_event_time = 0;
   U32DHMS loc_event_time = 0;
   U8BIT event_weekday = 0;
   U32BIT event_time;
   BOOLEAN do_compare;
   BOOLEAN match = FALSE;

   FUNCTION_START(AGS_GetNextResult);

   AGS_DBG(("AGS_GetNextResult"));

   while ((match == FALSE) &&
          (this_search->cur_service_idx < this_search->max_services) &&
          (this_search->cur_event_idx < this_search->num_entries))
   {
      cur_event_ptr = this_search->event_list[this_search->cur_event_idx];
      cur_service = this_search->service;

      if (cur_event_ptr != NULL)
      {
         do_compare = TRUE; // Will get set to false if not right day or time range

         utc_event_time = ADB_GetEventStartDateTime(cur_event_ptr);
         // User supplies local weekdays and time range, so convert event to local before comparisons
         loc_event_time = STB_GCConvertDHMS(utc_event_time, CONV_LOCAL);

         event_time = (DHMS_HOUR32(loc_event_time) * 60 * 60) +
            (DHMS_MINS32(loc_event_time) * 60) +
            DHMS_SECS(loc_event_time);
         event_weekday = (U8BIT)STB_GCGetDateWeekDay(DHMS_DATE(loc_event_time));
         AGS_DBG(("Event day %d search days %02x", event_weekday, this_search->search_days));

         // search_days is a mask with bit 0 representing MONDAY and bit 6 SUNDAY
         // so by raising 2 to the power of the day of the week of the event-1 (0-6)
         // we can check if the event day matches the search mask
         if (((U8BIT)pow(2, event_weekday - 1) & this_search->search_days) == 0)
         {
            do_compare = FALSE;
            AGS_DBG(("Not comparing... weekday not in search"));
         }

         if ((event_time < this_search->min_time) || (event_time > this_search->max_time))
         {
            do_compare = FALSE;
            AGS_DBG(("Not comparing... time not in search range"));
         }

         // If correct day and time range then proceed to string comparisons
         if (do_compare == TRUE)
         {
            AGS_DBG(("Comparing against..."));
            // Get the string(s) we'll be using for comparison from the events
            if (this_search->search_fields & AGS_SEARCH_TITLE)
            {
               event_name = ADB_GetEventName(cur_event_ptr);
               AGS_DBG(("Event Name [%s]", event_name));
               match = CheckMatch(this_search->search_str, event_name, this_search->match_type);
            }

            if ((match == FALSE) && (this_search->search_fields & AGS_SEARCH_DESC))
            {
               event_desc = ADB_GetEventDescription(cur_event_ptr);
               AGS_DBG(("Event Desc [%s]", event_desc));
               match = CheckMatch(this_search->search_str, event_desc, this_search->match_type);
            }

            if ((match == FALSE) && (this_search->search_fields & AGS_SEARCH_EXT_DESC))
            {
               event_ext_desc = ADB_GetEventExtendedDescription(cur_event_ptr);
               AGS_DBG(("Event Extended Desc [%s]", event_ext_desc));
               match = CheckMatch(this_search->search_str, event_ext_desc, this_search->match_type);
            }
         }
      }

      // Set the caller's return parameters if a match was found
      if (match == TRUE)
      {
         *event_id = ADB_GetEventId(cur_event_ptr);
         *serv_ptr = cur_service;
      }

      // Move onto the next event in this service
      this_search->cur_event_idx++;

      // If there are no more events, move onto the next service
      if (this_search->cur_event_idx >= this_search->num_entries)
      {
         // Free all the events from the old service
         ADB_ReleaseEventList(this_search->event_list, this_search->num_entries);
         this_search->event_list = NULL;
         this_search->num_entries = 0;

         // Move to the next service
         this_search->service = ADB_GetNextServiceInList(ADB_SERVICE_LIST_DIGITAL, this_search->service);

         // Get the events for the new service
         if (this_search->service != NULL)
         {
            AGS_DBG(("Populating event list"));
            ADB_GetEventSchedule(FALSE, this_search->service, &this_search->event_list, &this_search->num_entries);
            AGS_DBG(("Finished copying events"));
         }
         else
         {
            AGS_DBG(("No more services"));
            this_search->num_entries = 0;
            this_search->event_list = NULL;
         }

         this_search->cur_service_idx++;
         this_search->cur_event_idx = 0;
      }
   }

   FUNCTION_FINISH(AGS_GetNextResult);

   return match;
}

/**
 * @brief   Closes an open search and frees all associated resources
 * @param   ags_handle The handle of the open search
 */
void AGS_CloseSearch( void *ags_handle )
{
   S_EPG_SEARCH *this_search = ags_handle;

   FUNCTION_START(AGS_CloseSearch);

   AGS_DBG(("AGS_CloseSearch"));

   // Free any events still cached
   ADB_ReleaseEventList(this_search->event_list, this_search->num_entries);

   // Free the items allocated in this context
   STB_ReleaseUnicodeString(this_search->search_str);

   // Finally, free the memory for this search context
   STB_AppFreeMemory(ags_handle);

   FUNCTION_FINISH(AGS_CloseSearch);
}

//---local function definitions-----------------------------------------------
static BOOLEAN CheckMatch(U8BIT *search_string, U8BIT *compare_string, E_AGS_MATCH_TYPE type)
{
   BOOLEAN result = FALSE;
   U8BIT *token;
   U8BIT *tmp_string;
   U8BIT *save_ptr = NULL;
   U32BIT unicode_str_size;
   BOOLEAN ignore_case = FALSE;

   // Don't attempt to work with null strings
   if ((search_string == NULL) || (compare_string == NULL))
      return FALSE;

   if ((type & AGS_MATCH_IGNORE_CASE) != 0)
   {
      ignore_case = TRUE;
      type &= ~AGS_MATCH_IGNORE_CASE;
   }

   switch (type)
   {
      case AGS_MATCH_ANY_WORD:
      case AGS_MATCH_ALL_WORDS:
      {
         // Make a local copy of the search string so it can be tokenised
         unicode_str_size = STB_GetNumBytesInString(search_string);
         tmp_string = STB_AppGetMemory(unicode_str_size);
         if (tmp_string != NULL)
         {
            memcpy(tmp_string, search_string, unicode_str_size);
         }

         token = STB_UnicodeStringTokenise(tmp_string, &save_ptr);
         while (token != NULL)
         {
            if (STB_UnicodeStrStr(compare_string, token, ignore_case) != NULL)
            {
               result = TRUE;
               // May as well exit to save time... only need one match
               if (type == AGS_MATCH_ANY_WORD)
                  break;
            }
            else
            {
               // Undo any previous match on a non-matching word and exit
               if (type == AGS_MATCH_ALL_WORDS)
               {
                  result = FALSE;
                  break;
               }
            }
            token = STB_UnicodeStringTokenise(NULL, &save_ptr);
         }

         if (tmp_string != NULL)
         {
            STB_AppFreeMemory(tmp_string);
         }
         break;
      }

      // Check for EXACT match
      // i.e. only if event name is "SPECIAL PROG" and search is "SPECIAL PROG"
      case AGS_MATCH_EXACT_PHRASE:
      {
         if (STB_CompareUnicodeStrings(search_string, compare_string, TRUE, ignore_case) == 0)
         {
            result = TRUE;
         }
         break;
      }

      case AGS_MATCH_NOT_EQUAL:
      {
         if (STB_CompareUnicodeStrings(search_string, compare_string, TRUE, ignore_case) != 0)
         {
            result = TRUE;
         }
         break;
      }

      // Check for CONTAINS match
      // i.e. if event name is "SOME SPECIAL PROG SHOWING" and search is "SPECIAL PROG"
      case AGS_MATCH_CONTAINS_PHRASE:
      {
         if (STB_UnicodeStrStr(compare_string, search_string, ignore_case) != NULL)
         {
            result = TRUE;
         }
         break;
      }

      default:
      {
         break;
      }
   }

   return result;
}

