/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief
 * @file    ap_events.c
 * @date    March 2011
 * @author  Steve Ford
 */

/*#define OUTPUT_DEBUG*/

/*---includes for this file--------------------------------------------------*/
/* compiler library header files */
#include <stdio.h>
#include <string.h>
/* third party header files */

/* DVBCore header files */
#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbhwav.h"

#include "stberc.h"
#include "stbdpc.h"
#include "stbheap.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbllist.h"
#include "stbpvr.h"
#include "stberc.h"

#include "ap_si.h"
#include "ap_tmr.h"
#include "ap_events.h"
#include "ap_dbacc.h"
#include "ap_pvr.h"
#include "app.h"
#include "ap_state.h"
#include "ap_cntrl.h"

#ifdef COMMON_INTERFACE
#include "ap_ci_int.h"
#endif

#ifdef INTEGRATE_HBBTV
#include "hbbtv_api.h"
#endif

/*---constant definitions for this file--------------------------------------*/
#define MAX_EVENTS         500

#define TASK_STACK_SIZE    4096
#define TASK_PRIORITY      11

#ifdef OUTPUT_DEBUG
#define DBG(x) STB_SPDebugWrite x
#else
#define DBG(x)
#endif

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */
static void *event_queue;

static void *original_alternative_service;
static BOOLEAN classify_repeats = FALSE;

typedef struct aev_event_handler
{
   LINK_LIST_PTR_BLK ptrs;       /* prev/next handler in linked list */
   DVB_EVENT_HANDLER handler;
} AEV_EVENT_HANDLER;
LINK_LIST_HEADER event_handler_list;

void* event_handlers_mutex = NULL;

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */
static BOOLEAN STBEventHandler(BOOLEAN latched, BOOLEAN repeat, U16BIT path_class, U16BIT type,
   void *data, U32BIT data_size);
static void EventTask(void *param);
static BOOLEAN CheckForAlternativeService(U8BIT alt_serv_type);
static BOOLEAN ReturnFromAlternativeService(void);


/*---global function definitions---------------------------------------------*/

/*****************************************************************************
 * @brief
 ******************************************************************************/
BOOLEAN AEV_Initialise(DVB_EVENT_HANDLER event_handler)
{
   BOOLEAN retval;

   FUNCTION_START(AEV_Initialise);

   retval = TRUE;

   /* Create an event queue */
   event_queue = STB_OSCreateQueue(sizeof(S_EVENT_INFO), MAX_EVENTS);

   /* Register a handler to deal with events from the midware */
   STB_ERRegisterHandler(STBEventHandler);

   /* Initialise the external event handler list */
   STB_LLInitialiseHeader(&event_handler_list);

   /* Save the external event handler */
   /* Create a task to handle the events */
   event_handlers_mutex = STB_OSCreateMutex();
   if ((event_handlers_mutex == NULL) ||
         (event_handler && (APP_RegisterDVBEventHandler(event_handler) == FALSE)) ||
         (STB_OSCreateTask(EventTask, NULL, TASK_STACK_SIZE, TASK_PRIORITY, (U8BIT *)"EventTask") == NULL))
   {
      retval = FALSE;
   }

   original_alternative_service = NULL;

   FUNCTION_FINISH(AEV_Initialise);

   return(retval);
}

void AEV_Terminate(void)
{
   LINK_LIST_PTR_BLK * entry;


   while (NULL != (entry = STB_LLGetFirstBlock(&event_handler_list)))
   {
      STB_LLRemoveBlock(entry);
      STB_AppFreeMemory(entry);
   }

   if (event_handlers_mutex)
   {
      STB_OSDeleteMutex(event_handlers_mutex);
      event_handlers_mutex = NULL;
   }
}

BOOLEAN APP_RegisterDVBEventHandler(DVB_EVENT_HANDLER event_handler)
{
   AEV_EVENT_HANDLER *handler;
   BOOLEAN registered = FALSE;

   if (event_handler && event_handlers_mutex)
   {
      handler = STB_AppGetMemory(sizeof(AEV_EVENT_HANDLER));
      if (handler)
      {
         handler->handler = event_handler;
         STB_OSMutexLock(event_handlers_mutex);
         STB_LLAddBlockToEnd(&event_handler_list, (LINK_LIST_PTR_BLK *)handler);
         STB_OSMutexUnlock(event_handlers_mutex);
         registered = TRUE;
      }
   }
   return registered;
}

BOOLEAN APP_UnregisterDVBEventHandler(DVB_EVENT_HANDLER event_handler)
{
   LINK_LIST_PTR_BLK * entry;
   AEV_EVENT_HANDLER * handler;
   BOOLEAN unregistered = FALSE;

   if (event_handler && event_handlers_mutex)
   {
      STB_OSMutexLock(event_handlers_mutex);
      entry = STB_LLGetFirstBlock(&event_handler_list);
      while (entry)
      {
         handler = ((AEV_EVENT_HANDLER*)entry);
         if (handler->handler == event_handler)
         {
            STB_LLRemoveBlock(entry);
            unregistered = TRUE;
            STB_AppFreeMemory(handler);
            break;
         }
         entry = STB_LLGetNextBlock(entry);
      }
      STB_OSMutexUnlock(event_handlers_mutex);
   }
   return unregistered;
}

/**
 * @brief Enable classifying of repeat events by EV_CLASS_IS_REPEAT flag
 * @param enable TRUE - to turn on, FALSE - turn off (default)
 */
void APP_SetClassifyRepeatEvents(BOOLEAN enable)
{
   classify_repeats = enable;
}

/*---local function definitions----------------------------------------------*/

static void NotifyAll(U32BIT event, void *event_data, U32BIT data_size)
{
   LINK_LIST_PTR_BLK * entry;
   AEV_EVENT_HANDLER * handler;

   if (event_handlers_mutex)
   {
      STB_OSMutexLock(event_handlers_mutex);
      entry = STB_LLGetFirstBlock(&event_handler_list);
      while (entry)
      {
         handler = ((AEV_EVENT_HANDLER*)entry);
         handler->handler(event, event_data, data_size);
         entry = STB_LLGetNextBlock(entry);
      }
      STB_OSMutexUnlock(event_handlers_mutex);
   }
}


/*****************************************************************************
 * @brief
 ******************************************************************************/
static BOOLEAN STBEventHandler(BOOLEAN latched, BOOLEAN repeat, U16BIT class, U16BIT type,
   void *data, U32BIT data_size)
{
   S_EVENT_INFO event_info;
   BOOLEAN retval;

   FUNCTION_START(STBEventHandler);

   USE_UNWANTED_PARAM(latched);

   retval = TRUE;

   if (repeat && (class & EV_CLASS_CAN_REPEAT_FLAG) && classify_repeats)
   {
      class |= EV_CLASS_IS_REPEAT;
   }

   /* Build the 32 bit event code from the event parameters */
   event_info.event_code = (U32BIT)(class << 16) | type;

   if ((data != NULL) && (data_size > 0))
   {
      /* Allocate memory for the event data to be passed on */
      if ((event_info.data = STB_AppGetMemory(data_size)) != NULL)
      {
         memcpy(event_info.data, data, data_size);
         event_info.data_size = data_size;
      }
      else
      {
         retval = FALSE;
      }
   }
   else
   {
      event_info.data = NULL;
      event_info.data_size = 0;
   }

   if (retval)
   {
      /* Send the event to the queue */
      STB_OSWriteQueue(event_queue, (void *)&event_info, sizeof(S_EVENT_INFO), TIMEOUT_NEVER);
   }

   FUNCTION_FINISH(STBEventHandler);

   return(retval);
}

/*****************************************************************************
 * @brief
 ******************************************************************************/
static void EventTask(void *param)
{
   S_EVENT_INFO event_info;
   U8BIT path;
   void *serv_ptr;
   void *event_serv;
   U16BIT disk_id;
   U32BIT handle;
#if 0
   U32BIT heap_stats_time = STB_OSGetClockMilliseconds();
#endif
   BOOLEAN is_live;

   FUNCTION_START(EventTask);

   USE_UNWANTED_PARAM(param);

   DBG(("EventTask: started"));

   while (1)
   {
      if (STB_OSReadQueue(event_queue, (void *)&event_info, sizeof(S_EVENT_INFO), TIMEOUT_NEVER))
      {
         DBG(("EventTask: event 0x%08lx, data %p, %lu bytes", event_info.event_code,
              event_info.data, event_info.data_size));

         switch (event_info.event_code)
         {
            case STB_EVENT_TUNE_LOCKED:
            case STB_EVENT_TUNE_NOTLOCKED:
            case STB_EVENT_TUNE_SIGNAL_DATA_BAD:
            case STB_EVENT_TUNE_SIGNAL_DATA_OK:
            case STB_EVENT_SEARCH_SUCCESS:
            case STB_EVENT_SEARCH_FAIL:
            {
               ACTL_ActionEvent(event_info.event_code, event_info.data);
               break;
            }

            case STB_EVENT_AUDIO_DECODE_STARTED:
            case STB_EVENT_VIDEO_DECODE_STARTED:
            case STB_EVENT_AUDIO_DECODE_STOPPED:
            case STB_EVENT_VIDEO_DECODE_STOPPED:
            case STB_EVENT_AUDIO_DECODE_UNDERFLOW:
            case STB_EVENT_VIDEO_DECODE_UNDERFLOW:
            case STB_EVENT_SAMPLE_DECODE_STOPPED:
            case STB_EVENT_DECODE_LOCKED:
            {
               ACTL_ActionEvent(event_info.event_code, event_info.data);
               break;
            }

            case STB_EVENT_HDMI_CONNECTED:
            {
               ACTL_HDMIConnected();
               break;
            }

            case STB_EVENT_HDMI_DISCONNECTED:
            {
               ACTL_HDMIDisconnected();
               break;
            }

            case STB_EVENT_OTA_SW_UPGRADE_FOUND:
            case STB_EVENT_OTA_SW_UPGRADE_NOTFOUND:
            case STB_EVENT_OTA_SW_UPGRADE_ERROR:
            {
               ACTL_ActionEvent(event_info.event_code, event_info.data);
               break;
            }

            case STB_EVENT_DISK_FULL:
            {
               /* Stop any recordings being made on the given disk */
               disk_id = *(U16BIT *)event_info.data;

               for (path = 0; path < STB_DPGetNumPaths(); path++)
               {
                  if (STB_DPIsRecording(path, &handle) &&
                      (STB_PVRRecordingGetDiskId(handle) == disk_id))
                  {
                     APVR_StopRecording(handle);
                  }
               }
               break;
            }

            case APP_EVENT_SERVICE_NOT_RUNNING:
            {
               ACTL_StopSubtitles();

               /* Check for an alternative service */
               if (!CheckForAlternativeService(LINK_TYPE_SERVICE_REPLACEMENT))
               {
                  /* No alterative service found */
                  ACTL_ActionEvent(event_info.event_code, event_info.data);
               }
               break;
            }

            case APP_EVENT_SERVICE_AUDIO_PID_UPDATE:
            case APP_EVENT_SERVICE_VIDEO_PID_UPDATE:
            case APP_EVENT_SERVICE_RUNNING:
            {
               event_serv = *(void **)event_info.data;
               if ((path = STB_DPGetPathForService(event_serv)) != INVALID_RES_ID)
               {
                  if (STB_DPIsDecodingPath(path))
                  {
                     if ((original_alternative_service != NULL) &&
                         (event_serv == original_alternative_service))
                     {
                        /* A non-running / scrambled service that was switched away from has now
                         * started again, so switch back to it */
                        ACTL_StopSubtitles();
                        ReturnFromAlternativeService();
                     }
                     else
                     {
                        /* Handle the PID update for the service */
                        ACTL_ActionEvent(event_info.event_code, event_serv);
                     }
                  }
                  else
                  {
                     /* Handle the PID update for the service */
                     ACTL_ActionEvent(event_info.event_code, event_serv);
                  }
               }
               break;
            }

            case APP_EVENT_SERVICE_STREAMS_CHANGED:
            {
#if 0
               event_serv = *(void **)event_info.data;

               /* The service may be used by more than one path, so need to check them all */
               for (path = 0; path < STB_DPGetNumPaths(); path++)
               {
                  if ((ADB_GetTunedService(path) == event_serv) && STB_DPIsRecordingPath(path))
                  {
                     /* PIDs for the recording have changed */
                     APVR_PidsUpdated(path);
                  }
               }
#else
               ACTL_ActionEvent(event_info.event_code, event_info.data);
#endif
               break;
            }

            case APP_EVENT_SERVICE_SCRAMBLE_CHANGE:
            {
               event_serv = *(void **)event_info.data;
               if ((path = STB_DPGetLivePath()) != INVALID_RES_ID)
               {
                  serv_ptr = ADB_GetTunedService(path);
               }

               if ((path != INVALID_RES_ID) && (event_serv == serv_ptr))
               {
                  if (ADB_GetServiceScrambledFlag(serv_ptr))
                  {
                     ACTL_StopSubtitles();

                     if (!CheckForAlternativeService(LINK_TYPE_CA_REPLACEMENT_SERVICE))
                     {
                        /* No alternative service found, so handle the event here */
                        ACTL_ActionEvent(event_info.event_code, NULL);
                     }
                  }
                  else
                  {
                     /* Existing service has become unscrambled */
                     ACTL_ActionEvent(event_info.event_code, event_serv);
                  }
               }
               else if ((original_alternative_service != NULL) &&
                        (event_serv == original_alternative_service))
               {
                  if (!ADB_GetServiceScrambledFlag(original_alternative_service))
                  {
                     /* The original service is no longer scrambled so we can return to it */
                     ACTL_StopSubtitles();
                     ReturnFromAlternativeService();
                  }
               }
               break;
            }

            case APP_EVENT_SERVICE_SUBTITLE_UPDATE:
            case APP_EVENT_SERVICE_VIDEO_CODEC_CHANGED:
            case APP_EVENT_SERVICE_AUDIO_CODEC_CHANGED:
            {
               event_serv = *(void **)event_info.data;
               if ((path = STB_DPGetPathForService(event_serv)) != INVALID_RES_ID)
               {
                  ACTL_ActionEvent(event_info.event_code, event_serv);
               }
               break;
            }

            case STB_EVENT_TIMER_EXPIRE:
            {
               /* Check if this is a private timer for any module */
#ifdef COMMON_INTERFACE
               if (!ACI_HandlePrivateTimer(*((U32BIT *)event_info.data)))
#endif
               {
                  if (!APVR_HandlePrivateTimer(*((U32BIT *)event_info.data)))
                  {
                     if (!ACTL_HandlePrivateTimerEvent(*((U32BIT *)event_info.data)))
                     {
                        ATMR_HandleTimerEvent(*((U32BIT *)event_info.data));
                     }
                  }
               }
               break;
            }

            case STB_EVENT_TIMER_NOTIFY:
            {
#ifdef INTEGRATE_HBBTV
               if ((event_info.data != NULL) && (event_info.data_size == sizeof(U32BIT)))
               {
                  U32BIT timer_handle = *(U32BIT *)event_info.data;
                  if (timer_handle != INVALID_TIMER_HANDLE)
                  {
                     HBBTV_NotifyRecordingEvent(timer_handle, HBBTV_RECORDING_ABOUT_TO_START);
                  }
               }
#endif
               break;
            }

            case APP_EVENT_SERVICE_EIT_NOW_UPDATE:
            {
#ifdef NOW_EVENTS_LATCHED
               /* This event is latched so notify that it's now been handled.
                * The event is unlatched first to ensure that a new event coming in
                * while this one if being processed isn't lost, even if it means some
                * unecessary processing is performed, because it's safer than missing an event */
               STB_ERNotifyEvent(EVENT_CLASS(event_info.event_code), EVENT_TYPE(event_info.event_code));

               path = ACTL_GetActivePath();
               if ((path != INVALID_RES_ID) && STB_DPIsDecodingPath(path) &&
                   ((serv_ptr = ADB_GetTunedService(path)) != NULL))
               {
                  ACTL_ApplyParentalControl(path, serv_ptr);
                  ACTL_ApplyHDCP(serv_ptr);
               }

               event_serv = NULL;
#else
               event_serv = *(void **)event_info.data;

               for (path = 0; path < STB_DPGetNumPaths(); path++)
               {
                  if ((ADB_GetTunedService(path) == event_serv) && STB_DPIsDecodingPath(path))
                  {
                     ACTL_ApplyParentalControl(path, event_serv);
                     ACTL_ApplyHDCP(event_serv);
                  }
               }
#endif

               if (ATMR_CheckRecordStatus(TRUE, event_serv))
               {
                  /* A recording has been started */
                  ASTE_StartRecording();
               }

               APVR_EitUpdated();
               break;
            }

            case STB_EVENT_PVR_PLAY_START:
                STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_PLAY_STARTED, event_info.data, event_info.data_size);
                break;

            case STB_EVENT_PVR_PLAY_STOP:
                STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_PLAY_STOPPED, event_info.data, event_info.data_size);
                break;

            case STB_EVENT_PVR_PLAY_BOF:
                STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_PLAY_BOF, event_info.data, event_info.data_size);
                break;

            case STB_EVENT_PVR_PLAY_EOF:
                STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_PLAY_EOF, event_info.data, event_info.data_size);
                break;

            case APP_EVENT_PVR_RECORDING_FAILED:
            case STB_EVENT_PVR_REC_STOP:
            {
               if (ASTE_InStandby() && !APVR_IsRecordingInProgress())
               {
                  /* All recordings have finished and none are ready to start */
                  ACTL_EnterStandby();
               }
               break;
            }

            case APP_EVENT_SERVICE_EIT_SCHED_UPDATE:
            {
               /* This event is latched so notify that it's now been handled */
               STB_ERNotifyEvent(EVENT_CLASS(event_info.event_code), EVENT_TYPE(event_info.event_code));
               APVR_EitUpdated();
               break;
            }

            case STB_EVENT_FORCED_SERVICE_CHANGE:
            {
               path = *(U8BIT *)event_info.data;
               serv_ptr = STB_DPGetTunedService(path);
               is_live = STB_DPIsLivePath(path);
               ACTL_TuneToService(path, NULL, serv_ptr, FALSE, is_live);
               if (is_live == TRUE)
               {
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED,
                     &serv_ptr, sizeof(void *));
               }
               break;
            }

#ifdef COMMON_INTERFACE
            case STB_EVENT_CI_APP_INFO:
            {
               ACTL_ActionEvent(event_info.event_code, event_info.data);
               break;
            }
#endif

            default:
            {
               break;
            }
         }

         /* Now pass the event to the event handler that was registered on startup, if any,
          * which may pass the event to the UI */
         NotifyAll(event_info.event_code, event_info.data, event_info.data_size);

         /* The memory associated with the event can now be freed */
         if (event_info.data != NULL)
         {
            STB_AppFreeMemory(event_info.data);
         }
#if 0
         if (STB_OSGetClockDiff(heap_stats_time) >= 30000)
         {
            U32BIT cur_app, max_app, num_app, cur_mem, max_mem, num_mem;
            STB_GetHeapStats(&cur_app, &max_app, &num_app, &cur_mem, &max_mem, &num_mem);
            STB_SPDebugWrite("<<<<< Mem usage:\n");
            STB_SPDebugWrite("   App: Current=%luKB, Max=%luKB, Num=%lu\n", cur_app / 1024, max_app / 1024, num_app);
            STB_SPDebugWrite("   Sys: Current=%luKB, Max=%luKB, Num=%lu\n", cur_mem / 1024, max_mem / 1024, num_mem);
            heap_stats_time = STB_OSGetClockMilliseconds();
         }
#endif
      }
   }

   DBG(("EventTask: finished"));

   FUNCTION_FINISH(EventTask);
}

/*****************************************************************************
 * @brief   Checks for an alternative service and if found, tunes to it.
 *          This will usually be triggered when a service is not running or is scrambled.
 * @param   alt_serv_type - linkage type for the alterative service to be searched for
 * @return  TRUE if an alternative service is found
 ******************************************************************************/
static BOOLEAN CheckForAlternativeService(U8BIT alt_serv_type)
{
   void *serv_ptr;
   void *new_serv;
   U8BIT path;
   BOOLEAN retval;

   FUNCTION_START(CheckForAlternativeService);

   retval = FALSE;

   path = STB_DPGetLivePath();
   if (path != INVALID_RES_ID)
   {
      serv_ptr = ADB_GetTunedService(path);

      new_serv = ADB_GetAlternativeService(serv_ptr, alt_serv_type);
      if (new_serv != NULL)
      {
         /* Store the original service so that it can be returned to */
         original_alternative_service = serv_ptr;

         NotifyAll(APP_EVENT_SWITCH_ALTERNATIVE_SERVICE,&new_serv, sizeof(void *));
         ACTL_TuneToService(INVALID_RES_ID, NULL, new_serv, FALSE, TRUE);
         retval = TRUE;
      }
   }

   FUNCTION_FINISH(CheckForAlternativeService);

   return(retval);
}

/*!**************************************************************************
 * @brief   Changes back to a service that was left due to a switch to an alternative
 *          service as a result of something like a non-running event.
 * @return  TRUE if the service is changed, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ReturnFromAlternativeService(void)
{
   BOOLEAN retval;

   FUNCTION_START(ReturnFromAlternativeService);

   if (original_alternative_service != NULL)
   {
      NotifyAll(APP_EVENT_SWITCH_ALTERNATIVE_SERVICE,&original_alternative_service, sizeof(void *));
      ACTL_TuneToService(INVALID_RES_ID, NULL, original_alternative_service, FALSE, TRUE);
      retval = TRUE;
      original_alternative_service = NULL;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(ReturnFromAlternativeService);

   return(retval);
}

/****************************************************************************
 * End of file
 *****************************************************************************/

/*****************************************************************************
 * @brief
 ******************************************************************************/

