/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Initialises STB and creates a task to handle events, then enters idle loop.
 *
 * @file    ap_init.c
 * @date    April 2011
 * @author  Ocean Blue
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //debug functions
#include "stbhwos.h"
#include "stbhwav.h"

#include "stberc.h"
#include "stbinit.h"
#include "stbuni.h"
#include "stbdpc.h"
#include "stbheap.h"
#include "stbllist.h"
#include "stbsiflt.h"
#include "stbsitab.h"

#include "ap_dbacc.h"
#include "ap_cntrl.h"
#include "ap_cfg.h"
#include "app_nvm.h"
#include "ap_si.h"
#include "ap_events.h"
#include "app.h"
#include "ap_tmr.h"
#include "stbpvr.h"
#include "ap_pvr.h"
#include "ap_ipadd.h"

#ifdef COMMON_INTERFACE
#include "ap_ci_int.h"
#endif

#ifndef DEFAULT_ASCII_TABLE
#define DEFAULT_ASCII_TABLE   0
#endif

#ifdef PVR_LOG
FILE *pvr_log;
#endif

//---constant definitions for this file-------------------------------------------------------------


//---local typedefs, structs, enumerations for this file--------------------------------------------


//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)
static ACB_INFO_CALLBACK UiInfoCallback;

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)


//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------


//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

BOOLEAN APP_InitialiseDVB(DVB_EVENT_HANDLER event_handler, E_DVB_INIT_SUBS_TTXT subs_ttxt)
{
   E_STB_TTXT_SUBT_CONTROL subt_ctrl;
   BOOLEAN first_boot;
   U32BIT country_code;
   U8BIT lang_entry;
   U8BIT second_lang_entry;
   U8BIT text_lang_entry;
   U8BIT second_text_lang_entry;
   BOOLEAN aerial_pwr;
   U8BIT tuner;

   FUNCTION_START(APP_InitialiseDVB);

   /* Initialise the midware and platform - the order is important as it allows events
    * to be sent during initialisation */
   STB_OSInitialise();
   STB_ERInitialise();

   /* Initialise the event handler which will also start a task to monitor and handle system events */
   AEV_Initialise(event_handler);

   /* Now initialise the rest of the platform & midware */
   switch (subs_ttxt)
   {
      case DVB_INIT_TELETEXT_ONLY:
         subt_ctrl = EBU_SUBTITLES;
         break;
      case DVB_INIT_SUBTITLES_ONLY:
         subt_ctrl = DVB_SUBTITLES;
         break;
      case DVB_INIT_TELETEXT_AND_SUBTITLES:
         subt_ctrl = DVB_SUBTITLES_AND_TELETEXT;
         break;
      case DVB_INIT_NO_TELETEXT_OR_SUBTITLES:
      default:
         subt_ctrl = NO_TELETEXT_OR_SUBTITLES;
         break;
   }

   STB_Initialise(subt_ctrl);

   STB_SetDefaultAsciiTable(DEFAULT_ASCII_TABLE);

   /* Initialise the database and DVB settings/data */
   ADB_Initialise();
   APP_NvmInitialise();

   /* Do initial setting for IP network */
   AIP_ConnectToNetwork(FALSE);

   #ifdef COMMON_INTERFACE
   ACI_Initialise();
   #endif

   /* Initialise the nvm data, application SI and control handling */
   ASI_InitialiseAppSi();
   ACTL_InitialiseAppControl();

   first_boot = (BOOLEAN)APP_NvmRead(FIRST_BOOT_NVM);

   /* Do country specific initialisations */
   country_code = ACFG_GetCountry();

   ACFG_SetCountryIds(country_code, ACFG_GetRegionId(), ACFG_GetPrimaryAudioLangId(),
      ACFG_GetPrimaryTextLangId());

   /* Set default audio and subtitle language */
   lang_entry = ACFG_GetPrimaryAudioLangId();
   second_lang_entry = ACFG_GetSecondaryAudioLangId();
   if (second_lang_entry == ACFG_INVALID_LANG)
   {
      second_lang_entry = ACFG_GetDefaultSecondaryLangId();
      if (second_lang_entry != ACFG_INVALID_LANG)
      {
         ACFG_SetSecondaryAudioLangId(second_lang_entry);
      }
   }

   text_lang_entry = ACFG_GetPrimaryTextLangId();
   second_text_lang_entry = ACFG_GetSecondaryTextLangId();
   if (second_text_lang_entry == ACFG_INVALID_LANG)
   {
      second_text_lang_entry = ACFG_GetDefaultSecondaryLangId();
      if (second_text_lang_entry != ACFG_INVALID_LANG)
      {
         ACFG_SetSecondaryTextLangId(second_lang_entry);
      }
   }

   ADB_SetAudioLang(country_code, lang_entry);
   ADB_SetTextLang(country_code, text_lang_entry);

   if (second_lang_entry != ACFG_INVALID_LANG)
   {
      ADB_SetSecondaryAudioLang(country_code, second_lang_entry);
   }

   if (second_text_lang_entry != ACFG_INVALID_LANG)
   {
      ADB_SetSecondaryTextLang(country_code, second_text_lang_entry);
   }

   /* Set aerial power, if it's required */
   if (ACFG_GetAerialPowerOptionReqd(country_code))
   {
      if (first_boot)
      {
         aerial_pwr = ACFG_GetDefaultAerialPower(country_code);
         APP_NvmSave(AERIAL_POWER_ON_NVM, aerial_pwr, TRUE);
      }
      else
      {
         aerial_pwr = (BOOLEAN)APP_NvmRead(AERIAL_POWER_ON_NVM);
      }

      /* Set active aerial power setting for each tuner */
      for (tuner = 0; tuner < STB_HWGetTunerPaths(); tuner++)
      {
         STB_TuneActiveAerialPower(tuner, aerial_pwr);
      }
   }

   ATMR_Initialise();

   STB_PVRSetStandbyState(FALSE);
   APVR_Initialise();

   FUNCTION_FINISH(APP_InitialiseDVB);

   return(TRUE);
}

#if 0
void APP_TerminateDVB(void)
{
   FUNCTION_START(APP_TerminateDVB);

   APVR_Terminate();
   ATMR_Terminate();

   AEV_Terminate();

   FUNCTION_FINISH(APP_TerminateDVB);
}

#endif

void APP_RegisterUIInfoCallback(ACB_INFO_CALLBACK ui_info_callback)
{
   FUNCTION_START(APP_RegisterUIInfoCallback);
   UiInfoCallback = ui_info_callback;
   FUNCTION_FINISH(APP_RegisterUIInfoCallback);
}

BOOLEAN ACB_GetUIInformation(S_ACB_UI_INFO *info)
{
   BOOLEAN retval;

   FUNCTION_START(ACB_GetUIInfo);
   retval = UiInfoCallback ? UiInfoCallback(info) : FALSE;
   FUNCTION_FINISH(ACB_GetUIInfo);

   return retval;
}

