/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2010 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Contains the function for ip address
 *
 * @file    ap_ipadd.c
 * @date    27/04/2010
 * @author  Ocean Blue
 */

//#define DEBUG_IP

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //debug functions

#include "stbhwos.h"
#include "stbhwip.h"
#include "stbhwnet.h"
#include "stbheap.h"
#include "app_nvm.h"
#include "ap_ipadd.h"

//---constant definitions for this file-------------------------------------------------------------
#ifdef DEBUG_IP
#define DBG(X) STB_SPDebugWrite X
#else
#define DBG(X)
#endif

#define IP_TASK_QUEUE_SIZE 10
#define IP_TASK_STACK      1024
#define IP_TASK_PRIO       10
#define IP_TASK_TIMEOUT    30000
#define IP_TASK_PERIOD     1000

//---local typedefs, structs, enumerations for this file--------------------------------------------
typedef enum
{
   IPTASK_DHCP,
   IPTASK_STATIC
} E_IPTASK_COMMAND;

typedef struct
{
   E_IPTASK_COMMAND command;
   void *semaphore;
} S_IPTASK_PARAMS;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)


//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static void ConnectByStaticIP(void);
static void ConnectByDhcp(BOOLEAN wait_for_completion);
static void StartIpTask(E_IPTASK_COMMAND command, BOOLEAN wait_for_completion);
static void IpTask(void *param);

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Connect to network based on IP_MODE from NVM e.g. restore from IP NVM and connect to
 *          network
 * @param   wait_for_completion if TRUE and the mode is DHCP this function blocks until the DHCP
 *          has returned a result
 */
void AIP_ConnectToNetwork(BOOLEAN wait_for_completion)
{
   E_NET_IF_TYPE net_if;
   E_STB_IP_MODE ip_mode;
   U8BIT *essid;
   U8BIT *essid_password;
   E_IPTASK_COMMAND cmd;
   BOOLEAN connected;

   FUNCTION_START(AIP_ConnectToNetwork);

   net_if = (E_NET_IF_TYPE)APP_NvmRead(NET_IF_TYPE_NVM);
   if (net_if != NET_IF_NONE)
   {
      if (net_if == NET_IF_WIRELESS)
      {
         /* Need to connect to the wireless network before setting/acquiring an IP address */
         essid = APP_NvmReadString(ESSID_NVM);
         essid_password = APP_NvmReadString(ESSID_PASSWORD_NVM);

         STB_NWSelectInterface(NW_WIRELESS);

         connected = STB_NWConnectToAccessPoint(essid, essid_password);
      }
      else
      {
         STB_NWSelectInterface(NW_WIRED);
         connected = TRUE;
      }

      if (connected)
      {
         ip_mode = (E_STB_IP_MODE) APP_NvmRead(IP_MODE_NVM);
         if (ip_mode == IP_STATIC)
         {
            cmd = IPTASK_STATIC;
         }
         else
         {
            cmd = IPTASK_DHCP;
         }

         StartIpTask(cmd, wait_for_completion);
      }
   }

   FUNCTION_FINISH(AIP_ConnectToNetwork);
}

static void ConnectByStaticIP(void)
{
   U8BIT ip_address[4];
   U8BIT subnet_mask[4];
   U8BIT gateway[4];
   U8BIT dns[4];
   U32BIT ip_address_32;
   U32BIT subnet_mask_32;
   U32BIT gateway_32;
   U32BIT dns_32;

   FUNCTION_START(ConnectByStaticIP);

   // read ip from nvm and store in hw layer
   ip_address_32 = APP_NvmRead(IP_ADDRESS_NVM);
   subnet_mask_32 = APP_NvmRead(SUBNET_MASK_NVM);
   gateway_32 = APP_NvmRead(GATEWAY_IP_NVM);
   dns_32 = APP_NvmRead(DNS_SERVER_IP_NVM);

   ip_address[0] = (ip_address_32 & 0xFF000000) >> 24;
   ip_address[1] = (ip_address_32 & 0x00FF0000) >> 16;
   ip_address[2] = (ip_address_32 & 0x0000FF00) >> 8;
   ip_address[3] = (ip_address_32 & 0x000000FF);

   subnet_mask[0] = (subnet_mask_32 & 0xFF000000) >> 24;
   subnet_mask[1] = (subnet_mask_32 & 0x00FF0000) >> 16;
   subnet_mask[2] = (subnet_mask_32 & 0x0000FF00) >> 8;
   subnet_mask[3] = (subnet_mask_32 & 0x000000FF);

   gateway[0] = (gateway_32 & 0xFF000000) >> 24;
   gateway[1] = (gateway_32 & 0x00FF0000) >> 16;
   gateway[2] = (gateway_32 & 0x0000FF00) >> 8;
   gateway[3] = (gateway_32 & 0x000000FF);

   dns[0] = (dns_32 & 0xFF000000) >> 24;
   dns[1] = (dns_32 & 0x00FF0000) >> 16;
   dns[2] = (dns_32 & 0x0000FF00) >> 8;
   dns[3] = (dns_32 & 0x000000FF);

   STB_IPSetIPAddress(&ip_address[0]);
   STB_IPSetSubnetMask(&subnet_mask[0]);
   STB_IPSetGatewayIPAddress(&gateway[0]);
   STB_IPSetDnsServerIPAddress(&dns[0]);

   FUNCTION_FINISH(ConnectByStaticIP);
}

static void ConnectByDhcp(BOOLEAN wait_for_completion)
{
   FUNCTION_START(ConnectByDhcp);
   STB_IPGetIPByDhcp(wait_for_completion);
   FUNCTION_FINISH(ConnectByDhcp);
}

/**
 * @brief   Task Executes the required action on the selected network interface
 * @param   param Pointer to a structure of type S_IPTASK_PARAMS. The structure must be allocated by
 *          StartIpTask and freed by the task itself when it can return. The semaphore field of the
 *          S_IPTASK_PARAMS, if not NULL, must be freed by StartIpTask. This paraemeter must always
 *          be a valid pointer.
 */
static void IpTask(void *param)
{
   BOOLEAN wait;
   void *semaphore;
   E_IPTASK_COMMAND cmd;
   U32BIT time_start;
   E_NW_LINK_STATUS status;

   FUNCTION_START(IpTask);

   cmd = ((S_IPTASK_PARAMS *)param)->command;
   time_start = STB_OSGetClockMilliseconds();

   /* Wait for the link to become active (or for the timeout) */
   while (((status = STB_NWGetLinkStatus()) != NW_LINK_ACTIVE) && (STB_OSGetClockDiff(time_start) <= IP_TASK_TIMEOUT))
   {
      STB_OSTaskDelay(IP_TASK_PERIOD);
   }

   if (status == NW_LINK_ACTIVE)
   {
      switch (cmd)
      {
         case IPTASK_DHCP:
         {
            semaphore = ((S_IPTASK_PARAMS *)param)->semaphore;
            if (semaphore == NULL)
            {
               wait = FALSE;
            }
            else
            {
               wait = TRUE;
            }
            ConnectByDhcp(wait);
            if (semaphore != NULL)
            {
               STB_OSSemaphoreSignal(semaphore);
            }
            break;
         }

         case IPTASK_STATIC:
         {
            ConnectByStaticIP();
            break;
         }
      }
   }
#ifdef DEBUG_IP
   else
   {
      DBG(("%s: Could not execute command because link is not after afer %d ms", __FUNCTION__, IP_TASK_TIMEOUT));
   }
#endif

   STB_AppFreeMemory(param);

   FUNCTION_FINISH(IpTask);
}

/**
 * @brief   Starts a task that will execute the required operation on the selected network interface
 * @param   command Required operation
 * @param   wait_for_completion TRUE if the function can return only after the task has completed
 *          the reqruired operation.
 */
static void StartIpTask(E_IPTASK_COMMAND command, BOOLEAN wait_for_completion)
{
   S_IPTASK_PARAMS *params;
   void *semaphore = NULL;

   FUNCTION_START(StartIpTask);

   params = STB_AppGetMemory(sizeof(S_IPTASK_PARAMS));
   if (params != NULL)
   {
      if (wait_for_completion)
      {
         semaphore = STB_OSCreateCountSemaphore(0);
         params->semaphore = semaphore;
      }
      else
      {
         params->semaphore = NULL;
      }
      params->command = command;

      STB_OSCreateTask(IpTask, (void *)params, IP_TASK_STACK, IP_TASK_PRIO, (U8BIT *)"IpTask");

      if (wait_for_completion)
      {
         STB_OSSemaphoreWait(semaphore);
         DBG(("%s: command=%s wait_for_completion=TRUE. Completed", __FUNCTION__,
            (command == IPTASK_DHCP) ? "DHCP" : "Static"));
         STB_OSDeleteSemaphore(semaphore);
      }
   }
#ifdef DEBUG_IP
   else
   {
      DBG(("%s: failed allocating memory", __FUNCTION__));
   }
#endif

   FUNCTION_FINISH(StartIpTask);
}
