/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   PVR handling functions used by the application
 *
 * @file    ap_pvr.c
 * @date    11/06/2004
 * @author  Ocean Blue
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //debug functions

#include "stbhwos.h"
#include "stbhwav.h"
#include "stbpvrpr.h"
#include "stbhwdsk.h"

#include "stbgc.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbpvr.h"     // STB PVR
#include "stbheap.h"    // STB_AppFreeMemory
#include "stbuni.h"     // STB_ConcatUnicodeStrings()
#include "stbdsapi.h"
#include "stbpvrmsg.h"
#include "stbllist.h"
#include "stbebutt.h"
#include "stberc.h"
#include "ca_glue.h"
#include "app.h"

#ifdef INTEGRATE_HBBTV
#include "hbbtv_api.h"
#endif

#include "app_nvm.h"   // for APP_NvmReadPVRPlaylistCurrentAtTop
#include "ap_pvr.h"    // for PVR functionality
#include "ap_tmr.h"    // for timer functionality
#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_dbdef.h"
#include "ap_si.h"
#include "ap_cntrl.h"  // application stb layer control

#ifdef COMMON_INTERFACE
#include "stbci.h"
#include "stbcicc.h"
#include "stbcica.h"
#include "ap_ci.h"
#endif

#ifdef PVR_LOG
extern FILE *pvr_log;
extern void LogDateTime();
#endif

//---constant definitions for this file-------------------------------------------------------------
// enable debug printing     remove first comments
// NOTE: no brackets around x to allow for multiple line arguments
// ap_pvr.c
#define APP_PRINT_PVR_REC(x)          // STB_SPDebugWrite x   // recording
#define APP_PRINT_PVR_PAUSE(x)        // STB_SPDebugWrite x   // pause
#define APP_PRINT_PVR_PLAY_STATUS(x)  // STB_SPDebugWrite x   // play status
#define APP_PRINT_PVR_PLAY_TUNE(x)    // STB_SPDebugWrite x   // play status
#define APP_PRINT_PVR_FP(x)           // STB_SPDebugWrite x   // Freeview+ debug

// PVR constants
#define PVR_NUM_DUMMY_RECORDINGS       15

#define PVR_NORMAL_PLAY_SPEED          100

#define RECORD_WAIT_TIMEOUT            1000 // 1 seconds

#define MS_DIGIT                       0
#define LS_DIGIT                       1

#define PVR_100_PERCENT                100

#define CRID_DATE_TIMEOUT              91    /* This is the number of days that a CRID record should
                                              * remain in the CRID record list since it was last seen
                                              * in the EIT schedule data. */
#define CRID_MAX_SIZE                  64


#define PAT_PID                        0
#define CAT_PID                        1
#define EIT_PID                        18

//---local typedefs, structs, enumerations for this file--------------------------------------------
/*#define DEBUG_PVR*/

#ifdef DEBUG_PVR
#define DBG_PVR(X)    STB_SPDebugWrite X
#else
#define DBG_PVR(X)
#endif


//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)

// ---- services ----
static void *tuned_service_ptr;

// ---- play ----
static BOOLEAN inc_speed;
static BOOLEAN is_trick_mode_mute_on;
static U8BIT trick_mode_copy_volume_value;
static U32BIT playback_handle = 0;
static U8BIT playback_path = INVALID_RES_ID;
static U8BIT pause_path = INVALID_RES_ID;
static U8BIT monitor_si_path = INVALID_RES_ID;

// ---- pause ----
static U32BIT paused_recording;
static U16BIT paused_disk_id;
static U16BIT timeshift_buffer_size;

static void *eit_update_queue;
static void *process_crids_task;

static ADB_SERVICE_REC *playback_service = NULL;

static BOOLEAN encrypt_recordings;

#ifdef COMMON_INTERFACE
static U32BIT *cam_record_start_timer;
#endif

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)

// ---- services list ----

// ---- services ----

// ---- recording ----
static BOOLEAN GetRecording(U8BIT tuner_id, U8BIT **name_ptr);
static BOOLEAN DestroyRecording(U32BIT recording_handle);
static S16BIT IsPidInList(U16BIT pid, S_PVR_PID_INFO *pid_array, U16BIT num_pids);
static U16BIT FillPidList(void *service, S_PVR_PID_INFO **pid_array);
static void SetupForRecording(U8BIT path, void *service, BOOLEAN update);

// ---- playing ----

// is a recording being played
static BOOLEAN IsPlayingHandle(U32BIT *recording_handle_ptr);

static void EitUpdatedTask(void *param);
static void ProcessCridRecords(void);

static void FreePlaybackService(ADB_SERVICE_REC *service);
static ADB_SERVICE_REC* CreatePlaybackService(U16BIT service_id);
#ifdef COMMON_INTERFACE
static U32BIT NewCountdownTimer(U32BIT seconds);
#endif

//---local function definitions------------------------------------------------
/**
 *

 *
 * @brief   get live or timer recording stop name
 *
 * @param   tuner_type - tuner type (live or timerd)
 * @param   name_ptr - recording name, by reference
 *
 * @return   recording_ok - recordings ok
 *
 */
static BOOLEAN GetRecording(U8BIT path, U8BIT **name_ptr)
{
   BOOLEAN recording_ok;
   U32BIT recording_handle;

   FUNCTION_START(GetRecording);

   if (STB_DPIsRecording(path, &recording_handle))
   {
      *name_ptr = STB_PVRRecordingGetName(recording_handle);
      recording_ok = TRUE;
   }
   else
   {
      *name_ptr = NULL;
      recording_ok = FALSE;
   }

   FUNCTION_FINISH(GetRecording);

   return(recording_ok);
}

/**
 *

 *
 * @brief   destroy recording specified by recording handle
 *                (and stop any playing or recording)
 *
 * @param   recording_handle - recording handle to be destroyed
 *

 *
 */
static BOOLEAN DestroyRecording(U32BIT recording_handle)
{
   U32BIT handle;
   U8BIT path;
   BOOLEAN retval;

   FUNCTION_START(DestroyRecording);

   retval = FALSE;

   // stop recording and timers
   if (recording_handle != 0)
   {
      // stop playing
      if (IsPlayingHandle(&handle) == TRUE)
      {
         if (handle == recording_handle)
         {
            APP_PRINT_PVR_REC(("DestroyRecording: stop playing recording 0x%x", recording_handle));
            APVR_StopPlay(FALSE);
         }
      }

      // stop recording
      for (path = 0; path < STB_DPGetNumPaths(); path++)
      {
         if (STB_DPIsRecording(path, &handle))
         {
            if (handle == recording_handle)
            {
               STB_DPStopRecording(path);
               ATMR_DeleteRecordingTimer(recording_handle);
               APP_PRINT_PVR_REC(("DestroyRecording: StopRecordingType for path %d finished", path));
            }
         }
      }

      retval = STB_PVRDestroyRecording(recording_handle);
      if (retval && (recording_handle != paused_recording))
      {
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_DELETED,
            &recording_handle, sizeof(recording_handle));
      }
   }

   FUNCTION_FINISH(DestroyRecording);

   return(retval);
}

// ---- playing ----

/**
 *

 *
 * @brief   is a recording being played
 *
 * @param   recording_handle_ptr - pointer to handle of recording that is currently playing
 *
 * @return   is_playing - specified recording is playing
 *
 */
static BOOLEAN IsPlayingHandle(U32BIT *recording_handle_ptr)
{
   BOOLEAN is_playing;
   U8BIT path;

   FUNCTION_START(IsPlayingHandle);

   is_playing = FALSE;

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      if (STB_PVRIsPlaying(path, recording_handle_ptr) == TRUE)
      {
         is_playing = TRUE;
      }
   }

   FUNCTION_FINISH(IsPlayingHandle);

   return(is_playing);
}

/*!**************************************************************************
 * @brief   Checks whether the given PID is in the given list
 * @param   pid - PID to be searched for
 * @param   pid_array - array of PIDs to check
 * @param   num_pids - number of PIDs in the array
 * @return  Array index if the pid is found, -1 if not found
 ****************************************************************************/
static S16BIT IsPidInList(U16BIT pid, S_PVR_PID_INFO *pid_array, U16BIT num_pids)
{
   S16BIT i;
   S16BIT index;

   index = -1;

   for (i = 0; i < num_pids; i++)
   {
      if (pid_array[i].pid == pid)
      {
         index = i;
         break;
      }
   }

   return(index);
}

/*!**************************************************************************
 * @brief   Fills the given array with the PIDs that should be recorded for the given service
 * @param   service - the recorded service
 * @param   pid_array - pointer to an array that will be allocated and filled
 *                       with the PIDs for recording
 * @return  number of PIDs entered in the array
 ****************************************************************************/
static U16BIT FillPidList(void *service, S_PVR_PID_INFO **pid_array)
{
   U16BIT max_pids;
   U16BIT num_pids;
   S_PVR_PID_INFO *rec_pids;
   U16BIT pid;
   void **stream_list;
   U16BIT num_streams;
   U16BIT i;
   S16BIT index;
   U8BIT *pmt_data;
   U16BIT data_len;
   U16BIT *ca_pid_array;
   U16BIT num_ca_pids;

   FUNCTION_START(FillPidList);

   /* Find the maximum number of PIDs that will be recorded.
    * This may not be the number that will actually be recorded because data streams
    * won't be included in the list */
   max_pids = ADB_GetNumStreams(service, ADB_STREAM_LIST_ALL);

   pmt_data = ADB_GetServicePMTData(service, &data_len);
   if (pmt_data != NULL)
   {
      /* Get the array of PIDs the CA system wants to record, this will include ECMs */
      if ((num_ca_pids = STB_CAGetRecordingPids(pmt_data, &ca_pid_array)) > 0)
      {
         max_pids += num_ca_pids;
      }
   }
   else
   {
      num_ca_pids = 0;
   }

   /* Add the PCR PID and the number of SI/PSI PIDs to be recorded (PAT, CAT, EIT and PMT) */
   max_pids += 5;

   num_pids = 0;
   *pid_array = NULL;

   if ((rec_pids = (S_PVR_PID_INFO *)STB_AppGetMemory(max_pids * sizeof(S_PVR_PID_INFO))) != NULL)
   {
      if ((pid = ADB_GetServicePCRPid(service)) != 0)
      {
         rec_pids[num_pids].type = PVR_PID_TYPE_PCR;
         rec_pids[num_pids].pid = pid;
         num_pids++;
      }

      /* Add all video PIDs to the set of PIDs */
      ADB_GetStreamList(service, ADB_VIDEO_LIST_STREAM, &stream_list, &num_streams);
      if ((num_streams > 0) && (stream_list != NULL))
      {
         for (i = 0; i < num_streams; i++)
         {
            if ((pid = ADB_GetStreamPID(stream_list[i])) != 0)
            {
               /* Video PID is often the same as the PCR PID, so check for this and
                * add it to the list as a video PID too */
               if ((index = IsPidInList(pid, rec_pids, num_pids)) >= 0)
               {
                  if (rec_pids[index].type == PVR_PID_TYPE_PCR)
                  {
                     /* Video and PCR PIDs are the same, force the video PID to be added too */
                     index = -1;
                  }
               }

               if (index < 0)
               {
                  rec_pids[num_pids].type = PVR_PID_TYPE_VIDEO;
                  rec_pids[num_pids].pid = pid;

                  switch (ADB_GetStreamType(stream_list[i]))
                  {
                     case ADB_VIDEO_STREAM:
                        rec_pids[num_pids].u.video_codec = AV_VIDEO_CODEC_MPEG2;
                        break;

                     case ADB_H264_VIDEO_STREAM:
                        rec_pids[num_pids].u.video_codec = AV_VIDEO_CODEC_H264;
                        break;

                     case ADB_H265_VIDEO_STREAM:
                        rec_pids[num_pids].u.video_codec = AV_VIDEO_CODEC_H265;
                        break;

                     default:
                        rec_pids[num_pids].u.video_codec = AV_VIDEO_CODEC_AUTO;
                        break;
                  }

                  num_pids++;
               }
            }
         }

         ADB_ReleaseStreamList(stream_list, num_streams);
      }

      /* Add all audio PIDs to the set of PIDs */
      ADB_GetStreamList(service, ADB_AUDIO_LIST_STREAM, &stream_list, &num_streams);
      if ((num_streams > 0) && (stream_list != NULL))
      {
         for (i = 0; i < num_streams; i++)
         {
            if ((pid = ADB_GetStreamPID(stream_list[i])) != 0)
            {
               /* Audio PID can be the same as the PCR PID, so check for this and
                * add it to the list as an audio PID too */
               if ((index = IsPidInList(pid, rec_pids, num_pids)) >= 0)
               {
                  if (rec_pids[index].type == PVR_PID_TYPE_PCR)
                  {
                     /* Audio and PCR PIDs are the same, force the audio PID to be added too */
                     index = -1;
                  }
               }

               if (index < 0)
               {
                  rec_pids[num_pids].type = PVR_PID_TYPE_AUDIO;
                  rec_pids[num_pids].pid = pid;

                  switch (ADB_GetStreamType(stream_list[i]))
                  {
                     case ADB_AUDIO_STREAM:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_MP2;
                        break;

                     case ADB_AAC_AUDIO_STREAM:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_AAC;
                        break;

                     case ADB_HEAAC_AUDIO_STREAM:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_HEAAC;
                        break;

                     case ADB_AC3_AUDIO_STREAM:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_AC3;
                        break;

                     case ADB_EAC3_AUDIO_STREAM:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_EAC3;
                        break;

                     default:
                        rec_pids[num_pids].u.audio_codec = AV_AUDIO_CODEC_AUTO;
                        break;
                  }

                  num_pids++;
               }
            }
         }

         ADB_ReleaseStreamList(stream_list, num_streams);
      }

      ADB_GetStreamList(service, ADB_SUBTITLE_LIST_STREAM, &stream_list, &num_streams);
      if ((num_streams > 0) && (stream_list != NULL))
      {
         for (i = 0; i < num_streams; i++)
         {
            pid = ADB_GetStreamPID(stream_list[i]);
            if ((pid != 0) && (IsPidInList(pid, rec_pids, num_pids) < 0))
            {
               rec_pids[num_pids].type = PVR_PID_TYPE_SUBTITLES;
               rec_pids[num_pids].pid = pid;
               num_pids++;
            }
         }

         ADB_ReleaseStreamList(stream_list, num_streams);
      }

      ADB_GetStreamList(service, ADB_TTEXT_LIST_STREAM, &stream_list, &num_streams);
      if ((num_streams > 0) && (stream_list != NULL))
      {
         for (i = 0; i < num_streams; i++)
         {
            pid = ADB_GetStreamPID(stream_list[i]);
            if ((pid != 0) && (IsPidInList(pid, rec_pids, num_pids) < 0))
            {
               rec_pids[num_pids].type = PVR_PID_TYPE_TELETEXT;
               rec_pids[num_pids].pid = pid;
               num_pids++;
            }
         }

         ADB_ReleaseStreamList(stream_list, num_streams);
      }

      rec_pids[num_pids].type = PVR_PID_TYPE_SECTION;
      rec_pids[num_pids].pid = PAT_PID;
      num_pids++;

      rec_pids[num_pids].type = PVR_PID_TYPE_SECTION;
      rec_pids[num_pids].pid = CAT_PID;
      num_pids++;

      rec_pids[num_pids].type = PVR_PID_TYPE_SECTION;
      rec_pids[num_pids].pid = EIT_PID;
      num_pids++;

      pid = ADB_GetServicePmtPid(service);
      if ((pid != 0) && (pid != 0xFFFF))
      {
         rec_pids[num_pids].type = PVR_PID_TYPE_SECTION;
         rec_pids[num_pids].pid = pid;
         num_pids++;
      }

      if (num_ca_pids > 0)
      {
         for (i = 0; i < num_ca_pids; i++)
         {
            if (IsPidInList(ca_pid_array[i], rec_pids, num_pids) < 0)
            {
               rec_pids[num_pids].type = PVR_PID_TYPE_SECTION;
               rec_pids[num_pids].pid = ca_pid_array[i];
               num_pids++;
            }
         }

         STB_CAReleaseRecordingPids(ca_pid_array, num_ca_pids);
      }

      *pid_array = rec_pids;
   }

#ifdef PVR_LOG
   LogDateTime();
   fprintf(pvr_log, "FillPidList: LCN=%u, num_pids=%u\n", ADB_GetServiceLcn(service), num_pids);
   fflush(pvr_log);
#endif

   FUNCTION_FINISH(FillPidList);

   return(num_pids);
}

/*!**************************************************************************
 * @brief   Setup or update recording info, including the set of PIDs that are to be recorded
 * @param   path - recording path
 * @param   s_ptr - the service to be, or being, recorded
 * @param   update - TRUE if the recording has already been started and so
 *                   this is an update on what's being recorded
 ****************************************************************************/
static void SetupForRecording(U8BIT path, void *service, BOOLEAN update)
{
   U16BIT num_pids;
   S_PVR_PID_INFO *rec_pids;
   void *t_ptr;
   U16BIT serv_id, ts_id, onet_id;

   FUNCTION_START(SetupForRecording);

   if ((num_pids = FillPidList(service, &rec_pids)) > 0)
   {
      /* Set or update the PIDs to be recorded */
      if (update)
      {
         STB_PVRUpdateRecordingPids(path, num_pids, rec_pids);
      }
      else
      {
         STB_PVRSetRecordingPids(path, num_pids, rec_pids);

         /* Set the IDs for the service being recorded */
         t_ptr = ADB_GetServiceTransportPtr(service);
         if (t_ptr != NULL)
         {
            serv_id = ADB_GetServiceId(service);
            ts_id = ADB_GetTransportTid(t_ptr);
            onet_id = ADB_GetTransportOriginalNetworkId(t_ptr);

            STB_PVRSetRecoringTriplet(path, serv_id, ts_id, onet_id);
         }
      }

      STB_AppFreeMemory(rec_pids);
   }

   FUNCTION_FINISH(SetupForRecording);
}

//---global function definitions-----------------------------------------------

/**
 * @brief   Initialise all PVR resources: filters, recordings etc.
 */
void APVR_Initialise(void)
{
#ifdef COMMON_INTERFACE
   U8BIT i;
   U8BIT num_paths;
#endif

   FUNCTION_START(APVR_Initialise);

   inc_speed = TRUE;

   paused_recording = 0;
   paused_disk_id = INVALID_DISK_ID;
   timeshift_buffer_size = (U16BIT)APP_NvmRead(TIMESHIFT_BUFFER_SIZE_NVM);

   tuned_service_ptr = NULL;

   is_trick_mode_mute_on = FALSE;
   trick_mode_copy_volume_value = 0;

   encrypt_recordings = FALSE;

   playback_path = INVALID_RES_ID;

   eit_update_queue = STB_OSCreateQueue(sizeof(U8BIT), 5);
   if (eit_update_queue != NULL)
   {
      process_crids_task = STB_OSCreateTask(EitUpdatedTask, eit_update_queue, 4096, 7, (U8BIT *)"ProcessCrids");
   }

   /* Load and import recordings */
   STB_PVRUpdateRecordings(FALSE);

#ifdef COMMON_INTERFACE
   num_paths = STB_DPGetNumPaths();

   if ((cam_record_start_timer = (U32BIT *)STB_AppGetMemory(num_paths * sizeof(U32BIT))) != NULL)
   {
      for (i = 0; i < num_paths; i++)
      {
         cam_record_start_timer[i] = INVALID_TIMER_HANDLE;
      }
   }
#endif

   DBG_PVR(("APVR_Initialise: done"));

   FUNCTION_FINISH(APVR_Initialise);
}

/**
 * @brief   Destroy all PVR resources: filters, recordings etc.
 */
void APVR_Terminate(void)
{
   U8BIT q_msg = 0xFF;
   FUNCTION_START(APVR_Terminate);

   if (eit_update_queue != NULL)
   {
      if (process_crids_task)
      {
         do
         {
            STB_OSWriteQueue(eit_update_queue, (void *)&q_msg, sizeof(q_msg), TIMEOUT_NOW);
            STB_OSTaskDelay(8);
         }
         while (eit_update_queue != NULL);
         STB_OSTaskDelay(8);
         STB_OSDestroyTask( process_crids_task );
         process_crids_task = 0;
      }
   }

#ifdef COMMON_INTERFACE
   if (cam_record_start_timer != NULL)
   {
      STB_AppFreeMemory(cam_record_start_timer);
   }
#endif

   FUNCTION_FINISH(APVR_Terminate);
}

/**
 * @brief   Use to query whether the PVR function is available and ready to be used
 * @return  TRUE if available and ready, FALSE otherwise
 */
BOOLEAN APVR_IsInitialised(void)
{
   BOOLEAN initialised_ok;

   FUNCTION_START(APVR_IsInitialised);

   initialised_ok = STB_PVRIsInitialised();

   FUNCTION_FINISH(APVR_IsInitialised);

   return(initialised_ok);
}

/**
 * @brief   Sets the time, in seconds, when an event will be sent as a notification that a
 *          recording is about to start. This setting will apply to all recording timers that
 *          don't have a notify time already set. If this value is set to 0 then events will
 *          only be sent for recording timers that have their own notify time set.
 * @param   notify_time number of seconds before the recording is due to start when the event
 *                      will be sent. Setting 0 disables the notification.
 */
void APVR_SetNotifyTime(U16BIT notify_time)
{
   FUNCTION_START(APVR_SetNotifyTime);

   APP_NvmSave(PVR_NOTIFY_TIME_NVM, (U32BIT)notify_time, TRUE);

   FUNCTION_FINISH(APVR_SetNotifyTime);
}

/**
 * @brief   Returns the current recording notification time in seconds
 * @return  notification time in seconds, 0 means it's disabled
 */
U16BIT APVR_GetNotifyTime(void)
{
   FUNCTION_START(APVR_GetNotifyTime);
   FUNCTION_FINISH(APVR_GetNotifyTime);
   return((U16BIT)APP_NvmRead(PVR_NOTIFY_TIME_NVM));
}

/**
 * @brief   Sets the default time that will be added to the start of all recording timers created
 *          after this value is set. Positive values will start a recording before the scheduled
 *          start time and negative values will start it later than the scheduled time
 * @param   padding start padding time in seconds
 */
void APVR_SetStartPadding(S32BIT padding)
{
   FUNCTION_START(APVR_SetStartPadding);

   APP_NvmSave(RECORD_START_PADDING_NVM, (U32BIT)padding, TRUE);

   FUNCTION_FINISH(APVR_SetStartPadding);
}

/**
 * @brief   Sets the default time that will be added to the end of all recording timers created
 *          after this value is set. Positive values will end a recording after the scheduled
 *          end time and negative values will end it before the scheduled end time
 * @param   padding end padding time in seconds
 */
void APVR_SetEndPadding(S32BIT padding)
{
   FUNCTION_START(APVR_SetEndPadding);

   APP_NvmSave(RECORD_END_PADDING_NVM, (U32BIT)padding, TRUE);

   FUNCTION_FINISH(APVR_SetEndPadding);
}

/**
 * @brief   Returns the current setting for the start padding time added to all recording timers
 * @return  start padding value in seconds
 */
S32BIT APVR_GetStartPadding(void)
{
   FUNCTION_START(APVR_GetStartPadding);
   FUNCTION_FINISH(APVR_GetStartPadding);
   return((S32BIT)APP_NvmRead(RECORD_START_PADDING_NVM));
}

/**
 * @brief   Returns the current setting for the end padding time added to all recording timers
 * @return  end padding value in seconds
 */
S32BIT APVR_GetEndPadding(void)
{
   FUNCTION_START(APVR_GetEndPadding);
   FUNCTION_FINISH(APVR_GetEndPadding);
   return((S32BIT)APP_NvmRead(RECORD_END_PADDING_NVM));
}

// ----play list----

/**
 * @brief   Returns a list of all the existing recordings that can be played.
 *          All the arrays returned will be allocated by this function and
 *          should be freed by calling APVR_ReleasePlayList.
 * @param   handle_list address to return an array of recording handles
 * @param   name_list address to return an array of names for each recording,
 *          the U32BIT value will actually be a U8BIT* of a UTF-8 string
 * @param   rec_status_list address to return an array indicating whether each
 *          recording is currently being recorded, 0=no, 1=yes
 * @param   locked_list address to return an array indicating whether each
 *          recording is parentally locked, 0=no, 1=yes
 * @param   selected_list address to return an array indicating whether each
 *          recording is marked as selected, 0=no, 1=yes
 * @param   split_list address to return an array indicating whether each
 *          recording is part of a split recording, 0=no, 1=yes
 * @return  number of recordings
 */
U16BIT  APVR_GetPlayList(U32BIT **handle_list, U8BIT ***name_list, U32BIT **rec_status_list,
   U32BIT **locked_list, U32BIT **selected_list, U32BIT **split_list)
{
   U16BIT list_size;
   U16BIT index;
   U32BIT handle;
   U8BIT *crid;

   FUNCTION_START(APVR_GetPlayList);

   list_size = STB_PVRGetRecordingHandles(handle_list);
   if (list_size > 0)
   {
      /* If timeshift is in progress then the list will contain the recording for this as well,
       * so remove it if it's present */
      for (index = 0; index < list_size; index++)
      {
         if ((*handle_list)[index] == paused_recording)
         {
            /* Remove the timeshift recording from the list by moving all remaining items down 1 */
            for (index++; index < list_size; index++)
            {
               (*handle_list)[index-1] = (*handle_list)[index];
            }
            list_size--;
            break;
         }
      }

      *name_list = (U8BIT **)STB_AppGetMemory(list_size * sizeof(U8BIT*));
      *rec_status_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));
      *locked_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));
      *selected_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));
      *split_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));

      if ((*name_list != NULL) && (*rec_status_list != NULL) &&
          (*locked_list != NULL) && (*selected_list != NULL) &&
          (*split_list != NULL))
      {
         /* Copy each value into the arrays */
         for (index = 0; index < list_size; index++)
         {
            handle = (*handle_list)[index];

            (*name_list)[index] = STB_PVRRecordingGetName(handle);
            (*rec_status_list)[index] = (U32BIT)STB_PVRIsBeingRecorded(handle);
            (*locked_list)[index] = (U32BIT)STB_PVRRecordingGetLocked(handle);
            (*selected_list)[index] = (U32BIT)STB_PVRRecordingGetSelected(handle);

            crid = (U8BIT *)STB_AppGetMemory(CRID_MAX_SIZE);
            if (crid != NULL)
            {
               if (STB_PVRRecordingGetCrid(handle, crid, CRID_MAX_SIZE))
               {
                  (*split_list)[index] = (U32BIT)ADB_IsSplitProgrammeCrid(crid);
               }
               else
               {
                  (*split_list)[index] = FALSE;
               }
               STB_AppFreeMemory(crid);
            }
         }
      }
      else
      {
         STB_PVRReleaseRecordingHandles(*handle_list);
         *handle_list = NULL;

         if (*name_list != NULL)
         {
            STB_AppFreeMemory(*name_list);
            *name_list = NULL;
         }

         if (*rec_status_list != NULL)
         {
            STB_AppFreeMemory(*rec_status_list);
            *rec_status_list = NULL;
         }

         if (*locked_list != NULL)
         {
            STB_AppFreeMemory(*locked_list);
            *locked_list = NULL;
         }

         if (*selected_list != NULL)
         {
            STB_AppFreeMemory(*selected_list);
            *selected_list = NULL;
         }

         if (*split_list != NULL)
         {
            STB_AppFreeMemory(*split_list);
            *split_list = NULL;
         }

         list_size = 0;
      }
   }
   else
   {
      *handle_list = NULL;
      *name_list = NULL;
      *rec_status_list = NULL;
      *locked_list = NULL;
      *selected_list = NULL;
      *split_list = NULL;
   }

   FUNCTION_FINISH(APVR_GetPlayList);

   return(list_size);
}

/**
 * @brief   Frees the arrays allocated by APVR_GetPlayList
 * @param   handle_list array of recording handles
 * @param   name_list array of recording names
 * @param   rec_status_list array of recording status'
 * @param   locked_list array of locked status'
 * @param   selected_list array of selected status'
 */
void  APVR_ReleasePlayList(U32BIT *handle_list, U8BIT **name_list, U32BIT *rec_status_list,
   U32BIT *locked_list, U32BIT *selected_list, U32BIT *split_list)
{
   FUNCTION_START(APVR_ReleasePlayList);

   if (handle_list != NULL)
   {
      STB_PVRReleaseRecordingHandles(handle_list);
   }

   if (name_list != NULL)
   {
      STB_AppFreeMemory(name_list);
   }

   if (rec_status_list != NULL)
   {
      STB_AppFreeMemory(rec_status_list);
   }

   if (locked_list != NULL)
   {
      STB_AppFreeMemory(locked_list);
   }

   if (selected_list != NULL)
   {
      STB_AppFreeMemory(selected_list);
   }

   if (split_list != NULL)
   {
      STB_AppFreeMemory(split_list);
   }

   FUNCTION_FINISH(APVR_ReleasePlayList);
}

// ----playing----

/**
 * @brief   Starts playback of the given recording
 * @param   recording_handle handle of recorded programme to playback
 * @param   resume_playback TRUE if playback of the recording should resume from where
 *                          it was last stopped, FALSE to start from the beginning
 * @param   monitor_service service to be tuned to in order to monitor SI data whilst playback
 *                          is in progress. If this is NULL and the live path is still valid
 *                          and tuned to a service then this will be used to monitor SI data.
 *                          If neither are valid then no SI monitoring will be performed, which
 *                          may result in recordings failing to be started if they're set to
 *                          be event triggered.
 * @return  TRUE if playback starts, FALSE otherwise
 */
BOOLEAN APVR_PlayRecording(U32BIT recording_handle, BOOLEAN resume_playback, void *monitor_service)
{
   BOOLEAN playback_started;
   U8BIT path;
   U16BIT serv_id, ts_id, orig_net_id;

   FUNCTION_START(APVR_PlayRecording);

   playback_started = FALSE;

   APP_PRINT_PVR_PLAY_TUNE(("APVR_PlayRecording(0x%lx, resume=%u, monitor_service=%p)",
      recording_handle, resume_playback, monitor_service));

   if (STB_PVRIsValidHandle(recording_handle))
   {
      playback_started = TRUE;
      path = STB_DPGetLivePath();

      if (monitor_service == NULL)
      {
         if (path != INVALID_RES_ID)
         {
            monitor_service = ADB_GetTunedService(path);

            /* This is the service to return to when playback is stopped */
            tuned_service_ptr = monitor_service;
         }
         else
         {
            tuned_service_ptr = NULL;
         }
      }
      else if (path != INVALID_RES_ID)
      {
         /* This is the service to return to when playback is stopped */
         tuned_service_ptr = ADB_GetTunedService(path);
      }

      APP_PRINT_PVR_PLAY_TUNE(("APVR_PlayRecording: tuned_service_ptr = Ox%x", tuned_service_ptr));

      if (path != INVALID_RES_ID)
      {
         /* Stop the live path so playback can be started */
         ACTL_TuneOff(path);
         STB_DPReleasePath(path, RES_OWNER_NONE);
      }

      if (playback_path == INVALID_RES_ID)
      {
         /* Acquire a path to play back the recording */
         STB_PVRRecordingGetTriplet(recording_handle, &serv_id, &ts_id, &orig_net_id);
         playback_service = CreatePlaybackService(serv_id);
         playback_path = STB_DPAcquirePlaybackPath((void *)playback_service);
         APP_PRINT_PVR_PLAY_TUNE(("Acquired playback path %u", playback_path));
      }

      if (playback_path != INVALID_RES_ID)
      {
         /* set decoder source */
         STB_PVRStartPlayRunning(playback_path);

         playback_handle = recording_handle;

         playback_started = STB_PVRStartPlaying(playback_path, recording_handle, resume_playback);
         if (playback_started)
         {
            //ACTL_StartDecoding(playback_path, playback_service);  // FIXME: audio/video/pcr are not available yet
            //         STB_AVBlankVideo(0, FALSE);

            APP_PRINT_PVR_PLAY_TUNE(("Setting APP_SI_MODE_UPDATE on playback path %u", playback_path));
            ASI_SetAppSiMode(playback_path, APP_SI_MODE_UPDATE);
            STB_DPSetSearchMode(playback_path, FALSE);
            STB_DPStartSI(playback_path);

            if (monitor_service != NULL)
            {
               /* Acquire a new path to monitor SI data */
               monitor_si_path = ACTL_AcquirePathForService(monitor_service, FALSE, FALSE, NULL);
               APP_PRINT_PVR_PLAY_TUNE(("Path for monitoring SI data=%u", monitor_si_path));
               if (monitor_si_path != INVALID_RES_ID)
               {
                  ACTL_TuneToService(monitor_si_path, NULL, monitor_service, TRUE, FALSE);
               }
            }
         }
         else
         {
            playback_handle = 0;

            STB_DPReleasePath(playback_path, RES_OWNER_NONE);
            playback_path = INVALID_RES_ID;

            FreePlaybackService(playback_service);
            playback_service = NULL;

            /* Return to viewing the previous service */
            if (tuned_service_ptr != NULL)
            {
               ACTL_TuneToService(INVALID_RES_ID, NULL, tuned_service_ptr, FALSE, TRUE);
               tuned_service_ptr = NULL;
            }
         }
      }
   }

   FUNCTION_FINISH(APVR_PlayRecording);

   return(playback_started);
}

/**
 * @brief   Returns the handle of the recording currently being played back
 * @return  playback handle, which may be invalid if playback isn't taking place
 */
U32BIT APVR_GetPlaybackHandle(void)
{
   FUNCTION_START(APVR_GetPlaybackHandle);
   FUNCTION_FINISH(APVR_GetPlaybackHandle);
   return(playback_handle);
}

/**
 * @brief   Returns the PVR play status
 * @return  TRUE if a recoding is being played, FALSE otherwise
 */
BOOLEAN APVR_IsDecodingFile(void)
{
   BOOLEAN is_playing_file;
   U8BIT path;
   U32BIT recording_handle;

   FUNCTION_START(APVR_IsDecodingFile);

   is_playing_file = FALSE;

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      if (STB_PVRIsPlaying(path, &recording_handle) == TRUE)
      {
         is_playing_file = TRUE;
         APP_PRINT_PVR_PLAY_STATUS(("APVR_IsDecodingFile: STB_PVRIsPlaying(%u,...) returned TRUE", path));
      }
      else
      {
         APP_PRINT_PVR_PLAY_STATUS(("APVR_IsDecodingFile: STB_PVRIsPlaying(%u,...) returned FALSE", path));
      }
   }

   FUNCTION_FINISH(APVR_IsDecodingFile);

   return(is_playing_file);
}

/**
 * @brief   Returns whether playback is in progress
 * @return  TRUE if playback has been started, FALSE otherwise
 */
BOOLEAN APVR_IsPlaying(void)
{
   BOOLEAN is_playing;

   FUNCTION_START(APVR_IsPlaying);

   is_playing = FALSE;

   if (playback_path != INVALID_RES_ID)
   {
      if (STB_PVRIsPlayAudio(playback_path))
      {
         is_playing = TRUE;
      }
   }

   FUNCTION_FINISH(APVR_IsPlaying);

   return(is_playing);
}

/**
 * @brief   Returns the playback progress in time and as a percentage
 * @param   handle recording handle being played
 * @param   hours pointer for return of playback hours
 * @param   mins pointer for return of playback minutes
 * @param   secs pointer for return of playback seconds
 * @param   progress pointer for return of position in playback as a percentage
 * @return  TRUE if values are returned
 */
BOOLEAN APVR_GetPlaybackElapsedTime(U32BIT handle, U8BIT *hours, U8BIT *mins, U8BIT *secs, U8BIT *progress)
{
   BOOLEAN retval;
   U8BIT path;
   U8BIT audio_decoder, video_decoder;
   U32BIT space_used;
   U8BIT length_hour;
   U8BIT length_min;
   U8BIT length_sec;

   FUNCTION_START(APVR_GetPlaybackElapsedTime);

   retval = FALSE;

   path = APVR_GetPlaybackPath();
   if (path != INVALID_RES_ID)
   {
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);

      if ((audio_decoder == INVALID_RES_ID) ||
          !STB_PVRGetElapsedTime(audio_decoder, video_decoder, hours, mins, secs))
      {
         *hours = 0;
         *mins = 0;
         *secs = 0;
      }
      else
      {
         if (STB_PVRRecordingGetLength(handle, &length_hour, &length_min, &length_sec, &space_used))
         {
            U32BIT current = *hours * 3600 + *mins * 60 + *secs;
            U32BIT duration = length_hour * 3600 + length_min * 60 + length_sec;
               
            if ((length_hour > 0) || (length_min > 0) || (length_sec > 0))
            {
               /* Calculate the progress as a percentage */
               *progress = ((*hours * 3600 + *mins * 60 + *secs) * 100) /
                  (length_hour * 3600 + length_min * 60 + length_sec);
            }
            else
            {
               *progress = 0;
            }

            if ((current >= 0) && (current <= duration))
               retval = TRUE;
         }
      }
   }

   FUNCTION_FINISH(APVR_GetPlaybackElapsedTime);

   return(retval);
}

/**
 * @brief   If playback has been paused or is subject to some trick mode, calling this
 *          function will result in playback resuming at normal playback speed (i.e. 100%)
 */
void APVR_NormalPlay(void)
{
   FUNCTION_START(APVR_NormalPlay);

   if (playback_path != INVALID_RES_ID)
   {
      APVR_TrickModeMuteOff();

      // set to increment speed for FF
      inc_speed = TRUE;
      STB_PVRPlayNormal(playback_path);
      APP_PRINT_PVR_PLAY_STATUS(("Normal Play"));
   }

   FUNCTION_FINISH(APVR_NormalPlay);
}

/**
 * @brief   Stops playback and optionally returns to live TV
 * @param   return_to_live TRUE to return to live TV after stopping playback
 */
void APVR_StopPlay(BOOLEAN return_to_live)
{
   U32BIT recording_handle;
   U8BIT path;

   FUNCTION_START(APVR_StopPlay);

   APVR_TrickModeMuteOff();

   APP_PRINT_PVR_PLAY_TUNE(("Stop Play: recording 0x%x", recording_handle));

   /* Stop playback and release the decoder being used */
   if (playback_path != INVALID_RES_ID)
   {
      /* Stop the SI whether the pvr is playing or not, as this function is called also after EOF,
         if the user goes back to live */
      STB_DPStopSI(playback_path);

      STB_PVRSavePlayPosition(playback_path);

      /* If playback is in progress then stop it */
      ACTL_DecodeOff(playback_path);
      if (STB_PVRIsPlaying(playback_path, &recording_handle) == TRUE)
      {
         STB_PVRStopPlaying(playback_path);
      }

      STB_DPReleasePath(playback_path, RES_OWNER_NONE);
      playback_path = INVALID_RES_ID;

      FreePlaybackService(playback_service);
      playback_service = NULL;
   }

   if (monitor_si_path != INVALID_RES_ID)
   {
      ACTL_TuneOff(monitor_si_path);
      STB_DPReleasePath(monitor_si_path, RES_OWNER_NONE);
      monitor_si_path = INVALID_RES_ID;
   }

   STB_AVBlankVideo(0, TRUE);

   if (return_to_live)
   {
      if (tuned_service_ptr != NULL)
      {
         if (!ACTL_CanServiceBeViewed(tuned_service_ptr))
         {
            /* All tuners are being used and the service being watched when playback was started
             * can no longer be tuned to, so need to tune to one that can */
            tuned_service_ptr = NULL;
            for (path = 0; (path < STB_DPGetNumPaths()) && (tuned_service_ptr == NULL); path++)
            {
               tuned_service_ptr = ADB_GetTunedService(path);
            }
         }

         APP_PRINT_PVR_PLAY_TUNE(("APVR_StopPlay: returning to service = Ox%x", tuned_service_ptr));
         if (tuned_service_ptr != NULL)
         {
            ACTL_TuneToService(INVALID_RES_ID, NULL, tuned_service_ptr, FALSE, TRUE);
            tuned_service_ptr = NULL;
         }
      }
   }

   FUNCTION_FINISH(APVR_StopPlay);
}

/**
 * @brief   Pause playback
 */
void APVR_PausePlay(void)
{
   U8BIT path;

   FUNCTION_START(APVR_PausePlay);

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      if (STB_PVRIsPlayPause(path) == TRUE)
      {
         APP_PRINT_PVR_PLAY_STATUS(("Pause Play(%d): already paused so FrameInc", path));
         STB_PVRPlayFrameInc(path);
      }
      else
      {
         APP_PRINT_PVR_PLAY_STATUS(("Pause Play(%d): ", path));
         STB_PVRPlayPause(path);
      }
   }

   FUNCTION_FINISH(APVR_PausePlay);
}

/**
 * @brief   If paused, playback will go into the first available slow motion speed
 *          in a forwards direction. If speed is already > 0% then the next fast
 *          forward speed will be selected. If playback is in the reverse direction
 *          then the next slowest reverse play speed will be selected.
 */
void APVR_FFPlay(void)
{
   S16BIT play_speed;
   U8BIT path;

   FUNCTION_START(APVR_FFPlay);

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      APVR_TrickModeMuteOn();

      if (STB_PVRIsPlayPause(path) == TRUE)
      {
         APP_PRINT_PVR_PLAY_STATUS(("Fast Forward(%d): Paused so go into slow motion play", path));
         STB_PVRPlayFaster(path, TRUE);
      }
      else
      {
         play_speed = STB_PVRGetPlaySpeed(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));

         if ((play_speed >= PVR_NORMAL_PLAY_SPEED) ||
             (play_speed <= -PVR_NORMAL_PLAY_SPEED))
         {
            APP_PRINT_PVR_PLAY_STATUS(("Fast Forward(%d): Not slow motion, play faster", path));
            STB_PVRPlayFaster(path, FALSE);
         }
         else
         {
            /* Playback in slow motion so allow slow motion speeds */
            APP_PRINT_PVR_PLAY_STATUS(("Fast Forward(%d): Slow motion, play faster", path));
            STB_PVRPlayFaster(path, TRUE);
         }
      }
   }

   FUNCTION_FINISH(APVR_FFPlay);
}

/**
 * @brief   If paused, playback will go into the first available slow motion speed
 *          in a reverse direction. If speed is already < 0% then the next reverse
 *          speed will be selected. If playback is in the forwards direction then
 *          the next slowest forwards play speed will be selected.
 */
void APVR_FRPlay(void)
{
   S16BIT play_speed;
   U8BIT path;

   FUNCTION_START(APVR_FRPlay);

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      APVR_TrickModeMuteOn();

      if (STB_PVRIsPlayPause(path) == TRUE)
      {
         APP_PRINT_PVR_PLAY_STATUS(("Fast Rewind(%d): Paused so go into slow motion play", path));
         STB_PVRPlaySlower(path, TRUE);
      }
      else
      {
         play_speed = STB_PVRGetPlaySpeed(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));
         APP_PRINT_PVR_PLAY_STATUS(("Fast Rewind(%d): speed=%d", play_speed));

         if ((play_speed >= PVR_NORMAL_PLAY_SPEED) ||
             (play_speed <= -PVR_NORMAL_PLAY_SPEED))
         {
            APP_PRINT_PVR_PLAY_STATUS(("Fast Rewind(%d): Not slow motion, play slower", path));
            STB_PVRPlaySlower(path, FALSE);
         }
         else
         {
            /* Playback in slow motion so allow slow motion speeds */
            APP_PRINT_PVR_PLAY_STATUS(("Fast Rewind(%d): Slow motion, play slower", path));
            STB_PVRPlaySlower(path, TRUE);
         }
      }
   }

   FUNCTION_FINISH(APVR_FRPlay);
}

/**
 * @brief   Play in slow mothing mode
 */
void APVR_SlowMoPlay(void)
{
   U8BIT path;

   FUNCTION_START(APVR_SlowMoPlay);

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      APVR_TrickModeMuteOn();

      if (inc_speed == TRUE)
      {
         APP_PRINT_PVR_PLAY_STATUS(("Slow Motion(%d): play medium and inc_speed = FALSE", path));
         STB_PVRPlayMedium(path);
         inc_speed = FALSE;
      }

      if (STB_PVRIsPlayForward(path))
      {
         APP_PRINT_PVR_PLAY_STATUS(("Slow Motion(%d): APVR_FFPlay()", path));
         APVR_FFPlay();
      }
      else
      {
         APP_PRINT_PVR_PLAY_STATUS(("Slow Motion(%d): APVR_FRPlay()", path));
         APVR_FRPlay();
      }
   }

   FUNCTION_FINISH(APVR_SlowMoPlay);
}

/**
 *

 *
 * @brief   quick replay current playing recording
 *

 *

 *
 */
void APVR_QRPlay(void)
{
   FUNCTION_START(APVR_QRPlay);

   // TBD this function requires further work on STB layer
   APP_PRINT_PVR_PLAY_STATUS(("Quick Replay: TBD - so do nothing"));

   FUNCTION_FINISH(APVR_QRPlay);
}

/**
 *

 *
 * @brief   quick replay current playing recording
 *

 *

 *
 */
void APVR_JTLPlay(void)
{
   FUNCTION_START(APVR_JTLPlay);

   // TBD this function requires further work on STB layer
   APP_PRINT_PVR_PLAY_STATUS(("Jump To Live: TBD so do nothing"));

   FUNCTION_FINISH(APVR_JTLPlay);
}

/**
 * @brief   Returns the current play mode according to the current play speed
 * @return  play mode
 */
E_PVR_PLAY_STATUS APVR_GetPlayMode(void)
{
   U8BIT audio_decoder, video_decoder;
   E_PVR_PLAY_STATUS play_mode;
   S16BIT play_speed;
   U8BIT path;

   FUNCTION_START(APVR_GetPlayMode);

   play_mode = PVR_PLAY_STATUS_NULL;

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      audio_decoder = STB_DPGetPathAudioDecoder(path);
      video_decoder = STB_DPGetPathVideoDecoder(path);

      if (STB_PVRIsPlayStarted(audio_decoder, video_decoder))
      {
         play_speed = STB_PVRGetPlaySpeed(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));

         if (play_speed == 100)
         {
            play_mode = PVR_PLAY_STATUS_NORMAL;
         }
         else if (play_speed > PVR_NORMAL_PLAY_SPEED)
         {
            play_mode = PVR_PLAY_STATUS_FF;
         }
         else if (play_speed < -PVR_NORMAL_PLAY_SPEED)
         {
            play_mode = PVR_PLAY_STATUS_FR;
         }
         else if ((play_speed > 0) && (play_speed < PVR_NORMAL_PLAY_SPEED))
         {
            play_mode = PVR_PLAY_STATUS_SF;
         }
         else if ((play_speed < 0) && (play_speed > -PVR_NORMAL_PLAY_SPEED))
         {
            play_mode = PVR_PLAY_STATUS_SR;
         }
         else if (play_speed == 0)
         {
            play_mode = PVR_PLAY_STATUS_PAUSE;
         }
      }
      else
      {
         play_mode = PVR_PLAY_STATUS_STOP;
      }
   }

   FUNCTION_FINISH(APVR_GetPlayMode);

   return(play_mode);
}

/**
 * @brief   Returns the current play speed as a signed percentage value representing
 *          the speed, where 100% is normal playback speed.
 * @return  play speed
 */
S16BIT APVR_GetPlaySpeed(void)
{
   S16BIT play_speed;
   U8BIT path;

   FUNCTION_START(APVR_GetPlaySpeed);

   play_speed = 0;

   if (playback_path != INVALID_RES_ID)
   {
      path = playback_path;
   }
   else
   {
      path = STB_DPGetLivePath();
   }

   if (path != INVALID_RES_ID)
   {
      play_speed = STB_PVRGetPlaySpeed(STB_DPGetPathAudioDecoder(path), STB_DPGetPathVideoDecoder(path));
   }

   FUNCTION_FINISH(APVR_GetPlaySpeed);

   return(play_speed);
}

/**
 * @brief   Set current play speed as a signed percentage value representing
 *          the speed, where 100% is normal playback speed.
 * @param   speed Play speed as a percentage
 * @return  TRUE if successful
 */
BOOLEAN APVR_SetPlaySpeed(S16BIT speed)
{
   BOOLEAN result;

   FUNCTION_START(APVR_SetPlaySpeed);

   if (playback_path != INVALID_RES_ID)
   {
      result = STB_PVRSetPlaySpeed(STB_DPGetPathAudioDecoder(playback_path),
         STB_DPGetPathVideoDecoder(playback_path), speed);
#ifdef INTEGRATE_HBBTV
      if (result)
      {
         HBBTV_NotifyPlaySpeedChanged(speed);
      }
#endif
   }
   else
   {
      result = FALSE;
   }

   FUNCTION_FINISH(APVR_SetPlaySpeed);
   return result;
}

/**
 * @brief   Mute for trick mode
 */
void APVR_TrickModeMuteOn(void)
{
   U8BIT audio_decoder;

   FUNCTION_START(APVR_TrickModeMuteOn);

   if (!is_trick_mode_mute_on && (playback_path != INVALID_RES_ID))
   {
      audio_decoder = STB_DPGetPathAudioDecoder(playback_path);
      if (audio_decoder != INVALID_RES_ID)
      {
         is_trick_mode_mute_on = TRUE;
         trick_mode_copy_volume_value = STB_AVGetAudioVolume(audio_decoder);
         STB_AVSetAudioVolume(audio_decoder, ((trick_mode_copy_volume_value * 3) / 4));
      }
   }

   FUNCTION_FINISH(APVR_TrickModeMuteOn);
}

/**
 * @brief   Unmute for trick mode
 */
void APVR_TrickModeMuteOff(void)
{
   U8BIT audio_decoder;

   FUNCTION_START(APVR_TrickModeMuteOff);

   if (is_trick_mode_mute_on && (playback_path != INVALID_RES_ID))
   {
      audio_decoder = STB_DPGetPathAudioDecoder(playback_path);
      if (audio_decoder != INVALID_RES_ID)
      {
         is_trick_mode_mute_on = FALSE;
         STB_AVSetAudioVolume(audio_decoder, trick_mode_copy_volume_value);
         trick_mode_copy_volume_value = 0;
      }
   }

   FUNCTION_FINISH(APVR_TrickModeMuteOff);
}

// ---- recording ----

/**
 * @brief   Acquires a decode path for recording the given service and tunes to it
 * @param   onet_id original network id for the service
 * @param   trans_id transport id for the service
 * @param   service_id service id for the service
 * @param   new_tuned_service returns TRUE if tuning is started.
 *          FALSE means the path is already tuned to the service
 * @return  ID of path or INVALID_RES_ID on failure
 */
U8BIT APVR_PrepareNewRecording(U16BIT onet_id, U16BIT trans_id, U16BIT service_id, BOOLEAN *new_tuned_service)
{
   U8BIT path;
   void *s_ptr;
   S_ACTL_OWNER_INFO owner_info;

   FUNCTION_START(APVR_PrepareNewRecording);

   path = INVALID_RES_ID;
   *new_tuned_service = FALSE;

   s_ptr = ADB_FindServiceByIds(onet_id, trans_id, service_id);

   APP_PRINT_PVR_REC(("APVR_PrepareNewRecording(onet=0x%04x, tid=0x%04x, sid=0x%04x): preparing recording on service %p",
                      onet_id, trans_id, service_id, s_ptr));

   if (s_ptr != NULL)
   {
      owner_info.owner = RES_OWNER_DVB;
      owner_info.data = NULL;
      owner_info.data_size = 0;

      if ((path = ACTL_TuneToService(INVALID_RES_ID, &owner_info, s_ptr, FALSE, FALSE)) != INVALID_RES_ID)
      {
         *new_tuned_service = !ACTL_IsTuned(path);
      }
   }

   FUNCTION_FINISH(APVR_PrepareNewRecording);

   return(path);
}

/**
 * @brief   Starts a recording after any tuning has completed and sets the info to be stored with it
 * @param   disk_id disk to record onto, default disk will be used if given disk id is invalid
 * @param   path decode path to use for recording
 * @param   recording_name name to be given to recording
 * @param   event_id event id which will be used to obtain info to save with the recording
 * @param   prog_crid programme crid (only relevant for systems that broadcast CRIDs)
 * @param   other_crid series or recommendation crid
 * @param   rec_handle pointer to returned recording handle
 * @return  TRUE if the recording is started successfully, FALSE otherwise
 */
BOOLEAN APVR_StartNewRecording(U16BIT disk_id, U8BIT path, U8BIT *recording_name, U16BIT event_id,
   U8BIT *prog_crid, U8BIT *other_crid, U32BIT *rec_handle)
{
   BOOLEAN recording_started_ok;
   void *service;
   void *event_ptr;
   U8BIT *guidance;
   U8BIT guidance_type;
   U8BIT guidance_mode;
   U8BIT *name;
#ifdef COMMON_INTERFACE
   U8BIT slot_id;
   U16BIT service_id;
   U8BIT *pmt_data;
   U16BIT data_len;
   U8BIT uri[CIP_URI_LEN];
   S32BIT cam_pin;
   U8BIT pin_str[CIP_MAX_PIN_LENGTH + 1];
#endif
#if 0
   U8BIT *pvr_err_msg;
   U16BIT rec_date;
   U8BIT rec_hour, rec_min, rec_secs;
   BOOLEAN reverse;
#endif

   FUNCTION_START(APVR_StartNewRecording);

   recording_started_ok = FALSE;

   // start a recording:
   if (recording_name == NULL)
   {
      APP_PRINT_PVR_REC(("APVR_StartNewRecording(%d): Start Record (name!)", path));
   }
   else
   {
      APP_PRINT_PVR_REC(("APVR_StartNewRecording(%d): Start Record (%s)", path, recording_name));
   }

   /* Can the disk be used for the recording? */
   if (!STB_PVRCanDiskBeUsed(disk_id))
   {
      /* Disk is no longer valid so use the default disk */
      disk_id = STB_PVRGetDefaultDisk();
   }

   if (STB_PVRCanDiskBeUsed(disk_id))
   {
      // create and start recording
      if (STB_PVRCreateRecording(disk_id, recording_name, rec_handle) == TRUE)
      {
         APP_PRINT_PVR_REC(("APVR_StartNewRecording: Create Recording"));

         /* Check whether the recording should be encrypted */
         service = ADB_GetTunedService(path);
         event_ptr = NULL;

         if ((event_id == 0) || ((event_ptr = ADB_GetEvent(service, event_id)) == NULL))
         {
            /* No event specified, so check the now event */
            ADB_GetNowNextEvents(service, &event_ptr, NULL);
         }

         if (!encrypt_recordings && (STB_DPGetPathCISlot(path) == INVALID_RES_ID))
         {
            /* Recording without a CI slot, so check the event or service to see
             * whether the recording should be encrypted */
            if (event_ptr != NULL)
            {
               if (!ADB_GetEventDoNotScramble(event_ptr))
               {
                  DBG_PVR((">> This recording(event) needs to be encrypted\n"));
                  /* Recording must be encrypted */
                  STB_PVREncryptRecording(path, TRUE);
               }
               else
               {
                  STB_PVREncryptRecording(path, FALSE);
               }
            }
            else
            {
               /* Check whether the service is protected */
               if (!ADB_GetServiceDoNotScramble(service))
               {
                  DBG_PVR((">> This recording(service) needs to be encrypted\n"));
                  /* Recording must be encrypted */
                  STB_PVREncryptRecording(path, TRUE);
               }
               else
               {
                  STB_PVREncryptRecording(path, FALSE);
               }
            }
         }
         else
         {
            /* Recording needs to be encrypted, which could be because a CI slot was acquired
             * for the recording */
            STB_PVREncryptRecording(path, TRUE);
         }

         /* Setup the PIDs, etc, that are to be recorded */
         if (service)
         {
            SetupForRecording(path, service, FALSE);
         }

         if (STB_DPStartRecording(path, *rec_handle))
         {
#ifdef PVR_LOG
            LogDateTime();
            fprintf(pvr_log, "Start recording \"%s\", event id %u, path %u",
               recording_name, event_id, path);

            if (service != NULL)
            {
               name = ADB_GetServiceFullName(service, TRUE);
               fprintf(pvr_log, ", on %s", name);
               STB_ReleaseUnicodeString(name);
            }
            if ((prog_crid != NULL) && (strlen((char *)prog_crid) > 0))
            {
               fprintf(pvr_log, ", crid=%s", (char *)prog_crid);
            }
            fprintf(pvr_log, ", handle=0x%08lx\n", *rec_handle);
            fflush(pvr_log);
#endif
            APP_PRINT_PVR_REC(("APVR_StartNewRecording: recording 0x%x, \"%s\" started on path %d",
                               *rec_handle, recording_name, path));

            recording_started_ok = TRUE;

            if (service != NULL)
            {
               /* Store the short service name with the recording */
               name = ADB_GetServiceFullName(service, FALSE);
               STB_PVRRecordingSetServiceName(*rec_handle, name);
               STB_ReleaseUnicodeString(name);

               /* Set parental lock on the recording if the service is locked */
               STB_PVRRecordingSetParentalLock(*rec_handle, ADB_GetServiceLockedFlag(service));

               if ((prog_crid != NULL) && (strlen((char *)prog_crid) > 0))
               {
                  STB_PVRRecordingSetCrid(*rec_handle, prog_crid);
               }

               if ((other_crid != NULL) && (strlen((char *)other_crid) > 0))
               {
                  STB_PVRRecordingSetOtherCrid(*rec_handle, other_crid);
               }

               if (event_ptr != NULL)
               {
                  /* Set recording name to the name of the event if the recording name isn't given */
                  if ((recording_name != NULL) && !STB_IsStringEmpty(recording_name))
                  {
                     STB_PVRRecordingSetName(*rec_handle, recording_name);
                  }
                  else
                  {
                     STB_PVRRecordingSetName(*rec_handle, ADB_GetEventName(event_ptr));
                  }

                  /* Find the event description and save it with the recording */
                  STB_PVRRecordingSetDescription(*rec_handle, ADB_GetEventDescription(event_ptr));

                  STB_PVRRecordingSetExtendedDescription(*rec_handle,
                     ADB_GetEventExtendedDescription(event_ptr));

                  /* Only save guidance if there is some for the event */
                  guidance = ADB_GetEventGuidance(event_ptr, service, &guidance_type, &guidance_mode);
                  if (guidance != NULL)
                  {
                     STB_PVRRecordingSetGuidance(*rec_handle, guidance);
                  }
               }

#ifdef COMMON_INTERFACE
               slot_id = STB_DPGetPathCISlot(path);
               if (slot_id != INVALID_RES_ID)
               {
                  /* Need to ask CAM if the recording can proceed, so pause the recording
                   * and resume it when a reply is received */
                  if (STB_PVRPauseRecording(path))
                  {
                     service_id = ADB_GetServiceId(service);

                     if (STB_CiCcAuthenticated(slot_id))
                     {
                        pmt_data = ADB_GetServicePMTData(service, &data_len);
                        if (STB_CiCaSystemSupported(slot_id, pmt_data))
                        {
                           /* Service is scrambled, CAS supported and CAM is CI+, so
                            * need to get the initial URI and store it with the recording */
                           STB_CiCcGetUsageRulesInfo(slot_id, service_id, uri);
                           STB_PVRRecordingAddURI(*rec_handle, uri);
                        }
                     }

                     /* Get the pin for the slot */
                     cam_pin = ACI_ReadPinForSlot(slot_id);
                     if (cam_pin < 0)
                     {
                        /* There's no pin for the slot */
                        pin_str[0] = '\0';
                     }
                     else
                     {
                        sprintf((char *)pin_str, "%lu", (unsigned long)cam_pin);
                     }

                     /* Start a timer that will stop the recording if the CAM doesn't respond */
                     cam_record_start_timer[path] = NewCountdownTimer(10);

                     if (!STB_CiCcSendRecordStart(slot_id, service_id, pin_str))
                     {
                        /* Failed to notify the CAM, so recording can't continue */
                        ATMR_DeleteTimer(cam_record_start_timer[path]);
                        cam_record_start_timer[path] = INVALID_TIMER_HANDLE;

                        STB_PVRStopRecording(path);
                        recording_started_ok = FALSE;
                     }
                  }
                  else
                  {
                     /* Failed to pause recording so recording must not continue */
                     STB_PVRStopRecording(path);
                     recording_started_ok = FALSE;
                     APP_PRINT_PVR_REC(("APVR_StartNewRecording: Failed to pause recording"));
                  }
               }
#endif
            }

            /* The SI is needed on the recording path too (it can be running already if we're starting a live recording */
            APP_PRINT_PVR_REC(("APVR_StartNewRecording: Setting APP_SI_MODE_UPDATE on record path %u", path));
            ASI_SetAppSiMode(path, APP_SI_MODE_UPDATE);
            STB_DPSetSearchMode(path, FALSE);
            STB_DPStartSI(path);
         }
         else
         {
#ifdef PVR_LOG
            LogDateTime();
            fprintf(pvr_log, "Failed to start recording \"%s\", event id %u, path %u",
               recording_name, event_id, path);
            if (service != NULL)
            {
               name = ADB_GetServiceFullName(service, TRUE);
               fprintf(pvr_log, ", on %s", name);
               STB_ReleaseUnicodeString(name);
            }
            fprintf(pvr_log, "\n");
            fflush(pvr_log);
#endif
            APP_PRINT_PVR_REC(("APVR_StartNewRecording: failed to start recording 0x%x, \"%s\", path %d",
                               *rec_handle, recording_name, path));

            STB_PVRDestroyRecording(*rec_handle);
         }

         if (event_ptr != NULL)
         {
            ADB_ReleaseEventData(event_ptr);
         }

         if (recording_started_ok)
         {
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_STARTED,
               &path, sizeof(path));
         }
      }
      else
      {
         APP_PRINT_PVR_REC(("APVR_StartNewRecording: Failed to create recording"));
      }
   }
   else
   {
#ifdef PVR_LOG
      LogDateTime();
      fprintf(pvr_log, "Failed to start recording \"%s\", event id %u, path %u, no disk",
         recording_name, event_id, path);
      fprintf(pvr_log, "\n");
      fflush(pvr_log);
#endif
      APP_PRINT_PVR_REC(("APVR_StartNewRecording: No disk available to record onto"));

#if 0
      STB_GCGetGMTDateTime(&rec_date, &rec_hour, &rec_min, &rec_secs);

      pvr_err_msg = STB_FormatUnicodeString(FALSE, &reverse,
            UI_GetString(STR_PVR_REC_FAIL_NO_DISK), recording_name,
            UI_GetString(days_of_week_str_ids[STB_GCGetDateWeekDay((U16BIT)rec_date)]),
            STB_GCGetDateString((U16BIT)rec_date, (U8BIT)rec_hour, (U8BIT)rec_min, DATE_DMY),
            STB_GCGetTimeString((U16BIT)rec_date, (U8BIT)rec_hour, (U8BIT)rec_min, TIME_24H));

      if (pvr_err_msg != NULL)
      {
         STB_PVRAddMessage(pvr_err_msg);
         STB_ReleaseUnicodeString(pvr_err_msg);
      }
#endif
   }

   FUNCTION_FINISH(APVR_StartNewRecording);

   return(recording_started_ok);
}

/**
 * @brief   Stops the given recording
 * @param   recording_handle recording to be stopped
 * @return  TRUE if recording stopped ok, FALSE otherwise
 */
BOOLEAN APVR_StopRecording(U32BIT recording_handle)
{
   U8BIT path;
   BOOLEAN recording_stopped;
   U32BIT handle;
#ifdef COMMON_INTERFACE
   U8BIT slot_id;
#endif

   FUNCTION_START(APVR_StopRecording);

   recording_stopped = FALSE;

   /* Check whether the recording_handle is associated with any of the tuners */
   for (path = 0; path < STB_DPGetNumPaths(); path++)
   {
      if (STB_DPIsRecording(path, &handle) && (handle == recording_handle))
      {
#ifdef COMMON_INTERFACE
         if ((slot_id = STB_DPGetPathCISlot(path)) != INVALID_RES_ID)
         {
            STB_CiCcSendRecordStop(slot_id);
         }
#endif

         APP_PRINT_PVR_REC(("APVR_StopRecording: recording 0x%x is using path %d", recording_handle, path));

         if (!STB_DPIsLivePath(path))
         {
            STB_DPStopSI(path);
         }

         STB_DPStopRecording(path);
         ATMR_DeleteRecordingTimer(recording_handle);

         recording_stopped = TRUE;

         ACTL_TuneOff(path);
         STB_DPReleasePath(path, RES_OWNER_DVB);

         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_STOPPED,
            &handle, sizeof(handle));
         break;
      }
   }

   FUNCTION_FINISH(APVR_StopRecording);

   return(recording_stopped);
}

/**
 * @brief   Returns an array of recordings currently in progress.
 *          Use APVR_ReleaseActiveRecordingList to free the returned array.
 * @param   handle_list address of an array allocated by the function containing
 *          the recording handles
 * @return  number of recordings in progress
 */
U8BIT APVR_GetActiveRecordingList(U32BIT **handle_list)
{
   U8BIT list_size = 0;
   U8BIT num_paths;
   U8BIT path;
   BOOLEAN is_recording;
   U32BIT recording_handle;
   U8BIT index;

   FUNCTION_START(APVR_GetActiveRecordingList);

   /* Calculate list_size */
   num_paths = STB_DPGetNumPaths();
   for (path = 0; path < num_paths; path++)
   {
      is_recording = STB_DPIsRecording(path, &recording_handle);

      if (is_recording == TRUE)
      {
         list_size++;
      }
   }

   if (list_size > 0)
   {
      *handle_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));

      if (*handle_list != NULL)
      {
         index = 0;
         for (path = 0; path < num_paths; path++)
         {
            is_recording = STB_DPIsRecording(path, &recording_handle);

            if (is_recording == TRUE)
            {
               if (index < list_size)
               {
                  *handle_list[index] = recording_handle;
               }
            }
         }
      }
      else
      {
         list_size = 0;
      }
   }
   else
   {
      *handle_list = NULL;
   }

   FUNCTION_FINISH(APVR_GetActiveRecordingList);

   return(list_size);
}

/**
 * @brief   Frees the arrays allocated by APVR_GetActiveRecordingList
 * @param   handle_list array of recording handles
 */
void APVR_ReleaseActiveRecordingList(U32BIT *handle_list)
{
   FUNCTION_START(PVR_ReleaseActiveRecordingList);

   if (handle_list != NULL)
   {
      STB_AppFreeMemory(handle_list);
   }

   FUNCTION_FINISH(APVR_ReleaseActiveRecordingList);
}

/**
 * @brief   Get the name of the recording currently taking place on the given decode path
 * @param   path decode path being used for a recording
 * @param   name_ptr address of variable into which the name reference will be returned.
 *          This is actually a U8BIT*, but the string shouldn't be freed.
 * @return  TRUE if a recording is taking place on the given path and the name is returned,
 *          FALSE otherwise
 */
BOOLEAN APVR_GetPathRecordingName(U8BIT path, U8BIT **name_ptr)
{
   BOOLEAN recording_list_ok;

   FUNCTION_START(APVR_GetPathRecordingName);

   recording_list_ok = GetRecording(path, name_ptr);

   if (recording_list_ok)
   {
      APP_PRINT_PVR_REC(("APVR_GetTunerRecordingName(%d): recording \"%s\" is playing", path, *name_ptr));
   }

   FUNCTION_FINISH(APVR_GetPathRecordingName);

   return(recording_list_ok);
}

/**
 * @brief   Delete the given recording, including all files associated with it
 *          and remove it from the database of available recordings that can be
 *          played back
 * @param   handle recording to be deleted
 * @return  TRUE if the recording is deleted, FALSE otherwise
 */
BOOLEAN APVR_DeleteRecording(U32BIT handle)
{
   BOOLEAN recording_deleted_ok;

   FUNCTION_START(APVR_DeleteRecording);

   if (!STB_PVRRecordingGetLocked(handle))
   {
      recording_deleted_ok = DestroyRecording(handle);
   }
   else
   {
      recording_deleted_ok = FALSE;
   }

   FUNCTION_FINISH(APVR_DeleteRecording);

   return(recording_deleted_ok);
}

/**
 * @brief   Delete all the recordings marked as selected but not locked in the given list,
 *          including all files associated with it and remove it from the database
 *          of available recordings that can be played back.
 * @param   handles array of recording handles
 * @param   num_handles number of recordings in the array
 * @return  TRUE if at least one recording is deleted, FALSE otherwise
 */
BOOLEAN APVR_DeleteAllSelectedRecordings(U32BIT *handles, U16BIT num_handles)
{
   BOOLEAN recording_deleted;
   U16BIT i;

   FUNCTION_START(APVR_DeleteAllSelectedRecordings);

   recording_deleted = FALSE;

   if ((handles != NULL) && (num_handles > 0))
   {
      for (i = 0; i < num_handles; i++)
      {
         if (STB_PVRRecordingGetSelected(handles[i]) && !STB_PVRRecordingGetLocked(handles[i]))
         {
            DestroyRecording(handles[i]);
            recording_deleted = TRUE;
         }
      }
   }

   FUNCTION_FINISH(APVR_DeleteAllSelectedRecordings);

   return(recording_deleted);
}

/**
 * @brief   Un-select all recordings
 */
void APVR_UnselectAllRecordings(U32BIT *handles, U16BIT num_handles)
{
   U16BIT i;

   FUNCTION_START(APVR_UnselectAllRecordings);

   if ((handles != NULL) && (num_handles > 0))
   {
      for (i = 0; i < num_handles; i++)
      {
         STB_PVRRecordingSetSelected(handles[i], FALSE);
      }
   }

   FUNCTION_FINISH(APVR_UnselectAllRecordings);
}

/**
 * @brief   Checks if any recording in a list is selected
 * @param   handles array of recording handles
 * @param   num_handles number of recording handles in the array
 * @return  TRUE if at least one recording is selected, FALSE otherwise
 */
BOOLEAN APVR_AreSelectedRecordings(U32BIT *handles, U16BIT num_handles)
{
   BOOLEAN recording_selected;
   U16BIT i;

   FUNCTION_START(APVR_AreSelectedRecordings);

   recording_selected = FALSE;

   if ((handles != NULL) && (num_handles > 0))
   {
      for (i = 0; (i < num_handles) && !recording_selected; i++)
      {
         recording_selected = STB_PVRRecordingGetSelected(handles[i]);
      }
   }

   FUNCTION_FINISH(APVR_AreSelectedRecordings);

   return(recording_selected);
}

// ----pause recording----

/**
 * @brief   Sets the size of the timeshift buffer so that it's able to store
 *          a recording of the given length before it has to loop round. Whether
 *          this value can be accurately achieved may vary according to the support
 *          provided by the platform.
 * @param   time_in_mins timeshift buffer size in minutes
 */
void APVR_SetTimeshiftBufferSize(U16BIT time_in_mins)
{
   FUNCTION_START(APVR_SetTimeshiftBufferSize);
   timeshift_buffer_size = time_in_mins;
   APP_NvmSave(TIMESHIFT_BUFFER_SIZE_NVM, time_in_mins, TRUE);
   FUNCTION_FINISH(APVR_SetTimeshiftBufferSize);
}

/**
 * @brief   Returns the maximum size of the timeshift buffer in minutes
 * @return  timeshift buffer size in minutes
 */
U16BIT APVR_GetTimeshiftBufferSize(void)
{
   FUNCTION_START(APVR_GetTimeshiftBufferSize);
   FUNCTION_FINISH(APVR_GetTimeshiftBufferSize);
   return((U16BIT)APP_NvmRead(TIMESHIFT_BUFFER_SIZE_NVM));
}

/**
 * @brief   Checks whether timeshift has been started, which includes both recording and playback
 * @return  TRUE if timeshift is started, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftStarted(void)
{
   BOOLEAN retval;

   FUNCTION_START(APVR_IsTimeshiftStarted);

   if ((pause_path != INVALID_RES_ID) && (playback_path != INVALID_RES_ID))
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(APVR_IsTimeshiftStarted);

   return(retval);
}

/**
 * @brief   Checks whether the given path is the one that's being used for timeshift recording
 * @param   path decode path to be checked
 * @return  TRUE if the given path is the one being used for timeshift recording, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftRecordingPath(U8BIT path)
{
   BOOLEAN retval;

   FUNCTION_START(APVR_IsTimeshiftRecordingPath);

   if ((path != INVALID_RES_ID) && (pause_path == path))
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(APVR_IsTimeshiftRecordingPath);

   return(retval);
}

/**
 * @brief   Checks whether the given recording handle is the timeshift recording handle
 * @param   handle recording handle to be checked
 * @return  TRUE if the given recording handle is the timeshift recording handle, FALSE otherwise
 */
BOOLEAN APVR_IsTimeshiftRecording(U32BIT handle)
{
   BOOLEAN retval;

   FUNCTION_START(APVR_IsTimeshiftRecording);

   if ((handle != STB_PVR_INVALID_HANDLE) && (paused_recording == handle))
   {
      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   FUNCTION_FINISH(APVR_IsTimeshiftRecording);

   return(retval);
}

/**
 * @brief   Starts recording the live service ready for timeshift playback.
 *          The recording may not have started when this function returns, so
 *          an app should wait for the STB_EVENT_PVR_REC_START event before
 *          starting the paused playback using APVR_StartPausePlay.
 * @return  decode path used for the pause recording, INVALID_RES_ID on error
 */
U8BIT APVR_StartPauseRecord(void)
{
   U8BIT live_path;
   void *serv_ptr;
   void *transport;
   U16BIT onet_id;
   U16BIT trans_id;
   U16BIT service_id;
   U16BIT disk_id;
   BOOLEAN new_tuned_service;
   BOOLEAN record_started;
#ifdef COMMON_INTERFACE
   U8BIT slot_id;
#endif

   FUNCTION_START(APVR_StartPauseRecord);

   pause_path = INVALID_RES_ID;

   /* Pausing live TV records to the default disk */
   disk_id = STB_PVRGetDefaultDisk();

   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      serv_ptr = ADB_GetTunedService(live_path);
      if ((serv_ptr != NULL) && (disk_id != INVALID_DISK_ID))
      {
         /* Create and start recording */
         service_id = ADB_GetServiceId(serv_ptr);
         if ((transport = ADB_GetServiceTransportPtr(serv_ptr)) != NULL)
         {
            trans_id = ADB_GetTransportTid(transport);
            onet_id = ADB_GetTransportOriginalNetworkId(transport);
         }
         else
         {
            trans_id = ADB_INVALID_DVB_ID;
            onet_id = ADB_INVALID_DVB_ID;
         }

         /* Save the tuned service to be returned to when pause is stopped */
         tuned_service_ptr = serv_ptr;

         record_started = FALSE;

#ifdef INTEGRATE_HBBTV
         HBBTV_NotifyRecordingEvent(0, HBBTV_RECORDING_TS_STARTING);
#endif

         /* Acquire a decode path for recording.
          * If CI+ is enabled and the CAM has taken ownership of the live path/tuner then it may
          * not be possible to get a recording path, but going through the process of asking CI+
          * to release the path/tuner is just too much work, so in this case we'll just fail to
          * start pause live TV......for now. */
         pause_path = APVR_PrepareNewRecording(onet_id, trans_id, service_id, &new_tuned_service);
         if (pause_path != INVALID_RES_ID)
         {
            /* Set the flag for timeshift recording and max size of the timeshift buffer in seconds */
            STB_PVRStartRecordPaused(pause_path, timeshift_buffer_size * 60);

#ifdef COMMON_INTERFACE
            if ((slot_id = STB_DPGetPathCISlot(pause_path)) != INVALID_RES_ID)
            {
               /* Switch to timeshift mode */
              STB_CiCcSetRecordOperatingMode(slot_id, STB_CI_MODE_TIMESHIFT, service_id);
            }
#endif
            if (APVR_StartNewRecording(disk_id, pause_path, (U8BIT *)"PauseLiveTV", 0, 0, 0, &paused_recording) == TRUE)
            {
               paused_disk_id = disk_id;
               APP_PRINT_PVR_PAUSE(("APVR_StartPauseRecord: Recording live, handle=0x%04x", paused_recording));
               record_started = TRUE;
            }
#ifdef APP_PRINT_PVR_PAUSE
            else
            {
               APP_PRINT_PVR_PAUSE(("APVR_StartPauseRecord: Failed to start recording"));
            }
#endif

            if (!record_started)
            {
               /* Release the record path */
               ACTL_TuneToService(pause_path, NULL, NULL, FALSE, TRUE);
               STB_DPReleasePath(pause_path, RES_OWNER_DVB);
               pause_path = INVALID_RES_ID;
            }
         }
#if defined(APP_PRINT_PVR_PAUSE) || defined(INTEGRATE_HBBTV)
         else
         {
            APP_PRINT_PVR_PAUSE(("APVR_StartPauseRecord: Failed to get a path for recording"));
#ifdef INTEGRATE_HBBTV
            HBBTV_NotifyRecordingEvent(0, HBBTV_RECORDING_TS_STOPPED);
#endif
         }
#endif
      }
   }

   FUNCTION_FINISH(APVR_StartPauseRecord);

   return(pause_path);
}

/**
 * @brief   Starts playback of a live timeshift recording started with APVR_StartPauseRecord.
 *          This function should not be called until STB_EVENT_PVR_REC_START has been
 *          received with the correct recording path as an argument (as returned by
 *          APVR_StartPauseRecord).
 * @return  decode path used for the playback, INVALID_RES_ID on error
 */
U8BIT APVR_StartPausePlay(void)
{
   U8BIT live_path;
   U16BIT serv_id, ts_id, orig_net_id;

   FUNCTION_START(APVR_StartPausePlay);

   playback_path = INVALID_RES_ID;

   if (paused_recording != 0)
   {
      live_path = STB_DPGetLivePath();
      if (live_path != INVALID_RES_ID)
      {
         /* Stop the live path so playback can be started */
         ACTL_TuneOff(live_path);
         STB_DPReleasePath(live_path, RES_OWNER_NONE);
      }

      STB_PVRRecordingGetTriplet(paused_recording, &serv_id, &ts_id, &orig_net_id);
      playback_service = CreatePlaybackService(serv_id);

      playback_path = STB_DPAcquirePlaybackPath((void *)playback_service);
      if (playback_path != INVALID_RES_ID)
      {
         /* Set flag for playback */
         STB_PVRStartPlayPaused(playback_path);

         /* Start playback */
         if (STB_PVRStartPlaying(playback_path, paused_recording, FALSE))
         {
            ACTL_StartDecoding(playback_path, playback_service);
            STB_AVBlankVideo(0, FALSE);

            APP_PRINT_PVR_PAUSE(("%s: Setting APP_SI_MODE_UPDATE on playback path for live pause %u",
               __FUNCTION__, playback_path));

            ASI_SetAppSiMode(playback_path, APP_SI_MODE_UPDATE);
            STB_DPSetSearchMode(playback_path, FALSE);
            STB_DPStartSI(playback_path);

            playback_handle = paused_recording;

#if 0  /*FIXME: unnecessary */
            /* Acquire a new path to monitor live SI data. It should always be possible to get a path
             * for this service because this is the one being recorded for timeshift playback */
            monitor_si_path = ACTL_AcquirePathForService(tuned_service_ptr, FALSE, FALSE, NULL);
            APP_PRINT_PVR_PAUSE(("APVR_StartPausePlay: Path for monitoring SI data=%u", monitor_si_path));
            if (monitor_si_path != INVALID_RES_ID)
            {
               ACTL_TuneToService(monitor_si_path, NULL, tuned_service_ptr, TRUE, FALSE);
            }
#endif

#ifdef INTEGRATE_HBBTV
            /* Now playback has started for timeshift, notify HbbTV */
            HBBTV_NotifyRecordingEvent(0, HBBTV_RECORDING_TS_STARTED);
#endif
         }
         else
         {
            APP_PRINT_PVR_PAUSE(("APVR_StartPausePlay: Failed to start paused playback"));

            STB_DPReleasePath(playback_path, RES_OWNER_NONE);
            playback_path = INVALID_RES_ID;

            FreePlaybackService(playback_service);
            playback_service = NULL;
         }
      }
#ifdef APP_PRINT_PVR_PAUSE
      else
      {
         APP_PRINT_PVR_PAUSE(("APVR_StartPausePlay: Failed to get a playback path"));
      }
#endif
   }

   FUNCTION_FINISH(APVR_StartPausePlay);

   return(playback_path);
}

/**
 * @brief   Stops timeshifted playback and recording and optionally restarts decoding
 *          of the live signal.
 * @param   return_to_live TRUE to start decoding the live service
 */
void APVR_StopPauseRecord(BOOLEAN return_to_live)
{
   U32BIT handle;

   FUNCTION_START(APVR_StopPauseRecord);

   /* Stop playback and release the decoder being used */
   if (playback_path != INVALID_RES_ID)
   {
      STB_DPStopSI(playback_path);

      FreePlaybackService(playback_service);
      playback_service = NULL;

      /* If playback is in progress then stop it */
      ACTL_DecodeOff(playback_path);
      if (STB_PVRIsPlaying(playback_path, &handle) == TRUE)
      {
         STB_PVRStopPlaying(playback_path);
      }

      STB_DPReleasePath(playback_path, RES_OWNER_NONE);
      playback_path = INVALID_RES_ID;
   }

   if (pause_path != INVALID_RES_ID)
   {
      /* Stop the recording and release the path */
      STB_DPStopRecording(pause_path);
      ACTL_TuneToService(pause_path, NULL, NULL, FALSE, TRUE);
      STB_DPReleasePath(pause_path, RES_OWNER_DVB);
      pause_path = INVALID_RES_ID;
   }

   if (paused_recording != 0)
   {
      /* Delete any files associated with the pause live recording */
      APP_PRINT_PVR_PAUSE(("APVR_StopPauseRecord: stop and destroy recording 0x%x", paused_recording));
      DestroyRecording(paused_recording);
      paused_recording = 0;
   }

   if (monitor_si_path != INVALID_RES_ID)
   {
      APP_PRINT_PVR_PAUSE(("APVR_StopPauseRecord: Stop monitoring SI on path %u", monitor_si_path));

      ACTL_TuneOff(monitor_si_path);
      STB_DPReleasePath(monitor_si_path, RES_OWNER_NONE);
      monitor_si_path = INVALID_RES_ID;
   }

   if (return_to_live)
   {
      APP_PRINT_PVR_PAUSE(("APVR_StopPauseRecord: Return to live service %p", tuned_service_ptr));
      ACTL_TuneToService(INVALID_RES_ID, NULL, tuned_service_ptr, FALSE, TRUE);
   }

#ifdef INTEGRATE_HBBTV
   HBBTV_NotifyRecordingEvent(0, HBBTV_RECORDING_TS_STOPPED);
#endif

   FUNCTION_FINISH(APVR_StopPauseRecord);
}

/**
 * @brief   Returns TRUE if there are any recordings currently in progress.
 * @return  TRUE if recording in progress, FALSE otherwise
 */
BOOLEAN APVR_IsRecordingInProgress(void)
{
   U8BIT path, num_paths;
   U32BIT recording_handle;
   BOOLEAN is_recording, recordings;

   FUNCTION_START(APVR_IsRecordingInProgress);

   recordings = FALSE;

   num_paths = STB_DPGetNumPaths();

   for (path = 0; path < num_paths; path++)
   {
      is_recording = STB_DPIsRecording(path, &recording_handle);

      if ((is_recording) && (recording_handle == paused_recording))
      {
         /* This tuner is being used for pause so don't mark it as recording */
         is_recording = FALSE;
      }

      if (is_recording == TRUE)
      {
         recordings = TRUE;
      }
   }

   FUNCTION_FINISH(APVR_IsRecordingInProgress);

   return(recordings);
}

/**
 * @brief   Returns TRUE if given recording handle is still in progress on the
 *          given decode path.
 * @param   path decode path used for recording
 * @param   recording_handle recording
 * @return  TRUE if the recording is still in progress, FALSE otherwise
 */
BOOLEAN APVR_IsRecordingHandle(U8BIT path, U32BIT recording_handle)
{
   BOOLEAN is_recording;
   U32BIT recording_handle_found;

   FUNCTION_START(APVR_IsRecordingHandle);

   is_recording = FALSE;

   if (path != INVALID_RES_ID)
   {
      if (APVR_GetRecordingHandle(path, &recording_handle_found))
      {
         if (recording_handle == recording_handle_found)
         {
            is_recording = TRUE;
         }
      }
   }

   FUNCTION_FINISH(APVR_IsRecordingHandle);

   return(is_recording);
}

/**
 * @brief   Returns the handle of the recording taking place on the given decode path
 * @param   path decode path used for recording
 * @param   recording_handle_ptr address to return the recording handle,
 *          only valid on return if a recording is taking place
 * @return  TRUE if a recording is taking place and handle is returned, FALSE otherwise
 */
BOOLEAN APVR_GetRecordingHandle(U8BIT path, U32BIT *recording_handle_ptr)
{
   BOOLEAN is_recording;

   FUNCTION_START(APVR_GetRecordingHandle);

   is_recording = FALSE;

   // if tuner recording
   if (STB_DPIsRecording(path, recording_handle_ptr))
   {
      APP_PRINT_PVR_REC(("APVR_GetRecordingHandle(%d): found 0x%x", path, recording_handle_ptr));
      is_recording = TRUE;
   }

   FUNCTION_FINISH(APVR_GetRecordingHandle);

   return(is_recording);
}

/**
 * @brief   Returns the length in time and size in KB of the recording with the given handle
 * @param   handle - handle of recording being queried
 * @param   length_hours - length of the recording in hours, returned
 * @param   length_mins - length of the recording in minutes, returned
 * @param   length_secs - length of the recording in seconds, returned
 * @param   rec_size_kb - size of the recording in KB, returned
 * @return  TRUE if the length and size are being returned, FALSE otherwise
 */
BOOLEAN APVR_RecordingGetLength(U32BIT handle, U8BIT *length_hours, U8BIT *length_mins,
    U8BIT *length_secs, U32BIT *rec_size_kb )
{
    BOOLEAN ret;

    FUNCTION_START(APVR_RecordingGetLength);
    ret = STB_PVRRecordingGetLength (
        handle,
        length_hours,
        length_mins,
        length_secs,
        rec_size_kb
    );

    FUNCTION_FINISH(APVR_RecordingGetLength);

    return(ret);
}

// ---- disk ----

/**
 * @brief   Checks if there is enough space for the recording of the given event to take place
 * @brief   disk_id ID of disk to be queried
 * @param   event_ptr The event to check if there is any free space
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceForEvent(U16BIT disk_id, void *event_ptr)
{
   BOOLEAN free_space;

   FUNCTION_START(APVR_CheckSpaceForEvent);

   free_space = APVR_CheckSpaceDuration(disk_id, ADB_GetEventDuration(event_ptr));

   FUNCTION_FINISH(APVR_CheckSpaceForEvent);
   return(free_space);
}

/**
 * @brief   Check if there is enough space for the recording of the given duration to take place
 * @brief   disk_id ID of disk to be queried
 * @param   hours number of hours of the duration
 * @param   mins number of minutes of the duration
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceForDuration(U16BIT disk_id, U8BIT hours, U8BIT mins)
{
   U8BIT used_hour;
   U8BIT used_min;
   U8BIT free_hour;
   U8BIT free_min;
   U8BIT size_hour;
   U8BIT size_min;
   U16BIT total_time_needed;
   U16BIT total_space_left;
   BOOLEAN free_space;

   FUNCTION_START(APVR_CheckSpaceForDuration);

   APVR_GetDiskTime(disk_id, &used_hour, &used_min, &free_hour, &free_min, &size_hour, &size_min);

   /* Total up the event and the free space on disk durations */
   total_time_needed = (hours * 60) + mins;
   total_space_left = (free_hour * 60) + free_min;

   /* Check total time needed is less than whats left and also check for if anythings actually been used */
   if (total_time_needed < total_space_left)
   {
      free_space = TRUE;
   }
   else
   {
      free_space = FALSE;
   }

   FUNCTION_FINISH(APVR_CheckSpaceForDuration);

   return(free_space);
}

/**
 * @brief   Check if there is enough space for the recording of the given duration to take place
 * @brief   disk_id ID of disk to be queried
 * @param   duration Recording duration expressed as U32DHMS
 * @return  TRUE if there is space availiable, FALSE otherwise
 */
BOOLEAN APVR_CheckSpaceDuration(U16BIT disk_id, U32BIT duration)
{
   BOOLEAN free_space;
   FUNCTION_START(APVR_CheckSpaceDuration);
   free_space = APVR_CheckSpaceForDuration( disk_id,
         (U8BIT)(DHMS_DAYS(duration) * 24) + DHMS_HOUR(duration), DHMS_MINS(duration));
   FUNCTION_FINISH(APVR_CheckSpaceDuration);
   return(free_space);
}

/**
 * @brief   get total percentage of recording space used
 * @param   total_space_used total percentage of recording space used
 */
U8BIT APVR_GetTotalSpacePercentUsed(U16BIT disk_id)
{
   U32BIT size;
   U32BIT used;
   U8BIT total_space_used;

   FUNCTION_START(APVR_GetTotalSpacePercentUsed);

   if (!APVR_IsInitialised())
   {
      total_space_used = 100;
   }
   else
   {
      /* Calc % files used. All sizes are converted to MB to prevent overflow */
      size = STB_DSKGetSize(disk_id) / 1024;
      if (size > 0)
      {
         used = STB_DSKGetUsed(disk_id) / 1024;
         total_space_used = (U8BIT)((used * 100) / size);
      }
      else
      {
         total_space_used = 100;
      }
   }

   FUNCTION_FINISH(APVR_GetTotalSpacePercentUsed);

   return(total_space_used);
}

/**
 * @brief   Gets an estimate of disk space in time (hour / min): used, free and size
 * @param   disk_id ID of disk to be queried
 * @param   used_hour_ptr pointer to the variable containing the number of hours that the stored data is worth
 * @param   used_min_ptr pointer to the variable containing the number of minutes that the stored data is worth
 * @param   free_hour_ptr pointer to the variable containing the number of hours that the available space is worth
 * @param   free_min_ptr pointer to the variable containing the number of minutes that the available space is worth
 * @param   size_hour_ptr pointer to the variable containing the number of hours that the total disk size is worth
 * @param   size_min_ptr pointer to the variable containing the number of minutes that the total disk size is worth
 */
void APVR_GetDiskTime(U16BIT disk_id,
   U8BIT *used_hour_ptr, U8BIT *used_min_ptr,
   U8BIT *free_hour_ptr, U8BIT *free_min_ptr,
   U8BIT *size_hour_ptr, U8BIT *size_min_ptr)
{
   U16BIT total_min;

   FUNCTION_START(APVR_GetDiskTime);

   if (APVR_IsInitialised() == FALSE)
   {
      *used_hour_ptr = 0;
      *used_min_ptr = 0;

      *free_hour_ptr = 0;
      *free_min_ptr = 0;

      *size_hour_ptr = 0;
      *size_min_ptr = 0;
   }
   else
   {
      // hour and min:
      STB_PVRDiskUsed(disk_id, used_hour_ptr, used_min_ptr);
      STB_PVRDiskFree(disk_id, free_hour_ptr, free_min_ptr);
      total_min = (U16BIT)(*used_min_ptr) + (U16BIT)(*free_min_ptr);
      *size_min_ptr = (U8BIT)(total_min % 60);
      *size_hour_ptr = (*used_hour_ptr) + (*free_hour_ptr) + (U8BIT)(total_min / 60);
   }

   FUNCTION_FINISH(APVR_GetDiskTime);
}

/**
 * @brief   Ges disk space in Mbytes: used, free and size
 * @brief   disk_id ID of disk to be queried
 * @brief   used_mbyte_ptr amount of data stored on the disk
 * @brief   free_mbyte_ptr free space on the disk
 * @brief   size_mbyte_ptr total size of the disk
 */
void APVR_GetDiskMbyte(U16BIT disk_id, U32BIT *used_mbyte_ptr, U32BIT *free_mbyte_ptr, U32BIT *size_mbyte_ptr)
{
   U8BIT used_percent;

   FUNCTION_START(APVR_GetDiskMbyte);

   if (APVR_IsInitialised() == FALSE)
   {
      *used_mbyte_ptr = 0;
      *free_mbyte_ptr = 0;
      *size_mbyte_ptr = 0;
   }
   else
   {
      used_percent = APVR_GetTotalSpacePercentUsed(disk_id);
      *size_mbyte_ptr = STB_PVRDiskSize(disk_id);
      *used_mbyte_ptr = (used_percent * (*size_mbyte_ptr)) / PVR_100_PERCENT;
      *free_mbyte_ptr = ((PVR_100_PERCENT - used_percent) * (*size_mbyte_ptr)) / PVR_100_PERCENT;
   }

   FUNCTION_FINISH(APVR_GetDiskMbyte);
}

// ---- pause ----

/**
 * @brief   Calculates and returns the number of seconds behind live TV
 * @return  Number of seconds behind live
 */
U32BIT APVR_GetPauseProgress(void)
{
   U8BIT audio_decoder;
   U8BIT video_decoder;
   U8BIT length_hour, length_min, length_sec;
   U8BIT elapsed_hour, elapsed_min, elapsed_sec;
   U32BIT length, elapsed;
   U32BIT space_used;
   U32BIT pause_time;

   FUNCTION_START(APVR_GetPauseProgress);

   pause_time = 0;

   if (paused_recording != 0)
   {
      if (STB_PVRRecordingGetLength(paused_recording, &length_hour, &length_min, &length_sec, &space_used))
      {
         if (playback_path != INVALID_RES_ID)
         {
            audio_decoder = STB_DPGetPathAudioDecoder(playback_path);
            video_decoder = STB_DPGetPathVideoDecoder(playback_path);

            if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &elapsed_hour,
                   &elapsed_min, &elapsed_sec))
            {
               /* Calculate the number of seconds behind live TV */
               length = length_hour * 3600 + length_min * 60 + length_sec;
               elapsed = elapsed_hour * 3600 + elapsed_min * 60 + elapsed_sec;

               pause_time = length - elapsed;
            }
         }
      }
   }

   FUNCTION_FINISH(APVR_GetPauseProgress);

   return(pause_time);
}

/**
 * @brief   Calculates and returns the number of seconds in playback buffer
 * @return  Number of seconds of playback
 */
U32BIT APVR_GetPlaybackTime(void)
{
   U8BIT audio_decoder;
   U8BIT video_decoder;
   U8BIT elapsed_hour, elapsed_min, elapsed_sec;
   U32BIT elapsed;

   FUNCTION_START(APVR_GetPlaybackTime);

   elapsed = 0;

   if (playback_path != INVALID_RES_ID)
   {
      audio_decoder = STB_DPGetPathAudioDecoder(playback_path);
      video_decoder = STB_DPGetPathVideoDecoder(playback_path);

      if (STB_PVRGetElapsedTime(audio_decoder, video_decoder, &elapsed_hour,
             &elapsed_min, &elapsed_sec))
      {
         /* Calculate the number of seconds */
         elapsed = elapsed_hour * 3600 + elapsed_min * 60 + elapsed_sec;
      }
   }

   FUNCTION_FINISH(APVR_GetPlaybackTime);

   return(elapsed);
}

/**
 * @brief   Sets returns the number of seconds in playback buffer
 * @param   position_in_secs number of seconds from start of playback
 * @return  TRUE if successfully changed play back position
 */
BOOLEAN APVR_SetPlaybackTime(U32BIT position_in_secs)
{
   U8BIT audio_decoder;
   U8BIT video_decoder;
   BOOLEAN result;

   FUNCTION_START(APVR_SetPlaybackTime);

   if (playback_path != INVALID_RES_ID)
   {
      audio_decoder = STB_DPGetPathAudioDecoder(playback_path);
      video_decoder = STB_DPGetPathVideoDecoder(playback_path);
      result = STB_PVRPlaySetPosition(audio_decoder, video_decoder, position_in_secs);
   }
   else
   {
      result = FALSE;
   }

   FUNCTION_FINISH(APVR_SetPlaybackTime);
   return result;
}

/**
 * @brief   Returns the path currently acquired fro playback
 * @return  Decode path for playback
 */
U8BIT APVR_GetPlaybackPath(void)
{
   FUNCTION_START(APVR_GetPlaybackPath);
   FUNCTION_FINISH(APVR_GetPlaybackPath);
   return(playback_path);
}

#ifdef FREESAT_BUILD
/*!**************************************************************************
 * @brief   Works out whether a recording should be protected by a pin for playback
 *          based on the Freesat rules.
 * @param   handle - recording to be played
 * @return  TRUE if a pin should be requested, FALSE otherwise
 ****************************************************************************/
BOOLEAN APVR_IsRecordingPinLocked(U32BIT handle)
{
   BOOLEAN locked;
   U8BIT country_id;
   U16BIT date, local_date;
   U8BIT hours, mins, secs;
   U8BIT local_hours, local_mins, local_secs;

   FUNCTION_START(APVR_IsRecordingPinLocked);

   locked = FALSE;
   country_id = (U8BIT)APP_NvmRead(COUNTRY_ID_NVM);

   /* Check whether the recording requires a PIN to be entered before playback can start */
   if (ACTL_ParentalControlEnabled() &&
       STB_PVRRecordingGetParentalLock(handle) ||
       (STB_PVRRecordingHasGuidance(handle) &&
        STB_PVRRecordingGetDateTime(handle, &date, &hours, &mins, &secs) &&
        !ACFG_IsOutsideWatershedHours(country_id, hours, mins)))
   {
      /* The programme is subject to guidance so check
       * whether the current time is before the watershed */
      STB_GCGetGMTDateTime(&date, &hours, &mins, &secs);
      STB_GCConvertDateTime(date, hours, mins, secs, &local_date, &local_hours,
         &local_mins, &local_secs, CONV_LOCAL);

      if (!ACFG_IsOutsideWatershedHours(country_id, local_hours, local_mins))
      {
         locked = TRUE;
      }
   }

   FUNCTION_FINISH(APVR_IsRecordingPinLocked);

   return(locked);
}

#endif

/**
 * @brief   Determines whether the given programme CRID is already in the list of recorded events
 * @param   prog_crid full CRID (i.e. including default authority) of event to be searched for
 * @return  TRUE if a recording with the given CRID is found, FALSE otherwise
 */
BOOLEAN APVR_IsEventInRecordList(U8BIT *prog_crid)
{
   U32BIT handle;
   BOOLEAN recording_found;

   FUNCTION_START(APVR_IsEventInRecordList);

   recording_found = STB_PVRFindRecordingFromCrid(prog_crid, &handle);

   FUNCTION_FINISH(APVR_IsEventInRecordList);

   return(recording_found);
}

/**
 * @brief   Seaches the schedule for events that are part of the given series
 *          and that haven't already been recorded or aren't set to be recorded
 *          and creates a timer to record them.
 * @param   crid_ptr pointer to the CRID record
 */
void APVR_RecordSeries(void *crid_ptr)
{
   void **slist;
   void **elist;
   void *event_ptr;
   void *alt_event_ptr;
   U16BIT num_services;
   U16BIT num_events;
   U8BIT *series_crid_str;
   U8BIT *event_series_crid;
   U8BIT *prog_crid;
   U8BIT *series_crid;
   U16BIT si, ei;
   U8BIT ci, num_series_crids;
   U16BIT onet_id, tran_id, serv_id;
   U16BIT event_id;
   U32DHMS now_time;
   BOOLEAN proceed_with_recording;
   U32BIT timer;

   FUNCTION_START(APVR_RecordSeries);

   ADB_GetServiceList(ADB_SERVICE_LIST_ALL, &slist, &num_services);

   if ((num_services > 0) && (slist != NULL))
   {
      series_crid_str = ADB_GetCridString(crid_ptr);

      for (si = 0; si != num_services; si++)
      {
         /* Events on a service may be associated with different default CRID authorities
          * so it isn't possible to check that the service authorities for the series and
          * service match before going on to check each event, each event has to be checked
          * on all services */
         ADB_GetEventSchedule(FALSE, slist[si], &elist, &num_events);

         if ((num_events > 0) && (elist != NULL))
         {
            /* Get the current date/time to check for old events */
            now_time = STB_GCNowDHMSGmt();

            for (ei = 0; ei != num_events; ei++)
            {
               event_ptr = elist[ei];

               /* Check that the event isn't already in the past */
               if (ADB_GetEventStartDateTime(event_ptr) >= now_time)
               {
                  num_series_crids = ADB_GetEventNumSeriesCrids(event_ptr);

                  for (ci = 0; ci < num_series_crids; ci++)
                  {
                     event_series_crid = ADB_GetEventSeriesCrid(ci, slist[si], event_ptr);
                     if (event_series_crid != NULL)
                     {
                        if (STB_CompareStringsIgnoreCase(event_series_crid, series_crid_str) == 0)
                        {
                           /* This event is in the same series, check that it hasn't already
                            * been recorded and isn't set to be recorded */
                           proceed_with_recording = TRUE;

                           ADB_GetServiceIds(slist[si], &onet_id, &tran_id, &serv_id);

                           event_id = ADB_GetEventId(event_ptr);
                           prog_crid = ADB_GetEventProgrammeCrid(slist[si], event_ptr);

                           if ((prog_crid != NULL) && (strlen((char *)prog_crid) > 0))
                           {
                              if (APVR_IsEventInRecordList(prog_crid))
                              {
                                 /* This programme has already been recorded so don't re-record it */
                                 proceed_with_recording = FALSE;
                              }
                              else if (((timer = ATMR_FindTimerFromCrid(prog_crid)) != INVALID_TIMER_HANDLE) ||
                                 ((timer = ATMR_FindTimerFromEvent(onet_id, tran_id, serv_id, event_id)) != INVALID_TIMER_HANDLE))
                              {
                                 /* The event is already set to be recorded; check that it's being
                                  * recorded as part of the series */
                                 series_crid = ATMR_GetOtherCrid(timer);

                                 if ((series_crid != NULL) &&
                                    (STB_CompareStringsIgnoreCase(series_crid, series_crid_str) == 0))
                                 {
                                    /* The recording is part of the series so don't record it */
                                    proceed_with_recording = FALSE;
                                 }
                                 else if (ATMR_GetRecordingHandle(timer) != STB_PVR_INVALID_HANDLE)
                                 {
                                    /* The recording is already in progress so let it continue */
                                    proceed_with_recording = FALSE;
                                 }
                                 else
                                 {
                                    /* This programme is an individual recording or was being
                                     * recorded as a recommendation, but should now be
                                     * recorded as part of this series; delete the timer so
                                     * another one is created that includes the series CRID */
                                    ATMR_DeleteTimer(timer);
                                 }
                              }
#ifdef APP_PRINT_PVR_FP
                              if (proceed_with_recording)
                              {
                                 APP_PRINT_PVR_FP(("   recording series event with prog crid=\"%s\"", prog_crid));
                              }
#endif
                           }
                           else
                           {
                              /* As the event doesn't have a programme CRID the only check that
                               * can be made is to see if a timer exists with the same event and
                               * service IDs */
                              if (ATMR_FindTimerFromEvent(onet_id, tran_id, serv_id, event_id) != INVALID_TIMER_HANDLE)
                              {
                                 proceed_with_recording = FALSE;
                              }
#ifdef APP_PRINT_PVR_FP
                              else
                              {
                                 APP_PRINT_PVR_FP(("   recording series event from service 0x%x with event ID %d",
                                                   serv_id, event_id));
                              }
#endif
                           }

                           if (proceed_with_recording)
                           {
                              alt_event_ptr = ATMR_RecordEvent(slist[si], event_ptr, prog_crid,
                                    series_crid_str, FALSE, TRUE, ADB_GetCridDoNotDelete(crid_ptr));
                              if ((alt_event_ptr != NULL) && (alt_event_ptr != event_ptr))
                              {
                                 ADB_ReleaseEventData(alt_event_ptr);
                              }
                           }

                           if (prog_crid != NULL)
                           {
                              STB_AppFreeMemory(prog_crid);
                           }

                           /* Update the date this series was seen in the EIT */
                           ADB_UpdateCridEitDate(crid_ptr);
                        }

                        STB_AppFreeMemory(event_series_crid);
                     }
                  }
               }
            }

            ADB_ReleaseEventList(elist, num_events);
         }
      }

      ADB_ReleaseServiceList(slist, num_services);
   }

   FUNCTION_FINISH(APVR_RecordSeries);
}

/**
 * @brief   Searches the schedule for events that are part of the given recommendation
 *          and that haven't already been recorded or aren't set to be recorded
 *          and creates a timer to record them.
 * @param   crid_ptr pointer to the CRID record
 */
void APVR_RecordRecommendations(void *crid_ptr)
{
   void **slist;
   void **elist;
   void *alt_event_ptr;
   U16BIT num_services;
   U16BIT num_events;
   U8BIT *rec_crid_str;
   U8BIT *event_series_crid;
   U8BIT *prog_crid;
   U16BIT si, ei;
   U8BIT ci, num_series_crids;
   U16BIT onet_id, tran_id, serv_id;
   U16BIT event_id;
   BOOLEAN proceed_with_recording;

   FUNCTION_START(APVR_RecordRecommendations);

   ADB_GetServiceList(ADB_SERVICE_LIST_ALL, &slist, &num_services);

   if ((num_services > 0) && (slist != NULL))
   {
      rec_crid_str = ADB_GetCridString(crid_ptr);

      for (si = 0; si < num_services; si++)
      {
         /* Events on a service may be associated with different default CRID authorities
          * so it isn't possible to check that the service authorities for the recommendation
          * and service match before going on to check each event, each event has to be checked
          * on all services */
         ADB_GetEventSchedule(FALSE, slist[si], &elist, &num_events);

         if ((num_events > 0) && (elist != NULL))
         {
            for (ei = 0; ei < num_events; ei++)
            {
               proceed_with_recording = TRUE;

               /* A recommendation may either be a link to an individual event
                * or to a series, so both need to be checked as there's no way
                * to differentiate between them */
               prog_crid = ADB_GetEventProgrammeCrid(slist[si], elist[ei]);
               if (prog_crid != NULL)
               {
                  if (STB_CompareStringsIgnoreCase(prog_crid, rec_crid_str) == 0)
                  {
                     /* The recommendation is for this event, check that it hasn't already been
                      * recorded and isn't set to be recorded */
                     if (APVR_IsEventInRecordList(prog_crid) ||
                         (ATMR_FindTimerFromCrid(prog_crid) != INVALID_TIMER_HANDLE))
                     {
                        alt_event_ptr = ATMR_RecordEvent(slist[si], elist[ei], prog_crid, NULL,
                           TRUE, TRUE, ADB_GetCridDoNotDelete(crid_ptr));
                        if ((alt_event_ptr != NULL) && (alt_event_ptr != elist[ei]))
                        {
                           ADB_ReleaseEventData(alt_event_ptr);
                        }
                     }

                     /* The recommendation CRID has been matched against this programme CRID
                      * so it can't match a series */
                     proceed_with_recording = FALSE;
                  }

                  STB_AppFreeMemory(prog_crid);
               }

               if (proceed_with_recording)
               {
                  /* Check the recommendation CRID against all the series CRIDs of this event */
                  num_series_crids = ADB_GetEventNumSeriesCrids(elist[ei]);

                  for (ci = 0; ci < num_series_crids; ci++)
                  {
                     event_series_crid = ADB_GetEventSeriesCrid(ci, slist[si], elist[ei]);
                     if (event_series_crid != NULL)
                     {
                        //DBG_PVR(("    %d: series crid=\"%s\"\n", ei, event_series_crid));
                        if (STB_CompareStringsIgnoreCase(event_series_crid, rec_crid_str) == 0)
                        {
                           /* This event is in the same series */
                           ADB_GetServiceIds(slist[si], &onet_id, &tran_id, &serv_id);

                           event_id = ADB_GetEventId(elist[ei]);
                           prog_crid = ADB_GetEventProgrammeCrid(slist[si], elist[ei]);

                           if (prog_crid != NULL)
                           {
                              /* Check that it hasn't already been recorded and isn't set to be recorded */
                              if (APVR_IsEventInRecordList(prog_crid) ||
                                  (ATMR_FindTimerFromCrid(prog_crid) != INVALID_TIMER_HANDLE) ||
                                  (ATMR_FindTimerFromEvent(onet_id, tran_id, serv_id, event_id) != INVALID_TIMER_HANDLE))
                              {
                                 proceed_with_recording = FALSE;
                              }
                           }
                           else
                           {
                              /* As the event doesn't have a programme CRID the only check that
                               * can be made is to see if a timer exists with the same event and
                               * service IDs */
                              if (ATMR_FindTimerFromEvent(onet_id, tran_id, serv_id, event_id) != INVALID_TIMER_HANDLE)
                              {
                                 proceed_with_recording = FALSE;
                              }
                           }

                           if (proceed_with_recording)
                           {
                              alt_event_ptr = ATMR_RecordEvent(slist[si], elist[ei], prog_crid,
                                    event_series_crid, TRUE, TRUE, ADB_GetCridDoNotDelete(crid_ptr));

                              if ((alt_event_ptr != NULL) && (alt_event_ptr != elist[ei]))
                              {
                                 ADB_ReleaseEventData(alt_event_ptr);
                              }
                           }

                           if (prog_crid != NULL)
                           {
                              STB_AppFreeMemory(prog_crid);
                           }

                           /* Update the date this recommendation was seen in the EIT */
                           ADB_UpdateCridEitDate(crid_ptr);
                        }

                        STB_AppFreeMemory(event_series_crid);
                     }
                  }
               }
            }

            ADB_ReleaseEventList(elist, num_events);
         }
      }

      ADB_ReleaseServiceList(slist, num_services);
   }

   FUNCTION_FINISH(APVR_RecordRecommendations);
}

/**
 * @brief   If the current playback recording is a split event then the next
 *          chronological part of the event is found and its handle returned
 * @param   curr_handle handle of split recording when searching for the next part
 * @param   next_handle handle of the next recording to be played, returned
 * @return  TRUE if a recording is found, FALSE otherwise
 */
BOOLEAN APVR_FindNextSplitEvent(U32BIT curr_handle, U32BIT *next_handle)
{
   BOOLEAN found_next_recording;

   FUNCTION_START(APVR_FindNextSplitEvent);

   found_next_recording = FALSE;

   if (playback_path != INVALID_RES_ID)
   {
      found_next_recording = STB_PVRFindNextSplitRecording(curr_handle, next_handle);
   }

   FUNCTION_FINISH(APVR_FindNextSplitEvent);

   return(found_next_recording);
}

/**
 * @brief   A service instance is associated to the playback, this function returns its pointer.
 *          The handle can be used, for example, to call ADB_ServiceHasSubtitles as it's done for
 *          live channels.
 * @return  Playback service pointer
 */
void* APVR_GetPlaybackService(void)
{
   FUNCTION_START(APVR_GetPlaybackService);
   FUNCTION_FINISH(APVR_GetPlaybackService);
   return((void *)playback_service);
}

/**
 * @brief   Deals with any private timers started by the PVR module
 * @param   timer_handle handle of the timer that's been triggered
 * @return  TRUE if the timer is a PVR timer and has been dealt with, FALSE otherwise
 */
BOOLEAN APVR_HandlePrivateTimer(U32BIT timer_handle)
{
   BOOLEAN retval;
#ifdef COMMON_INTERFACE
   U8BIT path;
   U32BIT recording_handle;
#endif

   FUNCTION_START(APVR_HandlePrivateTimer);

   retval = FALSE;

#ifdef COMMON_INTERFACE
   for (path = 0; path < STB_DPGetNumPaths(); path++)
   {
      if (timer_handle == cam_record_start_timer[path])
      {
         retval = TRUE;

         /* No reply has been received from the CAM, so the recording can't proceed */
         if (STB_DPIsRecording(path, &recording_handle))
         {
            APVR_StopRecording(recording_handle);
            APVR_DeleteRecording(recording_handle);
         }

         STB_DPReleasePath(path, RES_OWNER_DVB);

         /* Delete the private timer */
         ATMR_DeleteTimer(timer_handle);
         cam_record_start_timer[path] = INVALID_TIMER_HANDLE;

         /* Send event to UI to inform it that a recording has failed */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);
      }
   }
#else
   USE_UNWANTED_PARAM(timer_handle);
#endif

   FUNCTION_FINISH(APVR_HandlePrivateTimer);

   return(retval);
}

#ifdef COMMON_INTERFACE
/**
 * @brief   When a recording is started that involves a CI CAM, a message is sent to check
 *          whether the recording can proceed, and this function handles the reply received
 * @param   start_succeeded TRUE if the reply indicates that the recording can proceed, FALSE
 *          otherwise
 * @param   slot_id the slot id associated with the reply
 */
void APVR_CIRecordReplyReceived(BOOLEAN start_succeeded, U8BIT slot_id)
{
   U8BIT path;
   U32BIT recording_handle;
   S_ACTL_OWNER_INFO owner_info;
   BOOLEAN stop_recording;

   FUNCTION_START(APVR_CIRecordReplyReceived);

   /* A slot can be shared with more than one path,
    * so handle this event for all paths using this slot */
   path = INVALID_RES_ID;
   while ((path = STB_DPIsCISlotInUse(path, slot_id, path)) != INVALID_RES_ID)
   {
      stop_recording = FALSE;

      if (STB_DPIsRecording(path, &recording_handle))
      {
         /* A reply has been received, so the timer created to handle the case of
          * no reply being received can be deleted */
         ATMR_DeleteTimer(cam_record_start_timer[path]);
         cam_record_start_timer[path] = INVALID_TIMER_HANDLE;

         if (start_succeeded)
         {
            /* Recording can be resumed */
            if (!STB_PVRResumeRecording(path))
            {
               /* Failed to resume recording, so have to stop it */
               stop_recording = TRUE;
            }
         }
         else
         {
            /* Recording can't be started */
            stop_recording = TRUE;
         }

         if (stop_recording)
         {
            /* Stop and delete the recording */
            APVR_StopRecording(recording_handle);
            APVR_DeleteRecording(recording_handle);

            /* Tune to a NULL service on the recording path */
            owner_info.owner = RES_OWNER_DVB;
            owner_info.data = NULL;
            owner_info.data_size = 0;

            ACTL_TuneToService(path, &owner_info, NULL, FALSE, TRUE);

            STB_DPReleasePath(path, RES_OWNER_DVB);

            /* Send event to UI to inform it that a recording has failed */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);
         }
      }
   }

   FUNCTION_FINISH(APVR_CIRecordReplyReceived);
}

#endif

static ADB_SERVICE_REC* CreatePlaybackService(U16BIT service_id)
{
   ADB_SERVICE_REC *service;

   FUNCTION_START(CreatePlaybackService);

   service = (ADB_SERVICE_REC *)STB_AppGetMemory(sizeof(ADB_SERVICE_REC));
   if (service != NULL)
   {
      memset(service, 0, sizeof(ADB_SERVICE_REC));
      service->serv_id = service_id;
   }

   FUNCTION_FINISH(CreatePlaybackService);

   return(service);
}

static void FreePlaybackService(ADB_SERVICE_REC *s_ptr)
{
   FUNCTION_START(FreePlaybackService);

   if (s_ptr != NULL)
   {
      ADB_DeleteServiceRec(s_ptr);
   }

   FUNCTION_FINISH(FreePlaybackService);
}

/**
 * @brief   Ensures that any EIT updates are processed to handle scenarios such
 *          as checking for new series recordings, etc.
 */
void APVR_EitUpdated(void)
{
   U8BIT q_msg = 0;

   FUNCTION_START(APVR_EitUpdated);

   if (eit_update_queue != NULL)
   {
      STB_OSWriteQueue(eit_update_queue, (void *)&q_msg, sizeof(q_msg), TIMEOUT_NOW);
   }

   FUNCTION_FINISH(APVR_EitUpdated);
}

/**
 * @brief   This function should be called when there's an update to the PIDs
 *          for a service that's being recorded. The PIDs being recorded are
 *          checked and any that are no longer valid will no longer be recorded
 *          and any new PIDs will be added.
 * @param   path recording path
 */
void APVR_PidsUpdated(U8BIT path)
{
   U32BIT handle;
   void *service;

   FUNCTION_START(APVR_PidsUpdated);

   if (STB_PVRIsRecording(path, &handle) == TRUE)
   {
      service = ADB_GetTunedService(path);
      if (service != NULL)
      {
         SetupForRecording(path, service, TRUE);
      }
   }

   FUNCTION_FINISH(APVR_PidsUpdated);
}

/**
 * @brief   Used to set whether all recordings should be encrypted or not. The default is that
 *          recordings aren't encrypted, in which case the do_not_scramble flag in the SI
 *          free-to-air content management descriptor is used to determine whether a recording
 *          should be encrypted or not.
 *          This setting is not persistent, so needs to be set each time the DVB stack is initialised
 * @param   state TRUE if recordings are to be encrypted, FALSE if not
 */
void APVR_EncryptRecordings(BOOLEAN state)
{
   FUNCTION_START(APVR_EncryptRecordings);
   encrypt_recordings = state;
   FUNCTION_FINISH(APVR_EncryptRecordings);
}

/*!**************************************************************************
 * @brief   Task entry point that performs all EIT update processing
 * @return  None
 ****************************************************************************/
static void EitUpdatedTask(void *queue)
{
   U8BIT q_msg;
   BOOLEAN read_queue;
   U16BIT num_q_reads;
   U32BIT last_update_time = STB_OSGetClockMilliseconds();

   FUNCTION_START(EitUpdatedTask);

   /* As EIT updates tend to arrive several at a time, no processing is performed
    * until there's a timeout reading the queue and processing only then occurs if
    * there has been at least one EIT update. */
   num_q_reads = 0;
   q_msg = 0;

   while (q_msg != 0xFF)
   {
      read_queue = STB_OSReadQueue(queue, &q_msg, sizeof(q_msg), 60000);
      if (read_queue)
      {
         /* A message has been read from the queue so increment the count */
         num_q_reads++;
      }

      if ((STB_OSGetClockDiff(last_update_time) >= 60000) && (num_q_reads != 0) && q_msg != 0xFF)
      {
         /* Reset the number of times the queue has been read and process the CRID records */
         num_q_reads = 0;

         /* Check whether the EIT update requires changes to any of the already scheduled recordings */
         ATMR_EitUpdated();

         /* Check whether the EIT update requires new recordings to be scheduled */
         ProcessCridRecords();

         last_update_time = STB_OSGetClockMilliseconds();
      }
   }
   eit_update_queue = NULL;
   STB_OSTaskDelay(1);
   STB_OSDestroyQueue( queue );
   FUNCTION_FINISH(EitUpdatedTask);
}

/*!**************************************************************************
 * @brief   Processes all CRID records to see whether there are any new events
 *          that need to be set for recording. This function is called as part
 *          of the EIT update background task.
 * @return  None
 ****************************************************************************/
static void ProcessCridRecords(void)
{
   void **crid_rec_list;
   U16BIT num_crid_recs;
   U16BIT i;
   U8BIT *crid_str;
   U32DHMS start_time, end_time, now_time;
   U16BIT eit_date, gmt_date;
   U16BIT serv_id;
   void *serv_ptr;
   void *event_ptr;

   FUNCTION_START(ProcessCridRecords);

   ADB_GetCridRecordList(&crid_rec_list, &num_crid_recs, TRUE, TRUE, TRUE);
#ifdef APP_PRINT_PVR_FP
   if (num_crid_recs > 0)
   {
      APP_PRINT_PVR_FP(("### Processing %d CRID records", num_crid_recs));
   }
#endif

   gmt_date = STB_GCGetGMTDate();

   for (i = 0; i < num_crid_recs; i++)
   {
      crid_str = ADB_GetCridString(crid_rec_list[i]);

      if (ADB_IsProgrammeCrid(crid_rec_list[i]) && ADB_IsSplitProgrammeCrid(crid_str))
      {
         /* This record represents a split event. Get the info to see whether
          * any additional parts of the split event are now available */
         start_time = ADB_GetCridDateTime(crid_rec_list[i]);
         serv_id = ADB_GetCridService(crid_rec_list[i]);

         APP_PRINT_PVR_FP(("  %d: Programme, CRID=\"%s\", time=%d.%d, service=0x%x", i, crid_str,
                           DHMS_HOUR(start_time), DHMS_MINS(start_time), serv_id));

         serv_ptr = ADB_FindServiceByIds(ADB_INVALID_DVB_ID, ADB_INVALID_DVB_ID, serv_id);

         /* Calculate the time this crid will no longer be valid */
         end_time = STB_GCCalculateDHMS(start_time, DHMS_CREATE(0, 3, 0, 0), CALC_ADD);
         now_time = STB_GCNowDHMSGmt();
         /* Only search for the crid if everything is valid and the time is less than 3 hours old */
         if ((crid_str != NULL) && (serv_ptr != NULL) && (end_time >= now_time))
         {
            /* Search backwards and forwards from the saved date/time */
            ATMR_RecordSplitEvent(serv_ptr, crid_str, start_time,
               ADB_GetCridDoNotDelete(crid_rec_list[i]), FALSE);
            ATMR_RecordSplitEvent(serv_ptr, crid_str, start_time,
               ADB_GetCridDoNotDelete(crid_rec_list[i]), TRUE);
         }
         else
         {
            /* Delete the record as it's no longer valid */
            ADB_DeleteCridRecord(crid_rec_list[i]);
         }
      }
      else
      {
         /* Check whether this CRID record should now be deleted from the record list */
         eit_date = ADB_GetCridEitDate(crid_rec_list[i]);

         if (gmt_date - eit_date >= CRID_DATE_TIMEOUT)
         {
            APP_PRINT_PVR_FP(("  %d: CRID record for \"%s\" has expired, GMT=%d, last seen=%d", i,
                              ADB_GetCridProgrammeName(crid_rec_list[i]), gmt_date, eit_date));
            ADB_DeleteCridRecord(crid_rec_list[i]);
         }
         else
         {
            if (ADB_IsProgrammeCrid(crid_rec_list[i]))
            {
               /* Check whether the event for this CRID is now in the EIT */
               event_ptr = ADB_FindEventFromCrid(crid_str, NULL, &serv_ptr);
               if (event_ptr != NULL)
               {
                  /* Set the event to be recorded */
                  ATMR_RecordEvent(serv_ptr, event_ptr, crid_str, NULL, FALSE, TRUE,
                     ADB_GetCridDoNotDelete(crid_rec_list[i]));

                  /* The CRID record can now be deleted */
                  ADB_DeleteCridRecord(crid_rec_list[i]);
               }
            }
            else if (ADB_IsSeriesCrid(crid_rec_list[i]))
            {
               APP_PRINT_PVR_FP(("  %d: Series, CRID=\"%s\" for \"%s\"", i,
                  ADB_GetCridString(crid_rec_list[i]), ADB_GetCridProgrammeName(crid_rec_list[i])));
               APVR_RecordSeries(crid_rec_list[i]);
            }
            else if (ADB_IsRecommendationCrid(crid_rec_list[i]))
            {
               APP_PRINT_PVR_FP(("  %d: Recommendation, CRID=\"%s\" for \"%s\"", i,
                  ADB_GetCridString(crid_rec_list[i]), ADB_GetCridProgrammeName(crid_rec_list[i])));
               APVR_RecordRecommendations(crid_rec_list[i]);
            }
            else
            {
               APP_PRINT_PVR_FP(("  %d: Unrecognised type of CRID record! CRID=\"%s\" for \"%s\"", i,
                                 ADB_GetCridString(crid_rec_list[i]), ADB_GetCridProgrammeName(crid_rec_list[i])));
            }
         }
      }
   }

   if (crid_rec_list != NULL)
   {
      STB_AppFreeMemory(crid_rec_list);
   }

#ifdef APP_PRINT_PVR_FP
   if (num_crid_recs > 0)
   {
      APP_PRINT_PVR_FP(("    Finished processing %d CRID records", num_crid_recs));
   }
#endif

   FUNCTION_FINISH(ProcessCridRecords);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Creates a new private timer that will expire the given number of seconds
 *          after the timer is created
 * @param   seconds - number of seconds until the timer expires
 * @return  timer handle
 ****************************************************************************/
static U32BIT NewCountdownTimer(U32BIT seconds)
{
   S_TIMER_INFO timer_info;
   U32BIT handle;
   U16BIT days;
   U8BIT hrs, mins, secs;

   FUNCTION_START(NewCountdownTimer);

   days = seconds / (3600 * 24);
   hrs  = (seconds / 3600) % 24;
   mins = (seconds / 60) % 60;
   secs =  seconds % 60;

   ATMR_InitialiseTimer(&timer_info, TIMER_TYPE_PRIVATE, NULL, NULL);

   timer_info.start_time = STB_GCCalculateDHMS( STB_GCNowDHMSGmt(), DHMS_CREATE(days,hrs,mins,secs), CALC_ADD);

   handle = ATMR_AddTimer(&timer_info);

   FUNCTION_FINISH(NewCountdownTimer);

   return(handle);
}
#endif


//**************************************************************************************************
// End of File
//**************************************************************************************************

