/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level SI task
 * @file    ap_si.c
 * @date    October 2003
 */
//#define DEBUG_PRINTING_ENABLED
// gives direct COM port access
#define AP_DEBUG

/*#define DEBUG_SI_CHAN_SEARCH*/
/*#define DEBUG_SI_STARTUP_SEARCH*/
/*#define DEBUG_SI_EVENT_SCHED_SEARCH*/
/*#define DEBUG_SI_UPDATE*/
/*#define DEBUG_SI_RECORDING*/
/*#define DEBUG_SI_PLAYBACK*/
/*#define DEBUG_SI_ICS*/
/*#define DEBUG_SI_TOT_SEARCH*/
/*#define DEBUG_CI_SI_UPDATE*/
/*#define DEBUG_SI_RCT*/
/*#define DEBUG_DYNAMIC_UPDATE*/

// prints table debug messages
#define DEBUG_SI_CAT
#define DEBUG_SI_PAT
#define DEBUG_SI_NIT
/*#define DEBUG_SI_SDT*/
#define DEBUG_SI_PMT
#define DEBUG_SI_PMT_VIDEO
#define DEBUG_SI_PMT_AUDIO
/*#define DEBUG_SI_EIT*/
/*#define DEBUG_SI_TOT*/
/*#define DEBUG_SI_TDT*/
/*#define DEBUG_SI_BAT*/
/*#define DEBUG_SI_UNT*/

//#define DEBUG_SI_CAT_PLAYBACK
/*#define DEBUG_SI_PAT_PLAYBACK*/
/*#define DEBUG_SI_PMT_PLAYBACK*/

// prints the transport parameters being stored into database
//#define DEBUG_TRANS_PARAMS

/*#define DEBUG_DVB_SSU*/
/*#define DEBUG_SI_DVB_SSU_SEARCH*/


//---includes for this file----------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files

#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbhwtun.h"
#include "stbhwav.h"

#include "stbheap.h"
#include "stberc.h"
#include "stbgc.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbllist.h"
#include "stbdsapi.h"
#include "stbuni.h"

#include "media_image.h"

#if defined(INCLUDE_DSMCC_FILE_REQUEST)
#include "dsm_client.h"
#include "dsm_control.h"
#endif
#ifdef INTEGRATE_HBBTV
#include "hbbtv_api.h"
#endif

#include "app.h"
#include "ap_cfg.h"
#include "ap_cfdat.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"
#include "ap_cntrl.h"
#include "ap_si.h"
#include "ap_pvr.h"
#include "app_nvm.h"

#ifdef COMMON_INTERFACE
#include "ap_ci_int.h"
#include "stbcicc.h"
#include "ap_ci.h"
#endif
#include "ca_glue.h"
#include "ap_ca.h"

#include "dba.h"

#ifdef ENABLE_LIBDVBPSI
#include <stdbool.h>
#include <dvbpsi/dvbpsi.h>
#include <dvbpsi/descriptor.h>
#include <dvbpsi/dr.h>
#include <dvbpsi/nit.h>
#endif

//---constant definitions for this file----------------------------------------
#ifdef AP_DEBUG
   #define AP_SI_PRINT(x) STB_SPDebugWrite x
   #define AP_SI_PRINT1(x) STB_SPDebugWrite x
#else
   #define AP_SI_PRINT(x)
#endif

// values passed to table request functions
#define DONT_CARE_ID_MATCH       0x0000
#define DONT_CARE_ID_MASK        0x0000

// table id values
#define EITPF_ACTUAL_TID            0x4E
#define EITPF_OTHER_TID             0x4F
#define EITSC_ACTUAL_TID            0x50     // range 0x50 to 0x5f
#define EITSC_OTHER_TID             0x60     // range 0x60 to 0x6f
#define EITPF_PLUS_TID              0xd1     /* Freesat EITpf++ table ID */
#define TDT_TID                     0x70
#define TOT_TID                     0x73

/* DVB descriptors */
#define AC3_DESC_DTAG               0x6a
#define EAC3_DESC_DTAG              0x7a

// timeout values
#define BAT_TIMEOUT_MS                 12000
#define CAT_TIMEOUT_MS                 2000
#define PAT_TIMEOUT_MS                 (800 + 10000)  // CNS PAT interval ~500ms
#define PMT_TIMEOUT_MS                 (3000 + 10000)
#define NIT_TIMEOUT_MS                 15000
#define SDT_TIMEOUT_MS                 12000
#define EIT_TIMEOUT_MS                 3000
#define TOT_TIMEOUT_MS                 32000
#define TDT_TIMEOUT_MS                 32000
#define SCHED_TIMEOUT_MS               3600000  /* 1 hour */
#define EVENT_SCHED_SEARCH_TIMEOUT_MS  10000
#define UNT_TIMEOUT_MS                 10000

#define PMT_UPDATE_MS            10000
#define SI_UPDATE_DELAY_MS       200

#define MAX_PMT_PRIORITY_LIST    4

#define DVB_SSU_LINKAGE_TYPE     0x09
#define DVB_DVBH1_LINKAGE_TYPE   0x0B
#define DVB_DVBH2_LINKAGE_TYPE   0x0C

// macros used as parameters to make code easier to understand
#define IGNORE_VERSION        TRUE
#define CHECK_VERSION         FALSE

#define EIT_ACTUAL_ONLY       TRUE
#define EIT_ACTUAL_OR_OTHER   FALSE

#define REPORT_PMT            TRUE
#define DONT_REPORT_PMT       FALSE

#define EIT_LIST_REQD         TRUE
#define EIT_LIST_NOT_REQD     FALSE

#define REPORT_CAT            TRUE
#define DONT_REPORT_CAT       FALSE

#define INVALID_SERVICE_ID    0xffffffff

/* Private data codes */
#define UK_DTT_PRIVATE_DATA_CODE    0x0000233a
#define FREESAT_PRIVATE_DATA_CODE   0x46534154

//---local typedefs, structs, enumerations for this file--------------------------------------------

typedef struct pmt_list_entry
{
   U16BIT serv_id;
   U16BIT ts_id;
   U16BIT on_id;
   U16BIT pid;
} PMT_LIST_ENTRY;

typedef struct eit_list_entry
{
   U16BIT serv_id;
   BOOLEAN got_eit;
} EIT_LIST_ENTRY;

typedef enum
{
   DB_ACCESS_UPDATE,
   DB_ACCESS_SEARCH,
   DB_ACCESS_PLAYBACK
} E_DB_ACCESS_MODE;

typedef struct s_si_event_data
{
   struct s_si_event_data *next;
   U8BIT index;
   U32BIT value;
} S_SI_EVENT_DATA;

typedef enum
{
   PMT_REQUEST_CURRENT,
   PMT_REQUEST_MONITOR,
   PMT_REQUEST_PRIORITY
} E_PMT_REQUEST_MODE;

typedef struct
{
   U16BIT tran_id;
   U16BIT onet_id;
} S_IDS;

typedef struct
{
   U16BIT bouquet_id;
   BOOLEAN received;
} S_ACTIVE_BOUQUET;

typedef struct s_dynamic_update_service
{
   /* Transport and service record of service being added */
   ADB_TRANSPORT_REC *add_t_ptr;
   ADB_SERVICE_REC *add_s_ptr;

   /* Transport and service record of service being deleted */
   ADB_TRANSPORT_REC *delete_t_ptr;
   ADB_SERVICE_REC *delete_s_ptr;

   struct s_dynamic_update_service *next;
} S_DYNAMIC_UPDATE_SERVICE;

typedef struct s_dynamic_update_transport
{
   ADB_TRANSPORT_REC *t_ptr;

   U16BIT num_lcns;
   SI_LCN_DESC *lcn_array;
   U16BIT num_hd_lcns;
   SI_LCN_DESC *hd_lcn_array;
   U16BIT num_nordig_lcns;
   SI_NORDIG_LCN_DESC *nordig_lcn_array;

   S_DYNAMIC_UPDATE_SERVICE *service_head;
   S_DYNAMIC_UPDATE_SERVICE *service_tail;

   struct s_dynamic_update_transport *next;
} S_DYNAMIC_UPDATE_TRANSPORT;

typedef struct
{
   BOOLEAN terrestrial;
   BOOLEAN cable;
   BOOLEAN satellite;
} S_DATABASE_UPDATES_ALLOWED;


//---local (static) variable declarations for this file------------------------
//   (internal variables declared static to make them local)

static E_APP_SI_MODE *required_si_mode;
static U32BIT *required_si_mode_ctrl;
static F_SIManager *current_manager;
static E_SEARCH_SERVICE_TYPE required_service_type;

static ADB_NETWORK_REC **current_network_rec;
static ADB_TRANSPORT_REC **current_transport_rec;
static ADB_SERVICE_REC **current_service_rec;
// used to track transport changes
static ADB_TRANSPORT_REC **last_transport_rec;

static void **pat_filter;
static void **pmt_filter;
static void **eit_filter;

static void **sched_filter;
static void **nit_filter;
static void **sdt_filter;
static void **bat_filter;
static void **tot_filter;
static void **tdt_filter;
static void **cat_filter;
static void **rct_filter;
static void **ait_filter;
static void **unt_filter = NULL;

static U32BIT *pat_start_timestamp;
static U32BIT *pmt_start_timestamp;
static U32BIT *nit_start_timestamp;
static U32BIT *sdt_start_timestamp;
static U32BIT *bat_start_timestamp;
static U32BIT *eit_start_timestamp;
static U32BIT *sched_start_timestamp;
static U32BIT *sched_timeout_ms;
static U32BIT *tot_start_timestamp;
static U32BIT *tdt_start_timestamp;
static U32BIT *si_update_delay_timestamp;
static U32BIT *pmt_update_timestamp;
static U32BIT *cat_start_timestamp;
static U32BIT *pmt_update_period_ms;
static U32BIT *last_timestamp;
static U32BIT *unt_start_timestamp;

static BOOLEAN *sdt_complete;
static BOOLEAN *bat_complete;
static BOOLEAN *nit_complete;
static BOOLEAN *pmts_complete;
static BOOLEAN *eits_complete;
static BOOLEAN *tot_complete;
static BOOLEAN *unt_complete;

static BOOLEAN *tot_already_received;

static BOOLEAN *pat_rcvd_on_this_trnsprt;

static PMT_LIST_ENTRY **pmt_list;
static U16BIT *num_pmt_list_entries;
static U16BIT *pmt_list_id;
static U16BIT *pmt_filter_pid;
static BOOLEAN *pmt_service_changed;
static BOOLEAN *report_pmt_allowed;
static BOOLEAN *pmt_reported;
static BOOLEAN *stop_pmt_reporting;
static E_PMT_REQUEST_MODE *pmt_request_mode;
static U32BIT pmt_priority_list[MAX_PMT_PRIORITY_LIST] = {INVALID_SERVICE_ID};
static void *si_pmt_list_sem;

static EIT_LIST_ENTRY **eit_list;
static U16BIT *num_eit_list_entries;

static U16BIT eit_schedule_limit;

// set during search to indicate that service list is ready to be read - i.e. sdt processing complete
static BOOLEAN *service_list_ready;

static S16BIT *last_reported_cat_version;
static S16BIT *last_reported_nit_version;
static U8BIT *last_reported_pmt_version;
static S16BIT *last_reported_unt_version;

static BOOLEAN use_bats_active;
static S_ACTIVE_BOUQUET *active_bouquet_ids;
static U16BIT num_active_bouquet_ids;

/* 0 is an unused reserved network ID so can be used to indicate the value hasn't been set */
static U16BIT active_network_id = 0;

static S16BIT last_playback_pmt_version = -1;
static S16BIT last_playback_pat_version = -1;

/* Data related to dynamic SI updates */
static S_DYNAMIC_UPDATE_TRANSPORT *dynamic_update_head;
static S_DYNAMIC_UPDATE_TRANSPORT *dynamic_update_tail;

// dvb ssu status data
static BOOLEAN ssu_refused;
static BOOLEAN dvb_ssu_mandatory;
static U8BIT dvb_oui[3] = {0x02, 0x01, 0x5a};

static F_EitParser eit_parser_func = NULL;
static F_BatTableUpdate update_bat_func = NULL;
static F_EitTableUpdate update_eit_func = NULL;
static F_NitTableUpdate update_nit_func = NULL;
static F_PmtTableUpdate update_pmt_func = NULL;
static F_SdtTableUpdate update_sdt_func = NULL;
static F_UntTableUpdate update_unt_func = NULL;

static F_EitSchedUpdateCB eit_sched_update_callback = NULL;

static S_DATABASE_UPDATES_ALLOWED db_updates_allowed = {TRUE, TRUE, TRUE};


//---local function prototypes for this file-----------------------------------
//   (internal functions declared static to make them local)

static void HandleSiEvent(U8BIT path, E_APP_SI_EVENT_TYPE event);
static void CheckForTimeout(U8BIT path, U32BIT timestamp, U32BIT timeout, U32BIT event);
static void ReceiveSiTable(void *filter_ptr, U32BIT ret_param, SI_TABLE_RECORD *table_rec);

static BOOLEAN ManageChannelSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageStartupSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageEventSchedSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageUpdate(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageMonitoring(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageTotSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);

#ifdef COMMON_INTERFACE
static BOOLEAN ManageCIPlusUpdate(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageCIPlusNoPatPmt(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
#endif

static BOOLEAN ProcessPatTablePlayback(SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageUpdatePlayback(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN ManageRecording(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static void StartEITScheduleFilter(U8BIT path);

static BOOLEAN ProcessSdtTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode,
   BOOLEAN ignore_version, BOOLEAN eit_list_reqd);
static BOOLEAN ProcessPatTable(SI_TABLE_RECORD *table_rec);
static BOOLEAN ProcessPmtTable(SI_TABLE_RECORD *table_rec, BOOLEAN ignore_version, BOOLEAN report_pmt,
   U16BIT *service_id, E_DB_ACCESS_MODE mode);
static BOOLEAN InternalProcessPmtTable(U8BIT path, ADB_SERVICE_REC *s_ptr, SI_TABLE_RECORD *table_rec,
   BOOLEAN db_services, E_DB_ACCESS_MODE mode);

static BOOLEAN ProcessNitTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode, BOOLEAN report_nit,
   BOOLEAN transport_changed);

static U32DHMS ReadEventStart(U8BIT *data_ptr);
static U32DHMS ReadEventDuration(U8BIT *data_ptr);
static BOOLEAN DeleteOutOfDateEvents(ADB_SERVICE_REC *s_ptr);
static BOOLEAN UpdateEvents(ADB_EVENT_REC *event_ptr, ADB_SERVICE_REC *s_ptr, BOOLEAN update_only);
static BOOLEAN DeleteEventsForPeriod(ADB_SERVICE_REC *s_ptr, U32DHMS start_time, U32DHMS end_time);

static void ProcessEitTable(SI_TABLE_RECORD *table_rec, BOOLEAN ignore_version, E_DB_ACCESS_MODE mode, BOOLEAN playback);

static void ProcessCatTable(U8BIT path, SI_TABLE_RECORD *table_rec, BOOLEAN report_cat, BOOLEAN transport_changed);
static void ProcessRctTable(SI_TABLE_RECORD *table_rec);
static BOOLEAN ProcessBatTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode, BOOLEAN report_bat);
static void ProcessUntTable(U8BIT path, SI_TABLE_RECORD *table_rec, BOOLEAN report_unt, BOOLEAN transport_changed);

static ADB_IMAGE_ICON* ProcessImageIcon(SI_IMAGE_ICON_DESC *si_icon);

#if defined(INCLUDE_DSMCC_FILE_REQUEST)
static void IconFileRequest(ADB_IMAGE_ICON *icon_ptr,U8BIT path);
#endif

static BOOLEAN MakeNewPmtRequest(U8BIT path);
static U16BIT GetPriorityListId(U8BIT path);
static void CancelTableRequests(U8BIT path, BOOLEAN this_path_only);
static void UpdateTransportParameters(U8BIT path);

static BOOLEAN ManageDvbSsuSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec);
static BOOLEAN CheckSsuLinkageDescForUpgrade(SI_LINKAGE_DESC_ENTRY *desc_ptr, U16BIT *onid_ptr,
   U16BIT *tid_ptr, U16BIT *sid_ptr);

static void CopyComponentTagArray(ADB_STREAM_REC *stream_rec, SI_PMT_STREAM_ENTRY *pmt_entry);
static BOOLEAN HaveStreamsChanged(ADB_STREAM_REC *slist1, ADB_STREAM_REC *slist2);

static void CopyLinkageDesc(SI_LINKAGE_DESC_ENTRY *new_list_ptr, SI_LINKAGE_DESC_ENTRY **list_ptr,
   SI_LINKAGE_DESC_ENTRY **last_entry_ptr, U16BIT *num_linkage_desc);
static void DeleteLinkageDescripterArray(SI_LINKAGE_DESC_ENTRY *list_ptr);

static BOOLEAN DynamicUpdateAddTransport(ADB_TRANSPORT_REC *transport, U16BIT num_lcns, SI_LCN_DESC * lcn_array,
   U16BIT num_hd_lcns, SI_LCN_DESC *hd_lcn_array, U16BIT num_nordig_lcns, SI_NORDIG_LCN_DESC *nordig_lcn_array);
static void DynamicUpdateAddService(ADB_TRANSPORT_REC *transport, ADB_SERVICE_REC *service, BOOLEAN check_move);
static void DynamicUpdateDeleteService(ADB_TRANSPORT_REC *transport, ADB_SERVICE_REC *service, BOOLEAN check_move);
static void ApplyDynamicUpdates(U8BIT path);
static void ClearDynamicUpdates(void);


//--------------------------------------------------------------------------------------------------
// local function definitions
//--------------------------------------------------------------------------------------------------


/**
 *

 *
 * @brief   Application si task
 *

 *

 *
 */
static void HandleSiEvent(U8BIT path, E_APP_SI_EVENT_TYPE event)
{
   BOOLEAN finished;
   U8BIT last_path;

   FUNCTION_START(HandleSiEvent);

   switch (event)
   {
      case STOP_SI:
      {
         AP_SI_PRINT(("APSI Stop(%d)", path));
         if (current_manager[path] != NULL)
         {
            finished = (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
            current_manager[path] = NULL;
         }
         break;
      }
      case START_SI_SEARCHING:
      {
         // check there is no manager running - if there is stop it
         if (current_manager[path] != NULL)
         {
            finished = (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
            current_manager[path] = NULL;
         }
         // decide which manager to start...
         switch (required_si_mode[path])
         {
            case APP_SI_MODE_CHANNEL_SEARCH:
            case APP_SI_MODE_CHANNEL_SEARCH_NO_NIT:
            {
               AP_SI_PRINT(("APSI(%d): channel search", path));
               current_manager[path] = ManageChannelSearch;
               break;
            }

            case APP_SI_MODE_STARTUP_SEARCH:
            {
               AP_SI_PRINT(("APSI(%d): startup search", path));
               current_manager[path] = ManageStartupSearch;
               break;
            }

            case APP_SI_MODE_TOT_SEARCH:
            {
               AP_SI_PRINT(("APSI(%d): tot search", path));
               current_manager[path] = ManageTotSearch;
               break;
            }

            case APP_SI_MODE_EVENT_PF_SEARCH:
            case APP_SI_MODE_EVENT_SCHED_SEARCH:
            case APP_SI_MODE_EVENT_PF_SCHED_SEARCH:
            {
               AP_SI_PRINT(("APSI(%d): schedule search", path));
               current_manager[path] = ManageEventSchedSearch;
               break;
            }

            case APP_SI_MODE_DVB_SSU_SEARCH:
            {
               AP_SI_PRINT(("APSI(%d): dvb ssu search", path));
               current_manager[path] = ManageDvbSsuSearch;
               break;
            }

            #ifdef COMMON_INTERFACE
            case APP_SI_MODE_CIPLUS_UPDATE:
            {
               AP_SI_PRINT(("APSI(%d): CI+ update search", path));
               current_manager[path] = ManageCIPlusUpdate;
               break;
            }

            case APP_SI_MODE_CIPLUS_NO_PAT_PMT:
            {
               AP_SI_PRINT(("APSI(%d): CI+ no PAT/PMT search", path));
               current_manager[path] = ManageCIPlusNoPatPmt;
               break;
            }
            #endif

            case APP_SI_MODE_USER_DEFINED:
            {
               AP_SI_PRINT(("APSI(%d): user defined search", path));
               break;
            }

            case APP_SI_MODE_NO_SI:
            case APP_SI_MODE_UPDATE:
            default:
            {
               AP_SI_PRINT(("APSI(%d): search failed - invalid mode %d", path, required_si_mode[path]));
               break;
            }
         }
         // start the new manager
         if (current_manager[path] != NULL)
         {
            finished = (current_manager[path])(path, APP_SI_START_MANAGER, NULL);
            if (finished == TRUE)
            {
               current_manager[path] = NULL;
            }
         }
         break;
      }

      case START_SI_UPDATING_NEW_TRANSPORT:
      {
         if (ACFG_GetCountry() == COUNTRY_CODE_UK)
         {
            ADB_TRANSPORT_REC *t_ptr;

            /* When changing transport the sdt versions stored in the transport are invalid:
             * "There is no requirement that actual and other tables should carry the
             * same version number." (Dbook7) */
            DBDEF_RequestAccess();
            t_ptr = DBDEF_GetNextTransportRec(NULL);
            while (t_ptr != NULL)
            {
               t_ptr->sdt_version = 0xFF;
               t_ptr = DBDEF_GetNextTransportRec(t_ptr);
            }
            DBDEF_ReleaseAccess();
         }

         // check there is no manager running - if there is stop it
         if (current_manager[path] != NULL)
         {
            finished = (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
            current_manager[path] = NULL;
         }

         if (required_si_mode[path] == APP_SI_MODE_UPDATE)
         {
            finished = FALSE;

            if (STB_DPIsLivePath(path))
            {
               AP_SI_PRINT(("APSI(%d): live", path));
               current_manager[path] = ManageUpdate;
               finished = ManageUpdate(path, APP_SI_START_MANAGER, NULL);
            }
            else if (STB_DPIsRecordingPath(path))
            {
               AP_SI_PRINT(("APSI(%d): recording", path));
               current_manager[path] = ManageRecording;
               finished = ManageRecording(path, APP_SI_START_MANAGER, NULL);
            }
            else if (path == APVR_GetPlaybackPath())
            {
               AP_SI_PRINT(("APSI(%d): update for playback", path));
               current_manager[path] = ManageUpdatePlayback;
               finished = ManageUpdatePlayback(path, APP_SI_START_MANAGER, NULL);
            }
            else if (STB_DPIsMonitoringPath(path))
            {
               AP_SI_PRINT(("APSI(%d): monitoring", path));
               current_manager[path] = ManageMonitoring;
               finished = ManageMonitoring(path, APP_SI_START_MANAGER, NULL);
            }
            else
            {
               AP_SI_PRINT(("APSI(%d): update", path));
               current_manager[path] = ManageUpdate;
               finished = ManageUpdate(path, APP_SI_START_MANAGER, NULL);
            }

            if (finished == TRUE)
            {
               current_manager[path] = NULL;
            }
         }
         else
         {
            AP_SI_PRINT(("APSI(%d): update failed - invalid mode %d", path, required_si_mode[path]));
         }
         break;
      }

      case START_SI_UPDATING_SAME_TRANSPORT:
      {
         // the ManageUpdate manager should be running
         if (required_si_mode[path] == APP_SI_MODE_UPDATE)
         {
            finished = FALSE;

            if (!STB_DPIsLivePath(path) && STB_DPIsRecordingPath(path))
            {
               AP_SI_PRINT(("APSI(%d): recording", path));

               if (current_manager[path] != NULL)
               {
                  finished = (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
                  current_manager[path] = NULL;
               }

               current_manager[path] = ManageRecording;
               finished = ManageRecording(path, APP_SI_START_MANAGER, NULL);
            }
            else
            {
               /* SI is only monitored on the live path */
               if (STB_DPIsLivePath(path))
               {
                  AP_SI_PRINT(("APSI(%d): update", path));

                  if (current_manager[path] != ManageUpdate)
                  {
                     // if current manager is not ManageUpdate stop it and start ManageUpdate
                     if (current_manager[path] != NULL)
                     {
                        (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
                     }

                     current_manager[path] = ManageUpdate;
                     finished = ManageUpdate(path, APP_SI_START_MANAGER, NULL);
                  }
                  else
                  {
                     // ManageUpdate is running - tell it about the channel change
                     finished = ManageUpdate(path, APP_SI_CHANNEL_CHANGE, NULL);
                  }
               }
               else if (STB_DPIsMonitoringPath(path))
               {
                  AP_SI_PRINT(("APSI(%d): monitoring", path));

                  if (current_manager[path] != ManageMonitoring)
                  {
                     // if current manager is not ManageUpdate stop it and start ManageUpdate
                     if (current_manager[path] != NULL)
                     {
                        (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
                     }

                     current_manager[path] = ManageMonitoring;
                     finished = ManageMonitoring(path, APP_SI_START_MANAGER, NULL);
                  }
                  else
                  {
                     // ManageUpdate is running - tell it about the channel change
                     finished = ManageMonitoring(path, APP_SI_CHANNEL_CHANGE, NULL);
                  }
               }
               else if (path == APVR_GetPlaybackPath())
               {
                  AP_SI_PRINT(("APSI(%d): update for playback", path));

                  if (current_manager[path] != ManageUpdatePlayback)
                  {
                     // if current manager is not ManageUpdatePlayback stop it and start ManageUpdatePlayback
                     (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
                     current_manager[path] = ManageUpdatePlayback;
                     finished = ManageUpdatePlayback(path, APP_SI_START_MANAGER, NULL);
                  }
                  else
                  {
                     // ManageUpdatePlayback is running - tell it about the channel change
                     finished = ManageUpdatePlayback(path, APP_SI_CHANNEL_CHANGE, NULL);
                  }
               }
            }

            if (finished == TRUE)
            {
               current_manager[path] = NULL;
            }
         }
         else
         {
            AP_SI_PRINT(("APSI(%d): update failed - invalid mode %d", path, required_si_mode[path]));
            if (current_manager[path] != NULL)
            {
               (current_manager[path])(path, APP_SI_STOP_MANAGER, NULL);
               current_manager[path] = NULL;
            }
         }
         break;
      }

      case SI_TIMEOUT:
      {
         U32BIT time_now, time_adjust;
         /* No events received for 1 second */
         if (path == INVALID_RES_ID)
         {
            /* Check timeouts on all paths */
            path = 0;
            last_path = STB_DPGetNumPaths();
         }
         else
         {
            /* Only check timeout for the specified path */
            last_path = path + 1;
         }

         for (; path < last_path; path++)
         {
            time_now = STB_OSGetClockMilliseconds();
            if ((STB_DPGetPathTuner(path) != INVALID_RES_ID) && (STB_DPGetTuneStatus(path) == TUNE_LOCKED))
            {
               time_adjust = 0;
            }
            else
            {
               time_adjust = time_now - last_timestamp[path];
            }
            last_timestamp[path] = time_now;

            if (current_manager[path] != NULL)
            {
               if (time_adjust != 0)
               {
                  if (cat_start_timestamp[path] != 0)
                  {
                     cat_start_timestamp[path] += time_adjust;
                  }
                  if (pat_start_timestamp[path] != 0)
                  {
                     pat_start_timestamp[path] += time_adjust;
                  }
                  if (pmt_start_timestamp[path] != 0)
                  {
                     pmt_start_timestamp[path] += time_adjust;
                  }
                  if (eit_start_timestamp[path] != 0)
                  {
                     eit_start_timestamp[path] += time_adjust;
                  }
                  if (sched_start_timestamp[path] != 0)
                  {
                     sched_start_timestamp[path] += time_adjust;
                  }
                  if (pmt_update_timestamp[path] != 0)
                  {
                     pmt_update_timestamp[path] += time_adjust;
                  }
                  if (si_update_delay_timestamp[path] != 0)
                  {
                     si_update_delay_timestamp[path] += time_adjust;
                  }
                  if (nit_start_timestamp[path] != 0)
                  {
                     nit_start_timestamp[path] += time_adjust;
                  }
                  if (sdt_start_timestamp[path] != 0)
                  {
                     sdt_start_timestamp[path] += time_adjust;
                  }
                  if (tot_start_timestamp[path] != 0)
                  {
                     tot_start_timestamp[path] += time_adjust;
                  }
                  if (tdt_start_timestamp[path] != 0)
                  {
                     tdt_start_timestamp[path] += time_adjust;
                  }
                  if (bat_start_timestamp[path] != 0)
                  {
                     bat_start_timestamp[path] += time_adjust;
                  }
                  if (unt_start_timestamp[path] != 0)
                  {
                     unt_start_timestamp[path] += time_adjust;
                  }
               }
               // check filter timeouts and pass on to current manager
               CheckForTimeout(path, cat_start_timestamp[path], CAT_TIMEOUT_MS, APP_SI_CAT_TIMEOUT);
               CheckForTimeout(path, pat_start_timestamp[path], PAT_TIMEOUT_MS, APP_SI_PAT_TIMEOUT);
               CheckForTimeout(path, pmt_start_timestamp[path], PMT_TIMEOUT_MS, APP_SI_PMT_TIMEOUT);
               CheckForTimeout(path, eit_start_timestamp[path], EIT_TIMEOUT_MS, APP_SI_EIT_TIMEOUT);
               CheckForTimeout(path, sched_start_timestamp[path], sched_timeout_ms[path],
                  APP_SI_SCHED_TIMEOUT);

               if (path != STB_DPGetPlaybackPath())
               {
                  CheckForTimeout(path, nit_start_timestamp[path], NIT_TIMEOUT_MS, APP_SI_NIT_TIMEOUT);
                  CheckForTimeout(path, sdt_start_timestamp[path], SDT_TIMEOUT_MS, APP_SI_SDT_TIMEOUT);
                  CheckForTimeout(path, tot_start_timestamp[path], TOT_TIMEOUT_MS, APP_SI_TOT_TIMEOUT);
                  CheckForTimeout(path, tdt_start_timestamp[path], TDT_TIMEOUT_MS, APP_SI_TDT_TIMEOUT);
                  CheckForTimeout(path, bat_start_timestamp[path], BAT_TIMEOUT_MS, APP_SI_BAT_TIMEOUT);
                  CheckForTimeout(path, unt_start_timestamp[path], UNT_TIMEOUT_MS, APP_SI_UNT_TIMEOUT);
               }

               // check control timeouts
               CheckForTimeout(path, pmt_update_timestamp[path], pmt_update_period_ms[path], APP_SI_PMT_UPDATE);
               CheckForTimeout(path, si_update_delay_timestamp[path], SI_UPDATE_DELAY_MS, APP_SI_UPDATE_DELAY_EXPIRED);

               // check application commands
               if (stop_pmt_reporting[path] == TRUE)
               {
                  stop_pmt_reporting[path] = FALSE;
                  finished = (current_manager[path])(path, APP_SI_STOP_PMT_REPORTING, NULL);
                  if (finished == TRUE)
                  {
                     current_manager[path] = NULL;
                  }
               }
            }
         }
         break;
      }
   }

   FUNCTION_FINISH(HandleSiEvent);
}

/**
 *

 *
 * @brief   Checks for timeout on the specified filter
 *
 * @param   timestamp - the start timestamp associated with the filter
 * @param   timeout   - the timeout period in milliseconds
 * @param   event     - the event to be sent to the current manager if a timeout has occurred
 *

 *
 */
static void CheckForTimeout(U8BIT path, U32BIT timestamp, U32BIT timeout, U32BIT event)
{
   BOOLEAN finished;

   FUNCTION_START(CheckForTimeout);

   if (timestamp != 0)
   {
      if (STB_OSGetClockDiff(timestamp) > timeout)
      {
         finished = (current_manager[path])(path, event, NULL);
         if (finished == TRUE)
         {
            current_manager[path] = NULL;
         }
      }
   }
   FUNCTION_FINISH(CheckForTimeout);
}

/**
 *

 *
 * @brief   callback function for si filter handling in STB layer to use to report si table
 *                arrived. Passes table on to manager
 *
 * @param   filter    - filter handle for filter which produced the table
 * @param   ret_param - return parameter setup when table requested
 * @param   table_rec - pointer to the table record received
 *

 *
 */
static void ReceiveSiTable(void *filter_ptr, U32BIT ret_param, SI_TABLE_RECORD *table_rec)
{
   U8BIT path;

   FUNCTION_START(ReceiveSiTable);

   USE_UNWANTED_PARAM(filter_ptr);

   if (table_rec != NULL)
   {
      path = table_rec->path;
   }
   else
   {
      AP_SI_PRINT(("===ReceiveSiTable: table_rec is NULL, so path = 0\n"));
      path = 0;
   }

   if (current_manager[path] != NULL)
   {
      // ret_param will contain the event code relevant for the table e.g. APP_SI_PAT_RECEIVED
      // pass event and table_rec to manager
      (current_manager[path])(path, ret_param, table_rec);
   }
   else
   {
      // there should always be a current manager, but in case there isn't release table_rec here
      // otherwise it won't get released and cancel all filters because there shouldn't be any
      // running
      STB_SIReleaseTableRecord(table_rec);
      CancelTableRequests(path, TRUE);
   }

   FUNCTION_FINISH(ReceiveSiTable);
}

/**
 *

 *
 * @brief   Manager for the full search algorithm
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageChannelSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   U32BIT freq;
   BOOLEAN finished;
   BOOLEAN success;
   U16BIT i;
   E_STB_DP_SIGNAL_TYPE tuner_type;
   BOOLEAN stop_search;
   U8BIT plp_id;
   BOOLEAN changed = FALSE;
   ADB_TRANSPORT_REC * t_ptr = NULL;

   FUNCTION_START(ManageChannelSearch);

   finished = FALSE;
   if (event == APP_SI_START_MANAGER)
   {
      // clear service list ready flag
      service_list_ready[path] = FALSE;

      // first ensure there is a transport record for the transport we are tuned to and set
      // current transport accordingly
      freq = STB_DPGetFrequency(path);
      tuner_type = STB_DPGetSignalType(path);

      DBDEF_RequestAccess();

      switch (tuner_type)
      {
         case SIGNAL_COFDM:
            plp_id = STB_DPGetTerrPLP(path);
            current_transport_rec[path] = DBDEF_FindTerrestrialTransportRec(freq, plp_id);
            if (current_transport_rec[path] == NULL)
            {
               /* Transport does not exist - add one */
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): start - new terrestrial transport (%s), PLP=%u", path,
                            ACTL_GetRfNameFromFreq(tuner_type, freq), plp_id));
               #endif
               current_transport_rec[path] = DBDEF_AddTerrestrialTransportRec(freq, plp_id, NULL);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%d): start - existing transport (%s)", path,
                            ACTL_GetRfNameFromFreq(tuner_type, freq)));
            }
            #endif
            break;
         case SIGNAL_QAM:
            current_transport_rec[path] = DBDEF_FindCableTransportRec(freq, SYMBOL_RATE_AUTO /*STB_DPGetSymbolRate(path)*/);
            if (current_transport_rec[path] == NULL)
            {
               /* Transport does not exist - add one */
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): start - new cable transport (%s), sym_rate=%u", path,
                            ACTL_GetRfNameFromFreq(tuner_type, freq), STB_DPGetSymbolRate(path)));
               #endif
               current_transport_rec[path] = DBDEF_AddCableTransportRec(freq,
                  STB_DPGetSymbolRate(path), NULL);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%d): start - existing transport (%s)", path,
                            ACTL_GetRfNameFromFreq(tuner_type, freq)));
            }
            #endif
            break;
         case SIGNAL_QPSK:
            current_transport_rec[path] = DBDEF_FindSatTransportRec(freq, STB_DPGetSymbolRate(path),
                  STB_DPGetPolarity(path), STB_DPGetDVBS2(path), STB_DPGetModulation(path),
                  ACTL_GetCurrentSatellite(path));
            if (current_transport_rec[path] == NULL)
            {
               /* Transport does not exist - add one */
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): start - new sat transport (%s), sym_rate=%u, polarity=%u",
                            path, ACTL_GetRfNameFromFreq(tuner_type, freq), STB_DPGetSymbolRate(path),
                            STB_DPGetPolarity(path)));
               #endif
               current_transport_rec[path] = DBDEF_AddSatTransportRec(freq, STB_DPGetSymbolRate(path),
                     STB_DPGetPolarity(path), STB_DPGetDVBS2(path), STB_DPGetModulation(path), NULL);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%d): start - existing transport (%s)", path,
                            ACTL_GetRfNameFromFreq(tuner_type, freq)));
            }
            #endif
            break;
         default:
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%u): Unknown tuner type, %u", path, tuner_type));
            #endif
            current_transport_rec[path] = NULL;
            break;
      }

      DBDEF_ReleaseAccess();

      if (current_transport_rec[path] != NULL)
      {
         DBDEF_SetTunedTransport(path, current_transport_rec[path]);

         current_network_rec[path] = current_transport_rec[path]->network;
         current_service_rec[path] = NULL;

         pat_rcvd_on_this_trnsprt[path] = FALSE;
         last_transport_rec[path] = NULL;

        if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH) {
            pmts_complete[path] = TRUE;
            sdt_complete[path] = FALSE;
            nit_complete[path] = FALSE;
            eits_complete[path] = TRUE;
            tot_complete[path] = TRUE;
            bat_complete[path] = FALSE;
        }
        else if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH_NO_NIT) {
            pmts_complete[path] = FALSE;
            sdt_complete[path] = TRUE;
            nit_complete[path] = TRUE;
            eits_complete[path] = FALSE;
            tot_complete[path] = FALSE;
            bat_complete[path] = TRUE;
        }
        else {
            pmts_complete[path] = FALSE;
            sdt_complete[path] = FALSE;
            nit_complete[path] = FALSE;
            eits_complete[path] = FALSE;
            tot_complete[path] = FALSE;
            bat_complete[path] = FALSE;
        }
        
        if (bat_complete[path] == FALSE) {
            for (i = 0; i < num_active_bouquet_ids; i++)
                active_bouquet_ids[i].received = FALSE;
        }

#ifdef COMMON_INTERFACE
         /* Ensure the routing of the TS is correct for acquiring the SDT for CI+ */
         if (ACI_SetSecureRouting(path))
#endif
         {
            if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH) {
                /* Start search for NIT */
                nit_start_timestamp[path] = STB_OSGetClockMilliseconds();
                nit_filter[path] = STB_SIRequestNit(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_NIT_RECEIVED);
            }
            else {
                /* Request TOT and TDT as TOT isn't always available */
                tot_start_timestamp[path] = STB_OSGetClockMilliseconds();
                tot_filter[path] = STB_SIRequestTot(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);

                tdt_start_timestamp[path] = STB_OSGetClockMilliseconds();
                tdt_filter[path] = STB_SIRequestTdt(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TDT_RECEIVED);
            }
         }
#ifdef COMMON_INTERFACE
         else
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): start - failed to route TS securely", path));
            #endif
            // report end of search
            STB_SISearchComplete(path, FALSE, NULL, 0);
            finished = TRUE;
         }
#endif
      }
      else
      {
         #ifdef DEBUG_SI_CHAN_SEARCH
         AP_SI_PRINT(("Chan search(%d): start - failed, no transport", path));
         #endif
         // report end of search
         STB_SISearchComplete(path, FALSE, NULL, 0);
         finished = TRUE;
      }
   }
   else if (event == APP_SI_STOP_MANAGER)
   {
      #ifdef DEBUG_SI_CHAN_SEARCH
      AP_SI_PRINT(("Chan search(%d): stop", path));
      #endif
      CancelTableRequests(path, FALSE);
      finished = TRUE;
   }
   else
   {
      switch (event)
      {
         case APP_SI_SDT_RECEIVED:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): SDT received (tid 0x%04x)", path, table_rec->xtid));
            #endif
            changed = ProcessSdtTable(table_rec, DB_ACCESS_SEARCH, IGNORE_VERSION, EIT_LIST_REQD);

            tuner_type = STB_DPGetSignalType(path);
            sdt_complete[path] = TRUE;

            DBDEF_RequestAccess();
            /* Iterate through transports */
            t_ptr = DBDEF_GetNextTransportRec(NULL);
            while (t_ptr != NULL)
            {
               if (t_ptr->sig_type == tuner_type) {
                  if (FALSE == t_ptr->sdt_received) {
                     sdt_complete[path] = FALSE;
                     break;
                  }
               }
               t_ptr = DBDEF_GetNextTransportRec(t_ptr);
            }
            DBDEF_ReleaseAccess();

            /* Finished with the filter - release it and flag sdt complete */
            if (TRUE == sdt_complete[path]) {
               service_list_ready[path] = TRUE;

               STB_SICancelTableRequest(sdt_filter[path]);
               sdt_filter[path] = NULL;
               sdt_start_timestamp[path] = 0;
               
               if (FALSE == bat_complete[path]) {
                   if (use_bats_active)
                   {
                      /* Start search for the BATs */
                      bat_start_timestamp[path] = STB_OSGetClockMilliseconds();
                      bat_filter[path] = STB_SIRequestBat(path, CONTINUOUS_REQUEST, DONT_CARE_ID_MATCH,
                            DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_BAT_RECEIVED);
                   }
               }
            }
            break;
         }

         case APP_SI_SDT_TIMEOUT:
         {
            /* timeout - no point in searching any further if there is no SDT. Set the complete
             * flags to indicate end of search */
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): SDT timeout", path));
            #endif

            /* Finished with the filter - release it and flag sdt complete */
            STB_SICancelTableRequest(sdt_filter[path]);
            sdt_filter[path] = NULL;
            sdt_start_timestamp[path] = 0;
            finished = TRUE;
            break;
         }

         case APP_SI_PAT_RECEIVED:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): PAT received (tid 0x%04x)", path, table_rec->xtid));
            #endif
            ProcessPatTable(table_rec);
            if (pmt_list[path] != NULL)
            {
               pmt_list_id[path] = 0;
               pmt_request_mode[path] = PMT_REQUEST_MONITOR;
               MakeNewPmtRequest(path);
            }
            else
            {
               pmts_complete[path] = TRUE;
            }

            /* Finished with the filter - release it */
            STB_SICancelTableRequest(pat_filter[path]);
            pat_filter[path] = NULL;
            pat_start_timestamp[path] = 0;
            break;
         }

         case APP_SI_PAT_TIMEOUT:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): PAT timeout", path));
            #endif

            /* Finished with the filter - release it */
            STB_SICancelTableRequest(pat_filter[path]);
            pat_filter[path] = NULL;
            pat_start_timestamp[path] = 0;
            finished = TRUE;
            break;
         }

         case APP_SI_PMT_RECEIVED:
         case APP_SI_PMT_TIMEOUT:
         {
            if (event == APP_SI_PMT_RECEIVED)
            {
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): PMT received (sid 0x%04x)", path, table_rec->xtid));
               #endif
               // process the pmt (regardless of version and do not report to other parties)
               ProcessPmtTable(table_rec, IGNORE_VERSION, DONT_REPORT_PMT, NULL, DB_ACCESS_SEARCH);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%d): PMT timeout (sid 0x%04x)", path,
                            pmt_list[path][pmt_list_id[path]].serv_id));
            }
            #endif

            if ((pmt_list_id[path] + 1) >= num_pmt_list_entries[path])
            {
               // finished list - release the filter and flag pmts complete
               pmts_complete[path] = TRUE;
               STB_SICancelTableRequest(pmt_filter[path]);
               pmt_filter[path] = NULL;
               pmt_start_timestamp[path] = 0;
            }
            else
            {
               // not finished yet - move on to next pmt...
               pmt_list_id[path]++;
               pmt_request_mode[path] = PMT_REQUEST_MONITOR;
               MakeNewPmtRequest(path);
            }
            break;
         }

         case APP_SI_NIT_RECEIVED:
         case APP_SI_NIT_TIMEOUT:
         {
            if (event == APP_SI_NIT_RECEIVED)
            {
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): NIT received (nid 0x%04x)", path, table_rec->xtid));
               #endif

               if (0xFFFF /*private network ID*/ == table_rec->xtid) {
                  break;
               }
               else if (table_rec->xtid < 0x10) {
                  // accept CDCAS test streams, network id: 0x01. 0x04
               }
               else if (0xA000 /*CNS network ID*/ != (table_rec->xtid & 0xFF00)) {
                  // FIXME: hardcode CNS network ID, 0xA030, 0xA031
                  break;
               }
               
               ProcessNitTable(table_rec, DB_ACCESS_SEARCH, FALSE, FALSE);
               nit_complete[path] = TRUE;
               AP_SI_PRINT(("Update(%u): NIT changed; restarting SDT collection", path));
               if (FALSE == sdt_complete[path]) {
                  sdt_start_timestamp[path] = STB_OSGetClockMilliseconds();
                  sdt_filter[path] = STB_SIRequestSdt(path, CONTINUOUS_REQUEST, TRUE, TRUE, DONT_CARE_ID_MATCH,
                      DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);
               }
            }
            else
            {
               if (current_transport_rec[path]->sig_type == SIGNAL_QPSK)
               {
                  DBDEF_RequestAccess();
                  current_network_rec[path] = DBDEF_FindOrAddPrivateNetwork(ACTL_GetCurrentSatellite(path));
                  if (current_network_rec[path] != NULL)
                  {
                     DBDEF_SetTunedNetwork(path, current_network_rec[path]);

                     /* Link the current transport to the network */
                     current_transport_rec[path]->network = current_network_rec[path];
                     DBA_SetRecordParent(current_transport_rec[path]->dba_rec,
                        current_network_rec[path]->dba_rec);
                     DBA_SaveRecord(current_transport_rec[path]->dba_rec);
                  }
                  DBDEF_ReleaseAccess();
               }
               // FIXME: for CNS network search
               if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH)
                  finished = TRUE;
               AP_SI_PRINT(("Chan search(%d): NIT timeout", path));
            }

            // finished with the filter - release it and flag nit complete
            STB_SICancelTableRequest(nit_filter[path]);
            nit_filter[path] = NULL;
            nit_start_timestamp[path] = 0;
            break;
         }

         case APP_SI_BAT_RECEIVED:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): BAT received (bouquet id 0x%04x, v%d)", path,
                         table_rec->xtid, table_rec->version));
            #endif
            ProcessBatTable(table_rec, DB_ACCESS_SEARCH, FALSE);

            /* Check whether all BATs have now been received.
             * If all BATs are to be collected then the search will continue until
             * there's a timeout because we don't know how many BATs there are */
            stop_search = FALSE;

            if (active_bouquet_ids != NULL)
            {
               stop_search = TRUE;

               for (i = 0; i < num_active_bouquet_ids; i++)
               {
                  if (!active_bouquet_ids[i].received)
                  {
                     /* Found one that hasn't been received yet */
                     stop_search = FALSE;
                     break;
                  }
               }
            }

            if (stop_search)
            {
               bat_complete[path] = TRUE;
               STB_SICancelTableRequest(bat_filter[path]);
               bat_filter[path] = NULL;
               bat_start_timestamp[path] = 0;

#if 1
               /* Start search for PAT */
               pat_start_timestamp[path] = STB_OSGetClockMilliseconds();
               pat_filter[path] = STB_SIRequestPat(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);

               /* Start search for eits on current transport only */
               eit_start_timestamp[path] = STB_OSGetClockMilliseconds();
               eit_filter[path] = STB_SIRequestEit(path, ONE_SHOT_REQUEST, EIT_NOW_NEXT_ACT,
                  DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff,
                  ReceiveSiTable, APP_SI_EIT_RECEIVED);
#endif
            }
            else
            {
               /* More BATs are needed, continue collecting */
               bat_start_timestamp[path] = STB_OSGetClockMilliseconds();
            }
            break;
         }

         case APP_SI_BAT_TIMEOUT:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): BAT timeout", path));
            #endif
            // finished with the filter - release it and flag BAT complete
            bat_complete[path] = TRUE;
            STB_SICancelTableRequest(bat_filter[path]);
            bat_filter[path] = NULL;
            bat_start_timestamp[path] = 0;
            // FIXME: for CNS network search
            if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH) {
               finished = TRUE;
            }
            else if (required_si_mode[path] == APP_SI_MODE_CHANNEL_SEARCH_NO_NIT) {
                ADB_NETWORK_REC * n_ptr = DBDEF_GetTunedNetwork(path);
               if (NULL != n_ptr) {
                  //FIXME: check CDCAS test streams, network ID: [1, 4]                  
                  if (n_ptr->net_id < 0x10 ) {

                  }
               }
               finished = TRUE;            
            }
            break;
         }

         case APP_SI_EIT_RECEIVED:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): EIT received (sid 0x%04x)", path, table_rec->xtid));
            #endif
            ProcessEitTable(table_rec, IGNORE_VERSION, DB_ACCESS_SEARCH, FALSE);

            // check if we have received all eits yet
            if (eit_list[path] != NULL)
            {
               eits_complete[path] = TRUE;
               for (i = 0; i < num_eit_list_entries[path]; i++)
               {
                  if (eit_list[path][i].got_eit == FALSE)
                  {
                     eits_complete[path] = FALSE;
                     break;
                  }
               }
            }
            if (eits_complete[path] == TRUE)
            {
               STB_SICancelTableRequest(eit_filter[path]);
               eit_filter[path] = NULL;
               eit_start_timestamp[path] = 0;

               STB_SICancelTableRequest(sched_filter[path]);
               sched_filter[path] = NULL;
               sched_start_timestamp[path] = 0;
            }
            break;
         }

         case APP_SI_EIT_TIMEOUT:
         {
            #ifdef DEBUG_SI_CHAN_SEARCH
            AP_SI_PRINT(("Chan search(%d): EIT timeout", path));
            #endif
            // finished with the filter - release it and flag nit complete
            eits_complete[path] = TRUE;
            if (eit_filter[path] != NULL)
            {
               STB_SICancelTableRequest(eit_filter[path]);
               eit_filter[path] = NULL;
               eit_start_timestamp[path] = 0;

               STB_SICancelTableRequest(sched_filter[path]);
               sched_filter[path] = NULL;
               sched_start_timestamp[path] = 0;
            }
            break;
         }

         case APP_SI_TOT_RECEIVED:
         case APP_SI_TOT_TIMEOUT:
         {
            if (event == APP_SI_TOT_RECEIVED)
            {
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%d): TOT received", path));
               #endif
               ASI_ProcessTotTable(table_rec);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%d): TOT timeout", path));
            }
            #endif

            // finished with the filter - release it and flag tot complete
            tot_complete[path] = TRUE;
            STB_SICancelTableRequest(tot_filter[path]);
            tot_filter[path] = NULL;
            tot_start_timestamp[path] = 0;
            break;
         }

         case APP_SI_TDT_RECEIVED:
         case APP_SI_TDT_TIMEOUT:
         {
            if (event == APP_SI_TDT_RECEIVED)
            {
               #ifdef DEBUG_SI_CHAN_SEARCH
               AP_SI_PRINT(("Chan search(%u): TDT received", path));
               #endif

               ASI_ProcessTdtTable(table_rec);
            }
            #ifdef DEBUG_SI_CHAN_SEARCH
            else
            {
               AP_SI_PRINT(("Chan search(%u): TDT timeout", path));
            }
            #endif

            /* Finished with the filter - release it and flag TOT complete
             * because it doesn't matter at this stage which has been seen */
            tot_complete[path] = TRUE;
            STB_SICancelTableRequest(tdt_filter[path]);
            tdt_filter[path] = NULL;
            tdt_start_timestamp[path] = 0;
            break;
         }
      }

      // check if search is complete - if so report fact
      if (finished || (sdt_complete[path] && nit_complete[path] && pmts_complete[path] &&
                       eits_complete[path] && tot_complete[path] && bat_complete[path]))
      {
         if (current_transport_rec[path] != NULL)
         {
            UpdateTransportParameters(path);
         }

         // report end of search
         #ifdef DEBUG_SI_CHAN_SEARCH
         AP_SI_PRINT(("Chan search(%d): search complete", path));
         #endif
         CancelTableRequests(path, FALSE);

         success = (sdt_complete[path] && nit_complete[path] && pmts_complete[path] &&
                    eits_complete[path] && tot_complete[path] && bat_complete[path]);

         STB_SISearchComplete(path, success, &current_transport_rec[path], sizeof(void *));
         finished = TRUE;
      }
   }

   FUNCTION_FINISH(ManageChannelSearch);
   return(finished);
}

/**
 *

 *
 * @brief   Manager for the TOT search. This search updates the system time on startup by
 *                searching for a valid TOT
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *
 * @return   TRUE when the manager has finished, FALSE otherwise
 *
 */
static BOOLEAN ManageTotSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;

   FUNCTION_START(ManageTotSearch);

   finished = FALSE;
   if (event == APP_SI_START_MANAGER)
   {
      // first ensure there is a transport record for the transport we are tuned to and set
      // current transport accordingly
      current_transport_rec[path] = DBDEF_GetTunedTransport(path);
      if (current_transport_rec[path] != NULL)
      {
         #ifdef DEBUG_SI_TOT_SEARCH
         AP_SI_PRINT(("Tot search(%d): start (%s)", path,
            ACTL_GetRfNameFromFreq(current_transport_rec[path]->sig_type, current_transport_rec[path]->frequency)));
         #endif

         /* The TOT only needs to be requested once on startup,
          * so don't request it if it's already been received */
         if (tot_already_received[path])
         {
            tot_complete[path] = TRUE;
            finished = TRUE;
            STB_SISearchComplete(path, TRUE, NULL, 0);
         }
         else
         {
            // start search for tot
            tot_complete[path] = FALSE;
            tot_start_timestamp[path] = STB_OSGetClockMilliseconds();
            tot_filter[path] = STB_SIRequestTot(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);
         }
      }
      else
      {
         #ifdef DEBUG_SI_TOT_SEARCH
         AP_SI_PRINT(("Tot search(%d): start - failed, no transport", path));
         #endif
         // report end of search
         STB_SISearchComplete(path, FALSE, NULL, 0);
         finished = TRUE;
      }
   }
   else if (event == APP_SI_STOP_MANAGER)
   {
      #ifdef DEBUG_SI_TOT_SEARCH
      AP_SI_PRINT(("Tot search(%d): stop", path));
      #endif
      CancelTableRequests(path, FALSE);
      finished = TRUE;
   }
   else
   {
      switch (event)
      {
         case APP_SI_TOT_RECEIVED:
         {
            #ifdef DEBUG_SI_TOT_SEARCH
            AP_SI_PRINT(("Tot search(%d): TOT received, search complete", path));
            #endif
            ASI_ProcessTotTable(table_rec);
            tot_already_received[path] = TRUE;
            STB_SISearchComplete(path, TRUE, NULL, 0);
            break;
         }
         case APP_SI_TOT_TIMEOUT:
         {
            #ifdef DEBUG_SI_TOT_SEARCH
            AP_SI_PRINT(("Tot search(%d): TOT timeout, search complete", path));
            #endif
            STB_SISearchComplete(path, FALSE, NULL, 0);
         }
      }

      finished = TRUE;
      // finished with the filter - release it and flag tot complete
      tot_complete[path] = TRUE;
      STB_SICancelTableRequest(tot_filter[path]);
      tot_filter[path] = NULL;
      tot_start_timestamp[path] = 0;
   }

   FUNCTION_FINISH(ManageTotSearch);
   return(finished);
}

/**
 *

 *
 * @brief   Manager for the startup search. This search updates the fields not stored in nvm
 *                on startup to fill in the missing information. It extracts the SDT, PAT, PMT and
 *                EIT tables. In processing the SDT only the non-NVM fields of existing service
 *                records are updated - no new records are added.
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageStartupSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   U16BIT i;

   FUNCTION_START(ManageStartupSearch);

   finished = FALSE;
   if (event == APP_SI_START_MANAGER)
   {
      // clear service list ready flag
      service_list_ready[path] = FALSE;

      // first ensure there is a transport record for the transport we are tuned to and set
      // current transport accordingly
      current_transport_rec[path] = DBDEF_GetTunedTransport(path);
      if (current_transport_rec[path] != NULL)
      {
         #ifdef DEBUG_SI_STARTUP_SEARCH
         AP_SI_PRINT(("Startup search(%d): start (%s)", path,
            ACTL_GetRfNameFromFreq(current_transport_rec[path]->sig_type, current_transport_rec[path]->frequency)));
         #endif

         current_network_rec[path] = current_transport_rec[path]->network;
         current_service_rec[path] = NULL;

         pat_rcvd_on_this_trnsprt[path] = FALSE;
         last_transport_rec[path] = NULL;
         pmts_complete[path] = FALSE;
         sdt_complete[path] = FALSE;
         eits_complete[path] = FALSE;
         tot_complete[path] = FALSE;

#ifdef COMMON_INTERFACE
         /* Ensure the routing of the TS is correct for acquiring the SDT for CI+ */
         if (ACI_SetSecureRouting(path))
#endif
         {
            // start search for sdt
            sdt_start_timestamp[path] = STB_OSGetClockMilliseconds();
            sdt_filter[path] = STB_SIRequestSdt(path, ONE_SHOT_REQUEST, TRUE, FALSE, DONT_CARE_ID_MATCH,
                  DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);

            /* The TOT only needs to be requested once on startup,
             * so don't request it if it's already been received */
            if (tot_already_received[path])
            {
               tot_complete[path] = TRUE;
            }
            else
            {
               // start search for tot
               tot_start_timestamp[path] = STB_OSGetClockMilliseconds();
               tot_filter[path] = STB_SIRequestTot(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);
            }

            // start search for PAT
            pat_start_timestamp[path] = STB_OSGetClockMilliseconds();
            pat_filter[path] = STB_SIRequestPat(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);
         }
#ifdef COMMON_INTERFACE
         else
         {
            #ifdef DEBUG_SI_STARTUP_SEARCH
            AP_SI_PRINT(("Startup search(%d): start - failed to route TS securely", path));
            #endif
            // report end of search
            STB_SISearchComplete(path, FALSE, NULL, 0);
            finished = TRUE;
         }
#endif
      }
      else
      {
         #ifdef DEBUG_SI_STARTUP_SEARCH
         AP_SI_PRINT(("Startup search(%d): start - failed, no transport", path));
         #endif
         // report end of search
         STB_SISearchComplete(path, TRUE, NULL, 0);
         finished = TRUE;
      }
   }
   else if (event == APP_SI_STOP_MANAGER)
   {
      #ifdef DEBUG_SI_STARTUP_SEARCH
      AP_SI_PRINT(("Startup search(%d): stop", path));
      #endif
      CancelTableRequests(path, FALSE);
      finished = TRUE;
   }
   else
   {
      switch (event)
      {
         case APP_SI_SDT_RECEIVED:
         {
            #ifdef DEBUG_SI_STARTUP_SEARCH
            AP_SI_PRINT(("Startup search(%d): SDT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
            #endif
            ProcessSdtTable(table_rec, DB_ACCESS_UPDATE, IGNORE_VERSION, EIT_LIST_REQD);

            // start search for eits on current transport only. Prefer not to get eits until sdt
            // has been received so that the eit_list is available. However, if the sdt has not
            // arrived start looking for events and use the eit timeout to indicate when eits are
            // complete
            eit_start_timestamp[path] = STB_OSGetClockMilliseconds();
            eit_filter[path] = STB_SIRequestEit(path, ONE_SHOT_REQUEST, EIT_NOW_NEXT_ACT, DONT_CARE_ID_MATCH,
                  DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);

            // finished with the filter - release it and flag sdt complete
            sdt_complete[path] = TRUE;
            service_list_ready[path] = TRUE;
            STB_SICancelTableRequest(sdt_filter[path]);
            sdt_filter[path] = NULL;
            sdt_start_timestamp[path] = 0;
            break;
         }

         case APP_SI_SDT_TIMEOUT:
            #ifdef DEBUG_SI_STARTUP_SEARCH
            AP_SI_PRINT(("Startup search(%d): SDT timeout", path));
            #endif

            /* If the SDT hasn't been received then there's no point trying to collect the rest
             * of the tables. E.g. the number of EITs to be collected is determined by the number
             * of services defined in the SDT. This will also speed up initialisation of the box
             * as the TOT takes longer to timeout. */
            sdt_complete[path] = TRUE;
            pmts_complete[path] = TRUE;
            eits_complete[path] = TRUE;
            tot_complete[path] = TRUE;

            // finished with the filter - release it
            service_list_ready[path] = TRUE;
            STB_SICancelTableRequest(sdt_filter[path]);
            sdt_filter[path] = NULL;
            sdt_start_timestamp[path] = 0;
            break;

         case APP_SI_PAT_RECEIVED:
         case APP_SI_PAT_TIMEOUT:
         {
            if (event == APP_SI_PAT_RECEIVED)
            {
               #ifdef DEBUG_SI_STARTUP_SEARCH
               AP_SI_PRINT(("Startup search(%d): PAT received (tid 0x%04x)", path, table_rec->xtid));
               #endif
               ProcessPatTable(table_rec);
               if (pmt_list[path] != NULL)
               {
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_MONITOR;
                  MakeNewPmtRequest(path);
               }
               else
               {
                  pmts_complete[path] = TRUE;
               }
            }
            else
            {
               #ifdef DEBUG_SI_STARTUP_SEARCH
               AP_SI_PRINT(("Startup search(%d): PAT timeout", path));
               #endif
               pmts_complete[path] = TRUE;
            }

            // finished with the filter - release it
            STB_SICancelTableRequest(pat_filter[path]);
            pat_filter[path] = NULL;
            pat_start_timestamp[path] = 0;
            break;
         }

         case APP_SI_PMT_RECEIVED:
         case APP_SI_PMT_TIMEOUT:
         {
            if (event == APP_SI_PMT_RECEIVED)
            {
               #ifdef DEBUG_SI_STARTUP_SEARCH
               AP_SI_PRINT(("Startup search(%d): PMT received (sid 0x%04x)", path, table_rec->xtid));
               #endif
               // process the pmt (regardless of version and do not report to other parties)
               ProcessPmtTable(table_rec, IGNORE_VERSION, DONT_REPORT_PMT, NULL, FALSE);
            }
            #ifdef DEBUG_SI_STARTUP_SEARCH
            else
            {
               AP_SI_PRINT(("Startup search(%d): PMT timeout (sid 0x%04x)", path,
                            pmt_list[path][pmt_list_id[path]].serv_id));
            }
            #endif

            if ((pmt_list_id[path] + 1) >= num_pmt_list_entries[path])
            {
               // finished list - release the filter and flag pmts complete
               pmts_complete[path] = TRUE;
               STB_SICancelTableRequest(pmt_filter[path]);
               pmt_filter[path] = NULL;
               pmt_start_timestamp[path] = 0;
            }
            else
            {
               // not finished yet - move on to next pmt...
               pmt_list_id[path]++;
               pmt_request_mode[path] = PMT_REQUEST_MONITOR;
               MakeNewPmtRequest(path);
            }
            break;
         }

         case APP_SI_EIT_RECEIVED:
         {
            #ifdef DEBUG_SI_STARTUP_SEARCH
            AP_SI_PRINT(("Startup search(%d): EIT received (sid 0x%04x)", path, table_rec->xtid));
            #endif
            ProcessEitTable(table_rec, IGNORE_VERSION, DB_ACCESS_SEARCH, FALSE);

            // check if we have received all eits yet
            if (eit_list[path] != NULL)
            {
               eits_complete[path] = TRUE;
               for (i = 0; i < num_eit_list_entries[path]; i++)
               {
                  if (eit_list[path][i].got_eit == FALSE)
                  {
                     eits_complete[path] = FALSE;
                     break;
                  }
               }
            }
            if (eits_complete[path] == TRUE)
            {
               STB_SICancelTableRequest(eit_filter[path]);
               eit_filter[path] = NULL;
               eit_start_timestamp[path] = 0;

               STB_SICancelTableRequest(sched_filter[path]);
               sched_filter[path] = NULL;
               sched_start_timestamp[path] = 0;
            }
            break;
         }

         case APP_SI_EIT_TIMEOUT:
         {
            #ifdef DEBUG_SI_STARTUP_SEARCH
            AP_SI_PRINT(("Startup search(%d): EIT timeout", path));
            #endif
            // finished with the filter - release it and flag nit complete
            eits_complete[path] = TRUE;
            if (eit_filter[path] != NULL)
            {
               STB_SICancelTableRequest(eit_filter[path]);
               eit_filter[path] = NULL;
               eit_start_timestamp[path] = 0;
            }

            if (sched_filter[path] != NULL)
            {
               STB_SICancelTableRequest(sched_filter[path]);
               sched_filter[path] = NULL;
               sched_start_timestamp[path] = 0;
            }
            break;
         }

         case APP_SI_TOT_RECEIVED:
         case APP_SI_TOT_TIMEOUT:
         {
            if (event == APP_SI_TOT_RECEIVED)
            {
               #ifdef DEBUG_SI_STARTUP_SEARCH
               AP_SI_PRINT(("Startup search(%d): TOT received", path));
               #endif
               ASI_ProcessTotTable(table_rec);

               tot_already_received[path] = TRUE;
            }
            #ifdef DEBUG_SI_STARTUP_SEARCH
            else
            {
               AP_SI_PRINT(("Startup search(%d): TOT timeout", path));
            }
            #endif

            // finished with the filter - release it and flag tot complete
            tot_complete[path] = TRUE;
            STB_SICancelTableRequest(tot_filter[path]);
            tot_filter[path] = NULL;
            tot_start_timestamp[path] = 0;
            break;
         }
      }

      // check if search is complete - if so report fact
      if (sdt_complete[path] && pmts_complete[path] && eits_complete[path] && tot_complete[path])
      {
         // report end of search
         #ifdef DEBUG_SI_STARTUP_SEARCH
         AP_SI_PRINT(("Startup search(%d): search complete", path));
         #endif
         CancelTableRequests(path, FALSE);
         STB_SISearchComplete(path, TRUE, NULL, 0);
         finished = TRUE;
      }
   }

   FUNCTION_FINISH(ManageStartupSearch);
   return(finished);
}

/**
 *

 *
 * @brief   Manager for the event schedule search
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageEventSchedSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;

   FUNCTION_START(ManageEventSchedSearch);

   finished = FALSE;
   if (event == APP_SI_START_MANAGER)
   {
      // first ensure there is a transport record for the transport we are tuned to and set
      // current transport accordingly
      current_transport_rec[path] = DBDEF_GetTunedTransport(path);
      if (current_transport_rec[path] != NULL)
      {
         #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
         AP_SI_PRINT(("Event sched search: start, freq %lu", current_transport_rec[path]->frequency));
         #endif

         // update transport fields
         current_network_rec[path] = current_transport_rec[path]->network;
         current_service_rec[path] = NULL;

         if ((required_si_mode[path] == APP_SI_MODE_EVENT_PF_SEARCH) ||
            (required_si_mode[path] == APP_SI_MODE_EVENT_PF_SCHED_SEARCH))
         {
            eit_start_timestamp[path] = STB_OSGetClockMilliseconds();
            eit_filter[path] = STB_SIRequestEit(path, ONE_SHOT_REQUEST, EIT_NOW_NEXT_ACT, DONT_CARE_ID_MATCH,
                  DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
         }

         if ((required_si_mode[path] == APP_SI_MODE_EVENT_SCHED_SEARCH) ||
            (required_si_mode[path] == APP_SI_MODE_EVENT_PF_SCHED_SEARCH))
         {
            StartEITScheduleFilter(path);
            sched_start_timestamp[path] = STB_OSGetClockMilliseconds();
            sched_timeout_ms[path] = EVENT_SCHED_SEARCH_TIMEOUT_MS;
         }
      }
      else
      {
         #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
         AP_SI_PRINT(("Event sched search: start - failed, no transport"));
         #endif
         // report end of search
         STB_SISearchComplete(path, TRUE, NULL, 0);
         finished = TRUE;
      }
   }
   else if (event == APP_SI_STOP_MANAGER)
   {
      #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
      AP_SI_PRINT(("Event sched search: stop"));
      #endif
      CancelTableRequests(path, FALSE);
      finished = TRUE;
   }
   else
   {
      switch (event)
      {
         case APP_SI_EIT_RECEIVED:
         {
            #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
            AP_SI_PRINT(("Event sched search: EIT 0x%02x received (sid 0x%04x)",
               table_rec->tid, table_rec->xtid));
            #endif
            ProcessEitTable(table_rec, IGNORE_VERSION, DB_ACCESS_UPDATE, FALSE);

            /* Reset the start time for the appropriate EIT data so the timeout doesn't occur */
            if ((eit_start_timestamp[path] != 0) &&
               ((table_rec->tid == EITPF_ACTUAL_TID) || (table_rec->tid == EITPF_PLUS_TID)))
            {
               eit_start_timestamp[path] = STB_OSGetClockMilliseconds();
            }
            else if (sched_start_timestamp[path] != 0)
            {
               sched_start_timestamp[path] = STB_OSGetClockMilliseconds();
            }
            break;
         }

         case APP_SI_EIT_TIMEOUT:
         {
            #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
            AP_SI_PRINT(("Event sched search: EIT timeout"));
            #endif
            STB_SICancelTableRequest(eit_filter[path]);
            eit_filter[path] = NULL;
            eit_start_timestamp[path] = 0;

            if (sched_filter[path] == NULL)
            {
               #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
               AP_SI_PRINT(("Event sched search: Search complete"));
               #endif
               /* Only EITp/f is being collected so the search is now finished */
               STB_SISearchComplete(path, TRUE, NULL, 0);
               finished = TRUE;
            }
            break;
         }

         case APP_SI_SCHED_TIMEOUT:
         {
            #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
            AP_SI_PRINT(("Event sched search: Schedule timeout"));
            #endif
            STB_SICancelTableRequest(sched_filter[path]);
            sched_filter[path] = NULL;
            sched_start_timestamp[path] = 0;

            if (eit_filter[path] == NULL)
            {
               #ifdef DEBUG_SI_EVENT_SCHED_SEARCH
               AP_SI_PRINT(("Event sched search: Search complete"));
               #endif
               /* Only EITp/f is being collected so the search is now finished */
               STB_SISearchComplete(path, TRUE, NULL, 0);
               finished = TRUE;
            }
            break;
         }
      }
   }

   FUNCTION_FINISH(ManageEventSchedSearch);
   return(finished);
}

/**
 *

 *
 * @brief   Manager for the update mode
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageUpdate(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   static BOOLEAN transport_changed;
   static BOOLEAN first_pmt;
   BOOLEAN finished;
   BOOLEAN pmt_list_changed;
   BOOLEAN sdt_changed;
   U16BIT serv_id;
   E_STB_DP_SIGNAL_TYPE tuner_type;

   FUNCTION_START(ManageUpdate);

   finished = FALSE;
   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = ADB_GetTunedService(path);
         current_transport_rec[path] = DBDEF_GetTunedTransport(path);
         current_network_rec[path] = DBDEF_GetTunedNetwork(path);

         if (current_transport_rec[path] != NULL)
         {
         #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): start, freq %luHz, (%s)", path,
               current_transport_rec[path]->frequency,
               ACTL_GetRfNameFromFreq(current_transport_rec[path]->sig_type, current_transport_rec[path]->frequency)));
         #endif

            pat_rcvd_on_this_trnsprt[path] = FALSE;
            pmt_service_changed[path] = TRUE;
            report_pmt_allowed[path] = TRUE;
            pmt_reported[path] = FALSE;
            si_update_delay_timestamp[path] = STB_OSGetClockMilliseconds();
         }

         if (last_transport_rec[path] != current_transport_rec[path])
         {
            transport_changed = TRUE;
            last_transport_rec[path] = current_transport_rec[path];
         }
         else
         {
            transport_changed = FALSE;
         }
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): stop", path));
         #endif
         CancelTableRequests(path, FALSE);
         ClearDynamicUpdates();
         finished = TRUE;
         break;
      }

      case APP_SI_UPDATE_DELAY_EXPIRED:
      {
         // start delay has expired - start filter requests...
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): start delay expired", path));
         #endif
         si_update_delay_timestamp[path] = 0;

         // start search for CAT, no timeout
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting CAT", path));
         #endif
         cat_start_timestamp[path] = 0;
         cat_filter[path] = STB_SIRequestCat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_CAT_RECEIVED);

         // start search for PAT, no timeout
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting PAT", path));
         #endif
         pat_start_timestamp[path] = STB_OSGetClockMilliseconds();
         pat_filter[path] = STB_SIRequestPat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);

         ASI_RestartSITables(path, !ADB_IsFreesatService(current_service_rec[path]));

#ifndef CUSTOMIZED_FOR_CNS
         if (current_service_rec[path]->rct_pid != 0)
         {
#ifdef DEBUG_SI_RCT
            AP_SI_PRINT(("Requesting RCT for service 0x%04x on PID %u",
                         current_service_rec[path]->serv_id, current_service_rec[path]->rct_pid));
#endif
            rct_filter[path] = STB_SIRequestRct(path, CONTINUOUS_REQUEST,
                  current_service_rec[path]->rct_pid, ReceiveSiTable, APP_SI_RCT_RECEIVED);
         }
#endif  // #ifndef CUSTOMIZED_FOR_CNS
         break;
      }

      case APP_SI_CHANNEL_CHANGE:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): Channel changed", path));
         #endif

         // setup status - start pmt update timer for start delay, pmts will then be re-started by
         // APP_SI_PMT_UPDATE event
         current_service_rec[path] = ADB_GetTunedService(path);
         pmt_service_changed[path] = TRUE;
         report_pmt_allowed[path] = TRUE;
         pmt_reported[path] = FALSE;
         pmt_update_timestamp[path] = STB_OSGetClockMilliseconds();
         pmt_update_period_ms[path] = SI_UPDATE_DELAY_MS;
         break;
      }

      case APP_SI_STOP_PMT_REPORTING:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): Stop pmt reporting", path));
         #endif

         // stop getting pmts
         if (pmt_filter[path] != NULL)
         {
            STB_SICancelTableRequest(pmt_filter[path]);
            pmt_filter[path] = NULL;
            pmt_start_timestamp[path] = 0;
            pmt_update_timestamp[path] = 0;
         }
         break;
      }

      case APP_SI_PAT_RECEIVED:
      {
         /* Check it is the right pat - can sometimes get the pat for the previous transport.
          * But if the tran_id for the current transport hasn't been set, such as when tuning
          * to a dynamically added transport, maybe from a delivery system descriptor, then
          * just take the PAT as it is because the actual tran_id won't be known */
         if ((current_transport_rec[path]->tran_id != 0) &&
             (table_rec->xtid != current_transport_rec[path]->tran_id))
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): Wrong PAT (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
            #endif
            STB_SIRestartTableRequest(pat_filter[path]);
         }
         else
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): PAT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
            #endif

            pat_start_timestamp[path] = 0;
            // process the pat table to build the new pmt list
            pmt_list_changed = ProcessPatTable(table_rec);
            if (pmt_list_changed == TRUE)
            {
               if ((pmt_list[path] != NULL) && (report_pmt_allowed[path] == TRUE))
               {
                  /* Find the current service in the pmt list and start pmt gathering
                   * from that service */
                  first_pmt = TRUE;
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  if (!MakeNewPmtRequest(path))
                  {
                     /* Service isn't in the PAT, so it isn't running */
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                        EV_SERVICE_NOT_RUNNING, &path, sizeof(U8BIT));
                  }
               }
               else
               {
                  // no list anymore - cancel existing pmt filter
                  if (pmt_filter[path] != NULL)
                  {
                     STB_SICancelTableRequest(pmt_filter[path]);
                     pmt_filter[path] = NULL;
                     pmt_start_timestamp[path] = 0;
                  }
               }

               /* Restart the pat filter - if the transport id has changed the filter will have more
                * than one table record. Calling Restart will clear out the filter records and start
                * again. It will mean we recieve the current table again but ProcessPatTable checks
                * for version so next time pmt_list_changed will be false */
               STB_SIRestartTableRequest(pat_filter[path]);
            }
         }
         break;
      }

      case APP_SI_PAT_TIMEOUT:
      {
         AP_SI_PRINT(("Chan update(%d): PAT timeout", path));

         /* Finished with the filter - release it */
         STB_SICancelTableRequest(pat_filter[path]);
         pat_filter[path] = NULL;
         pat_start_timestamp[path] = 0;
                           
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_NOT_RUNNING,
            &path, sizeof(U8BIT));
         break;
      }

      case APP_SI_PMT_RECEIVED:
      case APP_SI_PMT_TIMEOUT:
      case APP_SI_PMT_UPDATE:
      {
         if (event == APP_SI_PMT_RECEIVED)
         {
            // process the pmt (if new version and report to other parties)
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): PMT received (sid 0x%04x)", path, table_rec->xtid));
            #endif

            if (ProcessPmtTable(table_rec, pmt_reported[path] ? FALSE : TRUE, REPORT_PMT, &serv_id, DB_ACCESS_UPDATE) &&
                (serv_id == current_service_rec[path]->serv_id))
            {
               /* Acquire a CA descrambler if needed */
               if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
               {
                  /* The NIT, CAT and BAT need to be reported to the CA system, but
                   * in some cases they're received before the CA descrambler has been
                   * acquired, so the filters are reset to ensure they're received
                   * again and can be reported */
                  if (nit_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(nit_filter[path]);
                  }
                  if (cat_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(cat_filter[path]);
                  }
#ifndef CUSTOMIZED_FOR_CNS
                  if (bat_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(bat_filter[path]);
                  }
#endif
               }

#ifndef CUSTOMIZED_FOR_CNS
               /* PMT for current service has been received or updated,
                * start or restart monitoring the RCT */
               if (rct_filter[path] != NULL)
               {
#ifdef DEBUG_SI_RCT
                  AP_SI_PRINT(("Cancelling RCT request"));
#endif
                  STB_SICancelTableRequest(rct_filter[path]);
                  rct_filter[path] = NULL;
               }

               if (current_service_rec[path]->rct_pid != 0)
               {
#ifdef DEBUG_SI_RCT
                  AP_SI_PRINT(("Requesting RCT for service 0x%04x on PID %u",
                               serv_id, current_service_rec[path]->rct_pid));
#endif
                  rct_filter[path] = STB_SIRequestRct(path, CONTINUOUS_REQUEST,
                        current_service_rec[path]->rct_pid, ReceiveSiTable, APP_SI_RCT_RECEIVED);
               }
#endif  // #ifndef CUSTOMIZED_FOR_CNS

#ifdef INTEGRATE_HBBTV
               if (ait_filter[path] != NULL)
               {
                  AP_SI_PRINT(("Cancelling AIT request"));
                  STB_SICancelTableRequest(ait_filter[path]);
                  ait_filter[path] = NULL;
               }

               if (current_service_rec[path]->ait_pid != 0)
               {
                  ait_filter[path] = STB_SIRequestAit(path, CONTINUOUS_REQUEST,
                        current_service_rec[path]->ait_pid, ReceiveSiTable, APP_SI_AIT_RECEIVED);
               }
#endif
            }

#ifdef INTEGRATE_HBBTV
            if ((ait_filter[path] == NULL) && (current_service_rec[path]->ait_pid != 0))
            {
               ait_filter[path] = STB_SIRequestAit(path, CONTINUOUS_REQUEST,
                     current_service_rec[path]->ait_pid, ReceiveSiTable, APP_SI_AIT_RECEIVED);
            }
#endif

            first_pmt = FALSE;
            pmt_update_timestamp[path] = STB_OSGetClockMilliseconds();
            pmt_update_period_ms[path] = PMT_UPDATE_MS;
         }
         else
         {
            #ifdef DEBUG_SI_UPDATE
            if (event == APP_SI_PMT_TIMEOUT)
            {
               AP_SI_PRINT(("Update(%d): PMT timeout (sid 0x%04x)", path,
                            pmt_list[path][pmt_list_id[path]].serv_id));
            }
            #endif

            // move on to next pmt, wrap from last to first...
            if (pmt_list[path] != NULL)
            {
               if (pmt_filter[path] == NULL)
               {
                  // restart pmt filtering for the new current service
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
               }
               else
               {
                  // move on to the next pmt
                  if (pmt_request_mode[path] == PMT_REQUEST_CURRENT)
                  {
                     STB_OSSemaphoreWait(si_pmt_list_sem);

                     if (pmt_priority_list[0] != INVALID_SERVICE_ID)
                     {
                        pmt_request_mode[path] = PMT_REQUEST_PRIORITY;
                     }
                     else
                     {
                        pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                        pmt_list_id[path]++;
                        if (pmt_list_id[path] >= num_pmt_list_entries[path])
                        {
                           pmt_list_id[path] = 0;
                        }
                     }

                     STB_OSSemaphoreSignal(si_pmt_list_sem);
                  }
                  else if (pmt_request_mode[path] == PMT_REQUEST_PRIORITY)
                  {
                     pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                     pmt_list_id[path]++;
                     if (pmt_list_id[path] >= num_pmt_list_entries[path])
                     {
                        pmt_list_id[path] = 0;
                     }
                  }
                  else
                  {
                     pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  }
               }
               MakeNewPmtRequest(path);
            }
            else
            {
               pmt_start_timestamp[path] = 0;
               pmt_update_timestamp[path] = 0;
            }
         }
         break;
      }

      case APP_SI_NIT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%u): NIT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
         #endif

#ifdef CUSTOMIZED_FOR_CNS
         if (0xFFFF /*private network ID*/ == table_rec->xtid)
             break;

         // FIXME: hardcode CNS network ID, 0xA030, 0xA031
         if (0xA000 /*CNS network ID*/ != (table_rec->xtid & 0xFF00))
             break;
#endif

         if (ProcessNitTable(table_rec, DB_ACCESS_UPDATE, TRUE, transport_changed))
         {
            /* The NIT has changed so restart collection of the SDTs to get any updates */
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%u): NIT changed; restarting SDT collection", path));
            #endif

#ifdef CUSTOMIZED_FOR_CNS
            if (sdt_filter[path] != NULL)
            {
               STB_SICancelTableRequest(sdt_filter[path]);
               sdt_filter[path] = NULL;
               sdt_start_timestamp[path] = 0;
            }

            if (current_service_rec[path]->sdt_pid == 0)
            {
               #ifdef DEBUG_SI_UPDATE
               AP_SI_PRINT(("Update(%d): requesting SDT", path));
               #endif
               sdt_start_timestamp[path] = 0;
               sdt_filter[path] = STB_SIRequestSdt(path, CONTINUOUS_REQUEST, TRUE, TRUE, DONT_CARE_ID_MATCH,
                     DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);
            }
            else
            {
               #ifdef DEBUG_SI_UPDATE
               AP_SI_PRINT(("Update(%d): requesting SDT on PID %u", path, current_service_rec[path]->sdt_pid));
               #endif
               sdt_start_timestamp[path] = 0;
               sdt_filter[path] = STB_SIRequestSdtFromPid(path, current_service_rec[path]->sdt_pid,
                     CONTINUOUS_REQUEST, TRUE, TRUE, DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 1,
                     ReceiveSiTable, APP_SI_SDT_RECEIVED);
            }
#else
            ASI_RestartSdtFilter(path);
#endif  // #ifdef CUSTOMIZED_FOR_CNS
         }
         break;
      }

      case APP_SI_SDT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): SDT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
         #endif
         sdt_changed = ProcessSdtTable(table_rec, DB_ACCESS_UPDATE, CHECK_VERSION, EIT_LIST_NOT_REQD);
#ifndef CUSTOMIZED_FOR_CNS
         if (sdt_changed)
         {
            // restart the sdt filter - if the transport id has changed the filter will have more
            // than one table record. Calling Restart will clear out the filter records and start
            // again. It will mean we recieve the current table again but ProcessSdtTable checks
            // for version so next time sdt_changed will be false
            STB_SIRestartTableRequest(sdt_filter[path]);
         }
#else
         tuner_type = STB_DPGetSignalType(path);
         sdt_complete[path] = TRUE;
         
         DBDEF_RequestAccess();
         /* Iterate through transports */
         ADB_TRANSPORT_REC * t_ptr = DBDEF_GetNextTransportRec(NULL);
         while (t_ptr != NULL)
         {
            if (t_ptr->sig_type == tuner_type) {
               if (FALSE == t_ptr->sdt_received) {
                  sdt_complete[path] = FALSE;
                  break;
               }
            }
            t_ptr = DBDEF_GetNextTransportRec(t_ptr);
         }
         DBDEF_ReleaseAccess();

         /* Finished with the filter - release it and flag sdt complete */
         if (TRUE == sdt_complete[path]) {
            service_list_ready[path] = TRUE;

            STB_SICancelTableRequest(sdt_filter[path]);
            sdt_filter[path] = NULL;
            sdt_start_timestamp[path] = 0;

            /* force to process the BAT again */
            if (active_bouquet_ids != NULL) {
               for (U16BIT i = 0; i < num_active_bouquet_ids; i++) {
                  active_bouquet_ids[i].received = FALSE;
               }
            }

            if (use_bats_active) {
               /* Start search for the BATs */
               bat_start_timestamp[path] = STB_OSGetClockMilliseconds();
               bat_filter[path] = STB_SIRequestBat(path, CONTINUOUS_REQUEST, DONT_CARE_ID_MATCH,
                  DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_BAT_RECEIVED);
            }
         }
#endif  // #ifndef CUSTOMIZED_FOR_CNS
         break;
      }

      case APP_SI_EIT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): EIT received (sid 0x%04x)", path, table_rec->xtid));
         #endif
         ProcessEitTable(table_rec, CHECK_VERSION, DB_ACCESS_UPDATE, FALSE);
         break;
      }

      case APP_SI_TDT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): TDT received", path));
         #endif

         ASI_ProcessTdtTable(table_rec);

         // finished with the filter - release it and flag tdt complete
         STB_SICancelTableRequest(tdt_filter[path]);
         tdt_filter[path] = NULL;
         tdt_start_timestamp[path] = 0;
         break;
      }

      case APP_SI_TOT_RECEIVED:
      {
         // this is only used in DTG test
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): TOT received", path));
         #endif
         ASI_ProcessTotTable(table_rec);
         break;
      }

      case APP_SI_CAT_RECEIVED:
      case APP_SI_CAT_TIMEOUT:
      {
         if (event == APP_SI_CAT_RECEIVED)
         {
            // process the cat (if new version or transport - and report to other parties)
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): CAT received", path));
            #endif
            ProcessCatTable(path, table_rec, REPORT_CAT, transport_changed);
         }
         #ifdef DEBUG_SI_UPDATE
         else
         {
            AP_SI_PRINT(("Update(%d): CAT timeout", path));
         }
         #endif
         break;
      }

      case APP_SI_RCT_RECEIVED:
      {
#ifdef DEBUG_SI_RCT
         AP_SI_PRINT(("Update(%u): RCT received", path));
#endif
         ProcessRctTable(table_rec);
         break;
      }

      case APP_SI_SCHED_TIMEOUT:
      {
         /* EIT schedule filter needs to be restarted to force events to be re-received */
         STB_SICancelTableRequest(sched_filter[path]);
         sched_filter[path] = NULL;
         sched_start_timestamp[path] = 0;
         StartEITScheduleFilter(path);
         break;
      }

      case APP_SI_BAT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%u): BAT received (bouquet id 0x%04x, v%d)", path, table_rec->xtid, table_rec->version));
         #endif
         if (ProcessBatTable(table_rec, DB_ACCESS_UPDATE, TRUE))
         {
#ifndef CUSTOMIZED_FOR_CNS
            /* The BAT has changed so restart collection of the SDTs to get any updates */
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%u): BAT changed; restarting SDT collection", path));
            #endif
            ASI_RestartSdtFilter(path);
#endif
         }

#ifdef CUSTOMIZED_FOR_CNS
         /* Check whether all BATs have now been received.
          * If all BATs are to be collected then the search will continue until
          * there's a timeout because we don't know how many BATs there are */
         bat_complete[path] = TRUE;
         if (active_bouquet_ids != NULL) {
            for (U16BIT i = 0; i < num_active_bouquet_ids; i++) {
               if (!active_bouquet_ids[i].received) {
                  /* Found one that hasn't been received yet */
                  bat_complete[path] = FALSE;
                  break;
               }
            }
         }

         if (TRUE == bat_complete[path]) {
            STB_SICancelTableRequest(bat_filter[path]);
            bat_filter[path] = NULL;
            bat_start_timestamp[path] = 0;
         }
         else {
            /* More BATs are needed, continue collecting */
            bat_start_timestamp[path] = STB_OSGetClockMilliseconds();
         }
#endif  // #ifdef CUSTOMIZED_FOR_CNS
         break;
      }
#ifdef INTEGRATE_HBBTV
      case APP_SI_AIT_RECEIVED:
      {
         U16BIT i;
         SI_SECTION_RECORD *section;

         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%u): AIT received (test:%d,type:0x%04x, v%d)", path,
                      table_rec->xtid >> 15, table_rec->xtid & 0x7F, table_rec->version));
         #endif

         section = table_rec->section_list;
         for (i = 0; (i < table_rec->num_sect) && (section != NULL); i++)
         {
            HBBTV_ProcessAitSection(current_service_rec[path]->serv_id, &(section->data_start), section->data_len);
            section = section->next;
         }
         break;
      }
#endif
      case APP_SI_UNT_RECEIVED:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%u): UNT received (xtid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
         #endif
         ProcessUntTable(path, table_rec, TRUE, transport_changed);
         break;
      }
   }

   FUNCTION_FINISH(ManageUpdate);

   return(finished);
}

/**
 *

 *
 * @brief   Manager for the FCC update mode
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageMonitoring(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   static BOOLEAN transport_changed;
   static BOOLEAN first_pmt;
   BOOLEAN finished;
   BOOLEAN pmt_list_changed;
   U16BIT serv_id;

   FUNCTION_START(ManageMonitoring);

   finished = FALSE;
   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = ADB_GetTunedService(path);
         current_transport_rec[path] = DBDEF_GetTunedTransport(path);
         current_network_rec[path] = DBDEF_GetTunedNetwork(path);

         if (current_transport_rec[path] != NULL)
         {
         #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Monitoring(%d): start, freq %luHz, (%s)", path,
               current_transport_rec[path]->frequency,
               ACTL_GetRfNameFromFreq(current_transport_rec[path]->sig_type, current_transport_rec[path]->frequency)));
         #endif

            pat_rcvd_on_this_trnsprt[path] = FALSE;
            pmt_service_changed[path] = TRUE;
            report_pmt_allowed[path] = TRUE;
            pmt_reported[path] = FALSE;
            si_update_delay_timestamp[path] = STB_OSGetClockMilliseconds();
         }
         else {
            pat_rcvd_on_this_trnsprt[path] = TRUE;
            si_update_delay_timestamp[path] = STB_OSGetClockMilliseconds();
         }

         if (last_transport_rec[path] != current_transport_rec[path])
         {
            transport_changed = TRUE;
            last_transport_rec[path] = current_transport_rec[path];
         }
         else
         {
            transport_changed = FALSE;
         }
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Monitoring(%d): stop", path));
         #endif
         CancelTableRequests(path, FALSE);
         ClearDynamicUpdates();
         finished = TRUE;
         break;
      }

      case APP_SI_UPDATE_DELAY_EXPIRED:
      {
         // start delay has expired - start filter requests...
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Monitoring(%d): start delay expired", path));
         #endif
         si_update_delay_timestamp[path] = 0;

         if (NULL != current_service_rec[path]) {
            // start search for PAT, no timeout
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Monitoring(%d): requesting PAT", path));
            #endif
            pat_start_timestamp[path] = 0;
            pat_filter[path] = STB_SIRequestPat(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);
         }
         else {
            // DVB monitor
            // start search for CAT, no timeout
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): requesting CAT", path));
            #endif
            cat_start_timestamp[path] = 0;
            cat_filter[path] = STB_SIRequestCat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_CAT_RECEIVED);

            //ASI_RestartSITables(path, !ADB_IsFreesatService(current_service_rec[path]));
         }
         break;
      }

      case APP_SI_CHANNEL_CHANGE:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Monitoring(%d): Channel changed", path));
         #endif

         // setup status - start pmt update timer for start delay, pmts will then be re-started by
         // APP_SI_PMT_UPDATE event
         current_service_rec[path] = ADB_GetTunedService(path);
         pmt_service_changed[path] = TRUE;
         report_pmt_allowed[path] = TRUE;
         pmt_reported[path] = FALSE;
         pmt_update_timestamp[path] = STB_OSGetClockMilliseconds();
         pmt_update_period_ms[path] = SI_UPDATE_DELAY_MS;
         break;
      }

      case APP_SI_STOP_PMT_REPORTING:
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Monitoring(%d): Stop pmt reporting", path));
         #endif

         // stop getting pmts
         if (pmt_filter[path] != NULL)
         {
            STB_SICancelTableRequest(pmt_filter[path]);
            pmt_filter[path] = NULL;
            pmt_start_timestamp[path] = 0;
            pmt_update_timestamp[path] = 0;
         }
         break;
      }

      case APP_SI_PAT_RECEIVED:
      {
         /* Check it is the right pat - can sometimes get the pat for the previous transport.
          * But if the tran_id for the current transport hasn't been set, such as when tuning
          * to a dynamically added transport, maybe from a delivery system descriptor, then
          * just take the PAT as it is because the actual tran_id won't be known */
         if ((current_transport_rec[path]->tran_id != 0) &&
             (table_rec->xtid != current_transport_rec[path]->tran_id))
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Monitoring(%d): Wrong PAT (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
            #endif
            STB_SIRestartTableRequest(pat_filter[path]);
         }
         else
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Monitoring(%d): PAT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
            #endif

            // process the pat table to build the new pmt list
            pmt_list_changed = ProcessPatTable(table_rec);
            if (pmt_list_changed == TRUE)
            {
               if ((pmt_list[path] != NULL) && (report_pmt_allowed[path] == TRUE))
               {
                  /* Find the current service in the pmt list and start pmt gathering
                   * from that service */
                  first_pmt = TRUE;
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  if (!MakeNewPmtRequest(path))
                  {
                     /* Service isn't in the PAT, so it isn't running */
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                        EV_SERVICE_NOT_RUNNING, &path, sizeof(U8BIT));
                  }
               }
               else
               {
                  // no list anymore - cancel existing pmt filter
                  if (pmt_filter[path] != NULL)
                  {
                     STB_SICancelTableRequest(pmt_filter[path]);
                     pmt_filter[path] = NULL;
                     pmt_start_timestamp[path] = 0;
                  }
               }

               /* Restart the pat filter - if the transport id has changed the filter will have more
                * than one table record. Calling Restart will clear out the filter records and start
                * again. It will mean we recieve the current table again but ProcessPatTable checks
                * for version so next time pmt_list_changed will be false */
               STB_SIRestartTableRequest(pat_filter[path]);
            }
         }
         break;
      }

      case APP_SI_PMT_RECEIVED:
      case APP_SI_PMT_TIMEOUT:
      case APP_SI_PMT_UPDATE:
      {
         if (event == APP_SI_PMT_RECEIVED)
         {
            // process the pmt (if new version and report to other parties)
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Monitoring(%d): PMT received (sid 0x%04x)", path, table_rec->xtid));
            #endif

            if (ProcessPmtTable(table_rec, pmt_reported[path] ? FALSE : TRUE, REPORT_PMT, &serv_id, DB_ACCESS_UPDATE) &&
                (serv_id == current_service_rec[path]->serv_id))
            {
               ADB_SERVICE_REC * s_ptr = current_service_rec[path];
               if (TRUE == s_ptr->pmt_received) {
                  STB_DPSetEcmPIDs(path, s_ptr->ecm_pid, s_ptr->video_ecm_pid, s_ptr->audio_ecm_pid,
                     s_ptr->ttext_ecm_pid, 0, s_ptr->ad_ecm_pid);
                     
                  STB_DPSetDecodePIDs(path, s_ptr->pcr_pid, s_ptr->video_pid, s_ptr->audio_pid,
                     s_ptr->ttext_pid, 0, s_ptr->ad_pid);
               }
 
               /* Acquire a CA descrambler if needed */
               if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
               {
                  /* The NIT, CAT and BAT need to be reported to the CA system, but
                   * in some cases they're received before the CA descrambler has been
                   * acquired, so the filters are reset to ensure they're received
                   * again and can be reported */
                  if (nit_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(nit_filter[path]);
                  }
                  if (cat_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(cat_filter[path]);
                  }
#ifndef CUSTOMIZED_FOR_CNS
                  if (bat_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(bat_filter[path]);
                  }
#endif
               }

               /* PMT for current service has been received or updated */
            }

            first_pmt = FALSE;
            pmt_update_timestamp[path] = STB_OSGetClockMilliseconds();
            pmt_update_period_ms[path] = PMT_UPDATE_MS;
         }
         else
         {
            #ifdef DEBUG_SI_UPDATE
            if (event == APP_SI_PMT_TIMEOUT)
            {
               AP_SI_PRINT(("Monitoring(%d): PMT timeout (sid 0x%04x)", path,
                            pmt_list[path][pmt_list_id[path]].serv_id));
            }
            #endif

            // move on to next pmt, wrap from last to first...
            if (pmt_list[path] != NULL)
            {
               if (pmt_filter[path] == NULL)
               {
                  // restart pmt filtering for the new current service
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
               }
               else
               {
                  // move on to the next pmt
                  if (pmt_request_mode[path] == PMT_REQUEST_CURRENT)
                  {
                     STB_OSSemaphoreWait(si_pmt_list_sem);

                     if (pmt_priority_list[0] != INVALID_SERVICE_ID)
                     {
                        pmt_request_mode[path] = PMT_REQUEST_PRIORITY;
                     }
                     else
                     {
                        pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                        pmt_list_id[path]++;
                        if (pmt_list_id[path] >= num_pmt_list_entries[path])
                        {
                           pmt_list_id[path] = 0;
                        }
                     }

                     STB_OSSemaphoreSignal(si_pmt_list_sem);
                  }
                  else if (pmt_request_mode[path] == PMT_REQUEST_PRIORITY)
                  {
                     pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                     pmt_list_id[path]++;
                     if (pmt_list_id[path] >= num_pmt_list_entries[path])
                     {
                        pmt_list_id[path] = 0;
                     }
                  }
                  else
                  {
                     pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  }
               }
               MakeNewPmtRequest(path);
            }
            else
            {
               pmt_start_timestamp[path] = 0;
               pmt_update_timestamp[path] = 0;
            }
         }
         break;
      }

      case APP_SI_CAT_RECEIVED:
      case APP_SI_CAT_TIMEOUT:
      {
         if (event == APP_SI_CAT_RECEIVED)
         {
             // process the cat (if new version or transport - and report to other parties)
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): CAT received", path));
            #endif
            ProcessCatTable(path, table_rec, REPORT_CAT, transport_changed);
         }
         #ifdef DEBUG_SI_UPDATE
         else
         {
            AP_SI_PRINT(("Update(%d): CAT timeout", path));
         }
         #endif
         break;
      }
   }

   FUNCTION_FINISH(ManageMonitoring);

   return(finished);
}

static void StartEITScheduleFilter(U8BIT path)
{
   E_SI_SCHED_TABLE_REQ sched_table_req;

   if ((eit_schedule_limit == 0) || (eit_schedule_limit >= 8 * 24))
   {
      #ifdef DEBUG_SI_UPDATE
      AP_SI_PRINT(("Update(%d): requesting SCHED all", path));
      #endif
      sched_table_req = EIT_SCHED_ALL;
      sched_start_timestamp[path] = 0;
      sched_timeout_ms[path] = 0;
   }
   else if (eit_schedule_limit < 4 * 24)
   {
      #ifdef DEBUG_SI_UPDATE
      AP_SI_PRINT(("Update(%d): requesting SCHED 4 day", path));
      #endif
      sched_table_req = EIT_SCHED_ALL_4DAY;
      sched_start_timestamp[path] = STB_OSGetClockMilliseconds();
      sched_timeout_ms[path] = SCHED_TIMEOUT_MS;
   }
   else
   {
      #ifdef DEBUG_SI_UPDATE
      AP_SI_PRINT(("Update(%d): requesting SCHED 8 day", path));
      #endif
      sched_table_req = EIT_SCHED_ALL_8DAY;
      sched_start_timestamp[path] = STB_OSGetClockMilliseconds();
      sched_timeout_ms[path] = SCHED_TIMEOUT_MS;
   }

   /* Start search for eits on all transports, no timeout */
   sched_filter[path] = STB_SIRequestSched(path, CONTINUOUS_REQUEST, sched_table_req,
         DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
}

/**
 *

 *
 * @brief   Manager for the update mode during PVR playback
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageUpdatePlayback(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   BOOLEAN pmt_list_changed;
   BOOLEAN service_updated;
   U8BIT *pmt_data;
   U16BIT pmt_data_len;
   U16BIT *ca_ids;
   U16BIT num_ca_ids;

   FUNCTION_START(ManageUpdatePlayback);

   finished = FALSE;
   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = APVR_GetPlaybackService();

         /* Reset the table versions */
         last_playback_pmt_version = -1;
         last_playback_pat_version = -1;

         #ifdef DEBUG_SI_PLAYBACK
         AP_SI_PRINT(("UpdatePlayback(%d): start (playback service = %p)", path, current_service_rec[path]));
         #endif

         pat_rcvd_on_this_trnsprt[path] = FALSE;
         pmt_service_changed[path] = TRUE;
         report_pmt_allowed[path] = TRUE;
         pmt_reported[path] = FALSE;
         si_update_delay_timestamp[path] = 0;

         /* Start search for CAT, no timeout */
         cat_start_timestamp[path] = 0;
         cat_filter[path] = STB_SIRequestCat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_CAT_RECEIVED);

         /* Start search for PAT, no timeout */
         pmt_filter_pid[path] = 0;
         pat_start_timestamp[path] = 0;
         pat_filter[path] = STB_SIRequestPat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);

         /* Start search for eits on all transports, no timeout */
         eit_start_timestamp[path] = 0;
         eit_filter[path] = STB_SIRequestEit(path, CONTINUOUS_REQUEST, EIT_NOW_NEXT_ACT,
               current_service_rec[path]->serv_id, 0xffff, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
         #ifdef DEBUG_SI_PLAYBACK
         AP_SI_PRINT(("UpdatePlayback(%d): stop", path));
         #endif
         CancelTableRequests(path, FALSE);
         finished = TRUE;
         break;
      }

      case APP_SI_STOP_PMT_REPORTING:
      {
         #ifdef DEBUG_SI_PLAYBACK
         AP_SI_PRINT(("UpdatePlayback(%d): Stop pmt reporting", path));
         #endif

         // stop getting pmts
         if (pmt_filter[path] != NULL)
         {
            STB_SICancelTableRequest(pmt_filter[path]);
            pmt_filter[path] = NULL;
            pmt_start_timestamp[path] = 0;
            pmt_update_timestamp[path] = 0;
         }
         break;
      }

      case APP_SI_PAT_RECEIVED:
      {
         #ifdef DEBUG_SI_PLAYBACK
         AP_SI_PRINT(("UpdatePlayback(%d): PAT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
         #endif

         // process the pat table to build the new pmt list
         pmt_list_changed = ProcessPatTablePlayback(table_rec);
         if (pmt_list_changed == TRUE)
         {
            if (pmt_filter[path] != NULL)
            {
               STB_SICancelTableRequest(pmt_filter[path]);
               pmt_filter[path] = NULL;
            }

            /* ProcessPatTablePlayback has saved the pmt pid we're interested in */
            pmt_filter[path] = STB_SIRequestPmt(path, CONTINUOUS_REQUEST, pmt_filter_pid[path],
                  current_service_rec[path]->serv_id, 0xffff, 0xffff, ReceiveSiTable, APP_SI_PMT_RECEIVED);

            // restart the pat filter - if the transport id has changed the filter will have more
            // than one table record. Calling Restart will clear out the filter records and start
            // again. It will mean we recieve the current table again but ProcessPatTable checks
            // for version so next time pmt_list_changed will be false
            STB_SIRestartTableRequest(pat_filter[path]);
         }
         break;
      }

      case APP_SI_PMT_RECEIVED:
      case APP_SI_PMT_TIMEOUT:
      case APP_SI_PMT_UPDATE:
      {
         if (event == APP_SI_PMT_RECEIVED)
         {
            // process the pmt (if new version and report to other parties)
            #ifdef DEBUG_SI_PLAYBACK
            AP_SI_PRINT(("UpdatePlayback(%d): PMT received", path));
            #endif

            service_updated = ProcessPmtTable(table_rec, CHECK_VERSION, REPORT_PMT, NULL, DB_ACCESS_PLAYBACK);
            if (service_updated && (current_service_rec[path] != NULL))
            {
               if ((pmt_data = ADB_GetServicePMTData(current_service_rec[path], &pmt_data_len)) != NULL)
               {
                  /* Get the CA system IDs from the PMT */
                  if ((num_ca_ids = STB_SIGetPmtCaIdDescArray(pmt_data, &ca_ids)) != 0)
                  {
                     /* Check whether the CA system needs to be involved in the playback */
                     if (STB_CADescramblerRequiredForPlayback(ca_ids, num_ca_ids))
                     {
                        /* Acquire a CA descrambler if needed */
                        if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
                        {
                           /* The NIT, CAT and BAT need to be reported to the CA system, but in
                            * some cases they're received before the CA descrambler has been
                            * acquired, so the filters are reset to ensure they're received
                            * again and can be reported */
                           STB_SIRestartTableRequest(nit_filter[path]);
                           STB_SIRestartTableRequest(cat_filter[path]);
                           if (bat_filter[path] != NULL)
                           {
                              STB_SIRestartTableRequest(bat_filter[path]);
                           }
                        }
                     }

                     STB_SIReleaseCaIdDescArray(ca_ids, (U8BIT)num_ca_ids);
                  }
               }
            }

            pmt_update_timestamp[path] = 0;
         }
         else
         {
            #ifdef DEBUG_SI_PLAYBACK
            if (event == APP_SI_PMT_TIMEOUT)
            {
               AP_SI_PRINT(("UpdatePlayback(%d): PMT timeout", path));
            }
            else
            {
               AP_SI_PRINT(("UpdatePlayback(%d): #1# PMT update", path));
            }
            #endif
         }
         break;
      }

      case APP_SI_EIT_RECEIVED:
      {
         #ifdef DEBUG_SI_PLAYBACK
         AP_SI_PRINT(("UpdatePlayback(%d): EIT received (sid 0x%04x)", path, table_rec->xtid));
         #endif
         ProcessEitTable(table_rec, CHECK_VERSION, DB_ACCESS_UPDATE, TRUE);
         break;
      }

      case APP_SI_CAT_RECEIVED:
      case APP_SI_CAT_TIMEOUT:
      {
         if (event == APP_SI_CAT_RECEIVED)
         {
            // process the cat (if new version or transport - and report to other parties)
            #ifdef DEBUG_SI_PLAYBACK
            AP_SI_PRINT(("UpdatePlayback(%d): CAT received", path));
            #endif
            ProcessCatTable(path, table_rec, REPORT_CAT, FALSE);
         }
         #ifdef DEBUG_SI_PLAYBACK
         else
         {
            AP_SI_PRINT(("UpdatePlayback(%d): CAT timeout", path));
         }
         #endif

         break;
      }
   }

   FUNCTION_FINISH(ManageUpdatePlayback);
   return(finished);
}

#ifdef COMMON_INTERFACE
/*!**************************************************************************
 * @brief   Manages the CI+ search for the SDT using secure routing - i.e. not through
 *          the CAM if it isn't a CI+ CAM - and then proceeds to the normal
 *          update search when the SDT is received or times out.
 * @param   path - decode path
 * @param   event - event code
 * @param   table_rec - SI table record to go with the event, may be NULL
 * @return  TRUE if the search has finished, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ManageCIPlusUpdate(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   BOOLEAN acquired;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;

   FUNCTION_START(ManageCIPlusUpdate);

   finished = FALSE;

   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = ADB_GetTunedService(path);
         current_transport_rec[path] = ADB_GetTunedTransport(path);

         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ update(%u): start for sid 0x%04x", path, current_service_rec[path]->serv_id));
         #endif

         if (ACI_SetSecureRouting(path))
         {
            /* Can now request the SDT for the service */
            #ifdef DEBUG_CI_SI_UPDATE
            AP_SI_PRINT(("CI+ update(%u): Requesting SDT for tid 0x%04x", path,
                         current_transport_rec[path]->tran_id));
            #endif

            STB_CiCcSetSDTAcquisitionStatus(FALSE);
            sdt_start_timestamp[path] = STB_OSGetClockMilliseconds();
            sdt_filter[path] = STB_SIRequestSdt(path, CONTINUOUS_REQUEST, TRUE, FALSE,
                  DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);
         }
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ update(%u): stop", path));
         #endif
         CancelTableRequests(path, FALSE);
         finished = TRUE;
         break;
      }

      case APP_SI_SDT_RECEIVED:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ update(%u): SDT received (tid 0x%04x v%d)", path, table_rec->xtid,
                      table_rec->version));
         #endif

         if (!current_transport_rec[path]->sdt_received)
         {
            DBDEF_RequestAccess();
            DBA_LockDatabase();

            /* New transport added by a CICAM+ tuning command, update IDs before parsing the table */
            current_transport_rec[path]->tran_id = table_rec->xtid;
            DBA_SetFieldValue(current_transport_rec[path]->dba_rec, DBA_FIELD_TRANSPORT_ID,
               current_transport_rec[path]->tran_id);

            section_rec = table_rec->section_list;
            data_ptr = &(section_rec->data_start) + 8;
            current_transport_rec[path]->orig_net_id = (data_ptr[0] << 8) | data_ptr[1];
            DBA_SetFieldValue(current_transport_rec[path]->dba_rec, DBA_FIELD_ORIG_NET_ID,
               current_transport_rec[path]->orig_net_id);

            current_transport_rec[path]->sdt_received = TRUE;

            DBA_SaveRecord(current_transport_rec[path]->dba_rec);
            DBA_SaveDatabase();

            DBA_UnlockDatabase();
            DBDEF_ReleaseAccess();
         }

         ProcessSdtTable(table_rec, DB_ACCESS_UPDATE, IGNORE_VERSION, EIT_LIST_NOT_REQD);
         STB_CiCcSetSDTAcquisitionStatus(TRUE);

         /* Check whether the SDT for the current service has now been received */
         if (ADB_GetServiceSDTReceived(current_service_rec[path]))
         {
            STB_SICancelTableRequest(sdt_filter[path]);
            sdt_filter[path] = NULL;
            sdt_start_timestamp[path] = 0;

            /* Now need to re-evaluate the CAM requirements for the path */
            acquired = ACI_AcquireCISlot(path, current_service_rec[path]);

            if (!acquired)
            {
               /* Acquire a CA descrambler if needed */
               if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
               {
                  /* The NIT, CAT and BAT need to be reported to the CA system, but in
                   * some cases they're received before the CA descrambler has been
                   * acquired, so the filters are reset to ensure they're received
                   * again and can be reported */
                  STB_SIRestartTableRequest(nit_filter[path]);
                  STB_SIRestartTableRequest(cat_filter[path]);
                  if (bat_filter[path] != NULL)
                  {
                     STB_SIRestartTableRequest(bat_filter[path]);
                  }
               }
            }

            /* Now continue with normal update manager */
            #ifdef DEBUG_CI_SI_UPDATE
            AP_SI_PRINT(("CI+ update(%u): start update", path));
            #endif
            current_manager[path] = ManageUpdate;
            finished = ManageUpdate(path, APP_SI_START_MANAGER, NULL);
            if (finished)
            {
               current_manager[path] = NULL;
               STB_SISearchComplete(path, TRUE, NULL, 0);
            }
         }
         #ifdef DEBUG_CI_SI_UPDATE
         else
         {
            AP_SI_PRINT(("CI+ update(%u): required SDT not yet received, continue", path));
         }
         #endif
         break;
      }

      case APP_SI_SDT_TIMEOUT:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ update(%u): SDT timeout", path));
         #endif

         STB_CiCcSetSDTAcquisitionStatus(TRUE);

         STB_SICancelTableRequest(sdt_filter[path]);
         sdt_filter[path] = NULL;
         sdt_start_timestamp[path] = 0;

         /* Failed to receive the SDT, in which case the service can still be presented,
          * so now need to re-evaluate the CAM requirements for the path */
         acquired = ACI_AcquireCISlot(path, current_service_rec[path]);

         if (!acquired)
         {
            /* Acquire a CA descrambler if needed */
            if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
            {
               /* The NIT, CAT and BAT need to be reported to the CA system, but in
                * some cases they're received before the CA descrambler has been
                * acquired, so the filters are reset to ensure they're received
                * again and can be reported */
               STB_SIRestartTableRequest(nit_filter[path]);
               STB_SIRestartTableRequest(cat_filter[path]);
               if (bat_filter[path] != NULL)
               {
                  STB_SIRestartTableRequest(bat_filter[path]);
               }
            }
         }

         /* Now continue with normal update manager */
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ update(%u): start update", path));
         #endif
         current_manager[path] = ManageUpdate;
         finished = ManageUpdate(path, APP_SI_START_MANAGER, NULL);
         if (finished)
         {
            current_manager[path] = NULL;
            STB_SISearchComplete(path, FALSE, NULL, 0);
         }
         break;
      }
   }

   FUNCTION_FINISH(ManageCIPlusUpdate);

   return(finished);
}

/*!**************************************************************************
 * @brief   Manages the CI+ search where the SDT has to be acquired using secure routing
 *          and then monitors other SI tables but without PAT and PMT
 * @param   path - decode path
 * @param   event - event code
 * @param   table_rec - SI table record to go with the event, may be NULL
 * @return  TRUE if the search has finished, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ManageCIPlusNoPatPmt(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   BOOLEAN acquired;

   FUNCTION_START(ManageCIPlusNoPatPmt);

   finished = FALSE;

   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = ADB_GetTunedService(path);
         current_transport_rec[path] = ADB_GetTunedTransport(path);

         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ no PAT/PMT(%u): start for sid 0x%04x", path, current_service_rec[path]->serv_id));
         #endif

         if (ACI_SetSecureRouting(path))
         {
            /* Can now request the SDT for the service */
            #ifdef DEBUG_CI_SI_UPDATE
            AP_SI_PRINT(("CI+ no PAT/PMT(%u): Requesting SDT for tid 0x%04x", path,
                         current_transport_rec[path]->tran_id));
            #endif

            sdt_start_timestamp[path] = STB_OSGetClockMilliseconds();
            sdt_filter[path] = STB_SIRequestSdt(path, CONTINUOUS_REQUEST, TRUE, FALSE,
                  DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);
         }
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ no PAT/PMT(%u): stop", path));
         #endif
         CancelTableRequests(path, FALSE);
         finished = TRUE;
         break;
      }

      case APP_SI_SDT_RECEIVED:
      case APP_SI_SDT_TIMEOUT:
      {
         if (event == APP_SI_SDT_RECEIVED)
         {
            #ifdef DEBUG_CI_SI_UPDATE
            AP_SI_PRINT(("CI+ no PAT/PMT(%u): SDT received (tid 0x%04x v%d)", path, table_rec->xtid,
                         table_rec->version));
            #endif

            ProcessSdtTable(table_rec, DB_ACCESS_UPDATE, IGNORE_VERSION, EIT_LIST_NOT_REQD);

            /* Check whether the SDT for the current service has now been received */
            if (!ADB_GetServiceSDTReceived(current_service_rec[path]))
            {
               #ifdef DEBUG_CI_SI_UPDATE
               AP_SI_PRINT(("CI+ no PAT/PMT(%u): required SDT not yet received, continue search", path));
               #endif
               break;
            }
         }
         #ifdef DEBUG_CI_SI_UPDATE
         else /* APP_SI_SDT_TIMEOUT */
         {
            AP_SI_PRINT(("CI+ no PAT/PMT(%u): SDT timeout", path));
         }
         #endif

         STB_SICancelTableRequest(sdt_filter[path]);
         sdt_filter[path] = NULL;
         sdt_start_timestamp[path] = 0;

         /* Now need to re-evaluate the CAM requirements for the path */
         acquired = ACI_AcquireCISlot(path, current_service_rec[path]);

         if (!acquired)
         {
            /* Acquire a CA descrambler if needed */
            if (ACA_AcquireCADescrambler(path, current_service_rec[path]))
            {
               /* The NIT, CAT and BAT need to be reported to the CA system, but in
                * some cases they're received before the CA descrambler has been
                * acquired, so the filters are reset to ensure they're received
                * again and can be reported */
               STB_SIRestartTableRequest(nit_filter[path]);
               STB_SIRestartTableRequest(cat_filter[path]);
               if (bat_filter[path] != NULL)
               {
                  STB_SIRestartTableRequest(bat_filter[path]);
               }
            }
         }

         /* Notify that the CI+ tune has completed successfully */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_CIPLUS_TUNE_COMPLETED, &path, sizeof(path));

         /* Now start monitoring other SI tables */
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ no PAT/PMT(%u): starting other tables", path));
         #endif

         /* Start search for TDT and TOT, no timeout */
         tdt_start_timestamp[path] = 0;
         tdt_filter[path] = STB_SIRequestTdt(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TDT_RECEIVED);
         tot_start_timestamp[path] = 0;
         tot_filter[path] = STB_SIRequestTot(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);
         break;
      }

      case APP_SI_TDT_RECEIVED:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ no PAT/PMT(%d): TDT received", path));
         #endif

         ASI_ProcessTdtTable(table_rec);

         /* Finished with the filter so release it */
         STB_SICancelTableRequest(tdt_filter[path]);
         tdt_filter[path] = NULL;
         tdt_start_timestamp[path] = 0;
         break;
      }

      case APP_SI_TOT_RECEIVED:
      {
         #ifdef DEBUG_CI_SI_UPDATE
         AP_SI_PRINT(("CI+ no PAT/PMT(%d): TOT received", path));
         #endif
         ASI_ProcessTotTable(table_rec);
         break;
      }
   }

   FUNCTION_FINISH(ManageCIPlusNoPatPmt);

   return(finished);
}

#endif

/**
 *

 *
 * @brief   Manager for the recording mode
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageRecording(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   U16BIT eit_pid;

   FUNCTION_START(ManageRecording);

   finished = FALSE;

   switch (event)
   {
      case APP_SI_START_MANAGER:
      {
         current_service_rec[path] = ADB_GetTunedService(path);
         current_transport_rec[path] = ADB_GetTunedTransport(path);
         if (current_transport_rec[path] != NULL)
         {
#ifdef DEBUG_SI_RECORDING
            AP_SI_PRINT(("Recording(%u): start on transport %p (tid 0x%04x)", path,
               current_transport_rec[path], current_transport_rec[path]->tran_id));
#endif

            pat_rcvd_on_this_trnsprt[path] = FALSE;

            // start search for CAT, no timeout
            cat_start_timestamp[path] = 0;
            cat_filter[path] = STB_SIRequestCat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_CAT_RECEIVED);

            /* Start search for PAT */
            pat_start_timestamp[path] = 0;
            pat_filter[path] = STB_SIRequestPat(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_PAT_RECEIVED);

            /* Start search for eit now/next just on this transport */
            if (ADB_IsFreesatService(current_service_rec[path]))
            {
               eit_pid = 0;
               if (current_service_rec[path]->eit_pf_plus_pid != 0)
               {
                  eit_pid = current_service_rec[path]->eit_pf_plus_pid;
               }
               else if (current_service_rec[path]->eit_pf_pid != 0)
               {
                  eit_pid = current_service_rec[path]->eit_pf_pid;
               }

               if (eit_pid != 0)
               {
                  #ifdef DEBUG_SI_RECORDING
                  AP_SI_PRINT(("Recording(%u): requesting EIT on PID %u", path, eit_pid));
                  #endif
                  eit_start_timestamp[path] = 0;
                  eit_filter[path] = STB_SIRequestEitFromPid(path, eit_pid, CONTINUOUS_REQUEST, EIT_PF_PLUS,
                        DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
               }
            }
            else
            {
               eit_filter[path] = STB_SIRequestEit(path, CONTINUOUS_REQUEST, EIT_NOW_NEXT_ACT, DONT_CARE_ID_MATCH,
                     DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
            }
            
            // SSU UNT
            unt_start_timestamp[path] = 0;
            if (0 == current_service_rec[path]->unt_pid)
            {
               #ifdef DEBUG_SI_UPDATE
               AP_SI_PRINT(("Recording(%d): requesting UNT", path));
               #endif
               unt_filter[path] = STB_SIRequestUnt(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_UNT_RECEIVED);
            }
            else
            {
               #ifdef DEBUG_SI_UPDATE
               AP_SI_PRINT(("Recording(%d): requesting UNT on PID %u", path, current_service_rec[path]->unt_pid));
               #endif
               unt_filter[path] = STB_SIRequestUntFromPid(path, current_service_rec[path]->unt_pid,
                     CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_UNT_RECEIVED);
            }
         }
         break;
      }

      case APP_SI_STOP_MANAGER:
      {
#ifdef DEBUG_SI_RECORDING
         AP_SI_PRINT(("Recording(%d): stop", path));
#endif
         CancelTableRequests(path, FALSE);
         finished = TRUE;
         break;
      }

      case APP_SI_CAT_RECEIVED:
      case APP_SI_CAT_TIMEOUT:
      {
         if (event == APP_SI_CAT_RECEIVED)
         {
            // process the cat (if new version or transport - and report to other parties)
            #ifdef DEBUG_SI_RECORDING
            AP_SI_PRINT(("Recording(%d): CAT received", path));
            #endif
            ProcessCatTable(path, table_rec, REPORT_CAT, FALSE);
         }
         #ifdef DEBUG_SI_RECORDING
         else
         {
            AP_SI_PRINT(("recording(%d): CAT timeout", path));
         }
         #endif
         break;
      }

      case APP_SI_PAT_RECEIVED:
      {
         /* Check it is the right PAT */
         if (table_rec->xtid != current_transport_rec[path]->tran_id)
         {
#ifdef DEBUG_SI_RECORDING
            AP_SI_PRINT(("Recording(%d): Wrong PAT (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
#endif
            STB_SIRestartTableRequest(pat_filter[path]);
         }
         else
         {
#ifdef DEBUG_SI_RECORDING
            AP_SI_PRINT(("Recording(%d): PAT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
#endif

            /* Process the PAT table to build the new PMT list */
            if (ProcessPatTable(table_rec))
            {
               if (pmt_list[path] != NULL)
               {
                  /* Find the current service in the pmt list and start pmt gathering from that service */
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  MakeNewPmtRequest(path);
               }
               else
               {
                  /* No list anymore - cancel existing pmt filter */
                  if (pmt_filter[path] != NULL)
                  {
                     STB_SICancelTableRequest(pmt_filter[path]);
                     pmt_filter[path] = NULL;
                     pmt_start_timestamp[path] = 0;
                  }
               }

               /* Restart the pat filter - if the transport id has changed the filter will have more
                * than one table record. Calling Restart will clear out the filter records and start
                * again. It will mean we receive the current table again but ProcessPatTable checks
                * for version so next time pmt_list_changed will be false */
               STB_SIRestartTableRequest(pat_filter[path]);
            }
         }
         break;
      }

      case APP_SI_PMT_RECEIVED:
      case APP_SI_PMT_TIMEOUT:
      case APP_SI_PMT_UPDATE:
      {
         if (event == APP_SI_PMT_RECEIVED)
         {
            /* Process the pmt */
#ifdef DEBUG_SI_RECORDING
            AP_SI_PRINT(("Recording(%d): PMT received (sid 0x%04x)", path, table_rec->xtid));
#endif
            ProcessPmtTable(table_rec, CHECK_VERSION, DONT_REPORT_PMT, NULL, DB_ACCESS_UPDATE);
            pmt_update_timestamp[path] = STB_OSGetClockMilliseconds();
            pmt_update_period_ms[path] = PMT_UPDATE_MS;
         }
         else
         {
#ifdef DEBUG_SI_RECORDING
            if (event == APP_SI_PMT_TIMEOUT)
            {
               AP_SI_PRINT(("Recording(%d): PMT timeout (sid 0x%04x)", path,
                            pmt_list[path][pmt_list_id[path]].serv_id));
            }
            else
            {
               AP_SI_PRINT(("Recording(%d): PMT update", path));
            }
#endif

            /* Move on to next pmt, wrap from last to first */
            if (pmt_list[path] != NULL)
            {
               if (pmt_filter[path] == NULL)
               {
                  // restart pmt filtering for the new current service
                  pmt_list_id[path] = 0;
                  pmt_request_mode[path] = PMT_REQUEST_CURRENT;
               }
               else
               {
                  // move on to the next pmt
                  if (pmt_request_mode[path] == PMT_REQUEST_CURRENT)
                  {
                     STB_OSSemaphoreWait(si_pmt_list_sem);

                     if (pmt_priority_list[0] != INVALID_SERVICE_ID)
                     {
                        pmt_request_mode[path] = PMT_REQUEST_PRIORITY;
                     }
                     else
                     {
                        pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                        pmt_list_id[path]++;
                        if (pmt_list_id[path] >= num_pmt_list_entries[path])
                        {
                           pmt_list_id[path] = 0;
                        }
                     }

                     STB_OSSemaphoreSignal(si_pmt_list_sem);
                  }
                  else if (pmt_request_mode[path] == PMT_REQUEST_PRIORITY)
                  {
                     pmt_request_mode[path] = PMT_REQUEST_MONITOR;

                     pmt_list_id[path]++;
                     if (pmt_list_id[path] >= num_pmt_list_entries[path])
                     {
                        pmt_list_id[path] = 0;
                     }
                  }
                  else
                  {
                     pmt_request_mode[path] = PMT_REQUEST_CURRENT;
                  }
               }
               MakeNewPmtRequest(path);
            }
            else
            {
               pmt_start_timestamp[path] = 0;
               pmt_update_timestamp[path] = 0;
            }
         }
         break;
      }

      case APP_SI_EIT_RECEIVED:
      {
#ifdef DEBUG_SI_RECORDING
         AP_SI_PRINT(("Recording(%d): EIT received (sid 0x%04x)", path, table_rec->xtid));
#endif
         ProcessEitTable(table_rec, CHECK_VERSION, DB_ACCESS_UPDATE, FALSE);
         break;
      }

      case APP_SI_UNT_RECEIVED:
      {
         #ifdef DEBUG_SI_RECORDING
         AP_SI_PRINT(("Recording(%u): UNT received (tid 0x%04x v%d)", path, table_rec->xtid, table_rec->version));
         #endif
         ProcessUntTable(path, table_rec, TRUE, FALSE);
         break;
      }
   }

   FUNCTION_FINISH(ManageRecording);

   return(finished);
}


static BOOLEAN ValidServiceType(ADB_SERVICE_TYPE serv_type, BOOLEAN all_streams_free)
{
   BOOLEAN valid_service;

   FUNCTION_START(ValidServiceType);

   valid_service = FALSE;

   if (((serv_type == ADB_SERVICE_TYPE_TV) ||
        (serv_type == ADB_SERVICE_TYPE_RADIO) ||
        (serv_type == ADB_SERVICE_TYPE_DATA) ||
        (((serv_type == ADB_SERVICE_TYPE_AVC_RADIO) ||
         (serv_type == ADB_SERVICE_TYPE_AVC_SD_TV)) &&
         ((required_service_type & SEARCH_SERVICE_TYPE_ADVANCED_CODEC) != 0)) ||
        (((serv_type == ADB_SERVICE_TYPE_HD_TV) ||
         (serv_type == ADB_SERVICE_TYPE_MPEG2_HD)) &&
         ((required_service_type & SEARCH_SERVICE_TYPE_HD) != 0)) ||
        ((serv_type == ADB_SERVICE_TYPE_UHD_TV) &&
         ((required_service_type & SEARCH_SERVICE_TYPE_HEVC) != 0))))
   {
      if ((((required_service_type & SEARCH_SERVICE_TYPE_FREE_TO_AIR) != 0) && all_streams_free) ||
          (((required_service_type & SEARCH_SERVICE_TYPE_ENCRYPTED) != 0) && !all_streams_free))
      {
         valid_service = TRUE;
      }
   }

   FUNCTION_FINISH(ValidServiceType);

   return(valid_service);
}

/**
 * @brief   Processes the SDT table record to extract data for the database
 * @param   table_rec pointer to the sdt table record
 * @param   mode if DB_ACCESS_SEARCH then contents of the SDT is to be stored in
 *               the database, adding or updating existing records. If
 *               DB_ACCESS_UPDATE then only the non-nvm based fields of existing
 *               records are saved, no records are added
 * @param   ignore_version if TRUE then the sdt contents are to be analysed even if it is
 *                         the same version as previously received, if FALSE it will only
 *                         be analysed if the version has changed
 * @param   eit_list_reqd if TRUE a new eit_list is created, otherwise the eit_list will
 * @return  TRUE if SDT has been processed (i.e. changed), FALSE otherwise.
 */
static BOOLEAN ProcessSdtTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode,
   BOOLEAN ignore_version, BOOLEAN eit_list_reqd)
{
   BOOLEAN retval;
   SI_SDT_TABLE *sdt_table;
   SI_SDT_SERVICE_ENTRY *sdt_entry;
   SI_STRING_DESC *str_desc;
   SI_MULTILING_SERV_NAME_DESC *mlsn_desc_ptr;
   SI_MULTILING_SHORT_NAME_DESC *short_name_desc_ptr;
   SI_PREFERRED_NAME_DESC *pref_name_desc_ptr;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_ptr;
   ADB_ALT_SERV_REC *alt_serv_rec;
   ADB_ALT_SERV_REC **last_alt_serv_rec_next_ptr;
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *next_s_ptr;
   ADB_TRANSPORT_REC *t_ptr;
   BOOLEAN changed;
   U16BIT i;
   U16BIT j;
   U8BIT lang_id;
   U8BIT name_id;
   U16BIT eit_list_id;
   U8BIT path;
   BOOLEAN serv_scrambled;
   BOOLEAN serv_not_running;
   BOOLEAN service_updated;
   U8BIT serv_running_status;
   U32BIT ca_handle;

   FUNCTION_START(ProcessSdtTable);

   retval = FALSE;
   path = table_rec->path;
   eit_list_id = 0;

   sdt_table = STB_SIParseSdtTable(table_rec);
   if (sdt_table != NULL)
   {
      t_ptr = NULL;

      if (mode == DB_ACCESS_SEARCH)
      {
         t_ptr = current_transport_rec[path];

         if (t_ptr->sig_type != SIGNAL_COFDM)
         {
            if ((t_ptr = ADB_GetTransportFromIds(ADB_INVALID_DVB_ID, sdt_table->orig_net_id,
                       sdt_table->tran_id)) == NULL)
            {
               /* A transport with these IDs doesn't exist so use current transport */
               //t_ptr = current_transport_rec[path];  //FIXME: incorrectly overwite the existing transport
            }
            else
            {
#if 0
               if (t_ptr != current_transport_rec[path])
               {
                  /* A transport for this SDT has previously been created, but the tuning params
                   * weren't known at the time (i.e. it may have been created when processing a BAT,
                   * in which case it may also have service's associated with it), so the parent
                   * transport of any services that have the current transport as a parent are updated
                   * to point to the transport that's just been found, after which the old transport
                   * record can be deleted */
                  DBDEF_RequestAccess();

                  s_ptr = DBDEF_GetNextServiceRec(NULL);
                  while (s_ptr != NULL)
                  {
                     if (s_ptr->transport == current_transport_rec[path])
                     {
                        /* Change the parent transport record of this service */
                        s_ptr->transport = t_ptr;
                        DBA_SetRecordParent(s_ptr->dba_rec, t_ptr->dba_rec);  // FIXME: busyloop
                        DBA_SaveRecord(s_ptr->dba_rec);
                     }

                     s_ptr = DBDEF_GetNextServiceRec(s_ptr);
                  }

                  DBDEF_DeleteTransportRec(current_transport_rec[path]);
                  DBDEF_ReleaseAccess();

                  current_transport_rec[path] = t_ptr;
                  ADB_SetTunedTransport(path, t_ptr);
               }
#endif
            }
         }
      }
      else if (sdt_table->num_services > 0)
      {
         /* Find the transport for this SDT. This may not be found if this is an SDTother where
          * cross-carriage is used and the transport wasn't found when doing a service search */
         t_ptr = ADB_GetTransportFromIds(ADB_INVALID_DVB_ID, sdt_table->orig_net_id, sdt_table->tran_id);
      }

      if ((t_ptr != NULL) && ((t_ptr->network == NULL) || (t_ptr->network->profile_type != ADB_PROFILE_TYPE_CIPLUS)))
      {
         #ifdef DEBUG_SI_SDT
         if (mode == DB_ACCESS_SEARCH)
         {
            AP_SI_PRINT(("SDT Table: tran_id 0x%04x onet_id 0x%04x v%d %d services",
                         sdt_table->tran_id, sdt_table->orig_net_id, sdt_table->version,
                         sdt_table->num_services));
         }
         else
         {
            if ((t_ptr->sdt_version != sdt_table->version) || (ignore_version == TRUE))
            {
               AP_SI_PRINT(("SDT Table: tran_id 0x%04x onet_id 0x%04x v%d %d services - new",
                            sdt_table->tran_id, sdt_table->orig_net_id, sdt_table->version,
                            sdt_table->num_services));
            }
            else
            {
               AP_SI_PRINT(("SDT Table: tran_id 0x%04x onet_id 0x%04x v%d %d services - same",
                            sdt_table->tran_id, sdt_table->orig_net_id, sdt_table->version,
                            sdt_table->num_services));
            }
         }
         #endif

         if (eit_list_reqd == TRUE)
         {
            // make space for the eit_list
            if (eit_list[path] != NULL)
            {
               STB_AppFreeMemory(eit_list[path]);
            }
            num_eit_list_entries[path] = sdt_table->num_services;
            eit_list[path] = STB_AppGetMemory(num_eit_list_entries[path] * sizeof(EIT_LIST_ENTRY));
            eit_list_id = 0;
            memset(eit_list[path], 0, num_eit_list_entries[path] * sizeof(EIT_LIST_ENTRY));
         }

         // only process the table if we are in search mode, or the sdt version has changed
         if ((mode == DB_ACCESS_SEARCH) || (ignore_version == TRUE) || !t_ptr->sdt_received ||
             (t_ptr->sdt_version != sdt_table->version))
         {
            retval = TRUE;
            changed = FALSE;

            // save sdt data to the database...
            // get access to the database (may suspend)
            DBDEF_RequestAccess();

            t_ptr->sdt_received = TRUE;

            if (t_ptr->sdt_version != sdt_table->version)
            {
               // save version in transport record
               t_ptr->sdt_version = sdt_table->version;

               /* Save the latest version of the SDT */
               DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_VERSION, t_ptr->sdt_version);
               changed = TRUE;

               if (mode != DB_ACCESS_SEARCH)
               {
                  t_ptr->sdt_version_changed = TRUE;
               }
            }

            if (mode == DB_ACCESS_SEARCH)
            {
               /* Now doing a search, so version no longer changed */
               t_ptr->sdt_version_changed = FALSE;

               /* Save transport and onet ids in the transport record */
               if (t_ptr->tran_id != sdt_table->tran_id)
               {
                  t_ptr->tran_id = sdt_table->tran_id;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRANSPORT_ID, t_ptr->tran_id);
                  changed = TRUE;
               }

               /* Only set the onet id if it hasn't been set yet */
               if (t_ptr->orig_net_id == 0)
               {
                  t_ptr->orig_net_id = sdt_table->orig_net_id;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_ORIG_NET_ID, t_ptr->orig_net_id);
                  changed = TRUE;
               }

               if (changed)
               {
                  DBA_SaveRecord(t_ptr->dba_rec);
               }
            }

            // mark all services on this transport as unavailable - all services listed in the sdt will
            // be marked available in the following loop, leaving those services which are no longer in
            // the sdt marked unavailable
            s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr);
            while (s_ptr != NULL)
            {
               s_ptr->unavailable = TRUE;
               s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);
            }

            // examine each entry in the sdt...
            sdt_entry = sdt_table->service_list;
            while (sdt_entry != NULL)
            {
               /* Find service in database */
               if (mode == DB_ACCESS_SEARCH)
               {
                  s_ptr = DBDEF_FindServiceRec(sdt_entry->serv_id, t_ptr);
               }
               else
               {
                  s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                        t_ptr->tran_id, sdt_entry->serv_id);
               }

               if ((s_ptr == NULL) && ValidServiceType(sdt_entry->serv_type, sdt_entry->all_streams_free))
               {
                  if (mode == DB_ACCESS_SEARCH)
                  {
                     #ifdef DEBUG_SI_SDT
                     AP_SI_PRINT(("   add service sid 0x%04x", sdt_entry->serv_id));
                     #endif
                     s_ptr = DBDEF_AddServiceRec(sdt_entry->serv_id, t_ptr);
                     if (s_ptr != NULL)
                     {
                        s_ptr->new_service = TRUE;
                     }
                  }
                  else if ((mode == DB_ACCESS_UPDATE) && ASI_DatabaseUpdatesAllowed(t_ptr->sig_type) &&
                     ACFG_GetDynamicSIUpdate(t_ptr->sig_type, t_ptr->orig_net_id, ACFG_DYNAMIC_SI_UPDATE_SERVICE_ADD) &&
                     (dynamic_update_head != NULL))
                  {
                     #ifdef DEBUG_SI_SDT
                     AP_SI_PRINT(("   add service sid 0x%04x", sdt_entry->serv_id));
                     #endif
                     s_ptr = DBDEF_AddServiceRec(sdt_entry->serv_id, t_ptr);
                     if (s_ptr != NULL)
                     {
                        s_ptr->new_service = TRUE;

                        DynamicUpdateAddService(t_ptr, s_ptr,
                           ACFG_GetDynamicSIUpdate(t_ptr->sig_type, t_ptr->orig_net_id, ACFG_DYNAMIC_SI_UPDATE_SERVICE_MOVE));

                        /* The target region data may be needed for the allocation of LCNs so it's
                         * kept until after the dynamic update process has completed, when it will
                         * be freed */
                        s_ptr->target_region_list = sdt_entry->target_region_list;
                        sdt_entry->target_region_list = NULL;
                     }
                  }
               }

               while (s_ptr != NULL)
               {
                  #ifdef DEBUG_SI_SDT
                  AP_SI_PRINT((" Service 0x%04x", sdt_entry->serv_id));
                  if (s_ptr->name_str != NULL)
                  {
                     AP_SI_PRINT(("  name \"%s\"", s_ptr->name_str->str_ptr));
                  }
                  #endif

                  service_updated = FALSE;
                  s_ptr->found = TRUE;

                  /* Save the scambled state of the service to handle the change to/from FTA services */
                  serv_scrambled = s_ptr->scrambled;

                  if ((eit_list_reqd == TRUE) && (eit_list[path] != NULL))
                  {
                     // add service to eit list
                     if (eit_list_id < num_eit_list_entries[path])
                     {
                        eit_list[path][eit_list_id].serv_id = sdt_entry->serv_id;
                        eit_list_id++;
                     }
                  }

                  //--------------------------------------------------------------------------
                  // first store data in service record that is saved in nvm...
                  // (remember to free old data first if changed)

                  if (ASI_DatabaseUpdatesAllowed(t_ptr->sig_type))
                  {
                     // update CNS channel group mask
                     if (sdt_entry->sdt_code_desc != NULL)
                     {
                        if (sdt_entry->sdt_code_desc->data_length == 9) {
                           U8BIT *data = sdt_entry->sdt_code_desc->data;
                           U32BIT group_mask;
                           group_mask = data[8];
                           group_mask += (data[7] << 8);
                           group_mask += (data[6] << 16);
                           group_mask += (data[5] << 24);
                           if (DBDEF_SetServiceGroupMask(s_ptr, group_mask))
                           {
                              #ifdef DEBUG_SI_SDT
                              AP_SI_PRINT(("   channel group mask 0x%08X", group_mask));
                              #endif
                              service_updated = TRUE;
                           }
	                    }
                     }

                     // update service type
                     if (DBDEF_SetServiceType(s_ptr, sdt_entry->serv_type))
                     {
                        service_updated = TRUE;
                     }

                     // update service name
                     if (sdt_entry->name_str != NULL)
                     {
                        if (DBDEF_SetServiceName(s_ptr, sdt_entry->name_str->str_ptr))
                        {
                           #ifdef DEBUG_SI_SDT
                           AP_SI_PRINT(("   new name \"%s\"", sdt_entry->name_str->str_ptr));
                           #endif
                           service_updated = TRUE;
                        }
                     }
                     else
                     {
                        if (DBDEF_SetServiceName(s_ptr, NULL))
                        {
                           #ifdef DEBUG_SI_SDT
                           AP_SI_PRINT(("   new NULL name"));
                           #endif
                           service_updated = TRUE;
                        }
                     }
                  }

                  //--------------------------------------------------------------------------
                  // now store data in service record that is NOT saved in nvm...
                  // (remember to free old data first if appropriate)
                  s_ptr->unavailable = FALSE;
                  s_ptr->scrambled = (sdt_entry->all_streams_free == FALSE);
                  serv_running_status = s_ptr->running_status;
                  s_ptr->running_status = sdt_entry->running_status;
                  serv_not_running = s_ptr->not_running;

                  /* Check the running status of the service */
                  if ((sdt_entry->running_status == RUN_STATE_NOT_RUNNING) ||
                      (sdt_entry->running_status == RUN_STATE_STARTS_SOON) ||
                      (sdt_entry->running_status == RUN_STATE_OFF_AIR))
                  {
                     s_ptr->not_running = TRUE;
                  }
                  else
                  {
                     s_ptr->not_running = FALSE;
                  }

                  if (sdt_entry->fta_content_desc != NULL)
                  {
                     s_ptr->has_fta_desc = TRUE;
                     s_ptr->do_not_scramble = sdt_entry->fta_content_desc->do_not_scramble;
                  }

                  if (ASI_DatabaseUpdatesAllowed(t_ptr->sig_type))
                  {
                     // transfer provider string from sdt entry (not stored in nvm)
                     if (sdt_entry->provider_str != NULL)
                     {
                        DBDEF_SetServiceProviderName(s_ptr, sdt_entry->provider_str->str_ptr);
                     }
                     else
                     {
                        if (DBDEF_SetServiceProviderName(s_ptr, NULL))
                        {
                           service_updated = TRUE;
                        }
                     }

                     if (sdt_entry->short_name_str != NULL)
                     {
                        if (DBDEF_SetServiceShortName(s_ptr, sdt_entry->short_name_str->str_ptr))
                        {
                           service_updated = TRUE;
                        }
                     }
                     else
                     {
                        if (DBDEF_SetServiceShortName(s_ptr, NULL))
                        {
                           service_updated = TRUE;
                        }
                     }

                     // release all old multilingual and preferred names
                     for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
                     {
                        if (s_ptr->short_name_array[i] != NULL)
                        {
                           DBDEF_ReleaseString(s_ptr->short_name_array[i]);
                           s_ptr->short_name_array[i] = NULL;
                        }
                        if (s_ptr->provider_array[i] != NULL)
                        {
                           DBDEF_ReleaseString(s_ptr->provider_array[i]);
                           s_ptr->provider_array[i] = NULL;
                        }
                        for (j = 0; j < ADB_NUM_SERV_NAME_IDS; j++)
                        {
                           if (s_ptr->name_array[i][j] != NULL)
                           {
                              DBDEF_ReleaseString(s_ptr->name_array[i][j]);
                              s_ptr->name_array[i][j] = NULL;
                           }
                        }
                     }

                     /* Extract multilingual short service names */
                     short_name_desc_ptr = sdt_entry->multiling_short_name_array;
                     for (i = 0; i < sdt_entry->num_multiling_short_names; i++, short_name_desc_ptr++)
                     {
                        lang_id = ACFG_ConvertLangCodeToId(short_name_desc_ptr->lang_code);
                        if (lang_id != ACFG_INVALID_DB_LANG)
                        {
                           if (short_name_desc_ptr->name_str != NULL)
                           {
                              if (s_ptr->short_name_array[lang_id] == NULL)
                              {
                                 str_desc = short_name_desc_ptr->name_str;
                                 s_ptr->short_name_array[lang_id] = DBDEF_MakeString(
                                       short_name_desc_ptr->lang_code, str_desc->str_ptr, str_desc->nbytes);
                              }
                           }
                        }
                     }

                     // extract multilingual service names
                     mlsn_desc_ptr = sdt_entry->multiling_name_desc_array;
                     for (i = 0; i < sdt_entry->num_multiling_names; i++, mlsn_desc_ptr++)
                     {
                        lang_id = ACFG_ConvertLangCodeToId(mlsn_desc_ptr->lang_code);
                        if (lang_id != ACFG_INVALID_DB_LANG)
                        {
                           // if mulitlingual service name is defined transfer it to appropriate
                           // language entry for name_id=0 in names array
                           if (mlsn_desc_ptr->name_str != NULL)
                           {
                              if (s_ptr->name_array[lang_id][0] == NULL)
                              {
                                 str_desc = mlsn_desc_ptr->name_str;
                                 s_ptr->name_array[lang_id][0] = DBDEF_MakeString(
                                       mlsn_desc_ptr->lang_code, str_desc->str_ptr, str_desc->nbytes);
                              }
                           }

                           // if mulitlingual provider name is defined transfer it to appropriate
                           // language entry in the provider array
                           if (mlsn_desc_ptr->provider_str != NULL)
                           {
                              if (s_ptr->provider_array[lang_id] == NULL)
                              {
                                 str_desc = mlsn_desc_ptr->provider_str;
                                 s_ptr->provider_array[lang_id] = DBDEF_MakeString(
                                       mlsn_desc_ptr->lang_code, str_desc->str_ptr, str_desc->nbytes);
                              }
                           }
                        }
                     }

                     // now transfer preferred names to name array
                     pref_name_desc_ptr = sdt_entry->preferred_name_desc_array;
                     for (i = 0; i < sdt_entry->num_preferred_names; i++, pref_name_desc_ptr++)
                     {
                        if (pref_name_desc_ptr->name_str != NULL)
                        {
                           lang_id = ACFG_ConvertLangCodeToId(pref_name_desc_ptr->lang_code);
                           if (lang_id != ACFG_INVALID_DB_LANG)
                           {
                              if (pref_name_desc_ptr->name_id < ADB_NUM_SERV_NAME_IDS)
                              {
                                 name_id = pref_name_desc_ptr->name_id;
                                 if (s_ptr->name_array[lang_id][name_id] == NULL)
                                 {
                                    str_desc = pref_name_desc_ptr->name_str;
                                    s_ptr->name_array[lang_id][name_id] = DBDEF_MakeString(
                                          pref_name_desc_ptr->lang_code, str_desc->str_ptr, str_desc->nbytes);
                                 }
                              }
                           }
                        }
                     }
                  }

                  if ((sdt_entry->guidance != NULL) &&
                      ((sdt_entry->guidance->guidance_type == 0) || (sdt_entry->guidance->guidance_type == 1)))
                  {
                     for (i = 0; i < sdt_entry->guidance->num_langs; i++)
                     {
                        lang_id = ACFG_ConvertLangCodeToId(sdt_entry->guidance->lang_codes[i]);
                        if (lang_id != ACFG_INVALID_DB_LANG)
                        {
                           if (s_ptr->guidance[lang_id].text != NULL)
                           {
                              DBDEF_ReleaseString(s_ptr->guidance[lang_id].text);
                              s_ptr->guidance[lang_id].text = NULL;
                           }

                           s_ptr->guidance[lang_id].type = sdt_entry->guidance->guidance_type;
                           s_ptr->guidance[lang_id].mode = sdt_entry->guidance->guidance_mode;

                           s_ptr->guidance[lang_id].text = DBDEF_MakeString(
                                 sdt_entry->guidance->lang_codes[i],
                                 sdt_entry->guidance->strings[i]->str_ptr,
                                 sdt_entry->guidance->strings[i]->nbytes);
                        }
                     }
                  }
                  else if (s_ptr->guidance != NULL)
                  {
                     for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
                     {
                        if (s_ptr->guidance[i].text != NULL)
                        {
                           DBDEF_ReleaseString(s_ptr->guidance[i].text);
                           s_ptr->guidance[i].text = NULL;
                        }
                     }
                  }

                  // look for linkage descriptor defining alternative services
                  DBDEF_DeleteAltServList(s_ptr->alt_serv_list);
                  s_ptr->alt_serv_list = NULL;
                  last_alt_serv_rec_next_ptr = &(s_ptr->alt_serv_list);

                  linkage_desc_ptr = sdt_entry->linkage_desc_list;
                  while (linkage_desc_ptr != NULL)
                  {
                     if ((linkage_desc_ptr->link_type == LINK_TYPE_SERVICE_REPLACEMENT) ||
                         (linkage_desc_ptr->link_type == LINK_TYPE_CA_REPLACEMENT_SERVICE))
                     {
                        alt_serv_rec = STB_AppGetMemory(sizeof(ADB_ALT_SERV_REC));
                        if (alt_serv_rec != NULL)
                        {
                           alt_serv_rec->onet_id = linkage_desc_ptr->orig_net_id;
                           alt_serv_rec->tran_id = linkage_desc_ptr->tran_id;
                           alt_serv_rec->serv_id = linkage_desc_ptr->serv_id;
                           alt_serv_rec->link_type = linkage_desc_ptr->link_type;
                           alt_serv_rec->next = NULL;
                           *last_alt_serv_rec_next_ptr = alt_serv_rec;
                           last_alt_serv_rec_next_ptr = &(alt_serv_rec->next);
                           #ifdef DEBUG_SI_SDT
                           AP_SI_PRINT(("   add alt service type=0x%02x, 0x%04x/0x%04x/0x%04x",
                                        alt_serv_rec->link_type, alt_serv_rec->onet_id, alt_serv_rec->tran_id,
                                        alt_serv_rec->serv_id));
                           #endif
                        }
                     }
                     linkage_desc_ptr = linkage_desc_ptr->next;
                  }

                  if (sdt_entry->def_authority != NULL)
                  {
                     changed = TRUE;

                     if (s_ptr->def_authority != NULL)
                     {
                        /* Check whether the default authority has changed */
                        if (sdt_entry->def_authority->nbytes == s_ptr->def_authority->nbytes)
                        {
                           if (memcmp(s_ptr->def_authority->str_ptr, sdt_entry->def_authority->str_ptr,
                                  s_ptr->def_authority->nbytes) == 0)
                           {
                              /* The names are the same */
                              changed = FALSE;
                           }
                        }
                     }

                     if (changed)
                     {
                        /* Free the old authority string */
                        if (s_ptr->def_authority != NULL)
                        {
                           STB_AppFreeMemory(s_ptr->def_authority);
                           s_ptr->def_authority = NULL;
                        }

                        /* Add the new authority string */
                        s_ptr->def_authority = DBDEF_MakeString(0,
                              sdt_entry->def_authority->str_ptr, sdt_entry->def_authority->nbytes);
                        #ifdef DEBUG_SI_SDT
                        AP_SI_PRINT(("   SDT \"%s\" has default authority \"%s\"", s_ptr->name_str->str_ptr,
                                     s_ptr->def_authority->str_ptr));
                        #endif
                     }
                  }

                  if ((current_service_rec[path] == s_ptr) &&
                      (s_ptr->running_status != serv_running_status) &&
                      STB_DPGetPathCADescrambler(path, &ca_handle))
                  {
                     /* Notify the CA system that the running status has changed */
                     STB_CANotifyRunningStatus(ca_handle, s_ptr->running_status);
                  }

                  if (s_ptr->not_running != serv_not_running)
                  {
                     /* There has been a change in the running status of the service */
                     if ((current_service_rec[path] == s_ptr) && s_ptr->not_running)
                     {
                        /* The current service is no longer running */
                        #ifdef DEBUG_SI_SDT
                        AP_SI_PRINT(("   sid 0x%04x report service not running", s_ptr->serv_id));
                        #endif
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                           EV_SERVICE_NOT_RUNNING, &path, sizeof(U8BIT));
                        STB_SIReportCurrentPmt(s_ptr->serv_id, NULL, FALSE, FALSE);
                     }
                     else if ((s_ptr->not_running == FALSE) &&
                              ((s_ptr->video_pid != 0) || (s_ptr->audio_pid != 0) ||
                               (t_ptr != current_transport_rec[path])))
                     {
                        /* A service that was previously not running is now running */
                        #ifdef DEBUG_SI_SDT
                        AP_SI_PRINT(("   sid 0x%04x report service running", s_ptr->serv_id));
                        #endif
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_RUNNING,
                           &s_ptr, sizeof(s_ptr));
                     }
                  }
                  else
                  {
                     if (s_ptr->scrambled != serv_scrambled)
                     {
                        /* The service has changed to/from FTA.
                         * The service this change relates to needs to be known by whatever handles
                         * the event, but there isn't a way to send this info with the event as it
                         * would require too many bits and there are only 8 bits available (the top
                         * 8 bits of the class). So the service record is stored in a table and the
                         * index into that table is sent with the event, instead of the path. The
                         * event's handler must query this data, after which the table entry will
                         * be released so that it can be reused. */
                        #ifdef DEBUG_SI_SDT
                        AP_SI_PRINT(("   sid 0x%04x report scramble change", s_ptr->serv_id));
                        #endif
                        STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                           EV_SERVICE_SCRAMBLE_CHANGE, &s_ptr, sizeof(s_ptr));
                     }
                  }

                  if (mode == DB_ACCESS_SEARCH)
                  {
                     /* The target region data will only be kept until after the search has completed
                      * and all of the info gathered may be needed so just keep the lot and ensure it
                      * isn't deleted when the table gets deleted. */
                     s_ptr->target_region_list = sdt_entry->target_region_list;
                     sdt_entry->target_region_list = NULL;
                  }

                  if (service_updated)
                  {
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                        EV_SERVICE_UPDATED, &s_ptr, sizeof(s_ptr));
                  }

                  /* See if there's another service with the same IDs */
                  s_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                        t_ptr->tran_id, sdt_entry->serv_id);
               }

               if ((s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, t_ptr->orig_net_id,
                          t_ptr->tran_id, sdt_entry->serv_id)) != NULL)
               {
                  /* Delete the linkage array for repopulation */
                  DeleteLinkageDescripterArray(s_ptr->linkage_desc_list);
                  s_ptr->last_linkage_entry = NULL;
                  s_ptr->num_linkage_entries = 0;
                  s_ptr->linkage_desc_list = NULL;

                  linkage_desc_ptr = sdt_entry->linkage_desc_list;
                  while (linkage_desc_ptr != NULL)
                  {
                     /* Copy the linkage descripter to the service */
                     CopyLinkageDesc(linkage_desc_ptr,
                        &s_ptr->linkage_desc_list,
                        &s_ptr->last_linkage_entry,
                        &s_ptr->num_linkage_entries);

                     linkage_desc_ptr = linkage_desc_ptr->next;
                  }

#ifdef COMMON_INTERFACE
                  /* The CI protection desc should only be updated if the SDT has come
                   * from a trusted source */
                  if (ACI_IsTrustedPath(path) && (mode != DB_ACCESS_SEARCH))
                  {
                     s_ptr->sdt_received = TRUE;

                     if (s_ptr->ci_protection_desc != NULL)
                     {
                        STB_FreeMemory(s_ptr->ci_protection_desc);
                        s_ptr->ci_protection_desc = NULL;
                     }

                     /* Check whether the ci_protection_descriptor for the service needs to be updated */
                     if (sdt_entry->ci_protection_desc != NULL)
                     {
                        /* Save the raw desc data and clear the reference to it from the SDT service
                         * entry to ensure it isn't freed */
                        s_ptr->ci_protection_desc = sdt_entry->ci_protection_desc;
                        sdt_entry->ci_protection_desc = NULL;

                        /* Save the update time of the descriptor */
                        s_ptr->ci_prot_last_update = STB_OSGetClockRTC();
                     }
                  }
                  else
                  {
                     s_ptr->sdt_received = FALSE;
                  }
#endif
               }

               sdt_entry = sdt_entry->next;
            }

            /* For Nordig go through all services and delete the ones that are no longer available */
            if ((mode == DB_ACCESS_SEARCH) && ACFG_IsNordigCountry())
            {
               s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr);
               while (s_ptr != NULL)
               {
                  next_s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);

                  if (s_ptr->unavailable)
                  {
                     #ifdef DEBUG_SI_SDT
                     AP_SI_PRINT(("   sid 0x%04x deleted as not available", s_ptr->serv_id));
                     #endif
                     DBDEF_DeleteServiceRec(s_ptr);
                  }

                  s_ptr = next_s_ptr;
               }
            }
            else if ((mode == DB_ACCESS_UPDATE) && ASI_DatabaseUpdatesAllowed(t_ptr->sig_type) &&
               ACFG_GetDynamicSIUpdate(t_ptr->sig_type, t_ptr->orig_net_id, ACFG_DYNAMIC_SI_UPDATE_SERVICE_REMOVE) &&
               (dynamic_update_head != NULL))
            {
               /* Services that are no longer available should be deleted, if possible */
               for (s_ptr = DBDEF_GetNextServiceOnTransport(NULL, t_ptr); s_ptr != NULL; s_ptr = next_s_ptr)
               {
                  next_s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, t_ptr);

                  if (s_ptr->unavailable)
                  {
                     /* If moving of a service is supported then the deletion has to be delayed
                      * until all SDTs have been received to check whether this service is being
                      * moved or deleted. If it's being deleted then this will be done later */
                     if (ACFG_GetDynamicSIUpdate(t_ptr->sig_type, t_ptr->orig_net_id, ACFG_DYNAMIC_SI_UPDATE_SERVICE_MOVE))
                     {
                        DynamicUpdateDeleteService(t_ptr, s_ptr, TRUE);
                     }
                     else
                     {
                        /* The service can only be deleted if it isn't the current service */
                        if (current_service_rec[path] != s_ptr)
                        {
                           /* The service can be deleted */
                           #ifdef DEBUG_SI_SDT
                           AP_SI_PRINT(("   sid 0x%04x deleted", s_ptr->serv_id));
                           #endif
                           STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_DELETED,
                              &s_ptr, sizeof(s_ptr));

                           DBDEF_DeleteServiceRec(s_ptr);
                        }
                        else
                        {
                           /* The service can't be deleted so an event is sent to ask the
                            * controlling code to delete it */
                           #ifdef DEBUG_SI_SDT
                           AP_SI_PRINT(("   sid 0x%04x needs to be deleted", s_ptr->serv_id));
                           #endif
                           STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_DELETE_SERVICE,
                              &s_ptr, sizeof(s_ptr));
                        }
                     }
                  }
               }
            }

            if ((current_service_rec[path] != NULL) && (t_ptr == current_transport_rec[path]))
            {
               if (current_service_rec[path]->unavailable)
               {
                  #ifdef DEBUG_SI_SDT
                  AP_SI_PRINT(("   sid 0x%04x report not running", current_service_rec[path]->serv_id));
                  #endif
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_NOT_RUNNING,
                     &path, sizeof(U8BIT));
                  STB_SIReportCurrentPmt(current_service_rec[path]->serv_id, NULL, FALSE, FALSE);
               }
            }

            // release database access
            DBDEF_ReleaseAccess();
         }
      }

      ApplyDynamicUpdates(path);

      if (update_sdt_func != NULL)
      {
         /* Call the function that's been registered to handle additional SDT processing */
         (*update_sdt_func)(path, sdt_table, table_rec);
      }

#ifdef INTEGRATE_HBBTV
      if ((mode != DB_ACCESS_SEARCH) && retval)
      {
         HBBTV_NotifyServiceListChange();
      }
#endif

      // release the table
      STB_SIReleaseSdtTable(sdt_table);
   }

   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessSdtTable);
   return(retval);
}

/**
 *

 *
 * @brief   Processes the PAT table record to extract data for the database
 *
 * @param   table_rec - pointer to the pat table record
 *
 * @return   TRUE if pmt list has changed
 *
 */
static BOOLEAN ProcessPatTable(SI_TABLE_RECORD *table_rec)
{
   BOOLEAN retval;
   SI_PAT_TABLE *pat_table;
   SI_PAT_SERVICE_ENTRY *pat_entry;
   ADB_SERVICE_REC *s_ptr;
   U16BIT i;
   BOOLEAN found;
   U8BIT path;
   ADB_PMT_VERSION_REC *pmt_ver_rec;
   ADB_PMT_VERSION_REC *prev_ver_rec;

   FUNCTION_START(ProcessPatTable);

   retval = FALSE;

   // convert table_rec to pat_table
   pat_table = STB_SIParsePatTable(table_rec);

   path = table_rec->path;

   STB_SIReleaseTableRecord(table_rec);

   if (pat_table != NULL)
   {
      #ifdef DEBUG_SI_PAT
      if (pmt_list[path] == NULL)
      {
         AP_SI_PRINT(("PAT Table: tran_id 0x%04x v%d %d services",
                      pat_table->tran_id, pat_table->version, pat_table->num_services));
      }
      else
      {
         if (current_transport_rec[path]->pat_version != pat_table->version)
         {
            AP_SI_PRINT(("PAT Table: tran_id 0x%04x v%d %d services - new",
                         pat_table->tran_id, pat_table->version, pat_table->num_services));
         }
         else
         {
            AP_SI_PRINT(("PAT Table: tran_id 0x%04x v%d %d services - same",
                         pat_table->tran_id, pat_table->version, pat_table->num_services));
         }
      }
      #endif

      if ((pat_rcvd_on_this_trnsprt[path] == FALSE) || (current_transport_rec[path]->pat_version != pat_table->version))
      {
         retval = TRUE;
         current_transport_rec[path]->pat_version = pat_table->version;
         pat_rcvd_on_this_trnsprt[path] = TRUE;

         // delete old list if one exists
         if (pmt_list[path] != NULL)
         {
            STB_AppFreeMemory(pmt_list[path]);
         }

         // create new pmt list from entries in PAT
         num_pmt_list_entries[path] = pat_table->num_services;
         pmt_list[path] = STB_AppGetMemory(num_pmt_list_entries[path] * sizeof(PMT_LIST_ENTRY));
         pmt_list_id[path] = 0;
         if (pmt_list[path] != NULL)
         {
            pat_entry = pat_table->service_list;
            for (i = 0; ((i < num_pmt_list_entries[path]) && (pat_entry != NULL)); i++)
            {
               #ifdef DEBUG_SI_PAT
               AP_SI_PRINT(("   sid 0x%04x pmt 0x%04x", pat_entry->serv_id, pat_entry->pmt_pid));
               #endif
               pmt_list[path][i].serv_id = pat_entry->serv_id;
               pmt_list[path][i].ts_id = ADB_GetTransportTid(current_transport_rec[path]);
               pmt_list[path][i].on_id = ADB_GetTransportOriginalNetworkId(current_transport_rec[path]);
               pmt_list[path][i].pid = pat_entry->pmt_pid;

               pat_entry = pat_entry->next;
            }
            // confirm num_pmt_list_entries in case the loop exited with pat_entry == NULL
            num_pmt_list_entries[path] = i;
         }

         // save pat data to the database...
         DBDEF_RequestAccess();

         // search through all services on the current transport (if any) - if the service is not
         // listed in the pat table then mark it not running
         s_ptr = DBDEF_GetNextServiceOnTransport(NULL, current_transport_rec[path]);
         while (s_ptr != NULL)
         {
            found = FALSE;
            for (i = 0; i < num_pmt_list_entries[path]; i++)
            {
               if (pmt_list[path][i].serv_id == s_ptr->serv_id)
               {
                  found = TRUE;
                  break;
               }
            }

            if (found == FALSE)
            {
               if (s_ptr->not_running == FALSE)
               {
                  #ifdef DEBUG_SI_PAT
                  AP_SI_PRINT(("   sid 0x%04x marked not running", s_ptr->serv_id));
                  #endif
                  // service not listed in PAT - mark it not running and clear the details in the
                  // service that normally come from the pmt
                  s_ptr->not_running = TRUE;
                  DBDEF_DeleteStreamList(s_ptr->stream_list);
                  s_ptr->stream_list = NULL;
                  s_ptr->video_pid = 0;
                  s_ptr->audio_pid = 0;
                  s_ptr->subtitle_pid = 0;
                  s_ptr->pcr_pid = 0;

                  /* Delete the service's PMT version rec from the transport list */
                  pmt_ver_rec = current_transport_rec[path]->pmt_version_list;
                  prev_ver_rec = NULL;
                  while (pmt_ver_rec != NULL)
                  {
                     if (pmt_ver_rec->serv_id == s_ptr->serv_id)
                     {
                        /* Found the entry for this service.
                         * Check whether it's first in the list and remove it */
                        if (prev_ver_rec == NULL)
                        {
                           current_transport_rec[path]->pmt_version_list = pmt_ver_rec->next;
                        }
                        else
                        {
                           prev_ver_rec->next = pmt_ver_rec->next;
                        }

                        STB_AppFreeMemory(pmt_ver_rec);
                        pmt_ver_rec = NULL;
                     }
                     else
                     {
                        prev_ver_rec = pmt_ver_rec;
                        pmt_ver_rec = pmt_ver_rec->next;
                     }
                  }

                  /* If it's the current service tell the application it's no longer running */
                  if (s_ptr == current_service_rec[path])
                  {
                     #ifdef DEBUG_SI_PAT
                     AP_SI_PRINT(("   sid 0x%04x report not running", s_ptr->serv_id));
                     #endif
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_NOT_RUNNING,
                        &path, sizeof(U8BIT));
                     STB_SIReportCurrentPmt(s_ptr->serv_id, NULL, FALSE, FALSE);
                  }
               }
            }
            else
            {
               /* A PMT is present for this service, save it in the service structure */
               DBDEF_SetServicePmtPid(s_ptr, pmt_list[path][i].pid);
            }
            s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, current_transport_rec[path]);
         }

         // release database access
         DBDEF_ReleaseAccess();
      }
      // release the table
      STB_SIReleasePatTable(pat_table);
   }

   FUNCTION_FINISH(ProcessPatTable);
   return(retval);
}

/**
 *

 *
 * @brief   Processes the PAT table record to extract data for the database
 *
 * @param   table_rec - pointer to the pat table record
 *
 * @return   TRUE if pmt list has changed
 *
 */
static BOOLEAN ProcessPatTablePlayback(SI_TABLE_RECORD *table_rec)
{
   BOOLEAN retval;
   SI_PAT_TABLE *pat_table;
   SI_PAT_SERVICE_ENTRY *pat_entry;
   U16BIT i;
   U8BIT path;

   FUNCTION_START(ProcessPatTablePlayback);

   retval = FALSE;

   // convert table_rec to pat_table
   pat_table = STB_SIParsePatTable(table_rec);

   path = table_rec->path;

   STB_SIReleaseTableRecord(table_rec);

   if (pat_table != NULL)
   {
      if ((pat_rcvd_on_this_trnsprt[path] == FALSE) || (last_playback_pat_version != pat_table->version))
      {
         retval = TRUE;
         pat_rcvd_on_this_trnsprt[path] = TRUE;
         last_playback_pat_version = pat_table->version;

         pat_entry = pat_table->service_list;
         for (i = 0; ((i < pat_table->num_services) && (pat_entry != NULL)); i++)
         {
            if ((pat_entry->serv_id == current_service_rec[path]->serv_id) ||
                (current_service_rec[path]->serv_id == 0))
            {
               #ifdef DEBUG_SI_PAT_PLAYBACK
               AP_SI_PRINT(("ProcessPatTablePlayback: path=%u found sid 0x%04x pmt 0x%04x", path, pat_entry->serv_id, pat_entry->pmt_pid));
               #endif

               pmt_filter_pid[path] = pat_entry->pmt_pid;

               /* If serv_id == 0 in the current service, it means we were expecting a single service PAT.
                * Use the serv_id in the PAT to set the serv_id of the current service record */
               if (current_service_rec[path]->serv_id == 0)
               {
                  current_service_rec[path]->serv_id = pat_entry->serv_id;
               }
               break;
            }

            pat_entry = pat_entry->next;
         }
      }
      // release the table
      STB_SIReleasePatTable(pat_table);
   }

   FUNCTION_FINISH(ProcessPatTablePlayback);

   return(retval);
}

/**
 *

 *
 * @brief   Processes the PMT table record to extract data for the database
 *
 * @param   table_rec      - pointer to the pmt table record
 * @param   ignore_version - if TRUE then the pmt contents are to be analysed even if it is
 *                                 the same version as previously received, if FALSE it will only
 *                                 be analysed if the version has changed
 * @param   report_pmt     - if TRUE pmt will be reported to 3rd parties if this is the first
 *                                 copy after channel change or if it is a new version. if FALSE the
 *                                 pmt will not be reported (i.e. during search)
 * @param   service_id     - pointer to value to return service ID in, can be NULL
 *
 * @return   TRUE if a service is updated, FALSE otherwise
 *
 */
static BOOLEAN ProcessPmtTable(SI_TABLE_RECORD *table_rec, BOOLEAN ignore_version, BOOLEAN report_pmt,
   U16BIT *service_id, E_DB_ACCESS_MODE mode)
{
   ADB_PMT_VERSION_REC *pmt_ver_rec;
   BOOLEAN pmt_ver_changed;
   ADB_SERVICE_REC *s_ptr;
   U8BIT path;
   BOOLEAN service_updated;
   BOOLEAN pmt_received = FALSE;
#ifdef COMMON_INTERFACE
   void *owner_data;
   U32BIT data_size;
#endif

   FUNCTION_START(ProcessPmtTable);

   path = table_rec->path;

   // get access to the database (may suspend)
   DBDEF_RequestAccess();

   if (mode == DB_ACCESS_PLAYBACK)
   {
      if (last_playback_pmt_version != table_rec->version)
      {
         pmt_ver_changed = TRUE;
         last_playback_pmt_version = table_rec->version;
         #ifdef DEBUG_SI_PMT_PLAYBACK
         AP_SI_PRINT(("Playback, PMT table: serv_id 0x%04x v%d - update",
                      table_rec->xtid, table_rec->version));
         #endif
      }
      else
      {
         pmt_ver_changed = FALSE;
         #ifdef DEBUG_SI_PMT_PLAYBACK
         if (ignore_version == TRUE)
         {
            AP_SI_PRINT(("Playback, PMT table: serv_id 0x%04x v%d - same",
                         table_rec->xtid, table_rec->version));
         }
         #endif
      }
   }
   else
   {
      // find or add entry in the pmt history list
      pmt_ver_changed = TRUE;
      pmt_ver_rec = current_transport_rec[path]->pmt_version_list;
      while (pmt_ver_rec != NULL)
      {
         if (pmt_ver_rec->serv_id == table_rec->xtid)
         {
            // found entry - check if the version has changed
            if (pmt_ver_rec->version != table_rec->version)
            {
               pmt_ver_rec->version = table_rec->version;
               #ifdef DEBUG_SI_PMT
               AP_SI_PRINT(("PMT table: serv_id 0x%04x v%d - update",
                            table_rec->xtid, table_rec->version));
               #endif
            }
            else
            {
               pmt_ver_changed = FALSE;
               #ifdef DEBUG_SI_PMT
               if (ignore_version == TRUE)
               {
                  AP_SI_PRINT(("PMT table: serv_id 0x%04x v%d - same",
                               table_rec->xtid, table_rec->version));
               }
               #endif
            }
            break;
         }
         pmt_ver_rec = pmt_ver_rec->next;
      }
      if (pmt_ver_rec == NULL)
      {
         // no history entry - add one
         pmt_ver_rec = STB_AppGetMemory(sizeof(ADB_PMT_VERSION_REC));
         if (pmt_ver_rec != NULL)
         {
            pmt_ver_rec->serv_id = table_rec->xtid;
            pmt_ver_rec->version = table_rec->version;
            pmt_ver_rec->next = current_transport_rec[path]->pmt_version_list;
            current_transport_rec[path]->pmt_version_list = pmt_ver_rec;
            #ifdef DEBUG_SI_PMT
            AP_SI_PRINT(("PMT table: serv_id 0x%04x v%d - new",
                         table_rec->xtid, table_rec->version));
            #endif
         }
      }
   }

   if (service_id != NULL)
   {
      *service_id = table_rec->xtid;
   }

   service_updated = FALSE;

   if (mode == DB_ACCESS_PLAYBACK)
   {
      s_ptr = current_service_rec[path];
   }
   else
   {
      // find the service matching this pmt (should be one because it would have been added by
      // the SDT, if there isn't then skip this pmt)
      s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
            current_transport_rec[path]->orig_net_id, current_transport_rec[path]->tran_id,
            table_rec->xtid);
   }

#ifdef DEBUG_SI_PMT
   AP_SI_PRINT(("PMT table, service = %p, ignore_version=%u, pmt_ver_changed=%u", s_ptr,
                ignore_version, pmt_ver_changed));
#endif

   if (s_ptr != NULL)
   {
      /* Remember whether the PMT for the service had already been received, because after
       * being processed the flag will be TRUE */
      pmt_received = s_ptr->pmt_received;

      // if the service is marked not running, or we don't care about the version, or the version
      // has changed, or we don't yet have valid PIDs, then process the pmt
      if ((ignore_version == TRUE) || (pmt_ver_changed == TRUE))
      {
         service_updated = InternalProcessPmtTable(path, s_ptr, table_rec,
               ((mode == DB_ACCESS_PLAYBACK) ? FALSE : TRUE), mode);
      }
   }
   else
   {
      if (update_pmt_func != NULL)
      {
         /* Call the function that's been registered to handle additional PMT processing */
         (*update_pmt_func)(path, NULL, table_rec, NULL);
      }
   }

   // release database access
   DBDEF_ReleaseAccess();

   if (pmt_service_changed[path] && STB_DPIsLivePath(path) &&
       (s_ptr == current_service_rec[path]) && !pmt_received)
   {
      /* Service for live path has changed, inform the UI */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_CHANGED, &s_ptr, sizeof(void *));
   }

   // check if pmt needs reporting to other applications (e.g. ca systems or MHEG)
   if (report_pmt == TRUE)
   {
      if (report_pmt_allowed[path] == TRUE)
      {
         #ifdef DEBUG_SI_PMT
         AP_SI_PRINT(("Report pmt: serv_id 0x%04x v%d", table_rec->xtid, table_rec->version));
         #endif

#ifdef COMMON_INTERFACE
         if (pmt_service_changed[path] && STB_DPIsOwnedBy(path, RES_OWNER_CIPLUS))
         {
            /* The service has changed and the path is owned by CI+,
             * so inform it that the tune operation has completed */
            if ((owner_data = STB_DPGetOwnerData(path, &data_size)) != NULL)
            {
               if (table_rec->xtid == current_service_rec[path]->serv_id)
               {
                  ACI_TuneReply(path, *(U32BIT *)owner_data, CIP_TUNER_LOCKED);
               }
               else
               {
                  ACI_TuneReply(path, *(U32BIT *)owner_data, CIP_TUNER_SERVICE_NOT_FOUND);
               }
            }
         }
#endif

         if ((current_service_rec[path] != NULL) &&
             (table_rec->xtid == current_service_rec[path]->serv_id) &&
             (table_rec->version != last_reported_pmt_version[path]))
         {
            pmt_ver_changed = TRUE;
            last_reported_pmt_version[path] = table_rec->version;
         }
         else
         {
            pmt_ver_changed = FALSE;
         }

         STB_SIReportCurrentPmt(table_rec->xtid, table_rec, pmt_service_changed[path], pmt_ver_changed);

#ifdef COMMON_INTERFACE
         if (table_rec->section_list != NULL)
         {
            ACI_ProgramMapTableChanged(&(table_rec->section_list->data_start));
         }
#endif

         pmt_reported[path] = TRUE;
         pmt_service_changed[path] = FALSE;
      }
   }

   // free up the table record
   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessPmtTable);

   return(service_updated);
}

/*!**************************************************************************
 * @brief   Internal function to process a PMT table and update the database
 * @param   path - decode path the table was received from
 * @param   s_ptr - first service that the PMT is for
 * @param   table_rec - pointer to the SI table data
 * @param   db_services - TRUE if the database is to be searched for additional
 *                        identical services that the PMT is to be used for
 * @return  TRUE if a service is updated, FALSE otherwise
 ****************************************************************************/
static BOOLEAN InternalProcessPmtTable(U8BIT path, ADB_SERVICE_REC *s_ptr, SI_TABLE_RECORD *table_rec,
   BOOLEAN db_services, E_DB_ACCESS_MODE mode)
{
   SI_PMT_TABLE *pmt_table;
   ADB_STREAM_REC **last_stream_rec_next_ptr;
   SI_PMT_STREAM_ENTRY *pmt_entry;
   ADB_STREAM_REC *stream_rec = NULL;
   ADB_STREAM_REC *srec;
   ADB_STREAM_REC *old_stream_list;
   U16BIT i;
   U16BIT subt_pid;
   U16BIT subt_cpage;
   U16BIT subt_apage;
   BOOLEAN prev_not_running;
   BOOLEAN match_found;
   E_STB_DP_AUDIO_MODE audio_mode;
   ADB_STREAM_TYPE audio_type;
   ADB_STREAM_TYPE adesc_type;
   ADB_STREAM_TYPE video_type;
   U16BIT video_pid;
   U16BIT audio_pid;
   U16BIT adesc_pid;
   BOOLEAN service_updated;
   U16BIT ttxt_pid;
   U8BIT ttxt_mag;
   U8BIT ttxt_page;
#if defined(DEBUG_SI_PMT_AUDIO) || defined(DEBUG_SI_PMT_VIDEO)
   U8BIT tag_count;
#endif
#ifdef COMMON_INTERFACE
   U8BIT slot_id;
#endif

   FUNCTION_START(InternalProcessPmtTable);

   service_updated = FALSE;

   pmt_table = STB_SIParsePmtTable(table_rec);
   if (pmt_table != NULL)
   {
      /* The same service may have multiple LCNs (e.g. Finnish Nordig spec), which is handled by
       * having multiple copies of the same service, so the PMT may apply to more than one service */
      while (s_ptr != NULL)
      {
         if (s_ptr->pmt_data != NULL)
         {
            STB_AppFreeMemory(s_ptr->pmt_data);
            s_ptr->pmt_data = NULL;
            s_ptr->pmt_data_len = 0;
         }

         /* Store the PMT with the service for use by CI+ and other CA systems */
         s_ptr->pmt_data = STB_AppGetMemory(table_rec->section_list->data_len);
         if (s_ptr->pmt_data != NULL)
         {
            memcpy(s_ptr->pmt_data, &(table_rec->section_list->data_start),
               table_rec->section_list->data_len);
            s_ptr->pmt_data_len = table_rec->section_list->data_len;
         }

#ifdef COMMON_INTERFACE
         if ((slot_id = STB_DPGetPathCISlot(path)) != INVALID_RES_ID)
         {
            /* Re-evaluate state even if service is not scrambled */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_URI_UPDATED, &slot_id, sizeof(slot_id));
         }
#endif
         service_updated = TRUE;
         s_ptr->pmt_received = TRUE;

         /* Save the current PIDs and type so we can check whether they are changed */
         adesc_pid = DBDEF_GetReqdADPid(s_ptr, &audio_mode, &adesc_type, &match_found);
         video_pid = s_ptr->video_pid;
         audio_pid = s_ptr->audio_pid;
         video_type = s_ptr->video_type;
         audio_type = s_ptr->audio_type;

         DBDEF_GetReqdSubtitleParams(s_ptr, &subt_pid, &subt_cpage, &subt_apage);
         DBDEF_GetReqdTtextPid(s_ptr, TRUE, &ttxt_pid, &ttxt_mag, &ttxt_page);

         // extract the data from the pmt and add to the service record
         // all data from the pmt is stored in ram only so no need to update nvm. Remember to
         // free old data before the new data is added.

         /* Save the service's current stream list so it can be checked
          * to see if any have changed */
         old_stream_list = s_ptr->stream_list;

         s_ptr->stream_list = NULL;
         s_ptr->video_pid = 0;
         s_ptr->audio_pid = 0;
         s_ptr->subtitle_pid = 0;
         s_ptr->pcr_pid = pmt_table->pcr_pid;
         s_ptr->rct_pid = 0;

         prev_not_running = s_ptr->not_running;
         s_ptr->not_running = FALSE;

         if (pmt_table->ca_desc_array == NULL)
         {
            s_ptr->has_ca_descriptor = FALSE;
            s_ptr->ecm_pid = ADB_INVALID_DVB_ID;
         }
         else
         {
            s_ptr->has_ca_descriptor = TRUE;
            //FIXME
            s_ptr->ecm_pid = pmt_table->ca_desc_array[0].ca_pid;
         }

         /* Save the content protection level with the service so it can be used when content
          * protection is applied */
         s_ptr->content_protection_level = pmt_table->content_protection_level;

#ifdef INTEGRATE_HBBTV
         s_ptr->ait_pid = 0;
#endif

         // copy pmt data to service.
         last_stream_rec_next_ptr = &(s_ptr->stream_list);
         pmt_entry = pmt_table->stream_list;
         while (pmt_entry != NULL)
         {
            switch (pmt_entry->type)
            {
               case SI_STREAM_TYPE_VIDEO1:
               case SI_STREAM_TYPE_VIDEO2:
               case SI_STREAM_TYPE_H264:
               case SI_STREAM_TYPE_H265:
               {
                  // make new stream entry and add to end of list
                  stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                  if (stream_rec != NULL)
                  {
                     memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                     /* Now copying all tags as there may be multiple */
                     CopyComponentTagArray(stream_rec, pmt_entry);
                     stream_rec->pid = pmt_entry->pid;
                     if (pmt_entry->type == SI_STREAM_TYPE_H264)
                     {
                        stream_rec->type = ADB_H264_VIDEO_STREAM;
                     }
                     else if (pmt_entry->type == SI_STREAM_TYPE_H265)
                     {
                        stream_rec->type = ADB_H265_VIDEO_STREAM;
                     }
                     else
                     {
                        stream_rec->type = ADB_VIDEO_STREAM;
                     }
                     stream_rec->next = NULL;
                     *last_stream_rec_next_ptr = stream_rec;
                     last_stream_rec_next_ptr = &(stream_rec->next);

                     #ifdef DEBUG_SI_PMT_VIDEO
                     AP_SI_PRINT(("   video stream: pid 0x%04x", pmt_entry->pid));
                     for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                     {
                        AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                     }
                     #endif
                  }
                  break;
               }

               case SI_STREAM_TYPE_AUDIO1:
               case SI_STREAM_TYPE_AUDIO2:
               case SI_STREAM_TYPE_AAC:
               case SI_STREAM_TYPE_HEAAC:
               {
                  if (pmt_entry->num_iso_lang_entries == 0)
                  {
                     stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                     if (stream_rec != NULL)
                     {
                        memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                        /* Now copying all tags as there may be multiple */
                        CopyComponentTagArray(stream_rec, pmt_entry);
                        stream_rec->pid = pmt_entry->pid;
                        if (pmt_entry->type == SI_STREAM_TYPE_AAC)
                        {
                           if (pmt_entry->aac_descriptor != NULL)
                           {
                              switch (pmt_entry->aac_descriptor->profile_level)
                              {
                                 case 0x58:
                                 case 0x59:
                                 case 0x5a:
                                 case 0x5b:
                                    stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                                    break;

                                 case 0x60:
                                 case 0x61:
                                 case 0x62:
                                 case 0x63:
                                    stream_rec->type = ADB_HEAACv2_AUDIO_STREAM;
                                    break;

                                 default:
                                    stream_rec->type = ADB_AAC_AUDIO_STREAM;
                                    break;
                              }

                              if (pmt_entry->aac_descriptor->type_present)
                              {
                                 switch (pmt_entry->aac_descriptor->aac_type)
                                 {
                                    case 0x01:
                                       stream_rec->data.audio.mode = AUDIO_MONO;
                                       break;

                                    case 0x05:
                                       stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                       break;

                                    case 0x03:
                                    case 0x43:
                                    default:
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                 }
                              }
                              else
                              {
                                 /* Default to stereo */
                                 stream_rec->data.audio.mode = AUDIO_STEREO;
                              }
                           }
                           else
                           {
                              /* Default to AAC stereo */
                              stream_rec->type = ADB_AAC_AUDIO_STREAM;
                              stream_rec->data.audio.mode = AUDIO_STEREO;
                           }
                        }
                        else if (pmt_entry->type == SI_STREAM_TYPE_HEAAC)
                        {
                           if (pmt_entry->aac_descriptor != NULL)
                           {
                              switch (pmt_entry->aac_descriptor->profile_level)
                              {
                                 case 0x60:
                                 case 0x61:
                                 case 0x62:
                                 case 0x63:
                                    stream_rec->type = ADB_HEAACv2_AUDIO_STREAM;
                                    break;

                                 default:
                                    stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                                    break;
                              }

                              if (pmt_entry->aac_descriptor->type_present)
                              {
                                 switch (pmt_entry->aac_descriptor->aac_type)
                                 {
                                    case 0x01:
                                       stream_rec->data.audio.mode = AUDIO_MONO;
                                       break;

                                    case 0x05:
                                       stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                       break;

                                    case 0x03:
                                    case 0x43:
                                    default:
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                 }
                              }
                              else
                              {
                                 /* Default to stereo */
                                 stream_rec->data.audio.mode = AUDIO_STEREO;
                              }
                           }
                           else
                           {
                              /* Default to HE-AAC stereo */
                              stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                              stream_rec->data.audio.mode = AUDIO_STEREO;
                           }
                        }
                        else
                        {
                           stream_rec->type = ADB_AUDIO_STREAM;
                           stream_rec->data.audio.mode = AUDIO_UNDEF;
                        }
                        stream_rec->data.audio.lang_code = ACFG_UNDEFINED_DB_LANG_CODE;
                        stream_rec->data.audio.type = ADB_AUDIO_TYPE_UNDEFINED;

                        stream_rec->next = NULL;
                        *last_stream_rec_next_ptr = stream_rec;
                        last_stream_rec_next_ptr = &(stream_rec->next);

                        #ifdef DEBUG_SI_PMT_AUDIO
                        AP_SI_PRINT(("   %s audio stream: pid 0x%04x, type %u, mode %u",
                                     ((stream_rec->type == ADB_HEAACv2_AUDIO_STREAM) ? "HE-AACv2" :
                                      ((stream_rec->type == ADB_HEAAC_AUDIO_STREAM) ? "HE-AACv1" :
                                       ((stream_rec->type == ADB_AAC_AUDIO_STREAM) ? "AAC" : "MPEG"))),
                                     pmt_entry->pid,
                                     stream_rec->data.audio.type,
                                     stream_rec->data.audio.mode));
                        for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                        {
                           AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                        }
                        #endif
                     }
                  }
                  else
                  {
                     // make new stream entry and add to end of list. Add an entry for each iso
                     // lang desc with same tag and pid.
                     for (i = 0; i < pmt_entry->num_iso_lang_entries; i++)
                     {
                        stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                        if (stream_rec != NULL)
                        {
                           memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                           /* Now copying all tags as there may be multiple */
                           CopyComponentTagArray(stream_rec, pmt_entry);
                           stream_rec->pid = pmt_entry->pid;
                           if (pmt_entry->type == SI_STREAM_TYPE_AAC)
                           {
                              if (pmt_entry->aac_descriptor != NULL)
                              {
                                 switch (pmt_entry->aac_descriptor->profile_level)
                                 {
                                    case 0x58:
                                    case 0x59:
                                    case 0x5a:
                                    case 0x5b:
                                       stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                                       break;

                                    case 0x60:
                                    case 0x61:
                                    case 0x62:
                                    case 0x63:
                                       stream_rec->type = ADB_HEAACv2_AUDIO_STREAM;
                                       break;

                                    default:
                                       stream_rec->type = ADB_AAC_AUDIO_STREAM;
                                       break;
                                 }

                                 if (pmt_entry->aac_descriptor->type_present)
                                 {
                                    switch (pmt_entry->aac_descriptor->aac_type)
                                    {
                                       case 0x01:
                                          stream_rec->data.audio.mode = AUDIO_MONO;
                                          break;

                                       case 0x05:
                                          stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                          break;

                                       case 0x03:
                                       case 0x43:
                                       default:
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                    }
                                 }
                                 else
                                 {
                                    /* Default to stereo */
                                    stream_rec->data.audio.mode = AUDIO_STEREO;
                                 }
                              }
                              else
                              {
                                 /* Default to AAC stereo */
                                 stream_rec->type = ADB_AAC_AUDIO_STREAM;
                                 stream_rec->data.audio.mode = AUDIO_STEREO;
                              }
                           }
                           else if (pmt_entry->type == SI_STREAM_TYPE_HEAAC)
                           {
                              if (pmt_entry->aac_descriptor != NULL)
                              {
                                 switch (pmt_entry->aac_descriptor->profile_level)
                                 {
                                    case 0x60:
                                    case 0x61:
                                    case 0x62:
                                    case 0x63:
                                       stream_rec->type = ADB_HEAACv2_AUDIO_STREAM;
                                       break;

                                    default:
                                       stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                                       break;
                                 }

                                 if (pmt_entry->aac_descriptor->type_present)
                                 {
                                    switch (pmt_entry->aac_descriptor->aac_type)
                                    {
                                       case 0x01:
                                          stream_rec->data.audio.mode = AUDIO_MONO;
                                          break;

                                       case 0x05:
                                          stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                          break;

                                       case 0x03:
                                       case 0x43:
                                       default:
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                    }
                                 }
                                 else
                                 {
                                    /* Default to stereo */
                                    stream_rec->data.audio.mode = AUDIO_STEREO;
                                 }
                              }
                              else
                              {
                                 /* Default to HE-AAC stereo */
                                 stream_rec->type = ADB_HEAAC_AUDIO_STREAM;
                                 stream_rec->data.audio.mode = AUDIO_STEREO;
                              }
                           }
                           else
                           {
                              stream_rec->type = ADB_AUDIO_STREAM;
                              stream_rec->data.audio.mode = AUDIO_UNDEF;
                           }
                           stream_rec->data.audio.lang_code =
                              pmt_entry->iso_lang_desc_array[i].lang_code;
                           stream_rec->data.audio.type =
                              pmt_entry->iso_lang_desc_array[i].audio_type;

                           stream_rec->next = NULL;
                           *last_stream_rec_next_ptr = stream_rec;
                           last_stream_rec_next_ptr = &(stream_rec->next);

                           #ifdef DEBUG_SI_PMT_AUDIO
                           AP_SI_PRINT(("   %s audio stream: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                        ((stream_rec->type == ADB_HEAACv2_AUDIO_STREAM) ? "HE-AACv2" :
                                         ((stream_rec->type == ADB_HEAAC_AUDIO_STREAM) ? "HE-AACv1" :
                                          ((stream_rec->type == ADB_AAC_AUDIO_STREAM) ? "AAC" : "MPEG"))),
                                        pmt_entry->pid,
                                        ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                        ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                        (stream_rec->data.audio.lang_code & 0xff),
                                        stream_rec->data.audio.type,
                                        stream_rec->data.audio.mode));
                           for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                           {
                              AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                           }
                           #endif
                        }
                     }
                  }
                  break;
               }

               case SI_STREAM_TYPE_PES_PKT:
               {
                  if ((pmt_entry->ac3_descriptor != NULL) &&
                      ((pmt_entry->ac3_descriptor->dtag == AC3_DESC_DTAG) ||
                       (pmt_entry->ac3_descriptor->dtag == EAC3_DESC_DTAG)))
                  {
                     if (pmt_entry->num_iso_lang_entries == 0)
                     {
                        stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                        if (stream_rec != NULL)
                        {
                           memset(stream_rec, 0, sizeof(ADB_STREAM_REC));

                           /* Now copying all tags as there may be multiple */
                           CopyComponentTagArray(stream_rec, pmt_entry);

                           stream_rec->pid = pmt_entry->pid;
                           stream_rec->data.audio.type = ADB_AUDIO_TYPE_UNDEFINED;

                           if (pmt_entry->ac3_descriptor->dtag == AC3_DESC_DTAG)
                           {
                              stream_rec->type = ADB_AC3_AUDIO_STREAM;

                              if (pmt_entry->ac3_descriptor->component_type_flag)
                              {
                                 if ((pmt_entry->ac3_descriptor->component_type & 0x40) == 0)
                                 {
                                    /* This is a full service audio stream */
                                    stream_rec->data.audio.type = ADB_AUDIO_TYPE_CLEAN_EFFECTS;
                                 }
                                 else
                                 {
                                    switch ((pmt_entry->ac3_descriptor->component_type >> 3) & 0x07)
                                    {
                                       case 2:
                                          stream_rec->data.audio.type = ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED;
                                          break;
                                       case 3:
                                          stream_rec->data.audio.type = ADB_AUDIO_TYPE_FOR_HEARING_IMPAIRED;
                                          break;
                                       default:
                                          /* Default is undefined, which is already set */
                                          break;
                                    }
                                 }

                                 switch(pmt_entry->ac3_descriptor->component_type & 0x07)
                                 {
                                    case 0: /* Mono audio */
                                    case 1: /* 1+1 mode - dual mono */
                                       stream_rec->data.audio.mode = AUDIO_MONO;
                                       break;
                                    case 2: /* 2 channel stereo */
                                    case 3: /* 2 channel surround stereo */
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                    case 4: /* More than 2 channels */
                                       stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                       break;
                                    default: /* All other values are invalid for AC-3 */
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                 }
                              }
                              else
                              {
                                 /* Assume stereo audio in absense of definition */
                                 stream_rec->data.audio.mode = AUDIO_STEREO;
                              }
                           }
                           else
                           {
                              stream_rec->type = ADB_EAC3_AUDIO_STREAM;

                              if (pmt_entry->ac3_descriptor->component_type_flag)
                              {
                                 if ((pmt_entry->ac3_descriptor->component_type & 0x40) == 0)
                                 {
                                    /* This is a full service audio stream */
                                    stream_rec->data.audio.type = ADB_AUDIO_TYPE_CLEAN_EFFECTS;
                                 }
                                 else
                                 {
                                    switch ((pmt_entry->ac3_descriptor->component_type >> 3) & 0x07)
                                    {
                                       case 2:
                                          stream_rec->data.audio.type = ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED;
                                          break;
                                       case 3:
                                          stream_rec->data.audio.type = ADB_AUDIO_TYPE_FOR_HEARING_IMPAIRED;
                                          break;
                                       default:
                                          /* Default is undefined, which is already set */
                                          break;
                                    }
                                 }

                                 switch(pmt_entry->ac3_descriptor->component_type & 0x07)
                                 {
                                    case 0: /* Mono audio */
                                    case 1: /* 1+1 mode - dual mono */
                                       stream_rec->data.audio.mode = AUDIO_MONO;
                                       break;
                                    case 2: /* 2 channel stereo */
                                    case 3: /* 2 channel surround stereo */
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                    case 4: /* More than 2 channels */
                                    case 5: /* More than 5.1 channels */
                                       stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                       break;
                                    default: /* All other values are invalid for AC-3 */
                                       stream_rec->data.audio.mode = AUDIO_STEREO;
                                       break;
                                 }
                              }
                              else
                              {
                                 /* Assume multi-channel audio in absense of definition */
                                 stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                              }
                           }

                           stream_rec->data.audio.lang_code = ACFG_UNDEFINED_DB_LANG_CODE;

                           stream_rec->next = NULL;
                           *last_stream_rec_next_ptr = stream_rec;
                           last_stream_rec_next_ptr = &(stream_rec->next);

                           #ifdef DEBUG_SI_PMT_AUDIO
                           AP_SI_PRINT(("   %s audio stream: pid 0x%04x, type %u, mode %u",
                                        ((stream_rec->type == ADB_EAC3_AUDIO_STREAM) ? "E-AC3" : "AC3"),
                                        pmt_entry->pid,
                                        stream_rec->data.audio.type,
                                        stream_rec->data.audio.mode));
                           for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                           {
                              AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                           }
                           #endif
                        }
                     }
                     else
                     {
                        /* make new stream entry and add to end of list. Add an entry for each
                         * iso lang desc with same tag and pid. */
                        for (i = 0; i < pmt_entry->num_iso_lang_entries; i++)
                        {
                           stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                           if (stream_rec != NULL)
                           {
                              memset(stream_rec, 0, sizeof(ADB_STREAM_REC));

                              /* Now copying all tags as there may be multiple */
                              CopyComponentTagArray(stream_rec, pmt_entry);

                              stream_rec->pid = pmt_entry->pid;

                              if (pmt_entry->ac3_descriptor->dtag == AC3_DESC_DTAG)
                              {
                                 stream_rec->type = ADB_AC3_AUDIO_STREAM;

                                 if (pmt_entry->ac3_descriptor->component_type_flag)
                                 {
                                    switch(pmt_entry->ac3_descriptor->component_type & 0x07)
                                    {
                                       case 0: /* Mono audio */
                                       case 1: /* 1+1 mode - dual mono */
                                          stream_rec->data.audio.mode = AUDIO_MONO;
                                          break;
                                       case 2: /* 2 channel stereo */
                                       case 3: /* 2 channel surround stereo */
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                       case 4: /* More than 2 channels */
                                          stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                          break;
                                       default: /* All other values are invalid for AC-3 */
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                    }
                                 }
                                 else
                                 {
                                    /* Assume stereo audio in absense of definition */
                                    stream_rec->data.audio.mode = AUDIO_STEREO;
                                 }
                              }
                              else
                              {
                                 stream_rec->type = ADB_EAC3_AUDIO_STREAM;

                                 if (pmt_entry->ac3_descriptor->component_type_flag)
                                 {
                                    switch(pmt_entry->ac3_descriptor->component_type & 0x07)
                                    {
                                       case 0: /* Mono audio */
                                       case 1: /* 1+1 mode - dual mono */
                                          stream_rec->data.audio.mode = AUDIO_MONO;
                                          break;
                                       case 2: /* 2 channel stereo */
                                       case 3: /* 2 channel surround stereo */
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                       case 4: /* More than 2 channels */
                                       case 5: /* More than 5.1 channels */
                                          stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                          break;
                                       default: /* All other values are invalid for AC-3 */
                                          stream_rec->data.audio.mode = AUDIO_STEREO;
                                          break;
                                    }
                                 }
                                 else
                                 {
                                    /* Assume multi-channel audio in absense of definition */
                                    stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;
                                 }
                              }

                              stream_rec->data.audio.lang_code =
                                 pmt_entry->iso_lang_desc_array[i].lang_code;
                              stream_rec->data.audio.type =
                                 pmt_entry->iso_lang_desc_array[i].audio_type;

                              stream_rec->next = NULL;
                              *last_stream_rec_next_ptr = stream_rec;
                              last_stream_rec_next_ptr = &(stream_rec->next);

                              #ifdef DEBUG_SI_PMT_AUDIO
                              AP_SI_PRINT(("   %s audio stream: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                           ((stream_rec->type == ADB_EAC3_AUDIO_STREAM) ? "E-AC3" : "AC3"),
                                           pmt_entry->pid,
                                           ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                           ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                           (stream_rec->data.audio.lang_code & 0xff),
                                           stream_rec->data.audio.type,
                                           stream_rec->data.audio.mode));
                              for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                              {
                                 AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                              }
                              #endif
                           }
                        }
                     }
                  }
                  else
                  {
                     if (pmt_entry->subtitle_desc_array != NULL)
                     {
                        #ifdef DEBUG_SI_PMT
                        AP_SI_PRINT(("PMT: %d subtitle entries, PID = %d", pmt_entry->num_subtitle_entries,
                                     pmt_entry->pid));
                        #endif

                        // subtitle streams - make new stream entry for each subtitle descriptor
                        // and add to end of list
                        for (i = 0; i < pmt_entry->num_subtitle_entries; i++)
                        {
                           stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                           if (stream_rec != NULL)
                           {
                              memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                              /* Now copying all tags as there may be multiple */
                              CopyComponentTagArray(stream_rec, pmt_entry);
                              stream_rec->pid = pmt_entry->pid;
                              stream_rec->type = ADB_SUBTITLE_STREAM;
                              stream_rec->data.subtitle.lang_code =
                                 pmt_entry->subtitle_desc_array[i].lang_code;
                              stream_rec->data.subtitle.type =
                                 pmt_entry->subtitle_desc_array[i].type;
                              stream_rec->data.subtitle.composition_page =
                                 pmt_entry->subtitle_desc_array[i].composition_page;
                              stream_rec->data.subtitle.ancillary_page =
                                 pmt_entry->subtitle_desc_array[i].ancillary_page;

                              stream_rec->next = NULL;
                              *last_stream_rec_next_ptr = stream_rec;
                              last_stream_rec_next_ptr = &(stream_rec->next);
                           }
                        }
                     }
                     else
                     {
                        // teletext
                        if (pmt_entry->teletext_desc_array != NULL)
                        {
                           #ifdef DEBUG_SI_PMT
                           AP_SI_PRINT(("PMT: %d teletext entries, PID=%d", pmt_entry->num_teletext_entries,
                                        pmt_entry->pid));
                           #endif

                           // ttext streams - make new stream entry for each ttext descriptor
                           // and add to end of list
                           for (i = 0; i < pmt_entry->num_teletext_entries; i++)
                           {
                              #ifdef DEBUG_SI_PMT
                              AP_SI_PRINT(("   lang_code=0x%x, type=%d, magazine=%d, page=%d",
                                           pmt_entry->teletext_desc_array[i].lang_code,
                                           pmt_entry->teletext_desc_array[i].type,
                                           pmt_entry->teletext_desc_array[i].magazine,
                                           pmt_entry->teletext_desc_array[i].page));
                              #endif

                              stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                              if (stream_rec != NULL)
                              {
                                 memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                                 /* Now copying all tags as there may be multiple */
                                 CopyComponentTagArray(stream_rec, pmt_entry);
                                 stream_rec->pid = pmt_entry->pid;
                                 stream_rec->type = ADB_TTEXT_STREAM;
                                 stream_rec->data.ttext.lang_code =
                                    pmt_entry->teletext_desc_array[i].lang_code;
                                 stream_rec->data.ttext.type =
                                    pmt_entry->teletext_desc_array[i].type;
                                 stream_rec->data.ttext.magazine =
                                    pmt_entry->teletext_desc_array[i].magazine;
                                 stream_rec->data.ttext.page =
                                    pmt_entry->teletext_desc_array[i].page;

                                 stream_rec->next = NULL;
                                 *last_stream_rec_next_ptr = stream_rec;
                                 last_stream_rec_next_ptr = &(stream_rec->next);
                              }
                           }
                        }
                        else
                        {
                           #ifdef DEBUG_SI_PMT
                           AP_SI_PRINT(("PMT: data stream, PID=%d", pmt_entry->pid));
                           #endif

                           // not a subtitle stream - define it as a data stream
                           stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                           if (stream_rec != NULL)
                           {
                              memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                              /* Now copying all tags as there may be multiple */
                              CopyComponentTagArray(stream_rec, pmt_entry);
                              stream_rec->pid = pmt_entry->pid;
                              stream_rec->type = ADB_DATA_STREAM;

                              stream_rec->next = NULL;
                              *last_stream_rec_next_ptr = stream_rec;
                              last_stream_rec_next_ptr = &(stream_rec->next);
                           }
                        }
                     }
                  }
                  break;
               }

               case SI_STREAM_TYPE_AC3:
               {
                  if (pmt_entry->num_iso_lang_entries == 0)
                  {
                     stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                     if (stream_rec != NULL)
                     {
                        memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                        /* Now copying all tags as there may be multiple */
                        CopyComponentTagArray(stream_rec, pmt_entry);
                        stream_rec->pid = pmt_entry->pid;
                        stream_rec->type = ADB_AC3_AUDIO_STREAM;
                        stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;

                        stream_rec->data.audio.lang_code = ACFG_UNDEFINED_DB_LANG_CODE;
                        stream_rec->data.audio.type = ADB_AUDIO_TYPE_UNDEFINED;

                        stream_rec->next = NULL;
                        *last_stream_rec_next_ptr = stream_rec;
                        last_stream_rec_next_ptr = &(stream_rec->next);

                        #ifdef DEBUG_SI_PMT_AUDIO
                        AP_SI_PRINT(("   AC3 audio stream: pid 0x%04x, type %u, mode %u",
                                     pmt_entry->pid,
                                     stream_rec->data.audio.type,
                                     stream_rec->data.audio.mode));
                        for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                        {
                           AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                        }
                        #endif
                     }
                  }
                  else
                  {
                     /* make new stream entry and add to end of list. Add an entry for each
                      * iso lang desc with same tag and pid. */
                     for (i = 0; i < pmt_entry->num_iso_lang_entries; i++)
                     {
                        stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                        if (stream_rec != NULL)
                        {
                           memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                           /* Now copying all tags as there may be multiple */
                           CopyComponentTagArray(stream_rec, pmt_entry);
                           stream_rec->pid = pmt_entry->pid;
                           stream_rec->type = ADB_AC3_AUDIO_STREAM;
                           stream_rec->data.audio.mode = AUDIO_MULTICHANNEL;

                           stream_rec->data.audio.lang_code =
                              pmt_entry->iso_lang_desc_array[i].lang_code;
                           stream_rec->data.audio.type =
                              pmt_entry->iso_lang_desc_array[i].audio_type;

                           stream_rec->next = NULL;
                           *last_stream_rec_next_ptr = stream_rec;
                           last_stream_rec_next_ptr = &(stream_rec->next);

                           #ifdef DEBUG_SI_PMT_AUDIO
                           AP_SI_PRINT(("   AC3 audio stream: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                        pmt_entry->pid,
                                        ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                        ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                        (stream_rec->data.audio.lang_code & 0xff),
                                        stream_rec->data.audio.type,
                                        stream_rec->data.audio.mode));
                           for (tag_count = 0; tag_count < pmt_entry->num_tag_entries; tag_count++)
                           {
                              AP_SI_PRINT(("    tag: %x", pmt_entry->tag_array_ptr[tag_count]));
                           }
                           #endif
                        }
                     }
                  }
                  break;
               }

               case SI_STREAM_TYPE_PRIVATE:
               {
                  if (pmt_entry->has_rct)
                  {
                     /* Save the PID to be used by this service for collecting the RCT */
                     s_ptr->rct_pid = pmt_entry->pid;

                     /* No need to create a record for this stream */
                     break;
                  }

#ifdef INTEGRATE_HBBTV
                  #ifdef DEBUG_SI_PMT
                  AP_SI_PRINT(("PMT: has AIT: %s, num application signalling descriptors %d",
                     pmt_entry->has_ait ? "TRUE" : "FALSE", pmt_entry->num_app_sig_entries));
                  #endif
                  if (pmt_entry->has_ait)
                  {
                     /* "If more than one stream is signalled in the PMT for a service with an
                        application_signalling_descriptor, then the
                        application_signalling_descriptor for the stream containing the AIT for
                        the HbbTV application shall include the HbbTV application_type (0x0010)."
                        TS102796 section 7.2.3.1 */
                     if (s_ptr->ait_pid == 0)
                     {
                        /* Always take the first one, it will be overwritten by the next ones if they are valid */
                        s_ptr->ait_pid = pmt_entry->pid;
                     }
                     else if (pmt_entry->num_app_sig_entries > 0)
                     {
                        for (i = 0; i < pmt_entry->num_app_sig_entries; i++)
                        {
                           #ifdef DEBUG_SI_PMT
                           AP_SI_PRINT(("PMT: app_type=0x%x", pmt_entry->app_sig_desc_array[i].app_type));
                           #endif
                           if (pmt_entry->app_sig_desc_array[i].app_type == 0x0010)
                           {
                              s_ptr->ait_pid = pmt_entry->pid;
                           }
                        }
                     }
                  }
                  #ifdef DEBUG_SI_PMT
                  else
                  {
                     AP_SI_PRINT(("PMT: num application signalling descriptors %d but app_sig_desc_array is NULL ",
                        pmt_entry->num_app_sig_entries));
                  }
                  #endif
#endif

                  /* Allow to fall through if the stream isn't marked as defining an RCT PID */
               }

               default:
               {
                  // make new stream entry and add to end of list
                  stream_rec = STB_AppGetMemory(sizeof(ADB_STREAM_REC));
                  if (stream_rec != NULL)
                  {
                     memset(stream_rec, 0, sizeof(ADB_STREAM_REC));
                     /* Now copying all tags as there may be multiple */
                     CopyComponentTagArray(stream_rec, pmt_entry);
                     stream_rec->pid = pmt_entry->pid;
                     stream_rec->type = ADB_DATA_STREAM;

                     stream_rec->next = NULL;
                     *last_stream_rec_next_ptr = stream_rec;
                     last_stream_rec_next_ptr = &(stream_rec->next);
                  }
                  break;
               }
            }

            if (stream_rec != NULL)
            {
               if (pmt_entry->ca_desc_array == NULL)
               {
                  stream_rec->has_ca_descriptor = FALSE;
                  stream_rec->ecm_pid = ADB_INVALID_DVB_ID;
               }
               else
               {
                  stream_rec->has_ca_descriptor = TRUE;
                  //FIXME
                  stream_rec->ecm_pid = pmt_entry->ca_desc_array[0].ca_pid;
               }
            }

            pmt_entry = pmt_entry->next;
         }

         /* Set the audio channel setting for each audio stream */
         stream_rec = s_ptr->stream_list;
         while (stream_rec != NULL)
         {
            if (((stream_rec->type == ADB_AUDIO_STREAM) ||
                 (stream_rec->type == ADB_AAC_AUDIO_STREAM) ||
                 (stream_rec->type == ADB_HEAAC_AUDIO_STREAM) ||
                 (stream_rec->type == ADB_AC3_AUDIO_STREAM) ||
                 (stream_rec->type == ADB_EAC3_AUDIO_STREAM)) &&
                (stream_rec->pid != 0) && (stream_rec->data.audio.mode == AUDIO_UNDEF))
            {
               if (stream_rec->data.audio.type == ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED)
               {
                  stream_rec->data.audio.mode = AUDIO_MONO;
               }
               else
               {
                  /* Search for another audio stream with the same PID and audio type */
                  srec = stream_rec->next;
                  match_found = FALSE;

                  while ((srec != NULL) && !match_found)
                  {
                     if ((srec->type == stream_rec->type) && (srec->pid == stream_rec->pid) &&
                         (srec->data.audio.type != ADB_AUDIO_TYPE_FOR_VISUALLY_IMPAIRED))
                     {
                        /* Found another audio stream on the same PID,
                         * check whether the language is the same */
                        match_found = TRUE;

                        if (stream_rec->data.audio.lang_code == srec->data.audio.lang_code)
                        {
                           if (stream_rec->data.audio.type == srec->data.audio.type)
                           {
                              /* Left and right channels have the same language & type
                               * so mark them as stereo */
                              stream_rec->data.audio.mode = AUDIO_STEREO;
                              srec->data.audio.mode = AUDIO_STEREO;

                              #ifdef DEBUG_SI_PMT_AUDIO
                              AP_SI_PRINT(("  Stereo left/right audio stream: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                           stream_rec->pid,
                                           ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                           ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                           (stream_rec->data.audio.lang_code & 0xff),
                                           stream_rec->data.audio.type,
                                           stream_rec->data.audio.mode));
                              #endif
                           }
                           else
                           {
                              /* Left and right channels are different */
                              stream_rec->data.audio.mode = AUDIO_LEFT;
                              srec->data.audio.mode = AUDIO_RIGHT;

                              #ifdef DEBUG_SI_PMT_AUDIO
                              AP_SI_PRINT(("  Left/right audio stream with diff types: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                           stream_rec->pid,
                                           ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                           ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                           (stream_rec->data.audio.lang_code & 0xff),
                                           stream_rec->data.audio.type,
                                           stream_rec->data.audio.mode));
                              #endif
                           }
                        }
                        else
                        {
                           /* Left and right channels are different */
                           stream_rec->data.audio.mode = AUDIO_LEFT;
                           srec->data.audio.mode = AUDIO_RIGHT;

                           #ifdef DEBUG_SI_PMT_AUDIO
                           AP_SI_PRINT(("  Left/right audio stream with diff langs: pid 0x%04x, lang %c%c%c, type %u, mode %u",
                                        stream_rec->pid,
                                        ((stream_rec->data.audio.lang_code >> 16) & 0xff),
                                        ((stream_rec->data.audio.lang_code >> 8) & 0xff),
                                        (stream_rec->data.audio.lang_code & 0xff),
                                        stream_rec->data.audio.type,
                                        stream_rec->data.audio.mode));
                           #endif
                        }
                     }

                     srec = srec->next;
                  }

                  if (!match_found)
                  {
                     /* Single language code means the audio comes from the left channel but is stereo */
                     stream_rec->data.audio.mode = AUDIO_LEFT;
                  }
               }
            }

            stream_rec = stream_rec->next;
         }

         /* Now we have all the stream details, we can see if there are any changes from
          * the service's previous set of streams */
         if ((HaveStreamsChanged(old_stream_list, s_ptr->stream_list) == TRUE) && (mode != DB_ACCESS_SEARCH))
         {
            #ifdef DEBUG_SI_PMT
            AP_SI_PRINT(("PMT: streams changed for service %p (id=0x%04x)", s_ptr, s_ptr->serv_id));
            #endif
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_STREAMS_CHANGED,
               &s_ptr, sizeof(s_ptr));
         }

         /* The old list of streams can now be deleted */
         DBDEF_DeleteStreamList(old_stream_list);

         /* Now we have all the stream details we can get the correct pids
          * and subtitle details */
         s_ptr->audio_pid = DBDEF_GetReqdAudioPid(s_ptr, &audio_mode, &s_ptr->audio_type);
         stream_rec = DBDEF_FindStreamRecById(s_ptr, s_ptr->audio_pid);
         if (NULL != stream_rec) {
            s_ptr->audio_ecm_pid = stream_rec->ecm_pid;
         }
         
         DBDEF_GetReqdSubtitleParams(s_ptr, &(s_ptr->subtitle_pid), &(s_ptr->subtitle_cpage),
            &(s_ptr->subtitle_apage));

         DBDEF_GetReqdTtextPid(s_ptr, TRUE, &(s_ptr->ttext_pid), &(s_ptr->ttext_mag),
            &(s_ptr->ttext_page));

         s_ptr->video_pid = DBDEF_GetReqdVideoPid(s_ptr, &s_ptr->video_type);
         stream_rec = DBDEF_FindStreamRecById(s_ptr, s_ptr->video_pid);
         if (NULL != stream_rec) {
            s_ptr->video_ecm_pid = stream_rec->ecm_pid;
         }

         /* If any of the PIDs have changed or the service was previously not running
          * then send a message informing about the PID update */
         if (((video_pid != s_ptr->video_pid) || prev_not_running) && (mode != DB_ACCESS_SEARCH))
         {
            #ifdef DEBUG_SI_PMT
            AP_SI_PRINT(("PMT: report video PID update for service %p (id=0x%04x)", s_ptr, s_ptr->serv_id));
            #endif
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_VIDEO_PID_UPDATE,
               &s_ptr, sizeof(s_ptr));
         }

         if ((mode != DB_ACCESS_SEARCH) &&
             ((audio_pid != s_ptr->audio_pid) || prev_not_running ||
               (adesc_pid != DBDEF_GetReqdADPid(s_ptr, &audio_mode, &adesc_type, &match_found))))
         {
            #ifdef DEBUG_SI_PMT_AUDIO
            AP_SI_PRINT(("PMT: report audio PID update for service %p (id=0x%04x) pid=%x", s_ptr, s_ptr->serv_id,s_ptr->audio_pid));
            #endif
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_AUDIO_PID_UPDATE,
               &s_ptr, sizeof(s_ptr));
         }

         /* check whether the video type changed, if so, notify the application */
         if ((video_type != s_ptr->video_type) && (mode != DB_ACCESS_SEARCH))
         {
            #ifdef DEBUG_SI_PMT
            AP_SI_PRINT(("PMT: report video codec changed for service %p (id=0x%04x)", s_ptr, s_ptr->serv_id));
            #endif
            /* send an video codec changed event*/
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_VIDEO_CODEC_CHANGED,
               &s_ptr, sizeof(s_ptr));
         }

         /* check whether the audio type changed, if so, notify the application */
         if ((audio_type != s_ptr->audio_type) && (mode != DB_ACCESS_SEARCH))
         {
            #ifdef DEBUG_SI_PMT
            AP_SI_PRINT(("PMT: report audio codec changed for service %p (id=0x%04x)", s_ptr, s_ptr->serv_id));
            #endif
            /* send an video codec changed event*/
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_AUDIO_CODEC_CHANGED,
               &s_ptr, sizeof(s_ptr));
         }

         if (s_ptr == current_service_rec[path])
         {
            if (ACTL_AreSubtitlesStarted())
            {
               /* Check if parameters have changed */
               if (((subt_pid != s_ptr->subtitle_pid) ||
                    (subt_cpage != s_ptr->subtitle_cpage) ||
                    (subt_apage != s_ptr->subtitle_apage) ||
                    (ttxt_pid != s_ptr->ttext_pid) ||
                    (ttxt_mag != s_ptr->ttext_mag) ||
                    (ttxt_page != s_ptr->ttext_page)) &&
                   (mode != DB_ACCESS_SEARCH))
               {
                  #ifdef DEBUG_SI_PMT
                  AP_SI_PRINT(("PMT: report subtitle update for service %p (id=0x%04x)", s_ptr, s_ptr->serv_id));
                  #endif
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_SUBTITLE_UPDATE,
                     &s_ptr, sizeof(s_ptr));
               }
            }
         }

         if (update_pmt_func != NULL)
         {
            /* Call the function that's been registered to handle additional PMT processing */
            (*update_pmt_func)(path, pmt_table, table_rec, s_ptr);
         }

         if (db_services)
         {
            /* See if there's a duplicate service that this PMT also applies to */
            s_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID,
                  current_transport_rec[path]->orig_net_id, current_transport_rec[path]->tran_id,
                  s_ptr->serv_id);
         }
         else
         {
            s_ptr = NULL;
         }
      }

      // release pmt table
      STB_SIReleasePmtTable(pmt_table);
   }

   FUNCTION_FINISH(InternalProcessPmtTable);

   return(service_updated);
}

/*!**************************************************************************
 * @brief   Allocates and copies the tag array from the PMT stream to the service stream
 * @param   stream_rec - service stream the tag array is to be copied into
 * @param   pmt_entry - PMT stream the tag array is to be copied from
 ****************************************************************************/
static void CopyComponentTagArray(ADB_STREAM_REC *stream_rec, SI_PMT_STREAM_ENTRY *pmt_entry)
{
   U8BIT i;

   FUNCTION_START(CopyComponentTagArray);

   if (pmt_entry->tag_array_ptr != NULL)
   {
      stream_rec->tag_array = STB_AppGetMemory(pmt_entry->num_tag_entries);
      if (stream_rec->tag_array != NULL)
      {
         for (i = 0; i < pmt_entry->num_tag_entries; i++)
         {
            stream_rec->tag_array[i] = pmt_entry->tag_array_ptr[i];
         }
         stream_rec->num_tags = pmt_entry->num_tag_entries;
      }
   }

   FUNCTION_FINISH(CopyComponentTagArray);
}

/*!**************************************************************************
 * @brief   Compares the two stream lists to determine if they contain the same PIDs, ignoring differences in order
 * @param   slist1 - stream list 1
 * @param   slist2 - stream list 2
 * @return  TRUE if the PIDs are different or the number of streams is different, FALSE otherwise
 ****************************************************************************/
static BOOLEAN HaveStreamsChanged(ADB_STREAM_REC *slist1, ADB_STREAM_REC *slist2)
{
   BOOLEAN changed;
   ADB_STREAM_REC *srec1;
   ADB_STREAM_REC *srec2;

   FUNCTION_START(HaveStreamsChanged);

   changed = FALSE;

   if (DBDEF_NumStreamsInList(slist1) != DBDEF_NumStreamsInList(slist2))
   {
      changed = TRUE;
   }
   else
   {
      /* Need to check whether streams in one list are the same as streams in the other */
      srec1 = slist1;
      while ((srec1 != NULL) && !changed)
      {
         srec2 = slist2;
         while ((srec2 != NULL) && (srec2->pid != srec1->pid))
         {
            srec2 = srec2->next;
         }

         if (srec2 == NULL)
         {
            /* PID wasn't found so streams have changed */
            changed = TRUE;
         }

         srec1 = srec1->next;
      }
   }

   FUNCTION_FINISH(HaveStreamsChanged);

   return(changed);
}

/**
 * @brief   Processes the NIT table record to extract data for the database
 * @param   table_rec - pointer to the nit table record
 * @param   mode      - if DB_ACCESS_SEARCH then contents of the NIT is to be stored in
 *                            the database, adding or updating existing records. If
 *                            DB_ACCESS_UPDATE then only the ssu availability is checked
 * @param   report_nit - TRUE if the raw network record data can be reported to the midware
 *                             so it can be passed on to other interested modules
 * @param   transport_changed - TRUE if the current transport has changed, FALSE otherwise
 * @return  TRUE if the SDT collection needs to be restarted, FALSE otherwise
 */
static BOOLEAN ProcessNitTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode, BOOLEAN report_nit,
   BOOLEAN transport_changed)
{
   SI_NIT_TABLE *nit_table;
   SI_NIT_TRANSPORT_ENTRY *nit_entry;
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *s2_ptr;
   U8BIT i, j;
   BOOLEAN changed;
   U16BIT reqd_lcn;
   U8BIT path;
   ADB_TRANSPORT_REC *t_ptr;
   SI_MULTILING_NET_NAME_DESC *mlnn_desc_ptr;
   SI_STRING_DESC *str_desc;
   U8BIT lang_id;
   ADB_NETWORK_REC *n_ptr;
   SI_LINKAGE_DESC_ENTRY *linkage_desc_ptr;
   void *sat_ptr;
   BOOLEAN restart_sdts;
   SI_NIT_FREQUENCY_LIST_DESC *freq_list_ptr;

   FUNCTION_START(ProcessNitTable);

   restart_sdts = FALSE;

   path = table_rec->path;

   nit_table = STB_SIParseNitTable(table_rec);
   if (nit_table != NULL)
   {
      #ifdef DEBUG_SI_NIT
      AP_SI_PRINT(("NIT Table: net_id 0x%04x v%d",
                   nit_table->net_id, nit_table->version));
      #endif

      sat_ptr = ACTL_GetCurrentSatellite(path);

      // get access to the database (may suspend)
      DBDEF_RequestAccess();

      // get network record
      n_ptr = DBDEF_FindNetworkRec(nit_table->net_id, sat_ptr);
      if (n_ptr != NULL)
      {
         if (n_ptr->profile_type == ADB_PROFILE_TYPE_CIPLUS)
         {
            /* Don't change CI+ profiles */
            while (n_ptr != NULL)
            {
               n_ptr = DBDEF_GetNextNetworkRec(n_ptr);
               if ((n_ptr != NULL) && (n_ptr->net_id == nit_table->net_id) &&
                  (n_ptr->satellite == sat_ptr) && (n_ptr->profile_type != ADB_PROFILE_TYPE_CIPLUS))
               {
                  /* Found a matching non CI+ network, use it */
                  break;
               }
            }
         }

         if (n_ptr != NULL)
         {
            if (n_ptr->nit_version != nit_table->version)
            {
               // save version in network record
               n_ptr->nit_version = nit_table->version;

               if (mode != DB_ACCESS_SEARCH)
               {
                  n_ptr->nit_version_changed = TRUE;

                  DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_VERSION, n_ptr->nit_version);
                  DBA_SaveRecord(n_ptr->dba_rec);
               }
            }
            else
            {
               n_ptr->nit_version_changed = FALSE;
            }
         }
      }

      if (mode == DB_ACCESS_SEARCH)
      {
         if (n_ptr == NULL)
         {
            n_ptr = DBDEF_AddNetworkRec(nit_table->net_id, sat_ptr);
         }

         if (n_ptr != NULL)
         {
            //save this network as the tuned network
            DBDEF_SetTunedNetwork(path, n_ptr);
            current_network_rec[path] = n_ptr;

            if (current_transport_rec[path]->network != n_ptr)
            {
               // link the current transport to the network
               current_transport_rec[path]->network = n_ptr;
               DBA_SetRecordParent(current_transport_rec[path]->dba_rec, n_ptr->dba_rec);
               DBA_SaveRecord(current_transport_rec[path]->dba_rec);
            }

            /* Save version in network record */
            n_ptr->nit_version = nit_table->version;
            DBA_SetFieldValue(n_ptr->dba_rec, DBA_FIELD_VERSION, n_ptr->nit_version);

            /* Now doing a search, so version no longer changed */
            n_ptr->nit_version_changed = FALSE;

            if (n_ptr->satellite != sat_ptr)
            {
               if ((n_ptr->satellite = sat_ptr) != NULL)
               {
                  DBA_SetRecordParent(n_ptr->dba_rec, n_ptr->satellite->dba_rec);
               }
               else
               {
                  DBA_SetRecordParent(n_ptr->dba_rec, NULL);
               }
            }

            DBA_SaveRecord(n_ptr->dba_rec);

            /* The target region data will only be kept until after the search has completed
             * and all of the info gathered may be needed so just keep the lot and ensure it
             * isn't deleted when the table gets deleted. */
            n_ptr->target_region_name_list = nit_table->target_region_name_list;
            nit_table->target_region_name_list = NULL;

            n_ptr->target_region_list = nit_table->target_region_list;
            nit_table->target_region_list = NULL;
         }

         /* If the NIT contains linkage descriptors types 0xB or 0xC, then this transport is not DVB C/T/S */
         linkage_desc_ptr = nit_table->linkage_desc_list;
         s_ptr = NULL;
         while (linkage_desc_ptr != NULL)
         {
            if ((linkage_desc_ptr->link_type == DVB_DVBH1_LINKAGE_TYPE) ||
                (linkage_desc_ptr->link_type == DVB_DVBH2_LINKAGE_TYPE))
            {
#ifdef DEBUG_SI_NIT
               AP_SI_PRINT(("  linkage_type=0x%x (DVB-H) onid=0x%x tran_id=0x%x", linkage_desc_ptr->link_type,
                            linkage_desc_ptr->orig_net_id, linkage_desc_ptr->tran_id));
#endif
               /* Delete all the services on this transport that might have been added */
               s_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID,
                     linkage_desc_ptr->orig_net_id, linkage_desc_ptr->tran_id, ADB_INVALID_DVB_ID);
               while (s_ptr != NULL)
               {
                  s2_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID,
                        linkage_desc_ptr->orig_net_id, linkage_desc_ptr->tran_id, ADB_INVALID_DVB_ID);
#ifdef DEBUG_SI_NIT
                  AP_SI_PRINT(("  deleting non DVB-T/C/S sid=0x%x", s_ptr->serv_id));
#endif
                  /* Ignore services that are part of a CI+ profile */
                  if ((s_ptr->transport == NULL) ||
                     (s_ptr->transport->network == NULL) ||
                     (s_ptr->transport->network->profile_type != ADB_PROFILE_TYPE_CIPLUS))
                  {
                     DBDEF_DeleteServiceRec(s_ptr);
                  }
                  s_ptr = s2_ptr;
               }
            }
            linkage_desc_ptr = linkage_desc_ptr->next;
         }

         // extract logical channel numbers for the current transport only
         nit_entry = nit_table->transport_list;
         while (nit_entry != NULL)
         {
            /* Ignore transports that are part of a CI+ profile */
            if ((nit_entry->tran_id == current_transport_rec[path]->tran_id) &&
               ((current_transport_rec[path]->network == NULL) ||
               current_transport_rec[path]->network->profile_type != ADB_PROFILE_TYPE_CIPLUS))
            {
               if (current_transport_rec[path]->orig_net_id != nit_entry->orig_net_id)
               {
                  current_transport_rec[path]->orig_net_id = nit_entry->orig_net_id;
                  DBA_SetFieldValue(current_transport_rec[path]->dba_rec, DBA_FIELD_ORIG_NET_ID,
                     current_transport_rec[path]->orig_net_id);
               }

               /* Check for nordig version two LCN descripter, if this doesn't exsist then use normal lcn descripter*/
               if (nit_entry->nordig_lcn_desc_array != NULL)
               {
                  /* Mark all services as unavailable */
                  s_ptr = DBDEF_GetNextServiceOnTransport(NULL, current_transport_rec[path]);
                  while (s_ptr != NULL)
                  {
                     if (s_ptr->serv_lcn != 0)
                     {
                        s_ptr->unavailable = TRUE;
                     }
                     s_ptr = DBDEF_GetNextServiceOnTransport(s_ptr, current_transport_rec[path]);
                  }

                  for (j = 0; j < nit_entry->num_nordig_lcn_entries; j++)
                  {
                     for (i = 0; i < nit_entry->nordig_lcn_desc_array[j].num_services; i++)
                     {
                        s_ptr = DBDEF_FindServiceRec(nit_entry->nordig_lcn_desc_array[j].serv_array[i].serv_id,
                           current_transport_rec[path]);

                        if (s_ptr != NULL)
                        {
                           reqd_lcn = nit_entry->nordig_lcn_desc_array[j].serv_array[i].serv_lcn;

                           /* Set service hidden attribute for Nordig LCN descriptor version 2, a hidden service still
                              need to be selectable, so it's aways TRUE in this case, for other countries such as UK,
                              service attribute may be overwritten by service attribute descriptor */
                           s_ptr->hidden = !nit_entry->nordig_lcn_desc_array[j].serv_array[i].visible;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                           s_ptr->selectable = TRUE;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                           if (reqd_lcn != s_ptr->serv_lcn)
                           {
                              if (s_ptr->serv_lcn == 0)
                              {
                                 s_ptr->unavailable = FALSE;

                                 /* First time the LCN for this service has been set */
                                 s_ptr->serv_lcn = reqd_lcn;
                                 DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
#ifdef DEBUG_SI_NIT
                                 AP_SI_PRINT(("   LCN for 0x%04x, %u, hidden=%u", s_ptr->serv_id,
                                    reqd_lcn, s_ptr->hidden));
#endif
                              }
                              else
                              {
                                 /* The same service is being allocated a different LCN.
                                  * Check whether this LCN is already assigned to another service */
                                 s2_ptr = DBDEF_FindServiceRecByLcn(reqd_lcn, current_transport_rec[path], FALSE);
                                 if (s2_ptr == NULL)
                                 {
                                    /* Add a new duplicate service with this LCN */
#ifdef DEBUG_SI_NIT
                                    AP_SI_PRINT(("   copy service sid 0x%04x, lcn %u, new lcn %u",
                                                 s_ptr->serv_id, s_ptr->serv_lcn, reqd_lcn));
#endif
                                    s2_ptr = DBDEF_CopyServiceRec(s_ptr);
                                    if (s2_ptr != NULL)
                                    {
                                       s2_ptr->unavailable = FALSE;

                                       s2_ptr->new_service = TRUE;
                                       s2_ptr->serv_lcn = reqd_lcn;
                                       DBA_SetFieldValue(s2_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN,
                                          s2_ptr->serv_lcn);
                                       DBA_SaveRecord(s2_ptr->dba_rec);
                                    }
                                 }
                              }
                           }
                           else
                           {
                              s_ptr->unavailable = FALSE;
                           }
                           DBA_SaveRecord(s_ptr->dba_rec);
                        }
                     }
                  }
               }
               else
               {
                  for (i = 0; i < nit_entry->num_lcn_entries; i++)
                  {
                     s_ptr = DBDEF_FindServiceRec(nit_entry->lcn_desc_array[i].serv_id,
                        current_transport_rec[path]);

                     if (s_ptr != NULL)
                     {
                        reqd_lcn = nit_entry->lcn_desc_array[i].serv_lcn;

                        /* Set service hidden attribute for Nordig LCN descriptor version 1, a hidden service still
                           need to be selectable, so it's aways TRUE in this case */
                        s_ptr->hidden = !nit_entry->lcn_desc_array[i].visible;
                        DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                        s_ptr->selectable = TRUE;
                        DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                        if (reqd_lcn != s_ptr->serv_lcn)
                        {
                           s_ptr->serv_lcn = reqd_lcn;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
#ifdef DEBUG_SI_NIT
                           AP_SI_PRINT(("   LCN for 0x%04x, %u, hidden=%u", s_ptr->serv_id, reqd_lcn,
                              s_ptr->hidden));
#endif
                        }
                        DBA_SaveRecord(s_ptr->dba_rec);
                     }
                  }
               }

               /* Extract the service attributes - hidden/selectable flags */
               for (i = 0; i < nit_entry->num_serv_attr_entries; i++)
               {
                  s_ptr = DBDEF_FindServiceRec(nit_entry->serv_attr_array[i].serv_id,
                     current_transport_rec[path]);

                  if (s_ptr != NULL)
                  {
                     s_ptr->hidden = !nit_entry->serv_attr_array[i].service_visible;
                     DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                     s_ptr->selectable = nit_entry->serv_attr_array[i].service_selectable;
                     DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                     DBA_SaveRecord(s_ptr->dba_rec);
                  }
               }

               /* Now take into account any HD LCN assignments */
               for (i = 0; i < nit_entry->num_hd_lcn_entries; i++)
               {
                  s_ptr = DBDEF_FindServiceRec(nit_entry->hd_lcn_desc_array[i].serv_id,
                     current_transport_rec[path]);

                  if ((s_ptr != NULL) && (s_ptr->old_allocated_lcn == 0))
                  {
                     /* The HD LCN descriptor will be used at the end of the scan
                      * after the LCN conflicts and duplicated services have been
                      * dealt with. hd_lcn_desc will be freed by
                      * ADB_ReleaseDatabaseSearchData() */
                     s_ptr->hd_lcn_desc = STB_GetMemory(sizeof(SI_LCN_DESC));
                     memcpy(s_ptr->hd_lcn_desc, &nit_entry->hd_lcn_desc_array[i],
                        sizeof(SI_LCN_DESC));
                  }
               }
            }

            t_ptr = DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                  nit_entry->orig_net_id, nit_entry->tran_id);
            if ((t_ptr != NULL) && (t_ptr->network != NULL) &&
               (t_ptr->network->profile_type == ADB_PROFILE_TYPE_CIPLUS))
            {
               /* Ignore transports that are part of a CI+ profile */
               t_ptr = NULL;
            }

            if (nit_entry->del_sys_desc != NULL)
            {
               if (t_ptr == NULL)
               {
                  /* Add a new transport based on the system delivery descriptor */
                  switch (nit_entry->del_sys_desc_type)
                  {
                     case SI_DEL_SYS_DESC_TYPE_TERR:
#ifndef DVBC_IPQAM
                        if (current_transport_rec[path]->sig_type == SIGNAL_COFDM)
                        {
                           if (nit_entry->del_sys_desc->terr.is_t2)
                           {
                              if ((nit_entry->del_sys_desc->terr.u.t2.num_cells > 0) &&
                                  (nit_entry->del_sys_desc->terr.u.t2.cell[0].num_freqs > 0))
                              {
                                 if ((t_ptr = DBDEF_FindTerrestrialTransportRec(
                                            nit_entry->del_sys_desc->terr.u.t2.cell[0].freq_hz[0],
                                            nit_entry->del_sys_desc->terr.u.t2.plp_id)) == NULL)
                                 {
                                    #ifdef DEBUG_SI_NIT
                                    AP_SI_PRINT(("   Adding terr transport for %lu Hz, PLP %u",
                                                 nit_entry->del_sys_desc->terr.u.t2.cell[0].freq_hz[0],
                                                 nit_entry->del_sys_desc->terr.u.t2.plp_id));
                                    #endif
                                    t_ptr = DBDEF_AddTerrestrialTransportRec(
                                          nit_entry->del_sys_desc->terr.u.t2.cell[0].freq_hz[0],
                                          nit_entry->del_sys_desc->terr.u.t2.plp_id, n_ptr);
                                 }
                              }
                           }
                           else
                           {
                              if ((t_ptr = DBDEF_FindTerrestrialTransportRec(nit_entry->del_sys_desc->terr.u.t1.freq_hz, 0)) == NULL)
                              {
                                 #ifdef DEBUG_SI_NIT
                                 AP_SI_PRINT(("   Adding terr transport for %lu Hz",
                                              nit_entry->del_sys_desc->terr.u.t1.freq_hz));
                                 #endif
                                 t_ptr = DBDEF_AddTerrestrialTransportRec(nit_entry->del_sys_desc->terr.u.t1.freq_hz,
                                    0, n_ptr);
                              }
                           }
                        }
#endif
                        break;
                     case SI_DEL_SYS_DESC_TYPE_CABLE:
                        if (current_transport_rec[path]->sig_type == SIGNAL_QAM)
                        {
                           if ((t_ptr = DBDEF_FindCableTransportRec(nit_entry->del_sys_desc->cable.freq_hz,
                                      SYMBOL_RATE_AUTO)) == NULL)
                           {
                              #ifdef DEBUG_SI_NIT
                              AP_SI_PRINT(("   Adding cable transport for %lu Hz, symbol rate %u Kbits",
                                           nit_entry->del_sys_desc->cable.freq_hz,
                                           nit_entry->del_sys_desc->cable.symbol_rate));
                              #endif
                              t_ptr = DBDEF_AddCableTransportRec(nit_entry->del_sys_desc->cable.freq_hz,
                                    nit_entry->del_sys_desc->cable.symbol_rate, n_ptr);
                           }
                           else
                           {
                              /* Symbol rate read from a tuner isn't very accurate, so use the setting
                               * from the delivery descriptor */
                              t_ptr->u.cab.symbol_rate = (U16BIT)nit_entry->del_sys_desc->cable.symbol_rate;
                              DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, t_ptr->u.cab.symbol_rate);
                           }

                           if (NULL != t_ptr)
                           {
                              // EN 300 468
                              // Table 34: Modulation scheme for cable
                              E_STB_DP_CMODE cmode[] = {
                                 MODE_QAM_AUTO,
                                 MODE_QAM_16,
                                 MODE_QAM_32,
                                 MODE_QAM_64,
                                 MODE_QAM_128,
                                 MODE_QAM_256
                              };
                              DBDEF_SetCableTransportMode(t_ptr, cmode[nit_entry->del_sys_desc->cable.modulation]);
                           }
                        }
                        break;
                     case SI_DEL_SYS_DESC_TYPE_SAT:
                        if (current_transport_rec[path]->sig_type == SIGNAL_QPSK)
                        {
                           if ((t_ptr = DBDEF_FindSatTransportRec(nit_entry->del_sys_desc->sat.freq_hz,
                                      nit_entry->del_sys_desc->sat.sym_rate,
                                      nit_entry->del_sys_desc->sat.polarity,
                                      nit_entry->del_sys_desc->sat.dvb_s2,
                                      nit_entry->del_sys_desc->sat.modulation, sat_ptr)) == NULL)
                           {
                              #ifdef DEBUG_SI_NIT
                              AP_SI_PRINT(("   Adding sat transport for %lu Hz, symrate %u, polarity %u, DVB-S2 %u, modulation %u",
                                           nit_entry->del_sys_desc->sat.freq_hz,
                                           nit_entry->del_sys_desc->sat.sym_rate,
                                           nit_entry->del_sys_desc->sat.polarity,
                                           nit_entry->del_sys_desc->sat.dvb_s2,
                                           nit_entry->del_sys_desc->sat.modulation));
                              #endif
                              t_ptr = DBDEF_AddSatTransportRec(nit_entry->del_sys_desc->sat.freq_hz,
                                    nit_entry->del_sys_desc->sat.sym_rate,
                                    nit_entry->del_sys_desc->sat.polarity,
                                    nit_entry->del_sys_desc->sat.dvb_s2,
                                    nit_entry->del_sys_desc->sat.modulation, n_ptr);
                           }
                        }
                        break;
                     default:
                        break;
                  }
                  if (t_ptr != NULL)
                  {
                     DBDEF_SetTransportOrigNetworkId(t_ptr, nit_entry->orig_net_id);
                     DBDEF_SetTransportTransportId(t_ptr, nit_entry->tran_id);
                  }
               }
               else
               {
                  /* Set transport settings from the delivery descriptor */
                  switch (nit_entry->del_sys_desc_type)
                  {
                     case SI_DEL_SYS_DESC_TYPE_TERR:
                        break;
                     case SI_DEL_SYS_DESC_TYPE_CABLE:
                        /* Only set the symbol rate if the transport is DVB-C */
                        if (t_ptr->sig_type == SIGNAL_QAM)
                        {
                           /* Symbol rate read from a tuner isn't very accurate, so use the setting
                            * from the delivery descriptor */
                           t_ptr->u.cab.symbol_rate = (U16BIT)nit_entry->del_sys_desc->cable.symbol_rate;
                           DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, t_ptr->u.cab.symbol_rate);
                        }
                        break;
                     case SI_DEL_SYS_DESC_TYPE_SAT:
                        break;
                     default:
                        break;
                  }
               }

               if (t_ptr != NULL)
               {
                  DBDEF_SetTransportOrigNetworkId(t_ptr, nit_entry->orig_net_id);
               }
            }
            else {
               if (NULL == t_ptr) {
                  //FIXME: assume it's the current transport
                  t_ptr = current_transport_rec[path];
                  if (NULL != t_ptr) {
                     DBDEF_SetTransportOrigNetworkId(t_ptr, nit_entry->orig_net_id);
                     DBDEF_SetTransportTransportId(t_ptr, nit_entry->tran_id);
                  }
               }
            }

            if (t_ptr != NULL)
            {
               if (nit_entry->target_region_list != NULL)
               {
                  /* The target region data will only be kept until after the search has completed
                   * and all of the info gathered may be needed so just keep the lot and ensure it
                   * isn't deleted when the table gets deleted. */
                  t_ptr->u.terr.target_region_list = nit_entry->target_region_list;
                  nit_entry->target_region_list = NULL;
               }
               DBA_SaveRecord(t_ptr);
            }

            nit_entry = nit_entry->next;
         }
      }
      else if ((mode == DB_ACCESS_UPDATE) && (n_ptr != NULL) && n_ptr->nit_version_changed)
      {
         nit_entry = nit_table->transport_list;
         while (nit_entry != NULL)
         {
            t_ptr = DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                  nit_entry->orig_net_id, nit_entry->tran_id);
            if (t_ptr != NULL)
            {
               if (ASI_DatabaseUpdatesAllowed(t_ptr->sig_type) &&
                  ACFG_GetDynamicSIUpdate(t_ptr->sig_type, nit_entry->orig_net_id,
                  (ACFG_DYNAMIC_SI_UPDATE_LCNS |
                   ACFG_DYNAMIC_SI_UPDATE_SERVICE_ADD |
                   ACFG_DYNAMIC_SI_UPDATE_SERVICE_REMOVE)))
               {
                  /* Service addition or deletion means LCNs may change so LCN descriptors need to
                   * be saved so they can be applied following any changes to the service line up */
                  if (DynamicUpdateAddTransport(t_ptr,
                     nit_entry->num_lcn_entries, nit_entry->lcn_desc_array,
                     nit_entry->num_hd_lcn_entries, nit_entry->hd_lcn_desc_array,
                     nit_entry->num_nordig_lcn_entries, nit_entry->nordig_lcn_desc_array))
                  {
                     /* The structure for the Nordig LCNs isn't a simple array structure, so the
                      * LCN array from the transport will have been used directly and will be freed
                      * later, so the array is set to NULL to ensure it isn't freed with the NIT */
                     nit_entry->num_nordig_lcn_entries = 0;
                     nit_entry->nordig_lcn_desc_array = NULL;

                     /* The target region data may be needed for the allocation of LCNs so it's
                      * kept until after the dynamic update process has completed, when it will
                      * be freed */
                     t_ptr->u.terr.target_region_list = nit_entry->target_region_list;
                     nit_entry->target_region_list = NULL;

                     /* The SDT for this transport needs to be processed again, so mark it as not
                      * having been received yet so we can tell when it has been */
                     t_ptr->sdt_received = FALSE;
                     t_ptr->sdt_version = 0xFFFF;

                     restart_sdts = TRUE;
                  }
               }
            }

            nit_entry = nit_entry->next;
         }
      }

      if (n_ptr != NULL)
      {
         if (ASI_DatabaseUpdatesAllowed(STB_DPGetSignalType(path)))
         {
            if (nit_table->name_str != NULL)
            {
               DBDEF_SetNetworkName(n_ptr, nit_table->name_str->str_ptr);
            }
            else
            {
               DBDEF_SetNetworkName(n_ptr, NULL);
            }

            // now multilingual network names (not stored in nvm)
            // free old entries
            for (i = 0; i < ACFG_NUM_DB_LANGUAGES; i++)
            {
               if (n_ptr->name_array[i] != NULL)
               {
                  STB_AppFreeMemory(n_ptr->name_array[i]);
                  n_ptr->name_array[i] = NULL;
               }
            }

            // populate new entries
            mlnn_desc_ptr = nit_table->multiling_net_name_desc_array;
            for (i = 0; i < nit_table->num_multiling_net_names; i++, mlnn_desc_ptr++)
            {
               if (mlnn_desc_ptr->name_str != NULL)
               {
                  lang_id = ACFG_ConvertLangCodeToId(mlnn_desc_ptr->lang_code);
                  if (lang_id != ACFG_INVALID_DB_LANG)
                  {
                     if (n_ptr->name_array[lang_id] == NULL)
                     {
                        str_desc = mlnn_desc_ptr->name_str;
                        n_ptr->name_array[lang_id] = DBDEF_MakeString(mlnn_desc_ptr->lang_code,
                              str_desc->str_ptr, str_desc->nbytes);
                     }
                  }
               }
            }
         }

         if (nit_table->def_authority != NULL)
         {
            changed = TRUE;

            if (n_ptr->def_authority != NULL)
            {
               /* Check whether the default authority has changed */
               if (nit_table->def_authority->nbytes == n_ptr->def_authority->nbytes)
               {
                  if (memcmp(n_ptr->def_authority->str_ptr, nit_table->def_authority->str_ptr,
                         n_ptr->def_authority->nbytes) == 0)
                  {
                     /* The names are the same */
                     changed = FALSE;
                  }
               }
            }

            if (changed)
            {
               /* Free the old authority string */
               if (n_ptr->def_authority != NULL)
               {
                  STB_AppFreeMemory(n_ptr->def_authority);
                  n_ptr->def_authority = NULL;
               }

               /* Add the new authority string */
               n_ptr->def_authority = DBDEF_MakeString(0, nit_table->def_authority->str_ptr,
                     nit_table->def_authority->nbytes);
#ifdef DEBUG_SI_NIT
               AP_SI_PRINT(("   Network \"%s\" has default authority \"%s\"", n_ptr->name_str->str_ptr,
                            n_ptr->def_authority->str_ptr));
#endif
            }
         }

         if (nit_table->fta_content_desc != NULL)
         {
            n_ptr->has_fta_desc = TRUE;
            n_ptr->do_not_scramble = nit_table->fta_content_desc->do_not_scramble;
         }
      }

      nit_entry = nit_table->transport_list;
      while (nit_entry != NULL)
      {
         t_ptr = DBDEF_FindTransportRecByIds(NULL, nit_table->net_id, nit_entry->orig_net_id,
               nit_entry->tran_id);
         if ((t_ptr != NULL) && (t_ptr->network != NULL) &&
            (t_ptr->network->profile_type == ADB_PROFILE_TYPE_CIPLUS))
         {
            /* Ignore transports that are part of a CI+ profile */
            t_ptr = NULL;
         }

         if (t_ptr != NULL)
         {
            if (nit_entry->fta_content_desc != NULL)
            {
               t_ptr->has_fta_desc = TRUE;
               t_ptr->do_not_scramble = nit_entry->fta_content_desc->do_not_scramble;
            }

            if (nit_entry->def_authority != NULL)
            {
               changed = TRUE;

               if (t_ptr->def_authority != NULL)
               {
                  /* Check whether the default authority has changed */
                  if (nit_entry->def_authority->nbytes == t_ptr->def_authority->nbytes)
                  {
                     if (memcmp(t_ptr->def_authority->str_ptr, nit_entry->def_authority->str_ptr,
                            t_ptr->def_authority->nbytes) == 0)
                     {
                        /* The names are the same */
                        changed = FALSE;
                     }
                  }
               }

               if (changed)
               {
                  /* Free the old authority string */
                  if (t_ptr->def_authority != NULL)
                  {
                     STB_AppFreeMemory(t_ptr->def_authority);
                     t_ptr->def_authority = NULL;
                  }

                  /* Add the new authority string */
                  t_ptr->def_authority = DBDEF_MakeString(0, nit_entry->def_authority->str_ptr,
                        nit_entry->def_authority->nbytes);
#ifdef DEBUG_SI_NIT
                  AP_SI_PRINT(("   Transport 0x%x has default authority \"%s\"", t_ptr->tran_id,
                               t_ptr->def_authority->str_ptr));
#endif
               }
            }

            if (nit_entry->freq_list != NULL)
            {
               if (t_ptr->additional_frequencies != NULL)
               {
                  STB_AppFreeMemory(t_ptr->additional_frequencies);
                  t_ptr->additional_frequencies = NULL;
                  t_ptr->num_additional_frequencies = 0;
               }

               /* See if a list of additional frequencies exists for this transport based on its signal type */
               for (freq_list_ptr = nit_entry->freq_list; freq_list_ptr != NULL; freq_list_ptr = freq_list_ptr->next)
               {
                  if (((t_ptr->sig_type == SIGNAL_QPSK) &&
                       (freq_list_ptr->coding_type == FREQ_LIST_CODING_TYPE_SATELLITE)) ||
                      ((t_ptr->sig_type == SIGNAL_QAM) &&
                       (freq_list_ptr->coding_type == FREQ_LIST_CODING_TYPE_CABLE)) ||
                      ((t_ptr->sig_type == SIGNAL_COFDM) &&
                       (freq_list_ptr->coding_type == FREQ_LIST_CODING_TYPE_TERRESTRIAL)))
                  {
                     /* This is a list of additional frequencies for this transport.
                      * Save the frequencies with the transport so they can be used, if needed,
                      * the next time there's an attempt to tune to it */
                     t_ptr->additional_frequencies = STB_AppGetMemory(freq_list_ptr->num_frequencies * sizeof(U32BIT));
                     if (t_ptr->additional_frequencies != NULL)
                     {
                        memcpy(t_ptr->additional_frequencies, freq_list_ptr->frequency_array,
                           freq_list_ptr->num_frequencies * sizeof(U32BIT));
                        t_ptr->num_additional_frequencies = freq_list_ptr->num_frequencies;
                     }
                     break;
                  }
               }
            }
         }

         nit_entry = nit_entry->next;
      }

      if (n_ptr != NULL)
      {
         if (mode == DB_ACCESS_UPDATE)
         {
            /* Delete the linkage array for repopulation */
            DeleteLinkageDescripterArray(n_ptr->linkage_desc_list);
            n_ptr->last_linkage_entry = NULL;
            n_ptr->num_linkage_entries = 0;
            n_ptr->linkage_desc_list = NULL;

            /* Copy all network linkage descripters  */
            linkage_desc_ptr = nit_table->linkage_desc_list;
            while (linkage_desc_ptr != NULL)
            {
               CopyLinkageDesc(linkage_desc_ptr,
                  &n_ptr->linkage_desc_list,
                  &n_ptr->last_linkage_entry,
                  &n_ptr->num_linkage_entries);

               linkage_desc_ptr = linkage_desc_ptr->next;
            }
         }
      }

      // release database access
      DBDEF_ReleaseAccess();

      if (update_nit_func != NULL)
      {
         /* Call the function that's been registered to handle additional NIT processing */
         (*update_nit_func)(path, nit_table, table_rec);
      }

      // release the table - don't need it anymore
      STB_SIReleaseNitTable(nit_table);
   }

   if (report_nit && (transport_changed || (table_rec->version != last_reported_nit_version[path])))
   {
      #ifdef DEBUG_SI_NIT
      AP_SI_PRINT(("%s: reporting NIT 0x%04x, ver=%u", __FUNCTION__, table_rec->xtid, table_rec->version));
      #endif

      if (STB_SIReportNit(table_rec))
      {
         last_reported_nit_version[path] = table_rec->version;
      }
   }

   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessNitTable);

   return(restart_sdts);
}

/**
 *

 *
 * @brief   Processes the CAT table for reporting to CA systems
 *
 * @param   path
 * @param   table_rec      - pointer to the cat table record
 * @param   report_cat     - if TRUE cat will be reported to 3rd parties if this is the first
 *                                 copy after transport change or if it is a new version. if FALSE the
 *                                 cat will not be reported
 * @param   transport_changed
 *                               - if set TRUE indicates that a change in transport has occurred,
 *                                 FALSE otherwise.
 *

 *
 */
static void ProcessCatTable(U8BIT path, SI_TABLE_RECORD *table_rec, BOOLEAN report_cat, BOOLEAN transport_changed)
{
   FUNCTION_START(ProcessCatTable);

#ifdef DEBUG_SI_CAT
   AP_SI_PRINT(("%s(%u, report_cat=%u, transport_changed=%u): table_rec->version=%d last version=%d",
                __FUNCTION__, path, report_cat, transport_changed, table_rec->version, last_reported_cat_version[path]));
#endif

   // check if CAT needs reporting to other applications (e.g. ca systems)
#ifdef CUSTOMIZED_FOR_CNS
   if (report_cat)
#else
   if (report_cat && (transport_changed || (table_rec->version != last_reported_cat_version[path])))
#endif
   {
      #ifdef DEBUG_SI_CAT
      AP_SI_PRINT(("Reporting cat: version=%d num_sects=%d", table_rec->version, table_rec->num_sect));
      #endif

      if (STB_SIReportCat(table_rec))
      {
         last_reported_cat_version[path] = table_rec->version;
      }
   }

   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessCatTable);
}

/*!**************************************************************************
 * @brief   Processes the related content table (RCT) to provide trailer booking functionality.
 *          It's assumed that only the RCT for the current service will have been requested.
 * @param   table_rec - pointer to the SI table data
 ****************************************************************************/
static void ProcessRctTable(SI_TABLE_RECORD *table_rec)
{
   SI_RCT_TABLE *rct_table;
   SI_RCT_SUBTABLE *subtable;
   SI_RCT_SUBTABLE_DATA *subdata;
   SI_RCT_LINK_INFO *link_info;
   SI_RCT_PROMO_TEXT *promo_text;
   U8BIT path;
   U8BIT i, link;
   BOOLEAN valid_link;
   ADB_RCT_LINK_INFO *adb_link;
   ADB_IMAGE_ICON *icon_ptr;
   U8BIT lang_id;
   BOOLEAN links_cleared;
   BOOLEAN service_links_updated;

   FUNCTION_START(ProcessRctTable);

   path = table_rec->path;
   service_links_updated = FALSE;

   rct_table = STB_SIParseRctTable(table_rec);
   STB_SIReleaseTableRecord(table_rec);

   if (rct_table != NULL)
   {
      links_cleared = FALSE;

      /* Look for any RCT subtables that are for the current service */
      subtable = rct_table->subtables;
      while (subtable != NULL)
      {
         /* A service ID of 0 means the service defined by the PMT that defined the PID containing the RCT */
         if ((subtable->service_id == 0) || (subtable->service_id == current_service_rec[path]->serv_id))
         {
            if (!links_cleared)
            {
#ifdef DEBUG_SI_RCT
               AP_SI_PRINT(("Clearing RCT links for service 0x%04x", current_service_rec[path]->serv_id));
#endif
               /* Clear the existing links for the current service */
               ADB_ServiceReleaseRCTLinks(current_service_rec[path]);
               links_cleared = TRUE;
               service_links_updated = TRUE;
            }

            subdata = subtable->data;
            while (subdata != NULL)
            {
               /* Process the links */
               for (link = 0; link < subdata->link_count; link++)
               {
                  link_info = &subdata->link_array[link];

                  /* Check that this is a valid link as defined by D-Book 6.2.1 */
                  if ((link_info->link_type == RCT_LINK_TYPE_URI) &&
                      (link_info->how_related == RCT_HOW_RELATED_TVA_2007) &&
                      (link_info->uri_string != NULL))
                  {
                     switch (link_info->term_id)
                     {
                        case RCT_TERMID_IS_TRAILER_OF:
#ifdef DEBUG_SI_RCT
                           AP_SI_PRINT(("ProcessRctTable: Found TRAILER_OF link with uri \"%s\"",
                              link_info->uri_string));
#endif
                           valid_link = TRUE;
                           break;

                        case RCT_TERMID_IS_GROUP_TRAILER_OF:
#ifdef DEBUG_SI_RCT
                           AP_SI_PRINT(("ProcessRctTable: Found GROUP_TRAILER_OF link with uri \"%s\"",
                              link_info->uri_string));
#endif
                           valid_link = TRUE;
                           break;

                        default:
                           valid_link = FALSE;
                           break;
                     }

                     if (valid_link)
                     {
                        adb_link = (ADB_RCT_LINK_INFO *)STB_AppGetMemory(sizeof(ADB_RCT_LINK_INFO));
                        if (adb_link != NULL)
                        {
                           memset(adb_link, 0, sizeof(ADB_RCT_LINK_INFO));

                           if (link_info->term_id == RCT_TERMID_IS_GROUP_TRAILER_OF)
                           {
                              adb_link->is_group_trailer = TRUE;
                           }

                           adb_link->uri_string = (U8BIT *)STB_AppGetMemory(strlen((char *)link_info->uri_string) + 1);
                           if (adb_link->uri_string != NULL)
                           {
                              strncpy((char *)adb_link->uri_string, (char *)link_info->uri_string, strlen((char *)link_info->uri_string) + 1);

                              adb_link->can_use_default_icon = link_info->can_use_default_icon;
                              adb_link->icon_id = link_info->icon_id;

                              if ((link_info->event_desc != NULL) && (link_info->event_desc->name_str != NULL))
                              {
                                 adb_link->event_name = DBDEF_MakeString(link_info->event_desc->lang_code,
                                       link_info->event_desc->name_str->str_ptr,
                                       link_info->event_desc->name_str->nbytes);
                              }

                              /* Store promotional text strings for each defined language */
                              for (i = 0; i < link_info->num_items; i++)
                              {
                                 promo_text = &link_info->promo_text_array[i];

                                 if (promo_text->string != NULL)
                                 {
                                    lang_id = ACFG_ConvertLangCodeToId(promo_text->lang_code);
                                    if (lang_id == ACFG_INVALID_DB_LANG)
                                    {
                                       lang_id = 0;
                                    }

                                    if (adb_link->promo_text[lang_id] == NULL)
                                    {
                                       adb_link->promo_text[lang_id] = DBDEF_MakeString(promo_text->lang_code,
                                             promo_text->string->str_ptr, promo_text->string->nbytes);
                                    }
                                 }
                              }

#ifdef DEBUG_SI_RCT
                              AP_SI_PRINT(("ProcessRctTable: \"%s\" can_use_default=%u",
                                 adb_link->uri_string, adb_link->can_use_default_icon));
#endif
                              ADB_ServiceAddRCTLink(current_service_rec[path], adb_link);
                              service_links_updated = TRUE;

                              /* Save any icons defined in this link */
                              for (i = 0; i < link_info->num_icons; i++)
                              {
                                 if ((icon_ptr = ProcessImageIcon(&link_info->icon_array[i])) != NULL)
                                 {
                                    if (ADB_ServiceAddImageIcon(current_service_rec[path], icon_ptr))
                                    {
                                    #if defined(INCLUDE_DSMCC_FILE_REQUEST)
                                       IconFileRequest(icon_ptr,path);
                                    #endif
                                    }
                                 }
                              }
                           }
                           else
                           {
                              /* Out of memory! */
                              STB_AppFreeMemory(adb_link);
                           }
                        }
                     }
                  }
               }

               /* Add any icons defined by this subtable to the current service */
               for (i = 0; i < subdata->num_icons; i++)
               {
                  if ((icon_ptr = ProcessImageIcon(&subdata->icon_array[i])) != NULL)
                  {
                     if (ADB_ServiceAddImageIcon(current_service_rec[path], icon_ptr))
                     {
                     #if defined(INCLUDE_DSMCC_FILE_REQUEST)
                        IconFileRequest(icon_ptr,path);
                     #endif
                     }
                  }
               }

               subdata = subdata->next;
            }
         }

         subtable = subtable->next;
      }

      STB_SIReleaseRctTable(rct_table);

      if (service_links_updated)
      {
#ifdef DEBUG_SI_RCT
         AP_SI_PRINT(("ProcessRctTable: Sending PROMO_LINK_CHANGE event"));
#endif
         /* Send an event to the app to info it that RCT link info for the current service has changed */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_MHEG, EV_TYPE_MHEG_PROMO_LINK_CHANGE, NULL, 0);
      }
   }

   FUNCTION_FINISH(ProcessRctTable);
}

/*!**************************************************************************
 * @brief   Processes the bouquet association table (BAT)
 * @param   table_rec - pointer to the SI table data
 * @param   mode - mode in which the table is being processsed, SEARCH or UPDATE
 * @param   report_bat - TRUE if the BAT can be reported to other modules, FALSE otherwise
 * @return  TRUE if the SDT collection needs to be restarted, FALSE otherwise
 ****************************************************************************/
static BOOLEAN ProcessBatTable(SI_TABLE_RECORD *table_rec, E_DB_ACCESS_MODE mode, BOOLEAN report_bat)
{
   BOOLEAN process;
   SI_BAT_TABLE *bat_table;
   ADB_BAT_VERSION_REC *bat_version_rec;
   U8BIT list_id;
   U8BIT *name;
   U16BIT nchar;
   SI_BAT_TRANSPORT_ENTRY *trans_entry;
   ADB_TRANSPORT_REC *t_ptr;
   ADB_SATELLITE_REC *sat_ptr;
   SI_SERV_LIST_DESC *serv_desc;
   ADB_SERVICE_REC *s_ptr;
   BOOLEAN changed;
   U8BIT path;
   U16BIT i, j;
   BOOLEAN restart_sdts;
   BOOLEAN service_updated = FALSE;   

   FUNCTION_START(ProcessBatTable);

   restart_sdts = FALSE;

   /* Check whether this BAT needs to be processed */
   if (active_bouquet_ids != NULL)
   {
      process = FALSE;

      for (i = 0; i < num_active_bouquet_ids; i++)
      {
         if (active_bouquet_ids[i].bouquet_id == table_rec->xtid)
         {
            active_bouquet_ids[i].received = TRUE;
            process = TRUE;
            break;
         }
      }
   }
   else
   {
      /* All BATs are to be processed */
      process = TRUE;
   }

   if (process)
   {
      bat_table = STB_SIParseBatTable(table_rec);
      if (bat_table != NULL)
      {
         #ifdef DEBUG_SI_BAT
         AP_SI_PRINT(("BAT received: id=%u, num_trans=%u", bat_table->bouquet_id, bat_table->num_transports));

         if (bat_table->bouquet_name != NULL)
         {
            AP_SI_PRINT((" name=\"%s\"", bat_table->bouquet_name->str_ptr));
         }
         #endif

         path = table_rec->path;

         if ((sat_ptr = (ADB_SATELLITE_REC *)ACTL_GetCurrentSatellite(path)) != NULL)
         {
            /* Check the table version */
            bat_version_rec = sat_ptr->bat_version_list;
            while (bat_version_rec != NULL)
            {
               if (bat_version_rec->bouquet_id == bat_table->bouquet_id)
               {
                  if (bat_version_rec->version == bat_table->version)
                  {
                     /* Version hasn't changed so doesn't need to be processed */
#ifdef DEBUG_SI_BAT
                     AP_SI_PRINT(("  existing BAT version %u", bat_table->version));
#endif
                     process = FALSE;
                  }
                  break;
               }

               bat_version_rec = bat_version_rec->next;
            }

            if (bat_version_rec == NULL)
            {
               /* No version record for this BAT yet, so add one */
#ifdef DEBUG_SI_BAT
               AP_SI_PRINT(("  new BAT version %u", bat_table->version));
#endif
               bat_version_rec = (ADB_BAT_VERSION_REC *)STB_AppGetMemory(sizeof(ADB_BAT_VERSION_REC));
               if (bat_version_rec != NULL)
               {
                  bat_version_rec->bouquet_id = bat_table->bouquet_id;
                  bat_version_rec->next = sat_ptr->bat_version_list;
                  sat_ptr->bat_version_list = bat_version_rec;
               }
            }

            if (process && (bat_version_rec != NULL))
            {
               bat_version_rec->version = bat_table->version;
            }
         }

         if (process)
         {
            if (mode == DB_ACCESS_SEARCH)
            {
               /* No point creating a favourite list if the bouquet doesn't have a name */
               if (bat_table->bouquet_name != NULL)
               {
                  /* Create a favourite list for this bouquet */
                  if ((name = STB_ConvertStringToUTF8(bat_table->bouquet_name->str_ptr, &nchar, TRUE, 0)) != NULL)
                  {
                     if (ADB_AddFavouriteList(name, bat_table->bouquet_id, -1, &list_id))
                     {
                        DBDEF_RequestAccess();

                        // mark all services unavailable
                        s_ptr = DBDEF_GetNextServiceRec(NULL);
                        while (s_ptr != NULL)
                        {
                           s_ptr->found = FALSE;
                           s_ptr->unavailable = TRUE;
                           s_ptr->hidden = TRUE;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);  
                           s_ptr->selectable = FALSE;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);
                           DBA_SaveRecord(s_ptr->dba_rec);

                           s_ptr = DBDEF_GetNextServiceRec(s_ptr);
                        }


                        /* Go through each transport adding transports and services, as necessary, so
                         * the services can be added to the favourite list in the order they're defined
                         * in the BAT */
                        for (trans_entry = bat_table->transport_list; trans_entry != NULL;
                             trans_entry = trans_entry->next)
                        {
                           t_ptr = DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                                 trans_entry->orig_net_id, trans_entry->tran_id);

                           if ((trans_entry->num_serv_list_entries > 0) &&
                               (trans_entry->serv_list_desc_array != NULL))
                           {
                              if (t_ptr == NULL)
                              {
                                 #ifdef DEBUG_SI_BAT
                                 AP_SI_PRINT(("  New transport 0x%04x/0x%04x", trans_entry->orig_net_id,
                                              trans_entry->tran_id));
                                 #endif

                                 /* Create an empty transport record so the services can be added to it */
                                 switch (current_transport_rec[path]->sig_type)
                                 {
                                    case SIGNAL_COFDM:
                                       t_ptr = DBDEF_AddTerrestrialTransportRec(0, 0, NULL);
                                       break;

                                    case SIGNAL_QAM:
                                       t_ptr = DBDEF_AddCableTransportRec(0, 0, NULL);
                                       break;

                                    case SIGNAL_QPSK:
                                       t_ptr = DBDEF_AddSatTransportRec(0, 0, POLARITY_HORIZONTAL,
                                             FALSE, MOD_AUTO, NULL);
                                       break;

                                    default:
                                       break;
                                 }

                                 if (t_ptr != NULL)
                                 {
                                    DBDEF_SetTransportTransportId(t_ptr, trans_entry->tran_id);
                                    DBDEF_SetTransportOrigNetworkId(t_ptr, trans_entry->orig_net_id);
                                 }
                                 #ifdef DEBUG_SI_BAT
                                 else
                                 {
                                    AP_SI_PRINT(("   Failed to create transport!", __FUNCTION__));
                                 }
                                 #endif
                              }
                              #ifdef DEBUG_SI_BAT
                              else
                              {
                                 AP_SI_PRINT(("  Existing transport 0x%04x/0x%04x", trans_entry->orig_net_id,
                                              trans_entry->tran_id));
                              }
                              #endif

                              if (t_ptr != NULL)
                              {
                                 for (i = 0; i < trans_entry->num_serv_list_entries; i++)
                                 {
                                    serv_desc = &trans_entry->serv_list_desc_array[i];
                                    s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                                          t_ptr->orig_net_id, t_ptr->tran_id, serv_desc->serv_id);
                                    if (s_ptr == NULL)
                                    {
                                       #ifdef DEBUG_SI_BAT
                                       AP_SI_PRINT(("    New service 0x%04x", serv_desc->serv_id));
                                       #endif

                                       /* Create a new service */
                                       if ((s_ptr = DBDEF_AddServiceRec(serv_desc->serv_id, t_ptr)) != NULL)
                                       {
                                          s_ptr->found = TRUE;
                                          s_ptr->unavailable = FALSE;
                                          s_ptr->hidden = FALSE;
                                          DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);
                                          s_ptr->selectable = FALSE;
                                          DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);
                                          s_ptr->serv_type = serv_desc->serv_type;
                                          DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

                                          DBA_SaveRecord(s_ptr->dba_rec);
                                       }
                                       #ifdef DEBUG_SI_BAT
                                       else
                                       {
                                          AP_SI_PRINT(("     Failed to create service!", __FUNCTION__));
                                       }
                                       #endif
                                    }
                                    #ifdef DEBUG_SI_BAT
                                    else
                                    {
                                       AP_SI_PRINT(("    Existing service 0x%04x", serv_desc->serv_id));
                                    }
                                    #endif

                                    if (s_ptr != NULL)
                                    {
                                       s_ptr->found = TRUE;
                                       s_ptr->unavailable = FALSE;
                                       // FIXME: CNS logical channel number
                                       for (j = 0; j < trans_entry->num_lcn_entries; j++) {
                                          if (s_ptr->serv_id == trans_entry->lcn_desc_array[j].serv_id) {
                                            U16BIT reqd_lcn = trans_entry->lcn_desc_array[j].serv_lcn;
                                            
                                            /* Set service hidden attribute , a hidden service still
                                               need to be selectable, so it's aways TRUE in this case */
                                            s_ptr->hidden = FALSE; //!trans_entry->lcn_desc_array[j].visible;                                      
                                            DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                                            switch (s_ptr->serv_type){
                                               case 1:
                                                  s_ptr->selectable = TRUE;
                                                  break;
                                               case 2:
                                                  s_ptr->selectable = TRUE;
                                                  break;
                                               default:
                                                  s_ptr->selectable = FALSE;
                                                  break;
                                            }
                                            DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                                            if (reqd_lcn != s_ptr->serv_lcn) {
                                               s_ptr->serv_lcn = reqd_lcn;
                                               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_LCN, s_ptr->serv_lcn);
                                               s_ptr->allocated_lcn = reqd_lcn;
                                               DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
#ifdef DEBUG_SI_BAT
                                               AP_SI_PRINT(("   LCN for 0x%04x, %u, hidden=%u", s_ptr->serv_id, reqd_lcn,
                                                  s_ptr->hidden));
#endif
                                            }
                                            DBA_SaveRecord(s_ptr->dba_rec);
                                            break;               
                                          }
                                       }

                                       /* Add the service to the end of the bouquet's favourite list */
                                       changed = ADB_AddServiceToFavouriteList(list_id, s_ptr, -1);
                                       #ifdef DEBUG_SI_BAT
                                       if (!changed)
                                       {
                                          AP_SI_PRINT(("     Failed to add service to fav list 0x%02x", list_id));
                                       }
                                       #endif
                                    }
                                 }
                              }
                           }
                        }  // for (trans_entry = bat_table...)

                        DBDEF_ReleaseAccess();

                        if (ADB_GetNumServicesInList(ADB_LIST_TYPE_FROM_FAVLIST(list_id), TRUE) == 0)
                        {
                           #ifdef DEBUG_SI_BAT
                           AP_SI_PRINT(("%s: Deleting empty bouquet favourite list", __FUNCTION__));
                           #endif

                           /* No services have been added to the list so it can be deleted */
                           ADB_DeleteFavouriteList(list_id);
                        }
                     }

                     STB_ReleaseUnicodeString(name);
                  }
               }
            }
            else if (mode == DB_ACCESS_UPDATE)
            {
               /* To support dynamic addition and deletion of services, info from the transports
                * needs to be saved so it can be applied when all relevant SDTs have been received */
               trans_entry = bat_table->transport_list;
               while (trans_entry != NULL)
               {
                  t_ptr = DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                        trans_entry->orig_net_id, trans_entry->tran_id);
                        
#if 0
                  if (NULL == t_ptr) {
                     /* A new transport */
                     DBDEF_RequestAccess();

                     if ((trans_entry->num_serv_list_entries > 0) &&
                         (trans_entry->serv_list_desc_array != NULL))
                     {
                        #ifdef DEBUG_SI_BAT
                        AP_SI_PRINT(("  New transport 0x%04x/0x%04x", trans_entry->orig_net_id,
                                     trans_entry->tran_id));
                        #endif

                        /* Create an empty transport record so the services can be added to it */
                        switch (current_transport_rec[path]->sig_type)
                        {
                           case SIGNAL_COFDM:
                              t_ptr = DBDEF_AddTerrestrialTransportRec(0, 0, NULL);
                              break;

                           case SIGNAL_QAM:
                              t_ptr = DBDEF_AddCableTransportRec(0, 0, NULL);
                              break;

                           case SIGNAL_QPSK:
                              t_ptr = DBDEF_AddSatTransportRec(0, 0, POLARITY_HORIZONTAL,
                                    FALSE, MOD_AUTO, NULL);
                              break;

                           default:
                              break;
                        }

                        if (t_ptr != NULL)
                        {
                           DBDEF_SetTransportTransportId(t_ptr, trans_entry->tran_id);
                           DBDEF_SetTransportOrigNetworkId(t_ptr, trans_entry->orig_net_id);
                        }
                        #ifdef DEBUG_SI_BAT
                        else
                        {
                           AP_SI_PRINT(("   Failed to create transport!", __FUNCTION__));
                        }
                        #endif
                     }
                     DBDEF_ReleaseAccess();
                  }
#endif

                  if (t_ptr != NULL)
                  {
                     DBDEF_RequestAccess();
                     for (i = 0; i < trans_entry->num_serv_list_entries; i++)
                     {
                        serv_desc = &trans_entry->serv_list_desc_array[i];
                        s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                              t_ptr->orig_net_id, t_ptr->tran_id, serv_desc->serv_id);
                        if (s_ptr == NULL)
                        {
                           #ifdef DEBUG_SI_BAT
                           AP_SI_PRINT(("    New service 0x%04x", serv_desc->serv_id));
                           #endif

                           /* Create a new service */
                           if ((s_ptr = DBDEF_AddServiceRec(serv_desc->serv_id, t_ptr)) != NULL)
                           {
                              s_ptr->found = FALSE;
                              s_ptr->unavailable = TRUE;

                              s_ptr->serv_type = serv_desc->serv_type;
                              DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_TYPE, s_ptr->serv_type);

                              DBA_SaveRecord(s_ptr->dba_rec);
                              restart_sdts = TRUE;
                           }
                           #ifdef DEBUG_SI_BAT
                           else
                           {
                              AP_SI_PRINT(("     Failed to create service!", __FUNCTION__));
                           }
                           #endif                          
                        }
                        #ifdef DEBUG_SI_BAT
                        else
                        {
                           AP_SI_PRINT(("    Existing service 0x%04x", serv_desc->serv_id));
                        }
                        #endif

                        if (s_ptr != NULL)
                        {
                           BOOLEAN service_updated = FALSE;
                           // FIXME: CNS logical channel number
                           for (j = 0; j < trans_entry->num_lcn_entries; j++) {
                              if (s_ptr->serv_id == trans_entry->lcn_desc_array[j].serv_id) {
                                U16BIT reqd_lcn = trans_entry->lcn_desc_array[j].serv_lcn;

                                /* Set service hidden attribute , a hidden service still
                                   need to be selectable, so it's aways TRUE in this case */
                                s_ptr->hidden = !trans_entry->lcn_desc_array[j].visible;
                                DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);

                                s_ptr->selectable = TRUE;
                                DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                                if (reqd_lcn != s_ptr->serv_lcn) {
                                   s_ptr->serv_lcn = reqd_lcn;
                                   s_ptr->allocated_lcn = reqd_lcn;
                                   DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
                                   DBA_SaveRecord(s_ptr->dba_rec);
#ifdef DEBUG_SI_BAT
                                   AP_SI_PRINT(("   LCN for 0x%04x, %u, hidden=%u", s_ptr->serv_id, reqd_lcn,
                                      s_ptr->hidden));
#endif
                                   service_updated = TRUE;
                                   restart_sdts = TRUE;
                                }
                                break;               
                              }
                           }  // for (j = 0; j < trans_entry->num_lcn_entries; j++)

                           if (TRUE == service_updated) {
                              STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                                 EV_SERVICE_UPDATED, NULL, 0);
                           }
                        }
                     }  // for (i = 0; i < trans_entry->num_serv_list_entries; i++)
                     DBDEF_ReleaseAccess(); 

                     if (ASI_DatabaseUpdatesAllowed(t_ptr->sig_type) &&
                        ACFG_GetDynamicSIUpdate(t_ptr->sig_type, trans_entry->orig_net_id,
                        (ACFG_DYNAMIC_SI_UPDATE_LCNS |
                         ACFG_DYNAMIC_SI_UPDATE_SERVICE_ADD |
                         ACFG_DYNAMIC_SI_UPDATE_SERVICE_REMOVE)))
                     {
                        /* Service addition or deletion means LCNs may change so LCN descriptors need to
                         * be saved so they can be applied following any changes to the service line up */
                        if (DynamicUpdateAddTransport(t_ptr,
                           trans_entry->num_lcn_entries, trans_entry->lcn_desc_array, 0, NULL, 0, NULL))
                        {
                           /* The SDT for this transport needs to be processed again, so mark it as not
                            * having been received yet so we can tell when it has been */
                           t_ptr->sdt_received = FALSE;
                           t_ptr->sdt_version = 0xFFFF;  // FIXME

                           restart_sdts = TRUE;
                        }
                     }
                  }

                  trans_entry = trans_entry->next;
               }  // while (trans_entry != NULL)
            }

            if (sat_ptr != NULL)
            {
               if (bat_table->fta_content_desc != NULL)
               {
                  sat_ptr->has_fta_desc = TRUE;
                  sat_ptr->do_not_scramble = bat_table->fta_content_desc->do_not_scramble;
               }

               if (bat_table->def_authority != NULL)
               {
                  changed = TRUE;

                  if (sat_ptr->def_authority != NULL)
                  {
                     if (bat_table->def_authority->nbytes == sat_ptr->def_authority->nbytes)
                     {
                        /* String are the same length, check they're the same */
                        if (memcmp(bat_table->def_authority->str_ptr, sat_ptr->def_authority->str_ptr,
                               bat_table->def_authority->nbytes) == 0)
                        {
                           /* Strings are the same */
                           changed = FALSE;
                        }
                     }
                  }

                  if (changed)
                  {
                     if (sat_ptr->def_authority != NULL)
                     {
                        DBDEF_ReleaseString(sat_ptr->def_authority);
                     }

                     sat_ptr->def_authority = DBDEF_MakeString(0, bat_table->def_authority->str_ptr,
                           bat_table->def_authority->nbytes);
#ifdef DEBUG_SI_BAT
                     AP_SI_PRINT(("   Satellite has default authority \"%s\"",
                        sat_ptr->def_authority->str_ptr));
#endif
                  }
               }
               else if (sat_ptr->def_authority != NULL)
               {
                  /* Default authority has disappeared from the BAT, so delete it from the satellite */
                  DBDEF_ReleaseString(sat_ptr->def_authority);
                  sat_ptr->def_authority = NULL;
               }

               for (trans_entry = bat_table->transport_list; trans_entry != NULL;
                    trans_entry = trans_entry->next)
               {
                  t_ptr = DBDEF_FindTransportRecByIds(NULL, ADB_INVALID_DVB_ID,
                        trans_entry->orig_net_id, trans_entry->tran_id);
                  if (t_ptr != NULL)
                  {
                     if (trans_entry->def_authority != NULL)
                     {
                        changed = TRUE;

                        if (t_ptr->def_authority != NULL)
                        {
                           if (trans_entry->def_authority->nbytes == t_ptr->def_authority->nbytes)
                           {
                              /* String are the same length, check they're the same */
                              if (memcmp(trans_entry->def_authority->str_ptr,
                                     t_ptr->def_authority->str_ptr,
                                     trans_entry->def_authority->nbytes) == 0)
                              {
                                 /* Strings are the same */
                                 changed = FALSE;
                              }
                           }
                        }

                        if (changed)
                        {
                           if (t_ptr->def_authority != NULL)
                           {
                              DBDEF_ReleaseString(t_ptr->def_authority);
                           }

                           t_ptr->def_authority = DBDEF_MakeString(0,
                                 trans_entry->def_authority->str_ptr,
                                 trans_entry->def_authority->nbytes);
#ifdef DEBUG_SI_BAT
                           AP_SI_PRINT(("   Transport 0x%04x has default authority \"%s\"",
                              t_ptr->tran_id, t_ptr->def_authority->str_ptr));
#endif
                        }
                     }
                     else if (t_ptr->def_authority != NULL)
                     {
                        /* Default authority has disappeared from the BAT, so delete it from the transport */
                        DBDEF_ReleaseString(t_ptr->def_authority);
                        t_ptr->def_authority = NULL;
                     }
                  }
               }
            }
         }

         if (update_bat_func != NULL)
         {
            /* Call the function that's been registered to handle additional BAT processing */
            (*update_bat_func)(path, bat_table, table_rec);
         }

         STB_SIReleaseBatTable(bat_table);
      }

      if (report_bat)
      {
         #ifdef DEBUG_SI_BAT
         AP_SI_PRINT(("%s: reporting BAT 0x%04x, ver=%u", __FUNCTION__, table_rec->xtid, table_rec->version));
         #endif

         STB_SIReportBat(table_rec);
      }
   }

   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessBatTable);

   return(restart_sdts);
}

static void ProcessUntTable(U8BIT path, SI_TABLE_RECORD *table_rec, BOOLEAN report_unt, BOOLEAN transport_changed)
{
   SI_UNT_TABLE *unt_table;
   FUNCTION_START(ProcessUntTable);

   unt_table = STB_SIParseUntTable(table_rec);
   if (unt_table != NULL)
   {
      U8BIT path = table_rec->path;
      if (update_unt_func != NULL)
      {
          /* Call the function that's been registered to handle additional UNT processing */
          (*update_unt_func)(path, unt_table, table_rec);
      }
      STB_SIReleaseUntTable(unt_table);
   }
   
   STB_SIReleaseTableRecord(table_rec);

   FUNCTION_FINISH(ProcessUntTable);
}

/*!**************************************************************************
 * @brief   Processes an image icon descriptor, storing the info for use by the app
 * @param   si_icon - pointer to the SI icon structure
 * @return  a pointer to a newly allocated app image icon
 ****************************************************************************/
static ADB_IMAGE_ICON* ProcessImageIcon(SI_IMAGE_ICON_DESC *si_icon)
{
   ADB_IMAGE_ICON *icon_ptr;

   FUNCTION_START(ProcessImageIcon);

   icon_ptr = NULL;

   if (((si_icon->transport_mode == ICON_TRANS_LOCAL) || (si_icon->transport_mode == ICON_TRANS_URL)) &&
       (si_icon->icon_type != NULL) && (si_icon->data_len > 0) && (si_icon->icon_data != NULL))
   {
      /* Check the icon is a known type */
      if ((STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"image/png") == 0) ||
          (STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"/png") == 0) ||
          (STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"image/jpeg") == 0) ||
          (STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"/jpeg") == 0))
      {
         /* Create a structure for the new icon */
         icon_ptr = (ADB_IMAGE_ICON *)STB_AppGetMemory(sizeof(ADB_IMAGE_ICON));
         if (icon_ptr != NULL)
         {
            memset(icon_ptr, 0, sizeof(ADB_IMAGE_ICON));

            icon_ptr->icon_id = si_icon->icon_id;
            icon_ptr->transport_mode = si_icon->transport_mode;

            if ((STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"image/png") == 0) ||
                (STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"/png") == 0))
            {
               icon_ptr->icon_type = ICON_TYPE_PNG;
            }
            else if ((STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"image/jpeg") == 0) ||
                     (STB_CompareStringsIgnoreCase(si_icon->icon_type, (U8BIT *)"/jpeg") == 0))
            {
               icon_ptr->icon_type = ICON_TYPE_JPEG;
            }

            /* Set the default coordinate system which may be overridden below */
            icon_ptr->coord_system = ICON_COORDS_576;

            icon_ptr->position_defined = si_icon->position_defined;
            if (icon_ptr->position_defined)
            {
               /* Check for valid coord system and positions, otherwise allow app to use its
                * default pos. The specified position is scaled to output being used */
               switch (si_icon->coord_system)
               {
                  case ICON_COORDS_576:
                     if ((si_icon->x_pos < 720) && (si_icon->y_pos < 576))
                     {
                        icon_ptr->coord_system = si_icon->coord_system;
                        icon_ptr->x_pos = si_icon->x_pos;
                        icon_ptr->y_pos = si_icon->y_pos;
                     }
                     else
                     {
                        icon_ptr->position_defined = FALSE;
                     }
                     break;

                  case ICON_COORDS_720:
                     if ((si_icon->x_pos < 1280) && (si_icon->y_pos < 720))
                     {
                        icon_ptr->coord_system = si_icon->coord_system;
                        icon_ptr->x_pos = si_icon->x_pos;
                        icon_ptr->y_pos = si_icon->y_pos;
                     }
                     else
                     {
                        icon_ptr->position_defined = FALSE;
                     }
                     break;

                  case ICON_COORDS_1080:
                     if ((si_icon->x_pos < 1920) && (si_icon->y_pos < 1080))
                     {
                        icon_ptr->coord_system = si_icon->coord_system;
                        icon_ptr->x_pos = si_icon->x_pos;
                        icon_ptr->y_pos = si_icon->y_pos;
                     }
                     else
                     {
                        icon_ptr->position_defined = FALSE;
                     }
                     break;

                  default:
                     icon_ptr->position_defined = FALSE;
                     break;
               }
            }

            if (si_icon->transport_mode == ICON_TRANS_URL)
            {
               /* Store the icon's uri so it can be retrieved from the MHEG carousel */
               icon_ptr->icon_url = (U8BIT *)STB_AppGetMemory(si_icon->data_len);
               if (icon_ptr->icon_url != NULL)
               {
                  memcpy(icon_ptr->icon_url, si_icon->icon_data, si_icon->data_len);
               }
            }
            else
            {
               /* The icon data is contained in the SI data. Convert the file to an image
                * that can be displayed */
               if (icon_ptr->icon_type == ICON_TYPE_PNG)
               {
#ifdef DEBUG_SI_RCT
                  AP_SI_PRINT(("ProcessImageIcon: Inline PNG icon @ (%u, %u), size %u x %u",
                     icon_ptr->x_pos, icon_ptr->y_pos, icon_ptr->width, icon_ptr->height));
#endif
                  /* Interpret the data as a PNG image */
                  if (!STB_IMGConvertPNG(si_icon->icon_data, si_icon->data_len,
                         &icon_ptr->icon_data, &icon_ptr->data_len, &icon_ptr->width, &icon_ptr->height))
                  {
#ifdef DEBUG_SI_RCT
                     AP_SI_PRINT(("ProcessImageIcon: Failed to process PNG icon"));
#endif
                     DBDEF_DeleteImageIcons(icon_ptr);
                     icon_ptr = NULL;
                  }
               }
               else
               {
#ifdef DEBUG_SI_RCT
                  AP_SI_PRINT(("ProcessImageIcon: Inline JPEG icon @ (%u, %u), size %u x %u",
                     icon_ptr->x_pos, icon_ptr->y_pos, icon_ptr->width, icon_ptr->height));
#endif
                  /* Interpret the data as a JPEG image */
                  if (!STB_IMGConvertJPEG(si_icon->icon_data, si_icon->data_len,
                         &icon_ptr->icon_data, &icon_ptr->data_len, &icon_ptr->width, &icon_ptr->height))
                  {
#ifdef DEBUG_SI_RCT
                     AP_SI_PRINT(("ProcessImageIcon: Failed to process JPEG icon"));
#endif
                     DBDEF_DeleteImageIcons(icon_ptr);
                     icon_ptr = NULL;
                  }
               }
            }
         }
      }
   }

   FUNCTION_FINISH(ProcessImageIcon);

   return(icon_ptr);
}

#if defined(INCLUDE_DSMCC_FILE_REQUEST)
/*!**************************************************************************
 * @brief   Callback function for icons, passed to DSM file request, that processes
 *          an icon image when it's been received from the DSM carousel.
 * @param   user_data - pointer to the icon that this data is for
 * @param   file_data - data received from DSM-CC
 * @param   data_size - number of bytes received from DSM-CC
 ****************************************************************************/
static void IconFileReceived(E_FsStatus status, S_CONTENT *pContent)
{
   ADB_IMAGE_ICON *icon_ptr;

   FUNCTION_START(IconFileReceived);
   ASSERT( pContent != NULL );

   icon_ptr = (ADB_IMAGE_ICON *)pContent->user_data;
   if (status == FS_STATUS_OK)
   {
      ASSERT((pContent->data != NULL) && (pContent->size > 0));
      /* Process the image received based on the type of icon */
      if (icon_ptr->icon_type == ICON_TYPE_PNG)
      {
#ifdef DEBUG_SI_RCT
         AP_SI_PRINT(("IconFileReceived: PNG icon \"%s\" received", icon_ptr->icon_url));
#endif
         /* Interpret the data as a PNG image */
         if (!STB_IMGConvertPNG(pContent->data, pContent->size, &icon_ptr->icon_data,
            &icon_ptr->data_len, &icon_ptr->width, &icon_ptr->height))
         {
#ifdef DEBUG_SI_RCT
            AP_SI_PRINT(("IconFileReceived: Failed to process PNG icon"));
#endif
         }
      }
      else
      {
#ifdef DEBUG_SI_RCT
         AP_SI_PRINT(("IconFileReceived: JPEG icon \"%s\" received", icon_ptr->icon_url));
#endif
         /* Interpret the data as a JPEG image */
         if (!STB_IMGConvertJPEG(pContent->data, pContent->size, &icon_ptr->icon_data,
            &icon_ptr->data_len, &icon_ptr->width, &icon_ptr->height))
         {
#ifdef DEBUG_SI_RCT
            AP_SI_PRINT(("IconFileReceived: Failed to process JPEG icon"));
#endif
         }
      }
      ASSERT( pContent->destroy != NULL );
      pContent->destroy( pContent->fs_handle );
      icon_ptr->destroy_func = NULL;
      icon_ptr->dsm_handle = NULL;
   }
#ifdef DEBUG_SI_RCT
   else
   {
      AP_SI_PRINT(("IconFileReceived: Failed to obtain icon file \"%s\", status %u",
         icon_ptr->icon_url, status));
   }
#endif

   FUNCTION_FINISH(IconFileReceived);
}

static void IconFileRequest(ADB_IMAGE_ICON *icon_ptr,U8BIT path)
{
   E_FsStatus status;
   H_DsmControl dsmctrl;
   S_CONTENT dsm;

   FUNCTION_START(IconFileRequest);

   /* If the icon is signalled as being available from the MHEG
    * carousel, request the file from MHEG */
   if ((icon_ptr->transport_mode == ICON_TRANS_URL) &&
       (icon_ptr->icon_url != NULL))
   {
#ifdef DEBUG_SI_RCT
      AP_SI_PRINT(("IconFileRequest: Requesting icon file \"%s\"", icon_ptr->icon_url));
#endif
      dsm.user_data = icon_ptr;
      dsmctrl = DSMCC_FindInstance(current_service_rec[path]->serv_id,STB_DPGetPathDemux(path));
      if (dsmctrl != NULL)
      {
         status = DSMCC_ClientLoadObject(dsmctrl, icon_ptr->icon_url,
               LOAD_FLAGS_DEFAULT, IconFileReceived, &dsm);
         if (status == FS_STATUS_PENDING)
         {
            icon_ptr->destroy_func = dsm.destroy;
            icon_ptr->dsm_handle = dsm.fs_handle;
         }
         else if (status == FS_STATUS_OK)
         {
            if (icon_ptr->icon_type == ICON_TYPE_PNG)
            {
   #ifdef DEBUG_SI_RCT
               AP_SI_PRINT(("IconFileRequest: PNG icon received"));
   #endif
               /* Interpret the data as a PNG image */
               STB_IMGConvertPNG(dsm.data, dsm.size, &icon_ptr->icon_data, &icon_ptr->data_len,
                  &icon_ptr->width, &icon_ptr->height);
            }
            else
            {
   #ifdef DEBUG_SI_RCT
               AP_SI_PRINT(("IconFileRequest: JPEG icon received"));
   #endif
               /* Interpret the data as a JPEG image */
               STB_IMGConvertJPEG(dsm.data, dsm.size, &icon_ptr->icon_data, &icon_ptr->data_len,
                  &icon_ptr->width, &icon_ptr->height);
            }
            ASSERT( dsm.destroy != NULL );
            dsm.destroy( dsm.fs_handle );
         }
      }
   }

   FUNCTION_FINISH(IconFileRequest);
}

#endif /*INCLUDE_DSMCC_FILE_REQUEST*/

static void ProcessEitTable(SI_TABLE_RECORD *table_rec, BOOLEAN ignore_version,
   E_DB_ACCESS_MODE mode, BOOLEAN playback)
{
   U8BIT table_id;
   U8BIT version;
   U16BIT serv_id;
   U16BIT tran_id;
   U16BIT orig_net_id;
   SI_SECTION_RECORD *section_rec;
   U8BIT *data_ptr;
   U8BIT *data_end;
   U16BIT dloop_len;
   U8BIT *dloop_end;
   U16BIT dlen;
   U32BIT priv_data_code;
   ADB_SERVICE_REC *s_ptr;
   ADB_SERVICE_REC *s2_ptr;
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_DESC *event_desc;
   U16BIT i;
   U8BIT path;
   BOOLEAN abort;
   BOOLEAN now_next_updated;
   BOOLEAN sched_updated;
   U32DHMS gmt;
   U32DHMS start_date_time;
   U32DHMS limit_date_time;
   U8BIT sect_num, seg_last_sect_num, segment;
   SI_SECTION_RECORD *section_ptr;
   BOOLEAN complete_segment;
   U16BIT offset_hours;
   U32DHMS segment_start;

   FUNCTION_START(ProcessEitTable);

   if (table_rec != NULL)
   {
      now_next_updated = FALSE;
      sched_updated = FALSE;
      s_ptr = NULL;

      table_id = table_rec->tid;
      if ((table_id == EITPF_ACTUAL_TID) || (table_id == EITPF_PLUS_TID) ||
          ((playback == FALSE) && ((table_id == EITPF_OTHER_TID) ||
          ((table_id & 0xf0) == EITSC_ACTUAL_TID) || ((table_id & 0xf0) == EITSC_OTHER_TID))))
      {
         path = table_rec->path;

         version = table_rec->version;
         serv_id = table_rec->xtid;

#ifdef DEBUG_SI_EIT
         AP_SI_PRINT(("EIT: tid=0x%02x, serv=0x%04x, version=%u", table_id, serv_id, version));
#endif

         /* Get the transport and network IDs from the first section */
         section_rec = table_rec->section_list;
         if (section_rec != NULL)
         {
            /* Get pointer to start of section data */
            data_ptr = &(section_rec->data_start);

            /* Read transport id, original network id and last table id. If there is more than
             * one section all sections should contain the same value. */
            tran_id = (data_ptr[8] << 8) | data_ptr[9];
            orig_net_id = (data_ptr[10] << 8) | data_ptr[11];

            /* Get access to the database (may suspend) */
            DBDEF_RequestAccess();

            if (playback == TRUE)
            {
               s_ptr = current_service_rec[path];
            }
            else
            {
               /* Find service this event table refers to */
               if (mode == DB_ACCESS_SEARCH)
               {
                  /* EIT actual table only so restrict search to current transport */
                  s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                        current_transport_rec[path]->orig_net_id, current_transport_rec[path]->tran_id,
                        serv_id);
               }
               else
               {
                  /* Actual and other tables - find service by ids.
                   * IMPORTANT NOTE - in a multi-network system there may be more than one service
                   * matching the ids - must copy the events to all of them */
                  s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, orig_net_id,
                        tran_id, serv_id);
               }
            }

            if (s_ptr != NULL)
            {
               /* If EIT events have been limited, calculate the limit */
               if (eit_schedule_limit == 0)
               {
                  limit_date_time = 0;
               }
               else
               {
                  gmt = STB_GCNowDHMSGmt();

                  if (gmt == 0)
                  {
                     /* System date/time not yet known so can't calculate the limit */
                     limit_date_time = 0;
                  }
                  else
                  {
                     limit_date_time = STB_GCCalculateDHMS(gmt,
                           DHMS_CREATE(0, eit_schedule_limit, 0, 0), CALC_ADD);
                  }
               }

               /* Delete any events that are out of date */
               if (DeleteOutOfDateEvents(s_ptr))
               {
                  sched_updated = TRUE;
               }

               /* Loop through all sections in the list - they will be in section order */
               abort = FALSE;
               section_rec = table_rec->section_list;
               while ((section_rec != NULL) && !abort)
               {
                  /* Get pointer to section data */
                  data_ptr = &(section_rec->data_start);

                  /* If a complete segment has arrived, then any existing events covered by the
                   * period it represents can be deleted because the new events will replace them */
                  sect_num = data_ptr[6];

                  if (((sect_num % 8) == 0) &&
                     (((table_id & 0xf0) == EITSC_ACTUAL_TID) || ((table_id & 0xf0) == EITSC_OTHER_TID)))
                  {
                     /* Start of a segment, check whether the rest of the sections making up the
                      * segment are present */
                     segment = sect_num / 8;
                     seg_last_sect_num = data_ptr[12];
                     complete_segment = FALSE;

                     if (sect_num == seg_last_sect_num)
                     {
                        complete_segment = TRUE;
                     }
                     else
                     {
                        section_ptr = section_rec->next;
                        sect_num++;

                        /* Check the rest of the sections to see if the segment is complete */
                        while (!complete_segment && (section_ptr != NULL))
                        {
                           data_ptr = &(section_ptr->data_start);

                           /* Check this is the next expected section number */
                           if (data_ptr[6] == sect_num)
                           {
                              if (data_ptr[6] == seg_last_sect_num)
                              {
                                 /* Last section in the segment */
                                 complete_segment = TRUE;
                              }
                              else
                              {
                                 /* More sections in the segment, set the next section number */
                                 sect_num++;
                                 section_ptr = section_ptr->next;
                              }
                           }
                           else
                           {
                              /* The number of the next section isn't the expected one,
                               * so the sections provided don't make up a segment */
                              section_ptr = NULL;
                           }
                        }
                     }

                     if (complete_segment)
                     {
                        /* A complete segment is available, so the appropriate events can be
                         * deleted  because they're about to be updated.
                         * Each segment represents a 3 hour time slot, with 32 segments
                         * per sub-table, which is 4 days of events, so calculate the offset
                         * to the start of the segment in hours */
                        if ((table_id & 0xf0) == EITSC_ACTUAL_TID)
                        {
                           offset_hours = ((table_id - EITSC_ACTUAL_TID) * 32 + segment) * 3;
                        }
                        else
                        {
                           offset_hours = ((table_id - EITSC_OTHER_TID) * 32 + segment) * 3;
                        }

                        /* The schedule starts from midnight today */
                        segment_start = STB_GCCalculateDHMS(STB_GCCreateDHMS(STB_GCGetGMTDate(), 0, 0, 0),
                           STB_GCCreateDHMS(offset_hours / 24, offset_hours % 24, 0, 0), CALC_ADD);

                        /* Delete the events for the 3 hour period represented by this segment */
                        if (DeleteEventsForPeriod(s_ptr, segment_start,
                           STB_GCCalculateDHMS(segment_start, STB_GCCreateDHMS(0, 3, 0, 0), CALC_ADD)))
                        {
                           sched_updated = TRUE;
                        }
                     }
                  }

                  /* Get pointer to section data and end of section */
                  data_ptr = &(section_rec->data_start);
                  data_end = data_ptr + section_rec->data_len - 4;   // -4 for crc

                  /* Skip section header and IDs already read */
                  data_ptr += 14;

                  /* Read entry for each event */
                  while ((data_ptr < data_end) && !abort)
                  {
                     start_date_time = ReadEventStart(data_ptr);

                     if ((limit_date_time == 0) || (start_date_time <= limit_date_time))
                     {
                        e_ptr = STB_AppGetMemory(sizeof(ADB_EVENT_REC));
                        if (e_ptr != NULL)
                        {
                           memset(e_ptr, 0, sizeof(ADB_EVENT_REC));

                           e_ptr->event_id = (data_ptr[0] << 8) | data_ptr[1];
                           e_ptr->start = ReadEventStart(data_ptr);
                           e_ptr->duration = ReadEventDuration(data_ptr);
                           e_ptr->running_status = data_ptr[10] >> 5;
                           e_ptr->free_to_air = ((data_ptr[10] & 0x10) == 0);
                           e_ptr->version = version;

                           dloop_len = ((data_ptr[10] & 0x0f) << 8) | data_ptr[11];
                           data_ptr += 12;

#ifdef DEBUG_SI_EIT
                           AP_SI_PRINT((" event_id=%u, start=%u, duration=%u, len=%u",
                              e_ptr->event_id, e_ptr->start, e_ptr->duration, dloop_len));
#endif

                           priv_data_code = 0;
                           dloop_end = data_ptr + dloop_len;
                           while (data_ptr < dloop_end)
                           {
#ifdef DEBUG_SI_EIT
                              AP_SI_PRINT(("  dtag=0x%02x, dlen=%u", data_ptr[0], data_ptr[1]));
#endif
                              dlen = data_ptr[1];

                              /* Parse any descriptors here that contain data needed by DVBCore */
                              if (data_ptr[0] == PRIVATE_DATA_SPEC_DTAG)
                              {
                                 /* Keep track of the private data specifier so it can be passed
                                  * to the EIT parsing callback function */
                                 if (dlen == 4)
                                 {
                                    priv_data_code = (data_ptr[2] << 24) | (data_ptr[3] << 16) |
                                       (data_ptr[4] << 8) | data_ptr[5];
                                 }
                              }
                              else if (data_ptr[0] == FTA_CONTENT_DESC_DTAG)
                              {
                                 if (dlen == 1)
                                 {
                                    e_ptr->has_content_management_desc = TRUE;
                                    if ((data_ptr[2] & 0x08) != 0)
                                    {
                                       e_ptr->do_not_scramble = TRUE;
                                    }
                                    else
                                    {
                                       e_ptr->do_not_scramble = FALSE;
                                    }
                                 }
                              }
                              else
                              {
                                 /* Pass the descriptor to the registered callback function
                                  * so it can pull out any info it needs to */
                                 if (eit_parser_func != NULL)
                                 {
                                    (*eit_parser_func)(data_ptr[0], (U8BIT)dlen, data_ptr + 2, priv_data_code, e_ptr);
                                 }
                              }

                              /* Length is +2 for dtag and dlen */
                              dlen += 2;

                              event_desc = (ADB_EVENT_DESC *)STB_AppGetMemory(sizeof(ADB_EVENT_DESC) + dlen);
                              if (event_desc != NULL)
                              {
                                 event_desc->desc_data = (U8BIT *)event_desc + sizeof(ADB_EVENT_DESC);
                                 if (event_desc->desc_data != NULL)
                                 {
                                    memcpy(event_desc->desc_data, data_ptr, dlen);

                                    event_desc->next = NULL;

                                    if (e_ptr->desc_list_head == NULL)
                                    {
                                       e_ptr->desc_list_head = event_desc;
                                    }
                                    else
                                    {
                                       e_ptr->desc_list_tail->next = event_desc;
                                    }

                                    e_ptr->desc_list_tail = event_desc;
                                 }
                                 else
                                 {
                                    STB_AppFreeMemory(event_desc);
                                 }
                              }

                              data_ptr += dlen;
                           }

                           /* Add the event into the event list */
                           if ((table_id == EITPF_ACTUAL_TID) || (table_id == EITPF_OTHER_TID) ||
                               (table_id == EITPF_PLUS_TID))
                           {
                              if (section_rec->sect_num == 0)
                              {
                                 if (ignore_version || (s_ptr->now_event == NULL) ||
                                     ((s_ptr->now_event != NULL) &&
                                      ((s_ptr->now_event->event_id != e_ptr->event_id) ||
                                       (s_ptr->now_event->version != version))))
                                 {
                                    /* Now event */
                                    if (s_ptr->now_event != NULL)
                                    {
                                       DBDEF_DeleteEventList(s_ptr->now_event);
                                    }

                                    s_ptr->now_event = e_ptr;
                                    e_ptr = NULL;
                                    now_next_updated = TRUE;
                                 }
                              }
                              else if (section_rec->sect_num == 1)
                              {
                                 if (ignore_version || (s_ptr->next_event == NULL) ||
                                     ((s_ptr->next_event != NULL) &&
                                      ((s_ptr->next_event->event_id != e_ptr->event_id) ||
                                       (s_ptr->next_event->version != version))))
                                 {
                                    /* Next event */
                                    if (s_ptr->next_event != NULL)
                                    {
                                       DBDEF_DeleteEventList(s_ptr->next_event);
                                       s_ptr->next_event = NULL;
                                    }

                                    s_ptr->next_event = e_ptr;
                                    e_ptr = NULL;
                                    now_next_updated = TRUE;
                                 }
                              }
                              else if (table_id == EITPF_PLUS_TID)
                              {
                                 /* Freesat EITpf++ can have more than just the now/next events
                                  * in the pf++ table, so use it to update the schedule */
                                 if (UpdateEvents(e_ptr, s_ptr, TRUE))
                                 {
                                    e_ptr = NULL;
                                    sched_updated = TRUE;
                                 }
                              }
                           }
                           else
                           {
                              /* Insert this event into the event list */
                              if (UpdateEvents(e_ptr, s_ptr, FALSE))
                              {
                                 e_ptr = NULL;
                                 sched_updated = TRUE;
                              }
                           }

                           if (e_ptr != NULL)
                           {
                              /* The event hasn't been used so needs to be deleted */
                              DBDEF_DeleteEventList(e_ptr);
                              e_ptr = NULL;
                           }

                           /* In the update case check if there are any other services matching the same ids */
                           if ((mode == DB_ACCESS_UPDATE) && !playback)
                           {
                              s2_ptr = DBDEF_FindServiceRecByIds(s_ptr, ADB_INVALID_DVB_ID,
                                    orig_net_id, tran_id, serv_id);

                              while (s2_ptr != NULL)
                              {
                                 s2_ptr->now_event = s_ptr->now_event;
                                 s2_ptr->next_event = s_ptr->next_event;
                                 s2_ptr->event_schedule = s_ptr->event_schedule;

                                 /* Any more services? */
                                 s2_ptr = DBDEF_FindServiceRecByIds(s2_ptr, ADB_INVALID_DVB_ID,
                                       orig_net_id, tran_id, serv_id);
                              }
                           }
                        }
                        else
                        {
                           /* Failed to allocate memory so abort */
                           abort = TRUE;
                        }
                     }
                     else
                     {
                        /* Events are broadcast in chronological order, so any remaining events
                         * will also be outside the limit and can be skipped */
                        data_ptr = data_end;
                     }
                  }

                  /* Move on to next section */
                  section_rec = section_rec->next;
               }
            }

            /* If eit list is in use find entry corresponding to this service and mark it as done */
            if ((mode == DB_ACCESS_SEARCH) && (eit_list[path] != NULL) && (playback == FALSE))
            {
               for (i = 0; i < num_eit_list_entries[path]; i++)
               {
                  if (eit_list[path][i].serv_id == serv_id)
                  {
                     eit_list[path][i].got_eit = TRUE;
                     break;
                  }
               }
            }

            // release database access
            DBDEF_ReleaseAccess();
         }

         if (update_eit_func != NULL)
         {
            /* Call the function that's been registered to handle additional EIT processing */
            (*update_eit_func)(path, table_rec);
         }
      }

      // release eit table
      STB_SIReleaseTableRecord(table_rec);

      if (now_next_updated && (s_ptr->now_event != NULL))
      {
#ifdef NOW_EVENTS_LATCHED
         /* Send this event as a latched event. This means that if this type of event has already
          * been sent but it hasn't been dealt with then this one won't be queued. This prevents
          * the queue being flooded with events of this type, which can happen after startup. */
         STB_ERSendEvent(TRUE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_EIT_NOW_UPDATE, NULL, 0);
#else
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_EIT_NOW_UPDATE,
            &s_ptr, sizeof(s_ptr));
#endif

#ifdef INTEGRATE_HBBTV
         if (s_ptr == current_service_rec[path])
         {
            HBBTV_NotifyProgrammeChanged();
         }
#endif
      }

      if (sched_updated)
      {
         /* Send this event as a latched event. This means that if this type of event has already
          * been sent but it hasn't been dealt with then this one won't be queued. This prevents
          * the queue being flooded with events of this type, which can happen after startup. */
         STB_ERSendEvent(TRUE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_EIT_SCHED_UPDATE, NULL, 0);
      }
   }

   FUNCTION_FINISH(ProcessEitTable);
}

static U32DHMS ReadEventStart(U8BIT *data_ptr)
{
   U16BIT date;
   U8BIT hrs, mns, scs;

   FUNCTION_START(ReadEventStart);

   date = (data_ptr[2] << 8) | data_ptr[3];
   hrs = ((data_ptr[4] >> 4) * 10) + (data_ptr[4] & 0x0f);
   mns = ((data_ptr[5] >> 4) * 10) + (data_ptr[5] & 0x0f);
   scs = ((data_ptr[6] >> 4) * 10) + (data_ptr[6] & 0x0f);

   FUNCTION_FINISH(ReadEventStart);

   return DHMS_CREATE(date, hrs, mns, scs);
}

static U32DHMS ReadEventDuration(U8BIT *data_ptr)
{
   U32BIT days;
   U8BIT hrs, mns, scs;

   FUNCTION_START(ReadEventDuration);

   hrs = ((data_ptr[7] >> 4) * 10) + (data_ptr[7] & 0x0f);
   mns = ((data_ptr[8] >> 4) * 10) + (data_ptr[8] & 0x0f);
   scs = ((data_ptr[9] >> 4) * 10) + (data_ptr[9] & 0x0f);
   if (hrs > 23)
   {
     days = hrs / 24;
     hrs = hrs % 24;
   }
   else
   {
      days = 0;
   }

   FUNCTION_FINISH(ReadEventDuration);

   return DHMS_CREATE(days, hrs, mns, scs);
}

static BOOLEAN DeleteOutOfDateEvents(ADB_SERVICE_REC *s_ptr)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *prev_e_ptr;
   ADB_EVENT_REC *expired_e_ptr;
   U32DHMS now;
   U32DHMS end_date_time;
   BOOLEAN sched_updated = FALSE;

   /* Delete any events that are out of date */
   if (s_ptr->event_schedule != NULL)
   {
      prev_e_ptr = NULL;
      e_ptr = s_ptr->event_schedule;

      now = STB_GCNowDHMSGmt();

      do
      {
         end_date_time = STB_GCCalculateDHMS(e_ptr->start, e_ptr->duration, CALC_ADD);

         if (end_date_time < now)
         {
            /* Event has expired so test the next event */
            prev_e_ptr = e_ptr;
            e_ptr = e_ptr->next;
            s_ptr->num_events_in_schedule--;
            sched_updated = TRUE;
         }
         else
         {
            break;
         }
      }
      while (e_ptr != NULL);

      if (prev_e_ptr != NULL)
      {
         /* At this point, prev_e_ptr is pointing at the last event to be deleted
          * from the event list, so the list is terminated to delete these events */
         prev_e_ptr->next = NULL;

         /* keep the pointer to the expired part of the list */
         expired_e_ptr = s_ptr->event_schedule;

         /* e_ptr is the first event that hasn't expired so this now becomes the
          * first event in the schedule */
         s_ptr->event_schedule = e_ptr;

         /* notify about the expired events */
         e_ptr = expired_e_ptr;
         while (e_ptr)
         {
#ifdef DEBUG_SI_EIT
            AP_SI_PRINT(("[-] EIT %d v%d for LCN %d: deleting (expire)",
                  (int)e_ptr->event_id, (int)e_ptr->version,
                  (int)s_ptr->serv_lcn));
#endif
            ASI_NotifyEitSchedUpdate(s_ptr, e_ptr->event_id,
                  APP_SI_EIT_JOURNAL_TYPE_EXPIRE);
            e_ptr = e_ptr->next;
         }

         /* delete expired events */
         DBDEF_DeleteEventList(expired_e_ptr);
      }
   }
   return sched_updated;
}

static inline U32DHMS start(const ADB_EVENT_REC *e)
{
   return e->start;
}

static inline U32DHMS end(const ADB_EVENT_REC *e)
{
   return STB_GCCalculateDHMS(e->start, e->duration, CALC_ADD);
}

static inline BOOLEAN before(const ADB_EVENT_REC *e1, const ADB_EVENT_REC *e2)
{
   return start(e1) <= start(e2);
}

static inline BOOLEAN after(const ADB_EVENT_REC *e1, const ADB_EVENT_REC *e2)
{
   return before(e2, e1);
}

static inline void push(ADB_EVENT_REC **list, ADB_EVENT_REC *event)
{
   event->next = *list;
   *list = event;
}

static inline ADB_EVENT_REC *pop(ADB_EVENT_REC **list)
{
   ADB_EVENT_REC *event = *list;
   if (event) {
      *list = event->next;
      event->next = NULL;
   }
   /* else *list is already NULL because the last event->next was NULL
    * or *list was NULL to start with
    */

   return event;
}

static BOOLEAN DeleteEventsForPeriod(ADB_SERVICE_REC *s_ptr, U32DHMS start_time, U32DHMS end_time)
{
   ADB_EVENT_REC *e_ptr;
   ADB_EVENT_REC *prev_ptr;
   ADB_EVENT_REC *delete_ptr;
   ADB_EVENT_REC *prev_delete_ptr;
   BOOLEAN events_deleted;

   events_deleted = FALSE;
   prev_ptr = NULL;

#ifdef DEBUG_SI_EIT
   AP_SI_PRINT(("%s(%u): %u@%02u:%02u - %u@%02u:%02u", __FUNCTION__, s_ptr->serv_lcn,
      DHMS_DATE(start_time), DHMS_HOUR(start_time), DHMS_MINS(start_time),
      DHMS_DATE(end_time), DHMS_HOUR(end_time), DHMS_MINS(end_time)));
#endif
   e_ptr = s_ptr->event_schedule;
   while (e_ptr != NULL)
   {
      if (start(e_ptr) >= end_time)
      {
         /* No more events to delete */
         e_ptr = NULL;
      }
      else if (start(e_ptr) < start_time)
      {
         /* Event precedes the events to be deleted */
         prev_ptr = e_ptr;
         e_ptr = e_ptr->next;
      }
      else
      {
         /* Found the first event to be deleted.
          * Find the last event so they can all be deleted together */
         delete_ptr = e_ptr;
         if (prev_ptr != NULL)
         {
            prev_delete_ptr = prev_ptr;
         }
         else
         {
            prev_delete_ptr = NULL;
         }

         while ((e_ptr != NULL) && (start(e_ptr) < end_time))
         {
#ifdef DEBUG_SI_EIT
            AP_SI_PRINT(("[-] EIT %u v%u for LCN %u, %u@%02u:%02u - deleting",
                  e_ptr->event_id, e_ptr->version, s_ptr->serv_lcn,
                  DHMS_DATE(e_ptr->start), DHMS_HOUR(e_ptr->start), DHMS_MINS(e_ptr->start)));
#endif
            /* Notify that this event is going to be deleted */
            ASI_NotifyEitSchedUpdate(s_ptr, e_ptr->event_id, APP_SI_EIT_JOURNAL_TYPE_DELETE);

            prev_ptr = e_ptr;
            e_ptr = e_ptr->next;
         }

         /* Terminate the list of events to be deleted */
         prev_ptr->next = NULL;

         if (prev_delete_ptr == NULL)
         {
            s_ptr->event_schedule = e_ptr;
         }
         else
         {
            prev_delete_ptr->next = e_ptr;
         }

         DBDEF_DeleteEventList(delete_ptr);
         events_deleted = TRUE;
         e_ptr = NULL;
      }
   }

   return(events_deleted);
}

static ADB_EVENT_REC ** PruneEvents(ADB_EVENT_REC *event_ptr,
      ADB_SERVICE_REC *s_ptr, BOOLEAN update_only, BOOLEAN *is_update)
{
   ADB_EVENT_REC **walk_ptr = &(s_ptr->event_schedule);
   ADB_EVENT_REC **add_ptr = walk_ptr;
   ADB_EVENT_REC *prev_ptr;
   ADB_EVENT_REC * temp;
   BOOLEAN past_event = FALSE;
   BOOLEAN found_match = FALSE;
   BOOLEAN old_version;
   BOOLEAN overlap;

   FUNCTION_START(PruneEvents);

   /* walk through the event list and:
    * 1. find insertion point for this event
    * 2. delete all colliding items, including the old version, if present
    * The walk ends when we've found and deleted all overlapping items
    * (i.e. we past the end time of the event) and we've found and deleted
    * the old version of this event or we reached the end of the list.
    */
   prev_ptr = NULL;

   while ((*walk_ptr != NULL) && !(found_match && past_event))
   {
#if 0
      AP_SI_PRINT(("Found EIT event %d v%d, (%u-%u)",
            (int)(*walk_ptr)->event_id, (int)(*walk_ptr)->version,
            (unsigned)start(*walk_ptr), (unsigned)end(*walk_ptr)));
#endif

      /* we need to search the entire list for the original instance
       * of the event because the update can change the time and/or duration.
       */
      old_version = FALSE;
      if (!found_match && ((*walk_ptr)->event_id == event_ptr->event_id))
      {
         found_match = TRUE;
         if ((*walk_ptr)->version == event_ptr->version)
         {
#ifdef DEBUG_SI_EIT
            AP_SI_PRINT(("EIT event %d v%d is a duplicate",
                  (int)event_ptr->event_id, (int)event_ptr->version));
#endif
            add_ptr = NULL; /* don't add this event - we've got it already */
            break; /* stop the search */
         }
         else
         {
#ifdef DEBUG_SI_EIT
            AP_SI_PRINT(("EIT event update %d: v%d->v%d,  (%u-%u)->(%u-%u)",
                  (int)event_ptr->event_id,
                  (int)(*walk_ptr)->version, (int)event_ptr->version,
                  (unsigned)start(event_ptr), (unsigned)end(event_ptr),
                  (unsigned)start(*walk_ptr), (unsigned)end(*walk_ptr)
                  ));
#endif
            old_version = TRUE;
         }
      }

      /* check event time overlap and delete overlapping items;
       * this is sorted list so once we passed the end time of the event
       * all subsequent items are also after the event and no further
       * checks are needed because they can't overlap with received event.
       */
      overlap = FALSE;
      if (!past_event)
      {
         if (before(*walk_ptr, event_ptr))
         {
            /* possible insertion point after this item */
            add_ptr = &((*walk_ptr)->next);
         }
         else  /* possible collision - time overlaps */
         {
            past_event = after(*walk_ptr, event_ptr);
         }
      }
      /* else we're past the end time of the event - no overlap possible */

      /* delete unwanted item */
      if (old_version || (overlap && !update_only))
      {
         /* delete this item from the list */
         temp = pop(walk_ptr);
#ifdef DEBUG_SI_EIT
         AP_SI_PRINT(("[-] EIT %d v%d for LCN %d: deleting %s",
               (int)temp->event_id, (int)temp->version,
               (int)s_ptr->serv_lcn,
               found_match ? "(update)" : "(overlap)"));
#endif

         if (!found_match)
         {
            ASI_NotifyEitSchedUpdate(s_ptr, temp->event_id,
                  APP_SI_EIT_JOURNAL_TYPE_DELETE);
         }
         /* else send a single "update" instead of "del", "add" */

         DBDEF_DeleteEventList(temp);
         s_ptr->num_events_in_schedule--;

         if (prev_ptr != NULL)
         {
            add_ptr = &(prev_ptr->next);
         }
         else
         {
            add_ptr = &(s_ptr->event_schedule);
         }
      }
      else /* skip over this item and keep going */
      {
         prev_ptr = *walk_ptr;
         walk_ptr = &((*walk_ptr)->next);
      }
   }
   *is_update = found_match;

   FUNCTION_FINISH(PruneEvents);

   return add_ptr;
}

static BOOLEAN UpdateEvents(ADB_EVENT_REC *event_ptr, ADB_SERVICE_REC *s_ptr, BOOLEAN update_only)
{
   BOOLEAN event_used;
   ADB_EVENT_REC **add_ptr = NULL;
   BOOLEAN is_update = FALSE;

   FUNCTION_START(UpdateEvents);

   event_used = FALSE;

#ifdef DEBUG_SI_EIT
   AP_SI_PRINT(("EIT received event %u v%u for service %u,%u,%u LCN %u, (%u@%02u:%02u - %u@%02u:%02u)",
         event_ptr->event_id, event_ptr->version,
         s_ptr->transport->orig_net_id, s_ptr->transport->tran_id,
         s_ptr->serv_id, s_ptr->serv_lcn,
         DHMS_DATE(event_ptr->start), DHMS_HOUR(event_ptr->start), DHMS_MINS(event_ptr->start),
         DHMS_DATE(end(event_ptr)), DHMS_HOUR(end(event_ptr)), DHMS_MINS(end(event_ptr))));
#endif

   if (end(event_ptr) > STB_GCNowDHMSGmt())
   {
      add_ptr = PruneEvents(event_ptr, s_ptr, update_only, &is_update);
      if (add_ptr != NULL)
      {
#ifdef DEBUG_SI_EIT
         AP_SI_PRINT(("[+] EIT %u v%u for LCN %u: adding %s",
               event_ptr->event_id, event_ptr->version,
               s_ptr->serv_lcn,
               is_update ? "(update)" : "(new)"));
#endif
         push(add_ptr, event_ptr);
         s_ptr->num_events_in_schedule++;

         ASI_NotifyEitSchedUpdate(s_ptr, event_ptr->event_id, is_update ?
                     APP_SI_EIT_JOURNAL_TYPE_UPDATE :
                     APP_SI_EIT_JOURNAL_TYPE_ADD);

         event_used = TRUE;
      }
#ifdef DEBUG_SI_EIT
      else
      {
         AP_SI_PRINT(("[ ] EIT %u v%u for LCN %u: discarding, duplicate",
            event_ptr->event_id, event_ptr->version, s_ptr->serv_lcn));
      }
#endif
   }
#ifdef DEBUG_SI_EIT
   else
   {
      AP_SI_PRINT(("[ ] EIT %u v%u for LCN %u: discarding, ends in the past (%lu now)",
            event_ptr->event_id, event_ptr->version, s_ptr->serv_lcn,
            STB_GCNowDHMSGmt()));
   }
#endif
   FUNCTION_FINISH(UpdateEvents);

   return(event_used);
}

/**
 *

 *
 * @brief   Sets up pmt table request for the pmt specified by pmt_list[path][pmt_list_id[path]].
 *                If a pmt filter exists with the same pid then modifies the filter, otherwise
 *                sets up a new filter
 *

 *

 *
 */
static BOOLEAN MakeNewPmtRequest(U8BIT path)
{
   BOOLEAN service_found;
   U16BIT list_id;
   U16BIT i;

   FUNCTION_START(MakeNewPmtRequest);

   service_found = FALSE;

   if ((pmt_request_mode[path] == PMT_REQUEST_CURRENT) && (current_service_rec[path] != NULL))
   {
      list_id = 0;
      for (i = 0; i < num_pmt_list_entries[path]; i++)
      {
         if (pmt_list[path][i].serv_id == current_service_rec[path]->serv_id)
         {
            list_id = i;
            service_found = TRUE;
            break;
         }
      }
   }
   else if (pmt_request_mode[path] == PMT_REQUEST_PRIORITY)
   {
      list_id = GetPriorityListId(path);
   }
   else
   {
      list_id = pmt_list_id[path];
   }

   if ((pmt_filter[path] != NULL) && (pmt_filter_pid[path] == pmt_list[path][list_id].pid))
   {
      #ifdef DEBUG_SI_UPDATE
      AP_SI_PRINT(("Update(%u): PMT request (sid 0x%04x)", path, pmt_list[path][list_id].serv_id));
      #endif
      // filter exists with same pid - modify filter
      STB_SIModifyPmtRequest(pmt_filter[path], pmt_list[path][list_id].serv_id, 0xffff, 1);
      pmt_start_timestamp[path] = STB_OSGetClockMilliseconds();
      pmt_update_timestamp[path] = 0;
   }
   else
   {
      /* Different pid or filter doesn't exist yet - either way we need a new filter.
       * If a filter exists, cancel it, then request the new filter */
      if (pmt_filter[path] != NULL)
      {
         STB_SICancelTableRequest(pmt_filter[path]);
         pmt_filter[path] = NULL;
      }

      #ifdef DEBUG_SI_UPDATE
      AP_SI_PRINT(("Update(%u): PMT request (sid 0x%04x)", path, pmt_list[path][list_id].serv_id));
      #endif
      pmt_filter_pid[path] = pmt_list[path][list_id].pid;
      pmt_start_timestamp[path] = STB_OSGetClockMilliseconds();
      pmt_update_timestamp[path] = 0;
      pmt_filter[path] = STB_SIRequestPmt(path, ONE_SHOT_REQUEST, pmt_filter_pid[path],
            pmt_list[path][list_id].serv_id, 0xffff, 1, ReceiveSiTable, APP_SI_PMT_RECEIVED);
   }

   FUNCTION_FINISH(MakeNewPmtRequest);

   return(service_found);
}

static U16BIT GetPriorityListId(U8BIT path)
{
   U16BIT i, index;
   U16BIT list_id;
   U16BIT service_id;

   FUNCTION_START(GetPriorityListId);

   list_id = pmt_list_id[path];

   STB_OSSemaphoreWait(si_pmt_list_sem);

   if (pmt_priority_list[0] != INVALID_SERVICE_ID)
   {
      /* Find service in pmt_list */
      for (i = 0; i < num_pmt_list_entries[path]; i++)
      {
         if (pmt_list[path][i].serv_id == pmt_priority_list[0])
         {
            list_id = i;
            break;
         }
      }

      /* Rotate priority services, whether we found it or not */
      index = MAX_PMT_PRIORITY_LIST - 1;
      for (i = 1; i < MAX_PMT_PRIORITY_LIST; i++)
      {
         if (pmt_priority_list[i] == INVALID_SERVICE_ID)
         {
            index = i - 1;
            break;
         }
      }

      service_id = (U16BIT)pmt_priority_list[0];
      for (i = 0; i < index; i++)
      {
         pmt_priority_list[i] = pmt_priority_list[i + 1];
      }

      pmt_priority_list[index] = service_id;
   }

   STB_OSSemaphoreSignal(si_pmt_list_sem);

   FUNCTION_FINISH(GetPriorityListId);

   return list_id;
}

/**
 *

 *
 * @brief   Updates the signal strength value for the current transport record
 *

 *

 *
 */
static void UpdateTransportParameters(U8BIT path)
{
   U8BIT tuner;
   U8BIT sig_quality;
   U8BIT sig_strength;
   ADB_TRANSPORT_REC *t_ptr;
   E_STB_DP_TMODE tmode;
   E_STB_DP_TBWIDTH bwidth;
   E_STB_DP_TTYPE terr_type;
   E_STB_DP_CMODE cmode;
   U32BIT srate;

   FUNCTION_START(UpdateTransportParameters);

   tuner = STB_DPGetPathTuner(path);
   if (tuner != INVALID_RES_ID)
   {
      if ((t_ptr = current_transport_rec[path]) != NULL)
      {
         DBDEF_RequestAccess();

         switch (t_ptr->sig_type)
         {
            case SIGNAL_COFDM:
            {
               switch (STB_TuneGetActualTerrMode(tuner))
               {
                  case TUNE_MODE_COFDM_1K:   {tmode = MODE_COFDM_1K; break; }
                  case TUNE_MODE_COFDM_2K:   {tmode = MODE_COFDM_2K; break; }
                  case TUNE_MODE_COFDM_4K:   {tmode = MODE_COFDM_4K; break; }
                  case TUNE_MODE_COFDM_8K:   {tmode = MODE_COFDM_8K; break; }
                  case TUNE_MODE_COFDM_16K:  {tmode = MODE_COFDM_16K; break; }
                  case TUNE_MODE_COFDM_32K:  {tmode = MODE_COFDM_32K; break; }
                  default:                   {tmode = MODE_COFDM_UNDEFINED; break; }
               }
               if (t_ptr->u.terr.tmode != tmode)
               {
                  t_ptr->u.terr.tmode = tmode;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_MODE, tmode);
               }

               switch (STB_TuneGetActualTerrBwidth(tuner))
               {
                  case TUNE_TBWIDTH_8MHZ:   {bwidth = TBWIDTH_8MHZ; break; }
                  case TUNE_TBWIDTH_7MHZ:   {bwidth = TBWIDTH_7MHZ; break; }
                  case TUNE_TBWIDTH_6MHZ:   {bwidth = TBWIDTH_6MHZ; break; }
                  default:                  {bwidth = 0; break; }
               }
               if (t_ptr->u.terr.bwidth != bwidth)
               {
                  t_ptr->u.terr.bwidth = bwidth;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_BWIDTH, bwidth);
               }

               switch (STB_TuneGetSystemType(tuner))
               {
                  case TUNE_SYSTEM_TYPE_DVBT2: {terr_type = TERR_TYPE_DVBT2; break; }
                  case TUNE_SYSTEM_TYPE_DVBT:  {terr_type = TERR_TYPE_DVBT;  break; }
                  default:                   {terr_type = TERR_TYPE_UNKNOWN; break; }
               }

               if (t_ptr->u.terr.terr_type != terr_type)
               {
                  t_ptr->u.terr.terr_type = terr_type;
                  DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TTRAN_TERR_TYPE, terr_type);
               }
               break;
            }
            case SIGNAL_QAM:
            {
               /* Symbol rate is stored in Ksym/sec */
               if ((srate = STB_TuneGetActualSymbolRate(tuner)) != SYMBOL_RATE_AUTO)
               {
                  srate = srate / 1000;
                  DBDEF_SetCableTransportSymbolRate(t_ptr, (U16BIT)srate);
               }

               switch (STB_TuneGetActualCableMode(tuner))
               {
                  case TUNE_MODE_QAM_4:      {cmode = MODE_QAM_4; break; }
                  case TUNE_MODE_QAM_8:      {cmode = MODE_QAM_8; break; }
                  case TUNE_MODE_QAM_16:     {cmode = MODE_QAM_16; break; }
                  case TUNE_MODE_QAM_32:     {cmode = MODE_QAM_32; break; }
                  case TUNE_MODE_QAM_64:     {cmode = MODE_QAM_64; break; }
                  case TUNE_MODE_QAM_128:    {cmode = MODE_QAM_128; break; }
                  case TUNE_MODE_QAM_256:    {cmode = MODE_QAM_256; break; }
                  default:                   {cmode = MODE_QAM_AUTO; break; }
               }

               DBDEF_SetCableTransportMode(t_ptr, cmode);
               break;
            }
            case SIGNAL_QPSK:
            {
               /* Symbol rate is stored in Ksym/sec */
               if ((srate = STB_TuneGetActualSymbolRate(tuner)) != SYMBOL_RATE_AUTO)
               {
                  srate = srate / 1000;

                  if (t_ptr->u.sat.symbol_rate != srate)
                  {
                     t_ptr->u.sat.symbol_rate = (U16BIT)srate;
                     DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SRATE, srate);
                  }
               }
               break;
            }
            default:
            {
               break;
            }
         }

         /* Get signal quality in a scale 0 to 10 */
         sig_quality = STB_TuneGetDataIntegrity(tuner);
         DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SIGNAL_QUALITY, sig_quality);

         /* Get signal strength in a scale 0 to 10 */
         sig_strength = STB_TuneGetSignalStrength(tuner);
         DBA_SetFieldValue(t_ptr->dba_rec, DBA_FIELD_TRAN_SIGNAL_STRENGTH, sig_strength);

         /* Combine quality and strength into 1 byte in a system so that signals
          * of equal quality can be compared by strength. */
         current_transport_rec[path]->signal_level_at_search = GET_SIGNAL_STATUS(sig_quality, sig_strength);

         current_transport_rec[path]->available = TRUE;

         DBA_SaveRecord(t_ptr->dba_rec);
         DBDEF_ReleaseAccess();
      }
   }

   FUNCTION_FINISH(UpdateTransportParameters);
}

/**
 *

 *
 * @brief   Releases any active table requests
 *

 *

 *
 */
static void CancelTableRequests(U8BIT path, BOOLEAN this_path_only)
{
   if (!this_path_only && (rct_filter[path] != NULL))
   {
      STB_SICancelTableRequest(rct_filter[path]);
      rct_filter[path] = NULL;

      if (current_service_rec[path] != NULL)
      {
         ADB_ServiceReleaseRCTLinks(current_service_rec[path]);
      }
   }
   if (!this_path_only && (cat_filter[path] != NULL))
   {
      STB_SICancelTableRequest(cat_filter[path]);
      cat_filter[path] = NULL;
      cat_start_timestamp[path] = 0;
   }
   if (pat_filter[path] != NULL)
   {
      STB_SICancelTableRequest(pat_filter[path]);
      pat_filter[path] = NULL;
      pat_start_timestamp[path] = 0;
   }
   if (pmt_filter[path] != NULL)
   {
      STB_SICancelTableRequest(pmt_filter[path]);
      pmt_filter[path] = NULL;
      pmt_start_timestamp[path] = 0;
   }
   if (pmt_list[path] != NULL)
   {
      STB_AppFreeMemory(pmt_list[path]);
      pmt_list[path] = NULL;
      num_pmt_list_entries[path] = 0;
   }
   if (!this_path_only && (nit_filter[path] != NULL))
   {
      STB_SICancelTableRequest(nit_filter[path]);
      nit_filter[path] = NULL;
      nit_start_timestamp[path] = 0;
   }
   if (!this_path_only && (sdt_filter[path] != NULL))
   {
      STB_SICancelTableRequest(sdt_filter[path]);
      sdt_filter[path] = NULL;
      sdt_start_timestamp[path] = 0;
   }
   if (!this_path_only && (eit_list[path] != NULL))
   {
      STB_AppFreeMemory(eit_list[path]);
      eit_list[path] = NULL;
      num_eit_list_entries[path] = 0;
   }
   if (eit_filter[path] != NULL)
   {
      STB_SICancelTableRequest(eit_filter[path]);
      eit_filter[path] = NULL;
      eit_start_timestamp[path] = 0;
   }
   if (!this_path_only && (sched_filter[path] != NULL))
   {
      STB_SICancelTableRequest(sched_filter[path]);
      sched_filter[path] = NULL;
      sched_start_timestamp[path] = 0;
   }
   if (!this_path_only && (tot_filter[path] != NULL))
   {
      STB_SICancelTableRequest(tot_filter[path]);
      tot_filter[path] = NULL;
      tot_start_timestamp[path] = 0;
   }
   if (!this_path_only && (tdt_filter[path] != NULL))
   {
      STB_SICancelTableRequest(tdt_filter[path]);
      tdt_filter[path] = NULL;
      tdt_start_timestamp[path] = 0;
   }
   if (bat_filter[path] != NULL)
   {
      STB_SICancelTableRequest(bat_filter[path]);
      bat_filter[path] = NULL;
      bat_start_timestamp[path] = 0;
   }
   if (ait_filter[path] != NULL)
   {
      STB_SICancelTableRequest(ait_filter[path]);
      ait_filter[path] = NULL;
   }
   if (unt_filter[path] != NULL)
   {
      STB_SICancelTableRequest(unt_filter[path]);
      unt_filter[path] = NULL;
   }

   if (!this_path_only)
   {
      // clear flags
      report_pmt_allowed[path] = FALSE;
      pmt_reported[path] = FALSE;
      si_update_delay_timestamp[path] = 0;
   }

   pmt_update_timestamp[path] = 0;
}

/**
 *

 *
 * @brief   Manager for the dvb ssu search
 *
 * @param   event     - reason for the manager being called
 * @param   table_rec - pointer to the table record received if relevant to the event
 *                            (e.g. for any table received event)
 *

 *
 */
static BOOLEAN ManageDvbSsuSearch(U8BIT path, U32BIT event, SI_TABLE_RECORD *table_rec)
{
   BOOLEAN finished;
   U16BIT onet_id, tran_id, serv_id;

   FUNCTION_START(ManageDvbSsuSearch);

   finished = FALSE;
   if (event == APP_SI_START_MANAGER)
   {
      // first ensure there is a transport record for the transport we are tuned to and set
      // current transport accordingly
      current_transport_rec[path] = DBDEF_GetTunedTransport(path);
      if (current_transport_rec[path] != NULL)
      {
         #ifdef DEBUG_SI_DVB_SSU_SEARCH
         AP_SI_PRINT(("DVB SSU search: start (%s)",
            ACTL_GetRfNameFromFreq(current_transport_rec[path]->sig_type, current_transport_rec[path]->frequency)));
         #endif

         // update transport fields
         current_network_rec[path] = current_transport_rec[path]->network;
         current_service_rec[path] = NULL;

         // start search for NIT
         nit_start_timestamp[path] = STB_OSGetClockMilliseconds();

         if (active_network_id == 0)
         {
            nit_filter[path] = STB_SIRequestNit(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_NIT_RECEIVED);
         }
         else
         {
            nit_filter[path] = STB_SIRequestNitWithId(path, active_network_id, ONE_SHOT_REQUEST,
               ReceiveSiTable, APP_SI_NIT_RECEIVED);
         }
      }
      else
      {
         #ifdef DEBUG_SI_DVB_SSU_SEARCH
         AP_SI_PRINT(("DVB SSU search: start - failed, no transport"));
         #endif
         // report end of search
         STB_SISearchComplete(path, TRUE, NULL, 0);
         finished = TRUE;
      }
   }
   else if (event == APP_SI_STOP_MANAGER)
   {
      #ifdef DEBUG_SI_DVB_SSU_SEARCH
      AP_SI_PRINT(("DVB SSU search: stop"));
      #endif
      CancelTableRequests(path, FALSE);
      finished = TRUE;
   }
   else
   {
      switch (event)
      {
         case APP_SI_NIT_RECEIVED:
         {
            #ifdef DEBUG_SI_DVB_SSU_SEARCH
            AP_SI_PRINT(("DVB SSU search: NIT received (nid 0x%04x)", table_rec->xtid));
            #endif

            ProcessNitTable(table_rec, DB_ACCESS_UPDATE, TRUE, TRUE);

            // finished with the filter - release it
            STB_SICancelTableRequest(nit_filter[path]);
            nit_filter[path] = NULL;
            nit_start_timestamp[path] = 0;

            // report end of search
            CancelTableRequests(path, FALSE);

            if (ASI_GetNextOTALocation(current_network_rec[path], NULL, &onet_id, &tran_id,
               &serv_id) != NULL)
            {
               /* Possible OTA update found */
               #ifdef DEBUG_SI_DVB_SSU_SEARCH
               AP_SI_PRINT(("DVB SSU search: possible OTA update found, search succeeded"));
               #endif
               STB_SISearchComplete(path, TRUE, &current_transport_rec[path], sizeof(void *));
            }
            else
            {
               /* No OTA found */
             #ifndef INCLUDE_HWACOM_SSU
               #ifdef DEBUG_SI_DVB_SSU_SEARCH
               AP_SI_PRINT(("DVB SSU search: no OTA update found, search failed"));
               #endif
               STB_SISearchComplete(path, FALSE, &current_transport_rec[path], sizeof(void *));
             #else  // #ifndef INCLUDE_HWACOM_SSU
               // search SSU UNT
               unt_start_timestamp[path] = STB_OSGetClockMilliseconds();
               unt_filter[path] = STB_SIRequestUnt(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_UNT_RECEIVED);
               break;
             #endif  // #ifndef INCLUDE_HWACOM_SSU
            }
            finished = TRUE;
            break;
         }
         case APP_SI_NIT_TIMEOUT:
         {
            #ifdef DEBUG_SI_DVB_SSU_SEARCH
            AP_SI_PRINT(("DVB_SSU search: NIT timeout"));
            #endif

            // finished with the filter - release it
            STB_SICancelTableRequest(nit_filter[path]);
            nit_filter[path] = NULL;
            nit_start_timestamp[path] = 0;

            // report end of search
            #ifdef DEBUG_SI_DVB_SSU_SEARCH
            AP_SI_PRINT(("DVB SSU search: search failed"));
            #endif

            CancelTableRequests(path, FALSE);
            STB_SISearchComplete(path, FALSE, &current_transport_rec[path], sizeof(void *));
            finished = TRUE;
            break;
         }
         case APP_SI_UNT_RECEIVED:
         {
            ProcessUntTable(path, table_rec, TRUE, FALSE);
            // finished with the filter - release it
            STB_SICancelTableRequest(unt_filter[path]);
            unt_filter[path] = NULL;
            unt_start_timestamp[path] = 0;
            STB_SISearchComplete(path, TRUE, &current_transport_rec[path], sizeof(void *));
            finished = TRUE;
            break;
         }
         case APP_SI_UNT_TIMEOUT:
         {
            #ifdef DEBUG_SI_DVB_SSU_SEARCH
            AP_SI_PRINT(("DVB_SSU search: UNT timeout"));
            #endif

            // finished with the filter - release it
            STB_SICancelTableRequest(unt_filter[path]);
            unt_filter[path] = NULL;
            unt_start_timestamp[path] = 0;

            // report end of search
            #ifdef DEBUG_SI_DVB_SSU_SEARCH
            AP_SI_PRINT(("DVB SSU search: search failed"));
            #endif

            CancelTableRequests(path, FALSE);
            STB_SISearchComplete(path, FALSE, &current_transport_rec[path], sizeof(void *));
            finished = TRUE;
            break;
         }
      }
   }

   FUNCTION_FINISH(ManageDvbSsuSearch);
   return(finished);
}

/**
 *

 *
 * @brief   This copies a Linkage descripter to a linkage descripters linked list
 *
 * @param   new_list_ptr  - pointer to the new link list to be copied
 * @param   list_ptr      - pointer to the list of where it needs to be copied
 * @param   last_entry_ptr- pointer to the last entry in given linked list
 * @param   num_linkage_desc - pointer to the number of entries in the linked list
 *
 */
static void CopyLinkageDesc(SI_LINKAGE_DESC_ENTRY *new_list_ptr, SI_LINKAGE_DESC_ENTRY **list_ptr,
   SI_LINKAGE_DESC_ENTRY **last_entry_ptr, U16BIT *num_linkage_desc)
{
   SI_LINKAGE_DESC_ENTRY *desc_ptr;

   FUNCTION_START(CopyLinkageDesc);

   desc_ptr = (SI_LINKAGE_DESC_ENTRY *)STB_GetMemory(sizeof(SI_LINKAGE_DESC_ENTRY) + new_list_ptr->data_length);
   if (desc_ptr != NULL)
   {
      desc_ptr->next = NULL;
      desc_ptr->tran_id = new_list_ptr->tran_id;
      desc_ptr->orig_net_id = new_list_ptr->orig_net_id;
      desc_ptr->serv_id = new_list_ptr->serv_id;
      desc_ptr->link_type = new_list_ptr->link_type;
      desc_ptr->data_length = new_list_ptr->data_length;

      if (new_list_ptr->data_length != 0)
      {
         memcpy(&(desc_ptr->data), &(new_list_ptr->data), new_list_ptr->data_length);
      }

      // now add descriptor to linked list
      if (*last_entry_ptr == NULL)
      {
         // first entry in the list
         *list_ptr = desc_ptr;
      }
      else
      {
         // not the first entry
         (*last_entry_ptr)->next = desc_ptr;
      }
      *last_entry_ptr = desc_ptr;
      (*num_linkage_desc)++;

      #ifdef DEBUG_SI_SDT
      AP_SI_PRINT((" AP_SI----  Linkage desc: tid=0x%04x, onid=0x%04x, sid=0x%04x, type=%d, dlen=%d",
                   desc_ptr->tran_id, desc_ptr->orig_net_id, desc_ptr->serv_id,
                   desc_ptr->link_type, desc_ptr->data_length));
      #endif
   }

   FUNCTION_FINISH(CopyLinkageDesc);
}

/*!**************************************************************************
 * @brief   Deletes a linkage descripter rray
 * @param   list_ptr - array to delete
 ****************************************************************************/
static void DeleteLinkageDescripterArray(SI_LINKAGE_DESC_ENTRY *list_ptr)
{
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   SI_LINKAGE_DESC_ENTRY *tmp_ptr;

   FUNCTION_START(DeleteLinkageDescripterArray);

   desc_ptr = list_ptr;
   while (desc_ptr != NULL)
   {
      tmp_ptr = desc_ptr->next;
      STB_FreeMemory(desc_ptr);
      desc_ptr = tmp_ptr;
   }

   FUNCTION_FINISH(DeleteLinkageDescripterArray);
}

/**
 *

 *
 * @brief   Checks dvb ssu linkage descriptor to see if this software is for us
 *
 * @param   desc_ptr      - pointer to the decriptor to be checked
 * @param   onid_ptr      - pointer for the return of the link onid if upgrade is for us
 * @param   tid_ptr       - pointer for the return of the link tid if upgrade is for us
 * @param   sid_ptr       - pointer for the return of the link sid if upgrade is for us
 *
 * @return   TRUE if upgrade found, FALSE otherwise
 *
 */
static BOOLEAN CheckSsuLinkageDescForUpgrade(SI_LINKAGE_DESC_ENTRY *desc_ptr, U16BIT *onid_ptr,
   U16BIT *tid_ptr, U16BIT *sid_ptr)
{
   BOOLEAN retval;
   U8BIT *data_ptr;
   U8BIT oui_loop_len;
   U8BIT *oui_loop_end;
   U8BIT oui[3];
   U8BIT selector_len;

   FUNCTION_START(CheckSsuLinkageDescForUpgrade);

   ASSERT(desc_ptr != NULL);
   ASSERT(onid_ptr != NULL);
   ASSERT(tid_ptr != NULL);
   ASSERT(sid_ptr != NULL);

   #ifdef DEBUG_DVB_SSU
   AP_SI_PRINT(("   DVB SSU link desc 0x%04x/0x%04x/0x%04x len %d",
                desc_ptr->orig_net_id, desc_ptr->tran_id, desc_ptr->serv_id, desc_ptr->data_length));
   #endif

   retval = FALSE;

   // check data to see if upgrade applies to us
   data_ptr = &(desc_ptr->data);
   oui_loop_len = data_ptr[0];
   data_ptr++;
   oui_loop_end = data_ptr + oui_loop_len;
   while (data_ptr < oui_loop_end)
   {
      memcpy(oui, data_ptr, 3);
      selector_len = data_ptr[3];
      data_ptr += 4;
      #ifdef DEBUG_DVB_SSU
      AP_SI_PRINT(("   OUI=0x%02x%02x%02x", oui[0], oui[1], oui[2]));
      #endif
      if ((memcmp(oui, dvb_oui, 3) == 0) ||
         ((STB_HWGetOUI() != NULL) && (memcmp(oui, STB_HWGetOUI(), 3) == 0)))
      {
         *onid_ptr = desc_ptr->orig_net_id;
         *tid_ptr = desc_ptr->tran_id;
         *sid_ptr = desc_ptr->serv_id;
#ifdef DEBUG_DVB_SSU
         AP_SI_PRINT(("   DVB SSU: upgrade service detected, onid=0x%x, tid=0x%x, sid=0x%x",
                      *onid_ptr, *tid_ptr, *sid_ptr));
#endif
         retval = TRUE;
         break;
      }

      data_ptr += selector_len;
   }

   FUNCTION_FINISH(CheckSsuLinkageDescForUpgrade);
   return(retval);
}

//--------------------------------------------------------------------------------------------------
// global function definitions
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Initialises application SI handling
 */
void ASI_InitialiseAppSi(void)
{
   U8BIT path, num_paths;

   FUNCTION_START(ASI_InitialiseAppSi);

   /* Allocate arrays for each path */
   num_paths = STB_DPGetNumPaths();

   current_manager = (F_SIManager *)STB_AppGetMemory(sizeof(F_SIManager) * num_paths);
   required_si_mode = (E_APP_SI_MODE *)STB_AppGetMemory(sizeof(E_APP_SI_MODE) * num_paths);
   required_si_mode_ctrl = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);

   current_network_rec = (ADB_NETWORK_REC **)STB_AppGetMemory(sizeof(ADB_NETWORK_REC *) * num_paths);
   current_transport_rec = (ADB_TRANSPORT_REC **)STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC *) * num_paths);
   current_service_rec = (ADB_SERVICE_REC **)STB_AppGetMemory(sizeof(ADB_SERVICE_REC *) * num_paths);
   last_transport_rec = (ADB_TRANSPORT_REC **)STB_AppGetMemory(sizeof(ADB_TRANSPORT_REC *) * num_paths);

   sched_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   eit_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   pat_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   pmt_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   nit_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   sdt_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   bat_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   tot_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   tdt_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   cat_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   rct_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   ait_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);
   unt_filter = (void **)STB_AppGetMemory(sizeof(void *) * num_paths);

   pat_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   pmt_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   nit_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   sdt_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   bat_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   eit_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   sched_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   sched_timeout_ms = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   tot_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   tdt_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   si_update_delay_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   pmt_update_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   cat_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);

   pmt_update_period_ms = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   last_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);
   unt_start_timestamp = (U32BIT *)STB_AppGetMemory(sizeof(U32BIT) * num_paths);

   pmt_list_id = (U16BIT *)STB_AppGetMemory(sizeof(U16BIT) * num_paths);

   sdt_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   bat_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   nit_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   pmts_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   eits_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   tot_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   unt_complete = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);

   tot_already_received = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   pat_rcvd_on_this_trnsprt = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   pmt_list = (PMT_LIST_ENTRY **)STB_AppGetMemory(sizeof(PMT_LIST_ENTRY *) * num_paths);
   num_pmt_list_entries = (U16BIT *)STB_AppGetMemory(sizeof(U16BIT) * num_paths);
   pmt_filter_pid = (U16BIT *)STB_AppGetMemory(sizeof(U16BIT) * num_paths);
   pmt_service_changed = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   report_pmt_allowed = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   pmt_reported = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   stop_pmt_reporting = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   pmt_request_mode = (E_PMT_REQUEST_MODE *)STB_AppGetMemory(sizeof(E_PMT_REQUEST_MODE) * num_paths);
   eit_list = (EIT_LIST_ENTRY **)STB_AppGetMemory(sizeof(EIT_LIST_ENTRY *) * num_paths);
   num_eit_list_entries = (U16BIT *)STB_AppGetMemory(sizeof(U16BIT) * num_paths);

   service_list_ready = (BOOLEAN *)STB_AppGetMemory(sizeof(BOOLEAN) * num_paths);
   last_reported_cat_version = (S16BIT *)STB_AppGetMemory(sizeof(S16BIT) * num_paths);
   last_reported_nit_version = (S16BIT *)STB_AppGetMemory(sizeof(S16BIT) * num_paths);
   last_reported_pmt_version = (U8BIT *)STB_AppGetMemory(sizeof(U8BIT) * num_paths);
   last_reported_unt_version = (S16BIT *)STB_AppGetMemory(sizeof(S16BIT) * num_paths);

   for (path = 0; path < num_paths; path++)
   {
      current_manager[path] = NULL;
      required_si_mode[path] = STOP_SI;
      current_network_rec[path] = NULL;
      current_transport_rec[path] = NULL;
      current_service_rec[path] = NULL;
      last_transport_rec[path] = NULL;

      sched_filter[path] = NULL;
      eit_filter[path] = NULL;
      pat_filter[path] = NULL;
      pmt_filter[path] = NULL;
      nit_filter[path] = NULL;
      sdt_filter[path] = NULL;
      bat_filter[path] = NULL;
      tot_filter[path] = NULL;
      tdt_filter[path] = NULL;
      cat_filter[path] = NULL;
      rct_filter[path] = NULL;
      ait_filter[path] = NULL;
      unt_filter[path] = NULL;

      pat_start_timestamp[path] = 0;
      pmt_start_timestamp[path] = 0;
      nit_start_timestamp[path] = 0;
      sdt_start_timestamp[path] = 0;
      bat_start_timestamp[path] = 0;
      eit_start_timestamp[path] = 0;
      sched_start_timestamp[path] = 0;
      sched_timeout_ms[path] = 0;
      tot_start_timestamp[path] = 0;
      tdt_start_timestamp[path] = 0;
      si_update_delay_timestamp[path] = 0;
      pmt_update_timestamp[path] = 0;
      cat_start_timestamp[path] = 0;
      pmt_update_period_ms[path] = 0;
      last_timestamp[path] = 0;
      unt_start_timestamp[path] = 0;

      pmt_list_id[path] = 0;

      sdt_complete[path] = FALSE;
      bat_complete[path] = FALSE;
      nit_complete[path] = FALSE;
      pmts_complete[path] = FALSE;
      eits_complete[path] = FALSE;
      tot_complete[path] = FALSE;
      unt_complete[path] = FALSE;

      tot_already_received[path] = FALSE;
      pat_rcvd_on_this_trnsprt[path] = FALSE;
      pmt_list[path] = NULL;
      num_pmt_list_entries[path] = 0;
      pmt_filter_pid[path] = 0;
      pmt_service_changed[path] = FALSE;
      report_pmt_allowed[path] = FALSE;
      pmt_reported[path] = FALSE;
      stop_pmt_reporting[path] = FALSE;
      pmt_request_mode[path] = PMT_REQUEST_MONITOR;
      eit_list[path] = NULL;
      num_eit_list_entries[path] = 0;

      service_list_ready[path] = FALSE;
      last_reported_cat_version[path] = -1;
      last_reported_nit_version[path] = -1;
      last_reported_pmt_version[path] = 0;
      last_reported_unt_version[path] = -1;
   }

   required_service_type = SEARCH_SERVICE_TYPE_ALL;

   STB_SIRegisterAppSiEventHandler(HandleSiEvent);

   eit_schedule_limit = (U16BIT)APP_NvmRead(EIT_SCHED_LIMIT_NVM);

   /* Create a semaphore to protect access to the PMT priority list */
   si_pmt_list_sem = STB_OSCreateSemaphore();

   ASI_ClearPmtList();

   use_bats_active = FALSE;
   active_bouquet_ids = NULL;
   num_active_bouquet_ids = 0;

   active_network_id = 0;

   FUNCTION_FINISH(ASI_InitialiseAppSi);
}

/**
 * @brief   Sets application SI mode - used before STB_DPStartSI() is called.
 * @param   path decode path
 * @param   si_mode required mode
 */
void ASI_SetAppSiMode(U8BIT path, E_APP_SI_MODE si_mode)
{
   FUNCTION_START(APP_SetAppSiMode);

   if (path < STB_DPGetNumPaths())
   {
      required_si_mode[path] = si_mode;
   }

   FUNCTION_FINISH(APP_SetAppSiMode);
}

/**
 * @brief   reset application SI mode flag
 * @param   path decode path
 *
 */
void ASI_ResetAppSiModeFlag(U8BIT path)
{
    FUNCTION_START(ASI_ResetAppSiModeFlag);

    if (path < STB_DPGetNumPaths()) {
        required_si_mode_ctrl[path] = 0;
    }

    FUNCTION_FINISH(ASI_ResetAppSiModeFlag);
}

/**
 * @brief   set application SI mode flag - used before STB_DPStartSI() is called.
 * @param   path decode path
 * @param   si_mode_flag required mode flag
 * @param   enable TRUE or FALSE
 *
 */
void ASI_SetAppSiModeFlag(U8BIT path, E_ASI_MODE_FLAG si_mode_flag, BOOLEAN enable)
{
    FUNCTION_START(ASI_SetAppSiModeFlag);

    if (si_mode_flag < ASI_MODE_FLAG_MAX) {
        if (path < STB_DPGetNumPaths()) {
            if (TRUE == enable)
                required_si_mode_ctrl[path] |= (1 << si_mode_flag);
                enable = TRUE;
        }
    }

    FUNCTION_FINISH(ASI_SetAppSiModeFlag);
}

/**
 * @brief   get application SI mode flag
 * @param   path decode path
 * @param   si_mode_flag required mode flag
 *
 * @return   TRUE or FALSE
 */
BOOLEAN ASI_GetAppSiModeFlag(U8BIT path, E_ASI_MODE_FLAG si_mode_flag)
{
    BOOLEAN enable = FALSE;

    FUNCTION_START(ASI_GetAppSiModeFlag);

    if (si_mode_flag < ASI_MODE_FLAG_MAX) {
        if (path < STB_DPGetNumPaths()) {
            if (0 != (required_si_mode_ctrl[path] & (1 << si_mode_flag)))
                enable = TRUE;
        }
    }

    FUNCTION_FINISH(ASI_GetAppSiModeFlag);

    return enable;
}

/**
 *

 *
 * @brief   Reports the state of the service_list_ready flag which is set when the sdt has
 *                been processed in a search
 *
 * @param   U8BIT path
 *
 * @return   service_list_ready[path]
 *
 */
BOOLEAN ASI_CheckServiceListReadyDuringSearch(U8BIT path)
{
   FUNCTION_START(ASI_CheckServiceListReadyDuringSearch);
   FUNCTION_FINISH(ASI_CheckServiceListReadyDuringSearch);
   return(service_list_ready[path]);
}

/**
 * @brief   Returns the PID for the pmt of a given service (on live path)
 * @param   service_id provides the required service ID
 * @param   ts_id provides the required transport stream ID
 * @param   on_id provides the required original network ID
 * @return  the PID of the pmt for that service
 */
U16BIT ASI_GetPmtPid(U16BIT serv_id, U16BIT ts_id, U16BIT on_id)
{
   U16BIT i;
   U16BIT pid;
   U8BIT path;

   FUNCTION_START(ASI_GetPmtPid);

   pid = 0xffff;
   path = STB_DPGetLivePath();
   if (path != INVALID_RES_ID && pmt_list[path] != NULL)
   {
      for (i = 0; i < num_pmt_list_entries[path]; i++)
      {
         if ((pmt_list[path][i].serv_id == serv_id) &&
             ((ts_id == ADB_INVALID_DVB_ID) || (pmt_list[path][i].ts_id == ts_id)) &&
             ((on_id == ADB_INVALID_DVB_ID) || (pmt_list[path][i].on_id == on_id)))
         {
            pid = pmt_list[path][i].pid;
            break;
         }
      }
   }

   FUNCTION_FINISH(ASI_GetPmtPid);

   return pid;
}

/**
 * @brief   Prevents the current pmt being reported (e.g. to MHEG).
 */
void ASI_StopPmtReporting(U8BIT path)
{
   FUNCTION_START(ASI_StopPmtReporting);
   // set flag - state is checked in timer handling section
   stop_pmt_reporting[path] = TRUE;
   FUNCTION_FINISH(ASI_StopPmtReporting);
}

/**
 * @brief   Returns TRUE if pmt has been reported to third parties
 * @param   path decode path
 * @return  TRUE if reported, FALSE otherwise
 */
BOOLEAN ASI_PmtReported(U8BIT path)
{
   FUNCTION_START(ASI_PmtReported);
   FUNCTION_FINISH(ASI_PmtReported);
   return(pmt_reported[path]);
}

/**
 * @brief   Performs the neccessary actions for this module when entering/exiting standby
 *          according to the value passed in standby_state.
 * @param   standby_state if set TRUE indicates that the STB is entering standby
 *          mode and if set FALSE indicates shat the STB is exiting from standby mode.
 */
void ASI_SetStandbyState(BOOLEAN standby_state)
{
   U8BIT i, num_paths;

   FUNCTION_START(ASI_SetStandbyState);

   // If we are going into standby then nullify the last transport record pointer. This will ensure
   // that, on exiting from standby, tables which are normally reported (to third parties) when a
   // transport changes (e.g. CAT) are re-reported.
   if (standby_state == TRUE)
   {
      num_paths = STB_DPGetNumPaths();

      for (i = 0; i < num_paths; i++)
      {
         last_transport_rec[i] = NULL;
      }
   }

   FUNCTION_FINISH(ASI_SetStandbyState);
}

/**
 * @brief   Returns the DVB triplet for the next location containing a possible OTA update.
 * @param   network network on which the OTA search is being made
 * @param   prev_location if this is NULL then info from the first OTA location on the network will
 *                        be returned, otherwise it should be the value returned when this function
 *                        was last called so the next possible OTA location will be returned
 * @param   onid_ptr pointer to return the value of the original network ID for the possible OTA
 * @param   tdi_ptr pointer to return the value of the transport ID for the possible OTA
 * @param   sid_ptr pointer to return the value of the service ID for the possible OTA
 * @return  If NULL then no OTA locations are available, otherwise a possible OTA is available and
 *          the returned value should be passed in the next time this function is called
 */
void* ASI_GetNextOTALocation(void *network, void *prev_location, U16BIT *onid_ptr,
   U16BIT *tid_ptr, U16BIT *sid_ptr)
{
   ADB_NETWORK_REC *n_ptr;
   SI_LINKAGE_DESC_ENTRY *desc_ptr;
   U16BIT onid, tsid, sid;

   FUNCTION_START(ASI_GetNextOTALocation);

   desc_ptr = NULL;
   if (network != NULL)
   {
      n_ptr = (ADB_NETWORK_REC*)network;

      DBDEF_RequestAccess();

      if (prev_location != NULL)
      {
         /* Find the previous linkage descriptor in the network */
         for (desc_ptr = n_ptr->linkage_desc_list; desc_ptr != NULL; desc_ptr = desc_ptr->next)
         {
            if (desc_ptr == prev_location)
            {
               desc_ptr = desc_ptr->next;
               break;
            }
         }
      }
      else
      {
         /* Start the search from the first linkage descriptor in the network */
         desc_ptr = n_ptr->linkage_desc_list;
      }

      for ( ; desc_ptr != NULL; desc_ptr = desc_ptr->next)
      {
         if (CheckSsuLinkageDescForUpgrade(desc_ptr, &onid, &tsid, &sid))
         {
            *onid_ptr = onid;
            *tid_ptr = tsid;
            *sid_ptr = sid;
            break;
         }
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ASI_GetNextOTALocation);

   return(desc_ptr);
}

/**
 * @brief   Returns the flag indicating whether the SSU was refused
 * @return  TRUE if the SSU was refused, FALSE otherwise
 */
BOOLEAN ASI_SSURefused(void)
{
   return ssu_refused;
}

/**
 * @brief   Sets the flag indicating whether the SSU has been refused or not
 * @param   refuse TRUE to refuse the SSU, FALSE to accept
 */
void ASI_RefuseSSU(BOOLEAN refuse)
{
   ssu_refused = refuse;
}

/**
 * @brief   Sets the flag indicating whether the SSU can be refused by the user.
 * @param   mandatory TRUE if the SSU can't be refused by the user, FALSE otherwise
 */
void ASI_SSUSetMandatory(BOOLEAN mandatory)
{
   FUNCTION_START(ASI_SSUSetMandatory);
   dvb_ssu_mandatory = mandatory;
   FUNCTION_FINISH(ASI_SSUSetMandatory);
}

/**
 * @brief   Returns the flag indicating whether the SSU is mandatory and so can't be refused
 * @return  TRUE if the SSU is mandatory, FALSE otherwise
 */
BOOLEAN ASI_SSUGetMandatory(void)
{
   FUNCTION_START(ASI_SSUGetMandatory);
   FUNCTION_FINISH(ASI_SSUGetMandatory);
   return dvb_ssu_mandatory;
}

/**
 * @brief   Checks whether the NIT or SDT version numbers have changed, which
 *          may indicate a change to the service lineup requiring an update
 *          service scan.
 * @param   tuner_type check all networks and transports for this tuner type, or use
 *                     SIGNAL_NONE to check all networks and transports
 * @return  TRUE if one of the version numbers has changed, FALSE otherwise
 */
BOOLEAN ASI_CheckForServiceChange(E_STB_DP_SIGNAL_TYPE tuner_type)
{
   BOOLEAN service_change;
   ADB_TRANSPORT_REC *t_ptr;

   FUNCTION_START(ASI_CheckForServiceChange);

   service_change = FALSE;

   DBDEF_RequestAccess();

   /* Iterate through transports */
   t_ptr = DBDEF_GetNextTransportRec(NULL);
   while (!service_change && (t_ptr != NULL))
   {
      if ((tuner_type == SIGNAL_NONE) || (t_ptr->sig_type == tuner_type))
      {
         service_change = t_ptr->sdt_version_changed;
         if (!service_change && (t_ptr->network != NULL))
         {
            service_change = t_ptr->network->nit_version_changed;
         }
      }

      t_ptr = DBDEF_GetNextTransportRec(t_ptr);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ASI_CheckForServiceChange);

   return(service_change);
}

/**
 * @brief   Forces the SI demux filter collecting the CAT tables to be reset,
 *          so a previously processed version of the table won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartCatFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartCatFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (cat_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(cat_filter[path]);
      }

      last_reported_cat_version[path] = -1;
   }

   FUNCTION_FINISH(ASI_RestartCatFilter);
}

/**
 * @brief   Forces the SI demux filter collecting the NIT tables to be reset,
 *          so a previously processed version of the table won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartNitFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartNitFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (nit_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(nit_filter[path]);
      }

      last_reported_nit_version[path] = -1;
   }

   FUNCTION_FINISH(ASI_RestartNitFilter);
}

/**
 * @brief   Forces the SI demux filter collecting the SDT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartSdtFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartSdtFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (sdt_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(sdt_filter[path]);
      }
   }

   FUNCTION_FINISH(ASI_RestartSdtFilter);
}

/**
 * @brief   Forces the SI demux filter collecting the BAT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartBatFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartBatFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (bat_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(bat_filter[path]);
      }
   }

   FUNCTION_FINISH(ASI_RestartBatFilter);
}

/**
 * @brief   Forces the SI demux filter collecting the TOT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartTotFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartTotFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (tot_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(tot_filter[path]);
      }
   }

   FUNCTION_FINISH(ASI_RestartTotFilter);
}

/**
 * @brief   Forces the SI demux filter collecting the TDT tables to be reset,
 *          so a previously processed versions of the tables won't be ignored.
 * @param   path - decode path that will be affected
 */
void ASI_RestartTdtFilter(U8BIT path)
{
   FUNCTION_START(ASI_RestartTdtFilter);

   if (path < STB_DPGetNumPaths())
   {
      if (tdt_filter[path] != NULL)
      {
         STB_SIRestartTableRequest(tdt_filter[path]);
      }
   }

   FUNCTION_FINISH(ASI_RestartTdtFilter);
}

/**
 * @brief   Set the type for services that should be added during a service search
 * @param   service_type all, free-to-air, or scrambled
 */
void ASI_SetSearchServiceType(E_SEARCH_SERVICE_TYPE service_type)
{
   FUNCTION_START(ASI_SetSearchServiceType);
   required_service_type = service_type;
   FUNCTION_FINISH(ASI_SetSearchServiceType);
}

/**
 * @brief   Add the given service id to the list of services whose PMT will be requested
 *          with a higher priority than others
 * @param   service_id service id to be added to the list
 */
void ASI_AddServiceToPmtList(U16BIT service_id)
{
   U8BIT i, index = MAX_PMT_PRIORITY_LIST - 1;

   FUNCTION_START(ASI_AddServiceToPmtList);

   STB_OSSemaphoreWait(si_pmt_list_sem);

   for (i = 0; i < MAX_PMT_PRIORITY_LIST; i++)
   {
      if (pmt_priority_list[i] == service_id)
      {
         /* Service is already in the list */
         index = i;
         break;
      }
   }

   /* Move existing entries up so the new one can be added at the start */
   for (i = index; i > 0; i--)
   {
      pmt_priority_list[i] = pmt_priority_list[i - 1];
   }

   pmt_priority_list[0] = service_id;

   STB_OSSemaphoreSignal(si_pmt_list_sem);

   FUNCTION_FINISH(ASI_AddServiceToPmtList);
}

/**
 * @brief   Removes the service id from the PMT priority list
 * @param   service_id service id to be removed
 */
void ASI_RemoveServiceFromPmtList(U16BIT service_id)
{
   U8BIT i, index = MAX_PMT_PRIORITY_LIST;

   FUNCTION_START(ASI_RemoveServiceFromPmtList);

   STB_OSSemaphoreWait(si_pmt_list_sem);

   for (i = 0; i < MAX_PMT_PRIORITY_LIST; i++)
   {
      if (pmt_priority_list[i] == service_id)
      {
         index = i;
         break;
      }
   }

   if (index < MAX_PMT_PRIORITY_LIST)
   {
      for (i = index; i < MAX_PMT_PRIORITY_LIST - 1; i++)
      {
         pmt_priority_list[i] = pmt_priority_list[i + 1];
      }

      pmt_priority_list[MAX_PMT_PRIORITY_LIST - 1] = INVALID_SERVICE_ID;
   }

   STB_OSSemaphoreSignal(si_pmt_list_sem);

   FUNCTION_FINISH(ASI_RemoveServiceFromPmtList);
}

/**
 * @brief   Clears all service ids from the PMT priority list
 */
void ASI_ClearPmtList(void)
{
   U8BIT i;

   FUNCTION_START(ASI_ClearPmtList);

   STB_OSSemaphoreWait(si_pmt_list_sem);

   for (i = 0; i < MAX_PMT_PRIORITY_LIST; i++)
   {
      pmt_priority_list[i] = INVALID_SERVICE_ID;
   }

   STB_OSSemaphoreSignal(si_pmt_list_sem);

   FUNCTION_FINISH(ASI_ClearPmtList);
}

/**
 * @brief   Takes data for a raw PMT for the given service and processes it as if
 *          it had been received from the demux, also passing it to anything monitoring
 *          PMTs and CI+.
 * @param   path decode path
 * @param   s_ptr service to be updated with info extracted from the PMT
 * @param   pmt_data raw PMT data
 */
void ASI_ProcessPmt(U8BIT path, void *s_ptr, U8BIT *pmt_data)
{
   SI_TABLE_RECORD table_rec;
   SI_SECTION_RECORD *sect;
   U16BIT pmt_size;

   FUNCTION_START(ASI_ProcessPmt);

   /* Need to populate a table_rec before parsing the PMT data */
   pmt_size = ((pmt_data[1] & 0x0f) << 8) + pmt_data[2];

   if ((sect = (SI_SECTION_RECORD *)STB_AppGetMemory(sizeof(SI_SECTION_RECORD) + pmt_size)) != NULL)
   {
      sect->next = NULL;
      sect->sect_num = 0;
      sect->data_len = pmt_size;
      memcpy(&sect->data_start, pmt_data, pmt_size);

      table_rec.path = path;
      table_rec.tid = pmt_data[0];
      table_rec.version = pmt_data[5] & 0x1f;
      table_rec.xtid = (pmt_data[3] << 8) + pmt_data[4];
      table_rec.num_sect = 1;
      table_rec.section_list = sect;

      /* Can now "inject" the PMT */
      DBDEF_RequestAccess();
      InternalProcessPmtTable(path, (ADB_SERVICE_REC *)s_ptr, &table_rec, FALSE, DB_ACCESS_UPDATE);
      DBDEF_ReleaseAccess();

      STB_SIReportCurrentPmt(table_rec.xtid, &table_rec, TRUE, TRUE);

#ifdef COMMON_INTERFACE
      ACI_ProgramMapTableChanged(&(table_rec.section_list->data_start));
#endif

      STB_AppFreeMemory(sect);
   }

   FUNCTION_FINISH(ASI_ProcessPmt);
}

/**
 * @brief   Sets the number of hours of EIT data that's kept for each service
 *          that hasn't its had EIT schedule disabled.
 * @param   limit_hours number of hours to EIT data to be kept.
 *                        0 turns this feature off and all data will be kept.
 */
void ASI_SetEITScheduleLimit(U16BIT limit_hours)
{
   FUNCTION_START(ASI_SetEITScheduleLimit);

   eit_schedule_limit = limit_hours;
   APP_NvmSave(EIT_SCHED_LIMIT_NVM, eit_schedule_limit, TRUE);

   FUNCTION_FINISH(ASI_SetEITScheduleLimit);
}

/**
 * @brief   Returns the current setting for the number of hours of EIT data that's kept
 * @return  number of hours, 0 means all data is kept
 */
U16BIT ASI_GetEITScheduleLimit(void)
{
   FUNCTION_START(ASI_GetEITScheduleLimit);
   FUNCTION_FINISH(ASI_GetEITScheduleLimit);
   return(eit_schedule_limit);
}

/**
 * @brief   Enables or disables the collection of BATs as part of the SI processing
 *          and allows the bouquet IDs of the BATs to be collected to be specified.
 * @param   use_bats TRUE to enable the collection and processing of BATs, FALSE to disable
 * @param   bouquet_ids an array of the bouquet IDs that are to be used.
 *                        If this is NULL then all BATs will be used.
 * @param   num_ids number of bouquet IDs in the array
 */
void ASI_EnableBatCollection(BOOLEAN use_bats, U16BIT *bouquet_ids, U16BIT num_ids)
{
   U16BIT i;

   FUNCTION_START(ASI_EnableBatCollection);

   if (active_bouquet_ids != NULL)
   {
      /* Delete the current array of bouquet IDs */
      STB_AppFreeMemory(active_bouquet_ids);
      active_bouquet_ids = NULL;
      num_active_bouquet_ids = 0;
   }

   use_bats_active = use_bats;

   if (use_bats && (bouquet_ids != NULL) && (num_ids > 0))
   {
      /* Save the array of bouquet IDs to be collected */
      active_bouquet_ids = (S_ACTIVE_BOUQUET *)STB_AppGetMemory(sizeof(S_ACTIVE_BOUQUET) * num_ids);
      if (active_bouquet_ids != NULL)
      {
         num_active_bouquet_ids = num_ids;
         for (i = 0; i < num_ids; i++)
         {
            active_bouquet_ids[i].bouquet_id = bouquet_ids[i];
            active_bouquet_ids[i].received = FALSE;
         }
      }
   }

   FUNCTION_FINISH(ASI_EnableBatCollection);
}

/**
 * @brief   Cancels any existing SI filters and starts a new one for the UNT, SDT, TDT, TOT,
 *          NIT, EIT (pf, pf++ and sched) and BAT SI tables.
 * @param   path decode path the filters are using
 * @param   use_standard_pids TRUE if SI tables on the DVB standard PIDs are to be collected,
 *                            FALSE will only start a filter if a PID is defined for that table
 *                            on the current service.
 */
void ASI_RestartSITables(U8BIT path, BOOLEAN use_standard_pids)
{
   U16BIT eit_pid;

   FUNCTION_START(ASI_RestartSITables);

   #ifdef DEBUG_SI_UPDATE
   AP_SI_PRINT(("%s(%u, standard_pids=%u)", __FUNCTION__, path, use_standard_pids));
   #endif

   if ((path != INVALID_RES_ID) && (current_service_rec[path] != NULL))
   {
      if (TRUE == STB_DPIsMonitoringPath(path)) {
         if (required_si_mode[path] == APP_SI_MODE_UPDATE) {
            return;
         }
      }

#ifndef CUSTOMIZED_FOR_CNS
      if (sdt_filter[path] != NULL)
      {
         STB_SICancelTableRequest(sdt_filter[path]);
         sdt_filter[path] = NULL;
         sdt_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting SDT", path));
         #endif
         sdt_start_timestamp[path] = 0;
         sdt_filter[path] = STB_SIRequestSdt(path, CONTINUOUS_REQUEST, TRUE, TRUE, DONT_CARE_ID_MATCH,
               DONT_CARE_ID_MASK, 1, ReceiveSiTable, APP_SI_SDT_RECEIVED);
      }
      else if (current_service_rec[path]->sdt_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting SDT on PID %u", path, current_service_rec[path]->sdt_pid));
         #endif
         sdt_start_timestamp[path] = 0;
         sdt_filter[path] = STB_SIRequestSdtFromPid(path, current_service_rec[path]->sdt_pid,
               CONTINUOUS_REQUEST, TRUE, TRUE, DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 1,
               ReceiveSiTable, APP_SI_SDT_RECEIVED);
      }
#endif  // #ifndef CUSTOMIZED_FOR_CNS

      if (tdt_filter[path] != NULL)
      {
         STB_SICancelTableRequest(tdt_filter[path]);
         tdt_filter[path] = NULL;
         tdt_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting TDT", path));
         #endif
         tdt_start_timestamp[path] = 0;
         tdt_filter[path] = STB_SIRequestTdt(path, ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TDT_RECEIVED);
      }
      else if (current_service_rec[path]->tdt_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting TDT on PID %u", path, current_service_rec[path]->tdt_pid));
         #endif
         tdt_start_timestamp[path] = 0;
         tdt_filter[path] = STB_SIRequestTdtFromPid(path, current_service_rec[path]->tdt_pid,
               ONE_SHOT_REQUEST, ReceiveSiTable, APP_SI_TDT_RECEIVED);
      }

      if (tot_filter[path] != NULL)
      {
         STB_SICancelTableRequest(tot_filter[path]);
         tot_filter[path] = NULL;
         tot_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting TOT", path));
         #endif
         tot_start_timestamp[path] = 0;
         tot_filter[path] = STB_SIRequestTot(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);
      }
      else if (current_service_rec[path]->tot_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting TOT on PID %u", path, current_service_rec[path]->tot_pid));
         #endif
         tot_start_timestamp[path] = 0;
         tot_filter[path] = STB_SIRequestTotFromPid(path, current_service_rec[path]->tot_pid,
               CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_TOT_RECEIVED);
      }

      if (nit_filter[path] != NULL)
      {
         STB_SICancelTableRequest(nit_filter[path]);
         nit_filter[path] = NULL;
         nit_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting NIT", path));
         #endif
         nit_start_timestamp[path] = 0;

         if (active_network_id == 0)
         {
            nit_filter[path] = STB_SIRequestNit(path, CONTINUOUS_REQUEST, ReceiveSiTable,
               APP_SI_NIT_RECEIVED);
         }
         else
         {
            nit_filter[path] = STB_SIRequestNitWithId(path, active_network_id, CONTINUOUS_REQUEST,
               ReceiveSiTable, APP_SI_NIT_RECEIVED);
         }
      }
      else if (current_service_rec[path]->nit_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting NIT on PID %u", path, current_service_rec[path]->nit_pid));
         #endif
         nit_start_timestamp[path] = 0;
         nit_filter[path] = STB_SIRequestNitFromPid(path, current_service_rec[path]->nit_pid,
               FALSE, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_NIT_RECEIVED);
      }

      if (eit_filter[path] != NULL)
      {
         STB_SICancelTableRequest(eit_filter[path]);
         eit_filter[path] = NULL;
         eit_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting EIT", path));
         #endif
         eit_start_timestamp[path] = 0;
         eit_filter[path] = STB_SIRequestEit(path, CONTINUOUS_REQUEST, EIT_NOW_NEXT_ALL,
               DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
      }
      else
      {
         eit_pid = 0;
         if (current_service_rec[path]->eit_pf_plus_pid != 0)
         {
            eit_pid = current_service_rec[path]->eit_pf_plus_pid;
         }
         else if (current_service_rec[path]->eit_pf_pid != 0)
         {
            eit_pid = current_service_rec[path]->eit_pf_pid;
         }

         if (eit_pid != 0)
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): requesting EIT on PID %u", path, eit_pid));
            #endif
            eit_start_timestamp[path] = 0;
            eit_filter[path] = STB_SIRequestEitFromPid(path, eit_pid, CONTINUOUS_REQUEST, EIT_PF_PLUS,
                  DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_EIT_RECEIVED);
         }
      }

      if (sched_filter[path] != NULL)
      {
         STB_SICancelTableRequest(sched_filter[path]);
         sched_filter[path] = NULL;
         sched_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         /* Start schedule filter based on EIT memory usage requirements */
         StartEITScheduleFilter(path);
      }
      else if (current_service_rec[path]->eit_sched_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting SCHED on PID %u", path, current_service_rec[path]->eit_sched_pid));
         #endif
         sched_start_timestamp[path] = 0;
         sched_filter[path] = STB_SIRequestSchedFromPid(path, current_service_rec[path]->eit_sched_pid,
               CONTINUOUS_REQUEST, EIT_SCHED_ALL, DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff,
               ReceiveSiTable, APP_SI_EIT_RECEIVED);
      }

#ifndef CUSTOMIZED_FOR_CNS
      if (bat_filter[path] != NULL)
      {
         STB_SICancelTableRequest(bat_filter[path]);
         bat_filter[path] = NULL;
         bat_start_timestamp[path] = 0;
      }

      if (use_bats_active)
      {
         if (use_standard_pids)
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): requesting BAT", path));
            #endif
            bat_start_timestamp[path] = 0;
            bat_filter[path] = STB_SIRequestBat(path, CONTINUOUS_REQUEST, DONT_CARE_ID_MATCH,
                  DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable, APP_SI_BAT_RECEIVED);
         }
         else if (current_service_rec[path]->bat_pid != 0)
         {
            #ifdef DEBUG_SI_UPDATE
            AP_SI_PRINT(("Update(%d): requesting BAT on PID %u", path, current_service_rec[path]->bat_pid));
            #endif
            bat_start_timestamp[path] = 0;
            bat_filter[path] = STB_SIRequestBatFromPid(path, current_service_rec[path]->bat_pid,
                  CONTINUOUS_REQUEST, DONT_CARE_ID_MATCH, DONT_CARE_ID_MASK, 0xffff, ReceiveSiTable,
                  APP_SI_BAT_RECEIVED);
         }
      }
#endif  // #ifndef CUSTOMIZED_FOR_CNS

#ifndef CUSTOMIZED_FOR_CNS
      // SSU UNT
      if (unt_filter[path] != NULL)
      {
         STB_SICancelTableRequest(unt_filter[path]);
         unt_filter[path] = NULL;
         unt_start_timestamp[path] = 0;
      }

      if (use_standard_pids)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting UNT", path));
         #endif
         unt_start_timestamp[path] = 0;
         unt_filter[path] = STB_SIRequestUnt(path, CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_UNT_RECEIVED);
      }
      else if (current_service_rec[path]->unt_pid != 0)
      {
         #ifdef DEBUG_SI_UPDATE
         AP_SI_PRINT(("Update(%d): requesting UNT on PID %u", path, current_service_rec[path]->unt_pid));
         #endif
         unt_start_timestamp[path] = 0;
         unt_filter[path] = STB_SIRequestUntFromPid(path, current_service_rec[path]->unt_pid,
               CONTINUOUS_REQUEST, ReceiveSiTable, APP_SI_UNT_RECEIVED);
      }
#endif  // #ifndef CUSTOMIZED_FOR_CNS
   }

   FUNCTION_FINISH(ASI_RestartSITables);
}

/**
 * @brief   Sets a function that will be called when parsing an EIT table and a
 *          descriptor is found that the standard code doesn't know how to parse
 * @param   parser_func function pointer
 */
void ASI_SetEITParserFunction(F_EitParser parser_func)
{
   FUNCTION_START(ASI_SetEITParserFunction);
   eit_parser_func = parser_func;
   FUNCTION_FINISH(ASI_SetEITParserFunction);
}

/**
 * @brief   Sets a function that will be called when a BAT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateBatFunction(F_BatTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdateBatFunction);
   update_bat_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdateBatFunction);
}

/**
 * @brief   Sets a function that will be called when an EIT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateEitFunction(F_EitTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdateEitFunction);
   update_eit_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdateEitFunction);
}

/**
 * @brief   Sets a function that will be called when an NIT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateNitFunction(F_NitTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdateNitFunction);
   update_nit_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdateNitFunction);
}

/**
 * @brief   Sets a function that will be called when a PMT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdatePmtFunction(F_PmtTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdatePmtFunction);
   update_pmt_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdatePmtFunction);
}

/**
 * @brief   Sets a function that will be called when an SDT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateSdtFunction(F_SdtTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdateSdtFunction);
   update_sdt_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdateSdtFunction);
}

/**
 * @brief   Sets a function that will be called when an UNT table is received
 * @param   update_func function pointer
 */
void ASI_SetUpdateUntFunction(F_UntTableUpdate update_func)
{
   FUNCTION_START(ASI_SetUpdateUntFunction);
   update_unt_func = update_func;
   FUNCTION_FINISH(ASI_SetUpdateUntFunction);
}

/**
 * @brief   Processes an EIT table, partial or full, and updates the events of the service it is for
 * @param   table_rec SI table record containing the EIT data
 * @param   playback TRUE if the EIT is related to PVR playback
 */
void ASI_ProcessEitTable(SI_TABLE_RECORD *table_rec, BOOLEAN playback)
{
   FUNCTION_START(ASI_ProcessEitTable);
   ProcessEitTable(table_rec, TRUE, DB_ACCESS_UPDATE, playback);
   FUNCTION_FINISH(ASI_ProcessEitTable);
}

/**
 * @brief   Processes the TOT table record to extract data for the database
 * @param   table_rec pointer to the pat table record
 */
void ASI_ProcessTotTable(SI_TABLE_RECORD *table_rec)
{
   SI_TIME_TABLE *tot_table;
   SI_LTO_DESC *lto_desc_ptr;
   U16BIT i;
   U32BIT reqd_country_code;
   BOOLEAN regions_exist;
   U8BIT reqd_region_code;

   FUNCTION_START(ASI_ProcessTotTable);

   tot_table = STB_SIParseTimeTable(table_rec);
   STB_SIReleaseTableRecord(table_rec);
   if (tot_table != NULL)
   {
      #ifdef DEBUG_SI_TOT
      AP_SI_PRINT(("TOT table: %02d:%02d:%02d on %d",
                   tot_table->hrs, tot_table->mins, tot_table->secs, tot_table->date));
      #endif

      // set the time and date
      STB_GCSetGMTTime(tot_table->hrs, tot_table->mins, tot_table->secs);
      STB_GCSetGMTDate(tot_table->date);

      // check for time offset descriptor
      reqd_country_code = ACFG_GetCountry();
      regions_exist = ACFG_GetRegionCode(reqd_country_code, (U8BIT)APP_NvmRead(REGION_ID_NVM),
            &reqd_region_code);

      lto_desc_ptr = tot_table->lto_desc_array;
      for (i = 0; i < tot_table->num_lto_entries; i++, lto_desc_ptr++)
      {
         if (lto_desc_ptr->country_code == reqd_country_code)
         {
            if ((regions_exist == FALSE) ||
                ((regions_exist == TRUE) && (lto_desc_ptr->region == reqd_region_code)))
            {
               STB_GCSetLocalTimeChange(lto_desc_ptr->change_date,
                  lto_desc_ptr->change_hrs,
                  lto_desc_ptr->change_mins,
                  lto_desc_ptr->change_secs,
                  lto_desc_ptr->offset_hrs,
                  lto_desc_ptr->offset_mins,
                  lto_desc_ptr->next_offset_hrs,
                  lto_desc_ptr->next_offset_mins,
                  lto_desc_ptr->offset_negative);
               break;
            }
         }
      }

      STB_SIReleaseTimeTable(tot_table);

      /* Send event to the app indicating the time has been updated */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_TIME_CHANGED, NULL, 0);
   }

   FUNCTION_FINISH(ASI_ProcessTotTable);
}

/**
 * @brief   Processes the TDT table record to extract data for the database
 * @param   table_rec pointer to the pat table record
 */
void ASI_ProcessTdtTable(SI_TABLE_RECORD *table_rec)
{
   SI_TIME_TABLE *tdt_table;

   FUNCTION_START(ASI_ProcessTdtTable);

   tdt_table = STB_SIParseTimeTable(table_rec);
   STB_SIReleaseTableRecord(table_rec);
   if (tdt_table != NULL)
   {
      #ifdef DEBUG_SI_TDT
      AP_SI_PRINT(("TDT table: %02d:%02d:%02d on %d",
                   tdt_table->hrs, tdt_table->mins, tdt_table->secs, tdt_table->date));
      #endif

      // set the time and date
      STB_GCSetGMTTime(tdt_table->hrs, tdt_table->mins, tdt_table->secs);
      STB_GCSetGMTDate(tdt_table->date);

      STB_SIReleaseTimeTable(tdt_table);

      /* Send event to the app indicating the time has been updated */
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_TIME_CHANGED, NULL, 0);
   }

   FUNCTION_FINISH(ASI_ProcessTdtTable);
}

/**
 * @brief   Registers a function that will be called whenever an EIT event is added, updated,
 *          deleted or expires from a service's EIT schedule. The registered function should
 *          return control as soon as possible, otherwise processing of other EIT events will
 *          be affected.
 * @param   sched_update_callback function to be called
 */
void ASI_RegisterEitSchedUpdateCallback(F_EitSchedUpdateCB sched_update_callback)
{
   FUNCTION_START(ASI_RegisterEitSchedUpdateCallback);

   eit_sched_update_callback = sched_update_callback;

   FUNCTION_FINISH(ASI_RegisterEitSchedUpdateCallback);
}

/**
 * @brief   Calls the EIT schedule update callback function, if one has been registered,
 *          with details of the service and event
 * @param   serv_ptr service whose EIT schedule has changed
 * @param   event_id ID of the event that's been added, updated, deleted or expired
 * @param   type type of the change. For APP_SI_EIT_JOURNAL_TYPE_CLEAR, the event_id is irrelevant
 */
void ASI_NotifyEitSchedUpdate(void *serv_ptr, U16BIT event_id, E_APP_SI_EIT_JOURNAL_TYPE type)
{
   void *t_ptr;
   S_APP_SI_EIT_SCHED_UPDATE update;

   FUNCTION_START(ASI_NotifyEitSchedUpdate);

   if ((eit_sched_update_callback != NULL) && (serv_ptr != NULL))
   {
      if ((t_ptr = ADB_GetServiceTransportPtr(serv_ptr)) != NULL)
      {
         update.type = type;
         update.orig_net_id = ADB_GetTransportOriginalNetworkId(t_ptr);
         update.tran_id = ADB_GetTransportTid(t_ptr);
         update.serv_id = ADB_GetServiceId(serv_ptr);
         update.allocated_lcn = ADB_GetServiceLcn(serv_ptr);
         update.is_sched = TRUE;
         update.event_id = event_id;

         (*eit_sched_update_callback)(&update);
      }
   }

   FUNCTION_FINISH(ASI_NotifyEitSchedUpdate);
}

/**
 * @brief   Sets the network ID to be used for all future requests to get an NIT.
 *          If this value isn't set, the default behaviour is to request the NITactual,
 *          but when this value is set then the NIT received could be NITactual or NITother.
 *          To clear this setting this function should be called with a network ID of 0.
 * @param   network_id network ID of the NIT to be requested, or 0 to use NITactual
 */
void ASI_SetActiveNetworkId(U16BIT network_id)
{
   FUNCTION_START(ASI_SetActiveNetworkId);

   active_network_id = network_id;

   FUNCTION_FINISH(ASI_SetActiveNetworkId);
}

/**
 * @brief   Sets whether updates should be allowed to the service database from standard SI
 *          tables that are received when the SI engine is running in UPDATE mode. The default
 *          setting is for updates to be allowed and this setting doesn't persist across
 *          power cycles so needs to be explicitly set if updates shouldn't be allowed.
 *          If updates aren't allowed then this will prevent dynamic service additions, deletions
 *          and moves from being performed, even if these have been enabled, but it doesn't affect
 *          updates to the database when some form of search is being performed, such as a
 *          service search. This setting is used when the SDT, NIT and BAT tables are received.
 * @param   signal_type specifies the signal type that the setting will apply to. If this value
 *                      is given as SIGNAL_NONE, then the given flag will be applied to all
 *                      signal types (DVB-T, DVB-C and DVB-S).
 * @param   allow flag that defines whether service database updates should be allowed or not
 */
void ASI_AllowDatabaseUpdates(E_STB_DP_SIGNAL_TYPE signal_type, BOOLEAN allow)
{
   FUNCTION_START(ASI_AllowDatabaseUpdates);

   if (signal_type == SIGNAL_NONE)
   {
      db_updates_allowed.terrestrial = allow;
      db_updates_allowed.cable = allow;
      db_updates_allowed.satellite = allow;
   }
   else if (signal_type == SIGNAL_COFDM)
   {
      db_updates_allowed.terrestrial = allow;
   }
   else if (signal_type == SIGNAL_QAM)
   {
      db_updates_allowed.cable = allow;
   }
   else if (signal_type == SIGNAL_QPSK)
   {
      db_updates_allowed.satellite = allow;
   }

   FUNCTION_FINISH(ASI_AllowDatabaseUpdates);
}

/**
 * @brief   Returns whether service database updates are allowed for the given signal type
 * @param   signal_type signal type being queried
 * @return  TRUE if service updates are allowed, FALSE if not, or if SIGNAL_NONE is given
 */
BOOLEAN ASI_DatabaseUpdatesAllowed(E_STB_DP_SIGNAL_TYPE signal_type)
{
   BOOLEAN allowed;

   FUNCTION_START(ASI_DatabaseUpdatesAllowed);

   if (signal_type == SIGNAL_COFDM)
   {
      allowed = db_updates_allowed.terrestrial;
   }
   else if (signal_type == SIGNAL_QAM)
   {
      allowed = db_updates_allowed.cable;
   }
   else if (signal_type == SIGNAL_QPSK)
   {
      allowed = db_updates_allowed.satellite;
   }
   else
   {
      allowed = FALSE;
   }

   FUNCTION_FINISH(ASI_DatabaseUpdatesAllowed);

   return(allowed);
}


static BOOLEAN DynamicUpdateAddTransport(ADB_TRANSPORT_REC *transport, U16BIT num_lcns, SI_LCN_DESC * lcn_array,
   U16BIT num_hd_lcns, SI_LCN_DESC *hd_lcn_array, U16BIT num_nordig_lcns, SI_NORDIG_LCN_DESC *nordig_lcn_array)
{
   BOOLEAN retval;
   S_DYNAMIC_UPDATE_TRANSPORT *dyn_ts_ptr;

   retval = FALSE;

#ifdef DEBUG_DYNAMIC_UPDATE
   AP_SI_PRINT(("%s: tid 0x%04x, %u LCNs, %u HD LCNs, %u Nordig LCNs", __FUNCTION__, transport->tran_id,
      num_lcns, num_hd_lcns, num_nordig_lcns));
#endif

   /* Create a new transport entry to be added to the list */
   dyn_ts_ptr = (S_DYNAMIC_UPDATE_TRANSPORT *)STB_GetMemory(sizeof(S_DYNAMIC_UPDATE_TRANSPORT) +
      num_lcns * sizeof(SI_LCN_DESC) + num_hd_lcns * sizeof(SI_LCN_DESC));

   if (dyn_ts_ptr != NULL)
   {
      memset(dyn_ts_ptr, 0, sizeof(S_DYNAMIC_UPDATE_TRANSPORT));

      dyn_ts_ptr->t_ptr = transport;

      dyn_ts_ptr->num_lcns = num_lcns;
      dyn_ts_ptr->lcn_array = (SI_LCN_DESC *)(dyn_ts_ptr + 1);
      if (num_lcns != 0)
      {
         memcpy(dyn_ts_ptr->lcn_array, lcn_array, num_lcns * sizeof(SI_LCN_DESC));
      }

      dyn_ts_ptr->num_hd_lcns = num_hd_lcns;
      dyn_ts_ptr->hd_lcn_array = dyn_ts_ptr->lcn_array + dyn_ts_ptr->num_lcns;
      if (num_hd_lcns != 0)
      {
         memcpy(dyn_ts_ptr->hd_lcn_array, hd_lcn_array, num_hd_lcns * sizeof(SI_LCN_DESC));
      }

      /* The Nordig LCN structure isn't a simple structure to copy,
       * so the array passed in is used directly */
      dyn_ts_ptr->num_nordig_lcns = num_nordig_lcns;
      dyn_ts_ptr->nordig_lcn_array = nordig_lcn_array;

      if (dynamic_update_head == NULL)
      {
         dynamic_update_head = dyn_ts_ptr;
         dynamic_update_tail = dyn_ts_ptr;
      }
      else
      {
         dynamic_update_tail->next = dyn_ts_ptr;
         dynamic_update_tail = dyn_ts_ptr;
      }

      retval = TRUE;
   }

   return(retval);
}

static void DynamicUpdateAddService(ADB_TRANSPORT_REC *transport, ADB_SERVICE_REC *service, BOOLEAN check_move)
{
   S_DYNAMIC_UPDATE_TRANSPORT *dyn_ts_ptr;
   S_DYNAMIC_UPDATE_SERVICE *dyn_s_ptr;

#ifdef DEBUG_DYNAMIC_UPDATE
   AP_SI_PRINT(("%s: tid 0x%04x, sid 0x%04x, %s", __FUNCTION__, transport->tran_id, service->serv_id,
      (check_move ? "check move" : "")));
#endif

   if (check_move)
   {
      /* Check to see if there's been a service deleted with the same service ID
       * as the one being added. This will have been on a different transport so all
       * transports need to be checked. Without the support of a descriptor to define
       * the service move, the moved service can only be detected if it keeps the
       * same service ID */
      for (dyn_ts_ptr = dynamic_update_head, dyn_s_ptr = NULL; (dyn_ts_ptr != NULL) && (dyn_s_ptr == NULL);
         dyn_ts_ptr = dyn_ts_ptr->next)
      {
         for (dyn_s_ptr = dyn_ts_ptr->service_head; dyn_s_ptr != NULL; dyn_s_ptr = dyn_s_ptr->next)
         {
            if ((dyn_s_ptr->delete_s_ptr != NULL) && (dyn_s_ptr->delete_s_ptr->serv_id == service->serv_id))
            {
#ifdef DEBUG_DYNAMIC_UPDATE
               AP_SI_PRINT(("%s: Found deleted service with the same service ID, tid 0x%04x",
                  __FUNCTION__, dyn_s_ptr->delete_t_ptr->tran_id));
#endif
               /* A service with the same service ID is being deleted.
                * Update this entry with the details of the service being added */
               dyn_s_ptr->add_s_ptr = service;
               dyn_s_ptr->add_t_ptr = transport;
               break;
            }
         }

         if (dyn_s_ptr != NULL)
         {
            break;
         }
      }
   }
   else
   {
      dyn_ts_ptr = NULL;
   }

   if (dyn_ts_ptr == NULL)
   {
      /* Look for the transport the service is being added to */
      dyn_ts_ptr = dynamic_update_head;
      while ((dyn_ts_ptr != NULL) && (dyn_ts_ptr->t_ptr != transport))
      {
         dyn_ts_ptr = dyn_ts_ptr->next;
      }

      if (dyn_ts_ptr != NULL)
      {
         /* Create a new entry for this service */
         dyn_s_ptr = (S_DYNAMIC_UPDATE_SERVICE *)STB_GetMemory(sizeof(S_DYNAMIC_UPDATE_SERVICE));
         if (dyn_s_ptr != NULL)
         {
            memset(dyn_s_ptr, 0, sizeof(S_DYNAMIC_UPDATE_SERVICE));

            dyn_s_ptr->add_s_ptr = service;
            dyn_s_ptr->add_t_ptr = transport;

            /* Add the service */
            if (dyn_ts_ptr->service_head == NULL)
            {
               dyn_ts_ptr->service_head = dyn_s_ptr;
               dyn_ts_ptr->service_tail = dyn_s_ptr;
            }
            else
            {
               dyn_ts_ptr->service_tail->next = dyn_s_ptr;
               dyn_ts_ptr->service_tail = dyn_s_ptr;
            }
#ifdef DEBUG_DYNAMIC_UPDATE
            AP_SI_PRINT(("%s: New entry for service 0x%04x being added", __FUNCTION__, service->serv_id));
#endif
         }
      }
      else
      {
#ifdef DEBUG_DYNAMIC_UPDATE
         AP_SI_PRINT(("%s: Attempt to add service to a transport that isn't in the dynamic update list",
            __FUNCTION__));
#endif
      }
   }
}

static void DynamicUpdateDeleteService(ADB_TRANSPORT_REC *transport, ADB_SERVICE_REC *service, BOOLEAN check_move)
{
   S_DYNAMIC_UPDATE_TRANSPORT *dyn_ts_ptr;
   S_DYNAMIC_UPDATE_SERVICE *dyn_s_ptr;

#ifdef DEBUG_DYNAMIC_UPDATE
   AP_SI_PRINT(("%s: tid 0x%04x, sid 0x%04x, %s", __FUNCTION__, transport->tran_id, service->serv_id,
      (check_move ? "check move" : "")));
#endif

   if (check_move)
   {
      /* Check to see if there's been a service added with the same service ID
       * as the one being deleted. This will have been on a different transport so all
       * transports need to be checked. Without the support of a descriptor to define
       * the service move, the moved service can only be detected if it keeps the
       * same service ID */
      for (dyn_ts_ptr = dynamic_update_head, dyn_s_ptr = NULL; (dyn_ts_ptr != NULL) && (dyn_s_ptr == NULL);
         dyn_ts_ptr = dyn_ts_ptr->next)
      {
         for (dyn_s_ptr = dyn_ts_ptr->service_head; dyn_s_ptr != NULL; dyn_s_ptr = dyn_s_ptr->next)
         {
            if ((dyn_s_ptr->add_s_ptr != NULL) && (dyn_s_ptr->add_s_ptr->serv_id == service->serv_id))
            {
#ifdef DEBUG_DYNAMIC_UPDATE
               AP_SI_PRINT(("%s: Found added service with the same service ID, tid 0x%04x",
                  __FUNCTION__, dyn_s_ptr->add_t_ptr->tran_id));
#endif
               /* A service with the same service ID has been added.
                * Update this entry with the details of the service being deleted */
               dyn_s_ptr->delete_s_ptr = service;
               dyn_s_ptr->delete_t_ptr = transport;
               break;
            }
         }

         if (dyn_s_ptr != NULL)
         {
            break;
         }
      }
   }
   else
   {
      dyn_ts_ptr = NULL;
   }

   if (dyn_ts_ptr == NULL)
   {
      /* Look for the transport the service is being added to */
      dyn_ts_ptr = dynamic_update_head;
      while ((dyn_ts_ptr != NULL) && (dyn_ts_ptr->t_ptr != transport))
      {
         dyn_ts_ptr = dyn_ts_ptr->next;
      }

      if (dyn_ts_ptr != NULL)
      {
         /* Create a new entry for this service */
         dyn_s_ptr = (S_DYNAMIC_UPDATE_SERVICE *)STB_GetMemory(sizeof(S_DYNAMIC_UPDATE_SERVICE));
         if (dyn_s_ptr != NULL)
         {
            memset(dyn_s_ptr, 0, sizeof(S_DYNAMIC_UPDATE_SERVICE));

            dyn_s_ptr->delete_s_ptr = service;
            dyn_s_ptr->delete_t_ptr = transport;

            /* Add the service */
            if (dyn_ts_ptr->service_head == NULL)
            {
               dyn_ts_ptr->service_head = dyn_s_ptr;
               dyn_ts_ptr->service_tail = dyn_s_ptr;
            }
            else
            {
               dyn_ts_ptr->service_tail->next = dyn_s_ptr;
               dyn_ts_ptr->service_tail = dyn_s_ptr;
            }
         }
      }
      else
      {
#ifdef DEBUG_DYNAMIC_UPDATE
         AP_SI_PRINT(("%s: Attempt to delete service from a transport that isn't in the dynamic update list",
            __FUNCTION__));
#endif
      }
   }
}

static void ApplyDynamicUpdates(U8BIT path)
{
   BOOLEAN sdts_received;
   S_DYNAMIC_UPDATE_TRANSPORT *dyn_ts_ptr;
   S_DYNAMIC_UPDATE_SERVICE *dyn_s_ptr;
   ADB_SERVICE_REC *s_ptr;
   U16BIT reqd_lcn;
   U16BIT i, j;

   if (dynamic_update_head != NULL)
   {
      /* There are operations that may need to be applied; check that all SDTs have been received first */
      DBDEF_RequestAccess();

      sdts_received = TRUE;

      for (dyn_ts_ptr = dynamic_update_head; (dyn_ts_ptr != NULL) && sdts_received;
         dyn_ts_ptr = dyn_ts_ptr->next)
      {
#if 0
         if (TRUE != dyn_ts_ptr->t_ptr->sdt_received) {
             sdts_received = FALSE;
         }
#else
         sdts_received = dyn_ts_ptr->t_ptr->sdt_received;
#endif
      }

      DBDEF_ReleaseAccess();

      if (sdts_received)
      {
         /* All SDTs have been received so the required operations can be performed */
         DBDEF_RequestAccess();

         for (dyn_ts_ptr = dynamic_update_head; dyn_ts_ptr != NULL; dyn_ts_ptr = dyn_ts_ptr->next)
         {
#ifdef DEBUG_DYNAMIC_UPDATE
            AP_SI_PRINT(("%s: Apply updates for transport 0x%04x", __FUNCTION__, dyn_ts_ptr->t_ptr->tran_id));
#endif
            /* Iterate the service entries */
            for (dyn_s_ptr = dyn_ts_ptr->service_head; dyn_s_ptr != NULL; dyn_s_ptr = dyn_s_ptr->next)
            {
               /* Service addition will already have been performed, but an event needs to be sent */
               if ((dyn_s_ptr->add_s_ptr != NULL) && (dyn_s_ptr->delete_s_ptr == NULL))
               {
#ifdef DEBUG_DYNAMIC_UPDATE
                  AP_SI_PRINT(("%s: Added service 0x%04x to transport 0x%04x", __FUNCTION__,
                     dyn_s_ptr->add_s_ptr->serv_id, dyn_s_ptr->add_t_ptr->tran_id));
#endif
                  /* Send notification of the service that's been added */
                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_ADDED,
                     &dyn_s_ptr->add_s_ptr, sizeof(void *));
               }
               else if ((dyn_s_ptr->add_s_ptr != NULL) && (dyn_s_ptr->delete_s_ptr != NULL))
               {
                  /* Service has been moved.
                   * As the original service may be being referenced, the service to be deleted
                   * is updated with info from the service that's been added and its transport
                   * is changed, and the service that's been added is then deleted */
#ifdef DEBUG_DYNAMIC_UPDATE
                  AP_SI_PRINT(("%s: Move service 0x%04x (new sid 0x%04x) from 0x%04x to 0x%04x",
                     __FUNCTION__,
                     dyn_s_ptr->delete_s_ptr->serv_id, dyn_s_ptr->add_s_ptr->serv_id,
                     dyn_s_ptr->delete_t_ptr->tran_id, dyn_s_ptr->add_t_ptr->tran_id));
#endif
                  dyn_s_ptr->delete_s_ptr->transport = dyn_s_ptr->add_t_ptr;

                  if (dyn_s_ptr->delete_s_ptr->dba_rec != NULL)
                  {
                     DBA_SetRecordParent(dyn_s_ptr->delete_s_ptr->dba_rec, dyn_s_ptr->add_t_ptr->dba_rec);

                     DBA_SaveRecord(dyn_s_ptr->delete_s_ptr->dba_rec);
                  }

                  DBDEF_DeleteServiceRec(dyn_s_ptr->add_s_ptr);

                  if (ADB_GetTunedService(path) == dyn_s_ptr->delete_s_ptr)
                  {
                     /* Notify that the service has moved as a retune is required to stay on it */
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_MOVED,
                        &dyn_s_ptr->delete_s_ptr, sizeof(void *));
                  }
               }
               else if (dyn_s_ptr->delete_s_ptr != NULL)
               {
                  if (ADB_GetTunedService(path) != dyn_s_ptr->delete_s_ptr)
                  {
#ifdef DEBUG_DYNAMIC_UPDATE
                     AP_SI_PRINT(("%s: Delete service 0x%04x from transport 0x%04x", __FUNCTION__,
                        dyn_s_ptr->delete_s_ptr->serv_id, dyn_s_ptr->delete_t_ptr->tran_id));
#endif
                     /* Delete the service */
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_SERVICE_DELETED,
                        &dyn_s_ptr->delete_s_ptr, sizeof(dyn_s_ptr->delete_s_ptr));

                     DBDEF_DeleteServiceRec(dyn_s_ptr->delete_s_ptr);
                  }
                  else
                  {
                     /* This is the currently tuned service so can't be deleted yet */
#ifdef DEBUG_DYNAMIC_UPDATE
                     AP_SI_PRINT(("%s: Service 0x%04x on transport 0x%04x is the current service, not deleting",
                        __FUNCTION__, dyn_s_ptr->delete_s_ptr->serv_id, dyn_s_ptr->delete_t_ptr->tran_id));
#endif
                     STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_DELETE_SERVICE,
                        &dyn_s_ptr->delete_s_ptr, sizeof(dyn_s_ptr->delete_s_ptr));
                  }
               }
            }

            /* Now apply any LCN changes */
            if ((dyn_ts_ptr->num_nordig_lcns != 0) && (dyn_ts_ptr->nordig_lcn_array != NULL))
            {
               for (i = 0; i < dyn_ts_ptr->num_nordig_lcns; i++)
               {
                  for (j = 0; j < dyn_ts_ptr->nordig_lcn_array[i].num_services; j++)
                  {
                     s_ptr = DBDEF_FindServiceRec(dyn_ts_ptr->nordig_lcn_array[i].serv_array[j].serv_id,
                        dyn_ts_ptr->t_ptr);
                     if (s_ptr != NULL)
                     {
                        if (s_ptr->hidden != !dyn_ts_ptr->nordig_lcn_array[i].serv_array[j].visible)
                        {
                           s_ptr->hidden = !dyn_ts_ptr->nordig_lcn_array[i].serv_array[j].visible;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);
#ifdef DEBUG_DYNAMIC_UPDATE
                           AP_SI_PRINT(("%s: service 0x%04x, hidden flag now %u", __FUNCTION__,
                              s_ptr->serv_id, s_ptr->hidden));
#endif
                        }

                        s_ptr->selectable = TRUE;
                        DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                        reqd_lcn = dyn_ts_ptr->nordig_lcn_array[i].serv_array[j].serv_lcn;
                        if (reqd_lcn != s_ptr->serv_lcn)
                        {
                           /* LCN is being changed so reset the one that's currently allocated
                            * to this service so it's reassigned */
                           s_ptr->allocated_lcn = 0;
                           s_ptr->serv_lcn = reqd_lcn;
                           DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
#ifdef DEBUG_DYNAMIC_UPDATE
                           AP_SI_PRINT(("%s: service 0x%04x, LCN %u", __FUNCTION__, s_ptr->serv_id, reqd_lcn));
#endif
                        }

                        DBA_SaveRecord(s_ptr->dba_rec);
                     }
                  }
               }
            }
            else
            {
               for (i = 0; i < dyn_ts_ptr->num_lcns; i++)
               {
                  s_ptr = DBDEF_FindServiceRec(dyn_ts_ptr->lcn_array[i].serv_id, dyn_ts_ptr->t_ptr);
                  if (s_ptr != NULL)
                  {
                     /* Set service hidden attribute from LCN descriptor */
                     if (s_ptr->hidden != !dyn_ts_ptr->lcn_array[i].visible)
                     {
                        s_ptr->hidden = !dyn_ts_ptr->lcn_array[i].visible;
                        DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_HIDDEN, s_ptr->hidden);
#ifdef DEBUG_DYNAMIC_UPDATE
                        AP_SI_PRINT(("%s: service 0x%04x, hidden flag now %u", __FUNCTION__,
                           s_ptr->serv_id, s_ptr->hidden));
#endif
                     }

                     /* A hidden service still needs to be selectable, so it's aways TRUE in this case */
                     s_ptr->selectable = TRUE;
                     DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_SELECTABLE, s_ptr->selectable);

                     reqd_lcn = dyn_ts_ptr->lcn_array[i].serv_lcn;
                     if (reqd_lcn != s_ptr->serv_lcn)
                     {
                        /* LCN is being changed so reset the one that's currently allocated
                         * to this service so it's reassigned */
                        s_ptr->allocated_lcn = 0;
                        s_ptr->serv_lcn = reqd_lcn;
                        DBA_SetFieldValue(s_ptr->dba_rec, DBA_FIELD_SERV_REQ_LCN, s_ptr->serv_lcn);
#ifdef DEBUG_DYNAMIC_UPDATE
                        AP_SI_PRINT(("%s: service 0x%04x, LCN %u", __FUNCTION__, s_ptr->serv_id, reqd_lcn));
#endif
                     }

                     DBA_SaveRecord(s_ptr->dba_rec);
                  }
               }
            }

            if (dyn_ts_ptr->num_hd_lcns != 0)
            {
               /* Now take into account any HD LCN assignments */
               for (i = 0; i < dyn_ts_ptr->num_hd_lcns; i++)
               {
                  s_ptr = DBDEF_FindServiceRec(dyn_ts_ptr->hd_lcn_array[i].serv_id, dyn_ts_ptr->t_ptr);

                  if ((s_ptr != NULL) && (s_ptr->old_allocated_lcn == 0))
                  {
                     /* The HD LCN descriptor will be used when allocating LCNs and will be freed
                      * when ADB_ReleaseDatabaseSearchData is called */
                     s_ptr->hd_lcn_desc = STB_GetMemory(sizeof(SI_LCN_DESC));
                     memcpy(s_ptr->hd_lcn_desc, &dyn_ts_ptr->hd_lcn_array[i], sizeof(SI_LCN_DESC));
#ifdef DEBUG_DYNAMIC_UPDATE
                     AP_SI_PRINT(("%s: service 0x%04x has an HD LCN, %u", s_ptr->serv_id,
                        s_ptr->hd_lcn_desc->serv_lcn));
#endif
                  }
               }
            }
         }

         /* The dynamic update list can now be freed */
         ClearDynamicUpdates();

         DBDEF_AllocateLcns(STB_DPGetSignalType(path), TRUE);
         DBDEF_SortServicesByLcn();

         DBDEF_ReleaseAccess();

         ADB_ReleaseDatabaseSearchData();

         ADB_SaveDatabase();
      }
   }
}

static void ClearDynamicUpdates(void)
{
   S_DYNAMIC_UPDATE_TRANSPORT *next_ts_ptr;
   S_DYNAMIC_UPDATE_SERVICE *next_s_ptr;

#ifdef DEBUG_DYNAMIC_UPDATE
   AP_SI_PRINT(("%s", __FUNCTION__));
#endif

   for ( ; dynamic_update_head != NULL; dynamic_update_head = next_ts_ptr)
   {
      next_ts_ptr = dynamic_update_head->next;

      for ( ; dynamic_update_head->service_head != NULL; dynamic_update_head->service_head = next_s_ptr)
      {
         next_s_ptr = dynamic_update_head->service_head->next;

         STB_FreeMemory(dynamic_update_head->service_head);
      }

      if (dynamic_update_head->nordig_lcn_array != NULL)
      {
         STB_SIReleaseNordigLcn2DescArray(dynamic_update_head->nordig_lcn_array,
            dynamic_update_head->num_nordig_lcns);
      }

      STB_FreeMemory(dynamic_update_head);
   }

   dynamic_update_tail = NULL;
}

