/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Timer handling functions used by the application
 *
 * @file    ap_tmr.c
 * @date    27/05/2004
 * @author  Ocean Blue
 */

//---includes for this file-------------------------------------------------------------------------
// compiler library header files

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //obs debug functions

#include "stbhwdsk.h"
#include "stbhwos.h"
#include "stbhwc.h"     // STB_SPDebugWrite

#include "stbgc.h"
#include "stbdpc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbpvr.h"     // STB PVR
#include "stbpvrmsg.h"
#include "stbheap.h"    // STB_AppFreeMemory
#include "stbuni.h"     // STB_ConcatUnicodeStrings()
#include "stbllist.h"
#include "stberc.h"

#ifdef INTEGRATE_HBBTV
#include "hbbtv_api.h"
#endif

#include "app.h"
#include "ap_pvr.h"     // for PVR functionality

#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_tmr.h"
#include "ap_dbdef.h"
#include "ap_cntrl.h"
#include "ap_state.h"

#include "dba.h"

#ifdef PVR_LOG
extern FILE *pvr_log;
extern void LogDateTime();
#endif

//---constant definitions for this file-------------------------------------------------------------

#if 0
#ifdef FUNCTION_START
#undef FUNCTION_START
#endif
#define FUNCTION_START(X)  STB_SPDebugWrite(">> %s\n", #X)

#ifdef FUNCTION_FINISH
#undef FUNCTION_FINISH
#endif
#define FUNCTION_FINISH(X)  STB_SPDebugWrite("<< %s\n", #X)
#endif

/*#define TMR_DEBUG*/
#ifdef TMR_DEBUG
#define TMR_DBG(x)      STB_SPDebugWrite x
#else
#define TMR_DBG(x)
#endif

/*#define TMR_PVR_DEBUG*/
#ifdef TMR_PVR_DEBUG
#define TMR_PVR_DBG(x)  STB_SPDebugWrite x
#else
#define TMR_PVR_DBG(x)
#endif

/* The number of seconds added to a timer to allow an event triggered recording to overrun
 * before the recording is stopped */
#define EVENT_DURATION_OVERRUN         (2 * 60 * 60)
#define NORDIG_EVENT_DURATION_OVERRUN  (4 * 60 * 60)


//---local typedefs, structs, enumerations for this file--------------------------------------------

/* This structure is used for timer database filters */
typedef struct
{
   U32DHMS start_time;
   U32DHMS end_time;
} S_TIME_SLOT;

typedef struct
{
   U32BIT timer_handle;
   BOOLEAN is_recommendation;
   BOOLEAN do_not_delete;
   U8BIT prog_crid[TMR_PVR_CRID_LEN_MAX];
   U8BIT other_crid[TMR_PVR_CRID_LEN_MAX];
} S_ALT_EVENT_DATA;


//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)
static void *alt_event_queue;

//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static BOOLEAN DeleteTimer(U32BIT handle);

// ---- PVR action timer ----
static BOOLEAN ActionTimerRecStart(ADB_TIMER_REC *timer);
static void AlternateEventTask(void *param);

static void TimerTask(void *param);
static BOOLEAN StartRecordTimer(ADB_TIMER_REC *timer, BOOLEAN recordings_can_start, BOOLEAN *start_record);
static BOOLEAN HasTimerExpired(U32BIT handle);
static BOOLEAN RescheduleTimer(ADB_TIMER_REC *timer);

#ifdef TMR_DEBUG
static void DumpTimer(ADB_TIMER_REC *timer);
#endif

static void GetActualStartEndTime(U32DHMS timer_start, U32DHMS timer_duration, S32BIT start_padding, S32BIT end_padding,
   U32DHMS *start_time, U32DHMS *end_time);
static void SetTimerFields(ADB_TIMER_REC *timer, S_TIMER_INFO *info);
static U8BIT GetNumSimultaneousRecordings(U32BIT handle, U16BIT onet_id, U16BIT trans_id,
   U16BIT serv_id, U8BIT max_recordings, U32DHMS start_date_time,
   U32DHMS end_date_time, BOOLEAN include_start_padding, BOOLEAN include_end_padding,
   U32BIT **conflicting_timers);
static BOOLEAN SetStartPadding(ADB_TIMER_REC *timer, S32BIT padding);
static S32BIT GetStartPadding(ADB_TIMER_REC *timer);
static BOOLEAN SetEndPadding(ADB_TIMER_REC *timer, S32BIT padding);
static S32BIT GetEndPadding(ADB_TIMER_REC *timer);


//---global function definitions-----------------------------------------------

/**
 * @brief   Performs initialisation of the timers, reading existing entries from the database,
 *          rescheduling or deleting any timers that have been missed and starting background tasks
 */
void ATMR_Initialise(void)
{
   FUNCTION_START(ATMR_Initialise);

   TMR_DBG(("ATMR_Initialise"));

   /* Start the main timer background task */
   STB_OSCreateTask(TimerTask, NULL, 4096, 11, (U8BIT *)"TimerTask");

   /* Create queue and task used to find alternative events for recording */
   alt_event_queue = STB_OSCreateQueue(sizeof(S_ALT_EVENT_DATA), 30);
   STB_OSCreateTask(AlternateEventTask, NULL, 4096, 6, (U8BIT *)"AltEvent");

   FUNCTION_FINISH(ATMR_Initialise);
}

/**
 * @brief   Creates a new timer based on the information supplied
 * @param   info timer info used to create the timer
 * @return  timer handle, or INVALID_TIMER_HANDLE if creation fails
 */
U32BIT ATMR_AddTimer(S_TIMER_INFO *info)
{
   U32BIT handle;
   ADB_TIMER_REC *timer;
   void *s_ptr;

   FUNCTION_START(ATMR_AddTimer);

   handle = INVALID_TIMER_HANDLE;

   if ((info != NULL) && ((info->type == TIMER_TYPE_ALARM) || (info->type == TIMER_TYPE_SLEEP) ||
                          (info->type == TIMER_TYPE_PVR_RECORD) || (info->type == TIMER_TYPE_PRIVATE)))
   {
      DBDEF_RequestAccess();

      timer = DBDEF_AddTimerRec(!info->ram_only);
      if (timer != NULL)
      {
         SetTimerFields(timer, info);

         if (timer->type == TIMER_TYPE_PVR_RECORD)
         {
#ifdef INTEGRATE_HBBTV
            HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_NEWLY_SCHEDULED);
#endif
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_BOOKING_CREATED,
               (void *)&timer->handle, sizeof(timer->handle));

            /* If the timer is event triggered and the event has already started
             * then recording should be started now */
            s_ptr = ADB_FindServiceByIds(timer->u.record.orig_net_id, timer->u.record.transport_id,
               timer->u.record.service_id);
            ATMR_CheckRecordStatus(TRUE, s_ptr);
         }

         if (timer->dba_rec != NULL)
         {
            DBA_SaveRecord(timer->dba_rec);
         }

         handle = timer->handle;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ATMR_AddTimer);

   return(handle);
}

/**
 * @brief   Creates a timer based on the given event. If a recording timer is created, it will
 *          be set to record on the default disk, which can be changed afterwards using
 *          ATMR_SetDiskId.
 * @param   event_ptr - pointer to the event to be recorded
 * @param   serv_ptr service the event is on
 * @param   record TRUE if a recording timer is to be created, FALSE for an alarm timer
 * @param   event_triggered TRUE if the timer should be triggered by the change of event
 *                          rather than the time
 * @return  timer handle, or INVALID_TIMER_HANDLE if the timer couldn't be created
 */
U32BIT ATMR_AddTimerForEvent(void *event_ptr, void *serv_ptr, BOOLEAN record, BOOLEAN event_triggered)
{
   U32BIT handle;
   S_TIMER_INFO info;
   U8BIT *string;
   U16BIT str_len;
   U16BIT onet_id;
   U16BIT trans_id;
   U16BIT serv_id;

   FUNCTION_START(ATMR_AddTimerForEvent);

   handle = INVALID_TIMER_HANDLE;

   if ((event_ptr != NULL) && (serv_ptr != NULL))
   {
      memset(&info, 0, sizeof(info));

      info.frequency = TIMER_FREQ_ONCE;
      info.start_time = ADB_GetEventStartDateTime(event_ptr);

      if ((string = ADB_GetEventName(event_ptr)) != NULL)
      {
         if ((str_len = STB_GetNumBytesInString(string)) <= TMR_MAX_NAME_LENGTH)
         {
            memcpy(info.name, string, str_len);
         }
         else
         {
            /* Event name too long, truncate with null */
            memcpy(info.name, string, TMR_MAX_NAME_LENGTH);
            info.name[TMR_MAX_NAME_LENGTH - 1] = '\0';
         }

         STB_ReleaseUnicodeString(string);
      }

      ADB_GetServiceIds(serv_ptr, &onet_id, &trans_id, &serv_id);

      if (record)
      {
         info.type = TIMER_TYPE_PVR_RECORD;
         info.u.record.duration = ADB_GetEventDuration(event_ptr);
         info.u.record.service_id = serv_id;
         info.u.record.transport_id = trans_id;
         info.u.record.orig_net_id = onet_id;

         info.u.record.start_padding = 0;
         info.u.record.end_padding = 0;

         info.u.record.event_triggered = event_triggered;
         if (event_triggered)
         {
            info.u.record.event_id = ADB_GetEventId(event_ptr);
         }

         /* Set recording onto the default disk */
         info.u.record.disk_id = STB_PVRGetDefaultDisk();
      }
      else
      {
         info.type = TIMER_TYPE_ALARM;
         info.u.alarm.service_id = serv_id;
         info.u.alarm.transport_id = trans_id;
         info.u.alarm.orig_net_id = onet_id;
      }

      handle = ATMR_AddTimer(&info);
   }

   FUNCTION_FINISH(ATMR_AddTimerForEvent);

   return(handle);
}

/**
 * @brief   Updates the duration for an existing PVR recording timer
 * @param   handle handle of timer to be updated
 * @param   duration duration to be set in the timer
 * @return  TRUE if the timer is updated, FALSE otherwise
 */
BOOLEAN ATMR_UpdateTimerDuration(U32BIT handle, U32DHMS duration)
{
   BOOLEAN retval;
   ADB_TIMER_REC *timer;
   U32DHMS old_duration;
   U32DHMS time_diff;

   FUNCTION_START(ATMR_UpdateTimerDuration);

   retval = FALSE;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (timer->dba_rec != NULL)
      {
         /* Once a recording has started the duration held in memory may have been adjusted,
          * e.g. to prevent overrun if event triggered, so the value held in the database
          * is read (as this won't have been adjusted) to calculate the difference between
          * the old duration and the new, which can then be applied to the timer */
         DBA_GetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DURATION, &old_duration);
      }
      else
      {
         old_duration = timer->u.record.duration;
      }

      if (old_duration < duration)
      {
         /* Duration is being extended */
         time_diff = STB_GCCalculateDHMS(duration, old_duration, CALC_SUB);
         timer->u.record.duration = STB_GCCalculateDHMS(timer->u.record.duration, time_diff, CALC_ADD);
      }
      else
      {
         /* Duration is being shortened */
         time_diff = STB_GCCalculateDHMS(old_duration, duration, CALC_SUB);
         timer->u.record.duration = STB_GCCalculateDHMS(timer->u.record.duration, time_diff, CALC_SUB);
      }

      if (timer->dba_rec != NULL)
      {
         /* Update the database and save the change */
         DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DURATION, duration);
         DBA_SaveRecord(timer->dba_rec);
         ADB_SaveDatabase();
      }

      retval = TRUE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_UpdateTimerDuration);

   return(retval);
}

/**
 * @brief   Updates all the timer fields
 * @param   handle handle of timer to be updated
 * @param   info timer info used to update the timer
 * @return  TRUE if the timer exists and is deleted, FALSE otherwise
 */
BOOLEAN ATMR_UpdateTimer(U32BIT handle, S_TIMER_INFO *info)
{
   BOOLEAN retval;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_UpdateTimer);

   retval = FALSE;
   if ((info != NULL) && ((info->type == TIMER_TYPE_ALARM) || (info->type == TIMER_TYPE_SLEEP) ||
      (info->type == TIMER_TYPE_PVR_RECORD) || (info->type == TIMER_TYPE_PRIVATE)))
   {
      DBDEF_RequestAccess();

      timer = DBDEF_FindTimerRec(handle);
      if (timer != NULL)
      {
         SetTimerFields(timer, info);

         if (timer->dba_rec != NULL)
         {
            DBA_SaveRecord(timer->dba_rec);
         }

         retval = TRUE;
      }

      DBDEF_ReleaseAccess();
   }

   FUNCTION_FINISH(ATMR_UpdateTimer);

   return(retval);
}

/**
 * @brief   Delete the timer with the given handle
 * @param   handle handle of timer to be deleted
 * @return  TRUE if timer exists and is deleted, FALSE otherwise
 */
BOOLEAN ATMR_DeleteTimer(U32BIT handle)
{
   BOOLEAN timer_deleted;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_DeleteTimer);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      timer_deleted = DeleteTimer(handle);
   }
   else
   {
      timer_deleted = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_DeleteTimer);

   return(timer_deleted);
}

/**
 * @brief   Returns a list of all the timer handles and the number of items in the list.
 * @param   timer_list pointer to an array of timer handles, the array is allocated by this function
 * @param   list_size_ptr pointer to the number of items in the returned list
 * @param   list_type type of timers to put in the list
 * @param   date_time_order TRUE if the list is to be returned in date/time order,
 *                          FALSE returns the list in alphabetical name order
 * @return  TRUE if a list is returned, FALSE otherwise
 */
BOOLEAN ATMR_GetTimerList(U32BIT **timer_list, U16BIT *list_size_ptr, E_TIMER_TYPE list_type,
   BOOLEAN date_time_order)
{
   ADB_TIMER_REC *timer;
   U16BIT list_size, index;
   BOOLEAN result;

   FUNCTION_START(ATMR_GetTimerList);

   result = TRUE;

   DBDEF_RequestAccess();

   /* Sort the timer list */
   DBDEF_SortTimers(date_time_order);

   /* Find the number of timers of the type requested */
   for (list_size = 0, timer = DBDEF_GetNextTimerRec(NULL); timer != NULL;
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if ((list_type == TIMER_TYPE_ALL) || (timer->type == list_type))
      {
         list_size++;
      }
   }

   if (list_size > 0)
   {
      *timer_list = (U32BIT *)STB_AppGetMemory(list_size * sizeof(U32BIT));
      if (*timer_list == NULL)
      {
         list_size = 0;
         result = FALSE;
      }
      else
      {
         *list_size_ptr = list_size;

         for (index = 0, timer = DBDEF_GetNextTimerRec(NULL); timer != NULL;
              timer = DBDEF_GetNextTimerRec(timer))
         {
            if ((list_type == TIMER_TYPE_ALL) || (timer->type == list_type))
            {
               (*timer_list)[index] = timer->handle;
               index++;
            }
         }
      }
   }
   else
   {
      *timer_list = NULL;
      *list_size_ptr = 0;
      result = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetTimerList);

   return(result);
}

/**
 * @brief   Release the given array of timer handles
 * @param   timer_list - timer array to be freed
 * @param   list_size - number of items in the list
 */
void ATMR_ReleaseTimerList(U32BIT *timer_list, U16BIT list_size)
{
   FUNCTION_START(ATMR_ReleaseTimerList);

   USE_UNWANTED_PARAM(list_size);

   if (timer_list != NULL)
   {
      STB_AppFreeMemory(timer_list);
   }

   FUNCTION_FINISH(ATMR_ReleaseTimerList);
}

/**
 * @brief   Finds the timer using the given decode path and if it's a recording timer
 *          the recording will be started
 * @param   path decode path being used for the recording
 * @return  TRUE if the recording is started, FALSE otherwise
 */
BOOLEAN ATMR_StartRecord(U8BIT path)
{
   ADB_TIMER_REC *timer;
   U32BIT recording_handle;
   U32DHMS stop_date_time;
   BOOLEAN timer_found;
   BOOLEAN recording_started;

   FUNCTION_START(ATMR_StartRecord);

   TMR_PVR_DBG(("ATMR_StartRecord(%d)", path));

   DBDEF_RequestAccess();

   timer_found = FALSE;
   recording_started = FALSE;

   /* Find the recording timer waiting to be started on the given path */
   for (timer = DBDEF_GetNextTimerRec(NULL); !timer_found && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if ((timer->type == TIMER_TYPE_PVR_RECORD) && timer->starting && (timer->u.record.path == path))
      {
         /* Found the timer related to the recording to be started */
         timer_found = TRUE;

         /* Get starting timer values */
         stop_date_time = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);

         /* Stop timer in the future (stop >= now) */
         if (STB_GCIsFutureDateTime(DHMS_DATE(stop_date_time), DHMS_HOUR(stop_date_time),
                DHMS_MINS(stop_date_time), DHMS_SECS(stop_date_time)))
         {
            TMR_PVR_DBG(("ATMR_StartRecord: stop timer in future - start new recording"));

            STB_PVRStartRecordRunning(path);

#ifdef INTEGRATE_HBBTV
            HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_ACQUIRING_RESOURCES);
#endif

            /* Start new recording */
            if (APVR_StartNewRecording(timer->u.record.disk_id, path, timer->name,
                   timer->u.record.event_id, timer->u.record.prog_crid, timer->u.record.other_crid,
                   &recording_handle))
            {
               TMR_PVR_DBG(("ATMR_StartRecord(%u): Started recording \"%s\"", path, timer->name));

               /* Set the recording handle in the timer */
               timer->u.record.recording_handle = recording_handle;
               timer->u.record.programme_started = TRUE;
               timer->u.record.programme_finished = FALSE;

               timer->starting = FALSE;
               timer->started = TRUE;
               recording_started = TRUE;

               STB_PVRRecordingSetAdditionalInfo(recording_handle, timer->u.record.additional_info);
               STB_PVRRecordingSetStartPadding(recording_handle, GetStartPadding(timer));
               STB_PVRRecordingSetEndPadding(recording_handle, GetEndPadding(timer));
               STB_PVRRecordingSetLocked(recording_handle, timer->u.record.do_not_delete);

#ifdef INTEGRATE_HBBTV
               HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_STARTED);
#endif
            }
            else
            {
               /* Failed to start the recording */
               TMR_PVR_DBG(("ATMR_StartRecord(%u): Failed to start recording \"%s\"", path, timer->name));

               /* Mark the timer as not started and missed so that it's rescheduled, if possible */
               timer->missed = TRUE;
               timer->started = FALSE;
               timer->starting = FALSE;

#ifdef INTEGRATE_HBBTV
               HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_FAILED_UNKNOWN_ERROR);
#endif
            }
         }
         else
         {
            /* Stop time in the past, so make sure the timer is no longer referencing a recording */
            TMR_PVR_DBG(("ATMR_StartRecord(%u): Timer is marked as starting but the stop time is in the past", path));
            timer->u.record.recording_handle = STB_PVR_INVALID_HANDLE;

            /* Mark the timer as not started and missed so that it's rescheduled, if possible */
            timer->missed = TRUE;
            timer->starting = FALSE;
            timer->started = FALSE;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_StartRecord);

   return(recording_started);
}

/**
 * @brief   Handles the timer when a recording fails to start for some reason.
 *          This may result in the timer being deleted.
 * @param   path - recording path
 */
void ATMR_RecordingFailed(U8BIT path)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;

   FUNCTION_START(ATMR_RecordingFailed);

   DBDEF_RequestAccess();

   /* Find the timer with the given recording handle */
   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      if ((timer->type == TIMER_TYPE_PVR_RECORD) && timer->starting && (timer->u.record.path == path))
      {
         next_timer = NULL;

         if ((timer->frequency == TIMER_FREQ_ONCE) &&
             (STB_GetNumBytesInString(timer->u.record.prog_crid) == 0))
         {
            /* Timer isn't to be repeated and there's no crid so can't be rescheduled, so destroy it */
            TMR_PVR_DBG(("ATMR_RecordingFailed(%u): Destroy timer 0x%08lx", path, timer->handle));

            DeleteTimer(timer->handle);

            ADB_SaveDatabase();
         }
         else
         {
            /* Mark the timer as not started and missed so that it's rescheduled */
            timer->missed = TRUE;
            timer->starting = FALSE;
            timer->started = FALSE;
         }
      }
      else
      {
         next_timer = DBDEF_GetNextTimerRec(timer);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_RecordingFailed);
}

/**
 * @brief   Sets up the given timer info structure with default values for the given timer type
 *          using the service and event depending on timer type.
 * @param   timer_info pointer to timer info structure to be initialised
 * @param   timer_type type of timer to initialise the structure for
 * @param   serv_ptr service that will be used for alarm and PVR recording timers. It can be NULL,
 *          if not, it is used to initialise the service IDs in the alarm and PVR timers and to
 *          initialise the programme CRID field in PVR timers.
 * @param   event_ptr event that will be used for alarm and PVR recording timers. It can be NULL,
 *          if not, it is used to initialise start time and name in alarm and PVR timers and other
 *          event related fields in PVR timers. When event_ptr is specified for a PVR timer, it
 *          will be initialised as 'event triggered';
 * @return  TRUE if the structure was setup, FALSE otherwise
 */
BOOLEAN ATMR_InitialiseTimer(S_TIMER_INFO *timer_info, E_TIMER_TYPE timer_type, void *serv_ptr,
   void *event_ptr)
{
   BOOLEAN retval;
   U8BIT *name;
   U8BIT *crid;
   U32BIT name_len;
   U32DHMS now;
   U32DHMS time_diff;

   FUNCTION_START(ATMR_InitialiseTimer);

   retval = FALSE;

   if (timer_info != NULL)
   {
      memset(timer_info, 0, sizeof(*timer_info));

      /* Set default values */
      timer_info->type = timer_type;
      timer_info->frequency = TIMER_FREQ_ONCE;

      retval = TRUE;

      switch (timer_info->type)
      {
         case TIMER_TYPE_SLEEP:
         {
            /* Set sleep time to now assuming this will be modified later */
            timer_info->start_time = STB_GCNowDHMSGmt();
            break;
         }

         case TIMER_TYPE_ALARM:
         {
            timer_info->u.alarm.ramp_volume = FALSE;

            if (serv_ptr != NULL)
            {
               timer_info->u.alarm.change_service = TRUE;
               ADB_GetServiceIds(serv_ptr, &timer_info->u.alarm.orig_net_id,
                  &timer_info->u.alarm.transport_id, &timer_info->u.alarm.service_id);
            }
            else
            {
               timer_info->u.alarm.change_service = FALSE;
            }

            if (event_ptr != NULL)
            {
               /* Set the alarm based on the given event */
               timer_info->start_time = ADB_GetEventStartDateTime(event_ptr);
               if ((name = ADB_GetEventName(event_ptr)) != NULL)
               {
                  name_len = STB_GetNumBytesInString(name);
                  if (name_len > TMR_MAX_NAME_LENGTH)
                  {
                     memcpy(timer_info->name, name, TMR_MAX_NAME_LENGTH);
                     timer_info->name[TMR_MAX_NAME_LENGTH - 1] = '\0';
                  }
                  else
                  {
                     memcpy(timer_info->name, name, name_len);
                  }
               }
            }
            else
            {
               /* Set alarm time to now assuming this will be modified later */
               timer_info->start_time = STB_GCNowDHMSGmt();
            }
            break;
         }

         case TIMER_TYPE_PVR_RECORD:
         {
            timer_info->u.record.disk_id = STB_PVRGetDefaultDisk();

            if (serv_ptr != NULL)
            {
               ADB_GetServiceIds(serv_ptr, &timer_info->u.record.orig_net_id,
                  &timer_info->u.record.transport_id, &timer_info->u.record.service_id);
            }

            if (event_ptr != NULL)
            {
               if ((name = ADB_GetEventName(event_ptr)) != NULL)
               {
                  name_len = STB_GetNumBytesInString(name);
                  if (name_len > TMR_MAX_NAME_LENGTH)
                  {
                     memcpy(timer_info->name, name, TMR_MAX_NAME_LENGTH);
                     timer_info->name[TMR_MAX_NAME_LENGTH - 1] = '\0';
                  }
                  else
                  {
                     memcpy(timer_info->name, name, name_len);
                  }
               }

               timer_info->u.record.event_id = ADB_GetEventId(event_ptr);
               timer_info->u.record.event_triggered = TRUE;
               timer_info->start_time = ADB_GetEventStartDateTime(event_ptr);
               timer_info->u.record.duration = ADB_GetEventDuration(event_ptr);
               timer_info->u.record.start_padding = 0;
               timer_info->u.record.end_padding = 0;

               if (!STB_GCIsFutureDateTime(DHMS_DATE(timer_info->start_time),
                  DHMS_HOUR(timer_info->start_time), DHMS_MINS(timer_info->start_time),
                  DHMS_SECS(timer_info->start_time)))
               {
                  /* Event has already started so the recording needs to start immediately */
                  now = STB_GCNowDHMSGmt();
                  time_diff = STB_GCCalculateDHMS(now, timer_info->start_time, CALC_SUB);
                  timer_info->start_time = now;
                  timer_info->u.record.duration = STB_GCCalculateDHMS(timer_info->u.record.duration,
                     time_diff, CALC_SUB);
               }

               crid = ADB_GetEventFullProgrammeCrid(serv_ptr, event_ptr);
               if (crid != NULL)
               {
                  strncpy((char *)timer_info->u.record.prog_crid, (char *)crid, TMR_PVR_CRID_LEN_MAX);
                  STB_AppFreeMemory(crid);
               }
            }
            else
            {
               timer_info->u.record.event_triggered = FALSE;

               /* Set record time to now and default duration of 2 hours */
               timer_info->start_time = STB_GCNowDHMSGmt();
               timer_info->u.record.duration = DHMS_CREATE(0, 2, 0, 0);
            }
            break;
         }

         case TIMER_TYPE_PRIVATE:
         {
            break;
         }

         default:
         {
            retval = FALSE;
            break;
         }
      }
   }

   FUNCTION_FINISH(ATMR_InitialiseTimer);

   return(retval);
}

/**
 * @brief   Copies timer data for the given timer info the info structure provided
 * @param   handle timer handle the data is to be copied from
 * @param   timer_info will be filled with info from the timer
 * @return  TRUE if the timer is valid and data is returned, FALSE otherwise
 */
BOOLEAN ATMR_GetTimerInfo(U32BIT handle, S_TIMER_INFO *timer_info)
{
   BOOLEAN retval;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_GetTimerInfo);

   retval = FALSE;

   DBDEF_RequestAccess();

   if ((timer_info != NULL) && ((timer = DBDEF_FindTimerRec(handle)) != NULL))
   {
      memset(timer_info, 0, sizeof(S_TIMER_INFO));

      if (timer->dba_rec == NULL)
      {
         timer_info->ram_only = TRUE;
      }
      else
      {
         timer_info->ram_only = FALSE;
      }

      timer_info->type = timer->type;
      timer_info->frequency = timer->frequency;
      timer_info->start_time = timer->start_time;
      memcpy(&timer_info->name[0], &timer->name[0], TMR_MAX_NAME_LENGTH);

      switch (timer->type)
      {
         case TIMER_TYPE_SLEEP:
            /* No more info to be copied */
            break;

         case TIMER_TYPE_ALARM:
         {
            timer_info->u.alarm.change_service = timer->u.alarm.change_service;
            timer_info->u.alarm.orig_net_id = timer->u.alarm.orig_net_id;
            timer_info->u.alarm.transport_id = timer->u.alarm.transport_id;
            timer_info->u.alarm.service_id = timer->u.alarm.service_id;
            timer_info->u.alarm.ramp_volume = timer->u.alarm.ramp_volume;
            break;
         }

         case TIMER_TYPE_PVR_RECORD:
         {
            timer_info->u.record.duration = timer->u.record.duration;
            timer_info->u.record.event_triggered = timer->u.record.event_triggered;
            timer_info->u.record.event_id = timer->u.record.event_id;
            timer_info->u.record.orig_net_id = timer->u.record.orig_net_id;
            timer_info->u.record.transport_id = timer->u.record.transport_id;
            timer_info->u.record.service_id = timer->u.record.service_id;
            timer_info->u.record.disk_id = timer->u.record.disk_id;
            timer_info->u.record.recommendation = timer->u.record.recommendation;
            timer_info->u.record.start_padding = GetStartPadding(timer);
            timer_info->u.record.end_padding = GetEndPadding(timer);
            timer_info->u.record.notify_time = timer->u.record.notify_time;
            timer_info->u.record.do_not_delete = timer->u.record.do_not_delete;

            memcpy(&timer_info->u.record.prog_crid[0], &timer->u.record.prog_crid[0],
               TMR_PVR_CRID_LEN_MAX);
            memcpy(&timer_info->u.record.other_crid[0], &timer->u.record.other_crid[0],
               TMR_PVR_CRID_LEN_MAX);
            break;
         }

         default:
            /* Unknown or private timer */
            break;
      }

      retval = TRUE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetTimerInfo);

   return(retval);
}

/**
 * @brief   Returns the service for the recording timer on the given path
 * @param   path recording path
 * @return  service pointer, or NULL if timer not found or service referenced by timer not found
 */
void* ATMR_GetRecordService(U8BIT path)
{
   ADB_TIMER_REC *timer;
   void *s_ptr;

   FUNCTION_START(ATMR_GetRecordService);

   TMR_PVR_DBG(("ATMR_GetRecordService(%u)", path));

   DBDEF_RequestAccess();

   s_ptr = NULL;

   /* Find the recording timer on the given path */
   for (timer = DBDEF_GetNextTimerRec(NULL); (s_ptr == NULL) && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if ((timer->type == TIMER_TYPE_PVR_RECORD) && (timer->u.record.path == path))
      {
         /* Found the timer, get the service */
         s_ptr = ADB_FindServiceByIds(timer->u.record.orig_net_id, timer->u.record.transport_id,
               timer->u.record.service_id);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetRecordService);

   return(s_ptr);
}

/**
 * @brief   Get the name of the timer with the given handle
 * @param   handle timer handle
 * @return  pointer to timer name, this value shouldn't be freed
 */
U8BIT* ATMR_GetName(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U8BIT *retval;

   FUNCTION_START(ATMR_GetName);

   DBDEF_RequestAccess();

   retval = NULL;

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = timer->name;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetName);

   return(retval);
}

/**
 * @brief   Sets the name of the timer with the given handle
 * @param   handle timer handle
 */
void ATMR_SetName(U32BIT handle, U8BIT *name)
{
   ADB_TIMER_REC *timer;
   U32BIT size;

   FUNCTION_START(ATMR_SetName);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      size = strlen((char *)name) + 1;
      if (size >= TMR_MAX_NAME_LENGTH)
      {
         size = TMR_MAX_NAME_LENGTH - 1;
      }
      memcpy(timer->name, name, size);
      timer->name[size] = 0;

      if (timer->dba_rec != NULL)
      {
         DBA_SetFieldString(timer->dba_rec, DBA_FIELD_REC_NAME, timer->name, size);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetName);
}

/**
 * @brief   Get the start date & time of the timer with the given handle.
 *          The date/time returned will be in UTC
 * @param   handle timer handle
 * @return  UTC start date/time, or 0 if the timer handle isn't valid
 */
U32DHMS ATMR_GetStartDateTime(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U32DHMS retval;

   FUNCTION_START(ATMR_GetStartDateTime);

   DBDEF_RequestAccess();

   retval = 0;

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = timer->start_time;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetStartDateTime);

   return(retval);
}

/**
 * @brief   Returns the duration of the PVR record timer with the given handle
 * @param   handle timer handle
 * @return  timer duration, or 0 if the timer handle isn't valid
 */
U32DHMS ATMR_GetDuration(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U32DHMS retval;

   FUNCTION_START(ATMR_GetDuration);

   DBDEF_RequestAccess();

   retval = 0;

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      retval = timer->u.record.duration;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetDuration);

   return(retval);
}

/**
 * @brief   Get the end date & time of the timer with the given handle.
 *          The date/time returned will be in UTC
 * @param   handle timer handle
 * @return  UTC end date/time, or 0 if the timer handle isn't valid
 */
U32DHMS ATMR_GetEndDateTime(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U32DHMS retval;

   FUNCTION_START(ATMR_GetEndDateTime);

   DBDEF_RequestAccess();

   retval = 0;

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         retval = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);
      }
      else
      {
         retval = timer->start_time;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetEndDateTime);

   return(retval);
}

/**
 * @brief   Returns the type of the given timer
 * @param   handle timer handle
 * @return  type of the timer, or TIMER_TYPE_NONE if handle isn't valid
 */
E_TIMER_TYPE ATMR_GetType(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   E_TIMER_TYPE timer_type;

   FUNCTION_START(ATMR_GetType);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      timer_type = timer->type;
   }
   else
   {
      timer_type = TIMER_TYPE_NONE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetType);

   return(timer_type);
}

/**
 * @brief   Returns the frequency setting for the given timer
 * @param   handle timer handle
 * @return  frequency setting of the timer, or TIMER_FREQ_ONCE if handle isn't valid
 */
E_TIMER_FREQ ATMR_GetFrequency(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   E_TIMER_FREQ timer_freq;

   FUNCTION_START(ATMR_GetFrequency);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      timer_freq = timer->frequency;
   }
   else
   {
      timer_freq = TIMER_FREQ_ONCE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetFrequency);

   return(timer_freq);
}

/**
 * @brief   Returns the change service setting for an alarm timer
 * @param   handle timer handle
 * @return  value of change service setting, or FALSE if timer is invalid or isn't an alarm timer
 */
BOOLEAN ATMR_GetChangeService(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN change_service;

   FUNCTION_START(ATMR_GetChangeService);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_ALARM))
   {
      change_service = timer->u.alarm.change_service;
   }
   else
   {
      change_service = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetChangeService);

   return(change_service);
}

/**
 * @brief   Returns the ramp volume setting for an alarm timer
 * @param   handle timer handle
 * @return  value of ramp volume setting, or FALSE if timer is invalid or isn't an alarm timer
 */
BOOLEAN ATMR_GetRampVolume(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN ramp_volume;

   FUNCTION_START(ATMR_GetRampVolume);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_ALARM))
   {
      ramp_volume = timer->u.alarm.ramp_volume;
   }
   else
   {
      ramp_volume = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetRampVolume);

   return(ramp_volume);
}

/**
 * @brief   Returns the event ID for a PVR recording timer
 * @param   handle timer handle
 * @return  event ID, or 0 if the timer isn't valid or isn't a recording timer
 */
U16BIT ATMR_GetEventId(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U16BIT event_id;

   FUNCTION_START(ATMR_GetEventId);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      event_id = timer->u.record.event_id;
   }
   else
   {
      event_id = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetEventId);

   return(event_id);
}

/**
 * @brief   Returns the service ID for an alarm or PVR recording timer
 * @param   handle timer handle
 * @return  service ID, or ADB_INVALID_DVB_ID if the timer isn't valid or isn't
 *          an alarm or recording timer
 */
U16BIT ATMR_GetServiceId(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U16BIT service_id;

   FUNCTION_START(ATMR_GetServiceId);

   service_id = ADB_INVALID_DVB_ID;

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      if (timer->type == TIMER_TYPE_ALARM)
      {
         service_id = timer->u.alarm.service_id;
      }
      else if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         service_id = timer->u.record.service_id;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetServiceId);

   return(service_id);
}

/**
 * @brief   Returns the transport ID for an alarm or PVR recording timer
 * @param   handle timer handle
 * @return  transport ID, or ADB_INVALID_DVB_ID if the timer isn't valid or isn't
 *          an alarm or recording timer
 */
U16BIT ATMR_GetTransportId(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U16BIT trans_id;

   FUNCTION_START(ATMR_GetTransportId);

   trans_id = ADB_INVALID_DVB_ID;

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      if (timer->type == TIMER_TYPE_ALARM)
      {
         trans_id = timer->u.alarm.transport_id;
      }
      else if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         trans_id = timer->u.record.transport_id;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetTransportId);

   return(trans_id);
}

/**
 * @brief   Returns the original network ID for an alarm or PVR recording timer
 * @param   handle timer handle
 * @return  original network ID, or ADB_INVALID_DVB_ID if the timer isn't valid or isn't
 *          an alarm or recording timer
 */
U16BIT ATMR_GetOriginalNetworkId(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U16BIT onet_id;

   FUNCTION_START(ATMR_GetOriginalNetworkId);

   onet_id = ADB_INVALID_DVB_ID;

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      if (timer->type == TIMER_TYPE_ALARM)
      {
         onet_id = timer->u.alarm.orig_net_id;
      }
      else if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         onet_id = timer->u.record.orig_net_id;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetOriginalNetworkId);

   return(onet_id);
}

/**
 * @brief   Gets the timer's missed flag
 * @param   handle timer handle
 * @return  TRUE if the timer is marked as missed, FALSE otherwise
 */
BOOLEAN ATMR_GetMissed(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN missed;

   FUNCTION_START(ATMR_GetMissed);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      missed = timer->missed;
   }
   else
   {
      missed = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetMissed);

   return(missed);
}

/**
 * @brief   Returns a pointer to the programme CRID string from a recording timer.
 *          The returned value shouldn't be changed or freed.
 * @param   handle timer handle
 * @return  CRID string, NULL if there isn't a CRID or the timer isn't a recording timer
 */
U8BIT* ATMR_GetProgrammeCrid(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U8BIT *prog_crid;

   FUNCTION_START(ATMR_GetProgrammeCrid);

   prog_crid = NULL;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (strlen((char *)timer->u.record.prog_crid) > 0)
      {
         prog_crid = timer->u.record.prog_crid;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetProgrammeCrid);

   return(prog_crid);
}

/**
 * @brief   Does the timer have a series crid?
 * @param   handle timer handle
 * @return  TRUE if the timer is valid and is a recording timer and has a series CRID,
 *          FALSE otherwise
 */
BOOLEAN ATMR_HasSeriesCrid(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN retval;

   FUNCTION_START(ATMR_HasSeriesCrid);

   retval = FALSE;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (!timer->u.record.recommendation &&
          (strlen((char *)timer->u.record.other_crid) > 0))
      {
         retval = TRUE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_HasSeriesCrid);

   return(retval);
}

/**
 * @brief   Does the timer have a recommendation crid?
 * @param   handle timer handle
 * @return  TRUE if the timer is valid and is a recording timer and has a recommendation CRID,
 *          FALSE otherwise
 */
BOOLEAN ATMR_HasRecommendationCrid(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN retval;

   FUNCTION_START(ATMR_HasRecommendationCrid);

   DBDEF_RequestAccess();

   retval = FALSE;

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (timer->u.record.recommendation &&
          (strlen((char *)timer->u.record.other_crid) > 0))
      {
         retval = TRUE;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_HasRecommendationCrid);

   return(retval);
}

/**
 * @brief   Returns a pointer to the other CRID string from a recording timer.
 *          This will either be a series or recommendation CRID depending on the
 *          recommendation flag setting.
 *          The returned value shouldn't be changed or freed.
 * @param   handle timer handle
 * @return  CRID string, NULL if there isn't a CRID or the timer isn't a recording timer
 */
U8BIT* ATMR_GetOtherCrid(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U8BIT *crid;

   FUNCTION_START(ATMR_GetOtherCrid);

   crid = NULL;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (strlen((char *)timer->u.record.other_crid) > 0)
      {
         crid = timer->u.record.other_crid;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetOtherCrid);

   return(crid);
}

/**
 * @brief   Returns the disk id for the given timer if the timer is a PVR recording timer
 * @param   handle timer handle
 * @return  disk id if timer is valid and is a recording timer, INVALID_DISK_ID otherwise
 */
U16BIT ATMR_GetDiskId(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U16BIT disk_id;

   FUNCTION_START(ATMR_GetDiskId);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      disk_id = timer->u.record.disk_id;
   }
   else
   {
      disk_id = INVALID_DISK_ID;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetDiskId);

   return(disk_id);
}

/**
 * @brief   Set the disk for the given timer if the timer is a recording timer
 * @param   handle timer handle
 * @param   disk_id disk id
 */
void ATMR_SetDiskId(U32BIT handle, U16BIT disk_id)
{
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_SetDiskId);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      timer->u.record.disk_id = disk_id;

      if (timer->dba_rec != NULL)
      {
         DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DISKID, (U32BIT)disk_id);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetDiskId);
}

/**
 * @brief   Returns the recording handle associated with a PVR recording timer
 * @param   handle timer handle
 * @return  recording handle, or 0 if the timer isn't a PVR recording timer or no
 *          recording has been started
 */
U32BIT ATMR_GetRecordingHandle(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   U32BIT recording_handle;

   FUNCTION_START(ATMR_GetRecordingHandle);

   recording_handle = STB_PVR_INVALID_HANDLE;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      recording_handle = timer->u.record.recording_handle;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetRecordingHandle);

   return(recording_handle);
}

/**
 * @brief   Used by the DVB stack to handle an event for the given timer.
 *          If the timer requires the app to deal with it, such as for sleep or alarm
 *          timers, then another event will be sent to the app with an S_TIMER_INFO
 *          structure as the event data containing the timer's details to allow the
 *          app deal with it. The timer itself will have been updated or deleted.
 * @param   handle timer handle
 */
void ATMR_HandleTimerEvent(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   S_TIMER_INFO info;
   BOOLEAN delete_timer;
   BOOLEAN timers_updated;
   BOOLEAN stop_recording;
   U32BIT overrun_duration;
   U32DHMS start_date_time, end_date_time;
   U32BIT recording_handle;

   FUNCTION_START(ATMR_HandleTimerEvent);

   /* Action expired timers first so that back to back recordings will work
    * with any current recording being stopped before a new one is started.
    * Any expired timers that aren't to be rescheduled are marked for deletion */
   DBDEF_RequestAccess();

   delete_timer = FALSE;
   timers_updated = FALSE;

   timer = DBDEF_FindTimerRec(handle);
   if (timer != NULL)
   {
      /* Check whether it's expired */
      if (timer->expired)
      {
         TMR_DBG(("ATMR_HandleTimerEvent: Timer 0x%08lx has expired", timer->handle));

         memset(&info, 0, sizeof(info));

         switch (timer->type)
         {
            case TIMER_TYPE_SLEEP:
            {
               /* Clear the flag that indicates the timer is to be started */
               timer->starting = FALSE;

               if (timer->frequency == TIMER_FREQ_ONCE)
               {
                  /* Timer is no longer required so destroy it */
                  delete_timer = TRUE;
               }

               /* Setup details of the timer and send an event to the app */
               info.type = TIMER_TYPE_SLEEP;
               info.start_time = timer->start_time;
               info.frequency = timer->frequency;

               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_TIMER_TRIGGERED,
                  (void *)&info, sizeof(info));
               break;
            }

            case TIMER_TYPE_ALARM:
            {
               /* Clear the flag that indicates the timer is to be started */
               timer->starting = FALSE;

               if (timer->frequency == TIMER_FREQ_ONCE)
               {
                  /* Timer is no longer required so destroy it */
                  delete_timer = TRUE;
               }

               /* Setup details of the timer and send an event to the app */
               info.type = TIMER_TYPE_ALARM;
               info.start_time = timer->start_time;
               info.frequency = timer->frequency;
               memcpy(&info.name[0], &timer->name[0], sizeof(timer->name));
               info.u.alarm.change_service = timer->u.alarm.change_service;
               info.u.alarm.orig_net_id = timer->u.alarm.orig_net_id;
               info.u.alarm.transport_id = timer->u.alarm.transport_id;
               info.u.alarm.service_id = timer->u.alarm.service_id;
               info.u.alarm.ramp_volume = timer->u.alarm.ramp_volume;

               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_TIMER_TRIGGERED,
                  (void *)&info, sizeof(info));
               break;
            }

            case TIMER_TYPE_PVR_RECORD:
            {
               if (timer->u.record.event_triggered)
               {
                  stop_recording = FALSE;

                  if (timer->u.record.programme_finished)
                  {
                     /* The programme being recorded is no longer the now event and the end
                      * of the timer has triggered, so the recording should be stopped */
                     stop_recording = TRUE;
                     TMR_PVR_DBG(("Programme has finished and timer 0x%x has expired, stopping recording",
                        timer->handle));
                  }
                  else
                  {
                     /* The timer has triggered, which will include any padding, but the end of
                      * the programme hasn't been seen yet, so check that the recording doesn't
                      * overrun by more than's allowed */
                     if (ACFG_IsNordigCountry())
                     {
                        overrun_duration = NORDIG_EVENT_DURATION_OVERRUN;
                     }
                     else if (ACFG_GetCountry() == COUNTRY_CODE_UK)
                     {
                        overrun_duration = EVENT_DURATION_OVERRUN;
                     }
                     else
                     {
                        overrun_duration = 0;
                     }

                     GetActualStartEndTime(timer->start_time, timer->u.record.duration, 0,
                        overrun_duration, &start_date_time, &end_date_time);

                     if (!STB_GCIsFutureDateTime(DHMS_DATE(end_date_time), DHMS_HOUR(end_date_time),
                        DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
                     {
                        /* Programme has overrun and now needs to be stopped */
                        stop_recording = TRUE;
                        TMR_PVR_DBG(("Timer 0x%x has overrun, stopping recording", timer->handle));
                     }
                  }
               }
               else
               {
                  /* End of the timer triggers the end of recording */
                  stop_recording = TRUE;
                  TMR_PVR_DBG(("Timer 0x%x has expired, stopping recording", timer->handle));
               }

               if (stop_recording)
               {
                  if (timer->u.record.recording_handle != STB_PVR_INVALID_HANDLE)
                  {
#ifdef INTEGRATE_HBBTV
                     HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_COMPLETED);
#endif
                     /* Stop recording */
                     recording_handle = timer->u.record.recording_handle;
                     DBDEF_ReleaseAccess();
                     stop_recording = APVR_StopRecording(recording_handle);
                     DBDEF_RequestAccess();

                     if (stop_recording)
                     {
                        /* Stopping the recording may have deleted the timer,
                         * so the timer shouldn't be referenced anymore and the timers should be saved */
                        timers_updated = TRUE;
                        timer = NULL;
                     }
                     else
                     {
                        if (!STB_DPIsLivePath(timer->u.record.path))
                        {
                           /* Reset the tuned service on the path even though the recording was not
                            * stopped successfully for some reason. The tune state will be set to
                            * off and guarantee a tuning will be performed for the next recording
                            * which uses the same path */
                           ACTL_TuneToService(timer->u.record.path, NULL, NULL, FALSE, TRUE);
                        }

                        /* Release the path as well */
                        STB_DPReleasePath(timer->u.record.path, RES_OWNER_DVB);
                        timer->u.record.path = INVALID_RES_ID;
                     }
                  }
                  else
                  {
                     if (timer->u.record.path != INVALID_RES_ID)
                     {
                        /* Recording didn't start but resources need to be released */
                        if (!STB_DPIsLivePath(timer->u.record.path))
                        {
                           /* Reset the tuned service on the path even the recording was not started successfully
                            * for some reason. the tune state will be set to off and guarantee a tuning will be performed
                            * for the next recording which use the same path */
                           ACTL_TuneToService(timer->u.record.path, NULL, NULL, FALSE, TRUE);
                        }

                        STB_DPReleasePath(timer->u.record.path, RES_OWNER_DVB);
                        timer->u.record.path = INVALID_RES_ID;
                     }
                     else if (timer->frequency == TIMER_FREQ_ONCE)
                     {
                        /* Timer is no longer required so destroy it */
                        delete_timer = TRUE;
                     }
                  }
               }

               /* No event needs to be sent to the app for this timer because starting/stopping
                * of recording will cause events to be sent for the app to handle */
               break;
            }

            case TIMER_TYPE_PRIVATE:
               /* Private timers are handled elsewhere */
               break;

            default:
               TMR_DBG(("ATMR_HandleTimerEvent: Unrecognised expired timer 0x%08lx, type=0x%02x",
                        timer->handle, timer->type));
               break;
         }
      }

      if ((timer != NULL) && delete_timer)
      {
         TMR_DBG(("ATMR_HandleTimerEvent: deleting expired timer 0x%08lx", timer->handle));
         DeleteTimer(timer->handle);
         timer = NULL;

         timers_updated = TRUE;
      }

      if (timer != NULL)
      {
         if (timer->starting)
         {
            switch (timer->type)
            {
               case TIMER_TYPE_PVR_RECORD:
                  /* The recording should be started */
                  TMR_PVR_DBG(("ATMR_HandleTimerEvent: 0x%x, start recording", timer->handle));
                  if (ActionTimerRecStart(timer))
                  {
                     /* Update the state machine */
                     ASTE_StartRecording();
                  }

                  /* Timers may be updated whether the recording is started or not,
                   * so they need to be saved */
                  timers_updated = TRUE;
                  break;

               default:
                  /* Alarm and sleep timers shouldn't get be in this situation because they expire when started */
                  TMR_DBG(("ATMR_HandleTimerEvent: Timer type %d (0x%08lx) is set to start but hasn't been handled",
                           timer->type, timer->handle));
                  break;
            }
         }
      }
   }

   DBDEF_ReleaseAccess();

   if (timers_updated)
   {
      ADB_SaveDatabase();
   }

   FUNCTION_FINISH(ATMR_HandleTimerEvent);
}

/**
 * @brief   Checks all timers to see whether any recordings should be started or stopped
 *          as a result of the now event being changed.
 * @param   recordings_can_start TRUE if the event should be checked to start a recording
 * @param   service check recording status on this service only.
 *                  If NULL then all services are checked
 * @return  TRUE if a record can be/has been started
 */
BOOLEAN ATMR_CheckRecordStatus(BOOLEAN recordings_can_start, void *service)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;
   void *s_ptr;
   void *now_event;
   S32BIT event_id;
   BOOLEAN timers_updated;
   BOOLEAN record_started;

   FUNCTION_START(ATMR_CheckRecordStatus);

   timers_updated = FALSE;
   record_started = FALSE;

   DBDEF_RequestAccess();

   /* First check all the timers to see if any should be stopped.
    * This will be done when the initial now/next events are received on startup,
    * which will result in any timers that are no longer valid being deleted.
    */
   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);

      /* Make sure the timer isn't selected - user may have selected it in the timer screen */
      timer->selected = FALSE;

      /* If the timer has been started and has a valid service ID then check it */
      if (timer->started && (timer->type == TIMER_TYPE_PVR_RECORD) &&
         timer->u.record.event_triggered && (timer->u.record.event_id != 0))
      {
         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, timer->u.record.orig_net_id,
            timer->u.record.transport_id, timer->u.record.service_id);
         if (((service == NULL) && (s_ptr != NULL)) || (s_ptr == service))
         {
            /* Check the service's now event to see whether it's the same as the timer's event */
            ADB_GetNowNextEvents(s_ptr, &now_event, NULL);

            if (now_event != NULL)
            {
               if ((timer->u.record.event_id != ADB_GetEventId(now_event)) &&
                  timer->u.record.programme_started)
               {
                  if (!timer->u.record.programme_finished)
                  {
                     TMR_PVR_DBG(("%s: Programme for timer 0x%x has finished", __FUNCTION__, timer->handle));
                     timer->u.record.programme_finished = TRUE;
                  }

                  timers_updated |= HasTimerExpired(timer->handle);
               }

               ADB_ReleaseEventData(now_event);
            }
         }
      }
   }

   /* Go through the timers and delete any that are selected (marked for deletion) */
   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);

      if (timer->selected)
      {
         /* Delete marked timer */
         TMR_PVR_DBG(("%s: deleting expired timer 0x%08x", __FUNCTION__, timer->handle));
         DeleteTimer(timer->handle);
         timers_updated = TRUE;
      }
   }

   /* Check whether any now events should start a recording */
   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = DBDEF_GetNextTimerRec(timer))
   {
      /* Check whether the timer is associated with an event and whether that event has now started */
      if ((timer->type == TIMER_TYPE_PVR_RECORD) && !timer->missed && !timer->expired &&
         timer->u.record.event_triggered && !timer->u.record.programme_started)
      {
         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, timer->u.record.orig_net_id,
            timer->u.record.transport_id, timer->u.record.service_id);
         if (s_ptr != NULL)
         {
            ADB_GetNowNextEvents(s_ptr, &now_event, NULL);

            if (now_event != NULL)
            {
               event_id = ADB_GetEventId(now_event);
               if (timer->u.record.event_id == event_id)
               {
                  TMR_PVR_DBG(("%s: Programme for timer 0x%x has started", __FUNCTION__, timer->handle));
                  timer->u.record.programme_started = TRUE;
               }

               ADB_ReleaseEventData(now_event);
            }
         }
      }

      /* If the timer hasn't already been started then check whether it now needs to */
      if ((timer->type == TIMER_TYPE_PVR_RECORD) && !timer->started && !timer->missed &&
         (!timer->starting || (timer->u.record.path == INVALID_RES_ID)))
      {
         s_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID, timer->u.record.orig_net_id,
            timer->u.record.transport_id, timer->u.record.service_id);
         if (s_ptr != NULL)
         {
            if (!timer->u.record.event_triggered)
            {
               timers_updated |= StartRecordTimer(timer, recordings_can_start, &record_started);
            }
            else
            {
               ADB_GetNowNextEvents(s_ptr, &now_event, NULL);

               if (now_event != NULL)
               {
                  /* The recording can be started if the 'now' event is the one to be recorded */
                  if (timer->u.record.event_id == ADB_GetEventId(now_event))
                  {
                     if (recordings_can_start)
                     {
                        TMR_PVR_DBG(("%s: Timer 0x%x, start recording for event %lu @ %02u:%02u:%02u",
                           __FUNCTION__, timer->handle, timer->u.record.event_id,
                           DHMS_HOUR(timer->start_time), DHMS_MINS(timer->start_time),
                           DHMS_SECS(timer->start_time)));
                     }

                     timers_updated |= StartRecordTimer(timer, recordings_can_start, &record_started);
                  }

                  ADB_ReleaseEventData(now_event);
               }
            }
         }
#ifdef TMR_DEBUG
         else
         {
            TMR_PVR_DBG(("ATMR_CheckRecordStatus: Can't find service 0x%04x/0x%04x/0x%04x",
                     timer->u.record.orig_net_id, timer->u.record.transport_id, timer->u.record.service_id));
         }
#endif
      }
   }

   if (timers_updated)
   {
      ADB_SaveDatabase();
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_CheckRecordStatus);

   return(record_started);
}

/**
 * @brief   Checks each recording timer that's linked to an event to see whether
 *          the event is still in the schedule and whether the time has changed,
 *          and if it has changed then the timer is updated.
 */
void ATMR_EitUpdated(void)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;
   void *serv_ptr;
   void *event_ptr;
   U32DHMS event_start;
   U32DHMS event_duration;
   U32DHMS end_time;
   U32DHMS rec_end_time;
   BOOLEAN updated;
   BOOLEAN event_missed;
   S_ALT_EVENT_DATA alt_event_data;

   FUNCTION_START(ATMR_EitUpdated);

   DBDEF_RequestAccess();

   updated = FALSE;

   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);

      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         /* Only check the timer if it hasn't already expired */
         if (!timer->expired)
         {
            if ((timer->u.record.service_id != 0) && (timer->u.record.event_id != 0))
            {
               serv_ptr = DBDEF_FindServiceRecByIds(NULL, ADB_INVALID_DVB_ID,
                  timer->u.record.orig_net_id, timer->u.record.transport_id,
                  timer->u.record.service_id);
               if (serv_ptr != NULL)
               {
                  /* Check if the timer is already marked as missed */
                  if (timer->missed)
                  {
                     event_missed = TRUE;
                  }
                  else
                  {
                     event_missed = FALSE;

                     /* Find the event from the service's EIT list */
                     event_ptr = ADB_GetEvent(serv_ptr, timer->u.record.event_id);
                     if (event_ptr == NULL)
                     {
                        /* Event not found in the schedule */
                        TMR_PVR_DBG(("ATMR_EitUpdated: Event %lu on %lu @ %02lu:%02lu not found in schedule",
                                 timer->u.record.event_id, DHMS_DATE(timer->start_time),
                                 DHMS_HOUR(timer->start_time), DHMS_MINS(timer->start_time)));

                        event_missed = TRUE;
                     }
                     else
                     {
                        /* The event may still be in the schedule if it's already past,
                         * so check that the event isn't already over */
                        event_start = ADB_GetEventStartDateTime(event_ptr);
                        event_duration = ADB_GetEventDuration(event_ptr);
                        end_time = STB_GCCalculateDHMS(event_start, event_duration, CALC_ADD);

                        if (!timer->started &&
                           !STB_GCIsFutureDateTime(DHMS_DATE(end_time), DHMS_HOUR(end_time),
                               DHMS_MINS(end_time), DHMS_SECS(end_time)))
                        {
                           /* Event is in the past */
                           event_missed = TRUE;
                           TMR_PVR_DBG(("ATMR_EitUpdated: Event %d ending on %d @ %02d:%02d has been missed",
                                    timer->u.record.event_id, DHMS_DATE(end_time), DHMS_HOUR(end_time),
                                    DHMS_MINS(end_time)));
                        }
                        else
                        {
                           /* Check that the start time or duration hasn't changed.
                            * The actual start time may be different from the scheduled time if the
                            * recording is event triggered and has started early, so only stop the
                            * recording if this isn't the case */
                           if ((timer->start_time != event_start) &&
                              (timer->u.record.event_triggered && !timer->u.record.programme_started))
                           {
                              if ((timer->u.record.recording_handle != STB_PVR_INVALID_HANDLE) &&
                                 (event_start > timer->start_time))
                              {
                                 /* The event has been rescheduled to a later time, but the
                                  * recording has already started, which will either be due to
                                  * start padding, or the recording is time trigegred and the
                                  * start time for the event has passed. In either case, the
                                  * recording should be stopped and deleted and the timer reset so
                                  * it triggers again at the new time to restart the recording */
                                 if (!STB_DPIsLivePath(timer->u.record.path))
                                 {
                                    DBDEF_ReleaseAccess();
                                    STB_DPStopSI(timer->u.record.path);
                                    DBDEF_RequestAccess();
                                 }

                                 STB_DPStopRecording(timer->u.record.path);
                                 STB_DPReleasePath(timer->u.record.path, RES_OWNER_DVB);
                                 timer->u.record.path = INVALID_RES_ID;

                                 STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION,
                                    EV_PVR_RECORDING_STOPPED, &timer->u.record.recording_handle,
                                    sizeof(timer->u.record.recording_handle));

                                 APVR_DeleteRecording(timer->u.record.recording_handle);
                                 timer->u.record.recording_handle = STB_PVR_INVALID_HANDLE;

                                 timer->starting = FALSE;
                                 timer->started = FALSE;
                                 timer->u.record.notified = FALSE;
                              }

                              timer->start_time = event_start;
                              if (timer->dba_rec != NULL)
                              {
                                 DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_STARTTIME, event_start);
                              }
                              updated = TRUE;
                           }

                           if (timer->u.record.duration != event_duration)
                           {
                              /* The duration will be different if the event wasn't recorded
                               * from the beginning, so check the end time to decide whether
                               * the duration has actually been changed */
                              rec_end_time = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);

                              if (end_time != rec_end_time)
                              {
                                 timer->u.record.duration = event_duration;
                                 if (timer->dba_rec != NULL)
                                 {
                                    DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DURATION, event_duration);
                                 }
                                 updated = TRUE;
                              }
                           }

#ifdef INTEGRATE_HBBTV
                           if (updated)
                           {
                              HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_UPDATED);
                           }
#endif
                        }

                        ADB_ReleaseEventData(event_ptr);
                     }
                  }

                  if (event_missed)
                  {
                     /* Has the event been completely missed? */
                     end_time = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);

                     if (!STB_GCIsFutureDateTime(DHMS_DATE(end_time), DHMS_HOUR(end_time),
                            DHMS_MINS(end_time), 0))
                     {
                        /* Recording has been completely missed, check whether there's an
                         * alternate instance that can be recorded instead */
                        if (STB_GetNumBytesInString(timer->u.record.prog_crid) > 0)
                        {
                           /* Search for an alternate instance of the event.
                            * Searching the EIT data for alternative events can take a long time,
                            * so if the event isn't already marked as missed, mark it as missed,
                            * inform the user via a message and pass the info required to look
                            * for an alternative event to another background task */
                           if (!timer->missed)
                           {
                              TMR_PVR_DBG(("Missed event on %u @ %02u:%02u and ends on %d @ %02d:%02d (today=%d)\n",
                                           DHMS_DATE(timer->start_time), DHMS_HOUR(timer->start_time),
                                           DHMS_MINS(timer->start_time), DHMS_DATE(end_time), DHMS_HOUR(end_time),
                                           DHMS_MINS(end_time), STB_GCGetGMTDate()));

                              /* Mark the recording as missed */
                              timer->missed = TRUE;
                              if (timer->dba_rec != NULL)
                              {
                                 DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_MISSED, timer->missed);
                              }

                              updated = TRUE;
#ifdef INTEGRATE_HBBTV
                              HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_UPDATED);
#endif
                           }

                           /* Setup the data to passed to the background task */
                           alt_event_data.timer_handle = timer->handle;
                           alt_event_data.is_recommendation = timer->u.record.recommendation;
                           alt_event_data.do_not_delete = timer->u.record.do_not_delete;
                           strncpy((char *)alt_event_data.prog_crid, (char *)timer->u.record.prog_crid, TMR_PVR_CRID_LEN_MAX);
                           strncpy((char *)alt_event_data.other_crid, (char *)timer->u.record.other_crid, TMR_PVR_CRID_LEN_MAX);

                           STB_OSWriteQueue(alt_event_queue, (void *)&alt_event_data,
                              sizeof(alt_event_data), TIMEOUT_NOW);
                        }
                        else
                        {
                           /* Mark the recording as missed */
                           if (!timer->missed)
                           {
                              TMR_PVR_DBG(("Missed event on %u @ %02u:%02u and ends on %d @ %02d:%02d (today=%d)\n",
                                           DHMS_DATE(timer->start_time), DHMS_HOUR(timer->start_time),
                                           DHMS_MINS(timer->start_time), DHMS_DATE(end_time), DHMS_HOUR(end_time),
                                           DHMS_MINS(end_time), STB_GCGetGMTDate()));

                              /* As it isn't possible to find an alternative for the event, the timer
                               * should be deleted now a message has been added */
                              DeleteTimer(timer->handle);
                           }
                        }
                     }
                  }
               }
               else
               {
                  /* Service not found, delete the timer */
                  TMR_PVR_DBG(("ATMR_EitUpdated: Service not found, timer deleted!!"));
                  DeleteTimer(timer->handle);
               }
            }
         }
      }
   }

   DBDEF_ReleaseAccess();

   if (updated)
   {
      ADB_SaveDatabase();
   }

   FUNCTION_FINISH(ATMR_EitUpdated);
}

/**
 * @brief   Finds the timer for the given recording handle
 * @param   recording_handle recording handle associated with a timer
 * @return  timer handle, or INVALID_TIMER_HANDLE if the timer can't be found
 */
U32BIT ATMR_FindRecordingTimer(U32BIT recording_handle)
{
   U32BIT handle;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_FindRecordingTimer);

   handle = INVALID_TIMER_HANDLE;

   DBDEF_RequestAccess();

   /* Find the timer with the given recording handle */
   timer = DBDEF_GetNextTimerRec(NULL);
   while (timer != NULL)
   {
      if ((timer->type == TIMER_TYPE_PVR_RECORD) &&
         (timer->u.record.recording_handle == recording_handle))
      {
         handle = timer->handle;
         break;
      }

      timer = DBDEF_GetNextTimerRec(timer);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_FindRecordingTimer);

   return(handle);
}

/**
 * @brief   Delete the PVR record timer with the given recording handle
 * @param   recording_handle recording handle used by the timer to be deleted
 */
void ATMR_DeleteRecordingTimer(U32BIT recording_handle)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;

   FUNCTION_START(ATMR_DeleteRecordingTimer);

   DBDEF_RequestAccess();

   /* Find the timer with the given recording handle */
   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);

      if (timer->u.record.recording_handle == recording_handle)
      {
         if (timer->frequency == TIMER_FREQ_ONCE)
         {
            /* Destroy timer */
            TMR_PVR_DBG(("ATMR_DeleteRecordingTimer: Destroy timer 0x%08lx", timer->handle));
            DeleteTimer(timer->handle);

            /* Ensure any timer changes are saved to NVM */
            ADB_SaveDatabase();
         }
         else
         {
            /* Clear recording handle for next recording */
            timer->u.record.recording_handle = STB_PVR_INVALID_HANDLE;
            TMR_PVR_DBG(("ATMR_DeleteRecordingTimer: timer periodic, do not destroy, clear recording handle"));
         }
         break;
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_DeleteRecordingTimer);
}

/**
 * @brief   Counts the number of simultaneous recordings (EXT and PVR) between
 *          the given start and end dates/times.
 * @param   max_recordings the max recordings that can take place at the same time
 * @param   start_date_time start date and time of the period to be checked
 * @param   end_date_time end date and time of the period to be checked
 * @param   conflicting_timers pointer to the list of conflicting timers, the
 *          list must to be freed by the caller. If NULL it's ignored.
 * @return  Number of simultaneous recordings during the given period
 */
U8BIT ATMR_GetNumSimultaneousRecordings(U8BIT max_recordings, U32DHMS start_date_time,
   U32DHMS end_date_time, U32BIT **conflicting_timers)
{
   U8BIT num_recordings;

   FUNCTION_START(ATMR_GetNumSimultaneousRecordings);

   DBDEF_RequestAccess();

   num_recordings = GetNumSimultaneousRecordings(INVALID_TIMER_HANDLE, ADB_INVALID_DVB_ID,
      ADB_INVALID_DVB_ID, ADB_INVALID_DVB_ID, max_recordings, start_date_time, end_date_time, FALSE, FALSE,
      conflicting_timers);

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetNumSimultaneousRecordings);

   return(num_recordings);
}

/**
 * @brief   Adds a timer to perform a recording based on the given event and service.
 *          Conflicts are checked and a search is made for an alternative event if necessary.
 * @param   serv_ptr pointer to service on which recording is to take place
 * @param   event_ptr event to be recorded
 * @param   prog_crid programme CRID or NULL
 * @param   other_crid series or recommendation CRID, or NULL
 * @param   is_recommendation TRUE if the above crid is a recommendation crid, FALSE for series
 * @param   check_alternatives TRUE if alternative events are to be searched for if the specified
 *                             event can't be recorded
 * @param   do_not_delete flag that defines whether the recording should be marked as locked
 * @return  pointer to the event actually recorded, which may be different from the one given,
 *          or NULL if the event can't be recorded
 */
void* ATMR_RecordEvent(void *serv_ptr, void *event_ptr, U8BIT *prog_crid, U8BIT *other_crid,
   BOOLEAN is_recommendation, BOOLEAN check_alternatives, BOOLEAN do_not_delete)
{
   S_TIMER_INFO timer_info;
   U32DHMS start_time;
   U32DHMS end_time;
   U8BIT num_recordings;
   void *result_event_ptr = event_ptr;
#ifdef TMR_PVR_DEBUG
   U8BIT *event_prog_crid;
#endif
   U32BIT *conflicts;

   FUNCTION_START(ATMR_RecordEvent);

   /* Check whether this recording is possible */
   start_time = ADB_GetEventStartDateTime(event_ptr);
   end_time = ADB_GetEventEndDateTime(event_ptr);

   num_recordings = ATMR_GetNumSimultaneousRecordings(STB_HWGetNumRecorders(), start_time,
         end_time, &conflicts);

   if (num_recordings == STB_HWGetNumRecorders())
   {
      /* Recording this event isn't possible */
      if ((prog_crid != NULL) && check_alternatives)
      {
         /* Check whether there are any alternatives */
         result_event_ptr = ADB_FindEventFromCrid(prog_crid, event_ptr, &serv_ptr);
      }
      else
      {
         /* The event can't be recorded and as it doesn't have a programme CRID
          * so no alternatives can be searched for */
         result_event_ptr = NULL;
      }
   }

   if (result_event_ptr == NULL)
   {
      U16BIT serv_id, trans_id, onet_id;
      U8BIT t, n;
      void *s_ptr, *event;
      U8BIT *p_crid, *o_crid;

      for (t = 0; t < num_recordings; t++)
      {
         serv_id = ATMR_GetServiceId(conflicts[t]);
         trans_id = ATMR_GetTransportId(conflicts[t]);
         onet_id = ATMR_GetOriginalNetworkId(conflicts[t]);

         p_crid = NULL;
         o_crid = NULL;

         s_ptr = ADB_FindServiceByIds(onet_id, trans_id, serv_id);
         if (s_ptr != NULL)
         {
            event = ADB_GetEvent(s_ptr, ATMR_GetEventId(conflicts[t]));
            if (event != NULL)
            {
               p_crid = ADB_GetEventProgrammeCrid(s_ptr, event);
               if ((p_crid != NULL) && check_alternatives)
               {
                  /* Check whether there are any alternatives */
                  event = ADB_FindEventFromCrid(p_crid, event, &s_ptr);
                  if (event != NULL)
                  {
                     start_time = ADB_GetEventStartDateTime(event);
                     end_time = ADB_GetEventEndDateTime(event);

                     n = ATMR_GetNumSimultaneousRecordings(STB_HWGetNumRecorders(), start_time,
                        end_time, NULL);
                     if (n < STB_HWGetNumRecorders())
                     {
                  #ifdef TMR_PVR_DEBUG
                        event_prog_crid = ADB_GetEventFullProgrammeCrid(s_ptr, event);

                        TMR_PVR_DBG(("   Moving programmed event to \"%s\", start: %d:%d.%d, len: %d.%d, event=%d",
                                     ADB_GetEventName(event), ADB_GetEventStartDate(event),
                                     ADB_GetEventStartHour(event), ADB_GetEventStartMin(event),
                                     ADB_GetEventDurationHour(event), ADB_GetEventDurationMin(event),
                                     ADB_GetEventId(event)));
                        if (event_prog_crid != NULL)
                        {
                           TMR_PVR_DBG(("      prog CRID=\"%s\"", event_prog_crid));
                           STB_AppFreeMemory(event_prog_crid);
                        }
                  #endif

                        ATMR_InitialiseTimer(&timer_info, TIMER_TYPE_PVR_RECORD, s_ptr, event);

                        o_crid = ADB_GetEventSeriesCrid(0, s_ptr, event);
                        if (o_crid != NULL)
                        {
                           strncpy((char *)timer_info.u.record.other_crid, (char *)o_crid, TMR_PVR_CRID_LEN_MAX);
                           STB_AppFreeMemory(o_crid);
                        }

                        timer_info.u.record.recommendation = ATMR_HasRecommendationCrid(conflicts[t]);
                        timer_info.u.record.disk_id = STB_PVRGetDefaultDisk();
                        timer_info.u.record.event_triggered = TRUE;
                        timer_info.u.record.do_not_delete = do_not_delete;
                        timer_info.u.record.start_padding = 0;
                        timer_info.u.record.end_padding = 0;

                        /* Add the timer to record the event */
                        ATMR_AddTimer(&timer_info);

                        /* Now remove the conflicting timer */
                        ATMR_DeleteTimer(conflicts[t]);

                        result_event_ptr = event_ptr;
                     }
                  }
               }
            }
         }

         if (p_crid != NULL)
         {
            STB_AppFreeMemory(p_crid);
         }

         if (result_event_ptr != NULL)
         {
            break; /* Conflict resolved */
         }
      }
   }

   if (result_event_ptr != NULL)
   {
      ATMR_InitialiseTimer(&timer_info, TIMER_TYPE_PVR_RECORD, serv_ptr, result_event_ptr);

      if (other_crid != NULL)
      {
         strncpy((char *)timer_info.u.record.other_crid, (char *)other_crid, TMR_PVR_CRID_LEN_MAX);
      }

      timer_info.u.record.recommendation = is_recommendation;
      timer_info.u.record.disk_id = STB_PVRGetDefaultDisk();
      timer_info.u.record.event_triggered = TRUE;
      timer_info.u.record.do_not_delete = do_not_delete;

#ifdef TMR_PVR_DEBUG
      TMR_PVR_DBG(("   Recording event \"%s\", start: %d:%d.%d, len: %d.%d, event=%d",
                   ADB_GetEventName(result_event_ptr), DHMS_DATE(timer_info.start_time),
                   DHMS_HOUR(timer_info.start_time), DHMS_MINS(timer_info.start_time),
                   DHMS_HOUR(timer_info.u.record.duration), DHMS_MINS(timer_info.u.record.duration),
                   timer_info.u.record.event_id));

      event_prog_crid = ADB_GetEventFullProgrammeCrid(serv_ptr, result_event_ptr);
      if (event_prog_crid != NULL)
      {
         TMR_PVR_DBG(("      prog CRID=\"%s\"", event_prog_crid));
         STB_AppFreeMemory(event_prog_crid);
      }
#endif

      /* Add the timer to record the event */
      ATMR_AddTimer(&timer_info);
   }

   if (conflicts != NULL)
   {
      STB_AppFreeMemory(conflicts);
   }

   ADB_SaveDatabase();

   FUNCTION_FINISH(ATMR_RecordEvent);

   return(result_event_ptr);
}

/**
 * @brief   Searches for events within 3 hours of the given start date/time for an
 *          event with the given programme CRID and creates a timer to record any found.
 *          The search is performed forward or backward from the given start date/time
 *          as defined by the search direction argument.
 * @param   serv_ptr pointer to service on which search is to take place
 * @param   prog_crid programme CRID to be searched for
 * @param   start_date_time date and time to start searching from
 * @param   do_not_delete flag that defines whether the recording should be marked as locked
 * @param   search_forward TRUE if the search is to be made for events following the given date/time
 * @return  TRUE if an event is found and is set for recording, FALSE otherwise
 */
BOOLEAN ATMR_RecordSplitEvent(void *serv_ptr, U8BIT *prog_crid, U32DHMS start_date_time,
   BOOLEAN do_not_delete, BOOLEAN search_forward)
{
   void *event_ptr;
   void *alt_event_ptr;
   U32DHMS limit_date_time;
   U32DHMS end_date_time;
   U32DHMS next_date_time;
   U32DHMS event_start_time;
   U8BIT *event_crid;
   U16BIT service_id;
   BOOLEAN continue_search;
   BOOLEAN event_recorded;

   FUNCTION_START(ATMR_RecordSplitEvent);

   event_recorded = FALSE;

   service_id = ADB_GetServiceId(serv_ptr);

   if (search_forward)
   {
      /* Calculate the time beyond which searching should be stopped */
      limit_date_time = STB_GCCalculateDHMS(start_date_time, DHMS_CREATE(0, 3, 0, 0), CALC_ADD);

      next_date_time = start_date_time;
      continue_search = TRUE;

      while (continue_search)
      {
         TMR_PVR_DBG(("--- Searching for event after %d:%02d.%02d", DHMS_DATE(next_date_time),
                      DHMS_HOUR(next_date_time), DHMS_MINS(next_date_time)));

         event_ptr = ADB_GetLaterEvent(serv_ptr, DHMS_DATE(next_date_time),
               DHMS_HOUR(next_date_time), DHMS_MINS(next_date_time));

         if (event_ptr != NULL)
         {
            /* The event found must start within 3 hours of the previous event */
            event_start_time = ADB_GetEventStartDateTime(event_ptr);

            if (limit_date_time >= event_start_time)
            {
               event_crid = ADB_GetEventFullProgrammeCrid(serv_ptr, event_ptr);
#ifdef TMR_PVR_DEBUG
               if (event_crid != NULL)
               {
                  TMR_PVR_DBG(("   Found event in time limit, crid=\"%s\"", event_crid));
               }
               else
               {
                  TMR_PVR_DBG(("   Found event in time limit"));
               }
#endif

               /* Check that the CRIDs are the same */
               if ((event_crid != NULL) && (STB_CompareStringsIgnoreCase(event_crid, prog_crid) == 0))
               {
                  /* Event has the same CRID and is within 3 hours of the previous event.
                   * Make sure it isn't already booked for recording */
                  if (!ATMR_FindTimerFromCridAndEvent(prog_crid, service_id, ADB_GetEventId(event_ptr)))
                  {
                     TMR_PVR_DBG(("   Recording event @ %d:%02d.%02d, crid=\"%s\"",
                                  ADB_GetEventStartDate(event_ptr), ADB_GetEventStartHour(event_ptr),
                                  ADB_GetEventStartMin(event_ptr), event_crid));
                     TMR_PVR_DBG(("   Search forward: Recording event @ %d:%02d.%02d, crid=\"%s\", event_id=%u\n",
                                  ADB_GetEventStartDate(event_ptr), ADB_GetEventStartHour(event_ptr),
                                  ADB_GetEventStartMin(event_ptr), event_crid, ADB_GetEventId(event_ptr)));

                     /* Book this event for recording */
                     alt_event_ptr = ATMR_RecordEvent(serv_ptr, event_ptr, prog_crid, NULL,
                        FALSE, FALSE, do_not_delete);
                     if (alt_event_ptr != NULL)
                     {
                        event_recorded = TRUE;

                        if (alt_event_ptr != event_ptr)
                        {
                           /* An alternative event has been used so release the original */
                           ADB_ReleaseEventData(event_ptr);
                           event_ptr = alt_event_ptr;
                        }

                        /* Search for additional split events before and after this part */
                        ATMR_RecordSplitEvent(serv_ptr, prog_crid,
                           ADB_GetEventStartDateTime(event_ptr), do_not_delete, FALSE);

                        ATMR_RecordSplitEvent(serv_ptr, prog_crid,
                           ADB_GetEventStartDateTime(event_ptr), do_not_delete, TRUE);
                     }
                     else
                     {
                        /* Failed to record this part of the split event, so.... ? */
                     }
                  }
               }

               /* Continue the search after this event */
               next_date_time = STB_GCCalculateDHMS(event_start_time,
                     ADB_GetEventDuration(event_ptr), CALC_ADD);

               if (event_crid != NULL)
               {
                  STB_AppFreeMemory(event_crid);
               }
            }
            else
            {
               /* Event found is outside of the 3 hour limit */
               continue_search = FALSE;
               TMR_PVR_DBG(("   No more events within 3 hours"));
            }

            ADB_ReleaseEventData(event_ptr);
         }
         else
         {
            /* No later events found so stop the search but save the CRID so that when the
             * EIT is updated it can be checked for new events appearing */
            continue_search = FALSE;
            TMR_PVR_DBG(("   No more events"));
         }
      }
   }
   else /* Search backwards through the EPG */
   {
      /* Calculate the time beyond which searching should be stopped */
      limit_date_time = STB_GCCalculateDHMS(start_date_time, DHMS_CREATE(0, 3, 0, 0), CALC_SUB);

      next_date_time = start_date_time;
      continue_search = TRUE;

      while (continue_search)
      {
         TMR_PVR_DBG(("--- Searching for event before %d:%02d.%02d", DHMS_DATE(next_date_time),
                      DHMS_HOUR(next_date_time), DHMS_MINS(next_date_time)));

         event_ptr = ADB_GetEarlierEvent(serv_ptr, DHMS_DATE(next_date_time),
               DHMS_HOUR(next_date_time), DHMS_MINS(next_date_time));

         if (event_ptr != NULL)
         {
            /* The event found must finish within 3 hours of the previous event */
            event_start_time = ADB_GetEventStartDateTime(event_ptr);
            end_date_time = STB_GCCalculateDHMS(event_start_time, ADB_GetEventDuration(event_ptr), CALC_ADD);

            if (limit_date_time <= end_date_time)
            {
               event_crid = ADB_GetEventFullProgrammeCrid(serv_ptr, event_ptr);
#ifdef TMR_PVR_DEBUG
               if (event_crid != NULL)
               {
                  TMR_PVR_DBG(("   Found event in time limit, crid=\"%s\"", event_crid));
               }
               else
               {
                  TMR_PVR_DBG(("   Found event in time limit"));
               }
#endif

               /* Check that the CRIDs are the same */
               if ((event_crid != NULL) && (STB_CompareStringsIgnoreCase(event_crid, prog_crid) == 0))
               {
                  /* Event has the same CRID and is within 3 hours of the previous event.
                   * Make sure it isn't already booked for recording */
                  if (!ATMR_FindTimerFromCridAndEvent(prog_crid, service_id, ADB_GetEventId(event_ptr)))
                  {
                     TMR_PVR_DBG(("   Recording event @ %d:%02d.%02d, crid=\"%s\"",
                                  ADB_GetEventStartDate(event_ptr), ADB_GetEventStartHour(event_ptr),
                                  ADB_GetEventStartMin(event_ptr), event_crid));
                     TMR_PVR_DBG(("   Search backward: Recording event @ %d:%02d.%02d, crid=\"%s\", event_id=%u\n",
                                  ADB_GetEventStartDate(event_ptr), ADB_GetEventStartHour(event_ptr),
                                  ADB_GetEventStartMin(event_ptr), event_crid, ADB_GetEventId(event_ptr)));

                     /* Book this event for recording */
                     alt_event_ptr = ATMR_RecordEvent(serv_ptr, event_ptr, prog_crid, NULL,
                        do_not_delete, FALSE, FALSE);
                     if (alt_event_ptr != NULL)
                     {
                        event_recorded = TRUE;
                        if (alt_event_ptr != event_ptr)
                        {
                           /* An alternative event has been used so release the original */
                           ADB_ReleaseEventData(event_ptr);
                           event_ptr = alt_event_ptr;
                        }

                        /* Search for additional split events before and after this part */
                        ATMR_RecordSplitEvent(serv_ptr, prog_crid,
                           ADB_GetEventStartDateTime(event_ptr), do_not_delete, FALSE);

                        ATMR_RecordSplitEvent(serv_ptr, prog_crid,
                           ADB_GetEventStartDateTime(event_ptr), do_not_delete, TRUE);
                     }
                     else
                     {
                        /* Failed to record this part of the split event, so......? */
                     }
                  }
               }

               /* Continue the search before this event */
               next_date_time = ADB_GetEventStartDateTime(event_ptr);

               if (event_crid != NULL)
               {
                  STB_AppFreeMemory(event_crid);
               }
            }
            else
            {
               /* Event found is outside of the 3 hour limit */
               continue_search = FALSE;
               TMR_PVR_DBG(("   No events within 3 hours"));
            }

            ADB_ReleaseEventData(event_ptr);
         }
         else
         {
            /* No earlier  events found so stop the search but save the CRID so that when the
             * EIT is updated it can be checked for new events appearing */
            continue_search = FALSE;
            TMR_PVR_DBG(("   No more events"));
         }
      }
   }

   FUNCTION_FINISH(ATMR_RecordSplitEvent);

   return(event_recorded);
}

/**
 * @brief   Searches the timers for the first timer that will cause the DVB to wakeup that hasn't
 *          been missed. This will be an alarm or PVR recording timer.
 * @param   date_time pointer to return the date/time of the first timer found
 * @param   onet_id returns the original network ID of the service associated with the timer
 * @param   trans_id returns the transport ID of the service associated with the timer
 * @param   service_id returns the service ID of the service associated with the timer
 * @return  timer handle, or INVALID_TIMER_HANDLE if no timer is found
 */
U32BIT ATMR_GetFirstWakeupTime(U32DHMS *date_time, U16BIT *onet_id, U16BIT *trans_id,
   U16BIT *service_id)
{
   ADB_TIMER_REC *timer;
   U32BIT handle;
   U32DHMS end_date_time;

   FUNCTION_START(AMTR_GetFirstWakeupTime);

   DBDEF_RequestAccess();

   handle = INVALID_TIMER_HANDLE;
   *date_time = 0;

   /* As the timer list is sorted in date/time order, just need to find the first recording timer */
   DBDEF_SortTimers(TRUE);

   for (timer = DBDEF_GetNextTimerRec(NULL); (handle == INVALID_TIMER_HANDLE) && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if (timer->type == TIMER_TYPE_ALARM)
      {
         if (!timer->missed && STB_GCIsFutureDateTime(DHMS_DATE(timer->start_time),
                DHMS_HOUR(timer->start_time), DHMS_MINS(timer->start_time), DHMS_SECS(timer->start_time)))
         {
            *date_time = timer->start_time;
            *onet_id = timer->u.alarm.orig_net_id;
            *trans_id = timer->u.alarm.transport_id;
            *service_id = timer->u.alarm.service_id;
            handle = timer->handle;
         }
      }
      else if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         /* Has this timer already been missed? */
         end_date_time = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);

         TMR_PVR_DBG(("%s: handle=0x%08x, start=%u@%02u:%02u, end=%u@%02u:%02u, missed=%u",
                  __FUNCTION__, timer->handle, DHMS_DATE(timer->start_time), DHMS_HOUR(timer->start_time),
                  DHMS_MINS(timer->start_time), DHMS_DATE(end_date_time), DHMS_HOUR(end_date_time),
                  DHMS_MINS(end_date_time), timer->missed));

         if (!timer->missed && STB_GCIsFutureDateTime(DHMS_DATE(end_date_time),
                DHMS_HOUR(end_date_time), DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
         {
            *date_time = timer->start_time;
            *onet_id = timer->u.record.orig_net_id;
            *trans_id = timer->u.record.transport_id;
            *service_id = timer->u.record.service_id;
            handle = timer->handle;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(AMTR_GetFirstWakeupTime);

   return(handle);
}

/**
 * @brief   Searches the timers for a recording timer with the given event and service IDs
 * @param   onet_id original network ID or the service
 * @param   trans_id transport ID of the service
 * @param   serv_id service ID
 * @param   event_id event ID
 * @return  handle of timer, or INVALID_TIMER_HANDLE if no timer is found
 */
U32BIT ATMR_FindTimerFromEvent(U16BIT onet_id, U16BIT trans_id, U16BIT serv_id, U16BIT event_id)
{
   ADB_TIMER_REC *timer;
   U32BIT timer_handle;

   FUNCTION_START(ATMR_FindTimerFromEvent);

   DBDEF_RequestAccess();

   timer_handle = INVALID_TIMER_HANDLE;

   for (timer = DBDEF_GetNextTimerRec(NULL); (timer_handle == INVALID_TIMER_HANDLE) && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         if ((timer->u.record.orig_net_id == onet_id) && (timer->u.record.transport_id == trans_id) &&
             (timer->u.record.service_id == serv_id) && (timer->u.record.event_id == event_id))
         {
            timer_handle = timer->handle;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_FindTimerFromEvent);

   return(timer_handle);
}

/**
 * @brief   Searches the timers for a recording timer with the given programme CRID
 * @param   prog_crid the programme CRID to be searched for
 * @return  handle of timer, or INVALID_TIMER_HANDLE if no timer is found
 */
U32BIT ATMR_FindTimerFromCrid(U8BIT *prog_crid)
{
   ADB_TIMER_REC *timer;
   U32BIT timer_handle;

   FUNCTION_START(ATMR_FindTimerFromCrid);

   DBDEF_RequestAccess();

   timer_handle = INVALID_TIMER_HANDLE;

   for (timer = DBDEF_GetNextTimerRec(NULL); (timer_handle == INVALID_TIMER_HANDLE) && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         if (STB_CompareStringsIgnoreCase(timer->u.record.prog_crid, prog_crid) == 0)
         {
            timer_handle = timer->handle;
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_FindTimerFromCrid);

   return(timer_handle);
}

/**
 * @brief   Searches the timers for a recording timer with the given programme
 *          CRID and event ID.
 * @param   prog_crid the programme CRID to search for
 * @param   service_id service ID of the event
 * @param   event_id ID of the event
 * @return  handle of timer, or INVALID_TIMER_HANDLE if no timer is found
 */
U32BIT ATMR_FindTimerFromCridAndEvent(U8BIT *prog_crid, U16BIT service_id, U16BIT event_id)
{
   ADB_TIMER_REC *timer;
   U32BIT timer_handle;

   FUNCTION_START(ATMR_FindTimerFromCridAndEvent);

   DBDEF_RequestAccess();

   timer_handle = INVALID_TIMER_HANDLE;

   for (timer = DBDEF_GetNextTimerRec(NULL); (timer_handle == INVALID_TIMER_HANDLE) && (timer != NULL);
        timer = DBDEF_GetNextTimerRec(timer))
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         if (timer->u.record.service_id == service_id)
         {
            if (STB_CompareStringsIgnoreCase(timer->u.record.prog_crid, prog_crid) == 0)
            {
               if (timer->u.record.event_id == event_id)
               {
                  timer_handle = timer->handle;
               }
            }
         }
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_FindTimerFromCridAndEvent);

   return(timer_handle);
}

/**
 * @brief   Deletes any PVR record timers with the given series CRID
 * @param   crid the series/recommendation CRID to be searched for
 * @param   is_recommendation TRUE if the given crid is for a recommendation, FALSE otherwise
 */
void ATMR_DeleteTimersForSeriesRecommendations(U8BIT *crid, BOOLEAN is_recommendation)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;
   BOOLEAN timer_deleted;

   FUNCTION_START(ATMR_DeleteTimersForSeriesRecommendations);

   DBDEF_RequestAccess();

   TMR_PVR_DBG(("ATMR_DeleteTimersForSeriesRecommendations(\"%s\")", crid));

   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = next_timer)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);
      timer_deleted = FALSE;

      if (is_recommendation)
      {
         /* A recommendation could refer to a single event, so the programme crid also needs to
          * be checked against the given crid */
         if (STB_CompareStringsIgnoreCase(timer->u.record.prog_crid, crid) == 0)
         {
            TMR_PVR_DBG(("ATMR_DeleteTimersForSeriesRecommendations: Deleting timer 0x%08lx", timer->handle));
            DeleteTimer(timer->handle);
            timer_deleted = TRUE;
         }
      }

      if (!timer_deleted &&
          (STB_CompareStringsIgnoreCase(timer->u.record.other_crid, crid) == 0))
      {
         TMR_PVR_DBG(("ATMR_DeleteTimersForSeriesRecommendations: Deleting timer 0x%08lx", timer->handle));
         DeleteTimer(timer->handle);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_DeleteTimersForSeriesRecommendations);
}

/**
 * @brief   Sets the additioinal information string for the specified timer and commits the change
 *          to the database
 * @param   handle Timer handle
 * @param   info pointer to the string
 * @param   size number of bytes in the string
 */
void ATMR_SetAdditionalInfo(U32BIT handle, U8BIT *info, U32BIT size)
{
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_SetAdditionalInfo);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (size >= TMR_PVR_ADDINFO_LEN_MAX)
      {
         size = TMR_PVR_ADDINFO_LEN_MAX - 1;
      }

      memcpy(timer->u.record.additional_info, info, size);
      timer->u.record.additional_info[size] = 0;

      if (timer->dba_rec != NULL)
      {
         DBA_SetFieldString(timer->dba_rec, DBA_FIELD_TIMER_ADDITIONAL_INFO,
            timer->u.record.additional_info, size);
      }

      if (timer->started)
      {
         STB_PVRRecordingSetAdditionalInfo(timer->u.record.recording_handle, info);
      }
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetAdditionalInfo);
}

/**
 * @brief   Gets the additional information string associated with a timer.
 *          The name is allocated in UI temp memory.
 * @param   handle Timer handle
 * @param   size Pointer to variable containing the number of bytes in the string
 * @return  additional information string, or NULL
 */
U8BIT* ATMR_GetAdditionalInfo(U32BIT handle, U32BIT *size)
{
   ADB_TIMER_REC *timer;
   U8BIT *additional_info = NULL;

   FUNCTION_START(ATMR_SetAdditionalInfo);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      *size = strlen((char *)timer->u.record.additional_info) + 1;
      additional_info = STB_AppGetMemory(*size);
      strncpy((char *)additional_info, (char *)timer->u.record.additional_info, *size);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetAdditionalInfo);

   return additional_info;
}

/**
 * @brief   Returns the value of start padding associated with the specified timer
 *          The start padding is the number of seconds before (after, if negative) the actual
 *          timer start time the timer will be triggered.
 * @param   handle Timer handle
 * @return  start padding
 */
S32BIT ATMR_GetStartPadding(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   S32BIT retval;

   FUNCTION_START(ATMR_GetStartPadding);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = GetStartPadding(timer);
   }
   else
   {
      retval = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetStartPadding);

   return retval;
}

/**
 * @brief   Sets the value of start padding associated with the specified timer
 *          The start padding is the number of seconds before (after, if negative) the actual
 *          timer start time the timer will be triggered.
 * @param   handle Timer handle
 * @param   start_padding new value for the timer start padding
 * @return  TRUE if the start padding could be changed, FALSE otherwise (e.g. due to a conflict)
 */
BOOLEAN ATMR_SetStartPadding(U32BIT handle, S32BIT start_padding)
{
   BOOLEAN retval;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_SetStartPadding);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = SetStartPadding(timer, start_padding);

      if (retval && (timer->dba_rec != NULL))
      {
         DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_START_PADDING,
            timer->u.record.start_padding);
      }
   }
   else
   {
      retval = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetStartPadding);

   return retval;
}

/**
 * @brief   Returns the value of end padding associated with the specified timer
 *          The end padding is the number of seconds after (before, if negative) the actual
 *          timer end time the timer will expire.
 * @param   handle Timer handle
 * @return  end padding
 */
S32BIT ATMR_GetEndPadding(U32BIT handle)
{
   S32BIT retval;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_GetEndPadding);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = GetEndPadding(timer);
   }
   else
   {
      retval = 0;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetEndPadding);

   return retval;
}

/**
 * @brief   Sets the value of end padding associated with the specified timer
 *          The end padding is the number of seconds after (before, if negative) the actual
 *          timer end time the timer will expire.
 * @param   handle Timer handle
 * @param   end_padding new value for the timer end_padding
 * @return  TRUE if the end padding could be changed, FALSE otherwise (e.g. due to a conflict)
 */
BOOLEAN ATMR_SetEndPadding(U32BIT handle, S32BIT end_padding)
{
   BOOLEAN retval;
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_SetEndPadding);

   DBDEF_RequestAccess();

   if ((timer = DBDEF_FindTimerRec(handle)) != NULL)
   {
      retval = SetEndPadding(timer, end_padding);
   }
   else
   {
      retval = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetEndPadding);

   return retval;
}

/**
 * @brief   Sets the do_not_delete flag in a recording timer, which if set to TRUE, will lock
 *          any recordings that are started by the timer, to prevent them being deleted
 * @param   handle timer handle
 * @param   do_not_delete new setting for the do not delete flag
 * @return  TRUE if the timer is valid, FALSE otherwise
 */
BOOLEAN ATMR_SetDoNotDelete(U32BIT handle, BOOLEAN do_not_delete)
{
   ADB_TIMER_REC *timer;
   BOOLEAN retval;

   FUNCTION_START(ATMR_SetDoNotDelete);

   retval = FALSE;

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      timer->u.record.do_not_delete = do_not_delete;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_SetDoNotDelete);

   return(retval);
}

/**
 * @brief   Returns the setting of the do not delete flag for the given timer
 * @param   handle timer handle
 * @return  the value of the do not delete flag, or FALSE if the timer isn't valid
 */
BOOLEAN ATMR_GetDoNotDelete(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN retval;

   FUNCTION_START(ATMR_GetDoNotDelete);

   DBDEF_RequestAccess();

   if (((timer = DBDEF_FindTimerRec(handle)) != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      retval = timer->u.record.do_not_delete;
   }
   else
   {
      retval = FALSE;
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_GetDoNotDelete);

   return(retval);
}

/*!**************************************************************************
 * @brief   Prints details of all existing timers
 ****************************************************************************/
void ATMR_DumpAllTimers(void)
{
#ifdef TMR_DEBUG
   ADB_TIMER_REC *timer;

   FUNCTION_START(ATMR_DumpAllTimers);

   DBDEF_RequestAccess();

   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = DBDEF_GetNextTimerRec(timer))
   {
      DumpTimer(timer);
   }

   DBDEF_ReleaseAccess();

   FUNCTION_FINISH(ATMR_DumpAllTimers);
#endif
}

//---local function definitions------------------------------------------------

/**
 * @brief   Deletes the timer with the given handle.
 * @param   handle timer handle to be deleted
 * @return  TRUE if the timer is found in the filter and deleted, FALSE otherwise
 */
static BOOLEAN DeleteTimer(U32BIT handle)
{
   ADB_TIMER_REC *timer;
   BOOLEAN timer_deleted;

   FUNCTION_START(DeleteTimer);

   timer_deleted = FALSE;

   timer = DBDEF_FindTimerRec(handle);
   if (timer != NULL)
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         /* If the recording is stopped successfully then it will delete the timer,
          * so it's only deleted here if the recording isn't stopped for some reason */
         if ((timer->u.record.recording_handle == STB_PVR_INVALID_HANDLE) ||
            !APVR_StopRecording(timer->u.record.recording_handle))
         {
            DBDEF_DeleteTimerRec(timer);

#ifdef INTEGRATE_HBBTV
            HBBTV_NotifyRecordingEvent(handle, HBBTV_RECORDING_DELETED);
#endif
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_BOOKING_DELETED,
               (void *)&handle, sizeof(handle));
         }
      }
      else
      {
         DBDEF_DeleteTimerRec(timer);
      }

      timer_deleted = TRUE;
   }

   FUNCTION_FINISH(DeleteTimer);

   return(timer_deleted);
}

/**
 *

 *
 * @brief   action timer record start
 *

 *
 * @return   TRUE if recording is started successfully
 *
 */
static BOOLEAN ActionTimerRecStart(ADB_TIMER_REC *timer)
{
   BOOLEAN retval;
   U32DHMS gmt;
   U32DHMS stop_time;
   BOOLEAN new_tuned_service;
   U32BIT recording_handle;

   FUNCTION_START(ActionTimerRecStart);

   retval = FALSE;

   TMR_PVR_DBG(("ActionTimerRecStart()"));

   stop_time = STB_GCCalculateDHMS(timer->start_time, timer->u.record.duration, CALC_ADD);
   gmt = STB_GCNowDHMSGmt();

   /* Stop timer in the future (stop >= now) */
   if (stop_time > gmt)
   {
      TMR_PVR_DBG(("ActionTimerRecStart: stop timer in future - start new recording"));

#ifdef INTEGRATE_HBBTV
      HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_ACQUIRING_RESOURCES);
#endif

      /* Acquire a path for recording */
      timer->u.record.path = APVR_PrepareNewRecording(timer->u.record.orig_net_id,
            timer->u.record.transport_id, timer->u.record.service_id, &new_tuned_service);

      if (timer->u.record.path != INVALID_RES_ID)
      {
         /* The recording may not actually be started here but all is successful so far */
         retval = TRUE;

         TMR_PVR_DBG(("ActionTimerRecStart: APVR_PrepareNewRecording, path=%u", timer->u.record.path));

         /* If the tuner was being used for live viewing then it may not have been retuned
          * if the service to be recorded is the one that was already being viewed. If this
          * is the case then the recording needs to be started now as the events indicating
          * tuning status won't be sent and so the recording wouldn't start.
          */
         if (!new_tuned_service)
         {
            TMR_PVR_DBG(("ActionTimerRecStart: Starting recording \"%s\" with path %d",
                     timer->name, timer->u.record.path));

            STB_PVRStartRecordRunning(timer->u.record.path);

            timer->started = APVR_StartNewRecording(timer->u.record.disk_id, timer->u.record.path,
                  timer->name, timer->u.record.event_id, timer->u.record.prog_crid,
                  timer->u.record.other_crid, &recording_handle);

            if (timer->started)
            {
               /* Set the recording handle in the timer */
               timer->u.record.recording_handle = recording_handle;
               timer->u.record.programme_finished = FALSE;

               TMR_PVR_DBG(("ActionTimerRecStart: Recording started"));

               STB_PVRRecordingSetAdditionalInfo(recording_handle, timer->u.record.additional_info);
               STB_PVRRecordingSetStartPadding(recording_handle, GetStartPadding(timer));
               STB_PVRRecordingSetEndPadding(recording_handle, GetEndPadding(timer));
               STB_PVRRecordingSetLocked(recording_handle, timer->u.record.do_not_delete);

#ifdef INTEGRATE_HBBTV
               HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_STARTED);
#endif
            }
            else
            {
               /* Recording tried to start but failed */
               TMR_PVR_DBG(("ActionTimerRecStart: Failed to start recording"));

               ACTL_TuneOff(timer->u.record.path);
               STB_DPReleasePath(timer->u.record.path, RES_OWNER_DVB);
               timer->u.record.path = INVALID_RES_ID;

               /* Mark the timer as missed so that it's rescheduled */
               timer->missed = TRUE;

               if (timer->dba_rec != NULL)
               {
                  DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_MISSED, timer->missed);
               }

               /* Send event to UI to inform a recording has failed */
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);

#ifdef INTEGRATE_HBBTV
               HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_FAILED_UNKNOWN_ERROR);
#endif

               retval = FALSE;
            }

            timer->starting = FALSE;
         }
      }
      else
      {
         TMR_PVR_DBG(("ActionTimerRecStart: Failed to get a tuner to record"));

         /* Mark the timer as missed so that it's rescheduled */
         timer->starting = FALSE;
         timer->missed = TRUE;

         if (timer->dba_rec != NULL)
         {
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_MISSED, timer->missed);
         }

         /* Send event to UI to inform a recording has failed */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);

#ifdef INTEGRATE_HBBTV
         HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_FAILED_RESOURCE_LIMITATION);
#endif
      }
   }
   else
   {
      TMR_PVR_DBG(("ActionTimerRecStart: Stop time in the past, so recording has been missed"));

      /* Stop time in the past, mark it as missed so that it's rescheduled */
      timer->u.record.recording_handle = STB_PVR_INVALID_HANDLE;
      timer->missed = TRUE;
      timer->starting = FALSE;

      if (timer->dba_rec != NULL)
      {
         DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_MISSED, timer->missed);
      }

#ifdef INTEGRATE_HBBTV
      HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_FAILED_UNKNOWN_ERROR);
#endif
   }

   FUNCTION_FINISH(ActionTimerRecStart);

   return(retval);
}

/**
 * @brief   Background task that continually checks whether any timers need to be stopped or
 *          started and if they've expired will reschedule them if approriate or delete them.
 * @param   param unused
 */
static void TimerTask(void *param)
{
   ADB_TIMER_REC *timer;
   ADB_TIMER_REC *next_timer;
   U32DHMS start_date_time, end_date_time;
   BOOLEAN timer_updated;
   BOOLEAN start_record;

   FUNCTION_START(TimerTask);

   USE_UNWANTED_PARAM(param);

   /* Wait for the date/time to be set */
   while (STB_GCGetGMTDate() == 0)
   {
      STB_OSTaskDelay(100);
   }

   DBDEF_RequestAccess();

   timer_updated = FALSE;

   /* Check each timer to determine whether it's expired, missed, etc. */
   timer = DBDEF_GetNextTimerRec(NULL);
   while (timer != NULL)
   {
      next_timer = DBDEF_GetNextTimerRec(timer);

      /* Ignore timers that are already marked as missed */
      if (!timer->missed)
      {
         if (timer->type == TIMER_TYPE_PVR_RECORD)
         {
            GetActualStartEndTime(timer->start_time, timer->u.record.duration,
               GetStartPadding(timer), GetEndPadding(timer), &start_date_time,
               &end_date_time);
         }
         else
         {
            start_date_time = timer->start_time;
            end_date_time = timer->start_time;
         }

         if (!STB_GCIsFutureDateTime(DHMS_DATE(start_date_time), DHMS_HOUR(start_date_time),
                DHMS_MINS(start_date_time), DHMS_SECS(start_date_time)))
         {
            /* The start time of this timer has already passed, check whether it's expired */
            if (STB_GCIsFutureDateTime(DHMS_DATE(end_date_time), DHMS_HOUR(end_date_time),
                   DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
            {
               /* The end time hasn't yet been reached, so set the timer as starting */
               timer->starting = TRUE;

               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_SUCCESS, (void *)&timer->handle,
                  sizeof(timer->handle));
            }
            else
            {
               /* The timer has been missed, see if it can be rescheduled */
               timer_updated = TRUE;

               if (!RescheduleTimer(timer))
               {
                  /* Timer hasn't been rescheduled */
                  if ((timer->type != TIMER_TYPE_PVR_RECORD) ||
                      ((STB_GetNumBytesInString(timer->u.record.prog_crid) == 0) &&
                       (STB_GetNumBytesInString(timer->u.record.other_crid) == 0)))
                  {
                     /* The timer can be deleted */
                     DeleteTimer(timer->handle);
                  }
                  else
                  {
                     /* Set it as expired and leave it to the app to deal with */
                     timer->expired = TRUE;
                     timer_updated |= FALSE;
                  }
               }
            }
         }
      }

      timer = next_timer;
   }

   if (timer_updated)
   {
      /* Timers have been changed so save the database */
      ADB_SaveDatabase();
   }

   DBDEF_ReleaseAccess();

   while (1)
   {
      TMR_DBG(("Time now (GMT): %d/%d/%d %02d:%02d", STB_GCGetGMTDay(), STB_GCGetGMTMonth(),
         STB_GCGetGMTYear(), STB_GCGetGMTHour(), STB_GCGetGMTMin()));

      DBDEF_RequestAccess();

      timer_updated = FALSE;

      /* First check for any timers that need to be stopped.
       * This needs to be done first to ensure that back-to-back recordings are
       * stopped and started in the correct order */
      timer = DBDEF_GetNextTimerRec(NULL);
      while (timer != NULL)
      {
         /* Has this timer already been started, or is it still starting? */
         if (timer->started || timer->starting)
         {
            /* Check whether it's time to stop */
            timer_updated |= HasTimerExpired(timer->handle);
         }
         else if (timer->expired)
         {
            /* Timer has expired but is still running, so send the event again */
            TMR_DBG(("[%s:%d]: Timer 0x%x still expired", __FUNCTION__, __LINE__, timer->handle));

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_SUCCESS, (void *)&timer->handle,
               sizeof(timer->handle));
         }

         timer = DBDEF_GetNextTimerRec(timer);
      }

      /* Now check for any timers that are due to be started */
      timer = DBDEF_GetNextTimerRec(NULL);
      while (timer != NULL)
      {
         /* Has this timer already been started? */
         if (!timer->started && !timer->expired && !timer->missed)
         {
            if (timer->type == TIMER_TYPE_PVR_RECORD)
            {
               timer_updated |= StartRecordTimer(timer, TRUE, &start_record);
            }
            else
            {
               /* Check whether this timer should be started */
               if (!timer->starting && !timer->missed &&
                   !STB_GCIsFutureDateTime(DHMS_DATE(timer->start_time), DHMS_HOUR(timer->start_time),
                      DHMS_MINS(timer->start_time), DHMS_SECS(timer->start_time)))
               {
                  /* Timer should now be started */
                  TMR_DBG(("[%s:%d]: Timer 0x%x starting", __FUNCTION__, __LINE__, timer->handle));

                  timer->starting = TRUE;

                  STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_SUCCESS,
                     (void *)&timer->handle, sizeof(timer->handle));
               }
            }
         }

         timer = DBDEF_GetNextTimerRec(timer);
      }

      /* Ensure database is saved */
      if (timer_updated)
      {
         ADB_SaveDatabase();
      }

      DBDEF_ReleaseAccess();

      /* Wait five secs */
      STB_OSTaskDelay(5000);
   }

   FUNCTION_FINISH(TimerTask);
}

static BOOLEAN StartRecordTimer(ADB_TIMER_REC *timer, BOOLEAN recordings_can_start, BOOLEAN *start_record)
{
   BOOLEAN timer_updated;
   U32DHMS start_date_time, end_date_time;
   U32BIT notify_time;
   U32DHMS padding_start, padding_end;
   U8BIT max_recordings;
   U8BIT num_recordings;
   BOOLEAN start_timer;
   BOOLEAN conflict_resolved;
   U32BIT *conflicts;
   U8BIT i;
   S32BIT start_padding;
   S32BIT end_padding;
   S32BIT padding;
   U32DHMS time_diff;
   S32BIT padding_time;

   FUNCTION_START(StartRecordTimer);

   timer_updated = FALSE;
   *start_record = FALSE;

   start_padding = GetStartPadding(timer);
   end_padding = GetEndPadding(timer);

   GetActualStartEndTime(timer->start_time, timer->u.record.duration, start_padding, end_padding,
      &start_date_time, &end_date_time);

   /* Check whether a notification needs to be sent that this timer is about to start */
   if (!timer->u.record.notified)
   {
      notify_time = timer->u.record.notify_time;
      if (notify_time == 0)
      {
         notify_time = APVR_GetNotifyTime();
      }

      if (notify_time != 0)
      {
         notify_time = STB_GCCalculateDHMS(start_date_time,
               STB_GCCreateDHMSFromSeconds(notify_time), CALC_SUB);

         if (!STB_GCIsFutureDateTime(DHMS_DATE(notify_time), DHMS_HOUR(notify_time),
            DHMS_MINS(notify_time), DHMS_SECS(notify_time)))
         {
            TMR_PVR_DBG(("%s: Timer 0x%x being notified", __FUNCTION__, timer->handle));

            /* Send notification that the timer is going to be triggered */
            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_FAIL,
               (void *)&timer->handle, sizeof(timer->handle));

            timer->u.record.notified = TRUE;
         }
      }
   }

   /* Check whether this timer should be started */
   if (!timer->starting && !timer->missed &&
       (!STB_GCIsFutureDateTime(DHMS_DATE(start_date_time), DHMS_HOUR(start_date_time),
          DHMS_MINS(start_date_time), DHMS_SECS(start_date_time)) ||
       (timer->u.record.event_triggered && timer->u.record.programme_started)))
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         start_timer = FALSE;

         max_recordings = STB_HWGetNumRecorders();

         if (start_padding != 0)
         {
            /* Find the number of recordings in progress during the start padding */
            GetActualStartEndTime(timer->start_time, 0, start_padding, 0, &padding_start,
               &padding_end);
            padding_end = timer->start_time;

            num_recordings = GetNumSimultaneousRecordings(timer->handle, timer->u.record.orig_net_id,
               timer->u.record.transport_id, timer->u.record.service_id, max_recordings,
               padding_start, padding_end, TRUE, TRUE, &conflicts);

            if (num_recordings < max_recordings)
            {
               /* This recording can be started */
               start_timer = TRUE;
            }
            else
            {
               TMR_PVR_DBG(("[%s:%d]: Resolving conflict for timer 0x%x, @ %02u:%02u:%02u (padding %d/%d)",
                  __FUNCTION__, __LINE__, timer->handle, DHMS_HOUR(start_date_time),
                  DHMS_MINS(start_date_time), DHMS_SECS(start_date_time), start_padding, end_padding));

               /* Check all the conflicting timers to see whether the conflict can be resolved.
                * Timers are first checked to see if reducing padding is possible */
               for (i = 0, conflict_resolved = FALSE; !conflict_resolved && (i < max_recordings) &&
                  (conflicts[i] != INVALID_TIMER_HANDLE); i++)
               {
                  if ((ATMR_GetEndDateTime(conflicts[i]) < padding_end) &&
                     ((padding = ATMR_GetEndPadding(conflicts[i])) != 0))
                  {
                     /* Found a timer with end padding that conflicts with the start padding of this
                      * recording. The difference between the actual start and end times of the two
                      * is the total padding time available and this needs to be split between the
                      * two timers */
                     time_diff = timer->start_time - ATMR_GetEndDateTime(conflicts[i]);
                     padding_time = DHMS_MINS(time_diff) * 60 + DHMS_SECS(time_diff);

                     /* Back-to-back recordings are ignored this time in preference to ones where
                      * the padding can be reduced rather than being set to zero */
                     if (padding_time != 0)
                     {
                        if (ATMR_GetEndPadding(conflicts[i]) < (padding_time / 2))
                        {
                           /* End padding already less than the new padding time, so just the start
                            * padding will be reduced */
                           padding_time -= ATMR_GetEndPadding(conflicts[i]);

                           TMR_PVR_DBG(("[%s:%d]: Changing start padding of timer 0x%x from %d to %d",
                              __FUNCTION__, __LINE__, timer->handle, start_padding, padding_time));

                           SetStartPadding(timer, padding_time);
                        }
                        else if (start_padding < (padding_time / 2))
                        {
                           /* Start padding already less than the new padding time, so just the
                            * end padding will be reduced */
                           padding_time -= start_padding;

                           TMR_PVR_DBG(("[%s:%d]: Changing end padding of timer 0x%x from %d to %d",
                              __FUNCTION__, __LINE__, conflicts[i], ATMR_GetEndPadding(conflicts[i]),
                              padding_time));

                           ATMR_SetEndPadding(conflicts[i], padding_time);
                        }
                        else
                        {
                           /* The total padding time can be split between the two timers */
                           TMR_PVR_DBG(("[%s:%d]: Splitting padding:", __FUNCTION__, __LINE__));
                           TMR_PVR_DBG(("  timer 0x%x start padding %d to %d",
                              timer->handle, start_padding, padding_time / 2));

                           SetStartPadding(timer, padding_time / 2);

                           TMR_PVR_DBG(("  timer 0x%x end padding %d to %d", conflicts[i],
                              ATMR_GetEndPadding(conflicts[i]),
                              padding_time - GetStartPadding(timer)));

                           ATMR_SetEndPadding(conflicts[i], padding_time - GetStartPadding(timer));
                        }

                        conflict_resolved = TRUE;

                        /* Check whether the recording associated with the conflicted timer
                         * should now be stopped */
                        HasTimerExpired(conflicts[i]);
                     }
                  }
               }

               /* If the conflict hasn't been resolved, now check whether there are any back-to-back
                * recordings, which means all padding will need to be removed */
               for (i = 0; !conflict_resolved && (i < max_recordings) &&
                  (conflicts[i] != INVALID_TIMER_HANDLE); i++)
               {
                  if (timer->start_time == ATMR_GetEndDateTime(conflicts[i]))
                  {
                     /* Found a back-to-back recording, so remove the padding from both of them,
                      * but the recording can't be started yet */
                     TMR_PVR_DBG(("[%s:%d]: Found back-to-back recording with timer 0x%x, setting padding to 0",
                        __FUNCTION__, __LINE__, conflicts[i]));

                     SetStartPadding(timer, 0);
                     ATMR_SetEndPadding(conflicts[i], 0);
                     conflict_resolved = TRUE;

                     /* Check whether the recording associated with the conflicted timer
                      * should now be stopped */
                     HasTimerExpired(conflicts[i]);
                  }
               }

               if (conflict_resolved)
               {
                  TMR_PVR_DBG(("[%s:%d]: Timer 0x%x padding now %d/%d", __FUNCTION__, __LINE__,
                     timer->handle, GetStartPadding(timer), GetEndPadding(timer)));

                  for (i = 0; (i < max_recordings) && (conflicts[i] != INVALID_TIMER_HANDLE); i++)
                  {
                     TMR_PVR_DBG(("[%s:%d]: Timer 0x%x padding now %d/%d", __FUNCTION__, __LINE__,
                        conflicts[i], ATMR_GetStartPadding(conflicts[i]),
                        ATMR_GetEndPadding(conflicts[i])));
                  }

                  STB_AppFreeMemory(conflicts);
                  conflicts = NULL;

                  /* Check whether the resources are now available to start this recording */
                  start_padding = GetStartPadding(timer);
                  end_padding = GetEndPadding(timer);

                  GetActualStartEndTime(timer->start_time, 0, start_padding, 0,
                     &padding_start, &padding_end);
                  padding_end = timer->start_time;

                  num_recordings = GetNumSimultaneousRecordings(timer->handle,
                     timer->u.record.orig_net_id, timer->u.record.transport_id,
                     timer->u.record.service_id, max_recordings,
                     padding_start, padding_end, TRUE, TRUE, &conflicts);

                  if (num_recordings < max_recordings)
                  {
                     /* The conflict has been resolved, which means padding will have been changed,
                      * so check again whether the timer should be started yet */
                     GetActualStartEndTime(timer->start_time, timer->u.record.duration,
                        start_padding, end_padding, &start_date_time, &end_date_time);

                     if (!STB_GCIsFutureDateTime(DHMS_DATE(start_date_time), DHMS_HOUR(start_date_time),
                         DHMS_MINS(start_date_time), DHMS_SECS(start_date_time)))
                     {
                        /* This recording can be started */
                        start_timer = TRUE;
                     }
                  }
               }
            }

            if (conflicts != NULL)
            {
               STB_AppFreeMemory(conflicts);
            }
         }
         else
         {
            /* Check whether the resources are available to start this recording */
            num_recordings = GetNumSimultaneousRecordings(timer->handle, timer->u.record.orig_net_id,
               timer->u.record.transport_id, timer->u.record.service_id, max_recordings,
               timer->start_time, ATMR_GetEndDateTime(timer->handle), TRUE, TRUE, &conflicts);

            if (num_recordings < max_recordings)
            {
               if (!timer->u.record.event_triggered || timer->u.record.programme_started)
               {
                  /* This recording can be started */
                  start_timer = TRUE;
               }
            }
            else
            {
               TMR_PVR_DBG(("[%s:%d]: Resolving conflict for timer 0x%x, @ %02u:%02u:%02u (padding %d/%d)",
                  __FUNCTION__, __LINE__, timer->handle, DHMS_HOUR(start_date_time),
                  DHMS_MINS(start_date_time), DHMS_SECS(start_date_time), start_padding, end_padding));

               /* Check the conflicting recordings to see if they have end padding
                * that can be changed and so resolve the conflict */
               for (i = 0, conflict_resolved = FALSE; !conflict_resolved && (i < max_recordings) &&
                  (conflicts[i] != INVALID_TIMER_HANDLE); i++)
               {
                  if ((ATMR_GetEndDateTime(conflicts[i]) <= timer->start_time) &&
                     (ATMR_GetEndPadding(conflicts[i]) != 0))
                  {
                     /* Found a recording with end padding that can be changed */
                     time_diff = timer->start_time - ATMR_GetEndDateTime(conflicts[i]);
                     padding_time = DHMS_MINS(time_diff) * 60 + DHMS_SECS(time_diff);

                     TMR_PVR_DBG(("[%s:%d]: Changing end padding for timer 0x%x from %d to %d",
                        __FUNCTION__, __LINE__, conflicts[i], ATMR_GetEndPadding(conflicts[i]),
                        padding_time));

                     ATMR_SetEndPadding(conflicts[i], padding_time);
                     conflict_resolved = TRUE;

                     /* Check whether the recording associated with the conflicted timer
                      * should now be stopped */
                     HasTimerExpired(conflicts[i]);
                  }
               }

               if (conflict_resolved)
               {
                  TMR_PVR_DBG(("[%s:%d]: Timer 0x%x padding now %d/%d", __FUNCTION__, __LINE__,
                     timer->handle, start_padding, end_padding));

                  for (i = 0; (i < max_recordings) && (conflicts[i] != INVALID_TIMER_HANDLE); i++)
                  {
                     TMR_PVR_DBG(("[%s:%d]:: Timer 0x%x padding now %d/%d", __FUNCTION__, __LINE__,
                        conflicts[i], ATMR_GetStartPadding(conflicts[i]),
                        ATMR_GetEndPadding(conflicts[i])));
                  }

                  STB_AppFreeMemory(conflicts);
                  conflicts = NULL;

                  /* Check whether the resources are now available to start this recording */
                  num_recordings = GetNumSimultaneousRecordings(timer->handle,
                     timer->u.record.orig_net_id, timer->u.record.transport_id,
                     timer->u.record.service_id, max_recordings, timer->start_time,
                     ATMR_GetEndDateTime(timer->handle), TRUE, TRUE, &conflicts);

                  if (num_recordings < max_recordings)
                  {
                     if (!timer->u.record.event_triggered || timer->u.record.programme_started)
                     {
                        /* This recording can be started */
                        start_timer = TRUE;
                     }
                  }
               }
            }

            if (conflicts != NULL)
            {
               STB_AppFreeMemory(conflicts);
            }
         }
      }
      else
      {
         start_timer = TRUE;
      }

      if (start_timer)
      {
         *start_record = TRUE;

         if (recordings_can_start)
         {
            /* Timer should now be started but first check that it hasn't also expired */
            TMR_PVR_DBG(("[%s:%d]: Timer 0x%x starting", __FUNCTION__, __LINE__, timer->handle));

            timer->starting = TRUE;

            if (timer->u.record.duration == 0)
            {
               /* Timer type is just triggered by the start time, so this timer has now expired */
               TMR_PVR_DBG(("[%s:%d]: Timer 0x%x has expired due to 0 duration!", __FUNCTION__,
                  __LINE__, timer->handle));

               timer->expired = TRUE;
               timer_updated |= RescheduleTimer(timer);
            }

            STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_SUCCESS,
               (void *)&timer->handle, sizeof(timer->handle));
         }
      }
   }
   else if (recordings_can_start && timer->starting && !timer->missed &&
      (timer->u.record.path != INVALID_RES_ID))
   {
      if (!ATMR_StartRecord(timer->u.record.path))
      {
         /* Recording didn't start, so release the path */
         ACTL_TuneOff(timer->u.record.path);
         STB_DPReleasePath(timer->u.record.path, RES_OWNER_DVB);

         /* Send event to UI to inform a recording has failed */
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_RECORDING_FAILED, NULL, 0);
      }
   }

   FUNCTION_FINISH(StartRecordTimer);

   return(timer_updated);
}

static BOOLEAN HasTimerExpired(U32BIT handle)
{
   BOOLEAN expired;
   ADB_TIMER_REC *timer;
   U32DHMS start_date_time, end_date_time;
#ifdef TMR_DEBUG
   U8BIT start_day, wday, start_month;
   U16BIT start_year;
   U8BIT end_day, end_month;
   U16BIT end_year;
#endif

   FUNCTION_START(HasTimerExpired);

   expired = FALSE;

   timer = DBDEF_FindTimerRec(handle);
   if ((timer != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         GetActualStartEndTime(timer->start_time, timer->u.record.duration,
            GetStartPadding(timer), GetEndPadding(timer), &start_date_time, &end_date_time);
      }
      else
      {
         start_date_time = timer->start_time;
         end_date_time = timer->start_time;
      }

#ifdef TMR_DEBUG
      STB_GCGetMJDDateInfo(DHMS_DATE(start_date_time), &start_day, &wday, &start_month, &start_year);
      STB_GCGetMJDDateInfo(DHMS_DATE(end_date_time), &end_day, &wday, &end_month, &end_year);
      TMR_DBG(("Timer 0x%x, start:%d/%d/%d %02d:%02d, end %d/%d/%d %02d:%02d; started:%s starting:%s", timer->handle,
         start_day, start_month, start_year, DHMS_HOUR(start_date_time), DHMS_MINS(start_date_time),
         end_day, end_month, end_year, DHMS_HOUR(end_date_time), DHMS_MINS(end_date_time),
         timer->started ? "TRUE" : "FALSE", timer->starting ? "TRUE" : "FALSE"));
#endif

      if (timer->type == TIMER_TYPE_PVR_RECORD)
      {
         if (!timer->u.record.event_triggered &&
             !STB_GCIsFutureDateTime(DHMS_DATE(end_date_time), DHMS_HOUR(end_date_time),
                DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
         {
            TMR_PVR_DBG(("[%s:%d]: Timer 0x%x expired", __FUNCTION__, __LINE__, timer->handle));
            expired = TRUE;
         }
         else if (timer->u.record.event_triggered)
         {
            if (timer->u.record.programme_finished &&
               ((GetEndPadding(timer) == 0) || !STB_GCIsFutureDateTime(DHMS_DATE(end_date_time),
                DHMS_HOUR(end_date_time), DHMS_MINS(end_date_time), DHMS_SECS(end_date_time))))
            {
               /* The programme has finished or the end time has been reached, so mark the timer
                * as expired so the recording will be stopped if it isn't event triggered */
               TMR_PVR_DBG(("[%s:%d]: Timer 0x%x expired", __FUNCTION__, __LINE__, timer->handle));
               expired = TRUE;
            }
            else if (!timer->u.record.programme_started && !STB_GCIsFutureDateTime(DHMS_DATE(end_date_time),
                DHMS_HOUR(end_date_time), DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
            {
               /* Programme didn't appear in EITpf so hasn't started and the end time has now been
                * reached, so assume the event has been missed */
               TMR_PVR_DBG(("[%s:%d]: Event missed, timer 0x%x expired", __FUNCTION__, __LINE__, timer->handle));
               expired = TRUE;
            }
         }
      }
      else if (!STB_GCIsFutureDateTime(DHMS_DATE(end_date_time), DHMS_HOUR(end_date_time),
             DHMS_MINS(end_date_time), DHMS_SECS(end_date_time)))
      {
         TMR_DBG(("%s: Timer 0x%x expired", __FUNCTION__, timer->handle));
         expired = TRUE;
      }

      if (expired)
      {
         /* Timer has now completed/expired */
         timer->starting = FALSE;
         timer->expired = TRUE;

         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_TIMER, EV_TYPE_SUCCESS, (void *)&timer->handle,
            sizeof(timer->handle));

         if (RescheduleTimer(timer))
         {
            if (timer->type == TIMER_TYPE_PVR_RECORD)
            {
#ifdef INTEGRATE_HBBTV
               HBBTV_NotifyRecordingEvent(timer->handle, HBBTV_RECORDING_UPDATED);
#endif
               STB_ERSendEvent(FALSE, FALSE, EV_CLASS_APPLICATION, EV_PVR_BOOKING_CREATED,
                  (void *)&timer->handle, sizeof(timer->handle));
            }
         }
      }
   }

   FUNCTION_FINISH(HasTimerExpired);

   return(expired);
}

static BOOLEAN RescheduleTimer(ADB_TIMER_REC *timer)
{
   BOOLEAN rescheduled;
   U16BIT code;
   U8BIT hour, mins, secs;
   E_STB_GC_WEEKDAY wday;
   U8BIT oohour, oomin, nohour, nomin;
   BOOLEAN ooneg, noneg;
   BOOLEAN check_offset = FALSE;

   FUNCTION_START(RescheduleTimer);

   rescheduled = FALSE;

   /* Set flag to show the timer is no longer started */
   timer->started = FALSE;

   code = DHMS_DATE(timer->start_time);
   hour = DHMS_HOUR(timer->start_time);
   mins = DHMS_MINS(timer->start_time);
   secs = DHMS_SECS(timer->start_time);

   /* Get old local time offset */
   STB_GCGetLocalTimeChange(code, hour, mins, secs, &oohour, &oomin, &ooneg);

   /* Adjust date/time? */
   switch (timer->frequency)
   {
      case TIMER_FREQ_ONCE:
         /* Timer doesn't need to be rescheduled */
         break;

      case TIMER_FREQ_WEEKLY:
         /* Next week, same time */
         code += 7;
         check_offset = TRUE;
         rescheduled = TRUE;
         break;

      case TIMER_FREQ_WEEKENDDAYS:
         /* Next weekend day, same time */
         wday = STB_GCGetDateWeekDay(code);
         if (wday == WEEKDAY_MONDAY)
            code += 5;
         if (wday == WEEKDAY_TUESDAY)
            code += 4;
         if (wday == WEEKDAY_WEDNESDAY)
            code += 3;
         if (wday == WEEKDAY_THURSDAY)
            code += 2;
         if (wday == WEEKDAY_FRIDAY)
            code += 1;
         if (wday == WEEKDAY_SATURDAY)
            code += 1;
         if (wday == WEEKDAY_SUNDAY)
            code += 6;
         check_offset = TRUE;
         rescheduled = TRUE;
         break;

      case TIMER_FREQ_WEEKDAYS:
         /* Next week day, same time */
         wday = STB_GCGetDateWeekDay(code);
         if (wday == WEEKDAY_MONDAY)
            code += 1;
         if (wday == WEEKDAY_TUESDAY)
            code += 1;
         if (wday == WEEKDAY_WEDNESDAY)
            code += 1;
         if (wday == WEEKDAY_THURSDAY)
            code += 1;
         if (wday == WEEKDAY_FRIDAY)
            code += 3;
         if (wday == WEEKDAY_SATURDAY)
            code += 2;
         if (wday == WEEKDAY_SUNDAY)
            code += 1;
         check_offset = TRUE;
         rescheduled = TRUE;
         break;

      case TIMER_FREQ_DAILY:
         /* Tomorrow, same time */
         code += 1;
         check_offset = TRUE;
         rescheduled = TRUE;
         break;

      case TIMER_FREQ_HOURLY:
         /* Today, one hour later */
         hour = DHMS_HOUR(timer->start_time) + 1;
         if (hour > 23)
         {
            /* Or tomorrow? */
            code += 1;
            hour -= 24;
         }
         rescheduled = TRUE;
         break;

      default:
         break;
   }

   /* Adjust timer for new local time offset? */
   if (check_offset == TRUE)
   {
      STB_GCGetLocalTimeChange(code, hour, mins, secs, &nohour, &nomin, &noneg);
      if ((oohour != nohour) || (oomin != nomin) || (ooneg != noneg))
      {
         if (ooneg)
         {
            STB_GCCalculateDateTime(code, hour, mins, secs, oohour, oomin, 0,
               &code, &hour, &mins, &secs, CALC_SUB);
         }
         else
         {
            STB_GCCalculateDateTime(code, hour, mins, secs, oohour, oomin, 0,
               &code, &hour, &mins, &secs, CALC_ADD);
         }

         if (noneg)
         {
            STB_GCCalculateDateTime(code, hour, mins, secs, nohour, nomin, 0,
               &code, &hour, &mins, &secs, CALC_ADD);
         }
         else
         {
            STB_GCCalculateDateTime(code, hour, mins, secs, nohour, nomin, 0,
               &code, &hour, &mins, &secs, CALC_SUB);
         }
      }
   }

   /* Set new time */
   timer->start_time = DHMS_CREATE(code, hour, mins, secs);

   if (timer->dba_rec != NULL)
   {
      DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_STARTTIME, timer->start_time);
   }

   if (rescheduled && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      /* Reset the notified flag so the notification event is sent for the next scheduled recording */
      timer->u.record.notified = FALSE;
   }

   FUNCTION_FINISH(RescheduleTimer);

   return(rescheduled);
}

/**
 * @brief   Background task that searches the EIT data for alternatives to events
 *          that failed to record based on data from the orginal recording timer.
 *          If an event is found, the original timer is deleted and a new one created.
 * @param   param - unused
 */
static void AlternateEventTask(void *param)
{
   S_ALT_EVENT_DATA alt_event_data;
   void *serv_ptr;
   void *event_ptr;
   void *alt_event_ptr;
#ifdef TMR_PVR_DEBUG
   U32DHMS start_time;
#endif
   U8BIT *new_prog_crid;

   USE_UNWANTED_PARAM(param);

   while (TRUE)
   {
      if (STB_OSReadQueue(alt_event_queue, &alt_event_data, sizeof(alt_event_data), TIMEOUT_NEVER))
      {
         /* Look for an alternative event. As the recording has been missed, the original
          * event isn't required but may still be in the schedule */
         event_ptr = ADB_FindEventFromCrid(alt_event_data.prog_crid, NULL, &serv_ptr);
         if (event_ptr != NULL)
         {
#ifdef TMR_PVR_DEBUG
            start_time = ADB_GetEventStartDateTime(event_ptr);
#endif

            /* Delete the original timer */
            DBDEF_RequestAccess();
            DeleteTimer(alt_event_data.timer_handle);
            DBDEF_ReleaseAccess();

            TMR_PVR_DBG(("Alternative event found for \"%s\" on %lu @ %02lu:%02lu on LCN %d\n",
                     ADB_GetEventName(event_ptr), DHMS_DATE(start_time), DHMS_HOUR(start_time),
                     DHMS_MINS(start_time), ADB_GetServiceLcn(serv_ptr)));

            /* Set the new event to be recorded */
            new_prog_crid = ADB_GetEventProgrammeCrid(serv_ptr, event_ptr);
            alt_event_ptr = ATMR_RecordEvent(serv_ptr, event_ptr, new_prog_crid,
                  alt_event_data.other_crid, alt_event_data.is_recommendation, TRUE,
                  alt_event_data.do_not_delete);

            if ((alt_event_ptr != NULL) && (alt_event_ptr != event_ptr))
            {
               ADB_ReleaseEventData(alt_event_ptr);
            }

            STB_AppFreeMemory(new_prog_crid);
            ADB_ReleaseEventData(event_ptr);

            /* Force update NVM in case power is lost */
            ADB_SaveDatabase();
         }
      }
   }
}

/**
 * @brief   Returns the acutal start and end time, taking into consideration timer start and
 *          duration, start_padding and end_padding
 * @param   timer_start start time as set in the timer
 * @param   timer_duration duration as set in the timer
 * @param   start_padding padding to apply to the timer start time
 * @param   end_padding padding to apply to the timer end time
 * @param   start_time pointer to a U32DHMS value where to store the actual start time
 * @param   end_time pointer to a U32DHMS value where to store the actual end timr
 */
static void GetActualStartEndTime(U32DHMS timer_start, U32DHMS timer_duration, S32BIT start_padding, S32BIT end_padding,
   U32DHMS *start_time, U32DHMS *end_time)
{
   U32BIT pad_hours, pad_mins, pad_secs;

   if (start_padding == 0)
   {
      *start_time = timer_start;
   }
   else
   {
      pad_hours = 0;
      pad_mins = 0;
      pad_secs = (start_padding < 0) ? (-1 * start_padding) : start_padding;

      while (pad_secs > 59)
      {
         pad_secs -= 60;
         pad_mins++;
      }
      while (pad_mins > 59)
      {
         pad_mins -= 60;
         pad_hours++;
      }

      if (start_padding > 0)
      {
         *start_time = STB_GCCalculateDHMS(timer_start,
               STB_GCCreateDHMS(0, pad_hours, pad_mins, pad_secs), CALC_SUB);
      }
      else
      {
         *start_time = STB_GCCalculateDHMS(timer_start,
               STB_GCCreateDHMS(0, pad_hours, pad_mins, pad_secs), CALC_ADD);
      }
   }

   *end_time = STB_GCCalculateDHMS(timer_start, timer_duration, CALC_ADD);

   if (end_padding != 0)
   {
      pad_hours = 0;
      pad_mins = 0;
      pad_secs = (end_padding < 0) ? (-1 * end_padding) : end_padding;

      while (pad_secs > 59)
      {
         pad_secs -= 60;
         pad_mins++;
      }
      while (pad_mins > 59)
      {
         pad_mins -= 60;
         pad_hours++;
      }

      if (end_padding > 0)
      {
         *end_time = STB_GCCalculateDHMS(*end_time,
               STB_GCCreateDHMS(0, pad_hours, pad_mins, pad_secs), CALC_ADD);
      }
      else
      {
         *end_time = STB_GCCalculateDHMS(*end_time,
               STB_GCCreateDHMS(0, pad_hours, pad_mins, pad_secs), CALC_SUB);
      }
   }
}

/**
 * @brief   Sets the timer fields using the timer info structure without saving the its database
 *          record. The call to this function must be protected by DBDEF_RequestAccess /
 *          DBDEF_ReleaseAccess.
 * @param   timer Timer record
 * @param   info Timer info structure
 */
static void SetTimerFields(ADB_TIMER_REC *timer, S_TIMER_INFO *info)
{
   U16BIT name_len;

   /* Set rest of timer fields based on info provided */
   timer->type = info->type;

   if ((info->frequency != TIMER_FREQ_ONCE) &&
       (info->frequency != TIMER_FREQ_WEEKLY) &&
       (info->frequency != TIMER_FREQ_WEEKENDDAYS) &&
       (info->frequency != TIMER_FREQ_WEEKDAYS) &&
       (info->frequency != TIMER_FREQ_DAILY))
   {
      timer->frequency = TIMER_FREQ_ONCE;
   }
   else
   {
      timer->frequency = info->frequency;
   }

   timer->start_time = info->start_time;

   if ((name_len = STB_GetNumBytesInString(info->name)) != 0)
   {
      memcpy(timer->name, info->name, name_len);
   }
   else
   {
      timer->name[0] = 0;
   }

   if (timer->dba_rec != NULL)
   {
      DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_TYPE, timer->type);
      DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_FREQUENCY, timer->frequency);
      DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_STARTTIME, timer->start_time);
      DBA_SetFieldString(timer->dba_rec, DBA_FIELD_REC_NAME, timer->name, name_len);
   }

   switch (timer->type)
   {
      case TIMER_TYPE_ALARM:
      {
         timer->u.alarm.change_service = info->u.alarm.change_service;

         timer->u.alarm.orig_net_id = info->u.alarm.orig_net_id;
         timer->u.alarm.transport_id = info->u.alarm.transport_id;
         timer->u.alarm.service_id = info->u.alarm.service_id;

         timer->u.alarm.ramp_volume = info->u.alarm.ramp_volume;

         if (timer->dba_rec != NULL)
         {
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_ORIG_NET_ID, timer->u.alarm.orig_net_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TRANSPORT_ID, timer->u.alarm.transport_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_SERVICE_ID, timer->u.alarm.service_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_RAMPVOLUME, timer->u.alarm.ramp_volume);
         }
         break;
      }

      case TIMER_TYPE_PVR_RECORD:
      {
         timer->u.record.duration = info->u.record.duration;

         timer->u.record.event_triggered = info->u.record.event_triggered;
         timer->u.record.event_id = info->u.record.event_id;

         timer->u.record.orig_net_id = info->u.record.orig_net_id;
         timer->u.record.transport_id = info->u.record.transport_id;
         timer->u.record.service_id = info->u.record.service_id;

         timer->u.record.disk_id = info->u.record.disk_id;

         timer->u.record.recommendation = info->u.record.recommendation;

         if (info->u.record.start_padding != timer->u.record.start_padding)
         {
            timer->u.record.has_start_padding = TRUE;
            timer->u.record.start_padding = info->u.record.start_padding;
         }

         if (info->u.record.end_padding != timer->u.record.end_padding)
         {
            timer->u.record.has_end_padding = TRUE;
            timer->u.record.end_padding = info->u.record.end_padding;
         }

         timer->u.record.notify_time = info->u.record.notify_time;
         timer->u.record.do_not_delete = info->u.record.do_not_delete;

         if ((name_len = STB_GetNumBytesInString(info->u.record.prog_crid)) != 0)
         {
            memcpy(timer->u.record.prog_crid, info->u.record.prog_crid, name_len);
            if (timer->dba_rec != NULL)
            {
               DBA_SetFieldString(timer->dba_rec, DBA_FIELD_TIMER_CRID, timer->u.record.prog_crid, name_len);
            }
         }
         else
         {
            timer->u.record.prog_crid[0] = '\0';
         }

         if ((name_len = STB_GetNumBytesInString(info->u.record.other_crid)) != 0)
         {
            memcpy(timer->u.record.other_crid, info->u.record.other_crid, name_len);
            if (timer->dba_rec != NULL)
            {
               DBA_SetFieldString(timer->dba_rec, DBA_FIELD_TIMER_OTHERCRID, timer->u.record.other_crid, name_len);
            }
         }
         else
         {
            timer->u.record.other_crid[0] = '\0';
         }

         if (timer->dba_rec != NULL)
         {
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DURATION, timer->u.record.duration);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_EVENT_TRIGGERED,
               timer->u.record.event_triggered);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_EVENTID, timer->u.record.event_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_ORIG_NET_ID, timer->u.record.orig_net_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TRANSPORT_ID, timer->u.record.transport_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_SERVICE_ID, timer->u.record.service_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DISKID, timer->u.record.disk_id);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_CRID_RECOMMENDED, timer->u.record.recommendation);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_NOTIFY_TIME, timer->u.record.notify_time);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DO_NOT_DELETE, timer->u.record.do_not_delete);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_START_PADDING, timer->u.record.start_padding);
            DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_END_PADDING, timer->u.record.end_padding);
         }

         timer->u.record.path = INVALID_RES_ID;
         break;
      }

      default:
         break;
   }
}

/**
 * @brief   Counts the number of simultaneous recordings between the given start and end date/time.
 * @param   handle handle of timer that's being checked, can be INVALID_TIMER_HANDLE if the check
 *                 isn't related to a timer
 * @param   onet_id original network ID of the recording timer being checked
 * @param   trans_id transport ID of the recording timer being checked
 * @param   serv_id service ID of the recording timer being checked
 * @param   max_recordings the max recordings that can take place at the same time
 * @param   start_date_time start date and time of the period to be checked
 * @param   end_date_time end date and time of the period to be checked
 * @param   include_start_padding TRUE if start padding is to be included in the calculation
 * @param   include_end_padding TRUE if end padding is to be included in the calculation
 * @param   conflicting_timers pointer to the list of conflicting timers, the
 *          list must to be freed by the caller. If NULL it's ignored.
 * @return  Number of simultaneous recordings during the given period
 */
static U8BIT GetNumSimultaneousRecordings(U32BIT handle, U16BIT onet_id, U16BIT trans_id,
   U16BIT serv_id, U8BIT max_recordings, U32DHMS start_date_time,
   U32DHMS end_date_time, BOOLEAN include_start_padding, BOOLEAN include_end_padding,
   U32BIT **conflicting_timers)
{
   ADB_TIMER_REC *timer;
   U32DHMS timer_duration;
   U32DHMS timer_end, timer_start;
   S_TIME_SLOT *timer_slot;
   U8BIT slot, max_slot;
   BOOLEAN slot_found;

//   FUNCTION_START(GetNumSimultaneousRecordings);

   /*
    * This function works by allocating an array of time slots that represent each of the tuners,
    * given by the max number of recordings. The list of timers is iterated, after being sorted
    * into date/time order, and for each timer that overlaps with the given start/end times the
    * time slots (tuners) are checked to see whether any of them are free to be used for that time
    * period (does the timer start/end overlap with the time slot start/end?). If the times overlap
    * for each used time slot then a new slot is required, but if not, then a slot can be reused.
    * The maximum number of slots used defines the maximum number of simultaneous recordings that
    * will take place during the given time period.
    */
   max_slot = 0;

   TMR_PVR_DBG(("GetNumSimultaneousRecordings(0x%x): between %02d.%02d -> %02d.%02d, max_recordings=%d",
      handle, DHMS_HOUR(start_date_time), DHMS_MINS(start_date_time), DHMS_HOUR(end_date_time),
      DHMS_MINS(end_date_time), max_recordings));

   /* Allocate memory to store the max number of simultaneous timers */
   timer_slot = (S_TIME_SLOT *)STB_AppGetMemory(max_recordings * sizeof(S_TIME_SLOT));
   if (conflicting_timers != NULL)
   {
      *conflicting_timers = (U32BIT *)STB_AppGetMemory(max_recordings * sizeof(U32BIT));
      if (*conflicting_timers != NULL)
      {
         for (slot = 0; slot < max_recordings; slot++)
         {
            (*conflicting_timers)[slot] = INVALID_TIMER_HANDLE;
         }
      }
   }

   DBDEF_SortTimers(TRUE);

   for (timer = DBDEF_GetNextTimerRec(NULL); timer != NULL; timer = DBDEF_GetNextTimerRec(timer))
   {
      if ((timer->type == TIMER_TYPE_PVR_RECORD) &&
         ((handle == INVALID_TIMER_HANDLE) || (timer->handle != handle)) &&
         !timer->expired && !timer->missed)
      {
         /* Get the timer's duration as saved in the database, if possible, so any extensions
          * to allow for overrun are ignored */
         if (timer->dba_rec != NULL)
         {
            DBA_GetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_DURATION, &timer_duration);
         }
         else
         {
            timer_duration = timer->u.record.duration;
         }

         GetActualStartEndTime(timer->start_time, timer_duration,
            (include_start_padding ? GetStartPadding(timer) : 0),
            (include_end_padding ? GetEndPadding(timer) : 0),
            &timer_start, &timer_end);

         TMR_PVR_DBG(("GetNumSimultaneousRecordings: Timer 0x%08lx: %u @ %02d.%02d -> %02d.%02d",
            timer->handle, DHMS_DATE(timer_start), DHMS_HOUR(timer_start), DHMS_MINS(timer_start),
            DHMS_HOUR(timer_end), DHMS_MINS(timer_end)));

         /* Check whether this timer coincides with the given period, but ignore timers that
          * coincide but are on the same service, i.e. it's a back-to-back recording, because
          * this event can't clash with the one that follows it */
         if ((timer_end > start_date_time) && (timer_start < end_date_time) &&
            ((onet_id != timer->u.record.orig_net_id) ||
             (trans_id != timer->u.record.transport_id) ||
             (serv_id != timer->u.record.service_id)))
         {
            /* Timer coincides with given time period.
             * Check whether the timer needs to be inserted into a new slot */
            slot_found = FALSE;
            for (slot = 0; !slot_found && (slot < max_slot) && (slot < max_recordings); )
            {
               if (timer_start >= timer_slot[slot].end_time)
               {
                  TMR_PVR_DBG(("GetNumSimultaneousRecordings: Using slot %d", slot));

                  /* This slot can be used for this timer */
                  timer_slot[slot].start_time = timer_start;
                  timer_slot[slot].end_time = timer_end;

                  if ((conflicting_timers != NULL) && (*conflicting_timers != NULL))
                  {
                     TMR_PVR_DBG(("[%s:%d] Saving timer 0x%x\n", __FUNCTION__, __LINE__, timer->handle));
                     (*conflicting_timers)[slot] = timer->handle;
                  }

                  slot_found = TRUE;
               }
               else
               {
                  slot++;
               }
            }

            if (!slot_found && (max_slot < max_recordings))
            {
               TMR_PVR_DBG(("GetNumSimultaneousRecordings: New slot %d", max_slot));
               /* This timer needs a new slot */
               timer_slot[max_slot].start_time = timer_start;
               timer_slot[max_slot].end_time = timer_end;

               if ((conflicting_timers != NULL) && (*conflicting_timers != NULL))
               {
                  (*conflicting_timers)[max_slot] = timer->handle;
               }

               max_slot++;
            }
         }
      }
   }

   STB_AppFreeMemory(timer_slot);

//   FUNCTION_FINISH(GetNumSimultaneousRecordings);

   return(max_slot);
}

static BOOLEAN SetStartPadding(ADB_TIMER_REC *timer, S32BIT padding)
{
   BOOLEAN retval;

   if ((timer != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      timer->u.record.has_start_padding = TRUE;
      timer->u.record.start_padding = padding;

      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   return(retval);
}

static S32BIT GetStartPadding(ADB_TIMER_REC *timer)
{
   S32BIT padding;

   padding = 0;

   if ((timer != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (timer->u.record.has_start_padding)
      {
         padding = timer->u.record.start_padding;
      }
      else
      {
         padding = APVR_GetStartPadding();
      }
   }

   return(padding);
}

static BOOLEAN SetEndPadding(ADB_TIMER_REC *timer, S32BIT padding)
{
   BOOLEAN retval;

   if ((timer != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      timer->u.record.has_end_padding = TRUE;
      timer->u.record.end_padding = padding;

      if (timer->dba_rec != NULL)
      {
         DBA_SetFieldValue(timer->dba_rec, DBA_FIELD_TIMER_END_PADDING,
            timer->u.record.end_padding);
      }

      if (timer->started)
      {
         STB_PVRRecordingSetEndPadding(timer->u.record.recording_handle, timer->u.record.end_padding);
      }

      retval = TRUE;
   }
   else
   {
      retval = FALSE;
   }

   return(retval);
}

static S32BIT GetEndPadding(ADB_TIMER_REC *timer)
{
   S32BIT padding;

   padding = 0;

   if ((timer != NULL) && (timer->type == TIMER_TYPE_PVR_RECORD))
   {
      if (timer->u.record.has_end_padding)
      {
         padding = timer->u.record.end_padding;
      }
      else
      {
         padding = APVR_GetEndPadding();
      }
   }

   return(padding);
}

#ifdef TMR_DEBUG
/**
 * @brief   Prints details of given timer
 * @param   timer timer
 */
static void DumpTimer(ADB_TIMER_REC *timer)
{
   FUNCTION_START(DumpTimer);

   TMR_DBG(("Timer 0x%08lx: ", timer->handle));
   TMR_DBG(("  name=\"%s\"", timer->name));
   TMR_DBG(("  freq=%u", timer->frequency));
   TMR_DBG(("  starts on=%u %02u:%02u:%02u",
            DHMS_DATE(timer->start_time), DHMS_HOUR(timer->start_time), DHMS_MINS(timer->start_time),
            DHMS_SECS(timer->start_time)));

   switch (timer->type)
   {
      case TIMER_TYPE_SLEEP:
         TMR_DBG(("  type=SLEEP"));
         break;

      case TIMER_TYPE_ALARM:
         TMR_DBG(("  type=ALARM"));
         TMR_DBG(("  ramp volume=%u", timer->u.alarm.ramp_volume));
         TMR_DBG(("  change service=%u", timer->u.alarm.change_service));
         TMR_DBG(("  service=0x%04x/0x%04x/0x%04x", timer->u.alarm.orig_net_id,
                  timer->u.alarm.transport_id, timer->u.alarm.service_id));
         break;

      case TIMER_TYPE_PVR_RECORD:
         TMR_DBG(("  type=PVR RECORD"));
         TMR_DBG(("  duration=%02u:%02u", DHMS_HOUR(timer->u.record.duration),
                  DHMS_MINS(timer->u.record.duration)));
         TMR_DBG(("  event triggered=%u", timer->u.record.event_triggered));
         if (timer->u.record.event_triggered)
         {
            TMR_DBG(("  event id=%u", timer->u.record.event_id));
         }
         TMR_DBG(("  service=0x%04x/0x%04x/0x%04x", timer->u.record.orig_net_id,
                  timer->u.record.transport_id, timer->u.record.service_id));
         TMR_DBG(("  disk id=0x%04x", timer->u.record.disk_id));
         if (strlen((char *)timer->u.record.prog_crid) != 0)
         {
            TMR_DBG(("  prog crid=\"%s\"", timer->u.record.prog_crid));
         }
         if (strlen((char *)timer->u.record.other_crid) != 0)
         {
            TMR_DBG(("  other crid=\"%s\"", timer->u.record.other_crid));
            TMR_DBG(("  recommendation=%u", timer->u.record.recommendation));
         }
         TMR_DBG(("  start padding=%u", timer->u.record.start_padding));
         TMR_DBG(("  end padding=%u", timer->u.record.end_padding));
         TMR_DBG(("  notify time=%u", timer->u.record.notify_time));
         TMR_DBG(("  do_not_delete=%u", timer->u.record.do_not_delete));
         TMR_DBG(("  recording handle=%08x", timer->u.record.recording_handle));
         TMR_DBG(("  path=%u", timer->u.record.path));
         break;

      default:
         TMR_DBG(("  type=OTHER(0x%02x)", timer->type));
         break;
   }

   TMR_DBG(("  starting=%u", timer->starting));
   TMR_DBG(("  started=%u", timer->started));
   TMR_DBG(("  expired=%u", timer->expired));
   TMR_DBG(("  missed=%u", timer->missed));
   TMR_DBG(("  selected=%u", timer->selected));

   FUNCTION_FINISH(DumpTimer);
}

#endif /* TMR_DEBUG */

