/*******************************************************************************
 * Copyright  2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2004 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Non_Volatile_Memory data handling functions
 *
 * @file    app_nvm.c
 * @date    16/09/2003
 */

//#define NVM_DEBUG

//---includes for this file-------------------------------------------------------------------------
// compiler library header files
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

// third party header files

// Ocean Blue Software header files
#include "techtype.h"   //generic data definitions
#include "dbgfuncs.h"   //debug functions

#include "stbhwav.h"
#include "stbdpc.h"
#include "stbgc.h"
#include "stbsiflt.h"
#include "stbsitab.h"
#include "stbheap.h"
#include "vtctype.h"

#include "app.h"        // Application header file
#include "ap_cfg.h"
#include "ap_dbacc.h"
#include "ap_ipadd.h"

#include "app_nvm.h"    //nvm data functions header file

#include "dba.h"

//---constant definitions for this file-------------------------------------------------------------
#define DVB_DATA_BLOCK_ID        0x44564220     /* 'DVB ' */

#define WIRELESS_ESSID_MAX       33
#define WIRELESS_PASSWORD_MAX    64

#ifdef NVM_DEBUG
#define DBG(X) STB_SPDebugWrite X
#else
#define DBG(X)
#endif


//---local typedefs, structs, enumerations for this file--------------------------------------------
typedef struct s_nvm_field
{
   void *field_ptr;
   void *def_ptr;
   U32BIT field_size;
}
S_NVM_FIELD;

typedef struct s_nvm_data
{
   BOOLEAN nvm_first_boot;                 // FIRST_BOOT_NVM
   BOOLEAN nvm_standby_powersave;          // STANDBY_POWERSAVE_NVM
   U8BIT nvm_volume;                       // VOLUME_NVM
   E_STB_AV_ASPECT_MODE nvm_aspect_mode;   // ASPECT_MODE_NVM
   E_STB_AV_SOURCES nvm_scart_input_type;  // SCART_INPUT_TYPE_NVM
   U8BIT nvm_primary_audio_lang_entry;     // PRIMARY_AUDIO_LANG_NVM
   U8BIT nvm_secondary_audio_lang_entry;   // SECONDARY_AUDIO_LANG_NVM
   ADB_AUDIO_TYPE nvm_audio_type;          // AUDIO_TYPE_NVM
   U8BIT nvm_standby_state;                // STANDBY_STATE_NVM
   BOOLEAN nvm_searching;                  // SEARCHING_STATE_NVM
   U8BIT nvm_parental_lock;                // PARENTAL_LOCK_NVM
   U8BIT nvm_ics_parental_lock;            // ICS_PARENTAL_LOCK_NVM
   U16BIT nvm_system_pin_no;               // SYSTEM_PIN_NO_NVM
   BOOLEAN nvm_aerial_power_on;            // AERIAL_POWER_ON_NVM
   U32BIT nvm_country_code;                // COUNTRY_CODE_NVM
   U8BIT nvm_region_id;                    // REGION_ID_NVM
   U8BIT nvm_uhf_channel;                  // RF_CHANNEL_NUM_NVM
   E_STB_AV_ASPECT_RATIO nvm_aspect_ratio; // ASPECT_RATIO_NVM
   E_STB_AV_SOURCES nvm_tv_scart_type;     // TV_SCART_TYPE_NVM
   U8BIT nvm_wakeup_volume;                // WAKEUP_VOLUME_NVM
   E_STB_AV_VIDEO_FORMAT hdmi_resolution;  // HDMI_RESOLUTION_NVM
   E_NET_IF_TYPE nvm_net_if_type;          // NET_IF_TYPE_NVM
   E_STB_IP_MODE nvm_ip_mode;              // IP_MODE_NVM
   U32BIT nvm_ip_address;                  // IP_ADDRESS_NVM
   U32BIT nvm_subnet_mask;                 // SUBNET_MASK_NVM
   U32BIT nvm_gateway_ip;                  // GATEWAY_IP_NVM
   U32BIT nvm_dns_ip;                      // DNS_SERVER_IP_NVM
   U8BIT nvm_essid[WIRELESS_ESSID_MAX];
   U8BIT nvm_essid_password[WIRELESS_PASSWORD_MAX];
   U8BIT nvm_ad_volume;                    // AD_VOLUME_NVM
   E_STB_DIGITAL_AUDIO_TYPE nvm_spdif_output;   // SPDIF_OUTPUT_NVM
   U16BIT nvm_lip_sync_adjustment;         // LIP_SYNC_ADJUSTMENT_NVM
   U8BIT nvm_target_region_depth;          // TARGET_REGION_DEPTH_NVM
   U32BIT nvm_target_region_country;       // TARGET_REGION_COUNTRY_NVM
   U8BIT nvm_target_region_primary;        // TARGET_REGION_PRIMARY_NVM
   U8BIT nvm_target_region_secondary;      // TARGET_REGION_SECONDARY_NVM
   U16BIT nvm_target_region_tertiary;      // TARGET_REGION_TERTIARY_NVM
   U8BIT nvm_ota_type;                     // OTA_TYPE_NVM
   U16BIT nvm_ota_last_update_srch;        // OTA_LAST_UPDATE_SRCH_NVM
   U16BIT nvm_last_chan_srch;              // LAST_CHAN_SRCH_NVM
   U32BIT nvm_last_eit_update;             // LAST_EIT_UPDATE_NVM
   E_VIEW_REC_PREF nvm_watch_hd;           // WATCH_HD_PROGRAMMES_NVM
   E_VIEW_REC_PREF nvm_record_hd;          // RECORD_HD_PROGRAMMES_NVM
   U16BIT nvm_timeshift_buffer_size;       // TIMESHIFT_BUFFER_SIZE_NVM
   E_SUBTITLE_TYPE nvm_subtitle_type;      // SUBTITLE_TYPE_NVM
   U16BIT nvm_background_search_start;     // BACKGROUND_SEARCH_START_NVM
   U16BIT nvm_background_search_end;       // BACKGROUND_SEARCH_END_NVM
   BOOLEAN nvm_service_search_enabled;     // SERVICE_SEARCH_ENABLED_NVM
   BOOLEAN nvm_ssu_search_enabled;         // SSU_SEARCH_ENABLED_NVM
   U16BIT nvm_live_service_lcn;            // LIVE_SERVICE_LCN_NVM
   U8BIT nvm_primary_text_lang;            // PRIMARY_TEXT_LANG_NVM
   U8BIT nvm_secondary_text_lang;          // SECONDARY_TEXT_LANG_NVM
   E_STB_DIGITAL_AUDIO_TYPE nvm_hdmi_audio_output;   // HDMI_AUDIO_OUTPUT_NVM
   U16BIT nvm_eit_sched_limit;             // EIT_SCHED_LIMIT_NVM
   BOOLEAN nvm_prefer_hd_audio;            // PREFER_HD_AUDIO_NVM
   U16BIT nvm_pvr_notify_time;             // PVR_NOTIFY_TIME_NVM
   S32BIT nvm_record_start_padding;         // RECORD_START_PADDING_NVM
   S32BIT nvm_record_end_padding;           // RECORD_END_PADDING_NVM

#ifdef COMMON_INTERFACE
   U8BIT cicam_id0_0[4];
   U8BIT cicam_id0_1[4];
   S32BIT cicam_pin0;
   U32BIT pin0_used_time;
   U8BIT cicam_id1_0[4];
   U8BIT cicam_id1_1[4];
   S32BIT cicam_pin1;
   U32BIT pin1_used_time;
   U8BIT cicam_id2_0[4];
   U8BIT cicam_id2_1[4];
   S32BIT cicam_pin2;
   U32BIT pin2_used_time;
   U8BIT cicam_id3_0[4];
   U8BIT cicam_id3_1[4];
   S32BIT cicam_pin3;
   U32BIT pin3_used_time;
#endif

   //NOTE_1: ANY ADDITIONAL NVM VARIABLES ARE TO BE ADDED JUST ABOVE THIS COMMENT
   //        AND ABOVE THE "checksum" struct element below.
   //NOTE_2: AN INITIALISATION VALUE MUST ALSO BE SUPPLIED IN THE CORRESPONDING
   //        LOCATION IN THE static const default_data (defined below)
   //NOTE_3: AN ENTRY IN THE  field_definition struct MUST ALSO BE SUPPLIED IN THE CORRESPONDING
   //        LOCATION IN THE static const S_NVM_FIELD field_definition (defined below)
   //-------------------------------------------------------------------------

   /* NVM Data Checksum - U16BIT stored as byte array to ensure that it can be located on a byte boundary
      and there will be no padding between the last item of data and the checksum */
   U8BIT checksum[sizeof(U16BIT)];                        //0x0000
} S_NVM_DATA;

//---local (static) variable declarations for this file---------------------------------------------
//   (internal variables declared static to make them local)
static const S_NVM_DATA default_data =
{
   TRUE,                     // nvm_first_boot;
   TRUE,                     // nvm_standby_powersave;
   100,                      // nvm_volume;
   ASPECT_MODE_AUTO,      // nvm_aspect_mode;
   AV_SOURCE_VCR_COMPOSITE,  // nvm_scart_input_type;
   0,                        // nvm_primary_audio_lang_entry;
   ACFG_INVALID_LANG,        // nvm_secondary_audio_lang_entry - set to invalid
   ADB_AUDIO_TYPE_UNDEFINED, // nvm_audio_type;
   0,                        // nvm_standby_state;
   FALSE,                    // nvm_searching;
   PARENTAL_LOCK_OFF,        // nvm_parental_lock;
   PARENTAL_LOCK_OFF,        // nvm_ics_parental_lock;
   0,                        // nvm_system_pin_no;
   FALSE,                    // nvm_aerial_power_on;
   0,                        // nvm_country_code;
   ACFG_INVALID_REGION,      // nvm_region_id;
   36,                       // uhf_channel_id;
   ASPECT_RATIO_16_9,        // nvm_aspect_ratio;
   AV_SOURCE_ENCODER_RGB,    // nvm_tv_scart_type;
   50,                       // nvm_wakeup_volume;
   VIDEO_FORMAT_AUTO,        // hdmi_resolution;
   NET_IF_NONE,              // NET_IF_TYPE
   IP_DHCP,                  // IP_MODE_NVM
   0,                        // IP_ADDRESS_NVM
   0,                        // SUBNET_MASK_NVM
   0,                        // GATEWAY_IP_NVM
   0,                        // DNS_SERVER_IP_NVM
   "",                       // ESSID_NVM
   "",                       // ESSID_PASSWORD_NVM
   100,                      // AD_VOLUME_NVM
   DIGITAL_AUDIO_PCM,        // SPDIF output
   0,                        // Lip sync adjustment
   0,                        // nvm_target_region_depth;
   0,                        // nvm_target_region_country;
   0,                        // nvm_target_region_primary;
   0,                        // nvm_target_region_secondary;
   0,                        // nvm_target_region_tertiary;
   OTA_TYPE_AUTO,            // nvm_ota_type
   0,                        // nvm_ota_last_update_srch
   0,                        // nvm_last_chan_srch
   0,                        // nvm_last_eit_update
   VIEW_REC_NEVER,           // nvm_watch_hd
   VIEW_REC_NEVER,           // nvm_record_hd
   60,                       // nvm_timeshift_buffer_size
   SUBTITLE_NORMAL,          // nvm_subtitle_type
   120,                      // Background search start (2am)
   240,                      // Background search end (4am)
   TRUE,                     // Background service search enabled
   TRUE,                     // Background SSU search enabled
   0,                        // Live service LCN
   0,                        // nvm_primary_text_lang;
   ACFG_INVALID_LANG,        // nvm_secondary_text_lang;
   DIGITAL_AUDIO_PCM,        // HDMI audio output
   0,                        // EIT schedule limit, 0=disabled
   TRUE,                     // Prefer HD audio
   0,                        // PVR notify time
   0,                        // Start padding on a recording
   0,                        // End padding on a recording
#ifdef COMMON_INTERFACE
   {0, 0, 0, 0},
   {0, 0, 0, 0},
   -1,
   0,
   {0, 0, 0, 0},
   {0, 0, 0, 0},
   -1,
   0,
   {0, 0, 0, 0},
   {0, 0, 0, 0},
   -1,
   0,
   {0, 0, 0, 0},
   {0, 0, 0, 0},
   -1,
   0,
#endif
   {0x00, 0x00}              //U16BIT checksum (stored as a 2 byte array to guarantee there is no padding between last data item & chcksum)
};

static S_NVM_DATA current_data;

static const S_NVM_FIELD field_definition[] =
{
   {(void *)&(current_data.nvm_first_boot), (void *)&(default_data.nvm_first_boot), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_standby_powersave), (void *)&(default_data.nvm_standby_powersave), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_volume), (void *)&(default_data.nvm_volume), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_aspect_mode), (void *)&(default_data.nvm_aspect_mode), sizeof(E_STB_AV_ASPECT_MODE)},
   {(void *)&(current_data.nvm_scart_input_type), (void *)&(default_data.nvm_scart_input_type), sizeof(E_STB_AV_SOURCES)},
   {(void *)&(current_data.nvm_primary_audio_lang_entry), (void *)&(default_data.nvm_primary_audio_lang_entry), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_secondary_audio_lang_entry), (void *)&(default_data.nvm_secondary_audio_lang_entry), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_audio_type), (void *)&(default_data.nvm_audio_type), sizeof(ADB_AUDIO_TYPE)},
   {(void *)&(current_data.nvm_standby_state), (void *)&(default_data.nvm_standby_state), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_searching), (void *)&(default_data.nvm_searching), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_parental_lock), (void *)&(default_data.nvm_parental_lock), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_ics_parental_lock), (void *)&(default_data.nvm_ics_parental_lock), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_system_pin_no), (void *)&(default_data.nvm_system_pin_no), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_aerial_power_on), (void *)&(default_data.nvm_aerial_power_on), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_country_code), (void *)&(default_data.nvm_country_code), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_region_id), (void *)&(default_data.nvm_region_id), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_uhf_channel), (void *)&(default_data.nvm_uhf_channel), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_aspect_ratio), (void *)&(default_data.nvm_aspect_ratio), sizeof(E_STB_AV_ASPECT_RATIO)},
   {(void *)&(current_data.nvm_tv_scart_type), (void *)&(default_data.nvm_tv_scart_type), sizeof(E_STB_AV_SOURCES)},
   {(void *)&(current_data.nvm_wakeup_volume), (void *)&(default_data.nvm_wakeup_volume), sizeof(U8BIT)},
   {(void *)&(current_data.hdmi_resolution), (void *)&(default_data.hdmi_resolution), sizeof(E_STB_AV_VIDEO_FORMAT)},
   {(void *)&(current_data.nvm_net_if_type), (void *)&(default_data.nvm_net_if_type), sizeof(E_NET_IF_TYPE)},
   {(void *)&(current_data.nvm_ip_mode), (void *)&(default_data.nvm_ip_mode), sizeof(E_STB_IP_MODE)},
   {(void *)&(current_data.nvm_ip_address), (void *)&(default_data.nvm_ip_address), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_subnet_mask), (void *)&(default_data.nvm_subnet_mask), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_gateway_ip), (void *)&(default_data.nvm_gateway_ip), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_dns_ip), (void *)&(default_data.nvm_dns_ip), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_essid), (void *)&(default_data.nvm_essid), sizeof(current_data.nvm_essid)},
   {(void *)&(current_data.nvm_essid_password), (void *)&(default_data.nvm_essid_password), sizeof(current_data.nvm_essid_password)},
   {(void *)&(current_data.nvm_ad_volume), (void *)&(default_data.nvm_ad_volume), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_spdif_output), (void *)&(default_data.nvm_spdif_output), sizeof(E_STB_DIGITAL_AUDIO_TYPE)},
   {(void *)&(current_data.nvm_lip_sync_adjustment), (void *)&(default_data.nvm_lip_sync_adjustment), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_target_region_depth), (void *)&(default_data.nvm_target_region_depth), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_target_region_country), (void *)&(default_data.nvm_target_region_country), sizeof(U32BIT)},
   {(void *)&(current_data.nvm_target_region_primary), (void *)&(default_data.nvm_target_region_primary), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_target_region_secondary), (void *)&(default_data.nvm_target_region_secondary), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_target_region_tertiary), (void *)&(default_data.nvm_target_region_tertiary), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_ota_type), (void *)&(default_data.nvm_ota_type), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_ota_last_update_srch), (void *)&(default_data.nvm_ota_last_update_srch), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_last_chan_srch), (void *)&(default_data.nvm_last_chan_srch), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_last_eit_update), (void *)&(default_data.nvm_last_eit_update), sizeof(U32DHMS)},
   {(void *)&(current_data.nvm_watch_hd), (void *)&(default_data.nvm_watch_hd), sizeof(E_VIEW_REC_PREF)},
   {(void *)&(current_data.nvm_record_hd), (void *)&(default_data.nvm_record_hd), sizeof(E_VIEW_REC_PREF)},
   {(void *)&(current_data.nvm_timeshift_buffer_size), (void *)&(default_data.nvm_timeshift_buffer_size), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_subtitle_type), (void *)&(default_data.nvm_subtitle_type), sizeof(E_SUBTITLE_TYPE)},
   {(void *)&(current_data.nvm_background_search_start), (void *)&(default_data.nvm_background_search_start), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_background_search_end), (void *)&(default_data.nvm_background_search_end), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_service_search_enabled), (void *)&(default_data.nvm_service_search_enabled), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_ssu_search_enabled), (void *)&(default_data.nvm_ssu_search_enabled), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_live_service_lcn), (void *)&(default_data.nvm_live_service_lcn), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_primary_text_lang), (void *)&(default_data.nvm_primary_text_lang), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_secondary_text_lang), (void *)&(default_data.nvm_secondary_text_lang), sizeof(U8BIT)},
   {(void *)&(current_data.nvm_hdmi_audio_output), (void *)&(default_data.nvm_hdmi_audio_output), sizeof(E_STB_DIGITAL_AUDIO_TYPE)},
   {(void *)&(current_data.nvm_eit_sched_limit), (void *)&(default_data.nvm_eit_sched_limit), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_prefer_hd_audio), (void *)&(default_data.nvm_prefer_hd_audio), sizeof(BOOLEAN)},
   {(void *)&(current_data.nvm_pvr_notify_time), (void *)&(default_data.nvm_pvr_notify_time), sizeof(U16BIT)},
   {(void *)&(current_data.nvm_record_start_padding), (void *)&(default_data.nvm_record_start_padding), sizeof(S32BIT)},
   {(void *)&(current_data.nvm_record_end_padding), (void *)&(default_data.nvm_record_end_padding), sizeof(S32BIT)},
#ifdef COMMON_INTERFACE
   {(void *)&(current_data.cicam_id0_0[0]), (void *)&(default_data.cicam_id0_0[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_id0_1[0]), (void *)&(default_data.cicam_id0_1[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_pin0), (void *)&(default_data.cicam_pin0), sizeof(S32BIT)},
   {(void *)&(current_data.pin0_used_time), (void *)&(default_data.pin0_used_time), sizeof(U32DHMS)},
   {(void *)&(current_data.cicam_id1_0[0]), (void *)&(default_data.cicam_id1_0[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_id1_1[0]), (void *)&(default_data.cicam_id1_1[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_pin1), (void *)&(default_data.cicam_pin1), sizeof(S32BIT)},
   {(void *)&(current_data.pin1_used_time), (void *)&(default_data.pin1_used_time), sizeof(U32DHMS)},
   {(void *)&(current_data.cicam_id2_0[0]), (void *)&(default_data.cicam_id2_0[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_id2_1[0]), (void *)&(default_data.cicam_id2_1[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_pin2), (void *)&(default_data.cicam_pin2), sizeof(S32BIT)},
   {(void *)&(current_data.pin2_used_time), (void *)&(default_data.pin2_used_time), sizeof(U32DHMS)},
   {(void *)&(current_data.cicam_id3_0[0]), (void *)&(default_data.cicam_id3_0[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_id3_1[0]), (void *)&(default_data.cicam_id3_1[0]), sizeof(U8BIT[4])},
   {(void *)&(current_data.cicam_pin3), (void *)&(default_data.cicam_pin3), sizeof(S32BIT)},
   {(void *)&(current_data.pin3_used_time), (void *)&(default_data.pin3_used_time), sizeof(U32DHMS)}
#endif
};

static BOOLEAN nvmdata_is_initialised;
static BOOLEAN nvm_data_has_changed;


//---local function prototypes for this file--------------------------------------------------------
//   (internal functions declared static to make them local)
static BOOLEAN VerifyNvmChecksum(void);
static U16BIT ComputeNvmChecksum(void);
static void SaveNvmData(void);

//--------------------------------------------------------------------------------------------------
// LOCAL FUNCTION DEFINITIONS
//--------------------------------------------------------------------------------------------------

/**
 *

 *
 * @brief   verifies whether NVM data checksum
 *

 *
 * @return   TRUE if the checksum verifies, otherwise FALSE
 *
 */
static BOOLEAN VerifyNvmChecksum(void)
{
   BOOLEAN verify_status;
   U16BIT temp_checksum;

   FUNCTION_START(VerifyNvmChecksum);

   ASSERT(nvmdata_is_initialised == TRUE);

   temp_checksum = (current_data.checksum[0] << 8) | current_data.checksum[1];

   if (temp_checksum == ComputeNvmChecksum())
   {
      verify_status = TRUE;
   }
   else
   {
      verify_status = FALSE;
   }

   FUNCTION_FINISH(VerifyNvmChecksum);

   return verify_status;
}

/**
 *

 *
 * @brief   computes the checksum for the current NVM data
 *

 *
 * @return   value of checksum
 *
 */
static U16BIT ComputeNvmChecksum(void)
{
   U16BIT checksum;
   U16BIT nvmdata;
   U32BIT num_nvmdata_bytes;
   U8BIT *nvmdata_ptr;

   FUNCTION_START(ComputeNvmChecksum);

   nvmdata_ptr = (U8BIT *)&current_data;
   num_nvmdata_bytes = (U32BIT)(current_data.checksum - nvmdata_ptr);

   //init checksum with an arbitary seed value
   //This ensures that nvm data of all zeros is not mistaken for valid data
   nvmdata = 0xECD7;
   checksum = nvmdata;

   //adding two U16BIT numbers (as opposed to U16BIT and a U8BIT) substantially
   //increases the period between recurring ffff values (ie for the case of the
   //nvm data being all ones.
   while (num_nvmdata_bytes > 0)
   {
      nvmdata <<= 8;
      nvmdata += (*nvmdata_ptr);
      checksum += nvmdata;

      nvmdata_ptr++;
      num_nvmdata_bytes--;
   }

   FUNCTION_FINISH(ComputeNvmChecksum);

   return checksum;
}

/**
 *

 *
 * @brief   Saves the current nvm data to flash memory
 *

 *

 *
 */
static void SaveNvmData(void)
{
   U16BIT temp_checksum;

   FUNCTION_START(SaveNvmData);

   if (nvm_data_has_changed)
   {
      temp_checksum = ComputeNvmChecksum();
      current_data.checksum[0] = temp_checksum >> 8;
      current_data.checksum[1] = temp_checksum & 0xff;

      DBG(("SaveNvmData: saving dvb data"));
      DBA_LockDatabase();
      DBA_DataBlockWrite(DVB_DATA_BLOCK_ID, (U8BIT *)&current_data, sizeof(S_NVM_DATA));
      DBA_UnlockDatabase();

      nvm_data_has_changed = FALSE;
   }

   FUNCTION_FINISH(SaveNvmData);
}

//--------------------------------------------------------------------------------------------------
// GLOBAL FUNCTION DEFINITIONS
//--------------------------------------------------------------------------------------------------

/**
 * @brief   Initialises the DVB core's NVM data
 */
void APP_NvmInitialise(void)
{
   FUNCTION_START(APP_NvmInitialise);

   /* Read DVB data */
   if ((DBA_DataBlockSize(DVB_DATA_BLOCK_ID) != sizeof(current_data)) ||
       (DBA_DataBlockRead(DVB_DATA_BLOCK_ID, (U8BIT *)&current_data, sizeof(current_data)) != sizeof(current_data)))
   {
      /* Use default settings */
      DBG(("APP_NvmInitialise: Using defaults"));
      APP_NvmRestoreDefaults();
   }

   nvmdata_is_initialised = TRUE;
   nvm_data_has_changed = FALSE;

   if (!VerifyNvmChecksum())
   {
      DBG(("APP_NvmInitialise CRC ERROR, RESTORING DEFAULTS\n"));
      APP_NvmRestoreDefaults();
   }

   FUNCTION_FINISH(APP_NvmInitialise);
}

/**
 * @brief   Resets the DVB's settings to factory defaults
 */
void APP_NvmRestoreDefaults(void)
{
   FUNCTION_START(APP_NvmRestoreDefaults);

   memcpy(&current_data, &default_data, sizeof(S_NVM_DATA));
   nvm_data_has_changed = TRUE;

   DBG(("APP_NvmRestoreDefaults: restoring defaults"));
   SaveNvmData();

   FUNCTION_FINISH(APP_NvmRestoreDefaults);
}

/**
 * @brief   Returns the factory default value for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  factory default value
 */
U32BIT APP_NvmReadDefault(E_NVM_ITEMS nvm_item)
{
   U32BIT default_item_value = 0;

   FUNCTION_START(APP_NvmReadDefault);

   //ensure valid nvm item
   if (nvm_item < NUM_ITEMS_NVM)
   {
      switch (field_definition[nvm_item].field_size)
      {
         case 1:
         {
            default_item_value = (U32BIT)(*((U8BIT *)field_definition[nvm_item].def_ptr));
            break;
         }
         case 2:
         {
            default_item_value = (U32BIT)(*((U16BIT *)field_definition[nvm_item].def_ptr));
            break;
         }
         case 4:
         {
            default_item_value = (U32BIT)(*((U32BIT *)field_definition[nvm_item].def_ptr));
            break;
         }
         default:
         {
            ASSERT(0);
            break;
         }
      }
   }

   FUNCTION_FINISH(APP_NvmReadDefault);
   return(default_item_value);
}

/**
 * @brief   Returns the current value for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  current value
 */
U32BIT APP_NvmRead(E_NVM_ITEMS nvm_item)
{
   U32BIT nvm_item_value = 0;

   FUNCTION_START(APP_NvmRead);

   //ensure valid nvm item
   if (nvm_item < NUM_ITEMS_NVM)
   {
      switch (field_definition[nvm_item].field_size)
      {
         case 1:
         {
            nvm_item_value = (U32BIT)(*((U8BIT *)field_definition[nvm_item].field_ptr));
            break;
         }
         case 2:
         {
            nvm_item_value = (U32BIT)(*((U16BIT *)field_definition[nvm_item].field_ptr));
            break;
         }
         case 4:
         {
            nvm_item_value = (U32BIT)(*((U32BIT *)field_definition[nvm_item].field_ptr));
            break;
         }
         default:
         {
            ASSERT(0);
            break;
         }
      }
   }

   FUNCTION_FINISH(APP_NvmRead);
   return(nvm_item_value);
}

/**
 * @brief   Returns pointer to current string for the given DVB setting
 * @param   nvm_item - value to be read
 * @return  pointer to string
 */
U8BIT *APP_NvmReadString(E_NVM_ITEMS nvm_item)
{
   U8BIT *nvm_item_ptr;

   FUNCTION_START(APP_NvmReadString);

   //ensure valid nvm item
   if (nvm_item < NUM_ITEMS_NVM)
   {
      ASSERT(field_definition[nvm_item].field_size == sizeof(U8BIT*));
      nvm_item_ptr = (U8BIT*)field_definition[nvm_item].field_ptr;
   }
   else
   {
      nvm_item_ptr = NULL;
   }

   FUNCTION_FINISH(APP_NvmReadString);
   return(nvm_item_ptr);
}

/**
 * @brief   Sets the current value for the given DVB setting
 * @param   nvm_item - item to be read
 * @param   new_value - value for the item
 * @param   write_to_flash_now - if TRUE then all the current values will be saved.
 *                               When changing the values of several items, it will
 *                               be more efficient to only this this TRUE for the last item
 */
void APP_NvmSave(E_NVM_ITEMS nvm_item, U32BIT new_value, BOOLEAN write_to_flash_now)
{
   U32BIT nvm_item_value;

   FUNCTION_START(APP_NvmSave);

   //ensure valid nvm item
   if (nvm_item < NUM_ITEMS_NVM)
   {
      DBG(("APP_NvmSave: nvm_item = 0x%x, write_now=%s", nvm_item, write_to_flash_now ? "TRUE" : "FALSE"));
      switch (field_definition[nvm_item].field_size)
      {
         case 1:
         {
            //read current nvm value
            nvm_item_value = (U32BIT)(*((U8BIT *)field_definition[nvm_item].field_ptr));

            //has nvm value changed
            if (nvm_item_value != new_value)
            {
               *((U8BIT *)field_definition[nvm_item].field_ptr) = (U8BIT)new_value;
               nvm_data_has_changed = TRUE;
            }

            //write to flash if reqd
            if (write_to_flash_now == TRUE)
            {
               SaveNvmData();
            }
            break;
         }
         case 2:
         {
            //read current nvm value
            nvm_item_value = (U32BIT)(*((U16BIT *)field_definition[nvm_item].field_ptr));

            //has nvm value changed
            if (nvm_item_value != new_value)
            {
               *((U16BIT *)field_definition[nvm_item].field_ptr) = (U16BIT)new_value;
               nvm_data_has_changed = TRUE;
            }

            //write to flash if reqd
            if (write_to_flash_now == TRUE)
            {
               SaveNvmData();
            }
            break;
         }
         case 4:
         {
            //read current nvm value
            nvm_item_value = (U32BIT)(*((U32BIT *)field_definition[nvm_item].field_ptr));

            //has nvm value changed
            if (nvm_item_value != new_value)
            {
               *((U32BIT *)field_definition[nvm_item].field_ptr) = (U32BIT)new_value;
               nvm_data_has_changed = TRUE;
            }

            //write to flash if reqd
            if (write_to_flash_now == TRUE)
            {
               SaveNvmData();
            }
            break;
         }
         default:
         {
            ASSERT(0);
            break;
         }
      }
   }

   FUNCTION_FINISH(APP_NvmSave);
}

/**
 * @brief   Sets the current value for the given DVB setting
 * @param   nvm_item - item to be written
 * @param   str_ptr - pointer to string for the item
 * @param   write_to_flash_now - if TRUE then all the current values will be saved.
 *                               When changing the values of several items, it will
 *                               be more efficient to only this this TRUE for the last item
 */
void APP_NvmSaveString(E_NVM_ITEMS nvm_item, U8BIT *str_ptr, BOOLEAN write_to_flash_now)
{
   U8BIT *nvm_item_ptr;

   FUNCTION_START(APP_NvmSave);

   //ensure valid nvm item
   if (nvm_item < NUM_ITEMS_NVM)
   {
      ASSERT(field_definition[nvm_item].field_size == sizeof(U8BIT*));
      DBG(("APP_NvmSaveString: nvm_item = %s, write_now=%s", str_ptr, write_to_flash_now ? "TRUE" : "FALSE"));
      nvm_item_ptr = (U8BIT *)field_definition[nvm_item].field_ptr;
      if (strcmp((char *)nvm_item_ptr, (char *)str_ptr) != 0)
      {
         strncpy((char *)nvm_item_ptr, (char *)str_ptr, field_definition[nvm_item].field_size);
         nvm_data_has_changed = TRUE;
      }
      if (write_to_flash_now == TRUE)
      {
         SaveNvmData();
      }
   }
   FUNCTION_FINISH(APP_NvmSaveString);
}

/**
 * @brief   Saves DVB values immediately
 */
void APP_NvmSaveAllNow(void)
{
   FUNCTION_START(APP_NvmSaveAllNow);
   DBG(("APP_NvmSaveAllNow"));
   SaveNvmData();
   FUNCTION_FINISH(APP_NvmSaveAllNow);
}

/**
 * @brief   Returns the size in bytes the DVB module uses to save its settings
 * @return  size of DVB settings in bytes
 */
U32BIT APP_NvmGetDvbSize(void)
{
   FUNCTION_START(APP_NvmGetDvbSize);
   FUNCTION_FINISH(APP_NvmGetDvbSize);
   return(sizeof(S_NVM_DATA));
}

