/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2009 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI+ Glue - IP network functions
 * @file     ci_glue_net.c
 * @date     19 February 2009
 * @author   Omri Barel
 */
/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>
#include <stdarg.h>

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbci.h"
#include "stbhwnet.h"
#include "stbhwmem.h"

/*---constant definitions for this file--------------------------------------*/

#define MAX_IP_CONNECTIONS    10
#define IP_TASK_STACK_SIZE    4096
#define IP_TASK_PRIORITY      9
#define IP_RECV_BUFFER_SIZE   65536
#define IP_RECV_TIMEOUT       500

/*---local typedef enums for this file-------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

typedef struct network_connection
{
   U32BIT module;
   void *send_data_sem;
   void *socket;
   U8BIT *send_buffer;
   S32BIT send_length;
   void *recv_mutex;
   U8BIT *recv_buffer[2];
   S32BIT recv_num_bytes[2];
   U8BIT fill_index;
   U8BIT send_index;
   BOOLEAN recv_buffer_sent;
} S_NETWORK_CONNECTION;

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */

static void *ip_mutex = NULL;
static S_NETWORK_CONNECTION ip_connections[MAX_IP_CONNECTIONS];

static void IPReadData(S_NETWORK_CONNECTION *ipc);


/*!**************************************************************************
 * @brief   Find IP connection for the given module
 * @param   module - module
 * @return  Pointer to connection information, NULL if not found
 ****************************************************************************/
static S_NETWORK_CONNECTION* FindIPConnection(U32BIT module)
{
   U8BIT index;
   S_NETWORK_CONNECTION *ipc = NULL;

   FUNCTION_START(FindIPConnection);

   for (index = 0; index < MAX_IP_CONNECTIONS; index++)
   {
      if (ip_connections[index].module == module)
      {
         ipc = &ip_connections[index];
         break;
      }
   }

   FUNCTION_FINISH(FindIPConnection);

   return(ipc);
}

/*!**************************************************************************
 * @brief   Background task that sends data using a given network connection
 * @param   param - S_NETWORK_CONNECTION* to be used
 ****************************************************************************/
static void IPSendDataTask(void *param)
{
   S_NETWORK_CONNECTION *ipc = (S_NETWORK_CONNECTION *)param;
   S32BIT bytes_sent;
   U8BIT *bufptr;

   FUNCTION_START(IPSendDataTask);

   if (ipc != NULL)
   {
      while (1)
      {
         /* Wait for a communication request through the connection */
         STB_OSSemaphoreWait(ipc->send_data_sem);

         if (ipc->socket != NULL)
         {
            bufptr = ipc->send_buffer;

            do
            {
               if ((bytes_sent = STB_NWSend(ipc->socket, bufptr, ipc->send_length)) >= 0)
               {
                  if (ipc->send_length >= bytes_sent)
                  {
                     bufptr += bytes_sent;
                     ipc->send_length -= bytes_sent;
                  }
                  else
                  {
                     DBGPRINT(" Sending %u bytes, but sent %ld for module 0x%08x",
                             ipc->send_length, bytes_sent, ipc->module)

                     ipc->send_length = 0;
                  }
               }
#ifdef DEBUG_IP
               else
               {
                  DBGPRINT(" Failed to send data for module 0x%08x", ipc->module)
               }
#endif
            }
            while ((bytes_sent >= 0) && (ipc->send_length > 0));

            /* Whether data has been sent or the send failed, inform the CI+ stack that the buffer
             * is no longer being used */
            DBGPRINT(" Data sent for module 0x%08lx", ipc->module)

            STB_CINotifyIPStatus(ipc->module, STB_CI_IP_STATUS_DATA_SENT);
         }
      }
   }

   FUNCTION_FINISH(IPSendDataTask);
}

/*!**************************************************************************
 * @brief   Used by IPReceiveDataTask to read data on a socket and pass it to the CI+ stack
 * @param   ipc - network connection to be read
 ****************************************************************************/
static void IPReadData(S_NETWORK_CONNECTION *ipc)
{
   S32BIT bytes_free;
   U8BIT *bufptr;
   S32BIT bytes_read;

   FUNCTION_START(IPReadData);

   /* Check that there's space in the current recv buffer */
   STB_OSMutexLock(ipc->recv_mutex);

   bytes_free = IP_RECV_BUFFER_SIZE - ipc->recv_num_bytes[ipc->fill_index];
   if (bytes_free > 0)
   {
      bufptr = ipc->recv_buffer[ipc->fill_index] + ipc->recv_num_bytes[ipc->fill_index];
      bytes_read = STB_NWReceive(ipc->socket, bufptr, bytes_free);
      if (bytes_read > 0)
      {
         DBGPRINT("(0x%08lx): read %ld bytes into buffer %u", ipc->module, bytes_read, ipc->fill_index)

         ipc->recv_num_bytes[ipc->fill_index] += bytes_read;

         /* Send the buffer to the CI+ stack however much is in it */
         if (!ipc->recv_buffer_sent)
         {
            DBGPRINT("(0x%08lx): Buffer %u ready, sending", ipc->module, ipc->fill_index)

            /* Send this buffer and switch to filling the next buffer */
            ipc->send_index = ipc->fill_index;
            ipc->fill_index = (ipc->fill_index + 1) % 2;

            if (STB_CIHandleIPData(ipc->module, ipc->recv_buffer[ipc->send_index],
                   ipc->recv_num_bytes[ipc->send_index]))
            {
               ipc->recv_buffer_sent = TRUE;
            }
            else
            {
               DBGPRINT(": STB_CIHandleIPData failed for module 0x%08lx", ipc->module)

               /* Data failed to be sent, so just mark the buffer empty */
               ipc->recv_num_bytes[ipc->send_index] = 0;
            }
         }
#ifdef DEBUG_IP
         else
         {
            DBGPRINT("(0x%08lx): Buffer %u ready but can't send", ipc->module, ipc->fill_index)
         }
#endif
      }
      else
      {
         /* Server has closed the socket */
         DBGPRINT("(0x%08lx): Read failed, closing the connection", ipc->module)
         STB_CICloseIPConnection(ipc->module);
      }
   }

   STB_OSMutexUnlock(ipc->recv_mutex);

   FUNCTION_FINISH(IPReadData);
}

/*!**************************************************************************
 * @brief   Background task that receives data from any open network connections
 * @param   param - unused
 ****************************************************************************/
static void IPReceiveDataTask(void *param)
{
   U8BIT i;
   U16BIT num_socks;
   S_NW_SOCKSET read_sockets;
   S_NW_SOCKSET except_sockets;

   FUNCTION_START(IPReceiveDataTask);
   USE_UNWANTED_PARAM(param);

   while (1)
   {
      STB_OSMutexLock(ip_mutex);

      num_socks = 0;
      STB_NWSockZero(&read_sockets);
      STB_NWSockZero(&except_sockets);

      for (i = 0; i < MAX_IP_CONNECTIONS; i++)
      {
         if (ip_connections[i].socket != NULL)
         {
            STB_NWSockSet(ip_connections[i].socket, &read_sockets);
            STB_NWSockSet(ip_connections[i].socket, &except_sockets);
            num_socks++;
         }
      }

      STB_OSMutexUnlock(ip_mutex);

      if (num_socks > 0)
      {
         if (STB_NWSelect(&read_sockets, NULL, &except_sockets, IP_RECV_TIMEOUT) > 0)
         {
            /* Find the sockets that have data available to be read */
            STB_OSMutexLock(ip_mutex);

            for (i = 0; i < MAX_IP_CONNECTIONS; i++)
            {
               if (ip_connections[i].socket != NULL)
               {
                  /* Check except socket first */
                  if (STB_NWSockIsSet(ip_connections[i].socket, &except_sockets))
                  {
                     DBGPRINT(" except socket %u", i)
                  }

                  if (STB_NWSockIsSet(ip_connections[i].socket, &read_sockets))
                  {
                     /* Data available on this connection */
                     IPReadData(&ip_connections[i]);
                  }
               }
            }

            STB_OSMutexUnlock(ip_mutex);
         }
      }
      else
      {
         /* No connections open, so just delay and check again */
         STB_OSTaskDelay(IP_RECV_TIMEOUT);
      }
   }

   FUNCTION_FINISH(IPReceiveDataTask);
}

#ifdef DEBUG_TRACING
static void PrintIPAddress(S_STB_CI_IP_ADDR *addr)
{
   FUNCTION_START(PrintIPAddress);

   switch (addr->type)
   {
      case STB_CI_IP_ADDR_V4:
         PRINTDBG("%u.%u.%u.%u", addr->data.ipv4_addr[0], addr->data.ipv4_addr[1], addr->data.ipv4_addr[2],
                 addr->data.ipv4_addr[3])
         break;

      case STB_CI_IP_ADDR_V6:
         PRINTDBG("%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x",
                 addr->data.ipv6_addr[0], addr->data.ipv6_addr[1], addr->data.ipv6_addr[2],
                 addr->data.ipv6_addr[3], addr->data.ipv6_addr[4], addr->data.ipv6_addr[5],
                 addr->data.ipv6_addr[6], addr->data.ipv6_addr[7], addr->data.ipv6_addr[8],
                 addr->data.ipv6_addr[9], addr->data.ipv6_addr[10], addr->data.ipv6_addr[11],
                 addr->data.ipv6_addr[12], addr->data.ipv6_addr[13], addr->data.ipv6_addr[14],
                 addr->data.ipv6_addr[15])
         break;

      case STB_CI_IP_ADDR_HOSTNAME:
         PRINTDBG("\"%s\"", addr->data.hostname)
         break;
   }

   FUNCTION_FINISH(PrintIPAddress);
}

#endif /* DEBUG_TRACING */


/*---global function definitions---------------------------------------------*/

void CIP_GlueNetInitialise(void)
{
   U16BIT i;

   ip_mutex = STB_OSCreateMutex();
   if (ip_mutex != NULL)
   {
      for (i = 0; i != MAX_IP_CONNECTIONS; i++)
      {
         ip_connections[i].socket = NULL;
         if ((ip_connections[i].send_data_sem = STB_OSCreateSemaphore()) != NULL)
         {
            STB_OSSemaphoreWait(ip_connections[i].send_data_sem);

            /* Create a background task to handle communications through the connection */
            STB_OSCreateTask(IPSendDataTask, &ip_connections[i], IP_TASK_STACK_SIZE, IP_TASK_PRIORITY, (U8BIT *)"IPSendData");
         }

         ip_connections[i].recv_mutex = STB_OSCreateMutex();
      }
      /* Create a background task to handle receiving of data on all connections */
      STB_OSCreateTask(IPReceiveDataTask, NULL, IP_TASK_STACK_SIZE, IP_TASK_PRIORITY, (U8BIT *)"IPReceiveData");
   }
}

/*****************************************************************************
 *
 * Function Name: STB_CIGetMaxIPConnections
 *
 * Description:   This function is called by the CI+ stack to find out how
 *                many simultaneous connections the host can support.
 *
 *                If the host does not support IP connections, this function
 *                should return zero.
 *
 * Parameters:    Nothing
 *
 * Returns:       Number of simultaneous IP connections supported by the host
 *
 ****************************************************************************/
U16BIT STB_CIGetMaxIPConnections(void)
{
   FUNCTION_START(STB_CIGetMaxIPConnections);
   FUNCTION_FINISH(STB_CIGetMaxIPConnections);

   return MAX_IP_CONNECTIONS;
}

/*****************************************************************************
 *
 * Function Name: STB_CIOpenIPConnection
 *
 * Description:   This function is called by the CI+ stack to open an IP
 *                connection.
 *
 *                When the connection is opened, the host must call
 *                STB_CINotifyIPStatus with STB_CI_IP_STATUS_CONNECTED.
 *
 *                If the connection attempt times out, the host must call
 *                STB_CINotifyIPStatus with STB_CI_IP_STATUS_TIMEOUT.
 *
 *                If the host cannot open the connection, it must call
 *                STB_CINotifyIPStatus with STB_CI_IP_STATUS_DISCONNECTED.
 *
 * Note:          This function must not block
 *
 * Parameters:    module - low-speed communication module
 *                addr - IP address or host name
 *                port - port number on server
 *                protocol - IP protocol (TCP or UDP)
 *                timeout - timeout in seconds
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CIOpenIPConnection(U32BIT module, S_STB_CI_IP_ADDR *addr,
   U16BIT port, E_STB_CI_IP_PROTOCOL protocol,
   U8BIT timeout)
{
   BOOLEAN connected = FALSE;
   U8BIT index, i;
   E_NW_PROTOCOL nw_protocol;
   E_NW_AF af_type;
   void *socket;
   U8BIT address[48];
   E_NW_ERROR err;
   BOOLEAN valid;
   U16BIT num_addrs;
   S_NW_ADDR_INFO *nw_addrs;

   FUNCTION_START(STB_CIOpenIPConnection);
   USE_UNWANTED_PARAM(timeout);

   if (ip_mutex == NULL)
   {
      CIP_GlueNetInitialise();
   }

   for (index = 0; index < MAX_IP_CONNECTIONS; index++)
   {
      if (ip_connections[index].socket == NULL)
      {
         if (protocol == STB_CI_IP_PROTOCOL_TCP)
         {
            nw_protocol = NW_PROTOCOL_TCP;
         }
         else
         {
            nw_protocol = NW_PROTOCOL_UDP;
         }

         valid = TRUE;

         switch (addr->type)
         {
            case STB_CI_IP_ADDR_V4:
               af_type = NW_AF_INET;
               sprintf((char *)address, "%u.%u.%u.%u", addr->data.ipv4_addr[0], addr->data.ipv4_addr[1],
               addr->data.ipv4_addr[2], addr->data.ipv4_addr[3]);
               break;

            case STB_CI_IP_ADDR_HOSTNAME:
               if ((num_addrs = STB_NWLookupAddress(addr->data.hostname, &nw_addrs)) > 0)
               {
                  valid = FALSE;

                  for (i = 0; (i < num_addrs) && !valid; i++)
                  {
                     if (nw_addrs[i].type == NW_SOCK_STREAM)
                     {
                        /* Found an address that can be used */
                        af_type = nw_addrs[i].af;
                        nw_protocol = nw_addrs[i].protocol;
                        strcpy((char *)address, (char *)nw_addrs[i].addr);
                        valid = TRUE;
                     }
                  }

                  STB_MEMFreeSysRAM(nw_addrs);
               }
               else
               {
                  valid = FALSE;
               }
               break;

            case STB_CI_IP_ADDR_V6:
               af_type = NW_AF_INET6;
               sprintf((char *)address, "%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x",
               addr->data.ipv6_addr[0], addr->data.ipv6_addr[1], addr->data.ipv6_addr[2],
               addr->data.ipv6_addr[3], addr->data.ipv6_addr[4], addr->data.ipv6_addr[5],
               addr->data.ipv6_addr[6], addr->data.ipv6_addr[7], addr->data.ipv6_addr[8],
               addr->data.ipv6_addr[9], addr->data.ipv6_addr[10], addr->data.ipv6_addr[11],
               addr->data.ipv6_addr[12], addr->data.ipv6_addr[13], addr->data.ipv6_addr[14],
               addr->data.ipv6_addr[15]);
               break;

            default:
               valid = FALSE;
               break;
         }

         if (valid)
         {
            if ((socket = STB_NWOpenSocket(af_type, NW_SOCK_STREAM, nw_protocol, FALSE)) != NULL)
            {
               if ((err = STB_NWConnect(socket, address, port)) == NW_OK)
               {
                  STB_OSMutexLock(ip_mutex);

                  /* Have found a connection that can be used and have connected successfully */
                  ip_connections[index].socket = socket;
                  ip_connections[index].module = module;

                  STB_OSMutexLock(ip_connections[index].recv_mutex);

                  /* Allocate memory for the receive buffers */
                  for (i = 0; i < 2; i++)
                  {
                     ip_connections[index].recv_num_bytes[i] = 0;
                     ip_connections[index].recv_buffer[i] = STB_MEMGetSysRAM(IP_RECV_BUFFER_SIZE);
                     if (ip_connections[index].recv_buffer[i] == NULL)
                     {
                        DBGPRINT("(0x%08x): Failed to allocate memory for receive buffer", module)
                     }
                  }

                  ip_connections[index].fill_index = 0;
                  ip_connections[index].send_index = 0;
                  ip_connections[index].recv_buffer_sent = FALSE;

                  STB_OSMutexUnlock(ip_connections[index].recv_mutex);

                  STB_OSMutexUnlock(ip_mutex);

                  DBGPRINT("(0x%08x): Successfully connected to \"%s\", port %u",
                          module, address, port)

                  STB_CINotifyIPStatus(module, STB_CI_IP_STATUS_CONNECTED);

                  connected = TRUE;
                  break;
               }
               else
               {
                  /* Failed to connect, so close the socket */
                  STB_NWCloseSocket(socket);
                  DBGPRINT("(0x%08x): Failed to connect to \"%s\", port %u",
                          module, address, port)
                  break;
               }
            }
            else
            {
               DBGPRINT("(0x%08x): Failed to open socket", module)
               break;
            }
         }
      }
   }

   if (!connected)
   {
      #ifdef DEBUG_TRACING
      if (index >= MAX_IP_CONNECTIONS)
      {
         DBGPRINT("(0x%08x): No more connections available to connect to ", module)
         PrintIPAddress(addr);
      }
      #endif

      STB_CINotifyIPStatus(module, STB_CI_IP_STATUS_DISCONNECTED);
   }

   FUNCTION_FINISH(STB_CIOpenIPConnection);
}

/*****************************************************************************
 *
 * Function Name: STB_CISendIPData
 *
 * Description:   This function is called by the CI+ stack to send data to an
 *                IP connection.
 *
 *                The host is responsible for sending the data (as one block,
 *                if possible). If some of the data cannot be sent, the host
 *                must retry until the entire buffer is sent.
 *
 *                Once the buffer is sent, the host must call the function
 *                STB_CINotifyIPStatus to inform the CI+ stack. The buffer
 *                remains valid until the host calls the notification
 *                function. The CI+ stack does not call this function again
 *                until it receives a notification from the host.
 *
 * Note:          This function must not block
 *
 * Parameters:    module - low-speed communication module
 *                buffer - the buffer to send
 *                len - number of bytes to send
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CISendIPData(U32BIT module, U8BIT *buffer, U16BIT len)
{
   S_NETWORK_CONNECTION *ipc;

   FUNCTION_START(STB_CISendIPData);

   if ((ipc = FindIPConnection(module)) != NULL)
   {
      if (ipc->socket != NULL)
      {
         DBGPRINT("(0x%08x): len=%u", module, len)

         /* As the buffer is guaranteed to be valid until the send has completed,
          * the address can just be passed to the background task */
         ipc->send_buffer = buffer;
         ipc->send_length = len;

         STB_OSSemaphoreSignal(ipc->send_data_sem);
      }
   }

   FUNCTION_FINISH(STB_CISendIPData);
}

/*****************************************************************************
 *
 * Function Name: STB_CIReleaseIPData
 *
 * Description:   This function is called by the CI+ stack to inform the
 *                host that the IP data can be released.
 *
 *                This function is called once all the data provided by
 *                STB_CIHandleIPData has been consumed.
 *
 * Parameters:    module - low-speed communication module
 *                buffer - data buffer to release
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CIReleaseIPData(U32BIT module, U8BIT *buffer)
{
   S_NETWORK_CONNECTION *ipc;

   FUNCTION_START(STB_CIReleaseIPData);
   USE_UNWANTED_PARAM(buffer);

   if ((ipc = FindIPConnection(module)) != NULL)
   {
      STB_OSMutexLock(ipc->recv_mutex);

#ifdef DEBUG_TRACING
      if (ipc->recv_buffer[ipc->send_index] != buffer)
      {
         DBGPRINT("(0x%08lx): Expected buffer %u(%p) but received buffer %p", module,
                 ipc->send_index, ipc->recv_buffer[ipc->send_index], buffer)
      }
      else
      {
         DBGPRINT("(0x%08lx): Buffer %u", module, ipc->send_index)
      }
#endif

      ipc->recv_buffer_sent = FALSE;
      ipc->recv_num_bytes[ipc->send_index] = 0;

      /* Check to see if there's another buffer waiting to be sent */
      ipc->send_index = (ipc->send_index + 1) % 2;

      if (ipc->recv_num_bytes[ipc->send_index] > 0)
      {
         DBGPRINT("(0x%08lx): Sending buffer %u", module, ipc->send_index)

         if (STB_CIHandleIPData(ipc->module, ipc->recv_buffer[ipc->send_index],
                ipc->recv_num_bytes[ipc->send_index]))
         {
            ipc->recv_buffer_sent = TRUE;

            /* If this buffer is the one that was being filled then need to switch buffers */
            if (ipc->fill_index == ipc->send_index)
            {
               /* Switch the buffer being filled */
               ipc->fill_index = (ipc->fill_index + 1) % 2;
            }
         }
         else
         {
            DBGPRINT("(0x%08lx): STB_CIHandleIPData failed", module)

            /* CI+ stack didn't want the data so mark the buffer as empty */
            ipc->recv_num_bytes[ipc->send_index] = 0;
         }
      }

      STB_OSMutexUnlock(ipc->recv_mutex);
   }

   FUNCTION_FINISH(STB_CIReleaseIPData);
}

/*****************************************************************************
 *
 * Function Name: STB_CICloseIPConnection
 *
 * Description:   This function is called by the CI+ stack to close an IP
 *                connection.
 *
 * Parameters:    module - low-speed communication module
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CICloseIPConnection(U32BIT module)
{
   S_NETWORK_CONNECTION *ipc;
   U8BIT i;

   FUNCTION_START(STB_CICloseIPConnection);

   DBGPRINT("(0x%08x)", module)

   if ((ipc = FindIPConnection(module)) != NULL)
   {
      if (ipc->socket != NULL)
      {
         STB_OSMutexLock(ip_mutex);

         if (!STB_NWCloseSocket(ipc->socket))
         {
            DBGPRINT("(0x%08x) STB_NWCloseSocket failed", module)
         }
         ipc->socket = NULL;

         STB_OSMutexLock(ipc->recv_mutex);

         /* Any received data is no longer required so can be thrown away */
         for (i = 0; i < 2; i++)
         {
            if (ipc->recv_buffer[i] != NULL)
            {
               STB_MEMFreeSysRAM(ipc->recv_buffer[i]);
               ipc->recv_buffer[i] = NULL;
            }
         }

         STB_OSMutexUnlock(ipc->recv_mutex);

         STB_OSMutexUnlock(ip_mutex);

         STB_CINotifyIPStatus(module, STB_CI_IP_STATUS_DISCONNECTED);
      }
   }

   FUNCTION_FINISH(STB_CICloseIPConnection);
}

/**
 *  @brief   This function is called by the CI+ stack to find out the
 *           IP configuration information from the Host.
 *           The Host shall report on the primary adapter, i.e. the adapter
 *           currently used by the host for IP communication.
 *           The host should, in addition, use
 *           STB_CIGetCommsIPConfigDNSServersAtIndex() to obtain information
 *           about the DNS servers.
 *  @param   comms_ip_config IP configuration information of the Host
 */
void STB_CIGetCommsIPConfig(S_STB_CI_COMMS_IP_CONFIG *comms_ip_config)
{
   FUNCTION_START(STB_CIGetCommsIPConfig);
   USE_UNWANTED_PARAM(comms_ip_config);
   FUNCTION_FINISH(STB_CIGetCommsIPConfig);
}

/**
 *  @brief   This function is called by the CI+ stack to find out the
 *           IP address of a configured DNS server.
 *           The Host shall report on the primary adapter, i.e. the adapter
 *           currently used by the host for IP communication.
 *           The CI+ stack should first call STB_CIGetCommsIPConfig() to know
 *           the exact number of configured DNS servers. After that it must
 *           query each DNS server address index from 0 up to
 *           (num_DNS_servers - 1).
 *  @param   index relative index of the DNS server
 *  @param   DNS_server_to_populate buffer where the IP should be copied
 *  @return  TRUE if the request DNS IP is correctly filled succeeded,
 *           FALSE otherwise
 */
BOOLEAN STB_CIGetCommsIPConfigDNSServersAtIndex(U8BIT index,
                                                COMMS_IP_CONFIG_IP_ARRAY_T *DNS_server_to_populate)
{
   FUNCTION_START(STB_CIGetCommsIPConfigDNSServersAtIndex);
   USE_UNWANTED_PARAM(index);
   USE_UNWANTED_PARAM(DNS_server_to_populate);
   FUNCTION_FINISH(STB_CIGetCommsIPConfigDNSServersAtIndex);
   return FALSE;
}


/**
 * @brief   This function is called by the CI+ stack to open an multicast IP
 *          connection. If multicast is not supported this should return FALSE
 *          When the connection is opened, the host must call
 *          STB_CINotifyIPStatus with STB_CI_IP_STATUS_CONNECTED.
 *          If the connection attempt times out, the host must call
 *          STB_CINotifyIPStatus with STB_CI_IP_STATUS_TIMEOUT.
 *          If the host cannot open the connection, it must call
 *          STB_CINotifyIPStatus with STB_CI_IP_STATUS_DISCONNECTED.
 * @note    This function must not block
 * @param   module low-speed communication module
 * @param   descriptor the multicast descriptor, never NULL.
 *                     data should be copied from this parameter if
 *                     it is to be used after the call returns
 * @param   timeout timeout in seconds
 * @return FALSE if multicast is not supported
 */
BOOLEAN STB_CIOpenMulticastConnection(U32BIT module,
                                      S_STB_CI_MULTICAST_DESCRIPTOR *descriptor,
                                      U8BIT timeout)
{
   FUNCTION_START(STB_CIOpenMulticastConnection);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(descriptor);
   USE_UNWANTED_PARAM(timeout);
   FUNCTION_FINISH(STB_CIOpenMulticastConnection);
   return FALSE;
}


