/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2009 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief    CI+ Glue
 * @file     ci_plus_glue.c
 * @date     19 February 2009
 * @author   Omri Barel
 */

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */

/* third party header files */

/* DVBCore header files */
#include "techtype.h"
#include "dbgfuncs.h"
#include "stbhwos.h"
#include "stbgc.h"
#include "stberc.h"
#include "stbcicc.h"

#include "ap_cfg.h"
#include "ap_ci.h"
#include "ap_ci_support.h"
#include "ap_dbacc.h"

#include "stbci.h"

/*---constant definitions for this file--------------------------------------*/

//#define DEBUG_GLUE
#ifdef DEBUG_GLUE
#define DBG(x) STB_SPDebugWrite x
#else
#define DBG(x)
#endif

#define SAT_DEL_SYS_DTAG            0x43
#define CABLE_DEL_SYS_DTAG          0x44
#define TERR_DEL_SYS_DTAG           0x5a
#define EXT_DTAG                    0x7f
#define T2_DELIVERY_SYS_DTAG        0x04

#define DELIVERY_TYPE_DVBT          0x01
#define DELIVERY_TYPE_DVBS          0x02
#define DELIVERY_TYPE_DVBC          0x04
#define DELIVERY_TYPE_DVBT2         0x10


/*---local typedef enums for this file-------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local) */


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local) */


/*---global function definitions---------------------------------------------*/


/*!**************************************************************************
 * @brief    Notify application information
 * @param    module - module ID
 * @param    app_type - application type
 * @param    app_manf - application manufacturer
 * @param    manf_code - manufacturer code
 * @param    menu_string - top level menu string
 ****************************************************************************/
void STB_CINotifyAppInfo(U32BIT module, U8BIT app_type, U16BIT app_manf, U16BIT manf_code, U8BIT *menu_string)
{
   FUNCTION_START(STB_CINotifyAppInfo);

#ifdef DEBUG_GLUE
   DBG(("Application information: module=%x",module));
   DBG(("application_type          = 0x%02x", app_type));
   DBG(("application_manufacturer  = 0x%04x", app_manf));
   DBG(("manufacturer_code         = 0x%04x", manf_code));
   DBG(("menu_string               = \"%s\"", (char *)menu_string));
#else
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(app_type);
   USE_UNWANTED_PARAM(app_manf);
   USE_UNWANTED_PARAM(manf_code);
   USE_UNWANTED_PARAM(menu_string);
#endif

   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_APP_INFO, &module, sizeof(U32BIT));

   FUNCTION_FINISH(STB_CINotifyAppInfo);
}

/*!**************************************************************************
 * @brief    Notify module insertion
 * @param    slot_id - slot ID for module
 ****************************************************************************/
void STB_CINotifyModuleInsert(U8BIT slot_id)
{
   FUNCTION_START(STB_CINotifyModuleInsert);

   ACFG_SetHostCountryLanguageModuleId(0);

   STB_CiCcNotifyModuleInsert(slot_id);

   FUNCTION_FINISH(STB_CINotifyModuleInsert);
}

/*!**************************************************************************
 * @brief    Notify module removal
 * @param    slot_id - slot ID for module
 ****************************************************************************/
void STB_CINotifyModuleRemove(U8BIT slot_id)
{
   FUNCTION_START(STB_CINotifyModuleRemove);

   DBG(("slot_id=%d", slot_id));

   ACFG_SetHostCountryLanguageModuleId(0);

   STB_CiCcNotifyModuleRemove(slot_id);

   FUNCTION_FINISH(STB_CINotifyModuleRemove);
}

/*!**************************************************************************
 * @brief    Notify UI screen event
 * @param    module - module ID
 * @param    event - screen event
 ****************************************************************************/
void STB_CINotifyScreenEvent(U32BIT module, U8BIT event)
{
   FUNCTION_START(STB_CINotifyScreenEvent);

   switch (event)
   {
      case STB_CI_SCREEN_EVENT_REQUEST:
         DBG(("CI_SCREEN_EVENT_REQUEST %x", module));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_OPEN, &module, sizeof(U32BIT));
         break;
      case STB_CI_SCREEN_EVENT_CLOSE:
         DBG(("CI_SCREEN_EVENT_CLOSE %x", module));
         STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_CLOSE, &module, sizeof(U32BIT));
         break;
   }

   FUNCTION_FINISH(STB_CINotifyScreenEvent);
}

/*****************************************************************************
 *
 *  HOST CONTROL FUNCTIONS
 *
 ****************************************************************************/

/*!**************************************************************************
 * @brief    Handle Tune request from the CAM
 * @param    module - module ID
 * @param    nid - network ID
 * @param    onid - original network ID
 * @param    tsid - transport stream ID
 * @param    sid - service ID
 ****************************************************************************/
void STB_CITune(U32BIT module, U16BIT nid, U16BIT onid, U16BIT tsid, U16BIT sid)
{
   FUNCTION_START(STB_CITune);
   ACI_HcTune(module, nid, onid, tsid, sid);
   FUNCTION_FINISH(STB_CITune);
}

/*!**************************************************************************
 * @brief    Handle Replace message from the CAM
 * @param    module - module ID
 * @param    ref - reference for replacement operation
 * @param    replaced_pid - PID to be replaced
 * @param    replacement_pid - replacement PID
 ****************************************************************************/
void STB_CIReplace(U32BIT module, U8BIT ref, U16BIT replaced_pid, U16BIT replacement_pid)
{
   FUNCTION_START(STB_CIReplace);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ref);
   USE_UNWANTED_PARAM(replaced_pid);
   USE_UNWANTED_PARAM(replacement_pid);
   FUNCTION_FINISH(STB_CIReplace);
}

/*!**************************************************************************
 * @brief    Clear PID replacement
 * @param    module - module ID
 * @param    ref - replacement reference (provided by STB_CIReplace)
 ****************************************************************************/
void STB_CIClearReplace(U32BIT module, U8BIT ref)
{
   FUNCTION_START(STB_CIClearReplace);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ref);
   FUNCTION_FINISH(STB_CIClearReplace);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyHostControlSession
 *
 * Description:   This function is used by the CI stack to notify the host
 *                that a session with the host countrol resource has been
 *                opened by the module.
 *
 *                This signals to the host that the module intends to request
 *                that the host tunes to other transport streams / services.
 *
 *                Following this notification, the host must call the
 *                function CIP_AskRelease to request that the module closes
 *                the session. The function STB_CIAskReleaseReply will be
 *                called to let the host know whether the control over the
 *                tuner has been released or not.
 *
 * Parameters:    module - host control module
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyHostControlSession(U32BIT module)
{
   FUNCTION_START(STB_CINotifyHostControlSession);
   ACI_HcNotifyHostControlSession(module);
   FUNCTION_FINISH(STB_CINotifyHostControlSession);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyHostControlSessionClosed
 *
 * Description:   This function is used by the CI stack to notify the host
 *                that a session with the host control resource has been
 *                closed by the module.
 *
 *                This signals to the host that it has controls over the tuner
 *                and it does not need to call CIP_AskRelease before using
 *                it.
 *
 *                Following this call, any PIDs that have been replaced
 *                by STB_CIReplace should be restored, if possible.
 *
 * Parameters:    module - host control module
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyHostControlSessionClosed(U32BIT module)
{
   FUNCTION_START(STB_CINotifyHostControlSessionClosed);
   ACI_HcNotifyHostControlSessionClosed(module);
   FUNCTION_FINISH(STB_CINotifyHostControlSessionClosed);
}
/*****************************************************************************
 *
 * Function Name: STB_CIAskReleaseReply
 *
 * Description:   This function is called by the CI+ stack to send the
 *                reply of a release request to the host.
 *
 *                This function is called in response to a call to
 *                STB_CIAskRelease. The host must not assume that it has
 *                control over the tuner until this function is called.
 *
 * Parameters:    module - specifies module required
 *                reply - the reply to the release request
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CIAskReleaseReply(U32BIT module, U8BIT release_reply)
{
   FUNCTION_START(STB_CIAskReleaseReply);
   ACI_HcAskReleaseReply(module, release_reply);
   FUNCTION_FINISH(STB_CIAskReleaseReply);
}

/**
 * @brief   This function is used by the CI+ stack to ask the host
 *          what broadcast delivery systems are supported, and the
 *          likely success of a tune depending on the delivery system
 *          type.
 *          The host must reply calling the function STB_CITunerStatusReply.
 * @param   module specifies module required
 */
void STB_CITunerStatusRequest(U32BIT module)
{
   S_STB_CI_TUNER_STATUS tuner_status;
   S_STB_CI_TUNER_STATUS_DSD dsds[4];
   U8BIT tuners, i, cnt, done;

   FUNCTION_START(STB_CITunerStatusRequest);

   tuner_status.IP_tune_capable_flag = FALSE; /* IP tuning not supported */

   tuners = STB_HWGetTunerPaths();

   for (i = 0, cnt = 0, done = 0; i != tuners; i++)
   {
      switch (STB_TuneGetSupportedSystemType(i))
      {
         case TUNE_SYSTEM_TYPE_DVBT2:
            if ((done & DELIVERY_TYPE_DVBT2) == 0)
            {
               done |= DELIVERY_TYPE_DVBT2;
               dsds[cnt].connected_flag = FALSE;
               dsds[cnt].delivery_system_descriptor_tag = EXT_DTAG;
               dsds[cnt].descriptor_tag_extension = T2_DELIVERY_SYS_DTAG;
               cnt++;
            }
            /* Intentional fall through */
         case TUNE_SYSTEM_TYPE_DVBT:
            if ((done & DELIVERY_TYPE_DVBT) == 0)
            {
               done |= DELIVERY_TYPE_DVBT;
               dsds[cnt].connected_flag = FALSE;
               dsds[cnt].delivery_system_descriptor_tag = TERR_DEL_SYS_DTAG;
               dsds[cnt].descriptor_tag_extension = 0;
               cnt++;
            }
            break;
         case TUNE_SYSTEM_TYPE_DVBS:
         case TUNE_SYSTEM_TYPE_DVBS2:
            if ((done & DELIVERY_TYPE_DVBS) == 0)
            {
               done |= DELIVERY_TYPE_DVBS;
               dsds[cnt].connected_flag = FALSE;
               dsds[cnt].delivery_system_descriptor_tag = SAT_DEL_SYS_DTAG;
               dsds[cnt].descriptor_tag_extension = 0;
               cnt++;
            }
            break;
         case TUNE_SYSTEM_TYPE_DVBC:
            if ((done & DELIVERY_TYPE_DVBC) == 0)
            {
               done |= DELIVERY_TYPE_DVBC;
               dsds[cnt].connected_flag = FALSE;
               dsds[cnt].delivery_system_descriptor_tag = CABLE_DEL_SYS_DTAG;
               dsds[cnt].descriptor_tag_extension = 0;
               cnt++;
            }
            break;
         default:;
      }
   }
   tuner_status.num_dsd = cnt;
   tuner_status.dsds = dsds;

   STB_CITunerStatusReply(module, &tuner_status);

   FUNCTION_FINISH(STB_CITunerStatusRequest);
}

/**
 * @brief   This function is used by the CI+ stack to request that the
 *          host tunes to a transport stream using the given parameters.
 *          If the service_id is zero, the host should not report a PMT
 *          using STB_CIUpdatePmt.
 *          If the PMT is provided, it should be used when selecting
 *          elementary streams on the service. The PMT is not provided
 *          if service_id is zero.
 *          Following the tune operation (whether successful or not),
 *          the host must call the function STB_CITuneReply.
 * @param   module host control module
 * @param   service_id the service to tune to
 * @param   desc_loop_len length of descriptor loop in bytes
 * @param   desc_loop the descriptor loop
 * @param   pmt the PMT to use (or NULL)
 * @param   tune_quietly parameter specified in v3 of tune_broadcas t_request
 *                       message from CAM,
 *                       or STB_CI_TUNE_QUIETLY_UNKNOWN if not received with message
 * @param   keep_app_running parameter specified in v3 of tune_broadcast_request
 *                           message from CAM,
 *                           or STB_CI_KEEP_APP_RUNNING_UNKNOWN if not received with message
 */
void STB_CITuneBroadcastRequest(U32BIT module,
                                U16BIT service_id,
                                U16BIT desc_loop_len,
                                U8BIT *desc_loop,
                                U8BIT *pmt,
                                E_STB_CI_TUNE_QUIETLY_FLAG tune_quietly,
                                E_STB_CI_KEEP_APP_RUNNING_FLAG keep_app_running)
{
   E_CIP_TUNE_FLAGS flags;

   FUNCTION_START(STB_CITuneBroadcastRequest);

   flags = (tune_quietly == STB_CI_TUNE_QUIETLY_SET)? CIP_QUIET_TUNE : CIP_NORMAL_TUNE;
   if (keep_app_running == STB_CI_KEEP_APP_RUNNING_SET)
   {
      flags &= CIP_KEEP_APP_RUNING;
   }
   ACI_HcTuneBroadcastRequest(module, service_id, desc_loop_len, desc_loop, pmt, flags);

   FUNCTION_FINISH(STB_CITuneBroadcastRequest);
}

/**
 * @brief   This function is used by the CI+ stack to request that the
 *          host tunes to a LCN using the given parameters.
 *          Following the tune operation (whether successful or not),
 *          the host must call the function STB_CITuneReply.
 * @param   module host control module
 * @param   lcn of the service to tune, shall be within the range 0-9999 inclusive,
 *              the host shall resolve this to its active channel list
 * @param   tune_quietly parameter specified in v3 of tune_broadcast_request
 *                       message from CAM,
 *                       should never be STB_CI_TUNE_QUIETLY_UNKNOWN in this usage
 * @param   keep_app_running parameter specified in v3 of tune_broadcast_request
 *                           message from CAM,
 *                           should never be STB_CI_KEEP_APP_RUNNING_UNKNOWN in this usage
 */
void STB_CITuneLCNRequest(U32BIT module,
                          U16BIT lcn,
                          E_STB_CI_TUNE_QUIETLY_FLAG tune_quietly,
                          E_STB_CI_KEEP_APP_RUNNING_FLAG keep_app_running)
{
   E_CIP_TUNE_FLAGS flags;
   void *s_ptr;

   FUNCTION_START(STB_CITuneLCNRequest);

   s_ptr = ADB_FindServiceByLcn(ADB_SERVICE_LIST_ALL, lcn, TRUE);
   if (s_ptr != NULL)
   {
      flags = (tune_quietly == STB_CI_TUNE_QUIETLY_SET)? CIP_QUIET_TUNE : CIP_NORMAL_TUNE;
      if (keep_app_running == STB_CI_KEEP_APP_RUNNING_SET)
      {
         flags &= CIP_KEEP_APP_RUNING;
      }
      ACI_HcTuneService(module, s_ptr, flags);
   }
   else
   {
      STB_CITuneReply(module, STB_CI_TUNE_SERVICE_NOT_FOUND);
   }

   FUNCTION_FINISH(STB_CITuneLCNRequest);
}

/**
 * @brief   This function is used by the CI+ stack to request that the
 *          host tunes using the given DVB triplet parameters.
 *          Following the tune operation (whether successful or not),
 *          the host must call the function STB_CITuneReply.
 * @param   module host control module
 * @param   original_network_id network id of the triplet
 * @param   transport_stream_id transport stream id of the triplet
 * @param   service_id service id of the triplet
 * @param   delivery_system_descriptor_tag the delivery system type from which the service is requested.
 *                                         Tag values are defined in DVB SI specification
 * @param   descriptor_tag_extension only valid if delivery_system_descriptor_tag is 0x7F,
 *                                   as with delivery_system_descriptor_tag values are defined in DVB SI specification
 * @param   tune_quietly parameter specified in v3 of tune_broadcast_request
 *                       message from CAM,
 *                       should never be STB_CI_TUNE_QUIETLY_UNKNOWN in this usage
 * @param   keep_app_running parameter specified in v3 of tune_broadcast_request
 *                           message from CAM,
 *                           should never be STB_CI_KEEP_APP_RUNNING_UNKNOWN in this usage
 */
void STB_CITuneTripletRequest(U32BIT module,
                              U16BIT original_network_id,
                              U16BIT transport_stream_id,
                              U16BIT service_id,
                              U8BIT  delivery_system_descriptor_tag,
                              U8BIT  descriptor_tag_extension,
                              E_STB_CI_TUNE_QUIETLY_FLAG tune_quietly,
                              E_STB_CI_KEEP_APP_RUNNING_FLAG keep_app_running)
{
   E_CIP_TUNE_FLAGS flags;
   void *s_ptr;

   FUNCTION_START(STB_CITuneLCNRequest);

   /* ADB database does not provide function(s) that search a restricted list of DSD type */
   USE_UNWANTED_PARAM(delivery_system_descriptor_tag);
   USE_UNWANTED_PARAM(descriptor_tag_extension);

   s_ptr = ADB_FindServiceByIds(original_network_id, transport_stream_id, service_id);
   if (s_ptr != NULL)
   {
      flags = (tune_quietly == STB_CI_TUNE_QUIETLY_SET)? CIP_QUIET_TUNE : CIP_NORMAL_TUNE;
      if (keep_app_running == STB_CI_KEEP_APP_RUNNING_SET)
      {
         flags &= CIP_KEEP_APP_RUNING;
      }
      ACI_HcTuneService(module, s_ptr, flags);
   }
   else
   {
      STB_CITuneReply(module, STB_CI_TUNE_SERVICE_NOT_FOUND);
   }

   FUNCTION_FINISH(STB_CITuneLCNRequest);
}

/**
 * @brief   This function is used by the CI+ stack to request that the
 *          host tunes to a single IP-delivered service described
 *          by a ServiceLocation element.
 *          Following the tune operation (whether successful or not),
 *          the host must call the function STB_CITuneReply().
 *          If tune to IP is not supported STB_CITuneReply() must
 *          be called with STB_CI_TUNE_UNSUPPORTED_SYSTEM
 * @param   module host control module
 * @param   tune_quietly parameter specified in v3 of tune_ip_request
 *                       message from CAM,
 *                       should never be STB_CI_TUNE_QUIETLY_UNKNOWN in this usage
 * @param   keep_app_running parameter specified in v3 of tune_ip_request
 *                           message from CAM,
 *                           should never be STB_CI_KEEP_APP_RUNNING_UNKNOWN in this usage
 * @param   service_location_length the size of the data in service_location_data parameter
 * @param   service_location_data a text string which describes a valid XML description
 *                                containing a single ServiceLocation element conforming
 *                                to the XML schema defined in specification
 */
void STB_CITuneIPRequest(U32BIT module,
                         E_STB_CI_TUNE_QUIETLY_FLAG tune_quietly,
                         E_STB_CI_KEEP_APP_RUNNING_FLAG keep_app_running,
                         U16BIT service_location_length,
                         U8BIT *service_location_data)
{
   FUNCTION_START(STB_CITuneIPRequest);
   USE_UNWANTED_PARAM(tune_quietly);
   USE_UNWANTED_PARAM(keep_app_running);
   USE_UNWANTED_PARAM(service_location_length);
   USE_UNWANTED_PARAM(service_location_data);

   STB_CITuneReply(module, STB_CI_TUNE_UNSUPPORTED_SYSTEM);

   FUNCTION_FINISH(STB_CITuneIPRequest);
}

/*****************************************************************************
 *
 *  DATE AND TIME FUNCTIONS
 *
 ****************************************************************************/

/*!**************************************************************************
 * @brief    Return current date and time
 * @param    mjd - modified Julian date (date part only)
 * @param    hour - hours
 * @param    min - minutes
 * @param    secs - seconds
 * @param    offset - offset from UTC in minutes
 ****************************************************************************/
void STB_CIGetDateTime(U16BIT *mjd, U8BIT *hour, U8BIT *minute, U8BIT *second, S16BIT *offset)
{
   U8BIT offset_hr;
   U8BIT offset_min;
   BOOLEAN offset_neg;

   FUNCTION_START(STB_CIGetDateTime);

   STB_GCGetGMTDateTime(mjd, hour, minute, second);

   STB_GCGetLocalTimeOffset(&offset_hr, &offset_min, &offset_neg);
   *offset = 60 * offset_hr + offset_min;
   if (offset_neg)
   {
      *offset = -(*offset);
   }

   FUNCTION_FINISH(STB_CIGetDateTime);
}

/*****************************************************************************
 *
 *  HOST COUNTRY AND LANGUAGE FUNCTIONS
 *
 ****************************************************************************/

/*!**************************************************************************
 * @brief    Return country code
 * @param    module - module ID
 * @param    code - three letter country code
 ****************************************************************************/
void STB_CIGetHostCountryCode(U32BIT module, U8BIT *code)
{
   U32BIT retval;

   FUNCTION_START(STB_CIGetHostCountryCode);

   /* Store host country & language module */
   ACFG_SetHostCountryLanguageModuleId(module);

   /* Retrieve country code */
   retval = ACFG_GetCountry();

   /* Change lowercase to uppercase */
   retval &= ~0x00202020;

   code[0] = (U8BIT)(retval >> 16);
   code[1] = (U8BIT)(retval >> 8);
   code[2] = (U8BIT)retval;

   FUNCTION_FINISH(STB_CIGetHostCountryCode);
}

/*!**************************************************************************
 * @brief    Return language code
 * @param    module - module ID
 * @param    code - three letter language code
 ****************************************************************************/
void STB_CIGetHostLanguageCode(U32BIT module, U8BIT *code)
{
   U32BIT retval;

   FUNCTION_START(STB_CIGetHostLanguageCode);

   /* Store host country & language module */
   ACFG_SetHostCountryLanguageModuleId(module);

   /* Retrieve language code */
   retval = ACFG_ConvertLangIdToCode(ACFG_GetPrimaryAudioLangId());

   code[0] = (U8BIT)(retval >> 16);
   code[1] = (U8BIT)(retval >> 8);
   code[2] = (U8BIT)retval;

   FUNCTION_FINISH(STB_CIGetHostLanguageCode);
}


/*****************************************************************************
 *
 *  CAM UPGRADE FUNCTIONS
 *
 ****************************************************************************/

/*!**************************************************************************
 * @brief    Handle firmware upgrade notification
 * @param    slot_id - slot ID
 * @param    type - type of firmware upgrade (immediate/delayed)
 * @param    download_time - estimated download time in seconds (0 = unknown)
 * @return   CAM upgrade reply
 ****************************************************************************/
U8BIT STB_CINotifyFirmwareUpgrade(U8BIT slot_id, U8BIT type, U16BIT download_time)
{
   U8BIT reply;

   FUNCTION_START(STB_CINotifyFirmwareUpgrade);
   USE_UNWANTED_PARAM(slot_id);
   USE_UNWANTED_PARAM(type);
   USE_UNWANTED_PARAM(download_time);

   DBG(("STB_CINotifyFirmwareUpgrade"));
   reply = ACI_GetCamUpgradeMode();
   DBG(("reply = 0x%02x", reply));

   FUNCTION_FINISH(STB_CINotifyFirmwareUpgrade);

   return reply;
}

/*!**************************************************************************
 * @brief    Handle firmware upgrade progress notification
 * @param    slot_id - slot ID
 * @param    status - status of progress (percentage in 10% steps)
 ****************************************************************************/
void STB_CINotifyFirmwareUpgradeProgress(U8BIT slot_id, U8BIT status)
{
   FUNCTION_START(STB_CINotifyFirmwareUpgradeProgress);
   USE_UNWANTED_PARAM(slot_id);
   USE_UNWANTED_PARAM(status);

   DBG(("STB_CINotifyFirmwareUpgradeProgress"));
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_PROGRESS, NULL, 0);

   FUNCTION_FINISH(STB_CINotifyFirmwareUpgradeProgress);
}

/*!**************************************************************************
 * @brief    Handle firmware upgrade complete notification
 * @param    module - module ID
 ****************************************************************************/
void STB_CINotifyFirmwareUpgradeComplete(U8BIT slot_id)
{
   FUNCTION_START(STB_CINotifyFirmwareUpgradeComplete);
   USE_UNWANTED_PARAM(slot_id);

   DBG(("STB_CINotifyFirmwareUpgradeComplete"));
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_COMPLETE, NULL, 0);

   FUNCTION_FINISH(STB_CINotifyFirmwareUpgradeComplete);
}

/*!**************************************************************************
 * @brief    Handle firmware upgrade failure notification
 * @param    module - module ID
 ****************************************************************************/
void STB_CINotifyFirmwareUpgradeFailure(U8BIT slot_id)
{
   FUNCTION_START(STB_CINotifyFirmwareUpgradeFailure);
   USE_UNWANTED_PARAM(slot_id);

   DBG(("STB_CINotifyFirmwareUpgradeFailure"));
   STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_CAM_UPGRADE_FAILED, NULL, 0);

   FUNCTION_FINISH(STB_CINotifyFirmwareUpgradeFailure);
}

/*****************************************************************************
 *
 *  OPERATOR PROFILE FUNCTIONS
 *
 ****************************************************************************/

/*****************************************************************************
 *
 * Function Name: STB_CINotifyOperatorStatus
 *
 * Description:   This function is called by the CI+ stack to deliver the
 *                operator status to the host.
 *
 *                This should be called if the CICAM supports the Operator
 *                Profile. This provides the module to be used for other
 *                Operator Profile related calls.
 *
 * Parameters:    module - operator profile module
 *                status - operator status
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyOperatorStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status)
{
   FUNCTION_START(STB_CINotifyOperatorStatus);
   ACI_OpNotifyOperatorStatus(module, status);
   FUNCTION_FINISH(STB_CINotifyOperatorStatus);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyOperatorInfo
 *
 * Description:   This function is called by the CI+ stack to deliver the
 *                operator information to the host.
 *
 * Parameters:    module - operator profile module
 *                info_version - operator information version
 *                info - operator information or NULL (if not valid)
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyOperatorInfo(U32BIT module, U8BIT info_version,
   S_STB_CI_OPERATOR_INFO *info)
{
   FUNCTION_START(STB_CINotifyOperatorInfo);
   ACI_OpNotifyOperatorInfo(module, info_version, info);
   FUNCTION_FINISH(STB_CINotifyOperatorInfo);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyOperatorTune
 *
 * Description:   This function is called by the CI+ stack to request that
 *                the host performs the tune request as described in the
 *                provided descriptors.
 *
 *                The descriptor loop contains system delivery descriptors
 *                which the host must try in order until either an error
 *                condition is raised, the tune operation is successful or
 *                the descriptor loop is exhausted. The operation of the host
 *                should follow the pseudo code in section 14.7.5.11.
 *
 *                When the tune attempt is done, the host must call the
 *                function CIP_SendOperatorTuneStatus with the
 *                results of the tune attempt.
 *
 *                This function may be called during a profile search sequence.
 *
 * Parameters:    module - operator profile module
 *                desc_loop_len - length of descriptor loop in bytes
 *                desc_loop - the descriptor loop
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyOperatorTune(U32BIT module, U16BIT desc_loop_len, U8BIT *desc_loop)
{

   FUNCTION_START(STB_CINotifyOperatorTune);
   ACI_OpNotifyOperatorTune(module, desc_loop_len, desc_loop);
   FUNCTION_FINISH(STB_CINotifyOperatorTune);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyOperatorSearchStatus
 *
 * Description:   This function is called by the CI+ stack to deliver the
 *                operator search status to the host.
 *
 *                This function will be called at the end of a profile search
 *                sequence.
 *
 * Parameters:    module - operator profile module
 *                status - operator status
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyOperatorSearchStatus(U32BIT module, S_STB_CI_OPERATOR_STATUS *status)
{
   FUNCTION_START(STB_CINotifyOperatorSearchStatus);
   ACI_OpNotifyOperatorSearchStatus(module, status);
   FUNCTION_FINISH(STB_CINotifyOperatorSearchStatus);
}

/*****************************************************************************
 *
 * Function Name: STB_CINotifyOperatorNit
 *
 * Description:   This function is called by the CI+ stack to deliver the
 *                operator NIT to the host.
 *
 *                The CICAM delivers zero or more NIT sections to the host.
 *                Each section is a complete NIT section including the
 *                CRC-32 field, and the sections are provided without any
 *                padding between them.
 *
 *                The NIT section data must be copied if it used after this
 *                function returns.
 *
 * Parameters:    module - operator profile module
 *                nit_loop_length - length of NIT loop in bytes (may be 0)
 *                nit_sections - NIT section(s), if any
 *
 * Returns:       Nothing
 *
 ****************************************************************************/
void STB_CINotifyOperatorNit(U32BIT module, U16BIT nit_loop_length,
   U8BIT *nit_sections)
{
   FUNCTION_START(STB_CINotifyOperatorNit);
   ACI_OpNotifyOperatorNit(module, nit_loop_length, nit_sections);
   FUNCTION_FINISH(STB_CINotifyOperatorNit);
}

/**
 * @brief   This function is called by the CI+ stack to deliver the
 *          OSDT to the host.
 *          This notification should only occur after STB_CISendOperatorOSDTRequest()
 *          has been called.
 *          The OSD data must be copied if it used after this
 *          function returns.
 *
 * @note    This message is only supported for version 2 of the Operator Profile resource
 * @param   module operator profile module
 * @param   osdt_length length of OSDT in bytes (may be 0)
 * @param   osdt the data, if any
 */
void STB_CINotifyOperatorOSDTReply(U32BIT module, U32BIT osdt_length, U8BIT *osdt)
{
   FUNCTION_START(STB_CINotifyOperatorOSDTReply);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(osdt_length);
   USE_UNWANTED_PARAM(osdt);
   /* Implementation required when DVB supports IP delivered services */
   FUNCTION_FINISH(STB_CINotifyOperatorOSDTReply);
}

#ifdef CI_AUTH_RES
/**
 *  AUTHENTICATION FUNCTIONS
 */

/**
 * @brief   This function is called by the CI stack to send an
 *          authentication request from the module.
 * @note    This function is related to the Authentication resource
 *          and not to the Content Control resource.
 * @param   module authentication module
 * @param   auth_protocol_id authenticiation protocol ID
 * @param   auth_req authentication request
 * @param   auth_req_len length of authentication request in bytes
 */
void STB_CINotifyAuthRequest(U32BIT module, U16BIT auth_protocol_id,
   U8BIT *auth_req, U32BIT auth_req_len)
{
   FUNCTION_START(STB_CINotifyAuthRequest);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(auth_protocol_id);
   USE_UNWANTED_PARAM(auth_req);
   USE_UNWANTED_PARAM(auth_req_len);
   FUNCTION_FINISH(STB_CINotifyAuthRequest);
}
#endif  /* CI_AUTH_RES */

#ifdef CI_SAS_RES
/**
 *  SPECIFIC APPLICATION SUPPORT (SAS) FUNCTIONS
 */

/**
 * @brief   This function is used by the CI+ stack to notify the host
 *          that a session with the SAS resource has been opened by
 *          the module.
 *          Following this call, the host can call
 *          STB_CIRequestSasConnection to request a connection to the
 *          application part on the module.
 * @param   module SAS module
 */
void STB_CINotifySasSession(U32BIT module)
{
   FUNCTION_START(STB_CINotifySasSession);
   USE_UNWANTED_PARAM(module);
   FUNCTION_FINISH(STB_CINotifySasSession);
}

/**
 * @brief   This function is used by the CI+ stack to notify the host
 *          that a session with the SAS resource has been closed.
 *          Even after this call, there should be at least one open
 *          session to allow the host to request another connection.
 *          The host must not close the last open session to the SAS
 *          resource, otherwise it will not be able to open further
 *          connections to the application part on the module.
 * @param   module SAS module
 */
void STB_CINotifySasSessionClosed(U32BIT module)
{
   FUNCTION_START(STB_CINotifySasSessionClosed);
   USE_UNWANTED_PARAM(module);
   FUNCTION_FINISH(STB_CINotifySasSessionClosed);
}

/**
 * @brief   This function is called by the CI+ stack to confirm a
 *          connection request from the host.
 *          If the connection was been created successfully, the module
 *          should also open another session to the SAS resource, in
 *          which case the CI+ stack will call STB_CINotifySasSession.
 * @param   module SAS module
 * @param   app_id the private host application ID (64 bits)
 * @param   status session status:
 *          STB_CI_SAS_CONNECTION_ESTABLISHED
 *          STB_CI_SAS_CONNECTION_NO_APPLICATION
 *          STB_CI_SAS_CONNECTION_NOT_AVAILABLE
 * @note    The value of OIPF_APPLICATION_ID is 0x0108113101190000
 */
void STB_CIConfirmSasConnection(U32BIT module, U8BIT *app_id, U8BIT status)
{
   FUNCTION_START(STB_CIConfirmSasConnection);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(app_id);
   USE_UNWANTED_PARAM(status);
   FUNCTION_FINISH(STB_CIConfirmSasConnection);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a reply to
 *          a "send_msg" message to the host.
 *          The function delivers the "reply_msg" message. This message
 *          is described in the OIPF CSP specification as part of
 *          the HNI-CSP interface. It is mapped to the DAE API
 *          sendDRMMessage.
 *          The function should be called in response to a call to
 *          STB_CIOipfSendMsg by the host. The transaction ID should
 *          match the ID that was sent to the module through that
 *          function.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   transaction_id transaction ID
 * @param   status status code
 * @param   ca_info CA vendor specific information
 * @param   ca_info_len length of information in bytes (max=65000)
 * @note    This is an OIPF-specific function.
 * @note    The CA vendor specific information is optional in this
 *          function. If the module does not provide this data,
 *          then info=NULL and info_len=0.
 */
void STB_CINotifyOipfReplyMsg(U32BIT module, U16BIT ca_system_id,
   U32BIT transaction_id, U8BIT status,
   U8BIT *ca_info, U16BIT ca_info_len)
{
   FUNCTION_START(STB_CINotifyOipfReplyMsg);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(transaction_id);
   USE_UNWANTED_PARAM(status);
   USE_UNWANTED_PARAM(ca_info);
   USE_UNWANTED_PARAM(ca_info_len);
   FUNCTION_FINISH(STB_CINotifyOipfReplyMsg);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a
 *          "parental_control_info" message to the host.
 *          This message is described in the OIPF CSP specification
 *          as part of the HNI-CSP interface. It is mapped to the DAE API
 *          onParentalRatingChange and onParentalRatingError.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   info parental control information
 * @note    This is an OIPF-specific function.
 * @note    info->country_codes contains a sequence of pairs of letters
 *          describing the country codes using ISO 3166 alpha-2.
 *          If this datatype is not included in the message then
 *          info->country_codes will be NULL.
 * @note    info->parental_control_url is optional and may be NULL.
 */
void STB_CINotifyOipfParentalControlInfo(U32BIT module, U16BIT ca_system_id,
   S_STB_CI_OIPF_PARENTAL_CONTROL_INFO *info)
{
   FUNCTION_START(STB_CINotifyOipfParentalControlInfo);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(info);
   FUNCTION_FINISH(STB_CINotifyOipfParentalControlInfo);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a
 *          "rights_info" message to the host.
 *          This message is described in the OIPF CSP specification
 *          as part of the HNI-CSP interface. It is mapped to the DAE API
 *          onDRMRightsError.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   access_status 0=not descrambled (no access), 1=descrambled
 * @param   rights_issuer_url URL for connecting to the service provider
 * @param   url_len length of rights_issuer_url in bytes
 * @note    This is an OIPF-specific function.
 * @note    rights_issuer_url is optional and may be NULL.
 */
void STB_CINotifyOipfRightsInfo(U32BIT module, U16BIT ca_system_id,
   U8BIT access_status, U8BIT *rights_issuer_url,
   U16BIT url_len)
{
   FUNCTION_START(STB_CINotifyOipfRightsInfo);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(access_status);
   USE_UNWANTED_PARAM(rights_issuer_url);
   USE_UNWANTED_PARAM(url_len);
   FUNCTION_FINISH(STB_CINotifyOipfRightsInfo);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a
 *          "system_info" message to the host.
 *          This message is described in the OIPF CSP specification
 *          as part of the HNI-CSP interface. It is mapped to the DAE API
 *          onDRMSystemMessage.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   ca_info CA vendor specific information
 * @param   ca_info_len length of information in bytes (max=65000)
 * @note    This is an OIPF-specific function.
 */
void STB_CINotifyOipfSystemInfo(U32BIT module, U16BIT ca_system_id,
   U8BIT *ca_info, U16BIT ca_info_len)
{
   FUNCTION_START(STB_CINotifyOipfSystemInfo);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(ca_info);
   USE_UNWANTED_PARAM(ca_info_len);
   FUNCTION_FINISH(STB_CINotifyOipfSystemInfo);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a
 *          "can_play_content_reply" message to the host.
 *          This message is described in the OIPF CSP specification
 *          as part of the HNI-CSP interface.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   can_play_status - TRUE(1) can play, FALSE(0) = cannot
 * @note    This is an OIPF-specific function.
 */
void STB_CINotifyOipfCanPlayStatus(U32BIT module, U16BIT ca_system_id,
   BOOLEAN can_play_status)
{
   FUNCTION_START(STB_CINotifyOipfCanPlayStatus);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(can_play_status);
   FUNCTION_FINISH(STB_CINotifyOipfCanPlayStatus);
}

/**
 * @brief   This function is used by the CI+ stack to deliver a
 *          "can_record_content_reply" message to the host.
 *          This message is described in the OIPF CSP specification
 *          as part of the HNI-CSP interface.
 * @param   module SAS module
 * @param   ca_system_id CA system ID
 * @param   can_record_status TRUE(1) = can record, FALSE(0) = cannot
 * @note    This is an OIPF-specific function.
 */
void STB_CINotifyOipfCanRecordStatus(U32BIT module, U16BIT ca_system_id,
   BOOLEAN can_record_status)
{
   FUNCTION_START(STB_CINotifyOipfCanRecordStatus);
   USE_UNWANTED_PARAM(module);
   USE_UNWANTED_PARAM(ca_system_id);
   USE_UNWANTED_PARAM(can_record_status);
   FUNCTION_FINISH(STB_CINotifyOipfCanRecordStatus);
}

#endif /*CI_SAS_RES*/


/**
 * @brief   This function is called by the CI+ stack to find out the
 *          CSUV value when requested by the CAM.
 * @return  Critical Security Update Version
 */
U8BIT STB_CIGetCSUV(void)
{
   FUNCTION_START(STB_CIGetCSUV);
   FUNCTION_FINISH(STB_CIGetCSUV);
   return 0;
}

/**
 * @brief   This function is called by the CI stack when the CAM has requested
 *          host diagnostic screen to be displayed or to stop displaying.
 *          The host should respond with a STB_CIHDSConfirm() call, even if it is
 *          to indicate that the host does not implement a diagnostic screen.
 * @param   module specifies module required
 * @param   display_diagnostic_screen TRUE if requested to display the diagnostic screen,
 *                                    FALSE if requested to stop displaying the diagnostic screen
 */
void STB_CINotifyHDSRequest(U32BIT module, BOOLEAN display_diagnostic_screen)
{
   FUNCTION_START(STB_CINotifyHDSRequest);

   if (display_diagnostic_screen)
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_DIAGNOSTIC_SCREEN_SHOW, &module, sizeof(U32BIT));
   }
   else
   {
      STB_ERSendEvent(FALSE, FALSE, EV_CLASS_CI, EV_TYPE_CI_DIAGNOSTIC_SCREEN_HIDE, &module, sizeof(U32BIT));
   }

   FUNCTION_FINISH(STB_CINotifyHDSRequest);
}

/*---local function definitions----------------------------------------------*/

/****************************************************************************
** End of file
*****************************************************************************/
