/*******************************************************************************
 * Copyright  2017 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright  2011 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief   Application level HBBTV callback functions
 * @file    hbbtv_api.h
 * @date    December 2017
 */

// pre-processor mechanism so multiple inclusions don't cause compilation error
#ifndef _HBBTV_API_H
#define _HBBTV_API_H

#include "techtype.h"

//---Constant and macro definitions for public use-----------------------------

//---Enumerations for public use-----------------------------------------------

typedef enum
{
   HBBTV_CHANNEL_CHANGE_SUCCEEDED,
   HBBTV_CHANNEL_CHANGE_WRONG_TUNER,
   HBBTV_CHANNEL_CHANGE_NO_SIGNAL,
   HBBTV_CHANNEL_CHANGE_TUNER_IN_USE,
   HBBTV_CHANNEL_CHANGE_PARENTAL_LOCKED,
   HBBTV_CHANNEL_CHANGE_ENCRYPTED,
   HBBTV_CHANNEL_CHANGE_UNKNOWN_CHANNEL,
   HBBTV_CHANNEL_CHANGE_INTERRUPTED,
   HBBTV_CHANNEL_CHANGE_RECORDING_IN_PROGRESS,
   HBBTV_CHANNEL_CHANGE_INSUFFICIENT_RESOURCES,
   HBBTV_CHANNEL_CHANGE_CHANNEL_NOT_IN_TS,
   HBBTV_CHANNEL_CHANGE_UNKNOWN_ERROR
} E_HBBTV_CHANNEL_CHANGE_STATUS;

typedef enum
{
   HBBTV_RECORDING_NEWLY_SCHEDULED,
   HBBTV_RECORDING_ABOUT_TO_START,
   HBBTV_RECORDING_ACQUIRING_RESOURCES,
   HBBTV_RECORDING_STARTED,
   HBBTV_RECORDING_UPDATED,
   HBBTV_RECORDING_COMPLETED,
   HBBTV_RECORDING_PART_COMPLETED,
   HBBTV_RECORDING_FAILED_RESOURCE_LIMITATION,
   HBBTV_RECORDING_FAILED_INSUFFICIENT_STORAGE,
   HBBTV_RECORDING_FAILED_TUNER_CONFLICT,
   HBBTV_RECORDING_FAILED_NOT_ALLOWED,
   HBBTV_RECORDING_FAILED_UNKNOWN_ERROR,
   HBBTV_RECORDING_DELETED,
   HBBTV_RECORDING_TS_STARTING,           /* Timeshift related events */
   HBBTV_RECORDING_TS_STARTED,
   HBBTV_RECORDING_TS_STOPPED,
   HBBTV_RECORDING_TS_NOT_POSSIBLE,
   HBBTV_RECORDING_TS_NOT_ALLOWED,
   HBBTV_RECORDING_TS_FAILED
} E_HBBTV_RECORDING_EVENT;

//---Global type defs for public use-------------------------------------------

//---Global Function prototypes for public use---------------------------------

/**
 * @brief   Requests the HbbTV engine to process the specified AIT. The HbbTV engine expects the
 *          relevant AITs only (the first one after HBBTV_Start and when the version changes).
 *          If more than one stream is signalled in the PMT for a service with an
 *          application_signalling_descriptor, then the application_signalling_descriptor for the
 *          stream containing the AIT for the HbbTV application shall include the HbbTV
 *          application_type (0x0010).
 * @param   service_id Service ID he AIT refers to
 * @param   data pointer to the buffer containing the AIT row data. It can be released as soon as
 *          the function returns.
 * @param   nbytes number of bytes contained in data
 */
void HBBTV_ProcessAitSection(U16BIT service_id, U8BIT *data, U32BIT nbytes);

/**
 * @brief   Notifies the HbbTV ending that it must update Service list. This will cause the engine
 *          to subsequently call HBBTV_ObtainServiceList().
 */
void HBBTV_NotifyServiceListChange(void);

/**
 * @brief   Notifies the HbbTV engine that the parental rating of the currently playing service has
 *          changed.
 * @param   blocked TRUE if the current service is blocked by the parental control system.
 */
void HBBTV_NotifyParentalRatingChange(BOOLEAN blocked);

/**
 * @brief   Notifies the HbbTV engine of a change in the status of the service identified by the
 *          specified DVB triplet.
 * @param   onet_id Original Network ID
 * @param   trans_id Transport Stream ID
 * @param   serv_id Service ID
 * @param   status New status
 */
void HBBTV_NotifyChannelChangeStatus(U16BIT onet_id, U16BIT trans_id, U16BIT serv_id,
   E_HBBTV_CHANNEL_CHANGE_STATUS status);

/**
 * @brief   Notifies the HbbTV engine that the present/following events have changed on the current
 *          service.
 */
void HBBTV_NotifyProgrammeChanged(void);

/**
 * @brief   Notifies the HbbTV engine of a recording event
 * @param   id scheduled or completed recording ID
 * @param   event recording event
 */
void HBBTV_NotifyRecordingEvent(U32BIT id, E_HBBTV_RECORDING_EVENT event);

/**
 * @brief   Notify the video broadcast object that the timeshift play speed has changed
 * @param   speed speed represented as a signed percentage value, with 100 as normal speed
 */
void HBBTV_NotifyPlaySpeedChanged(S16BIT speed);

#endif
