/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 * 
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      System Interface, Configuration
 * @file       conf.c
 * @date       January 2014
 * @author     Sergio Panseri
 */
 
/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwtun.h"
#include "stbheap.h"
#include "stbdpc.h"
#include "stbsitab.h"
#include "ap_cfg.h"
#include "ap_cntrl.h"

#include "hbbtv_sif_types.h"
#include "hbbtv_sif_conf.h"


/*---constant definitions for this file--------------------------------------*/

#define SYSTEM_FLAG_DVBT   1
#define SYSTEM_FLAG_DVBT2  2
#define SYSTEM_FLAG_DVBS   4
#define SYSTEM_FLAG_DVBS2  8
#define SYSTEM_FLAG_DVBC   16


/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

static U8BIT ConvertToStringLang(U8BIT lang_id1, U8BIT lang_id2, U8BIT **string);


/*---local function definitions----------------------------------------------*/

static U8BIT ConvertToStringLang(U8BIT lang_id1, U8BIT lang_id2, U8BIT **string)
{
   U8BIT *ptr = NULL;
   U32BIT lang1, lang2;
   U8BIT len;

   len = 0;
   if (lang_id1 != ACFG_INVALID_LANG)
   {
      lang1 = ACFG_ConvertLangIdToCode(lang_id1);
      len = 4;
   }
   if (lang_id2 != ACFG_INVALID_LANG)
   {
      lang2 = ACFG_ConvertLangIdToCode(lang_id2);
      len += 4;
   }
   if (len > 0)
   {
      ptr = STB_AppGetMemory(len);
      if (ptr != NULL)
      {
         ptr[0] = ((U8BIT *)(&lang1))[2];
         ptr[1] = ((U8BIT *)(&lang1))[1];
         ptr[2] = ((U8BIT *)(&lang1))[0];
         ptr[3] = 0;
         if (len > 4)
         {
            ptr[3] = ',';
            ptr[4] = ((U8BIT *)(&lang2))[2];
            ptr[5] = ((U8BIT *)(&lang2))[1];
            ptr[6] = ((U8BIT *)(&lang2))[0];
            ptr[7] = 0;
         }
      }
   }
   *string = ptr;

   return len;
}


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Retrieves a string containing a comma-separated set of languages to be used for audio 
 *          playback, in order of preference. The returned string must be freed with 
 *          HBBTV_ReleaseLangString.
 * @param   lang_string Pointer to the returned string 
 * @param   size Pointer to the number of bytes in the 
 *          returned string
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_ALLOCATING_MEMORYThe string couldn't be allocated
 */
E_HBBTV_ERR HBBTV_GetPrefAudioLang(U8BIT **lang_string, U32BIT *size)
{
   E_HBBTV_ERR result;
   U8BIT id;

   FUNCTION_START(HBBTV_GetPrefAudioLang);

   id = ACFG_GetSecondaryAudioLangId();
   if (id == ACFG_INVALID_LANG)
   {
      id = ACFG_GetDefaultSecondaryLangId();
   }

   *size = ConvertToStringLang(ACFG_GetPrimaryAudioLangId(), id, lang_string);
   if (*size > 0)
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_OTHER;
   }

   FUNCTION_FINISH(HBBTV_GetPrefAudioLang);

   return result;
}

/**
 * @brief   Retrieves a string containing a comma-separated set of languages to be used for 
 *          subtitles, in order of preference. The returned string must be freed with 
 *          HBBTV_ReleaseLangString.
 * @param   lang_string Pointer to the returned string
 * @param   size Pointer to the number of bytes in the returned string
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_ALLOCATING_MEMORYThe string couldn't be allocated
 */
E_HBBTV_ERR HBBTV_GetPrefSubtLang(U8BIT **lang_string, U32BIT *size)
{
   E_HBBTV_ERR result;
   U8BIT id;

   FUNCTION_START(HBBTV_GetPrefSubtLang);

   id = ACFG_GetSecondaryTextLangId();
   if (id == ACFG_INVALID_LANG)
   {
      id = ACFG_GetDefaultSecondaryLangId();
   }

   *size = ConvertToStringLang(ACFG_GetPrimaryTextLangId(), id, lang_string);
   if (*size > 0)
   {
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_OTHER;
   }

   FUNCTION_FINISH(HBBTV_GetPrefSubtLang);

   return result;
}

/**
 * @brief   Retrieves a string containing the three character string representing country code
 *          identifying the country in which the receiver is deployed. The returned string must be 
 *          freed with HBBTV_ReleaseLangString.
 * @param   lang_string Pointer to the returned string
 * @param   size Pointer to the number of bytes in the returned string
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_ALLOCATING_MEMORYThe string couldn't be allocated
 */
E_HBBTV_ERR HBBTV_GetPrefCountryId(U8BIT **lang_string, U32BIT *size)
{
   E_HBBTV_ERR result;
   U32BIT lang_code;

   FUNCTION_START(HBBTV_GetPrefCountryId);

   *lang_string = STB_AppGetMemory(4);
   if (*lang_string == NULL)
   {
      *size = 0;
      result = HBBTV_ERR_OTHER;
   }
   else
   {
      lang_code = ACFG_GetCountry();
      (*lang_string)[0] = (lang_code >> 16) & 0xFF;
      (*lang_string)[1] = (lang_code >> 8) & 0xFF;
      (*lang_string)[2] = lang_code & 0xFF;
      (*lang_string)[3] = 0;
      *size = 4;
      result = HBBTV_OK;
   }

   FUNCTION_FINISH(HBBTV_GetPrefCountryId);

   return result;
}

/**
 * @brief   Frees the string obtained from HBBTV_GetPrefAudioLang, HBBTV_GetPrefSubtLang, 
 *          HBBTV_GetPrefUiLang or HBBTV_GetPrefCountryId.
 * @param   lang_string Pointer to string to be freed
 */
void HBBTV_ReleaseLangString(U8BIT *lang_string)
{
   FUNCTION_START(HBBTV_ReleaseLangString);

   STB_AppFreeMemory(lang_string);

   FUNCTION_FINISH(HBBTV_ReleaseLangString);
}

/**
 * @brief   Retrieves the string describing the supported Broadcast Delivery Systems
 *          (DVB_S, DVB_C, DVB_T, DVB_C2, DVB_T2 or DVB_S2) as defined in Section
 *          9.2, Table 15, under "UI Profile Name Fragment". the HbbTV engine will call
 *          HBBTV_ReleaseDeliverySystemsString when the string is not needed any 
 *          more.
 * @return  Pointer to an allocated buffer containing a string describing the list of supported 
 *          broadcast deliverys systems (e.g. "+DVB_T2+DVB_T+DVB_C", note that the first '+' 
 *          character is needed). The buffer will be freed by the HbbTV engine using 
 *          HBBTV_ReleaseDeliverySystemsString.
 */
U8BIT *HBBTV_GetDeliverySystemsString(void)
{
   U8BIT num_tuners, i;
   E_STB_TUNE_SYSTEM_TYPE system;
   U8BIT flags;
   U32BIT length;
   U8BIT *str_ptr, *systems;

   FUNCTION_START(HBBTV_GetDeliverySystemsString);

   flags = 0;
   length = 0;
   num_tuners = STB_HWGetTunerPaths();
   for (i = 0; i < num_tuners; i++)
   {
      system = STB_TuneGetSupportedSystemType(i);
      switch (system)
      {
         case TUNE_SYSTEM_TYPE_DVBT2:
            flags |= SYSTEM_FLAG_DVBT2;
            length += 7; /* "+DVB_T2" */
            /* Intentionally falling through to the next case */
         case TUNE_SYSTEM_TYPE_DVBT:
            flags |= SYSTEM_FLAG_DVBT;
            length += 6; /* "+DVB_T" */
            break;
         case TUNE_SYSTEM_TYPE_DVBS2:
            flags |= SYSTEM_FLAG_DVBS2;
            length += 7; /* "+DVB_S2" */
            /* Intentionally falling through to the next case */
         case TUNE_SYSTEM_TYPE_DVBS:
            flags |= SYSTEM_FLAG_DVBS;
            length += 6; /* "+DVB_S" */
            break;
         case TUNE_SYSTEM_TYPE_DVBC:
            flags |= SYSTEM_FLAG_DVBC;
            length += 6; /* "+DVB_C" */
            break;
         default:
            break;
      }
   }

   str_ptr = STB_AppGetMemory(length + 1);
   systems = str_ptr;
   if (str_ptr != NULL)
   {
      str_ptr[0] = 0; /* Termination, just in case... */

      if ((flags & SYSTEM_FLAG_DVBT2) != 0)
      {
         strcpy((char *)str_ptr, "+DVB_T2");
         str_ptr += 7;
      }
      if ((flags & SYSTEM_FLAG_DVBT) != 0)
      {
         strcpy((char *)str_ptr, "+DVB_T");
         str_ptr += 6;
      }
      if ((flags & SYSTEM_FLAG_DVBS2) != 0)
      {
         strcpy((char *)str_ptr, "+DVB_S2");
         str_ptr += 7;
      }
      if ((flags & SYSTEM_FLAG_DVBS) != 0)
      {
         strcpy((char *)str_ptr, "+DVB_S");
         str_ptr += 6;
      }
      if ((flags & SYSTEM_FLAG_DVBC) != 0)
      {
         strcpy((char *)str_ptr, "+DVB_C");
         str_ptr += 6; /* Not actually needed, but left it to extend the list easily if needed */
      }
   }

   FUNCTION_FINISH(HBBTV_GetDeliverySystemsString);

   return systems;
}

/**
 * @brief   Frees the Broadacast Delivery Systems string, returned by HBBTV_GetDeliverySystemsString.
 * @param   systems Pointer to string to be freed
 */
void HBBTV_ReleaseDeliverySystemsString(U8BIT *systems)
{
   FUNCTION_START(HBBTV_ReleaseDeliverySystemsString);

   STB_AppFreeMemory(systems);

   FUNCTION_FINISH(HBBTV_ReleaseDeliverySystemsString);
}

/**
 * @brief   Query whether parental control is enabled and the selected age, if applicable. The age 
 *          is the actual age, so for DVB the valid values are 4 to 18, inclusive.
 * @param   enabled Used to return a boolean defining whether parental control is enabled (TRUE), or
 *          not (FALSE).
 * @param   age Pointer to returned age value. Should be returned as 0 if parental control is 
 *          disabled or age isn't set
 */
void HBBTV_GetParentalControlSettings(BOOLEAN* enabled, U8BIT* age)
{
   FUNCTION_START(HBBTV_GetParentalControlSettings);
   
   *enabled = ACTL_ParentalControlEnabled();
   *age = ACTL_GetParentalControlAge();

   FUNCTION_FINISH(HBBTV_GetParentalControlSettings);
}

