/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 *
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 *
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      System Interface, Decoding
 * @file       decoding.c
 * @date       January 2014
 * @author     Sergio Panseri
 */

//#define DEBUG

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbdpc.h"
#include "stbvtc.h"
#include "ap_dbacc.h"
#include "ap_cntrl.h"

#include "hbbtv_sif_types.h"
#include "hbbtv_sif_decoding.h"

/*---constant definitions for this file--------------------------------------*/
#ifdef DEBUG
#define DBG(string, vars ...) STB_SPDebugWrite("[%s:%d] "string, __FUNCTION__, __LINE__, ##vars)
#else
#define DBG(string, vars ...)
#endif

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

static void SetVideo(U8BIT path, U16BIT pcr_pid, U16BIT video_pid, E_STB_DP_VIDEO_CODEC codec);
static void SetAudio(U8BIT path, U16BIT pcr_pid, U16BIT audio_pid, E_STB_DP_AUDIO_CODEC codec);
static void GetDefaultVideo(void *service, U16BIT *pid, E_STB_DP_VIDEO_CODEC *codec);
static void GetDefaultAudio(void *service, U16BIT *pid, E_STB_DP_AUDIO_CODEC *codec);
static BOOLEAN GetRequestedVideoStream(void *service, S16BIT tag, U16BIT *pid, E_STB_DP_VIDEO_CODEC *codec);
static BOOLEAN GetRequestedAudioStream(void *service, S16BIT tag, U16BIT *pid, E_STB_DP_AUDIO_CODEC *codec);


/*---local function definitions----------------------------------------------*/

static void SetVideo(U8BIT path, U16BIT pcr_pid, U16BIT video_pid, E_STB_DP_VIDEO_CODEC codec)
{
   FUNCTION_START(SetVideo);

   if ((video_pid != STB_DPGetVideoPID(path)) || (STB_DPGetVideoStatus(path) != DECODE_RUNNING))
   {
      STB_DPStopVideoDecoding(path);
      STB_DPSetVideoCodec(path, codec);
      STB_DPSetVideoPID(path, video_pid);
      STB_DPSetPCRPID(path, pcr_pid);
   }
#ifdef DEBUG
   else
   {
      DBG("Same video PID %d", video_pid);
   }
#endif
   STB_DPStartVideoDecoding(path);

   FUNCTION_FINISH(SetVideo);
}

static void SetAudio(U8BIT path, U16BIT pcr_pid, U16BIT audio_pid, E_STB_DP_AUDIO_CODEC codec)
{
   FUNCTION_START(SetAudio);

   if ((audio_pid != STB_DPGetAudioPID(path)) || (STB_DPGetAudioStatus(path) != DECODE_RUNNING))
   {
      STB_DPStopAudioDecoding(path);
      STB_DPSetAudioCodec(path, codec);
      STB_DPSetAudioPID(path, audio_pid);
      STB_DPSetPCRPID(path, pcr_pid);
   }
#ifdef DEBUG
   else
   {
      DBG("Same audio PID %d", audio_pid);
   }
#endif
   STB_DPStartAudioDecoding(path);

   FUNCTION_FINISH(SetAudio);
}

static void GetDefaultVideo(void *service, U16BIT *pid, E_STB_DP_VIDEO_CODEC *codec)
{
   FUNCTION_START(GetDefaultVideo);

   *pid = ADB_GetServiceVideoPid(service);
   *codec = ADB_GetVideoCodecFromStream(ADB_GetServiceVideoType(service));

   /* H265 isn't defined as a supported format in the OIPFs specs */
   if(*codec == VIDEO_CODEC_H265)
   {
     *codec = VIDEO_CODEC_AUTO;
   }

   FUNCTION_FINISH(GetDefaultVideo);
}

static void GetDefaultAudio(void *service, U16BIT *pid, E_STB_DP_AUDIO_CODEC *codec)
{
   FUNCTION_START(GetDefaultAudio);

   *pid = ADB_GetServiceAudioPid(service);
   *codec = ADB_GetAudioCodecFromStream(ADB_GetServiceAudioType(service));

   FUNCTION_FINISH(GetDefaultAudio);
}

static BOOLEAN GetRequestedVideoStream(void *service, S16BIT tag, U16BIT *pid, E_STB_DP_VIDEO_CODEC *codec)
{
   BOOLEAN result = FALSE;
   void **stream_list;
   U16BIT num, i;
   U8BIT num_tags, j;

   FUNCTION_START(GetRequestedVideoStream);

   *pid = 0;
   *codec = VIDEO_CODEC_AUTO;
   ADB_GetStreamList(service, ADB_VIDEO_LIST_STREAM, &stream_list, &num);
   if ((num > 0) && (stream_list != NULL))
   {
      for (i = 0; (i < num) && (*pid == 0); i++)
      {
         num_tags = ADB_GetStreamNumTags(stream_list[i]);
         for (j = 0; j < num_tags; j++)
         {
            if (tag == ADB_GetStreamTag(stream_list[i], j))
            {
               *pid = ADB_GetStreamPID(stream_list[i]);
               *codec = ADB_GetVideoCodecFromStream(ADB_GetStreamType(stream_list[i]));
               /* H265 isn't defined as a supported format in the OIPFs specs */
               if(*codec == VIDEO_CODEC_H265)
               {
                 *codec = VIDEO_CODEC_AUTO;
               }
               result = TRUE;
               break;
            }
         }
      }
      ADB_ReleaseStreamList(stream_list, num);
   }

   FUNCTION_FINISH(GetRequestedVideoStream);

   return result;
}

static BOOLEAN GetRequestedAudioStream(void *service, S16BIT tag, U16BIT *pid, E_STB_DP_AUDIO_CODEC *codec)
{
   BOOLEAN result = FALSE;
   void **stream_list;
   U16BIT num, i;
   U8BIT num_tags, j;

   FUNCTION_START(GetRequestedAudioStream);

   *pid = 0;
   *codec = AUDIO_CODEC_AUTO;
   ADB_GetStreamList(service, ADB_AUDIO_LIST_STREAM, &stream_list, &num);
   if ((num > 0) && (stream_list != NULL))
   {
      for (i = 0; (i < num) && (*pid == 0); i++)
      {
         num_tags = ADB_GetStreamNumTags(stream_list[i]);
         for (j = 0; j < num_tags; j++)
         {
            if (tag == ADB_GetStreamTag(stream_list[i], j))
            {
               *pid = ADB_GetStreamPID(stream_list[i]);
               *codec = ADB_GetAudioCodecFromStream(ADB_GetStreamType(stream_list[i]));
               result = TRUE;
               break;
            }
         }
      }
      ADB_ReleaseStreamList(stream_list, num);
   }

   FUNCTION_FINISH(GetRequestedAudioStream);

   return result;
}


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Function used to notify the controlling application whether the HbbTV engine is taking
 *          control of the A/V presentation.
 * @param   control TRUE when the HbbTV engine is in control of the A/V, FALSE otherwise.
 */
void HBBTV_ControlDVBPresentation(BOOLEAN control)
{
   FUNCTION_START(HBBTV_ControlDVBPresentation);

   DBG("control = %s", control ? "TRUE" : "FALSE");
   if (control)
   {
      STB_VTSetHbbtvEnable(TRUE);
   }
   else
   {
      STB_VTSetHbbtvVideoWindow(NULL);
      STB_VTSetHbbtvEnable(FALSE);
   }

   FUNCTION_FINISH(HBBTV_ControlDVBPresentation);
}

/**
 * @brief   Instructs the controlling application to present the audio specified
 *          by its component tag.
 * @param   component_tag Component tag of the audio to be presented. If the component_tag value is
 *          -1, the default audio elementary stream must be used.
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_PresentDVBAudio(S16BIT component_tag)
{
   E_HBBTV_ERR result;
   U8BIT live_path;
   void *service;
   U16BIT requested_pid, pcr;
   E_STB_DP_AUDIO_CODEC requested_codec;

   FUNCTION_START(HBBTV_PresentDVBAudio);

   DBG("component_tag=%d", component_tag);
   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      service = STB_DPGetTunedService(live_path);
   }
   else
   {
      service = NULL;
   }

   if (service != NULL)
   {
      pcr = ADB_GetServicePCRPid(service);
      if (component_tag == -1)
      {
         GetDefaultAudio(service, &requested_pid, &requested_codec);
      }
      else
      {
         if (!GetRequestedAudioStream(service, component_tag, &requested_pid, &requested_codec))
         {
            requested_pid = 0;
         }
      }
      if (requested_pid != 0)
      {
         SetAudio(live_path, pcr, requested_pid, requested_codec);
         result = HBBTV_OK;
      }
      else
      {
         /* Stream could not be found */
         result = HBBTV_ERR_BAD_PARAMETER;
         DBG("Stream with tag %d not found", component_tag);
      }
   }
   else
   {
      /* TODO: service might be not known yet (e.g. TuneDSD), should we retry later? */
      result = HBBTV_ERR_OTHER;
      DBG("Current service is NULL");
   }

   FUNCTION_FINISH(HBBTV_PresentDVBAudio);

   return result;
}

/**
 * @brief   Instructs the controlling application to stop the audio presentation
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_StopDVBAudio(void)
{
   E_HBBTV_ERR result;
   U8BIT live_path;

   FUNCTION_START(HBBTV_StopDVBAudio);

   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      DBG("Calling STB_DPStopAudioDecoding(%u)", live_path);
      STB_DPStopAudioDecoding(live_path);
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_OTHER;
   }

   FUNCTION_FINISH(HBBTV_StopDVBAudio);

   return result;
}

/**
 * @brief   Instructs the controlling application to present the video specified
 *          by its component tag.
 * @param   component_tag Component tag of the video to be presented. If the component_tag value is
 *          -1, the default video elementary stream must be used.
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_PresentDVBVideo(S16BIT component_tag)
{
   E_HBBTV_ERR result;
   void *service;
   U16BIT requested_pid, pcr;
   E_STB_DP_VIDEO_CODEC requested_codec;
   U8BIT live_path;

   FUNCTION_START(HBBTV_PresentDVBVideo);

   DBG("component_tag=%d", component_tag);

   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      service = STB_DPGetTunedService(live_path);
   }
   else
   {
      service = NULL;
   }

   service = STB_DPGetTunedService(live_path);
   if (service != NULL)
   {
      pcr = ADB_GetServicePCRPid(service);
      if (component_tag == -1)
      {
         GetDefaultVideo(service, &requested_pid, &requested_codec);
      }
      else
      {
         if (!GetRequestedVideoStream(service, component_tag, &requested_pid, &requested_codec))
         {
            requested_pid = 0;
         }
      }
      if (requested_pid != 0)
      {
         SetVideo(live_path, pcr, requested_pid, requested_codec);
         result = HBBTV_OK;
      }
      else
      {
         /* Stream could not be found */
         result = HBBTV_ERR_BAD_PARAMETER;
         DBG("Stream with tag %d not found", component_tag);
      }
   }
   else
   {
      /* TODO: service might be not known yet (e.g. TuneDSD), should we retry later? */
      result = HBBTV_ERR_OTHER;
      DBG("Current service is NULL");
   }

   FUNCTION_FINISH(HBBTV_PresentDVBVideo);

   return result;
}

/**
 * @brief   Instructs the controlling application to stop the video presentation
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_StopDVBVideo(void)
{
   E_HBBTV_ERR result;
   U8BIT live_path;

   FUNCTION_START(HBBTV_StopDVBVideo);

   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      DBG("Calling STB_DPStopVideoDecoding(%u)", live_path);
      STB_DPStopVideoDecoding(live_path);
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_OTHER;
   }

   FUNCTION_FINISH(HBBTV_StopDVBVideo);

   return result;
}

/**
 * @brief   Instructs the controlling application to present the subtitles specified
 *          by their component tag.
 * @param   component_tag Component tag of the subtitle to be presented. If the component_tag value is
 *          -1, the default video elementary stream must be used.
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_PresentDVBSubtitle(S16BIT component_tag)
{
   E_HBBTV_ERR result = HBBTV_ERR_OTHER;
   U8BIT live_path;
   void *service;
   void **stream_list;
   U16BIT num, i;
   U8BIT num_tags, j;
   U16BIT pid;
   ADB_STREAM_TYPE stream_type;
   U32BIT lang_code;
   ADB_SUBTITLE_TYPE sub_type;
   U8BIT ttext_type;

   FUNCTION_START(HBBTV_PresentDVBSubtitle);

   live_path = STB_DPGetLivePath();
   if (live_path != INVALID_RES_ID)
   {
      service = STB_DPGetTunedService(live_path);
   }
   else
   {
      service = NULL;
   }

   if (service != NULL)
   {
      pid = 0;
      if (component_tag != -1)
      {
         ADB_GetStreamList(service, ADB_SUBTITLE_LIST_STREAM | ADB_TTEXT_SUBT_LIST_STREAM,
            &stream_list, &num);
         if ((num > 0) && (stream_list != NULL))
         {
            for (i = 0; (i < num) && (pid == 0); i++)
            {
               num_tags = ADB_GetStreamNumTags(stream_list[i]);
               for (j = 0; j < num_tags; j++)
               {
                  if (component_tag == ADB_GetStreamTag(stream_list[i], j))
                  {
                     pid = ADB_GetStreamPID(stream_list[i]);
                     stream_type = ADB_GetStreamType(stream_list[i]);
                     if (stream_type == ADB_SUBTITLE_STREAM)
                     {
                        lang_code = ADB_GetSubtitleStreamLangCode(stream_list[i]);
                        sub_type = ADB_GetSubtitleStreamType(stream_list[i]);
                     }
                     else if (stream_type == ADB_TTEXT_STREAM)
                     {
                        lang_code = ADB_GetTtextStreamLangCode(stream_list[i]);
                        ttext_type = ADB_GetTtextStreamType(stream_list[i]);
                     }
                     break;
                  }
               }
            }
            ADB_ReleaseStreamList(stream_list, num);
         }
      }

      if ((pid != 0) || (component_tag == -1))
      {
         if (ACTL_AreSubtitlesDisplayed() == TRUE)
         {
            ACTL_StopSubtitles();
         }

         if (component_tag == -1)
         {
            /* Reset to default */
            ADB_SetReqdSubtitleStreamSettings((void *)service, FALSE, 0, 0);
            ADB_SetReqdTeletextStreamSettings((void *)service, FALSE, 0, 0);
         }
         else 
         {
            if (stream_type == ADB_SUBTITLE_STREAM)
            {
               ADB_SetReqdSubtitleStreamSettings((void *)service, TRUE, lang_code, sub_type);
            }
            else if (stream_type == ADB_TTEXT_STREAM)
            {
               ADB_SetReqdTeletextStreamSettings((void *)service, TRUE, lang_code, ttext_type);
            }
         }

         ACTL_StartSubtitles();
         result = HBBTV_OK;
      }
   }

   FUNCTION_FINISH(HBBTV_PresentDVBSubtitle);

   return result;
}

/**
 * @brief   Instructs the controlling application to stop the subtitle presentation
 * @return  - HBBTV_OK success
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_StopDVBSubtitle(void)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_StopDVBVideo);

   ACTL_StopSubtitles();
   result = HBBTV_OK;

   FUNCTION_FINISH(HBBTV_StopDVBVideo);

   return result;
}

/**
 * @brief   Changes the presentation window of the DVB video
 * @param   rect Rectangle representing the position where the video should be presented.
 *          Values are in HbbTV 1280x720 coordinates.
 */
void HBBTV_SetVideoRectangle(S_HBBTV_RECT *rect)
{
   S_RECTANGLE output;

   FUNCTION_START(HBBTV_SetVideoRectangle);

   output.left = rect->left;
   output.top = rect->top;
   output.width = rect->width;
   output.height = rect->height;

   STB_VTSetHbbtvVideoWindow(&output);

   FUNCTION_FINISH(HBBTV_SetVideoRectangle);
}

