/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 * 
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      System Interface, Media player
 * @file       mediaplayer.c
 * @date       January 2014
 * @author     Sergio Panseri
 */

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>
#include "stbhwav.h"
#include "stbhwmediaplayer.h"
#include "stbheap.h"
#include "stbdpc.h"
#include "hbbtv_sif_types.h"
#include "hbbtv_sif_mediaplayer.h"
#include "hbbtv.h"


/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

static E_STB_MP_COMPONENT_TYPE ConvertCompTypeToPlatform(E_HBBTV_COMPONENT_TYPE type);
static void ConvertComponentToPlatform(S_STB_MP_COMPONENT_DETAILS *plat_comp, 
   S_HBBTV_COMPONENT_DETAILS *hbbtv_comp);
static void ConvertComponentToHbbtv(S_HBBTV_COMPONENT_DETAILS *hbbtv_comp, 
   S_STB_MP_COMPONENT_DETAILS *plat_comp);
static void MpCallback(void *handle, E_STB_MP_STATE state);
static E_HBBTV_MP_STATE ConvertToHbbtvState(E_STB_MP_STATE state);


/*---local function definitions----------------------------------------------*/

static E_STB_MP_COMPONENT_TYPE ConvertCompTypeToPlatform(E_HBBTV_COMPONENT_TYPE type)
{
   E_STB_MP_COMPONENT_TYPE plat_type;

   FUNCTION_START(ConvertCompTypeToPlatform);

   switch (type)
   {
      case HBBTV_COMPONENT_VIDEO:
      {
         plat_type = STB_MP_COMPONENT_VIDEO;
         break;
      }
      case HBBTV_COMPONENT_AUDIO:
      {
         plat_type = STB_MP_COMPONENT_AUDIO;
         break;
      }
      default:
      {
         plat_type = STB_MP_COMPONENT_SUBTITLE;
         break;
      }      
   }

   FUNCTION_FINISH(ConvertCompTypeToPlatform);
   
   return plat_type;
}

static void ConvertComponentToHbbtv(S_HBBTV_COMPONENT_DETAILS *hbbtv_comp, 
   S_STB_MP_COMPONENT_DETAILS *plat_comp)
{
   FUNCTION_START(ConvertComponentToHbbtv);

   hbbtv_comp->active = plat_comp->active;
   hbbtv_comp->component_tag = plat_comp->component_tag;
   hbbtv_comp->encrypted = plat_comp->encrypted;
   hbbtv_comp->pid = plat_comp->pid;
   switch (plat_comp->type)
   {
      case STB_MP_COMPONENT_VIDEO:
      {
         hbbtv_comp->type = HBBTV_COMPONENT_VIDEO;
         break;
      }
      case STB_MP_COMPONENT_AUDIO:
      {
         hbbtv_comp->type = HBBTV_COMPONENT_AUDIO;
         break;
      }
      default:
      {
         hbbtv_comp->type = HBBTV_COMPONENT_SUBTITLE;
         break;
      }
   }

   hbbtv_comp->av.audio.audio_channels = plat_comp->av.audio.num_channels;
   hbbtv_comp->av.audio.audio_description = plat_comp->av.audio.audio_description;
   switch (plat_comp->av.audio.encoding)
   {
      case AV_AUDIO_CODEC_HEAAC:
      {
         hbbtv_comp->av.audio.encoding = HBBTV_AUDIO_HEAAC;
         break;
      }
      case AV_AUDIO_CODEC_EAC3:
      {
         hbbtv_comp->av.audio.encoding = HBBTV_AUDIO_E_AC3;
         break;
      }
      case AV_AUDIO_CODEC_MP3:
      {
         hbbtv_comp->av.audio.encoding = HBBTV_AUDIO_MP3;
         break;
      }
      default:
      {
         hbbtv_comp->av.audio.encoding = HBBTV_AUDIO_MPEG2;
         break;
      }
   }
   hbbtv_comp->av.audio.lang_code[0] = ((U8BIT *)(&plat_comp->av.audio.lang_code))[2];
   hbbtv_comp->av.audio.lang_code[1] = ((U8BIT *)(&plat_comp->av.audio.lang_code))[1];
   hbbtv_comp->av.audio.lang_code[2] = ((U8BIT *)(&plat_comp->av.audio.lang_code))[0];
   hbbtv_comp->av.audio.lang_code[3] = 0;
   
   switch(plat_comp->av.video.aspect_ratio)
   {
      case ASPECT_RATIO_4_3:
      {
         hbbtv_comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_4x3;
         break;
      }
      default:
      {
         hbbtv_comp->av.video.aspect_ratio = HBBTV_VIDEO_ASPECT_RATIO_16x9;
         break;
      }
   }
   switch (plat_comp->av.video.encoding)
   {
      case AV_VIDEO_CODEC_H264:
      {
         hbbtv_comp->av.video.encoding = HBBTV_VIDEO_H_264;
         break;
      }
      default:
      {
         hbbtv_comp->av.video.encoding = HBBTV_VIDEO_MPEG2;
         break;
      }
   }
   hbbtv_comp->av.video.frame_rate = plat_comp->av.video.frame_rate;
   hbbtv_comp->av.video.hd = plat_comp->av.video.hd;

   hbbtv_comp->av.subtitle.hearing_impaired = plat_comp->av.subtitle.hearing_impaired;
   hbbtv_comp->av.subtitle.lang_code[0] = ((U8BIT *)(&plat_comp->av.subtitle.lang_code))[2];
   hbbtv_comp->av.subtitle.lang_code[1] = ((U8BIT *)(&plat_comp->av.subtitle.lang_code))[1];
   hbbtv_comp->av.subtitle.lang_code[2] = ((U8BIT *)(&plat_comp->av.subtitle.lang_code))[0];
   hbbtv_comp->av.subtitle.lang_code[3] = 0;

   FUNCTION_FINISH(ConvertComponentToHbbtv);
}

static void ConvertComponentToPlatform(S_STB_MP_COMPONENT_DETAILS *plat_comp, 
   S_HBBTV_COMPONENT_DETAILS *hbbtv_comp)
{
   FUNCTION_START(ConvertComponentToPlatform);

   plat_comp->component_tag = hbbtv_comp->component_tag;
   plat_comp->pid = hbbtv_comp->pid;
   plat_comp->encrypted = hbbtv_comp->encrypted;
   plat_comp->active = hbbtv_comp->active;
   plat_comp->type = ConvertCompTypeToPlatform(hbbtv_comp->type);

   switch (hbbtv_comp->av.audio.encoding)
   {
      case HBBTV_AUDIO_MPEG2:
      {
         plat_comp->av.audio.encoding = AV_AUDIO_CODEC_MP2;
         break;
      }
      case HBBTV_AUDIO_HEAAC:
      {
         plat_comp->av.audio.encoding = AV_AUDIO_CODEC_HEAAC;
         break;
      }
      case HBBTV_AUDIO_E_AC3:
      {
         plat_comp->av.audio.encoding = AV_AUDIO_CODEC_EAC3;
         break;
      }
      case HBBTV_AUDIO_MP3:
      {
         plat_comp->av.audio.encoding = AV_AUDIO_CODEC_MP3;
         break;
      }
   }
   plat_comp->av.audio.lang_code = hbbtv_comp->av.audio.lang_code[2] + 
      (hbbtv_comp->av.audio.lang_code[1] << 8) + (hbbtv_comp->av.audio.lang_code[0] << 16);
   plat_comp->av.audio.num_channels = hbbtv_comp->av.audio.audio_channels;
   plat_comp->av.audio.audio_description = hbbtv_comp->av.audio.audio_description;

   switch (hbbtv_comp->av.video.encoding)
   {
      case HBBTV_VIDEO_MPEG2:
      {
         plat_comp->av.video.encoding = AV_VIDEO_CODEC_MPEG2;
         break;
      }
      case HBBTV_VIDEO_H_264:
      {
         plat_comp->av.video.encoding = AV_VIDEO_CODEC_H264;
         break;
      }
   }
   plat_comp->av.video.hd = hbbtv_comp->av.video.hd;
   plat_comp->av.video.frame_rate = hbbtv_comp->av.video.frame_rate;
   switch (hbbtv_comp->av.video.aspect_ratio)
   {
      case HBBTV_VIDEO_ASPECT_RATIO_16x9:
      {
         plat_comp->av.video.aspect_ratio = ASPECT_RATIO_4_3;
         break;
      }
      case HBBTV_VIDEO_ASPECT_RATIO_4x3:
      {
         plat_comp->av.video.aspect_ratio = ASPECT_RATIO_16_9;
         break;
      }
   }
      
   plat_comp->av.subtitle.lang_code = hbbtv_comp->av.subtitle.lang_code[2] + 
      (hbbtv_comp->av.subtitle.lang_code[1] << 8) + (hbbtv_comp->av.subtitle.lang_code[0] << 16);;
   plat_comp->av.subtitle.hearing_impaired = hbbtv_comp->av.subtitle.hearing_impaired;

   FUNCTION_FINISH(ConvertComponentToPlatform);
}

static E_HBBTV_MP_STATE ConvertToHbbtvState(E_STB_MP_STATE state)
{
   E_HBBTV_MP_STATE retval;

   FUNCTION_START(ConvertToHbbtvState);

   switch (state)
   {
      case STB_MP_STATE_STOPPED:
      {
         retval = HBBTV_MP_STOPPED;
         break;
      }
      case STB_MP_STATE_PLAYING:
      {
         retval = HBBTV_MP_PLAYING;
         break;
      }
      case STB_MP_STATE_PAUSED:
      {
         retval = HBBTV_MP_PAUSED;
         break;
      }
      case STB_MP_STATE_CONNECTING:
      {
         retval = HBBTV_MP_CONNECTING;
         break;
      }
      case STB_MP_STATE_BUFFERING:
      {
         retval = HBBTV_MP_BUFFERING;
         break;
      }
      case STB_MP_STATE_FINISHED:
      {
         retval = HBBTV_MP_EOF;
         break;
      }
      case STB_MP_STATE_ERROR:
      {
         retval = HBBTV_MP_ERROR;
         break;
      }
   }

   FUNCTION_FINISH(ConvertToHbbtvState);

   return retval;
}

static void MpCallback(void *handle, E_STB_MP_STATE state)
{
   FUNCTION_START(MpCallback);
   HBBTV_MPNotifyState(handle, ConvertToHbbtvState(state));
   FUNCTION_FINISH(MpCallback);
}


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Initialises Media Player with the source URL.
 * @return  Handle
 * @param   source_url Source url of the file
 */
void* HBBTV_MPInit(U8BIT *source_url)
{
   void *result;

   FUNCTION_START(HBBTV_MPInit);

   result = STB_MPInit(source_url);
   if (result != NULL)
   {
      STB_MPRegisterCallback(result, MpCallback);
   }

   FUNCTION_FINISH(HBBTV_MPInit);

   return result;
}

/**
 * @brief   Releases the Media Player and the associated resources
 * @param   ihdl Handle returned by HBBTV_MPInit
 */
void HBBTV_MPExit(void* ihdl)
{
   FUNCTION_START(HBBTV_MPExit);

   STB_MPExit(ihdl);

   FUNCTION_FINISH(HBBTV_MPExit);
}

/**
 * @brief   Starts playing file given by HBBTV_MPInit
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   params Start parameters, see the definition of S_HBBTV_MP_START_PARAMS.
 */
E_HBBTV_ERR HBBTV_MPStart(void* ihdl, S_HBBTV_MP_START_PARAMS* params)
{
   E_HBBTV_ERR result;
   S_STB_MP_START_PARAMS start_params;

   FUNCTION_START(HBBTV_MPStart);

   if ((ihdl != NULL) && (params != NULL))
   {
      start_params.cache = params->cache;
      start_params.loops = params->loops;
      if (STB_MPStart(ihdl, &start_params) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPStart);

   return result;
}

/**
 * @brief   Stops playing file
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 */
E_HBBTV_ERR HBBTV_MPStop(void* ihdl)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_MPStop);

   if (ihdl != NULL)
   {
      if (STB_MPStop(ihdl) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPStop);

   return result;
}

/**
 * @brief   Pauses playing of file
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 */
E_HBBTV_ERR HBBTV_MPPause(void* ihdl)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_MPPause);

   if (ihdl != NULL)
   {
      if (STB_MPPause(ihdl) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPPause);

   return result;
}

/**
 * @brief   Resumes playing of file
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 */
E_HBBTV_ERR HBBTV_MPResume(void* ihdl)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_MPResume);

   if (ihdl != NULL)
   {
      if (STB_MPResume(ihdl) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPResume);

   return result;
}

/**
 * @brief   Changes position and size of the video presentation window
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   rect Pointer to the rectangle defining the presentation window. Values are HbbTV 
 *          1280x720 coordinates.
 */
E_HBBTV_ERR HBBTV_MPResize(void *ihdl, S_HBBTV_RECT *rect)
{
   E_HBBTV_ERR result;
   S_RECTANGLE r;
   U8BIT path, vid_path = INVALID_RES_ID;
   U16BIT width, height;
   float vertical, horizontal;

   FUNCTION_START(HBBTV_MPResize);

   if ((ihdl != NULL) && (rect != NULL))
   {
      path = STB_DPGetLivePath();
      if (path != INVALID_RES_ID)
      {
         vid_path = STB_DPGetPathVideoDecoder(path);
      }

      if (vid_path != INVALID_RES_ID)
      {
         STB_AVGetScreenSize(vid_path, &width, &height);
         
         horizontal = (float)width / 1280;
         vertical = (float)height / 720;

         r.left = rect->left * horizontal;
         r.top = rect->top * vertical;
         r.height = rect->height * vertical;
         r.width = rect->width * horizontal;
      }
      else
      {
         /* Failed to determine the decoder path, don't convert coordinates */
         r.left = rect->left;
         r.top = rect->top;
         r.height = rect->height;
         r.width = rect->width;
      }

      if (STB_MPResize(ihdl, &r) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPResize);

   return result;
}

/**
 * @brief   Gets the current play position
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   begin Pointer to return start position in millisecs (usually zero)
 * @param   current Pointer to return current position in millisecs
 * @param   end Pointer to return end position in millisecs
 */
E_HBBTV_ERR HBBTV_MPGetTimes(void* ihdl, U32BIT *begin, U32BIT *current, U32BIT *end)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_MPGetTimes);

   if ((ihdl != NULL) && (begin != NULL) && (current != NULL) && (end != NULL))
   {
      if (STB_MPGetTimes(ihdl, begin, current, end) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPGetTimes);

   return result;
}

/**
 * @brief   Sets the play position
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   position Position in in millisecs in the clip
 */
E_HBBTV_ERR HBBTV_MPSeek(void* ihdl, U32BIT position)
{
   E_HBBTV_ERR result;

   FUNCTION_START(HBBTV_MPSeek);

   if (ihdl != NULL)
   {
      if (STB_MPSeek(ihdl, position) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPSeek);

   return result;
}

/**
 * @brief   Retrieves the list of available components. Provided HBBTV_OK is returned, the HbbTV
 *          engine will call HBBTV_MPReleaseComponentList to release allocated data on the same task
 *          and immediately after it has copied the data.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   type Required component type
 * @param   num_ptr Pointer to number of components (Not NULL)
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS Pointer (Not NULL)
 */
E_HBBTV_ERR HBBTV_MPObtainComponentList(void* ihdl, E_HBBTV_COMPONENT_TYPE type, U32BIT *num_ptr, 
   S_HBBTV_COMPONENT_DETAILS** list_ptr)
{
   E_HBBTV_ERR result;
   E_STB_MP_COMPONENT_TYPE plat_type;
   S_STB_MP_COMPONENT_DETAILS *plat_list, *plat_comp;
   S_HBBTV_COMPONENT_DETAILS *hbbtv_comp;
   U32BIT i;

   FUNCTION_START(HBBTV_MPObtainComponentList);

   if ((ihdl != NULL) && (num_ptr != NULL) && (list_ptr != NULL))
   {
      plat_type = ConvertCompTypeToPlatform(type);
      if (STB_MPObtainComponentList(ihdl, plat_type, num_ptr, &plat_list) == HW_OK)
      {
         if (*num_ptr > 0)
         {
            *list_ptr = STB_AppGetMemory(*num_ptr * sizeof(S_HBBTV_COMPONENT_DETAILS));
            if (*list_ptr != NULL)
            {
               hbbtv_comp = *list_ptr;
               plat_comp = plat_list;
               for (i = 0; i < *num_ptr; i++, hbbtv_comp++, plat_comp++)
               {
                  ConvertComponentToHbbtv(hbbtv_comp, plat_comp);
               }
               STB_MPReleaseComponentList(ihdl, plat_list);
            }
         }
         if ((*list_ptr != NULL) || (*num_ptr == 0))
         {
            result = HBBTV_OK;
         }
         else
         {
            *num_ptr = 0;
            result = HBBTV_ERR_OTHER;
         }
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPObtainComponentList);

   return result;
}

/**
 * @brief   Release data allocated by HBBTV_MPObtainComponentList. Always called immediately after
 *          the HbbTV engine has copied data.
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   list_ptr Pointer to S_HBBTV_COMPONENT_DETAILS
 */
void HBBTV_MPReleaseComponentList(void* ihdl, S_HBBTV_COMPONENT_DETAILS* list_ptr)
{
   FUNCTION_START(HBBTV_MPReleaseComponentList);
   USE_UNWANTED_PARAM(ihdl);
   
   STB_AppFreeMemory(list_ptr);

   FUNCTION_FINISH(HBBTV_MPReleaseComponentList);
}

/**
 * @brief   Replaces any component of the same type with the specified one. This function can be 
 *          called before or after the media player has been started.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   component Pointer to the component to be rendered
 */
E_HBBTV_ERR HBBTV_MPSelectComponent(void* ihdl, S_HBBTV_COMPONENT_DETAILS *component)
{
   E_HBBTV_ERR result;
   S_STB_MP_COMPONENT_DETAILS plat_comp;

   FUNCTION_START(HBBTV_MPSelectComponent);

   if ((ihdl != NULL) && (component != NULL))
   {
      ConvertComponentToPlatform(&plat_comp, component);
      if (STB_MPSelectComponent(ihdl, &plat_comp) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPSelectComponent);

   return result;
}

/**
 * @brief   Stops rendering the specified component.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER
 *          - HBBTV_ERR_OTHER controlling application specific error
 * @param   ihdl Handle returned by HBBTV_MPInit
 * @param   component Pointer to the component to be stopped
 */
E_HBBTV_ERR HBBTV_MPUnselectComponent (void* ihdl, S_HBBTV_COMPONENT_DETAILS *component)
{
   E_HBBTV_ERR result;
   S_STB_MP_COMPONENT_DETAILS plat_comp;

   FUNCTION_START(HBBTV_MPUnselectComponent);

   if ((ihdl != NULL) && (component != NULL))
   {
      ConvertComponentToPlatform(&plat_comp, component);
      if (STB_MPUnselectComponent(ihdl, &plat_comp) == HW_OK)
      {
         result = HBBTV_OK;
      }
      else
      {
         result = HBBTV_ERR_OTHER;
      }
   }
   else
   {
      result = HBBTV_ERR_BAD_PARAMETER;
   }

   FUNCTION_FINISH(HBBTV_MPUnselectComponent);

   return result;
}

/**
 * @brief   Retrieves the error code from the media player. The HbbTV engine uses this function to
 *          retrieve the error code from the media player and expects it to be available as soon as 
 *          it receives the ERROR event via HBBTV_MPNotifyState. The HbbTV engine might call 
 *          HBBTV_MPGetError within HBBTV_MPNotifyState.
 * @return  E_HBBTV_MP_ERROR
 * @param   ihdl Handle returned by HBBTV_MPInit
 */
E_HBBTV_MP_ERROR HBBTV_MPGetError (void* ihdl)
{
   E_HBBTV_MP_ERROR result;
   E_STB_MP_ERROR err;

   FUNCTION_START(HBBTV_MPGetError);

   result = HBBTV_MP_UNDEFINED;
   if (ihdl != NULL)
   {
      err = STB_MPGetError(ihdl);
      switch (err)
      {
         case STB_MP_NO_ERROR:
         {
            result = HBBTV_MP_NO_ERROR;
            break;
         }
         case STB_MP_FORMAT_NOT_SUPPORTED:
         {
            result = HBBTV_MP_FORMAT_NOT_SUPPORTED;
            break;
         }
         case STB_MP_CONNECTION_ERROR:
         {
            result = HBBTV_MP_CONNECTION_ERROR;
            break;
         }
         case STB_MP_UNDEFINED:
         {
            result = HBBTV_MP_UNDEFINED;
            break;
         }
         case STB_MP_NO_RESOURCES:
         {
            result = HBBTV_MP_NO_RESOURCES;
            break;
         }
         case STB_MP_CORRUPT:
         {
            result = HBBTV_MP_CORRUPT;
            break;
         }
         case STB_MP_NOT_AVAILABLE:
         {
            result = HBBTV_MP_NOT_AVAILABLE;
            break;
         }
         case STB_MP_NOT_AVAILABLE_POSITION:
         {
            result = HBBTV_MP_NOT_AVAILABLE_POSITION;
            break;
         }
         case STB_MP_BLOCKED:
         {
            result = HBBTV_MP_BLOCKED;
            break;
         }
      }
   }

   FUNCTION_FINISH(HBBTV_MPGetError);

   return result;
}
