/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 * 
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      System Interface, Configuration
 * @file       conf.c
 * @date       January 2014
 * @author     Sergio Panseri
 */
 
/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>
#include <string.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwnet.h"
#include "stbhwip.h"
#include "stbheap.h"

#include "hbbtv.h"
#include "hbbtv_sif_network.h"


/*---constant definitions for this file--------------------------------------*/

#ifdef DEBUG
#define DBG(string, vars ...) STB_SPDebugWrite("[%s:%d] "string, __FUNCTION__, __LINE__, ##vars)
#else
#define DBG(string, vars ...)
#endif


/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/* DVBCore only has the concept of one interface */
static S_HBBTV_NW_INTERFACE *interface = NULL;


/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/


/*---global function definitions---------------------------------------------*/

/**
 * @brief   Retrieves the current status of the broadband network connection. This MUST be a 
 *          non-blocking function, returning results immediately.
 * @param   p_status Current status of the network connection
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_BAD_PARAMETER invalid parameter
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_NWGetStatus(E_HBBTV_NW_STATUS *p_status)
{
   E_HBBTV_ERR result;
   E_NW_LINK_STATUS link_status;

   FUNCTION_START(HBBTV_NWGetStatus);

   link_status = STB_NWGetLinkStatus();
   switch (link_status)
   {
      case NW_LINK_ACTIVE:
      {
         *p_status = HBBTV_NW_STATUS_ACTIVE;
         break;
      }
      case NW_LINK_INACTIVE:
      {
         *p_status = HBBTV_NW_STATUS_INACTIVE;
         break;
      }
      case NW_LINK_DISABLED:
      {
         *p_status = HBBTV_NW_STATUS_DISABLED;
         break;
      }
   }

   result = HBBTV_OK;

   FUNCTION_FINISH(HBBTV_NWGetStatus);

   return result;
}

/**
 * @brief   Retrieves the list of available network interfaces. Provided HBBTV_OK is returned,
 *          the HbbTV engine will call HBBTV_NWReleaseInterfaceList to release allocated data on the
 *          same task and immediately after it has copied the data.
 * @param   list Pointer to the list of pointers
 * @param   num Pointer to the number of elements in the returned list
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_NWGetInterfaceList(S_HBBTV_NW_INTERFACE **list, U32BIT *num)
{
   E_HBBTV_ERR result;
   E_NW_INTERFACE iface;
   U8BIT mac_addr[6];

   FUNCTION_START(HBBTV_NWGetInterfaceList);

   if (interface == NULL)
   {
      interface = STB_AppGetMemory(sizeof(S_HBBTV_NW_INTERFACE));
      if (interface != NULL)
      {
         interface->handle = interface; // Any pointer
         iface = STB_NWGetSelectedInterface();
         if (STB_NWGetMACAddress(iface, mac_addr))
         {
            interface->mac.zlen = 18; /* 6 2-digit fields separated by 5 ':' */
            interface->mac.zptr = STB_AppGetMemory(interface->mac.zlen);
            sprintf((char *)interface->mac.zptr, "%02x:%02x:%02x:%02x:%02x:%02x", mac_addr[0], mac_addr[1], 
               mac_addr[2], mac_addr[3], mac_addr[4], mac_addr[5]);
         }
         else
         {
            STB_AppFreeMemory(interface);
            interface = NULL;
            DBG("STB_NWGetMACAddress returned FALSE");
         }
      }
      else
      {
         DBG("Failed allocating %d bytes", sizeof(S_HBBTV_NW_INTERFACE));
      }
   }

   if (interface != NULL)
   {
      *list = interface;
      *num = 1;
      result = HBBTV_OK;
   }
   else
   {
      *num = 0;
       result = HBBTV_ERR_OTHER;
   }

   FUNCTION_FINISH(HBBTV_NWGetInterfaceList);

   return result;
}

/**
 * @brief   Releases data allocated by HBBTV_NWGetInterfaceList (e.g. String data). Always called
 *          immediately after the HbbTV engine has copied data.
 * @param   list list to be released
 * @param   num Number of elements in the list
 */
void HBBTV_NWReleaseInterfaceList(S_HBBTV_NW_INTERFACE *list, U32BIT num)
{
   FUNCTION_START(HBBTV_NWReleaseInterfaceList);
   USE_UNWANTED_PARAM(list);
   USE_UNWANTED_PARAM(num);
   /* Nothing to do, interface is static */
   FUNCTION_FINISH(HBBTV_NWReleaseInterfaceList);
}

/**
 * @brief   Retrieves a string containing the interface IP address. The HbbTV engine will free the 
 *          string contained in the zptr field by calling HBBTV_OSFree
 * @param   handle Handle of the interface as returned in the S_HBBTV_NW_INTERFACE structure by
 *          HBBTV_NWGetInterfaceList.
 * @param   ip String containing the interface's IP address.
 * @return  - HBBTV_OK on success
 *          - HBBTV_ERR_OTHER controlling application specific error
 */
E_HBBTV_ERR HBBTV_NWGetIpAddress(void *handle, S_HBBTV_STRING *ip)
{
   E_HBBTV_ERR result;
   U8BIT ip_addr[4];

   FUNCTION_START(HBBTV_NWGetIpAddress);
   USE_UNWANTED_PARAM(handle);

   if (STB_IPGetIPAddress(ip_addr))
   {
      ip->zptr = STB_AppGetMemory(16); /* 4 3-digit fields separated by 3 '.' */
      sprintf((char *)ip->zptr, "%d.%d.%d.%d", ip_addr[0], ip_addr[1], ip_addr[2], ip_addr[3]);
      ip->zlen = strlen((char *)ip->zptr); 
      result = HBBTV_OK;
   }
   else
   {
      result = HBBTV_ERR_OTHER;
      DBG("STB_IPGetIPAddress returned FALSE");
   }

   FUNCTION_FINISH(HBBTV_NWGetIpAddress);

   return result;
}

/**
 * @brief   Retrieves the interface enabled/disabled status
 * @param   handle Handle of the interface as returned in the S_HBBTV_NW_INTERFACE structure by
 *          HBBTV_NWGetInterfaceList.
 * @return  TRUE if the interface is enabled, FALSE if it is disabled
 */
BOOLEAN HBBTV_NWGetEnabled(void *handle)
{
   BOOLEAN result;

   FUNCTION_START(HBBTV_NWGetEnabled);

   USE_UNWANTED_PARAM(handle);

   if (STB_NWGetLinkStatus() != NW_LINK_DISABLED)
   {
      result = TRUE;
   }
   else
   {
      result = FALSE;
   }

   FUNCTION_FINISH(HBBTV_NWGetEnabled);

   return result;
}

/**
 * @brief   Retrieves the interface connected/disconnected status
 * @param   handle Handle of the interface as returned in the S_HBBTV_NW_INTERFACE structure by
 *          HBBTV_NWGetInterfaceList.
 * @return  TRUE if the interface is connected, FALSE if it is disconnected
 */
BOOLEAN HBBTV_NWGetConnected(void *handle)
{
   BOOLEAN result;

   FUNCTION_START(HBBTV_NWGetConnected);

   USE_UNWANTED_PARAM(handle);
   
   if (STB_NWGetLinkStatus() == NW_LINK_ACTIVE)
   {
      result = TRUE;
   }
   else
   {
      result = FALSE;
   }

   FUNCTION_FINISH(HBBTV_NWGetConnected);

   return result;
}
