/*******************************************************************************
 * Copyright © 2014 The DTVKit Open Software Foundation Ltd (www.dtvkit.org)
 * Copyright © 2014 Ocean Blue Software Ltd
 *
 * This file is part of a DTVKit Software Component
 * You are permitted to copy, modify or distribute this file subject to the terms
 * of the DTVKit 1.0 Licence which can be found in licence.txt or at www.dtvkit.org
 * 
 * THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 * EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * If you or your organisation is not a member of DTVKit then you have access
 * to this source code outside of the terms of the licence agreement
 * and you are expected to delete this and any associated files immediately.
 * Further information on DTVKit, membership and terms can be found at www.dtvkit.org
 *******************************************************************************/
/**
 * @brief      System Interface, Operating System
 * @file       os.c
 * @date       January 2014
 * @author     Sergio Panseri
 */

/*---includes for this file--------------------------------------------------*/

/* compiler library header files */
#include <stdio.h>

/* third party header files */

#include <techtype.h>
#include <dbgfuncs.h>

#include "stbhwos.h"
#include "stbheap.h"

#include "hbbtv_sif_types.h"


/*---constant definitions for this file--------------------------------------*/

/*---local typedef structs for this file-------------------------------------*/

/*---local (static) variable declarations for this file----------------------*/
/*   (internal variables declared static to make them local)                 */

/*---local function prototypes for this file---------------------------------*/
/*   (internal functions declared static to make them local)                 */

/*---local function definitions----------------------------------------------*/

/*---global function definitions---------------------------------------------*/

/**
 * @brief   Allocates the specified number of bytes
 * @param   size Required size of new memory block.
 * @return  Pointer to memory location.
 */
void* HBBTV_OSAlloc(U32BIT size)
{
   void *result;

   FUNCTION_START(HBBTV_OSAlloc);

   result = STB_AppGetMemory(size);

   FUNCTION_FINISH(HBBTV_OSAlloc);

   return result;
}

/**
 * @brief   Releases previously allocated memory.
 * @param   ptr Pointer to the memory block to be freed
 */
void HBBTV_OSFree(void *ptr)
{
   FUNCTION_START(HBBTV_OSFree);

   STB_AppFreeMemory(ptr);

   FUNCTION_FINISH(HBBTV_OSFree);
}

/**
 * @brief   Creates a new mutex.
 * @return  New mutex handle
 */
void* HBBTV_OSCreateMutex(void)
{
   void *result;

   FUNCTION_START(HBBTV_OSCreateMutex);

   result = STB_OSCreateMutex();

   FUNCTION_FINISH(HBBTV_OSCreateMutex);

   return result;
}

/**
 * @brief   Deletes an exisiting mutex.
 * @param   mutex Mutex handle
 */
void HBBTV_OSDeleteMutex(void* mutex)
{
   FUNCTION_START(HBBTV_OSDeleteMutex);

   STB_OSDeleteMutex(mutex);

   FUNCTION_FINISH(HBBTV_OSDeleteMutex);
}

/**
 * @brief   Locks the specified mutex.
 * @param   mutex Mutex handle
 */
void HBBTV_OSMutexLock(void* mutex)
{
   FUNCTION_START(HBBTV_OSMutexLock);

   STB_OSMutexLock(mutex);

   FUNCTION_FINISH(HBBTV_OSMutexLock);
}

/**
 * @brief   Unlocks the specified mutex.
 * @param   mutex Mutex handle
 */
void HBBTV_OSMutexUnlock(void* mutex)
{
   FUNCTION_START(HBBTV_OSMutexUnlock);

   STB_OSMutexUnlock(mutex);

   FUNCTION_FINISH(HBBTV_OSMutexUnlock);
}

/**
 * @brief   Creates a new counting semaphore, with the count set to value
 * @param   value Initial value of count
 * @return  New semaphore handle
 */
void* HBBTV_OSCreateCountSemaphore(U32BIT value)
{
   void *result;

   FUNCTION_START(HBBTV_OSCreateCountSemaphore);

   result = STB_OSCreateCountSemaphore(value);

   FUNCTION_FINISH(HBBTV_OSCreateCountSemaphore);

   return result;
}

/**
 * @brief   Deletes an exisiting semaphore.
 * @param   semaphore Semaphore handle
 */
void HBBTV_OSDeleteSemaphore(void* semaphore)
{
   FUNCTION_START(HBBTV_OSDeleteSemaphore);

   STB_OSDeleteSemaphore(semaphore);

   FUNCTION_FINISH(HBBTV_OSDeleteSemaphore);
}

/**
 * @brief   Waits for specified semaphore to be cleared.
 * @param   semaphore Semaphore handle
 */
void HBBTV_OSSemaphoreWait(void* semaphore)
{
   FUNCTION_START(HBBTV_OSSemaphoreWait);

   STB_OSSemaphoreWait(semaphore);

   FUNCTION_FINISH(HBBTV_OSSemaphoreWait);
}

/**
 * @brief   Waits a specified timeout period for a specified semaphore to be cleared.
 * @param   semaphore Semaphore handle
 * @param   timeout The timeout in milliseconds, HBBTV_OS_TIMEOUT_NOW or HBBTV_OS_TIMEOUT_NEVER
 * @return  TRUE if the semaphore was obtained, FALSE otherwise.
 */
BOOLEAN HBBTV_OSSemaphoreWaitTimeout(void* semaphore, U16BIT timeout)
{
   BOOLEAN result;

   FUNCTION_START(HBBTV_OSWriteQueue);

   result = STB_OSSemaphoreWaitTimeout(semaphore, timeout);

   FUNCTION_FINISH(HBBTV_OSWriteQueue);

   return result;
}

/**
 * @brief   Clears specified semaphore.
 * @param   semaphore Semaphore handle
 */
void HBBTV_OSSemaphoreSignal(void* semaphore)
{
   FUNCTION_START(HBBTV_OSSemaphoreSignal);

   STB_OSSemaphoreSignal(semaphore);

   FUNCTION_FINISH(HBBTV_OSSemaphoreSignal);
}

/**
 * @brief   Creates a new task.
 * @param   function Pointer to callback function
 * @param   param Parameter of above function
 * @param   stack Size of stack required in bytes
 * @param   priority 0 - 15 (15 is highest)
 * @param   name Name to assign to task
 * @return  New task handle
 */
void* HBBTV_OSCreateTask(void (*function)(void*), void *param, U32BIT stack, U8BIT priority, 
   U8BIT *name)
{
   void *result;

   FUNCTION_START(HBBTV_OSCreateTask);

   result = STB_OSCreateTask(function, param, stack, priority, name);

   FUNCTION_FINISH(HBBTV_OSCreateTask);

   return result;
}

/**
 * @brief   Destroys the specified task
 * @param   task_id Task handle
 */
void HBBTV_OSDestroyTask(void *task_id)
{
   FUNCTION_START(HBBTV_OSDestroyTask);

   STB_OSDestroyTask(task_id);

   FUNCTION_FINISH(HBBTV_OSDestroyTask);
}

/**
 * @brief   Suspends current task for given period.
 * @param   time Number of milliseconds the task will be suspended
 */
void HBBTV_OSTaskDelay(U16BIT time)
{
   FUNCTION_START(HBBTV_OSTaskDelay);

   STB_OSTaskDelay(time);

   FUNCTION_FINISH(HBBTV_OSTaskDelay);
}

/**
 * @brief   Returns the number of milliseconds since the system booted
 * @return  Time in milliseconds
 */
U32BIT HBBTV_OSGetClockMilliseconds(void)
{
   U32BIT result;

   FUNCTION_START(HBBTV_OSGetClockMilliseconds);

   result = STB_OSGetClockMilliseconds();

   FUNCTION_FINISH(HBBTV_OSGetClockMilliseconds);

   return result;
}

/**
 * @brief   Returns the current time
 * @return  Current time expressed in number of seconds passed since midnight (UTC) 1 January 1970
 */
U32BIT HBBTV_OSGetTimeNow(void)
{
   U32BIT result;

   FUNCTION_START(HBBTV_OSGetTimeNow);

   result = STB_OSGetClockRTC();

   FUNCTION_FINISH(HBBTV_OSGetTimeNow);

   return result;
}

/**
 * @brief   Creates a new message queue.
 * @param   msg_size Size of each message in queue
 * @param   msg_max Maximum number of messages in queue
 * @return  New queue handle
 */
void* HBBTV_OSCreateQueue(U16BIT msg_size, U16BIT msg_max)
{
   void *result;

   FUNCTION_START(HBBTV_OSCreateQueue);

   result = STB_OSCreateQueue(msg_size, msg_max);

   FUNCTION_FINISH(HBBTV_OSCreateQueue);

   return result;
}

/**
 * @brief   Deletes the specified message queue.
 * @param   queue Queue handle
 */
void HBBTV_OSDeleteQueue (void* queue)
{
   FUNCTION_START(HBBTV_OSDeleteQueue);

   STB_OSDestroyQueue(queue);

   FUNCTION_FINISH(HBBTV_OSDeleteQueue);
}

/**
 * @brief   Reads a message from specified queue.
 * @param   queue Queue handle
 * @param   msg Pointer to the buffer where the received message will be copied
 * @param   msg_size Size of message
 * @param   timeout Timeout value (ms)
 * @return  TRUE if message arrived, FALSE if timed out.
 */
BOOLEAN HBBTV_OSReadQueue(void* queue, void* msg, U16BIT msg_size, U16BIT timeout)
{
   BOOLEAN result;

   FUNCTION_START(HBBTV_OSReadQueue);

   result = STB_OSReadQueue(queue, msg, msg_size, timeout);

   FUNCTION_FINISH(HBBTV_OSReadQueue);

   return result;
}

/**
 * @brief   Writes a message to specified queue.
 * @param   queue Queue handle
 * @param   msg Pointer to the message to be written
 * @param   msg_size Size of message
 * @param   timeout Timeout value (ms)
 * @return  TRUE if message arrived, FALSE if timed out.
 */
BOOLEAN HBBTV_OSWriteQueue(void* queue, void* msg, U16BIT msg_size, U16BIT timeout)
{
   BOOLEAN result;

   FUNCTION_START(HBBTV_OSWriteQueue);

   result = STB_OSWriteQueue(queue, msg, msg_size, timeout);

   FUNCTION_FINISH(HBBTV_OSWriteQueue);

   return result;
}
